/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: PGPSelectKeysDialog.cp,v 1.24 1999/05/14 07:24:37 wprice Exp $
____________________________________________________________________________*/

#include <LPushButton.h>
#include <LTableMonoGeometry.h>
#include <LTableArrayStorage.h>
#include <LTableMultiSelector.h>
#include <LTableSingleSelector.h>
#include <PP_Messages.h>
#include <UDrawingUtils.h>
#include <UGAColorRamp.h>
#include <URegistrar.h>
#include <UTextTraits.h>

#include "pgpMem.h"
#include "pgpErrors.h"
#include "pgpKeys.h"
#include "pgpUtilities.h"
#include "pgpClientLib.h"
#include "pgpClientPrefs.h"

#include "pflPrefTypes.h"

#include "MacErrors.h"
#include "MacCursors.h"
#include "MacQuickdraw.h"
#include "MacStrings.h"

#include "StSaveHeapZone.h"
#include "StSaveCurResFile.h"
#include "CString.h"
#include "CPGPException.h"

#include "PGPclientLibUtils.h"
#include "PGPclientLibDialogs.h"

#include "PGPSelectKeysDialog.h"


const ResIDT	Txtr_Geneva9				=	4748;

const ResIDT	ppob_SelectKeysID			=	4753;

const PaneIDT	caption_Prompt				=	1003;
const PaneIDT	table_UserIDs				=	1004;
const PaneIDT	button_SelectAll			=	1005;
const PaneIDT	button_SelectNone			=	1006;
const PaneIDT	button_OK					=	'bOK ';

const MessageT	msg_UserIDs					=	table_UserIDs;
const MessageT	msg_SelectAll				=	button_SelectAll;
const MessageT	msg_SelectNone				=	button_SelectNone;

const SInt16	kValidityColumnLeft			=	8;
const SInt16	kVerticalIndent				=	3;

const ResIDT	icsx_ValidKey	 			=	3014;
const ResIDT	icsx_InvalidKey 			=	3015;
const ResIDT	icsx_AxiomaticKey			=	3017;

const ResIDT	acur_BeachBall				=	4747;

const ResIDT	ppat_BarberPattern	 		=	4747;

const SInt16	kBarHeight					=	10;
const SInt16	kMaxValidity				=	3;

	static PGPError
RealPGPSelectKeysDialog(
	PGPContextRef			context,
	PGPSelectKeysSettings	settings,
	PGPPrefRef				clientPrefsRef,
	const char *			prompt,
	PGPKeySetRef			fromSet,							
	PGPKeySetRef			validitySet,
	PGPKeySetRef			*selectedSet)					
{
	CComboError	err;
	DialogRef	skDialog;
	
	
	RegisterClass_( CSelectKeysDialog );
	RegisterClass_( CSelectKeysTable );
	
	// Create standard Macintosh window and overlay PowerPlant onto it
	skDialog = 
			CPGPModalGrafPortView::CreateDialog( ppob_SelectKeysID );
	if( IsntNull( skDialog ) )
	{
		CSelectKeysDialog *skGrafPortView = nil;

		skGrafPortView =
				(CSelectKeysDialog *) GetWRefCon( skDialog );
		pgpAssert( IsntNull( skGrafPortView ) );
		
		err = skGrafPortView->Init( context, settings, clientPrefsRef, prompt,
									fromSet, validitySet, selectedSet);
		if( err.IsntError() )
		{
			ModalFilterUPP	filterUPP;
			MessageT		dismissMessage;
			short			itemHit = 0;
		
			InitCursor();
			
			SetPort( skDialog );
			ShowWindow( skDialog );
			SelectWindow( skDialog );
	
			skGrafPortView->Refresh();
			
			filterUPP = NewModalFilterProc(
						CPGPModalGrafPortView::ModalFilterProcHandler );
			
			skGrafPortView->SetDismissMessage( msg_Nothing );

			while( itemHit == 0 )
			{
				ModalDialog( filterUPP, &itemHit );
			}
		
			dismissMessage = skGrafPortView->GetDismissMessage();
			if( dismissMessage == msg_OK )
			{
				err = skGrafPortView->GetKeys();
			}
			else
			{
				err.pgpErr = kPGPError_UserAbort;
			}
		
			DisposeRoutineDescriptor( filterUPP );
		}
		
		delete skGrafPortView;
		DisposeDialog( skDialog );
	}
	else
	{
		err.pgpErr = kPGPError_OutOfMemory;
	}
	
	return err.ConvertToPGPError();
}

	PGPError
PGPSelectKeysDialog(
	PGPContextRef			context,
	PGPSelectKeysSettings	settings,
	const char *			prompt,
	PGPKeySetRef			fromSet,							
	PGPKeySetRef			validitySet,
	PGPKeySetRef			*selectedSet)					
{
	PGPError			err;
	PGPclientLibState	state;
	
	PGPValidatePtr( selectedSet );
	*selectedSet	= kInvalidPGPKeySetRef;
	PGPValidateParam( IsntNull( context ) );
	PGPValidateParam( IsntNull( fromSet ) );
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = RealPGPSelectKeysDialog( context, settings, state.clientPrefsRef,
					prompt, fromSet, validitySet, selectedSet );
	}
		
	ExitPGPclientLib( &state );
	
	return( err );
}

CSelectKeysDialog::CSelectKeysDialog(
	LStream *	inStream)
		: CPGPModalGrafPortView(inStream)
{
}

CSelectKeysDialog::~CSelectKeysDialog()
{
}

	CComboError
CSelectKeysDialog::Init(
	PGPContextRef			inContext,
	PGPSelectKeysSettings	settings,
	PGPPrefRef				inClientPrefsRef,
	const char *			inPrompt,
	PGPKeySetRef			inFromSet,
	PGPKeySetRef			inValiditySet,
	PGPKeySetRef *			outSelectedSet)
{
	CComboError	err;
	
	if (IsntNull( inPrompt ) )
	{
		FindPaneByID(caption_Prompt)->SetDescriptor(CString(inPrompt));
	}
	
	mSingleSelect = ( settings & kPGPSelectKeysSingleKey ) != 0;
	
	if( mSingleSelect )
	{
		FindPaneByID(button_SelectNone)->Hide();
		FindPaneByID(button_SelectAll)->Hide();
	}
	
	err = mUserIDTable->SetTableInfo( inContext, inClientPrefsRef,
							inFromSet, inValiditySet, mSingleSelect,
							outSelectedSet );
	if( err.IsntError() )
	{	
		if( settings & kPGPSelectKeysImportVariation )
		{
			Str255	buttonName;
			
			GetIndString(	buttonName, kDialogStringListResID,
							kImportButtonNameIndex );
			FindPaneByID(button_OK)->SetDescriptor( buttonName );
			mUserIDTable->SelectAllCells();
		}
		AdjustButtons();
	}
	
	return( err );
}

	CComboError
CSelectKeysDialog::GetKeys(void)
{
	pgpAssert( IsntNull( mUserIDTable ) );
	
	return( mUserIDTable->GetKeys() );
}

	void
CSelectKeysDialog::FinishCreateSelf()
{
	LPushButton *	button;
	
	CPGPModalGrafPortView::FinishCreateSelf();
	
	button = (LPushButton *) FindPaneByID(button_SelectAll);
	button->AddListener(this);

	button = (LPushButton *) FindPaneByID(button_SelectNone);
	button->AddListener(this);
	
	mUserIDTable = (CSelectKeysTable *) FindPaneByID(table_UserIDs);
	pgpAssert( IsntNull( mUserIDTable ) );
	
	mUserIDTable->AddListener(this);
	SetLatentSub( mUserIDTable );
}

	void
CSelectKeysDialog::AdjustButtons()
{
	TableIndexT	numRows;
	TableIndexT	numColumns;
	STableCell	theCell(0, 0);
	TableIndexT	numSelectedRows = 0;
	
	mUserIDTable->GetTableSize( numRows, numColumns );
	
	while( mUserIDTable->GetNextSelectedCell( theCell ) )
		++numSelectedRows;
		
	if ( numSelectedRows == 0 )
	{
		mOKButton->Disable();
		FindPaneByID(button_SelectNone)->Disable();
	}
	else
	{
		mOKButton->Enable();
		FindPaneByID(button_SelectNone)->Enable();
	}
	
	if ( numSelectedRows == numRows )
	{
		FindPaneByID(button_SelectAll)->Disable();
	}
	else
	{
		FindPaneByID(button_SelectAll)->Enable();
	}
}

	void
CSelectKeysDialog::ListenToMessage(
	MessageT	inMessage,
	void *		ioParam)
{
	switch (inMessage)
	{
		case coltable_SelectChange:
			AdjustButtons();
			break;
		case coltable_DoubleClick:
			if( mSingleSelect )
				mOKButton->SimulateHotSpotClick( kControlButtonPart );
			break;
		
		case msg_SelectAll:
			mUserIDTable->SelectAllCells();
			AdjustButtons();
			break;

		case msg_SelectNone:
			mUserIDTable->UnselectAllCells();
			AdjustButtons();
			break;
		
		default:
			CPGPModalGrafPortView::ListenToMessage( inMessage, ioParam );
			break;
	}
}

CSelectKeysTable::CSelectKeysTable(
	LStream *	inStream)
		: CColumnTable(inStream),
		mFromSet(kInvalidPGPKeySetRef),
		mBarberPixPat(nil),
		mKeyIter(kInvalidPGPKeyIterRef),
		mSelectedSet(nil)
{
}

CSelectKeysTable::~CSelectKeysTable()
{
	if(mKeyIter != kInvalidPGPKeyIterRef)
		PGPFreeKeyIter(mKeyIter);
	if(mBarberPixPat != nil)
		::DisposePixPat(mBarberPixPat);
}

	CComboError
CSelectKeysTable::AddKeysToTable(
	PGPKeySetRef	inFromSet )
{
	PGPKeyListRef		keyList;
	CComboError			err;
	
	err.pgpErr = PGPOrderKeySet( inFromSet, kPGPUserIDOrdering, &keyList);
	if( err.IsntError() )
	{
		err.pgpErr = PGPNewKeyIter(	keyList, &mKeyIter);
		if( err.IsntError() )
		{
			PGPKeyRef	key;

			while ( ( PGPKeyIterNext( mKeyIter, &key ) == kPGPError_NoErr )
					&& ( key != kInvalidPGPKeyRef ) )
			{
				InsertRows(	1, LArray::index_Last, &key, sizeof(PGPKeyRef) );
			}
		}
		
		PGPFreeKeyList( keyList );
	}
	
	return( err );
}

	CComboError
CSelectKeysTable::SetTableInfo(
	PGPContextRef	inContext,
	PGPPrefRef		inClientPrefsRef,
	PGPKeySetRef	inFromSet,
	PGPKeySetRef	inValiditySet,
	PGPBoolean		inSingleSelect,
	PGPKeySetRef *	outSelectedSet)
{
	CComboError			err;
	PGPUInt32			numKeys;

	// Make sure that there are some keys to show
	err.pgpErr = PGPCountKeys( inFromSet, &numKeys );
	if( err.IsntError() )
	{
		if( numKeys > 0 )
		{
			// Save off our sets
			mSelectedSet 	= outSelectedSet;
			mFromSet 		= inFromSet;
			
			mBarberPixPat = GetPixPat(ppat_BarberPattern);
			pgpAssert( IsntNull( mBarberPixPat ) );
		
			err.pgpErr = PGPGetPrefBoolean(	inClientPrefsRef,
									kPGPPrefDisplayMarginalValidity,
									&mShowMarginalValidity);
			if( err.IsntError() )
			{
				err.pgpErr = PGPGetPrefBoolean(	inClientPrefsRef,
										kPGPPrefMarginalIsInvalid,
										&mMarginalIsInvalid);
			}
			
			if( err.IsntError() )
			{
				err = AddKeysToTable( inFromSet );
			}
			
			// Adjust if single select
			if( err.IsntError() && ( inSingleSelect ) )
			{
				delete mTableSelector;
				SetTableSelector(new LTableSingleSelector(this));
			}

			// Propagate validity
			if( IsntNull( inValiditySet ) && err.IsntError() )
			{
				PGPKeySetRef	combinedSet = kInvalidPGPKeySetRef;
			
				// Create combined set
				err.pgpErr = PGPNewKeySet( inContext, &combinedSet );
				if( err.IsntError() )
				{
					err.pgpErr = PGPAddKeys( inFromSet, combinedSet);
					if( err.IsntError() )
					{
						err.pgpErr = PGPAddKeys( inValiditySet, combinedSet);
					}
				}

				if( err.IsntError() )
				{
					AnimatedCursorRef	cursorRef = nil;

					err.err = Get1AnimatedCursor( acur_BeachBall, &cursorRef );
					pgpAssert( err.IsntError() );

					err.pgpErr = PGPCheckKeyRingSigs( inFromSet, combinedSet,
									TRUE, EventHandler, cursorRef );

					DisposeAnimatedCursor( cursorRef );
				}

				if( err.IsntError() )
				{
					err.pgpErr = PGPPropagateTrust( combinedSet );
				}
				
				PGPFreeKeySet( combinedSet );
			}
		}
		else
		{
			err.pgpErr = kPGPError_BadParams;
		}
	}

	return( err );
}

	CComboError
CSelectKeysTable::GetKeys()
{
	CComboError		err;

	err.pgpErr = PGPNewEmptyKeySet( mFromSet, mSelectedSet );
	if( err.IsntError() )
	{
		STableCell		theCell(0, 0);

		while( GetNextSelectedCell( theCell ) )
		{
			PGPKeySetRef	tempSet;
			PGPKeyRef		key;
			UInt32			len = sizeof( key );
			PGPKeySetRef	singleKeySet = kInvalidPGPKeySetRef;

			tempSet = *mSelectedSet;
			
			GetCellData( theCell, &key, len );
			
			err.pgpErr = PGPNewSingletonKeySet( key, &singleKeySet );
			if( err.IsntError() )
			{
				err.pgpErr = PGPUnionKeySets( tempSet, singleKeySet,
										mSelectedSet);
				
				PGPFreeKeySet( singleKeySet );
			}
			
			PGPFreeKeySet( tempSet );
			
			if( err.IsError() )
				break;
		}
	}

	return( err );
}

	Boolean
CSelectKeysTable::DrawCellCustom(
	const STableCell		&inCell,
	const Rect				&inLocalRect,
	Boolean					)
{
	RGBColor				trustColor = { 52223, 52223, 65535 };
	PGPBoolean				isAxiomatic;
	PGPInt32				validity;
	UInt32					len = sizeof(PGPKeyRef);
	PGPKeyRef				key;
	Rect					barRect;
			
	GetCellData( inCell, &key, len );
	PGPGetKeyBoolean( key, kPGPKeyPropIsAxiomatic, &isAxiomatic);
	PGPGetKeyNumber( key, kPGPKeyPropValidity, &validity);
	
	barRect.left	= inLocalRect.left + kValidityColumnLeft;
	barRect.top		= inLocalRect.top + kVerticalIndent;
	barRect.right	= inLocalRect.right - kValidityColumnLeft;
	barRect.bottom	= barRect.top + kBarHeight;
	FrameRect( &barRect);
	InsetRect( &barRect, 1, 1);

	if( isAxiomatic && ( mBarberPixPat != nil) )
	{
		PenPixPat( mBarberPixPat );
		PaintRect( &barRect );
		PenNormal();
	}
	else
	{
		SInt16 		width;
		SInt16		barPercent;
		RGBColor	tempColor;
		
		switch (validity)
		{
			default:
			case kPGPValidity_Unknown:
			case kPGPValidity_Invalid:
				barPercent = 0;
				break;
			case kPGPValidity_Marginal:
				barPercent = 50;
				break;
			case kPGPValidity_Complete:
				barPercent = 100;
				break;
		}
		RGBForeColor( &trustColor );
		PaintRect( &barRect );
		width = barRect.right - barRect.left;
		barRect.right -= (width * (100 - barPercent)) / 100;
		tempColor = UGAColorRamp::GetColor(colorRamp_Gray9);
		RGBForeColor( &tempColor );
		PaintRect( &barRect );
	}

	return FALSE;
}

	Boolean
CSelectKeysTable::GetCellDrawData(
	STableCell			inCell,
	ResIDT				&iconID,
	Int16				&indentLevel,
	Str255				data,
	StyleParameter		&style )
{
	PGPError			pgpErr;
	PGPKeyRef			key;
	UInt32				len = sizeof(PGPKeyRef);
	TableIndexT			col;
	PGPInt32			algorithm;
	Boolean				custom = FALSE;
	
	indentLevel;	style;
	col = inCell.col;
	inCell.col = 1;
	GetCellData( inCell, &key, len );
	switch( col )
	{
		case 1:		// User ID
		{
			PGPBoolean			secretKey;
			PGPBoolean			expiredKey;
			PGPBoolean			revokedKey;
			PGPBoolean			disabledKey;
			PGPBoolean			splitKey;
				
			pgpErr = PGPGetKeyNumber(	key,
										kPGPKeyPropAlgID,
										&algorithm);
			PGPThrowIfPGPError_(pgpErr);

			pgpErr = PGPGetKeyBoolean(	key,
										kPGPKeyPropIsSecret,
										&secretKey);
			PGPThrowIfPGPError_(pgpErr);

			pgpErr = PGPGetKeyBoolean(	key,
										kPGPKeyPropIsRevoked,
										&revokedKey);
			PGPThrowIfPGPError_(pgpErr);

			pgpErr = PGPGetKeyBoolean(	key,
										kPGPKeyPropIsExpired,
										&expiredKey);
			PGPThrowIfPGPError_(pgpErr);
			pgpErr = PGPGetKeyBoolean(	key,
										kPGPKeyPropIsDisabled,
										&disabledKey);
			PGPThrowIfPGPError_(pgpErr);
			
			pgpErr = PGPGetKeyBoolean(	key,
										kPGPKeyPropIsSecretShared,
										&splitKey );
			PGPThrowIfPGPError_(pgpErr);

			if( algorithm == kPGPPublicKeyAlgorithm_DSA )
			{
				if( revokedKey )
					iconID = kPGPIconID_DSARevokedKey;
				else if( expiredKey )
					iconID = kPGPIconID_DSAExpiredKey;
				else if( splitKey )
					iconID = kPGPIconID_DSASplitKeyPair;
				else if( secretKey )
					iconID = kPGPIconID_DSAKeyPair;
				else if( disabledKey )
					iconID = kPGPIconID_DSADisabledKey;
				else
					iconID = kPGPIconID_DSAKey;
			}
			else
			{
				if( revokedKey )
					iconID = kPGPIconID_RSARevokedKey;
				else if( expiredKey )
					iconID = kPGPIconID_RSAExpiredKey;
				else if( splitKey )
					iconID = kPGPIconID_RSASplitKeyPair;
				else if( secretKey )
					iconID = kPGPIconID_RSAKeyPair;
				else if( disabledKey )
					iconID = kPGPIconID_RSADisabledKey;
				else
					iconID = kPGPIconID_RSAKey;
			}
			
			// Draw the UserID
			len = sizeof(Str255) - 1;
			pgpErr = PGPGetPrimaryUserIDNameBuffer(key,
				sizeof( Str255 ) - 1, (char *)&data[1], &len);
			if(pgpErr != kPGPError_BufferTooSmall)
			{
				PGPThrowIfPGPError_(pgpErr);
			}
			data[0] = len;
			break;
		}
		case 2:		// Validity
		{
			PGPBoolean	isAxiomatic;
			PGPInt32	validity;
					
			if( mShowMarginalValidity )
			{
				custom = TRUE;
				break;
			}
			pgpErr = PGPGetKeyBoolean(	key,
										kPGPKeyPropIsAxiomatic,
										&isAxiomatic);
			PGPThrowIfPGPError_(pgpErr);
			pgpErr = PGPGetKeyNumber(	key,
										kPGPKeyPropValidity,
										&validity);
			PGPThrowIfPGPError_(pgpErr);
			
			if( isAxiomatic )
				iconID = icsx_AxiomaticKey;
			else if( validity == kPGPValidity_Complete )
				iconID = icsx_ValidKey;
			else if( validity == kPGPValidity_Marginal )
			{
				if( mMarginalIsInvalid )
					iconID = icsx_InvalidKey;
				else
					iconID = icsx_ValidKey;
			}
			else
				iconID = icsx_InvalidKey;
			indentLevel = 1;
			break;
		}
		case 3:		// Creation
		{
			PGPTime			creation;
			
			pgpErr = PGPGetKeyTime(	key,
									kPGPKeyPropCreation,
									&creation);
			PGPThrowIfPGPError_(pgpErr);
			creation = PGPTimeToMacTime(creation);
			::DateString(creation, shortDate, data, nil);
			break;
		}
		case 4:		// Size
		{
			PGPInt32		bits;
			PGPSubKeyRef	subKey;		
			
			pgpErr = PGPGetKeyNumber(	key,
										kPGPKeyPropBits,
										&bits);
			PGPThrowIfPGPError_(pgpErr);
			pgpErr = PGPGetKeyNumber(	key,
										kPGPKeyPropAlgID,
										&algorithm);
			PGPThrowIfPGPError_(pgpErr);
			if(algorithm == kPGPPublicKeyAlgorithm_DSA)
			{
				PGPInt32	bits2 = 0;
				Str31		string2;
				
				GetIndString( data, kDialogStringListResID,
						kDSSDHKeySizeFormatStrIndex );
				
				PGPKeyIterSeek(mKeyIter, key);
				pgpErr = PGPKeyIterNextSubKey(mKeyIter, &subKey);
				if (IsntPGPError(pgpErr) && PGPSubKeyRefIsValid(subKey))
				{
					pgpErr = PGPGetSubKeyNumber(	subKey,
													kPGPKeyPropBits,
													&bits2);
					PGPThrowIfPGPError_(pgpErr);
				}
				if( bits2 > 0 )
				{
					::NumToString(bits2, string2);
					PrintPString( data, data, string2 );

					::NumToString(bits, string2);
					PrintPString( data, data, string2 );
				}
				else
					::NumToString(bits, data);
			}
			else
				::NumToString(bits, data);
			break;
		}
	}
	return custom;
}

	PGPError
CSelectKeysTable::EventHandler(
	PGPContextRef	context,
	PGPEvent *		event,
	PGPUserValue	userValue)
{
	(void) context;
	(void) event;
	
	if( userValue != 0 )
		AnimateCursor((AnimatedCursorRef) userValue);
	
	return kPGPError_NoErr;
}

