/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: PGPclientLibDialogs.cp,v 1.35 1999/04/04 11:12:00 wprice Exp $
____________________________________________________________________________*/

#include <string.h>

#include <Gestalt.h>
#include <LowMem.h>
#include <Sound.h>

#include <LStaticText.h>
#include <PP_Messages.h>

#include "MacStrings.h"
#include "pflPrefTypes.h"
#include "pgpKeys.h"
#include "pgpMem.h"
#include "pgpOptionList.h"
#include "pgpUserInterface.h"

#include "CPGPModalGrafPortView.h"
#include "pgpAdminPrefs.h"
#include "pgpClientPrefs.h"
#include "PGPclientLib.h"
#include "PGPclientLibDialogs.h"
#include "PGPclientLibUtils.h"
#include "PGPOpenPrefs.h"

enum
{
	kMacBinaryPopupNoneMenuItem	= 1,
	kMacBinaryPopupSmartMenuItem,
	kMacBinaryPopupYesMenuItem
};


	static MessageT
PGPUIWarningAlertInternal(
	WarningAlertType 	alertType,
	WarningAlertStyle 	alertStyle,
	ConstStr255Param	string)
{
	MessageT				dismissMessage = msg_Cancel;
	GrafPtr					savePort;
	AlertStdAlertParamRec	alertParams;
	AlertType				dialogType;
	SInt16					itemHit;
	WindowPtr				frontWindow;
	short					saveWindowKind;
	
	GetPort( &savePort );
	
	pgpClearMemory( &alertParams, sizeof( alertParams ) );
	
	alertParams.position 	= kWindowDefaultPosition;
	alertParams.filterProc	= NewModalFilterProc( 
								CPGPModalGrafPortView::ModalFilterProcHandler );
	
	switch( alertStyle )
	{
		case kWAOKStyle:
			alertParams.defaultText		= (StringPtr) kAlertDefaultOKText;
			alertParams.defaultButton	= kAlertStdAlertOKButton;
			break;
			
		case kWAOKCancelStyle:
			alertParams.defaultText		= (StringPtr) kAlertDefaultOKText;
			alertParams.cancelText		= (StringPtr) kAlertDefaultCancelText;
			alertParams.defaultButton	= kAlertStdAlertOKButton;
			alertParams.cancelButton	= kAlertStdAlertCancelButton;
			break;
			
		default:
			pgpDebugMsg("PGPUIWarningAlertInternal(): Unsupported alert style");
			break;
	}
	
	switch( alertType )
	{
		case kWANoteAlertType:
			dialogType = kAlertNoteAlert;
			break;

		case kWACautionAlertType:
			dialogType = kAlertCautionAlert;
			break;

		case kWAStopAlertType:
			dialogType = kAlertStopAlert;
			break;

		default:
			pgpDebugMsg("PGPUIWarningAlertInternal(): Unsupported alert type");
			dialogType = kAlertPlainAlert;
			break;
	}
	
	frontWindow = FrontWindow();
	if( IsntNull( frontWindow ) )
	{
		saveWindowKind = GetWindowKind( frontWindow );
		SetWindowKind( frontWindow, kApplicationWindowKind );
	}
	
	if( StandardAlert( dialogType, (StringPtr) string, NULL, &alertParams,
				&itemHit ) == noErr )
	{
		if( itemHit == kAlertStdAlertOKButton )
		{
			dismissMessage = msg_OK;
		}
	}
	
	if( IsntNull( frontWindow ) )
	{
		SetWindowKind( frontWindow, saveWindowKind );
	}
	
	DisposeRoutineDescriptor( alertParams.filterProc );
	SetPort( savePort );

	return( dismissMessage );
}

	MessageT
PGPUIWarningAlert(
	WarningAlertType 	alertType,
	WarningAlertStyle 	alertStyle,
	ConstStr255Param	baseString,
	ConstStr255Param	str1,
	ConstStr255Param	str2,
	ConstStr255Param	str3 )
{
	Str255		msg;
	
	InsertPStrings( baseString, str1, str2, str3, msg );
	
	return( PGPUIWarningAlertInternal( alertType, alertStyle, msg ) );
}


	
	MessageT
PGPUIWarningAlert(
	WarningAlertType 	alertType,
	WarningAlertStyle 	alertStyle,
	ResID				stringListResID,
	short				stringListIndex,
	ConstStr255Param	str1,
	ConstStr255Param	str2 ,
	ConstStr255Param	str3 )
{
	Str255		baseString;
	Str255		msg;
	
	GetIndString( baseString, stringListResID, stringListIndex );
	InsertPStrings( baseString, str1, str2, str3, msg );
	
	return( PGPUIWarningAlertInternal( alertType, alertStyle, msg ) );
}

	static PGPOptionListRef
NewCheckboxOption(
	PGPContextRef		context,
	PGPUInt32			itemID,
	ResID				strListResID,
	short				titleStrListIndex,
	short				descStrListIndex,
	Boolean				initialValue,
	PGPOptionListRef	subOptions,
	PGPUInt32			*resultPtr )
{
	char	title[256];
	char	desc[256];

	GetIndCString( title, strListResID, titleStrListIndex );
	
	if( descStrListIndex != 0 )
	{
		GetIndCString( desc, strListResID, descStrListIndex );
	}
	else
	{
		*desc = 0;
	}
	
	return( PGPOUICheckbox( context, itemID, title, desc, initialValue,
					resultPtr, subOptions != NULL ? subOptions :
					PGPONullOption( context ), PGPOLastOption( context ) ) );
}

	static char *
GetIndCStringAlloc(
	ResID 	stringListResID,
	short 	stringListIndex)
{
	char		str[256];
	char		*ptr;
	PGPUInt32	strLength;
	
	GetIndCString( str, stringListResID, stringListIndex );
	strLength = strlen( str );
	
	ptr = (char *) pgpAlloc( strLength + 1 );
	if( IsntNull( ptr ) )
	{
		pgpCopyMemory( str, ptr, strLength + 1 );
	}
	
	return( ptr );
}

	static PGPOptionListRef
NewPopupListOption(
	PGPContextRef	context,
	PGPUInt32		itemID,
	ResID			strListResID,
	short			titleStrListIndex,
	short			descStrListIndex,
	short			firstItemStrListIndex,
	short			lastItemStrListIndex,
	Boolean			initialValue,
	PGPUInt32		*resultPtr )
{
	char				title[256];
	char				desc[256];
	char				*listItems[32];
	PGPUInt32			numListItems;
	PGPUInt32 			listIndex;
	PGPOptionListRef	optionList;
	
	numListItems = lastItemStrListIndex - firstItemStrListIndex + 1;
	pgpAssert( numListItems <= 32 );
	
	GetIndCString( title, strListResID, titleStrListIndex );
	
	if( descStrListIndex != 0 )
	{
		GetIndCString( desc, strListResID, descStrListIndex );
	}
	else
	{
		*desc = 0;
	}
	
	pgpClearMemory( listItems, sizeof( listItems ) );
	
	for( listIndex = 0; listIndex < numListItems; listIndex++ )
	{
		listItems[listIndex] = GetIndCStringAlloc( strListResID,
					firstItemStrListIndex + listIndex );
		pgpAssert( IsntNull( listItems[listIndex] ) );
	}
	
	optionList = PGPOUIPopupList( context, itemID, title, desc, numListItems,
					listItems, initialValue, resultPtr,
					PGPOLastOption( context ) );
					
	for( listIndex = 0; listIndex < numListItems; listIndex++ )
	{
		/* PGPOUIPopupList() copied our strings. Free our copies */
		if( IsntNull( listItems[listIndex] ) )
			pgpFree( listItems[listIndex] );
	}

	return( optionList );
}

	static PGPOptionListRef
TextOutputCheckboxOptionList(
	PGPContextRef 	context,
	PGPUInt32		itemID,
	PGPBoolean		checked,
	PGPUInt32		*resultPtr)
{
	return( NewCheckboxOption( context, itemID, kOptionDialogStringListResID,
					kTextOutputTitleStrIndex, kTextOutputDescStrIndex,
					checked, NULL, resultPtr ) );
}

	static PGPOptionListRef
DetachedSignatureCheckboxOptionList(
	PGPContextRef 	context,
	PGPUInt32		itemID,
	PGPBoolean		checked,
	PGPUInt32		*resultPtr)
{
	return( NewCheckboxOption( context, itemID, kOptionDialogStringListResID,
					kDetachedSignatureTitleStrIndex,
					kDetachedSignatureDescStrIndex, checked, NULL, resultPtr ) );
}

	static PGPOptionListRef
SDACheckboxOptionList(
	PGPContextRef 	context,
	PGPUInt32		itemID,
	PGPBoolean		checked,
	PGPUInt32		*resultPtr)
{
	return( NewCheckboxOption( context, itemID, kOptionDialogStringListResID,
					kSDATitleStrIndex, kSDADescStrIndex,
					checked, NULL, resultPtr ) );
}

	static PGPOptionListRef
ConvEncryptCheckboxOptionList(
	PGPContextRef 	context,
	PGPUInt32		itemID,
	PGPBoolean		checked,
	PGPUInt32		*resultPtr,
	PGPUInt32		showSDA,
	PGPBoolean		sdaChecked,
	PGPUInt32		*sdaResultPtr)
{
	return( NewCheckboxOption( context, itemID, kOptionDialogStringListResID,
					kConvEncryptTitleStrIndex, kConvEncryptDescStrIndex,
					checked, showSDA ? SDACheckboxOptionList( context, itemID + 1,
					sdaChecked, sdaResultPtr ) : NULL, resultPtr ) );
}

	static PGPOptionListRef
FYEOCheckboxOptionList(
	PGPContextRef 	context,
	PGPUInt32		itemID,
	PGPBoolean		checked,
	PGPUInt32		*resultPtr)
{
	return( NewCheckboxOption( context, itemID, kOptionDialogStringListResID,
					kFYEOTitleStrIndex, kFYEODescStrIndex,
					checked, NULL, resultPtr ) );
}

	static PGPOptionListRef
WipeOriginalCheckboxOptionList(
	PGPContextRef 	context,
	PGPUInt32		itemID,
	PGPBoolean		checked,
	PGPUInt32		*resultPtr)
{
	return( NewCheckboxOption( context, itemID, kOptionDialogStringListResID,
					kWipeOriginalTitleStrIndex, kWipeOriginalDescStrIndex,
					checked, NULL, resultPtr ) );
}

	static PGPOptionListRef
MacBinaryPopupOptionList(
	PGPContextRef 	context,
	PGPUInt32		itemID,
	PGPUInt32		initialValue,
	PGPUInt32		*resultPtr)
{
	return( NewPopupListOption( context, itemID, kOptionDialogStringListResID,
					kMacBinaryTitleStrIndex, kMacBinaryDescStrIndex,
					kFirstMacBinaryMenuItemStrIndex,
					kLastMacBinaryMenuItemStrIndex,
					initialValue, resultPtr ) );
}

	static PGPUInt32
GetMacBinaryPopupInitialValue(
	PGPGetPassphraseSettings	defaultSettings,
	PGPPrefRef					clientPrefsRef)
{
	PGPUInt32	macBinaryMenuItem;
	
	/* MacBinary always on by default */
	macBinaryMenuItem = kMacBinaryPopupYesMenuItem;

	if ( ( defaultSettings & kPGPUISettingsNoMacBinary ) != 0 )
	{
		macBinaryMenuItem = kMacBinaryPopupNoneMenuItem;
	}
	else if ( ( defaultSettings & kPGPUISettingsSmartMacBinary ) != 0 )
	{
		macBinaryMenuItem = kMacBinaryPopupSmartMenuItem;
	}
	else if ( ( defaultSettings & kPGPUISettingsForceMacBinary ) != 0 )
	{
		macBinaryMenuItem = kMacBinaryPopupYesMenuItem;
	}
	else
	{
		PGPUInt32	macBinaryPref;
		
		if( IsntPGPError( PGPGetPrefNumber( clientPrefsRef,
				kPGPPrefMacBinaryDefault, &macBinaryPref ) ) )
		{
			switch( macBinaryPref )
			{
				case kPGPPrefMacBinaryOff:
					macBinaryMenuItem = kMacBinaryPopupNoneMenuItem;
					break;
				
				case kPGPPrefMacBinarySmart:
					macBinaryMenuItem = kMacBinaryPopupSmartMenuItem;
					break;
			}
		}
	}
	
	return( macBinaryMenuItem );
}
	
	static PGPError
pgpClientSigningPassphraseDialog(
	PGPContextRef				context,
	PGPPrefRef					clientPrefsRef,				
	PGPKeySetRef				allKeys,				
	const char 					*prompt,				
	PGPGetPassphraseOptions		dialogOptions,			
	PGPGetPassphraseSettings	defaultSettings,		
	PGPKeyRef					defaultSigningKey,		
	char						passphrase[256],
	PGPGetPassphraseSettings	*userSettings,			
	PGPKeyRef 					*actualSigningKey)
{
	PGPError			err;
	PGPOptionListRef	optionList = kInvalidPGPOptionListRef;
	char				*allocatedPassphrase = NULL;
	
	*passphrase = 0;
	
	err = PGPBuildOptionList( context, &optionList,
				PGPOUIOutputPassphrase( context, &allocatedPassphrase ),
				PGPOLastOption( context ) );
	if( IsntPGPError( err ) )
	{
		PGPUInt32	detachedSignature	= 0;
		PGPUInt32	textOutput			= 0;
		PGPUInt32	macBinary			= 0;
		Boolean		haveFileOptions 	= FALSE;

		if( IsntNull( prompt ) )
		{
			err = PGPAppendOptionList( optionList,
						PGPOUIDialogPrompt( context, prompt ),
						PGPOLastOption( context ) );
		}
		
		if( IsntPGPError( err ) &&
			( dialogOptions & kPGPGetPassphraseOptionsHideFileOptions ) == 0 )
		{
			/* Setup options dialog */
			
			haveFileOptions = TRUE;

			if( (defaultSettings & kPGPGetPassphraseSettingsDetachedSig) != 0 )
				detachedSignature = 1;

			if( (defaultSettings & kPGPUISettingsTextOutput) != 0 )
				textOutput = 1;
			
			macBinary = GetMacBinaryPopupInitialValue( defaultSettings,
								clientPrefsRef );
								
			err = PGPAppendOptionList( optionList,
						PGPOUIDialogOptions( context,
							DetachedSignatureCheckboxOptionList( context, 1,
								detachedSignature, &detachedSignature ),
							TextOutputCheckboxOptionList( context, 2,
								textOutput, &textOutput ),
							MacBinaryPopupOptionList( context, 3, macBinary,
								&macBinary ),
							PGPOLastOption( context ) ),
						PGPOLastOption( context ) );
		}

		if( IsntPGPError( err ) )
		{
			err = PGPSigningPassphraseDialog( context, allKeys,
						actualSigningKey,
						optionList,
						PGPOUIDefaultKey( context, defaultSigningKey ),
						PGPOLastOption( context ) );
		}

		if( haveFileOptions && ( IsntPGPError( err ) ||
				err == kPGPError_KeyUnusableForSignature ) )
		{
			*userSettings = 0;
			
			if( detachedSignature != 0 )
				*userSettings |= kPGPGetPassphraseSettingsDetachedSig;
				
			if( textOutput != 0 )
				*userSettings |= kPGPUISettingsTextOutput;
				
			if( macBinary == kMacBinaryPopupNoneMenuItem )
			{	
				*userSettings |= kPGPUISettingsNoMacBinary;
			}
			else if( macBinary == kMacBinaryPopupSmartMenuItem )
			{
				*userSettings |= kPGPUISettingsSmartMacBinary;
			}
			else
			{
				*userSettings |= kPGPUISettingsForceMacBinary;
			}
		}
		
		CopyCString( allocatedPassphrase, passphrase );
		
		PGPFreeOptionList( optionList );
	}
	
	if( IsntNull( allocatedPassphrase ) )
		PGPFreeData( allocatedPassphrase );
		
	return( err );
}

	PGPError
PGPClientSigningPassphraseDialog(
	PGPContextRef				context,				
	PGPKeySetRef				allKeys,				
	const char 					*prompt,				
	PGPGetPassphraseOptions		dialogOptions,			
	PGPGetPassphraseSettings	defaultSettings,		
	PGPKeyRef					defaultSigningKey,		
	char						passphrase[256],
	PGPGetPassphraseSettings	*userSettings,			
	PGPKeyRef 					*actualSigningKey)
{
	PGPError			err;
	PGPclientLibState	state;
		
	if ( IsntNull( passphrase ) )
		*passphrase	= 0;
	if ( IsntNull( actualSigningKey ) )
		*actualSigningKey = NULL;
	if ( IsntNull( userSettings ) )
		pgpClearMemory( userSettings, sizeof( *userSettings ) );
		
	PGPValidatePtr( passphrase );
	pgpClearMemory( passphrase, 256 );
	PGPValidatePtr( userSettings );
	PGPValidatePtr( actualSigningKey );
	
	AssertAddrNullOrValid( prompt, uchar, "PGPClientSigningPassphraseDialog" );
	AssertAddrNullOrValid( defaultSigningKey, VoidAlign,
				"PGPClientSigningPassphraseDialog" );
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = pgpClientSigningPassphraseDialog( context, state.clientPrefsRef,
						allKeys, prompt, dialogOptions, defaultSettings,
						defaultSigningKey, passphrase, userSettings,
						actualSigningKey );
	}
	
	ExitPGPclientLib( &state );
	
	return( err );
}

#if PGP_BUSINESS_SECURITY

	static PGPError
SetupOutgoingADK(
	PGPContextRef		context,
	PGPOptionListRef	recipientOptions,
	PGPPrefRef			adminPrefsRef,
	PGPKeySetRef		allKeys)
{
	PGPError	err = kPGPError_NoErr;
	PGPBoolean	haveOutgoingADK;
	
	err = PGPGetPrefBoolean( adminPrefsRef, kPGPPrefUseOutgoingADK,
						&haveOutgoingADK );
	if( IsntPGPError( err ) && haveOutgoingADK )
	{
		PGPByte		*adKeyID;
		PGPSize		adKeyIDSize;
		PGPBoolean	enforceOutgoingADK;
		PGPKeyRef	outgoingADK = kInvalidPGPKeyRef;
		
		if( IsPGPError( PGPGetPrefBoolean( adminPrefsRef,
					kPGPPrefEnforceOutgoingADK, &enforceOutgoingADK ) ) )
		{
			enforceOutgoingADK = TRUE;
		}
		
		err = PGPGetPrefData( adminPrefsRef, kPGPPrefOutgoingADKID,
						&adKeyIDSize, &adKeyID );
		if( IsntPGPError( err ) )
		{
			PGPKeyID	keyID;
			PGPUInt32	outADKKeyAlgorithm;
			
			err = PGPGetPrefNumber( adminPrefsRef,
						kPGPPrefOutADKPublicKeyAlgorithm,
						&outADKKeyAlgorithm );
			if( IsntPGPError( err ) )
			{
				err = PGPImportKeyID( adKeyID, &keyID );
				
				if( IsntPGPError( err ) )
				{
					err = PGPGetKeyByKeyID( allKeys, &keyID,
						(PGPPublicKeyAlgorithm) outADKKeyAlgorithm,
						&outgoingADK );
				}
			}
			
			(void) PGPDisposePrefData( adminPrefsRef, adKeyID);
		}
		
		if( IsntPGPError( err ) &&
			PGPKeyRefIsValid( outgoingADK ) )
		{
			PGPRecipientSpec	spec;
		
			pgpClearMemory( &spec, sizeof( spec ) );
			
			spec.type	= kPGPRecipientSpecType_Key;
			spec.locked	= enforceOutgoingADK;
			spec.u.key	= outgoingADK;
			
			err = PGPAppendOptionList( recipientOptions,
						PGPOUIDefaultRecipients( context, 1, &spec ),
						PGPOLastOption( context ) );
		}
		else
		{
			if( enforceOutgoingADK )
			{
				SysBeep( 1 );
				
 				PGPUIWarningAlert( kWACautionAlertType,
 						kWAOKStyle,
						kRecipientDialogStringListResID,
						kCorporateADKDisabledOrMissingStrIndex );
				
				err = kPGPError_UserAbort;
			}
			else
			{
				err = kPGPError_NoErr;
			}
		}
	}
	
	return( err );
}

	static PGPError
SetupAdminParams(
	PGPContextRef		context,
	PGPOptionListRef	recipientOptions,
	PGPPrefRef			adminPrefsRef,
	PGPKeySetRef		allKeys)
{
	PGPError			err = kPGPError_NoErr;
	
	pgpAssert( PGPPrefRefIsValid( adminPrefsRef ) );
	
	err = SetupOutgoingADK( context, recipientOptions, adminPrefsRef,
				allKeys );
	if( IsPGPError( err ) )
	{
	
	}
	
	return( err );
}

	static PGPError
HaveUnsignedRecipient(
	PGPKeySetRef	allKeys,
	PGPKeySetRef	recipients,
	PGPKeyRef		corporateKey,
	PGPBoolean		*haveUnsignedKeyPtr )
{
	PGPError		err;
	PGPKeyListRef	keyList;
	PGPBoolean		haveUnsignedKey = FALSE;
		
	err = PGPOrderKeySet( recipients, kPGPAnyOrdering, &keyList );
	if( IsntPGPError( err ) )
	{
		PGPKeyIterRef	iter;

		err = PGPNewKeyIter( keyList, &iter );
		if( IsntPGPError( err ) )
		{
			PGPKeyRef		key;
			
			err = PGPKeyIterNext( iter, &key );
			while( IsntPGPError( err ) && ! haveUnsignedKey )
			{
				PGPUserIDRef	userID;
				PGPBoolean		signedByCorporateKey = FALSE;
				
				/* check all user ids for a corporate sig */
				/* only one need be signed to qualify */
				
				err = PGPKeyIterNextUserID( iter, &userID );
				while( IsntPGPError( err ) && ! signedByCorporateKey )
				{
					PGPSigRef	sig;
					
					err = PGPKeyIterNextUIDSig( iter, &sig );
					while( IsntPGPError( err ) && ! signedByCorporateKey )
					{
						PGPKeyRef	certifierKey;
					
						if( IsntPGPError( PGPGetSigCertifierKey( sig,
								allKeys, &certifierKey ) ) )
						{
							if( PGPCompareKeys( certifierKey, corporateKey,
									kPGPKeyIDOrdering ) == 0 )
							{
								signedByCorporateKey = TRUE;
								break;
							}
						}
					
						if( IsntPGPError( err ) )
							err = PGPKeyIterNextUIDSig( iter, &sig );
					}
					
					if ( err == kPGPError_EndOfIteration )
						err	= kPGPError_NoErr;

					if( IsntPGPError( err ) )
						err = PGPKeyIterNextUserID( iter, &userID );
				}

				if ( err == kPGPError_EndOfIteration )
					err	= kPGPError_NoErr;
				
				if( ! signedByCorporateKey )
					haveUnsignedKey = TRUE;
					
				if( IsntPGPError( err ) )
					err = PGPKeyIterNext( iter, &key );
			}
			
			if ( err == kPGPError_EndOfIteration )
				err	= kPGPError_NoErr;
				
			PGPFreeKeyIter( iter );
		}
		
		PGPFreeKeyList( keyList );
	}
	
	*haveUnsignedKeyPtr = haveUnsignedKey;
	
	return( err );
}

#endif	// PGP_BUSINESS_SECURITY

	static PGPError
pgpClientRecipientDialog(
	PGPContextRef			context,					
	PGPPrefRef				clientPrefsRef,				
	PGPPrefRef				adminPrefsRef,				
	PGPtlsContextRef		tlsContext,
	PGPKeySetRef			allKeys,					
	PGPUInt32				numDefaultRecipients,		
	const PGPRecipientSpec	*defaultRecipients,			
	PGPRecipientOptions		dialogOptions,				
	PGPRecipientSettings	defaultSettings,
	PGPRecipientSettings	*userSettings,			
	PGPKeySetRef			*actualRecipients,			
	PGPKeySetRef			*newKeys)
{
	PGPError						err = kPGPError_NoErr;
	PGPOptionListRef				recipientOptions;
	PGPBoolean						showMarginalValidity;
	PGPBoolean						marginalIsInvalid;
	PGPBoolean						forceShowDialog;
	StPGPPreserveKeyServerStorage	keyserverStorage;

	
	(void) adminPrefsRef;
	
	if( IsntNull( newKeys ) )
		*newKeys = kInvalidPGPKeySetRef;
		
	if( ( dialogOptions & kPGPRecipientOptionsAlwaysShowDialog ) != 0 )
	{
		forceShowDialog = TRUE;
	}
	else
	{
		forceShowDialog = ( numDefaultRecipients == 0 );;
	}
	
	(void) PGPGetPrefBoolean( clientPrefsRef, kPGPPrefDisplayMarginalValidity,
				&showMarginalValidity);
	(void) PGPGetPrefBoolean( clientPrefsRef, kPGPPrefMarginalIsInvalid,
				&marginalIsInvalid);

	err = PGPBuildOptionList( context, &recipientOptions,
				PGPOUIDisplayMarginalValidity( context, showMarginalValidity ),
				PGPOUIIgnoreMarginalValidity( context, marginalIsInvalid ),
				PGPOLastOption( context ) );
	if( IsntPGPError( err ) )
	{
		PGPBoolean			allowConvEncrypt 	= TRUE;
		PGPKeyServerEntry	*ksEntries 			= NULL;
		PGPKeyServerSpec	*serverList 		= NULL;
		PGPUInt32			numKSEntries 		= 0;
		PGPUInt32			serverIndex;
		
	#if PGP_BUSINESS_SECURITY
		PGPKeyRef			corporateKeyRef 		= kInvalidPGPKeyRef;
		PGPBoolean			warnNotSignedByCorpKey	= FALSE;
	#endif
	
		PGPAdditionalRecipientRequestEnforcement	arrEnforcement;
		
		arrEnforcement = kPGPARREnforcement_Warn;
		
		err = PGPCreateKeyServerPath( clientPrefsRef, "", &ksEntries,
					&numKSEntries);
		if( IsntPGPError( err ) && IsntNull( ksEntries ) && numKSEntries > 0 )
		{
			serverList = (PGPKeyServerSpec *) PGPNewData(
								PGPGetContextMemoryMgr( context ),
								numKSEntries * sizeof( *serverList ),
								kPGPMemoryMgrFlags_Clear );
			if( IsntNull( serverList ) )
			{
				for( serverIndex = 0; serverIndex < numKSEntries;
							serverIndex++ )
				{
					err = PGPNewKeyServerFromHostName( context,
											ksEntries[serverIndex].serverDNS,
											ksEntries[serverIndex].serverPort,
											ksEntries[serverIndex].protocol,
											kPGPKeyServerAccessType_Normal,
											kPGPKeyServerKeySpace_Normal,
											&serverList[serverIndex].server );
					if( IsPGPError( err ) )
						break;
					
					serverList[serverIndex].serverName =
							ksEntries[serverIndex].serverDNS;
				}
				
				if( IsntPGPError( err ) )
				{
					PGPBoolean	searchForKeys = FALSE;
					
					(void) PGPGetPrefBoolean( clientPrefsRef,
								kPGPPrefKeyServerSyncUnknownKeys, &searchForKeys );
					err = PGPAppendOptionList( recipientOptions,
								PGPOUIKeyServerUpdateParams( context,
									numKSEntries, serverList, tlsContext,
									searchForKeys, newKeys, PGPOLastOption( context ) ),
								PGPOLastOption( context ) );
				}
			}
			else
			{
				err = kPGPError_OutOfMemory;
			}
		}

		if( IsntPGPError( err ) && numDefaultRecipients != 0 )
		{
			err = PGPAppendOptionList( recipientOptions,
					PGPOUIDefaultRecipients( context, numDefaultRecipients,
						defaultRecipients ),
					PGPOLastOption( context ) );
		}

		if( IsntPGPError( err ) )
		{
			FSSpec	fileSpec;
			
			if( IsntPGPError( PGPGetGroupsFileFSSpec( &fileSpec ) ) )
			{
				PGPGroupSetRef	groupSet;
				
				if ( IsntPGPError( PGPNewGroupSetFromFSSpec( context,
							&fileSpec, &groupSet) ) )
				{
					err = PGPAppendOptionList( recipientOptions,
								PGPOUIRecipientGroups( context, groupSet ),
								PGPOLastOption( context ) );
								
					(void) PGPFreeGroupSet( groupSet );
				}
			}
		}
		
		// Add default key, if set to do so.
		if( IsntPGPError( err ) )
		{
			PGPBoolean	encryptToSelf;
			
			err = PGPGetPrefBoolean( clientPrefsRef, kPGPPrefEncryptToSelf,
						&encryptToSelf );
			if( IsntPGPError( err ) && encryptToSelf )
			{
				PGPKeyRef	defaultKey;
				
				if( IsntPGPError( PGPGetDefaultPrivateKey( allKeys,
									&defaultKey ) ) )
				{
					PGPRecipientSpec	spec;
				
					pgpClearMemory( &spec, sizeof( spec ) );
					
					spec.type	= kPGPRecipientSpecType_Key;
					spec.u.key	= defaultKey;
					
					err = PGPAppendOptionList( recipientOptions,
								PGPOUIDefaultRecipients( context, 1, &spec ),
								PGPOLastOption( context ) );
				}
				else
				{
					SysBeep( 1 );
					
					PGPUIWarningAlert( kWACautionAlertType, kWAOKStyle,
						kRecipientDialogStringListResID,
						kNoDefaultPrivateKeyFoundStrIndex );
				}
			}
		}
			
	#if PGP_BUSINESS_SECURITY
		if( IsntPGPError( err ) && PGPPrefRefIsValid( adminPrefsRef ) )
		{
			PGPBoolean	enforceRemoteADKClass;
			
			if( IsPGPError( PGPGetPrefBoolean( adminPrefsRef,
								kPGPPrefAllowConventionalEncryption,
								&allowConvEncrypt ) ) )
			{
				allowConvEncrypt = FALSE;
			}

			if( IsntPGPError( PGPGetPrefBoolean( adminPrefsRef,
								kPGPPrefEnforceRemoteADKClass,
								&enforceRemoteADKClass ) ) &&
				enforceRemoteADKClass )
			{
				arrEnforcement = kPGPARREnforcement_Strict;
			}
			
			err = SetupAdminParams( context, recipientOptions, adminPrefsRef,
							allKeys );
			if( IsntPGPError( err ) )
			{
				err = PGPGetPrefBoolean( adminPrefsRef,
							kPGPPrefWarnNotCertByCorp,
							&warnNotSignedByCorpKey );
				if( IsntPGPError( err ) && warnNotSignedByCorpKey )
				{
					PGPByte		*corpKeyID;
					PGPSize		corpKeyIDSize;
					
					err = PGPGetPrefData( adminPrefsRef,
								kPGPPrefCorpKeyID, &corpKeyIDSize,
								&corpKeyID );
					if( IsntPGPError( err ) )
					{
						PGPUInt32	corpKeyAlgorithm;
						
						err = PGPGetPrefNumber( adminPrefsRef,
									kPGPPrefCorpKeyPublicKeyAlgorithm,
									&corpKeyAlgorithm );
						if( IsntPGPError( err ) )
						{
							PGPKeyID	keyID;
							
							err = PGPImportKeyID( corpKeyID, &keyID );
							
							if( IsntPGPError( err ) )
							{
								err = PGPGetKeyByKeyID ( allKeys, &keyID,
											(PGPPublicKeyAlgorithm)
											corpKeyAlgorithm,
											&corporateKeyRef );
							}
						}

						PGPDisposePrefData( adminPrefsRef, corpKeyID );
					}
					
					if( IsPGPError( err ) ||
						! PGPKeyRefIsValid( corporateKeyRef ) )
					{
						/*
						** Tell the user that we don't have the
						** corporate key and abort the dialog.
						*/
						
						SysBeep( 1 );
						
						PGPUIWarningAlert( kWAStopAlertType, kWAOKStyle,
								kRecipientDialogStringListResID,
								kCorporateSigningKeyMissingStrIndex );
						
						err = kPGPError_UserAbort;
					}
				}
			}
		}
	#endif // PGP_BUSINESS_SECURITY
		
		if( IsntPGPError( err ) )
		{
			PGPUInt32	conventionalEncrypt	= 0;
			PGPUInt32	textOutput			= 0;
			PGPUInt32	macBinary			= 0;
			PGPUInt32	wipeOriginal		= 0;
			PGPUInt32	showFYEO			= 0;
			PGPUInt32	fYEO 				= 0;
			PGPUInt32	sda					= 0;
			Boolean		showSDA				= FALSE;
			Boolean		haveFileOptions 	= FALSE;
			
			if( allowConvEncrypt )
			{
				if( (defaultSettings & kPGPRecipientSettingsConvEncrypt) != 0 )
					conventionalEncrypt = 1;
					
				if( ( dialogOptions & kPGPRecipientOptionsShowSDA ) != 0 )
				{
					showSDA = TRUE;
					
					if( (defaultSettings & kPGPRecipientSettingsSDA) != 0 )
						sda = 1;
				}
			}
			
			if( ( dialogOptions & kPGPRecipientOptionsShowFYEO ) != 0 )
			{
				showFYEO = 1;
				if( (defaultSettings & kPGPRecipientSettingsFYEO) != 0 )
					fYEO = 1;
			}

			if( ( dialogOptions & kPGPRecipientOptionsHideFileOptions ) == 0 )
			{
				/* Setup file options */
				
				haveFileOptions = TRUE;

				if( (defaultSettings & kPGPUISettingsTextOutput) != 0 )
					textOutput = 1;
				
				if( (defaultSettings & kPGPRecipientSettingsWipeOriginal) != 0 )
					wipeOriginal = 1;

				macBinary = GetMacBinaryPopupInitialValue( defaultSettings,
									clientPrefsRef );
									
				err = PGPAppendOptionList( recipientOptions,
							PGPOUIDialogOptions( context,
								allowConvEncrypt ? ConvEncryptCheckboxOptionList(
									context, 1, conventionalEncrypt, &conventionalEncrypt,
									showSDA, sda, &sda ) :
									PGPONullOption( context ),
								TextOutputCheckboxOptionList( context, 3,
									textOutput, &textOutput ),
								MacBinaryPopupOptionList( context, 4, macBinary,
									&macBinary ),
								WipeOriginalCheckboxOptionList( context, 5,
									wipeOriginal, &wipeOriginal ),
								(showFYEO) ? FYEOCheckboxOptionList( context, 6, fYEO, &fYEO ) :
									PGPONullOption( context ),
								PGPOLastOption( context ) ),
							PGPOLastOption( context ) );
			}
			else
			{

				err = PGPAppendOptionList( recipientOptions,
							PGPOUIDialogOptions( context,
								allowConvEncrypt ? ConvEncryptCheckboxOptionList(
									context, 1, conventionalEncrypt, &conventionalEncrypt, FALSE,
									0, NULL ) :
									PGPONullOption( context ),
								(showFYEO) ? FYEOCheckboxOptionList( context, 2, fYEO, &fYEO ) :
									PGPONullOption( context ),
								PGPOLastOption( context ) ),
							PGPOLastOption( context ) );
			}
			
			if( IsntPGPError( err ) )
			{
				PGPBoolean	warnOnADKs;
				
				err = PGPGetPrefBoolean( clientPrefsRef, kPGPPrefWarnOnADK,
							&warnOnADKs );
				if( IsntPGPError( err ) )
				{			
					err = PGPRecipientDialog( context, allKeys,
								forceShowDialog, actualRecipients,
								recipientOptions,
								PGPOUIEnforceAdditionalRecipientRequests(
									context, arrEnforcement, warnOnADKs ),
								PGPOLastOption( context ) );
				}
				
				/*
				** Tell the user if there are no keys selected and no
				** conventional encryption.
				*/
				
				if( sda != 0 )
					conventionalEncrypt = 1;
					
				if( IsntPGPError( err ) && conventionalEncrypt == 0 )
				{
					PGPUInt32	numKeys;
					
					err = PGPCountKeys( *actualRecipients, &numKeys );
					if( IsntPGPError( err ) && numKeys == 0 )
					{
						SysBeep( 1 );
						
						PGPUIWarningAlert( kWAStopAlertType,
 									kWAOKStyle,
									kRecipientDialogStringListResID,
									kMustSelectKeyOrConvEncryptStrIndex );

						err = kPGPError_UserAbort;
					}
				}
				
			#if PGP_BUSINESS_SECURITY
				if( IsntPGPError( err ) && warnNotSignedByCorpKey && conventionalEncrypt != 0 )
				{
					PGPBoolean	haveUnsignedKey = FALSE;
					
					pgpAssert( PGPKeyRefIsValid( corporateKeyRef ) );

					err = HaveUnsignedRecipient( allKeys,
								*actualRecipients, corporateKeyRef,
								&haveUnsignedKey );
					if( IsntPGPError( err ) && haveUnsignedKey )
					{
						MessageT	dialogMessage;
						
						SysBeep( 1 );
						
						dialogMessage = PGPUIWarningAlert(
										kWACautionAlertType,
										kWAOKCancelStyle,
										kRecipientDialogStringListResID,
										kKeysNotSignedByCorporateKeyStrIndex );
						if( dialogMessage == msg_Cancel )
							err = kPGPError_UserAbort;
					}
				}
			#endif
			
				if( IsntPGPError( err ) )
				{
					*userSettings = 0;
					
					if( conventionalEncrypt != 0 )
					{
						*userSettings |= kPGPRecipientSettingsConvEncrypt;
						
						if( PGPKeySetRefIsValid( *actualRecipients ) )
						{
							PGPFreeKeySet( *actualRecipients );
							*actualRecipients = kInvalidPGPKeySetRef;
						}
					}
						
					if( textOutput != 0 )
						*userSettings |= kPGPUISettingsTextOutput;

					if( sda != 0 )
						*userSettings |= kPGPRecipientSettingsSDA;

					if( wipeOriginal != 0 )
						*userSettings |= kPGPRecipientSettingsWipeOriginal;
						
					if( fYEO != 0 )
						*userSettings |= kPGPRecipientSettingsFYEO;

					if( macBinary == kMacBinaryPopupNoneMenuItem )
					{	
						*userSettings |= kPGPUISettingsNoMacBinary;
					}
					else if( macBinary == kMacBinaryPopupSmartMenuItem )
					{
						*userSettings |= kPGPUISettingsSmartMacBinary;
					}
					else
					{
						*userSettings |= kPGPUISettingsForceMacBinary;
					}
				}

				if( IsPGPError( err ) )
				{
					if( PGPKeySetRefIsValid( *actualRecipients ) )
					{
						PGPFreeKeySet( *actualRecipients );
						*actualRecipients = kInvalidPGPKeySetRef;
					}
					
					if( IsntNull( newKeys ) && PGPKeySetRefIsValid( *newKeys ) )
					{
						PGPFreeKeySet( *newKeys );
						*newKeys = kInvalidPGPKeySetRef;
					}
				}
			}
		}
		
		if( IsntNull( ksEntries ) )
			PGPDisposeKeyServerPath( ksEntries );

		if( IsntNull( serverList ) )
		{
			for( serverIndex = 0; serverIndex < numKSEntries; serverIndex++ )
			{
				if( PGPKeyServerRefIsValid( serverList[serverIndex].server ) )
					PGPFreeKeyServer( serverList[serverIndex].server );
			}

			PGPFreeData( serverList );
		}
		
		PGPFreeOptionList( recipientOptions );
	}
		
	return( err );
}

	PGPError
PGPClientRecipientDialog(
	PGPContextRef			context,					
	PGPtlsContextRef		tlsContext,
	PGPKeySetRef			allKeys,					
	PGPUInt32				numDefaultRecipients,		
	const PGPRecipientSpec	*defaultRecipients,			
	PGPRecipientOptions		dialogOptions,				
	PGPRecipientSettings	defaultSettings,
	PGPRecipientSettings	*userSettings,			
	PGPKeySetRef			*actualRecipients,			
	PGPKeySetRef			*newKeys)
{
	PGPError			err;
	PGPclientLibState	state;
		
	if ( IsntNull( userSettings ) )
		*userSettings = 0;
	if ( IsntNull( actualRecipients ) )
		*actualRecipients = kInvalidPGPKeySetRef;
	if ( IsntNull( newKeys ) )
		*newKeys = kInvalidPGPKeySetRef;
				
	PGPValidatePtr( context );
	PGPValidatePtr( allKeys );
	PGPValidatePtr( userSettings );
	PGPValidatePtr( actualRecipients );
	
	if( numDefaultRecipients != 0 )
	{
		PGPValidatePtr( defaultRecipients );
	}
	else if( IsntNull( defaultRecipients ) )
	{
		PGPValidateParam( numDefaultRecipients != 0 )
	}
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = PGPKeyServerInit();
		if( IsntPGPError( err ) )
		{
			err = pgpClientRecipientDialog( context, state.clientPrefsRef,
						state.adminPrefsRef, tlsContext, allKeys,
						numDefaultRecipients, defaultRecipients,
						dialogOptions, defaultSettings, userSettings,
						actualRecipients, newKeys );
			
			PGPKeyServerCleanup();
		}
	}
	
	ExitPGPclientLib( &state );
	
	return( err );
}	


	static PGPError
pgpClientDecryptionPassphraseDialog(
	PGPContextRef		context,
	PGPPrefRef			clientPrefsRef,				
	PGPtlsContextRef	tlsContext,
	const char			*prompt,						
	PGPKeySetRef		recipientKeys,
	const PGPKeyID		keyIDList[],		
	PGPUInt32			keyIDCount,
	PGPOptionListRef	additionalOptions,
	char				**passphrase,
	PGPKeyRef 			*decryptionKey,
	PGPKeySetRef		*newKeys)
{
	PGPError						err;
	PGPOptionListRef				optionList 	= kInvalidPGPOptionListRef;
	PGPKeyServerSpec				*serverList = NULL;
	PGPUInt32						serverIndex;
	PGPUInt32						numKSEntries = 0;
	StPGPPreserveKeyServerStorage	keyserverStorage;
			
	err = PGPBuildOptionList( context, &optionList,
				PGPOUIOutputPassphrase( context, passphrase ),
				IsntNull( prompt ) ? PGPOUIDialogPrompt( context, prompt ) : 
					PGPONullOption( context ),
				PGPOptionListRefIsValid( additionalOptions ) ?
					additionalOptions : PGPONullOption( context ),
				PGPOLastOption( context ) );
	if( IsntPGPError( err ) && IsntNull( newKeys ) )
	{
		PGPKeyServerEntry	*ksEntries 	= NULL;

		err = PGPCreateKeyServerPath( clientPrefsRef, "", &ksEntries,
					&numKSEntries);
		if( IsntPGPError( err ) && IsntNull( ksEntries ) && numKSEntries > 0 )
		{
			serverList = (PGPKeyServerSpec *) PGPNewData(
								PGPGetContextMemoryMgr( context ),
								numKSEntries * sizeof( *serverList ),
								kPGPMemoryMgrFlags_Clear );
			if( IsntNull( serverList ) )
			{
				for( serverIndex = 0; serverIndex < numKSEntries;
							serverIndex++ )
				{
					err = PGPNewKeyServerFromHostName( context,
											ksEntries[serverIndex].serverDNS,
											ksEntries[serverIndex].serverPort,
											ksEntries[serverIndex].protocol,
											kPGPKeyServerAccessType_Normal,
											kPGPKeyServerKeySpace_Normal,
											&serverList[serverIndex].server );
					if( IsPGPError( err ) )
						break;
					
					serverList[serverIndex].serverName =
							ksEntries[serverIndex].serverDNS;
				}
				
				if( IsntPGPError( err ) )
				{
					err = PGPAppendOptionList( optionList,
								PGPOUIKeyServerUpdateParams( context,
									numKSEntries, serverList, tlsContext,
									FALSE, newKeys, PGPOLastOption( context ) ),
								PGPOLastOption( context ) );
				}
			}
			else
			{
				err = kPGPError_OutOfMemory;
			}
		}

		if( IsntNull( ksEntries ) )
			PGPDisposeKeyServerPath( ksEntries );
	}
	
	if( IsntPGPError( err ) )
	{
		err = PGPDecryptionPassphraseDialog( context, recipientKeys,
					keyIDCount, keyIDList, decryptionKey, optionList,
					PGPOLastOption( context ) );
	}
	
	if( PGPOptionListRefIsValid( optionList ) )
		PGPFreeOptionList( optionList );
	
	if( IsntNull( serverList ) )
	{
		for( serverIndex = 0; serverIndex < numKSEntries; serverIndex++ )
		{
			if( PGPKeyServerRefIsValid( serverList[serverIndex].server ) )
				PGPFreeKeyServer( serverList[serverIndex].server );
		}

		PGPFreeData( serverList );
	}
	
	return( err );
}

	PGPError
PGPClientDecryptionPassphraseDialog(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	const char			*prompt,
	PGPKeySetRef		recipientKeys,
	const PGPKeyID		keyIDList[],		
	PGPUInt32			keyIDCount,
	PGPOptionListRef	additionalOptions,
	char				**passphrase,
	PGPKeyRef 			*decryptionKey,
	PGPKeySetRef		*newKeys)
{
	PGPError			err;
	PGPclientLibState	state;

	if ( IsntNull( passphrase ) )
		*passphrase = NULL;
	if ( IsntNull( decryptionKey ) )
		*decryptionKey = kInvalidPGPKeyRef;
	if ( IsntNull( newKeys ) )
		*newKeys = kInvalidPGPKeySetRef;
				
	PGPValidatePtr( context );
	PGPValidatePtr( recipientKeys );
	PGPValidatePtr( passphrase );
	PGPValidatePtr( decryptionKey );

	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = PGPKeyServerInit();
		if( IsntPGPError( err ) )
		{
			err = pgpClientDecryptionPassphraseDialog( context,
						state.clientPrefsRef,
						tlsContext, prompt, recipientKeys, keyIDList,
						keyIDCount, additionalOptions, passphrase,
						decryptionKey, newKeys );
				
			PGPKeyServerCleanup();
		}
	}
	
	ExitPGPclientLib( &state );
				
	return( err );
}





