/*____________________________________________________________________________
	Copyright (C) 1997-1998 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: CPGPtoolsFileTask.cp,v 1.72 1999/04/04 11:12:28 wprice Exp $
____________________________________________________________________________*/

#include <LArray.h>
#include <LComparator.h>
#include <PP_Messages.h>
#include <UDesktop.h>

#include "pgpFileSpec.h"
#include "pgpShareFile.h"
#include "pgpUserInterface.h"
#include "pgpUtilities.h"

#include "MacErrors.h"
#include "MacFiles.h"
#include "MacStrings.h"
#include "pgpMem.h"
#include "pgpUtilities.h"

#include "CPGPtoolsProcessingList.h"

#include "CPGPtoolsFileTask.h"
#include "CPGPtoolsTaskProgressDialog.h"
#include "CResultsWindow.h"
#include "PGPtoolsEncryptDecrypt.h"
#include "PGPtoolsResources.h"
#include "PGPtoolsUtils.h"
#include "CPGPToolsWipeTask.h"
#include "CWarningAlert.h"


CPGPtoolsFileTask::CPGPtoolsFileTask(
	PGPtoolsOperation operation,
	const FSSpec *fileSpec) : CPGPtoolsCryptoTask( operation )
{
	OSStatus	err;
	CInfoPBRec	cpb;
	
	mWorkingFileSpec	= *fileSpec;
	mSourceSpec 		= *fileSpec;
	
	err	= FSpGetCatInfo( fileSpec, &cpb );
	pgpAssertNoErr( err );
	mSourceIsFolder	= cpbIsFolder( &cpb );
	
	mProcessingList	= new CPGPtoolsProcessingList();
	if ( IsntNull( mProcessingList ) )
	{
		err	= mProcessingList->BuildSourceTree( &mSourceSpec );
		pgpAssertMsg( IsntErr( err ), "error building source tree" );
		if ( IsErr( err ) )
		{
			delete mProcessingList;
			mProcessingList	= NULL;
		}
	}
	
	pgpClearMemory( &mDetachedSigInputSpec, sizeof( mDetachedSigInputSpec ) );
}

CPGPtoolsFileTask::~CPGPtoolsFileTask(void)
{
	delete mProcessingList;
	mProcessingList	= NULL;
}

// MW cannot seem to compile this function without this option
#pragma global_optimizer on

	CToolsError
CPGPtoolsFileTask::EncryptSignFileInternal(
	CEncodeParams 			*params,
	const ProcessingInfo	*srcInfo,
	const FSSpec 			*destSpec )
{
	CToolsError				err;
	PGPOptionListRef		signingOptions	= kInvalidPGPOptionListRef;
	PGPContextRef			c	= mContext;
	PGPBoolean				clearSign 	= FALSE;
	PGPBoolean				armorOutput = FALSE;
	PGPLocalEncodingFlags	encodingFlags;
		
	pgpAssert( PGPContextRefIsValid( mContext ) );
	pgpAssertAddrValid( params, CEncodeParams );

	params->progressDialog->SetNewProgressItem( srcInfo->srcSpec.name );

	encodingFlags = params->encodingFlags;
	if( encodingFlags == kPGPLocalEncoding_None &&
		srcInfo->dataSize == 0 &&
		srcInfo->resSize != 0 )
	{
		// MacBinary is off, but the file is a resource fork-only
		// file. This case is illogical. Turn on MacBinary for this file.
	
		encodingFlags = kPGPLocalEncoding_Force;
	}
	
	if( params->textOutput )
	{
		armorOutput = TRUE;
		
		if( srcInfo->fInfo.fdType == 'TEXT' &&
			params->signing &&
			! params->encrypting &&
			encodingFlags != kPGPLocalEncoding_Force &&
			! params->detachedSignature )
		{
			// Clear sign TEXT files in the non-MacBinary,
			// non-detached sig case
			
			clearSign 	= TRUE;
			armorOutput = FALSE;
		}
	}
	
	if( params->signing )
	{
		err.pgpErr = PGPBuildOptionList( c, &signingOptions, 
				PGPOSignWithKey( c, params->signingKey,
					params->signingWithSplitKey ?
						PGPOPasskeyBuffer( c, params->signingPasskey,
							 params->signingPasskeySize ) :
						PGPOPassphrase( c, params->signingPassphrase ),
					PGPOLastOption( c ) ),
				PGPOClearSign( c, clearSign ),
				PGPOLastOption( c ) );
		
		if( err.IsntError() && params->detachedSignature )
		{
			err.pgpErr = PGPAppendOptionList( signingOptions,
						PGPODetachedSig( c,
							PGPOLastOption( c ) ),
						PGPOLastOption( c ) );
		
			/* text output should have alreday been set up */
		}
	}
	else
	{
		/* no signing options; create dummy list */
		err.pgpErr	= PGPBuildOptionList( c,
						&signingOptions, PGPOLastOption( c ) );
	}
	
	if( err.IsntError() )
	{
		err.pgpErr = PGPGuaranteeMinimumEntropy( c );
	}
	
	if( err.IsntError() )
	{
		CEncodeEventHandlerData	data( this, params );
		
		err.pgpErr = PGPEncode( c,
					PGPOInputFileFSSpec( c, &srcInfo->srcSpec ),
					PGPOOutputFileFSSpec( c, destSpec ),
					PGPODataIsASCII( c, srcInfo->fInfo.fdType == 'TEXT' ),
					PGPOArmorOutput( c, armorOutput ),
					signingOptions,
					PGPOptionListRefIsValid( params->encodeOptions ) ?
						params->encodeOptions : PGPONullOption( c ),
					PGPOEventHandler( c,
						sEncodeEventHandlerProc, &data ),
					PGPOSendNullEvents( c, TRUE ),
					PGPOLocalEncoding( c, encodingFlags ),
					PGPOForYourEyesOnly( c, params->fyeo ),
					PGPOLastOption( c  ) );
	}

	if ( PGPOptionListRefIsValid( signingOptions ) )
		PGPFreeOptionList( signingOptions );
	
	if( ShouldReportError( err ) )
	{
		err = ReportError( err,
			params->encrypting ? kFileCannotBeEncryptedStrIndex :
			kFileCannotBeSignedStrIndex );
	}

	return( err );
}

#pragma global_optimizer reset

	OSStatus
CPGPtoolsFileTask::GetOutputFolderSpec(
	Boolean			encryptSign,
	const FSSpec *	srcSpec,
	FSSpec *		destSpec )
{
	OSStatus	err	= noErr;
	const uchar	kPGPSuffix[]	= "\p PGP";
	
	*destSpec	= *srcSpec;
	
	if ( encryptSign )
	{
		if ( StrLength( destSpec->name ) +
				StrLength( kPGPSuffix ) > kMaxHFSFileNameLength )
		{
			destSpec->name[ 0 ]	=
				kMaxHFSFileNameLength - StrLength( kPGPSuffix );
		}
		AppendPString( kPGPSuffix, destSpec->name );
	}
	else
	{	/* decrypting/verifying */
		if ( PStringHasSuffix( srcSpec->name, kPGPSuffix, FALSE) )
		{
			/* strip the ".pgp" off */
			destSpec->name[ 0 ]	-= StrLength( kPGPSuffix );
		}
	}
	
	err	= FSpGetUniqueSpec( destSpec, destSpec );
	
	return( err );
}


/*____________________________________________________________________________
	Determine if the file should be processed.
	
	While it might make sense to not encrypt already encrypted items or
	encrypt detached sigs, there are also cases where you might want
	to do this. so don't prevent it.
	
	The same idea applies when signing already signed or encrypted items.
____________________________________________________________________________*/
	Boolean
CPGPtoolsFileTask::ShouldEncryptSignFile(
	Boolean			encrypting,
	const FSSpec *	spec )
{
	OSStatus		err;
	Boolean			shouldProcess	= TRUE;
	
	// FileKindInfo	info;
	// err	= GetPGPFileKindInfo( spec, &info );
	if ( encrypting )
	{
		/* without UI to ask the user, we can't reasonably forbid signing */
	}
	else
	{
		/* without UI to ask the user, we can't reasonably forbid signing */
	}
	
	if ( shouldProcess )
	{
		CInfoPBRec	cpb;
		
		err	= FSpGetCatInfo( spec, &cpb );
		if ( IsntErr( err ) )
		{
			/* don't do aliases or invisible files */
			if ( cpbFInfo( &cpb ).fdFlags & ( kIsInvisible | kIsAlias ) )
				shouldProcess	= FALSE;
		}
	}
	
	return( shouldProcess );
}
	

	CToolsError
CPGPtoolsFileTask::EncryptSignFolder(
	CEncodeParams *	params )
{
	CToolsError	err;
	FSSpec		destTreeSpec;
	
	(void)params;
	pgpAssert( IsntNull( mProcessingList ) );
	
	err.err	= mProcessingList->PrepareForIteration();
	if ( err.IsntError() && ! mProcessInPlace)
	{
		err.err	= GetOutputFolderSpec( TRUE,
					&mSourceSpec, &destTreeSpec );
		if ( err.IsntError() )
		{
			err.err	= mProcessingList->CreateDestTree( &destTreeSpec );
		}
	}
	
	if ( err.IsntError() )
	{
		UInt32	numFiles	= mProcessingList->GetNumFiles();
		
		for( UInt32 index = 0; index < numFiles; ++index )
		{
			ProcessingInfo	info;
			FSSpec			destSpec;
			
			mProcessingList->GetIndFile( index, &info );
			
			mWorkingFileSpec	= info.srcSpec;
			
			if ( mProcessInPlace )
			{
				destSpec.vRefNum	= info.srcSpec.vRefNum;
				destSpec.parID		= info.srcSpec.parID;
			}
			else
			{
				destSpec.vRefNum	= info.destVRefNum;
				destSpec.parID		= info.destParID;
			}
			if ( ShouldEncryptSignFile( params->encrypting,
					&info.srcSpec ) )
			{
				GetDefaultEncryptSignName( info.srcSpec.name,
					params->textOutput,
					params->detachedSignature, destSpec.name );
				pgpAssert( ! FSpEqual( &info.srcSpec, &destSpec ) );
				
				err	= EncryptSignFileInternal( params, &info,
					&destSpec );
			}
			else if ( ! mProcessInPlace )
			{
				CopyPString( info.srcSpec.name, destSpec.name );
				err.err	= FSpCopyFiles( PGPGetContextMemoryMgr( mContext ),
								&info.srcSpec, &destSpec );
				pgpAssertNoErr( err.err );
			}
			
			if ( err.IsError() )
				break;
		}
	}
	
	return( err );
}

 
	CToolsError
CPGPtoolsFileTask::EncryptSign(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	CEncodeParams *		params)
{
	CToolsError	err;

	mContext	= context;
	mTLSContext	= tlsContext;
	
	pgpAssert( (! params->wipeOriginal) || params->encrypting );
	
	/* send output to a duplicate folder hierarchy 
		unless we are doing detached sigs */
	mProcessInPlace	= params->detachedSignature;

	if ( mSourceIsFolder )
	{
		err	= EncryptSignFolder( params );
	}
	else
	{
		FSSpec			destSpec;
		ProcessingInfo	info;
		
		mProcessingList->GetIndFile( 0, &info );
			
		GetDefaultEncryptSignFSSpec( &info.srcSpec,
				params->textOutput,
				params->detachedSignature,
				&destSpec );

		if( ( GetTaskModifiers() & optionKey ) != 0 )
		{
			Str255				prompt;
			short				strIndex;
			StandardFileReply	sfReply;
			
			if( ( params->encrypting && params->signing ) ||
				params->detachedSignature )
			{
				strIndex = kSaveAsPromptStrIndex;
			}
			else if( params->encrypting )
			{
				strIndex = kSaveEncryptedFileAsPromptStrIndex;
			}
			else
			{
				strIndex = kSaveSignedFileAsPromptStrIndex;
			}
			
			GetIndString( prompt, kPGPtoolsMiscStringsResID, strIndex );
			
			LMSetSFSaveDisk( -destSpec.vRefNum );
			LMSetCurDirStore( destSpec.parID );
			
			StandardPutFile( prompt, destSpec.name, &sfReply );
			if( sfReply.sfGood )
			{
				destSpec = sfReply.sfFile;
			}
			else
			{
				err.pgpErr = kPGPError_UserAbort;
			}
		}
		else
		{
			(void)FSpGetUniqueSpec( &destSpec, &destSpec );
		}
		
		if( err.IsntError() )
		{
			err	= EncryptSignFileInternal( params, &info,
				&destSpec );
		}
	}
	
	if ( err.IsntError() &&
		params->wipeOriginal )
	{
		CPGPtoolsWipeTask	task( &mSourceSpec );
		CPGPtoolsTaskProgressDialog * progress =
					params->progressDialog;
		ByteCount		progressBytes;
		
		progress->SetProgressOperation( kPGPtoolsWipeOperation );
		err.err	= task.CalcProgressBytes( context, &progressBytes);
		if ( err.IsntError() )
		{
			progress->SetTotalOperations( progressBytes );
			err	= task.Wipe( context, progress );
		}
	}

	mContext	= kInvalidPGPContextRef;
	return( err );
}


	CToolsError
CPGPtoolsFileTask::DecryptVerifyEncryptedFile(
	CDecodeParams *			params,
	const ProcessingInfo *	srcSpecInfo,
	const FSSpec *			destSpec)
{
	CDecodeEventHandlerData	data( mContext, mTLSContext, this, params );
	PGPContextRef			c	= mContext;
	CToolsError				err;
	PGPKeySetRef			newKeysSet;
	
	mWorkingFileSpec = srcSpecInfo->srcSpec;
	
	// Assume both forks are verified
	mSignatureData.didNotVerifyResourceFork = FALSE;
	
	err.pgpErr = PGPNewKeySet( mContext, &newKeysSet );
	if( err.IsntError() )
	{
		PGPOptionListRef		decodeOptions = kInvalidPGPOptionListRef;

		err.pgpErr = PGPBuildOptionList( c, &decodeOptions,
						PGPOKeySetRef( c, params->allKeys ),
						PGPOImportKeysTo( c, newKeysSet ),
						PGPOEventHandler( c, sDecodeEventHandlerProc,
							&data ),
						PGPOSendNullEvents( c, TRUE ),
						PGPOLastOption( c ) );
		if( err.IsntError() )
		{
			err.pgpErr = PGPDecode( c,
					PGPOInputFileFSSpec( c, &srcSpecInfo->srcSpec ),
					IsntNull( destSpec ) ?
						PGPOOutputFileFSSpec( c, destSpec ) :
						PGPONullOption( c ),
					decodeOptions,
					PGPOLastOption( c ) );
			
			if ( err.IsntError() &&
					mSignatureDataValid &&
						mSignatureData.data.checked &&
							( ! mSignatureData.data.verified ) )
			{
				/*
				sig checked, but didn't verify.  Output should have been
				created. Try again with no output, existing passphrase,
				and local encoding off (controlled by 
				data.retryingSigVerification ).
				*/
				data.retryingSigVerification			= TRUE;
				mSignatureData.didNotVerifyResourceFork = TRUE;
				mSignatureDataValid						= FALSE;
				
				err.pgpErr = PGPDecode( c,
						PGPOInputFileFSSpec( c, &srcSpecInfo->srcSpec ),
						PGPODiscardOutput( c, TRUE ),
						decodeOptions,
						PGPOLastOption( c ) );
						
				pgpAssert( mSignatureDataValid );
			}
			
			if( err.IsntError() && ( data.outputData != NULL ) )
			{
				err.pgpErr = PGPForYourEyesOnlyDialog( c, (char *) data.outputData );
				pgpClearMemory( data.outputData, data.outputDataSize );
				PGPFreeData( data.outputData );
				data.outputData = NULL;
			}
			
			if( err.IsntError() &&
				PGPKeySetRefIsValid( data.clientHandlerData.newKeySet ) )
			{
				err.pgpErr = PGPAddKeys( data.clientHandlerData.newKeySet,
									newKeysSet );
				if( err.IsntError() )
				{
					err.pgpErr = PGPCommitKeyRingChanges( newKeysSet );
				}
			}
			
			if( err.IsntError() )
			{
				PGPUInt32	numNewKeys;
			
				err.pgpErr = PGPCountKeys( newKeysSet, &numNewKeys );
				if( err.IsntError() && numNewKeys > 0 )
				{
					// Offer to import the keys
					PGPKeySetRef	importSet;
					char			prompt[256];
					
					GetIndCString( prompt, kDialogStringsListResID,
							kSelectiveImportFoundKeysPromptStrIndex );
					
					UDesktop::Deactivate();
						err.pgpErr = PGPSelectKeysDialog( c, 
											kPGPSelectKeysImportVariation,
											prompt, newKeysSet,
											params->allKeys, &importSet );
					UDesktop::Activate();
					
					if( err.IsntError() )
					{
						err.pgpErr = PGPSharedAddKeysToDefaultKeyring(
												importSet );
						
						(void) PGPFreeKeySet( importSet );
					}
				}
			}
		
			PGPFreeOptionList( decodeOptions );
		}
		
		PGPFreeKeySet( newKeysSet );
	}
	
	return( err );
}

	CToolsError
CPGPtoolsFileTask::DecryptKeyShareFile(
	CDecodeParams *			params,
	const ProcessingInfo *	srcSpecInfo,
	const FSSpec *			destSpec)
{
	CToolsError		err;
	
	Str255		message;
	MessageT	dialogResult;

	GetIndString( message, kDialogStringsListResID,
				kDecryptKeyShareFileMsgStrIndex );
	
	if( mSourceIsFolder )
	{
		dialogResult = CWarningAlert::DisplayCustom(
							dialog_DecryptKeyShare, kWACustomAlertType,
							kWACustomAlertStyle, message,
							srcSpecInfo->srcSpec.name );
	}
	else
	{
		dialogResult = CWarningAlert::Display( kWANoteAlertType,
							kWAOKCancelStyle, message,
							srcSpecInfo->srcSpec.name );
	}
	
	if( dialogResult == msg_OK )
	{
		const FSSpec	*shareFileSpec;
			
		if( mSourceIsFolder )
		{
			// Must copy the file first as the conversion is done in place.
			
			pgpAssert( IsntNull( destSpec ) );
			err.err	= FSpCopyFiles( PGPGetContextMemoryMgr( mContext ),
							&srcSpecInfo->srcSpec, destSpec );
							
			shareFileSpec = destSpec;
		}
		else
		{
			shareFileSpec = &srcSpecInfo->srcSpec;
		}
		
		if( err.IsntError() )
		{
			PFLFileSpecRef	shareFileSpecRef;

			err.pgpErr = PFLNewFileSpecFromFSSpec(
								PGPGetContextMemoryMgr( mContext ),
								shareFileSpec, &shareFileSpecRef );

			if( err.IsntError() )
			{
				PGPShareFileRef	shareFileRef;
				
				err.pgpErr = PGPOpenShareFile( shareFileSpecRef,
									&shareFileRef );
				if( err.IsntError() )
				{
					PGPOptionListRef		optionsDecode;
					CDecodeEventHandlerData	data( mContext, mTLSContext, this,
													params );
					PGPShareRef				shares;
					
					optionsDecode = kInvalidPGPOptionListRef;
					shares = kInvalidPGPShareRef;
					
					// decrypt specified share file
					err.pgpErr = PGPBuildOptionList( mContext, &optionsDecode,
										PGPOKeySetRef( mContext,
											params->allKeys ),
										PGPOEventHandler( mContext,
											sDecodeEventHandlerProc, &data ),
										PGPOLastOption( mContext ) );
					if( err.IsntError() )
					{
						err.pgpErr = PGPCopySharesFromFile( mContext,
											shareFileRef, optionsDecode,
											&shares );
					}
					
					if( err.IsntError() )
					{
						PGPRecipientSettings	recipientSettings;
						PGPKeySetRef			recipients;
						PGPKeySetRef			newKeys;
						PGPOptionListRef		optionsEncode;
							
						recipients 		= kInvalidPGPKeySetRef;
						newKeys 		= kInvalidPGPKeySetRef;
						optionsEncode 	= kInvalidPGPOptionListRef;
						
						UDesktop::Deactivate();
						err.pgpErr = PGPClientRecipientDialog( mContext,
										mTLSContext, params->allKeys, 0, nil,
										kPGPRecipientOptionsHideFileOptions,
										kPGPRecipientSettingsDefault,
										&recipientSettings, &recipients,
										&newKeys );
						UDesktop::Activate();

						if( err.IsntError() && PGPKeySetRefIsValid( newKeys ) )
						{
							PGPKeySetRef	importSet;
							char			prompt[256];
							
							GetIndCString( prompt, kDialogStringsListResID,
									kSelectiveImportFoundKeysPromptStrIndex );
							
							err.pgpErr = PGPSelectKeysDialog( mContext,
									kPGPSelectKeysImportVariation,
									prompt, newKeys, params->allKeys,
									&importSet );
							if( err.IsntError() )
							{
								(void) PGPFreeKeySet( params->allKeys );
								params->allKeys = kInvalidPGPKeySetRef;
								
								err.pgpErr = PGPSharedAddKeysToDefaultKeyring(
											importSet );
								if( err.IsntError() )
								{
									err.pgpErr = PGPOpenDefaultKeyRings(
													mContext, 0,
													&params->allKeys );
								}
								
								(void) PGPFreeKeySet( importSet );
							}
							else if( err.pgpErr == kPGPError_UserAbort )
							{
								err.pgpErr = kPGPError_NoErr;
							}
							
							(void) PGPFreeKeySet( newKeys );
						}
						
						if( err.IsntError() )
						{
							if( ( recipientSettings &
									kPGPRecipientSettingsConvEncrypt ) != 0 )
							{
								char	*passphrase = NULL;
								char	prompt[256];
								
								GetIndCString( prompt,
										kDialogStringsListResID,
										kConventionalPassphrasePromptStrIndex );
										
								UDesktop::Deactivate();
								err.pgpErr =
									PGPConventionalEncryptionPassphraseDialog(
											mContext,
											PGPOUIDialogPrompt( mContext,
												prompt ),
											PGPOUIOutputPassphrase( mContext,
												&passphrase ),
											PGPOLastOption( mContext ) );
								UDesktop::Activate();
								
								if( err.IsntError() )
								{
									err.pgpErr = PGPBuildOptionList(
													mContext, &optionsEncode,
													PGPOConventionalEncrypt(
														mContext,
														PGPOPassphrase(
															mContext,
															passphrase ),
														PGPOLastOption(
															mContext ) ),
													PGPOLastOption( mContext ) );
								}
								
								if( IsntNull( passphrase ) )
									PGPFreeData( passphrase );
							}
							else
							{
								err.pgpErr = PGPBuildOptionList( mContext,
												&optionsEncode,
												PGPOEncryptToKeySet( mContext,
													recipients ),
												PGPOLastOption( mContext ) );
							}
						}
	
						if( err.IsntError() )
						{
							err.pgpErr = PGPGuaranteeMinimumEntropy( mContext );
						}
						
						if( err.IsntError() )
						{
							err.pgpErr = PGPCopySharesToFile( mContext,
											shareFileRef, optionsEncode,
											shares );

							if( err.IsntError() )
							{
								err.pgpErr = PGPSaveShareFile( shareFileRef );
							}
						}

						if( PGPKeySetRefIsValid( recipients ) )
							PGPFreeKeySet( recipients );
							
						if( PGPOptionListRefIsValid( optionsEncode ) )
							PGPFreeOptionList( optionsEncode );
					}

					if( PGPOptionListRefIsValid( optionsDecode ) )
						PGPFreeOptionList( optionsDecode );
						
					if( PGPShareRefIsValid( shares ) )
						PGPFreeShares( shares );

					PGPFreeShareFile( shareFileRef );
				}
			}
			
			PFLFreeFileSpec( shareFileSpecRef );
			
			// Delete copied share file on error.
			if( err.IsError() && mSourceIsFolder )
				(void) FSpDelete( destSpec );
		}
	}
	else if( dialogResult == msg_Cancel )
	{
		err.pgpErr = kPGPError_UserAbort;
	}
	else
	{
		/* Skip button. Copy to destination if we're processing a folder. */
		
		if( mSourceIsFolder )
		{
			err.err	= FSpCopyFiles( PGPGetContextMemoryMgr( mContext ),
					&srcSpecInfo->srcSpec, destSpec );
		}
	}
	
	return( err );
}

	CToolsError
CPGPtoolsFileTask::DecryptVerifyFileInternal(
	CDecodeParams *			params,
	const ProcessingInfo *	srcSpecInfo,
	const FSSpec *			destSpec )
{
	CToolsError			err;
	Boolean				forcePrompt	= FALSE;
	PGPOptionListRef	signingOptions	= kInvalidPGPOptionListRef;	
	FileKindInfo		info;
	
#if PGP_DEBUG
	pgpAssert( PGPContextRefIsValid( mContext ) );
	pgpAssertAddrValid( params, CEncodeParams );
	if ( IsntNull( destSpec ) )
		pgpAssert( ! FSpEqual( &srcSpecInfo->srcSpec, destSpec ) );
#endif

	params->progressDialog->SetNewProgressItem(
				srcSpecInfo->srcSpec.name );
	
	(void) GetPGPFileKindInfo( &srcSpecInfo->srcSpec, &info );
	
	if( ( info.maybePGPFile || info.isPGPFile ) ||
		! mSourceIsFolder )	/* Always try to decrypt "top level" files */
	{
		if( info.isKeyShareFile )
		{
			err = DecryptKeyShareFile( params, srcSpecInfo,
						destSpec );
		}
		else
		{
			err = DecryptVerifyEncryptedFile( params, srcSpecInfo,
						destSpec );
			if( err.IsntError() &&
				mSourceIsFolder &&
				IsntNull( destSpec ) &&
				! FSpExists( destSpec ) )
			{
				/*
				** "Decrypting" the file did not really do so i.e. there was no
				** data to decrypt. Copy the file instead
				*/
				
				err.err = FSpCopyFiles( PGPGetContextMemoryMgr( mContext ),
							&srcSpecInfo->srcSpec, destSpec );
			}
		}
	}
	else
	{
		/* if it's a file job, don't copy it */
		/* but if it's a folder, copy non-relevant items */
		if ( mSourceIsFolder )
		{
			pgpAssert( IsntNull( destSpec ) );
			err.err	= FSpCopyFiles( PGPGetContextMemoryMgr( mContext ),
							&srcSpecInfo->srcSpec, destSpec );
		}
	}
	
	if( err.IsntError( ) )
	{
		if( mSignatureDataValid )
		{
			CResultsWindow::NewResult( mDetachedSigInputSpec.name[0] != 0 ?
						mDetachedSigInputSpec.name : srcSpecInfo->srcSpec.name,
						&mSignatureData);
		}
	}

	if( ShouldReportError( err ) )
	{
		err = ReportError( err, kFileCannotBeDecryptedVerifiedStrIndex );
	}

	return( err );
}


	CToolsError
CPGPtoolsFileTask::DecryptVerifyFolder(
	CDecodeParams *	params)
{
	CToolsError	err;
	FSSpec		destTreeSpec;
	
	(void)params;
	pgpAssert( IsntNull( mProcessingList ) );
	
	err.err	= mProcessingList->PrepareForIteration();
	if ( err.IsntError() && ! mProcessInPlace)
	{
		err.err	= GetOutputFolderSpec( FALSE,
					&mSourceSpec, &destTreeSpec );
		if ( err.IsntError() )
		{
			err.err	= mProcessingList->CreateDestTree( &destTreeSpec );
		}
	}
	
	if ( err.IsntError() )
	{
		UInt32	numFiles	= mProcessingList->GetNumFiles();
		
		for( UInt32 index = 0; index < numFiles; ++index )
		{
			ProcessingInfo	info;
			FSSpec			destSpec;
			FileKindInfo	fileInfo;
			
			mProcessingList->GetIndFile( index, &info );
			mWorkingFileSpec	= info.srcSpec;
			
			if ( mProcessInPlace )
			{
				destSpec.vRefNum	= info.srcSpec.vRefNum;
				destSpec.parID		= info.srcSpec.parID;
			}
			else
			{
				destSpec.vRefNum	= info.destVRefNum;
				destSpec.parID		= info.destParID;
			}

			(void) GetPGPFileKindInfo( &info.srcSpec, &fileInfo );
	
			if ( fileInfo.maybePGPFile || fileInfo.isPGPFile )
			{
				GetDefaultDecryptVerifyName( info.srcSpec.name,
					destSpec.name );
				pgpAssert( ! FSpEqual( &info.srcSpec, &destSpec ) );
				
				err	= DecryptVerifyFileInternal( params,
						&info, &destSpec );
			}
			else if ( ! mProcessInPlace )
			{
				CopyPString( info.srcSpec.name, destSpec.name );
				err.err	= FSpCopyFiles( PGPGetContextMemoryMgr( mContext ),
									&info.srcSpec, &destSpec );
				pgpAssertNoErr( err.err );
			}
			
			if ( err.IsError() )
				break;
		}
	}
	
	return( err );
}

							
	CToolsError
CPGPtoolsFileTask::DecryptVerify(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	CDecodeParams		*params)
{
	CToolsError				err;
	
	pgpAssertAddrValid( params, CDecodeParams );
	
	mContext		= context;
	mTLSContext		= tlsContext;
	mProcessInPlace	= FALSE;
	
	if ( mSourceIsFolder )
	{
		err	= DecryptVerifyFolder( params );
		
	}
	else
	{
		ProcessingInfo	info;
		
		mProcessingList->GetIndFile( 0, &info );

		err	= DecryptVerifyFileInternal( params, &info, NULL );
	}

	mContext	= kInvalidPGPContextRef;
	return( err );
}

	void
CPGPtoolsFileTask::BuildErrorMessage(
	CToolsError 	err,
	short 		errorStrIndex,
	StringPtr	msg)
{
	Str255			errorStr;

	GetIndString( msg, kErrorStringListResID, errorStrIndex );

	PrintPString( msg, msg, mWorkingFileSpec.name );
	
	GetErrorString( err, errorStr );
	PrintPString( msg, msg, errorStr );
}


	OSStatus
CPGPtoolsFileTask::CalcProgressBytes(
	PGPContextRef	context,
	ByteCount *		progressBytes )
{
	OSStatus	err	= noErr;
	
	(void)context;
	pgpAssert( IsntNull( mProcessingList ) );
	if ( IsNull( mProcessingList ) )
		return( paramErr );
	*progressBytes	= 0;
	
	err	= mProcessingList->PrepareForIteration();
	if ( IsntErr( err ) )
	{
		UInt32	numFiles	= mProcessingList->GetNumFiles();
		Str32	sigFileSuffix;
		
		GetIndString( sigFileSuffix, kPGPtoolsMiscStringsResID,
					kDetachedSigFileSuffixStrIndex );

		for( UInt32 index = 0; index < numFiles; ++index )
		{
			ProcessingInfo	info;
			
			mProcessingList->GetIndFile( index, &info );

			*progressBytes += info.resSize + info.dataSize;

			/* Account for bytes in target of detached signature */
			
			if( info.fInfo.fdType == kPGPMacFileType_DetachedSig ||
				PStringHasSuffix( info.srcSpec.name, sigFileSuffix, FALSE ) )
			{
				FSSpec		testSpec = info.srcSpec;
				CInfoPBRec	cpb;
				
				testSpec.name[0] -= sigFileSuffix[0];
				
				if( FSpGetCatInfo( &testSpec, &cpb ) == noErr &&
					cpbIsFile( &cpb ) )
				{
					*progressBytes += cpbDataForkSize( &cpb ) + cpbResForkSize( &cpb );
				}
			}
		}
	}

	return( err );
}




	PGPError
CPGPtoolsFileTask::DecodeOutputEvent(
	PGPEvent *					event,
	CDecodeEventHandlerData	*	data)
{
	PGPError	err = kPGPError_NoErr;
	FSSpec		outSpec;

	(void)data;
		
	GetDefaultDecryptVerifyFSSpec( &mWorkingFileSpec, &outSpec );
	
	if( ( GetTaskModifiers() & optionKey ) != 0 &&
		! mSourceIsFolder )
	{
		Str255				prompt;
		StandardFileReply	sfReply;
		
		GetIndString( prompt, kPGPtoolsMiscStringsResID, kSaveDecryptedFileAsPromptStrIndex );
		
		LMSetSFSaveDisk( -outSpec.vRefNum );
		LMSetCurDirStore( outSpec.parID );
		
		StandardPutFile( prompt, outSpec.name, &sfReply );
		if( sfReply.sfGood )
		{
			outSpec = sfReply.sfFile;
		}
		else
		{
			err = kPGPError_UserAbort;
		}
	}
	else
	{
		(void) FSpGetUniqueSpec( &outSpec, &outSpec );
	}
	
	if( IsntPGPError( err ) )
	{
		err = PGPAddJobOptions( event->job,
					PGPOOutputFileFSSpec( mContext, &outSpec ),
					PGPOLastOption( mContext ) );
	}
	
	return( err );
}



	PGPError
CPGPtoolsFileTask::DecodeDetachedSignatureEvent(
	PGPEvent *					event,
	CDecodeEventHandlerData	*	data)
{
	PGPContextRef	c	= mContext;
	PGPError		err	= kPGPError_NoErr;
	
	/* see if there is an appropriately named source file */
	if( ! data->retryingSigVerification )
	{	/* caution: use just macErr here */
		OSStatus	macErr;
		CInfoPBRec	cpb;
	
		if ( GetDetachedSigFileSourceSpec( &mWorkingFileSpec,
			&mDetachedSigInputSpec ) )
		{
			macErr	= FSpGetCatInfo( &mDetachedSigInputSpec, &cpb );
			if ( IsntErr( macErr ) )
			{
				if ( cpbIsFolder( &cpb ) )
					macErr	= fnfErr;
			}
		}
		if ( IsErr( macErr ) )
		{
			macErr = PromptForSignatureSourceFile( &mWorkingFileSpec,
							&mDetachedSigInputSpec );
		}
		
		err = MacErrorToPGPError( macErr );
	}
	
	
	if ( IsntPGPError( err ) )
	{
		CInfoPBRec				cpb;
		PGPLocalEncodingFlags	encodingFlags;
		
		/* first we'll try MacBinary on (Force), then if we're doing
			a retry, we'll try it off */
		if ( data->retryingSigVerification )
			encodingFlags	= kPGPLocalEncoding_None;
		else
			encodingFlags	= kPGPLocalEncoding_Force;
			
		(void)FSpGetCatInfo( &mWorkingFileSpec, &cpb );
		
		/* require CRC to be valid, unless it's not created by us */
		/* at least one brain-dead program doesn't set the CRC */
		if ( cpbFileCreator( &cpb ) != kPGPMacFileCreator_Tools )
			encodingFlags	|= kPGPLocalEncoding_NoMacBinCRCOkay;
		
		(void)FSpGetCatInfo( &mDetachedSigInputSpec, &cpb );

		mSignatureData.haveResourceFork = ( cpbResForkSize( &cpb ) != 0 );
			
		err	= PGPAddJobOptions( event->job,
			PGPODetachedSig( c, 
				PGPOInputFileFSSpec( c, &mDetachedSigInputSpec ),
				PGPOLocalEncoding( c, encodingFlags),
				PGPOLastOption( c ) ),
			PGPOLastOption( c ) );
	}
	
	return( err );
}


	PGPError
CPGPtoolsFileTask::DecodeEventHandler(
	PGPEvent *					event,
	CDecodeEventHandlerData *	data)
{
	PGPError	err = kPGPError_NoErr;
	
	switch( event->type )
	{
		default:
		{
			err	= CPGPtoolsCryptoTask::DecodeEventHandler( event, data );
			break;
		}
		
		case kPGPEvent_OutputEvent:
		{
			if( event->data.outputData.forYourEyesOnly )
			{
				err = CPGPtoolsCryptoTask::DecodeEventHandler( event, data );
			}
			else
			{
				err = DecodeOutputEvent( event, data );
			}
			break;
		}
		
		case  kPGPEvent_DetachedSignatureEvent:
		{
			err = DecodeDetachedSignatureEvent( event, data );
			break;
		}
		
		case kPGPEvent_EndLexEvent:
		{
			err = CPGPtoolsCryptoTask::DecodeEventHandler( event, data );
			break;
		}
	}
	
	return( err );
}



	void
CPGPtoolsFileTask::GetDecryptPassphrasePromptString(StringPtr prompt)
{
	GetIndString( prompt, kDialogStringsListResID,
						kGetPassphraseForFilePromptStrIndex );
	PrintPString( prompt, prompt, mWorkingFileSpec.name );
}

	void
CPGPtoolsFileTask::GetTaskItemName(StringPtr name)
{
	CopyPString( mWorkingFileSpec.name, name );
}


	OSStatus
CPGPtoolsFileTask::CanUseFYEO(
	Boolean *	outCanUseFYEO)
{
	OSStatus	result = noErr;
	
	*outCanUseFYEO = false;
	if( ! mSourceIsFolder )
	{
		CInfoPBRec	cpb;
		
		result = FSpGetCatInfo( &mSourceSpec, &cpb );
		if( ( result == noErr ) && ( cpbFileType( &cpb ) == 'TEXT' ) )
		{
			*outCanUseFYEO = true;
		}
	}
	
	return result;
}
