/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc. and it's affiliated companies
	All rights reserved.

	$Id: PGPsdaMacCommon.cp,v 1.2 1999/02/25 11:10:09 heller Exp $
____________________________________________________________________________*/

#include <string.h>

#include "MacEnvirons.h"
#include "pgpMem.h"

#include "PGPsdaMacCommon.h"
#include "SHA.h"


	static void
HashBuf(
	const void 	*buf,
	PGPSize 	length,
	uchar 		hashBytes[kSHS_DIGESTSIZE])
{
	SHA			hash;
	SHA::Digest	hashResult;
		
	pgpAssertAddrValid( buf, uchar );
	pgpAssertAddrValid( hashBytes, uchar );
	
	hash.Update( (uchar *) buf, length );
	hash.Final( &hashResult );
	
	pgpCopyMemory( hashResult.bytes, hashBytes, kSHS_DIGESTSIZE );
}

	static void
SaltKey(
	const CASTKey			*keyin,
	CASTKey 				*keyout,
	const PassphraseSalt 	*salt,
	PGPUInt32 				*hashReps)
{
	uchar		j;
	PGPUInt32	i, k;
	PGPUInt32	endTicks;
	SHA			hash;
	SHA::Digest	hashResult;
	
	pgpAssertAddrValid( keyin, EncryptedCASTKey );
	pgpAssertAddrValid( keyout, EncryptedCASTKey );
	pgpAssertAddrValid( salt, uchar );
	pgpAssertAddrValid( hashReps, short );
	
	hash.Update( salt->saltBytes, sizeof( salt->saltBytes ) );
	
	if( *hashReps != 0)
	{
		// Hash the passphrase and a rotating byte counter the specified
		// number of times into the hash field with the 8 bytes of salt from
		// above in order to form the session key.

		k = *hashReps;

		for( i = 0, j = 0; i < k; i++, j++ )
		{
			hash.Update( keyin->keyBytes, sizeof( keyin->keyBytes ) );
			hash.Update( &j, 1 );
		}
	}
	else
	{
		// This is a new SDA.  We hash the
		// passphrase in with a rotating counter byte an arbitrary number of
		// times based on the processing power of the computer we're running
		// on up to a maximum of 16000.

		endTicks = LMGetTicks() + 30;
		for( i = 0, j = 0; ( LMGetTicks() < endTicks ) && ( i < 16000 );
					i++, j++ )
		{
			hash.Update( keyin->keyBytes, sizeof( keyin->keyBytes ) );
			hash.Update(&j, 1);
		}
		
		*hashReps = i;
	}
	
	hash.Final( &hashResult );
	
	pgpCopyMemory( hashResult.bytes, keyout->keyBytes,
			sizeof( keyout->keyBytes ) );
}

	void
HashSaltPassphrase(
	const char				*passphrase,
	const PassphraseSalt 	*salt,
	PGPUInt32 				*hashReps,
	CASTKey					*key)
{
	uchar	hash[kSHS_DIGESTSIZE];
	
	HashBuf( passphrase, strlen( passphrase ), hash );
	SaltKey( (const CASTKey *) hash, key, salt, hashReps );
}

	OSStatus
ComputeCheckBytes(
	CipherProcRef			cipherProc,
	CipherContext			*cipherContext,
	const CASTKey			*key,
	PGPsdaArchiveCheckBytes	*checkBytes)
{
	PGPByte		inBuffer[512];
	PGPByte		outBuffer[512];
	OSStatus	err;
	UInt32		blockNumber = 529583;	/* Arbitrary */
	
	HoldBuffer( inBuffer, sizeof( inBuffer ) );
	
	/* Check bytes are simply the CAST key encrypted to itself.
	** Since our encryptor is from PGdisk and is block based, we encrypt
	** 512 bytes with a fixed, arbitrary block number.
	*/

	pgpAssert( sizeof( *checkBytes ) <= sizeof( *key ) );
	
	pgpClearMemory( inBuffer, sizeof( inBuffer ) );
	pgpCopyMemory( key, inBuffer, sizeof( *key ) );
	
	err = CipherProc_CallEncrypt( cipherProc, cipherContext, blockNumber,
					1, inBuffer, outBuffer );
	if( IsntErr( err ) )
	{
		pgpCopyMemory( outBuffer, checkBytes, sizeof( *checkBytes ) );
	}
	
	UnholdBuffer( inBuffer, sizeof( inBuffer ), TRUE );

	return( err );
}

	PGPBoolean
VerifyCheckBytes(
	CipherProcRef			cipherProc,
	CipherContext			*cipherContext,
	const CASTKey			*key,
	PGPsdaArchiveCheckBytes	*checkBytes)
{
	PGPsdaArchiveCheckBytes	calcCheckBytes;
	PGPBoolean				bytesMatch = FALSE;
	
	HoldBuffer( &calcCheckBytes, sizeof( calcCheckBytes ) );
	
	if( IsntErr( ComputeCheckBytes( cipherProc, cipherContext, key,
				&calcCheckBytes ) ) )
	{
		bytesMatch = pgpMemoryEqual( checkBytes, &calcCheckBytes,
							sizeof( calcCheckBytes ) );
	}
	
	UnholdBuffer( &calcCheckBytes, sizeof( calcCheckBytes ), TRUE );

	return( bytesMatch );
}

	void
HoldBuffer(
	void 	*buffer,
	PGPSize	bufferSize)
{
	if( VirtualMemoryIsOn() )
	{
		HoldMemory( buffer, bufferSize );
	}
}

	void
UnholdBuffer(
	void 		*buffer,
	PGPSize		bufferSize,
	PGPBoolean	clearBuffer)
{
	if( clearBuffer )
	{
		pgpClearMemory( buffer, bufferSize );
	}

	if( VirtualMemoryIsOn() )
	{
		UnholdMemory( buffer, bufferSize );
	}
}

	OSStatus
AllocateSDABuffer(
	PGPSize 	bufferSize,
	SDABuffer	*buffer)
{
	OSStatus	err = noErr;
	
	pgpAssert( bufferSize != 0 );
	pgpAssertAddrValid( buffer, SDABuffer );

	pgpClearMemory( buffer, sizeof( *buffer ) );
	
	buffer->data = (PGPByte *) NewPtr( bufferSize );
	if( IsntNull( buffer->data ) )
	{
		buffer->size 	= bufferSize;
		buffer->used	= 0;
		buffer->pos		= 0;
	}
	else
	{
		err = memFullErr;
	}
	
	return( err );
}

	void
FreeSDABuffer(SDABuffer *buffer)
{
	pgpAssertAddrValid( buffer, SDABuffer );

	if( IsntNull( buffer->data ) )
	{
		DisposePtr( (Ptr) buffer->data );
	}
	
	pgpClearMemory( buffer, sizeof( *buffer ) );
}