/*

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]
The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.
*/

#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <assert.h>
#include <time.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <ctype.h>

#include "pgplib.h"

void
print_info(FILE *fd)
{
    
    char *s = 
"
   This keyserver is build on a library that makes it possible to use
   public keys in software (and not only on the command line by means of 
   PGP).
   There are routines to search PGP keyrings for keys, to generate packets 
   that are fully compatible with those PGP writes, and to parse (and 
   unarmor, uncompress and decrypt) packets written by PGP.  With it you 
   can also create files (literal, compressed and armored) that can be 
   manipulated by PGP.

   If the server turns out to be useful for the PGP community, that would be 
   nice since it is a spin-off rather than a target.  The server handles
   one connection at the time (rather than fork'ing) to avoid having to 
   reload the database for every request.

   The library has been written as part of the PASTA project; the security
   effort of PASTA is undertaken by Arne Helme (arne@acm.org) and myself,
   with the usual cast of students.  We are investigating the effects on
   system design when _personal_ machines are introduced.

   The cryptographic engine is SSLeay
      * ftp://ftp.psy.uq.oz.au/pub/Crypto/SSL/ - SSLeay source
      * ftp://ftp.psy.uq.oz.au/pub/Crypto/SSLapps/ - SSL applications
      * http://www.psy.uq.oz.au/~ftp/Crypto/ssl.html - SSLeay Programmer
         Reference
      * ftp://ftp.psy.uq.oz.au/pub/Crypto/SSLapps/PORT4-5 porting notes
   I have used the BN library, RSA-code and IDEA code.

   If you have any comments or suggestions, feel free to send me an 
   e-mail.

   Tage Stabell-Kul  (tage@acm.org)
   Department of Computer Science
   University of Troms
   Norway
";
    fprintf(fd, "%s\n", s);

}


FILE	*logfile;


void
get_key( char *arg, FILE *fd)
{
    u_long	short_keyid;
    u_quad_t	key_id, *keyptr;
    PGPuserid_t	**uid;
    int		num_keys;
    int		key_len;
    int		ret;
    char	*string;
    u_char 	*buf, *armor_buf;
    int		buflen, armor_len;
    /* scratch */
    int		index,i,j;

    key_id = 0LL;
    short_keyid = 0L;

    if ( ! strncmp(arg, "ARMOR ", 6)) {
	ret = sscanf(arg+6, "%qx", &key_id);
	if ( ret != 1 ) {
	    fprintf(fd, "
You should type \"ARMOR KeyID\" (without quotes) where KeyID is a real
KeyID in the format 0x033367ab71fbd469.  You can find the real KeyID
of any key with this server.\n");
	    return;
	}
	i = get_pub_key_record(key_id, &buf, &buflen);
	if ( i != 0 ) {
	    if ( i == -1 ) {
		fprintf(logfile, 
			"Error from get_pub_key_record on key %#.16llx\n",
			key_id);
		fprintf(fd, "Some error from get_pub_key_record\n");
		return;
	    }
	    else {
		fprintf(fd, "
No such key (are you sure you provided a real KeyID (64 bits long)).\n");
		return;
	    }
	    assert(0);
	}
	/* Found a key */
	i = armor_buffer(buf, buflen, (char **)&armor_buf, &armor_len);
	fprintf(fd, "%s\n", armor_buf);
	free(buf);
	free(armor_buf);
	return;
    } /* if ARMOR */

    if ( arg[0] == '0' && toupper(arg[1]) == 'X') {
	key_len = strlen(arg);
	switch ( key_len ) {
	case 10:
	    ret = sscanf(arg, "%lx", &short_keyid);
	    break;
	case 18:{

#ifdef __NetBSD__

	    ret = sscanf(arg, "%qx", &key_id);
#else

	    /* 
	     * arrrggghhhh.... this is FreeBSD
	     *
	     * We have verified that the string starts with 0[Xx], let us 
	     * discard two char's.  Then loop over the string ensuring
	     * the all chars are [0-9a-fA-F], then assign them two
	     * and two into a byte.
	     */
	    key_id = atoq(arg);
	    assert(key_id != 0);
#endif

	    break;
	}
	default:
	    ret = 0;
	    break;
	}

	if ( ret != 1 ) {
	    fprintf(fd, 
		    "[0x01234576 | 0x0123456789012345 | FooBar]\n");
	    return;
	}

	if ( short_keyid != 0 ) {
	    /* Find the real key */
	    int		i;
	    u_quad_t	tmpkey;
	    
	    i = find_shortid(arg, &tmpkey);
	    if ( i == -1 )
		return;
	    if ( tmpkey == 0 ) {
		fprintf(fd, "Found no matching key\n");
		return;
	    }
	    key_id = tmpkey;
	}
	/* A real key */
	i = get_pub_key_record(key_id, &buf, &buflen);
	switch ( i ) {
	case -1:
	    fprintf(fd, "Some error\n");
	    return;
	case 1:
	    fprintf(fd, "No (correct) key found\n");
	    return;
	case 0:
	    break;
	default:
	    assert(0);
	}
	/* Print what we found */
	print_keyrecords(buf, buflen, fd);
	free(buf);
	return;
    }
    else {
	/* Not a key but a string */
	string = arg;
	while ( isspace(*string) )
	    string += 1;
	
	if ( strlen(string) < 3 ) {
	    fprintf(fd, 
"Sorry, that is a little too short.  I will get exhausted if I am to find
all keys that matches such a sort pattern.\n");
	    return;
	}
	if ( !strcasecmp(string, "com") || !strcasecmp(string, "edu") ||
	     !strcasecmp(string, "org") || !strcasecmp(string, "acm"))
	{
	    fprintf(fd,
"Can it be necessary to check whether there are UserID's with e-mail 
addresses within them ?
If you _really_, _really_ need to torment me like this, try looking for
.com (or .edu or .org) instead.\n");
	    return;
 	}
	if ( !strcasecmp(string, ".com") || !strcasecmp(string, ".edu") ||
	     !strcasecmp(string, ".org")) {
	    fprintf(fd,
"Well, sorry.  If you just want to be a pain-in-the-back I am sure you
can find other means.  If you have any (any at all) useful reason for this
request, I will be very happy to hear from you.  Try \"INFO\" instead.\n");
	    return;
 	}

	/* See if the string has been saved exactly */
	
	num_keys = find_keys_from_string(string, 0, &keyptr);
	    
	if ( num_keys == -1) {
	    fprintf(fd,  "Some error in find_userid\n");
	    return;
	}
	if ( num_keys == 0 ) {
	    fprintf(fd, "Found no keys that matched\n");
	    return;
	}
	fprintf(fd, "Found these keys:\n");
	for(index=0; index < num_keys; index++ ) {
	    fprintf(fd, "\t%#.16llx\n", keyptr[index]);
	    uid = NULL;
	    i = find_uids_with_key(keyptr[index], &uid);
	    if ( i == -1 ) {
		fprintf(fd, "Some internal error (code 0x56345)\n");
		fprintf(logfile, "Some internal error\n");
		return;
	    }
	    if ( i == 1 ) {
		fprintf(fd, "No matching UID\n");
		continue;
	    }
	    assert(i==0);
	    for(j=0; uid[j] != NULL; j++) {
		fprintf(fd, "\t\t%s\n", uid[j]->name);
		free_uid(uid[j]);
	    }
	    assert(uid != NULL);
	    free(uid);
	}
	free(keyptr);
	return;
    }
}



void
usage(void)
{
    fprintf(stderr, "keyserver [-d ] -p port -f file [-l file]\n");
    exit(1);
}

#define MAXLINE  80

char Arg[MAXLINE];


int
main(int argc, char *argv[])
{

    int		ch;
    extern char	*optarg;
    int		port;
    char 	*database;
    int		server;
    int		s;
    struct sockaddr_in	sin;

    /* scratch */
    char	*pp;
    int		i;
    
    port = 0;
    server = 0;
    logfile  = NULL;
    database = NULL;
    while ((ch = getopt (argc, argv, "dp:f:l:")) != -1) {
	switch (ch) {
	case 'd':
	    server = 1;
	    break;
	case 'f':
	    database = optarg;
	    i = keydb_init(optarg, READER);
	    if ( i != 0 )
		usage();
	    break;
	case 'l':
	    logfile = fopen(optarg, "a+");
	    if ( logfile == NULL ) {
		perror("fopen");
		usage();
	    }
	    i = setvbuf(logfile, (char *)NULL, _IOLBF, 0);
	    if ( i == EOF ) {
		fprintf(stderr, "Could not set vbuf\n");
		exit(1);
	    }
	    break;
	case 'p':
	    errno = 0;
	    i = (int) strtol(optarg, &pp, 10);
	    if ( i > 0xffff) {
		fprintf(stderr, "Portnumber to large\n");
		usage();
	    }
	    if ( pp == NULL || *pp != 0)
		usage();
	    port = i;
	    break;
	case '?':
	default:
	    usage();
	}
    }
    
    if ( port == 0 )
	usage();
    if ( logfile == NULL )
	logfile = stderr;
    if ( database == NULL )
	usage();
    s = socket(AF_INET, SOCK_STREAM, 0);
    if ( s == -1 ) {
	perror("socket");
	exit(1);
    }

    memset(&sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_port = htons(port);
    sin.sin_addr.s_addr = htonl(INADDR_ANY);
    i = bind(s, (struct sockaddr *) &sin, sizeof(sin));
    if ( i < 0 ) {
	perror("bind");
	exit(1);
    }
    listen(s, 5);

    if ( server == 1 ) {
	if ( daemon(1, 0) < 0 ) {
	    perror("daemon");
	    exit(1);
	}
    }

    /* 
     * If you didn't specify -l but did use -d, your error messages
     * are lost at this point 
     */


/* **************************************************************
 * OK, here is the keyserver.  We accept a single connection,
 * read the request, and try to do something with it.
 * ************************************************************** */
    
    
    while ( 1 ) {
	int 			sock, len;
	struct sockaddr_in 	from;
	struct hostent		*hp;
	char  		what_time[26];	/* Local representaion of the time */
	struct timeval 	started, now;	/* To get the above */
	double		total_cpu, sess_cpu;
	int		total_sec;
	fd_set		selfd;
	struct timeval	tval;
	FILE		*fd;
	char		*p;

	char        line[MAXLINE];

	len = sizeof(from);

	sock = accept(s, (struct sockaddr *)&from, &len);
	if ( sock < 0 ) {
	    fprintf(logfile, "%s : %s\n", what_time, strerror(errno));
	    continue;
	}

	gettimeofday(&started, (struct timezone *) 0);
	sprintf(what_time, "%s", ctime( (time_t*)&started.tv_sec));
	
	total_cpu = clock();

	hp = gethostbyaddr((char *)&from.sin_addr, len, AF_INET);
	if ( hp != NULL ) {
	    fprintf(logfile, "%s [%s] %s", hp->h_name, 
		    inet_ntoa(from.sin_addr), what_time);
	}
	else {
	    fprintf(logfile, "Error from gethostbyaddr: %s\n", 
		    hstrerror(h_errno));
	}

	fd = fdopen(sock, "a+");
	if ( fd == NULL ) {
	    fprintf(logfile, "fdopen: %s\n", strerror(errno));
	    close(sock);
	    continue;
	}	

	fprintf(fd, "\nWelcome to the PASTA keyserver.  Local time is %s\n",
		what_time);

	fprintf(fd,
"Type a string (\"Tage Stabell-Kul\") which will be searched for 
among the available UserIDs (not found due to the Norvegian character), 
a UserID (\"Tage Stabell-Kulo <tage@acm.org>\") which will be looked 
up (or, failing that, searched for), a short keyID the way PGP 
prints them (\"0x71FBD469\") or a real keyID the way PGP actually 
stores them  (\"0x033367ab71fbd469\") or the word \"INFO\" (IN CAPITALS).  
Do not expect the result to look like what you get from PGP.
  
If you do ARMOR 0x033367ab71fbd469 (or some other KeyID) you will  
get that key with associated UserID and sertificates armored (PGP -kxa).
The usefulness of this was pointed out by Peter N. Wan.

The server is running on an old HP9000/433 (that is MC68040).\n\n
>");
	fflush(fd);

	FD_ZERO(&selfd);
	FD_SET(sock, &selfd);
	/* Let the other end have 30 sec's to read this text. */
	tval.tv_sec = 30;
	tval.tv_usec = 0;
	i = select(32, &selfd, NULL, NULL, &tval);
	if ( i == -1 ) {
	    fprintf(logfile, "Error from select\n");
	    perror("select");
	    exit(1);
	}
	if ( i == 0 ) {
	    fprintf(fd, "Timeout - Sorry\n");
	    fclose(fd);			/* fclose fflush'es */
	    close(sock);
	    continue;
	}
	if ( fgets(line, MAXLINE, fd) == NULL) {
	    fclose(fd);
	    close(sock);
	    continue;
	}

	fprintf(logfile, "\t%s", line);

	if ( strncmp(line, "INFO\r\n", 6)) {
	    /* Not INFO */
	    p = rindex(line, '\r');
	    if ( p != NULL)
		*p = NULL;
	    /* Then, just to be sure */
	    p = rindex(line, '\n');
	    if ( p != NULL ) 
		*p = NULL;

	    /* Keep a copy of the arguments on the command line */
	    strncpy(Arg, line, MAXLINE);

	    fprintf(fd, "----------------\n");
	    get_key(line, fd);
	    fprintf(fd, "----------------");
	}
	else {
	    /* Asked for INOF */
	    print_info(fd);	
	}

	/* Tell him how much time we used */
	gettimeofday(&now, (struct timezone  *)0);
	
	if (started.tv_usec > now.tv_usec) {
	    now.tv_usec += 1000000;
	    now.tv_sec--;
	}
	total_sec =  (now.tv_sec  - started.tv_sec);
	
	
	sess_cpu = clock() - total_cpu;

	fprintf(logfile, "\t%d seconds (of which %.2f CPU seconds)\n", 
		total_sec, sess_cpu / (double)CLOCKS_PER_SEC);

	fprintf(fd, "\n\tYou spent %d seconds, (of which %.2f sec CPU)\n", 
		total_sec, sess_cpu / (double)CLOCKS_PER_SEC);
	fprintf(fd, "\tIn total, %.2f seconds CPU consumed by this server\n",
		clock()/(double)CLOCKS_PER_SEC);
	fprintf(fd, 
"\n\tThank you for using this experimental keyserver
\tPlease report any problems (or suggestions) to 
\tTage Stabell-Kul (tage@acm.org).\n\n");

	fclose(fd);
	close(sock);
	continue;
	
    }
}
