/*  Hey Emacs, switch to -*-C-*-, please 
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.
*/


#include <errno.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

/* SSLeay*/
#include <rsa.h>
#include <md5.h>

#include "pgplib.h"
#include "pgp_m.h"


/*
 * Convert a signature into a buffer.
 */

int
signature2buf(PGPsig_t	*sig, 
	      u_char	**buf)
{
    int 	sofar;
    u_char 	*tmp;
    u_short	us;
    u_long	ul;
    /* scratch */
    int		i;

    /* If the signature has been generated (rather than read) we
     * don't know the size it will occupy.  According to PGFORMAT.DOC
     * the MPI starts at position 22.  The added 2 is for the MPI
     * prefix.
     */
    if ( sig->length == 0 )
	sig->length = BN_num_bytes(sig->I) + 22 + 2;

    tmp = malloc(sig->length);
    if ( tmp == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    
    sofar = 0;
    tmp[sofar] = CTB_SKE;
    sofar++;


    /* Length is 2 bytes */
    assert ( ctb_llength(*tmp) == 2 );

    us = htons( (u_short) sig->length - 1 - 2);
    memcpy(tmp+sofar, &us, 2);
    sofar += 2;
    
    tmp[sofar] = sig->version;
    sofar++;

    tmp[sofar] = sig->included;
    sofar++;

    tmp[sofar] = sig->classification;
    sofar++;

    ul = htonl(sig->timestamp);
    memcpy(tmp+sofar, &ul, 4);
    sofar += 4;
    
    copy_quad(sig->key_id, tmp+sofar);
    sofar += 8;

    tmp[sofar] = sig->pk_alg;
    assert(sig->pk_alg == RSA_ALGORITHM_BYTE);
    sofar++;

    tmp[sofar] = sig->hash_alg;
    assert(sig->hash_alg = MD5_ALGRITHM_BYTE);
    sofar++;

    tmp[sofar] = sig->ck[0];
    sofar += 1;
    tmp[sofar] = sig->ck[1];
    sofar += 1;

    /* MPI prefix */
    i = BN_num_bits(sig->I);
    us = htons((u_short) i);
    memcpy(tmp+sofar, &us, 2);
    sofar += 2;

    i = BN_num_bytes(sig->I);

    if ( sofar + i > sig->length ) {
	fprintf(stderr, "Inconsistency, signature buffer too small\n");
	free(tmp);
	return(-1);
    }
    (void) BN_bn2bin(sig->I, tmp+sofar);

    assert(sofar+i == sig->length);

    *buf = tmp;
    return(sofar+i);
}    

/*
 * Convert a buffer into a signature.  What the signature is for
 * can be seen in the signature byte.
 */

int
buf2signature(u_char	*buf, 
	      int 	buflen, 
	      PGPsig_t 	*sig)
{
    u_long	l;
    int	        i;
    u_char 	*q;	
    int		howfar;

    q = 0;

    if ( buf == NULL || sig == NULL )
	return(-1);

    if ( ! is_ctb_type(*buf, CTB_SKE_TYPE) ) {
	fprintf(stderr, "buf2sig called with wrong packet\n");
	return(-1);
    }
    
    howfar = 0;
    memset(sig, 0, sizeof(PGPsig_t));

    /* skip CTB */
    howfar += 1;
    
    /* and length */
    howfar += ctb_llength(*buf);
    
    sig->version = buf[howfar];
    if (sig->version != VERSION_BYTE_NEW && sig->version != VERSION_BYTE_OLD) {
	fprintf(stderr, "Probably not correct version\n");
	goto bad;
    }
    howfar += 1;
    
    sig->included = buf[howfar];
    howfar += 1;
    if ( sig->included != 5 ) {
	fprintf(stderr, "Probably not correct implicit\n");
	goto bad;
    }
    
    sig->classification = buf[howfar];
    howfar += 1;

    l = 0;
    memcpy( &l, buf + howfar, SIZEOF_TIMESTAMP);
    sig->timestamp = ntohl(l);
    if ( sig->timestamp == 0 ) {
	fprintf(stderr, "Probably incorrect timestamp\n");
	goto bad;
    }
    howfar += SIZEOF_TIMESTAMP;

    sig->key_id = to_quad(buf+howfar);
    howfar += 8;

    sig->pk_alg = buf[howfar];
    if ( sig->pk_alg !=  RSA_ALGORITHM_BYTE ) {
	fprintf(stderr, "Probably incorrect pk_algorithm\n");
	goto bad;
    }
    howfar += 1;

    sig->hash_alg= buf[howfar];
    if ( sig->hash_alg !=  MD5_ALGRITHM_BYTE) {
	fprintf(stderr, "Probably incorrect ck_algorithm\n");
	goto bad;
    }
    howfar += 1;

    sig->ck[0] = buf[howfar];
    howfar += 1;
    sig->ck[1] = buf[howfar];
    howfar += 1;

    i = mpi2bignum(buf+howfar, buflen - howfar, &sig->I);
    if ( i == 0 )
	goto bad;
    howfar += i;

    sig->length = howfar;

    return(0);

bad:
    BN_free(sig->I);
    free(q);
    free(sig);
    return(-1);
}

void
free_signature(PGPsig_t *sig)
{
    if ( sig == NULL)
	return;
    if ( sig->I != NULL )
	BN_free(sig->I);
}


/* The following is from PGFORMAT.DOC:
 * ===========================
 * 
 * Message digest "packet"
 * -----------------------
 * 
 * The Message digest has no CTB packet framing.  It is stored
 * packetless and naked, with padding, encrypted inside the MPI in the
 * Signature packet.  
 * 
 * PGP versions 2.3 and later use a new format for encoding the message
 * digest into the MPI in the signature packet, a format which is
 * compatible with RFC1425 (formerly RFC1115).  This format is accepted
 * but not written by version 2.2.  The older format used by versions 2.2
 * is acepted by versions up to 2.4, but the RSAREF code in 2.5 is
 * not capable of parsing it.
 *
 * PGP versions 2.3 and later encode the message digest (MD) into the 
 * MPI as follows:
 *       MSB               .   .   .                  LSB
 *        0   1   FF(n bytes)   0   ASN(18 bytes)   MD(16 bytes)
 * See RFC1423 for an explanation of the meaning of the ASN string.
 * It is the following 18 byte long hex value:
 *        3020300c06082a864886f70d020505000410
 * Enough bytes of FF padding are added to make the length of this
 * whole string equal to the number of bytes in the modulus.
 * 
 * There is no checksum included.  The padding serves to verify that the
 * correct RSA key was used.
 * =============== (End of PGFORMAT.DOC)

 * When trying to understand how these routines work, remember that the 
 * only time you use your secret key is to _sign_ a hash (of something)
 * and the only time you use someones public key is to _verify_ that
 * they used their secret key.  In other words, private_encrypt is only
 * used (by you) when you make a signature (on a hash).  Likevise, you
 * use public_decrypt to verify the someone else used their secret key to
 * encrypt a hash.  And so, you use public_encrypt to encrypt something
 * that only the other party will be able to recover (always a DEK, what
 * else ?).
 */

static int
make_bufferhash(u_char		*data,
		int		dlen,
		PGPsig_t	*sig,
		u_char		digest[16])
{	
    u_long	ul;
    MD5_CTX	context;

    assert(sig->hash_alg == MD5_ALGRITHM_BYTE);

    MD5_Init(&context);
    MD5_Update(&context, data, dlen);

    /* 5 = 1 + 4 => signature classification + timestamp */
    assert(sig->included == 5);

    MD5_Update(&context, &sig->classification, 1);
    ul = htonl(sig->timestamp);
    MD5_Update(&context, (unsigned char *)&ul, 4);

    MD5_Final(digest, &context);
    return(0);
}
 

/* 
 *  ASN.1 Distinguished Encoding Rules (DER) encoding for RSA/MD5.
 */
static u_char asn1_array[] = {		/* PKCS 01 block type 01 data */
    0x30,0x20,0x30,0x0c,0x06,0x08,
    0x2a,0x86,0x48,0x86,0xf7,0x0d,
    0x02,0x05,0x05,0x00,0x04,0x10 };


/*
 * Verify that a signature is correct.  We take a public key,
 * a signature and a buffer (with a length).  Create a hash and 
 * compare it with what we find inside the sig->I, using the ASN.1
 * to verify decryption.
 * Return -1 on error, 1 on incorrect decryption and 0 on sucess.
 */

int
verify_signature(u_char		*data,
		 int		dlen,
		 PGPsig_t   	*sig,
		 PGPpubkey_t	*signkey)
{
    RSA		*rsa_info;
    u_char	*sigbuf;
    u_char	our_digest[16+2]; /* The extra is checksum */
    u_char	*found_digest;
    int		buflen;
    int		ret;
    /* scratch */
    int		i, j;

    if ( sig == NULL || signkey == NULL || data == NULL || dlen == 0 ) {
	fprintf(stderr, "Illegal parameter\n");
	return(-1);
    }
    
    /* We have to make sig->I into a buffer.  This is a pity since rsa_*
     * turns it into a number again but they want a buffer.
     */
    buflen = BN_num_bytes(sig->I);
    sigbuf = calloc(buflen, 1);
    if ( sigbuf == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    /* We make buflen just as large, to be sure.  In fact, there is 
     * no other size that makes sense.
     */
    found_digest =  calloc(buflen, 1);
    if ( found_digest == NULL ) {
	fprintf(stderr, "Out of memory\n");
	free(sigbuf);
	return(-1);
    }
    BN_bn2bin(sig->I, sigbuf);
    rsa_info = RSA_new();
    rsa_info->n = BN_dup(signkey->N);
    rsa_info->e = BN_dup(signkey->E);

    assert( sig->pk_alg == RSA_ALGORITHM_BYTE);

    i = RSA_public_decrypt(buflen, sigbuf, found_digest, rsa_info);
    RSA_free(rsa_info);
    free(sigbuf);

    if ( i == -1 ) {
	fprintf(stderr, "Didn't decrypt properly\n");
	free(found_digest);
	return(-1);
    }
    /* The result should contain 18 bytes of ASN.1 data followed
     * by 16 bytes of MD5 (= 34 bytes).
     */
    if ( i != 16 + sizeof(asn1_array) ) {
	free(found_digest);
	return(1);
    }
    /* 
     * Verify that the correct key was used by comparing the ASN.1 
     * data.
     */
    for(i=0; i < sizeof(asn1_array); i++ ) {
	if ( found_digest[i] != asn1_array[i] ) {
	    free(found_digest);
	    return(1);
	}
    }

    /* Construct our own hash */
    i = make_bufferhash(data, dlen, sig, our_digest);
    if ( i == -1 ) {
	fprintf(stderr, "Could not make hash\n");
	free(found_digest);
	return(-1);
    }
    
    /* 
     * Well, are they identical ? 
     * An interesting question is: What if the digests are identical but
     * the checksums are different ?
     * Since I don't know the answer, I don't known what to do if this
     * turns out to be the case; I'll ignore it (suggestions
     * are most welcome).
     */
    ret = 0;
    for(j = sizeof(asn1_array), i=0; i < 16 ; i++ ) {
	if ( our_digest[i] != found_digest[i+j]) {
	    ret = 1;
	    break;
	}
    }	
    free(found_digest);
    return(ret);
}


/*
 * To verify a key-uid binding we concatenate them and verify
 * if the signature is correct.
 */

int
verify_keysig(PGPpubkey_t	*pubkey,
	      PGPuserid_t	*uid,
	      PGPsig_t		*sig,
	      PGPpubkey_t	*signkey)
{
    int 	len;
    int		ret;
    u_char	*buf, *p;
    int		hlen; /* not used but don't delete, see the ifdef below */

    len = pubkey2buf(pubkey, &buf);
    if ( len == -1 )
	return(-1);

/* **************************************************************
 * There is a BUG in the way PGP makes signatures on keys.
 * According to PGFORMAT.DOC (line 220) this is what is supposed to
 * happen:
 *   The packet headers (CTB and length fields) for the public key packet
 *   and the user ID packet are both omitted from the signature
 *   calculation for a key certification.  
 * However, the key is read as it appears on file, contrary to what it 
 * it supposed to do.
 * 
 * I not able to evaluate the security implications of this.  I suggest
 * you send comments to Stale Schumacher at pgp-bugs@ifi.uio.no.
 * He is known to take your comments seriously.
 * 
 * ************************************************************** */

#ifdef NotDef
    hlen = 1 + ctb_llength(buf[0]);	/* CTB + length */
#else
    hlen = 0;
#endif

    p = realloc(buf, len+strlen(uid->name));
    if ( p == NULL ) {
	free(buf);
	return(-1);
    }
    buf = p;
    memcpy(buf+len, uid->name, strlen(uid->name));
    /* hlen then used here to offset over the CTB and length */
    ret = verify_signature(buf+hlen, len-hlen+strlen(uid->name), 
			 sig, signkey);
    return(ret);
}


/*
 * Sign a buffer with a secret key.  The result is places in a
 * PGPsec_t which can be converted into a PGP package if required.
 * The signature->classification must be set by the caller (there is
 * no way we can know).
 */

int
make_signature(u_char		*buffer, 
	       int 		bufsize,
	       u_char		class,
	       PGPseckey_t	*key, 
	       PGPsig_t 	*sig)
{
    
    u_char 	mdigest[18+16];
    RSA		*rsa_info;
    u_char	*rsa_res;
    int		rsa_len;
    /* scratch */
    int		i;
    	
    assert( sizeof(asn1_array) == 18 );

    if ( key == NULL || buffer == NULL || bufsize < 1 || sig == NULL ) {
	fprintf(stderr, "Illegal parameter\n");
	return(-1);
    }
    
    sig->classification = class;

    sig->version = VERSION_BYTE_NEW;
    sig->timestamp = (u_long) time(NULL);
    sig->included = 5;
    sig->pk_alg = RSA_ALGORITHM_BYTE;
    sig->hash_alg = MD5_ALGRITHM_BYTE;
    sig->key_id = BN_get_keyid(key->N);

    memcpy(mdigest, asn1_array, 18);
    i = make_bufferhash(buffer, bufsize, sig, mdigest+18);
    
    /* First two bytes of the hash (for some definition of "first") */
    memcpy(sig->ck, mdigest+18, 2);

    rsa_info = RSA_new();
    rsa_info->d = key->D;
    rsa_info->n = key->N;
    rsa_res = malloc(RSA_size(rsa_info));
    if ( rsa_res == NULL ) {
	RSA_free(rsa_info);
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }

    rsa_len = RSA_private_encrypt(18+16, mdigest, rsa_res, rsa_info);
    if ( rsa_len == -1 ) {
	RSA_free(rsa_info);
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }

    sig->I = BN_bin2bn(rsa_res, rsa_len, NULL);

    return(0);
}

int
sign_buffer(u_char	*buffer, 
	    int 	bufsize,
	    PGPseckey_t	*key, 
	    PGPsig_t 	*sig)

{
    return(make_signature(buffer, bufsize, SB_SIGNATURE_BYTE, key, sig));
}
