package Ssh;

use strict;
use Expect;
use IO::Stty;
use Carp;

my $Debugging = 0;
my $Verbose = 0;

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self = {};
    bless ($self, $class);
    $self->initialize();
    return $self;
}

sub initialize {
    my $self = shift;
    $self->{"-printsub"} = \&defaultprintsub;
    $self->{"-inputsub"} = \&defaultinputsub;
    $self->{"-passwordsub"} = \&defaultpasswordsub;
    $self->{"-yesnodialog"} = \&defaultyes_no_dialog;
    $self->{"-inputdialog"} = \&defaultinput_dialog;
    $self->{"-passworddialog"} = \&defaultpassword_dialog;
}

sub user {
    my ($self,$user) = @_;

    $self->{"-user"} = $user;
}

sub keyfile {
    my ($self,$keyfile) = @_;

    $self->{"-keyfile"} = $keyfile;
}

sub cipher {
    my ($self,$cipher) = @_;

    $self->{"-cipher"} = $cipher;
}

sub port {
    my ($self,$port) = @_;

    $self->{"-port"} = $port;
}

sub remoteuser {
    my ($self,$remoteuser) = @_;

    $self->{"-remoteuser"} = $remoteuser;
}

sub remotehost {
    my ($self,$remotehost) = @_;

    $self->{"-remotehost"} = $remotehost;
}


sub debug {
    my $self = shift;
    confess "usage: thing->debug(level)"    unless @_ == 1;
    my $level = shift;
    if (ref($self))  {
	$self->{"_DEBUG"} = $level;         # just myself
    } else {
	$Debugging        = $level;         # whole class
    }
}

sub verbose {
    my $self = shift;
    confess "usage: thing->verbose(level)"    unless @_ == 1;
    my $level = shift;
    if (ref($self))  {
	$self->{"_VERBOSE"} = $level;         # just myself
    } else {
	$Verbose = $level;         # whole class
    }
}

sub DESTROY {
    my $self = shift;
    if ($Debugging || $self->{"_DEBUG"}) {
	carp "Destroying $self " . $self->name;
    }
}

sub secure_copy {
    my ($self,@args) = @_;

    my $i;
    my $totalargs = @args;
    my %args;
    for($i=0;$i<$totalargs-1;$i+=2) {
	my $key = $args[$i];
	my $value = $args[$i+1];
	$args{$key} = $value;
    }
    my $remotesystem = $args{'-gateway'} || $args{'-remotesystem'} || $self->{'-gateway'} || $self->{'remotesystem'} or return (-1,"No Remote System or Gateway specified.");
    my $filename = $args{'-filename'} || return (-1,"No file specified!");
    my $remoteuser = $args{'-remoteuser'} || $self->{'-remoteuser'};
    my $remotepath = $args{'-remotepath'} || $self->{'-remotepath'};
    
    my @parameters = ( "-v", "-C", "${filename}");
    croak "No remote host specified" if (!$self->{"-remotehost"});
    if ($args{"-keyfile"} || $self->{"-keyfile"}) {
	my $keyfile = $args{"-keyfile"} || $self->{"-keyfile"};
	push(@parameters,"-i",$keyfile);
    }
    if ($args{"-path"} || $self->{"-path"}) {
	my $path = $args{"-path"} || $self->{"-path"};
	push(@parameters,"-P",$path);
    }
    if ($args{"-cipher"} || $self->{"-cipher"}) {
	my $cipher = $args{"-cipher"} || $self->{"-cipher"};
	push(@parameters,"-c",$cipher);
    }
    if ($args{'-remoteuser'} || $self->{'-remoteuser'}) {
	my $remoteuser = $args{'-remoteuser'} || $self->{'-remoteuser'};
	push(@parameters,"$remoteuser\@$remotesystem:$remotepath"); 
    }
    else {
	my $remotesystem = $self->{"-remotehost"};
	push(@parameters,"$remotesystem:$remotepath");
    }
    my $expectobj = Expect->spawn("scp", @parameters);
    if ($Debugging || $self->{"_DEBUG"}) {
	$expectobj->log_stdout(1);
    }
    else {
	$expectobj->log_stdout(0);
    }
    my ($error, $retval) = $self->ssh_core($expectobj,undef);
    return ($error,$retval);
}

sub secure_shell {
    my ($self,@args) = @_;

    my $i;
    my $totalargs = @args;
    my %args;
    for($i=0;$i<$totalargs-1;$i+=2) {
	my $key = $args[$i];
	my $value = $args[$i+1];
	$args{$key} = $value;
    }
    my $remotesystem = $args{'-gateway'} || $args{'-remotesystem'} || $self->{'-gateway'} || $self->{'remotesystem'} or return (-1,"No Remote System or Gateway specified.");
    my $remotecommand = $args{'-remotecommand'} || $args{'-remotecommand'} or return (-1,"No Remote Command specified.");
    my $expectstring = $args{'-expectstring'} || $self->{'-expectstring'} || undef;

    my @parameters = ( "-v", "-C");
    croak "No remote host specified" if (!$self->{"-remotehost"});
    if ($args{"-keyfile"} || $self->{"-keyfile"}) {
	my $keyfile = $args{"-keyfile"} || $self->{"-keyfile"};
	push(@parameters,"-i",$keyfile);
    }
    if ($args{"-path"} || $self->{"-path"}) {
	my $path = $args{"-path"} || $self->{"-path"};
	push(@parameters,"-P",$path);
    }
    if ($args{"-cipher"} || $self->{"-cipher"}) {
	my $cipher = $args{"-cipher"} || $self->{"-cipher"};
	push(@parameters,"-c",$cipher);
    }
    if ($self->{"-remoteuser"}) {
	my $remoteuser = $self->{"-remoteuser"};
	push(@parameters,"-l",$remoteuser);
    }
    push(@parameters,"$remotesystem", "$remotecommand");

    my $expectobj = Expect->spawn("ssh", @parameters);
    if ($Debugging || $self->{"_DEBUG"}) {
	$expectobj->log_stdout(1);
    }
    else {
	$expectobj->log_stdout(0);
    }
    my ($error, $retval) = $self->ssh_core($expectobj,$expectstring);
    return ($error,$retval);
}

sub ssh_ppp {
    my ($self,$localip,$remoteip,$timer) = @_;
    
    my $parameters;
    my $remotehost = $self->{"-remotehost"} || $self->{"-gateway"};
    croak "No remote host specified" if (!$remotehost);
    croak "No keyfile specified" if (!$self->{"-keyfile"});
    my $keyfile = $self->{"-keyfile"};
    $parameters = "ssh-ppp -i$keyfile -h \"115200 lock debug modem crtscts ${localip}:${remoteip}\" -o \"115200 lock debug modem crtscts ${remoteip}:${localip}\" -b$timer $remotehost &";
    
    system($parameters);
    sleep($timer);
}

sub ssh_core {
    my ($self,$expectobj,$expectstring) = @_;

    my ($matched_pattern_position,$error,$successfully_matching_string,$before_match,$after_match);
    $matched_pattern_position = 0;
    $successfully_matching_string = "";
    $error = 0;
    while ((defined ($matched_pattern_position)) && ($matched_pattern_position < 7) && ((!defined($error)) || (!$error))) {
	if (defined (@$expectstring)) {
	    ($matched_pattern_position,$error,$successfully_matching_string,$before_match,$after_match) = $expectobj->expect(30, '-re', 'Enter passphrase for RSA key [^:]+: ', '-re', '^.*?s password: ', 'Host key not found from the list of known hosts.', 'Are you sure you want to continue connecting (yes/no)? ','Permission denied', 'lost connection', @$expectstring);
	}
	else {
	    ($matched_pattern_position,$error,$successfully_matching_string,$before_match,$after_match) = $expectobj->expect(30, '-re', 'Enter passphrase for RSA key [^:]+: ', '-re', '^.*?s password: ', 'Host key not found from the list of known hosts.', 'Are you sure you want to continue connecting (yes/no)? ','Permission denied', 'lost connection');
	}
	if (defined($error)) {
	    undef $error  if ($error eq "2:EOF");
	}
	if (defined($matched_pattern_position)) {
	    my $string;
	    if ($matched_pattern_position == 1) {
		$string = $self->password_dialog($successfully_matching_string);
		print $expectobj "$string\r";
	    }
	    elsif ($matched_pattern_position == 2) {
		$string = $self->password_dialog($successfully_matching_string);
		print $expectobj "$string\r";
	    }
	    elsif ($matched_pattern_position == 4) {
		$string = "";
		while (($string ne "yes") && ($string ne "no")) {
		    my $alertstring;
		    if ($before_match =~ /HOST IDENTIFICATION HAS CHANGED/) {
			$alertstring = "Host Identification has changed, someone could be doing something nasty. $successfully_matching_string ";
		    }
		    else {
			$alertstring = "Host Key not in known_hosts. $successfully_matching_string ";
		    }
		    $string = $self->yes_no_dialog($alertstring);
		}
		print $expectobj "$string\r";
		if ($string eq "no") {
		    return (-1, "User interrupted.");
		}
	    }
	    elsif ($matched_pattern_position == 5) {
		return(-1,$successfully_matching_string);
	    }
	    elsif ($matched_pattern_position == 6) {
		return(-1,$successfully_matching_string);
	    }
	}
    }
    my ($matched_pattern_position1,$error1,$successfully_matching_string1,$before_match1,$after_match1) = $expectobj->expect(30,'Bad passphrase', 'Permission denied', 'lost connection', undef);
    if (@$expectstring) {
	return ($error1,$successfully_matching_string);
    }
    return ($error1,$successfully_matching_string1);
}

sub defaultprintsub {
    my $self = shift;

    print @_;
}

sub defaultinputsub { 
    my $self = shift;

    my $input = <>; 
    return $input; 
}

sub defaultpasswordsub {
    my $self = shift;

    my $old_mode=IO::Stty::stty(\*STDIN,'-g');
    # Turn off echoing.
  IO::Stty::stty(\*STDIN,'-echo');
    # Do whatever.. grab input maybe?
    my $read_password = $self->inputsub();
    # Now restore the old mode.
  IO::Stty::stty(\*STDIN,$old_mode);
    chop $read_password;
    return $read_password;
}

sub defaultinput_dialog {
    my $self = shift;

    $self->printsub();
    my $string = $self->inputsub();
    return $string;
}

sub defaultyes_no_dialog {
    my $self = shift;

    $self->printsub();
    my $string = $self->inputsub();
    return $string;
}

sub defaultpassword_dialog {
    my $self = shift;

    $self->printsub();
    my $string = $self->passwordsub();
    return $string;
}

sub printsub {
    my $self = shift;

    &{ $self->{"-printsub"} }($self,@_);
}

sub inputsub {
    my $self = shift;

    &{ $self->{"-inputsub"} }($self,@_);
}

sub passwordsub {
    my $self = shift;

    &{ $self->{"-passwordsub"} }($self,@_);
}

sub input_dialog {
    my $self = shift;

    &{ $self->{"-inputdialog"} }($self,@_);
}

sub yes_no_dialog {
    my $self = shift;

    &{ $self->{"-yesnodialog"} }($self,@_);
}

sub password_dialog {
    my $self = shift;

    &{ $self->{"-passworddialog"} }($self,@_);
}

sub change_printsub {
    my $self = shift;

    $self->{"-printsub"} = shift;
}

sub change_inputsub {
    my $self = shift;

    $self->{"-inputsub"} = shift;
}

sub change_passwordsub {
    my $self = shift;

    $self->{"-passwordsub"} = shift;
}

sub change_input_dialog {
    my $self = shift;

    $self->{"-inputdialog"} = shift;
}

sub change_yes_no_dialog {
    my $self = shift;

    $self->{"-yesnodialog"} = shift;
}

sub change_password_dialog {
    my $self = shift;

    $self->{"-passworddialog"} = shift;
}

1;
