/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995-1997  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997-1998  Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include "freedom.h"

int
pgpdecrypt(const char *encrypted, const char *decrypted)
{
  int lockfd, passfd, status;
  pid_t pid = getpid();
  char passenv[BUFSIZ], pathenv[BUFSIZ];
  FILE *infile, *outfile;

  /* PGP needs exclusive access to randseed.bin */
  lockfd = open("pgp.lock", O_WRONLY | O_CREAT | O_TRUNC, 0660);
  flock(lockfd, LOCK_EX);
  write(lockfd, &pid, sizeof(pid_t));

  /* call PGP to do decryption */
  if (!fork()) { /* child */
    fclose(stderr); /* shut up! */
    infile = fopen(encrypted, "r");
    dup2(fileno(infile), STDIN_FILENO);
    fclose(infile);
    outfile = fopen(decrypted, "w");
    dup2(fileno(outfile), STDOUT_FILENO);
    fclose(outfile);
    passfd = open(PASS_FILE, O_RDONLY);
    sprintf(passenv, "PGPPASSFD=%d", passfd);
    putenv(passenv);
    sprintf(pathenv, "PGPPATH=%s", PGP_DIR);
    putenv(pathenv);
    if (USE_SYSLOG)
      syslog(LOG_DEBUG, "Decrypting message");
    execl(PGP, basename(PGP), "+batchmode", "+verbose=0", "-f", NULL);
  }
  else /* parent */
    wait(&status);

  flock(lockfd, LOCK_UN);
  close(lockfd);
  return status;
}

int
encrypt_key(const char *toencrypt, const char *toappendto, const char *key)
{
  int lockfd, status;
  pid_t pid = getpid();
  char pathenv[BUFSIZ];
  FILE *infile, *outfile;

  /* PGP needs exclusive access to randseed.bin */
  lockfd = open("pgp.lock", O_WRONLY | O_CREAT | O_TRUNC, 0660);
  flock(lockfd, LOCK_EX);
  write(lockfd, &pid, sizeof(pid_t));

  /* call PGP to do encryption and append encrypted text to message */
  if (!fork()) { /* child */
    fclose(stderr); /* shut up! */
    infile = fopen(toencrypt, "r");
    dup2(fileno(infile), STDIN_FILENO);
    fclose(infile);
    outfile = fopen(toappendto, "a");
    fputs("**\n", outfile);
    dup2(fileno(outfile), STDOUT_FILENO);
    fclose(outfile);
    sprintf(pathenv, "PGPPATH=%s", PGP_DIR);
    putenv(pathenv);
    if (USE_SYSLOG)
      syslog(LOG_DEBUG, "Encrypting message");
    execl(PGP, basename(PGP), "-fact", "+batchmode", "+armorlines=0",
	  "+verbose=0", "-z", key, NULL);
  }
  else /* parent */
    wait(&status);

  flock(lockfd, LOCK_UN);
  close(lockfd);
  unlink(toencrypt);
  return status;
}

int
append_url(const char *urlfilename, const char *toappendto, const char *url)
{
  int lockfd, status;
  pid_t pid = getpid();
  char pathenv[BUFSIZ];
  FILE *infile, *outfile;
  struct stat urlfilestat;

  /* URL is blocked; write error message to outfile */
  if (blocked(url, URL_BLOCK)) {
    if (USE_SYSLOG)
      syslog(LOG_NOTICE, "Ignoring request");
    outfile = fopen(toappendto, "a");
    fprintf(outfile, "You are not allowed to download the URL\n");
    fprintf(outfile, "%s\n", url);
    fprintf(outfile, "through the %s.\n", REMAILER_NAME);
    fclose(outfile);
    return 0;
  }

  /* call Wget to retrieve the requested URL */
  if (!fork()) { /* child */
    outfile = fopen(toappendto, "a");
    dup2(fileno(outfile), STDOUT_FILENO);
    fclose(outfile);
    if (USE_SYSLOG)
      syslog(LOG_INFO, "Downloading URL %s", url);
    execl(WGET, basename(WGET), "-nv", "-r", "-l1", "-goff", "-O", urlfilename,
	  url, NULL);
  }
  else /* parent */
    wait(&status);
  if (USE_SYSLOG)
    syslog(LOG_INFO, "Download finished");

  stat(urlfilename, &urlfilestat);
  if (urlfilestat.st_size == 0L ||
      (SIZE_LIMIT && urlfilestat.st_size > SIZE_LIMIT)) {
    if (urlfilestat.st_size == 0L) {
      if (USE_SYSLOG)
	syslog(LOG_NOTICE, "Empty URL %s", url);
    }
    else if (SIZE_LIMIT && urlfilestat.st_size > SIZE_LIMIT) {
      if (USE_SYSLOG)
	syslog(LOG_NOTICE, "Oversized URL %s: %ld bytes",
	       url, urlfilestat.st_size);
      outfile = fopen(toappendto, "a");
      fprintf(outfile, "You are not allowed to download more than\n");
      fprintf(outfile, "%d bytes\n", SIZE_LIMIT);
      fprintf(outfile, "through the %s.\n", REMAILER_NAME);
      fclose(outfile);
    }
    if (USE_SYSLOG)
      syslog(LOG_DEBUG, "Deleting %s", urlfilename);
    unlink(urlfilename);
    return status;
  }

  /* call PGP to ASCII-armor and append URL file to message */
  /* PGP needs exclusive access to randseed.bin */
  lockfd = open("pgp.lock", O_WRONLY | O_CREAT | O_TRUNC, 0660);
  flock(lockfd, LOCK_EX);
  write(lockfd, &pid, sizeof(pid_t));

  if (!fork()) { /* child */
    fclose(stderr); /* shut up! */
    infile = fopen(urlfilename, "r");
    dup2(fileno(infile), STDIN_FILENO);
    outfile = fopen(toappendto, "a");
    fputs("\n", outfile);
    dup2(fileno(outfile), STDOUT_FILENO);
    fclose(outfile);
    sprintf(pathenv, "PGPPATH=%s", PGP_DIR);
    putenv(pathenv);
    execl(PGP, basename(PGP), "-fa", "+batchmode", "+armorlines=0",
	  "+verbose=0", NULL);
  }
  else /* parent */
    wait(&status);

  flock(lockfd, LOCK_UN);
  close(lockfd);
  unlink(urlfilename);
  return status;
}
