/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  codecForExchangeKeysResponse,
  DenominationPubKey,
  DenomKeyType,
  Duration,
  ExchangeKeysResponse,
  Logger,
  TalerCorebankApiClient,
} from "@gnu-taler/taler-util";
import {
  createPlatformHttpLib,
  readSuccessResponseJsonOrThrow,
} from "@gnu-taler/taler-util/http";
import { makeNoFeeCoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  ExchangeService,
  getTestHarnessPaytoForLabel,
  GlobalTestState,
  MerchantService,
  setupDb,
} from "../harness/harness.js";
import {
  applyTimeTravelV2,
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/environments.js";

const logger = new Logger("test-exchange-timetravel.ts");

interface DenomInfo {
  denomPub: DenominationPubKey;
  expireDeposit: string;
}

function getDenomInfoFromKeys(ek: ExchangeKeysResponse): DenomInfo[] {
  const denomInfos: DenomInfo[] = [];
  for (const denomGroup of ek.denominations) {
    switch (denomGroup.cipher) {
      case "RSA":
      case "RSA+age_restricted": {
        let ageMask = 0;
        if (denomGroup.cipher === "RSA+age_restricted") {
          ageMask = denomGroup.age_mask;
        }
        for (const denomIn of denomGroup.denoms) {
          const denomPub: DenominationPubKey = {
            age_mask: ageMask,
            cipher: DenomKeyType.Rsa,
            rsa_public_key: denomIn.rsa_pub,
          };
          denomInfos.push({
            denomPub,
            expireDeposit: AbsoluteTime.stringify(
              AbsoluteTime.fromProtocolTimestamp(denomIn.stamp_expire_deposit),
            ),
          });
        }
        break;
      }
      case "CS+age_restricted":
      case "CS":
        logger.warn("Clause-Schnorr denominations not supported");
        continue;
      default:
        logger.warn(
          `denomination type ${(denomGroup as any).cipher} not supported`,
        );
        continue;
    }
  }
  return denomInfos;
}

const http = createPlatformHttpLib({
  enableThrottling: false,
});

/**
 * Basic time travel test.
 */
export async function runExchangeTimetravelTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: db.connStr,
  });

  let receiverName = "Exchange";
  let exchangeBankUsername = "exchange";
  let exchangeBankPassword = "mypw-password";
  let exchangePaytoUri = getTestHarnessPaytoForLabel(exchangeBankUsername);

  await exchange.addBankAccount("1", {
    wireGatewayAuth: {
      username: exchangeBankUsername,
      password: exchangeBankPassword,
    },
    wireGatewayApiBaseUrl: new URL(
      "accounts/exchange/taler-wire-gateway/",
      bank.baseUrl,
    ).href,
    accountPaytoUri: exchangePaytoUri,
  });

  bank.setSuggestedExchange(exchange, exchangePaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "admin-password",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  exchange.addCoinConfigList(makeNoFeeCoinConfig("TESTKUDOS"));

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  const { accessToken: adminAccessToken } = await merchant.addInstanceWithWireAccount({
    id: "admin",
    name: "Default Instance",
    paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [getTestHarnessPaytoForLabel("minst1")],
  }, {adminAccessToken});

  console.log("setup done!");

  const { walletClient } = await createWalletDaemonWithClient(t, {
    name: "default",
  });

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:15",
  });
  await wres.withdrawalFinishedCond;

  const keysResp1 = await http.fetch(exchange.baseUrl + "keys");
  const keys1 = await readSuccessResponseJsonOrThrow(
    keysResp1,
    codecForExchangeKeysResponse(),
  );
  console.log(
    "keys 1 (before time travel):",
    JSON.stringify(keys1, undefined, 2),
  );

  // Travel into the future, the deposit expiration is two years
  // into the future.
  console.log("applying first time travel");
  await applyTimeTravelV2(
    Duration.toMilliseconds(Duration.fromSpec({ days: 400 })),
    {
      walletClient,
      exchange,
      merchant,
    },
  );

  const keysResp2 = await http.fetch(exchange.baseUrl + "keys");
  const keys2 = await readSuccessResponseJsonOrThrow(
    keysResp2,
    codecForExchangeKeysResponse(),
  );
  console.log(
    "keys 2 (after time travel):",
    JSON.stringify(keys2, undefined, 2),
  );

  const denomPubs1 = getDenomInfoFromKeys(keys1);
  const denomPubs2 = getDenomInfoFromKeys(keys2);

  const dps2 = new Set(denomPubs2.map((x) => x.denomPub));

  console.log("=== KEYS RESPONSE 1 ===");

  console.log(
    "list issue date",
    AbsoluteTime.stringify(
      AbsoluteTime.fromProtocolTimestamp(keys1.list_issue_date),
    ),
  );
  console.log("num denoms", denomPubs1.length);
  console.log("denoms", JSON.stringify(denomPubs1, undefined, 2));

  console.log("=== KEYS RESPONSE 2 ===");

  console.log(
    "list issue date",
    AbsoluteTime.stringify(
      AbsoluteTime.fromProtocolTimestamp(keys2.list_issue_date),
    ),
  );
  console.log("num denoms", denomPubs2.length);
  console.log("denoms", JSON.stringify(denomPubs2, undefined, 2));

  for (const da of denomPubs1) {
    let found = false;
    for (const db of denomPubs2) {
      const d1 = da.denomPub;
      const d2 = db.denomPub;
      if (DenominationPubKey.cmp(d1, d2) === 0) {
        found = true;
        break;
      }
    }
    if (!found) {
      console.log("=== ERROR ===");
      console.log(
        `denomination with public key ${da.denomPub} is not present in new /keys response`,
      );
      console.log(
        `the new /keys response was issued ${AbsoluteTime.stringify(
          AbsoluteTime.fromProtocolTimestamp(keys2.list_issue_date),
        )}`,
      );
      console.log(
        `however, the missing denomination has stamp_expire_deposit ${da.expireDeposit}`,
      );
      console.log("see above for the verbatim /keys responses");
      t.assertTrue(false);
    }
  }
}

runExchangeTimetravelTest.suites = ["exchange"];
