#include "globals.h"
#include "readbatch.h"
#include "scanargs.h"
#include "hdf.h"

char *BatchTokens[] = {
	"layoutBounds",
	"objectList",
	"styleList",
	""
};

#define LAYOUTBOUNDS	1
#define OBJECTLIST		2
#define STYLELIST		3

extern Object *LoadObject();

/************************************************************************
*																		*
*	int ReadBatchParameters(fName, layout)								*
*	Read in Layout Batch parameters:									*
*	1.	DFopen the HDF file fName which should be a Layout batch file	*
*	2.	Read in the parameter information including:					*
*		a.	Layout bounds - Not necessarily the same as the dims		*
*		b.	Object list - contains either image bounds or counter info.	*
*	3.	Close the HDF file.												*
*																		*
************************************************************************/
int ReadBatchParameters(fName, layout)
	unsigned char *fName;
	LayoutMask *layout;
{
	unsigned char *dfile;
	unsigned char *s, *stringPtr, *scanPtr;
	int token;
	Object **objectList;
	unsigned char accessMsg[150];
	int length, size, i;
	StyleRec *styleList;
	
	if(NULL == (dfile = DFopen(fName, DFACC_READ, -1))) {
		printf("Error opening file %s.\n", fName);
		return(-1);
	}
	sprintf(accessMsg, "Error: %s is not a valid Layout batch file.\n", fName);
	s = mDFgetelement(fName, dfile, LAYBAT_TAG, 1, &length, accessMsg);
	if(s == NULL) return(-1);
	stringPtr = s;
	layout->objectCount = 0;
	layout->styleCount = 0;

	while((token = getToken(&stringPtr, &scanPtr, BatchTokens)) != EOS) {
		switch(token) {
			case LAYOUTBOUNDS:
				sscanf(scanPtr, "%u, %u, %u, %u",
					&layout->bounds.top, &layout->bounds.left,
					&layout->bounds.bottom, &layout->bounds.right);
				break;
			case OBJECTLIST:
				size = scanInt(&scanPtr);
				objectList = (Object **)mMalloc(size*sizeof(Object *));
				if(objectList == NULL) {
					free(s);
					DFclose(dfile);
					return(-1);
				}
				layout->objectList = objectList;
				layout->objectCount = size;
				for(;size > 0; size--)
					*objectList++ = LoadObject(fName, dfile, scanInt(&scanPtr));
				break;
			case STYLELIST:
				size = scanInt(&scanPtr);
				styleList = (StyleRec *)mMalloc(size*sizeof(StyleRec));
				if(styleList == NULL) {
					free(s);
					DFclose(dfile);
					return(-1);
				}
				layout->styleList = styleList;
				layout->styleCount = size;
				for(;size > 0; size--, styleList++) {
					styleList->ascent = scanInt(&scanPtr);
					for(i=0; i < 10; i++)
						styleList->offset[i] = scanInt(&scanPtr);
				}
				break;
		}
	}
	free(s);
	DFclose(dfile);
	return(0);
}

/************************************************************************
*																		*
*	Image *nextImage(layout, seqNo)										*
*	Given seqNo, nextImage returns the image with the sequence number	*
*	which follows seqNo.  This is an inefficient way of sorting but		*
*	considering there are ussually only no more than 2 or 3 images		*
*	its not worth worrying about.										*
*																		*
************************************************************************/
Image *nextImage(layout, seqNo)
	LayoutMask *layout;
	int seqNo;
{
	int i;
	Image *image, *tempImage;
	
	image = NULL;
	for(i = 0; i < layout->objectCount; i++)
		if((layout->objectList[i])->kind == ImageKind) {
			tempImage = (Image *)(layout->objectList[i]);
			if(tempImage->seqNumber > seqNo)
				if(image == NULL)
					image = tempImage;
				else if (tempImage->seqNumber < image->seqNumber)
					image = tempImage;
		}
	if(image == NULL)
		printf("Error while processing image sequence numbers.");
	return(image);
}

/************************************************************************
*																		*
*	int VerifyImageBounds(layout, scanData)								*
*	Verify that the dimensions of each image match the dimensions of	*
*	the bounds in the image bounds list.								*
*	For each imageBounds and imageSet									*
*		For each image in the set										*
*			if the dimensions of the image does not match the bounds	*
*				return -1												*
*	return 0															*
*	This routine assumes layout->imageCount == scanData->imageCount		*
*																		*
************************************************************************/
int VerifyImageBounds(layout, scanData)
	LayoutMask *layout;
	ScanData *scanData;
{
	int i,k, seqNo;
	int dimx, dimy, x, y;
	Rect *bounds;
	unsigned char **imageNames;
	int isPal;
	Image *image;
	
	for(i = 0, seqNo = -1; i < scanData->imageCount; i++) {
		image = nextImage(layout, seqNo);
		if(image == NULL)
			return(-1);
		seqNo = image->seqNumber;
		bounds = &image->object.mask.bounds;
		imageNames = scanData->imageList[i];
		image->nameList = imageNames;
		dimx = bounds->right - bounds->left;
		dimy = bounds->bottom - bounds->top;
		for(k = 0; k < scanData->seqCount; k++) {
			if(DFR8getdims(imageNames[k], &x, &y, &isPal) < 0) {
				printf("Error reading image file '%s'.\n", imageNames[k]);
				return(-1);
			}
			if(x != dimx || y != dimy) {
				printf("Dimensions of '%s' don't match those in '%s'.\n",
					imageNames[k], scanData->maskName);
				printf("'%s' dims = %u, %u;  '%s' dims = %u %u\n",
						imageNames[k], x, y, scanData->maskName, dimx, dimy);
				return(-1);
			}
		}
	}
	return(0);
}


/************************************************************************
*																		*
*	int ReadInBackGround(fName, layout)									*
*	Read in the layout background image.								*
*																		*
************************************************************************/
int ReadInBackGround(fName, layout)
	unsigned char *fName;
	LayoutMask *layout;
{
	if((layout->layout = ReadImage(fName, &layout->lx, 
							&layout->ly, layout->palette)) 
			== NULL)
		return(-1);
	return(0);
}




/************************************************************************
*																		*
*	int ReadInCounterBitMaps(fName, layout)								*
*	Read in all the font bit maps and store them with the appropriate	*
*	Style record.														*
*																		*
************************************************************************/
int ReadInCounterBitMaps(fName, layout)
	unsigned char *fName;
	LayoutMask *layout;
{
	int i;
	StyleRec *style;
	unsigned char *mask;


	for(i=0, style = layout->styleList; i < layout->styleCount; i++, style++) {
		mask = ReadImage(fName, &style->rowBytes, &style->height, NULL);
		if(mask == NULL)
			return(-1);
		if((style->mask = Expand1to8(mask, style->rowBytes*8, style->height)) == NULL)
			return(-1);
		free(mask);
	}
	return(0);
}


/************************************************************************
*																		*
*	int ReadInMasks(fName, layout)										*
*	Read in the object masks, verify the rigNumber, and expand the data.*
*																		*
************************************************************************/
int ReadInMasks(fName, layout)
	unsigned char *fName;
	LayoutMask *layout;
{
	int x, y, x1, xdims, ydims;
	unsigned char *mask;
	int i, lastRef;
	Object *object;

	for(i = 0; i < layout->objectCount; i++) {
		object = layout->objectList[i];
		if(object->maskRigNo > 0) {
			if((mask = ReadImage(fName, &x, &y, NULL)) == NULL)
				return(-1);
			/* Check parameters - shouldn't be wrong */
			lastRef = DFR8lastref();
			if(object->maskRigNo != lastRef) {
				printf("Wrong rig reference number - expected:%u actual:%u\n",
						layout->objectList[i]->maskRigNo, lastRef);
				return(-1);
			}
			xdims = object->mask.bounds.right - object->mask.bounds.left;
			ydims = object->mask.bounds.bottom - object->mask.bounds.top;
			x1 = xdims >> 3;
			if((x1 << 3) < xdims) x1++;
			if((ydims != y) || (x1 != x)) {
				printf("Object and mask dimensions do not match in Layout file '%s'.\n", fName);
				return(-1);
			}
			if((object->mask.data = Expand1to8(mask, xdims, ydims)) == NULL)
				return(-1);
			free(mask);
		}
	}
	return(0);
}



/************************************************************************
*																		*
*	int ReadBatch(layout, scanData)										*
*	To load a Layout batch perform the following steps:					*
*	1.	Read in Layout Batch parameters.								*
*	2.	Verify that the number of animated images in layout match 		*
		the -s count.													*
*	3.	Verify that all the images in imageList are HDF and match the 	*
*		dimensions of the imageBounds.									*
*	4.	Read in the layout background and mask images.					*
*	5.	If all is well return (0) else return (-1)						*
*																		*
************************************************************************/
int ReadBatch(layout, scanData)
	LayoutMask *layout;
	ScanData *scanData;
{
	int imageCount;

	if(ReadBatchParameters(scanData->maskName, layout) != 0)
		return(-1);
	imageCount = CountObjects(layout->objectList, layout->objectCount, ImageKind);
	if(imageCount  != scanData->imageCount) {
		printf("The number of -s options must match the number of Layout images.\n");
		printf("# of -s options = %u;   # of Layout images = %u\n",
				scanData->imageCount, imageCount);
		return(-1);
	}
	if(VerifyImageBounds(layout, scanData) != 0)
		return(-1);
	if(ReadInBackGround(scanData->maskName, layout) != 0)
		return(-1);
	if(ReadInCounterBitMaps(scanData->maskName, layout) != 0)
		return(-1);
	if(ReadInMasks(scanData->maskName, layout) != 0)
		return(-1);
	return(0);
}

