/****************************************************************/
/*                                                              */
/*                                                              */
/*                            BrewCalc                          */
/*                                                              */
/*                A Home Brewer's Recipe Formulator             */
/*                                                              */
/*                                                              */
/****************************************************************/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*          Copyright (C) 1995   Richard J. Procassini          */
/*                                                              */
/*             (See file 'brewcalc.h' for warranty              */
/*               and redistribution information.)               */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

/*  Include Files  */
#include <curses.h>
#include <math.h>
#include <stdio.h>

#include "brewcalc.h"

/****************************************************************/


void main(void)
/****************************************************************/
/*  Routine main is the driver code for the BrewCalc: A Home    */
/*  Brewer's Recipe Formulator.                                 */
/****************************************************************/
{
    char answer;
    
    /* Read the recipe data base. */
    Read_Recipe_Data_Base();
    
    /* Read the data from the style, malt and hops data base files. */
    Read_Styles();  
    Read_Malts();
    Read_Hops();
    
    /* Initialize the weights and volumes units/conversion factors. */
    Set_Units();
    
    /* Initialize the CURSES terminal package. */
    Start_Up();
    
    while (1)
    {
	/* Draw a box around the window. */
	erase();
	box(stdscr, '*', '*');

	/* Determine what the user wants to do. */
	mvaddstr(1, 37, "BrewCalc V");
	printw("%.1f", VERSION);
	mvaddstr(3, 17, "(N)ew Recipe , (L)oad Recipe , (D)elete Recipe , (Q)uit");
	mvaddstr(4, 37, "Enter Command:");
	move(4, 52);
	get_string(&answer, 1, 1);
	
	if ((answer == 'n') || (answer == 'N'))
	{
	    /* Formulate a recipe. */
	    New_Recipe();
	}
	else if ((answer == 'l') || (answer == 'L'))
	{
	    /* Load a recipe from the recipe data base. */
	    Load_Recipe();
	}
	else if ((answer == 'd') || (answer == 'D'))
	{
	    /* Delete a recipe from the recipe data base. */
	    Delete_Recipe();
	}
	else if ((answer == 'q') || (answer == 'Q'))
	{
	    /* Write the recipe data base if necessary. */
	    if (data_base_modified)
	    {
		Write_Recipe_Data_Base();
	    }
	    
	    mvaddstr(4, 5, "                                                                           ");
	    move(4, 35);
	    standout();
	    printw("Exiting BrewCalc!");
	    refresh();
	    sleep(3);
	    standend();
	    
	    /* Close the CURSES terminal package. */
	    Shut_Down();
	    
	    /* Free dynamic memory and exit from BrewCalc. */
	    Quit();
	}
    }
    
}  /* End main */


void Read_Recipe_Data_Base()
/****************************************************************/
/*  Routine Read_Recipe_Data_Base reads the recipe data base    */
/*  binary file.                                                */
/****************************************************************/
{
    int bytes_read;
    int i;
    int j;
    FILE *file_pointer;
    
    /* Initialize some variables. */
    num_recipes        = 0;
    max_num_recipes    = 0;
    
    /* Open the recipe data base binary file. */
    file_pointer = fopen("recipes.rdb", "r");
    if (file_pointer == NULL)
    {
	fprintf(stderr, "BrewCalc: Read_Recipe_Data_Base: Open of recipes.rdb failed!\n");
	exit(1);
    }
    
    /* Read the number of recipes in the recipe data base. */
    bytes_read = 0;
    bytes_read = fread(&num_recipes, sizeof(char), sizeof(int),
		       file_pointer);
    if (bytes_read != sizeof(int))
    {
	fprintf(stderr, "BrewCalc: Read_Recipe_Data_Base: Error reading the variable\n");
	fprintf(stderr, "\t num_recipes from recipes.rdb!\n");
	exit(1);
    }
    
    /* Allocate space for the recipe data base. */
    max_num_recipes  = num_recipes + 5;
    recipe_data_base = (beer *)malloc(max_num_recipes*sizeof(beer));
    ordered_recipes  = (order *)malloc(max_num_recipes*sizeof(order));
    
    /* Read the existing entries in the recipe data base. */
    bytes_read = 0;
    bytes_read = fread(recipe_data_base, sizeof(char), num_recipes*sizeof(beer),
		       file_pointer);
    if (bytes_read != num_recipes*sizeof(beer))
    {
	fprintf(stderr, "BrewCalc: Read_Recipe_Data_Base: Error reading num_recipes\n");
	fprintf(stderr, "\t recipes from recipes.rdb!\n");
	exit(1);
    }
    
    /* Close the recipe data base binary file. */
    fclose(file_pointer);
    
    /* Initialize the 5 additional entries at the end of the recipe data base. */
    Clear_Extra_Recipes();
    
    /* Convert underscores to blanks in recipe names. */
    for (j = 0; j < num_recipes; j++)
    {
	for (i = 0; i < NAMESIZE; i++)
	{
	    if (recipe_data_base[j].name[i] == '_')
	    {
		recipe_data_base[j].name[i] = ' ';
	    }
	}
    }
    
    /* Sort the names of the recipes in the data base. */
    for (i = 0; i < num_recipes; i++)
    {
	for (j = 0; j < NAMESIZE; j++)
	{
	    ordered_recipes[i].name[j] = recipe_data_base[i].name[j];
	}
	ordered_recipes[i].data_base_location = i;
    }
    Sort_Recipe_Names();
    
    /* Indicate that the recipe data base is in its "virgin" state. */
    data_base_modified = 0;
    
}  /* End Read_Recipe_Data_Base */


void Read_Styles()
/****************************************************************/
/*  Routine Read_Styles reads the AHA style data base file.     */
/****************************************************************/
{
    int i;
    int j;
    FILE *file_pointer;
    
    /* Open the AHA style data base file. */
    file_pointer = fopen("aha.dat", "r");
    if (file_pointer == NULL)
    {
	fprintf(stderr, "BrewCalc: Read_Styles: Open of aha.dat failed!\n");
	exit(1);
    }
    
    /* Allocate space for the AHA style data base. */
    styles = (style *)malloc(MAX_NUM_DB_STYLES*sizeof(style));
    
    /* Initialize the AHA style data base. */
    num_styles = 0;
    for (j = 0; j <= (MAX_NUM_DB_STYLES - 1); j++)
    {
	for (i = 0; i < NAMESIZE; i++)
	{
	    styles[j].name[i] = 0;
	}
	styles[j].original_gravity_min = 0;
	styles[j].original_gravity_max = 0;
	styles[j].alcohol_min = 0;
	styles[j].alcohol_max = 0;
	styles[j].IBU_min = 0;
	styles[j].IBU_max = 0;
	styles[j].color_min = 0;
	styles[j].color_max = 0;
    }
    
    /* Read the AHA style data base. */
    while (fscanf(file_pointer, "%s %f %f %f %f %d %d %f %f",
		  &(styles[num_styles].name),
		  &(styles[num_styles].original_gravity_min),
		  &(styles[num_styles].original_gravity_max),
		  &(styles[num_styles].alcohol_min),
		  &(styles[num_styles].alcohol_max),
		  &(styles[num_styles].IBU_min),
		  &(styles[num_styles].IBU_max),
		  &(styles[num_styles].color_min),
		  &(styles[num_styles].color_max)) != EOF)
    {
	num_styles++;
	if (num_styles > MAX_NUM_DB_STYLES)
	{
	    fprintf(stderr, "BrewCalc: Read_Styles: num_styles is greater than MAX_NUM_DB_STYLES!\n");
	    exit(1);
	}
    };
    
    /* Close the AHA style data base file. */
    fclose(file_pointer);
    
    /* Convert underscores to blanks in style names. */
    for (j = 0; j <= (num_styles - 1); j++)
    {
	for (i = 0; i < NAMESIZE; i++)
	{
	    if (styles[j].name[i] == '_')
	    {
		styles[j].name[i] = ' ';
	    }
	}
    }
    
}  /* End Read_Styles */


void Read_Malts()
/****************************************************************/
/*  Routine Read_Malts reads the malts (grains and adjuncts)    */
/*  data base file.                                             */
/****************************************************************/
{
    int i;
    int j;
    FILE *file_pointer;
    
    /* Open the malt data base file. */
    file_pointer = fopen("malt.dat", "r");
    if (file_pointer == NULL)
    {
	fprintf(stderr, "BrewCalc: Read_Malts: Open of malt.dat failed!\n");
	exit(1);
    }
    
    /* Allocate space for the malt data base. */
    malts = (malt *)malloc(MAX_NUM_DB_MALTS*sizeof(malt));
    
    /* Initialize the malt data base. */
    num_malts = 0;
    for (j = 0; j <= (MAX_NUM_DB_MALTS - 1); j++)
    {
	for (i = 0; i < NAMESIZE; i++)
	{
	    malts[j].name[i] = 0;
	}
	malts[j].gravity = 0;
	malts[j].color = 0;
	malts[j].converted = 0;
    }
    
    /* Read the malt data base. */
    while (fscanf(file_pointer,"%s %f %f %d",
		  &(malts[num_malts].name),
		  &(malts[num_malts].gravity),
		  &(malts[num_malts].color),
		  &(malts[num_malts].converted)) != EOF)
    {
	num_malts++;
	if (num_malts > MAX_NUM_DB_MALTS)
	{
	    fprintf(stderr, "BrewCalc: Read_Malts: num_malts is greater than MAX_NUM_DB_MALTS!\n");
	    exit(1);
	}
    };
    
    /* Close the malt data base file. */
    fclose(file_pointer);
    
    /* Convert underscores to blanks in malt names. */
    for (j = 0; j <= (num_malts - 1); j++)
    {
	for (i = 0; i < NAMESIZE; i++)
	{
	    if (malts[j].name[i] == '_')
	    {
		malts[j].name[i] = ' ';
	    }
	}
    }
    
}  /* End Read_Malts */


void Read_Hops()
/****************************************************************/
/*  Routine Read_Hops reads the hops data base file.            */
/****************************************************************/
{
    int i;
    int j;
    FILE *file_pointer;
    
    /* Open the hop data base file. */
    file_pointer = fopen("hop.dat", "r");
    if (file_pointer == NULL)
    {
	fprintf(stderr, "BrewCalc: Read_Hops: Open of hop.dat failed!\n");
	exit(1);
    }
    
    /* Allocate space for the hop data base. */
    hops = (hop *)malloc(MAX_NUM_DB_HOPS*sizeof(hop));
    
    /* Initialize the hop data base. */
    num_hops = 0;
    for (j = 0; j <= (MAX_NUM_DB_HOPS - 1); j++)
    {
	for (i = 0; i < NAMESIZE; i++)
	{
	    hops[j].name[i] = 0;
	}
	hops[j].alpha_acid_percentage = 0;
    }
    
    /* Read the hop data base. */
    while (fscanf(file_pointer, "%s %f",
		  &(hops[num_hops].name),
		  &(hops[num_hops].alpha_acid_percentage)) != EOF)
    {
	num_hops++;
	if (num_hops > MAX_NUM_DB_HOPS)
	{
	    fprintf(stderr, "BrewCalc: Read_Hops: num_hops is greater than MAX_NUM_DB_HOPS!\n");
	    exit(1);
	}
    };
    
    /* Close the hop data base file. */
    fclose(file_pointer);
    
    /* Convert underscores to blanks in hop names. */
    for (j = 0; j <= (num_hops - 1); j++)
    {
	for (i = 0; i < NAMESIZE; i++)
	{
	    if (hops[j].name[i] == '_')
	    {
		hops[j].name[i] = ' ';
	    }
	}
    }
    
}  /* End Read_Hops */


void Set_Units()
/****************************************************************/
/*  Routine Set_Units initializes the weights and volumes       */
/*  units/conversion factors.                                   */
/****************************************************************/
{
    int i;
    int j;
    
    /* Allocate space for the units/conversion factors structures. */
    volume_units      = (unit *)malloc(2*sizeof(unit));
    malt_weight_units = (unit *)malloc(4*sizeof(unit));
    hop_weight_units  = (unit *)malloc(2*sizeof(unit));
    
    /* Initialize the unit names. */
    for (j = 0; j < 4; j++)
    {
	if (j <= 1)
	{
	    for (i = 0; i < 8; i++)
	    {
		volume_units[j].name[i]     = 0;
		hop_weight_units[j].name[i] = 0;
	    }
	}
	for (i = 0; i < 8; i++)
	{
	    malt_weight_units[j].name[i] = 0;
	}
    }
    
    /* Volume units/conversion factors. */
    sscanf("gallons", "%s", volume_units[0].name);
    sscanf("liters", "%s", volume_units[1].name);
    volume_units[0].conversion_factor = 1.0;
    volume_units[1].conversion_factor = 2.645503e-1;
    
    /* Malt weight units/conversion factors. */
    sscanf("lb", "%s", malt_weight_units[0].name);
    sscanf("oz", "%s", malt_weight_units[1].name);
    sscanf("kg", "%s", malt_weight_units[2].name);
    sscanf("g", "%s", malt_weight_units[3].name);
    malt_weight_units[0].conversion_factor = 1.0;
    malt_weight_units[1].conversion_factor = 6.25e-2;
    malt_weight_units[2].conversion_factor = 2.2;
    malt_weight_units[3].conversion_factor = 2.20e-3;
    
    /* Hop weight units/conversion factors. */
    sscanf("oz", "%s", hop_weight_units[0].name);
    sscanf("g", "%s", hop_weight_units[1].name);
    hop_weight_units[0].conversion_factor = 1.0;
    hop_weight_units[1].conversion_factor = 3.524229e-2;
    
}  /* End Set_Units */


void Start_Up()
/****************************************************************/
/*  Routine Start_Up initializes the CURSES terminal package.    */
/****************************************************************/
{
    
    erase_char = get_erase();
    initscr();
    raw();
    noecho();
    
}  /* End Start_Up */


void New_Recipe()
/****************************************************************/
/*  Routine New_Recipe allows the user to formulate a new       */
/*  recipe.                                                     */
/****************************************************************/
{
    int i;
    char answer;
    
    /* Initialize some data. */
    data_base_location = -1;
    for (i = 0; i < NAMESIZE; i++)
    {
	current_brew.name[i] = ' ';
    }
    
    /* Edit the recipe. */
    Edit_Recipe(1);
    
    while (1)
    {
	mvaddstr(23+MAX_NUM_MALTS+MAX_NUM_HOPS, 14,
		 "(E)dit Recipe , (S)ave Recipe , (P)rint Recipe , (M)ain Menu");
	mvaddstr(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 37,
		 "Enter Command:");
	move(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 52);
	get_string(&answer, 1, 1);
	
	if ((answer == 'e') || (answer == 'E'))
	{
	    /* Edit the recipe. */
	    Edit_Recipe(0);
	}
	else if ((answer == 's') || (answer == 'S'))
	{
	    /* Save a recipe to the recipe data base. */
            Save_Recipe();
	}
	else if ((answer == 'p') || (answer == 'P'))
	{
	    /* Print a summary of the recipe to an ASCII file. */
	    Print_Recipe_File();
	}
	else if ((answer == 'm') || (answer == 'M'))
	{
	    /* Return to the main menu. */
	    goto Done;
	}
    }
    
 Done:
    
}  /* End New_Recipe */


void Load_Recipe()
/****************************************************************/
/*  Routine Load_Recipe allows the user to select a recipe      */
/*  from the recipe data base and to make it the current        */
/*  recipe.                                                     */
/****************************************************************/
{
    int i;
    int j;
    int location;
    char answer;
    
    /* Select a recipe from the recipe data base. */
    mvaddstr(4, 5, "                                                                           ");
    mvaddstr(4, 28, "Load Recipe:");
    refresh();
    move(4, 41);
    location = pick_one_recipe(ordered_recipes, num_recipes);
    refresh();
    data_base_location = ordered_recipes[location].data_base_location;
    j                  = data_base_location;
    
    /* If the selected recipe is not the "(NONE)" recipe, then load it. */
    if (data_base_location != 0)
    {
	/* Copy the selected recipe into the current recipe structure. */
	for (i = 0; i < NAMESIZE; i++)
	{
	    current_brew.name[i] = recipe_data_base[j].name[i];
	}
	for (i = 0; i < MAX_NUM_MALTS; i++)
	{
	    current_brew.malts[i]          = recipe_data_base[j].malts[i];
	    current_brew.malt_amounts[i]   = recipe_data_base[j].malt_amounts[i];
	    current_brew.malt_colors[i]    = recipe_data_base[j].malt_colors[i];
	    current_brew.malt_gravities[i] = recipe_data_base[j].malt_gravities[i];
	    current_brew.malt_units[i]     = recipe_data_base[j].malt_units[i];
	}
	for (i = 0; i < MAX_NUM_HOPS; i++)
	{
	    current_brew.hops[i]        = recipe_data_base[j].hops[i];
	    current_brew.hop_amounts[i] = recipe_data_base[j].hop_amounts[i];
	    current_brew.hop_times[i]   = recipe_data_base[j].hop_times[i];
	    current_brew.hop_IBUs[i]    = recipe_data_base[j].hop_IBUs[i];
	    current_brew.hop_units[i]   = recipe_data_base[j].hop_units[i];
	}
	current_brew.style                 = recipe_data_base[j].style;
	current_brew.total_volume          = recipe_data_base[j].total_volume;
	current_brew.total_volume_units    = recipe_data_base[j].total_volume_units;
	current_brew.boil_volume           = recipe_data_base[j].boil_volume;
	current_brew.boil_volume_units     = recipe_data_base[j].boil_volume_units;
	current_brew.num_malts             = recipe_data_base[j].num_malts;
	current_brew.num_hops              = recipe_data_base[j].num_hops;
	current_brew.extraction_efficiency = recipe_data_base[j].extraction_efficiency;
	current_brew.apparent_attenuation  = recipe_data_base[j].apparent_attenuation;
	current_brew.real_attenuation      = recipe_data_base[j].real_attenuation;
	current_brew.original_gravity      = recipe_data_base[j].original_gravity;
	current_brew.final_gravity         = recipe_data_base[j].final_gravity;
	current_brew.alcohol_by_volume     = recipe_data_base[j].alcohol_by_volume;
	current_brew.color                 = recipe_data_base[j].color;
	current_brew.IBUs                  = recipe_data_base[j].IBUs;
	
	/* Draw a box around the window. */
	erase();
	box(stdscr, '*', '*');
	
	/* Write the general description data for the current brew. */
	mvaddstr(1, 37, "BrewCalc V");
	printw("%.1f", VERSION);
	mvaddstr(3, 34, "Name: ");
	move(3, 41);
	printw("%-25s", current_brew.name);
	mvaddstr(4, 33, "Style: ");
	move(4, 41);
	printw("%-25s", styles[current_brew.style].name);
	mvaddstr(5, 26, "Total Volume: ");
	move(5, 41);
	printw("%4.1f", current_brew.total_volume);
	move(5, 46);
	printw("%-8s", volume_units[current_brew.total_volume_units].name);
	mvaddstr(6, 27, "Boil Volume: ");
	move(6, 41);
	printw("%4.1f", current_brew.boil_volume);
	move(6, 46);
	printw("%-8s", volume_units[current_brew.boil_volume_units].name);
	mvaddstr(7, 27, "Attenuation: ");
	move(7, 41);
	printw("%4.1f", current_brew.apparent_attenuation);
	mvaddstr(7, 46, "percent");
	mvaddstr(8, 28, "Efficiency: ");
	move(8, 41);
	printw("%4.1f", current_brew.extraction_efficiency);
	mvaddstr(8, 46, "percent");
	refresh();
	
	/* Calculate the real attenuation of the yeast. */
	Real_Attenuation();
	
	/* Write the malt information for the current brew. */
	mvaddstr(10, 5, "Malt/Adjunct Name");
	mvaddstr(11, 5, "-----------------");
	mvaddstr(10, 37, "Qty.");
	mvaddstr(11, 37, "----");
	mvaddstr(10, 44, "Unit");
	mvaddstr(11, 44, "----");
	mvaddstr(10, 52, "Color");
	mvaddstr(11, 52, "-----");
	mvaddstr(10, 60, "Gravity");
	mvaddstr(11, 60, "-------");
	for (i = 0; i < current_brew.num_malts; i++)
	{
	    move(12+i, 5);
	    printw("%-25s", current_brew.malts[i].name);
	    move(12+i, 35);
	    printw("%6.2f", current_brew.malt_amounts[i]);
	    move(12+i, 46);
	    printw("%2s", malt_weight_units[current_brew.malt_units[i]].name);
	    move(12+i, 51);
	    printw("%6.2f", current_brew.malt_colors[i]);
	    move(12+i, 61);
	    printw("%5.3f", current_brew.malt_gravities[i]);
	}
	refresh();
	
	/* Write the hop information for the current brew. */
	mvaddstr(13+MAX_NUM_MALTS, 5, "Hop Name");
	mvaddstr(14+MAX_NUM_MALTS, 5, "--------");
	mvaddstr(13+MAX_NUM_MALTS, 38, "AA%");
	mvaddstr(14+MAX_NUM_MALTS, 38, "---");
	mvaddstr(13+MAX_NUM_MALTS, 45, "Qty.");
	mvaddstr(14+MAX_NUM_MALTS, 45, "----");
	mvaddstr(13+MAX_NUM_MALTS, 52, "Unit");
	mvaddstr(14+MAX_NUM_MALTS, 52, "----");
	mvaddstr(13+MAX_NUM_MALTS, 59, "Time");
	mvaddstr(14+MAX_NUM_MALTS, 59, "----");
	mvaddstr(13+MAX_NUM_MALTS, 67, "IBUs");
	mvaddstr(14+MAX_NUM_MALTS, 67, "----");
	for (i = 0; i < current_brew.num_hops; i++)
	{
	    move(15+MAX_NUM_MALTS+i, 5);
	    printw("%-25s", current_brew.hops[i].name);
	    move(15+MAX_NUM_MALTS+i, 36);
	    printw("%5.2f", current_brew.hops[i].alpha_acid_percentage);
	    move(15+MAX_NUM_MALTS+i, 44);
	    printw("%5.2f", current_brew.hop_amounts[i]);
	    move(15+MAX_NUM_MALTS+i, 54);
	    printw("%2s", hop_weight_units[current_brew.hop_units[i]].name);
	    move(15+MAX_NUM_MALTS+i, 60);
	    printw("%2d", current_brew.hop_times[i]);
	    move(15+MAX_NUM_MALTS+i, 66);
	    printw("%5.2f", current_brew.hop_IBUs[i]);
	}
	refresh();
	
	/* Compare the results to the style limits. */
	Print_Guidelines();
	
	while (1)
	{
	    mvaddstr(23+MAX_NUM_MALTS+MAX_NUM_HOPS, 14,
		     "(E)dit Recipe , (S)ave Recipe , (P)rint Recipe , (M)ain Menu");
	    mvaddstr(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 37,
		     "Enter Command:");
	    move(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 52);
	    get_string(&answer, 1, 1);
	    
	    if ((answer == 'e') || (answer == 'E'))
	    {
		/* Edit the recipe. */
		Edit_Recipe(0);
	    }
	    else if ((answer == 's') || (answer == 'S'))
	    {
		/* Save a recipe to the recipe data base. */
		Save_Recipe();
	    }
	    else if ((answer == 'p') || (answer == 'P'))
	    {
		/* Print a summary of the recipe to an ASCII file. */
		Print_Recipe_File();
	    }
	    else if ((answer == 'm') || (answer == 'M'))
	    {
		/* Return to the main menu. */
		goto Done;
	    }
	}
	
    Done:
    }
    
}  /* End Load_Recipe */


void Delete_Recipe()
/****************************************************************/
/*  Routine Delete_Recipe allows the user to delete a recipe    */
/*  from the recipe data base.                                  */
/****************************************************************/
{
    int i;
    int j;
    int location;
    int name_length;
    char answer;
    
    /* Select a recipe to delete from the recipe data base. */
    mvaddstr(4, 5, "                                                                           ");
    mvaddstr(4, 27, "Delete Recipe:");
    refresh();
    move(4, 42);
    location = pick_one_recipe(ordered_recipes, num_recipes);
    refresh();
    mvaddstr(4, 5, "                                                                           ");

    /* If the selected recipe is not the "(NONE)" recipe, then delete it. */
    if (location != 0)
    {
	name_length = strlen(ordered_recipes[location].name);
	move(4, 30-(name_length/2));
	printw("Confirm: Delete Recipe '%-s'?", ordered_recipes[location].name);
	refresh();
	move(4, 58+(name_length/2));
	get_string(&answer, 1, 1);
	
	if ((answer == 'y') || (answer == 'Y'))
	{
	    location = ordered_recipes[location].data_base_location;
	    
	    for (j = (location + 1); j < max_num_recipes; j++)
	    {
		/* Copy the selected recipe into the current recipe structure. */
		for (i = 0; i < NAMESIZE; i++)
		{
		    recipe_data_base[j-1].name[i] = recipe_data_base[j].name[i];
		}
		for (i = 0; i < MAX_NUM_MALTS; i++)
		{
		    recipe_data_base[j-1].malts[i]          = recipe_data_base[j].malts[i];
		    recipe_data_base[j-1].malt_amounts[i]   = recipe_data_base[j].malt_amounts[i];
		    recipe_data_base[j-1].malt_colors[i]    = recipe_data_base[j].malt_colors[i];
		    recipe_data_base[j-1].malt_gravities[i] = recipe_data_base[j].malt_gravities[i];
		    recipe_data_base[j-1].malt_units[i]     = recipe_data_base[j].malt_units[i];
		}
		for (i = 0; i < MAX_NUM_HOPS; i++)
		{
		    recipe_data_base[j-1].hops[i]        = recipe_data_base[j].hops[i];
		    recipe_data_base[j-1].hop_amounts[i] = recipe_data_base[j].hop_amounts[i];
		    recipe_data_base[j-1].hop_times[i]   = recipe_data_base[j].hop_times[i];
		    recipe_data_base[j-1].hop_IBUs[i]    = recipe_data_base[j].hop_IBUs[i];
		    recipe_data_base[j-1].hop_units[i]   = recipe_data_base[j].hop_units[i];
		}
		recipe_data_base[j-1].style                 = recipe_data_base[j].style;
		recipe_data_base[j-1].total_volume          = recipe_data_base[j].total_volume;
		recipe_data_base[j-1].total_volume_units    = recipe_data_base[j].total_volume_units;
		recipe_data_base[j-1].boil_volume           = recipe_data_base[j].boil_volume;
		recipe_data_base[j-1].boil_volume_units     = recipe_data_base[j].boil_volume_units;
		recipe_data_base[j-1].num_malts             = recipe_data_base[j].num_malts;
		recipe_data_base[j-1].num_hops              = recipe_data_base[j].num_hops;
		recipe_data_base[j-1].extraction_efficiency = recipe_data_base[j].extraction_efficiency;
		recipe_data_base[j-1].apparent_attenuation  = recipe_data_base[j].apparent_attenuation;
		recipe_data_base[j-1].real_attenuation      = recipe_data_base[j].real_attenuation;
		recipe_data_base[j-1].original_gravity      = recipe_data_base[j].original_gravity;
		recipe_data_base[j-1].final_gravity         = recipe_data_base[j].final_gravity;
		recipe_data_base[j-1].alcohol_by_volume     = recipe_data_base[j].alcohol_by_volume;
		recipe_data_base[j-1].color                 = recipe_data_base[j].color;
		recipe_data_base[j-1].IBUs                  = recipe_data_base[j].IBUs;
	    }
	    /* Adjust the number of recipes in the recipe data base. */
	    num_recipes--;
	    
	    /* Indicate that the recipe data base has been modified. */
	    data_base_modified = 1;
	    
	    mvaddstr(4, 5, "                                                                           ");
	    move(4, 37);
	    standout();
	    printw("Recipe Deleted!");
	    refresh();
	    sleep(3);
	    standend();
	    
	    /* Sort the names of the recipes in the data base. */
	    for (i = 0; i < num_recipes; i++)
	    {
		for (j = 0; j < NAMESIZE; j++)
		{
		    ordered_recipes[i].name[j] = recipe_data_base[i].name[j];
		}
		ordered_recipes[i].data_base_location = i;
	    }
	    Sort_Recipe_Names();
	}
    }
    
}  /* End Delete_Recipe */


void Write_Recipe_Data_Base()
/****************************************************************/
/*  Routine Write_Recipe_Data_Base writes the recipe data       */
/*  base binary file.                                           */
/****************************************************************/
{
    int bytes_written;
    int i;
    int j;
    FILE *file_pointer;
    
    /* Convert blanks to underscores in recipe names. */
    for (j = 0; j < num_recipes; j++)
    {
	for (i = 0; i < NAMESIZE; i++)
	{
	    if (recipe_data_base[j].name[i] == ' ')
	    {
		recipe_data_base[j].name[i] = '_';
	    }
	}
    }
    
    /* Open the recipe data base binary file. */
    file_pointer = fopen("recipes.rdb", "w");
    if (file_pointer == NULL)
    {
	fprintf(stderr, "BrewCalc: Write_Recipe_Data_Base: Open of recipes.rdb failed!\n");
	exit(1);
    }
    
    /* Write the number of recipes to the recipe data base. */
    bytes_written = 0;
    bytes_written = fwrite(&num_recipes, sizeof(char), sizeof(int),
			   file_pointer);
    if (bytes_written != sizeof(int))
    {
	fprintf(stderr, "BrewCalc: Write_Recipe_Data_Base: Error writing the variable\n");
	fprintf(stderr, "\t num_recipes to recipes.rdb!\n");
	exit(1);
    }
    
    /* Write the recipe data base to the binary file. */
    bytes_written = 0;
    bytes_written = fwrite(recipe_data_base, sizeof(char), num_recipes*sizeof(beer),
			   file_pointer);
    if (bytes_written != num_recipes*sizeof(beer))
    {
	fprintf(stderr, "BrewCalc: Write_Recipe_Data_Base: Error writing num_recipes\n");
	fprintf(stderr, "\t recipes to recipes.rdb!\n");
	exit(1);
    }
    
    /* Close the recipe data base binary file. */
    fclose(file_pointer);
    
}  /* End Write_Recipe_Data_Base */


void Shut_Down()
/****************************************************************/
/*  Routine Shut_Down closes the CURSES terminal package.       */
/****************************************************************/
{
    
    erase();
    refresh();
    mvcur(0, COLS - 1, LINES - 1, 0);
    nl();
    endwin();

}  /* End Shut_Down */


void Quit()
/****************************************************************/
/*  Routine Quit frees dynamically-allocated data structures    */
/*  and exits from BrewCalc.                                    */
/****************************************************************/
{
    
    /* Free dynamically-allocated data structures. */
    free(styles);
    free(malts);
    free(hops);
    free(volume_units);
    free(malt_weight_units);
    free(hop_weight_units);
    free(recipe_data_base);
    free(ordered_recipes);
    
    /* Exit from BrewCalc. */
    exit(0);
    
}  /* End Quit */


void Clear_Extra_Recipes()
/****************************************************************/
/*  Routine Clear_Extra_Recipes initializes the additional 5    */
/*  entries at the end of the recipe data base.                 */
/****************************************************************/
{
    int i;
    int j;
    int k;
    
    /* Initialize the 5 additional entries */
    /* at the end of the recipe data base. */
    for (j = num_recipes; j < max_num_recipes; j++)
    {
	for (i = 0; i < NAMESIZE; i++)
	{
	    recipe_data_base[j].name[i] = 0;
	    ordered_recipes[j].name[i] = 0;
	}
	for (i = 0; i < MAX_NUM_MALTS; i++)
	{
	    for (k = 0; k < NAMESIZE; k++)
	    {
		recipe_data_base[j].malts[i].name[k] = 0;
	    }
	    recipe_data_base[j].malts[i].gravity = 0;
	    recipe_data_base[j].malts[i].color = 0;
	    recipe_data_base[j].malts[i].converted = 0;
	    recipe_data_base[j].malt_amounts[i] = 0;
	    recipe_data_base[j].malt_colors[i] = 0;
	    recipe_data_base[j].malt_gravities[i] = 0;
	    recipe_data_base[j].malt_units[i] = 0;
	}
	for (i = 0; i < MAX_NUM_HOPS; i++)
	{
	    for (k = 0; k < NAMESIZE; k++)
	    {
		recipe_data_base[j].hops[i].name[k] = 0;
	    }
	    recipe_data_base[j].hops[i].alpha_acid_percentage = 0;
	    recipe_data_base[j].hop_amounts[i] = 0;
	    recipe_data_base[j].hop_times[i] = 0;
	    recipe_data_base[j].hop_IBUs[i] = 0;
	    recipe_data_base[j].hop_units[i] = 0;
	}
	recipe_data_base[j].style = 0;
	recipe_data_base[j].total_volume = 0;
	recipe_data_base[j].total_volume_units = 0;
	recipe_data_base[j].boil_volume = 0;
	recipe_data_base[j].boil_volume_units = 0;
	recipe_data_base[j].num_malts = 0;
	recipe_data_base[j].num_hops = 0;
	recipe_data_base[j].extraction_efficiency = 0;
	recipe_data_base[j].apparent_attenuation = 0;
	recipe_data_base[j].real_attenuation = 0;
	recipe_data_base[j].original_gravity = 0;
	recipe_data_base[j].final_gravity = 0;
	recipe_data_base[j].alcohol_by_volume = 0;
	recipe_data_base[j].color = 0;
	recipe_data_base[j].IBUs = 0;
    }
    
}  /* End Clear_Extra_Recipes */


void Sort_Recipe_Names()
/****************************************************************/
/*  Routine Sort_Recipe_Names lexically orders the names of     */
/*  the recipes in the recipe data base.                        */
/****************************************************************/
{
    int i;
    int int_temp;
    int j;
    int k;
    char char_temp[NAMESIZE];
    
    for (i = 1; i < num_recipes; i++)
    {
	for (j = (i + 1); j < num_recipes; j++)
	{
	    if (strcasecmp(ordered_recipes[i].name,
			   ordered_recipes[j].name) > 0)
	    {
		for (k = 0; k < NAMESIZE; k++)
		{
		    char_temp[k]               = ordered_recipes[i].name[k];
		    ordered_recipes[i].name[k] = ordered_recipes[j].name[k];
		    ordered_recipes[j].name[k] = char_temp[k];
		}
		int_temp                              = ordered_recipes[i].data_base_location;
		ordered_recipes[i].data_base_location = ordered_recipes[j].data_base_location;
		ordered_recipes[j].data_base_location = int_temp;
	    }
	}
    }
    
}  /* End Sort_Recipe_Names */


void Edit_Recipe(initial_edit)
/****************************************************************/
/*  Routine Edit_Recipe allows the user to edit a new or        */
/*  existing recipe.                                            */
/****************************************************************/
int initial_edit;
{
    
    /* If this is an initial editing session, then clear the */
    /* window.  Otherwise, clear the command options lines. */
    if (initial_edit)
    {
	erase();
    }
    else
    {
	mvaddstr(23+MAX_NUM_MALTS+MAX_NUM_HOPS, 5,
		 "                                                                           ");
	mvaddstr(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 5,
		 "                                                                           ");
    }
    
    /* Draw a box around the window. */
    box(stdscr, '*', '*');
    
    /* Input the general description data for the current brew. */
    mvaddstr(1, 37, "BrewCalc V");
    printw("%.1f", VERSION);
    mvaddstr(3, 34, "Name: ");
    mvaddstr(4, 33, "Style: ");
    mvaddstr(5, 26, "Total Volume: ");
    mvaddstr(6, 27, "Boil Volume: ");
    mvaddstr(7, 27, "Attenuation: ");
    mvaddstr(7, 46, "percent");
    mvaddstr(8, 28, "Efficiency: ");
    mvaddstr(8, 46, "percent");
    refresh();
    
    /* Input the name of the brew. */
    move(3, 41);
    refresh();
    get_string(current_brew.name, NAMESIZE, 1);
    
    /* Input the style of the brew.*/
    move(4, 41);
    refresh();
    current_brew.style = pick_one_style(styles, num_styles);
    
    /* Input the total volume. */
    if (initial_edit)
    {
	move(5, 46);
	printw("gallons");
	refresh();
	move(5, 41);
	refresh();
	current_brew.total_volume = get_float(5.0, 4, 1);
    }
    else
    {
	move(5, 41);
	refresh();
	current_brew.total_volume = get_float(current_brew.total_volume, 4, 1);
    }
    move(5, 46);
    refresh();
    current_brew.total_volume_units = pick_one_unit(volume_units, 2);
    
    /* Input the boil volume. */
    if (initial_edit)
    {
	move(6, 46);
	printw("gallons");
	refresh();
	move(6, 41);
	refresh();
	current_brew.boil_volume = get_float(2.5, 4, 1);
    }
    else
    {
	move(6, 41);
	refresh();
	current_brew.boil_volume = get_float(current_brew.boil_volume, 4, 1);
    }
    move(6, 46);
    refresh();
    current_brew.boil_volume_units = pick_one_unit(volume_units, 2);
    
    /* Input the apparent attenuation of the yeast. */
    if (initial_edit)
    {
	move(7, 41);
	refresh();
	current_brew.apparent_attenuation = get_float(75.0, 4, 1);
    }
    else
    {
	move(7, 41);
	refresh();
	current_brew.apparent_attenuation = get_float(current_brew.apparent_attenuation, 4, 1);
    }
    
    /* Calculate the real attenuation of the yeast. */
    Real_Attenuation();
    
    /* Input the extraction efficiency of the mashing process. */
    if (initial_edit)
    {
	move(8, 41);
	refresh();
	current_brew.extraction_efficiency = get_float(75.0, 4, 1);
    }
    else
    {
	move(8, 41);
	refresh();
	current_brew.extraction_efficiency = get_float(current_brew.extraction_efficiency, 4, 1);
    }
    
    /* Select the grains and adjuncts used in the current brew. */
    Select_Malts(initial_edit);
    
    /* Calculate the original and final gravities of the current brew. */
    Total_Gravity();
    
    /* Calculate the alcohol by volume of the current brew. */
    Total_Alcohol();
    
    /* Calculate the color of the current brew. */
    Total_Color();
    
    /* Select the hops used in the current brew. */
    Select_Hops(initial_edit);
    
    /* Calculate the bitterness of the current brew. */
    Total_IBUs();
    
    /* Compare the results to the style limits. */
    Print_Guidelines();
    
}  /* End Edit_Recipe */


void Save_Recipe()
/****************************************************************/
/*  Routine Save_Recipe allows the user to save a new or        */
/*  existing recipe in the recipe data base.                    */
/****************************************************************/
{
    int i;
    int j;
    
    if (data_base_location == -1)
    {
	/* A new recipe is to be saved to the recipe data base. */
	if (num_recipes == max_num_recipes)
	{
	    /* Add 5 more blank entries to the recipe data base. */
	    max_num_recipes =+ 5;
	    
	    /* Increase the size of the recipe data base. */
	    recipe_data_base = (beer *)realloc(recipe_data_base,
					       max_num_recipes*sizeof(beer));
	    
	    /* Initialize the 5 additional entries */
	    /* at the end of the recipe data base. */
	    Clear_Extra_Recipes();
	}
	/* Adjust the number of recipes in the recipe data base. */
	num_recipes++;
	
	/* Set the location of the new recipe in the recipe data base. */
	data_base_location = num_recipes;
	j                  = data_base_location - 1;
    }
    else
    {
	/* An existing recipe is to be saved to the recipe data base. */
	j          = data_base_location;
    }
    
    /* Indicate that the recipe data base has been modified. */
    data_base_modified = 1;
    
    /* Copy the current recipe into the data base recipe structure. */
    for (i = 0; i < NAMESIZE; i++)
    {
	recipe_data_base[j].name[i] = current_brew.name[i];
    }
    for (i = 0; i < MAX_NUM_MALTS; i++)
    {
	recipe_data_base[j].malts[i]          = current_brew.malts[i];
	recipe_data_base[j].malt_amounts[i]   = current_brew.malt_amounts[i];
	recipe_data_base[j].malt_colors[i]    = current_brew.malt_colors[i];
	recipe_data_base[j].malt_gravities[i] = current_brew.malt_gravities[i];
	recipe_data_base[j].malt_units[i]     = current_brew.malt_units[i];
    }
    for (i = 0; i < MAX_NUM_HOPS; i++)
    {
	recipe_data_base[j].hops[i]        = current_brew.hops[i];
	recipe_data_base[j].hop_amounts[i] = current_brew.hop_amounts[i];
	recipe_data_base[j].hop_times[i]   = current_brew.hop_times[i];
	recipe_data_base[j].hop_IBUs[i]    = current_brew.hop_IBUs[i];
	recipe_data_base[j].hop_units[i]   = current_brew.hop_units[i];
    }
    recipe_data_base[j].style                 = current_brew.style;
    recipe_data_base[j].total_volume          = current_brew.total_volume;
    recipe_data_base[j].total_volume_units    = current_brew.total_volume_units;
    recipe_data_base[j].boil_volume           = current_brew.boil_volume;
    recipe_data_base[j].boil_volume_units     = current_brew.boil_volume_units;
    recipe_data_base[j].num_malts             = current_brew.num_malts;
    recipe_data_base[j].num_hops              = current_brew.num_hops;
    recipe_data_base[j].extraction_efficiency = current_brew.extraction_efficiency;
    recipe_data_base[j].apparent_attenuation  = current_brew.apparent_attenuation;
    recipe_data_base[j].real_attenuation      = current_brew.real_attenuation;
    recipe_data_base[j].original_gravity      = current_brew.original_gravity;
    recipe_data_base[j].final_gravity         = current_brew.final_gravity;
    recipe_data_base[j].alcohol_by_volume     = current_brew.alcohol_by_volume;
    recipe_data_base[j].color                 = current_brew.color;
    recipe_data_base[j].IBUs                  = current_brew.IBUs;
    
    mvaddstr(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 5,
	     "                                                                           ");
    move(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 37);
    standout();
    printw("Recipe Saved!");
    refresh();
    sleep(3);
    standend();
    mvaddstr(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 5,
	     "                                                                           ");
    refresh();
    
    /* Sort the names of the recipes in the data base. */
    for (i = 0; i < num_recipes; i++)
    {
	for (j = 0; j < NAMESIZE; j++)
	{
	    ordered_recipes[i].name[j] = recipe_data_base[i].name[j];
	}
	ordered_recipes[i].data_base_location = i;
    }
    Sort_Recipe_Names();
    
}  /* End Save_Recipe */

    
void Real_Attenuation()
/****************************************************************/
/*  Routine Real_Attenuation calculates the real attenuation    */
/*  of the yeast based upon the input apparent attenuation.     */
/****************************************************************/
{
    
    current_brew.real_attenuation = 0.8192 *
	current_brew.apparent_attenuation;
    
}  /* End Real_Attenuation */


void Select_Malts(initial_selection)
/****************************************************************/
/*  Routine Select_Malts allows the user to select up to        */
/*  MAX_NUM_MALTS grains and adjuncts from the malts data       */
/*  base file.                                                  */
/****************************************************************/
int initial_selection;
{
    int i;
    int j;
    int temp = 0;
    
    mvaddstr(10, 5, "Malt/Adjunct Name");
    mvaddstr(11, 5, "-----------------");
    mvaddstr(10, 37, "Qty.");
    mvaddstr(11, 37, "----");
    mvaddstr(10, 44, "Unit");
    mvaddstr(11, 44, "----");
    mvaddstr(10, 52, "Color");
    mvaddstr(11, 52, "-----");
    mvaddstr(10, 60, "Gravity");
    mvaddstr(11, 60, "-------");
    
    /* Initialize the malt data for the current brew. */
    current_brew.num_malts = 0;
    for (i = 0; i < MAX_NUM_MALTS; i++)
    {
	current_brew.malt_colors[i] = 0;
	current_brew.malt_gravities[i] = 0;
    }
    if (initial_selection)
    {
	for (i = 0; i < MAX_NUM_MALTS; i++)
	{
	    current_brew.malt_amounts[i] = 0;
	}
    }
    
    /* Select the malts for the current brew. */
    for (i = 0; i < MAX_NUM_MALTS; i++)
    {
	move(12+i, 5);
	refresh();
	/* Select one of the malts from the data base. */
	temp = pick_one_malt(malts, num_malts);
	if (temp == 0)
	{
	    /* End malt selection. */
	    for (j = 0; (j + i) < MAX_NUM_MALTS; j++)
	    {
		mvaddstr(12+i+j, 5,
			 "                                                                           ");
	    }
	    i = MAX_NUM_MALTS;
	}
	else
	{
	    /* Accept the specified malt. */
	    current_brew.num_malts++;
	    current_brew.malts[i] = malts[temp];
	    
	    /* Input the amount of the specified malt. */
	    move(12+i, 35);
	    if (initial_selection)
	    {
		current_brew.malt_amounts[i] = get_float(0.0, 6, 2);
	    }
	    else
	    {
		current_brew.malt_amounts[i] = get_float(current_brew.malt_amounts[i], 6, 2);
	    }
	    refresh();
	    
	    /* Select the weight unit of the specified malt. */
	    if (initial_selection)
	    {
		move(12+i, 46);
		printw("lb");
		refresh();
	    }
	    move(12+i, 46);
	    current_brew.malt_units[i] = pick_one_unit(malt_weight_units, 4);
	    refresh();
	    
	    /* Calculate the contribution of the specified malt */
	    /* to the color of the brew. */
	    current_brew.malt_colors[i] = 
		Color((current_brew.total_volume*
		       volume_units[current_brew.total_volume_units].conversion_factor),
		      (current_brew.malt_amounts[i]*
		       malt_weight_units[current_brew.malt_units[i]].conversion_factor),
		      current_brew.malts[i].color);
	    move(12+i, 51);
	    printw("%6.2f", current_brew.malt_colors[i]);
	    refresh();
	    
	    /* Calculate the contribution of the specified malt */
	    /* to the specific gravity of the brew. */
	    current_brew.malt_gravities[i] = 
		Specific_Gravity((current_brew.total_volume*
				  volume_units[current_brew.total_volume_units].conversion_factor),
				 (current_brew.malt_amounts[i]*
				  malt_weight_units[current_brew.malt_units[i]].conversion_factor),
				 current_brew.malts[i].gravity);
	    if (current_brew.malts[i].converted == 0)
	    {
		current_brew.malt_gravities[i] =
		    current_brew.extraction_efficiency* 
			current_brew.malt_gravities[i]/100;
	    }
	    move(12+i, 61);
	    printw("%5.3f", current_brew.malt_gravities[i]);
	    refresh();
	}
    }
    
}  /* End Select_Malts */


void Total_Gravity()
/****************************************************************/
/*  Routine Total_Gravity calculates the original specific      */
/*  gravity of the wort by summing over all the chosen grains   */
/*  and adjuncts.                                               */
/****************************************************************/
{
    int i;
    
    /* Calculate the original gravity by summing over */
    /* all of the malts used in the current brew. */
    current_brew.original_gravity = 0;
    for (i = 0; i < current_brew.num_malts; i++)
    {
	current_brew.original_gravity = current_brew.original_gravity + 
	    current_brew.malt_gravities[i];
    }
    
    /* Calculate the final gravity from the original */
    /* gravity and apparent attenuation of the brew. */
    current_brew.final_gravity = (current_brew.original_gravity)*
	(1 - (current_brew.apparent_attenuation)/100) + 1;
    
    current_brew.original_gravity = current_brew.original_gravity + 1;
  
}  /* End Total_Gravity */


void Total_Alcohol()
/****************************************************************/
/*  Routine Total_Alcohol calculates the alcohol by volume in   */
/*  the finished beer based upon the original and final spec-   */
/*  ific gravities of the wort.                                 */
/****************************************************************/
{
    
    /* Calculate the alcohol by volume from the */
    /* original and final gravities of the brew.. */
    current_brew.alcohol_by_volume =
	Alcohol_by_Volume(current_brew.original_gravity,
			  current_brew.final_gravity);
  
}  /* End Total_Alcohol */


void Total_Color()
/****************************************************************/
/*  Routine Total_Color calculates the color of the wort by     */
/*  summing over all the chosen grains and adjuncts.            */
/****************************************************************/
{
    int i;
    
    /* Calculate the color by summing over */
    /* all of the malts used in the current brew. */
    current_brew.color = 0;
    for (i = 0; i < current_brew.num_malts; i++)
    {
	current_brew.color = current_brew.color + 
	    current_brew.malt_colors[i];
    }
    
}  /* End Total_Color */


void Select_Hops(initial_selection)
/****************************************************************/
/*  Routine Select_Hops allows the user to select up to         */
/*  MAX_NUM_HOPS hops from the hops data base file.             */
/****************************************************************/
int initial_selection;
{
    int i;
    int j;
    int temp = 0;
    
    mvaddstr(13+MAX_NUM_MALTS, 5, "Hop Name");
    mvaddstr(14+MAX_NUM_MALTS, 5, "--------");
    mvaddstr(13+MAX_NUM_MALTS, 38, "AA%");
    mvaddstr(14+MAX_NUM_MALTS, 38, "---");
    mvaddstr(13+MAX_NUM_MALTS, 45, "Qty.");
    mvaddstr(14+MAX_NUM_MALTS, 45, "----");
    mvaddstr(13+MAX_NUM_MALTS, 52, "Unit");
    mvaddstr(14+MAX_NUM_MALTS, 52, "----");
    mvaddstr(13+MAX_NUM_MALTS, 59, "Time");
    mvaddstr(14+MAX_NUM_MALTS, 59, "----");
    mvaddstr(13+MAX_NUM_MALTS, 67, "IBUs");
    mvaddstr(14+MAX_NUM_MALTS, 67, "----");
    
    /* Initialize the hop data for the current brew. */
    current_brew.num_hops = 0;
    for (i = 0; i < MAX_NUM_HOPS; i++)
    {
	current_brew.hop_IBUs[i] = 0;
    }
    if (initial_selection)
    {
	for (i = 0; i < MAX_NUM_HOPS; i++)
	{
	    current_brew.hops[i].alpha_acid_percentage = 0;
	    current_brew.hop_amounts[i] = 0;
	    current_brew.hop_times[i] = 0;
	}
    }
    
    /* Select the hops for the current brew. */
    for (i = 0; i < MAX_NUM_HOPS; i++)
    {
	move(15+MAX_NUM_MALTS+i, 5);
	refresh();
	/* Select one of the hops from the data base. */
	temp = pick_one_hop(hops, num_hops);
	if (temp == 0)
	{
	    /* End hop selection. */
	    for (j = 0; (j + i) < MAX_NUM_HOPS; j++)
	    {
		mvaddstr(15+MAX_NUM_MALTS+i+j, 5,
			 "                                                                           ");
	    }
	    i = MAX_NUM_HOPS;
	}
	else
	{
	    /* Accept the specified hop. */
	    current_brew.num_hops++;
	    current_brew.hops[i] = hops[temp];
	    
	    /* Input the alpha acid percentage of the specified hop. */
	    move(15+MAX_NUM_MALTS+i, 36);
	    current_brew.hops[i].alpha_acid_percentage =
		get_float(current_brew.hops[i].alpha_acid_percentage, 5, 2);
	    refresh();
	    
	    /* Input the amount of the specified hop. */
	    move(15+MAX_NUM_MALTS+i, 44);
	    if (initial_selection)
	    {
		current_brew.hop_amounts[i] = get_float(0.0, 5, 2);
	    }
	    else
	    {
		current_brew.hop_amounts[i] = get_float(current_brew.hop_amounts[i], 5, 2);
	    }
	    refresh();
	    
	    /* Select the weight unit of the specified hop. */
	    if (initial_selection)
	    {
		move(15+MAX_NUM_MALTS+i, 54);
		printw("oz");
		refresh();
	    }
	    move(15+MAX_NUM_MALTS+i, 54);
	    current_brew.hop_units[i] = pick_one_unit(hop_weight_units, 2);
	    refresh();
	    
	    /* Input the boil time of the specified hop. */
	    move(15+MAX_NUM_MALTS+i, 60);
	    if (initial_selection)
	    {
		current_brew.hop_times[i] = get_int(0, 2);
	    }
	    else
	    {
		current_brew.hop_times[i] = get_int(current_brew.hop_times[i], 2);
	    }
	    refresh();
	    
	    /* Calculate the contribution of the specified hop */
	    /* to the bitterness of the brew. */
	    current_brew.hop_IBUs[i] = (current_brew.hop_amounts[i]*
		hop_weight_units[current_brew.hop_units[i]].conversion_factor)*
		    IBU_per_oz(current_brew.hops[i].alpha_acid_percentage,
			       current_brew.hop_times[i],
			       (current_brew.total_volume*
				volume_units[current_brew.total_volume_units].conversion_factor),
			       (current_brew.boil_volume*
				volume_units[current_brew.boil_volume_units].conversion_factor),
			       current_brew.original_gravity);
	    move(15+MAX_NUM_MALTS+i, 66);
	    printw("%5.2f", current_brew.hop_IBUs[i]);
	}
    }
    
}  /* End Select_Hops */


void Total_IBUs()
/****************************************************************/
/*  Routine Total_IBUs calculates the bitterness of the wort    */
/*  (IBUs) by summing over all the chosen hops.                 */
/****************************************************************/
{
    int i;
    
    /* Calculate the bitterness by summing over */
    /* all of the hops used in the current brew. */
    current_brew.IBUs = 0;
    for (i = 0; i < current_brew.num_hops; i++)
    {
	current_brew.IBUs = current_brew.IBUs + 
	    current_brew.hop_IBUs[i];
    }
    
}  /* End Total_IBUs */


void Print_Guidelines()
/****************************************************************/
/*  Routine Print_Guidelines prints the minimum and maximum     */
/*  values of original gravity, alcohol by volume, color and    */
/*  IBUs for the chosen style, along with the values for your   */
/*  recipe.                                                     */
/****************************************************************/
{
    int brew;
    
    brew = current_brew.style;
    
    mvaddstr(16+MAX_NUM_MALTS+MAX_NUM_HOPS, 34,
	     "Min");
    mvaddstr(17+MAX_NUM_MALTS+MAX_NUM_HOPS, 34,
	     "---");
    mvaddstr(16+MAX_NUM_MALTS+MAX_NUM_HOPS, 43,
	     "Yours");
    mvaddstr(17+MAX_NUM_MALTS+MAX_NUM_HOPS, 43,
	     "-----");
    mvaddstr(16+MAX_NUM_MALTS+MAX_NUM_HOPS, 54,
	     "Max");
    mvaddstr(17+MAX_NUM_MALTS+MAX_NUM_HOPS, 54,
	     "---");
    mvaddstr(18+MAX_NUM_MALTS+MAX_NUM_HOPS, 24,
	     "O.G.:");
    mvaddstr(19+MAX_NUM_MALTS+MAX_NUM_HOPS, 21,
	     "Alcohol:");
    mvaddstr(20+MAX_NUM_MALTS+MAX_NUM_HOPS, 23,
	     "Color:");
    mvaddstr(21+MAX_NUM_MALTS+MAX_NUM_HOPS, 24,
	     "IBUs:");
    
    /* Print the actual and style limits for */
    /* the original gravity of the brew. */
    move(18+MAX_NUM_MALTS+MAX_NUM_HOPS, 33);
    printw("%5.3f", styles[brew].original_gravity_min);
    move(18+MAX_NUM_MALTS+MAX_NUM_HOPS, 43);
    printw("%5.3f", current_brew.original_gravity);
    move(18+MAX_NUM_MALTS+MAX_NUM_HOPS, 53);
    printw("%5.3f", styles[brew].original_gravity_max);
    
    /* Print the actual and style limits for */
    /* the alcohol by volume of the brew. */
    move(19+MAX_NUM_MALTS+MAX_NUM_HOPS, 34);
    printw("%4.1f", styles[brew].alcohol_min);
    move(19+MAX_NUM_MALTS+MAX_NUM_HOPS, 44);
    printw("%4.1f", current_brew.alcohol_by_volume);
    move(19+MAX_NUM_MALTS+MAX_NUM_HOPS, 54);
    printw("%4.1f", styles[brew].alcohol_max);
  
    /* Print the actual and style limits for */
    /* the color of the brew. */
    move(20+MAX_NUM_MALTS+MAX_NUM_HOPS, 33);
    printw("%5.1f", styles[brew].color_min);
    move(20+MAX_NUM_MALTS+MAX_NUM_HOPS, 43);
    printw("%5.1f", current_brew.color);
    move(20+MAX_NUM_MALTS+MAX_NUM_HOPS, 53);
    printw("%5.1f", styles[brew].color_max);
    
    /* Print the actual and style limits for */
    /* the bitterness of the brew. */
    move(21+MAX_NUM_MALTS+MAX_NUM_HOPS, 33);
    printw("%5.1f", (float)styles[brew].IBU_min);
    move(21+MAX_NUM_MALTS+MAX_NUM_HOPS, 43);
    printw("%5.1f", current_brew.IBUs);
    move(21+MAX_NUM_MALTS+MAX_NUM_HOPS, 53);
    printw("%5.1f", (float)styles[brew].IBU_max);
    
    refresh();
    
}  /* End Print_Guidelines */


void Print_Recipe_File()
/****************************************************************/
/*  Routine Print_Recipe_File prints summary information about  */
/*  the recipe to an ASCII file.                                */
/****************************************************************/
{
    int i;
    char filename[NAMESIZE];
    FILE *file_pointer;
    
    /* Initialize the name of the recipe file. */
    for (i = 0; i < NAMESIZE; i++)
    {
	filename[i] = ' ';
    }
    
    /* Input the name of the recipe file. */
    mvaddstr(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 5,
	     "                                                                           ");
    mvaddstr(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 21,
	     "Name of Recipe File:");
    move(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 42);
    get_string(filename, NAMESIZE, 0);
    mvaddstr(24+MAX_NUM_MALTS+MAX_NUM_HOPS, 5,
	     "                                                                           ");
    
    /* Open the recipe file. */
    file_pointer = fopen(filename, "w");
    if (file_pointer == NULL)
    {
	fprintf(stderr, "BrewCalc: Print_Recipe_File: Open of %s failed!\n", filename);
	exit(1);
    }
    
    /* Write the general description data for the current brew. */
    fprintf(file_pointer, "\n\n\t\t********************************\n");
    fprintf(file_pointer, "\t\t*       BrewCalc - V%.1f        *\n",
	    VERSION);
    fprintf(file_pointer, "\t\t*         Recipe Report        *\n");
    fprintf(file_pointer, "\t\t********************************\n");
    
    fprintf(file_pointer, "\n\n");
    
    fprintf(file_pointer, "Recipe Name:\t\t\t%s\n",
	    current_brew.name);
    fprintf(file_pointer, "Style:\t\t\t\t%s\n",
	    &styles[current_brew.style]);
    fprintf(file_pointer, "Total Volume:\t\t\t%4.1f %-7s\n",
	    current_brew.total_volume,
	    volume_units[current_brew.total_volume_units].name);
    fprintf(file_pointer, "Boil Volume:\t\t\t%4.1f %-7s\n",
	    current_brew.boil_volume,
	    volume_units[current_brew.boil_volume_units].name);
    fprintf(file_pointer, "Apparent Attenuation:\t\t%4.1f percent\n",
	    current_brew.apparent_attenuation);
    fprintf(file_pointer, "Extraction Efficiency:\t\t%4.1f percent\n",
	    current_brew.extraction_efficiency);
    
    /* Write the grain and adjunct data for the current brew. */
    fprintf(file_pointer, "\n\n\t\t      Grains/Adjuncts\n\t\t      ***************\n\n");
    
    fprintf(file_pointer, "Type\t\t\t\t  Qty.\t Unit\t Color\t Gravity\n");
    fprintf(file_pointer, "----\t\t\t\t  ----\t ----\t -----\t -------\n");
    for (i = 0; i < current_brew.num_malts; i++)
    {
	fprintf(file_pointer, "%-25s\t %5.2f\t %4s\t %5.2f\t %7.3f\n",
		current_brew.malts[i].name,
		current_brew.malt_amounts[i],
		malt_weight_units[current_brew.malt_units[i]].name,
		current_brew.malt_colors[i],
		current_brew.malt_gravities[i]);
    }
    fprintf(file_pointer, "\t\t\t\t\t\t -----\t -------\n");
    fprintf(file_pointer, "Total Color/Gravity:\t\t\t\t %5.2f\t %7.3f\n",
	    current_brew.color, current_brew.original_gravity);
    
    /* Write the hop data for the current brew. */
    fprintf(file_pointer, "\n\n\t\t\t   Hops\n\t\t\t   ****\n\n");
    
    fprintf(file_pointer, "Type\t\t\t\t   AA\t  Qty.\t Unit\t  Time\t  IBUs\t\n");
    fprintf(file_pointer, "----\t\t\t\t   --\t  ----\t ----\t  ----\t  ----\t\n");
    for (i =0; i < current_brew.num_hops; i++)
    {
	fprintf(file_pointer, "%-25s\t %5.2f\t %5.2f\t %4s\t %4d\t %5.2f\n",
		current_brew.hops[i].name,
		current_brew.hops[i].alpha_acid_percentage,
		current_brew.hop_amounts[i],
		hop_weight_units[current_brew.hop_units[i]].name,
		current_brew.hop_times[i],
		current_brew.hop_IBUs[i]);
    }
    fprintf(file_pointer, "\t\t\t\t\t\t\t\t -----\n");
    fprintf(file_pointer, "Total IBUs:\t\t\t\t\t\t\t %5.2f",
	    current_brew.IBUs);
    
    fprintf(file_pointer, "\n\n");
    
    /* Close the recipe file. */
    fclose(file_pointer);
    
}  /* End Print_Recipe_File */


float Hop_Utilization(minutes)
/****************************************************************/
/*  Routine Hop_Utilization calculates the utilization of the   */
/*  hops for the specified boiling time.  This formula is due   */
/*  to Jackie Rager, Zymurgy Hops and Beer Special Issue 1990.  */
/****************************************************************/
float minutes;
{
    float hop_utilization;
    
    hop_utilization = (18.10907 + 13.86204*
		       (float)tanh((minutes - 31.32275)/18.26774))/
			   100;
    
    return(hop_utilization);
    
}  /* End Hop_Utilization */


float IBU_per_oz(alpha, boil_time, total_volume, boil_volume, gravity)
/****************************************************************/
/*  Routine IBU_per_oz calculates the bitterness (IBUs) per     */
/*  ounce of hops.  This formula is due to Jackie Rager, Zym-   */
/*  urgy Hops and Beer Special Issue 1990.                      */
/****************************************************************/
float alpha;
float boil_time;
float total_volume;
float boil_volume;
float gravity;
{
    float ibu_per_oz;
    float adjustment = 0;
    
    if (gravity > 1.050)
    {
	adjustment = ((1 + (gravity - 1)*(total_volume/boil_volume)) - 1.050)
	    / 0.2;
    }
    else
    {
	adjustment = 0;
    }
    
    ibu_per_oz = Hop_Utilization(boil_time)*(alpha/100)*7490/
	(total_volume*(1 + adjustment));
    
    return(ibu_per_oz);
    
}  /* End IBU_per_oz */


float Specific_Gravity(batch_size, weight, gravity)
/****************************************************************/
/*  Routine Specific_Gravity calculates the contribution to     */
/*  the specific gravity of the wort due to a single grain or   */
/*  adjunct for the specified weight (pounds) and total volume  */
/*  (gallons).                                                  */
/****************************************************************/
float batch_size;
float weight;
float gravity;
{
    float partial_specific_gravity;
    
    partial_specific_gravity = (weight*(gravity - 1))/batch_size;
    
    return(partial_specific_gravity);
    
}  /* End Specific_Gravity */


float Color(batch_size, weight, color)
/****************************************************************/
/*  Routine Color calculates the contribution to the color of   */
/*  the wort due to a single grain/adjunct for the specified    */
/*  weight (pounds) and total volume (gallons).                 */
/****************************************************************/
float batch_size;
float weight;
float color;
{
    float partial_color;
    
    partial_color = (weight*color)/batch_size;
    
    return(partial_color);
    
}  /* End Color */


float Alcohol_by_Weight(original_gravity, final_gravity)
/****************************************************************/
/*  Routine Alcohol_by_Weight calculates the percentage of      */
/*  alcohol by weight of the finished beer based upon the       */
/*  original and final specific gravities of the wort.          */
/****************************************************************/
float original_gravity;
float final_gravity;
{
    float alcohol_by_weight;
    
    alcohol_by_weight = (original_gravity - final_gravity)/
	(2.310742e-2 - 1.301758e-2*original_gravity);
    
    return(alcohol_by_weight);
    
}  /* End Alcohol_by_Weight */


float Alcohol_by_Volume(original_gravity, final_gravity)
/****************************************************************/
/*  Routine Alcohol_by_Volume calculates the percentage of      */
/*  alcohol by volume of the finished beer based upon the       */
/*  original and final specific gravities of the wort.          */
/****************************************************************/
float original_gravity;
float final_gravity;
{
    float alcohol_by_volume;
    
    alcohol_by_volume = (original_gravity - final_gravity)/
	(1.892960e-2 - 1.066400e-2*original_gravity);
    
    return(alcohol_by_volume);
    
}  /* End Alcohol_by_Volume */
