/******************************************/
/* readplan.c                             */
/*                                        */
/* Functions to read the files of         */
/* VGA-Planets  (RST- and DAT-Files,      */
/* not TRN-Files, don't know the format)  */
/*                                        */
/* If you have problems with the binary   */
/* format of your machine, change the     */
/* Makefile or look in readWord, readLong,*/
/* readInt and readString                 */
/*                                        */
/* (c) 1995 C. Quix                       */
/* Email: Christoph_Quix@p17.ks.fido.de   */
/******************************************/
#include <stdlib.h>
#include "planets.h"


void skip(FILE *fp, WORD i)  {
/* Skip i bytes of file fp */
	char ch;
	WORD j;
	
	for(j=0;j<i;j++) fread(&ch,1,1,fp);
}

WORD readWord(FILE *fp)  {
/* Read one WORD (2 Bytes, 0x1234 is stored as 34 12) from fp */
	WORD w;
#ifdef PC
	fread(&w,1,2,fp);
#else
	fread(((void *)&w)+1,1,1,fp);
	fread(&w,1,1,fp);
#endif
	return w;
}

INT readInt(FILE *fp)  {
/* Read one INT (2 Bytes, 0x1234 is stored as 34 12) from fp */
	INT i;
#ifdef PC
	fread(&i,1,2,fp);
#else
	fread(((void *)&i)+1,1,1,fp);
	fread(&i,1,1,fp);
#endif
	return i;
}

LONG readLong(FILE *fp)  {
/* Read one LONG (4 Bytes, 0x12345678 is stored as 78 56 34 12) from fp) */
	LONG l;
#ifdef PC
	fread(&l,1,4,fp);
#else
	fread(((void *)&l)+3,1,1,fp);
	fread(((void *)&l)+2,1,1,fp);
	fread(((void *)&l)+1,1,1,fp);
	fread(&l,1,1,fp);
#endif
	return l;
}

void readString(FILE *fp, char *str, WORD length)  {
/* Important: the memory for str must be already allocated AND
 *            the string is NOT terminated by NULL! */
	fread(str,1,length,fp);
}


/*************************/
/* Read a RST-File       */
/*************************/
RstFile* readRst(FILE *fp)  {
	RstFile *rst;
	
	rst=(RstFile *) malloc(sizeof(RstFile));
	
	skip(fp,33);
	
	rst->numberOfShips=readWord(fp);
	rst->shipInfo=readShips(fp, rst->numberOfShips);

	rst->numberOfAlienShips=readWord(fp);
	rst->alienShipInfo=readAlienShips(fp, rst->numberOfAlienShips);
	
	rst->numberOfPlanets=readWord(fp);
	rst->planetInfo=readPlanets(fp,rst->numberOfPlanets);
	
	rst->numberOfStarbases=readWord(fp);
	rst->starbaseInfo=readStarbases(fp,rst->numberOfStarbases);

	rst->numberOfMessages=readWord(fp);
	rst->messageInfo=readMessages(fp,rst->numberOfMessages,0);
	
	readShipXY(fp,rst->shipXYInfo);
	readString(fp,rst->timestamp,18);
	readScores(fp,rst->scoreInfo);
	
	rst->playernumber=readWord(fp);
	readString(fp,rst->password,20);
	readString(fp,rst->checkInfo,12);
	rst->turnNumber=readWord(fp);
	rst->unknown=readWord(fp);
	
	rst->numberOfVCRs=readWord(fp);
	rst->vcrInfo=readVCRs(fp,rst->numberOfVCRs);
	
	return rst;
}


RstFile* readDatFiles(WORD raceNumber, char *path)  {
	
	RstFile *rst;
	FILE *fp;
	char fname[120];
	
	rst=(RstFile *) malloc(sizeof(RstFile));

	/* SHIP??.DAT */
	sprintf(fname,"%s/ship%d.dat",path,raceNumber);
	fp=fopen(fname,"rb");
	if (fp)  {
		rst->numberOfShips=readWord(fp);
		rst->shipInfo=readShips(fp, rst->numberOfShips);
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open %s", fname);
		help();
	}

	/* TARGET??.DAT */
	sprintf(fname,"%s/target%d.dat",path,raceNumber);
	fp=fopen(fname,"rb");
	if (fp)  {
		rst->numberOfAlienShips=readWord(fp);
		rst->alienShipInfo=readAlienShips(fp, rst->numberOfAlienShips);
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open %s", fname);
		help();
	}

	/* PDATA??.DAT */
	sprintf(fname,"%s/pdata%d.dat",path,raceNumber);
	fp=fopen(fname,"rb");
	if (fp)  {
		rst->numberOfPlanets=readWord(fp);
		rst->planetInfo=readPlanets(fp,rst->numberOfPlanets);
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open %s", fname);
		help();
	}

	/* BDATA??.DAT */
	sprintf(fname,"%s/bdata%d.dat",path,raceNumber);
	fp=fopen(fname,"rb");
	if (fp)  {
		rst->numberOfStarbases=readWord(fp);
		rst->starbaseInfo=readStarbases(fp,rst->numberOfStarbases);
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open %s", fname);
		help();
	}

	/* MDATA??.DAT */
	sprintf(fname,"%s/mdata%d.dat",path,raceNumber);
	fp=fopen(fname,"rb");
	if (fp)  {
		rst->numberOfMessages=readWord(fp);
		rst->messageInfo=readMessages(fp,rst->numberOfMessages,1);
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open %s", fname);
		help();
	}
	
	/* SHIPXY??.DAT */
	sprintf(fname,"%s/shipxy%d.dat",path,raceNumber);
	fp=fopen(fname,"rb");
	if (fp)  {
		readShipXY(fp,rst->shipXYInfo);
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open %s", fname);
		help();
	}
	
	/* GEN??.DAT */
	sprintf(fname,"%s/gen%d.dat",path,raceNumber);
	fp=fopen(fname,"rb");
	if (fp)  {
		readString(fp,rst->timestamp,18);
		readScores(fp,rst->scoreInfo);
		rst->playernumber=readWord(fp);
		readString(fp,rst->password,20);
		/* readString(fp,rst->checkInfo,12);   Don't know? */
		skip(fp,25);    
		rst->turnNumber=readWord(fp);
		rst->unknown=readWord(fp);
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open %s", fname);
		help();
	}

	/* VCR??.DAT */
	sprintf(fname,"%s/vcr%d.dat",path,raceNumber);
	fp=fopen(fname,"rb");
	if (fp)  {
		rst->numberOfVCRs=readWord(fp);
		rst->vcrInfo=readVCRs(fp,rst->numberOfVCRs);
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open %s", fname);
		help();
	}
	
	return rst;
}


Ship* readShips(FILE *fp, WORD num)  {
	Ship *s;
	WORD i;
	
	s=(Ship *) malloc(sizeof(Ship)*num);

	for(i=0;i<num; i++)  {
		s[i].id=readWord(fp);
		s[i].owner=readWord(fp);
		readString(fp,s[i].friendlyCode,3);
		s[i].warp=readWord(fp);
		s[i].deltax=readInt(fp);
		s[i].deltay=readInt(fp);
		s[i].x=readWord(fp);
		s[i].y=readWord(fp);
		s[i].engineIndex=readWord(fp);
		s[i].hullType=readWord(fp);
		s[i].beamIndex=readWord(fp);
		s[i].numberOfBeams=readWord(fp);
		s[i].numberOfBays=readWord(fp);
		s[i].torpIndex=readWord(fp);
		s[i].numberOfFightTorp=readWord(fp);
		s[i].numberOfLaunchers=readWord(fp);
		s[i].mission=readWord(fp);
		s[i].enemy=readWord(fp);
		s[i].towed=readWord(fp);
		s[i].damage=readWord(fp);
		s[i].crew=readWord(fp);
		s[i].clans=readWord(fp);
		readString(fp,s[i].name,20);
		s[i].fuel=readWord(fp);
		s[i].trit=readWord(fp);
		s[i].dura=readWord(fp);
		s[i].moly=readWord(fp);
		s[i].supplies=readWord(fp);
		s[i].clansToPlanet=readWord(fp);
		s[i].fuelToPlanet=readWord(fp);
		s[i].tritToPlanet=readWord(fp);
		s[i].duraToPlanet=readWord(fp);
		s[i].molyToPlanet=readWord(fp);
		s[i].suppliesToPlanet=readWord(fp);
		s[i].planetID=readWord(fp);
		s[i].clansToShip=readWord(fp);
		s[i].fuelToShip=readWord(fp);
		s[i].tritToShip=readWord(fp);
		s[i].duraToShip=readWord(fp);
		s[i].molyToShip=readWord(fp);
		s[i].suppliesToShip=readWord(fp);
		s[i].shipID=readWord(fp);
		s[i].intercept=readWord(fp);
		s[i].megaCredits=readWord(fp);
	}
	
	return s;
}


AlienShip* readAlienShips(FILE *fp, WORD num)  {
	AlienShip *a;
	WORD i;
	
	a=(AlienShip*) malloc(sizeof(AlienShip)*num);
	
	for(i=0;i<num;i++)  {
		a[i].id=readWord(fp);
		a[i].owner=readWord(fp);
		a[i].warp=readWord(fp);
		a[i].x=readWord(fp);
		a[i].y=readWord(fp);
		a[i].hull=readWord(fp);
		a[i].angle=readWord(fp);
		readString(fp, a[i].name, 20);
	}
	
	return a;
}

Planet* readPlanets(FILE *fp, WORD num) {
	Planet *p;
	WORD i;
	
	p=(Planet*) malloc(sizeof(Planet)*num);
	
	for(i=0;i<num;i++) {
		p[i].owner=readWord(fp);
		p[i].id=readWord(fp);
		readString(fp,p[i].friendlyCode,3);
		p[i].mines=readWord(fp);
		p[i].facs=readWord(fp);
		p[i].defposts=readWord(fp);
		p[i].fuel=readLong(fp);
		p[i].trit=readLong(fp);
		p[i].dura=readLong(fp);
		p[i].moly=readLong(fp);
		p[i].clans=readLong(fp);
		p[i].supplies=readLong(fp);
		p[i].megaCredits=readLong(fp);
		p[i].fuelGround=readLong(fp);
		p[i].tritGround=readLong(fp);
		p[i].duraGround=readLong(fp);
		p[i].molyGround=readLong(fp);
		p[i].fuelMiningRate=readWord(fp);
		p[i].tritMiningRate=readWord(fp);
		p[i].duraMiningRate=readWord(fp);
		p[i].molyMiningRate=readWord(fp);
		p[i].colonistTax=readWord(fp);
		p[i].nativeTax=readWord(fp);
		p[i].colonistStatus=readInt(fp);
		p[i].nativeStatus=readInt(fp);
		p[i].government=readWord(fp);
		p[i].natives=readLong(fp);
		p[i].nativeRace=readWord(fp);
		p[i].temperature=100-readWord(fp);
		p[i].buildStarbase=readWord(fp);
	}
	return p;
}

Starbase*    readStarbases(FILE *fp, WORD num)  {
	Starbase *s;
	WORD i,j;
	
	s=(Starbase*) malloc(sizeof(Starbase)*num);
	
	for(i=0;i<num;i++) {
		s[i].id=readWord(fp);
		s[i].owner=readWord(fp);
		s[i].defposts=readWord(fp);
		s[i].damage=readWord(fp);
		s[i].engineTechLevel=readWord(fp);
		s[i].hullTechLevel=readWord(fp);
		s[i].beamTechLevel=readWord(fp);
		s[i].torpTechLevel=readWord(fp);
		for(j=0;j<MAXENG;j++)
			s[i].engines[j]=readWord(fp);
		for(j=0;j<20;j++)
			s[i].hulls[j]=readWord(fp);
		for(j=0;j<MAXBEAM;j++)
			s[i].beams[j]=readWord(fp);
		for(j=0;j<MAXTORP;j++)
			s[i].launchers[j]=readWord(fp);
		for(j=0;j<MAXTORP;j++)
			s[i].torps[j]=readWord(fp);
		s[i].fighters=readWord(fp);
		s[i].fixID=readWord(fp);
		s[i].fixShip=readWord(fp);
		s[i].mission=readWord(fp);
		s[i].ship=readWord(fp);
		s[i].shipEngineIndex=readWord(fp);
		s[i].shipBeamIndex=readWord(fp);
		s[i].shipBeamsOnShip=readWord(fp);
		s[i].shipTorpIndex=readWord(fp);
		s[i].shipTorpsOnShip=readWord(fp);
		s[i].shipFightersOnShip=readWord(fp);
	}
	return s;
}


Message* readMessages(FILE *fp, WORD num, WORD isDat)  {
	Message *m;
	WORD i,j;
	char ch;
	
	m=(Message*) malloc(sizeof(Message)*num);
	
	for(i=0;i<num;i++) {
		m[i].offset=readLong(fp);
		m[i].length=readWord(fp);
		m[i].text=(char*) malloc(sizeof(char)*m[i].length);
	}
	if (!isDat)
		fread(&ch,1,1,fp); /* Unused NULL-Byte */
	
	for(i=0;i<num;i++)  {
		readString(fp,m[i].text,m[i].length);
		for (j=0;j<m[i].length;j++)
			m[i].text[j]-=13;
	}
	return m;
}


Vcr* readVCRs(FILE *fp, WORD num)  {
	Vcr *v;
	WORD i;
	
	v=(Vcr*) malloc(sizeof(Vcr)*num);
	
	for(i=0;i<num;i++) {
		v[i].unknown[0]=readWord(fp);
		v[i].unknown[1]=readWord(fp);
		v[i].unknown[2]=readWord(fp);
		v[i].unknown[3]=readWord(fp);
		v[i].mass1=readWord(fp);
		v[i].mass2=readWord(fp);
		readString(fp,v[i].name1,20);
		v[i].damage1=readWord(fp);
		v[i].crew1=readWord(fp);
		v[i].id1=readWord(fp);
		v[i].race1=readWord(fp);
		v[i].bitmap1=readWord(fp);
		v[i].beamIndex1=readWord(fp);
		v[i].numberOfBeams1=readWord(fp);
		v[i].numberOfBays1=readWord(fp);
		v[i].torpIndex1=readWord(fp);
		v[i].numberOfTorpFighters1=readWord(fp);
		v[i].numberOfLaunchers1=readWord(fp);
		readString(fp,v[i].name2,20);
		v[i].damage2=readWord(fp);
		v[i].crew2=readWord(fp);
		v[i].id2=readWord(fp);
		v[i].race2=readWord(fp);
		v[i].bitmap2=readWord(fp);
		v[i].beamIndex2=readWord(fp);
		v[i].numberOfBeams2=readWord(fp);
		v[i].numberOfBays2=readWord(fp);
		v[i].torpIndex2=readWord(fp);
		v[i].numberOfTorpFighters2=readWord(fp);
		v[i].numberOfLaunchers2=readWord(fp);
		v[i].shield1=readWord(fp);
		v[i].shield2=readWord(fp);
	}
	return v;
}

void readShipXY(FILE *fp, ShipXY *sh)  {
	WORD i;
	
	for(i=0;i<MAXSHIP;i++)  {
		sh[i].x=readWord(fp);
		sh[i].y=readWord(fp);
		sh[i].owner=readWord(fp);
		sh[i].mass=readWord(fp);
	}
}
		
	
void readScores(FILE *fp, Score *scores)  {
	WORD i;

	for(i=0;i<MAXRACE;i++)  {
		scores[i].planets=readWord(fp);
		scores[i].ships=readWord(fp);
		scores[i].freighters=readWord(fp);
		scores[i].bases=readWord(fp);
	}
}

		
/***************************************************/
/* Functions to read general data                  */
/***************************************************/

void readHullSpec()  {
	
	extern HullSpec hullSpecs[MAXHULL];
	
	FILE *fp;
	WORD i;
	
	extern char GlobalPath[200];
	char fname[250];
	
	strcpy(fname,GlobalPath);
	strcat(fname,"/hullspec.dat");
	
	fp=fopen(fname,"rb");
	
	if (fp)  {
		for (i=0; i<MAXHULL; i++)  {
			readString(fp,hullSpecs[i].name,30);
			hullSpecs[i].picture=readWord(fp);
			hullSpecs[i].unknown=readWord(fp);
			hullSpecs[i].trit=readWord(fp);
			hullSpecs[i].dura=readWord(fp);
			hullSpecs[i].moly=readWord(fp);
			hullSpecs[i].fuelTank=readWord(fp);
			hullSpecs[i].crew=readWord(fp);
			hullSpecs[i].numberOfEngines=readWord(fp);
			hullSpecs[i].mass=readWord(fp);
			hullSpecs[i].techLevel=readWord(fp);
			hullSpecs[i].cargo=readWord(fp);
			hullSpecs[i].fighterBays=readWord(fp);
			hullSpecs[i].torpBays=readWord(fp);
			hullSpecs[i].beams=readWord(fp);
			hullSpecs[i].megaCredits=readWord(fp);
		}
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open hullspec.dat\n");
		help();
	}
}


void readTrueHulls()  {
	
	extern TrueHull trueHulls[MAXRACE];
	
	FILE *fp;
	WORD i,j;
	
	extern char GlobalPath[200];
	char fname[250];
	
	strcpy(fname,GlobalPath);
	strcat(fname,"/truehull.dat");

	fp=fopen(fname,"rb");
	
	if (fp)  {
		for (i=0; i<MAXRACE; i++)  {
			for (j=0;j<MAXHULLRACE; j++)  {
				trueHulls[i].hullIndex[j]=readWord(fp);
			}
		}
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open truehull.dat\n");
		help();
	}
}


void readBeamSpec()  {
	
	extern BeamSpec beamSpecs[MAXBEAM];
	
	FILE *fp;
	WORD i;
	
	extern char GlobalPath[200];
	char fname[250];
	
	strcpy(fname,GlobalPath);
	strcat(fname,"/beamspec.dat");
	
	fp=fopen(fname,"rb");
	
	if (fp)  {
		for (i=0; i<MAXBEAM; i++)  {
			readString(fp,beamSpecs[i].name,20);
			beamSpecs[i].megaCredits=readWord(fp);
			beamSpecs[i].trit=readWord(fp);
			beamSpecs[i].dura=readWord(fp);
			beamSpecs[i].moly=readWord(fp);
			beamSpecs[i].mass=readWord(fp);
			beamSpecs[i].techLevel=readWord(fp);
			beamSpecs[i].kill=readWord(fp);
			beamSpecs[i].explosive=readWord(fp);
		}
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open beamspec.dat\n");
		help();
	}
}


void readTorpSpec()  {
	
	extern TorpSpec torpSpecs[MAXTORP];
	
	FILE *fp;
	WORD i;
	
	extern char GlobalPath[200];
	char fname[250];
	
	strcpy(fname,GlobalPath);
	strcat(fname,"/torpspec.dat");
	
	fp=fopen(fname,"rb");
	
	if (fp)  {
		for (i=0; i<MAXTORP; i++)  {
			readString(fp,torpSpecs[i].name,20);
			torpSpecs[i].megaCreditsTorp=readWord(fp);
			torpSpecs[i].megaCreditsLauncher=readWord(fp);
			torpSpecs[i].trit=readWord(fp);
			torpSpecs[i].dura=readWord(fp);
			torpSpecs[i].moly=readWord(fp);
			torpSpecs[i].mass=readWord(fp);
			torpSpecs[i].techLevel=readWord(fp);
			torpSpecs[i].kill=readWord(fp);
			torpSpecs[i].explosive=readWord(fp);
		}
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open torpspec.dat\n");
		help();
	}
}


void readEngSpec()  {
	
	extern EngSpec engSpecs[MAXENG];
	
	FILE *fp;
	WORD i,j;
	
	extern char GlobalPath[200];
	char fname[250];
	
	strcpy(fname,GlobalPath);
	strcat(fname,"/engspec.dat");
	
	fp=fopen(fname,"rb");
	
	if (fp)  {
		for (i=0; i<MAXENG; i++)  {
			readString(fp,engSpecs[i].name,20);
			engSpecs[i].megaCredits=readWord(fp);
			engSpecs[i].trit=readWord(fp);
			engSpecs[i].dura=readWord(fp);
			engSpecs[i].moly=readWord(fp);
			engSpecs[i].techLevel=readWord(fp);
			for(j=0;j<9;j++)
				engSpecs[i].fuelUse[j]=readLong(fp);
		}
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open engspec.dat\n");
		help();
	}
}


void readXYPlan()  {
	extern XYPlan xyPlanetData[MAXPLANET];
	
	FILE *fp;
	WORD i;
	
	extern char GlobalPath[200];
	char fname[250];
	
	strcpy(fname,GlobalPath);
	strcat(fname,"/xyplan.dat");
	
	fp=fopen(fname,"rb");
	
	if (fp)  {
		for(i=0;i<MAXPLANET;i++)  {
			xyPlanetData[i].x=readWord(fp);
			xyPlanetData[i].y=readWord(fp);
			xyPlanetData[i].unknown=readWord(fp);
		}
		fclose(fp);
	}
	else  {
		fprintf(stderr,"Unable to open xyplan.dat\n");
		help();
	}
}


void readPlanetNames()  {
	extern PlanetName planetNames[MAXPLANET];
	
	FILE *fp;
	WORD i;
	
	extern char GlobalPath[200];
	char fname[250];
	
	strcpy(fname,GlobalPath);
	strcat(fname,"/planet.nm");

	fp=fopen(fname,"rb");
	if (fp) {
		for (i=0; i<MAXPLANET; i++)  {
			readString(fp, planetNames[i].name, 20);
		}
	}
	else  {
		fprintf(stderr,"Unable to open planet.nm\n");
		help();
	}
}


void readRaceNames()  {
	extern RaceName raceNames[MAXRACE];
	
	FILE *fp;
	WORD i;
	
	extern char GlobalPath[200];
	char fname[250];
	
	strcpy(fname,GlobalPath);
	strcat(fname,"/race.nm");
	
	fp=fopen(fname,"rb");
	if (fp) {
		for (i=0;i<MAXRACE;i++)
			readString(fp,raceNames[i].longName,30);
		for (i=0;i<MAXRACE;i++)
			readString(fp,raceNames[i].pluralName,20);
		for (i=0;i<MAXRACE;i++)
			readString(fp,raceNames[i].shortName,12);
	}
	else  {
		fprintf(stderr,"Unable to open race.nm\n");
		help();
	}
}
