package Bio::Tradis::CommandLine::RunMapping;
$Bio::Tradis::CommandLine::RunMapping::VERSION = '1.4.3';
# ABSTRACT: Perform mapping


use Moose;
use Getopt::Long qw(GetOptionsFromArray);
use Cwd 'abs_path';
use Bio::Tradis::Map;

has 'args'        => ( is => 'ro', isa => 'ArrayRef', required => 1 );
has 'script_name' => ( is => 'ro', isa => 'Str',      required => 1 );
has 'fastqfile'   => ( is => 'rw', isa => 'Str',      required => 0 );
has 'reference'   => ( is => 'rw', isa => 'Str',      required => 0 );
has 'help'        => ( is => 'rw', isa => 'Bool',     required => 0 );
has 'refname' =>
  ( is => 'rw', isa => 'Str', required => 0, default => 'ref.index' );
has 'outfile' =>
  ( is => 'rw', isa => 'Str', required => 0, default => 'mapped.sam' );
has 'smalt' => ( is => 'rw', isa => 'Maybe[Int]', required => 0, default => 0 );
has 'smalt_k' => ( is => 'rw', isa => 'Maybe[Int]', required => 0 );
has 'smalt_s' => ( is => 'rw', isa => 'Maybe[Int]', required => 0 );
has 'smalt_y' => ( is => 'rw', isa => 'Maybe[Num]', required => 0, default => 0.96 );
has 'smalt_r' => ( is => 'rw', isa => 'Maybe[Int]', required => 0, default => -1 );
has 'smalt_n' => ( is => 'rw', isa => 'Maybe[Int]', required => 0, default => 1 );


sub BUILD {
    my ($self) = @_;

    my ( $fastqfile, $ref, $refname, $outfile, $smalt, $smalt_k, $smalt_s, $smalt_y, $smalt_r,$smalt_n,  $help );

    GetOptionsFromArray(
        $self->args,
        'f|fastqfile=s'   => \$fastqfile,
        'r|reference=s'   => \$ref,
        'rn|refname=s'    => \$refname,
        'o|outfile=s'     => \$outfile,
        's|smalt=i'    => \$smalt,
	'sk|smalt_k=i'    => \$smalt_k,
	'ss|smalt_s=i'    => \$smalt_s,
	'sy|smalt_y=f'    => \$smalt_y,
	'sr|smalt_r=i'    => \$smalt_r,
	'n|smalt_n=i'     => \$smalt_n,
        'h|help'          => \$help
    );

    $self->fastqfile( abs_path($fastqfile) ) if ( defined($fastqfile) );
    $self->reference( abs_path($ref) )       if ( defined($ref) );
    $self->refname($refname)                 if ( defined($refname) );
    $self->outfile( abs_path($outfile) )     if ( defined($outfile) );
    $self->smalt( $smalt )               if ( defined($smalt) );
    $self->smalt_k( $smalt_k )               if ( defined($smalt_k) );
    $self->smalt_s( $smalt_s )               if ( defined($smalt_s) );
    $self->smalt_y( $smalt_y )               if ( defined($smalt_y) );
    $self->smalt_r( $smalt_r )               if ( defined($smalt_r) );
		$self->smalt_n( $smalt_n )               if ( defined($smalt_n) );
    $self->help($help)                       if ( defined($help) );

	# print usage text if required parameters are not present
	($fastqfile && $ref) or die $self->usage_text;
}

sub run {
    my ($self) = @_;

    if ( defined( $self->help ) ) {
    #if ( scalar( @{ $self->args } ) == 0 ) {
          $self->usage_text;
    }

    my $mapping = Bio::Tradis::Map->new(
        fastqfile => $self->fastqfile,
        reference => $self->reference,
        refname   => $self->refname,
        outfile   => $self->outfile,
        smalt => $self->smalt,
	smalt_k   => $self->smalt_k,
	smalt_s   => $self->smalt_s,
	smalt_y   => $self->smalt_y,
	smalt_r   => $self->smalt_r
    );
    $mapping->index_ref;
    $mapping->do_mapping;
}

sub usage_text {
      print <<USAGE;
Indexes the reference genome and maps the given fastq file. If smalt is to be used set -s or --smalt and 
-sk and -s as options for indexing which are calculated for the length of
the read as follows unless otherwise specified ( --smalt_k & 
--smalt_s options )
Read length    | k  |  s
---------------+----+-----
<70            | 13 |  4
>70 & <100     | 13 |  6
>100           | 20 |  13

Usage: run_mapping -f file.fastq -r ref.fa [options]

Options:
-f        : fastq file to map
-r        : reference in fasta format
-rn       : reference index name (optional. default: ref.index)
-o        : mapped SAM output name (optional. default: mapped.sam)
-k        : minimum seed length for BWA mapping (optional)
--smalt   : smalt to be used as aligner (oprional. default: bwa)
--smalt_k : custom k-mer value for SMALT mapping
--smalt_s : custom step size for SMALT mapping
--smalt_r : custom r value for SMALT mapping
-n        : number of threads to use for mapping and samtools sort (optional. default = 1)

USAGE
      exit;
}

__PACKAGE__->meta->make_immutable;
no Moose;
1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Bio::Tradis::CommandLine::RunMapping - Perform mapping

=head1 VERSION

version 1.4.3

=head1 SYNOPSIS

Takes a reference genome and indexes it.
Maps given fastq files to ref.

=head1 AUTHOR

Carla Cummins <path-help@sanger.ac.uk>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2013 by Wellcome Trust Sanger Institute.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
