package Hash::MD5;

use utf8;
use 5.006;
use strict;
use warnings FATAL => 'all';
use Digest::MD5 qw(md5_hex);
use vars qw($VERSION @EXPORT_OK);

require Exporter;
*import    = \&Exporter::import;
@EXPORT_OK = qw(sum_hash sum_array);

###############################################################################
##############                 PRIVATE SUBS              ######################
###############################################################################
sub _sort_keys {
	my ($hashref) = @_;
	my $serial    = scalar( keys %$hashref );
	my $ret       = '';
	foreach ( sort { $a cmp $b } keys %$hashref ) {
		if 		( ref $hashref->{$_} eq 'HASH' ) { $ret = $ret . unpack( 'H*', $_ ) . _sort_keys( ( defined $hashref->{$_} ? $hashref->{$_} : 'undef' ) ); }
		elsif 	( ref $hashref->{$_} eq 'ARRAY' ){ $ret = $ret . unpack( 'H*', $_ ) . _sort_array(( defined $hashref->{$_} ? $hashref->{$_} : 'undef' ) ); }
		else {
			my $var = (defined $hashref->{$_}) ? $hashref->{$_} : 'undef';
			$serial = $serial . length( $var );
			utf8::encode($_);
			utf8::encode($var);
			$ret = $ret . unpack( 'H*', $_ ) . unpack( 'H*', $var );
		}
	}
	return $ret . $serial;
}

sub _sort_array {
	my ($arrayref) = @_;
	my $serial     = scalar(@$arrayref);
	my $ret        = '';
	foreach (@$arrayref) {
		if 		( ref $_ eq 'HASH' ) { $ret = $ret . _sort_keys($_); }
		elsif 	( ref $_ eq 'ARRAY') { $ret = $ret . _sort_array($_);}
		else {
			$serial = $serial . length($_);
			utf8::encode($_);
			$ret = $ret . unpack( 'H*', $_ );
		}
	}
	return $ret . $serial;
}

=pod

=head1 NAME

Hash::MD5 - MD5 checksum for choosen hashref

=head1 VERSION

Version 0.05

=cut

our $VERSION = '0.05';

=head1 SYNOPSIS

		use Hash::MD5 qw(sum_hash);
		
		...
		my $hashref = {some => 'hashref'};
		print sum_hash( $hashref ), $/;
		...



		use Hash::MD5 qw(sum_array);
		
		...
		my $arrayref = ['some', 'arrayref'];
		print sum_array( $arrayref ), $/;
		...

=head1 SUBROUTINES/METHODS

=head2 sum_hash

=cut

sub sum_hash {
	my ($arg) = @_;
	return md5_hex( _sort_keys($arg) );
}

=head2 sum_array

=cut

sub sum_array {
	my ($arg) = @_;
	return md5_hex( _sort_array($arg) );
}

=head1 AUTHOR

Mario Zieschang, C<< <mario.zieschang at combase.de> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-hash-md5 at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Hash-MD5>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.


=head1 SUPPORT

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Hash-MD5>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Hash-MD5>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Hash-MD5>

=item * Search CPAN

L<http://search.cpan.org/dist/Hash-MD5/>

=back


=head1 MOTIVATION

I need a package which I can grab the uniqueness of a Hashrefs in a md5 sum.
This I will compare to later processing.
My first approach was to use encode_json string.
Unfortunately, the resort was not very consistent.

So I wrote my first cpan packet in order also to learn.


=head1 LICENSE AND COPYRIGHT

Copyright 2015 Mario Zieschang.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1;    # End of Hash::MD5
