package Validate::CodiceFiscale;
use v5.24;
use Carp;
use experimental qw< signatures >;
{ our $VERSION = '0.003002' }

use List::Util 'sum';
use Time::Local 'timegm';
use JSON::PP 'decode_json';
use Exporter 'import';

our @EXPORT_OK = qw< assert_valid_cf decode_cf is_valid_cf validate_cf r >;

# PUBLIC interface

sub assert_valid_cf ($cf, %options) {
   my $errors = validate_cf($cf, all_errors => 0, %options) or return;

   defined(my $ecb = $options{on_error})
     or croak join ', ', $errors->@*;

   my $exception = $ecb->($errors->@*);
   die $exception;    # just as a fallback, $ecb might throw by itself
} ## end sub assert_valid_cf

sub decode_cf ($cf, %options) {
   return _decode_and_validate($cf, %options, all_errors => 1);
}

sub is_valid_cf ($cf, %options) {
   my $error = 0;
   _validate_cf($cf, $options{data}, sub { $error = 1; return 0 });
   return !$error;
}

sub validate_cf ($cf, %options) {
   my $r = _decode_and_validate($cf, %options);
   my $errors = $r->{errors} // [];
   return scalar($errors->@*) ? $errors : undef;
} ## end sub validate_cf

# The following is useful for one-lines:
#
#     $ perl -MValidate::CodiceFiscale=r -er bcadfe88a48h501p
#
sub r (@args) {
   @args = @ARGV unless @args;
   my $i = 0;
   my $n = 0;
   for my $cf (@ARGV) {
      if (my $errors = validate_cf($cf)) {
         say "$i not ok - " . join(', ', $errors->@*);
         ++$n;
      }
      else {
         say "$i ok - $cf";
      }
      ++$i;
   } ## end for my $cf (@ARGV)
   return $n ? 1 : 0;
} ## end sub r

exit r(@ARGV) unless caller();    # modulino

# PRIVATE interface

sub _decode_and_validate ($cf, %options) {
   my $data = $options{data} // undef;

   my $collect_all_errors = $options{all_errors} // 1;
   my @errors;
   my $callback = sub ($msg) {
      push @errors, $msg;
      return $collect_all_errors;
   };

   my $r = _validate_cf($cf, $data, $callback);
   $r->{errors} = \@errors;
   return $r;
}

sub _validate_cf ($cf, $data, $cb) {
   state $consonant = qr{(?imxs:[BCDFGHJKLMNPQRSTVWXYZ])};
   state $vowel     = qr{(?imxs:[AEIOU])};
   state $namish    = qr{(?imxs:
         $consonant  $consonant  $consonant  # includes CCX, CXX, XXX
      |  $consonant  $consonant  $vowel
      |  $consonant  $vowel      $vowel
      |  $consonant  $vowel      X
      |  $vowel      $vowel      $vowel
      |  $vowel      $vowel      X
      |  $vowel      X           X
   )};
   state $digitish = qr{(?imxs:[0-9LMNPQRSTUV])};

   if (length($cf) != 16) {
      $cb->('invalid length');
      return {};
   }

   $cf = uc($cf);
   my %portions = (
      surname  => substr($cf, 0,  3),
      name     => substr($cf, 3,  3),
      date     => substr($cf, 6,  5),
      place    => substr($cf, 11, 4),
      checksum => substr($cf, 15, 1),
   );
   my $retval = {portions => \%portions};

   return $retval
     if $portions{name} !~ m{\A$namish\z}mxs
     && !$cb->('invalid name');

   return $retval
     if $portions{surname} !~ m{\A$namish\z}mxs
     && !$cb->('invalid surname');

   my ($y, $m, $d, $sex) = _expand_date($portions{date}, $data);
   $retval->@{qw< year month day sex >} = ($y, $m, $d, $sex);
   return $retval
     if !_is_valid_cf_date($y, $m, $d)
     && !$cb->('invalid birth date');

   my $date;
   $date = $retval->{date} = sprintf('%04d-%02d-%02d', $y, $m, $d)
     if defined($y);

   if (defined(my $p = _place_name_for($portions{place}, $date))) {
      $retval->{place} = $p;
   }
   else {
      return $retval unless $cb->('invalid birth place');
   }

   my $checksum = _cf_checksum($cf);
   return $retval
     if $checksum ne substr($cf, -1, 1)
     && !$cb->("invalid checksum (should be: $checksum)");

   return $retval unless $data;

   if (defined(my $surname = $data->{surname})) {
      return $retval
        if substr($cf, 0, 3) ne _compact_surname($surname)
        && !$cb->('surname mismatch');
   }
   if (defined(my $name = $data->{name})) {
      return $retval
        if substr($cf, 3, 3) ne _compact_name($name)
        && !$cb->('name mismatch');
   }
   if (defined(my $birthdate = $data->{date})) {
      my ($male, $female) = _compact_birthdates($birthdate);
      my $got = _normalized_birthdate(substr($cf, 6, 5));
      return $retval
           if ($got ne $male)
        && ($got ne $female)
        && !$cb->('birth date mismatch');
   } ## end if (defined(my $birthdate...))
   if (defined(my $sex = $data->{sex})) {
      my $got = _normalized_birthdate(substr($cf, 6, 5));
      my $day = substr($got, -2, 2) + 0;
      return $retval
        if ((lc($sex) eq 'm' && $day > 31)
         || (lc($sex) eq 'f' && $day < 41))
        && !$cb->('sex mismatch');
   } ## end if (defined(my $sex = ...))
   if (defined(my $place = $data->{place})) {
      my $got = $retval->{place} //
         _normalized_birthplace($portions{place});
      return $retval
        if fc($got) ne fc($place)
        && !$cb->('birth place mismatch');
   } ## end if (defined(my $place ...))

   return $retval;
} ## end sub _validate_cf

sub _cf_checksum ($cf) {
   state $odd_checksums = {
      0 => 1,
      1 => 0,
      2 => 5,
      3 => 7,
      4 => 9,
      5 => 13,
      6 => 15,
      7 => 17,
      8 => 19,
      9 => 21,
      A => 1,
      B => 0,
      C => 5,
      D => 7,
      E => 9,
      F => 13,
      G => 15,
      H => 17,
      I => 19,
      J => 21,
      K => 2,
      L => 4,
      M => 18,
      N => 20,
      O => 11,
      P => 3,
      Q => 6,
      R => 8,
      S => 12,
      T => 14,
      U => 16,
      V => 10,
      W => 22,
      X => 25,
      Y => 24,
      Z => 23,
     },
     my $even_checksums = {
      0 => 0,
      1 => 1,
      2 => 2,
      3 => 3,
      4 => 4,
      5 => 5,
      6 => 6,
      7 => 7,
      8 => 8,
      9 => 9,
      A => 0,
      B => 1,
      C => 2,
      D => 3,
      E => 4,
      F => 5,
      G => 6,
      H => 7,
      I => 8,
      J => 9,
      K => 10,
      L => 11,
      M => 12,
      N => 13,
      O => 14,
      P => 15,
      Q => 16,
      R => 17,
      S => 18,
      T => 19,
      U => 20,
      V => 21,
      W => 22,
      X => 23,
      Y => 24,
      Z => 25,
     };
   state $checksums_for = [$odd_checksums, $even_checksums];
   my @chars = split m{}mxs, substr($cf, 0, 15);    # no checksum
   my $sum = sum map { $checksums_for->[$_ % 2]{$chars[$_]} } 0 .. $#chars;
   chr(ord('A') + ($sum % 26));
} ## end sub _cf_checksum

sub _normalized_string ($string, @positions) {
   state $letters   = [qw< L M N P Q R S T U V >];
   state $digit_for = {map { $letters->[$_] => $_ } 0 .. $letters->$#*};
   for my $i (@positions) {
      my $current = substr($string, $i, 1);
      substr($string, $i, 1, $digit_for->{$current})
        if exists $digit_for->{$current};
   }
   return $string;
} ## end sub _normalized_string

sub _normalized_birthplace ($place) { _normalized_string($place, 1 .. 3) }
sub _normalized_birthdate ($date) { _normalized_string($date, 0, 1, 3, 4) }

sub _expand_date ($date, $opts) {
   state $mlf       = [split m{}mxs, 'ABCDEHLMPRST'];
   state $month_for = {map { $mlf->[$_] => $_ } 0 .. $mlf->$#*};

   $date = _normalized_birthdate($date);
   my ($y, $mc, $d) = $date =~ m{\A(\d\d)([ABCDEHLMPRST])(\d\d)\z}mxs
     or return;
   my $m = 1 + $month_for->{$mc};
   $_ += 0 for ($d, $y);
   my $sex = $d > 40 ? 'F' : 'M';
   $d -= 40 if $d > 40;

   # century: the initial digits of a year
   if (defined(my $years_baseline = ($opts // {})->{years_baseline})) {
      $y += $years_baseline;
   }
   else {    # whatever in the last 100 years
      my $this_year = 1900 + (localtime)[5];
      $y += 100 * int($this_year / 100);
      $y -= 100 if $y > $this_year;
   }

   return ($y, $m, $d, $sex);
} ## end sub _expand_date

sub _is_valid_cf_date ($y, $m, $d) {
   return !!(eval { timegm(30, 30, 12, $d, $m - 1, $y); 1 });
}

sub _compact_birthdates ($birthdate) {
   state $month_letter_for = ['', split m{}mxs, 'ABCDEHLMPRST'];
   my ($y, $m, $d) = split m{\D}mxs, $birthdate;
   ($y, $d) = ($d, $y) if $d > 31;
   $y %= 100;
   $m = $month_letter_for->[$m + 0];
   map { sprintf '%02d%s%02d', $y, $m, $_ } ($d, $d + 40);
} ## end sub _compact_birthdates

sub _compact_surname ($surname) {
   my ($cs, $vs) = _consonants_and_vowels($surname);
   my @retval = ($cs->@*, $vs->@*, ('X') x 3);
   return join '', @retval[0 .. 2];
}

sub _compact_name ($name) {
   my ($cs, $vs) = _consonants_and_vowels($name);
   splice $cs->@*, 1, 1 if $cs->@* > 3;
   my @retval = ($cs->@*, $vs->@*, ('X') x 3);
   return join '', @retval[0 .. 2];
} ## end sub _compact_name

sub _consonants_and_vowels ($string) {
   my (@consonants, @vowels);
   for my $char (grep { m{[A-Z]}mxs } split m{}mxs, uc($string)) {
      if   ($char =~ m{[AEIOU]}mxs) { push @vowels,     $char }
      else                          { push @consonants, $char }
   }
   return (\@consonants, \@vowels);
} ## end sub _consonants_and_vowels

sub _places {
   state $retval = do {
      local $/;
      binmode DATA, ':raw';
      (my $json = readline(DATA)) =~ s{\n+}{}gmxs;
      decode_json($json);
   };
}

sub _place_name_for ($place, $birthdate) {
   state $place_for = _places();
   my $record = $place_for->{_normalized_birthplace($place)} or return;
   return "[$record->[-1]{name}]" unless defined($birthdate);
   for my $candidate ($record->@*) {
      next if $birthdate gt $candidate->{end};
      last if $birthdate lt $candidate->{start};
      return $candidate->{name};
   }
   return;
} ## end sub _place_name_for

1;

__DATA__
{"G935":[{"start":"1957-04-07","end":"9999-12-31","name":"POSTA FIBRENO"}],"
G846":[{"start":"1866-11-19","end":"1902-02-06","name":"PONTE DI PIAVE"},{"n
ame":"PONTE DI PIAVE","start":"1902-02-07","end":"1907-05-09"},{"name":"PONT
E DI PIAVE","start":"1907-05-10","end":"9999-12-31"}],"M082":[{"name":"VITER
BO","end":"1927-01-11","start":"1871-01-15"},{"name":"VITERBO","start":"1927
-01-12","end":"1928-05-18"},{"name":"VITERBO","start":"1928-05-19","end":"19
46-11-11"},{"end":"9999-12-31","start":"1946-11-12","name":"VITERBO"}],"B508
":[{"end":"1927-01-11","start":"1863-04-23","name":"CAMPIGLIA CERVO"},{"star
t":"1927-01-12","end":"1992-04-15","name":"CAMPIGLIA CERVO"},{"start":"1992-
04-16","end":"2015-12-31","name":"CAMPIGLIA CERVO"}],"A402":[{"start":"1861-
03-17","end":"9999-12-31","name":"ARIELLI"}],"I608":[{"end":"9999-12-31","st
art":"1861-03-17","name":"SENIGALLIA"}],"B255":[{"start":"1863-07-13","end":
"9999-12-31","name":"BUGLIO IN MONTE"}],"E780":[{"name":"MACCHIA VALFORTORE"
,"end":"1928-06-30","start":"1861-03-17"},{"name":"MACCHIA VALFORTORE","star
t":"1946-04-09","end":"1970-03-02"},{"name":"MACCHIA VALFORTORE","start":"19
70-03-03","end":"9999-12-31"}],"G831":[{"name":"PONTEBBA","start":"1866-11-1
9","end":"1924-08-29"},{"end":"1928-05-18","start":"1924-08-30","name":"PONT
EBBA"},{"end":"1968-04-05","start":"1928-05-19","name":"PONTEBBA"},{"name":"
PONTEBBA","start":"1968-04-06","end":"9999-12-31"}],"F153":[{"start":"1861-0
3-17","end":"9999-12-31","name":"MESE"}],"L292":[{"start":"1863-03-30","end"
:"1928-10-15","name":"TORRICELLA VERZATE"},{"name":"TORRICELLA VERZATE","sta
rt":"1946-12-19","end":"9999-12-31"}],"L321":[{"name":"TRAMATZA","start":"18
61-03-17","end":"1928-05-12"},{"end":"1974-08-19","start":"1951-01-12","name
":"TRAMATZA"},{"start":"1974-08-20","end":"9999-12-31","name":"TRAMATZA"}],"
E495":[{"end":"1927-01-11","start":"1861-03-17","name":"LAVENO"},{"start":"1
927-01-12","end":"1928-01-27","name":"LAVENO"}],"F942":[{"end":"9999-12-31",
"start":"1861-03-17","name":"NOTARESCO"}],"C274":[{"start":"1861-03-17","end
":"1928-12-31","name":"CASTELSPINA"},{"end":"9999-12-31","start":"1954-01-24
","name":"CASTELSPINA"}],"L899":[{"name":"VIGONOVO","end":"9999-12-31","star
t":"1866-11-19"}],"A024":[{"name":"ACERRA","start":"1861-03-17","end":"1927-
01-11"},{"end":"9999-12-31","start":"1927-01-12","name":"ACERRA"}],"L382":[{
"end":"1968-04-05","start":"1866-11-19","name":"TREPPO GRANDE"},{"name":"TRE
PPO GRANDE","start":"1968-04-06","end":"9999-12-31"}],"B724":[{"name":"CARAP
ELLE","start":"1958-01-14","end":"9999-12-31"}],"B801":[{"name":"CAROBBIO DE
GLI ANGELI","end":"9999-12-31","start":"1928-02-21"}],"D641":[{"name":"FOBEL
LO","end":"1927-01-11","start":"1861-03-17"},{"name":"FOBELLO","end":"9999-1
2-31","start":"1927-01-12"}],"A230":[{"start":"1862-12-29","end":"9999-12-31
","name":"ALTAVILLA SILENTINA"}],"C514":[{"start":"1861-03-17","end":"1928-0
3-23","name":"CERIGNOLA"},{"name":"CERIGNOLA","end":"9999-12-31","start":"19
28-03-24"}],"B075":[{"name":"BOSCONERO","start":"1861-03-17","end":"9999-12-
31"}],"E196":[{"start":"1861-03-17","end":"9999-12-31","name":"GROPPARELLO"}
],"H977":[{"start":"1862-11-07","end":"9999-12-31","name":"SAN MANGO PIEMONT
E"}],"B905":[{"end":"1929-04-12","start":"1863-02-06","name":"CASALNUOVO DI 
NAPOLI"},{"start":"1929-04-13","end":"9999-12-31","name":"CASALNUOVO DI NAPO
LI"}],"B662":[{"start":"1867-07-22","end":"9999-12-31","name":"PONTE NELLE A
LPI"}],"D403":[{"name":"EMPOLI","end":"9999-12-31","start":"1861-03-17"}],"C
916":[{"name":"COMASINE","start":"1920-10-16","end":"1923-02-13"},{"name":"C
OMASINE","start":"1923-02-14","end":"1928-08-11"}],"C619":[{"name":"CHIAROMO
NTE","start":"1861-03-17","end":"1928-05-18"},{"end":"1947-01-09","start":"1
928-05-19","name":"CHIAROMONTE"},{"end":"9999-12-31","start":"1947-01-10","n
ame":"CHIAROMONTE"}],"H007":[{"end":"9999-12-31","start":"1863-05-06","name"
:"PRATOLA PELIGNA"}],"M021":[{"name":"VILLA SANTA LUCIA DEGLI ABRUZZI","end"
:"1951-02-01","start":"1910-07-26"},{"start":"1951-02-02","end":"9999-12-31"
,"name":"VILLA SANTA LUCIA DEGLI ABRUZZI"}],"D939":[{"name":"GAUNA","start":
"1861-03-17","end":"1927-01-11"},{"name":"GAUNA","start":"1927-01-12","end":
"1929-05-14"}],"I729":[{"start":"1920-10-16","end":"1923-02-13","name":"SILA
NDRO"},{"start":"1923-02-14","end":"1927-01-11","name":"SILANDRO"},{"name":"
SILANDRO","end":"1928-05-11","start":"1927-01-12"},{"start":"1928-05-12","en
d":"9999-12-31","name":"SILANDRO"}],"G887":[{"end":"9999-12-31","start":"187
1-10-16","name":"STELLA CILENTO"}],"F460":[{"name":"MONTE CAVALLO","end":"99
99-12-31","start":"1863-01-26"}],"D636":[{"end":"9999-12-31","start":"1868-0
9-15","name":"FLORIDIA"}],"G538":[{"end":"1945-09-21","start":"1929-02-10","
name":"PIAGGINE"},{"start":"1945-09-22","end":"9999-12-31","name":"PIAGGINE"
}],"H434":[{"start":"1861-03-17","end":"9999-12-31","name":"ROCCARASO"}],"I5
62":[{"end":"1968-04-05","start":"1866-11-19","name":"SEDEGLIANO"},{"name":"
SEDEGLIANO","end":"2014-02-03","start":"1968-04-06"},{"name":"SEDEGLIANO","e
nd":"9999-12-31","start":"2014-02-04"}],"A169":[{"start":"1861-03-17","end":
"1928-06-28","name":"ALBOGNO"}],"A341":[{"start":"1936-04-25","end":"1945-04
-08","name":"APRILIA"},{"name":"APRILIA","end":"9999-12-31","start":"1945-04
-09"}],"B124":[{"end":"1927-12-28","start":"1861-03-17","name":"BRAONE"},{"e
nd":"9999-12-31","start":"1949-08-27","name":"BRAONE"}],"A888":[{"end":"1873
-04-09","start":"1861-03-17","name":"BISIO"}],"Z301":[{"name":"Algeria","sta
rt":"0000-01-01","end":"9999-12-31"}],"E573":[{"name":"LICATA","start":"1861
-03-17","end":"9999-12-31"}],"A537":[{"end":"1923-02-13","start":"1920-10-16
","name":"BADIA"},{"name":"BADIA","end":"1927-01-11","start":"1923-02-14"},{
"name":"BADIA","end":"9999-12-31","start":"1927-01-12"}],"D098":[{"name":"CO
SSIRANO","start":"1861-03-17","end":"1929-01-23"}],"A476":[{"end":"9999-12-3
1","start":"1880-07-12","name":"ASSO"}],"L044":[{"end":"1923-02-07","start":
"1921-01-05","name":"TAPOGLIANO"},{"name":"TAPOGLIANO","end":"1928-11-27","s
tart":"1923-02-08"},{"start":"1946-12-24","end":"1968-04-05","name":"TAPOGLI
ANO"},{"name":"TAPOGLIANO","start":"1968-04-06","end":"2006-02-08"},{"start"
:"2006-02-09","end":"2008-12-31","name":"TAPOGLIANO"}],"F046":[{"end":"1927-
01-11","start":"1861-03-17","name":"MASSIMINO"},{"start":"1927-01-12","end":
"9999-12-31","name":"MASSIMINO"}],"E607":[{"name":"LIPOMO","start":"1861-03-
17","end":"9999-12-31"}],"E796":[{"start":"1920-10-16","end":"1923-02-13","n
ame":"MADRANO"},{"end":"1929-01-24","start":"1923-02-14","name":"MADRANO"}],
"H738":[{"name":"SAMASSI","start":"1861-03-17","end":"1974-08-19"},{"end":"2
005-12-31","start":"1974-08-20","name":"SAMASSI"},{"end":"2016-04-27","start
":"2006-01-01","name":"SAMASSI"},{"name":"SAMASSI","start":"2016-04-28","end
":"9999-12-31"}],"I815":[{"name":"SOMAGLIA","start":"1882-03-03","end":"1992
-04-15"},{"name":"SOMAGLIA","start":"1992-04-16","end":"1997-07-25"},{"name"
:"SOMAGLIA","start":"1997-07-26","end":"9999-12-31"}],"A336":[{"name":"APRIA
NO","start":"1921-01-05","end":"1923-02-07"},{"name":"APRIANO","start":"1923
-02-08","end":"1924-03-08"},{"name":"APRIANO","start":"1924-03-09","end":"19
31-12-21"}],"I129":[{"name":"SAN POTITO ULTRA","end":"9999-12-31","start":"1
863-02-06"}],"A522":[{"name":"AVOLA","start":"1861-03-17","end":"9999-12-31"
}],"B428":[{"end":"1937-07-16","start":"1861-03-17","name":"CALTAGIRONE"},{"
start":"1937-07-17","end":"1976-05-26","name":"CALTAGIRONE"},{"end":"9999-12
-31","start":"1976-05-27","name":"CALTAGIRONE"}],"F031":[{"end":"1927-11-08"
,"start":"1861-03-17","name":"MASSAMA"}],"A769":[{"end":"9999-12-31","start"
:"1863-02-23","name":"BELVEDERE OSTRENSE"}],"I911":[{"start":"1861-03-17","e
nd":"1928-06-18","name":"SPINETO"},{"start":"1928-06-19","end":"1946-12-16",
"name":"SPINETO SCRIVIA"},{"name":"SPINETO SCRIVIA","start":"1946-12-17","en
d":"9999-12-31"}],"A581":[{"name":"BAJARDO","start":"1861-03-17","end":"9999
-12-31"}],"D523":[{"end":"9999-12-31","start":"1861-03-17","name":"FEISOGLIO
"}],"B069":[{"end":"9999-12-31","start":"1866-11-19","name":"BOSARO"}],"E180
":[{"end":"9999-12-31","start":"1861-03-17","name":"GRIMALDI"}],"M344":[{"en
d":"9999-12-31","start":"2015-01-01","name":"PREDAIA"}],"F753":[{"start":"18
61-03-17","end":"1924-01-13","name":"MORTANO"}],"L683":[{"end":"1928-09-22",
"start":"1861-03-17","name":"VARGO"}],"A175":[{"name":"ALBUZZANO","end":"999
9-12-31","start":"1872-11-01"}],"E213":[{"name":"GROTTOLE","start":"1861-03-
17","end":"1927-01-11"},{"name":"GROTTOLE","start":"1927-01-12","end":"9999-
12-31"}],"I064":[{"name":"SAN NOVO","start":"1861-03-17","end":"1870-08-01"}
],"C552":[{"name":"CERVETERI","start":"1871-01-15","end":"1926-11-29"},{"nam
e":"CERVETERI","end":"1935-03-21","start":"1926-11-30"},{"end":"1949-07-08",
"start":"1935-03-22","name":"CERVETERI"},{"end":"1970-05-26","start":"1949-0
7-09","name":"CERVETERI"},{"end":"1990-03-24","start":"1970-05-27","name":"C
ERVETERI"},{"end":"9999-12-31","start":"1990-03-25","name":"CERVETERI"}],"H9
10":[{"end":"1928-04-16","start":"1861-03-17","name":"SAN GIOVANNI BIANCO"},
{"name":"SAN GIOVANNI BIANCO","start":"1928-04-17","end":"9999-12-31"}],"F08
7":[{"name":"MEDOLLA","end":"9999-12-31","start":"1861-03-17"}],"A376":[{"na
me":"ARCORE","start":"1869-10-11","end":"1963-02-19"},{"name":"ARCORE","star
t":"1963-02-20","end":"2009-06-29"},{"name":"ARCORE","end":"9999-12-31","sta
rt":"2009-06-30"}],"L501":[{"name":"URBISAGLIA","start":"1861-03-17","end":"
1936-10-29"},{"start":"1946-10-27","end":"9999-12-31","name":"URBISAGLIA"},{
"end":"1946-10-26","start":"1936-10-30","name":"URBISAGLIA BONSERVIZI"}],"B1
50":[{"start":"1861-03-17","end":"1927-01-11","name":"BRENTA"},{"end":"1928-
01-17","start":"1927-01-12","name":"BRENTA"},{"name":"BRENTA","start":"1953-
09-01","end":"9999-12-31"}],"Z401":[{"name":"Canada","end":"9999-12-31","sta
rt":"0000-01-01"}],"F071":[{"start":"1861-03-17","end":"1923-06-28","name":"
MAZZOLENI E FALGHERA"},{"name":"MAZZOLENI E FALGHERA","start":"1923-06-29","
end":"1927-08-03"}],"A441":[{"end":"1902-01-16","start":"1872-06-10","name":
"ARSAGO"},{"name":"ARSAGO","start":"1902-01-17","end":"1927-01-11"},{"start"
:"1927-01-12","end":"1948-02-05","name":"ARSAGO"},{"name":"ARSAGO SEPRIO","s
tart":"1948-02-06","end":"9999-12-31"}],"L264":[{"end":"1928-06-27","start":
"1861-03-17","name":"TORRE DE' RATTI"}],"F266":[{"name":"MOGGIO UDINESE","en
d":"1968-04-05","start":"1866-11-19"},{"end":"9999-12-31","start":"1968-04-0
6","name":"MOGGIO UDINESE"}],"F901":[{"end":"1929-05-27","start":"1863-04-10
","name":"NIZZA DI SICILIA"},{"end":"9999-12-31","start":"1948-12-25","name"
:"NIZZA DI SICILIA"}],"E533":[{"name":"LENZIMA","start":"1920-10-16","end":"
1923-02-13"},{"start":"1923-02-14","end":"1929-02-13","name":"LENZIMA"}],"A5
77":[{"name":"BAGNOREA","end":"1922-07-25","start":"1871-01-15"},{"name":"BA
GNOREGIO","end":"1927-01-11","start":"1922-07-26"},{"name":"BAGNOREGIO","end
":"1927-12-30","start":"1927-01-12"},{"start":"1927-12-31","end":"9999-12-31
","name":"BAGNOREGIO"}],"A603":[{"name":"BALSORANO","end":"9999-12-31","star
t":"1861-03-17"}],"G969":[{"name":"PRADAMANO","end":"1960-03-24","start":"18
66-11-19"},{"start":"1960-03-25","end":"1968-04-05","name":"PRADAMANO"},{"en
d":"9999-12-31","start":"1968-04-06","name":"PRADAMANO"}],"G666":[{"start":"
1861-03-17","end":"1927-01-11","name":"PILA"},{"end":"9999-12-31","start":"1
927-01-12","name":"PILA"}],"H778":[{"name":"SAN BIAGIO PLATANI","end":"9999-
12-31","start":"1864-06-23"}],"D568":[{"name":"FICAROLO","end":"9999-12-31",
"start":"1866-11-19"}],"D856":[{"end":"1928-06-13","start":"1861-03-17","nam
e":"GAIOLA"},{"name":"GAIOLA","start":"1946-07-03","end":"9999-12-31"}],"B32
8":[{"name":"CADELBOSCO DI SOPRA","start":"1866-01-01","end":"9999-12-31"}],
"A436":[{"end":"2007-07-12","start":"1863-01-26","name":"ARQUATA SCRIVIA"},{
"start":"2007-07-13","end":"9999-12-31","name":"ARQUATA SCRIVIA"}],"A217":[{
"name":"ALMENNO SAN SALVATORE","end":"9999-12-31","start":"1885-01-15"}],"I5
94":[{"end":"9999-12-31","start":"1893-04-10","name":"SELVA DI PROGNO"}],"A1
35":[{"start":"1863-07-13","end":"1978-02-11","name":"ALBAREDO PER SAN MARCO
"},{"name":"ALBAREDO PER SAN MARCO","start":"1978-02-12","end":"2006-08-08"}
,{"name":"ALBAREDO PER SAN MARCO","start":"2006-08-09","end":"9999-12-31"}],
"B249":[{"name":"BUDRIO","start":"1869-01-30","end":"9999-12-31"}],"B842":[{
"start":"1861-03-17","end":"9999-12-31","name":"CARSOLI"}],"F251":[{"start":
"1861-03-17","end":"9999-12-31","name":"MISTRETTA"}],"B694":[{"name":"CAPREZ
ZO","start":"1861-03-17","end":"1992-05-22"},{"name":"CAPREZZO","end":"9999-
12-31","start":"1992-05-23"}],"E711":[{"end":"1927-01-11","start":"1861-03-1
7","name":"LOZZOLO"},{"name":"LOZZOLO","start":"1927-01-12","end":"9999-12-3
1"}],"D602":[{"end":"1927-01-11","start":"1869-07-31","name":"FINALE PIA"},{
"start":"1927-01-12","end":"1927-01-12","name":"FINALE PIA"}],"B463":[{"end"
:"1992-05-22","start":"1861-03-17","name":"CAMBIASCA"},{"start":"1992-05-23"
,"end":"9999-12-31","name":"CAMBIASCA"}],"D965":[{"name":"GENGA","end":"1937
-05-28","start":"1861-03-17"},{"start":"1937-05-29","end":"9999-12-31","name
":"GENGA"}],"L808":[{"name":"VESPOLATE","start":"1861-03-17","end":"9999-12-
31"}],"I244":[{"name":"SANTA MARIA IMBARO","end":"9999-12-31","start":"1861-
03-17"}],"B599":[{"start":"1861-03-17","end":"1936-10-21","name":"CANEVINO"}
,{"name":"CANEVINO","start":"1947-11-26","end":"2018-12-31"}],"M148":[{"end"
:"1928-05-28","start":"1861-03-17","name":"ZAPPELLO"}],"H026":[{"start":"186
1-03-17","end":"1901-01-25","name":"PREGNANA"},{"end":"9999-12-31","start":"
1901-01-26","name":"PREGNANA MILANESE"}],"I699":[{"name":"SETTIMO SAN PIETRO
","end":"1928-06-30","start":"1862-09-29"},{"name":"SETTIMO SAN PIETRO","end
":"1974-08-19","start":"1946-04-02"},{"end":"9999-12-31","start":"1974-08-20
","name":"SETTIMO SAN PIETRO"}],"G651":[{"start":"1868-07-12","end":"9999-12
-31","name":"PIEVE SAN GIACOMO"}],"D861":[{"start":"1863-05-06","end":"9999-
12-31","name":"GALATI MAMERTINO"}],"I996":[{"name":"SUELLO","end":"1927-11-0
8","start":"1861-03-17"},{"name":"SUELLO","end":"1992-04-15","start":"1955-0
8-19"},{"end":"9999-12-31","start":"1992-04-16","name":"SUELLO"}],"A510":[{"
end":"1928-11-30","start":"1861-03-17","name":"AVENONE"}],"C895":[{"name":"C
OLOGNO MONZESE","start":"1867-01-14","end":"9999-12-31"}],"E929":[{"end":"19
27-01-11","start":"1861-03-17","name":"MARCHIROLO"},{"name":"MARCHIROLO","en
d":"1928-10-07","start":"1927-01-12"},{"name":"MARCHIROLO","end":"1978-02-11
","start":"1955-08-19"},{"end":"9999-12-31","start":"1978-02-12","name":"MAR
CHIROLO"}],"G871":[{"start":"1861-03-17","end":"1927-01-11","name":"PONZA"},
{"name":"PONZA","start":"1927-01-12","end":"1934-10-25"},{"end":"1934-12-17"
,"start":"1934-10-26","name":"PONZA"},{"start":"1934-12-18","end":"1935-07-3
0","name":"PONZA"},{"end":"1937-05-19","start":"1935-07-31","name":"PONZA"},
{"name":"PONZA","end":"1945-04-08","start":"1937-05-20"},{"end":"9999-12-31"
,"start":"1945-04-09","name":"PONZA"}],"F618":[{"start":"1862-12-29","end":"
9999-12-31","name":"MONTE SAN GIACOMO"}],"A735":[{"end":"9999-12-31","start"
:"1861-03-17","name":"BEINETTE"}],"E626":[{"name":"LIVORNO FERRARIS","end":"
1927-01-11","start":"1924-07-21"},{"name":"LIVORNO FERRARIS","end":"9999-12-
31","start":"1927-01-12"},{"name":"LIVORNO PIEMONTE","start":"1863-04-23","e
nd":"1924-07-20"}],"G005":[{"name":"OFFIDA","end":"9999-12-31","start":"1861
-03-17"}],"A302":[{"end":"9999-12-31","start":"1867-08-05","name":"ANNONE VE
NETO"}],"E111":[{"name":"GORZEGNO","end":"9999-12-31","start":"1861-03-17"}]
,"H885":[{"end":"9999-12-31","start":"1863-03-30","name":"SAN GIORGIO DI LOM
ELLINA"}],"Z342":[{"name":"Seychelles","end":"9999-12-31","start":"0000-01-0
1"}],"G218":[{"name":"PADERNO D'ADDA","end":"1928-07-28","start":"1863-02-23
"},{"end":"1992-04-15","start":"1947-10-02","name":"PADERNO D'ADDA"},{"end":
"9999-12-31","start":"1992-04-16","name":"PADERNO D'ADDA"}],"C991":[{"end":"
1968-04-05","start":"1866-11-19","name":"CORDENONS"},{"start":"1968-04-06","
end":"9999-12-31","name":"CORDENONS"}],"I883":[{"end":"1879-08-28","start":"
1861-03-17","name":"SPADAFORA SAN PIETRO"}],"A050":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"ACQUAVIVA COLLECROCE"}],"G975":[{"name":"PRAIA A MA
RE","end":"1937-07-17","start":"1928-05-12"},{"name":"PRAIA A MARE","end":"9
999-12-31","start":"1937-07-18"}],"L421":[{"start":"1866-11-19","end":"1968-
04-05","name":"TRICESIMO"},{"end":"9999-12-31","start":"1968-04-06","name":"
TRICESIMO"}],"E395":[{"name":"LACCHIARELLA","start":"1869-03-22","end":"1995
-05-04"},{"name":"LACCHIARELLA","end":"1998-04-18","start":"1995-05-05"},{"n
ame":"LACCHIARELLA","start":"1998-04-19","end":"9999-12-31"}],"H981":[{"end"
:"1928-07-27","start":"1862-12-12","name":"SAN MARCO ARGENTANO"},{"name":"SA
N MARCO ARGENTANO","end":"1937-07-17","start":"1928-07-28"},{"name":"SAN MAR
CO ARGENTANO","start":"1937-07-18","end":"9999-12-31"}],"D676":[{"end":"1927
-01-11","start":"1862-12-29","name":"FONTANETTO DA PO"},{"end":"1955-04-28",
"start":"1927-01-12","name":"FONTANETTO DA PO"},{"end":"9999-12-31","start":
"1955-04-29","name":"FONTANETTO PO"}],"G578":[{"end":"1929-04-11","start":"1
866-11-19","name":"PIAVON"}],"L615":[{"start":"1920-10-16","end":"1923-02-13
","name":"VALLES"},{"start":"1923-02-14","end":"1927-01-11","name":"VALLES"}
,{"start":"1927-01-12","end":"1929-05-30","name":"VALLES"}],"B750":[{"name":
"CARCIATO","end":"1923-02-13","start":"1920-10-16"},{"name":"CARCIATO","star
t":"1923-02-14","end":"1928-09-05"}],"H922":[{"name":"SAN GIOVANNI LA PUNTA"
,"end":"9999-12-31","start":"1861-03-17"}],"L482":[{"end":"9999-12-31","star
t":"1861-03-17","name":"UCRIA"}],"I092":[{"name":"SAN PIETRO AL NATISONE","e
nd":"1968-04-05","start":"1869-01-22"},{"end":"9999-12-31","start":"1968-04-
06","name":"SAN PIETRO AL NATISONE"}],"E285":[{"end":"9999-12-31","start":"1
861-03-17","name":"ILLORAI"}],"H474":[{"name":"RODELLO","start":"1861-03-17"
,"end":"9999-12-31"}],"A270":[{"start":"1861-03-17","end":"9999-12-31","name
":"ANCARANO"}],"C020":[{"name":"CASSINA RIZZARDI","end":"9999-12-31","start"
:"1861-03-17"}],"H334":[{"name":"RIVALTA BORMIDA","end":"9999-12-31","start"
:"1863-02-16"}],"B035":[{"name":"BORGO SAN GIACOMO","start":"1863-02-23","en
d":"1927-12-31"},{"name":"BORGO SAN GIACOMO","end":"9999-12-31","start":"192
8-01-01"}],"F360":[{"end":"1928-06-18","start":"1866-11-19","name":"MONFUMO"
},{"name":"MONFUMO","end":"9999-12-31","start":"1945-05-01"}],"H178":[{"name
":"RANZIANO","start":"1921-01-05","end":"1923-02-07"},{"end":"1927-01-11","s
tart":"1923-02-08","name":"RANZIANO"},{"start":"1927-01-12","end":"1928-04-1
4","name":"RANZIANO"},{"end":"1947-09-15","start":"1928-04-15","name":"RANZI
ANO"}],"H937":[{"start":"1861-03-17","end":"9999-12-31","name":"SAN GODENZO"
}],"D009":[{"start":"1861-03-17","end":"2008-05-23","name":"CORLEONE"},{"nam
e":"CORLEONE","start":"2008-05-24","end":"9999-12-31"}],"D303":[{"name":"DIN
AMI","end":"1992-04-15","start":"1861-03-17"},{"name":"DINAMI","start":"1992
-04-16","end":"9999-12-31"}],"I980":[{"end":"1927-01-11","start":"1861-03-17
","name":"STRONA"},{"name":"STRONA","end":"1992-04-15","start":"1927-01-12"}
,{"end":"9999-12-31","start":"1992-04-16","name":"STRONA"}],"G920":[{"name":
"PORTO SAN GIORGIO","end":"2009-07-11","start":"1877-12-03"},{"name":"PORTO 
SAN GIORGIO","end":"9999-12-31","start":"2009-07-12"}],"I279":[{"name":"SANT
'ANGELO ALL'ESCA","start":"1861-03-17","end":"9999-12-31"}],"Z213":[{"start"
:"0000-01-01","end":"9999-12-31","name":"Corea del Sud"}],"D113":[{"end":"19
27-01-11","start":"1870-09-01","name":"COSTANZANA"},{"name":"COSTANZANA","en
d":"9999-12-31","start":"1927-01-12"}],"C255":[{"name":"CASTEL ROZZONE","sta
rt":"1861-03-17","end":"9999-12-31"}],"M339":[{"name":"MACCAGNO CON PINO E V
EDDASCA","start":"2014-02-04","end":"9999-12-31"}],"M033":[{"end":"1928-02-2
0","start":"1861-03-17","name":"VILLAVIANI"}],"C508":[{"name":"CERETTO LOMEL
LINA","end":"9999-12-31","start":"1955-11-13"},{"start":"1863-03-30","end":"
1955-11-12","name":"CERRETTO LOMELLINO"}],"G457":[{"name":"PERLETTO","start"
:"1861-03-17","end":"9999-12-31"}],"A686":[{"end":"1928-11-20","start":"1861
-03-17","name":"BARZAN\u00d2"},{"end":"1953-12-05","start":"1928-11-21","nam
e":"BARZAN\u00d2"},{"start":"1953-12-06","end":"1992-04-15","name":"BARZAN\u
00d2"},{"end":"9999-12-31","start":"1992-04-16","name":"BARZAN\u00d2"}],"I87
2":[{"name":"SOVERATO","start":"1861-03-17","end":"9999-12-31"}],"A754":[{"n
ame":"BELLOMBRA","start":"1924-01-14","end":"1929-01-13"}],"A989":[{"name":"
BORGARELLO","end":"1929-03-20","start":"1861-03-17"},{"start":"1958-08-28","
end":"9999-12-31","name":"BORGARELLO"}],"B274":[{"name":"BURCEI","start":"18
61-03-17","end":"1974-08-19"},{"name":"BURCEI","end":"2016-04-27","start":"1
974-08-20"},{"end":"9999-12-31","start":"2016-04-28","name":"BURCEI"}],"D280
":[{"name":"DERVIO","end":"1928-01-25","start":"1861-03-17"},{"start":"1928-
01-26","end":"1950-04-04","name":"DERVIO"},{"start":"1950-04-05","end":"1992
-04-15","name":"DERVIO"},{"name":"DERVIO","start":"1992-04-16","end":"9999-1
2-31"}],"C075":[{"name":"CASTEL D'AIANO","end":"9999-12-31","start":"1861-03
-17"}],"A225":[{"start":"1861-03-17","end":"9999-12-31","name":"ALTAMURA"}],
"I059":[{"end":"1928-08-23","start":"1908-04-30","name":"SANNICOLA"},{"start
":"1928-08-24","end":"9999-12-31","name":"SANNICOLA"}],"B514":[{"start":"192
0-10-16","end":"1923-02-13","name":"CAMPITELLO"},{"end":"1926-02-07","start"
:"1923-02-14","name":"CAMPITELLO"},{"name":"CAMPITELLO DI FASSA","end":"1999
-12-07","start":"1956-11-04"},{"name":"CAMPITELLO DI FASSA","start":"1999-12
-08","end":"9999-12-31"}],"I353":[{"name":"SANTORSO","start":"1866-11-19","e
nd":"9999-12-31"}],"D390":[{"start":"1861-03-17","end":"1929-05-20","name":"
EBOLI"},{"start":"1929-05-21","end":"9999-12-31","name":"EBOLI"}],"I614":[{"
end":"9999-12-31","start":"1861-03-17","name":"SENNORI"}],"G823":[{"name":"P
ONSO","end":"9999-12-31","start":"1866-11-19"}],"L333":[{"name":"TRAREGO","e
nd":"1928-09-08","start":"1861-03-17"},{"name":"TRAREGO","end":"1929-10-28",
"start":"1928-09-09"},{"name":"TRAREGO VIGGIONA","end":"1992-05-22","start":
"1929-10-29"},{"name":"TRAREGO VIGGIONA","end":"9999-12-31","start":"1992-05
-23"}],"C801":[{"name":"COASSOLO TORINESE","start":"1863-08-10","end":"9999-
12-31"}],"L039":[{"name":"TALMASSONS","start":"1866-11-19","end":"1968-04-05
"},{"name":"TALMASSONS","end":"9999-12-31","start":"1968-04-06"}],"C724":[{"
name":"CIRIMIDO","start":"1861-03-17","end":"9999-12-31"}],"I519":[{"name":"
SCENA","start":"1920-10-16","end":"1923-02-13"},{"name":"SCENA","start":"192
3-02-14","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31","name"
:"SCENA"}],"A112":[{"end":"1927-11-07","start":"1861-03-17","name":"AIRUNO"}
,{"end":"1992-04-15","start":"1927-11-08","name":"AIRUNO"},{"name":"AIRUNO",
"end":"9999-12-31","start":"1992-04-16"}],"B054":[{"name":"BORNO","end":"196
3-02-12","start":"1861-03-17"},{"name":"BORNO","end":"9999-12-31","start":"1
963-02-13"}],"L473":[{"start":"1861-03-17","end":"1928-06-14","name":"TURRI"
},{"name":"TURRI","end":"1974-08-19","start":"1946-03-08"},{"name":"TURRI","
start":"1974-08-20","end":"2005-12-31"},{"name":"TURRI","end":"2016-04-27","
start":"2006-01-01"},{"end":"9999-12-31","start":"2016-04-28","name":"TURRI"
}],"E301":[{"name":"INGRIA","start":"1861-03-17","end":"1927-01-11"},{"start
":"1927-01-12","end":"1945-10-04","name":"INGRIA"},{"name":"INGRIA","start":
"1945-10-05","end":"1957-03-22"},{"name":"INGRIA","start":"1957-03-23","end"
:"9999-12-31"}],"A647":[{"end":"1927-01-11","start":"1861-03-17","name":"BAR
DINETO"},{"name":"BARDINETO","start":"1927-01-12","end":"9999-12-31"}],"B619
":[{"name":"CANOSA DI PUGLIA","start":"1863-01-19","end":"2009-07-26"},{"nam
e":"CANOSA DI PUGLIA","start":"2009-07-27","end":"9999-12-31"}],"C662":[{"na
me":"CHIUSI","start":"1861-03-17","end":"9999-12-31"}],"H355":[{"end":"9999-
12-31","start":"1871-08-01","name":"RIVOLI"}],"B916":[{"start":"1861-03-17",
"end":"1927-01-11","name":"CASALUCE"},{"name":"CASALUCE","end":"1929-04-23",
"start":"1927-01-12"},{"start":"1946-12-21","end":"9999-12-31","name":"CASAL
UCE"}],"C905":[{"name":"COLOSIMI","end":"9999-12-31","start":"1861-03-17"}],
"M151":[{"end":"1956-04-08","start":"1929-02-24","name":"ZAVATTARELLO VALVER
DE"}],"F598":[{"start":"1861-03-17","end":"9999-12-31","name":"MONTERIGGIONI
"}],"D546":[{"start":"1872-04-26","end":"9999-12-31","name":"PIANOPOLI"}],"G
648":[{"end":"1923-11-23","start":"1861-03-17","name":"PIEVE FOSCIANA"},{"st
art":"1923-11-24","end":"9999-12-31","name":"PIEVE FOSCIANA"}],"I265":[{"nam
e":"SANT'ANDREA","start":"1907-06-21","end":"1927-01-11"},{"start":"1927-01-
12","end":"1962-02-07","name":"SANT'ANDREA"},{"end":"9999-12-31","start":"19
62-02-08","name":"SANT'ANDREA DEL GARIGLIANO"}],"A690":[{"name":"BASCAP\u00c
8","start":"1872-11-01","end":"9999-12-31"}],"C124":[{"end":"1924-03-08","st
art":"1923-04-27","name":"CASTEL IABLANIZZA"},{"end":"1927-11-03","start":"1
924-03-09","name":"CASTEL IABLANIZZA"},{"start":"1927-11-04","end":"1947-09-
15","name":"CASTEL IABLANIZZA"},{"name":"IABLANIZZA","end":"1923-02-07","sta
rt":"1921-01-05"},{"name":"IABLANIZZA","start":"1923-02-08","end":"1923-04-2
6"}],"A712":[{"name":"BASTIDA PANCARANA","end":"2006-05-11","start":"1885-04
-17"},{"start":"2006-05-12","end":"9999-12-31","name":"BASTIDA PANCARANA"}],
"L998":[{"end":"1974-08-19","start":"1861-03-17","name":"VILLAPUTZU"},{"name
":"VILLAPUTZU","start":"1974-08-20","end":"1986-04-02"},{"name":"VILLAPUTZU"
,"end":"2016-04-27","start":"1986-04-03"},{"name":"VILLAPUTZU","end":"9999-1
2-31","start":"2016-04-28"}],"D944":[{"name":"GAVI","start":"1861-03-17","en
d":"2007-07-12"},{"start":"2007-07-13","end":"9999-12-31","name":"GAVI"}],"H
263":[{"name":"RHEMES SAINT GEORGES","end":"1927-01-11","start":"1861-03-17"
},{"start":"1927-01-12","end":"1928-05-12","name":"RHEMES SAINT GEORGES"},{"
name":"RHEMES SAINT GEORGES","end":"1977-01-01","start":"1946-03-10"},{"name
":"RH\u00caMES-SAINT-GEORGES","end":"9999-12-31","start":"1977-01-02"}],"C42
8":[{"name":"CELENZA SUL TRIGNO","end":"9999-12-31","start":"1864-02-10"}],"
D531":[{"name":"FENEGR\u00d2","start":"1861-03-17","end":"9999-12-31"}],"H44
9":[{"name":"ROCCASTRADA","start":"1861-03-17","end":"1905-06-29"},{"name":"
ROCCASTRADA","start":"1905-06-30","end":"9999-12-31"}],"D296":[{"name":"DIAN
O CASTELLO","start":"1871-05-01","end":"1923-12-21"},{"end":"9999-12-31","st
art":"1925-08-22","name":"DIANO CASTELLO"}],"E960":[{"name":"MARLIANA","star
t":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31"
,"name":"MARLIANA"}],"D347":[{"start":"1902-01-24","end":"9999-12-31","name"
:"DORMELLETTO"}],"F248":[{"start":"1882-06-19","end":"1928-05-11","name":"MI
SSAGLIA"},{"name":"MISSAGLIA","start":"1928-05-12","end":"1992-04-15"},{"end
":"9999-12-31","start":"1992-04-16","name":"MISSAGLIA"}],"F023":[{"name":"MA
SSA","start":"1868-10-10","end":"1938-12-17"},{"end":"9999-12-31","start":"1
946-03-04","name":"MASSA"}],"F329":[{"name":"MONASTEROLO CASOTTO","start":"1
956-02-22","end":"9999-12-31"},{"name":"MONASTEROLO CASSOTO","start":"1862-1
2-19","end":"1928-05-03"},{"name":"MONASTEROLO CASSOTO","start":"1946-12-05"
,"end":"1956-02-21"}],"L595":[{"name":"VALLE AURINA","end":"1958-08-26","sta
rt":"1929-02-21"},{"end":"9999-12-31","start":"1958-08-27","name":"VALLE AUR
INA"}],"D713":[{"start":"1909-01-23","end":"1961-12-25","name":"FORMIGNANA"}
,{"start":"1961-12-26","end":"2018-12-31","name":"FORMIGNANA"}],"M166":[{"na
me":"ZERBO","start":"1861-03-17","end":"9999-12-31"}],"D386":[{"end":"9999-1
2-31","start":"1861-03-17","name":"DURAZZANO"}],"C069":[{"name":"CASTELCIVIT
A","start":"1864-02-02","end":"9999-12-31"}],"L627":[{"start":"1881-05-13","
end":"1927-04-07","name":"VALLO DI NERA"},{"name":"VALLO DI NERA","start":"1
930-09-14","end":"9999-12-31"}],"C363":[{"name":"CAVAGLI\u00c0","start":"186
1-03-17","end":"1927-01-11"},{"end":"1992-04-15","start":"1927-01-12","name"
:"CAVAGLI\u00c0"},{"start":"1992-04-16","end":"9999-12-31","name":"CAVAGLI\u
00c0"}],"F891":[{"name":"NICORVO","start":"1861-03-17","end":"9999-12-31"}],
"A154":[{"end":"9999-12-31","start":"1866-11-19","name":"ALBETTONE"}],"B012"
:[{"name":"MOTTEGGIANA","start":"1868-02-14","end":"9999-12-31"}],"E442":[{"
end":"1947-03-15","start":"1928-09-09","name":"LANZA"}],"L245":[{"name":"TOR
RE ANNUNZIATA","start":"1877-03-09","end":"1928-04-27"},{"name":"TORRE ANNUN
ZIATA","end":"1946-02-27","start":"1928-04-28"},{"start":"1946-02-28","end":
"9999-12-31","name":"TORRE ANNUNZIATA"}],"I652":[{"name":"SERRARA FONTANA","
end":"1938-11-12","start":"1861-03-17"},{"end":"9999-12-31","start":"1945-09
-22","name":"SERRARA FONTANA"}],"E863":[{"end":"1927-01-11","start":"1861-03
-17","name":"MALNATE"},{"name":"MALNATE","start":"1927-01-12","end":"1928-01
-17"},{"end":"9999-12-31","start":"1928-01-18","name":"MALNATE"}],"A458":[{"
start":"1866-11-19","end":"9999-12-31","name":"ARZERGRANDE"}],"D587":[{"end"
:"1992-04-15","start":"1867-07-19","name":"FILADELFIA"},{"name":"FILADELFIA"
,"start":"1992-04-16","end":"9999-12-31"}],"F995":[{"name":"OCCIMIANO","star
t":"1861-03-17","end":"9999-12-31"}],"D838":[{"end":"1923-02-07","start":"19
21-01-05","name":"GABROVIZZA"},{"name":"GABROVIZZA","end":"1927-01-11","star
t":"1923-02-08"},{"start":"1927-01-12","end":"1928-08-23","name":"GABROVIZZA
"}],"L632":[{"start":"1862-11-10","end":"1928-04-16","name":"VALLORIA MARITT
IMA"}],"L197":[{"name":"TOLVE","end":"9999-12-31","start":"1861-03-17"}],"I0
45":[{"name":"SAN MICHELE SALENTINO","end":"9999-12-31","start":"1928-12-13"
}],"F816":[{"name":"MURO LECCESE","start":"1862-11-28","end":"9999-12-31"}],
"L814":[{"end":"1927-01-11","start":"1871-01-15","name":"VETRALLA"},{"name":
"VETRALLA","start":"1927-01-12","end":"9999-12-31"}],"D571":[{"name":"FI\u00
c8","end":"1923-02-13","start":"1920-10-16"},{"name":"FI\u00c8","end":"1927-
01-11","start":"1923-02-14"},{"name":"FI\u00c8","end":"1971-08-17","start":"
1927-01-12"},{"start":"1971-08-18","end":"9999-12-31","name":"FI\u00c8 ALLO 
SCILIAR"}],"E828":[{"name":"MAGR\u00c8","end":"1928-03-05","start":"1866-11-
19"},{"name":"MAGR\u00c8 VICENTINO","end":"1928-05-18","start":"1928-03-06"}
],"H349":[{"name":"RIVERA","end":"1928-03-05","start":"1861-03-17"}],"H043":
[{"start":"1861-03-17","end":"9999-12-31","name":"PRESEGLIE"}],"L740":[{"end
":"9999-12-31","start":"1863-01-19","name":"VENTIMIGLIA DI SICILIA"}],"D447"
:[{"end":"1928-09-24","start":"1863-02-23","name":"FABBRICA CURONE"},{"name"
:"FABBRICA CURONE","end":"9999-12-31","start":"1928-09-25"}],"M365":[{"name"
:"PIEVE DI BONO-PREZZO","start":"2016-01-01","end":"9999-12-31"}],"G912":[{"
name":"PORTOFERRAIO","start":"1861-03-17","end":"1951-03-09"},{"name":"PORTO
FERRAIO","start":"1951-03-10","end":"9999-12-31"}],"F068":[{"name":"MAZZIN",
"start":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-14","end":"1926-0
7-21","name":"MAZZIN"},{"end":"9999-12-31","start":"1952-05-17","name":"MAZZ
IN"}],"D185":[{"start":"1861-03-17","end":"1927-01-11","name":"CROSIO DELLA 
VALLE"},{"end":"9999-12-31","start":"1927-01-12","name":"CROSIO DELLA VALLE"
}],"E582":[{"end":"1956-11-10","start":"1928-06-14","name":"LIETO COLLE"}],"
C463":[{"end":"1927-01-11","start":"1861-03-17","name":"CENGIO"},{"start":"1
927-01-12","end":"1928-07-17","name":"CENGIO"},{"name":"CENGIO","end":"1928-
12-12","start":"1928-07-18"},{"name":"CENGIO","end":"9999-12-31","start":"19
28-12-13"}],"A867":[{"start":"1921-01-05","end":"1923-02-07","name":"BIGLIA"
},{"name":"BIGLIA","start":"1923-02-08","end":"1927-01-11"},{"name":"BIGLIA"
,"start":"1927-01-12","end":"1928-04-14"}],"C694":[{"start":"1920-10-16","en
d":"1923-02-13","name":"CIMEGO"},{"start":"1923-02-14","end":"1928-03-12","n
ame":"CIMEGO"},{"name":"CIMEGO","end":"2015-12-31","start":"1946-12-24"}],"D
486":[{"end":"9999-12-31","start":"1861-03-17","name":"FANANO"}],"A120":[{"n
ame":"ALANNO","end":"1927-01-11","start":"1861-03-17"},{"end":"9999-12-31","
start":"1927-01-12","name":"ALANNO"}],"C842":[{"name":"SEGNACCO","end":"1928
-04-29","start":"1878-08-13"}],"C249":[{"name":"CASTELPONZONE","start":"1861
-03-17","end":"1934-11-19"}],"H684":[{"name":"SALABERTANO","end":"1955-04-04
","start":"1937-06-16"},{"start":"1861-03-17","end":"1937-06-15","name":"SAL
BERTRAND"},{"end":"9999-12-31","start":"1955-04-05","name":"SALBERTRAND"}],"
E521":[{"start":"1861-03-17","end":"1928-05-11","name":"LEMNA"}],"D878":[{"n
ame":"GALLIERA","start":"1861-03-17","end":"9999-12-31"}],"G189":[{"name":"O
TRICOLI","start":"1861-03-17","end":"1927-01-11"},{"end":"9999-12-31","start
":"1927-01-12","name":"OTRICOLI"}],"H756":[{"name":"SAMUGHEO","start":"1861-
03-17","end":"1974-08-19"},{"start":"1974-08-20","end":"9999-12-31","name":"
SAMUGHEO"}],"L672":[{"name":"VARANO DE' MELEGARI","start":"1873-11-03","end"
:"9999-12-31"}],"I760":[{"name":"SIROR","end":"1923-02-13","start":"1920-10-
16"},{"end":"1927-11-23","start":"1923-02-14","name":"SIROR"},{"name":"SIROR
","end":"2015-12-31","start":"1947-01-24"}],"H228":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"REITANO"}],"F429":[{"name":"MONTAQUILA","start":"18
61-03-17","end":"1970-03-02"},{"end":"9999-12-31","start":"1970-03-03","name
":"MONTAQUILA"}],"H589":[{"name":"ROTELLO","end":"1970-03-02","start":"1861-
03-17"},{"name":"ROTELLO","end":"9999-12-31","start":"1970-03-03"}],"L065":[
{"end":"1928-05-12","start":"1866-11-19","name":"TAVAGNACCO"},{"name":"TAVAG
NACCO","end":"1947-01-08","start":"1928-05-13"},{"end":"1968-04-05","start":
"1947-01-09","name":"TAVAGNACCO"},{"end":"9999-12-31","start":"1968-04-06","
name":"TAVAGNACCO"}],"A358":[{"end":"1927-01-11","start":"1861-03-17","name"
:"ARBORIO"},{"name":"ARBORIO","end":"9999-12-31","start":"1927-01-12"}],"E34
2":[{"end":"1983-07-13","start":"1863-07-13","name":"ISOLATO"},{"name":"MADE
SIMO","end":"9999-12-31","start":"1983-07-14"}],"C599":[{"start":"1861-03-17
","end":"9999-12-31","name":"CHERASCO"}],"D211":[{"name":"CUPRAMONTANA","end
":"9999-12-31","start":"1862-05-06"}],"B234":[{"name":"BRUZZANO ZEFFIRIO","e
nd":"9999-12-31","start":"1885-01-15"}],"G016":[{"name":"OLCENENGO","start":
"1861-03-17","end":"1927-01-11"},{"name":"OLCENENGO","end":"9999-12-31","sta
rt":"1927-01-12"}],"A720":[{"name":"BAUCO","start":"1871-01-15","end":"1907-
02-03"},{"start":"1907-02-04","end":"1927-01-11","name":"BOVILLE ERNICA"},{"
start":"1927-01-12","end":"9999-12-31","name":"BOVILLE ERNICA"}],"G357":[{"n
ame":"PASSERANO","end":"1929-04-13","start":"1861-03-17"}],"G789":[{"start":
"1861-03-17","end":"9999-12-31","name":"POLINAGO"}],"I827":[{"name":"SONCINO
","start":"1861-03-17","end":"9999-12-31"}],"H156":[{"end":"9999-12-31","sta
rt":"1861-03-17","name":"RADICOFANI"}],"G954":[{"name":"POZZILLI","start":"1
861-03-17","end":"1911-07-27"},{"start":"1911-07-28","end":"1927-10-13","nam
e":"POZZILLI"},{"name":"POZZILLI","end":"1970-03-02","start":"1937-02-24"},{
"end":"9999-12-31","start":"1970-03-03","name":"POZZILLI"}],"B895":[{"name":
"CASAL VELINO","start":"1893-06-02","end":"9999-12-31"}],"G556":[{"start":"1
863-02-23","end":"9999-12-31","name":"PIANELLO DEL LARIO"}],"E084":[{"start"
:"1861-03-17","end":"1928-07-28","name":"GONI"},{"name":"GONI","start":"1946
-02-07","end":"1974-08-19"},{"start":"1974-08-20","end":"2016-04-27","name":
"GONI"},{"name":"GONI","end":"9999-12-31","start":"2016-04-28"}],"I239":[{"n
ame":"SANTA MARIA DEL MONTE","end":"1927-01-11","start":"1861-03-17"},{"name
":"SANTA MARIA DEL MONTE","end":"1927-12-28","start":"1927-01-12"}],"I832":[
{"end":"1934-10-25","start":"1871-01-15","name":"SONNINO"},{"name":"SONNINO"
,"start":"1934-10-26","end":"1934-12-17"},{"name":"SONNINO","start":"1934-12
-18","end":"1945-04-08"},{"name":"SONNINO","start":"1945-04-09","end":"9999-
12-31"}],"B991":[{"name":"CASORZO","end":"1935-04-07","start":"1861-03-17"},
{"name":"CASORZO","end":"2022-03-02","start":"1935-04-08"},{"name":"CASORZO 
MONFERRATO","start":"2022-03-03","end":"9999-12-31"}],"Z711":[{"name":"Isole
 Marshall","start":"0000-01-01","end":"9999-12-31"}],"L259":[{"name":"TORRE 
DEL GRECO","start":"1861-03-17","end":"9999-12-31"}],"M379":[{"name":"TERRE 
ROVERESCHE","end":"9999-12-31","start":"2017-01-01"}],"M073":[{"end":"1923-0
2-07","start":"1921-01-05","name":"VISCO"},{"start":"1923-02-08","end":"1968
-04-05","name":"VISCO"},{"name":"VISCO","start":"1968-04-06","end":"9999-12-
31"}],"L433":[{"end":"9999-12-31","start":"1866-11-19","name":"TRISSINO"}],"
Z533":[{"name":"Saint Kitts e Nevis","start":"0000-01-01","end":"9999-12-31"
}],"L797":[{"end":"1992-04-15","start":"1861-03-17","name":"VERUCCHIO"},{"na
me":"VERUCCHIO","start":"1992-04-16","end":"9999-12-31"}],"L909":[{"start":"
1867-09-02","end":"1968-04-05","name":"VILLA SANTINA"},{"end":"9999-12-31","
start":"1968-04-06","name":"VILLA SANTINA"}],"D490":[{"start":"1871-08-01","
end":"1905-12-31","name":"FARA D'ADDA"},{"name":"FARA GERA D'ADDA","start":"
1906-01-01","end":"9999-12-31"}],"L606":[{"end":"1916-06-04","start":"1861-0
3-17","name":"VALLE INFERIORE MOSSO"},{"name":"VALLE MOSSO","start":"1916-06
-05","end":"1927-01-11"},{"name":"VALLE MOSSO","end":"1929-06-14","start":"1
927-01-12"},{"end":"1992-04-15","start":"1929-06-15","name":"VALLE MOSSO"},{
"start":"1992-04-16","end":"2018-12-31","name":"VALLE MOSSO"}],"C750":[{"end
":"1927-01-11","start":"1861-03-17","name":"CITT\u00c0 SANT'ANGELO"},{"name"
:"CITT\u00c0 SANT'ANGELO","start":"1927-01-12","end":"9999-12-31"}],"F604":[
{"name":"MONTERONI DI LECCE","start":"1864-04-01","end":"9999-12-31"}],"I453
":[{"name":"SASSELLO","end":"1927-01-11","start":"1861-03-17"},{"name":"SASS
ELLO","end":"1970-02-24","start":"1927-01-12"},{"start":"1970-02-25","end":"
9999-12-31","name":"SASSELLO"}],"H161":[{"name":"RAGOGNA","start":"1866-11-1
9","end":"1928-02-15"},{"start":"1928-02-16","end":"1968-04-05","name":"RAGO
GNA"},{"end":"9999-12-31","start":"1968-04-06","name":"RAGOGNA"}],"E643":[{"
name":"LOCCA","end":"1923-02-13","start":"1920-10-16"},{"end":"1928-02-08","
start":"1923-02-14","name":"LOCCA"}],"L140":[{"name":"TERTENIA","end":"1927-
01-11","start":"1861-03-17"},{"name":"TERTENIA","end":"2005-12-31","start":"
1927-01-12"},{"end":"2016-04-27","start":"2006-01-01","name":"TERTENIA"},{"s
tart":"2016-04-28","end":"9999-12-31","name":"TERTENIA"}],"H455":[{"name":"R
OCCELLA VALDEMONE","end":"9999-12-31","start":"1863-04-10"}],"G204":[{"name"
:"OZZANO MONFERRATO","end":"9999-12-31","start":"1863-08-10"}],"G561":[{"sta
rt":"1861-03-17","end":"9999-12-31","name":"PIANFEI"}],"M253":[{"start":"186
1-03-17","end":"1929-02-07","name":"SICIGNANO"},{"name":"SICIGNANO DEGLI ALB
URNI","end":"9999-12-31","start":"1929-02-08"}],"L373":[{"start":"1869-02-01
","end":"1923-10-13","name":"TRENNO"}],"L079":[{"start":"1920-10-16","end":"
1923-02-13","name":"TAVON"},{"name":"TAVON","end":"1929-01-23","start":"1923
-02-14"}],"B020":[{"start":"1861-03-17","end":"1926-01-21","name":"BORGOMARO
"},{"name":"BORGOMARO","start":"1926-01-22","end":"1928-12-07"},{"start":"19
28-12-08","end":"1954-06-03","name":"BORGOMARO"},{"end":"9999-12-31","start"
:"1954-06-04","name":"BORGOMARO"}],"E401":[{"start":"1861-03-17","end":"9999
-12-31","name":"LAERRU"}],"D785":[{"end":"1923-03-18","start":"1863-04-13","
name":"FRASSO SABINO"},{"end":"1927-01-11","start":"1923-03-19","name":"FRAS
SO SABINO"},{"name":"FRASSO SABINO","start":"1927-01-12","end":"9999-12-31"}
],"F509":[{"start":"1861-03-17","end":"2009-07-11","name":"MONTEFORTINO"},{"
start":"2009-07-12","end":"9999-12-31","name":"MONTEFORTINO"}],"I021":[{"nam
e":"SAN MARZANOTTO","start":"1861-03-17","end":"1929-06-04"}],"E385":[{"name
":"JERAGO CON BESNATE ED ORAGO","start":"1892-04-04","end":"1907-07-01"}],"D
850":[{"end":"9999-12-31","start":"1864-06-23","name":"GAGLIANO ATERNO"}],"G
660":[{"start":"1861-03-17","end":"1947-09-15","name":"PIGNA"},{"name":"PIGN
A","end":"9999-12-31","start":"1947-09-16"}],"A430":[{"name":"AROSIO","end":
"1929-05-16","start":"1861-03-17"},{"end":"9999-12-31","start":"1950-04-05",
"name":"AROSIO"}],"H991":[{"start":"1864-02-14","end":"9999-12-31","name":"S
AN MARTINO SULLA MARRUCINA"}],"I893":[{"name":"SPERONE","start":"1861-03-17"
,"end":"9999-12-31"}],"F139":[{"start":"1861-03-17","end":"1992-04-15","name
":"MERCATO SARACENO"},{"name":"MERCATO SARACENO","start":"1992-04-16","end":
"9999-12-31"}],"B156":[{"start":"1861-03-17","end":"1992-01-29","name":"BRES
CELLO"},{"name":"BRESCELLO","start":"1992-01-30","end":"9999-12-31"}],"D203"
:[{"end":"9999-12-31","start":"1862-12-19","name":"CUMIGNANO SUL NAVIGLIO"}]
,"Z103":[{"name":"Belgio","start":"0000-01-01","end":"9999-12-31"}],"E948":[
{"end":"1923-02-13","start":"1920-10-16","name":"MARGONE"},{"end":"1928-04-1
8","start":"1923-02-14","name":"MARGONE"}],"A143":[{"name":"ALBAVILLA","end"
:"1935-07-12","start":"1928-07-29"},{"name":"ALBAVILLA","start":"1935-07-13"
,"end":"9999-12-31"}],"H895":[{"name":"SAN GIORGIO DI NOGARO","end":"1928-04
-16","start":"1866-11-19"},{"end":"1940-12-19","start":"1928-04-17","name":"
SAN GIORGIO DI NOGARO"},{"start":"1940-12-20","end":"1946-12-23","name":"SAN
 GIORGIO DI NOGARO"},{"name":"SAN GIORGIO DI NOGARO","end":"1968-04-05","sta
rt":"1946-12-24"},{"name":"SAN GIORGIO DI NOGARO","start":"1968-04-06","end"
:"1968-07-22"},{"start":"1968-07-23","end":"9999-12-31","name":"SAN GIORGIO 
DI NOGARO"}],"F260":[{"end":"1992-04-15","start":"1963-11-24","name":"TAVAZZ
ANO CON VILLAVESCO"},{"start":"1992-04-16","end":"9999-12-31","name":"TAVAZZ
ANO CON VILLAVESCO"},{"start":"1879-06-09","end":"1963-11-23","name":"VILLAV
ESCO"}],"C885":[{"start":"1866-11-19","end":"1968-04-05","name":"COLLOREDO D
I MONTE ALBANO"},{"end":"9999-12-31","start":"1968-04-06","name":"COLLOREDO 
DI MONTE ALBANO"}],"H234":[{"name":"RENATE VEDUGGIO","start":"1929-04-11","e
nd":"1956-04-07"}],"A952":[{"name":"BOLZANO","start":"1920-10-16","end":"192
3-02-13"},{"start":"1923-02-14","end":"1925-01-16","name":"BOLZANO"},{"name"
:"BOLZANO","start":"1925-01-17","end":"1927-01-11"},{"start":"1927-01-12","e
nd":"9999-12-31","name":"BOLZANO"}],"A370":[{"start":"1891-06-13","end":"999
9-12-31","name":"ARCINAZZO ROMANO"}],"E545":[{"name":"LESA BELGIRATE","end":
"1947-12-22","start":"1928-04-15"}],"Z242":[{"name":"Timor Leste","end":"999
9-12-31","start":"0000-01-01"}],"H020":[{"name":"PREDORE","end":"9999-12-31"
,"start":"1861-03-17"}],"L723":[{"name":"VELO D'ASTICO","start":"1867-09-02"
,"end":"9999-12-31"}],"D142":[{"name":"CREMA","end":"1928-05-28","start":"18
75-04-01"},{"start":"1928-05-29","end":"9999-12-31","name":"CREMA"}],"C334":
[{"name":"CASTRI DI LECCE","end":"9999-12-31","start":"1892-01-08"}],"I990":
[{"start":"1861-03-17","end":"9999-12-31","name":"STURNO"}],"A202":[{"name":
"ALIMENA","start":"1861-03-17","end":"9999-12-31"}],"A809":[{"name":"BERTINO
RO","start":"1861-03-17","end":"1992-04-15"},{"name":"BERTINORO","end":"9999
-12-31","start":"1992-04-16"}],"D617":[{"name":"FIUMALBO","end":"1936-07-25"
,"start":"1861-03-17"},{"start":"1936-07-26","end":"9999-12-31","name":"FIUM
ALBO"}],"B455":[{"end":"9999-12-31","start":"1861-03-17","name":"CAMAIORE"}]
,"C178":[{"name":"CASTELLO D'ALIFE","start":"1862-11-10","end":"1927-01-11"}
,{"name":"CASTELLO D'ALIFE","end":"1945-08-31","start":"1927-01-12"},{"start
":"1945-09-01","end":"1970-10-27","name":"CASTELLO D'ALIFE"},{"name":"CASTEL
LO DEL MATESE","start":"1970-10-28","end":"9999-12-31"}],"C937":[{"name":"CO
MUN NUOVO","end":"9999-12-31","start":"1861-03-17"}],"L492":[{"name":"UMBRIA
TICO","start":"1861-03-17","end":"1992-04-15"},{"start":"1992-04-16","end":"
9999-12-31","name":"UMBRIATICO"}],"B161":[{"name":"BRESSANVIDO","start":"186
6-11-19","end":"9999-12-31"}],"I082":[{"name":"SAN PIER D'ISONZO","start":"1
923-04-27","end":"1948-01-09"},{"name":"SAN PIER D'ISONZO","end":"9999-12-31
","start":"1948-01-10"},{"name":"SAN PIETRO DELL'ISONZO","end":"1923-02-07",
"start":"1921-01-05"},{"name":"SAN PIETRO DELL'ISONZO","end":"1923-04-26","s
tart":"1923-02-08"}],"E295":[{"end":"1928-12-06","start":"1863-02-16","name"
:"INCISA BELBO"},{"start":"1928-12-07","end":"1929-05-27","name":"INCISA SCA
PACCINO"},{"end":"1935-04-07","start":"1929-05-28","name":"INCISA SCAPACCINO
"},{"name":"INCISA SCAPACCINO","end":"9999-12-31","start":"1935-04-08"}],"E8
34":[{"start":"1861-03-17","end":"1935-06-05","name":"MAIDA"},{"name":"MAIDA
","start":"1935-06-06","end":"9999-12-31"}],"C474":[{"end":"1927-01-11","sta
rt":"1861-03-17","name":"CEPAGATTI"},{"name":"CEPAGATTI","start":"1927-01-12
","end":"9999-12-31"}],"G318":[{"name":"PAOLISI","end":"9999-12-31","start":
"1861-03-17"}],"C922":[{"start":"1861-03-17","end":"1927-01-11","name":"COME
RIO"},{"name":"COMERIO","end":"1927-07-26","start":"1927-01-12"},{"start":"1
927-07-27","end":"1929-05-18","name":"COMERIO"},{"name":"COMERIO","start":"1
929-05-19","end":"1957-08-12"},{"end":"9999-12-31","start":"1957-08-13","nam
e":"COMERIO"}],"A914":[{"start":"1861-03-17","end":"1927-01-11","name":"BOCC
IOLETO"},{"name":"BOCCIOLETO","start":"1927-01-12","end":"9999-12-31"}],"L12
3":[{"name":"TERRANUOVA BRACCIOLINI","start":"1868-08-09","end":"9999-12-31"
}],"D742":[{"name":"FOSSANO","start":"1861-03-17","end":"9999-12-31"}],"F351
":[{"end":"9999-12-31","start":"1861-03-17","name":"MONDOV\u00cc"}],"M064":[
{"end":"1923-02-13","start":"1920-10-16","name":"VION"},{"name":"VION","end"
:"1928-12-13","start":"1923-02-14"}],"B043":[{"end":"9999-12-31","start":"18
61-03-17","name":"BORGO TICINO"}],"B349":[{"name":"CAERANO DI SAN MARCO","en
d":"1928-05-03","start":"1872-03-06"},{"name":"CAERANO DI SAN MARCO","end":"
9999-12-31","start":"1945-05-01"}],"A317":[{"end":"1992-05-22","start":"1928
-05-31","name":"ANTRONA SCHIERANCO"},{"end":"9999-12-31","start":"1992-05-23
","name":"ANTRONA SCHIERANCO"}],"C778":[{"name":"CIVITELLA ALFEDENA","start"
:"1861-03-17","end":"9999-12-31"}],"A056":[{"name":"ADERN\u00d2","start":"18
61-03-17","end":"1929-08-22"},{"end":"9999-12-31","start":"1929-08-23","name
":"ADRANO"}],"I947":[{"name":"STELLANELLO","end":"1927-01-11","start":"1861-
03-17"},{"start":"1927-01-12","end":"9999-12-31","name":"STELLANELLO"}],"E41
3":[{"start":"1861-03-17","end":"9999-12-31","name":"LAJATICO"}],"B761":[{"n
ame":"CARELLA CON MARIAGA","start":"1861-03-17","end":"1927-07-14"}],"A516":
[{"name":"AVIANO","start":"1866-11-19","end":"1953-07-15"},{"name":"AVIANO",
"end":"1968-04-05","start":"1953-07-16"},{"name":"AVIANO","start":"1968-04-0
6","end":"9999-12-31"}],"I344":[{"name":"SANT'IPPOLITO","end":"1928-06-29","
start":"1861-03-17"},{"end":"1998-09-03","start":"1928-06-30","name":"SANT'I
PPOLITO"},{"name":"SANT'IPPOLITO","end":"9999-12-31","start":"1998-09-04"}],
"E620":[{"start":"1861-03-17","end":"1927-01-11","name":"LIVERI"},{"end":"99
99-12-31","start":"1927-01-12","name":"LIVERI"}],"L364":[{"end":"1954-10-25"
,"start":"1866-11-19","name":"TREGNAGO"},{"start":"1954-10-26","end":"9999-1
2-31","name":"TREGNAGO"}],"I689":[{"name":"SESTOLA","end":"9999-12-31","star
t":"1861-03-17"}],"I986":[{"name":"STROZZA","end":"9999-12-31","start":"1861
-03-17"}],"M129":[{"name":"VOLTRI","start":"1861-03-17","end":"1926-02-05"}]
,"B589":[{"end":"9999-12-31","start":"1878-06-14","name":"CANDIANA"}],"A061"
:[{"end":"9999-12-31","start":"1866-11-19","name":"AFFI"}],"A276":[{"start":
"1920-10-16","end":"1923-02-13","name":"ANDOGNO"},{"name":"ANDOGNO","start":
"1923-02-14","end":"1928-01-25"}],"F739":[{"name":"MORNICO LOSANA","start":"
1863-03-30","end":"9999-12-31"}],"L967":[{"start":"1866-11-19","end":"9999-1
2-31","name":"VILLAMARZANA"}],"B756":[{"name":"CARDETO","end":"1977-02-26","
start":"1861-03-17"},{"end":"9999-12-31","start":"1977-02-27","name":"CARDET
O"}],"ND":[{"end":"1911-07-04","start":"1861-03-17","name":"ARIZZANO"},{"end
":"1927-01-11","start":"1863-01-05","name":"CAMPIGLIA SOANA"},{"name":"CAMPI
GLIA SOANA","end":"1929-01-22","start":"1927-01-12"},{"start":"1861-03-17","
end":"1927-01-11","name":"CARAVINO"},{"start":"1927-01-12","end":"1929-03-02
","name":"CARAVINO"},{"name":"CUGLIATE","end":"1927-01-11","start":"1861-03-
17"},{"name":"CUGLIATE","start":"1927-01-12","end":"1928-10-07"},{"start":"1
863-07-13","end":"1927-01-11","name":"CUVEGLIO IN VALLE"},{"start":"1927-01-
12","end":"1928-02-15","name":"CUVEGLIO IN VALLE"},{"start":"1861-03-17","en
d":"1928-05-31","name":"GONNOSN\u00d2"},{"start":"1923-04-27","end":"1924-03
-08","name":"MONTE CHILOVI"},{"start":"1924-03-09","end":"1924-10-30","name"
:"MONTE CHILOVI"},{"start":"1921-01-05","end":"1923-02-07","name":"MONTEFRED
DO"},{"name":"MONTEFREDDO","start":"1923-02-08","end":"1923-04-26"},{"start"
:"1921-01-05","end":"1923-02-07","name":"OCCISLA-CLANZO"},{"start":"1923-02-
08","end":"1923-02-08","name":"OCCISLA-CLANZO"},{"name":"PRATO DI VENOSTA","
start":"1920-10-16","end":"1923-02-13"},{"end":"1927-01-11","start":"1923-02
-14","name":"PRATO DI VENOSTA"},{"start":"1927-01-12","end":"1929-05-14","na
me":"PRATO DI VENOSTA"},{"name":"RUINAS","end":"1928-06-14","start":"1861-03
-17"},{"start":"1861-03-17","end":"1928-08-11","name":"SAN SIRO"},{"start":"
1920-10-16","end":"1923-02-13","name":"SENALE"},{"start":"1923-02-14","end":
"1928-06-30","name":"SENALE"},{"name":"SENALE","end":"1948-03-13","start":"1
947-03-12"},{"end":"1974-09-17","start":"1948-03-14","name":"SENALE"},{"star
t":"1861-03-17","end":"1928-05-04","name":"SIAMANNA"},{"name":"TORTOL\u00cc"
,"end":"1927-01-11","start":"1861-03-17"},{"name":"TORTOL\u00cc","end":"1928
-05-11","start":"1927-01-12"}],"B684":[{"name":"CAPRAIA E LIMITE","end":"999
9-12-31","start":"1863-04-13"}],"Z703":[{"name":"Isole Cook (NZ)","start":"0
000-01-01","end":"9999-12-31"}],"A743":[{"name":"BELLA","end":"9999-12-31","
start":"1861-03-17"}],"F567":[{"name":"POLLENZA","start":"1862-10-24","end":
"9999-12-31"}],"B228":[{"name":"BRUSIMPIANO","end":"1927-01-11","start":"186
1-03-17"},{"start":"1927-01-12","end":"1928-09-05","name":"BRUSIMPIANO"},{"s
tart":"1928-09-06","end":"9999-12-31","name":"BRUSIMPIANO"}],"D670":[{"name"
:"FONTANAFREDDA","start":"1866-11-19","end":"1968-04-05"},{"name":"FONTANAFR
EDDA","end":"2003-04-16","start":"1968-04-06"},{"start":"2003-04-17","end":"
9999-12-31","name":"FONTANAFREDDA"}],"I584":[{"name":"SELLA DELLE TRINCEE","
start":"1923-04-27","end":"1927-01-11"},{"name":"SELLA DELLE TRINCEE","end":
"1928-08-23","start":"1927-01-12"},{"end":"1923-02-07","start":"1921-01-05",
"name":"SELLO"},{"name":"SELLO","end":"1923-04-26","start":"1923-02-08"}],"A
417":[{"end":"1923-02-13","start":"1920-10-16","name":"ARMO"},{"start":"1923
-02-14","end":"1929-02-28","name":"ARMO"}],"C813":[{"name":"CODEVILLA","star
t":"1861-03-17","end":"9999-12-31"}],"I148":[{"end":"9999-12-31","start":"18
61-03-17","name":"SAN SALVO"}],"G714":[{"name":"PISTOLESA","start":"1861-03-
17","end":"1927-01-11"},{"name":"PISTOLESA","end":"1992-04-15","start":"1927
-01-12"},{"name":"PISTOLESA","end":"1998-12-31","start":"1992-04-16"}],"B449
":[{"start":"1951-07-05","end":"1958-05-27","name":"CALVI SAN NAZZARO"},{"st
art":"1863-02-06","end":"1951-07-04","name":"SAN NAZZARO CALVI"}],"B263":[{"
start":"1861-03-17","end":"1928-06-13","name":"BULGORELLO"}],"E559":[{"name"
:"LETTOPALENA","start":"1861-03-17","end":"1928-04-02"},{"name":"LETTOPALENA
","end":"9999-12-31","start":"1947-05-22"}],"I444":[{"name":"SARSINA","start
":"1861-03-17","end":"1965-01-16"},{"end":"1992-04-15","start":"1965-01-17",
"name":"SARSINA"},{"start":"1992-04-16","end":"9999-12-31","name":"SARSINA"}
],"D088":[{"name":"COSIO VALTELLINO","start":"1863-07-13","end":"9999-12-31"
}],"E654":[{"end":"1927-11-23","start":"1884-11-06","name":"LOGRATO"},{"end"
:"1947-08-14","start":"1927-11-24","name":"LOGRATO"},{"end":"9999-12-31","st
art":"1947-08-15","name":"LOGRATO"}],"F165":[{"name":"MEZZAGO","start":"1861
-03-17","end":"2009-06-29"},{"start":"2009-06-30","end":"9999-12-31","name":
"MEZZAGO"}],"E786":[{"name":"MACHERIO","start":"1861-03-17","end":"2009-06-2
9"},{"start":"2009-06-30","end":"9999-12-31","name":"MACHERIO"}],"E313":[{"e
nd":"9999-12-31","start":"1870-08-02","name":"INVERUNO"}],"E019":[{"start":"
1861-03-17","end":"9999-12-31","name":"GIAVE"}],"C738":[{"name":"CISSONE","e
nd":"1929-01-24","start":"1861-03-17"},{"name":"CISSONE","end":"9999-12-31",
"start":"1947-06-17"}],"G840":[{"name":"PONTEDASSIO","end":"1928-02-20","sta
rt":"1861-03-17"},{"name":"PONTEDASSIO","start":"1928-02-21","end":"9999-12-
31"}],"A898":[{"end":"9999-12-31","start":"1861-03-17","name":"BIZZARONE"}],
"E747":[{"end":"2005-12-31","start":"1947-10-26","name":"LUOGOSANTO"},{"star
t":"2006-01-01","end":"2016-04-27","name":"LUOGOSANTO"},{"name":"LUOGOSANTO"
,"start":"2016-04-28","end":"9999-12-31"}],"H552":[{"start":"1920-10-16","en
d":"1923-02-13","name":"RONZONE"},{"end":"1928-08-06","start":"1923-02-14","
name":"RONZONE"},{"start":"1953-01-04","end":"9999-12-31","name":"RONZONE"}]
,"F097":[{"end":"1925-10-31","start":"1861-03-17","name":"MELDOLA"},{"name":
"MELDOLA","start":"1925-11-01","end":"1926-02-20"},{"name":"MELDOLA","end":"
1992-04-15","start":"1926-02-21"},{"end":"9999-12-31","start":"1992-04-16","
name":"MELDOLA"}],"C910":[{"start":"1861-03-17","end":"9999-12-31","name":"C
OLZATE"}],"D101":[{"name":"COSSOMBRATO","end":"1935-04-07","start":"1861-03-
17"},{"name":"COSSOMBRATO","end":"9999-12-31","start":"1935-04-08"}],"L693":
[{"name":"VASIA","start":"1861-03-17","end":"1928-04-16"},{"name":"VASIA","s
tart":"1928-04-17","end":"9999-12-31"}],"A241":[{"end":"1926-01-10","start":
"1881-04-22","name":"ALTOPASCIO"},{"name":"ALTOPASCIO","start":"1926-01-11",
"end":"9999-12-31"}],"D630":[{"name":"FLAIBANO","start":"1927-11-02","end":"
1968-04-05"},{"end":"2014-02-03","start":"1968-04-06","name":"FLAIBANO"},{"n
ame":"FLAIBANO","start":"2014-02-04","end":"9999-12-31"},{"name":"SANT' ODOR
ICO","start":"1866-11-19","end":"1927-11-01"}],"L464":[{"name":"TULA","end":
"1988-08-05","start":"1861-03-17"},{"name":"TULA","start":"1988-08-06","end"
:"9999-12-31"}],"F466":[{"start":"1872-09-16","end":"1937-11-03","name":"MON
TECELIO"}],"G943":[{"name":"POVE","end":"1950-07-19","start":"1866-11-19"},{
"name":"POVE DEL GRAPPA","start":"1950-07-20","end":"9999-12-31"}],"H012":[{
"name":"PR\u00c8","start":"1920-10-16","end":"1923-02-13"},{"start":"1923-02
-14","end":"1928-04-03","name":"PR\u00c8"}],"E190":[{"end":"1927-10-12","sta
rt":"1861-03-17","name":"GRONA"}],"F779":[{"end":"9999-12-31","start":"1861-
03-17","name":"MOTTA SAN GIOVANNI"}],"L168":[{"name":"TIGLIOLE","end":"1929-
04-11","start":"1861-03-17"},{"name":"TIGLIOLE","end":"1935-04-07","start":"
1929-04-12"},{"name":"TIGLIOLE","start":"1935-04-08","end":"9999-12-31"}],"A
591":[{"end":"9999-12-31","start":"1863-01-05","name":"BALDISSERO TORINESE"}
],"G152":[{"end":"1928-05-22","start":"1861-03-17","name":"OSASIO"},{"end":"
9999-12-31","start":"1946-12-19","name":"OSASIO"}],"A236":[{"name":"ALTISSIM
O","end":"9999-12-31","start":"1866-11-19"}],"E147":[{"name":"GRASSANO","sta
rt":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31
","name":"GRASSANO"}],"L889":[{"start":"1861-03-17","end":"1928-01-07","name
":"VIGNONE"},{"start":"1948-04-08","end":"1992-05-22","name":"VIGNONE"},{"na
me":"VIGNONE","end":"9999-12-31","start":"1992-05-23"}],"D824":[{"name":"FUR
CI SICULO","start":"1919-08-04","end":"1928-12-29"},{"start":"1928-12-30","e
nd":"1932-03-26","name":"FURCI SICULO"},{"end":"1938-08-06","start":"1932-03
-27","name":"FURCI SICULO"},{"name":"FURCI SICULO","end":"9999-12-31","start
":"1938-08-07"}],"D701":[{"name":"FORINO","end":"9999-12-31","start":"1861-0
3-17"}],"A330":[{"start":"1861-03-17","end":"9999-12-31","name":"APOLLOSA"}]
,"L282":[{"start":"1861-03-17","end":"1929-06-17","name":"TORRETTA"},{"name"
:"TORRETTA","end":"9999-12-31","start":"1929-06-18"}],"E485":[{"start":"1875
-09-07","end":"9999-12-31","name":"LAURINO"}],"H274":[{"start":"1922-12-02",
"end":"1992-04-15","name":"RICCIONE"},{"start":"1992-04-16","end":"9999-12-3
1","name":"RICCIONE"}],"F040":[{"name":"MASSE DI SIENA","start":"1876-12-28"
,"end":"1904-05-22"}],"A470":[{"end":"9999-12-31","start":"1868-07-01","name
":"ASOLA"}],"E790":[{"name":"MACUGNAGA","start":"1861-03-17","end":"1992-05-
22"},{"name":"MACUGNAGA","start":"1992-05-23","end":"9999-12-31"}],"L768":[{
"start":"1861-03-17","end":"2019-02-07","name":"VERMEZZO"}],"G752":[{"start"
:"1861-03-17","end":"9999-12-31","name":"POGGIBONSI"}],"M092":[{"name":"VITU
LAZIO","start":"1882-12-18","end":"1927-01-11"},{"name":"VITULAZIO","start":
"1927-01-12","end":"1928-03-13"},{"start":"1946-04-13","end":"9999-12-31","n
ame":"VITULAZIO"}],"B355":[{"end":"1928-04-02","start":"1861-03-17","name":"
CAGLIO"},{"end":"9999-12-31","start":"1947-10-01","name":"CAGLIO"}],"I748":[
{"name":"SINDIA","start":"1861-03-17","end":"1927-01-11"},{"name":"SINDIA","
end":"9999-12-31","start":"1927-01-12"}],"H916":[{"name":"SAN GIOVANNI ILARI
ONE","start":"1866-11-19","end":"1924-01-13"},{"end":"9999-12-31","start":"1
924-01-14","name":"SAN GIOVANNI ILARIONE"}],"G114":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"ORMEA"}],"H619":[{"name":"ROVIGNO","end":"1922-02-0
1","start":"1921-01-05"},{"start":"1922-02-02","end":"1923-02-07","name":"RO
VIGNO"},{"name":"ROVIGNO","start":"1923-02-08","end":"1923-04-26"},{"start":
"1923-04-27","end":"1928-09-15","name":"ROVIGNO D'ISTRIA"},{"end":"1947-09-1
5","start":"1928-09-16","name":"ROVIGNO D'ISTRIA"}],"C007":[{"end":"9999-12-
31","start":"1861-03-17","name":"CASSIGLIO"}],"H054":[{"start":"1921-01-05",
"end":"1923-02-07","name":"PERVACINA"},{"end":"1923-04-26","start":"1923-02-
08","name":"PERVACINA"},{"end":"1927-01-11","start":"1923-04-27","name":"PRE
VACINA"},{"start":"1927-01-12","end":"1928-04-16","name":"PREVACINA"}],"E874
":[{"name":"MAMOIADA","start":"1861-03-17","end":"1927-01-11"},{"name":"MAMO
IADA","end":"9999-12-31","start":"1927-01-12"}],"C434":[{"name":"CELLA DI BO
BBIO","end":"1929-05-28","start":"1864-02-14"}],"G418":[{"start":"1861-03-17
","end":"9999-12-31","name":"PEIA"}],"F765":[{"name":"MOSCUFO","end":"1927-0
1-11","start":"1861-03-17"},{"end":"9999-12-31","start":"1927-01-12","name":
"MOSCUFO"}],"E186":[{"name":"GRISOLIA CIPOLLINA","end":"1948-04-29","start":
"1861-03-17"}],"C374":[{"start":"1861-03-17","end":"1928-06-13","name":"CAVA
LLASCA"},{"start":"1956-11-10","end":"2016-12-31","name":"CAVALLASCA"}],"L39
2":[{"name":"TRESIVIO","end":"1929-01-31","start":"1867-01-06"},{"name":"TRE
SIVIO","start":"1929-02-01","end":"9999-12-31"}],"C138":[{"name":"CASTELLANO
","start":"1920-10-16","end":"1923-02-13"},{"name":"CASTELLANO","start":"192
3-02-14","end":"1929-02-20"}],"L221":[{"name":"TORLINO","end":"1961-03-31","
start":"1861-03-17"},{"end":"9999-12-31","start":"1961-04-01","name":"TORLIN
O VIMERCATI"}],"C977":[{"start":"1861-03-17","end":"9999-12-31","name":"CONZ
ANO"}],"H514":[{"start":"1928-05-02","end":"1954-06-29","name":"ROMANS D'ISO
NZO"},{"start":"1954-06-30","end":"9999-12-31","name":"ROMANS D'ISONZO"}],"E
703":[{"name":"LOVER","end":"1923-02-13","start":"1920-10-16"},{"start":"192
3-02-14","end":"1928-10-29","name":"LOVER"}],"A425":[{"start":"1861-03-17","
end":"9999-12-31","name":"ARNESANO"}],"F744":[{"name":"MORRA DE SANCTIS","en
d":"9999-12-31","start":"1934-06-22"},{"end":"1934-06-21","start":"1863-02-0
6","name":"MORRA IRPINO"}],"L746":[{"start":"1939-06-04","end":"1992-05-22",
"name":"VERBANIA"},{"start":"1992-05-23","end":"9999-12-31","name":"VERBANIA
"}],"G077":[{"name":"OPACCHIASELLA","start":"1921-01-05","end":"1923-02-07"}
,{"name":"OPACCHIASELLA","end":"1927-01-11","start":"1923-02-08"},{"name":"O
PACCHIASELLA","start":"1927-01-12","end":"1928-08-23"},{"start":"1928-08-24"
,"end":"1947-09-15","name":"OPACCHIASELLA"}],"E094":[{"start":"1861-03-17","
end":"9999-12-31","name":"GORGONZOLA"}],"G907":[{"name":"PORTO VALTRAVAGLIA"
,"end":"1927-01-11","start":"1863-07-13"},{"start":"1927-01-12","end":"1928-
08-15","name":"PORTO VALTRAVAGLIA"},{"name":"PORTO VALTRAVAGLIA","start":"19
28-08-16","end":"9999-12-31"}],"I751":[{"end":"1927-01-11","start":"1861-03-
17","name":"SINISCOLA"},{"name":"SINISCOLA","end":"9999-12-31","start":"1927
-01-12"}],"G799":[{"name":"POLLUTRI","start":"1861-03-17","end":"9999-12-31"
}],"M353":[{"start":"2016-01-01","end":"9999-12-31","name":"BORGO LARES"}],"
L273":[{"end":"9999-12-31","start":"1861-03-17","name":"TORREMAGGIORE"}],"M0
59":[{"name":"VINCI","start":"1861-03-17","end":"9999-12-31"}],"F810":[{"end
":"9999-12-31","start":"1889-07-15","name":"SALCEDO"}],"B885":[{"name":"CASA
LE MONFERRATO","start":"1861-03-17","end":"9999-12-31"}],"L731":[{"name":"VE
NDROGNO","end":"1992-04-15","start":"1861-03-17"},{"start":"1992-04-16","end
":"2019-12-31","name":"VENDROGNO"}],"G304":[{"start":"1873-05-10","end":"999
9-12-31","name":"PANCARANA"}],"C156":[{"name":"CASTELLETTO D'ERRO","start":"
1861-03-17","end":"9999-12-31"}],"B937":[{"name":"CASARGO","end":"1928-05-24
","start":"1861-03-17"},{"name":"CASARGO","start":"1928-05-25","end":"1992-0
4-15"},{"name":"CASARGO","end":"9999-12-31","start":"1992-04-16"}],"B178":[{
"end":"9999-12-31","start":"1863-07-13","name":"BRIGNANO GERA D'ADDA"}],"I47
9":[{"name":"SAVOGNA","end":"1923-02-07","start":"1921-01-05"},{"start":"192
3-02-08","end":"1923-04-26","name":"SAVOGNA"},{"end":"1927-01-11","start":"1
923-04-27","name":"SAVOGNA D'ISONZO"},{"name":"SAVOGNA D'ISONZO","end":"1927
-12-28","start":"1927-01-12"},{"name":"SAVOGNA D'ISONZO","end":"2009-01-28",
"start":"1951-02-15"},{"name":"SAVOGNA D'ISONZO-SOVODNJE OB SO\u010cI","star
t":"2009-01-29","end":"9999-12-31"}],"H035":[{"end":"1923-02-13","start":"19
20-10-16","name":"PREMIONE"},{"start":"1923-02-14","end":"1928-03-12","name"
:"PREMIONE"}],"D795":[{"start":"1861-03-17","end":"1927-11-20","name":"FRERO
LA"}],"C455":[{"name":"CENATE D'ARGON","start":"1929-05-18","end":"1947-06-2
3"}],"L513":[{"name":"USSARAMANNA","start":"1861-03-17","end":"1927-11-03"},
{"name":"USSARAMANNA","start":"1946-02-28","end":"1974-08-19"},{"name":"USSA
RAMANNA","end":"2005-12-31","start":"1974-08-20"},{"name":"USSARAMANNA","sta
rt":"2006-01-01","end":"2016-04-27"},{"name":"USSARAMANNA","start":"2016-04-
28","end":"9999-12-31"}],"G257":[{"name":"PALATA","start":"1861-03-17","end"
:"1970-03-02"},{"name":"PALATA","start":"1970-03-03","end":"9999-12-31"}],"B
334":[{"name":"CADERO CON GRAGLIO","start":"1877-11-08","end":"1927-01-11"},
{"name":"CADERO CON GRAGLIO","start":"1927-01-12","end":"1928-08-06"}],"M279
":[{"name":"PRIOLO GARGALLO","start":"1979-08-10","end":"9999-12-31"}],"L359
":[{"start":"1866-11-19","end":"1928-04-06","name":"TRECENTA"},{"name":"TREC
ENTA","start":"1928-04-07","end":"9999-12-31"}],"L053":[{"end":"1927-01-11",
"start":"1923-04-27","name":"TARNOVA DELLA SELVA"},{"end":"1947-09-15","star
t":"1927-01-12","name":"TARNOVA DELLA SELVA"},{"name":"TERNOVA","start":"192
1-01-05","end":"1923-02-07"},{"name":"TERNOVA","end":"1923-04-26","start":"1
923-02-08"}],"B922":[{"end":"1927-01-11","start":"1861-03-17","name":"CASAMA
RCIANO"},{"start":"1927-01-12","end":"1928-09-07","name":"CASAMARCIANO"},{"n
ame":"CASAMARCIANO","start":"1928-09-08","end":"1946-07-02"},{"end":"9999-12
-31","start":"1946-07-03","name":"CASAMARCIANO"}],"H750":[{"start":"1920-10-
16","end":"1923-02-13","name":"SAMOCLEVO"},{"start":"1923-02-14","end":"1928
-12-31","name":"SAMOCLEVO"}],"B474":[{"name":"CAMERINO","start":"1861-03-17"
,"end":"9999-12-31"}],"I766":[{"name":"SIVIANO","end":"1928-02-15","start":"
1861-03-17"}],"C161":[{"name":"CASTELLETTO MOLINA","start":"1861-03-17","end
":"1935-04-07"},{"start":"1935-04-08","end":"9999-12-31","name":"CASTELLETTO
 MOLINA"}],"I033":[{"start":"1863-02-16","end":"1928-02-21","name":"SAN MICH
ELE D'ASTI"}],"I339":[{"name":"SANTI COSMA E DAMIANO","end":"1927-01-11","st
art":"1861-03-17"},{"start":"1927-01-12","end":"1928-10-28","name":"SANTI CO
SMA E DAMIANO"},{"name":"SANTI COSMA E DAMIANO","end":"1984-03-05","start":"
1947-05-23"},{"end":"9999-12-31","start":"1984-03-06","name":"SANTI COSMA E 
DAMIANO"}],"A126":[{"name":"ALBAGIARA","start":"1964-06-09","end":"1974-08-1
9"},{"name":"ALBAGIARA","start":"1974-08-20","end":"9999-12-31"},{"end":"196
4-06-08","start":"1959-04-08","name":"OLLASTA"},{"name":"OLLASTRA USELLUS","
end":"1927-10-09","start":"1861-03-17"}],"D480":[{"name":"FALLO","end":"1928
-11-09","start":"1861-03-17"},{"name":"FALLO","end":"9999-12-31","start":"19
64-04-09"}],"L787":[{"end":"9999-12-31","start":"1863-01-05","name":"VERRUA 
SAVOIA"}],"F657":[{"name":"MONTEVECCHIA","start":"1861-03-17","end":"1928-05
-22"},{"start":"1966-09-22","end":"1992-04-15","name":"MONTEVECCHIA"},{"end"
:"9999-12-31","start":"1992-04-16","name":"MONTEVECCHIA"}],"F913":[{"name":"
NOCERA SUPERIORE","start":"1861-03-17","end":"9999-12-31"}],"H328":[{"start"
:"1862-11-10","end":"1928-03-06","name":"RIVA LIGURE"},{"name":"RIVA LIGURE"
,"start":"1954-09-18","end":"9999-12-31"}],"B778":[{"name":"CARIMATE","start
":"1861-03-17","end":"1928-10-29"},{"start":"1928-10-30","end":"1928-12-01",
"name":"CARIMATE"},{"name":"CARIMATE","start":"1928-12-02","end":"1950-04-04
"},{"name":"CARIMATE","start":"1950-04-05","end":"9999-12-31"}],"C349":[{"st
art":"1861-03-17","end":"9999-12-31","name":"CASTROVILLARI"}],"D195":[{"end"
:"9999-12-31","start":"1862-12-29","name":"CUCCARO VETERE"}],"C043":[{"name"
:"CASTAGN\u00c8","end":"1923-02-13","start":"1920-10-16"},{"end":"1929-01-24
","start":"1923-02-14","name":"CASTAGN\u00c8"}],"F877":[{"start":"1861-03-17
","end":"1928-07-17","name":"NESSO"},{"end":"9999-12-31","start":"1928-07-18
","name":"NESSO"}],"E592":[{"name":"LIMENA","end":"9999-12-31","start":"1866
-11-19"}],"G550":[{"name":"PIAN D'ARTOGNE","start":"1929-02-21","end":"1957-
02-13"}],"E242":[{"start":"1863-02-06","end":"1928-01-25","name":"GUARDIA PI
EMONTESE"},{"start":"1945-02-22","end":"9999-12-31","name":"GUARDIA PIEMONTE
SE"}],"L445":[{"name":"TROFARELLO","start":"1861-03-17","end":"1956-04-24"},
{"name":"TROFARELLO","start":"1956-04-25","end":"9999-12-31"}],"D311":[{"sta
rt":"1920-10-16","end":"1923-02-13","name":"DOBBIACO"},{"end":"1927-01-11","
start":"1923-02-14","name":"DOBBIACO"},{"start":"1927-01-12","end":"1928-03-
12","name":"DOBBIACO"},{"end":"9999-12-31","start":"1928-03-13","name":"DOBB
IACO"}],"H150":[{"start":"1861-03-17","end":"9999-12-31","name":"RACCONIGI"}
],"I166":[{"name":"SAN SPERATE","start":"1861-03-17","end":"1974-08-19"},{"n
ame":"SAN SPERATE","end":"2016-04-27","start":"1974-08-20"},{"name":"SAN SPE
RATE","start":"2016-04-28","end":"9999-12-31"}],"A505":[{"name":"AUZZA","end
":"1923-02-07","start":"1921-01-05"},{"end":"1927-01-11","start":"1923-02-08
","name":"AUZZA"},{"name":"AUZZA","end":"1928-03-12","start":"1927-01-12"}],
"C761":[{"name":"CIVIDATE MALEGNO","end":"1947-11-04","start":"1928-05-22"}]
,"A258":[{"name":"AMATRICE","end":"1927-01-11","start":"1861-03-17"},{"name"
:"AMATRICE","start":"1927-01-12","end":"9999-12-31"}],"A726":[{"name":"BAZZA
NO","end":"2013-12-31","start":"1861-03-17"}],"E635":[{"name":"LOCANA","end"
:"1927-01-11","start":"1861-03-17"},{"name":"LOCANA","end":"1945-10-04","sta
rt":"1927-01-12"},{"name":"LOCANA","start":"1945-10-05","end":"9999-12-31"}]
,"G010":[{"end":"1927-11-01","start":"1861-03-17","name":"OGLIANICO"},{"name
":"OGLIANICO","start":"1946-07-18","end":"9999-12-31"}],"L187":[{"name":"TOC
ENO","start":"1861-03-17","end":"1928-06-28"},{"start":"1947-08-03","end":"1
992-05-22","name":"TOCENO"},{"name":"TOCENO","start":"1992-05-23","end":"999
9-12-31"}],"E849":[{"name":"MALCHINA","start":"1923-04-27","end":"1928-09-08
"},{"start":"1921-01-05","end":"1923-02-07","name":"MANCHINIE"},{"end":"1923
-04-26","start":"1923-02-08","name":"MANCHINIE"}],"A908":[{"start":"1861-03-
17","end":"1927-01-11","name":"BOBBIATE"},{"name":"BOBBIATE","end":"1927-12-
28","start":"1927-01-12"}],"E103":[{"name":"GORLE","end":"9999-12-31","start
":"1861-03-17"}],"B026":[{"name":"BORGO PACE","end":"1998-09-03","start":"18
61-03-17"},{"name":"BORGO PACE","end":"9999-12-31","start":"1998-09-04"}],"C
589":[{"start":"1861-03-17","end":"1928-05-11","name":"CEVA"},{"name":"CEVA"
,"end":"1946-11-11","start":"1928-05-12"},{"start":"1946-11-12","end":"1947-
01-14","name":"CEVA"},{"start":"1947-01-15","end":"9999-12-31","name":"CEVA"
}],"H599":[{"name":"ROVEGNO","end":"1923-07-22","start":"1861-03-17"},{"star
t":"1923-07-23","end":"9999-12-31","name":"ROVEGNO"}],"F144":[{"name":"MERET
O DI TOMBA","start":"1940-02-11","end":"1968-04-05"},{"name":"MERETO DI TOMB
A","start":"1968-04-06","end":"2012-07-18"},{"name":"MERETO DI TOMBA","end":
"2014-02-03","start":"2012-07-19"},{"name":"MERETO DI TOMBA","end":"9999-12-
31","start":"2014-02-04"},{"name":"MERETTO DI TOMBA","end":"1940-02-10","sta
rt":"1866-11-19"}],"L146":[{"start":"1861-03-17","end":"1923-12-21","name":"
TERZORIO"},{"end":"1928-03-06","start":"1925-08-22","name":"TERZORIO"},{"end
":"9999-12-31","start":"1947-03-28","name":"TERZORIO"}],"I465":[{"name":"SAU
ZE DI CESANA","end":"1928-12-13","start":"1882-06-09"},{"name":"SAUZE DI CES
ANA","start":"1947-07-25","end":"1955-08-18"},{"name":"SAUZE DI CESANA","end
":"9999-12-31","start":"1955-08-19"}],"H694":[{"name":"SALE","start":"1861-0
3-17","end":"9999-12-31"}],"I151":[{"name":"SAN SEBASTIANO AL VESUVIO","star
t":"1877-04-09","end":"1953-06-17"},{"name":"SAN SEBASTIANO AL VESUVIO","sta
rt":"1953-06-18","end":"1981-04-20"},{"start":"1981-04-21","end":"9999-12-31
","name":"SAN SEBASTIANO AL VESUVIO"}],"H842":[{"name":"SAN FILIPPO DEL MELA
","start":"1861-03-17","end":"9999-12-31"}],"G199":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"OVIGLIO"}],"D067":[{"end":"2015-12-31","start":"186
1-03-17","name":"CORTEOLONA"}],"C228":[{"name":"CASTELNUOVO BOCCA D'ADDA","e
nd":"1992-04-15","start":"1861-03-17"},{"start":"1992-04-16","end":"9999-12-
31","name":"CASTELNUOVO BOCCA D'ADDA"}],"L131":[{"start":"1861-03-17","end":
"9999-12-31","name":"TERRASINI"}],"D496":[{"name":"FARA VICENTINO","end":"99
99-12-31","start":"1866-11-19"}],"H249":[{"name":"REVIANO FOLAS","end":"1923
-02-13","start":"1920-10-16"},{"name":"REVIANO FOLAS","start":"1923-02-14","
end":"1929-02-13"}],"C684":[{"name":"CIGOGNOLA","end":"9999-12-31","start":"
1861-03-17"}],"A078":[{"name":"AGNELLENGO","start":"1861-03-17","end":"1928-
06-13"}],"M265":[{"start":"1971-07-10","end":"9999-12-31","name":"VAJONT"}],
"F448":[{"name":"MONTECALVO IRPINO","end":"9999-12-31","start":"1862-11-10"}
],"C756":[{"start":"1920-10-16","end":"1923-02-13","name":"CIVEZZANO"},{"sta
rt":"1923-02-14","end":"9999-12-31","name":"CIVEZZANO"}],"G524":[{"start":"1
863-05-06","end":"9999-12-31","name":"PETTORANO SUL GIZIO"}],"F822":[{"name"
:"MUSEI","end":"1974-08-19","start":"1861-03-17"},{"start":"1974-08-20","end
":"2005-12-31","name":"MUSEI"},{"name":"MUSEI","start":"2006-01-01","end":"2
016-04-27"},{"name":"MUSEI","end":"9999-12-31","start":"2016-04-28"}],"C263"
:[{"name":"CASTEL SAN NICCOL\u00d2","end":"9999-12-31","start":"1868-04-01"}
],"D286":[{"end":"2009-06-29","start":"1861-03-17","name":"DESIO"},{"name":"
DESIO","start":"2009-06-30","end":"9999-12-31"}],"E675":[{"name":"LONGIANO",
"start":"1861-03-17","end":"1932-10-19"},{"end":"1992-04-15","start":"1932-1
0-20","name":"LONGIANO"},{"name":"LONGIANO","end":"9999-12-31","start":"1992
-04-16"}],"L988":[{"end":"9999-12-31","start":"1866-11-19","name":"VILLANOVA
 MARCHESANA"}],"C449":[{"name":"CELLINO ATTANASIO","end":"9999-12-31","start
":"1863-07-13"}],"H428":[{"start":"1861-03-17","end":"9999-12-31","name":"RO
CCAPALUMBA"}],"B813":[{"end":"1928-02-22","start":"1861-03-17","name":"CARPA
NZANO"},{"name":"CARPANZANO","start":"1937-07-18","end":"9999-12-31"}],"F229
":[{"end":"1926-08-23","start":"1868-01-01","name":"MIRA"},{"end":"1927-09-0
3","start":"1926-08-24","name":"MIRA"},{"name":"MIRA","end":"9999-12-31","st
art":"1927-09-04"}],"F588":[{"start":"1920-10-16","end":"1923-02-13","name":
"MONTEPONENTE"},{"name":"MONTEPONENTE","end":"1927-01-11","start":"1923-02-1
4"},{"end":"1928-09-08","start":"1927-01-12","name":"MONTEPONENTE"}],"F837":
[{"name":"NANNO","end":"1923-02-13","start":"1920-10-16"},{"name":"NANNO","e
nd":"1928-09-05","start":"1923-02-14"},{"start":"1947-03-15","end":"2015-12-
31","name":"NANNO"}],"B738":[{"end":"1974-08-19","start":"1862-09-29","name"
:"VILLASIMIUS"},{"name":"VILLASIMIUS","start":"1974-08-20","end":"2016-04-27
"},{"name":"VILLASIMIUS","start":"2016-04-28","end":"9999-12-31"}],"E062":[{
"name":"GIUSSAGO","start":"1872-11-01","end":"1929-02-07"},{"end":"9999-12-3
1","start":"1929-02-08","name":"GIUSSAGO"}],"A680":[{"name":"BARTESATE","end
":"1928-02-09","start":"1861-03-17"}],"H124":[{"start":"1866-11-19","end":"2
013-12-27","name":"QUERO"}],"G926":[{"name":"PORTO VIRO","end":"1938-02-10",
"start":"1932-12-18"},{"name":"PORTO VIRO","end":"9999-12-31","start":"1995-
01-01"},{"name":"TAGLIO DI PORTO VIRO","end":"1932-12-17","start":"1929-01-1
3"}],"G629":[{"start":"1861-03-17","end":"9999-12-31","name":"PIETRASTORNINA
"}],"D411":[{"name":"ENTRATICO","end":"1928-06-13","start":"1861-03-17"},{"e
nd":"1947-11-25","start":"1928-06-14","name":"ENTRATICO"},{"end":"9999-12-31
","start":"1947-11-26","name":"ENTRATICO"}],"L345":[{"name":"TRAVERSELLA","e
nd":"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"1929-04-
18","name":"TRAVERSELLA"},{"start":"1929-04-19","end":"1945-10-04","name":"T
RAVERSELLA"},{"start":"1945-10-05","end":"9999-12-31","name":"TRAVERSELLA"}]
,"I365":[{"name":"SANTO STEFANO AL MARE","start":"1861-03-17","end":"1923-12
-05"},{"end":"1925-08-21","start":"1923-12-06","name":"SANTO STEFANO AL MARE
"},{"end":"1928-03-06","start":"1925-08-22","name":"SANTO STEFANO AL MARE"},
{"name":"SANTO STEFANO AL MARE","start":"1954-09-18","end":"9999-12-31"}],"D
597":[{"end":"9999-12-31","start":"1861-03-17","name":"FILOTTRANO"}],"F985":
[{"end":"1928-06-26","start":"1861-03-17","name":"NURECI"},{"name":"NURECI",
"end":"1974-08-19","start":"1951-01-12"},{"name":"NURECI","end":"9999-12-31"
,"start":"1974-08-20"}],"G708":[{"name":"PISCOPIO","end":"1937-05-28","start
":"1861-03-17"}],"B910":[{"name":"CASALPUSTERLENGO","end":"1929-05-27","star
t":"1861-03-17"},{"start":"1929-05-28","end":"1992-04-15","name":"CASALPUSTE
RLENGO"},{"name":"CASALPUSTERLENGO","start":"1992-04-16","end":"9999-12-31"}
],"A038":[{"name":"ACQUANEGRA SUL CHIESE","start":"1868-07-01","end":"9999-1
2-31"}],"D625":[{"start":"1861-03-17","end":"1927-10-10","name":"FIUMENERO"}
],"L171":[{"start":"1861-03-17","end":"1927-01-11","name":"TINA"},{"name":"T
INA","start":"1927-01-12","end":"1928-03-07"}],"D396":[{"end":"1923-02-13","
start":"1920-10-16","name":"ELLE"},{"end":"1927-01-11","start":"1923-02-14",
"name":"ELLE"},{"name":"ELLE","start":"1927-01-12","end":"1928-12-14"}],"F34
8":[{"name":"MONDOLFO","start":"1861-03-17","end":"1998-09-03"},{"end":"9999
-12-31","start":"1998-09-04","name":"MONDOLFO"}],"D247":[{"end":"1984-12-14"
,"start":"1866-11-19","name":"DANTA"},{"start":"1984-12-15","end":"9999-12-3
1","name":"DANTA DI CADORE"}],"F881":[{"name":"NEVIANO","start":"1861-03-17"
,"end":"1928-08-23"},{"name":"NEVIANO","start":"1928-08-24","end":"9999-12-3
1"}],"A107":[{"name":"AILOCHE","end":"1927-01-11","start":"1861-03-17"},{"en
d":"1992-04-15","start":"1927-01-12","name":"AILOCHE"},{"name":"AILOCHE","en
d":"9999-12-31","start":"1992-04-16"}],"H069":[{"name":"PRIOLA","start":"186
1-03-17","end":"9999-12-31"}],"H363":[{"name":"ROASCIO","start":"1861-03-17"
,"end":"1928-05-11"},{"name":"ROASCIO","start":"1947-01-15","end":"9999-12-3
1"}],"Z612":[{"end":"9999-12-31","start":"0000-01-01","name":"Trinidad e Tob
ago"}],"M045":[{"start":"1927-11-03","end":"9999-12-31","name":"VILLONGO"}],
"L585":[{"name":"VALL'ALTA","start":"1861-03-17","end":"1928-02-15"}],"L233"
:[{"start":"1861-03-17","end":"9999-12-31","name":"TORRACA"}],"E966":[{"star
t":"1861-03-17","end":"1926-01-21","name":"MARO CASTELLO"}],"D290":[{"end":"
1928-02-22","start":"1861-03-17","name":"SCIGLIANO"},{"start":"1928-02-23","
end":"1937-07-17","name":"SCIGLIANO"},{"start":"1937-07-18","end":"9999-12-3
1","name":"SCIGLIANO"}],"E669":[{"end":"1928-08-07","start":"1861-03-17","na
me":"LONGANO"},{"start":"1934-02-25","end":"1970-03-02","name":"LONGANO"},{"
end":"9999-12-31","start":"1970-03-03","name":"LONGANO"}],"G108":[{"name":"O
RIO AL SERIO","start":"1863-07-13","end":"1954-11-25"},{"name":"ORIO AL SERI
O","end":"2010-02-12","start":"1954-11-26"},{"name":"ORIO AL SERIO","start":
"2010-02-13","end":"9999-12-31"}],"G022":[{"end":"9999-12-31","start":"1872-
11-16","name":"OLEVANO ROMANO"}],"A325":[{"start":"1864-03-11","end":"1928-0
7-29","name":"ANZOLA D'OSSOLA"},{"start":"1947-08-03","end":"1992-05-22","na
me":"ANZOLA D'OSSOLA"},{"start":"1992-05-23","end":"9999-12-31","name":"ANZO
LA D'OSSOLA"}],"I253":[{"start":"1861-03-17","end":"1928-04-01","name":"SANT
A MARINA"},{"start":"1946-02-28","end":"9999-12-31","name":"SANTA MARINA"}],
"L771":[{"name":"VERNANTE","start":"1861-03-17","end":"9999-12-31"}],"H255":
[{"name":"REZZAGO","start":"1861-03-17","end":"1928-04-02"},{"name":"REZZAGO
","end":"9999-12-31","start":"1947-10-01"}],"G404":[{"name":"PEDAVENA","end"
:"9999-12-31","start":"1866-11-19"}],"D540":[{"end":"9999-12-31","start":"18
61-03-17","name":"FERLA"}],"A696":[{"name":"BASELICE","end":"9999-12-31","st
art":"1861-03-17"}],"A999":[{"name":"BORGHETTO SANTO SPIRITO","end":"1927-01
-11","start":"1861-03-17"},{"name":"BORGHETTO SANTO SPIRITO","end":"9999-12-
31","start":"1927-01-12"}],"A707":[{"name":"BASSIANO","start":"1871-01-15","
end":"1934-10-25"},{"start":"1934-10-26","end":"1934-12-17","name":"BASSIANO
"},{"end":"1945-04-08","start":"1934-12-18","name":"BASSIANO"},{"name":"BASS
IANO","start":"1945-04-09","end":"9999-12-31"}],"I800":[{"end":"9999-12-31",
"start":"1861-03-17","name":"SOLETO"}],"G037":[{"start":"1863-05-06","end":"
1927-01-11","name":"OLIVETO LUCANO"},{"name":"OLIVETO LUCANO","end":"9999-12
-31","start":"1927-01-12"}],"E201":[{"end":"9999-12-31","start":"1861-03-17"
,"name":"GROSOTTO"}],"E564":[{"name":"LEVICE","start":"1861-03-17","end":"99
99-12-31"}],"H508":[{"end":"9999-12-31","start":"1861-03-17","name":"ROMANEN
GO"}],"B434":[{"name":"CALUSCO D'ADDA","start":"1883-12-17","end":"9999-12-3
1"}],"E951":[{"name":"MARIANO COMENSE","start":"1862-12-29","end":"9999-12-3
1"}],"Z313":[{"start":"0000-01-01","end":"9999-12-31","name":"Costa d'Avorio
"}],"B007":[{"end":"1962-03-09","start":"1898-07-01","name":"BORGO A MOZZANO
"},{"name":"BORGO A MOZZANO","start":"1962-03-10","end":"9999-12-31"}],"C355
":[{"name":"CATONA","start":"1867-01-25","end":"1927-08-02"}],"H662":[{"name
":"SAGLIANO MICCA","end":"1927-01-11","start":"1863-04-23"},{"name":"SAGLIAN
O MICCA","end":"1929-05-21","start":"1927-01-12"},{"name":"SAGLIANO MICCA","
end":"1992-04-15","start":"1948-03-21"},{"name":"SAGLIANO MICCA","start":"19
92-04-16","end":"9999-12-31"}],"H905":[{"name":"SAN GIOVANNI ALLA CASTAGNA",
"start":"1861-03-17","end":"1924-01-11"}],"I073":[{"end":"1927-01-11","start
":"1862-12-29","name":"SAN PAOLO BEL SITO"},{"end":"9999-12-31","start":"192
7-01-12","name":"SAN PAOLO BEL SITO"}],"I379":[{"end":"1923-02-13","start":"
1920-10-16","name":"SAN VALENTINO ALLA MUTA"},{"end":"1923-04-26","start":"1
923-02-14","name":"SAN VALENTINO ALLA MUTA"},{"name":"SAN VALENTINO ALLA MUT
TA","end":"1927-01-11","start":"1923-04-27"},{"start":"1927-01-12","end":"19
28-06-15","name":"SAN VALENTINO ALLA MUTTA"}],"E855":[{"name":"MALFA","start
":"1910-01-10","end":"9999-12-31"}],"D380":[{"end":"9999-12-31","start":"195
7-01-11","name":"DUGENTA"}],"I903":[{"name":"SPILAMBERTO","start":"1861-03-1
7","end":"9999-12-31"}],"B977":[{"name":"CASNATE CON BERNATE","end":"9999-12
-31","start":"1937-10-10"}],"B138":[{"name":"BREMBATE DI SOPRA","end":"1928-
01-06","start":"1861-03-17"},{"end":"9999-12-31","start":"1948-02-26","name"
:"BREMBATE DI SOPRA"}],"I439":[{"start":"1920-10-16","end":"1923-02-13","nam
e":"SARNONICO"},{"name":"SARNONICO","start":"1923-02-14","end":"1928-08-06"}
,{"name":"SARNONICO","start":"1953-01-04","end":"1964-06-09"},{"end":"9999-1
2-31","start":"1964-06-10","name":"SARNONICO"}],"B374":[{"end":"1928-04-14",
"start":"1861-03-17","name":"CALABRITTO"},{"name":"CALABRITTO","start":"1928
-04-15","end":"9999-12-31"}],"H724":[{"start":"1861-03-17","end":"1992-04-15
","name":"SALUDECIO"},{"end":"9999-12-31","start":"1992-04-16","name":"SALUD
ECIO"}],"H801":[{"end":"1992-04-15","start":"1869-12-10","name":"SAN CLEMENT
E"},{"end":"9999-12-31","start":"1992-04-16","name":"SAN CLEMENTE"}],"L459":
[{"name":"TUFILLO","end":"9999-12-31","start":"1861-03-17"}],"M160":[{"name"
:"ZELO SURRIGONE","end":"2019-02-07","start":"1861-03-17"}],"G081":[{"end":"
9999-12-31","start":"1933-07-30","name":"OPPIDO LUCANO"},{"name":"PALMIRA","
end":"1933-07-29","start":"1863-05-06"}],"E700":[{"end":"1927-01-11","start"
:"1861-03-17","name":"LOTZORAI"},{"end":"1928-05-12","start":"1927-01-12","n
ame":"LOTZORAI"},{"name":"LOTZORAI","end":"2005-12-31","start":"1946-03-25"}
,{"end":"2016-04-27","start":"2006-01-01","name":"LOTZORAI"},{"name":"LOTZOR
AI","start":"2016-04-28","end":"9999-12-31"}],"I224":[{"name":"SANTA MARGHER
ITA DI BELICE","start":"1863-01-19","end":"9999-12-31"}],"C165":[{"start":"1
871-07-10","end":"9999-12-31","name":"CASTELLETTO STURA"}],"M128":[{"name":"
VOLTORRE","end":"1927-01-11","start":"1861-03-17"},{"name":"VOLTORRE","start
":"1927-01-12","end":"1928-01-11"}],"I688":[{"start":"1869-03-15","end":"192
7-01-11","name":"SESTO CALENDE"},{"name":"SESTO CALENDE","start":"1927-01-12
","end":"1928-04-02"},{"name":"SESTO CALENDE","start":"1928-04-03","end":"19
29-06-12"},{"start":"1929-06-13","end":"9999-12-31","name":"SESTO CALENDE"}]
,"A482":[{"end":"9999-12-31","start":"1882-07-10","name":"ATELLA"}],"H046":[
{"name":"PRESEZZO","start":"1879-06-02","end":"1962-08-01"},{"start":"1962-0
8-02","end":"1964-02-29","name":"PRESEZZO"},{"name":"PRESEZZO","start":"1964
-03-01","end":"9999-12-31"}],"B588":[{"start":"1863-01-05","end":"1927-01-11
","name":"CANDIA CANAVESE"},{"name":"CANDIA CANAVESE","end":"1928-06-14","st
art":"1927-01-12"},{"start":"1928-06-15","end":"1945-10-04","name":"CANDIA C
ANAVESE"},{"name":"CANDIA CANAVESE","start":"1945-10-05","end":"9999-12-31"}
],"M002":[{"name":"VILLARBASSE","start":"1890-10-07","end":"1956-10-06"},{"s
tart":"1956-10-07","end":"9999-12-31","name":"VILLARBASSE"}],"A615":[{"end":
"9999-12-31","start":"1861-03-17","name":"BARAGIANO"}],"F738":[{"name":"MORN
ICO AL SERIO","end":"9999-12-31","start":"1863-07-13"}],"M424":[{"name":"VER
MEZZO CON ZELO","end":"9999-12-31","start":"2019-02-08"}],"B229":[{"name":"B
RUSNENGO","start":"1861-03-17","end":"1927-01-11"},{"end":"1992-04-15","star
t":"1927-01-12","name":"BRUSNENGO"},{"name":"BRUSNENGO","end":"9999-12-31","
start":"1992-04-16"}],"L270":[{"name":"TORREGLIA","end":"9999-12-31","start"
:"1866-11-19"}],"M350":[{"start":"2016-01-01","end":"9999-12-31","name":"ALT
OPIANO DELLA VIGOLANA"}],"F813":[{"name":"MURIALDO","end":"1927-01-11","star
t":"1861-03-17"},{"name":"MURIALDO","start":"1927-01-12","end":"9999-12-31"}
],"B822":[{"end":"9999-12-31","start":"1863-02-06","name":"CARPIGNANO SALENT
INO"}],"D791":[{"end":"1928-11-09","start":"1869-04-26","name":"FRATTE ROSA"
},{"name":"FRATTE ROSA","end":"1998-09-03","start":"1937-07-11"},{"name":"FR
ATTE ROSA","end":"9999-12-31","start":"1998-09-04"}],"L735":[{"start":"1861-
03-17","end":"1929-06-12","name":"VENETICO"},{"name":"VENETICO","end":"9999-
12-31","start":"1940-01-24"}],"B348":[{"start":"1882-06-12","end":"1929-01-1
3","name":"CA' EMO"}],"C779":[{"start":"1861-03-17","end":"1913-12-31","name
":"CIVITELLA CASANOVA"},{"start":"1914-01-01","end":"1927-01-11","name":"CIV
ITELLA CASANOVA"},{"name":"CIVITELLA CASANOVA","end":"9999-12-31","start":"1
927-01-12"}],"B881":[{"name":"CASALE CREMASCO-VIDOLASCO","end":"9999-12-31",
"start":"1934-03-07"}],"L302":[{"end":"9999-12-31","start":"1872-10-30","nam
e":"TORRITA TIBERINA"}],"H753":[{"end":"1927-01-11","start":"1861-03-17","na
me":"SAMONE"},{"start":"1927-01-12","end":"1945-10-04","name":"SAMONE"},{"na
me":"SAMONE","end":"9999-12-31","start":"1945-10-05"}],"C097":[{"name":"CAST
EL DI SASSO","end":"1927-01-11","start":"1863-07-13"},{"name":"CASTEL DI SAS
SO","start":"1927-01-12","end":"1945-08-31"},{"name":"CASTEL DI SASSO","end"
:"9999-12-31","start":"1945-09-01"}],"L050":[{"name":"TARCENTO","end":"1928-
04-28","start":"1866-11-19"},{"end":"1968-04-05","start":"1928-04-29","name"
:"TARCENTO"},{"end":"2004-03-31","start":"1968-04-06","name":"TARCENTO"},{"e
nd":"9999-12-31","start":"2004-04-01","name":"TARCENTO"}],"I030":[{"start":"
1863-01-05","end":"9999-12-31","name":"SAN MAURO TORINESE"}],"C466":[{"start
":"1861-03-17","end":"9999-12-31","name":"CENTALLO"}],"G807":[{"end":"9999-1
2-31","start":"1863-02-16","name":"POMARO MONFERRATO"}],"H087":[{"start":"19
48-07-11","end":"9999-12-31","name":"PUGLIANELLO"}],"F910":[{"name":"NOCERA 
TERINESE","start":"1863-02-06","end":"9999-12-31"}],"D483":[{"start":"1871-0
1-15","end":"1927-01-11","name":"FALVATERRA"},{"end":"9999-12-31","start":"1
927-01-12","name":"FALVATERRA"}],"B985":[{"end":"9999-12-31","start":"1861-0
3-17","name":"CASOLI"}],"A421":[{"end":"1927-01-11","start":"1871-01-15","na
me":"ARNARA"},{"name":"ARNARA","end":"9999-12-31","start":"1927-01-12"}],"L1
35":[{"start":"1861-03-17","end":"1928-06-26","name":"TERRAZZANO"}],"E687":[
{"name":"LORENZAGO","end":"1940-08-31","start":"1866-11-19"},{"end":"9999-12
-31","start":"1940-09-01","name":"LORENZAGO DI CADORE"}],"M261":[{"name":"CA
STRO","start":"1975-05-28","end":"9999-12-31"}],"L217":[{"start":"1861-03-17
","end":"1927-01-11","name":"TORGNON"},{"name":"TORGNON","end":"1939-10-20",
"start":"1927-01-12"},{"start":"1946-01-30","end":"9999-12-31","name":"TORGN
ON"},{"start":"1939-10-21","end":"1946-01-29","name":"TORGNONE"}],"M076":[{"
name":"VISINO","end":"1928-01-25","start":"1861-03-17"}],"C040":[{"end":"999
9-12-31","start":"1863-07-13","name":"CASTEL CASTAGNA"}],"G553":[{"name":"PI
ANE CRATI","start":"1863-01-19","end":"9999-12-31"}],"I155":[{"end":"9999-12
-31","start":"1861-03-17","name":"SANSEPOLCRO"}],"A808":[{"start":"1920-10-1
6","end":"1923-02-13","name":"BERSONE"},{"end":"1928-03-06","start":"1923-02
-14","name":"BERSONE"},{"name":"BERSONE","end":"2014-12-31","start":"1952-09
-20"}],"C179":[{"name":"CASTELLO VALSOLDA","end":"1927-06-09","start":"1862-
12-29"}],"H153":[{"end":"1911-06-13","start":"1861-03-17","name":"RADDA"},{"
name":"RADDA IN CHIANTI","start":"1911-06-14","end":"9999-12-31"}],"I461":[{
"end":"9999-12-31","start":"1861-03-17","name":"SASSOFERRATO"}],"H942":[{"na
me":"SAN GREGORIO DA SASSOLA","end":"9999-12-31","start":"1872-09-16"}],"G31
9":[{"start":"1955-02-15","end":"1958-02-15","name":"PAPARELLA SAN MARCO"},{
"start":"1958-02-16","end":"9999-12-31","name":"VALDERICE"}],"E100":[{"name"
:"GORLAGO","end":"9999-12-31","start":"1861-03-17"}],"F374":[{"name":"MONLEA
LE","end":"1929-01-23","start":"1861-03-17"},{"start":"1947-09-04","end":"19
47-12-02","name":"MONLEALE"},{"name":"MONLEALE","end":"9999-12-31","start":"
1947-12-03"}],"C765":[{"start":"1871-01-15","end":"1927-01-11","name":"CIVIT
A CASTELLANA"},{"name":"CIVITA CASTELLANA","start":"1927-01-12","end":"9999-
12-31"}],"L376":[{"start":"1921-01-05","end":"1923-02-07","name":"TRENTA D'I
SONZO"},{"start":"1923-02-08","end":"1927-01-11","name":"TRENTA D'ISONZO"},{
"name":"TRENTA D'ISONZO","end":"1928-02-20","start":"1927-01-12"}],"M256":[{
"start":"1974-08-02","end":"9999-12-31","name":"SCANZANO JONICO"}],"E631":[{
"start":"1861-03-17","end":"1927-06-01","name":"LIZZOLA"}],"D315":[{"name":"
DOGLIOLA","start":"1861-03-17","end":"9999-12-31"}],"A264":[{"name":"AMENO",
"start":"1861-03-17","end":"9999-12-31"}],"Z337":[{"name":"Zimbabwe","end":"
9999-12-31","start":"0000-01-01"}],"A501":[{"end":"1961-03-08","start":"1866
-11-19","name":"AURONZO"},{"name":"AURONZO DI CADORE","start":"1961-03-09","
end":"9999-12-31"}],"F977":[{"name":"NULVI","start":"1861-03-17","end":"1980
-07-10"},{"start":"1980-07-11","end":"9999-12-31","name":"NULVI"}],"F138":[{
"start":"1869-08-24","end":"1934-12-06","name":"MERCATO SAN SEVERINO"},{"end
":"9999-12-31","start":"1945-08-23","name":"MERCATO SAN SEVERINO"},{"name":"
SAN SEVERINO ROTA","start":"1934-12-07","end":"1935-10-29"},{"name":"SANSEVE
RINO ROTA","end":"1945-08-22","start":"1935-10-30"}],"I456":[{"name":"SASSO 
DI BORDIGHERA","end":"1928-04-03","start":"1862-11-10"}],"F007":[{"name":"MA
SCIAGO PRIMO","end":"1927-01-11","start":"1863-02-23"},{"start":"1927-01-12"
,"end":"1997-07-25","name":"MASCIAGO PRIMO"},{"name":"MASCIAGO PRIMO","start
":"1997-07-26","end":"9999-12-31"}],"E646":[{"name":"LOCULI","start":"1861-0
3-17","end":"1927-01-11"},{"name":"LOCULI","start":"1927-01-12","end":"1928-
02-01"},{"end":"9999-12-31","start":"1946-03-08","name":"LOCULI"}],"L577":[{
"name":"VALGANNA","end":"1927-01-11","start":"1861-03-17"},{"end":"9999-12-3
1","start":"1927-01-12","name":"VALGANNA"}],"E949":[{"start":"1868-07-01","e
nd":"1929-12-05","name":"MARIANA"},{"end":"9999-12-31","start":"1929-12-06",
"name":"MARIANA MANTOVANA"}],"H460":[{"name":"ROCCHETTA NERVINA","start":"18
62-11-10","end":"1947-09-15"},{"start":"1947-09-16","end":"9999-12-31","name
":"ROCCHETTA NERVINA"}],"F434":[{"name":"MONTE CREMASCO","end":"9999-12-31",
"start":"1862-12-19"}],"L436":[{"end":"1927-01-11","start":"1861-03-17","nam
e":"TRIVERO"},{"name":"TRIVERO","end":"1992-04-15","start":"1927-01-12"},{"s
tart":"1992-04-16","end":"2018-12-31","name":"TRIVERO"}],"L603":[{"end":"999
9-12-31","start":"1861-03-17","name":"VALLEDOLMO"}],"D191":[{"start":"1861-0
3-17","end":"1928-03-24","name":"CRUSINALLO"}],"Z322":[{"start":"0000-01-01"
,"end":"9999-12-31","name":"Kenya"}],"M041":[{"start":"1861-03-17","end":"99
99-12-31","name":"VILLETTA BARREA"}],"H618":[{"name":"ROVIANO","end":"9999-1
2-31","start":"1871-01-15"}],"I749":[{"end":"1928-05-31","start":"1861-03-17
","name":"SINI"},{"name":"SINI","end":"1974-08-19","start":"1946-02-07"},{"s
tart":"1974-08-20","end":"9999-12-31","name":"SINI"}],"L581":[{"name":"VALGR
EGHENTINO","start":"1861-03-17","end":"1927-10-08"},{"name":"VALGREGHENTINO"
,"end":"1992-04-15","start":"1927-10-09"},{"name":"VALGREGHENTINO","end":"99
99-12-31","start":"1992-04-16"}],"F885":[{"name":"NIBBIANO","start":"1861-03
-17","end":"1929-02-13"},{"end":"1938-02-09","start":"1929-02-14","name":"NI
BBIANO"},{"end":"1950-02-18","start":"1938-02-10","name":"NIBBIANO"},{"name"
:"NIBBIANO","start":"1950-02-19","end":"2017-12-31"}],"B810":[{"name":"CAROV
ILLI","start":"1861-03-17","end":"1970-03-02"},{"end":"9999-12-31","start":"
1970-03-03","name":"CAROVILLI"}],"B744":[{"name":"CARBONERA","end":"9999-12-
31","start":"1866-11-19"}],"M036":[{"name":"VILLAZZANO","end":"1923-02-13","
start":"1920-10-16"},{"name":"VILLAZZANO","end":"1926-09-30","start":"1923-0
2-14"}],"G419":[{"name":"PEIO","end":"1923-02-13","start":"1920-10-16"},{"na
me":"PEIO","start":"1923-02-14","end":"1928-08-11"},{"end":"9999-12-31","sta
rt":"1928-08-12","name":"PEIO"}],"L175":[{"start":"1875-06-11","end":"9999-1
2-31","name":"TIRANO"}],"D621":[{"name":"FIUME","end":"1911-10-31","start":"
1866-11-19"},{"end":"1968-04-05","start":"1911-11-01","name":"FIUME VENETO"}
,{"start":"1968-04-06","end":"9999-12-31","name":"FIUME VENETO"}],"A683":[{"
end":"1992-04-15","start":"1861-03-17","name":"BARZAGO"},{"start":"1992-04-1
6","end":"9999-12-31","name":"BARZAGO"}],"C139":[{"start":"1869-03-11","end"
:"1927-04-13","name":"CASTELLANZA"},{"end":"9999-12-31","start":"1927-04-14"
,"name":"CASTELLANZA"}],"I361":[{"name":"SANTO STEFANO TICINO","start":"1862
-12-01","end":"1997-07-25"},{"end":"9999-12-31","start":"1997-07-26","name":
"SANTO STEFANO TICINO"}],"D116":[{"end":"1923-02-13","start":"1920-10-16","n
ame":"COSTASAVINA"},{"start":"1923-02-14","end":"1929-01-24","name":"COSTASA
VINA"}],"Z216":[{"end":"9999-12-31","start":"0000-01-01","name":"Filippine"}
],"F981":[{"name":"NURAGUS","start":"1861-03-17","end":"1927-01-11"},{"start
":"1927-01-12","end":"2005-12-31","name":"NURAGUS"},{"start":"2006-01-01","e
nd":"2016-04-27","name":"NURAGUS"},{"end":"9999-12-31","start":"2016-04-28",
"name":"NURAGUS"}],"L888":[{"name":"VIGNOLO","end":"9999-12-31","start":"186
1-03-17"}],"D682":[{"name":"FONTECHIARI","end":"1927-01-11","start":"1862-10
-27"},{"name":"FONTECHIARI","start":"1927-01-12","end":"9999-12-31"}],"L537"
:[{"name":"VAIANO","start":"1949-09-08","end":"1984-08-15"},{"name":"VAIANO"
,"end":"1992-04-15","start":"1984-08-16"},{"name":"VAIANO","start":"1992-04-
16","end":"2007-11-14"},{"start":"2007-11-15","end":"9999-12-31","name":"VAI
ANO"}],"L341":[{"end":"1910-09-08","start":"1861-03-17","name":"TRAVEDONA"},
{"start":"1910-09-09","end":"1927-01-11","name":"TRAVEDONA"},{"name":"TRAVED
ONA","start":"1927-01-12","end":"1927-09-03"}],"D415":[{"start":"1951-01-25"
,"end":"9999-12-31","name":"ERACLEA"},{"start":"1866-11-19","end":"1951-01-2
4","name":"GRISOLERA"}],"B657":[{"end":"1946-02-28","start":"1938-04-28","na
me":"CAPITELLO"},{"name":"POLICASTRO DEL GOLFO","end":"1938-04-27","start":"
1928-04-01"}],"F922":[{"name":"NOGAROLE VICENTINO","end":"9999-12-31","start
":"1867-09-02"}],"I557":[{"start":"1921-01-05","end":"1923-02-07","name":"SE
BREGHE"},{"end":"1927-01-11","start":"1923-02-08","name":"SEBREGHE"},{"name"
:"SEBREGHE","start":"1927-01-12","end":"1928-07-28"}],"F474":[{"start":"1861
-03-17","end":"1998-09-03","name":"MONTECICCARDO"},{"end":"2020-06-30","star
t":"1998-09-04","name":"MONTECICCARDO"}],"A044":[{"end":"1957-12-14","start"
:"1865-12-25","name":"ACQUASANTA"},{"name":"ACQUASANTA TERME","start":"1957-
12-15","end":"9999-12-31"}],"L476":[{"end":"1918-02-14","start":"1864-03-11"
,"name":"TURRO MILANESE"}],"L522":[{"name":"UZZANO","start":"1895-05-10","en
d":"1928-01-25"},{"start":"1928-01-26","end":"1963-04-08","name":"UZZANO"},{
"end":"1991-05-14","start":"1963-04-09","name":"UZZANO"},{"start":"1991-05-1
5","end":"9999-12-31","name":"UZZANO"}],"G826":[{"name":"PONT CANAVESE","sta
rt":"1861-03-17","end":"1927-01-11"},{"end":"1945-10-04","start":"1927-01-12
","name":"PONT CANAVESE"},{"name":"PONT CANAVESE","start":"1945-10-05","end"
:"9999-12-31"}],"E671":[{"end":"9999-12-31","start":"1866-11-19","name":"LON
GARE"}],"F334":[{"end":"1927-01-11","start":"1861-03-17","name":"MONATE"},{"
end":"1927-09-03","start":"1927-01-12","name":"MONATE"}],"L336":[{"end":"199
2-05-22","start":"1861-03-17","name":"TRASQUERA"},{"end":"9999-12-31","start
":"1992-05-23","name":"TRASQUERA"}],"L769":[{"name":"VERMIGLIO","start":"192
0-10-16","end":"1923-02-13"},{"start":"1923-02-14","end":"9999-12-31","name"
:"VERMIGLIO"}],"F937":[{"name":"NORMA","start":"1871-01-15","end":"1934-10-2
5"},{"end":"1934-12-17","start":"1934-10-26","name":"NORMA"},{"name":"NORMA"
,"start":"1934-12-18","end":"1945-04-08"},{"start":"1945-04-09","end":"9999-
12-31","name":"NORMA"}],"H360":[{"name":"RO","end":"2018-12-31","start":"190
9-01-23"}],"I356":[{"name":"SANTO STEFANO QUISQUINA","start":"1863-01-19","e
nd":"9999-12-31"}],"I017":[{"end":"1929-04-20","start":"1863-02-16","name":"
SAN MARZANO OLIVETO"},{"start":"1947-06-23","end":"9999-12-31","name":"SAN M
ARZANO OLIVETO"}],"L230":[{"name":"TORO","end":"1970-03-02","start":"1861-03
-17"},{"name":"TORO","start":"1970-03-03","end":"9999-12-31"}],"H805":[{"nam
e":"SAN CONO","end":"9999-12-31","start":"1861-03-17"}],"A382":[{"end":"1929
-04-19","start":"1861-03-17","name":"ARDENNO"},{"start":"1929-04-20","end":"
1964-01-27","name":"ARDENNO"},{"start":"1964-01-28","end":"1999-07-26","name
":"ARDENNO"},{"name":"ARDENNO","start":"1999-07-27","end":"9999-12-31"}],"G0
85":[{"start":"1861-03-17","end":"1928-01-26","name":"ORASSO"}],"D293":[{"en
d":"1923-12-21","start":"1861-03-17","name":"DIANO ARENTINO"},{"name":"DIANO
 ARENTINO","start":"1925-08-22","end":"9999-12-31"}],"I250":[{"end":"1928-06
-28","start":"1861-03-17","name":"SANTA MARIA MAGGIORE E CRANA"}],"E851":[{"
name":"MALEGNO","start":"1861-03-17","end":"1928-05-21"},{"start":"1947-11-0
4","end":"9999-12-31","name":"MALEGNO"}],"H071":[{"name":"PROCENO","start":"
1871-01-15","end":"1927-01-11"},{"name":"PROCENO","start":"1927-01-12","end"
:"9999-12-31"}],"H266":[{"start":"1861-03-17","end":"1927-01-11","name":"RIA
LTO"},{"name":"RIALTO","end":"9999-12-31","start":"1927-01-12"}],"C351":[{"s
tart":"1861-03-17","end":"1926-03-17","name":"CATANIA"},{"end":"9999-12-31",
"start":"1926-03-18","name":"CATANIA"}],"H901":[{"name":"SAN GIOVANNI VALDAR
NO","start":"1863-04-13","end":"9999-12-31"}],"D543":[{"start":"1861-03-17",
"end":"1927-01-11","name":"FERNO"},{"start":"1927-01-12","end":"9999-12-31",
"name":"FERNO"}],"E955":[{"end":"1927-01-11","start":"1861-03-17","name":"MA
RIGLIANO"},{"name":"MARIGLIANO","end":"9999-12-31","start":"1927-01-12"}],"L
169":[{"name":"TIGNALE","start":"1861-03-17","end":"1929-01-04"},{"name":"TI
GNALE","start":"1929-01-05","end":"9999-12-31"}],"I803":[{"start":"1861-03-1
7","end":"9999-12-31","name":"SOLIGNANO"}],"B877":[{"name":"CASALE DI SCODOS
IA","end":"9999-12-31","start":"1867-08-26"}],"M324":[{"end":"9999-12-31","s
tart":"2014-01-01","name":"POGGIO TORRIANA"}],"Z502":[{"name":"Bahamas","end
":"9999-12-31","start":"0000-01-01"}],"E205":[{"name":"GROTTAGLIE","end":"19
23-12-21","start":"1861-03-17"},{"end":"9999-12-31","start":"1923-12-22","na
me":"GROTTAGLIE"}],"L402":[{"start":"1866-11-19","end":"9999-12-31","name":"
TREVIGNANO"}],"C567":[{"name":"CESARA","start":"1861-03-17","end":"1928-02-2
0"},{"name":"CESARA","end":"1947-07-08","start":"1928-02-21"},{"name":"CESAR
A","start":"1947-07-09","end":"1955-03-11"},{"name":"CESARA","end":"1992-05-
22","start":"1955-03-12"},{"name":"CESARA","start":"1992-05-23","end":"9999-
12-31"}],"B448":[{"end":"9999-12-31","start":"1861-03-17","name":"CALVIGNASC
O"}],"A542":[{"start":"1861-03-17","end":"9999-12-31","name":"BADOLATO"}],"H
251":[{"name":"REVIGLIASCO TORINESE","start":"1863-01-05","end":"1928-12-29"
}],"Z310":[{"start":"0000-01-01","end":"9999-12-31","name":"Comore"}],"I070"
:[{"name":"SAN PAOLO DELLA VALLE","end":"1928-02-21","start":"1863-02-16"}],
"E866":[{"end":"1923-02-13","start":"1920-10-16","name":"MALOSCO"},{"end":"1
928-06-29","start":"1923-02-14","name":"MALOSCO"},{"name":"MALOSCO","end":"2
019-12-31","start":"1947-03-27"}],"E558":[{"start":"1861-03-17","end":"1927-
01-11","name":"LETTOMANOPPELLO"},{"name":"LETTOMANOPPELLO","start":"1927-01-
12","end":"9999-12-31"}],"I900":[{"start":"1920-10-16","end":"1923-02-13","n
ame":"SPIGNES"},{"name":"SPIGNES","end":"1923-04-26","start":"1923-02-14"},{
"name":"SPINGA","end":"1927-01-11","start":"1923-04-27"},{"start":"1927-01-1
2","end":"1929-05-30","name":"SPINGA"}],"B144":[{"name":"BRENNA","start":"18
61-03-17","end":"2009-08-22"},{"end":"9999-12-31","start":"2009-08-23","name
":"BRENNA"}],"A292":[{"name":"ANGIARI","start":"1866-11-19","end":"9999-12-3
1"}],"D383":[{"name":"DUINO AURISINA","end":"1947-09-15","start":"1928-09-09
"},{"name":"DUINO AURISINA","start":"1947-09-16","end":"2009-01-28"},{"name"
:"DUINO AURISINA-DEVIN NABRE\u017dINA","end":"9999-12-31","start":"2009-01-2
9"}],"A321":[{"end":"9999-12-31","start":"1861-03-17","name":"ANZI"}],"D089"
:[{"name":"COSOLETO","end":"9999-12-31","start":"1861-03-17"}],"C366":[{"nam
e":"CAVAGLIO SAN DONNINO","start":"1862-12-29","end":"1927-12-31"}],"L775":[
{"start":"1861-03-17","end":"1984-08-15","name":"VERNIO"},{"name":"VERNIO","
end":"1992-04-15","start":"1984-08-16"},{"name":"VERNIO","start":"1992-04-16
","end":"9999-12-31"}],"L024":[{"end":"9999-12-31","start":"1861-03-17","nam
e":"TAGGIA"}],"C739":[{"name":"CISTERNA D'ASTI","end":"1935-04-07","start":"
1863-02-16"},{"end":"1951-02-19","start":"1935-04-08","name":"CISTERNA D'AST
I"},{"name":"CISTERNA D'ASTI","start":"1951-02-20","end":"9999-12-31"}],"F02
6":[{"start":"1861-03-17","end":"2013-12-31","name":"MASSA FISCAGLIA"}],"E01
8":[{"end":"1942-09-22","start":"1939-12-28","name":"GIARRE RIPOSTO"},{"name
":"IONIA","start":"1942-09-23","end":"1945-10-25"}],"D716":[{"name":"FORNERO
","end":"1928-01-26","start":"1884-08-05"}],"M163":[{"name":"ZENSON DI PIAVE
","start":"1869-01-23","end":"9999-12-31"}],"A899":[{"end":"1927-01-11","sta
rt":"1861-03-17","name":"BIZZOZERO"},{"name":"BIZZOZERO","start":"1927-01-12
","end":"1927-12-28"}],"A909":[{"name":"BOBBIO","start":"1861-03-17","end":"
1923-07-22"},{"name":"BOBBIO","start":"1923-07-23","end":"1927-06-27"},{"sta
rt":"1927-06-28","end":"9999-12-31","name":"BOBBIO"}],"D853":[{"end":"1929-0
4-10","start":"1861-03-17","name":"GAGLIOLE"},{"name":"GAGLIOLE","start":"19
34-03-07","end":"9999-12-31"}],"G663":[{"name":"PIGNOLA","end":"9999-12-31",
"start":"1935-03-05"},{"end":"1928-03-13","start":"1863-01-19","name":"PIGNO
LA DI BASILICATA"}],"A797":[{"name":"BERGOGNA","end":"1923-02-07","start":"1
921-01-05"},{"name":"BERGOGNA","end":"1927-01-11","start":"1923-02-08"},{"st
art":"1927-01-12","end":"1928-01-06","name":"BERGOGNA"},{"name":"BERGOGNA","
end":"1947-09-15","start":"1928-01-07"}],"A606":[{"end":"9999-12-31","start"
:"1861-03-17","name":"BANARI"}],"G494":[{"name":"PESCO SANNITA","end":"9999-
12-31","start":"1948-01-18"},{"name":"PESCOLAMAZZA","start":"1861-03-17","en
d":"1948-01-17"}],"E477":[{"name":"LAUDES","end":"1923-02-13","start":"1920-
10-16"},{"end":"1927-01-11","start":"1923-02-14","name":"LAUDES"},{"end":"19
28-01-26","start":"1927-01-12","name":"LAUDES"}],"C588":[{"name":"CETRARO","
end":"9999-12-31","start":"1861-03-17"}],"A433":[{"end":"1927-01-11","start"
:"1861-03-17","name":"ARPINO"},{"end":"9999-12-31","start":"1927-01-12","nam
e":"ARPINO"}],"C837":[{"name":"COLFOSCO","end":"1923-02-13","start":"1920-10
-16"},{"name":"COLFOSCO","end":"1925-09-15","start":"1923-02-14"}],"H598":[{
"name":"ROVATO","start":"1861-03-17","end":"2000-04-13"},{"end":"9999-12-31"
,"start":"2000-04-14","name":"ROVATO"}],"E536":[{"start":"1861-03-17","end":
"1927-01-11","name":"LEONFORTE"},{"name":"LEONFORTE","start":"1927-01-12","e
nd":"9999-12-31"}],"B165":[{"end":"1923-02-13","start":"1920-10-16","name":"
BREZ"},{"name":"BREZ","end":"1928-12-31","start":"1923-02-14"},{"name":"BREZ
","end":"1947-01-09","start":"1929-01-01"},{"end":"2019-12-31","start":"1947
-01-10","name":"BREZ"}],"E291":[{"end":"9999-12-31","start":"1928-12-16","na
me":"IMPRUNETA"}],"E322":[{"name":"IPPLIS","start":"1866-11-19","end":"1928-
04-15"}],"Z100":[{"name":"Albania","start":"0000-01-01","end":"9999-12-31"}]
,"G198":[{"name":"OVARO","end":"1968-04-05","start":"1872-01-01"},{"start":"
1968-04-06","end":"9999-12-31","name":"OVARO"}],"E934":[{"end":"9999-12-31",
"start":"1861-03-17","name":"MARCIGNAGO"}],"D200":[{"start":"1861-03-17","en
d":"1927-01-11","name":"CUGLIERI"},{"name":"CUGLIERI","end":"1974-08-19","st
art":"1927-01-12"},{"start":"1974-08-20","end":"9999-12-31","name":"CUGLIERI
"}],"F263":[{"start":"1920-10-16","end":"1923-02-13","name":"MOENA"},{"name"
:"MOENA","start":"1923-02-14","end":"1928-10-06"},{"name":"MOENA","start":"1
928-10-07","end":"9999-12-31"}],"C822":[{"start":"1920-10-16","end":"1923-02
-13","name":"COGNOLA"},{"name":"COGNOLA","start":"1923-02-14","end":"1926-09
-30"}],"A140":[{"name":"ALBARETTO VALLE DI MACRA","start":"1863-03-30","end"
:"1928-05-30"}],"C229":[{"end":"9999-12-31","start":"1861-03-17","name":"CAS
TELNUOVO BORMIDA"}],"E541":[{"end":"9999-12-31","start":"1861-03-17","name":
"LERCARA FRIDDI"}],"A079":[{"end":"1927-01-11","start":"1861-03-17","name":"
AGNONA"},{"start":"1927-01-12","end":"1928-09-05","name":"AGNONA"}],"F449":[
{"start":"1863-03-30","end":"9999-12-31","name":"MONTECALVO VERSIGGIA"}],"E3
37":[{"start":"1861-03-17","end":"9999-12-31","name":"ISNELLO"}],"A373":[{"s
tart":"1870-04-07","end":"1923-12-21","name":"ARCOLA"},{"start":"1923-12-22"
,"end":"1928-12-12","name":"ARCOLA"},{"name":"ARCOLA","start":"1928-12-13","
end":"9999-12-31"}],"H248":[{"name":"REVERE","end":"2017-12-31","start":"186
6-11-19"}],"L720":[{"name":"VELLEZZO BELLINI","start":"1872-11-01","end":"99
99-12-31"}],"D427":[{"end":"1923-02-07","start":"1921-01-05","name":"ERSEL I
N MONTE"},{"start":"1923-02-08","end":"1927-01-11","name":"ERSEL IN MONTE"},
{"start":"1927-01-12","end":"1928-04-28","name":"ERSEL IN MONTE"}],"H329":[{
"name":"RIVA VALDOBBIA","end":"1927-01-11","start":"1862-12-29"},{"end":"201
8-12-31","start":"1927-01-12","name":"RIVA VALDOBBIA"}],"C881":[{"name":"COL
LI IN PUSTERIA","end":"1923-02-13","start":"1920-10-16"},{"name":"COLLI IN P
USTERIA","start":"1923-02-14","end":"1927-01-11"},{"start":"1927-01-12","end
":"1929-03-26","name":"COLLI IN PUSTERIA"}],"H023":[{"name":"PREGASINA","sta
rt":"1920-10-16","end":"1923-02-13"},{"end":"1925-09-14","start":"1923-02-14
","name":"PREGASINA"}],"B779":[{"name":"CARINARO","end":"1927-01-11","start"
:"1861-03-17"},{"name":"CARINARO","start":"1927-01-12","end":"1929-04-12"},{
"end":"9999-12-31","start":"1946-07-03","name":"CARINARO"}],"I993":[{"name":
"SUCCIVO","end":"1927-01-11","start":"1861-03-17"},{"end":"1928-05-29","star
t":"1927-01-12","name":"SUCCIVO"},{"end":"9999-12-31","start":"1946-04-26","
name":"SUCCIVO"}],"C348":[{"start":"1861-03-17","end":"9999-12-31","name":"C
ASTROREGIO"}],"D950":[{"name":"GAZZADA","start":"1861-03-17","end":"1927-01-
11"},{"end":"1927-06-30","start":"1927-01-12","name":"GAZZADA"}],"D372":[{"s
tart":"1861-03-17","end":"9999-12-31","name":"DRONERO"}],"H891":[{"start":"1
872-01-01","end":"1968-04-05","name":"SAN GIORGIO DELLA RICHINVELDA"},{"end"
:"1990-11-28","start":"1968-04-06","name":"SAN GIORGIO DELLA RICHINVELDA"},{
"name":"SAN GIORGIO DELLA RICHINVELDA","start":"1990-11-29","end":"9999-12-3
1"}],"D432":[{"end":"1927-10-09","start":"1861-03-17","name":"ESCOVEDU"}],"I
774":[{"start":"1919-08-04","end":"2009-07-11","name":"SMERILLO"},{"name":"S
MERILLO","start":"2009-07-12","end":"9999-12-31"}],"B466":[{"end":"1970-03-0
2","start":"1896-11-22","name":"SANT'ELENA SANNITA"},{"end":"9999-12-31","st
art":"1970-03-03","name":"SANT'ELENA SANNITA"}],"A852":[{"name":"BIBBONA","s
tart":"1906-04-18","end":"1925-11-29"},{"name":"BIBBONA","end":"9999-12-31",
"start":"1925-11-30"}],"E381":[{"start":"1861-03-17","end":"1970-03-02","nam
e":"JELSI"},{"start":"1970-03-03","end":"9999-12-31","name":"JELSI"}],"A259"
:[{"start":"1861-03-17","end":"9999-12-31","name":"AMBIVERE"}],"I025":[{"nam
e":"SAN MAURIZIO D'OPAGLIO","end":"1928-08-11","start":"1861-03-17"},{"name"
:"SAN MAURIZIO D'OPAGLIO","start":"1948-05-08","end":"9999-12-31"}],"E848":[
{"end":"9999-12-31","start":"1866-11-19","name":"MALCESINE"}],"H995":[{"name
":"SAN MARTINO AL CIMINO","start":"1872-10-02","end":"1927-01-11"},{"end":"1
928-05-18","start":"1927-01-12","name":"SAN MARTINO AL CIMINO"}],"B097":[{"n
ame":"BOVA","start":"1861-03-17","end":"1908-04-23"},{"end":"9999-12-31","st
art":"1908-04-24","name":"BOVA"}],"L120":[{"start":"1871-01-15","end":"1928-
09-24","name":"TERRACINA"},{"start":"1928-09-25","end":"1933-09-12","name":"
TERRACINA"},{"name":"TERRACINA","end":"1934-10-25","start":"1933-09-13"},{"n
ame":"TERRACINA","end":"1934-12-17","start":"1934-10-26"},{"name":"TERRACINA
","start":"1934-12-18","end":"1935-07-17"},{"name":"TERRACINA","end":"1937-0
5-28","start":"1935-07-18"},{"start":"1937-05-29","end":"1945-04-08","name":
"TERRACINA"},{"end":"9999-12-31","start":"1945-04-09","name":"TERRACINA"}],"
I478":[{"name":"SAVOGNA","end":"1968-04-05","start":"1866-11-19"},{"name":"S
AVOGNA","end":"9999-12-31","start":"1968-04-06"}],"B179":[{"name":"BRIGNANO-
FRASCATA","start":"1928-06-19","end":"1928-12-08"},{"name":"BRIGNANO-FRASCAT
A","start":"1928-12-09","end":"1947-11-02"},{"start":"1947-11-03","end":"999
9-12-31","name":"BRIGNANO-FRASCATA"}],"B040":[{"end":"1928-01-06","start":"1
861-03-17","name":"BORGOSATOLLO"},{"name":"BORGOSATOLLO","start":"1928-01-07
","end":"1955-11-25"},{"end":"9999-12-31","start":"1955-11-26","name":"BORGO
SATOLLO"}],"I886":[{"name":"SPARONE","end":"1927-01-11","start":"1861-03-17"
},{"name":"SPARONE","end":"1945-10-04","start":"1927-01-12"},{"name":"SPARON
E","end":"9999-12-31","start":"1945-10-05"}],"G362":[{"name":"PASTENA","star
t":"1861-03-17","end":"1927-01-11"},{"name":"PASTENA","end":"9999-12-31","st
art":"1927-01-12"}],"M278":[{"end":"2005-12-31","start":"1979-05-26","name":
"VILLAPERUCCIO"},{"name":"VILLAPERUCCIO","start":"2006-01-01","end":"2016-04
-27"},{"start":"2016-04-28","end":"9999-12-31","name":"VILLAPERUCCIO"}],"A06
5":[{"name":"AFRICO","start":"1861-03-17","end":"1978-06-09"},{"start":"1978
-06-10","end":"9999-12-31","name":"AFRICO"}],"I174":[{"start":"1864-03-11","
end":"9999-12-31","name":"SANTA CRISTINA GELA"}],"E410":[{"name":"LAGOSANTO"
,"end":"9999-12-31","start":"1861-03-17"}],"I338":[{"name":"SANTICOLO","star
t":"1861-03-17","end":"1927-09-15"}],"E623":[{"name":"LIVO","end":"1928-05-2
2","start":"1861-03-17"},{"end":"9999-12-31","start":"1950-04-05","name":"LI
VO"}],"A197":[{"start":"1863-02-16","end":"9999-12-31","name":"ALICE BEL COL
LE"}],"D306":[{"start":"1861-03-17","end":"1928-06-30","name":"DISSIMO"}],"B
765":[{"name":"CARENTINO","start":"1861-03-17","end":"1928-11-30"},{"end":"9
999-12-31","start":"1955-11-26","name":"CARENTINO"}],"Z700":[{"name":"Austra
lia","end":"9999-12-31","start":"0000-01-01"}],"G798":[{"name":"POLLONE","st
art":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"1992-04-1
5","name":"POLLONE"},{"end":"9999-12-31","start":"1992-04-16","name":"POLLON
E"}],"A740":[{"start":"1862-11-07","end":"1998-09-03","name":"BELFORTE ALL'I
SAURO"},{"name":"BELFORTE ALL'ISAURO","start":"1998-09-04","end":"9999-12-31
"}],"M058":[{"start":"1861-03-17","end":"1935-04-07","name":"VINCHIO"},{"end
":"9999-12-31","start":"1935-04-08","name":"VINCHIO"}],"I847":[{"end":"1927-
01-11","start":"1861-03-17","name":"SORDEVOLO"},{"name":"SORDEVOLO","start":
"1927-01-12","end":"1992-04-15"},{"end":"9999-12-31","start":"1992-04-16","n
ame":"SORDEVOLO"}],"D673":[{"start":"1861-03-17","end":"9999-12-31","name":"
FONTANELLATO"}],"F355":[{"start":"1861-03-17","end":"1928-06-27","name":"MON
ESIGLIO"},{"end":"1929-05-22","start":"1928-06-28","name":"MONESIGLIO"},{"en
d":"1946-07-02","start":"1929-05-23","name":"MONESIGLIO"},{"end":"1946-12-27
","start":"1946-07-03","name":"MONESIGLIO"},{"name":"MONESIGLIO","end":"9999
-12-31","start":"1946-12-28"}],"C810":[{"start":"1927-05-27","end":"1929-04-
11","name":"COCQUIO-TREVISAGO"},{"name":"COCQUIO-TREVISAGO","end":"9999-12-3
1","start":"1929-04-12"}],"I134":[{"end":"1926-02-05","start":"1863-04-23","
name":"SAN QUIRICO IN VAL POLCEVERA"}],"A595":[{"start":"1861-03-17","end":"
1927-09-11","name":"BALLABIO INFERIORE"}],"A627":[{"end":"1926-07-15","start
":"1866-11-19","name":"BARBARANO"},{"start":"1926-07-16","end":"1939-09-06",
"name":"BARBARANO VICENTINO"},{"name":"BARBARANO VICENTINO","start":"1939-09
-07","end":"1947-12-21"},{"name":"BARBARANO VICENTINO","end":"2018-02-16","s
tart":"1947-12-22"}],"F156":[{"name":"MESOLA","end":"1962-10-09","start":"18
61-03-17"},{"name":"MESOLA","start":"1962-10-10","end":"9999-12-31"}],"L154"
:[{"name":"TEULADA","end":"1974-08-19","start":"1861-03-17"},{"start":"1974-
08-20","end":"2016-04-27","name":"TEULADA"},{"start":"2016-04-28","end":"999
9-12-31","name":"TEULADA"}],"B260":[{"name":"BUIE D'ISTRIA","start":"1921-01
-05","end":"1923-02-07"},{"end":"1977-04-03","start":"1923-02-08","name":"BU
IE D'ISTRIA"}],"I378":[{"name":"SAN VALENTINO AL BRENNERO","start":"1920-10-
16","end":"1923-02-13"},{"start":"1923-02-14","end":"1927-01-11","name":"SAN
 VALENTINO AL BRENNERO"},{"end":"1929-02-14","start":"1927-01-12","name":"SA
N VALENTINO AL BRENNERO"}],"Z205":[{"start":"0000-01-01","end":"9999-12-31",
"name":"Bhutan"}],"E502":[{"start":"1866-11-19","end":"9999-12-31","name":"L
AZISE"}],"D105":[{"name":"COSTA DI ROVIGO","end":"9999-12-31","start":"1867-
07-22"}],"I438":[{"name":"SARNO","end":"9999-12-31","start":"1861-03-17"}],"
B139":[{"name":"BREMBATE DI SOTTO","end":"1928-08-14","start":"1861-03-17"}]
,"A245":[{"start":"1861-03-17","end":"1925-06-11","name":"ALZANO"},{"name":"
ALZANO SCRIVIA","start":"1925-06-12","end":"1928-06-18"},{"name":"ALZANO SCR
IVIA","end":"9999-12-31","start":"1946-07-18"}],"G843":[{"start":"1861-03-17
","end":"1928-11-06","name":"PONTEDERA"},{"name":"PONTEDERA","end":"1929-06-
17","start":"1928-11-07"},{"start":"1929-06-18","end":"1931-06-27","name":"P
ONTEDERA"},{"name":"PONTEDERA","end":"2007-04-03","start":"1931-06-28"},{"st
art":"2007-04-04","end":"9999-12-31","name":"PONTEDERA"}],"A632":[{"end":"19
84-08-15","start":"1861-03-17","name":"BARBERINO DI MUGELLO"},{"end":"9999-1
2-31","start":"1984-08-16","name":"BARBERINO DI MUGELLO"}],"E310":[{"start":
"1872-10-15","end":"1961-12-25","name":"INVERNO"},{"name":"INVERNO E MONTELE
ONE","start":"1961-12-26","end":"9999-12-31"}],"G462":[{"name":"PEROSA CANAV
ESE","start":"1863-01-05","end":"1927-01-11"},{"end":"1928-06-26","start":"1
927-01-12","name":"PEROSA CANAVESE"},{"name":"PEROSA CANAVESE","end":"9999-1
2-31","start":"1946-12-24"}],"L458":[{"end":"1970-03-02","start":"1861-03-17
","name":"TUFARA"},{"end":"9999-12-31","start":"1970-03-03","name":"TUFARA"}
],"C744":[{"name":"CITT\u00c0 DELLA PIEVE","start":"1861-03-17","end":"9999-
12-31"}],"E668":[{"name":"LONDA","end":"1926-01-21","start":"1861-03-17"},{"
name":"LONDA","start":"1926-01-22","end":"1929-03-08"},{"name":"LONDA","end"
:"1973-05-11","start":"1929-03-09"},{"name":"LONDA","end":"9999-12-31","star
t":"1973-05-12"}],"G109":[{"end":"1927-01-11","start":"1864-04-01","name":"O
RIO CANAVESE"},{"name":"ORIO CANAVESE","start":"1927-01-12","end":"1945-10-0
4"},{"end":"9999-12-31","start":"1945-10-05","name":"ORIO CANAVESE"}],"H604"
:[{"name":"ROVERBELLA","start":"1866-11-19","end":"9999-12-31"}],"L690":[{"n
ame":"VARZI","end":"1929-05-27","start":"1872-12-16"},{"end":"1946-12-18","s
tart":"1929-05-28","name":"VARZI"},{"start":"1946-12-19","end":"9999-12-31",
"name":"VARZI"}],"C913":[{"end":"1871-06-01","start":"1861-03-17","name":"CO
MAIRANO"}],"D406":[{"start":"1928-07-29","end":"1929-05-22","name":"ENDINE G
AIANO"},{"name":"ENDINE GAIANO","start":"1929-05-23","end":"2006-12-29"},{"e
nd":"9999-12-31","start":"2006-12-30","name":"ENDINE GAIANO"}],"C657":[{"nam
e":"CHIUSANICO","end":"1928-04-16","start":"1861-03-17"},{"name":"CHIUSANICO
","start":"1928-04-17","end":"9999-12-31"}],"D633":[{"name":"FLERES","end":"
1923-02-13","start":"1920-10-16"},{"name":"FLERES","start":"1923-02-14","end
":"1927-01-11"},{"name":"FLERES","end":"1929-02-14","start":"1927-01-12"}],"
F161":[{"end":"9999-12-31","start":"1866-11-19","name":"MESTRINO"}],"G940":[
{"name":"POSTUA","start":"1861-03-17","end":"1927-01-11"},{"end":"9999-12-31
","start":"1927-01-12","name":"POSTUA"}],"F455":[{"end":"1928-10-12","start"
:"1861-03-17","name":"MONTECASTELLO"},{"end":"9999-12-31","start":"1956-01-2
6","name":"MONTECASTELLO"}],"A998":[{"name":"BORGHETTO DI BORBERA","start":"
1861-03-17","end":"1928-06-27"},{"start":"1928-06-28","end":"1928-10-29","na
me":"BORGHETTO DI BORBERA"},{"end":"1929-04-11","start":"1928-10-30","name":
"BORGHETTO DI BORBERA"},{"end":"9999-12-31","start":"1929-04-12","name":"BOR
GHETTO DI BORBERA"}],"E193":[{"end":"9999-12-31","start":"1868-07-02","name"
:"GRONTARDO"}],"L712":[{"name":"VEGLIO","end":"1927-01-11","start":"1861-03-
17"},{"end":"1992-04-15","start":"1927-01-12","name":"VEGLIO"},{"start":"199
2-04-16","end":"9999-12-31","name":"VEGLIO"}],"H509":[{"end":"9999-12-31","s
tart":"1863-07-13","name":"ROMANO DI LOMBARDIA"}],"L225":[{"start":"1868-01-
01","end":"9999-12-31","name":"TORNATA"}],"E422":[{"end":"9999-12-31","start
":"1861-03-17","name":"LALLIO"}],"G997":[{"end":"1927-01-11","start":"1861-0
3-17","name":"PRATIGLIONE"},{"start":"1927-01-12","end":"1945-10-04","name":
"PRATIGLIONE"},{"name":"PRATIGLIONE","end":"9999-12-31","start":"1945-10-05"
}],"E974":[{"end":"1980-08-09","start":"1861-03-17","name":"MARSALA"},{"name
":"MARSALA","start":"1980-08-10","end":"9999-12-31"}],"G709":[{"name":"PISIN
O","end":"1922-02-01","start":"1921-01-05"},{"end":"1923-02-07","start":"192
2-02-02","name":"PISINO"},{"end":"1928-11-30","start":"1923-02-08","name":"P
ISINO"},{"name":"PISINO","start":"1928-12-01","end":"1947-09-15"}],"G394":[{
"name":"PAZZANO","start":"1861-03-17","end":"9999-12-31"}],"A039":[{"end":"9
999-12-31","start":"1868-01-01","name":"ACQUANEGRA CREMONESE"}],"A333":[{"na
me":"APPIANO","end":"1926-09-23","start":"1867-04-19"},{"end":"1927-11-01","
start":"1926-09-24","name":"APPIANO GENTILE"},{"name":"APPIANO GENTILE","sta
rt":"1927-11-02","end":"1950-04-04"},{"name":"APPIANO GENTILE","end":"9999-1
2-31","start":"1950-04-05"}],"E377":[{"name":"ITTIRI","end":"9999-12-31","st
art":"1861-03-17"}],"F761":[{"end":"1938-02-18","start":"1861-03-17","name":
"MOSCAZZANO"},{"name":"MOSCAZZANO","start":"1938-02-19","end":"9999-12-31"}]
,"H816":[{"name":"SAN DANIELE DEL FRIULI","end":"1928-02-15","start":"1867-0
9-02"},{"name":"SAN DANIELE DEL FRIULI","end":"1961-03-08","start":"1928-02-
16"},{"name":"SAN DANIELE DEL FRIULI","start":"1961-03-09","end":"1968-04-05
"},{"end":"9999-12-31","start":"1968-04-06","name":"SAN DANIELE DEL FRIULI"}
],"F043":[{"name":"SAN MARCO EVANGELISTA","start":"1862-11-10","end":"1904-0
2-25"},{"end":"1927-01-11","start":"1904-02-26","name":"SAN MARCO EVANGELIST
A"},{"end":"1928-06-27","start":"1927-01-12","name":"SAN MARCO EVANGELISTA"}
,{"name":"SAN MARCO EVANGELISTA","end":"9999-12-31","start":"1975-05-22"}],"
F349":[{"end":"1927-06-17","start":"1861-03-17","name":"MONDONICO"}],"E437":
[{"name":"LANDRIANO","end":"9999-12-31","start":"1872-11-01"}],"A473":[{"nam
e":"ASSAGO","start":"1861-03-17","end":"9999-12-31"}],"C877":[{"start":"1862
-11-24","end":"9999-12-31","name":"COLLI DEL TRONTO"}],"L112":[{"start":"186
3-04-10","end":"2011-08-11","name":"TERMINI IMERESE"},{"start":"2011-08-12",
"end":"9999-12-31","name":"TERMINI IMERESE"}],"E793":[{"end":"9999-12-31","s
tart":"1868-07-05","name":"MADIGNANO"}],"E004":[{"end":"1927-07-26","start":
"1861-03-17","name":"GHILARZA"},{"start":"1927-07-27","end":"1934-11-01","na
me":"GHILARZA"},{"end":"1958-08-13","start":"1934-11-02","name":"GHILARZA"},
{"name":"GHILARZA","start":"1958-08-15","end":"1974-08-19"},{"name":"GHILARZ
A","end":"1979-04-19","start":"1974-08-20"},{"start":"1979-04-20","end":"999
9-12-31","name":"GHILARZA"}],"H068":[{"name":"PRIOCCA","start":"1861-03-17",
"end":"9999-12-31"}],"E576":[{"name":"LICENZA","end":"9999-12-31","start":"1
899-07-26"}],"B351":[{"end":"9999-12-31","start":"1861-03-17","name":"CAGGIA
NO"}],"L647":[{"start":"1939-10-21","end":"1946-01-29","name":"VALSAVARA"},{
"name":"VALSAVARANCHE","end":"1927-01-11","start":"1861-03-17"},{"start":"19
27-01-12","end":"1939-10-20","name":"VALSAVARANCHE"},{"name":"VALSAVARANCHE"
,"start":"1946-01-30","end":"1977-01-01"},{"end":"9999-12-31","start":"1977-
01-02","name":"VALSAVARENCHE"}],"I734":[{"name":"SILIQUA","end":"1974-08-19"
,"start":"1861-03-17"},{"start":"1974-08-20","end":"2016-04-27","name":"SILI
QUA"},{"start":"2016-04-28","end":"9999-12-31","name":"SILIQUA"}],"D924":[{"
name":"GARGNANO","end":"9999-12-31","start":"1861-03-17"}],"I667":[{"name":"
SERRENTI","start":"1861-03-17","end":"1974-08-19"},{"end":"2005-12-31","star
t":"1974-08-20","name":"SERRENTI"},{"start":"2006-01-01","end":"2016-04-27",
"name":"SERRENTI"},{"name":"SERRENTI","end":"9999-12-31","start":"2016-04-28
"}],"D472":[{"end":"1928-05-22","start":"1863-02-23","name":"FALCONARA MARIT
TIMA"},{"start":"1948-04-10","end":"2010-03-01","name":"FALCONARA MARITTIMA"
},{"end":"9999-12-31","start":"2010-03-02","name":"FALCONARA MARITTIMA"}],"L
686":[{"name":"VARMO","start":"1866-11-19","end":"1968-04-05"},{"end":"9999-
12-31","start":"1968-04-06","name":"VARMO"}],"L989":[{"end":"1928-11-29","st
art":"1861-03-17","name":"VILLANOVA MONTELEONE"},{"end":"1947-10-25","start"
:"1928-11-30","name":"VILLANOVA MONTELEONE"},{"start":"1947-10-26","end":"99
99-12-31","name":"VILLANOVA MONTELEONE"}],"F756":[{"name":"MORTEGLIANO","end
":"1968-04-05","start":"1892-06-06"},{"name":"MORTEGLIANO","end":"2006-07-19
","start":"1968-04-06"},{"name":"MORTEGLIANO","end":"9999-12-31","start":"20
06-07-20"}],"C448":[{"end":"1927-01-11","start":"1862-11-28","name":"CELLINO
 SAN MARCO"},{"name":"CELLINO SAN MARCO","start":"1927-01-12","end":"9999-12
-31"}],"L754":[{"name":"VERDERIO","end":"1905-04-24","start":"1872-02-16"}],
"B567":[{"name":"CAMPOSPINOSO","start":"1861-03-17","end":"1928-11-09"},{"na
me":"CAMPOSPINOSO","end":"2008-08-15","start":"1948-01-20"},{"start":"2008-0
8-16","end":"9999-12-31","name":"CAMPOSPINOSO"}],"D327":[{"name":"DOLZAGO","
end":"1992-04-15","start":"1861-03-17"},{"end":"9999-12-31","start":"1992-04
-16","name":"DOLZAGO"}],"F228":[{"name":"MIONE","start":"1866-11-19","end":"
1871-12-31"}],"H429":[{"start":"1865-12-25","end":"9999-12-31","name":"ROCCA
 PIA"}],"E216":[{"name":"GRUGLIASCO","end":"9999-12-31","start":"1861-03-17"
}],"F589":[{"name":"MONTE PORZIO","end":"1998-09-03","start":"1861-03-17"},{
"name":"MONTE PORZIO","start":"1998-09-04","end":"9999-12-31"}],"D705":[{"na
me":"FORLIMPOPOLI","end":"1992-04-15","start":"1861-03-17"},{"start":"1992-0
4-16","end":"9999-12-31","name":"FORLIMPOPOLI"}],"E481":[{"name":"LAUREGNO",
"start":"1920-10-16","end":"1923-02-13"},{"name":"LAUREGNO","end":"1928-09-0
6","start":"1923-02-14"},{"name":"LAUREGNO","start":"1948-03-14","end":"9999
-12-31"}],"B739":[{"end":"2018-12-31","start":"1867-06-24","name":"CARBONARA
 DI PO"}],"G628":[{"name":"PIETRASANTA","start":"1861-03-17","end":"1913-05-
10"},{"name":"PIETRASANTA","end":"9999-12-31","start":"1913-05-11"}],"D526":
[{"name":"FELINO","start":"1861-03-17","end":"9999-12-31"}],"B366":[{"end":"
9999-12-31","start":"1861-03-17","name":"CAIOLO"}],"G284":[{"name":"PALMANOV
A","start":"1866-11-19","end":"1968-04-05"},{"start":"1968-04-06","end":"999
9-12-31","name":"PALMANOVA"}],"D332":[{"start":"1884-07-04","end":"1928-05-0
3","name":"DOMODOSSOLA"},{"name":"DOMODOSSOLA","end":"1950-09-08","start":"1
928-05-04"},{"name":"DOMODOSSOLA","start":"1950-09-09","end":"1959-03-17"},{
"start":"1959-03-18","end":"1992-05-22","name":"DOMODOSSOLA"},{"start":"1992
-05-23","end":"9999-12-31","name":"DOMODOSSOLA"}],"C144":[{"end":"1928-07-27
","start":"1861-03-17","name":"CASTELLARO DE' GIORGI"}],"B068":[{"end":"1927
-01-11","start":"1861-03-17","name":"BOSA"},{"start":"1927-01-12","end":"199
7-02-06","name":"BOSA"},{"name":"BOSA","start":"1997-02-07","end":"2005-12-3
1"},{"name":"BOSA","start":"2006-01-01","end":"9999-12-31"}],"E732":[{"end":
"1923-02-07","start":"1921-01-05","name":"LUICO"},{"end":"1927-01-11","start
":"1923-02-08","name":"LUICO"},{"start":"1927-01-12","end":"1928-05-01","nam
e":"LUICO"}],"G923":[{"end":"1929-01-12","start":"1873-04-10","name":"PORTO 
TOLLE"},{"start":"1929-01-13","end":"1936-09-28","name":"PORTO TOLLE"},{"nam
e":"PORTO TOLLE","end":"1938-02-09","start":"1936-09-29"},{"start":"1938-02-
10","end":"9999-12-31","name":"PORTO TOLLE"}],"H351":[{"name":"RIVIERA D'ADD
A","start":"1928-04-03","end":"1970-03-17"}],"D110":[{"name":"COSTA DI MEZZA
TE","start":"1861-03-17","end":"1927-08-18"},{"name":"COSTA DI MEZZATE","end
":"9999-12-31","start":"1964-06-27"},{"name":"COSTA DI MONTICELLI","start":"
1927-08-19","end":"1955-05-13"},{"name":"COSTA DI MONTICELLI","end":"1964-06
-26","start":"1955-05-14"}],"C901":[{"name":"COLONNELLA","start":"1861-03-17
","end":"1963-04-27"},{"name":"COLONNELLA","start":"1963-04-28","end":"9999-
12-31"}],"Z210":[{"end":"9999-12-31","start":"0000-01-01","name":"Cina"}],"M
030":[{"name":"VILLAURBANA","end":"1928-05-03","start":"1861-03-17"},{"start
":"1928-05-04","end":"1947-12-12","name":"VILLAURBANA"},{"name":"VILLAURBANA
","start":"1947-12-13","end":"1974-08-19"},{"start":"1974-08-20","end":"9999
-12-31","name":"VILLAURBANA"}],"E305":[{"start":"1861-03-17","end":"1992-04-
15","name":"INTROBIO"},{"name":"INTROBIO","start":"1992-04-16","end":"9999-1
2-31"}],"E727":[{"name":"LUGNACCO","end":"1927-01-11","start":"1861-03-17"},
{"name":"LUGNACCO","end":"1945-10-04","start":"1927-01-12"},{"name":"LUGNACC
O","start":"1945-10-05","end":"2018-12-31"}],"B816":[{"start":"1861-03-17","
end":"1928-09-24","name":"CARPEGNA"},{"start":"1928-09-25","end":"1947-01-18
","name":"CARPEGNA"},{"end":"1998-09-03","start":"1947-01-19","name":"CARPEG
NA"},{"start":"1998-09-04","end":"9999-12-31","name":"CARPEGNA"}],"C805":[{"
start":"1923-04-27","end":"1927-01-11","name":"COBBIA"},{"name":"COBBIA","st
art":"1927-01-12","end":"1928-03-12"},{"start":"1921-01-05","end":"1923-02-0
7","name":"COBILAGLAVA"},{"name":"COBILAGLAVA","start":"1923-02-08","end":"1
923-04-26"}],"C071":[{"name":"CASTEL COLONNA","start":"1921-11-23","end":"20
13-12-31"},{"name":"TOMBA DI SENIGALLIA","start":"1861-03-17","end":"1921-11
-22"}],"C266":[{"end":"9999-12-31","start":"1872-11-24","name":"CASTEL SAN P
IETRO ROMANO"}],"G095":[{"name":"ORGIANO","end":"1909-01-22","start":"1866-1
1-19"},{"name":"ORGIANO","end":"9999-12-31","start":"1909-01-23"}],"A392":[{
"name":"ARGELATO","end":"9999-12-31","start":"1861-03-17"}],"D283":[{"end":"
1928-02-15","start":"1863-07-13","name":"DESENZANO AL SERIO"}],"A221":[{"sta
rt":"1861-03-17","end":"1927-01-11","name":"ALPETTE"},{"name":"ALPETTE","sta
rt":"1927-01-12","end":"1945-10-04"},{"end":"9999-12-31","start":"1945-10-05
","name":"ALPETTE"}],"H366":[{"name":"ROATTO","end":"1928-12-30","start":"18
61-03-17"},{"start":"1947-06-23","end":"9999-12-31","name":"ROATTO"}],"I424"
:[{"end":"9999-12-31","start":"1871-01-15","name":"SARACINESCO"}],"I972":[{"
name":"STRAMENTIZZO","end":"1923-02-13","start":"1920-10-16"},{"end":"1925-0
3-18","start":"1923-02-14","name":"STRAMENTIZZO"}],"I350":[{"start":"1861-03
-17","end":"1992-04-15","name":"SANT'ONOFRIO"},{"end":"9999-12-31","start":"
1992-04-16","name":"SANT'ONOFRIO"}],"A889":[{"name":"BISTAGNO","start":"1861
-03-17","end":"9999-12-31"}],"G820":[{"end":"1927-01-11","start":"1861-03-17
","name":"PONDERANO"},{"start":"1927-01-12","end":"1992-04-15","name":"PONDE
RANO"},{"name":"PONDERANO","end":"9999-12-31","start":"1992-04-16"}],"A282":
[{"name":"ANDRATE","end":"1927-01-11","start":"1861-03-17"},{"start":"1927-0
1-12","end":"1929-03-01","name":"ANDRATE"},{"name":"ANDRATE","end":"1945-10-
04","start":"1929-03-02"},{"name":"ANDRATE","start":"1945-10-05","end":"1954
-07-14"},{"start":"1954-07-15","end":"9999-12-31","name":"ANDRATE"}],"D099":
[{"name":"COSSOGNO","start":"1861-03-17","end":"1992-05-22"},{"start":"1992-
05-23","end":"9999-12-31","name":"COSSOGNO"}],"D393":[{"name":"ELENA","end":
"1927-01-11","start":"1897-04-01"},{"start":"1927-01-12","end":"1927-03-16",
"name":"ELENA"}],"H739":[{"end":"1928-06-29","start":"1861-03-17","name":"SA
MATZAI"},{"start":"1928-06-30","end":"1946-02-07","name":"SAMATZAI"},{"end":
"1974-08-19","start":"1946-02-08","name":"SAMATZAI"},{"end":"2016-04-27","st
art":"1974-08-20","name":"SAMATZAI"},{"end":"9999-12-31","start":"2016-04-28
","name":"SAMATZAI"}],"L330":[{"start":"1861-03-17","end":"9999-12-31","name
":"TRAONA"}],"G167":[{"start":"1867-08-26","end":"9999-12-31","name":"OSPEDA
LETTO EUGANEO"}],"B429":[{"start":"1861-03-17","end":"1968-04-06","name":"CA
LTANISSETTA"},{"name":"CALTANISSETTA","end":"1987-03-07","start":"1968-04-07
"},{"name":"CALTANISSETTA","end":"9999-12-31","start":"1987-03-08"}],"L470":
[{"start":"1861-03-17","end":"9999-12-31","name":"TURATE"}],"I128":[{"start"
:"1861-03-17","end":"9999-12-31","name":"SAN POSSIDONIO"}],"D777":[{"start":
"1863-08-10","end":"1928-05-30","name":"FRASSINELLO MONFERRATO"},{"end":"192
8-08-15","start":"1928-05-31","name":"FRASSINELLO MONFERRATO"},{"start":"195
0-02-11","end":"9999-12-31","name":"FRASSINELLO MONFERRATO"}],"I002":[{"star
t":"1873-01-14","end":"9999-12-31","name":"SAN MARTINO SANNITA"}],"C251":[{"
start":"1861-03-17","end":"1929-04-10","name":"CASTELRAIMONDO"},{"name":"CAS
TELRAIMONDO","end":"1934-03-06","start":"1929-04-11"},{"name":"CASTELRAIMOND
O","end":"9999-12-31","start":"1934-03-07"}],"A768":[{"end":"1928-05-21","st
art":"1861-03-17","name":"BELSITO"},{"name":"BELSITO","end":"9999-12-31","st
art":"1937-07-18"}],"F991":[{"start":"1958-01-26","end":"1965-01-07","name":
"NUXIS"},{"start":"1965-01-08","end":"1974-08-19","name":"NUXIS"},{"end":"20
05-12-31","start":"1974-08-20","name":"NUXIS"},{"name":"NUXIS","start":"2006
-01-01","end":"2016-04-27"},{"end":"9999-12-31","start":"2016-04-28","name":
"NUXIS"}],"L241":[{"start":"1862-12-19","end":"9999-12-31","name":"TORRE MON
DOV\u00cc"}],"A464":[{"start":"1861-03-17","end":"1923-03-18","name":"ASCREA
"},{"name":"ASCREA","start":"1923-03-19","end":"1927-01-11"},{"name":"ASCREA
","end":"1968-06-20","start":"1927-01-12"},{"start":"1968-06-21","end":"9999
-12-31","name":"ASCREA"}],"H139":[{"end":"1928-07-27","start":"1861-03-17","
name":"QUINZANELLO"}],"G767":[{"end":"9999-12-31","start":"1861-03-17","name
":"POGGIOREALE"}],"A693":[{"name":"BASELGA","end":"1923-02-13","start":"1920
-10-16"},{"start":"1923-02-14","end":"1928-05-21","name":"BASELGA"},{"name":
"BASELGA","start":"1947-01-15","end":"1955-03-01"},{"start":"1955-03-02","en
d":"1968-07-02","name":"BASELGA DI VEZZANO"}],"G634":[{"name":"PIEVE EMANUEL
E","start":"1862-11-03","end":"9999-12-31"}],"I256":[{"start":"1863-02-06","
end":"1927-01-11","name":"SANT'AMBROGIO SUL GARIGLIANO"},{"name":"SANT'AMBRO
GIO SUL GARIGLIANO","end":"9999-12-31","start":"1927-01-12"}],"C618":[{"end"
:"9999-12-31","start":"1861-03-17","name":"CHIARI"}],"F895":[{"name":"NIELLA
 TANARO","end":"9999-12-31","start":"1861-03-17"}],"G539":[{"name":"PIAGGINE
 SOPRANE","end":"1929-02-09","start":"1861-03-17"}],"D938":[{"name":"GATTINA
RA","start":"1861-03-17","end":"1927-01-11"},{"end":"9999-12-31","start":"19
27-01-12","name":"GATTINARA"}],"I728":[{"end":"1928-12-15","start":"1861-03-
17","name":"SIGNA"},{"end":"9999-12-31","start":"1928-12-16","name":"SIGNA"}
],"D177":[{"name":"CRODO","start":"1861-03-17","end":"1928-11-09"},{"name":"
CRODO","end":"1929-05-06","start":"1928-11-10"},{"name":"CRODO","start":"192
9-05-07","end":"1937-08-19"},{"name":"CRODO","start":"1937-08-20","end":"199
2-05-22"},{"end":"9999-12-31","start":"1992-05-23","name":"CRODO"}],"H260":[
{"end":"1928-08-11","start":"1861-03-17","name":"REZZONICO"}],"L236":[{"star
t":"1861-03-17","end":"1923-11-28","name":"TORRAZZA"}],"A168":[{"name":"ALBO
GASIO","end":"1927-06-09","start":"1861-03-17"}],"L591":[{"name":"VALLE DI M
ADDALONI","end":"1927-01-11","start":"1862-11-10"},{"name":"VALLE DI MADDALO
NI","start":"1927-01-12","end":"1945-08-31"},{"name":"VALLE DI MADDALONI","e
nd":"9999-12-31","start":"1945-09-01"}],"E963":[{"name":"MARMORA","end":"192
8-02-20","start":"1861-03-17"},{"end":"1946-04-25","start":"1928-02-21","nam
e":"MARMORA"},{"name":"MARMORA","start":"1946-04-26","end":"9999-12-31"}],"F
020":[{"name":"MASONE","end":"9999-12-31","start":"1861-03-17"}],"E132":[{"n
ame":"GRAGNANO TREBBIENSE","end":"9999-12-31","start":"1862-11-28"}],"D535":
[{"name":"FENILE","end":"1927-10-21","start":"1861-03-17"}],"D710":[{"end":"
1938-02-18","start":"1861-03-17","name":"FORMIGARA"},{"end":"1984-07-26","st
art":"1938-02-19","name":"FORMIGARA"},{"end":"9999-12-31","start":"1984-07-2
7","name":"FORMIGARA"}],"L417":[{"name":"TRIBUSSA","end":"1923-02-07","start
":"1921-01-05"},{"name":"TRIBUSSA","start":"1923-02-08","end":"1927-01-11"},
{"start":"1927-01-12","end":"1928-01-25","name":"TRIBUSSA"}],"B509":[{"end":
"1925-11-29","start":"1862-10-20","name":"CAMPIGLIA MARITTIMA"},{"start":"19
25-11-30","end":"1929-06-03","name":"CAMPIGLIA MARITTIMA"},{"end":"1949-08-0
3","start":"1929-06-04","name":"CAMPIGLIA MARITTIMA"},{"end":"9999-12-31","s
tart":"1949-08-04","name":"CAMPIGLIA MARITTIMA"}],"I906":[{"name":"SPINADESC
O","end":"9999-12-31","start":"1861-03-17"}],"I609":[{"start":"1861-03-17","
end":"1928-06-25","name":"SENIS"},{"start":"1928-06-26","end":"1946-02-07","
name":"SENIS"},{"name":"SENIS","start":"1946-02-08","end":"1951-01-11"},{"en
d":"1974-08-19","start":"1951-01-12","name":"SENIS"},{"end":"9999-12-31","st
art":"1974-08-20","name":"SENIS"}],"C360":[{"name":"CAVA MANARA","start":"18
72-10-15","end":"9999-12-31"}],"I504":[{"start":"1861-03-17","end":"9999-12-
31","name":"SCANSANO"}],"E127":[{"start":"1861-03-17","end":"1992-04-15","na
me":"GRAFFIGNANA"},{"end":"9999-12-31","start":"1992-04-16","name":"GRAFFIGN
ANA"}],"Z316":[{"end":"9999-12-31","start":"0000-01-01","name":"Gabon"}],"I0
76":[{"end":"1935-04-07","start":"1928-02-21","name":"SAN PAOLO SOLBRITO"},{
"end":"9999-12-31","start":"1935-04-08","name":"SAN PAOLO SOLBRITO"}],"E860"
:[{"name":"MALLARE","start":"1861-03-17","end":"1927-01-11"},{"name":"MALLAR
E","end":"9999-12-31","start":"1927-01-12"}],"I261":[{"start":"1861-03-17","
end":"1927-01-11","name":"SAN TAMMARO"},{"name":"SAN TAMMARO","start":"1927-
01-12","end":"1929-02-06"},{"name":"SAN TAMMARO","start":"1947-09-25","end":
"9999-12-31"}],"L898":[{"start":"1861-03-17","end":"9999-12-31","name":"VIGO
NE"}],"D844":[{"name":"GAGGI","start":"1939-12-22","end":"9999-12-31"},{"end
":"1939-12-21","start":"1861-03-17","name":"KAGGI"}],"B604":[{"start":"1871-
01-15","end":"1927-01-11","name":"CANINO"},{"name":"CANINO","start":"1927-01
-12","end":"9999-12-31"}],"G579":[{"end":"1927-09-13","start":"1863-07-13","
name":"PIAZZA BREMBANA"},{"end":"1978-02-11","start":"1956-08-26","name":"PI
AZZA BREMBANA"},{"name":"PIAZZA BREMBANA","end":"9999-12-31","start":"1978-0
2-12"}],"Z237":[{"end":"9999-12-31","start":"0000-01-01","name":"Qatar"}],"D
978":[{"start":"1871-01-15","end":"9999-12-31","name":"GERANO"}],"D137":[{"e
nd":"9999-12-31","start":"1861-03-17","name":"CRECCHIO"}],"C153":[{"end":"99
99-12-31","start":"1868-01-01","name":"CASTELLEONE"}],"D215":[{"end":"1947-1
0-20","start":"1927-10-13","name":"CURDOMO"}],"Z115":[{"name":"Grecia","star
t":"0000-01-01","end":"9999-12-31"}],"A364":[{"end":"1928-12-13","start":"18
61-03-17","name":"ARCELLASCO"}],"A967":[{"name":"BONDO","start":"1920-10-16"
,"end":"1923-02-13"},{"start":"1923-02-14","end":"1928-02-23","name":"BONDO"
},{"name":"BONDO","end":"2015-12-31","start":"1947-01-07"}],"L061":[{"name":
"TAURANO","end":"1939-08-10","start":"1861-03-17"},{"name":"TAURANO","start"
:"1946-12-19","end":"9999-12-31"}],"M017":[{"name":"VILLA SAN FIORANO","end"
:"1929-01-21","start":"1861-03-17"},{"start":"1929-01-22","end":"1929-08-24"
,"name":"VILLA SAN FIORANO"},{"end":"2009-06-29","start":"1929-08-25","name"
:"VILLASANTA"},{"start":"2009-06-30","end":"9999-12-31","name":"VILLASANTA"}
],"F274":[{"start":"1867-03-15","end":"9999-12-31","name":"MOIANO"}],"L276":
[{"start":"1861-03-17","end":"9999-12-31","name":"TORRE PALLAVICINA"}],"C942
":[{"name":"CONCADIRAME","start":"1866-11-19","end":"1927-09-03"}],"M356":[{
"end":"9999-12-31","start":"2016-01-01","name":"CONT\u00c0"}],"L743":[{"name
":"VENZONE","end":"1968-04-05","start":"1866-11-19"},{"name":"VENZONE","end"
:"9999-12-31","start":"1968-04-06"}],"D122":[{"name":"COTRONE","start":"1861
-03-17","end":"1929-03-11"},{"start":"1929-03-12","end":"1992-04-15","name":
"CROTONE"},{"start":"1992-04-16","end":"9999-12-31","name":"CROTONE"}],"E525
":[{"start":"1861-03-17","end":"1929-02-07","name":"LENNO"},{"name":"LENNO",
"start":"1947-10-03","end":"2014-02-03"}],"Z222":[{"end":"9999-12-31","start
":"0000-01-01","name":"India"}],"H040":[{"name":"PREPOTTO","start":"1878-11-
23","end":"1968-04-05"},{"start":"1968-04-06","end":"9999-12-31","name":"PRE
POTTO"}],"E706":[{"end":"9999-12-31","start":"1861-03-17","name":"LOZIO"}],"
H179":[{"name":"RANZI PIETRA","start":"1861-03-17","end":"1927-01-11"},{"sta
rt":"1927-01-12","end":"1929-01-24","name":"RANZI PIETRA"}],"E697":[{"name":
"LOSETO","start":"1861-03-17","end":"1937-04-01"}],"G674":[{"start":"1861-03
-17","end":"1928-12-05","name":"PINEROLO"},{"start":"1928-12-06","end":"9999
-12-31","name":"PINEROLO"}],"D008":[{"end":"9999-12-31","start":"1861-03-17"
,"name":"CORIO"}],"F916":[{"end":"1958-05-27","start":"1861-03-17","name":"N
OCIGLIA"},{"start":"1958-05-28","end":"1975-05-19","name":"NOCIGLIA"},{"star
t":"1975-05-20","end":"9999-12-31","name":"NOCIGLIA"}],"L914":[{"name":"BASS
ANO","end":"1923-02-07","start":"1921-01-05"},{"name":"BASSANO","end":"1923-
04-26","start":"1923-02-08"},{"start":"1923-04-27","end":"1926-10-28","name"
:"VILLABASSA DI SENOSECCHIA"}],"E928":[{"start":"1861-03-17","end":"1927-12-
23","name":"MARCHENO"},{"end":"9999-12-31","start":"1927-12-24","name":"MARC
HENO"}],"D181":[{"start":"1861-03-17","end":"1929-07-02","name":"CROPANI"},{
"end":"1946-02-27","start":"1929-07-03","name":"CROPANI"},{"start":"1946-02-
28","end":"1957-01-17","name":"CROPANI"},{"start":"1957-01-18","end":"9999-1
2-31","name":"CROPANI"}],"F619":[{"start":"1882-05-12","end":"1923-03-18","n
ame":"MONTE SAN GIOVANNI IN SABINA"},{"end":"1927-01-11","start":"1923-03-19
","name":"MONTE SAN GIOVANNI IN SABINA"},{"start":"1927-01-12","end":"9999-1
2-31","name":"MONTE SAN GIOVANNI IN SABINA"}],"A123":[{"name":"ALATRI","star
t":"1871-01-15","end":"1927-01-11"},{"name":"ALATRI","start":"1927-01-12","e
nd":"9999-12-31"}],"L317":[{"name":"TRABIA","start":"1861-03-17","end":"2011
-08-11"},{"name":"TRABIA","start":"2011-08-12","end":"9999-12-31"}],"M361":[
{"start":"2016-01-01","end":"9999-12-31","name":"TRE VILLE"}],"L056":[{"end"
:"9999-12-31","start":"1861-03-17","name":"TARTANO"}],"F054":[{"start":"1861
-03-17","end":"9999-12-31","name":"MATINO"}],"I763":[{"start":"1867-06-03","
end":"2013-12-31","name":"SISSA"}],"G219":[{"start":"1878-12-05","end":"1929
-01-24","name":"PADERNO CREMONESE"}],"I041":[{"name":"SAN MICHELE EXTRA","st
art":"1867-08-26","end":"1927-02-15"}],"E172":[{"name":"GRIANTE","end":"9999
-12-31","start":"1861-03-17"}],"D575":[{"start":"1865-08-13","end":"1910-07-
30","name":"FIESOLE"},{"start":"1910-07-31","end":"9999-12-31","name":"FIESO
LE"}],"H765":[{"start":"1861-03-17","end":"9999-12-31","name":"SAN BASILE"}]
,"L516":[{"name":"USSEGLIO","end":"9999-12-31","start":"1861-03-17"}],"F514"
:[{"end":"9999-12-31","start":"1866-11-19","name":"MONTEGALDA"}],"G812":[{"n
ame":"POMIGLIANO D'ARCO","start":"1861-03-17","end":"1929-04-12"},{"name":"P
OMIGLIANO D'ARCO","end":"9999-12-31","start":"1929-04-13"}],"E405":[{"end":"
1928-01-25","start":"1861-03-17","name":"LAGLIO"},{"end":"1928-05-30","start
":"1928-01-26","name":"LAGLIO"},{"start":"1948-03-26","end":"9999-12-31","na
me":"LAGLIO"},{"end":"1948-03-25","start":"1928-05-31","name":"LAGLIO-BRIENN
O"}],"L202":[{"name":"TONARA","start":"1861-03-17","end":"1927-01-11"},{"nam
e":"TONARA","start":"1927-01-12","end":"9999-12-31"}],"B248":[{"start":"1959
-05-01","end":"2005-12-31","name":"BUDONI"},{"name":"BUDONI","start":"2006-0
1-01","end":"2011-01-20"},{"name":"BUDONI","end":"2016-04-27","start":"2011-
01-21"},{"start":"2016-04-28","end":"9999-12-31","name":"BUDONI"}],"D781":[{
"start":"1861-03-17","end":"1927-01-11","name":"FRASSINETTO"},{"name":"FRASS
INETTO","end":"1945-10-04","start":"1927-01-12"},{"name":"FRASSINETTO","star
t":"1945-10-05","end":"1957-03-22"},{"name":"FRASSINETTO","end":"9999-12-31"
,"start":"1957-03-23"}],"A723":[{"end":"1929-06-01","start":"1861-03-17","na
me":"BAUSO"}],"H451":[{"name":"ROCCAVERANO","start":"1861-03-17","end":"1935
-04-07"},{"start":"1935-04-08","end":"9999-12-31","name":"ROCCAVERANO"}],"L8
09":[{"name":"VESSALICO","start":"1861-03-17","end":"1929-03-15"},{"name":"V
ESSALICO","start":"1929-03-16","end":"9999-12-31"}],"G565":[{"start":"1866-1
1-19","end":"9999-12-31","name":"PIANIGA"}],"I163":[{"end":"1913-05-08","sta
rt":"1861-03-17","name":"SAN SOSSIO"},{"end":"9999-12-31","start":"1913-05-0
9","name":"SAN SOSSIO BARONIA"}],"A492":[{"start":"1861-03-17","end":"1927-0
1-11","name":"ATZARA"},{"end":"9999-12-31","start":"1927-01-12","name":"ATZA
RA"}],"B598":[{"name":"CANEVA","start":"1866-11-19","end":"1968-04-05"},{"na
me":"CANEVA","end":"9999-12-31","start":"1968-04-06"}],"M149":[{"name":"ZARA
","end":"1923-02-07","start":"1921-01-05"},{"end":"1947-09-15","start":"1923
-02-08","name":"ZARA"}],"E772":[{"end":"9999-12-31","start":"1861-03-17","na
me":"LUZZARA"}],"H165":[{"name":"RUVIANO","end":"1927-01-11","start":"1862-1
2-29"},{"start":"1927-01-12","end":"1945-08-31","name":"RUVIANO"},{"end":"99
99-12-31","start":"1945-09-01","name":"RUVIANO"}],"I698":[{"name":"SETTIME",
"start":"1861-03-17","end":"1935-04-07"},{"start":"1935-04-08","end":"9999-1
2-31","name":"SETTIME"}],"C046":[{"end":"9999-12-31","start":"1861-03-17","n
ame":"CASTAGNITO"}],"M070":[{"start":"1861-03-17","end":"9999-12-31","name":
"VISANO"}],"Z530":[{"start":"0000-01-01","end":"9999-12-31","name":"Isole Ca
yman"}],"L430":[{"name":"TRIORA","end":"1904-01-21","start":"1861-03-17"},{"
start":"1904-01-22","end":"1947-10-06","name":"TRIORA"},{"name":"TRIORA","st
art":"1947-10-07","end":"9999-12-31"}],"I927":[{"start":"1866-11-19","end":"
9999-12-31","name":"SPRESIANO"}],"D664":[{"name":"FONDRA","start":"1861-03-1
7","end":"1928-08-14"}],"F552":[{"name":"MONTELUPONE","end":"9999-12-31","st
art":"1861-03-17"}],"G854":[{"start":"1861-03-17","end":"1927-01-11","name":
"PONT SAINT MARTIN"},{"name":"PONT SAINT MARTIN","end":"1929-05-02","start":
"1927-01-12"},{"start":"1946-01-30","end":"1946-03-09","name":"PONT SAINT MA
RTIN"},{"name":"PONT SAINT MARTIN","end":"1977-01-01","start":"1946-03-10"},
{"name":"PONTE SAN MARTINO","end":"1946-01-29","start":"1929-05-03"},{"end":
"9999-12-31","start":"1977-01-02","name":"PONT-SAINT-MARTIN"}],"D737":[{"end
":"1970-03-02","start":"1863-01-19","name":"FOSSALTO"},{"end":"9999-12-31","
start":"1970-03-03","name":"FOSSALTO"}],"C753":[{"name":"CIVELLO","end":"192
8-05-30","start":"1861-03-17"}],"H097":[{"name":"QUADRELLE","start":"1861-03
-17","end":"9999-12-31"}],"B995":[{"name":"CASSAGO","end":"1927-06-07","star
t":"1861-03-17"}],"D493":[{"name":"FARA IN SABINA","start":"1863-04-13","end
":"1923-03-18"},{"name":"FARA IN SABINA","end":"1927-01-11","start":"1923-03
-19"},{"name":"FARA IN SABINA","end":"1937-10-25","start":"1927-01-12"},{"na
me":"FARA IN SABINA","start":"1937-10-26","end":"9999-12-31"}],"C087":[{"nam
e":"CASTEL DE' RATTI","end":"1928-06-27","start":"1861-03-17"}],"I450":[{"na
me":"SASSA","start":"1861-03-17","end":"1927-09-20"}],"E640":[{"start":"1861
-03-17","end":"9999-12-31","name":"LOCATELLO"}],"H466":[{"start":"1861-03-17
","end":"1935-04-07","name":"ROCCHETTA PALAFEA"},{"name":"ROCCHETTA PALAFEA"
,"end":"9999-12-31","start":"1935-04-08"}],"I324":[{"start":"1861-03-17","en
d":"1951-12-31","name":"SANT'ELPIDIO A MARE"},{"start":"1952-01-01","end":"2
009-07-11","name":"SANT'ELPIDIO A MARE"},{"end":"9999-12-31","start":"2009-0
7-12","name":"SANT'ELPIDIO A MARE"}],"I932":[{"name":"STAFFOLO","start":"186
1-03-17","end":"1928-05-28"},{"name":"STAFFOLO","start":"1928-05-29","end":"
1946-02-05"},{"start":"1946-02-06","end":"9999-12-31","name":"STAFFOLO"}],"L
143":[{"start":"1861-03-17","end":"1929-02-14","name":"TERZO"},{"name":"TERZ
O","start":"1947-11-26","end":"9999-12-31"}],"D722":[{"name":"FORNO","start"
:"1861-03-17","end":"1928-01-26"}],"B891":[{"name":"CASALETTO VAPRIO","end":
"9999-12-31","start":"1861-03-17"}],"F012":[{"name":"MASERADA","end":"1925-0
6-18","start":"1866-11-19"},{"start":"1925-06-19","end":"9999-12-31","name":
"MASERADA SUL PIAVE"}],"D569":[{"name":"FICARRA","end":"9999-12-31","start":
"1861-03-17"}],"H779":[{"name":"SAN BIAGIO SARACINISCO","end":"1927-01-11","
start":"1864-11-07"},{"end":"9999-12-31","start":"1927-01-12","name":"SAN BI
AGIO SARACINISCO"}],"E106":[{"name":"GORNO","start":"1861-03-17","end":"9999
-12-31"}],"L370":[{"name":"TREMEZZINA","start":"1929-02-08","end":"1947-10-0
3"}],"L952":[{"end":"1939-09-06","start":"1866-11-19","name":"VILLAGA"},{"na
me":"VILLAGA","start":"1947-12-22","end":"9999-12-31"}],"G968":[{"start":"18
61-03-17","end":"9999-12-31","name":"PRADALUNGA"}],"E057":[{"name":"GIULIANO
 DI ROMA","start":"1872-10-18","end":"1927-01-11"},{"name":"GIULIANO DI ROMA
","end":"9999-12-31","start":"1927-01-12"}],"D227":[{"end":"1906-06-05","sta
rt":"1872-01-01","name":"CURTATONE"},{"end":"1925-03-20","start":"1906-06-06
","name":"CURTATONE"},{"end":"1943-02-23","start":"1925-03-21","name":"CURTA
TONE"},{"name":"CURTATONE","start":"1943-02-24","end":"9999-12-31"}],"F328":
[{"start":"1863-07-13","end":"1928-07-29","name":"MONASTEROLO DEL CASTELLO"}
,{"name":"MONASTEROLO DEL CASTELLO","start":"1947-05-25","end":"9999-12-31"}
],"Z127":[{"name":"Polonia","start":"0000-01-01","end":"9999-12-31"}],"E316"
:[{"name":"INVORIO SUPERIORE","start":"1861-03-17","end":"1928-04-02"}],"H87
7":[{"end":"9999-12-31","start":"1861-03-17","name":"SANGINETO"}],"M025":[{"
name":"VILLASOR","start":"1861-03-17","end":"1974-08-19"},{"end":"2016-04-27
","start":"1974-08-20","name":"VILLASOR"},{"start":"2016-04-28","end":"9999-
12-31","name":"VILLASOR"}],"B901":[{"name":"CASALMORO","start":"1868-07-01",
"end":"9999-12-31"}],"E783":[{"end":"9999-12-31","start":"1861-03-17","name"
:"MACERATA"}],"C068":[{"name":"CASTEL CELLESI","start":"1871-01-15","end":"1
927-01-11"},{"end":"1927-12-30","start":"1927-01-12","name":"CASTEL CELLESI"
}],"G384":[{"end":"1974-08-19","start":"1861-03-17","name":"PAULILATINO"},{"
end":"9999-12-31","start":"1974-08-20","name":"PAULILATINO"}],"Z132":[{"name
":"Svezia","end":"9999-12-31","start":"0000-01-01"}],"F150":[{"start":"1921-
01-05","end":"1923-02-07","name":"MERNA"},{"start":"1923-02-08","end":"1927-
01-11","name":"MERNA"},{"name":"MERNA","start":"1927-01-12","end":"1927-12-2
8"},{"start":"1927-12-29","end":"1947-09-15","name":"MERNA"}],"D232":[{"name
":"CUSINO","end":"9999-12-31","start":"1861-03-17"}],"B266":[{"end":"9999-12
-31","start":"1861-03-17","name":"BUONABITACOLO"}],"B071":[{"name":"BOSCO MA
RENGO","start":"1863-02-16","end":"1929-06-14"},{"name":"BOSCO MARENGO","end
":"9999-12-31","start":"1929-06-15"}],"E517":[{"name":"LEI","start":"1861-03
-17","end":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":"LE
I"}],"B805":[{"start":"1869-03-11","end":"1927-01-11","name":"CARONNO MILANE
SE"},{"name":"CARONNO MILANESE","start":"1927-01-12","end":"1940-03-22"},{"n
ame":"CARONNO PERTUSELLA","start":"1940-03-23","end":"9999-12-31"}],"C816":[
{"name":"CODOGNO","end":"1992-04-15","start":"1869-02-01"},{"name":"CODOGNO"
,"start":"1992-04-16","end":"9999-12-31"}],"A459":[{"end":"9999-12-31","star
t":"1866-11-19","name":"ARZIGNANO"}],"D839":[{"name":"GABY","start":"1952-04
-19","end":"9999-12-31"}],"G987":[{"name":"PRASCO","end":"9999-12-31","start
":"1861-03-17"}],"D645":[{"start":"1921-01-05","end":"1923-02-07","name":"FO
GLIANO"},{"name":"FOGLIANO","start":"1923-02-08","end":"1923-04-26"},{"start
":"1923-04-27","end":"1939-07-14","name":"FOGLIANO DI MONFALCONE"},{"name":"
FOGLIANO REDIPUGLIA","start":"1939-07-15","end":"1948-01-09"},{"end":"9999-1
2-31","start":"1948-01-10","name":"FOGLIANO REDIPUGLIA"}],"G294":[{"start":"
1861-03-17","end":"9999-12-31","name":"PALOMBARO"}],"A233":[{"name":"ALTIDON
A","start":"1861-03-17","end":"2009-07-11"},{"name":"ALTIDONA","start":"2009
-07-12","end":"9999-12-31"}],"E277":[{"name":"IDRESCA D'ISONZO","end":"1923-
02-07","start":"1921-01-05"},{"start":"1923-02-08","end":"1927-01-11","name"
:"IDRESCA D'ISONZO"},{"start":"1927-01-12","end":"1928-05-01","name":"IDRESC
A D'ISONZO"}],"G649":[{"name":"PIEVEPELAGO","end":"9999-12-31","start":"1870
-10-01"}],"A771":[{"name":"LIZZANO IN BELVEDERE","start":"1862-10-31","end":
"9999-12-31"}],"G946":[{"start":"1923-04-27","end":"1927-06-08","name":"POVE
RIO"},{"start":"1921-01-05","end":"1923-02-07","name":"POVIR"},{"start":"192
3-02-08","end":"1923-04-26","name":"POVIR"}],"F599":[{"end":"2009-07-11","st
art":"1861-03-17","name":"MONTE RINALDO"},{"start":"2009-07-12","end":"9999-
12-31","name":"MONTE RINALDO"}],"L325":[{"name":"TRAMONTI DI SOTTO","start":
"1866-11-19","end":"1968-04-05"},{"end":"9999-12-31","start":"1968-04-06","n
ame":"TRAMONTI DI SOTTO"}],"E491":[{"name":"LA VALLE","start":"1920-10-16","
end":"1923-02-13"},{"name":"LA VALLE","end":"1927-01-11","start":"1923-02-14
"},{"name":"LA VALLE","start":"1927-01-12","end":"1928-06-15"},{"name":"LA V
ALLE","start":"1964-06-10","end":"9999-12-31"}],"L696":[{"start":"1861-03-17
","end":"1970-03-02","name":"VASTOGIRARDI"},{"name":"VASTOGIRARDI","start":"
1970-03-03","end":"9999-12-31"}],"F694":[{"name":"MONTORO SUPERIORE","start"
:"1861-03-17","end":"2013-12-02"}],"L707":[{"name":"VEDESETA","end":"9999-12
-31","start":"1861-03-17"}],"G544":[{"end":"1928-04-16","start":"1861-03-17"
,"name":"PIANAVIA"}],"F842":[{"end":"1975-05-31","start":"1861-03-17","name"
:"NARD\u00d2"},{"start":"1975-06-01","end":"9999-12-31","name":"NARD\u00d2"}
],"B251":[{"name":"BUGGIANO","start":"1883-05-21","end":"1928-01-25"},{"name
":"BUGGIANO","start":"1928-01-26","end":"9999-12-31"}],"L999":[{"name":"VILL
AR DORA","start":"1885-01-26","end":"1928-03-05"},{"start":"1955-07-10","end
":"9999-12-31","name":"VILLAR DORA"}],"D400":[{"name":"ELSANE","start":"1923
-04-27","end":"1924-03-08"},{"name":"ELSANE","start":"1924-03-09","end":"192
7-11-03"},{"end":"1936-09-02","start":"1927-11-04","name":"ELSANE"},{"name":
"ELSANE","end":"1947-09-15","start":"1936-09-03"},{"start":"1921-01-05","end
":"1923-02-07","name":"GELSANE"},{"name":"GELSANE","end":"1923-04-26","start
":"1923-02-08"}],"F463":[{"end":"9999-12-31","start":"1861-03-17","name":"MO
NTECCHIO EMILIA"}],"G931":[{"start":"1866-11-19","end":"9999-12-31","name":"
POSINA"}],"C429":[{"name":"CELENZA VALFORTORE","start":"1862-11-10","end":"9
999-12-31"}],"F249":[{"end":"9999-12-31","start":"1861-03-17","name":"MISSAN
ELLO"}],"H448":[{"name":"ROCCASPINALVETI","start":"1861-03-17","end":"9999-1
2-31"}],"I058":[{"name":"SAN NICOLA DA CRISSA","start":"1863-07-13","end":"1
992-04-15"},{"name":"SAN NICOLA DA CRISSA","start":"1992-04-16","end":"9999-
12-31"}],"E570":[{"end":"1927-01-11","start":"1862-09-08","name":"LIBERI"},{
"name":"LIBERI","start":"1927-01-12","end":"1945-08-31"},{"name":"LIBERI","s
tart":"1945-09-01","end":"9999-12-31"}],"A171":[{"name":"ALBONESE","end":"99
99-12-31","start":"1861-03-17"}],"H744":[{"name":"SAMBUCA PISTOIESE","end":"
1927-01-11","start":"1864-04-01"},{"name":"SAMBUCA PISTOIESE","end":"9999-12
-31","start":"1927-01-12"}],"L038":[{"name":"TALLA","start":"1861-03-17","en
d":"9999-12-31"}],"I518":[{"name":"SCELMO","end":"1923-02-13","start":"1920-
10-16"},{"name":"SCELMO","start":"1923-02-14","end":"1928-03-12"}],"L107":[{
"end":"1923-02-13","start":"1920-10-16","name":"TERLAGO"},{"name":"TERLAGO",
"end":"1928-05-21","start":"1923-02-14"},{"start":"1928-05-22","end":"1947-0
1-14","name":"TERLAGO"},{"end":"2015-12-31","start":"1947-01-15","name":"TER
LAGO"}],"H810":[{"name":"SAN CRISTOFORO","start":"1861-03-17","end":"9999-12
-31"}],"A585":[{"name":"BAIRO TORRE","start":"1929-01-15","end":"1945-10-04"
},{"end":"1955-04-29","start":"1945-10-05","name":"BAIRO TORRE"}],"B618":[{"
start":"1889-04-26","end":"1953-02-16","name":"CANONICA D'ADDA"},{"name":"CA
NONICA D'ADDA","end":"9999-12-31","start":"1953-02-17"}],"B360":[{"start":"1
920-10-16","end":"1923-02-13","name":"CAGN\u00d2"},{"name":"CAGN\u00d2","end
":"1928-09-05","start":"1923-02-14"},{"name":"CAGN\u00d2","start":"1950-05-1
3","end":"2019-12-31"}],"I278":[{"start":"1861-03-17","end":"9999-12-31","na
me":"SANT'ANGELO A FASANELLA"}],"D520":[{"start":"1861-03-17","end":"1927-11
-01","name":"FAVRIA"},{"name":"FAVRIA","start":"1946-07-18","end":"9999-12-3
1"}],"F035":[{"name":"MASSAROSA","end":"9999-12-31","start":"1870-01-02"}],"
I915":[{"end":"1928-07-29","start":"1861-03-17","name":"SPINONE"}],"M174":[{
"name":"ZIBELLO","end":"2015-12-31","start":"1861-03-17"}],"C509":[{"start":
"1861-03-17","end":"9999-12-31","name":"CERGNAGO"}],"M338":[{"start":"2014-0
2-04","end":"9999-12-31","name":"CORNALE E BASTIDA"}],"E183":[{"end":"1923-0
2-07","start":"1921-01-05","name":"GRISIGNANA"},{"start":"1923-02-08","end":
"1977-04-03","name":"GRISIGNANA"}],"A988":[{"name":"BORETTO","start":"1861-0
3-17","end":"9999-12-31"}],"L680":[{"name":"VARENNA","start":"1861-03-17","e
nd":"1928-08-14"},{"start":"1928-08-15","end":"1953-06-16","name":"VARENNA"}
,{"name":"VARENNA","end":"1992-04-15","start":"1953-06-17"},{"name":"VARENNA
","end":"9999-12-31","start":"1992-04-16"}],"Z732":[{"end":"9999-12-31","sta
rt":"0000-01-01","name":"Tuvalu"}],"C604":[{"name":"CHIALAMBERTO","end":"999
9-12-31","start":"1861-03-17"}],"F750":[{"name":"MORSANO AL TAGLIAMENTO","st
art":"1866-11-19","end":"1968-04-05"},{"name":"MORSANO AL TAGLIAMENTO","end"
:"9999-12-31","start":"1968-04-06"}],"I811":[{"start":"1861-03-17","end":"19
28-03-06","name":"SOLTO"}],"E210":[{"start":"1871-01-15","end":"1927-01-11",
"name":"GROTTE DI CASTRO"},{"name":"GROTTE DI CASTRO","start":"1927-01-12","
end":"9999-12-31"}],"G042":[{"name":"OLIVOLA","start":"1861-03-17","end":"19
28-05-30"},{"name":"OLIVOLA","start":"1950-02-11","end":"9999-12-31"}],"A345
":[{"name":"AQUILA DEGLI ABRUZZI","end":"1927-09-19","start":"1863-05-06"},{
"end":"1940-01-10","start":"1927-09-20","name":"AQUILA DEGLI ABRUZZI"},{"nam
e":"L'AQUILA","start":"1940-01-11","end":"1947-09-30"},{"start":"1947-10-01"
,"end":"9999-12-31","name":"L'AQUILA"}],"Z305":[{"start":"0000-01-01","end":
"9999-12-31","name":"Burundi"}],"H913":[{"name":"SAN GIOVANNI DI BIEDA","sta
rt":"1871-01-15","end":"1927-01-11"},{"end":"1927-10-19","start":"1927-01-12
","name":"SAN GIOVANNI DI BIEDA"},{"end":"1961-11-08","start":"1945-09-15","
name":"SAN GIOVANNI DI BIEDA"},{"name":"VILLA SAN GIOVANNI IN TUSCIA","end":
"9999-12-31","start":"1961-11-09"}],"H657":[{"name":"SACILE","start":"1866-1
1-19","end":"1968-04-05"},{"start":"1968-04-06","end":"9999-12-31","name":"S
ACILE"}],"B942":[{"start":"1863-02-23","end":"2017-12-31","name":"CASASCO D'
INTELVI"}],"L908":[{"end":"9999-12-31","start":"1875-06-11","name":"VILLA DI
 TIRANO"}],"D865":[{"name":"GALBIATE","end":"1928-02-08","start":"1861-03-17
"},{"end":"1929-04-19","start":"1928-02-09","name":"GALBIATE"},{"start":"192
9-04-20","end":"1937-10-26","name":"GALBIATE"},{"name":"GALBIATE","start":"1
937-10-27","end":"1992-04-15"},{"name":"GALBIATE","end":"9999-12-31","start"
:"1992-04-16"}],"Z106":[{"start":"0000-01-01","end":"9999-12-31","name":"Sta
to della Citt\u00e0 del Vaticano"}],"D206":[{"start":"1920-10-16","end":"192
3-02-13","name":"CUNEVO"},{"name":"CUNEVO","end":"1928-10-16","start":"1923-
02-14"},{"end":"2015-12-31","start":"1947-03-30","name":"CUNEVO"}],"B153":[{
"name":"BRENTONICO","end":"1923-02-13","start":"1920-10-16"},{"name":"BRENTO
NICO","end":"1924-01-13","start":"1923-02-14"},{"name":"BRENTONICO","start":
"1924-01-14","end":"9999-12-31"}],"A146":[{"name":"ALBERA LIGURE","end":"192
9-05-27","start":"1863-01-26"},{"name":"ALBERA LIGURE","start":"1947-09-04",
"end":"1955-11-25"},{"start":"1955-11-26","end":"9999-12-31","name":"ALBERA 
LIGURE"}],"E530":[{"name":"LENTATE SUL SEVESO","end":"2004-04-09","start":"1
869-02-24"},{"name":"LENTATE SUL SEVESO","end":"2009-12-17","start":"2004-04
-10"},{"name":"LENTATE SUL SEVESO","start":"2009-12-18","end":"9999-12-31"}]
,"F255":[{"name":"MOCCHIE","start":"1861-03-17","end":"1936-07-07"}],"D961":
[{"end":"1907-09-29","start":"1861-03-17","name":"GEMMANO"},{"name":"GEMMANO
","start":"1907-09-30","end":"1920-03-14"},{"name":"GEMMANO","end":"1992-04-
15","start":"1920-03-15"},{"end":"9999-12-31","start":"1992-04-16","name":"G
EMMANO"}],"E715":[{"end":"9999-12-31","start":"1884-07-07","name":"LUCCA"}],
"I896":[{"end":"1937-07-17","start":"1928-05-19","name":"SPEZZANO DELLA SILA
"},{"end":"9999-12-31","start":"1937-07-18","name":"SPEZZANO DELLA SILA"}],"
A600":[{"start":"1861-03-17","end":"1927-01-11","name":"BALMUCCIA"},{"end":"
9999-12-31","start":"1927-01-12","name":"BALMUCCIA"}],"L078":[{"name":"TAVOL
ETO","end":"1998-09-03","start":"1861-03-17"},{"name":"TAVOLETO","start":"19
98-09-04","end":"9999-12-31"}],"A131":[{"end":"9999-12-31","start":"1863-04-
10","name":"ALBANO DI LUCANIA"}],"F508":[{"end":"9999-12-31","start":"1867-0
8-26","name":"MONTEFORTE D'ALPONE"}],"G788":[{"start":"1882-05-29","end":"99
99-12-31","name":"SAN PIETRO IN CERRO"}],"D554":[{"name":"FERRERE","end":"19
35-04-07","start":"1861-03-17"},{"name":"FERRERE","start":"1935-04-08","end"
:"1951-02-19"},{"name":"FERRERE","end":"9999-12-31","start":"1951-02-20"}],"
F905":[{"start":"1920-10-16","end":"1923-02-13","name":"NOARNA"},{"start":"1
923-02-14","end":"1929-02-20","name":"NOARNA"}],"F662":[{"end":"9999-12-31",
"start":"1906-05-07","name":"MONTEVIALE"}],"B460":[{"name":"CAMASTRA","end":
"1929-05-15","start":"1861-03-17"},{"start":"1946-03-25","end":"1986-03-08",
"name":"CAMASTRA"},{"start":"1986-03-09","end":"9999-12-31","name":"CAMASTRA
"}],"D659":[{"end":"1928-07-29","start":"1861-03-17","name":"FOMARCO"}],"G86
9":[{"name":"PONTOGLIO","end":"9999-12-31","start":"1861-03-17"}],"L505":[{"
name":"URURI","start":"1861-03-17","end":"1970-03-02"},{"name":"URURI","star
t":"1970-03-03","end":"9999-12-31"}],"M134":[{"name":"VOLZANA","start":"1921
-01-05","end":"1923-02-07"},{"start":"1923-02-08","end":"1927-01-11","name":
"VOLZANA"},{"name":"VOLZANA","start":"1927-01-12","end":"1928-05-11"}],"D956
":[{"start":"1866-11-19","end":"9999-12-31","name":"GAZZO"}],"I238":[{"start
":"1958-08-28","end":"1970-03-02","name":"SANTA MARIA DEL MOLISE"},{"name":"
SANTA MARIA DEL MOLISE","end":"9999-12-31","start":"1970-03-03"},{"end":"195
8-08-27","start":"1861-03-17","name":"SANT'ANGELO IN GROTTE"}],"F075":[{"sta
rt":"1861-03-17","end":"1928-05-30","name":"MEANO"}],"A445":[{"end":"9999-12
-31","start":"1906-01-05","name":"ARSITA"},{"name":"BACUCCO","start":"1861-0
3-17","end":"1906-01-04"}],"F801":[{"start":"1869-02-11","end":"1992-04-15",
"name":"MULAZZANO"},{"name":"MULAZZANO","start":"1992-04-16","end":"2004-06-
07"},{"name":"MULAZZANO","start":"2004-06-08","end":"9999-12-31"}],"F724":[{
"name":"MORFASSO","end":"9999-12-31","start":"1870-06-01"}],"A187":[{"start"
:"1861-03-17","end":"9999-12-31","name":"ALFEDENA"}],"L726":[{"name":"VENALZ
IO","start":"1937-06-16","end":"1947-09-15"},{"start":"1947-09-16","end":"19
67-10-28","name":"VENALZIO"},{"start":"1861-03-17","end":"1937-06-15","name"
:"VENAUS"},{"start":"1967-10-29","end":"9999-12-31","name":"VENAUS"}],"D014"
:[{"end":"1923-02-07","start":"1921-01-05","name":"CORMONS"},{"end":"1927-01
-11","start":"1923-02-08","name":"CORMONS"},{"end":"1928-09-08","start":"192
7-01-12","name":"CORMONS"},{"start":"1928-09-09","end":"1947-09-15","name":"
CORMONS"},{"name":"CORMONS","end":"1951-02-14","start":"1947-09-16"},{"start
":"1951-02-15","end":"1955-05-13","name":"CORMONS"},{"name":"CORMONS","start
":"1955-05-14","end":"9999-12-31"}],"M378":[{"start":"2017-01-01","end":"999
9-12-31","name":"MONTALCINO"}],"L258":[{"name":"TORRE DE' PICENARDI","start"
:"1868-01-01","end":"9999-12-31"}],"G262":[{"start":"1863-01-05","end":"1927
-01-11","name":"PALAZZO CANAVESE"},{"end":"1929-05-25","start":"1927-01-12",
"name":"PALAZZO CANAVESE"},{"name":"PALAZZO CANAVESE","end":"9999-12-31","st
art":"1946-12-26"}],"H822":[{"end":"1927-01-11","start":"1861-03-17","name":
"SAN DONACI"},{"name":"SAN DONACI","start":"1927-01-12","end":"9999-12-31"}]
,"B031":[{"start":"1869-05-17","end":"9999-12-31","name":"BORGORICCO"}],"A51
3":[{"end":"1927-02-15","start":"1866-11-19","name":"AVESA"}],"D272":[{"star
t":"1861-03-17","end":"1929-06-01","name":"DENICE"},{"name":"DENICE","start"
:"1946-07-03","end":"9999-12-31"}],"D879":[{"name":"GALLIERA VENETA","start"
:"1867-08-26","end":"9999-12-31"}],"G188":[{"name":"OTRANTO","start":"1861-0
3-17","end":"9999-12-31"}],"C248":[{"name":"CASTELPLANIO","start":"1861-03-1
7","end":"1928-05-11"},{"name":"CASTELPLANIO","end":"1946-11-01","start":"19
28-05-12"},{"name":"CASTELPLANIO","end":"9999-12-31","start":"1946-11-02"}],
"F428":[{"name":"MONTAPPONE","end":"2009-07-11","start":"1861-03-17"},{"name
":"MONTAPPONE","start":"2009-07-12","end":"9999-12-31"}],"H229":[{"name":"RE
MANZACCO","end":"1960-03-24","start":"1866-11-19"},{"name":"REMANZACCO","sta
rt":"1960-03-25","end":"1968-04-05"},{"start":"1968-04-06","end":"2006-11-29
","name":"REMANZACCO"},{"name":"REMANZACCO","end":"9999-12-31","start":"2006
-11-30"}],"E416":[{"end":"9999-12-31","start":"1861-03-17","name":"LAINO"}],
"B046":[{"name":"BORGO VERCELLI","start":"1869-04-26","end":"1927-01-11"},{"
start":"1927-01-12","end":"9999-12-31","name":"BORGO VERCELLI"}],"H588":[{"s
tart":"1869-04-26","end":"9999-12-31","name":"ROTELLA"}],"L611":[{"name":"VA
LLEPIETRA","start":"1871-01-15","end":"1936-09-24"},{"start":"1936-09-25","e
nd":"9999-12-31","name":"VALLEPIETRA"}],"H837":[{"start":"1920-10-16","end":
"1923-02-13","name":"SAN FELICE"},{"start":"1923-02-14","end":"1928-06-30","
name":"SAN FELICE"},{"start":"1947-03-12","end":"1948-03-13","name":"SAN FEL
ICE"},{"end":"1974-09-17","start":"1948-03-14","name":"SAN FELICE"}],"A053":
[{"name":"ACRI","start":"1861-03-17","end":"9999-12-31"}],"E281":[{"name":"I
GLESIAS","start":"1861-03-17","end":"1938-01-07"},{"start":"1938-01-08","end
":"1939-04-03","name":"IGLESIAS"},{"name":"IGLESIAS","start":"1939-04-04","e
nd":"1940-02-11"},{"end":"1945-04-20","start":"1940-02-12","name":"IGLESIAS"
},{"name":"IGLESIAS","start":"1945-04-21","end":"1974-08-19"},{"name":"IGLES
IAS","start":"1974-08-20","end":"2005-12-31"},{"end":"2016-04-27","start":"2
006-01-01","name":"IGLESIAS"},{"name":"IGLESIAS","start":"2016-04-28","end":
"9999-12-31"}],"A359":[{"start":"1861-03-17","end":"1974-08-19","name":"ARBU
S"},{"name":"ARBUS","end":"2005-12-31","start":"1974-08-20"},{"name":"ARBUS"
,"end":"2016-04-27","start":"2006-01-01"},{"name":"ARBUS","start":"2016-04-2
8","end":"9999-12-31"}],"I880":[{"start":"1861-03-17","end":"9999-12-31","na
me":"SOZZAGO"}],"A787":[{"name":"BERBENNO DI VALTELLINA","end":"9999-12-31",
"start":"1864-04-01"}],"F124":[{"name":"MENDATICA MONTEGROSSO","start":"1928
-07-18","end":"1950-02-17"}],"L126":[{"start":"1863-04-10","end":"9999-12-31
","name":"TERRANOVA DI POLLINO"}],"C598":[{"name":"CHARVENSOD","end":"1927-0
1-11","start":"1861-03-17"},{"start":"1927-01-12","end":"1928-03-20","name":
"CHARVENSOD"},{"name":"CHARVENSOD","start":"1946-05-15","end":"9999-12-31"}]
,"G484":[{"end":"9999-12-31","start":"1861-03-17","name":"PESCASSEROLI"}],"L
425":[{"name":"TRIGGIANO","end":"1934-02-08","start":"1861-03-17"},{"start":
"1934-02-09","end":"9999-12-31","name":"TRIGGIANO"}],"E391":[{"name":"JOVEN\
u00c7AN","start":"1861-03-17","end":"1927-01-11"},{"end":"1928-03-20","start
":"1927-01-12","name":"JOVEN\u00c7AN"},{"name":"JOVEN\u00c7AN","end":"9999-1
2-31","start":"1946-05-15"}],"D052":[{"start":"1861-03-17","end":"1935-04-07
","name":"CORTAZZONE"},{"name":"CORTAZZONE","start":"1935-04-08","end":"9999
-12-31"}],"Z525":[{"end":"9999-12-31","start":"0000-01-01","name":"Isole Ver
gini britanniche"}],"E222":[{"name":"GRUMES","start":"1920-10-16","end":"192
3-02-13"},{"name":"GRUMES","start":"1923-02-14","end":"1928-09-05"},{"end":"
1952-05-16","start":"1928-09-06","name":"GRUMES"},{"start":"1952-05-17","end
":"2015-12-31","name":"GRUMES"}],"H985":[{"name":"SAN MARCO IN LAMIS","start
":"1861-03-17","end":"9999-12-31"}],"C995":[{"name":"COREGLIA LIGURE","end":
"9999-12-31","start":"1888-02-03"}],"B753":[{"start":"1861-03-17","end":"192
7-01-11","name":"CARDANA"},{"start":"1927-01-12","end":"1928-01-21","name":"
CARDANA"}],"E829":[{"name":"MAGR\u00c8 ALL'ADIGE","start":"1928-06-27","end"
:"1948-03-13"},{"end":"1952-05-16","start":"1948-03-14","name":"MAGR\u00c8 A
LL'ADIGE"},{"name":"MAGR\u00c8 ALL'ADIGE","end":"1971-09-21","start":"1952-0
5-17"},{"name":"MAGR\u00c8 SULLA STRADA DEL VINO","end":"9999-12-31","start"
:"1971-09-22"}],"A746":[{"name":"BELLANTE","start":"1861-03-17","end":"1928-
05-18"},{"name":"BELLANTE","start":"1928-05-19","end":"1928-08-06"},{"start"
:"1928-08-07","end":"1929-04-11","name":"BELLANTE"},{"name":"BELLANTE","end"
:"9999-12-31","start":"1929-04-12"}],"G001":[{"name":"ODOLO","start":"1861-0
3-17","end":"9999-12-31"}],"E237":[{"start":"1861-03-17","end":"1927-01-11",
"name":"GUARDABOSONE"},{"start":"1927-01-12","end":"9999-12-31","name":"GUAR
DABOSONE"}],"A273":[{"start":"1861-03-17","end":"1928-06-01","name":"ANDALO"
},{"name":"ANDALO VALTELLINO","start":"1928-06-02","end":"9999-12-31"}],"H34
8":[{"end":"9999-12-31","start":"1861-03-17","name":"RIVELLO"}],"D512":[{"st
art":"1863-04-23","end":"9999-12-31","name":"FAVALE DI MALVARO"}],"C329":[{"
end":"1984-09-26","start":"1902-07-01","name":"CASTIRAGA VIDARDO"},{"end":"1
992-04-15","start":"1984-09-27","name":"CASTIRAGA VIDARDO"},{"name":"CASTIRA
GA VIDARDO","end":"9999-12-31","start":"1992-04-16"}],"H881":[{"end":"9999-1
2-31","start":"1863-04-10","name":"SAN GIORGIO ALBANESE"}],"E115":[{"start":
"1861-03-17","end":"1928-03-26","name":"GOTTASECCA"},{"start":"1947-01-10","
end":"9999-12-31","name":"GOTTASECCA"}],"F363":[{"end":"2004-10-25","start":
"1861-03-17","name":"MONGHIDORO"},{"name":"MONGHIDORO","start":"2004-10-26",
"end":"9999-12-31"}],"F069":[{"name":"MAZZO MILANESE","start":"1862-11-03","
end":"1928-06-26"}],"C891":[{"start":"1920-10-16","end":"1923-02-13","name":
"COLOGNA-GAVAZZO"},{"name":"COLOGNA-GAVAZZO","end":"1929-04-11","start":"192
3-02-14"}],"A731":[{"end":"9999-12-31","start":"1881-02-14","name":"BEDONIA"
}],"I983":[{"start":"1861-03-17","end":"1928-09-25","name":"STROPINO"}],"G87
5":[{"end":"9999-12-31","start":"1868-02-14","name":"PONZANO VENETO"}],"D300
":[{"name":"DIGNANO","end":"1968-04-05","start":"1866-11-19"},{"name":"DIGNA
NO","end":"2014-02-03","start":"1968-04-06"},{"start":"2014-02-04","end":"99
99-12-31","name":"DIGNANO"}],"A272":[{"end":"1957-01-16","start":"1861-03-17
","name":"ANDALI"},{"start":"1957-01-17","end":"9999-12-31","name":"ANDALI"}
],"C022":[{"name":"CASSINASCO","end":"1935-04-07","start":"1861-03-17"},{"na
me":"CASSINASCO","start":"1935-04-08","end":"9999-12-31"}],"L795":[{"end":"9
999-12-31","start":"1861-03-17","name":"VERTOVA"}],"D513":[{"end":"1927-01-1
1","start":"1873-09-29","name":"VALSINNI"},{"end":"9999-12-31","start":"1927
-01-12","name":"VALSINNI"}],"F362":[{"start":"1923-04-27","end":"1927-01-11"
,"name":"MONGHEZZO DI FUORI"},{"name":"MONGHEZZO DI FUORI","start":"1927-01-
12","end":"1929-03-24"},{"start":"1920-10-16","end":"1923-02-13","name":"MON
GHEZZO ESTERNO"},{"name":"MONGHEZZO ESTERNO","start":"1923-02-14","end":"192
3-04-26"}],"C108":[{"name":"CASTROLIBERO","end":"9999-12-31","start":"1863-0
4-10"}],"I982":[{"end":"1992-04-15","start":"1861-03-17","name":"STRONGOLI"}
,{"start":"1992-04-16","end":"9999-12-31","name":"STRONGOLI"}],"D359":[{"nam
e":"DOZIO","start":"1861-03-17","end":"1927-10-09"}],"E223":[{"name":"GRUMO 
APPULA","end":"1927-11-07","start":"1863-01-19"},{"start":"1927-11-08","end"
:"1946-11-11","name":"GRUMO APPULA"},{"name":"GRUMO APPULA","start":"1946-11
-12","end":"9999-12-31"}],"D053":[{"end":"1927-04-02","start":"1892-07-22","
name":"CORTE"}],"H457":[{"end":"1927-07-26","start":"1861-03-17","name":"ROC
HEMOLLES"}],"C037":[{"name":"CASSOLA","start":"1866-11-19","end":"9999-12-31
"}],"H920":[{"name":"SAN GIOVANNI IN GALDO","start":"1861-03-17","end":"1970
-03-02"},{"end":"9999-12-31","start":"1970-03-03","name":"SAN GIOVANNI IN GA
LDO"}],"B752":[{"end":"1927-01-11","start":"1861-03-17","name":"CARCOFORO"},
{"end":"9999-12-31","start":"1927-01-12","name":"CARCOFORO"}],"L480":[{"name
":"UBOLDO","end":"1927-01-11","start":"1861-03-17"},{"name":"UBOLDO","start"
:"1927-01-12","end":"1928-02-15"},{"name":"UBOLDO","end":"9999-12-31","start
":"1950-07-19"}],"D787":[{"start":"1863-04-13","end":"9999-12-31","name":"FR
ATTA TODINA"}],"C404":[{"end":"9999-12-31","start":"1861-03-17","name":"CAVO
UR"}],"I090":[{"start":"1863-01-26","end":"9999-12-31","name":"SAN PIETRO VA
L LEMINA"}],"B114":[{"name":"BRACCIANO","start":"1871-01-15","end":"1926-11-
29"},{"name":"BRACCIANO","start":"1926-11-30","end":"1935-03-21"},{"name":"B
RACCIANO","end":"9999-12-31","start":"1935-03-22"}],"A052":[{"end":"1929-02-
13","start":"1861-03-17","name":"ACQUI"},{"start":"1929-02-14","end":"1929-0
5-13","name":"ACQUI"},{"start":"1929-05-14","end":"1947-11-25","name":"ACQUI
"},{"end":"1956-05-23","start":"1947-11-26","name":"ACQUI"},{"start":"1956-0
5-24","end":"9999-12-31","name":"ACQUI TERME"}],"F798":[{"start":"1863-04-10
","end":"9999-12-31","name":"MUGNANO DEL CARDINALE"}],"I825":[{"name":"SOMMO
","start":"1866-01-14","end":"9999-12-31"}],"D746":[{"name":"MONTEBELLO JONI
CO","start":"1890-04-22","end":"9999-12-31"}],"E048":[{"name":"GIOVO","start
":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-14","end":"9999-12-31",
"name":"GIOVO"}],"B897":[{"name":"CASALINO","start":"1869-04-26","end":"9999
-12-31"}],"A512":[{"start":"1861-03-17","end":"1927-01-11","name":"AVERSA"},
{"name":"AVERSA","start":"1927-01-12","end":"1929-04-11"},{"start":"1929-04-
12","end":"1945-08-31","name":"AVERSA"},{"end":"1946-07-02","start":"1945-09
-01","name":"AVERSA"},{"name":"AVERSA","start":"1946-07-03","end":"9999-12-3
1"}],"I119":[{"name":"SAN PIETRO VERNOTICO","end":"1927-01-11","start":"1861
-03-17"},{"name":"SAN PIETRO VERNOTICO","start":"1927-01-12","end":"9999-12-
31"}],"H823":[{"start":"1866-11-19","end":"1929-04-19","name":"SAN DON\u00c0
 DI PIAVE"},{"name":"SAN DON\u00c0 DI PIAVE","start":"1929-04-20","end":"193
6-12-30"},{"name":"SAN DON\u00c0 DI PIAVE","end":"9999-12-31","start":"1936-
12-31"}],"C081":[{"start":"1861-03-17","end":"9999-12-31","name":"CASTELDELF
INO"}],"B418":[{"end":"1935-04-07","start":"1861-03-17","name":"CALLIANO"},{
"start":"1935-04-08","end":"2022-08-11","name":"CALLIANO"},{"name":"CALLIANO
 MONFERRATO","start":"2022-08-12","end":"9999-12-31"}],"D273":[{"start":"192
0-10-16","end":"1923-02-13","name":"DENNO"},{"start":"1923-02-14","end":"192
8-10-29","name":"DENNO"},{"name":"DENNO","end":"1952-09-19","start":"1928-10
-30"},{"end":"2008-10-14","start":"1952-09-20","name":"DENNO"},{"end":"9999-
12-31","start":"2008-10-15","name":"DENNO"}],"D731":[{"name":"FORTEZZA","end
":"1973-08-14","start":"1942-01-27"},{"end":"9999-12-31","start":"1973-08-15
","name":"FORTEZZA"},{"start":"1923-04-27","end":"1927-01-11","name":"MEZZAS
ELVA"},{"name":"MEZZASELVA","start":"1927-01-12","end":"1942-01-26"},{"end":
"1923-02-13","start":"1920-10-16","name":"MEZZASELVA ALL'ISARCO"},{"name":"M
EZZASELVA ALL'ISARCO","start":"1923-02-14","end":"1923-04-26"}],"I921":[{"na
me":"SPOLETO","end":"1927-04-07","start":"1861-03-17"},{"start":"1927-04-08"
,"end":"1930-09-13","name":"SPOLETO"},{"name":"SPOLETO","end":"9999-12-31","
start":"1930-09-14"}],"A300":[{"name":"CASTELLO DI ANNONE","start":"1863-02-
16","end":"1935-04-07"},{"end":"9999-12-31","start":"1935-04-08","name":"CAS
TELLO DI ANNONE"}],"Z340":[{"start":"0000-01-01","end":"9999-12-31","name":"
Sant'Elena"}],"H091":[{"end":"9999-12-31","start":"1861-03-17","name":"PUMEN
ENGO"}],"E691":[{"name":"LORETO APRUTINO","end":"1927-01-11","start":"1863-0
7-13"},{"end":"9999-12-31","start":"1927-01-12","name":"LORETO APRUTINO"}],"
A865":[{"start":"1861-03-17","end":"1927-01-11","name":"BIESTRO"},{"name":"B
IESTRO","start":"1927-01-12","end":"1928-07-27"}],"G449":[{"end":"1908-01-21
","start":"1861-03-17","name":"PERETO"},{"end":"9999-12-31","start":"1908-01
-22","name":"PERETO"}],"B714":[{"name":"CAPRONNO","end":"1927-01-11","start"
:"1861-03-17"},{"end":"1928-05-30","start":"1927-01-12","name":"CAPRONNO"}],
"F198":[{"name":"MIGLIARINO","end":"1963-07-30","start":"1883-12-17"},{"star
t":"1963-07-31","end":"1986-10-18","name":"MIGLIARINO"},{"start":"1986-10-19
","end":"2013-12-31","name":"MIGLIARINO"}],"Z246":[{"name":"Yemen","end":"99
99-12-31","start":"0000-01-01"}],"D146":[{"end":"1927-12-27","start":"1861-0
3-17","name":"CREMEZZANO"}],"C524":[{"start":"1863-02-23","end":"9999-12-31"
,"name":"CERRETO D'ESI"}],"G263":[{"start":"1861-03-17","end":"9999-12-31","
name":"PALAZZO ADRIANO"}],"A206":[{"end":"9999-12-31","start":"1866-11-19","
name":"ALLEGHE"}],"A961":[{"name":"BONASSOLA","end":"1923-12-21","start":"18
61-03-17"},{"name":"BONASSOLA","start":"1923-12-22","end":"9999-12-31"}],"I7
19":[{"end":"1974-08-19","start":"1947-12-13","name":"SIAMANNA-SIAPICCIA"},{
"start":"1974-08-20","end":"1974-09-12","name":"SIAMANNA-SIAPICCIA"}],"H648"
:[{"name":"SABBIA","start":"1861-03-17","end":"1927-01-11"},{"name":"SABBIA"
,"start":"1927-01-12","end":"2017-12-31"}],"C926":[{"end":"9999-12-31","star
t":"1861-03-17","name":"COMIGNAGO"}],"C629":[{"start":"1861-03-17","end":"19
27-01-11","name":"CHIESANUOVA"},{"start":"1927-01-12","end":"1928-09-25","na
me":"CHIESANUOVA"},{"name":"CHIESANUOVA","end":"9999-12-31","start":"1948-01
-15"}],"M011":[{"name":"VILLAROSA","end":"1927-01-11","start":"1861-03-17"},
{"start":"1927-01-12","end":"1937-03-01","name":"VILLAROSA"},{"name":"VILLAR
OSA","end":"9999-12-31","start":"1937-03-02"}],"L067":[{"end":"1909-04-13","
start":"1893-01-02","name":"TAVARNELLE"},{"name":"TAVARNELLE VAL DI PESA","s
tart":"1909-04-14","end":"2018-12-31"}],"F663":[{"end":"1927-01-11","start":
"1861-03-17","name":"MONTEVIASCO"},{"name":"MONTEVIASCO","start":"1927-01-12
","end":"1928-08-06"}],"D131":[{"name":"CRANDOLA","end":"1928-05-31","start"
:"1861-03-17"},{"end":"1962-07-02","start":"1957-05-03","name":"CRANDOLA"},{
"start":"1962-07-03","end":"1992-04-15","name":"CRANDOLA VALSASSINA"},{"name
":"CRANDOLA VALSASSINA","end":"9999-12-31","start":"1992-04-16"}],"B840":[{"
start":"1861-03-17","end":"9999-12-31","name":"CARROSIO"}],"L496":[{"start":
"1861-03-17","end":"1974-08-19","name":"URAS"},{"end":"9999-12-31","start":"
1974-08-20","name":"URAS"}],"F494":[{"name":"MONTEFALCONE DI VAL FORTORE","e
nd":"9999-12-31","start":"1863-02-06"}],"I086":[{"start":"1912-12-18","end":
"9999-12-31","name":"SAN PIERO PATTI"},{"name":"SAN PIETRO SOPRA PATTI","sta
rt":"1861-03-17","end":"1912-12-17"}],"D600":[{"name":"FINALE LIGURE","end":
"9999-12-31","start":"1927-01-12"}],"L195":[{"name":"TOLMEZZO","start":"1866
-11-19","end":"1968-04-05"},{"name":"TOLMEZZO","end":"9999-12-31","start":"1
968-04-06"}],"I047":[{"name":"SAN NAZARIO","end":"2019-01-29","start":"1866-
11-19"}],"C708":[{"name":"CINISI","end":"9999-12-31","start":"1861-03-17"}],
"L311":[{"start":"1861-03-17","end":"1928-07-28","name":"TOSCOLANO"}],"M367"
:[{"end":"9999-12-31","start":"2016-01-01","name":"POLESINE ZIBELLO"}],"D445
":[{"name":"EUPILIO","start":"1927-07-15","end":"9999-12-31"}],"B943":[{"end
":"1992-04-15","start":"1874-11-17","name":"CASATENOVO"},{"start":"1992-04-1
6","end":"9999-12-31","name":"CASATENOVO"}],"A659":[{"start":"1861-03-17","e
nd":"1878-06-21","name":"BARGANO"}],"A956":[{"start":"1861-03-17","end":"999
9-12-31","name":"BOMBA"}],"B152":[{"name":"BRENTINO BELLUNO","start":"1928-1
0-17","end":"9999-12-31"}],"A014":[{"start":"1938-07-09","end":"9999-12-31",
"name":"ACATE"},{"name":"BISCARI","end":"1927-01-11","start":"1861-03-17"},{
"start":"1927-01-12","end":"1937-07-16","name":"BISCARI"},{"start":"1937-07-
17","end":"1938-07-08","name":"BISCARI"}],"D187":[{"name":"CROVA","end":"192
7-01-11","start":"1891-03-06"},{"name":"CROVA","end":"9999-12-31","start":"1
927-01-12"}],"Z733":[{"name":"Vanuatu","end":"9999-12-31","start":"0000-01-0
1"}],"L597":[{"start":"1861-03-17","end":"9999-12-31","name":"VALLE CASTELLA
NA"}],"C550":[{"start":"1861-03-17","end":"9999-12-31","name":"CERVERE"}],"G
349":[{"start":"1877-10-31","end":"1928-02-22","name":"PARUZZARO"},{"name":"
PARUZZARO","start":"1928-02-23","end":"1947-11-27"},{"start":"1947-11-28","e
nd":"1960-09-28","name":"PARUZZARO"},{"name":"PARUZZARO","end":"9999-12-31",
"start":"1960-09-29"}],"G043":[{"end":"9999-12-31","start":"1991-07-10","nam
e":"OLLASTRA"},{"start":"1861-03-17","end":"1928-03-12","name":"OLLASTRA SIM
AXIS"},{"end":"1974-08-19","start":"1946-04-13","name":"OLLASTRA SIMAXIS"},{
"start":"1974-08-20","end":"1991-07-09","name":"OLLASTRA SIMAXIS"}],"D171":[
{"end":"1923-12-21","start":"1919-11-29","name":"CRISPIANO"},{"end":"9999-12
-31","start":"1923-12-22","name":"CRISPIANO"}],"L625":[{"start":"1871-01-15"
,"end":"9999-12-31","name":"VALLINFREDA"}],"H912":[{"end":"9999-12-31","star
t":"1922-09-19","name":"SAN GIOVANNI DEL DOSSO"}],"L396":[{"start":"1866-11-
19","end":"1929-04-04","name":"TREVENZUOLO"},{"end":"9999-12-31","start":"19
29-04-05","name":"TREVENZUOLO"}],"F394":[{"name":"MONTAGNANA","start":"1866-
11-19","end":"9999-12-31"}],"L247":[{"name":"TORRE BAIRO","start":"1861-03-1
7","end":"1927-01-11"},{"start":"1927-01-12","end":"1929-01-15","name":"TORR
E BAIRO"},{"end":"1961-10-02","start":"1955-04-29","name":"TORRE BAIRO"},{"n
ame":"TORRE CANAVESE","start":"1961-10-03","end":"9999-12-31"}],"G761":[{"en
d":"9999-12-31","start":"1861-03-17","name":"POGGIO IMPERIALE"}],"F709":[{"n
ame":"MORANSENGO","start":"1861-03-17","end":"1928-11-30"},{"name":"MORANSEN
GO","start":"1947-06-23","end":"2022-12-31"}],"F997":[{"end":"9999-12-31","s
tart":"1861-03-17","name":"ODALENGO GRANDE"}],"D585":[{"name":"FIGU-GONNOSN\
u00d2","start":"1947-10-17","end":"1964-06-08"},{"start":"1964-06-09","end":
"1974-08-19","name":"GONNOSN\u00d2"},{"start":"1974-08-20","end":"9999-12-31
","name":"GONNOSN\u00d2"}],"E182":[{"start":"1861-03-17","end":"1915-12-19",
"name":"GRINZANE"},{"name":"GRINZANE CAVOUR","start":"1915-12-20","end":"193
1-02-05"},{"name":"GRINZANE CAVOUR","end":"9999-12-31","start":"1947-12-22"}
],"E919":[{"name":"MARATEA","end":"9999-12-31","start":"1861-03-17"}],"F628"
:[{"name":"MONTE SAN SAVINO","end":"9999-12-31","start":"1861-03-17"}],"E616
":[{"name":"LISSAGO","start":"1861-03-17","end":"1927-01-11"},{"name":"LISSA
GO","end":"1927-12-28","start":"1927-01-12"}],"A520":[{"name":"AVIO","end":"
1923-02-13","start":"1920-10-16"},{"name":"AVIO","start":"1923-02-14","end":
"1928-04-16"},{"name":"AVIO","end":"9999-12-31","start":"1928-04-17"}],"F284
":[{"end":"9999-12-31","start":"1861-03-17","name":"MOLFETTA"}],"L286":[{"en
d":"1923-03-18","start":"1876-03-17","name":"TORRI IN SABINA"},{"end":"1927-
01-11","start":"1923-03-19","name":"TORRI IN SABINA"},{"name":"TORRI IN SABI
NA","start":"1927-01-12","end":"9999-12-31"}],"E121":[{"start":"1921-01-05",
"end":"1923-02-07","name":"GRACOVA SERRAVALLE"},{"name":"GRACOVA SERRAVALLE"
,"start":"1923-02-08","end":"1927-01-11"},{"end":"1947-09-15","start":"1927-
01-12","name":"GRACOVA SERRAVALLE"}],"D988":[{"end":"1992-04-15","start":"18
61-03-17","name":"GEROCARNE"},{"name":"GEROCARNE","start":"1992-04-16","end"
:"9999-12-31"}],"G589":[{"name":"PICCIANO","start":"1861-03-17","end":"1927-
01-11"},{"name":"PICCIANO","end":"9999-12-31","start":"1927-01-12"}],"G684":
[{"start":"1863-01-05","end":"9999-12-31","name":"PIOBESI TORINESE"}],"H189"
:[{"name":"RASUN-ANTERSELVA","end":"9999-12-31","start":"1955-03-02"}],"G756
":[{"end":"1923-03-18","start":"1861-03-17","name":"POGGIO BUSTONE"},{"end":
"1927-01-11","start":"1923-03-19","name":"POGGIO BUSTONE"},{"name":"POGGIO B
USTONE","start":"1927-01-12","end":"9999-12-31"}],"G228":[{"end":"9999-12-31
","start":"1861-03-17","name":"PAESANA"}],"M096":[{"start":"1866-11-19","end
":"1968-04-05","name":"VIVARO"},{"start":"1968-04-06","end":"9999-12-31","na
me":"VIVARO"}],"Z511":[{"name":"Honduras","end":"9999-12-31","start":"0000-0
1-01"}],"L411":[{"name":"TREZZO SULL'ADDA","end":"9999-12-31","start":"1869-
02-01"}],"D345":[{"end":"1927-01-11","start":"1861-03-17","name":"DORGALI"},
{"start":"1927-01-12","end":"9999-12-31","name":"DORGALI"}],"C199":[{"name":
"CASTELLUCCIO INFERIORE","end":"1928-05-11","start":"1861-03-17"},{"start":"
1935-06-16","end":"9999-12-31","name":"CASTELLUCCIO INFERIORE"}],"B660":[{"n
ame":"CAPIZZI","start":"1861-03-17","end":"9999-12-31"}],"F843":[{"name":"NA
RDODIPACE","end":"1978-12-28","start":"1902-01-22"},{"name":"NARDODIPACE","e
nd":"1992-04-15","start":"1978-12-29"},{"end":"9999-12-31","start":"1992-04-
16","name":"NARDODIPACE"}],"C077":[{"name":"CASTELDARNE","end":"1927-01-11",
"start":"1923-04-27"},{"name":"CASTELDARNE","end":"1929-03-24","start":"1927
-01-12"},{"name":"CASTELLO DI PUSTERIA","start":"1920-10-16","end":"1923-02-
13"},{"end":"1923-04-26","start":"1923-02-14","name":"CASTELLO DI PUSTERIA"}
],"A227":[{"name":"ALTAVILLA MONFERRATO","end":"9999-12-31","start":"1863-02
-16"}],"F462":[{"start":"1948-04-30","end":"9999-12-31","name":"MONTECCHIO"}
],"E804":[{"name":"MAGHERNO","start":"1861-03-17","end":"1937-03-31"},{"end"
:"1948-03-20","start":"1937-04-01","name":"MAGHERNO"},{"end":"9999-12-31","s
tart":"1948-03-21","name":"MAGHERNO"}],"E721":[{"name":"LUCINO","end":"1928-
06-14","start":"1861-03-17"}],"H556":[{"name":"ROS\u00c0","end":"9999-12-31"
,"start":"1866-11-19"}],"H868":[{"end":"9999-12-31","start":"1864-04-01","na
me":"SAN GIACOMO FILIPPO"}],"I560":[{"name":"SECONDIGLIANO","start":"1861-03
-17","end":"1926-07-03"}],"L380":[{"end":"1929-01-23","start":"1861-03-17","
name":"TRENZANO"},{"name":"TRENZANO","start":"1929-01-24","end":"9999-12-31"
}],"C304":[{"end":"1938-02-18","start":"1863-01-26","name":"CASTIGLIONE D'AD
DA"},{"start":"1938-02-19","end":"1992-04-15","name":"CASTIGLIONE D'ADDA"},{
"end":"9999-12-31","start":"1992-04-16","name":"CASTIGLIONE D'ADDA"}],"H789"
:[{"name":"SAN CARLO CANAVESE","end":"1927-12-30","start":"1903-04-25"},{"st
art":"1947-01-09","end":"9999-12-31","name":"SAN CARLO CANAVESE"},{"name":"S
AN CARLO DI CIRI\u00c8","end":"1903-04-24","start":"1863-01-05"}],"G156":[{"
name":"OSILO","end":"1980-07-10","start":"1861-03-17"},{"name":"OSILO","end"
:"9999-12-31","start":"1980-07-11"}],"H954":[{"end":"1927-01-11","start":"18
66-07-12","name":"SAN LEUCIO"},{"name":"SAN LEUCIO","start":"1927-01-12","en
d":"1928-06-27"}],"A839":[{"name":"BEZZECCA","start":"1920-10-16","end":"192
3-02-13"},{"name":"BEZZECCA","start":"1923-02-14","end":"1928-02-08"},{"end"
:"1952-09-19","start":"1928-02-09","name":"BEZZECCA"},{"end":"2009-12-31","s
tart":"1952-09-20","name":"BEZZECCA"}],"A645":[{"name":"BARDELLO","start":"1
861-03-17","end":"1927-01-11"},{"end":"2022-12-31","start":"1927-01-12","nam
e":"BARDELLO"}],"H016":[{"name":"PRECOTTO","start":"1861-03-17","end":"1920-
07-15"}],"D459":[{"end":"9999-12-31","start":"1861-03-17","name":"FAETO"}],"
Z605":[{"start":"0000-01-01","end":"9999-12-31","name":"Ecuador"}],"C907":[{
"name":"COLSANO","start":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-
14","end":"1927-01-11","name":"COLSANO"},{"name":"COLSANO","end":"1929-02-14
","start":"1927-01-12"}],"C799":[{"name":"CLUSIO","end":"1923-02-13","start"
:"1920-10-16"},{"name":"CLUSIO","end":"1927-01-11","start":"1923-02-14"},{"n
ame":"CLUSIO","start":"1927-01-12","end":"1928-01-26"}],"H357":[{"name":"RIV
OLTA D'ADDA","end":"9999-12-31","start":"1863-08-10"}],"D233":[{"end":"9999-
12-31","start":"1861-03-17","name":"CUSIO"}],"Z133":[{"name":"Svizzera","sta
rt":"0000-01-01","end":"9999-12-31"}],"L290":[{"end":"1927-01-11","start":"1
871-01-15","name":"TORRICE"},{"start":"1927-01-12","end":"9999-12-31","name"
:"TORRICE"}],"E268":[{"end":"1927-01-11","start":"1861-03-17","name":"GURONE
"},{"end":"1928-01-17","start":"1927-01-12","name":"GURONE"}],"G455":[{"name
":"PERITO","start":"1861-03-17","end":"9999-12-31"}],"H204":[{"start":"1861-
03-17","end":"1929-04-18","name":"REA"},{"name":"REA","end":"2004-08-27","st
art":"1954-07-15"},{"name":"REA","start":"2004-08-28","end":"2005-01-07"},{"
name":"REA","start":"2005-01-08","end":"9999-12-31"}],"H561":[{"end":"1927-0
1-11","start":"1906-08-26","name":"ROSAZZA"},{"name":"ROSAZZA","end":"1992-0
4-15","start":"1927-01-12"},{"start":"1992-04-16","end":"9999-12-31","name":
"ROSAZZA"}],"D771":[{"name":"FRASCAROLO","start":"1861-03-17","end":"9999-12
-31"}],"L540":[{"start":"1862-12-29","end":"1927-01-11","name":"VAIRANO PATE
NORA"},{"end":"1945-08-31","start":"1927-01-12","name":"VAIRANO PATENORA"},{
"end":"9999-12-31","start":"1945-09-01","name":"VAIRANO PATENORA"}],"M080":[
{"end":"1927-01-11","start":"1861-03-17","name":"VISTRORIO"},{"name":"VISTRO
RIO","end":"1945-10-04","start":"1927-01-12"},{"start":"1945-10-05","end":"9
999-12-31","name":"VISTRORIO"}],"G161":[{"name":"OSNAGO","end":"1992-04-15",
"start":"1861-03-17"},{"name":"OSNAGO","start":"1992-04-16","end":"9999-12-3
1"}],"A400":[{"end":"1929-01-12","start":"1867-07-22","name":"ARIANO NEL POL
ESINE"},{"start":"1929-01-13","end":"1938-02-09","name":"ARIANO NEL POLESINE
"},{"name":"ARIANO NEL POLESINE","start":"1938-02-10","end":"9999-12-31"}],"
F109":[{"name":"MELISSANO","start":"1921-05-12","end":"9999-12-31"}],"E782":
[{"end":"9999-12-31","start":"1861-03-17","name":"MACELLO"}],"B318":[{"start
":"1921-12-31","end":"1923-02-07","name":"CACCIA"},{"name":"CACCIA","end":"1
925-08-22","start":"1923-02-08"}],"D723":[{"name":"FORNO","start":"1920-10-1
6","end":"1923-02-13"},{"name":"FORNO","end":"1928-10-06","start":"1923-02-1
4"}],"I933":[{"name":"STAGHIGLIONE","end":"1928-11-28","start":"1861-03-17"}
],"L142":[{"end":"9999-12-31","start":"1913-07-13","name":"TERZIGNO"}],"A781
":[{"start":"1861-03-17","end":"9999-12-31","name":"BENETUTTI"}],"B108":[{"s
tart":"1920-10-16","end":"1923-02-13","name":"BOZZANA"},{"name":"BOZZANA","e
nd":"1928-12-31","start":"1923-02-14"}],"I409":[{"name":"SAN VITTORE OLONA",
"end":"1928-10-06","start":"1864-04-01"},{"end":"9999-12-31","start":"1928-1
0-07","name":"SAN VITTORE OLONA"}],"F013":[{"start":"1866-11-19","end":"9999
-12-31","name":"MASI"}],"F319":[{"name":"MOMPEO","start":"1861-03-17","end":
"1923-03-18"},{"name":"MOMPEO","start":"1923-03-19","end":"1927-01-11"},{"en
d":"9999-12-31","start":"1927-01-12","name":"MOMPEO"}],"H831":[{"name":"SAN 
FELE","end":"9999-12-31","start":"1861-03-17"}],"L953":[{"end":"1927-01-11",
"start":"1861-03-17","name":"VILLAGRANDE STRISAILI"},{"name":"VILLAGRANDE ST
RISAILI","start":"1927-01-12","end":"2005-12-31"},{"name":"VILLAGRANDE STRIS
AILI","end":"2016-04-27","start":"2006-01-01"},{"name":"VILLAGRANDE STRISAIL
I","start":"2016-04-28","end":"9999-12-31"}],"L617":[{"name":"VALLE SALIMBEN
E","end":"1902-02-16","start":"1871-06-01"},{"name":"VALLE SALIMBENE","end":
"1939-07-12","start":"1902-02-17"},{"name":"VALLE SALIMBENE","start":"1939-0
7-13","end":"9999-12-31"}],"B022":[{"start":"1867-08-19","end":"1928-07-28",
"name":"BORGONATO"}],"E287":[{"start":"1861-03-17","end":"1928-07-28","name"
:"IMBERSAGO"},{"name":"IMBERSAGO","end":"1992-04-15","start":"1948-03-30"},{
"name":"IMBERSAGO","start":"1992-04-16","end":"9999-12-31"}],"F553":[{"end":
"9999-12-31","start":"1863-02-06","name":"MONTEMAGGIORE BELSITO"}],"B404":[{
"name":"CALDONAZZO","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-
02-14","end":"1929-01-23","name":"CALDONAZZO"},{"name":"CALDONAZZO","end":"1
947-05-16","start":"1929-01-24"},{"name":"CALDONAZZO","start":"1947-05-17","
end":"1988-08-02"},{"name":"CALDONAZZO","end":"9999-12-31","start":"1988-08-
03"}],"D492":[{"name":"FARA NOVARESE","start":"1863-04-23","end":"9999-12-31
"}],"M209":[{"end":"1974-08-19","start":"1965-01-02","name":"SANT'ANNA ARRES
I"},{"name":"SANT'ANNA ARRESI","start":"1974-08-20","end":"2005-12-31"},{"na
me":"SANT'ANNA ARRESI","start":"2006-01-01","end":"2016-04-27"},{"name":"SAN
T'ANNA ARRESI","end":"9999-12-31","start":"2016-04-28"}],"C752":[{"start":"1
861-03-17","end":"1992-04-15","name":"CIVATE"},{"end":"9999-12-31","start":"
1992-04-16","name":"CIVATE"}],"B037":[{"end":"9999-12-31","start":"1861-03-1
7","name":"BORGO SAN MARTINO"}],"H935":[{"name":"SAN GIUSTINO","end":"9999-1
2-31","start":"1861-03-17"}],"H846":[{"end":"9999-12-31","start":"1861-03-17
","name":"SAN FLORO"}],"C897":[{"end":"9999-12-31","start":"1867-08-26","nam
e":"COLOGNOLA AI COLLI"}],"L850":[{"name":"VICOPISANO","start":"1884-09-16",
"end":"1928-06-29"},{"end":"1929-06-17","start":"1928-06-30","name":"VICOPIS
ANO"},{"start":"1929-06-18","end":"9999-12-31","name":"VICOPISANO"}],"A493":
[{"name":"AUDITORE","start":"1861-03-17","end":"1998-09-03"},{"name":"AUDITO
RE","end":"2018-12-31","start":"1998-09-04"}],"G434":[{"start":"1861-03-17",
"end":"9999-12-31","name":"PENNADOMO"}],"E773":[{"end":"9999-12-31","start":
"1861-03-17","name":"LUZZI"}],"A737":[{"end":"9999-12-31","start":"1867-08-2
6","name":"BELFIORE"}],"A664":[{"name":"BARIANO","start":"1861-03-17","end":
"9999-12-31"}],"C114":[{"name":"CASTEL FRENTANO","start":"1864-02-10","end":
"9999-12-31"}],"I830":[{"end":"9999-12-31","start":"1861-03-17","name":"SONG
AVAZZO"}],"G007":[{"end":"1992-05-22","start":"1861-03-17","name":"OGGEBBIO"
},{"name":"OGGEBBIO","end":"9999-12-31","start":"1992-05-23"}],"H887":[{"sta
rt":"1862-11-28","end":"9999-12-31","name":"SAN GIORGIO PIACENTINO"}],"E596"
:[{"start":"1863-01-26","end":"1905-03-09","name":"LIMONE SAN GIOVANNI"},{"e
nd":"9999-12-31","start":"1905-03-10","name":"LIMONE SUL GARDA"}],"H538":[{"
name":"RONCO BIELLESE","start":"1863-04-23","end":"1927-01-11"},{"start":"19
27-01-12","end":"1992-04-15","name":"RONCO BIELLESE"},{"end":"9999-12-31","s
tart":"1992-04-16","name":"RONCO BIELLESE"}],"L203":[{"name":"TONCO","start"
:"1861-03-17","end":"1935-04-07"},{"end":"9999-12-31","start":"1935-04-08","
name":"TONCO"}],"A722":[{"name":"BAUNEI","start":"1861-03-17","end":"1927-01
-11"},{"end":"1960-03-10","start":"1927-01-12","name":"BAUNEI"},{"end":"2005
-12-31","start":"1960-03-11","name":"BAUNEI"},{"name":"BAUNEI","end":"2016-0
4-27","start":"2006-01-01"},{"name":"BAUNEI","start":"2016-04-28","end":"999
9-12-31"}],"G138":[{"end":"1865-12-24","start":"1861-03-17","name":"ORTI"}],
"E994":[{"end":"1902-06-30","start":"1869-02-08","name":"MARUDO"},{"name":"M
ARUDO","end":"1992-04-15","start":"1902-07-01"},{"name":"MARUDO","start":"19
92-04-16","end":"9999-12-31"}],"G977":[{"name":"PRALBOINO","end":"1928-08-14
","start":"1861-03-17"},{"name":"PRALBOINO","start":"1928-08-15","end":"1947
-12-21"},{"start":"1947-12-22","end":"9999-12-31","name":"PRALBOINO"}],"E246
":[{"start":"1861-03-17","end":"9999-12-31","name":"GUARDIA PERTICARA"}],"E3
97":[{"name":"LACEDONIA","end":"9999-12-31","start":"1861-03-17"}],"G374":[{
"start":"1871-01-15","end":"1927-01-11","name":"PATRICA"},{"name":"PATRICA",
"start":"1927-01-12","end":"9999-12-31"}],"C418":[{"start":"1861-03-17","end
":"9999-12-31","name":"CEDRASCO"}],"B081":[{"start":"1861-03-17","end":"1929
-07-09","name":"BOSISIO"},{"name":"BOSISIO PARINI","end":"1992-04-15","start
":"1929-07-10"},{"start":"1992-04-16","end":"2004-06-07","name":"BOSISIO PAR
INI"},{"name":"BOSISIO PARINI","start":"2004-06-08","end":"9999-12-31"}],"A5
69":[{"end":"9999-12-31","start":"1862-09-08","name":"BAGNOLO MELLA"}],"I162
":[{"end":"9999-12-31","start":"2003-01-01","name":"SAN SIRO"}],"A215":[{"na
me":"VILLA D'ALM\u00c8","start":"1885-01-15","end":"1927-07-30"},{"name":"VI
LLA D'ALM\u00c8","start":"1951-09-11","end":"9999-12-31"}],"D364":[{"end":"1
992-04-15","start":"1861-03-17","name":"DRAPIA"},{"name":"DRAPIA","end":"999
9-12-31","start":"1992-04-16"}],"B524":[{"end":"9999-12-31","start":"1866-11
-19","name":"CAMPODARSEGO"}],"E173":[{"name":"GRICIGNANO DI AVERSA","end":"1
927-01-11","start":"1872-01-01"},{"end":"1929-04-12","start":"1927-01-12","n
ame":"GRICIGNANO DI AVERSA"},{"end":"9999-12-31","start":"1946-07-03","name"
:"GRICIGNANO DI AVERSA"}],"A137":[{"name":"ALBAREDO D'ADIGE","start":"1867-0
8-26","end":"9999-12-31"}],"A978":[{"start":"1875-06-22","end":"9999-12-31",
"name":"BONORVA"}],"C714":[{"start":"1867-08-05","end":"9999-12-31","name":"
CINTO CAOMAGGIORE"}],"I288":[{"name":"SANT'ANGELO LE FRATTE","end":"9999-12-
31","start":"1861-03-17"}],"E340":[{"end":"1927-01-11","start":"1869-07-06",
"name":"ISOLA DEL LIRI"},{"end":"9999-12-31","start":"1927-01-12","name":"IS
OLA DEL LIRI"}],"M184":[{"end":"1928-03-26","start":"1861-03-17","name":"ZOG
NO"},{"end":"2002-10-25","start":"1928-03-27","name":"ZOGNO"},{"name":"ZOGNO
","start":"2002-10-26","end":"9999-12-31"}],"I624":[{"end":"1974-08-19","sta
rt":"1861-03-17","name":"SERDIANA"},{"end":"2016-04-27","start":"1974-08-20"
,"name":"SERDIANA"},{"end":"9999-12-31","start":"2016-04-28","name":"SERDIAN
A"}],"G813":[{"end":"9999-12-31","start":"1928-04-28","name":"POMPEI"}],"D96
7":[{"name":"GENOLA","start":"1861-03-17","end":"9999-12-31"}],"I529":[{"end
":"9999-12-31","start":"1861-03-17","name":"SCHIGNANO"}],"A122":[{"name":"AL
ASSIO","end":"1927-01-11","start":"1861-03-17"},{"name":"ALASSIO","start":"1
927-01-12","end":"9999-12-31"}],"C840":[{"end":"2015-12-31","start":"1872-03
-11","name":"COLLAGNA"}],"G738":[{"end":"1923-02-07","start":"1921-01-05","n
ame":"PLEZZO"},{"end":"1927-01-11","start":"1923-02-08","name":"PLEZZO"},{"n
ame":"PLEZZO","end":"1928-05-03","start":"1927-01-12"},{"end":"1947-09-15","
start":"1928-05-04","name":"PLEZZO"}],"B629":[{"name":"CANTALUPO LIGURE","en
d":"1929-05-27","start":"1863-01-26"},{"end":"9999-12-31","start":"1947-09-0
4","name":"CANTALUPO LIGURE"}],"B926":[{"name":"CASANOVA DI UGGIATE","end":"
1903-05-09","start":"1864-03-11"},{"name":"CASANOVA LANZA","start":"1903-05-
10","end":"1928-05-03"}],"L670":[{"end":"9999-12-31","start":"1861-03-17","n
ame":"VARALLO POMBIA"}],"A008":[{"end":"1927-01-11","start":"1861-03-17","na
me":"ABBATEGGIO"},{"name":"ABBATEGGIO","start":"1927-01-12","end":"1929-05-2
2"},{"start":"1947-01-10","end":"9999-12-31","name":"ABBATEGGIO"}],"I762":[{
"name":"SISINI","start":"1861-03-17","end":"1927-08-27"}],"Z223":[{"name":"I
ndonesia","start":"0000-01-01","end":"9999-12-31"}],"D123":[{"name":"COTRONE
I","end":"1992-04-15","start":"1861-03-17"},{"name":"COTRONEI","end":"9999-1
2-31","start":"1992-04-16"}],"L742":[{"name":"VENTOTENE","end":"1934-10-25",
"start":"1861-03-17"},{"name":"VENTOTENE","start":"1934-10-26","end":"1934-1
2-17"},{"name":"VENTOTENE","end":"1935-07-30","start":"1934-12-18"},{"start"
:"1935-07-31","end":"1937-05-19","name":"VENTOTENE"},{"name":"VENTOTENE","st
art":"1937-05-20","end":"1945-04-08"},{"name":"VENTOTENE","end":"9999-12-31"
,"start":"1945-04-09"}],"F807":[{"start":"1866-11-19","end":"1924-01-13","na
me":"MURANO"}],"A181":[{"start":"1863-01-19","end":"9999-12-31","name":"ALES
SANDRIA DELLA ROCCA"}],"B708":[{"name":"CAPRILE","start":"1861-03-17","end":
"1927-01-11"},{"name":"CAPRILE","end":"1992-04-15","start":"1927-01-12"},{"s
tart":"1992-04-16","end":"9999-12-31","name":"CAPRILE"}],"G910":[{"start":"1
861-03-17","end":"1970-03-02","name":"PORTOCANNONE"},{"name":"PORTOCANNONE",
"start":"1970-03-03","end":"9999-12-31"}],"E580":[{"name":"LICUSATI","end":"
1928-10-06","start":"1861-03-17"}],"A575":[{"start":"1866-11-19","end":"9999
-12-31","name":"BAGNOLO SAN VITO"}],"C152":[{"name":"CASTELLEONE DI SUASA","
start":"1864-11-07","end":"9999-12-31"}],"C943":[{"name":"CONCAMARISE","end"
:"1929-05-29","start":"1866-11-19"},{"start":"1948-03-31","end":"9999-12-31"
,"name":"CONCAMARISE"}],"I398":[{"end":"1927-01-11","start":"1863-04-13","na
me":"SAN VITO IN MONTE"},{"start":"1927-01-12","end":"1929-02-04","name":"SA
N VITO IN MONTE"}],"E440":[{"name":"LANISCHIE","end":"1947-09-15","start":"1
929-05-07"}],"H969":[{"name":"SAN LORENZO NUOVO","start":"1871-01-15","end":
"1927-01-11"},{"name":"SAN LORENZO NUOVO","end":"9999-12-31","start":"1927-0
1-12"}],"H666":[{"name":"SAGRON MIS","end":"1923-02-13","start":"1920-10-16"
},{"start":"1923-02-14","end":"1927-11-23","name":"SAGRON MIS"},{"start":"19
47-01-24","end":"9999-12-31","name":"SAGRON MIS"}],"G778":[{"name":"POLA","e
nd":"1923-02-07","start":"1921-01-05"},{"name":"POLA","start":"1923-02-08","
end":"1934-01-25"},{"name":"POLA","end":"1947-09-15","start":"1934-01-26"}],
"I650":[{"name":"SERRA PEDACE","start":"1861-03-17","end":"2017-05-04"}],"B5
50":[{"end":"1970-03-02","start":"1861-03-17","name":"CAMPOMARINO"},{"name":
"CAMPOMARINO","start":"1970-03-03","end":"9999-12-31"}],"L101":[{"name":"TEO
R","end":"1968-04-05","start":"1866-11-19"},{"start":"1968-04-06","end":"201
3-12-31","name":"TEOR"}],"D693":[{"name":"FORCHIA","end":"9999-12-31","start
":"1861-03-17"}],"D464":[{"name":"FAGNANO CASTELLO","end":"1928-07-17","star
t":"1863-01-19"},{"start":"1928-07-18","end":"1934-03-06","name":"FAGNANO CA
STELLO"},{"end":"9999-12-31","start":"1934-03-07","name":"FAGNANO CASTELLO"}
],"L630":[{"end":"1938-12-21","start":"1866-11-19","name":"VALLONARA"}],"E13
3":[{"name":"GRAMMICHELE","end":"9999-12-31","start":"1861-03-17"}],"A177":[
{"name":"ALCARA LI FUSI","start":"1868-01-01","end":"9999-12-31"}],"M388":[{
"name":"CASSANO SPINOLA","start":"2018-01-01","end":"9999-12-31"}],"D168":[{
"end":"1928-03-06","start":"1862-12-29","name":"CREVOLA D'OSSOLA"},{"end":"1
961-08-16","start":"1928-03-07","name":"CREVOLA D'OSSOLA"},{"name":"CREVOLAD
OSSOLA","end":"1992-05-22","start":"1961-08-17"},{"name":"CREVOLADOSSOLA","e
nd":"9999-12-31","start":"1992-05-23"}],"F085":[{"end":"1928-04-02","start":
"1861-03-17","name":"MEDOLAGO"},{"name":"MEDOLAGO","end":"9999-12-31","start
":"1970-03-17"}],"B010":[{"end":"1928-06-29","start":"1861-03-17","name":"BO
RGO DI TERZO"},{"name":"BORGO DI TERZO","start":"1947-11-04","end":"9999-12-
31"}],"G026":[{"end":"1927-06-16","start":"1863-02-23","name":"OLGIATE MOLGO
RA"},{"name":"OLGIATE MOLGORA","end":"1928-02-20","start":"1927-06-17"},{"na
me":"OLGIATE MOLGORA","start":"1953-09-13","end":"1992-04-15"},{"name":"OLGI
ATE MOLGORA","start":"1992-04-16","end":"9999-12-31"}],"A710":[{"start":"186
1-03-17","end":"9999-12-31","name":"BASTIA UMBRA"}],"M298":[{"name":"STATTE"
,"start":"1993-05-01","end":"9999-12-31"}],"H651":[{"start":"1861-03-17","en
d":"1928-02-23","name":"SABBIONCELLO"}],"A535":[{"end":"9999-12-31","start":
"1919-02-03","name":"BACOLI"}],"E962":[{"end":"9999-12-31","start":"1872-01-
01","name":"MARMIROLO"}],"E605":[{"name":"LIONI","end":"9999-12-31","start":
"1861-03-17"}],"I817":[{"end":"9999-12-31","start":"1861-03-17","name":"SOMA
NO"}],"L088":[{"name":"TELTI","end":"2005-12-31","start":"1963-09-27"},{"sta
rt":"2006-01-01","end":"2016-04-27","name":"TELTI"},{"end":"9999-12-31","sta
rt":"2016-04-28","name":"TELTI"}],"I498":[{"start":"1861-03-17","end":"9999-
12-31","name":"SCANDOLARA RIPA D'OGLIO"}],"A692":[{"end":"9999-12-31","start
":"1861-03-17","name":"BASCIANO"}],"A844":[{"end":"1929-03-27","start":"1861
-03-17","name":"BIANDRATE"},{"start":"1929-03-28","end":"9999-12-31","name":
"BIANDRATE"}],"A110":[{"end":"9999-12-31","start":"1861-03-17","name":"AIROL
A"}],"G981":[{"start":"1866-11-19","end":"9999-12-31","name":"PRAMAGGIORE"}]
,"I794":[{"name":"SOLBIATE OLONA","end":"1927-01-11","start":"1893-01-27"},{
"name":"SOLBIATE OLONA","start":"1927-01-12","end":"9999-12-31"}],"H218":[{"
start":"1868-07-01","end":"9999-12-31","name":"REDONDESCO"}],"I309":[{"end":
"9999-12-31","start":"1863-01-19","name":"SANTA SOFIA D'EPIRO"}],"I003":[{"n
ame":"SAN MARTINO BUON ALBERGO","end":"1927-10-06","start":"1866-11-19"},{"n
ame":"SAN MARTINO BUON ALBERGO","start":"1927-10-07","end":"1928-01-27"},{"s
tart":"1928-01-28","end":"9999-12-31","name":"SAN MARTINO BUON ALBERGO"}],"F
419":[{"end":"1928-12-29","start":"1872-11-06","name":"MONTALTO DI CASTRO"},
{"name":"MONTALTO DI CASTRO","start":"1928-12-30","end":"9999-12-31"}],"E511
":[{"name":"LEGGIUNO-SANGIANO","end":"1963-09-13","start":"1928-01-07"}],"B0
77":[{"end":"1928-04-27","start":"1877-03-09","name":"BOSCOTRECASE"},{"name"
:"BOSCOTRECASE","end":"1980-04-04","start":"1946-02-28"},{"end":"9999-12-31"
,"start":"1980-04-05","name":"BOSCOTRECASE"}],"H975":[{"name":"SAN MANGO SUL
 CALORE","end":"9999-12-31","start":"1863-05-06"}],"C660":[{"name":"CHIUSAVE
CCHIA","start":"1861-03-17","end":"1923-12-05"},{"name":"CHIUSAVECCHIA","end
":"1925-08-21","start":"1923-12-06"},{"end":"1958-08-27","start":"1925-08-22
","name":"CHIUSAVECCHIA"},{"end":"9999-12-31","start":"1958-08-28","name":"C
HIUSAVECCHIA"}],"B799":[{"name":"CARNEDO","start":"1920-10-16","end":"1923-0
2-13"},{"name":"CARNEDO","start":"1923-02-14","end":"1923-04-26"},{"start":"
1923-04-27","end":"1927-01-11","name":"CORNEDO"},{"name":"CORNEDO","start":"
1927-01-12","end":"1929-04-12"},{"end":"9999-12-31","start":"1929-04-13","na
me":"CORNEDO ALL'ISARCO"}],"B907":[{"end":"9999-12-31","start":"1868-07-01",
"name":"CASALOLDO"}],"I973":[{"end":"1927-01-11","start":"1871-01-15","name"
:"STRANGOLAGALLI"},{"name":"STRANGOLAGALLI","start":"1927-01-12","end":"9999
-12-31"}],"Z121":[{"name":"Malta","start":"0000-01-01","end":"9999-12-31"}],
"D392":[{"name":"EGNA","start":"1920-10-16","end":"1923-02-13"},{"end":"1948
-03-13","start":"1923-02-14","name":"EGNA"},{"name":"EGNA","end":"2005-05-03
","start":"1948-03-14"},{"name":"EGNA","start":"2005-05-04","end":"9999-12-3
1"}],"D221":[{"start":"1890-04-28","end":"1927-10-12","name":"CURNO"},{"end"
:"9999-12-31","start":"1947-10-20","name":"CURNO"}],"E051":[{"name":"GIRONIC
O","end":"2014-02-03","start":"1861-03-17"}],"A283":[{"start":"1866-11-19","
end":"1968-04-05","name":"ANDREIS"},{"name":"ANDREIS","start":"1968-04-06","
end":"9999-12-31"}],"H871":[{"name":"SAN GIACOMO IN COLLE","end":"1923-02-07
","start":"1921-01-05"},{"end":"1947-09-15","start":"1923-02-08","name":"SAN
 GIACOMO IN COLLE"}],"B304":[{"name":"BUTTAPIETRA","start":"1866-11-19","end
":"9999-12-31"}],"D889":[{"name":"GALZIGNANO","start":"1866-11-19","end":"19
79-08-27"},{"start":"1979-08-28","end":"9999-12-31","name":"GALZIGNANO TERME
"}],"G178":[{"end":"9999-12-31","start":"1861-03-17","name":"OSSI"}],"G937":
[{"name":"POSTALESIO","end":"9999-12-31","start":"1861-03-17"}],"B257":[{"st
art":"1861-03-17","end":"1928-02-22","name":"BUGNATE"}],"F826":[{"name":"MUS
ILE","end":"1920-06-24","start":"1866-11-19"},{"start":"1920-06-25","end":"1
936-12-30","name":"MUSILE DI PIAVE"},{"name":"MUSILE DI PIAVE","start":"1936
-12-31","end":"9999-12-31"}],"L824":[{"end":"1928-08-12","start":"1861-03-17
","name":"VEZZO"}],"L701":[{"name":"VECCANA","start":"1861-03-17","end":"192
7-01-11"},{"name":"VECCANA","start":"1927-01-12","end":"1928-07-28"}],"E818"
:[{"end":"9999-12-31","start":"1866-11-19","name":"MAGNACAVALLO"}],"D282":[{
"end":"1923-02-07","start":"1921-01-05","name":"DESCLA"},{"start":"1923-02-0
8","end":"1927-01-11","name":"DESCLA"},{"name":"DESCLA","start":"1927-01-12"
,"end":"1928-05-11"}],"A393":[{"end":"1932-01-11","start":"1861-03-17","name
":"ARGENTA"},{"name":"ARGENTA","end":"1938-12-24","start":"1932-01-12"},{"en
d":"9999-12-31","start":"1938-12-25","name":"ARGENTA"}],"G334":[{"name":"PAR
ETO","end":"9999-12-31","start":"1880-01-26"}],"A099":[{"name":"AIDUSSINA","
start":"1921-01-05","end":"1923-02-07"},{"name":"AIDUSSINA","start":"1923-02
-08","end":"1927-01-11"},{"end":"1928-07-27","start":"1927-01-12","name":"AI
DUSSINA"},{"name":"AIDUSSINA","end":"1947-09-15","start":"1928-07-28"}],"C31
8":[{"start":"1863-07-13","end":"9999-12-31","name":"CASTIGLION FIBOCCHI"}],
"G922":[{"name":"PORTOSCUSO","end":"1940-02-12","start":"1861-03-17"},{"name
":"PORTOSCUSO","end":"1974-08-19","start":"1945-04-21"},{"name":"PORTOSCUSO"
,"start":"1974-08-20","end":"2005-12-31"},{"name":"PORTOSCUSO","start":"2006
-01-01","end":"2016-04-27"},{"start":"2016-04-28","end":"9999-12-31","name":
"PORTOSCUSO"}],"A777":[{"name":"BEMA","end":"9999-12-31","start":"1861-03-17
"}],"E497":[{"end":"1928-11-09","start":"1861-03-17","name":"LAVENONE"},{"na
me":"LAVENONE","end":"9999-12-31","start":"1928-11-10"}],"G474":[{"end":"999
9-12-31","start":"1928-11-27","name":"PERTICA ALTA"}],"H578":[{"name":"ROSSA
NA","start":"1861-03-17","end":"9999-12-31"}],"D768":[{"end":"9999-12-31","s
tart":"1861-03-17","name":"FRANCOFONTE"}],"E066":[{"name":"GIUSVALLA","end":
"1927-01-11","start":"1861-03-17"},{"end":"9999-12-31","start":"1927-01-12",
"name":"GIUSVALLA"}],"I870":[{"start":"1861-03-17","end":"1928-11-26","name"
:"SOVAZZA"}],"E271":[{"name":"GUSSAGO","start":"1861-03-17","end":"9999-12-3
1"}],"M119":[{"start":"1861-03-17","end":"9999-12-31","name":"VOLPARA"}],"E3
68":[{"end":"1927-01-11","start":"1861-03-17","name":"ISSIGLIO"},{"end":"194
5-10-04","start":"1927-01-12","name":"ISSIGLIO"},{"name":"ISSIGLIO","start":
"1945-10-05","end":"9999-12-31"}],"H304":[{"name":"RIOMAGGIORE","start":"187
1-06-01","end":"1923-12-21"},{"name":"RIOMAGGIORE","end":"9999-12-31","start
":"1923-12-22"}],"C789":[{"name":"CLANEZZO","start":"1861-03-17","end":"1927
-04-01"},{"name":"UBIALE CLANEZZO","end":"9999-12-31","start":"1927-04-02"}]
,"C954":[{"end":"9999-12-31","start":"1861-03-17","name":"CONDOFURI"}],"B871
":[{"start":"1861-03-17","end":"1970-03-02","name":"CASALCIPRANO"},{"start":
"1970-03-03","end":"9999-12-31","name":"CASALCIPRANO"}],"C016":[{"name":"CAS
SINA MARIAGA","end":"1928-01-25","start":"1861-03-17"}],"L390":[{"end":"2018
-12-31","start":"1961-12-26","name":"TRESIGALLO"}],"B005":[{"name":"BORGIO V
EREZZI","start":"1933-05-24","end":"9999-12-31"}],"H907":[{"start":"1861-03-
17","end":"9999-12-31","name":"SAN GIOVANNI A PIRO"}],"C357":[{"name":"CATTO
LICA","end":"1992-04-15","start":"1895-12-20"},{"name":"CATTOLICA","end":"99
99-12-31","start":"1992-04-16"}],"H799":[{"name":"SAN CIPRIANO PO","end":"19
42-11-29","start":"1863-03-30"},{"end":"9999-12-31","start":"1942-11-30","na
me":"SAN CIPRIANO PO"}],"D333":[{"start":"1861-03-17","end":"1974-08-19","na
me":"DOMUS DE MARIA"},{"end":"2016-04-27","start":"1974-08-20","name":"DOMUS
 DE MARIA"},{"end":"9999-12-31","start":"2016-04-28","name":"DOMUS DE MARIA"
}],"D039":[{"start":"1866-11-19","end":"1928-06-14","name":"CORREZZO"}],"E85
7":[{"name":"MALGOLO","end":"1928-11-26","start":"1928-06-16"},{"name":"SALT
ER-MALGOLO","end":"1923-02-13","start":"1920-10-16"},{"name":"SALTER-MALGOLO
","end":"1928-06-15","start":"1923-02-14"}],"B218":[{"start":"1861-03-17","e
nd":"1930-11-05","name":"BRUNATE"},{"name":"BRUNATE","end":"1943-08-12","sta
rt":"1930-11-06"},{"end":"9999-12-31","start":"1947-11-03","name":"BRUNATE"}
],"D473":[{"end":"9999-12-31","start":"1863-01-19","name":"FALCONARA ALBANES
E"}],"B975":[{"start":"1863-02-23","end":"1928-06-13","name":"CASLINO AL PIA
NO"}],"H077":[{"end":"1929-01-01","start":"1928-06-14","name":"PROVAGLIO"},{
"name":"PROVAGLIO VAL SABBIA","end":"9999-12-31","start":"1929-01-02"}],"C55
6":[{"name":"CERVIGNANO","end":"1923-02-07","start":"1921-01-05"},{"end":"19
23-04-26","start":"1923-02-08","name":"CERVIGNANO"},{"end":"1928-05-01","sta
rt":"1923-04-27","name":"CERVIGNANO DEL FRIULI"},{"start":"1928-05-02","end"
:"1968-04-05","name":"CERVIGNANO DEL FRIULI"},{"name":"CERVIGNANO DEL FRIULI
","start":"1968-04-06","end":"9999-12-31"}],"C868":[{"start":"1953-06-21","e
nd":"9999-12-31","name":"COLLERETTO GIACOSA"},{"name":"COLLERETTO PARELLA","
end":"1927-01-11","start":"1861-03-17"},{"name":"COLLERETTO PARELLA","start"
:"1927-01-12","end":"1929-04-12"},{"name":"COLLERETTO PARELLA","start":"1947
-10-11","end":"1953-06-20"}],"B578":[{"end":"9999-12-31","start":"1866-11-19
","name":"CANARO"}],"A472":[{"name":"ASPRA SABINA","end":"1923-03-18","start
":"1861-03-17"},{"end":"1927-01-11","start":"1923-03-19","name":"ASPRA SABIN
A"},{"start":"1927-01-12","end":"1927-08-18","name":"ASPRA SABINA"},{"end":"
1939-07-03","start":"1927-08-19","name":"ASPRA SABINA"},{"start":"1939-07-04
","end":"1939-12-11","name":"ASPRA SABINA"},{"end":"1947-08-16","start":"193
9-12-12","name":"ASPRA SABINA"},{"name":"CASPERIA","start":"1947-08-17","end
":"9999-12-31"}],"F042":[{"name":"MASSERANO","start":"1861-03-17","end":"192
7-01-11"},{"end":"1992-04-15","start":"1927-01-12","name":"MASSERANO"},{"sta
rt":"1992-04-16","end":"9999-12-31","name":"MASSERANO"}],"A924":[{"start":"1
921-01-05","end":"1922-02-01","name":"BOGLIUNO"},{"start":"1922-02-02","end"
:"1923-02-07","name":"BOGLIUNO"},{"start":"1923-02-08","end":"1947-09-15","n
ame":"BOGLIUNO"}],"E792":[{"name":"MADERNO","end":"1928-07-28","start":"1861
-03-17"}],"L113":[{"name":"TERMOLI","end":"1928-05-02","start":"1861-03-17"}
,{"name":"TERMOLI","start":"1928-05-03","end":"1946-03-24"},{"name":"TERMOLI
","end":"1970-03-02","start":"1946-03-25"},{"name":"TERMOLI","start":"1970-0
3-03","end":"9999-12-31"}],"A327":[{"start":"1861-03-17","end":"1998-09-03",
"name":"APECCHIO"},{"name":"APECCHIO","start":"1998-09-04","end":"9999-12-31
"}],"G750":[{"end":"1923-11-28","start":"1861-03-17","name":"POGGI"}],"M090"
:[{"name":"VITTORITO","end":"9999-12-31","start":"1861-03-17"}],"I678":[{"na
me":"SESSAME","start":"1861-03-17","end":"1927-07-05"},{"name":"SESSAME","st
art":"1927-07-06","end":"1935-04-07"},{"name":"SESSAME","end":"9999-12-31","
start":"1935-04-08"}],"E423":[{"end":"9999-12-31","start":"1861-03-17","name
":"LA LOGGIA"}],"H257":[{"name":"REZZO","end":"1928-04-14","start":"1861-03-
17"},{"start":"1928-04-15","end":"9999-12-31","name":"REZZO"}],"A705":[{"nam
e":"TRONZANO LAGO MAGGIORE","end":"1927-01-11","start":"1863-12-14"},{"name"
:"TRONZANO LAGO MAGGIORE","start":"1927-01-12","end":"9999-12-31"}],"C561":[
{"start":"1861-03-17","end":"1927-01-11","name":"CESA"},{"name":"CESA","star
t":"1927-01-12","end":"1945-08-31"},{"name":"CESA","start":"1945-09-01","end
":"9999-12-31"}],"E610":[{"end":"9999-12-31","start":"1861-03-17","name":"LI
SCATE"}],"A332":[{"name":"APPIANO","end":"1923-02-13","start":"1920-10-16"},
{"name":"APPIANO","end":"1927-01-11","start":"1923-02-14"},{"name":"APPIANO"
,"start":"1927-01-12","end":"1971-09-21"},{"name":"APPIANO SULLA STRADA DEL 
VINO","start":"1971-09-22","end":"9999-12-31"}],"G035":[{"end":"1924-02-03",
"start":"1861-03-17","name":"OLIVASTRI"}],"L280":[{"name":"TORRE SANTA SUSAN
NA","end":"1927-01-11","start":"1861-03-17"},{"end":"9999-12-31","start":"19
27-01-12","name":"TORRE SANTA SUSANNA"}],"A526":[{"name":"AZZANELLO","end":"
9999-12-31","start":"1861-03-17"}],"C204":[{"name":"CASTEL MAGGIORE","end":"
9999-12-31","start":"1861-03-17"}],"I367":[{"start":"1861-03-17","end":"1901
-04-30","name":"SANTO STEFANO BELBO"},{"start":"1901-05-01","end":"1928-06-1
3","name":"SANTO STEFANO BELBO"},{"name":"SANTO STEFANO BELBO","start":"1928
-06-14","end":"1947-05-22"},{"end":"9999-12-31","start":"1947-05-23","name":
"SANTO STEFANO BELBO"}],"D595":[{"name":"FILIGNANO","start":"1884-04-18","en
d":"1970-03-02"},{"name":"FILIGNANO","start":"1970-03-03","end":"9999-12-31"
}],"F987":[{"name":"NUS","start":"1861-03-17","end":"1927-01-11"},{"name":"N
US","end":"1928-10-15","start":"1927-01-12"},{"name":"NUS","start":"1928-10-
16","end":"1946-05-14"},{"end":"9999-12-31","start":"1946-05-15","name":"NUS
"}],"H010":[{"end":"1968-04-05","start":"1866-11-19","name":"PRAVISDOMINI"},
{"end":"9999-12-31","start":"1968-04-06","name":"PRAVISDOMINI"}],"L713":[{"e
nd":"1927-01-11","start":"1872-08-26","name":"VEJANO"},{"name":"VEJANO","end
":"9999-12-31","start":"1927-01-12"}],"E192":[{"name":"GRONE","start":"1861-
03-17","end":"1928-06-29"},{"end":"9999-12-31","start":"1947-11-04","name":"
GRONE"}],"F384":[{"name":"MONSUMMANO","start":"1861-03-17","end":"1928-01-25
"},{"end":"1948-07-08","start":"1928-01-26","name":"MONSUMMANO"},{"end":"999
9-12-31","start":"1948-07-09","name":"MONSUMMANO TERME"}],"L386":[{"start":"
1869-04-01","end":"9999-12-31","name":"TRESANA"}],"G150":[{"name":"ORZIVECCH
I","start":"1861-03-17","end":"9999-12-31"}],"D627":[{"start":"1921-01-05","
end":"1923-02-07","name":"FIUMICELLO"},{"end":"1924-01-14","start":"1923-02-
08","name":"FIUMICELLO"},{"name":"FIUMICELLO","end":"1968-04-05","start":"19
47-01-09"},{"start":"1968-04-06","end":"2018-01-31","name":"FIUMICELLO"}],"I
964":[{"name":"STORO","start":"1920-10-16","end":"1923-02-13"},{"name":"STOR
O","end":"1928-04-16","start":"1923-02-14"},{"name":"STORO","start":"1928-04
-17","end":"1953-09-11"},{"name":"STORO","start":"1953-09-12","end":"9999-12
-31"}],"Z145":[{"name":"Lettonia","end":"9999-12-31","start":"0000-01-01"}],
"D245":[{"start":"1927-08-20","end":"1963-02-06","name":"DALMINE"},{"start":
"1963-02-07","end":"9999-12-31","name":"DALMINE"}],"H550":[{"end":"1923-02-0
7","start":"1921-01-05","name":"RONZINA"},{"name":"RONZINA","end":"1927-01-1
1","start":"1923-02-08"},{"name":"RONZINA","end":"1928-03-12","start":"1927-
01-12"}],"I566":[{"name":"SEDRIANO","end":"2002-10-25","start":"1861-03-17"}
,{"start":"2002-10-26","end":"9999-12-31","name":"SEDRIANO"}],"I858":[{"name
":"SORISOLE","end":"9999-12-31","start":"1861-03-17"}],"A105":[{"name":"AIET
A","start":"1861-03-17","end":"1928-05-11"},{"name":"AIETA","start":"1937-07
-18","end":"9999-12-31"}],"G328":[{"name":"PARCINES","end":"1923-02-13","sta
rt":"1920-10-16"},{"start":"1923-02-14","end":"1927-01-11","name":"PARCINES"
},{"end":"9999-12-31","start":"1927-01-12","name":"PARCINES"}],"C912":[{"end
":"9999-12-31","start":"1861-03-17","name":"COMACCHIO"}],"B666":[{"name":"CA
PO D'ORLANDO","end":"9999-12-31","start":"1925-07-10"}],"M047":[{"name":"VIL
LONGO SANT'ALESSANDRO","end":"1927-11-03","start":"1861-03-17"}],"B969":[{"s
tart":"1896-07-03","end":"1926-01-21","name":"CASTEL DI CASIO"},{"name":"CAS
TEL DI CASIO","start":"1926-01-22","end":"9999-12-31"}],"L587":[{"start":"19
20-10-16","end":"1923-02-13","name":"VALLARGA"},{"name":"VALLARGA","start":"
1923-02-14","end":"1927-01-11"},{"name":"VALLARGA","start":"1927-01-12","end
":"1929-03-24"}],"L838":[{"end":"1893-04-09","start":"1861-03-17","name":"VI
CCHIO"},{"name":"VICCHIO","end":"9999-12-31","start":"1893-04-10"}],"D632":[
{"end":"1927-01-11","start":"1861-03-17","name":"FLECCHIA"},{"end":"1928-03-
13","start":"1927-01-12","name":"FLECCHIA"}],"F931":[{"name":"NONE","start":
"1861-03-17","end":"9999-12-31"}],"E503":[{"end":"1901-02-19","start":"1861-
03-17","name":"LAZZARONE"},{"name":"VILLABELLA","start":"1901-02-20","end":"
1938-06-18"}],"G249":[{"name":"PALADINA","end":"1929-05-16","start":"1861-03
-17"},{"name":"PALADINA","end":"1970-06-01","start":"1929-05-17"},{"end":"99
99-12-31","start":"1970-06-02","name":"PALADINA"}],"I011":[{"end":"9999-12-3
1","start":"1861-03-17","name":"SAN MARTINO IN RIO"}],"C189":[{"name":"CASTE
LLO DI FIEMME","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14
","end":"1925-03-17","name":"CASTELLO DI FIEMME"},{"start":"1925-03-18","end
":"1973-09-25","name":"CASTELLO DI FIEMME"},{"start":"1973-09-26","end":"999
9-12-31","name":"CASTELLO-MOLINA DI FIEMME"}],"G694":[{"end":"1933-12-11","s
tart":"1866-11-19","name":"PIOVENE"},{"end":"9999-12-31","start":"1933-12-12
","name":"PIOVENE ROCCHETTE"}],"E677":[{"name":"LONGOBARDI","start":"1861-03
-17","end":"1928-05-18"},{"start":"1937-07-18","end":"9999-12-31","name":"LO
NGOBARDI"}],"A633":[{"start":"1893-01-02","end":"2018-12-31","name":"BARBERI
NO VAL D'ELSA"}],"L546":[{"end":"9999-12-31","start":"1893-08-30","name":"VA
LBREVENNA"}],"H199":[{"name":"RAVENNA","end":"1939-12-04","start":"1861-03-1
7"},{"start":"1939-12-05","end":"1982-04-14","name":"RAVENNA"},{"name":"RAVE
NNA","end":"9999-12-31","start":"1982-04-15"}],"F544":[{"start":"1861-03-17"
,"end":"9999-12-31","name":"MONTELEPRE"}],"G842":[{"name":"PONTE DELL'OLIO",
"end":"9999-12-31","start":"1861-03-17"}],"M086":[{"end":"1927-01-11","start
":"1871-01-15","name":"VITORCHIANO"},{"start":"1927-01-12","end":"9999-12-31
","name":"VITORCHIANO"}],"G463":[{"start":"1863-01-26","end":"1928-05-30","n
ame":"PEROSA ARGENTINA"},{"name":"PEROSA ARGENTINA","start":"1928-05-31","en
d":"1954-10-16"},{"name":"PEROSA ARGENTINA","start":"1954-10-17","end":"9999
-12-31"}],"A406":[{"start":"1861-03-17","end":"1927-09-20","name":"ARISCHIA"
}],"D998":[{"end":"9999-12-31","start":"1861-03-17","name":"GESUALDO"}],"F64
9":[{"name":"DORNBERG","start":"1921-01-05","end":"1923-02-07"},{"name":"DOR
NBERG","end":"1923-04-26","start":"1923-02-08"},{"start":"1923-04-27","end":
"1927-01-11","name":"MONTESPINO"},{"end":"1928-04-16","start":"1927-01-12","
name":"MONTESPINO"},{"name":"MONTESPINO","end":"1947-09-15","start":"1928-04
-17"}],"I551":[{"start":"1866-11-19","end":"9999-12-31","name":"SCORZ\u00c8"
}],"F946":[{"name":"NOVALE","start":"1866-11-19","end":"1928-08-23"}],"G599"
:[{"name":"PIEDIMONTE DEL CALVARIO","end":"1927-01-11","start":"1923-04-27"}
,{"start":"1927-01-12","end":"1927-01-12","name":"PIEDIMONTE DEL CALVARIO"},
{"end":"1923-02-07","start":"1921-01-05","name":"PODGORA"},{"name":"PODGORA"
,"end":"1923-04-26","start":"1923-02-08"}],"F835":[{"end":"1923-02-13","star
t":"1920-10-16","name":"NAGO-TORBOLE"},{"end":"1929-05-28","start":"1923-02-
14","name":"NAGO-TORBOLE"},{"name":"NAGO-TORBOLE","end":"9999-12-31","start"
:"1957-07-03"}],"L944":[{"name":"VILLAFRANCA SICULA","end":"9999-12-31","sta
rt":"1863-01-19"}],"L531":[{"name":"VAGLIO SERRA","start":"1863-02-16","end"
:"1935-04-07"},{"name":"VAGLIO SERRA","start":"1935-04-08","end":"9999-12-31
"}],"L296":[{"name":"TORRICELLA DEL PIZZO","start":"1861-03-17","end":"9999-
12-31"}],"F294":[{"name":"MOLISE","start":"1861-03-17","end":"1970-03-02"},{
"end":"9999-12-31","start":"1970-03-03","name":"MOLISE"}],"B651":[{"end":"99
99-12-31","start":"1861-03-17","name":"CAPESTRANO"}],"L347":[{"name":"TRAVES
IO","start":"1866-11-19","end":"1928-05-28"},{"end":"1968-04-05","start":"19
28-05-29","name":"TRAVESIO"},{"end":"9999-12-31","start":"1968-04-06","name"
:"TRAVESIO"}],"H926":[{"end":"9999-12-31","start":"1861-03-17","name":"SAN G
IOVANNI ROTONDO"}],"G124":[{"name":"ORSARA BORMIDA","end":"9999-12-31","star
t":"1863-04-13"}],"H629":[{"start":"1921-01-05","end":"1923-02-07","name":"R
UDA"},{"name":"RUDA","end":"1928-04-14","start":"1923-02-08"},{"name":"RUDA"
,"start":"1928-04-15","end":"1948-03-25"},{"name":"RUDA","start":"1948-03-26
","end":"1968-04-05"},{"end":"2012-06-20","start":"1968-04-06","name":"RUDA"
},{"start":"2012-06-21","end":"9999-12-31","name":"RUDA"}],"F484":[{"start":
"1861-03-17","end":"9999-12-31","name":"MONTECRETO"}],"L486":[{"name":"UGGIA
TE","end":"1938-01-17","start":"1861-03-17"}],"C648":[{"name":"CHITIGNANO","
end":"9999-12-31","start":"1861-03-17"}],"I096":[{"name":"SAN PIETRO AVELLAN
A","start":"1861-03-17","end":"1970-03-02"},{"start":"1970-03-03","end":"999
9-12-31","name":"SAN PIETRO AVELLANA"}],"L878":[{"end":"9999-12-31","start":
"1867-08-26","name":"VIGHIZZOLO D'ESTE"}],"D672":[{"name":"FONTANELLA","star
t":"1861-03-17","end":"9999-12-31"}],"I467":[{"start":"1870-01-02","end":"19
23-12-21","name":"SAVA"},{"name":"SAVA","start":"1923-12-22","end":"9999-12-
31"}],"F188":[{"name":"MEZZOMERICO","end":"9999-12-31","start":"1861-03-17"}
],"G428":[{"name":"PELLIZZANO","start":"1920-10-16","end":"1923-02-13"},{"st
art":"1923-02-14","end":"1928-09-08","name":"PELLIZZANO"},{"name":"PELLIZZAN
O","start":"1954-07-17","end":"9999-12-31"}],"D065":[{"end":"1927-08-05","st
art":"1861-03-17","name":"CORTENOVA"},{"start":"1927-08-06","end":"1992-04-1
5","name":"CORTENOVA"},{"name":"CORTENOVA","start":"1992-04-16","end":"9999-
12-31"}],"E681":[{"name":"LONGONE SABINO","end":"1923-03-18","start":"1863-0
4-13"},{"name":"LONGONE SABINO","start":"1923-03-19","end":"1927-01-11"},{"s
tart":"1927-01-12","end":"9999-12-31","name":"LONGONE SABINO"}],"L211":[{"st
art":"1920-10-16","end":"1923-02-13","name":"TORCEGNO"},{"name":"TORCEGNO","
start":"1923-02-14","end":"1928-05-12"},{"end":"9999-12-31","start":"1947-01
-15","name":"TORCEGNO"}],"M267":[{"end":"9999-12-31","start":"1975-05-20","n
ame":"ZAPPONETA"}],"G069":[{"end":"1923-02-13","start":"1920-10-16","name":"
ONIES"},{"end":"1927-01-11","start":"1923-02-14","name":"ONIES"},{"start":"1
927-01-12","end":"1928-12-14","name":"ONIES"}],"A306":[{"name":"ANTERIVO","s
tart":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-14","end":"1925-09-
26","name":"ANTERIVO"},{"end":"1948-03-13","start":"1947-06-17","name":"ANTE
RIVO"},{"end":"9999-12-31","start":"1948-03-14","name":"ANTERIVO"}],"D197":[
{"name":"CUCEGLIO","start":"1861-03-17","end":"1927-01-11"},{"start":"1927-0
1-12","end":"1945-10-04","name":"CUCEGLIO"},{"name":"CUCEGLIO","start":"1945
-10-05","end":"9999-12-31"}],"F875":[{"start":"1861-03-17","end":"1926-02-18
","name":"NESE"},{"end":"1928-03-26","start":"1926-02-19","name":"NESE"},{"n
ame":"NESE","end":"1938-01-17","start":"1928-03-27"},{"name":"NESE","end":"1
939-09-19","start":"1938-01-18"}],"F001":[{"start":"1861-03-17","end":"1928-
02-20","name":"MARZI"},{"end":"9999-12-31","start":"1937-07-18","name":"MARZ
I"}],"D814":[{"end":"2017-02-13","start":"1861-03-17","name":"FUBINE"},{"nam
e":"FUBINE MONFERRATO","start":"2017-02-14","end":"9999-12-31"}],"L571":[{"n
ame":"VALENZANO","start":"1861-03-17","end":"9999-12-31"}],"E622":[{"name":"
LIVINALLONGO","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14"
,"end":"1933-04-24","name":"LIVINALLONGO"},{"start":"1933-04-25","end":"9999
-12-31","name":"LIVINALLONGO DEL COL DI LANA"}],"D740":[{"name":"FOSSALTA DI
 PIAVE","start":"1867-08-05","end":"9999-12-31"}],"L447":[{"start":"1861-03-
17","end":"9999-12-31","name":"TROIA"}],"F971":[{"name":"NOZZA","end":"1928-
09-24","start":"1861-03-17"}],"A507":[{"name":"AVELENGO","end":"1927-01-11",
"start":"1923-04-27"},{"start":"1927-01-12","end":"1931-06-13","name":"AVELE
NGO"},{"name":"AVELENGO","end":"9999-12-31","start":"1957-06-12"},{"name":"A
VELIGNA","start":"1920-10-16","end":"1923-02-13"},{"name":"AVELIGNA","end":"
1923-04-26","start":"1923-02-14"}],"Z331":[{"name":"Mauritania","end":"9999-
12-31","start":"0000-01-01"}],"L185":[{"end":"9999-12-31","start":"1863-02-0
6","name":"TOCCO CAUDIO"}],"E764":[{"end":"1923-02-13","start":"1920-10-16",
"name":"LUSON"},{"name":"LUSON","end":"1927-01-11","start":"1923-02-14"},{"e
nd":"9999-12-31","start":"1927-01-12","name":"LUSON"}],"E637":[{"name":"LOCA
TE BERGAMASCO","start":"1863-07-13","end":"1928-01-06"}],"A673":[{"name":"BA
RONE CANAVESE","start":"1861-03-17","end":"1927-01-11"},{"name":"BARONE CANA
VESE","start":"1927-01-12","end":"1945-10-04"},{"name":"BARONE CANAVESE","st
art":"1945-10-05","end":"9999-12-31"}],"H708":[{"name":"SALICE SALENTINO","s
tart":"1864-04-01","end":"1927-01-11"},{"start":"1927-01-12","end":"1927-04-
13","name":"SALICE SALENTINO"},{"end":"9999-12-31","start":"1927-04-14","nam
e":"SALICE SALENTINO"}],"D433":[{"name":"ESILLE","start":"1937-06-16","end":
"1953-07-15"},{"name":"EXILLES","start":"1861-03-17","end":"1937-06-15"},{"s
tart":"1953-07-16","end":"9999-12-31","name":"EXILLES"}],"B846":[{"name":"CA
RTOCETO","end":"1998-09-03","start":"1861-03-17"},{"name":"CARTOCETO","end":
"9999-12-31","start":"1998-09-04"}],"B935":[{"end":"1927-01-11","start":"186
1-03-17","name":"CASAPULLA"},{"name":"CASAPULLA","end":"1929-02-06","start":
"1927-01-12"},{"end":"9999-12-31","start":"1946-12-19","name":"CASAPULLA"}],
"H037":[{"start":"1861-03-17","end":"1928-07-27","name":"PREMOSELLO"},{"end"
:"1960-02-12","start":"1928-07-28","name":"PREMOSELLO"},{"start":"1960-02-13
","end":"1992-05-22","name":"PREMOSELLO-CHIOVENDA"},{"start":"1992-05-23","e
nd":"9999-12-31","name":"PREMOSELLO-CHIOVENDA"}],"D797":[{"start":"1861-03-1
7","end":"9999-12-31","name":"FRESONARA"}],"L490":[{"end":"1923-02-13","star
t":"1920-10-16","name":"ULTIMO"},{"name":"ULTIMO","start":"1923-02-14","end"
:"1927-01-11"},{"name":"ULTIMO","start":"1927-01-12","end":"1960-04-26"},{"n
ame":"ULTIMO","start":"1960-04-27","end":"9999-12-31"}],"A853":[{"name":"BIB
IANA","end":"9999-12-31","start":"1861-03-17"}],"D909":[{"end":"1913-07-12",
"start":"1861-03-17","name":"GARAGUSO"},{"end":"1927-01-11","start":"1913-07
-13","name":"GARAGUSO"},{"start":"1927-01-12","end":"9999-12-31","name":"GAR
AGUSO"}],"G508":[{"end":"1992-04-15","start":"1863-02-06","name":"PETILIA PO
LICASTRO"},{"name":"PETILIA POLICASTRO","end":"9999-12-31","start":"1992-04-
16"}],"D606":[{"name":"FIORANO AL SERIO","start":"1863-07-13","end":"1927-06
-01"},{"name":"FIORANO AL SERIO","end":"9999-12-31","start":"1948-12-21"}],"
C457":[{"name":"CENATE SOTTO","start":"1861-03-17","end":"1927-11-30"},{"end
":"9999-12-31","start":"1947-06-24","name":"CENATE SOTTO"}],"E468":[{"name":
"LATERINA","end":"2017-12-31","start":"1861-03-17"}],"H404":[{"name":"ROCCA 
DI PAPA","end":"9999-12-31","start":"1871-01-15"}],"G255":[{"name":"PALANZAN
O","start":"1861-03-17","end":"9999-12-31"}],"C920":[{"start":"1866-11-19","
end":"9999-12-31","name":"COMELICO SUPERIORE"}],"Z240":[{"end":"9999-12-31",
"start":"0000-01-01","name":"Siria"}],"H022":[{"name":"PREGANZIOL","start":"
1866-11-19","end":"9999-12-31"}],"D140":[{"name":"CREDERA","end":"1928-07-28
","start":"1868-07-13"}],"B831":[{"name":"CARPUGNINO","start":"1861-03-17","
end":"1928-09-25"}],"A200":[{"end":"1927-01-11","start":"1861-03-17","name":
"ALIFE"},{"name":"ALIFE","start":"1927-01-12","end":"1945-08-31"},{"start":"
1945-09-01","end":"9999-12-31","name":"ALIFE"}],"I992":[{"name":"SUBIACO","e
nd":"9999-12-31","start":"1871-01-15"}],"D373":[{"name":"DRUENT","end":"1937
-06-15","start":"1861-03-17"},{"name":"DRUENTO","start":"1937-06-16","end":"
9999-12-31"}],"D079":[{"start":"1920-10-16","end":"1923-02-13","name":"CORVA
RA IN BADIA"},{"end":"1925-09-14","start":"1923-02-14","name":"CORVARA IN BA
DIA"},{"name":"CORVARA IN BADIA","start":"1938-10-17","end":"9999-12-31"},{"
name":"LADINIA","end":"1927-01-11","start":"1925-09-15"},{"start":"1927-01-1
2","end":"1938-10-16","name":"LADINIA"}],"L785":[{"start":"1861-03-17","end"
:"1927-01-11","name":"VERRONE"},{"name":"VERRONE","end":"1992-04-15","start"
:"1927-01-12"},{"name":"VERRONE","end":"9999-12-31","start":"1992-04-16"}],"
E164":[{"name":"GREMIASCO","start":"1861-03-17","end":"1928-12-18"},{"name":
"GREMIASCO","start":"1947-09-24","end":"1963-11-08"},{"name":"GREMIASCO","st
art":"1963-11-09","end":"9999-12-31"}],"F655":[{"name":"MONTEVAGO","start":"
1861-03-17","end":"9999-12-31"}],"H108":[{"end":"1927-01-11","start":"1861-0
3-17","name":"QUARONA"},{"end":"1928-09-05","start":"1927-01-12","name":"QUA
RONA"},{"end":"9999-12-31","start":"1939-10-04","name":"QUARONA"}],"G801":[{
"end":"1967-11-23","start":"1861-03-17","name":"POLPENAZZE"},{"name":"POLPEN
AZZE DEL GARDA","start":"1967-11-24","end":"9999-12-31"}],"G724":[{"name":"P
IZZOFERRATO","end":"9999-12-31","start":"1861-03-17"}],"A427":[{"end":"1928-
02-21","start":"1861-03-17","name":"AROLA"},{"name":"AROLA","start":"1955-03
-12","end":"1992-05-22"},{"name":"AROLA","end":"9999-12-31","start":"1992-05
-23"}],"C823":[{"name":"COGOLETO","start":"1861-03-17","end":"1927-01-11"},{
"start":"1927-01-12","end":"1933-05-19","name":"COGOLETO"},{"name":"COGOLETO
","end":"9999-12-31","start":"1933-05-20"}],"H081":[{"start":"1920-10-16","e
nd":"1923-02-13","name":"PROVES"},{"end":"1928-09-06","start":"1923-02-14","
name":"PROVES"},{"name":"PROVES","end":"9999-12-31","start":"1948-03-14"}],"
F262":[{"end":"1928-10-15","start":"1861-03-17","name":"MODUGNO"},{"end":"99
99-12-31","start":"1928-10-16","name":"MODUGNO"}],"F869":[{"end":"1947-09-15
","start":"1924-02-07","name":"NERESINE"}],"C091":[{"end":"9999-12-31","star
t":"1934-03-07","name":"CASTEL DI IUDICA"}],"A372":[{"end":"1923-02-13","sta
rt":"1920-10-16","name":"ARCO"},{"name":"ARCO","start":"1923-02-14","end":"1
929-02-13"},{"end":"9999-12-31","start":"1929-02-14","name":"ARCO"}],"A950":
[{"name":"BOLTIERE","start":"1861-03-17","end":"9999-12-31"}],"G075":[{"name
":"ONORE","start":"1861-03-17","end":"1927-11-19"},{"name":"ONORE","end":"99
99-12-31","start":"1958-09-10"}],"F788":[{"name":"MOZZATE","start":"1861-03-
17","end":"1928-09-22"},{"name":"MOZZATE","start":"1953-09-13","end":"9999-1
2-31"}],"G662":[{"name":"PIGNATARO INTERAMNA","end":"1927-01-11","start":"18
62-11-24"},{"name":"PIGNATARO INTERAMNA","end":"9999-12-31","start":"1927-01
-12"}],"L658":[{"name":"VALVERDE","start":"1951-05-06","end":"9999-12-31"}],
"G905":[{"name":"PORTIGLIOLA","start":"1861-03-17","end":"9999-12-31"}],"D85
2":[{"end":"9999-12-31","start":"1861-03-17","name":"GAGLIATO"}],"B538":[{"n
ame":"CAMPO LIGURE","end":"9999-12-31","start":"1884-04-21"}],"A432":[{"end"
:"1904-08-15","start":"1861-03-17","name":"ARPAISE"},{"name":"ARPAISE","end"
:"1918-01-17","start":"1904-08-16"},{"name":"ARPAISE","start":"1918-01-18","
end":"9999-12-31"}],"B887":[{"name":"CASALETTO LODIGIANO","start":"1870-08-0
2","end":"1992-04-15"},{"name":"CASALETTO LODIGIANO","end":"9999-12-31","sta
rt":"1992-04-16"}],"I638":[{"end":"1923-02-07","start":"1921-01-05","name":"
SERPENIZZA"},{"start":"1923-02-08","end":"1927-01-11","name":"SERPENIZZA"},{
"name":"SERPENIZZA","end":"1928-05-03","start":"1927-01-12"}],"D259":[{"star
t":"1861-03-17","end":"1974-08-19","name":"DECIMOMANNU"},{"name":"DECIMOMANN
U","end":"9999-12-31","start":"1974-08-20"}],"E323":[{"start":"1861-03-17","
end":"1927-01-11","name":"IRGOLI"},{"name":"IRGOLI","end":"1928-02-01","star
t":"1927-01-12"},{"name":"IRGOLI","start":"1946-03-08","end":"9999-12-31"}],
"E029":[{"end":"1927-01-11","start":"1861-03-17","name":"GIGNOD"},{"name":"G
IGNOD","end":"1928-03-20","start":"1927-01-12"},{"end":"9999-12-31","start":
"1946-05-15","name":"GIGNOD"}],"Z159":[{"name":"Montenegro","start":"0000-01
-01","end":"9999-12-31"}],"D382":[{"start":"1921-01-05","end":"1923-02-07","
name":"DUINO"},{"start":"1923-02-08","end":"1928-09-08","name":"DUINO"}],"A2
93":[{"name":"ANGOLO","end":"1928-05-11","start":"1861-03-17"},{"end":"1959-
07-25","start":"1928-05-12","name":"ANGOLO"},{"start":"1959-07-26","end":"19
63-07-16","name":"ANGOLO"},{"end":"9999-12-31","start":"1963-07-17","name":"
ANGOLO TERME"}],"G234":[{"name":"PAGLIARA","start":"1914-07-20","end":"1938-
08-06"},{"name":"PAGLIARA","start":"1938-08-07","end":"9999-12-31"}],"B357":
[{"name":"CAGNANO VARANO","end":"9999-12-31","start":"1862-11-10"}],"L641":[
{"name":"VALMOZZOLA","start":"1861-03-17","end":"9999-12-31"}],"C005":[{"nam
e":"CASSANO SPINOLA","start":"1861-03-17","end":"1929-04-11"},{"name":"CASSA
NO SPINOLA","end":"1947-11-25","start":"1929-04-12"},{"name":"CASSANO SPINOL
A","end":"2017-12-31","start":"1947-11-26"}],"B016":[{"start":"1861-03-17","
end":"9999-12-31","name":"BORGOLAVEZZARO"}],"C871":[{"name":"COLLESANO","sta
rt":"1861-03-17","end":"1927-10-20"},{"end":"1961-04-10","start":"1927-10-21
","name":"COLLESANO"},{"name":"COLLESANO","start":"1961-04-11","end":"9999-1
2-31"}],"B954":[{"name":"CASEI GEROLA","end":"9999-12-31","start":"1861-03-1
7"}],"B789":[{"name":"CARLOFORTE","start":"1861-03-17","end":"1974-08-19"},{
"name":"CARLOFORTE","end":"2005-12-31","start":"1974-08-20"},{"end":"2016-04
-27","start":"2006-01-01","name":"CARLOFORTE"},{"name":"CARLOFORTE","start":
"2016-04-28","end":"9999-12-31"}],"E431":[{"name":"LAMPEDUSA E LINOSA","end"
:"9999-12-31","start":"1875-08-09"}],"G895":[{"start":"1861-03-17","end":"99
99-12-31","name":"PORTALBERA"}],"M162":[{"name":"ZENEVREDO","end":"9999-12-3
1","start":"1861-03-17"}],"B868":[{"end":"9999-12-31","start":"1862-12-29","
name":"CASALBUONO"}],"B556":[{"end":"1954-07-22","start":"1861-03-17","name"
:"CAMPOREALE"},{"end":"9999-12-31","start":"1954-07-23","name":"CAMPOREALE"}
],"L403":[{"name":"TREVILLE","start":"1861-03-17","end":"9999-12-31"}],"E371
":[{"start":"1861-03-17","end":"1927-01-11","name":"ISSOGNE"},{"start":"1927
-01-12","end":"9999-12-31","name":"ISSOGNE"}],"Z503":[{"name":"Costa Rica","
end":"9999-12-31","start":"0000-01-01"}],"L939":[{"name":"VILLADOSE","start"
:"1866-11-19","end":"1929-01-12"},{"name":"VILLADOSE","end":"9999-12-31","st
art":"1929-01-13"}],"A543":[{"end":"1928-01-29","start":"1861-03-17","name":
"BAGAGGERA"}],"F634":[{"start":"1861-03-17","end":"1928-06-26","name":"MONTE
 SAN VITO"},{"name":"MONTE SAN VITO","start":"1928-06-27","end":"1956-08-08"
},{"name":"MONTE SAN VITO","start":"1956-08-09","end":"9999-12-31"}],"L636":
[{"name":"VALMALA","end":"1928-06-09","start":"1861-03-17"},{"name":"VALMALA
","start":"1947-09-25","end":"2018-12-31"}],"F767":[{"name":"MOSSA","start":
"1921-01-05","end":"1923-02-07"},{"start":"1923-02-08","end":"1927-01-11","n
ame":"MOSSA"},{"name":"MOSSA","start":"1927-01-12","end":"1928-01-01"},{"nam
e":"MOSSA","end":"9999-12-31","start":"1955-08-19"}],"E446":[{"start":"1861-
03-17","end":"1924-01-11","name":"LAORCA"}],"C975":[{"name":"CONVERSANO","en
d":"9999-12-31","start":"1861-03-17"}],"H660":[{"name":"SAGA","start":"1921-
01-05","end":"1923-02-07"},{"end":"1927-01-11","start":"1923-02-08","name":"
SAGA"},{"start":"1927-01-12","end":"1928-05-03","name":"SAGA"}],"I656":[{"en
d":"9999-12-31","start":"1861-03-17","name":"SERRATA"}],"C218":[{"name":"CAS
TELNOVO DI SOTTO","start":"1861-03-17","end":"9999-12-31"}],"I784":[{"start"
:"1861-03-17","end":"1928-02-09","name":"SOLANAS"}],"G991":[{"end":"1907-03-
07","start":"1898-04-15","name":"PRATA SANNITA"},{"name":"PRATA SANNITA","en
d":"1927-01-11","start":"1907-03-08"},{"end":"1945-08-31","start":"1927-01-1
2","name":"PRATA SANNITA"},{"name":"PRATA SANNITA","end":"9999-12-31","start
":"1945-09-01"}],"I959":[{"name":"STIMIGLIANO","start":"1861-03-17","end":"1
923-03-18"},{"name":"STIMIGLIANO","end":"1927-01-11","start":"1923-03-19"},{
"end":"9999-12-31","start":"1927-01-12","name":"STIMIGLIANO"}],"E145":[{"nam
e":"GRANTORTO","end":"9999-12-31","start":"1866-11-19"}],"D542":[{"end":"200
9-07-11","start":"1877-12-03","name":"FERMO"},{"end":"9999-12-31","start":"2
009-07-12","name":"FERMO"}],"I209":[{"name":"SANT'AGOSTINO","end":"1928-06-2
6","start":"1861-03-17"},{"name":"SANT'AGOSTINO","end":"1959-10-02","start":
"1928-06-27"},{"name":"SANT'AGOSTINO","start":"1959-10-03","end":"2016-12-31
"}],"I802":[{"name":"SOLIERA","end":"9999-12-31","start":"1861-03-17"}],"E48
7":[{"end":"1939-08-10","start":"1871-08-01","name":"LAURO"},{"end":"1946-12
-18","start":"1939-08-11","name":"LAURO"},{"name":"LAURO","end":"9999-12-31"
,"start":"1946-12-19"}],"C578":[{"name":"CESIO","start":"1861-03-17","end":"
1923-12-21"},{"name":"CESIO","start":"1925-08-22","end":"1929-03-15"},{"name
":"CESIO","start":"1929-03-16","end":"9999-12-31"}],"B204":[{"end":"9999-12-
31","start":"1861-03-17","name":"BROSSASCO"}],"D292":[{"end":"9999-12-31","s
tart":"1862-12-29","name":"TEGGIANO"}],"M409":[{"name":"TRESIGNANA","start":
"2019-01-01","end":"9999-12-31"}],"A089":[{"end":"9999-12-31","start":"1927-
07-27","name":"AGRIGENTO"},{"start":"1861-03-17","end":"1927-07-26","name":"
GIRGENTI"}],"D321":[{"name":"DOLEGNA","end":"1923-02-07","start":"1921-01-05
"},{"start":"1923-02-08","end":"1923-04-26","name":"DOLEGNA"},{"name":"DOLEG
NA DEL COLLIO","start":"1923-04-27","end":"1927-01-11"},{"name":"DOLEGNA DEL
 COLLIO","start":"1927-01-12","end":"1928-03-06"},{"end":"1947-09-15","start
":"1928-03-07","name":"DOLEGNA DEL COLLIO"},{"name":"DOLEGNA DEL COLLIO","en
d":"9999-12-31","start":"1947-09-16"}],"B561":[{"end":"9999-12-31","start":"
1863-05-06","name":"CAMPOROTONDO ETNEO"}],"A383":[{"end":"9999-12-31","start
":"1861-03-17","name":"ARDESIO"}],"A716":[{"start":"1861-03-17","end":"1928-
06-27","name":"BATTIFOLLO"},{"name":"BATTIFOLLO","end":"9999-12-31","start":
"1947-01-10"}],"G020":[{"start":"1861-03-17","end":"1928-02-22","name":"OLEG
GIO CASTELLO"},{"name":"OLEGGIO CASTELLO","end":"9999-12-31","start":"1947-1
1-28"}],"I661":[{"start":"1862-10-24","end":"9999-12-31","name":"SERRAVALLE 
DI CHIENTI"}],"D899":[{"end":"1992-04-15","start":"1861-03-17","name":"GAMBE
TTOLA"},{"end":"9999-12-31","start":"1992-04-16","name":"GAMBETTOLA"}],"E745
":[{"end":"9999-12-31","start":"1861-03-17","name":"LUNGRO"}],"F095":[{"name
":"MELARA","start":"1866-11-19","end":"9999-12-31"}],"Z610":[{"end":"9999-12
-31","start":"0000-01-01","name":"Paraguay"}],"M398":[{"end":"9999-12-31","s
tart":"2018-01-01","name":"CELLIO CON BREIA"}],"D683":[{"end":"1927-01-11","
start":"1862-09-08","name":"FONTEGRECA"},{"name":"FONTEGRECA","end":"1945-08
-31","start":"1927-01-12"},{"name":"FONTEGRECA","start":"1945-09-01","end":"
9999-12-31"}],"A621":[{"name":"BARATILI SAN PIETRO","end":"1927-10-11","star
t":"1864-11-07"},{"name":"BARATILI SAN PIETRO","start":"1946-02-07","end":"1
974-08-01"},{"name":"BARATILI SAN PIETRO","start":"1974-08-02","end":"1974-0
8-19"},{"name":"BARATILI SAN PIETRO","end":"9999-12-31","start":"1974-08-20"
}],"C969":[{"name":"CONTIGLIANO","start":"1880-04-05","end":"1923-03-18"},{"
start":"1923-03-19","end":"1927-01-11","name":"CONTIGLIANO"},{"start":"1927-
01-12","end":"1928-02-20","name":"CONTIGLIANO"},{"name":"CONTIGLIANO","end":
"9999-12-31","start":"1946-03-09"}],"B912":[{"start":"1866-11-19","end":"194
1-02-06","name":"CASAL SERUGO"},{"name":"CASALSERUGO","start":"1941-02-07","
end":"9999-12-31"}],"C666":[{"end":"1923-02-13","start":"1920-10-16","name":
"CHIZZOLA"},{"end":"1928-09-05","start":"1923-02-14","name":"CHIZZOLA"}],"A1
16":[{"name":"ALA","start":"1920-10-16","end":"1923-02-13"},{"name":"ALA","e
nd":"1928-09-05","start":"1923-02-14"},{"name":"ALA","start":"1928-09-06","e
nd":"9999-12-31"}],"F923":[{"name":"NOICATTARO","end":"1934-02-08","start":"
1863-01-19"},{"start":"1934-02-09","end":"9999-12-31","name":"NOICATTARO"}],
"A415":[{"name":"ARMENTO","end":"9999-12-31","start":"1861-03-17"}],"E060":[
{"name":"GIUNGANO","start":"1861-03-17","end":"9999-12-31"}],"I876":[{"name"
:"SOVERZENE","start":"1866-11-19","end":"9999-12-31"}],"A682":[{"start":"186
1-03-17","end":"1927-01-11","name":"BARZA"},{"end":"1928-06-14","start":"192
7-01-12","name":"BARZA"}],"L098":[{"end":"1923-02-13","start":"1920-10-16","
name":"TEODONA"},{"end":"1923-04-26","start":"1923-02-14","name":"TEODONA"},
{"end":"1927-01-11","start":"1923-04-27","name":"TEODONE"},{"start":"1927-01
-12","end":"1928-04-04","name":"TEODONE"}],"B189":[{"name":"BRISINO","start"
:"1861-03-17","end":"1927-03-01"}],"D044":[{"name":"CORSANO","start":"1861-0
3-17","end":"9999-12-31"}],"I488":[{"start":"1920-10-16","end":"1923-02-13",
"name":"SCALE"},{"end":"1923-04-26","start":"1923-02-14","name":"SCALE"},{"n
ame":"SCALERES","start":"1923-04-27","end":"1927-01-11"},{"end":"1929-05-07"
,"start":"1927-01-12","name":"SCALERES"}],"C651":[{"name":"CHIURO","end":"99
99-12-31","start":"1861-03-17"}],"F820":[{"name":"MUSCOLINE","start":"1866-0
1-14","end":"1928-07-27"},{"start":"1928-07-28","end":"9999-12-31","name":"M
USCOLINE"}],"Z368":[{"name":"Eritrea","start":"0000-01-01","end":"9999-12-31
"}],"F167":[{"name":"MEZZANA MORTIGLIENGO","start":"1863-04-23","end":"1927-
01-11"},{"start":"1927-01-12","end":"1992-04-15","name":"MEZZANA MORTIGLIENG
O"},{"name":"MEZZANA MORTIGLIENGO","start":"1992-04-16","end":"9999-12-31"}]
,"M288":[{"start":"1986-04-03","end":"2016-04-27","name":"CASTIADAS"},{"name
":"CASTIADAS","end":"9999-12-31","start":"2016-04-28"}],"I184":[{"end":"9999
-12-31","start":"1864-11-07","name":"SANTA DOMENICA VITTORIA"}],"H840":[{"en
d":"9999-12-31","start":"1912-01-07","name":"SAN FERMO DELLA BATTAGLIA"},{"n
ame":"VERGOSA","end":"1912-01-06","start":"1861-03-17"}],"I548":[{"start":"1
861-03-17","end":"9999-12-31","name":"SCORDIA"}],"B648":[{"start":"1884-07-0
7","end":"1913-07-12","name":"CAPANNORI"},{"end":"1926-01-10","start":"1913-
07-13","name":"CAPANNORI"},{"end":"9999-12-31","start":"1926-01-11","name":"
CAPANNORI"}],"L602":[{"end":"1923-02-07","start":"1921-01-05","name":"VALLE 
D'ISTRIA"},{"name":"VALLE D'ISTRIA","end":"1947-09-15","start":"1923-02-08"}
],"E159":[{"start":"1939-04-22","end":"9999-12-31","name":"GRAZZANO BADOGLIO
"},{"name":"GRAZZANO MONFERRATO","start":"1868-12-31","end":"1935-04-07"},{"
start":"1935-04-08","end":"1939-04-21","name":"GRAZZANO MONFERRATO"}],"L965"
:[{"end":"9999-12-31","start":"1861-03-17","name":"VILLAMAINA"}],"D808":[{"s
tart":"1861-03-17","end":"1928-09-24","name":"FRONTONE"},{"name":"FRONTONE",
"end":"1998-09-03","start":"1946-02-06"},{"name":"FRONTONE","end":"9999-12-3
1","start":"1998-09-04"}],"I298":[{"name":"SANT'ANTONIO RUINAS","start":"186
1-03-17","end":"1928-06-14"},{"end":"1974-08-19","start":"1951-01-12","name"
:"SANT'ANTONIO RUINAS"},{"name":"SANT'ANTONIO RUINAS","end":"1985-05-11","st
art":"1974-08-20"},{"name":"VILLA SANT'ANTONIO","start":"1985-05-12","end":"
9999-12-31"}],"M194":[{"end":"9999-12-31","start":"1866-11-19","name":"ZOVEN
CEDO"}],"F565":[{"name":"MONTEMIGNAIO","end":"9999-12-31","start":"1868-04-0
1"}],"H714":[{"start":"1866-11-19","end":"9999-12-31","name":"SALIZZOLE"}],"
E240":[{"name":"GUARDA VENETA","end":"9999-12-31","start":"1866-11-19"}],"F9
68":[{"name":"NOVIGLIO","start":"1870-08-02","end":"9999-12-31"}],"H152":[{"
name":"RACIGNES","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-
14","end":"1923-04-26","name":"RACIGNES"},{"name":"RACINES","start":"1923-04
-27","end":"1927-01-11"},{"start":"1927-01-12","end":"1928-04-11","name":"RA
CINES"},{"start":"1928-04-12","end":"9999-12-31","name":"RACINES"}],"I841":[
{"start":"1861-03-17","end":"1928-10-15","name":"SORANO"},{"name":"SORANO","
start":"1928-10-16","end":"9999-12-31"}],"I388":[{"name":"SAN VINCENZO LA CO
STA","end":"1928-05-11","start":"1863-02-06"},{"end":"9999-12-31","start":"1
937-07-18","name":"SAN VINCENZO LA COSTA"}],"D501":[{"end":"1927-01-11","sta
rt":"1861-03-17","name":"FARINDOLA"},{"name":"FARINDOLA","start":"1927-01-12
","end":"9999-12-31"}],"H943":[{"start":"1863-01-19","end":"9999-12-31","nam
e":"SAN GREGORIO MAGNO"}],"G012":[{"start":"1939-10-21","end":"1946-01-29","
name":"OIASSE"},{"name":"OYACE","start":"1861-03-17","end":"1927-01-11"},{"s
tart":"1927-01-12","end":"1939-10-20","name":"OYACE"},{"name":"OYACE","start
":"1946-01-30","end":"9999-12-31"}],"A315":[{"end":"1927-01-11","start":"186
1-03-17","name":"ANTRODOCO"},{"name":"ANTRODOCO","end":"1928-05-11","start":
"1927-01-12"},{"end":"1939-11-14","start":"1928-05-12","name":"ANTRODOCO"},{
"start":"1939-11-15","end":"1946-12-16","name":"ANTRODOCO"},{"name":"ANTRODO
CO","start":"1946-12-17","end":"1957-12-13"},{"name":"ANTRODOCO","end":"9999
-12-31","start":"1957-12-14"}],"D264":[{"end":"1927-01-11","start":"1861-03-
17","name":"DEGO"},{"start":"1927-01-12","end":"1929-04-11","name":"DEGO"},{
"name":"DEGO","end":"9999-12-31","start":"1929-04-12"}],"I836":[{"end":"1928
-10-08","start":"1861-03-17","name":"SOPRAPONTE"}],"I945":[{"end":"1992-04-1
5","start":"1861-03-17","name":"STEFANACONI"},{"name":"STEFANACONI","end":"9
999-12-31","start":"1992-04-16"}],"E590":[{"end":"1992-04-15","start":"1861-
03-17","name":"LIMBADI"},{"name":"LIMBADI","end":"9999-12-31","start":"1992-
04-16"}],"A191":[{"name":"ALFONSINE","end":"9999-12-31","start":"1861-03-17"
}],"L568":[{"name":"VALEGGIO","start":"1861-03-17","end":"9999-12-31"}],"G55
2":[{"start":"1861-03-17","end":"2013-12-31","name":"PIAN DI SCO"}],"F854":[
{"start":"1861-03-17","end":"1928-11-26","name":"NAVONO"}],"L856":[{"end":"9
999-12-31","start":"1866-11-19","name":"VIDOR"}],"E547":[{"name":"LESIGNANO 
DE' BAGNI","end":"9999-12-31","start":"1861-03-17"}],"L979":[{"end":"9999-12
-31","start":"1867-08-26","name":"VILLANOVA DI CAMPOSAMPIERO"}],"F674":[{"st
art":"1882-06-19","end":"1992-04-15","name":"MONTICELLO BRIANZA"},{"start":"
1992-04-16","end":"9999-12-31","name":"MONTICELLO BRIANZA"}],"B920":[{"end":
"9999-12-31","start":"1861-03-17","name":"CASALVOLONE"}],"L676":[{"name":"VA
RCO SABINO","end":"1923-03-18","start":"1873-08-25"},{"start":"1923-03-19","
end":"1927-01-11","name":"VARCO SABINO"},{"name":"VARCO SABINO","start":"192
7-01-12","end":"1968-06-20"},{"name":"VARCO SABINO","end":"9999-12-31","star
t":"1968-06-21"}],"H752":[{"end":"9999-12-31","start":"1861-03-17","name":"S
AMOLACO"}],"E331":[{"end":"1923-02-13","start":"1920-10-16","name":"ISCHIA"}
,{"name":"ISCHIA","end":"1929-01-24","start":"1923-02-14"}],"B457":[{"name":
"CAMANDONA","end":"1927-01-11","start":"1861-03-17"},{"name":"CAMANDONA","en
d":"1965-02-15","start":"1927-01-12"},{"start":"1965-02-16","end":"1992-04-1
5","name":"CAMANDONA"},{"name":"CAMANDONA","start":"1992-04-16","end":"9999-
12-31"}],"D615":[{"name":"FISCIANO","end":"9999-12-31","start":"1869-08-24"}
],"D482":[{"name":"FALOPPIO","end":"9999-12-31","start":"1928-05-12"}],"C935
":[{"start":"1861-03-17","end":"1932-04-11","name":"COMUNANZA"},{"start":"19
45-09-14","end":"9999-12-31","name":"COMUNANZA"},{"start":"1932-04-12","end"
:"1945-09-13","name":"COMUNANZA DEL LITTORIO"}],"C846":[{"start":"1863-02-06
","end":"9999-12-31","name":"COLLE SANNITA"}],"E346":[{"name":"ISOLABONA","e
nd":"9999-12-31","start":"1861-03-17"}],"E297":[{"name":"INCUDINE","start":"
1861-03-17","end":"1927-11-23"},{"name":"INCUDINE","end":"1947-11-02","start
":"1927-11-24"},{"name":"INCUDINE","end":"9999-12-31","start":"1947-11-03"}]
,"G274":[{"name":"PALESTRINA","end":"9999-12-31","start":"1871-01-15"}],"F57
9":[{"name":"MONTENERO SABINO","start":"1861-03-17","end":"1923-03-18"},{"en
d":"1927-01-11","start":"1923-03-19","name":"MONTENERO SABINO"},{"name":"MON
TENERO SABINO","end":"9999-12-31","start":"1927-01-12"}],"A791":[{"end":"191
2-10-30","start":"1861-03-17","name":"BEREGAZZO"},{"end":"1928-02-20","start
":"1912-11-01","name":"BEREGAZZO CON FIGLIARO"},{"name":"BEREGAZZO CON FIGLI
ARO","start":"1948-03-21","end":"9999-12-31"}],"L009":[{"end":"1927-01-11","
start":"1871-01-15","name":"SUPINO"},{"name":"SUPINO","start":"1927-01-12","
end":"9999-12-31"}],"C831":[{"start":"1861-03-17","end":"1928-11-30","name":
"COLCAVAGNO"},{"end":"1998-08-31","start":"1947-12-24","name":"COLCAVAGNO"}]
,"L303":[{"end":"1928-03-13","start":"1861-03-17","name":"TORRITA"},{"end":"
9999-12-31","start":"1928-03-14","name":"TORRITA DI SIENA"}],"E471":[{"name"
:"LATIANO","end":"1927-01-11","start":"1861-03-17"},{"end":"9999-12-31","sta
rt":"1927-01-12","name":"LATIANO"}],"E387":[{"start":"1861-03-17","end":"192
7-01-11","name":"JERZU"},{"end":"2005-12-31","start":"1927-01-12","name":"JE
RZU"},{"end":"2016-04-27","start":"2006-01-01","name":"JERZU"},{"name":"JERZ
U","start":"2016-04-28","end":"9999-12-31"}],"F812":[{"end":"1927-01-11","st
art":"1861-03-17","name":"MURIAGLIO"},{"name":"MURIAGLIO","end":"1929-05-03"
,"start":"1927-01-12"}],"G514":[{"name":"PETRIANO","end":"1928-02-20","start
":"1861-03-17"},{"start":"1947-01-19","end":"1998-09-03","name":"PETRIANO"},
{"name":"PETRIANO","end":"9999-12-31","start":"1998-09-04"}],"B091":[{"name"
:"BOTTICINO","end":"1928-11-26","start":"1928-03-29"},{"start":"1928-11-27",
"end":"9999-12-31","name":"BOTTICINO"}],"H418":[{"name":"ROCCALUMERA","end":
"1914-07-19","start":"1880-04-26"},{"start":"1914-07-20","end":"1929-05-27",
"name":"ROCCALUMERA"},{"name":"ROCCALUMERA","start":"1929-05-28","end":"1948
-12-24"},{"name":"ROCCALUMERA","start":"1948-12-25","end":"9999-12-31"}],"B8
23":[{"name":"CARPIGNANO SESIA","start":"1863-04-23","end":"1928-02-22"},{"e
nd":"1948-01-16","start":"1928-02-23","name":"CARPIGNANO SESIA"},{"start":"1
948-01-17","end":"9999-12-31","name":"CARPIGNANO SESIA"}],"F219":[{"name":"M
INERBIO","start":"1869-01-30","end":"9999-12-31"}],"E759":[{"name":"LUSERNET
TA","end":"1928-03-26","start":"1861-03-17"},{"start":"1947-01-09","end":"99
99-12-31","name":"LUSERNETTA"}],"E984":[{"name":"MARTIGNANO","end":"9999-12-
31","start":"1861-03-17"}],"G054":[{"end":"1929-05-13","start":"1861-03-17",
"name":"OLTRESSENDA ALTA"},{"start":"1958-09-05","end":"9999-12-31","name":"
OLTRESSENDA ALTA"}],"H897":[{"name":"SAN GIORGIO IN BOSCO","end":"1986-05-26
","start":"1882-07-20"},{"end":"9999-12-31","start":"1986-05-27","name":"SAN
 GIORGIO IN BOSCO"}],"E586":[{"name":"LIGOSULLO","end":"1968-04-05","start":
"1866-11-19"},{"name":"LIGOSULLO","end":"2018-01-31","start":"1968-04-06"}],
"M003":[{"start":"1861-03-17","end":"1927-01-11","name":"VILLARBOIT"},{"name
":"VILLARBOIT","start":"1927-01-12","end":"9999-12-31"}],"M309":[{"start":"2
001-10-15","end":"9999-12-31","name":"FONTE NUOVA"}],"D421":[{"end":"1962-05
-25","start":"1861-03-17","name":"ERBUSCO"},{"end":"9999-12-31","start":"196
2-05-26","name":"ERBUSCO"}],"C538":[{"name":"CERRO VERONESE","end":"1928-05-
01","start":"1867-08-26"},{"name":"CERRO VERONESE","start":"1947-10-05","end
":"9999-12-31"}],"A483":[{"name":"ATELLA DI NAPOLI","start":"1928-05-29","en
d":"1945-08-31"},{"start":"1945-09-01","end":"1946-04-26","name":"ATELLA DI 
NAPOLI"}],"H114":[{"end":"9999-12-31","start":"1948-03-11","name":"QUARTO"}]
,"G916":[{"end":"1938-12-24","start":"1861-03-17","name":"PORTOMAGGIORE"},{"
name":"PORTOMAGGIORE","start":"1938-12-25","end":"1959-10-08"},{"start":"195
9-10-09","end":"1960-03-11","name":"PORTOMAGGIORE"},{"start":"1960-03-12","e
nd":"9999-12-31","name":"PORTOMAGGIORE"}],"G619":[{"name":"PIETRA MARAZZI","
end":"1928-10-12","start":"1861-03-17"},{"start":"1928-10-13","end":"1929-01
-14","name":"PIETRA MARAZZI"},{"name":"PIETRA MARAZZI","start":"1929-01-15",
"end":"1956-01-25"},{"end":"9999-12-31","start":"1956-01-26","name":"PIETRA 
MARAZZI"}],"I665":[{"end":"1923-02-13","start":"1920-10-16","name":"SERRAVAL
LE ALL'ADIGE"},{"end":"1928-09-05","start":"1923-02-14","name":"SERRAVALLE A
LL'ADIGE"}],"A290":[{"start":"1861-03-17","end":"1927-01-11","name":"ANGERA"
},{"name":"ANGERA","end":"1928-01-12","start":"1927-01-12"},{"start":"1928-0
1-13","end":"1928-05-30","name":"ANGERA"},{"name":"ANGERA","start":"1928-05-
31","end":"1957-01-24"},{"name":"ANGERA","end":"9999-12-31","start":"1957-01
-25"}],"I902":[{"name":"SPIGNO SATURNIA","end":"1927-01-11","start":"1863-02
-06"},{"start":"1927-01-12","end":"1934-10-25","name":"SPIGNO SATURNIA"},{"s
tart":"1934-10-26","end":"1934-12-17","name":"SPIGNO SATURNIA"},{"start":"19
34-12-18","end":"1945-04-08","name":"SPIGNO SATURNIA"},{"start":"1945-04-09"
,"end":"9999-12-31","name":"SPIGNO SATURNIA"}],"C188":[{"name":"CASTELLO DI 
CISTERNA","end":"1927-01-11","start":"1862-10-06"},{"name":"CASTELLO DI CIST
ERNA","end":"9999-12-31","start":"1927-01-12"}],"G248":[{"start":"1861-03-17
","end":"1928-08-15","name":"PAITONE"},{"end":"9999-12-31","start":"1947-05-
12","name":"PAITONE"}],"H198":[{"name":"RAVELLO","end":"1929-03-09","start":
"1861-03-17"},{"start":"1929-03-10","end":"1946-04-25","name":"RAVELLO"},{"s
tart":"1946-04-26","end":"9999-12-31","name":"RAVELLO"}],"E136":[{"name":"GR
ANAROLO DELL'EMILIA","start":"1875-10-26","end":"9999-12-31"}],"D325":[{"sta
rt":"1866-11-19","end":"9999-12-31","name":"DOLO"}],"B565":[{"name":"CAMPOSA
NO","start":"1861-03-17","end":"1927-01-11"},{"end":"1984-02-27","start":"19
27-01-12","name":"CAMPOSANO"},{"name":"CAMPOSANO","start":"1984-02-28","end"
:"9999-12-31"}],"D696":[{"name":"FORENZA","end":"9999-12-31","start":"1861-0
3-17"}],"G598":[{"name":"PIEDIMONTE SAN GERMANO","start":"1862-12-29","end":
"1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":"PIEDIMONTE SA
N GERMANO"}],"C484":[{"end":"1929-05-13","start":"1861-03-17","name":"CERANO
VA"},{"end":"1963-06-20","start":"1947-10-02","name":"CERANOVA"},{"name":"CE
RANOVA","start":"1963-06-21","end":"9999-12-31"}],"D707":[{"end":"9999-12-31
","start":"1871-01-15","name":"FORMELLO"}],"F648":[{"name":"MONTESPERTOLI","
end":"9999-12-31","start":"1861-03-17"}],"L400":[{"name":"TREVIGLIO","start"
:"1861-03-17","end":"9999-12-31"}],"D999":[{"name":"GHEDI","start":"1861-03-
17","end":"9999-12-31"}],"A540":[{"name":"BADIA CALAVENA","end":"9999-12-31"
,"start":"1866-11-19"}],"H663":[{"end":"1929-05-27","start":"1891-06-09","na
me":"SAGLIANO DI CRENNA"}],"E141":[{"name":"GRANDOLA ED UNITI","start":"1927
-10-13","end":"1950-04-04"},{"name":"GRANDOLA ED UNITI","end":"9999-12-31","
start":"1950-04-05"}],"Z312":[{"name":"Repubblica Democratica del Congo","en
d":"9999-12-31","start":"0000-01-01"}],"H494":[{"start":"1871-01-15","end":"
9999-12-31","name":"ROIATE"}],"I072":[{"name":"SAN PAOLO DI CIVITATE","start
":"1862-11-10","end":"9999-12-31"}],"C971":[{"name":"CONTRADA","end":"9999-1
2-31","start":"1861-03-17"}],"B854":[{"end":"9999-12-31","start":"1861-03-17
","name":"CARVICO"}],"L227":[{"name":"TORNIMPARTE","end":"9999-12-31","start
":"1861-03-17"}],"G995":[{"end":"1927-01-11","start":"1907-03-08","name":"PR
ATELLA"},{"end":"1945-08-31","start":"1927-01-12","name":"PRATELLA"},{"name"
:"PRATELLA","end":"9999-12-31","start":"1945-09-01"}],"E375":[{"name":"ITRI"
,"end":"1927-01-11","start":"1861-03-17"},{"name":"ITRI","start":"1927-01-12
","end":"1934-10-25"},{"end":"1934-12-17","start":"1934-10-26","name":"ITRI"
},{"name":"ITRI","end":"1945-04-08","start":"1934-12-18"},{"start":"1945-04-
09","end":"9999-12-31","name":"ITRI"}],"E435":[{"name":"LANCIANO","end":"999
9-12-31","start":"1861-03-17"}],"C875":[{"start":"1861-03-17","end":"1970-03
-02","name":"COLLETORTO"},{"end":"9999-12-31","start":"1970-03-03","name":"C
OLLETORTO"}],"L232":[{"name":"TORRA","end":"1923-02-13","start":"1920-10-16"
},{"name":"TORRA","end":"1929-01-23","start":"1923-02-14"}],"A380":[{"name":
"ARDAULI","start":"1861-03-17","end":"1974-08-19"},{"start":"1974-08-20","en
d":"9999-12-31","name":"ARDAULI"}],"I859":[{"name":"SORLI","start":"1861-03-
17","end":"1928-10-29"}],"G891":[{"start":"1866-11-19","end":"1968-04-05","n
ame":"PORPETTO"},{"name":"PORPETTO","end":"9999-12-31","start":"1968-04-06"}
],"I252":[{"end":"1957-05-01","start":"1928-08-12","name":"SANTA MARIA REZZO
NICO"},{"start":"1957-05-02","end":"2002-12-31","name":"SANTA MARIA REZZONIC
O"}],"B968":[{"start":"1861-03-17","end":"1950-05-27","name":"CASINO"},{"nam
e":"CASTELSILANO","end":"1992-04-15","start":"1950-05-28"},{"end":"9999-12-3
1","start":"1992-04-16","name":"CASTELSILANO"}],"G329":[{"name":"PAR\u00c8",
"end":"1928-06-13","start":"1861-03-17"},{"start":"1956-11-10","end":"2014-0
2-03","name":"PAR\u00c8"}],"G023":[{"name":"OLEVANO SUL TUSCIANO","end":"999
9-12-31","start":"1863-03-30"}],"C001":[{"name":"CASSANO ALBESE","end":"1928
-09-22","start":"1862-12-29"}],"L839":[{"name":"VICENO","end":"1928-11-10","
start":"1861-03-17"}],"L645":[{"start":"1861-03-17","end":"1927-01-11","name
":"VALPRATO"},{"end":"1929-01-22","start":"1927-01-12","name":"VALPRATO"}],"
A286":[{"start":"1920-10-16","end":"1923-02-13","name":"ANDRIANO"},{"name":"
ANDRIANO","start":"1923-02-14","end":"1927-01-11"},{"start":"1927-01-12","en
d":"1929-01-01","name":"ANDRIANO"},{"start":"1953-05-23","end":"9999-12-31",
"name":"ANDRIANO"}],"L520":[{"name":"USTIE","start":"1921-01-05","end":"1923
-02-07"},{"name":"USTIE","end":"1927-01-11","start":"1923-02-08"},{"end":"19
28-07-27","start":"1927-01-12","name":"USTIE"}],"B579":[{"end":"1923-02-13",
"start":"1920-10-16","name":"CANAZEI"},{"start":"1923-02-14","end":"1926-02-
07","name":"CANAZEI"},{"name":"CANAZEI","start":"1926-02-08","end":"1956-11-
03"},{"start":"1956-11-04","end":"1999-12-07","name":"CANAZEI"},{"name":"CAN
AZEI","start":"1999-12-08","end":"9999-12-31"}],"Z354":[{"name":"Burkina Fas
o","end":"9999-12-31","start":"0000-01-01"}],"Z613":[{"name":"Uruguay","star
t":"0000-01-01","end":"9999-12-31"}],"C655":[{"name":"CHIUSA DI SAN MICHELE"
,"start":"1863-01-05","end":"9999-12-31"}],"H362":[{"name":"ROASCHIA","end":
"1928-03-06","start":"1861-03-17"},{"end":"9999-12-31","start":"1946-04-26",
"name":"ROASCHIA"}],"I976":[{"end":"1927-02-28","start":"1861-03-17","name":
"STRESA"},{"end":"1992-05-22","start":"1946-10-27","name":"STRESA"},{"name":
"STRESA","start":"1992-05-23","end":"9999-12-31"},{"start":"1927-03-01","end
":"1928-05-28","name":"STRESA BORROMEO"},{"start":"1928-05-29","end":"1946-1
0-26","name":"STRESA BORROMEO"}],"I679":[{"end":"1966-10-14","start":"1861-0
3-17","name":"SESSANO"},{"name":"SESSANO DEL MOLISE","end":"1970-03-02","sta
rt":"1966-10-15"},{"end":"9999-12-31","start":"1970-03-03","name":"SESSANO D
EL MOLISE"}],"I006":[{"name":"SAN MARTINO DE' CALVI","end":"1956-08-26","sta
rt":"1927-09-14"}],"D680":[{"name":"FONTE","end":"9999-12-31","start":"1866-
11-19"}],"F457":[{"name":"MONTECASTRILLI","end":"1927-01-11","start":"1861-0
3-17"},{"end":"1975-04-23","start":"1927-01-12","name":"MONTECASTRILLI"},{"s
tart":"1975-04-24","end":"9999-12-31","name":"MONTECASTRILLI"}],"I574":[{"na
me":"SEGNO","end":"1927-01-11","start":"1861-03-17"},{"end":"1929-05-29","st
art":"1927-01-12","name":"SEGNO"}],"B674":[{"end":"1938-10-16","start":"1861
-03-17","name":"CAPOSELE"},{"start":"1938-10-17","end":"9999-12-31","name":"
CAPOSELE"}],"F920":[{"end":"1923-02-13","start":"1920-10-16","name":"NOGARED
O"},{"end":"1929-02-20","start":"1923-02-14","name":"NOGAREDO"},{"end":"9999
-12-31","start":"1955-03-02","name":"NOGAREDO"}],"A411":[{"name":"ARIZZANO S
UPERIORE","start":"1911-07-04","end":"1928-01-07"}],"M118":[{"start":"1866-1
1-19","end":"1925-06-07","name":"VOLPAGO"},{"end":"9999-12-31","start":"1925
-06-08","name":"VOLPAGO DEL MONTELLO"}],"A597":[{"name":"BALLAO","start":"18
61-03-17","end":"1974-08-19"},{"name":"BALLAO","end":"2016-04-27","start":"1
974-08-20"},{"name":"BALLAO","end":"9999-12-31","start":"2016-04-28"}],"C788
":[{"start":"1925-09-26","end":"1947-09-15","name":"CLANA"}],"E063":[{"start
":"1871-01-01","end":"2009-06-29","name":"GIUSSANO"},{"name":"GIUSSANO","end
":"9999-12-31","start":"2009-06-30"}],"I214":[{"name":"SANT'ALESSIO IN ASPRO
MONTE","start":"1864-06-23","end":"9999-12-31"}],"E369":[{"name":"ISSIME","s
tart":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"1952-04-
18","name":"ISSIME"},{"name":"ISSIME","start":"1952-04-19","end":"9999-12-31
"}],"H798":[{"name":"SAN CIPRIANO D'AVERSA","end":"1927-01-11","start":"1862
-12-29"},{"name":"SAN CIPRIANO D'AVERSA","start":"1927-01-12","end":"1928-05
-23"},{"start":"1946-03-25","end":"1973-03-13","name":"SAN CIPRIANO D'AVERSA
"},{"end":"9999-12-31","start":"1973-03-14","name":"SAN CIPRIANO D'AVERSA"}]
,"A625":[{"name":"BARBANIA","start":"1861-03-17","end":"1928-02-20"},{"name"
:"BARBANIA","end":"1938-05-31","start":"1928-02-21"},{"start":"1938-06-01","
end":"1947-03-29","name":"BARBANIA"},{"name":"BARBANIA","end":"9999-12-31","
start":"1947-03-30"}],"E736":[{"start":"1895-10-10","end":"1927-01-11","name
":"LULA"},{"start":"1927-01-12","end":"9999-12-31","name":"LULA"}],"D038":[{
"end":"2009-06-29","start":"1861-03-17","name":"CORREZZANA"},{"name":"CORREZ
ZANA","end":"9999-12-31","start":"2009-06-30"}],"Z207":[{"end":"9999-12-31",
"start":"0000-01-01","name":"Brunei Darussalam"}],"B219":[{"end":"1927-01-11
","start":"1861-03-17","name":"BRUNELLO"},{"name":"BRUNELLO","start":"1927-0
1-12","end":"1928-01-12"},{"name":"BRUNELLO","start":"1956-03-04","end":"999
9-12-31"}],"C262":[{"start":"1861-03-17","end":"9999-12-31","name":"CASTEL S
AN LORENZO"}],"D107":[{"start":"1866-11-19","end":"9999-12-31","name":"COSTA
BISSARA"}],"F823":[{"name":"MUSELLARO","end":"1927-01-11","start":"1861-03-1
7"},{"end":"1928-09-24","start":"1927-01-12","name":"MUSELLARO"}],"M414":[{"
start":"2019-01-01","end":"9999-12-31","name":"QUAREGNA CERRETO"}],"A396":[{
"name":"ARGUELLO","end":"1928-05-01","start":"1861-03-17"},{"start":"1947-03
-30","end":"9999-12-31","name":"ARGUELLO"}],"A247":[{"name":"ALZANO MAGGIORE
","start":"1861-03-17","end":"1927-06-06"}],"E741":[{"name":"LUMEZZANE SANT'
APOLLONIO","end":"1921-04-28","start":"1861-03-17"},{"end":"1928-01-06","sta
rt":"1921-04-29","name":"LUMEZZANE SANT'APOLLONIO"}],"C869":[{"start":"1861-
03-17","end":"1910-06-21","name":"COLLESALVETTI"},{"name":"COLLESALVETTI","s
tart":"1910-06-22","end":"1925-11-29"},{"start":"1925-11-30","end":"1931-10-
02","name":"COLLESALVETTI"},{"end":"9999-12-31","start":"1931-10-03","name":
"COLLESALVETTI"}],"B812":[{"name":"CARPANETO","end":"1929-10-05","start":"18
61-03-17"},{"name":"CARPANETO PIACENTINO","end":"9999-12-31","start":"1929-1
0-06"}],"F091":[{"end":"2018-02-16","start":"1866-11-19","name":"MEGLIADINO 
SAN FIDENZIO"}],"A195":[{"name":"ALIA","start":"1861-03-17","end":"9999-12-3
1"}],"H843":[{"name":"SAN FIOR","start":"1866-11-19","end":"9999-12-31"}],"I
534":[{"name":"SCIARA","start":"1861-03-17","end":"9999-12-31"}],"H462":[{"e
nd":"1928-02-20","start":"1861-03-17","name":"ROCCHETTA BELBO"},{"name":"ROC
CHETTA BELBO","end":"9999-12-31","start":"1947-03-15"}],"F868":[{"start":"18
71-01-15","end":"1928-12-29","name":"NEPI"},{"start":"1928-12-30","end":"195
5-05-26","name":"NEPI"},{"start":"1955-05-27","end":"9999-12-31","name":"NEP
I"}],"L554":[{"name":"VALDASTICO","end":"1980-06-19","start":"1940-09-13"},{
"name":"VALDASTICO","end":"9999-12-31","start":"1980-06-20"}],"F556":[{"end"
:"1935-04-07","start":"1861-03-17","name":"MONTEMAGNO"},{"name":"MONTEMAGNO"
,"start":"1935-04-08","end":"1953-01-24"},{"name":"MONTEMAGNO","start":"1953
-01-25","end":"9999-12-31"}],"I941":[{"end":"1928-09-22","start":"1861-03-17
","name":"STAZZANO"},{"start":"1928-09-23","end":"1929-04-11","name":"STAZZA
NO"},{"start":"1929-04-12","end":"9999-12-31","name":"STAZZANO"}],"B634":[{"
name":"CANTELLO","end":"1927-01-11","start":"1895-08-02"},{"name":"CANTELLO"
,"end":"1968-07-09","start":"1927-01-12"},{"start":"1968-07-10","end":"9999-
12-31","name":"CANTELLO"}],"B767":[{"end":"1927-01-11","start":"1861-03-17",
"name":"CARESANA"},{"end":"9999-12-31","start":"1927-01-12","name":"CARESANA
"}],"Z320":[{"end":"9999-12-31","start":"0000-01-01","name":"Guinea-Bissau"}
],"E102":[{"start":"1870-08-02","end":"1916-04-30","name":"GORLA MINORE"},{"
end":"1927-01-11","start":"1916-05-01","name":"GORLA MINORE"},{"name":"GORLA
 MINORE","start":"1927-01-12","end":"9999-12-31"}],"L659":[{"name":"VALVERDE
","end":"1929-02-23","start":"1861-03-17"},{"name":"VALVERDE","end":"2018-12
-31","start":"1956-04-08"}],"D505":[{"start":"1867-11-25","end":"9999-12-31"
,"name":"FARRA DI SOLIGO"}],"F954":[{"name":"NOVATE BRIANZA","end":"1927-06-
09","start":"1863-02-23"}],"F789":[{"name":"MOZZECANE","start":"1866-11-19",
"end":"9999-12-31"}],"L956":[{"end":"2013-11-11","start":"1928-05-31","name"
:"VILLA GUARDIA"},{"end":"9999-12-31","start":"2013-11-12","name":"VILLA GUA
RDIA"}],"B539":[{"name":"CAMPOGALLIANO","start":"1861-03-17","end":"9999-12-
31"}],"A311":[{"name":"ANTIGNANA","end":"1923-02-07","start":"1921-01-05"},{
"start":"1923-02-08","end":"1947-09-15","name":"ANTIGNANA"}],"I936":[{"name"
:"STAITI","end":"9999-12-31","start":"1885-01-15"}],"D726":[{"start":"1866-1
1-19","end":"2016-02-22","name":"FORNO DI ZOLDO"}],"I845":[{"name":"SORBOLO"
,"end":"1965-02-16","start":"1861-03-17"},{"name":"SORBOLO","start":"1965-02
-17","end":"2018-12-31"}],"I639":[{"name":"SERRA SAN BRUNO","start":"1863-02
-06","end":"1992-04-15"},{"start":"1992-04-16","end":"9999-12-31","name":"SE
RRA SAN BRUNO"}],"L014":[{"name":"SUSEGANA","start":"1889-08-16","end":"9999
-12-31"}],"Z158":[{"name":"Serbia","end":"9999-12-31","start":"0000-01-01"}]
,"F016":[{"start":"1959-10-09","end":"1989-01-13","name":"MASI TORELLO"},{"n
ame":"MASI TORELLO","end":"9999-12-31","start":"1989-01-14"}],"F357":[{"name
":"MONFESTINO IN SERRAMAZZONI","start":"1861-03-17","end":"1948-03-14"},{"en
d":"9999-12-31","start":"1948-03-15","name":"SERRAMAZZONI"}],"D258":[{"name"
:"DAZIO","start":"1861-03-17","end":"1964-01-27"},{"start":"1964-01-28","end
":"9999-12-31","name":"DAZIO"}],"E028":[{"name":"GIGNESE","start":"1861-03-1
7","end":"1928-08-11"},{"name":"GIGNESE","start":"1928-08-12","end":"1992-05
-22"},{"end":"9999-12-31","start":"1992-05-23","name":"GIGNESE"}],"E243":[{"
name":"GUARDIAGRELE","end":"9999-12-31","start":"1861-03-17"}],"G509":[{"nam
e":"PETINA","start":"1861-03-17","end":"9999-12-31"}],"L865":[{"name":"VIGAN
O SAN MARTINO","end":"1928-06-29","start":"1863-08-10"},{"name":"VIGANO SAN 
MARTINO","start":"1947-11-04","end":"9999-12-31"}],"D908":[{"name":"GARABIOL
O","start":"1861-03-17","end":"1927-01-11"},{"name":"GARABIOLO","start":"192
7-01-12","end":"1928-01-07"}],"H940":[{"name":"SAN GREGORIO DI CATANIA","sta
rt":"1862-12-29","end":"9999-12-31"}],"L206":[{"end":"1957-02-06","start":"1
864-11-07","name":"TORANO CASTELLO"},{"start":"1957-02-07","end":"9999-12-31
","name":"TORANO CASTELLO"}],"F204":[{"start":"1861-03-17","end":"1869-02-13
","name":"MIGNETTE"}],"E469":[{"start":"1861-03-17","end":"1923-12-21","name
":"LATERZA"},{"name":"LATERZA","end":"9999-12-31","start":"1923-12-22"}],"F5
61":[{"name":"MONTE MARENZO","start":"1861-03-17","end":"1992-04-15"},{"name
":"MONTE MARENZO","end":"9999-12-31","start":"1992-04-16"}],"A067":[{"name":
"AGAZZANO","start":"1861-03-17","end":"9999-12-31"}],"E593":[{"end":"1911-04
-04","start":"1861-03-17","name":"LIMIDO"},{"end":"9999-12-31","start":"1911
-04-05","name":"LIMIDO COMASCO"}],"L961":[{"start":"1861-03-17","end":"9999-
12-31","name":"VILLALFONSINA"}],"A496":[{"name":"AULLA","start":"1881-04-22"
,"end":"9999-12-31"}],"M314":[{"start":"2010-01-01","end":"9999-12-31","name
":"COMANO TERME"}],"D078":[{"end":"1927-01-11","start":"1861-03-17","name":"
CORVARA"},{"end":"9999-12-31","start":"1927-01-12","name":"CORVARA"}],"E776"
:[{"start":"1861-03-17","end":"1927-01-11","name":"MACCAGNO INFERIORE"},{"st
art":"1927-01-12","end":"1928-01-07","name":"MACCAGNO INFERIORE"}],"H109":[{
"end":"9999-12-31","start":"1959-10-03","name":"QUARRATA"},{"name":"TIZZANA"
,"start":"1861-03-17","end":"1927-01-11"},{"name":"TIZZANA","end":"1959-10-0
2","start":"1927-01-12"}],"G604":[{"name":"PIETRAMONTECORVINO","start":"1862
-11-10","end":"9999-12-31"}],"D425":[{"name":"ERPELLE-COSINA","end":"1947-09
-15","start":"1923-02-08"}],"B923":[{"name":"CASAMASSIMA","end":"1929-07-09"
,"start":"1861-03-17"},{"name":"CASAMASSIMA","start":"1929-07-10","end":"999
9-12-31"}],"L052":[{"name":"TARCETTA","end":"1929-01-24","start":"1866-11-19
"}],"G068":[{"start":"1861-03-17","end":"9999-12-31","name":"ONETA"}],"E884"
:[{"end":"9999-12-31","start":"1861-03-17","name":"MANERBIO"}],"I032":[{"nam
e":"SAN MAURO LA BRUCA","end":"9999-12-31","start":"1861-03-17"}],"F912":[{"
name":"NOCERA INFERIORE","start":"1861-03-17","end":"9999-12-31"}],"H394":[{
"end":"9999-12-31","start":"1861-03-17","name":"ROCCADASPIDE"}],"C987":[{"en
d":"9999-12-31","start":"1866-11-19","name":"CORBOLA"}],"L512":[{"start":"18
61-03-17","end":"1974-08-19","name":"USSANA"},{"end":"2016-04-27","start":"1
974-08-20","name":"USSANA"},{"end":"9999-12-31","start":"2016-04-28","name":
"USSANA"}],"G816":[{"name":"POMPONESCO","start":"1868-07-01","end":"9999-12-
31"}],"I027":[{"name":"SAN MAURO DI ROMAGNA","end":"1932-11-08","start":"186
3-01-15"},{"name":"SAN MAURO PASCOLI","start":"1932-11-09","end":"1990-11-07
"},{"end":"1992-04-15","start":"1990-11-08","name":"SAN MAURO PASCOLI"},{"en
d":"9999-12-31","start":"1992-04-16","name":"SAN MAURO PASCOLI"}],"H997":[{"
name":"SAN MARTINO CANAVESE","end":"1927-01-11","start":"1863-01-05"},{"end"
:"1928-06-25","start":"1927-01-12","name":"SAN MARTINO CANAVESE"},{"end":"19
29-05-02","start":"1928-06-26","name":"SAN MARTINO CANAVESE"},{"name":"SAN M
ARTINO CANAVESE","start":"1929-05-03","end":"1937-07-26"},{"start":"1937-07-
27","end":"1941-02-04","name":"SAN MARTINO CANAVESE"},{"start":"1941-02-05",
"end":"1945-10-04","name":"SAN MARTINO CANAVESE"},{"end":"1946-12-23","start
":"1945-10-05","name":"SAN MARTINO CANAVESE"},{"name":"SAN MARTINO CANAVESE"
,"start":"1946-12-24","end":"9999-12-31"}],"C384":[{"end":"1922-08-02","star
t":"1866-11-19","name":"CAVASO"},{"name":"CAVASO DEL TOMBA","start":"1922-08
-03","end":"9999-12-31"}],"H709":[{"name":"SALICE CALABRO","end":"1927-08-02
","start":"1864-06-23"}],"E176":[{"end":"1927-09-03","start":"1867-07-22","n
ame":"GRIGNANO DI POLESINE"}],"B095":[{"name":"BOTTRIGHE","end":"1924-01-13"
,"start":"1866-11-19"},{"end":"1929-01-12","start":"1924-01-14","name":"BOTT
RIGHE"}],"A795":[{"end":"9999-12-31","start":"1866-11-19","name":"BERGANTINO
"}],"H628":[{"name":"RUBIERA","start":"1861-03-17","end":"9999-12-31"}],"C64
9":[{"end":"9999-12-31","start":"1861-03-17","name":"CHIUDUNO"}],"E475":[{"n
ame":"LATTARICO","end":"1928-05-11","start":"1861-03-17"},{"name":"LATTARICO
","start":"1934-03-07","end":"1957-02-06"},{"name":"LATTARICO","start":"1957
-02-07","end":"9999-12-31"}],"H284":[{"start":"1920-10-16","end":"1923-02-13
","name":"RIFIANO"},{"start":"1923-02-14","end":"1927-01-11","name":"RIFIANO
"},{"name":"RIFIANO","start":"1927-01-12","end":"9999-12-31"}],"C835":[{"end
":"1928-02-22","start":"1861-03-17","name":"COLERE"},{"end":"9999-12-31","st
art":"1947-05-22","name":"COLERE"}],"L272":[{"name":"TORRE LE NOCELLE","end"
:"9999-12-31","start":"1861-03-17"}],"C946":[{"name":"CONCERVIANO","end":"19
10-09-08","start":"1861-03-17"},{"name":"CONCERVIANO","end":"1923-03-18","st
art":"1910-09-09"},{"name":"CONCERVIANO","end":"1927-01-11","start":"1923-03
-19"},{"name":"CONCERVIANO","end":"9999-12-31","start":"1927-01-12"}],"M352"
:[{"end":"9999-12-31","start":"2016-01-01","name":"BORGO CHIESE"}],"C294":[{
"name":"CASTIGLION DORA","start":"1939-10-21","end":"1946-01-29"},{"name":"C
H\u00c2TILLON","start":"1861-03-17","end":"1927-01-11"},{"name":"CH\u00c2TIL
LON","end":"1928-03-28","start":"1927-01-12"},{"end":"1939-10-20","start":"1
928-03-29","name":"CH\u00c2TILLON"},{"name":"CH\u00c2TILLON","end":"1946-03-
24","start":"1946-01-30"},{"name":"CH\u00c2TILLON","end":"9999-12-31","start
":"1946-03-25"}],"L879":[{"name":"VIGLIANO D'ASTI","end":"1935-04-07","start
":"1863-02-16"},{"end":"9999-12-31","start":"1935-04-08","name":"VIGLIANO D'
ASTI"}],"B820":[{"end":"9999-12-31","start":"1861-03-17","name":"CARPIANO"}]
,"B167":[{"name":"BRIAGLIA","end":"9999-12-31","start":"1861-03-17"}],"E702"
:[{"name":"LOVENO SOPRA MENAGGIO","start":"1862-12-29","end":"1929-05-13"}],
"D611":[{"name":"FIORENZUOLA D'ARDA","start":"1866-03-03","end":"9999-12-31"
}],"F189":[{"name":"MIAGLIANO","start":"1861-03-17","end":"1927-01-11"},{"na
me":"MIAGLIANO","start":"1927-01-12","end":"1929-05-21"},{"end":"1992-04-15"
,"start":"1955-01-19","name":"MIAGLIANO"},{"name":"MIAGLIANO","end":"9999-12
-31","start":"1992-04-16"}],"C931":[{"name":"COMMEZZADURA","end":"9999-12-31
","start":"1928-10-07"}],"G429":[{"end":"1923-02-13","start":"1920-10-16","n
ame":"PELUGO"},{"end":"1928-06-26","start":"1923-02-14","name":"PELUGO"},{"s
tart":"1946-12-19","end":"9999-12-31","name":"PELUGO"}],"D126":[{"start":"18
61-03-17","end":"9999-12-31","name":"COVO"}],"A480":[{"name":"ASUNI","start"
:"1861-03-17","end":"1974-08-19"},{"end":"9999-12-31","start":"1974-08-20","
name":"ASUNI"}],"Z226":[{"name":"Israele","start":"0000-01-01","end":"9999-1
2-31"}],"E335":[{"name":"ISERNIA","start":"1861-03-17","end":"1928-08-06"},{
"end":"1934-02-24","start":"1928-08-07","name":"ISERNIA"},{"name":"ISERNIA",
"start":"1934-02-25","end":"1937-02-23"},{"name":"ISERNIA","end":"1970-03-02
","start":"1937-02-24"},{"name":"ISERNIA","end":"9999-12-31","start":"1970-0
3-03"}],"C544":[{"start":"1866-11-19","end":"9999-12-31","name":"CERVARESE S
ANTA CROCE"}],"E207":[{"name":"GROTTAMMARE","start":"1861-03-17","end":"9999
-12-31"}],"G031":[{"end":"1927-01-11","start":"1861-03-17","name":"OLIENA"},
{"end":"9999-12-31","start":"1927-01-12","name":"OLIENA"}],"C565":[{"name":"
CESANO BOSCONE","start":"1861-03-17","end":"9999-12-31"}],"I489":[{"end":"19
11-12-31","start":"1861-03-17","name":"SCALEA"},{"start":"1912-01-01","end":
"9999-12-31","name":"SCALEA"}],"L099":[{"end":"1925-11-01","start":"1861-03-
17","name":"TEODORANO"}],"B188":[{"start":"1861-03-17","end":"9999-12-31","n
ame":"BRISIGHELLA"}],"L393":[{"name":"TRESNURAGHES","end":"1927-01-11","star
t":"1861-03-17"},{"name":"TRESNURAGHES","start":"1927-01-12","end":"1974-08-
19"},{"start":"1974-08-20","end":"9999-12-31","name":"TRESNURAGHES"}],"A701"
:[{"name":"BASSANELLO","start":"1871-01-15","end":"1927-01-11"},{"name":"BAS
SANELLO","start":"1927-01-12","end":"1949-09-07"},{"start":"1949-09-08","end
":"9999-12-31","name":"VASANELLO"}],"A824":[{"start":"1922-02-02","end":"192
3-02-07","name":"BERSEZ"},{"end":"1923-04-26","start":"1923-02-08","name":"B
ERSEZ"},{"start":"1923-04-27","end":"1924-03-08","name":"BERSEZIO DEL QUARNA
RO"},{"start":"1924-03-09","end":"1929-05-21","name":"BERSEZIO DEL QUARNARO"
}],"D330":[{"start":"1866-11-19","end":"1957-09-02","name":"DOMEGGE"},{"name
":"DOMEGGE DI CADORE","start":"1957-09-03","end":"9999-12-31"}],"G046":[{"en
d":"9999-12-31","start":"1861-03-17","name":"OLMEDO"}],"D470":[{"name":"FALC
ADE","start":"1866-11-19","end":"9999-12-31"}],"L777":[{"start":"1861-03-17"
,"end":"1928-01-25","name":"VEROLANUOVA"},{"start":"1928-01-26","end":"1948-
03-20","name":"VEROLANUOVA"},{"end":"9999-12-31","start":"1948-03-21","name"
:"VEROLANUOVA"}],"M289":[{"start":"1988-06-14","end":"9999-12-31","name":"MA
SSA DI SOMMA"}],"B484":[{"end":"9999-12-31","start":"1861-03-17","name":"CAM
ISANO"}],"H807":[{"start":"1864-01-29","end":"1992-04-15","name":"SAN COSTAN
TINO CALABRO"},{"end":"9999-12-31","start":"1992-04-16","name":"SAN COSTANTI
NO CALABRO"}],"G087":[{"name":"ORBASSANO","end":"9999-12-31","start":"1861-0
3-17"}],"L110":[{"end":"1923-02-13","start":"1920-10-16","name":"TERMENAGO"}
,{"start":"1923-02-14","end":"1928-09-08","name":"TERMENAGO"}],"G753":[{"end
":"9999-12-31","start":"1867-06-24","name":"POGGIO RUSCO"}],"M093":[{"name":
"VITULANO","start":"1861-03-17","end":"9999-12-31"}],"M399":[{"end":"9999-12
-31","start":"2018-02-01","name":"TREPPO LIGOSULLO"}],"C854":[{"name":"COLLE
 D'ANCHISE","end":"1928-05-12","start":"1861-03-17"},{"start":"1946-04-24","
end":"1970-03-02","name":"COLLE D'ANCHISE"},{"start":"1970-03-03","end":"999
9-12-31","name":"COLLE D'ANCHISE"}],"B971":[{"start":"1863-07-13","end":"999
9-12-31","name":"CASIRATE D'ADDA"}],"E420":[{"end":"1923-02-13","start":"192
0-10-16","name":"LAION"},{"end":"1927-01-11","start":"1923-02-14","name":"LA
ION"},{"name":"LAION","start":"1927-01-12","end":"1929-06-03"},{"start":"192
9-06-04","end":"9999-12-31","name":"LAION"}],"B001":[{"end":"1992-04-15","st
art":"1861-03-17","name":"BORGHI"},{"end":"9999-12-31","start":"1992-04-16",
"name":"BORGHI"}],"E354":[{"start":"1866-11-19","end":"1905-12-31","name":"I
SOLA DI MALO"},{"name":"ISOLA VICENTINA","end":"9999-12-31","start":"1906-01
-01"}],"C968":[{"name":"CONTESSA ENTELLINA","end":"9999-12-31","start":"1875
-07-14"}],"E613":[{"start":"1864-11-07","end":"9999-12-31","name":"LISCIANO 
NICCONE"}],"E957":[{"start":"1861-03-17","end":"2011-08-11","name":"MARINEO"
},{"start":"2011-08-12","end":"9999-12-31","name":"MARINEO"}],"B875":[{"name
":"CASALE LITTA","end":"1927-01-11","start":"1869-03-11"},{"start":"1927-01-
12","end":"1958-05-27","name":"CASALE LITTA"},{"name":"CASALE LITTA","start"
:"1958-05-28","end":"9999-12-31"}],"L283":[{"start":"1861-03-17","end":"1928
-04-06","name":"TORRE UZZONE"}],"M104":[{"end":"1928-06-13","start":"1861-03
-17","name":"VOBARNO"},{"name":"VOBARNO","start":"1928-06-14","end":"9999-12
-31"}],"L535":[{"start":"1862-12-19","end":"9999-12-31","name":"VAIANO CREMA
SCO"}],"G942":[{"end":"1928-03-12","start":"1861-03-17","name":"POTENZA"},{"
start":"1928-03-13","end":"1935-03-04","name":"POTENZA"},{"name":"POTENZA","
start":"1935-03-05","end":"9999-12-31"}],"L710":[{"end":"9999-12-31","start"
:"1866-11-19","name":"VEGGIANO"}],"I208":[{"end":"1927-05-17","start":"1865-
10-25","name":"SANT'AGNELLO"},{"name":"SANT'AGNELLO","start":"1946-04-24","e
nd":"9999-12-31"}],"B655":[{"end":"1992-04-15","start":"1861-03-17","name":"
CAPISTRANO"},{"end":"9999-12-31","start":"1992-04-16","name":"CAPISTRANO"}],
"D417":[{"start":"1906-11-13","end":"1928-01-25","name":"ERBA INCINO"}],"H31
9":[{"end":"9999-12-31","start":"1892-06-20","name":"RIPARBELLA"}],"H013":[{
"end":"1911-08-02","start":"1861-03-17","name":"PRECACORE"},{"name":"SAMO","
start":"1911-08-03","end":"1928-07-27"},{"name":"SAMO","start":"1946-03-09",
"end":"9999-12-31"}],"G153":[{"name":"OSCHIRI","start":"1861-03-17","end":"2
005-12-31"},{"start":"2006-01-01","end":"2016-04-27","name":"OSCHIRI"},{"nam
e":"OSCHIRI","start":"2016-04-28","end":"9999-12-31"}],"I555":[{"start":"186
1-03-17","end":"1935-04-07","name":"SCURZOLENGO"},{"end":"9999-12-31","start
":"1935-04-08","name":"SCURZOLENGO"}],"F831":[{"name":"MUTIGNANO","end":"192
7-12-21","start":"1861-03-17"},{"name":"PINETO","end":"1933-01-24","start":"
1929-05-21"},{"name":"PINETO","end":"9999-12-31","start":"1933-01-25"}],"C57
9":[{"name":"CESNOLA","start":"1861-03-17","end":"1927-01-11"},{"name":"CESN
OLA","start":"1927-01-12","end":"1929-05-04"}],"H553":[{"name":"ROPPOLO","st
art":"1861-03-17","end":"1927-01-11"},{"end":"1992-04-15","start":"1927-01-1
2","name":"ROPPOLO"},{"name":"ROPPOLO","end":"9999-12-31","start":"1992-04-1
6"}],"A088":[{"start":"1863-04-23","end":"9999-12-31","name":"AGRATE CONTURB
IA"}],"C674":[{"end":"9999-12-31","start":"1861-03-17","name":"CICALA"}],"M4
08":[{"end":"9999-12-31","start":"2019-01-01","name":"BARBERINO TAVARNELLE"}
],"L692":[{"name":"VAS","start":"1866-11-19","end":"2013-12-27"}],"D898":[{"
end":"9999-12-31","start":"1861-03-17","name":"GAMBERALE"}],"F846":[{"end":"
9999-12-31","start":"1861-03-17","name":"NARZOLE"}],"L844":[{"start":"1861-0
3-17","end":"1927-01-11","name":"VICO DI PANTANO"},{"name":"VICO DI PANTANO"
,"end":"1927-03-09","start":"1927-01-12"},{"name":"VILLA LITERNO","end":"194
5-08-31","start":"1927-03-10"},{"name":"VILLA LITERNO","end":"9999-12-31","s
tart":"1945-09-01"}],"F935":[{"name":"NORCIA","start":"1861-03-17","end":"99
99-12-31"}],"E500":[{"start":"1920-10-16","end":"1923-02-13","name":"LAVIS"}
,{"name":"LAVIS","end":"1958-06-17","start":"1923-02-14"},{"name":"LAVIS","e
nd":"9999-12-31","start":"1958-06-18"}],"I864":[{"start":"1861-03-17","end":
"9999-12-31","name":"SORTINO"}],"D241":[{"name":"CUZZAGO","start":"1861-03-1
7","end":"1928-07-28"}],"E312":[{"name":"INVERSO PORTE","end":"1928-05-30","
start":"1861-03-17"}],"B788":[{"end":"1928-04-17","start":"1866-11-19","name
":"CARLINO"},{"end":"1968-04-05","start":"1946-12-24","name":"CARLINO"},{"en
d":"9999-12-31","start":"1968-04-06","name":"CARLINO"}],"A630":[{"name":"BAR
BARIGA","end":"1927-12-23","start":"1861-03-17"},{"name":"BARBARIGA","end":"
9999-12-31","start":"1927-12-24"}],"A101":[{"end":"1927-08-05","start":"1864
-01-16","name":"AIELLO DEL SABATO"},{"name":"AIELLO DEL SABATO","start":"192
7-08-06","end":"1950-02-10"},{"end":"9999-12-31","start":"1950-02-11","name"
:"AIELLO DEL SABATO"}],"F887":[{"name":"NIBIONNO","end":"1992-04-15","start"
:"1861-03-17"},{"end":"9999-12-31","start":"1992-04-16","name":"NIBIONNO"}],
"F538":[{"start":"1862-10-24","end":"1929-02-23","name":"MONTELEONE DI PUGLI
A"},{"start":"1929-02-24","end":"9999-12-31","name":"MONTELEONE DI PUGLIA"}]
,"G460":[{"name":"PERNO","end":"1930-12-31","start":"1861-03-17"}],"C812":[{
"name":"CODEVIGO","end":"9999-12-31","start":"1866-11-19"}],"B869":[{"name":
"CASALBUTTANO ED UNITI","end":"9999-12-31","start":"1867-09-06"}],"L938":[{"
end":"1958-09-04","start":"1929-05-14","name":"VILLA D'OGNA"},{"end":"9999-1
2-31","start":"1958-09-05","name":"VILLA D'OGNA"}],"L177":[{"start":"1861-03
-17","end":"9999-12-31","name":"TIRIOLO"}],"D236":[{"end":"1992-04-15","star
t":"1861-03-17","name":"CUTRO"},{"name":"CUTRO","end":"9999-12-31","start":"
1992-04-16"}],"I958":[{"start":"1920-10-16","end":"1923-02-13","name":"STILV
ES"},{"start":"1923-02-14","end":"1927-01-11","name":"STILVES"},{"name":"STI
LVES","start":"1927-01-12","end":"1928-12-09"}],"B262":[{"name":"BULGAROGRAS
SO","end":"1927-11-01","start":"1861-03-17"},{"end":"9999-12-31","start":"19
50-04-05","name":"BULGAROGRASSO"}],"C219":[{"end":"1957-11-04","start":"1871
-05-01","name":"CASTELNOVO NE' MONTI"},{"name":"CASTELNOVO NE' MONTI","start
":"1957-11-05","end":"9999-12-31"}],"D591":[{"end":"1927-01-11","start":"187
1-01-15","name":"FILETTINO"},{"start":"1927-01-12","end":"1938-11-26","name"
:"FILETTINO"},{"name":"FILETTINO","end":"9999-12-31","start":"1945-08-17"},{
"name":"FILETTINO GRAZIANI","start":"1938-11-27","end":"1945-08-16"}],"C767"
:[{"end":"1914-10-29","start":"1871-01-15","name":"CIVITA LAVINIA"},{"name":
"LANUVIO","start":"1914-10-30","end":"1935-03-06"},{"start":"1935-03-07","en
d":"9999-12-31","name":"LANUVIO"}],"C634":[{"name":"CHIEVE","start":"1861-03
-17","end":"9999-12-31"}],"L181":[{"start":"1861-03-17","end":"9999-12-31","
name":"TITO"}],"A742":[{"start":"1861-03-17","end":"1928-04-15","name":"BELG
IRATE"},{"start":"1947-12-23","end":"1992-05-22","name":"BELGIRATE"},{"name"
:"BELGIRATE","start":"1992-05-23","end":"9999-12-31"}],"F218":[{"start":"186
6-11-19","end":"9999-12-31","name":"MINERBE"}],"D317":[{"start":"1866-11-19"
,"end":"9999-12-31","name":"DOLC\u00c8"}],"D056":[{"name":"CORTE DE' CORTESI
 CON CIGNONE","end":"9999-12-31","start":"1868-01-01"}],"H419":[{"start":"18
71-01-15","end":"1927-01-11","name":"ROCCALVECCE"},{"name":"ROCCALVECCE","st
art":"1927-01-12","end":"1928-05-19"}],"E226":[{"end":"9999-12-31","start":"
1866-11-19","name":"GRUMOLO DELLE ABBADESSE"}],"E758":[{"start":"1878-02-18"
,"end":"1928-03-26","name":"LUSERNA SAN GIOVANNI"},{"end":"1947-01-08","star
t":"1928-03-27","name":"LUSERNA SAN GIOVANNI"},{"end":"9999-12-31","start":"
1947-01-09","name":"LUSERNA SAN GIOVANNI"}],"Z335":[{"start":"0000-01-01","e
nd":"9999-12-31","name":"Nigeria"}],"F975":[{"name":"NUGHEDU DI SAN NICOL\u0
0d2","end":"1996-08-12","start":"1861-03-17"},{"start":"1996-08-13","end":"9
999-12-31","name":"NUGHEDU SAN NICOL\u00d2"}],"F005":[{"end":"9999-12-31","s
tart":"1861-03-17","name":"MASCALUCIA"}],"L575":[{"name":"VALFLORIANA","end"
:"1923-02-13","start":"1920-10-16"},{"end":"9999-12-31","start":"1923-02-14"
,"name":"VALFLORIANA"}],"C539":[{"end":"9999-12-31","start":"1861-03-17","na
me":"CERSOSIMO"}],"M308":[{"name":"CAVALLINO-TREPORTI","end":"9999-12-31","s
tart":"1999-04-02"}],"G618":[{"start":"1861-03-17","end":"9999-12-31","name"
:"PIETRALUNGA"}],"D516":[{"end":"1923-02-13","start":"1920-10-16","name":"FA
VER"},{"end":"1928-09-05","start":"1923-02-14","name":"FAVER"},{"name":"FAVE
R","end":"2015-12-31","start":"1952-05-17"}],"F871":[{"name":"NEROLA","end":
"9999-12-31","start":"1871-01-15"}],"E685":[{"name":"LOREGLIA","end":"1928-0
1-26","start":"1861-03-17"},{"name":"LOREGLIA","end":"1992-05-22","start":"1
955-06-09"},{"name":"LOREGLIA","start":"1992-05-23","end":"9999-12-31"}],"L9
78":[{"start":"1863-04-23","end":"1927-01-11","name":"VILLANOVA BIELLESE"},{
"end":"1992-04-15","start":"1927-01-12","name":"VILLANOVA BIELLESE"},{"start
":"1992-04-16","end":"9999-12-31","name":"VILLANOVA BIELLESE"}],"L137":[{"na
me":"TERRES","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14",
"end":"1928-10-16","name":"TERRES"},{"name":"TERRES","end":"2015-12-31","sta
rt":"1947-03-30"}],"L215":[{"name":"TORELLA DEL SANNIO","start":"1863-01-19"
,"end":"1970-03-02"},{"end":"9999-12-31","start":"1970-03-03","name":"TORELL
A DEL SANNIO"}],"E412":[{"name":"LAGUNDO","end":"1923-02-13","start":"1920-1
0-16"},{"end":"1927-01-11","start":"1923-02-14","name":"LAGUNDO"},{"name":"L
AGUNDO","start":"1927-01-12","end":"1958-04-15"},{"name":"LAGUNDO","start":"
1958-04-16","end":"1974-09-17"},{"end":"9999-12-31","start":"1974-09-18","na
me":"LAGUNDO"}],"I157":[{"name":"SAN SEVERINO LUCANO","end":"9999-12-31","st
art":"1863-04-10"}],"H826":[{"name":"SAN DONATO DI LECCE","start":"1863-02-0
6","end":"9999-12-31"}],"D061":[{"name":"CORTEMAGGIORE","start":"1861-03-17"
,"end":"9999-12-31"}],"D276":[{"end":"1923-02-13","start":"1920-10-16","name
":"DERMULO"},{"name":"DERMULO","start":"1923-02-14","end":"1929-01-23"}],"D7
43":[{"end":"1939-07-13","start":"1883-06-21","name":"FOSSARMATO"}],"L122":[
{"name":"TERRALBA","end":"1928-11-02","start":"1861-03-17"},{"end":"1931-02-
19","start":"1928-11-03","name":"TERRALBA"},{"end":"1947-01-23","start":"193
1-02-20","name":"TERRALBA"},{"name":"TERRALBA","end":"1948-05-11","start":"1
947-01-24"},{"end":"1974-08-19","start":"1948-05-12","name":"TERRALBA"},{"st
art":"1974-08-20","end":"2011-01-20","name":"TERRALBA"},{"name":"TERRALBA","
end":"9999-12-31","start":"2011-01-21"}],"B042":[{"name":"BORGO TARO","start
":"1884-04-25","end":"1928-06-27"},{"name":"BORGO VAL DI TARO","end":"1946-0
2-26","start":"1928-06-28"},{"name":"BORGO VAL DI TARO","start":"1946-02-27"
,"end":"9999-12-31"}],"F578":[{"start":"1861-03-17","end":"9999-12-31","name
":"MONTENERODOMO"}],"L008":[{"name":"SUPERSANO","end":"1963-03-29","start":"
1861-03-17"},{"name":"SUPERSANO","end":"9999-12-31","start":"1963-03-30"}],"
A670":[{"start":"1861-03-17","end":"1927-12-26","name":"BARNI"},{"end":"9999
-12-31","start":"1950-04-05","name":"BARNI"}],"F969":[{"name":"NOVILLARA","e
nd":"1929-04-10","start":"1861-03-17"}],"L664":[{"start":"1968-07-23","end":
"9999-12-31","name":"VANZAGHELLO"}],"D430":[{"end":"1927-01-11","start":"186
1-03-17","name":"ESCALAPLANO"},{"name":"ESCALAPLANO","end":"2005-12-31","sta
rt":"1927-01-12"},{"end":"2016-04-27","start":"2006-01-01","name":"ESCALAPLA
NO"},{"start":"2016-04-28","end":"9999-12-31","name":"ESCALAPLANO"}],"L737":
[{"name":"VENIANO","end":"1927-11-01","start":"1861-03-17"},{"end":"9999-12-
31","start":"1950-04-05","name":"VENIANO"}],"F666":[{"start":"1861-03-17","e
nd":"1929-03-25","name":"MONTEZEMOLO"},{"start":"1947-03-30","end":"1957-07-
19","name":"MONTEZEMOLO"},{"name":"MONTEZEMOLO","start":"1957-07-20","end":"
9999-12-31"}],"I083":[{"start":"1863-04-10","end":"1928-05-03","name":"SAN P
IER FEDELE"}],"I389":[{"end":"1998-12-08","start":"1864-06-23","name":"SAN V
INCENZO VALLE ROVETO"},{"name":"SAN VINCENZO VALLE ROVETO","end":"9999-12-31
","start":"1998-12-09"}],"L493":[{"name":"UNCHIO","start":"1861-03-17","end"
:"1929-05-24"}],"A850":[{"name":"BIBBIANO","end":"9999-12-31","start":"1861-
03-17"}],"I757":[{"name":"SIRIS","end":"1928-03-13","start":"1861-03-17"},{"
name":"SIRIS","start":"1962-05-22","end":"1974-08-19"},{"end":"9999-12-31","
start":"1974-08-20","name":"SIRIS"}],"G901":[{"end":"1928-06-29","start":"18
61-03-17","name":"PORTESE"}],"C923":[{"name":"COMERO","start":"1861-03-17","
end":"1928-04-02"}],"D143":[{"name":"CREMELLA","start":"1861-03-17","end":"1
928-11-21"},{"start":"1953-12-06","end":"1992-04-15","name":"CREMELLA"},{"st
art":"1992-04-16","end":"9999-12-31","name":"CREMELLA"}],"L722":[{"end":"999
9-12-31","start":"1893-04-10","name":"VELO VERONESE"}],"Z243":[{"start":"000
0-01-01","end":"9999-12-31","name":"Turchia"}],"B384":[{"name":"CALATABIANO"
,"end":"9999-12-31","start":"1861-03-17"}],"A203":[{"end":"9999-12-31","star
t":"1861-03-17","name":"ALIMINUSA"}],"G071":[{"end":"1927-01-11","start":"18
61-03-17","name":"ONIFERI"},{"end":"9999-12-31","start":"1927-01-12","name":
"ONIFERI"}],"G266":[{"name":"PALAZZOLO VERCELLESE","end":"1927-01-11","start
":"1862-12-29"},{"start":"1927-01-12","end":"9999-12-31","name":"PALAZZOLO V
ERCELLESE"}],"L569":[{"start":"1871-01-15","end":"1927-01-11","name":"VALENT
ANO"},{"name":"VALENTANO","start":"1927-01-12","end":"9999-12-31"}],"D370":[
{"name":"DRIZZONA","end":"2018-12-31","start":"1868-01-01"}],"D952":[{"name"
:"GAZZANIGA","end":"1927-06-01","start":"1861-03-17"},{"end":"1948-12-20","s
tart":"1927-06-02","name":"GAZZANIGA"},{"start":"1948-12-21","end":"9999-12-
31","name":"GAZZANIGA"}],"G805":[{"name":"POMARETTO","start":"1861-03-17","e
nd":"1928-05-30"},{"start":"1954-10-17","end":"9999-12-31","name":"POMARETTO
"}],"H085":[{"name":"PRUNETTO","end":"1928-06-27","start":"1861-03-17"},{"st
art":"1946-07-03","end":"9999-12-31","name":"PRUNETTO"}],"B987":[{"end":"192
7-01-11","start":"1902-01-17","name":"CASORATE SEMPIONE"},{"end":"9999-12-31
","start":"1927-01-12","name":"CASORATE SEMPIONE"}],"Z102":[{"name":"Austria
","start":"0000-01-01","end":"9999-12-31"}],"D202":[{"name":"CUMIANA","start
":"1861-03-17","end":"1928-05-22"},{"start":"1928-05-23","end":"9999-12-31",
"name":"CUMIANA"}],"C167":[{"name":"CASTELLETTO UZZONE","end":"1929-03-23","
start":"1861-03-17"},{"name":"CASTELLETTO UZZONE","start":"1929-03-24","end"
:"9999-12-31"}],"C820":[{"start":"1919-08-04","end":"2009-06-29","name":"COG
LIATE"},{"start":"2009-06-30","end":"9999-12-31","name":"COGLIATE"}],"F651":
[{"end":"1928-04-30","start":"1861-03-17","name":"MONTEU DA PO"},{"end":"999
9-12-31","start":"1946-12-19","name":"MONTEU DA PO"}],"A142":[{"start":"1861
-03-17","end":"1943-08-12","name":"ALBATE"}],"L781":[{"start":"1866-11-19","
end":"1927-02-15","name":"VERONA"},{"name":"VERONA","start":"1927-02-16","en
d":"1927-05-11"},{"name":"VERONA","end":"1927-10-06","start":"1927-05-12"},{
"name":"VERONA","start":"1927-10-07","end":"1932-08-18"},{"start":"1932-08-1
9","end":"1933-09-01","name":"VERONA"},{"name":"VERONA","end":"1975-01-28","
start":"1933-09-02"},{"end":"9999-12-31","start":"1975-01-29","name":"VERONA
"}],"I549":[{"name":"SCORRANO","end":"9999-12-31","start":"1861-03-17"}],"B2
94":[{"name":"BUSSI SUL TIRINO","start":"1889-11-09","end":"1927-01-11"},{"s
tart":"1927-01-12","end":"9999-12-31","name":"BUSSI SUL TIRINO"}],"B946":[{"
end":"9999-12-31","start":"1946-07-30","name":"CASAVATORE"}],"B835":[{"end":
"1911-04-17","start":"1866-11-19","name":"CARR\u00c8"},{"name":"CARR\u00c8",
"end":"9999-12-31","start":"1911-04-18"}],"B649":[{"start":"1933-07-20","end
":"1935-03-06","name":"CAPENA"},{"end":"9999-12-31","start":"1935-03-07","na
me":"CAPENA"},{"start":"1871-01-15","end":"1933-07-19","name":"LEPRIGNANO"}]
,"A617":[{"start":"1873-06-23","end":"1938-11-12","name":"BARANO D'ISCHIA"},
{"name":"BARANO D'ISCHIA","start":"1945-09-22","end":"9999-12-31"}],"E158":[
{"name":"GRAZZANISE","end":"1907-06-30","start":"1861-03-17"},{"name":"GRAZZ
ANISE","start":"1907-07-01","end":"1927-01-11"},{"end":"1945-08-31","start":
"1927-01-12","name":"GRAZZANISE"},{"name":"GRAZZANISE","start":"1945-09-01",
"end":"9999-12-31"}],"D809":[{"name":"FRONTONE-SERRA","start":"1928-09-25","
end":"1946-02-05"}],"A953":[{"name":"BOLZANO NOVARESE","end":"1928-02-22","s
tart":"1861-03-17"},{"name":"BOLZANO NOVARESE","end":"9999-12-31","start":"1
947-05-23"}],"G251":[{"name":"PALAGIANELLO","end":"1923-12-21","start":"1907
-07-02"},{"name":"PALAGIANELLO","start":"1923-12-22","end":"9999-12-31"}],"I
299":[{"name":"SANT'ANTONIO A TREBBIA","start":"1862-11-28","end":"1923-07-2
2"}],"B544":[{"name":"CAMPOLIETO","start":"1861-03-17","end":"1970-03-02"},{
"start":"1970-03-03","end":"9999-12-31","name":"CAMPOLIETO"}],"E454":[{"end"
:"1929-05-13","start":"1861-03-17","name":"LARDIRAGO"},{"name":"LARDIRAGO","
end":"1947-10-01","start":"1929-05-14"},{"name":"LARDIRAGO","end":"1947-11-2
7","start":"1947-10-02"},{"name":"LARDIRAGO","start":"1947-11-28","end":"196
3-06-20"},{"start":"1963-06-21","end":"9999-12-31","name":"LARDIRAGO"}],"I64
4":[{"end":"9999-12-31","start":"1861-03-17","name":"SERRADIFALCO"}],"E320":
[{"name":"JOLANDA DI SAVOIA","start":"1911-04-20","end":"9999-12-31"},{"name
":"LE VENEZIE","end":"1911-04-19","start":"1909-01-23"}],"I892":[{"name":"SP
ERLONGA","end":"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end
":"1934-10-25","name":"SPERLONGA"},{"name":"SPERLONGA","start":"1934-10-26",
"end":"1934-12-17"},{"name":"SPERLONGA","end":"1945-04-08","start":"1934-12-
18"},{"name":"SPERLONGA","end":"9999-12-31","start":"1945-04-09"}],"D720":[{
"name":"FORNI DI SOTTO","end":"1968-04-05","start":"1866-11-19"},{"start":"1
968-04-06","end":"9999-12-31","name":"FORNI DI SOTTO"}],"L427":[{"name":"TRI
NIT\u00c0","end":"9999-12-31","start":"1861-03-17"}],"I930":[{"name":"SQUINZ
ANO","end":"1972-05-08","start":"1861-03-17"},{"name":"SQUINZANO","start":"1
972-05-09","end":"9999-12-31"}],"Z527":[{"end":"9999-12-31","start":"0000-01
-01","name":"Santa Lucia"}],"C394":[{"end":"1938-02-18","start":"1869-02-01"
,"name":"CAVENAGO D'ADDA"},{"start":"1938-02-19","end":"1992-04-15","name":"
CAVENAGO D'ADDA"},{"name":"CAVENAGO D'ADDA","start":"1992-04-16","end":"9999
-12-31"}],"F010":[{"name":"MASERA","start":"1861-03-17","end":"1992-05-22"},
{"name":"MASERA","end":"9999-12-31","start":"1992-05-23"}],"H987":[{"end":"1
935-04-07","start":"1898-04-25","name":"SAN MARTINO ALFIERI"},{"name":"SAN M
ARTINO ALFIERI","start":"1935-04-08","end":"9999-12-31"}],"B085":[{"name":"B
OTRICELLO","start":"1957-01-17","end":"9999-12-31"}],"C997":[{"end":"1928-01
-26","start":"1864-03-11","name":"CORENNO PLINIO"}],"H384":[{"name":"ROCCABI
ANCA","start":"1861-03-17","end":"9999-12-31"}],"C709":[{"name":"CINO","end"
:"9999-12-31","start":"1861-03-17"}],"L950":[{"name":"VILLAFRANCA TIRRENA","
start":"1929-06-02","end":"1952-12-13"},{"name":"VILLAFRANCA TIRRENA","start
":"1952-12-14","end":"9999-12-31"}],"L372":[{"name":"TREMOSINE","start":"186
1-03-17","end":"1929-01-04"},{"start":"1929-01-05","end":"2013-11-11","name"
:"TREMOSINE"},{"end":"9999-12-31","start":"2013-11-12","name":"TREMOSINE SUL
 GARDA"}],"E235":[{"end":"9999-12-31","start":"1861-03-17","name":"GUANZATE"
}],"Z532":[{"name":"Antigua e Barbuda","start":"0000-01-01","end":"9999-12-3
1"}],"F550":[{"end":"9999-12-31","start":"1862-12-19","name":"MONTELUPO ALBE
SE"}],"L432":[{"name":"TRISOBBIO","end":"9999-12-31","start":"1861-03-17"}],
"A658":[{"start":"1861-03-17","end":"9999-12-31","name":"BARGAGLI"}],"Z326":
[{"start":"0000-01-01","end":"9999-12-31","name":"Libia"}],"E117":[{"end":"1
928-05-11","start":"1861-03-17","name":"GOTTRO"}],"I452":[{"name":"SASSARI",
"start":"1861-03-17","end":"1988-08-05"},{"name":"SASSARI","end":"9999-12-31
","start":"1988-08-06"}],"E642":[{"name":"LOCAVIZZA DI CANALE","start":"1921
-01-05","end":"1923-02-07"},{"name":"LOCAVIZZA DI CANALE","end":"1927-01-11"
,"start":"1923-02-08"},{"name":"LOCAVIZZA DI CANALE","end":"1928-01-25","sta
rt":"1927-01-12"}],"E894":[{"name":"MANTA","end":"9999-12-31","start":"1861-
03-17"}],"G877":[{"end":"9999-12-31","start":"1861-03-17","name":"PONZONE"}]
,"H931":[{"name":"SAN GIUSEPPE VESUVIANO","end":"9999-12-31","start":"1895-0
1-07"}],"A490":[{"end":"1927-01-11","start":"1861-03-17","name":"ATTIGLIANO"
},{"start":"1927-01-12","end":"9999-12-31","name":"ATTIGLIANO"}],"E770":[{"s
tart":"1861-03-17","end":"1928-06-13","name":"LUZZANA"},{"name":"LUZZANA","e
nd":"9999-12-31","start":"1947-11-26"}],"G448":[{"start":"1870-07-01","end":
"1928-01-29","name":"PEREGO"},{"start":"1953-09-13","end":"1992-04-15","name
":"PEREGO"},{"start":"1992-04-16","end":"2015-01-29","name":"PEREGO"}],"F199
":[{"end":"2013-12-31","start":"1963-07-31","name":"MIGLIARO"}],"H544":[{"na
me":"RONCOLA","start":"1861-03-17","end":"9999-12-31"}],"I833":[{"start":"19
21-01-05","end":"1923-02-07","name":"SONZIA"},{"name":"SONZIA","end":"1927-0
1-11","start":"1923-02-08"},{"name":"SONZIA","end":"1928-02-20","start":"192
7-01-12"},{"name":"SONZIA","start":"1928-02-21","end":"1947-09-15"}],"M072":
[{"end":"1927-01-11","start":"1861-03-17","name":"VISCIANO"},{"start":"1927-
01-12","end":"9999-12-31","name":"VISCIANO"}],"H649":[{"end":"1927-08-19","s
tart":"1864-02-02","name":"SABBIO BERGAMASCO"}],"C284":[{"name":"CASTELVETER
E IN VAL FORTORE","start":"1863-02-06","end":"9999-12-31"}],"H835":[{"name":
"SAN FELICE SUL PANARO","start":"1862-12-29","end":"9999-12-31"}],"H946":[{"
name":"SAN LAZZARO REALE","start":"1862-11-10","end":"1928-12-07"}],"L200":[
{"start":"1928-12-22","end":"9999-12-31","name":"TON"}],"I718":[{"name":"SIA
MANNA","start":"1974-09-13","end":"9999-12-31"}],"G144":[{"name":"ORTOVERO",
"end":"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"9999-1
2-31","name":"ORTOVERO"}],"C628":[{"name":"CHIESA","start":"1861-03-17","end
":"1940-07-23"},{"end":"9999-12-31","start":"1940-07-24","name":"CHIESA IN V
ALMALENCO"}],"A785":[{"end":"9999-12-31","start":"1885-12-25","name":"BENTIV
OGLIO"}],"H294":[{"name":"RIMINI","end":"1922-12-01","start":"1861-03-17"},{
"name":"RIMINI","end":"1956-03-03","start":"1922-12-02"},{"name":"RIMINI","e
nd":"1992-04-15","start":"1956-03-04"},{"start":"1992-04-16","end":"9999-12-
31","name":"RIMINI"}],"E170":[{"end":"9999-12-31","start":"1861-03-17","name
":"GREZZAGO"}],"I414":[{"start":"1866-11-19","end":"9999-12-31","name":"SAN 
ZENO DI MONTAGNA"}],"L306":[{"start":"1861-03-17","end":"9999-12-31","name":
"TORTORELLA"}],"F304":[{"end":"1992-04-15","start":"1861-03-17","name":"MOLT
ENO"},{"start":"1992-04-16","end":"9999-12-31","name":"MOLTENO"}],"A571":[{"
name":"BAGNOLO PIEMONTE","end":"9999-12-31","start":"1862-12-19"}],"F799":[{
"name":"MUGNANO DI NAPOLI","start":"1863-02-06","end":"9999-12-31"}],"F907":
[{"end":"9999-12-31","start":"1861-03-17","name":"NOCARA"}],"G810":[{"end":"
1938-02-09","start":"1936-10-22","name":"POMETO"},{"name":"POMETO","start":"
1938-02-10","end":"1939-02-04"},{"name":"POMETO","start":"1939-02-05","end":
"1947-11-26"}],"E343":[{"start":"1863-07-13","end":"9999-12-31","name":"ISOL
A DEL GRAN SASSO D'ITALIA"}],"E049":[{"end":"1927-01-11","start":"1861-03-17
","name":"GIRASOLE"},{"name":"GIRASOLE","end":"1928-05-12","start":"1927-01-
12"},{"start":"1946-03-25","end":"2005-12-31","name":"GIRASOLE"},{"end":"201
6-04-27","start":"2006-01-01","name":"GIRASOLE"},{"name":"GIRASOLE","start":
"2016-04-28","end":"9999-12-31"}],"B419":[{"name":"CALLIANO","start":"1920-1
0-16","end":"1923-02-13"},{"end":"1929-03-25","start":"1923-02-14","name":"C
ALLIANO"},{"end":"9999-12-31","start":"1947-07-09","name":"CALLIANO"}],"C462
":[{"name":"CENESI","end":"1927-01-11","start":"1861-03-17"},{"name":"CENESI
","start":"1927-01-12","end":"1929-06-03"}],"L507":[{"name":"USCIO","start":
"1877-11-03","end":"9999-12-31"}],"G744":[{"start":"1923-04-27","end":"1927-
01-11","name":"POCRAI DEL PIRO"},{"end":"1928-01-25","start":"1927-01-12","n
ame":"POCRAI DEL PIRO"},{"name":"PODICRAI","end":"1923-02-07","start":"1921-
01-05"},{"end":"1923-04-26","start":"1923-02-08","name":"PODICRAI"}],"I118":
[{"start":"1861-03-17","end":"1927-09-15","name":"SAN PIETRO PULA"},{"name":
"SAN PIETRO PULA","start":"1948-05-07","end":"1950-01-28"},{"start":"1950-01
-29","end":"1974-08-19","name":"VILLA SAN PIETRO"},{"end":"9999-12-31","star
t":"1974-08-20","name":"VILLA SAN PIETRO"}],"M214":[{"name":"BADESI","start"
:"1969-04-15","end":"2005-12-31"},{"start":"2006-01-01","end":"2016-04-27","
name":"BADESI"},{"name":"BADESI","end":"9999-12-31","start":"2016-04-28"}],"
A447":[{"end":"1932-09-24","start":"1866-11-19","name":"ARTA"},{"name":"ARTA
","start":"1932-09-25","end":"1946-11-14"},{"name":"ARTA","end":"1959-03-03"
,"start":"1946-11-15"},{"name":"ARTA","end":"1965-08-14","start":"1959-03-04
"},{"name":"ARTA TERME","start":"1965-08-15","end":"1968-04-05"},{"end":"999
9-12-31","start":"1968-04-06","name":"ARTA TERME"}],"F077":[{"name":"MECHEL"
,"end":"1923-02-13","start":"1920-10-16"},{"end":"1928-09-08","start":"1923-
02-14","name":"MECHEL"}],"L673":[{"name":"VARAPODIO","end":"9999-12-31","sta
rt":"1861-03-17"}],"A185":[{"start":"1873-07-16","end":"1928-08-23","name":"
ALEZIO"},{"start":"1928-08-24","end":"9999-12-31","name":"ALEZIO"}],"A486":[
{"name":"ATINA","end":"1927-01-11","start":"1861-03-17"},{"name":"ATINA","st
art":"1927-01-12","end":"9999-12-31"}],"D120":[{"name":"COSTIGLIOLE SALUZZO"
,"start":"1861-03-17","end":"9999-12-31"}],"Z220":[{"start":"0000-01-01","en
d":"9999-12-31","name":"Giordania"}],"H042":[{"end":"1927-01-11","start":"18
61-03-17","name":"PR\u00c9 SAINT DIDIER"},{"name":"PR\u00c9 SAINT DIDIER","e
nd":"1939-10-20","start":"1927-01-12"},{"start":"1946-01-30","end":"1977-01-
01","name":"PR\u00c9 SAINT DIDIER"},{"end":"9999-12-31","start":"1977-01-02"
,"name":"PR\u00c9-SAINT-DIDIER"},{"end":"1946-01-29","start":"1939-10-21","n
ame":"SAN DESIDERIO TERME"}],"M006":[{"name":"VILLA RENDENA","end":"1923-02-
13","start":"1920-10-16"},{"name":"VILLA RENDENA","start":"1923-02-14","end"
:"1928-06-27"},{"start":"1928-06-28","end":"2015-12-31","name":"VILLA RENDEN
A"}],"G913":[{"name":"PORTOFINO","end":"9999-12-31","start":"1861-03-17"}],"
D867":[{"name":"GALEATA","end":"1923-03-18","start":"1861-03-17"},{"name":"G
ALEATA","end":"1925-10-31","start":"1923-03-19"},{"end":"1927-03-15","start"
:"1925-11-01","name":"GALEATA"},{"end":"1929-06-17","start":"1927-03-16","na
me":"GALEATA"},{"start":"1929-06-18","end":"1933-07-26","name":"GALEATA"},{"
name":"GALEATA","end":"1992-04-15","start":"1933-07-27"},{"end":"9999-12-31"
,"start":"1992-04-16","name":"GALEATA"}],"G657":[{"end":"9999-12-31","start"
:"1861-03-17","name":"PIEVE TORINA"}],"E583":[{"name":"LIGNANA","end":"1927-
01-11","start":"1861-03-17"},{"name":"LIGNANA","end":"9999-12-31","start":"1
927-01-12"}],"A878":[{"name":"BIONE","start":"1861-03-17","end":"9999-12-31"
}],"D358":[{"start":"1868-08-18","end":"9999-12-31","name":"DOVERA"}],"F257"
:[{"start":"1861-03-17","end":"9999-12-31","name":"MODENA"}],"B826":[{"name"
:"CARPINETO SINELLO","start":"1863-02-06","end":"9999-12-31"}],"E717":[{"sta
rt":"1861-03-17","end":"1928-06-26","name":"LUCERNATE"}],"A211":[{"end":"201
7-12-31","start":"1861-03-17","name":"ALLUVIONI CAMBI\u00d2"}],"C940":[{"nam
e":"CONCA DEI MARINI","end":"1928-10-29","start":"1863-02-06"},{"end":"1947-
10-25","start":"1928-10-30","name":"CONCA DEI MARINI"},{"name":"CONCA DEI MA
RINI","start":"1947-10-26","end":"9999-12-31"}],"E443":[{"name":"LANZADA","e
nd":"9999-12-31","start":"1861-03-17"}],"F037":[{"end":"1927-01-11","start":
"1861-03-17","name":"MASSAZZA"},{"name":"MASSAZZA","start":"1927-01-12","end
":"1992-04-15"},{"name":"MASSAZZA","start":"1992-04-16","end":"9999-12-31"}]
,"E862":[{"name":"MALLES VENOSTA","end":"9999-12-31","start":"1928-01-26"}],
"I917":[{"name":"SPINOSO","start":"1861-03-17","end":"9999-12-31"}],"I653":[
{"end":"9999-12-31","start":"1861-03-17","name":"SERRA SAN QUIRICO"}],"Z506"
:[{"name":"El Salvador","end":"9999-12-31","start":"0000-01-01"}],"D690":[{"
start":"1895-01-07","end":"1928-12-04","name":"SAN GIOVANNI TEATINO"},{"star
t":"1928-12-05","end":"9999-12-31","name":"SAN GIOVANNI TEATINO"}],"I314":[{
"start":"1934-04-07","end":"1951-05-05","name":"SANTA VENERINA"},{"name":"SA
NTA VENERINA","start":"1951-05-06","end":"9999-12-31"}],"L406":[{"start":"18
67-06-02","end":"9999-12-31","name":"TREVISO BRESCIANO"}],"B553":[{"start":"
1894-08-03","end":"1927-06-01","name":"CAMPO NELL'ELBA"},{"name":"CAMPO NELL
'ELBA","end":"9999-12-31","start":"1927-06-02"}],"E269":[{"name":"GURRO","en
d":"1992-05-22","start":"1861-03-17"},{"name":"GURRO","start":"1992-05-23","
end":"9999-12-31"}],"F404":[{"name":"MONTALDO BORMIDA","end":"9999-12-31","s
tart":"1863-02-16"}],"L633":[{"start":"1861-03-17","end":"9999-12-31","name"
:"VALMACCA"}],"A546":[{"end":"9999-12-31","start":"1861-03-17","name":"BAGHE
RIA"}],"E130":[{"end":"1927-01-11","start":"1861-03-17","name":"GRAGLIA"},{"
start":"1927-01-12","end":"1992-04-15","name":"GRAGLIA"},{"start":"1992-04-1
6","end":"9999-12-31","name":"GRAGLIA"}],"F022":[{"end":"1940-02-07","start"
:"1863-05-06","name":"MASSA D'ALBE"},{"start":"1940-02-08","end":"1959-12-28
","name":"MASSA D'ALBE"},{"start":"1959-12-29","end":"1960-08-12","name":"MA
SSA D'ALBE"},{"start":"1960-08-13","end":"9999-12-31","name":"MASSA D'ALBE"}
],"D712":[{"start":"1861-03-17","end":"1927-01-11","name":"FORMIGLIANA"},{"n
ame":"FORMIGLIANA","start":"1927-01-12","end":"1956-11-09"},{"start":"1956-1
1-10","end":"9999-12-31","name":"FORMIGLIANA"}],"A944":[{"end":"1937-12-05",
"start":"1861-03-17","name":"BOLOGNA"},{"start":"1937-12-06","end":"9999-12-
31","name":"BOLOGNA"}],"E601":[{"name":"LINATE AL LAMBRO","start":"1916-04-1
5","end":"1934-03-16"},{"start":"1861-03-17","end":"1916-04-14","name":"MEZZ
ATE"}],"A296":[{"end":"9999-12-31","start":"1883-12-21","name":"ANGUILLARA V
ENETA"}],"A347":[{"end":"9999-12-31","start":"1862-12-29","name":"AQUILONIA"
}],"Z307":[{"start":"0000-01-01","end":"9999-12-31","name":"Capo Verde"}],"A
531":[{"name":"AZZATE","end":"1927-01-11","start":"1861-03-17"},{"end":"1928
-01-12","start":"1927-01-12","name":"AZZATE"},{"end":"1956-03-03","start":"1
928-01-13","name":"AZZATE"},{"name":"AZZATE","start":"1956-03-04","end":"999
9-12-31"}],"B013":[{"name":"BORGOFRANCO SUL PO","start":"1867-06-24","end":"
2018-12-31"}],"B319":[{"name":"CACCURI","start":"1861-03-17","end":"1928-05-
21"},{"name":"CACCURI","end":"1946-12-23","start":"1928-05-22"},{"name":"CAC
CURI","end":"1992-04-15","start":"1946-12-24"},{"name":"CACCURI","start":"19
92-04-16","end":"1999-01-12"},{"start":"1999-01-13","end":"9999-12-31","name
":"CACCURI"}],"H655":[{"end":"9999-12-31","start":"1866-11-19","name":"SACCO
LONGO"}],"C362":[{"end":"1938-02-18","start":"1861-03-17","name":"CAVACURTA"
},{"name":"CAVACURTA","end":"1992-04-15","start":"1938-02-19"},{"end":"2017-
12-31","start":"1992-04-16","name":"CAVACURTA"}],"F108":[{"start":"1861-03-1
7","end":"1992-04-15","name":"MELISSA"},{"name":"MELISSA","end":"9999-12-31"
,"start":"1992-04-16"}],"A713":[{"name":"BASTIGLIA","end":"9999-12-31","star
t":"1861-03-17"}],"H262":[{"name":"RHEMES NOTRE DAME","start":"1861-03-17","
end":"1927-01-11"},{"start":"1927-01-12","end":"1928-05-12","name":"RHEMES N
OTRE DAME"},{"start":"1946-03-10","end":"1977-01-01","name":"RHEMES NOTRE DA
ME"},{"name":"RH\u00caMES-NOTRE-DAME","end":"9999-12-31","start":"1977-01-02
"}],"F081":[{"name":"MEDEA","start":"1921-01-05","end":"1923-02-07"},{"name"
:"MEDEA","start":"1923-02-08","end":"1927-01-11"},{"name":"MEDEA","start":"1
927-01-12","end":"1928-09-08"},{"name":"MEDEA","end":"9999-12-31","start":"1
955-05-14"}],"H869":[{"name":"SAN GIACOMO","start":"1920-10-16","end":"1923-
02-13"},{"name":"SAN GIACOMO","end":"1928-12-31","start":"1923-02-14"}],"Z25
4":[{"name":"Georgia","end":"9999-12-31","start":"0000-01-01"}],"D154":[{"en
d":"1927-01-11","start":"1861-03-17","name":"CRESCENTINO"},{"end":"9999-12-3
1","start":"1927-01-12","name":"CRESCENTINO"}],"A386":[{"start":"1861-03-17"
,"end":"1992-04-15","name":"ARENA"},{"end":"9999-12-31","start":"1992-04-16"
,"name":"ARENA"}],"L105":[{"start":"1861-03-17","end":"1927-01-11","name":"T
ERELLE"},{"start":"1927-01-12","end":"9999-12-31","name":"TERELLE"}],"A587":
[{"start":"1861-03-17","end":"9999-12-31","name":"BALANGERO"}],"A838":[{"end
":"1947-09-30","start":"1928-04-15","name":"BEVILACQUA-BOSCHI"}],"C798":[{"e
nd":"1927-12-28","start":"1873-08-25","name":"CLUSANE SUL LAGO"}],"D458":[{"
start":"1861-03-17","end":"1985-01-13","name":"FAENZA"},{"start":"1985-01-14
","end":"9999-12-31","name":"FAENZA"}],"F926":[{"end":"1927-01-11","start":"
1861-03-17","name":"NOLI"},{"end":"9999-12-31","start":"1927-01-12","name":"
NOLI"}],"L924":[{"name":"VILLACIDRO","end":"1974-08-19","start":"1861-03-17"
},{"start":"1974-08-20","end":"2005-12-31","name":"VILLACIDRO"},{"name":"VIL
LACIDRO","start":"2006-01-01","end":"2016-04-27"},{"end":"9999-12-31","start
":"2016-04-28","name":"VILLACIDRO"}],"E918":[{"name":"MARASSI","end":"1873-1
1-09","start":"1861-03-17"}],"F629":[{"end":"1927-04-13","start":"1863-04-13
","name":"MONTE SANTA MARIA TIBERINA"},{"name":"MONTE SANTA MARIA TIBERINA",
"start":"1927-04-14","end":"9999-12-31"}],"A113":[{"name":"AISONE","start":"
1861-03-17","end":"9999-12-31"}],"E275":[{"end":"1928-03-26","start":"1861-0
3-17","name":"IATRINOLI"}],"H484":[{"name":"RO\u00c8 VOLCIANO","start":"1928
-09-09","end":"9999-12-31"},{"name":"VOLCIANO","start":"1861-03-17","end":"1
928-09-08"}],"L472":[{"start":"1861-03-17","end":"1929-07-09","name":"TURI"}
,{"end":"9999-12-31","start":"1929-07-10","name":"TURI"}],"C494":[{"name":"C
ERCIVENTO","end":"1968-04-05","start":"1866-11-19"},{"end":"9999-12-31","sta
rt":"1968-04-06","name":"CERCIVENTO"}],"G588":[{"name":"PIAZZOLO","end":"198
0-06-14","start":"1861-03-17"},{"name":"PIAZZOLO","start":"1980-06-15","end"
:"9999-12-31"}],"D686":[{"end":"9999-12-31","start":"1866-11-19","name":"FON
ZASO"}],"D989":[{"end":"1928-12-31","start":"1863-02-23","name":"GEROLANUOVA
"}],"D754":[{"name":"FRAGAGNANO","end":"1923-12-21","start":"1861-03-17"},{"
name":"FRAGAGNANO","start":"1923-12-22","end":"9999-12-31"}],"G837":[{"end":
"9999-12-31","start":"1861-03-17","name":"PONTECHIANALE"}],"L327":[{"start":
"1861-03-17","end":"9999-12-31","name":"TRANA"}],"C663":[{"name":"CHIUSI DEL
LA VERNA","end":"9999-12-31","start":"1928-05-19"},{"end":"1928-05-18","star
t":"1861-03-17","name":"CHIUSI IN CASENTINO"}],"H188":[{"name":"RASSA","end"
:"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"9999-12-31"
,"name":"RASSA"}],"L705":[{"end":"2014-02-03","start":"1928-08-07","name":"V
EDDASCA"}],"I352":[{"start":"1871-01-15","end":"1928-12-29","name":"SANT'ORE
STE"},{"end":"1941-01-20","start":"1928-12-30","name":"SANT'ORESTE"},{"name"
:"SANT'ORESTE","start":"1941-01-21","end":"9999-12-31"}],"I970":[{"name":"ST
RAMBINO","start":"1861-03-17","end":"1927-01-11"},{"end":"1928-06-15","start
":"1927-01-12","name":"STRAMBINO"},{"name":"STRAMBINO","end":"9999-12-31","s
tart":"1946-12-24"}],"G229":[{"name":"PAESE","start":"1866-11-19","end":"192
0-04-11"},{"start":"1920-04-12","end":"9999-12-31","name":"PAESE"}],"L526":[
{"name":"VACRI","start":"1861-03-17","end":"9999-12-31"}],"G822":[{"name":"P
ONSACCO","end":"9999-12-31","start":"1861-03-17"}],"F524":[{"name":"MONTE GR
IMANO","start":"1861-03-17","end":"1926-03-03"},{"name":"MONTE GRIMANO","end
":"1929-05-13","start":"1926-03-04"},{"end":"1947-11-24","start":"1929-05-14
","name":"MONTE GRIMANO"},{"start":"1947-11-25","end":"1998-09-03","name":"M
ONTE GRIMANO"},{"start":"1998-09-04","end":"2002-06-06","name":"MONTE GRIMAN
O"},{"name":"MONTE GRIMANO TERME","end":"9999-12-31","start":"2002-06-07"}],
"A280":[{"name":"ANDORNO MICCA","end":"1948-03-20","start":"1929-05-21"},{"n
ame":"ANDORNO MICCA","end":"1955-01-18","start":"1948-03-21"},{"start":"1955
-01-19","end":"1992-04-15","name":"ANDORNO MICCA"},{"end":"9999-12-31","star
t":"1992-04-16","name":"ANDORNO MICCA"}],"C198":[{"name":"CASTELLUCCIO DEI S
AURI","end":"9999-12-31","start":"1861-03-17"}],"L332":[{"name":"TRAPPETO","
end":"9999-12-31","start":"1954-07-23"}],"D225":[{"end":"1992-05-22","start"
:"1928-01-26","name":"CURSOLO-ORASSO"},{"name":"CURSOLO-ORASSO","start":"199
2-05-23","end":"2018-12-31"}],"E055":[{"start":"1861-03-17","end":"9999-12-3
1","name":"GIULIANA"}],"Z125":[{"name":"Norvegia","end":"9999-12-31","start"
:"0000-01-01"}],"H875":[{"name":"SAN GIMIGNANO","start":"1861-03-17","end":"
9999-12-31"}],"H001":[{"name":"PRATO SESIA","end":"9999-12-31","start":"1863
-04-23"}],"M027":[{"name":"VILLASTELLONE","start":"1861-03-17","end":"9999-1
2-31"}],"G348":[{"start":"1861-03-17","end":"2008-05-23","name":"PARTINICO"}
,{"start":"2008-05-24","end":"9999-12-31","name":"PARTINICO"}],"A390":[{"nam
e":"AREZZO","end":"9999-12-31","start":"1861-03-17"}],"G881":[{"name":"PORAN
O","start":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"999
9-12-31","name":"PORANO"}],"E730":[{"end":"9999-12-31","start":"1861-03-17",
"name":"LUGO"}],"H971":[{"name":"SAN LUCIDO","start":"1861-03-17","end":"999
9-12-31"}],"D112":[{"end":"1992-04-15","start":"1870-03-01","name":"COSTA MA
SNAGA"},{"name":"COSTA MASNAGA","end":"9999-12-31","start":"1992-04-16"}],"E
515":[{"end":"9999-12-31","start":"1866-11-19","name":"LEGNARO"}],"B807":[{"
name":"CARONNO GHIRINGHELLO","end":"1927-01-11","start":"1861-03-17"},{"star
t":"1927-01-12","end":"1940-03-22","name":"CARONNO GHIRINGHELLO"},{"start":"
1940-03-23","end":"9999-12-31","name":"CARONNO VARESINO"}],"M032":[{"end":"9
999-12-31","start":"1866-11-19","name":"VILLAVERLA"}],"G985":[{"name":"PRARO
LO","end":"1927-01-11","start":"1861-03-17"},{"end":"9999-12-31","start":"19
27-01-12","name":"PRAROLO"}],"F708":[{"end":"9999-12-31","start":"1863-07-13
","name":"MORANO CALABRO"}],"I873":[{"name":"SOVERE","start":"1861-03-17","e
nd":"1929-02-13"},{"name":"SOVERE","end":"2006-12-29","start":"1929-02-14"},
{"start":"2006-12-30","end":"9999-12-31","name":"SOVERE"}],"D735":[{"start":
"1861-03-17","end":"1919-06-26","name":"FOSDINOVO"},{"name":"FOSDINOVO","end
":"1983-07-20","start":"1919-06-27"},{"end":"9999-12-31","start":"1983-07-21
","name":"FOSDINOVO"}],"I925":[{"name":"SPORMINORE","end":"1923-02-13","star
t":"1920-10-16"},{"end":"1928-10-29","start":"1923-02-14","name":"SPORMINORE
"},{"end":"9999-12-31","start":"1947-07-09","name":"SPORMINORE"}],"H095":[{"
name":"PUTIFIGARI","end":"1928-11-29","start":"1861-03-17"},{"name":"PUTIFIG
ARI","start":"1947-10-26","end":"9999-12-31"}],"D510":[{"name":"FAUGLIA","en
d":"1902-08-21","start":"1861-03-17"},{"start":"1902-08-22","end":"9999-12-3
1","name":"FAUGLIA"}],"B709":[{"name":"CAPRINO VERONESE","end":"9999-12-31",
"start":"1867-08-26"}],"B997":[{"name":"CASSANO IRPINO","start":"1863-02-06"
,"end":"9999-12-31"}],"C085":[{"end":"1920-10-18","start":"1861-03-17","name
":"CASTEL DEL PIANO"},{"name":"CASTEL DEL PIANO","end":"1956-06-12","start":
"1920-10-19"},{"name":"CASTEL DEL PIANO","end":"9999-12-31","start":"1956-06
-13"}],"B394":[{"name":"CALCINATO","start":"1861-03-17","end":"9999-12-31"}]
,"D302":[{"start":"1920-10-16","end":"1923-02-13","name":"DIMARO"},{"end":"1
928-09-05","start":"1923-02-14","name":"DIMARO"},{"name":"DIMARO","end":"195
3-09-11","start":"1928-09-06"},{"start":"1953-09-12","end":"2015-12-31","nam
e":"DIMARO"}],"I704":[{"end":"9999-12-31","start":"1861-03-17","name":"SETTI
NGIANO"}],"I821":[{"name":"SOMMACAMPAGNA","start":"1866-11-19","end":"1975-0
1-28"},{"name":"SOMMACAMPAGNA","start":"1975-01-29","end":"9999-12-31"}],"E2
20":[{"end":"1927-03-11","start":"1861-03-17","name":"GRUMELLO DEL PIANO"}],
"D050":[{"start":"1861-03-17","end":"1928-12-01","name":"CORTANDONE"},{"star
t":"1947-10-02","end":"9999-12-31","name":"CORTANDONE"}],"H923":[{"name":"SA
N GIOVANNI LIPIONI","start":"1861-03-17","end":"9999-12-31"}],"I093":[{"name
":"SAN PIETRO A MAIDA","start":"1861-03-17","end":"1935-06-05"},{"name":"SAN
 PIETRO A MAIDA","end":"9999-12-31","start":"1935-06-06"}],"L483":[{"name":"
UDINE","start":"1866-11-19","end":"1968-04-05"},{"name":"UDINE","end":"9999-
12-31","start":"1968-04-06"}],"E407":[{"start":"1861-03-17","end":"1928-01-1
7","name":"LAGO"},{"name":"LAGO","end":"9999-12-31","start":"1928-01-18"}],"
A979":[{"name":"BONVICINO","start":"1887-06-06","end":"9999-12-31"}],"I882":
[{"end":"1929-06-12","start":"1861-03-17","name":"SPADAFORA SAN MARTINO"}],"
A676":[{"end":"1927-01-11","start":"1861-03-17","name":"BARRAFRANCA"},{"name
":"BARRAFRANCA","end":"9999-12-31","start":"1927-01-12"}],"E034":[{"start":"
1861-03-17","end":"9999-12-31","name":"GINESTRA DEGLI SCHIAVONI"}],"I289":[{
"name":"SANT'ANGELO LIMOSANO","start":"1861-03-17","end":"1928-08-12"},{"nam
e":"SANT'ANGELO LIMOSANO","start":"1933-04-26","end":"1970-03-02"},{"name":"
SANT'ANGELO LIMOSANO","end":"9999-12-31","start":"1970-03-03"}],"G567":[{"na
me":"PIANO PORLEZZA","start":"1863-07-13","end":"1928-05-11"}],"H820":[{"nam
e":"SAN DIDERO","end":"9999-12-31","start":"1861-03-17"}],"G739":[{"start":"
1921-01-05","end":"1923-02-07","name":"PLISCOVIZZA DELLA MADONNA"},{"start":
"1923-02-08","end":"1927-01-11","name":"PLISCOVIZZA DELLA MADONNA"},{"start"
:"1927-01-12","end":"1928-08-23","name":"PLISCOVIZZA DELLA MADONNA"}],"I528"
:[{"name":"SCHIERANCO","start":"1861-03-17","end":"1928-05-31"}],"H167":[{"n
ame":"RAISSA","start":"1920-10-16","end":"1923-02-13"},{"name":"RAISSA","sta
rt":"1923-02-14","end":"1923-04-26"},{"name":"RISCONE","end":"1927-01-11","s
tart":"1923-04-27"},{"name":"RISCONE","start":"1927-01-12","end":"1928-04-04
"}],"D270":[{"start":"1861-03-17","end":"1928-07-27","name":"DELLO"},{"name"
:"DELLO","start":"1928-07-28","end":"9999-12-31"}],"A303":[{"name":"ANOIA","
start":"1861-03-17","end":"9999-12-31"}],"A009":[{"end":"1947-09-15","start"
:"1931-12-22","name":"ABBAZIA"}],"B628":[{"start":"1861-03-17","end":"1928-0
5-12","name":"CANTALUPA"},{"start":"1954-07-15","end":"9999-12-31","name":"C
ANTALUPA"}],"Z343":[{"start":"0000-01-01","end":"9999-12-31","name":"Senegal
"}],"B284":[{"start":"1861-03-17","end":"1927-10-20","name":"BUSANO"},{"star
t":"1946-12-24","end":"9999-12-31","name":"BUSANO"}],"L791":[{"name":"VERTEM
ATE","start":"1861-03-17","end":"1929-05-29"}],"D376":[{"name":"DUALCHI","st
art":"1861-03-17","end":"1927-01-11"},{"name":"DUALCHI","start":"1927-01-12"
,"end":"1928-03-12"},{"name":"DUALCHI","end":"1946-04-26","start":"1939-02-1
7"},{"name":"DUALCHI","end":"9999-12-31","start":"1946-04-27"}],"L315":[{"na
me":"TOVO SAN GIACOMO","start":"1863-04-23","end":"1903-02-01"},{"name":"TOV
O SAN GIACOMO","end":"1927-01-11","start":"1903-02-02"},{"name":"TOVO SAN GI
ACOMO","start":"1927-01-12","end":"1929-01-23"},{"name":"TOVO SAN GIACOMO","
start":"1929-01-24","end":"9999-12-31"}],"D441":[{"name":"ESPORLATU","end":"
1928-11-29","start":"1861-03-17"},{"end":"9999-12-31","start":"1946-11-02","
name":"ESPORLATU"}],"H539":[{"name":"RONCO CANAVESE","start":"1863-01-05","e
nd":"1927-01-11"},{"name":"RONCO CANAVESE","start":"1927-01-12","end":"1945-
10-04"},{"end":"9999-12-31","start":"1945-10-05","name":"RONCO CANAVESE"}],"
G260":[{"name":"PALAZZO PIGNANO","start":"1863-03-27","end":"1929-05-03"},{"
name":"PALAZZO PIGNANO","end":"9999-12-31","start":"1929-05-04"}],"G139":[{"
name":"ORTIGNANO RAGGIOLO","start":"1875-10-26","end":"9999-12-31"}],"H767":
[{"name":"SAN BASSANO","end":"9999-12-31","start":"1861-03-17"}],"D577":[{"n
ame":"FIESSO UMBERTIANO","end":"9999-12-31","start":"1867-07-22"}],"H634":[{
"name":"RUFFR\u00c8","start":"1920-10-16","end":"1923-02-13"},{"start":"1923
-02-14","end":"1928-08-07","name":"RUFFR\u00c8"},{"end":"2005-05-10","start"
:"1953-01-04","name":"RUFFR\u00c8"},{"name":"RUFFR\u00c8-MENDOLA","start":"2
005-05-11","end":"9999-12-31"}],"F660":[{"name":"MONTEVERDE","start":"1861-0
3-17","end":"9999-12-31"}],"D436":[{"start":"1927-10-09","end":"1992-04-15",
"name":"ESINO LARIO"},{"end":"9999-12-31","start":"1992-04-16","name":"ESINO
 LARIO"}],"B843":[{"start":"1861-03-17","end":"1929-03-16","name":"CARTARI E
 CALDERARA"}],"D603":[{"start":"1861-03-17","end":"1928-06-30","name":"FINER
O"}],"B462":[{"start":"1861-03-17","end":"9999-12-31","name":"CAMBIANO"}],"A
856":[{"start":"1861-03-17","end":"1927-12-31","name":"BIDON\u00cc"},{"start
":"1951-01-12","end":"1974-08-19","name":"BIDON\u00cc"},{"name":"BIDON\u00cc
","start":"1974-08-20","end":"9999-12-31"}],"L191":[{"end":"9999-12-31","sta
rt":"1861-03-17","name":"TOLENTINO"}],"A568":[{"name":"BAGNOLI DI SOPRA","st
art":"1867-08-26","end":"9999-12-31"}],"E532":[{"start":"1861-03-17","end":"
9999-12-31","name":"LENTINI"}],"Z235":[{"end":"9999-12-31","start":"0000-01-
01","name":"Oman"}],"D135":[{"end":"1928-11-10","start":"1861-03-17","name":
"CRAVEGNA"}],"I408":[{"name":"SAN VITTORE DEL LAZIO","start":"1862-11-24","e
nd":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":"SAN VITTO
RE DEL LAZIO"}],"F318":[{"end":"9999-12-31","start":"1861-03-17","name":"MOM
PANTERO"}],"D217":[{"end":"1927-01-11","start":"1861-03-17","name":"CURIGLIA
"},{"name":"CURIGLIA","start":"1927-01-12","end":"1928-08-06"},{"end":"9999-
12-31","start":"1928-08-07","name":"CURIGLIA CON MONTEVIASCO"}],"B109":[{"en
d":"9999-12-31","start":"1861-03-17","name":"BOZZOLE"}],"Z117":[{"start":"00
00-01-01","end":"9999-12-31","name":"Islanda"}],"E326":[{"name":"IRSINA","en
d":"1927-01-11","start":"1895-04-12"},{"start":"1927-01-12","end":"9999-12-3
1","name":"IRSINA"}],"A965":[{"end":"9999-12-31","start":"1861-03-17","name"
:"BONDENO"}],"A602":[{"name":"BALSAMO","end":"1928-11-02","start":"1861-03-1
7"}],"M015":[{"end":"9999-12-31","start":"1861-03-17","name":"VILLAR SAN COS
TANZO"}],"B940":[{"end":"1968-04-05","start":"1867-09-02","name":"CASARSA DE
LLA DELIZIA"},{"end":"9999-12-31","start":"1968-04-06","name":"CASARSA DELLA
 DELIZIA"}],"I104":[{"end":"1929-04-19","start":"1861-03-17","name":"SAN PIE
TRO DI MONTEROSSO"}],"E527":[{"name":"LENOLA","start":"1861-03-17","end":"19
27-01-11"},{"name":"LENOLA","start":"1927-01-12","end":"1934-10-25"},{"start
":"1934-10-26","end":"1934-12-17","name":"LENOLA"},{"name":"LENOLA","end":"1
945-04-08","start":"1934-12-18"},{"name":"LENOLA","start":"1945-04-09","end"
:"9999-12-31"}],"M208":[{"end":"9999-12-31","start":"1968-11-28","name":"LAM
EZIA TERME"}],"A861":[{"start":"1861-03-17","end":"1927-12-03","name":"BIENN
O"},{"name":"BIENNO","start":"1927-12-04","end":"1947-06-17"},{"name":"BIENN
O","start":"1947-06-18","end":"9999-12-31"}],"E695":[{"start":"1861-03-17","
end":"9999-12-31","name":"LORSICA"}],"C826":[{"name":"COGORNO","start":"1861
-03-17","end":"9999-12-31"}],"G179":[{"start":"1861-03-17","end":"1963-02-12
","name":"OSSIMO"},{"name":"OSSIMO","start":"1963-02-13","end":"9999-12-31"}
],"Z730":[{"name":"Papua Nuova Guinea","end":"9999-12-31","start":"0000-01-0
1"}],"D888":[{"end":"1927-01-11","start":"1861-03-17","name":"GALTELL\u00cc"
},{"end":"1928-02-01","start":"1927-01-12","name":"GALTELL\u00cc"},{"name":"
GALTELL\u00cc","start":"1946-03-08","end":"9999-12-31"}],"L682":[{"name":"VA
RESE","end":"1927-01-11","start":"1861-03-17"},{"name":"VARESE","end":"1927-
12-28","start":"1927-01-12"},{"name":"VARESE","start":"1927-12-29","end":"19
29-06-17"},{"end":"1950-12-23","start":"1929-06-18","name":"VARESE"},{"name"
:"VARESE","start":"1950-12-24","end":"9999-12-31"}],"D537":[{"end":"1927-01-
11","start":"1861-03-17","name":"FENIS"},{"name":"FENIS","end":"1928-10-16",
"start":"1927-01-12"},{"end":"1977-01-01","start":"1946-05-15","name":"FENIS
"},{"end":"9999-12-31","start":"1977-01-02","name":"F\u00c9NIS"}],"H674":[{"
name":"SAINT PIERRE","start":"1861-03-17","end":"1927-01-11"},{"end":"1928-0
5-12","start":"1927-01-12","name":"SAINT PIERRE"},{"start":"1946-03-10","end
":"1977-01-01","name":"SAINT PIERRE"},{"start":"1977-01-02","end":"9999-12-3
1","name":"SAINT-PIERRE"}],"D341":[{"name":"DONGO","end":"1934-04-23","start
":"1861-03-17"},{"start":"1934-04-24","end":"9999-12-31","name":"DONGO"}],"L
415":[{"name":"TRIBIANO","end":"1940-02-22","start":"1861-03-17"},{"end":"99
99-12-31","start":"1940-02-23","name":"TRIBIANO"}],"C553":[{"name":"CERVIA",
"start":"1861-03-17","end":"9999-12-31"}],"E212":[{"end":"9999-12-31","start
":"1861-03-17","name":"GROTTERIA"}],"Z515":[{"name":"Nicaragua","end":"9999-
12-31","start":"0000-01-01"}],"E819":[{"end":"1968-07-22","start":"1869-03-2
5","name":"MAGNAGO"},{"start":"1968-07-23","end":"9999-12-31","name":"MAGNAG
O"}],"G040":[{"name":"OLIVETO LARIO","end":"1992-04-15","start":"1927-12-29"
},{"name":"OLIVETO LARIO","end":"9999-12-31","start":"1992-04-16"}],"A098":[
{"name":"AIDONE","end":"1927-01-11","start":"1861-03-17"},{"name":"AIDONE","
end":"9999-12-31","start":"1927-01-12"}],"D476":[{"name":"FALERNA","start":"
1861-03-17","end":"9999-12-31"}],"D522":[{"start":"1869-10-11","end":"1927-0
1-11","name":"ORCO FEGLINO"},{"name":"ORCO FEGLINO","start":"1927-01-12","en
d":"9999-12-31"}],"B362":[{"end":"1927-01-11","start":"1861-03-17","name":"C
AIAZZO"},{"end":"1945-08-31","start":"1927-01-12","name":"CAIAZZO"},{"start"
:"1945-09-01","end":"9999-12-31","name":"CAIAZZO"}],"E125":[{"name":"GRADO",
"start":"1921-01-05","end":"1923-02-07"},{"name":"GRADO","end":"1948-01-09",
"start":"1923-02-08"},{"start":"1948-01-10","end":"9999-12-31","name":"GRADO
"}],"C319":[{"start":"1861-03-17","end":"9999-12-31","name":"CASTIGLION FIOR
ENTINO"}],"C013":[{"end":"1928-06-26","start":"1895-01-07","name":"PERO"},{"
name":"PERO","end":"9999-12-31","start":"1928-06-27"}],"D336":[{"start":"192
0-10-16","end":"1923-02-13","name":"DON"},{"name":"DON","end":"1928-06-15","
start":"1923-02-14"},{"name":"DON","start":"1952-09-20","end":"2015-12-31"}]
,"D769":[{"end":"1927-01-11","start":"1861-03-17","name":"FRANCOLISE"},{"end
":"1945-08-31","start":"1927-01-12","name":"FRANCOLISE"},{"start":"1945-09-0
1","end":"9999-12-31","name":"FRANCOLISE"}],"H579":[{"start":"1861-03-17","e
nd":"2018-03-30","name":"ROSSANO"}],"M157":[{"name":"ZELBIO-VELESO","start":
"1928-04-15","end":"1948-03-26"}],"H812":[{"name":"SAN DAMIANO MACRA","start
":"1862-12-19","end":"1929-01-21"},{"name":"SAN DAMIANO MACRA","end":"9999-1
2-31","start":"1929-01-22"}],"A523":[{"end":"1928-07-31","start":"1861-03-17
","name":"AVOLASCA"},{"end":"1929-04-19","start":"1928-08-01","name":"AVOLAS
CA"},{"name":"AVOLASCA","end":"9999-12-31","start":"1929-04-20"}],"D581":[{"
name":"FIGLIARO","start":"1928-02-21","end":"1931-07-07"},{"start":"1931-07-
08","end":"1948-03-21","name":"MIRABELLO COMASCO"}],"I308":[{"end":"1992-04-
15","start":"1861-03-17","name":"SANTA SEVERINA"},{"end":"9999-12-31","start
":"1992-04-16","name":"SANTA SEVERINA"}],"H219":[{"name":"REFRANCORE","start
":"1861-03-17","end":"1935-04-07"},{"name":"REFRANCORE","start":"1935-04-08"
,"end":"1953-01-24"},{"name":"REFRANCORE","end":"9999-12-31","start":"1953-0
1-25"}],"E426":[{"name":"LAMA MOCOGNO","end":"9999-12-31","start":"1861-03-1
7"}],"G765":[{"name":"POGGIO NATIVO","end":"1923-03-18","start":"1861-03-17"
},{"end":"1927-01-11","start":"1923-03-19","name":"POGGIO NATIVO"},{"end":"1
949-07-09","start":"1927-01-12","name":"POGGIO NATIVO"},{"start":"1949-07-10
","end":"9999-12-31","name":"POGGIO NATIVO"}],"B798":[{"name":"CARNATE","end
":"2009-06-29","start":"1909-07-04"},{"name":"CARNATE","end":"9999-12-31","s
tart":"2009-06-30"}],"E572":[{"end":"1923-02-07","start":"1921-01-05","name"
:"LIBUSSINA"},{"name":"LIBUSSINA","start":"1923-02-08","end":"1927-01-11"},{
"start":"1927-01-12","end":"1928-05-01","name":"LIBUSSINA"}],"F897":[{"end":
"1923-10-13","start":"1861-03-17","name":"NIGUARDA"}],"L621":[{"start":"1920
-10-16","end":"1923-02-13","name":"VALLE SAN SILVESTRO"},{"name":"VALLE SAN 
SILVESTRO","end":"1927-01-11","start":"1923-02-14"},{"end":"1928-03-13","sta
rt":"1927-01-12","name":"VALLE SAN SILVESTRO"}],"D175":[{"start":"1861-03-17
","end":"1901-07-12","name":"CROCEFIESCHI"},{"end":"9999-12-31","start":"190
1-07-13","name":"CROCEFIESCHI"}],"F114":[{"name":"MELLE","end":"1928-06-09",
"start":"1861-03-17"},{"name":"MELLE","start":"1928-06-10","end":"1947-09-24
"},{"name":"MELLE","start":"1947-09-25","end":"9999-12-31"}],"L116":[{"name"
:"TERNENGO","end":"1927-01-11","start":"1861-03-17"},{"name":"TERNENGO","sta
rt":"1927-01-12","end":"1992-04-15"},{"end":"9999-12-31","start":"1992-04-16
","name":"TERNENGO"}],"D402":[{"name":"EMARESE","end":"1927-01-11","start":"
1861-03-17"},{"end":"1928-05-02","start":"1927-01-12","name":"EMARESE"},{"en
d":"1977-01-01","start":"1946-05-15","name":"EMARESE"},{"start":"1977-01-02"
,"end":"9999-12-31","name":"EMAR\u00c8SE"}],"B663":[{"start":"1871-01-15","e
nd":"1927-01-11","name":"CAPODIMONTE"},{"end":"9999-12-31","start":"1927-01-
12","name":"CAPODIMONTE"}],"F840":[{"start":"1861-03-17","end":"1974-08-19",
"name":"NARBOLIA"},{"name":"NARBOLIA","start":"1974-08-20","end":"9999-12-31
"}],"M299":[{"start":"1993-11-05","end":"1995-09-20","name":"BOVILLE"}],"B49
4":[{"name":"CASTEL CAMPAGNANO","end":"1927-01-11","start":"1916-07-07"},{"s
tart":"1927-01-12","end":"1945-08-31","name":"CASTEL CAMPAGNANO"},{"start":"
1945-09-01","end":"9999-12-31","name":"CASTEL CAMPAGNANO"},{"end":"1916-07-0
6","start":"1862-11-24","name":"CASTELLO DI CAMPAGNANO"}],"I563":[{"start":"
1866-11-19","end":"9999-12-31","name":"SEDICO"}],"G165":[{"name":"OSPEDALETT
O D'ALPINOLO","start":"1863-05-06","end":"9999-12-31"}],"I499":[{"end":"1923
-03-18","start":"1875-08-30","name":"SCANDRIGLIA"},{"start":"1923-03-19","en
d":"1927-01-11","name":"SCANDRIGLIA"},{"start":"1927-01-12","end":"9999-12-3
1","name":"SCANDRIGLIA"}],"G451":[{"name":"PERGINE","start":"1869-08-20","en
d":"1928-02-15"},{"name":"PERGINE VALDARNO","end":"2017-12-31","start":"1928
-02-16"}],"L383":[{"start":"1861-03-17","end":"1972-05-08","name":"TREPUZZI"
},{"name":"TREPUZZI","end":"9999-12-31","start":"1972-05-09"}],"B198":[{"nam
e":"BROLO","start":"1861-03-17","end":"9999-12-31"}],"L089":[{"end":"1923-02
-13","start":"1920-10-16","name":"TELVE"},{"start":"1923-02-14","end":"1928-
05-12","name":"TELVE"},{"start":"1947-01-15","end":"1980-01-01","name":"TELV
E"},{"end":"9999-12-31","start":"1980-01-02","name":"TELVE"}],"H565":[{"star
t":"1861-03-17","end":"9999-12-31","name":"ROSE"}],"D775":[{"end":"1923-02-1
3","start":"1920-10-16","name":"FRASSILONGO"},{"start":"1923-02-14","end":"1
929-03-24","name":"FRASSILONGO"},{"name":"FRASSILONGO","end":"9999-12-31","s
tart":"1948-01-25"}],"D024":[{"name":"CORNETO TARQUINIA","end":"1922-05-07",
"start":"1872-09-25"},{"start":"1922-05-08","end":"1928-12-29","name":"TARQU
INIA"},{"name":"TARQUINIA","end":"9999-12-31","start":"1928-12-30"}],"F714":
[{"end":"1927-01-11","start":"1861-03-17","name":"MORCA"},{"name":"MORCA","s
tart":"1927-01-12","end":"1929-03-01"}],"L716":[{"start":"1864-11-07","end":
"9999-12-31","name":"VELEZZO LOMELLINA"}],"Z130":[{"end":"9999-12-31","start
":"0000-01-01","name":"San Marino"}],"E904":[{"name":"MARANELLO","end":"9999
-12-31","start":"1861-03-17"}],"G779":[{"name":"POLAVENO","end":"9999-12-31"
,"start":"1861-03-17"}],"A641":[{"name":"BARCO","start":"1861-03-17","end":"
1927-12-23"}],"F152":[{"name":"MESAGNE","end":"1927-01-11","start":"1861-03-
17"},{"end":"9999-12-31","start":"1927-01-12","name":"MESAGNE"}],"D230":[{"n
ame":"CUSANO MUTRI","start":"1867-03-19","end":"9999-12-31"}],"H968":[{"end"
:"1868-07-04","start":"1861-03-17","name":"SAN LORENZO MONDINARI"}],"Z601":[
{"name":"Bolivia","start":"0000-01-01","end":"9999-12-31"}],"L293":[{"name":
"TORRICELLA IN SABINA","start":"1876-09-16","end":"1923-03-18"},{"name":"TOR
RICELLA IN SABINA","start":"1923-03-19","end":"1927-01-11"},{"name":"TORRICE
LLA IN SABINA","end":"9999-12-31","start":"1927-01-12"}],"F943":[{"start":"1
861-03-17","end":"1950-11-11","name":"NOTO"},{"name":"NOTO","end":"1981-04-1
5","start":"1950-11-12"},{"end":"1989-08-16","start":"1981-04-16","name":"NO
TO"},{"name":"NOTO","start":"1989-08-17","end":"9999-12-31"}],"E307":[{"name
":"INTRODACQUA","start":"1861-03-17","end":"9999-12-31"}],"L543":[{"name":"V
ALAS","start":"1923-04-27","end":"1927-01-11"},{"start":"1927-01-12","end":"
1928-12-30","name":"VALAS"},{"name":"VELASIO","start":"1920-10-16","end":"19
23-02-13"},{"name":"VELASIO","end":"1923-04-26","start":"1923-02-14"}],"E725
":[{"end":"1927-11-07","start":"1861-03-17","name":"LUDRIANO"}],"A403":[{"na
me":"ARIENZO","start":"1861-03-17","end":"1927-01-11"},{"name":"ARIENZO","st
art":"1927-01-12","end":"1928-09-25"},{"start":"1946-12-21","end":"9999-12-3
1","name":"ARIENZO"}],"C807":[{"end":"1928-11-29","start":"1875-06-04","name
":"COCCONATO"},{"end":"1935-04-07","start":"1928-11-30","name":"COCCONATO"},
{"name":"COCCONATO","end":"1947-01-09","start":"1935-04-08"},{"end":"1947-06
-22","start":"1947-01-10","name":"COCCONATO"},{"name":"COCCONATO","end":"999
9-12-31","start":"1947-06-23"}],"M389":[{"start":"2018-01-01","end":"9999-12
-31","name":"ALTO SERMENZA"}],"M083":[{"name":"VITICUSO","end":"1927-01-11",
"start":"1902-07-23"},{"name":"VITICUSO","start":"1927-01-12","end":"9999-12
-31"}],"D169":[{"end":"1927-01-11","start":"1862-12-29","name":"CREVOLA SESI
A"},{"name":"CREVOLA SESIA","end":"1929-03-01","start":"1927-01-12"}],"E506"
:[{"name":"LECCE","start":"1861-03-17","end":"9999-12-31"}],"H854":[{"end":"
1928-04-17","start":"1861-03-17","name":"SAN GALLO"}],"E074":[{"name":"GODRA
NO","start":"1861-03-17","end":"2011-08-11"},{"start":"2011-08-12","end":"99
99-12-31","name":"GODRANO"}],"G097":[{"name":"ORGOSOLO","end":"1927-01-11","
start":"1861-03-17"},{"name":"ORGOSOLO","start":"1927-01-12","end":"9999-12-
31"}],"M035":[{"end":"1945-08-31","start":"1928-03-13","name":"VILLA VOLTURN
O"},{"name":"VILLA VOLTURNO","end":"1946-04-13","start":"1945-09-01"}],"F174
":[{"end":"1965-02-16","start":"1861-03-17","name":"MEZZANI"},{"name":"MEZZA
NI","end":"2018-12-31","start":"1965-02-17"}],"L176":[{"end":"1923-02-13","s
tart":"1920-10-16","name":"TIRES"},{"name":"TIRES","start":"1923-02-14","end
":"1927-01-11"},{"name":"TIRES","start":"1927-01-12","end":"9999-12-31"}],"G
982":[{"end":"1928-05-30","start":"1861-03-17","name":"PRAMOLLO"},{"name":"P
RAMOLLO","end":"9999-12-31","start":"1954-06-30"}],"F338":[{"start":"1861-03
-17","end":"1928-04-15","name":"MONCHIERO"},{"end":"9999-12-31","start":"194
7-03-15","name":"MONCHIERO"}],"D237":[{"end":"1907-07-01","start":"1861-03-1
7","name":"CUTROFIANO"},{"name":"CUTROFIANO","start":"1907-07-02","end":"999
9-12-31"}],"E512":[{"name":"LEGNAGO","end":"9999-12-31","start":"1866-11-19"
}],"Z215":[{"start":"0000-01-01","end":"9999-12-31","name":"Emirati Arabi Un
iti"}],"C253":[{"start":"1861-03-17","end":"1935-04-07","name":"CASTEL ROCCH
ERO"},{"name":"CASTEL ROCCHERO","end":"9999-12-31","start":"1935-04-08"}],"A
048":[{"name":"ACQUAVIVA DELLE FONTI","start":"1863-01-19","end":"9999-12-31
"}],"F478":[{"name":"MONTECOPIOLO","end":"1998-09-03","start":"1861-03-17"},
{"name":"MONTECOPIOLO","end":"2021-06-16","start":"1998-09-04"},{"name":"MON
TECOPIOLO","end":"9999-12-31","start":"2021-06-17"}],"H279":[{"start":"1920-
10-16","end":"1923-02-13","name":"RIDANNA"},{"name":"RIDANNA","start":"1923-
02-14","end":"1927-01-11"},{"end":"1928-04-12","start":"1927-01-12","name":"
RIDANNA"}],"D829":[{"name":"FUSIGNANO","end":"9999-12-31","start":"1861-03-1
7"}],"G090":[{"name":"ORCIANO PISANO","start":"1862-10-20","end":"1927-01-11
"},{"start":"1957-07-21","end":"9999-12-31","name":"ORCIANO PISANO"}],"Z122"
:[{"end":"9999-12-31","start":"0000-01-01","name":"Isola di Man"}],"D391":[{
"end":"1927-08-12","start":"1861-03-17","name":"EDOLO"},{"end":"1928-12-13",
"start":"1927-08-13","name":"EDOLO"},{"end":"1948-03-25","start":"1928-12-14
","name":"EDOLO"},{"end":"9999-12-31","start":"1948-03-26","name":"EDOLO"}],
"E052":[{"start":"1861-03-17","end":"9999-12-31","name":"GISSI"}],"D222":[{"
end":"9999-12-31","start":"1928-06-15","name":"CURON VENOSTA"}],"H872":[{"na
me":"SANGIANO","end":"1927-01-11","start":"1861-03-17"},{"name":"SANGIANO","
end":"1928-01-07","start":"1927-01-12"},{"name":"SANGIANO","end":"9999-12-31
","start":"1963-09-14"}],"C800":[{"name":"CLUSONE","end":"1919-12-14","start
":"1861-03-17"},{"start":"1919-12-15","end":"9999-12-31","name":"CLUSONE"}],
"F886":[{"name":"NIBBIOLA","end":"9999-12-31","start":"1861-03-17"}],"L884":
[{"start":"1920-10-16","end":"1923-02-13","name":"VIGNOLA"},{"start":"1923-0
2-14","end":"1929-01-24","name":"VIGNOLA"}],"H519":[{"end":"9999-12-31","sta
rt":"1861-03-17","name":"ROMETTA"}],"G825":[{"name":"PONTASSIEVE","start":"1
861-03-17","end":"9999-12-31"}],"L335":[{"start":"1866-11-19","end":"1968-04
-05","name":"TRASAGHIS"},{"end":"9999-12-31","start":"1968-04-06","name":"TR
ASAGHIS"}],"L702":[{"name":"VECCHIANO","end":"9999-12-31","start":"1861-03-1
7"}],"F847":[{"name":"NASINO","start":"1861-03-17","end":"1927-01-11"},{"nam
e":"NASINO","start":"1927-01-12","end":"9999-12-31"}],"C379":[{"start":"1861
-03-17","end":"1927-11-01","name":"CAVANDONE"}],"C073":[{"start":"1866-11-19
","end":"1928-06-18","name":"CASTELCUCCO"},{"name":"CASTELCUCCO","start":"19
46-04-26","end":"9999-12-31"}],"D281":[{"name":"DESANA","start":"1861-03-17"
,"end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31","name":"DESANA
"}],"A223":[{"start":"1861-03-17","end":"9999-12-31","name":"ALSENO"}],"I355
":[{"end":"1923-02-07","start":"1921-01-05","name":"BAINSIZZA SANTO SPIRITO"
},{"name":"BAINSIZZA SANTO SPIRITO","start":"1923-02-08","end":"1923-04-26"}
,{"end":"1927-01-11","start":"1923-04-27","name":"SANTO SPIRITO DELLA BAINSI
ZZA"},{"name":"SANTO SPIRITO DELLA BAINSIZZA","end":"1928-08-15","start":"19
27-01-12"}],"B748":[{"start":"1861-03-17","end":"1927-01-11","name":"CARCARE
"},{"start":"1927-01-12","end":"9999-12-31","name":"CARCARE"}],"E879":[{"end
":"1927-07-07","start":"1863-02-23","name":"MANDELLO DEL LARIO"},{"end":"192
8-01-25","start":"1927-07-08","name":"MANDELLO DEL LARIO"},{"name":"MANDELLO
 DEL LARIO","end":"1992-04-15","start":"1928-01-26"},{"name":"MANDELLO DEL L
ARIO","start":"1992-04-16","end":"9999-12-31"}],"G921":[{"start":"1952-01-01
","end":"2009-07-11","name":"PORTO SANT'ELPIDIO"},{"end":"9999-12-31","start
":"2009-07-12","name":"PORTO SANT'ELPIDIO"}],"C439":[{"start":"1861-03-17","
end":"9999-12-31","name":"CELLATICA"}],"H353":[{"start":"1861-03-17","end":"
9999-12-31","name":"RIVISONDOLI"}],"H059":[{"end":"1929-03-25","start":"1861
-03-17","name":"PRIERO"},{"start":"1947-03-30","end":"1957-07-19","name":"PR
IERO"},{"name":"PRIERO","end":"9999-12-31","start":"1957-07-20"}],"C903":[{"
name":"COLORINA","start":"1861-03-17","end":"9999-12-31"}],"D416":[{"name":"
ERBA","end":"1906-11-12","start":"1861-03-17"},{"start":"1928-01-26","end":"
1928-12-13","name":"ERBA"},{"name":"ERBA","end":"1928-12-15","start":"1928-1
2-14"},{"end":"1935-07-12","start":"1928-12-16","name":"ERBA"},{"name":"ERBA
","end":"9999-12-31","start":"1935-07-13"}],"G119":[{"start":"1861-03-17","e
nd":"1927-01-11","name":"OROSEI"},{"name":"OROSEI","end":"9999-12-31","start
":"1927-01-12"}],"H614":[{"name":"ROVESCALA","end":"9999-12-31","start":"187
8-06-14"}],"L475":[{"start":"1861-03-17","end":"1927-01-11","name":"TURRIVAL
IGNANI"},{"name":"TURRIVALIGNANI","end":"9999-12-31","start":"1927-01-12"}],
"E272":[{"end":"9999-12-31","start":"1861-03-17","name":"GUSSOLA"}],"E014":[
{"name":"GIARDINI","start":"1861-03-17","end":"1978-07-01"},{"start":"1978-0
7-02","end":"9999-12-31","name":"GIARDINI-NAXOS"}],"I263":[{"end":"1927-04-0
7","start":"1895-04-26","name":"SANT'ANATOLIA DI NARCO"},{"start":"1930-09-1
4","end":"9999-12-31","name":"SANT'ANATOLIA DI NARCO"}],"L028":[{"start":"19
28-01-29","end":"1947-06-23","name":"TAGLIOLO BELFORTE"}],"E659":[{"start":"
1861-03-17","end":"1967-08-17","name":"LOMAZZO"},{"start":"1967-08-18","end"
:"9999-12-31","name":"LOMAZZO"}],"L102":[{"name":"TEORA","end":"9999-12-31",
"start":"1861-03-17"}],"E956":[{"name":"MARINA DI GIOIOSA IONICA","end":"999
9-12-31","start":"1948-07-22"}],"I449":[{"end":"1923-12-21","start":"1861-03
-17","name":"SARZANA"},{"name":"SARZANA","start":"1923-12-22","end":"9999-12
-31"}],"B148":[{"name":"SOMBRENO","start":"1864-06-23","end":"1929-05-16"}],
"H806":[{"name":"SAN COSMO ALBANESE","end":"9999-12-31","start":"1897-02-13"
}],"E554":[{"end":"1927-01-11","start":"1861-03-17","name":"LETINO"},{"end":
"1945-08-31","start":"1927-01-12","name":"LETINO"},{"end":"9999-12-31","star
t":"1945-09-01","name":"LETINO"}],"G086":[{"name":"ORATINO","end":"1928-05-1
2","start":"1861-03-17"},{"end":"1970-03-02","start":"1947-05-22","name":"OR
ATINO"},{"start":"1970-03-03","end":"9999-12-31","name":"ORATINO"}],"G719":[
{"name":"PIVERONE","start":"1861-03-17","end":"1927-01-11"},{"end":"1929-05-
25","start":"1927-01-12","name":"PIVERONE"},{"end":"9999-12-31","start":"194
6-12-26","name":"PIVERONE"}],"B444":[{"end":"9999-12-31","start":"1958-05-28
","name":"CALVI"}],"H265":[{"end":"9999-12-31","start":"1861-03-17","name":"
RIACE"}],"F890":[{"name":"NICOLOSI","end":"9999-12-31","start":"1861-03-17"}
],"G047":[{"name":"OLMENETA","end":"9999-12-31","start":"1861-03-17"}],"M328
":[{"name":"CRESPINA LORENZANA","start":"2014-01-01","end":"9999-12-31"}],"F
774":[{"start":"1861-03-17","end":"1927-01-11","name":"MOTTA DE' CONTI"},{"n
ame":"MOTTA DE' CONTI","end":"9999-12-31","start":"1927-01-12"}],"L776":[{"n
ame":"VERNOLE","end":"9999-12-31","start":"1861-03-17"}],"H652":[{"start":"1
868-07-01","end":"9999-12-31","name":"SABBIONETA"}],"C365":[{"name":"CAVAGLI
O D'AGOGNA","start":"1863-04-23","end":"1928-05-02"},{"end":"1947-11-27","st
art":"1928-05-03","name":"CAVAGLIO D'AGOGNA"},{"name":"CAVAGLIO D'AGOGNA","e
nd":"9999-12-31","start":"1947-11-28"}],"D530":[{"name":"FELTRE","start":"18
66-11-19","end":"9999-12-31"}],"F025":[{"name":"MASSA E COZZILE","end":"1928
-01-25","start":"1861-03-17"},{"name":"MASSA E COZZILE","end":"9999-12-31","
start":"1928-01-26"}],"L593":[{"name":"VALLE LOMELLINA","end":"9999-12-31","
start":"1863-03-30"}],"E961":[{"name":"MARMENTINO","start":"1861-03-17","end
":"1927-12-30"},{"end":"9999-12-31","start":"1955-03-12","name":"MARMENTINO"
}],"D715":[{"start":"1861-03-17","end":"1970-03-02","name":"FORNELLI"},{"nam
e":"FORNELLI","end":"9999-12-31","start":"1970-03-03"}],"M150":[{"start":"18
61-03-17","end":"1923-07-22","name":"ZAVATTARELLO"},{"start":"1923-07-23","e
nd":"1927-01-26","name":"ZAVATTARELLO"},{"name":"ZAVATTARELLO","end":"1929-0
2-23","start":"1927-01-27"},{"end":"9999-12-31","start":"1956-04-08","name":
"ZAVATTARELLO"}],"E206":[{"name":"GROTTAMINARDA","end":"9999-12-31","start":
"1861-03-17"}],"L469":[{"name":"TURANO","start":"1869-02-13","end":"1929-02-
13"},{"end":"1938-02-18","start":"1929-02-14","name":"TURANO LODIGIANO"},{"n
ame":"TURANO LODIGIANO","end":"1992-04-15","start":"1938-02-19"},{"start":"1
992-04-16","end":"9999-12-31","name":"TURANO LODIGIANO"}],"F993":[{"end":"19
27-01-11","start":"1861-03-17","name":"OCCHIEPPO SUPERIORE"},{"start":"1927-
01-12","end":"1992-04-15","name":"OCCHIEPPO SUPERIORE"},{"start":"1992-04-16
","end":"9999-12-31","name":"OCCHIEPPO SUPERIORE"}],"Z722":[{"name":"Isole P
itcairn","start":"0000-01-01","end":"9999-12-31"}],"L243":[{"name":"TORRE CA
JETANI","end":"1927-01-11","start":"1872-10-18"},{"name":"TORRE CAJETANI","s
tart":"1927-01-12","end":"9999-12-31"}],"A691":[{"name":"BASCHI","start":"18
61-03-17","end":"1927-01-11"},{"name":"BASCHI","start":"1927-01-12","end":"1
948-04-29"},{"end":"9999-12-31","start":"1948-04-30","name":"BASCHI"}],"E865
":[{"start":"1861-03-17","end":"9999-12-31","name":"MALONNO"}],"I349":[{"nam
e":"SANT'OMOBONO IMAGNA","start":"1927-08-03","end":"2004-08-13"},{"name":"S
ANT'OMOBONO TERME","end":"2014-02-03","start":"2004-08-14"}],"Z110":[{"name"
:"Francia","start":"0000-01-01","end":"9999-12-31"}],"D210":[{"end":"9999-12
-31","start":"1862-11-24","name":"CUPRA MARITTIMA"}],"C166":[{"end":"9999-12
-31","start":"1861-03-17","name":"CASTELLETTO SOPRA TICINO"}],"D183":[{"end"
:"1938-12-21","start":"1866-11-19","name":"CROSARA"}],"A121":[{"name":"ALANO
 DI PIAVE","end":"9999-12-31","start":"1867-04-12"}],"B344":[{"name":"CADINE
","end":"1923-02-13","start":"1920-10-16"},{"name":"CADINE","start":"1923-02
-14","end":"1926-09-30"}],"E520":[{"end":"9999-12-31","start":"1861-03-17","
name":"LEMIE"}],"H045":[{"start":"1878-08-02","end":"1927-01-11","name":"PRE
SENZANO"},{"end":"1945-08-31","start":"1927-01-12","name":"PRESENZANO"},{"st
art":"1945-09-01","end":"9999-12-31","name":"PRESENZANO"}],"B947":[{"start":
"1927-02-18","end":"9999-12-31","name":"CASAZZA"},{"name":"MOLOGNO","end":"1
927-02-17","start":"1861-03-17"}],"M069":[{"name":"VIRLE PIEMONTE","end":"19
28-05-22","start":"1864-04-01"},{"name":"VIRLE PIEMONTE","end":"9999-12-31",
"start":"1946-12-19"}],"M363":[{"name":"VILLE D'ANAUNIA","start":"2016-01-01
","end":"9999-12-31"}],"I761":[{"end":"1928-11-21","start":"1861-03-17","nam
e":"SIRTORI"},{"end":"1992-04-15","start":"1953-12-06","name":"SIRTORI"},{"n
ame":"SIRTORI","end":"9999-12-31","start":"1992-04-16"}],"A919":[{"name":"BO
FFALORA D'ADDA","end":"1927-12-26","start":"1863-01-26"},{"name":"BOFFALORA 
D'ADDA","start":"1927-12-27","end":"1992-04-15"},{"end":"9999-12-31","start"
:"1992-04-16","name":"BOFFALORA D'ADDA"}],"A616":[{"name":"BARANELLO","start
":"1861-03-17","end":"1970-03-02"},{"end":"9999-12-31","start":"1970-03-03",
"name":"BARANELLO"}],"L741":[{"end":"1923-12-05","start":"1861-03-17","name"
:"VENTIMIGLIA"},{"start":"1923-12-06","end":"1925-08-21","name":"VENTIMIGLIA
"},{"name":"VENTIMIGLIA","end":"9999-12-31","start":"1925-08-22"}],"A182":[{
"name":"ALESSANDRIA","start":"1861-03-17","end":"1929-06-14"},{"name":"ALESS
ANDRIA","start":"1929-06-15","end":"1972-03-21"},{"end":"9999-12-31","start"
:"1972-03-22","name":"ALESSANDRIA"}],"I589":[{"name":"SELLIA","start":"1861-
03-17","end":"1957-01-17"},{"name":"SELLIA","start":"1957-01-18","end":"9999
-12-31"}],"G267":[{"end":"1981-04-15","start":"1863-07-13","name":"PALAZZOLO
 ACREIDE"},{"start":"1981-04-16","end":"1989-08-16","name":"PALAZZOLO ACREID
E"},{"end":"9999-12-31","start":"1989-08-17","name":"PALAZZOLO ACREIDE"}],"C
465":[{"start":"1861-03-17","end":"1928-04-14","name":"CENOVA"}],"M428":[{"n
ame":"PRESICCE-ACQUARICA","end":"9999-12-31","start":"2019-05-15"}],"B689":[
{"name":"MARZABOTTO","end":"9999-12-31","start":"1882-07-17"}],"F734":[{"end
":"9999-12-31","start":"1871-01-15","name":"MORLUPO"}],"F667":[{"name":"MONT
I","start":"1861-03-17","end":"1963-09-26"},{"name":"MONTI","end":"2005-12-3
1","start":"1963-09-27"},{"end":"2016-04-27","start":"2006-01-01","name":"MO
NTI"},{"start":"2016-04-28","end":"9999-12-31","name":"MONTI"}],"L736":[{"st
art":"1883-02-02","end":"1916-12-31","name":"VENEZIA"},{"end":"1923-06-10","
start":"1917-01-01","name":"VENEZIA"},{"start":"1923-06-11","end":"1924-01-1
3","name":"VENEZIA"},{"name":"VENEZIA","start":"1924-01-14","end":"1926-08-2
3"},{"name":"VENEZIA","start":"1926-08-24","end":"1927-09-03"},{"name":"VENE
ZIA","start":"1927-09-04","end":"1999-04-16"},{"start":"1999-04-17","end":"9
999-12-31","name":"VENEZIA"}],"B584":[{"name":"CANDELA","start":"1861-03-17"
,"end":"9999-12-31"}],"C151":[{"start":"1861-03-17","end":"1927-01-11","name
":"CASTELLENGO"},{"start":"1927-01-12","end":"1930-04-06","name":"CASTELLENG
O"}],"I684":[{"start":"1869-08-11","end":"1928-12-15","name":"SESTO FIORENTI
NO"},{"name":"SESTO FIORENTINO","end":"9999-12-31","start":"1928-12-16"}],"I
756":[{"name":"SIRIGNANO","end":"9999-12-31","start":"1861-03-17"}],"M124":[
{"name":"VOLTAGO","end":"1964-06-26","start":"1866-11-19"},{"name":"VOLTAGO 
AGORDINO","end":"9999-12-31","start":"1964-06-27"}],"D570":[{"end":"1927-01-
11","start":"1887-07-22","name":"FICULLE"},{"name":"FICULLE","start":"1927-0
1-12","end":"9999-12-31"}],"H760":[{"start":"1863-02-23","end":"1928-11-26",
"name":"SAN BARTOLOMEO VAL CAVARGNA"},{"end":"1950-04-04","start":"1928-11-2
7","name":"SAN BARTOLOMEO VAL CAVARGNA"},{"name":"SAN BARTOLOMEO VAL CAVARGN
A","start":"1950-04-05","end":"9999-12-31"}],"L063":[{"name":"TAURIANOVA","s
tart":"1928-03-27","end":"1946-04-25"},{"name":"TAURIANOVA","start":"1946-04
-26","end":"9999-12-31"}],"L369":[{"start":"1874-06-30","end":"9999-12-31","
name":"TREMESTIERI ETNEO"}],"A782":[{"start":"1861-03-17","end":"9999-12-31"
,"name":"BENEVELLO"}],"L141":[{"end":"1928-05-12","start":"1861-03-17","name
":"TERZANO"}],"G560":[{"name":"PIANEZZE","end":"9999-12-31","start":"1866-11
-19"}],"F438":[{"end":"1923-02-13","start":"1920-10-16","name":"MONTE A TRAM
ONTANA"},{"start":"1923-02-14","end":"1923-04-26","name":"MONTE A TRAMONTANA
"},{"name":"MONTE DI TRAMONTANA","end":"1927-01-11","start":"1923-04-27"},{"
end":"1928-05-12","start":"1927-01-12","name":"MONTE DI TRAMONTANA"}],"H239"
:[{"name":"RESCALDA","start":"1861-03-17","end":"1869-03-10"}],"B893":[{"end
":"9999-12-31","start":"1861-03-17","name":"CASALGRANDE"}],"H832":[{"name":"
SAN FELICE DI SCOVOLO","start":"1863-01-26","end":"1928-06-29"}],"B021":[{"n
ame":"BORGOMASINO","end":"1927-01-11","start":"1861-03-17"},{"name":"BORGOMA
SINO","start":"1927-01-12","end":"1929-03-01"},{"name":"BORGOMASINO","end":"
1945-10-04","start":"1929-03-02"},{"end":"1947-09-29","start":"1945-10-05","
name":"BORGOMASINO"},{"name":"BORGOMASINO","end":"9999-12-31","start":"1947-
09-30"}],"E400":[{"name":"LACONI","start":"1861-03-17","end":"1927-01-11"},{
"name":"LACONI","start":"1927-01-12","end":"2005-12-31"},{"end":"9999-12-31"
,"start":"2006-01-01","name":"LACONI"}],"F134":[{"name":"MERCALLO","end":"19
27-01-11","start":"1861-03-17"},{"name":"MERCALLO","start":"1927-01-12","end
":"9999-12-31"}],"L136":[{"name":"TERRAZZO","start":"1866-11-19","end":"9999
-12-31"}],"A268":[{"name":"ANACAPRI","start":"1861-03-17","end":"1927-06-01"
},{"name":"ANACAPRI","start":"1946-04-02","end":"9999-12-31"}],"M075":[{"end
":"1923-02-07","start":"1921-01-05","name":"VISINADA"},{"start":"1923-02-08"
,"end":"1947-09-15","name":"VISINADA"}],"D491":[{"start":"1864-06-23","end":
"9999-12-31","name":"FARA OLIVANA CON SOLA"}],"C751":[{"start":"1861-03-17",
"end":"1927-01-11","name":"CITTIGLIO"},{"name":"CITTIGLIO","start":"1927-01-
12","end":"1928-01-17"},{"name":"CITTIGLIO","start":"1928-01-18","end":"1953
-08-31"},{"name":"CITTIGLIO","end":"9999-12-31","start":"1953-09-01"}],"I156
":[{"end":"9999-12-31","start":"1863-08-10","name":"SAN SEVERINO MARCHE"}],"
H827":[{"name":"SAN DONATO MILANESE","start":"1870-08-02","end":"1932-10-20"
},{"end":"9999-12-31","start":"1932-10-21","name":"SAN DONATO MILANESE"}],"F
378":[{"name":"MONRUPINO","end":"1947-09-15","start":"1932-05-26"},{"name":"
MONRUPINO","end":"2009-01-28","start":"1947-09-16"},{"end":"9999-12-31","sta
rt":"2009-01-29","name":"MONRUPINO-REPENTABOR"},{"end":"1923-02-07","start":
"1921-01-05","name":"REPNO"},{"name":"REPNO","end":"1923-04-26","start":"192
3-02-08"},{"end":"1932-05-25","start":"1923-04-27","name":"RUPIN GRANDE"}],"
D277":[{"end":"9999-12-31","start":"1861-03-17","name":"DERNICE"}],"I455":[{
"start":"1861-03-17","end":"9999-12-31","name":"SASSINORO"}],"E839":[{"end":
"9999-12-31","start":"1861-03-17","name":"MAIORI"}],"C479":[{"end":"1927-01-
11","start":"1871-01-15","name":"CEPRANO"},{"name":"CEPRANO","end":"9999-12-
31","start":"1927-01-12"}],"E645":[{"name":"LOCOROTONDO","start":"1861-03-17
","end":"9999-12-31"}],"L435":[{"end":"1970-03-02","start":"1861-03-17","nam
e":"TRIVENTO"},{"name":"TRIVENTO","start":"1970-03-03","end":"9999-12-31"}],
"E232":[{"name":"GUALTIERI","start":"1861-03-17","end":"9999-12-31"}],"B990"
:[{"name":"CASORIA","end":"1946-07-29","start":"1861-03-17"},{"name":"CASORI
A","end":"1958-03-14","start":"1946-07-30"},{"end":"9999-12-31","start":"195
8-03-15","name":"CASORIA"}],"D517":[{"name":"FAVERZANO","start":"1861-03-17"
,"end":"1928-02-15"}],"C766":[{"start":"1861-03-17","end":"9999-12-31","name
":"CIVITA D'ANTINO"}],"L375":[{"name":"TRENTA","end":"2017-05-04","start":"1
861-03-17"}],"M255":[{"name":"MOLINA ATERNO","start":"1889-07-15","end":"195
1-12-31"},{"name":"MOLINA ATERNO","end":"9999-12-31","start":"1952-01-01"}],
"D783":[{"name":"FRASSINORO","end":"9999-12-31","start":"1870-10-01"}],"A804
":[{"name":"BERNATE TICINO","start":"1862-12-01","end":"1980-05-13"},{"end":
"9999-12-31","start":"1980-05-14","name":"BERNATE TICINO"}],"A721":[{"name":
"BAULADU","end":"1928-05-12","start":"1861-03-17"},{"start":"1946-03-25","en
d":"1974-08-19","name":"BAULADU"},{"start":"1974-08-20","end":"9999-12-31","
name":"BAULADU"}],"D057":[{"start":"1868-03-05","end":"9999-12-31","name":"C
ORTE DE' FRATI"}],"E227":[{"name":"GUAGNANO","end":"1927-01-11","start":"186
1-03-17"},{"start":"1927-01-12","end":"1927-04-13","name":"GUAGNANO"},{"name
":"GUAGNANO","start":"1927-04-14","end":"9999-12-31"}],"H453":[{"name":"ROCC
AVIONE","start":"1861-03-17","end":"1928-03-06"},{"name":"ROCCAVIONE","end":
"1946-04-25","start":"1928-03-07"},{"name":"ROCCAVIONE","start":"1946-04-26"
,"end":"9999-12-31"}],"D316":[{"name":"DOGNA","start":"1866-11-19","end":"19
68-04-05"},{"end":"9999-12-31","start":"1968-04-06","name":"DOGNA"}],"I161":
[{"name":"SAN SILVESTRO","end":"1868-09-29","start":"1861-03-17"},{"name":"S
AN SILVESTRO","start":"1868-09-30","end":"1879-05-12"}],"B082":[{"name":"BOS
NASCO","end":"9999-12-31","start":"1861-03-17"}],"C339":[{"start":"1925-02-2
7","end":"1925-10-31","name":"CASTROCARO E TERRA DEL SOLE"},{"name":"CASTROC
ARO E TERRA DEL SOLE","end":"1962-05-25","start":"1925-11-01"},{"start":"196
2-05-26","end":"1992-04-15","name":"CASTROCARO TERME E TERRA DEL SOLE"},{"na
me":"CASTROCARO TERME E TERRA DEL SOLE","end":"9999-12-31","start":"1992-04-
16"},{"end":"1923-03-18","start":"1871-12-07","name":"TERRA DEL SOLE E CASTR
OCARO"},{"start":"1923-03-19","end":"1925-02-26","name":"TERRA DEL SOLE E CA
STROCARO"}],"C033":[{"name":"CASSINETTA DI LUGAGNANO","start":"1862-12-01","
end":"9999-12-31"}],"E569":[{"name":"LEZZENO","start":"1861-03-17","end":"99
99-12-31"}],"B253":[{"name":"BUGIALLO","start":"1861-03-17","end":"1928-05-1
1"}],"Z206":[{"name":"Myanmar/Birmania","end":"9999-12-31","start":"0000-01-
01"}],"M020":[{"name":"VILLA SANTA CATERINA","start":"1920-10-16","end":"192
3-02-13"},{"start":"1923-02-14","end":"1927-01-11","name":"VILLA SANTA CATER
INA"},{"end":"1928-04-04","start":"1927-01-12","name":"VILLA SANTA CATERINA"
}],"A246":[{"name":"ALZANO LOMBARDO","end":"1938-01-17","start":"1927-06-07"
},{"name":"ALZANO LOMBARDO","start":"1938-01-18","end":"1939-09-19"},{"name"
:"ALZANO LOMBARDO","start":"1939-09-20","end":"9999-12-31"}],"F461":[{"end":
"9999-12-31","start":"1880-01-16","name":"MONTECCHIA DI CROSARA"}],"A397":[{
"name":"ARGUSTO","start":"1861-03-17","end":"9999-12-31"}],"E722":[{"name":"
LUCITO","end":"1970-03-02","start":"1861-03-17"},{"name":"LUCITO","start":"1
970-03-03","end":"9999-12-31"}],"G933":[{"end":"9999-12-31","start":"1866-11
-19","name":"POSSAGNO"}],"A994":[{"end":"1928-04-03","start":"1863-01-26","n
ame":"BORGHETTO SAN NICOL\u00d2"}],"G409":[{"name":"PEDERSANO","end":"1923-0
2-13","start":"1920-10-16"},{"end":"1929-02-20","start":"1923-02-14","name":
"PEDERSANO"}],"A596":[{"name":"BALLABIO SUPERIORE","start":"1861-03-17","end
":"1927-09-11"}],"B800":[{"start":"1861-03-17","end":"1928-02-20","name":"CA
ROBBIO"}],"D640":[{"name":"FLUSSIO","start":"1861-03-17","end":"1927-01-11"}
,{"end":"1928-02-13","start":"1927-01-12","name":"FLUSSIO"},{"end":"2005-12-
31","start":"1946-07-18","name":"FLUSSIO"},{"name":"FLUSSIO","end":"9999-12-
31","start":"2006-01-01"}],"A231":[{"name":"ALTAVILLA VICENTINA","start":"18
67-09-02","end":"9999-12-31"}],"F155":[{"end":"9999-12-31","start":"1861-03-
17","name":"MESERO"}],"H608":[{"start":"1866-11-19","end":"1909-02-11","name
":"ROVER\u00c8 DI VELO"},{"end":"9999-12-31","start":"1909-02-12","name":"RO
VER\u00c8 VERONESE"}],"E664":[{"start":"1920-10-16","end":"1923-02-13","name
":"LONA-LASES"},{"name":"LONA-LASES","start":"1923-02-14","end":"1928-09-08"
},{"name":"LONA-LASES","end":"9999-12-31","start":"1952-05-17"}],"E737":[{"n
ame":"LUMARZO","start":"1861-03-17","end":"9999-12-31"}],"A773":[{"end":"999
9-12-31","start":"1863-04-10","name":"BELVEDERE MARITTIMO"}],"E493":[{"start
":"1861-03-17","end":"9999-12-31","name":"LAVELLO"}],"G830":[{"end":"1923-02
-13","start":"1920-10-16","name":"PONTE ALL'ISARCO"},{"name":"PONTE ALL'ISAR
CO","start":"1923-02-14","end":"1927-01-11"},{"name":"PONTE ALL'ISARCO","sta
rt":"1927-01-12","end":"1938-10-16"},{"start":"1938-10-17","end":"9999-12-31
","name":"PONTE GARDENA"}],"F456":[{"start":"1863-04-13","end":"9999-12-31",
"name":"MONTE CASTELLO DI VIBIO"}],"C748":[{"name":"CITTANOVA D'ISTRIA","end
":"1923-02-07","start":"1921-01-05"},{"name":"CITTANOVA D'ISTRIA","start":"1
923-02-08","end":"1977-04-03"}],"L454":[{"name":"TRUCCAZZANO","end":"9999-12
-31","start":"1869-02-01"}],"I007":[{"name":"SAN MARTINO DEL LAGO","end":"99
99-12-31","start":"1868-01-01"}],"B073":[{"name":"BOSCO CHIESANUOVA","start"
:"1866-11-19","end":"9999-12-31"}],"H729":[{"end":"9999-12-31","start":"1894
-04-20","name":"SALVE"}],"B379":[{"name":"CALANNA","end":"9999-12-31","start
":"1861-03-17"}],"L320":[{"start":"1861-03-17","end":"1928-02-09","name":"TR
AFFIUME"}],"I434":[{"end":"9999-12-31","start":"1861-03-17","name":"SARMATO"
}],"D772":[{"name":"FRASCATA","start":"1861-03-17","end":"1928-06-18"}],"H56
2":[{"name":"ROSCIANO","end":"1927-01-11","start":"1861-03-17"},{"end":"9999
-12-31","start":"1927-01-12","name":"ROSCIANO"}],"A287":[{"name":"ANELA","st
art":"1861-03-17","end":"9999-12-31"}],"I374":[{"start":"1861-03-17","end":"
1974-08-19","name":"SANTU LUSSURGIU"},{"name":"SANTU LUSSURGIU","end":"9999-
12-31","start":"1974-08-20"}],"L158":[{"end":"9999-12-31","start":"1861-03-1
7","name":"THIESI"}],"B903":[{"name":"VILLAPIANA","start":"1863-01-19","end"
:"9999-12-31"}],"D405":[{"name":"ENDINE","end":"1928-07-28","start":"1861-03
-17"}],"B439":[{"name":"CALVATONE","start":"1861-03-17","end":"9999-12-31"}]
,"E781":[{"end":"1928-05-30","start":"1861-03-17","name":"MACCIO"}],"I977":[
{"start":"1861-03-17","end":"9999-12-31","name":"STREVI"}],"I138":[{"end":"1
923-12-05","start":"1861-03-17","name":"SANREMO"},{"name":"SANREMO","end":"1
925-08-21","start":"1923-12-06"},{"name":"SANREMO","end":"1928-04-03","start
":"1925-08-22"},{"name":"SANREMO","end":"1956-04-20","start":"1928-04-04"},{
"name":"SANREMO","start":"1956-04-21","end":"9999-12-31"}],"C148":[{"name":"
CASTELLAZZO BORMIDA","end":"1929-06-14","start":"1863-02-16"},{"name":"CASTE
LLAZZO BORMIDA","start":"1929-06-15","end":"1994-12-13"},{"name":"CASTELLAZZ
O BORMIDA","start":"1994-12-14","end":"9999-12-31"}],"G288":[{"name":"PALMI"
,"end":"9999-12-31","start":"1861-03-17"}],"H129":[{"start":"1866-11-19","en
d":"9999-12-31","name":"QUINGENTOLE"}],"I813":[{"end":"1928-04-02","start":"
1861-03-17","name":"SOLZA"},{"name":"SOLZA","end":"9999-12-31","start":"1970
-03-17"}],"G624":[{"name":"PIETRAPERZIA","end":"1927-01-11","start":"1861-03
-17"},{"name":"PIETRAPERZIA","start":"1927-01-12","end":"9999-12-31"}],"H911
":[{"end":"2016-12-31","start":"1877-07-16","name":"SAN GIOVANNI D'ASSO"}],"
D172":[{"start":"1861-03-17","end":"1928-03-12","name":"CRISSOLO"},{"start":
"1928-03-13","end":"1946-07-02","name":"CRISSOLO"},{"end":"9999-12-31","star
t":"1946-07-03","name":"CRISSOLO"}],"F224":[{"end":"1927-01-11","start":"187
9-07-28","name":"MINTURNO"},{"start":"1927-01-12","end":"1934-10-25","name":
"MINTURNO"},{"name":"MINTURNO","start":"1934-10-26","end":"1934-12-17"},{"na
me":"MINTURNO","end":"1945-04-08","start":"1934-12-18"},{"end":"9999-12-31",
"start":"1945-04-09","name":"MINTURNO"}],"A580":[{"name":"BAIANO","start":"1
861-03-17","end":"9999-12-31"}],"L758":[{"end":"9999-12-31","start":"1861-03
-17","name":"VERDUNO"}],"G762":[{"start":"1861-03-17","end":"1946-11-08","na
me":"POGGIOMARINO"},{"name":"POGGIOMARINO","end":"9999-12-31","start":"1946-
11-09"}],"C444":[{"end":"9999-12-31","start":"1862-12-29","name":"CELLE DI B
ULGHERIA"}],"F688":[{"name":"MONTORFANO","end":"9999-12-31","start":"1861-03
-17"}],"G529":[{"start":"1861-03-17","end":"1963-02-06","name":"PEZZAZE"},{"
name":"PEZZAZE","end":"9999-12-31","start":"1963-02-07"}],"E181":[{"name":"G
RIMALDO","start":"1923-04-27","end":"1927-01-11"},{"name":"GRIMALDO","end":"
1928-03-07","start":"1927-01-12"},{"end":"1923-02-13","start":"1920-10-16","
name":"VILLA IN SELVA"},{"end":"1923-04-26","start":"1923-02-14","name":"VIL
LA IN SELVA"}],"H815":[{"name":"SAN DANIELE PO","end":"9999-12-31","start":"
1955-11-18"},{"name":"SAN DANIELE RIPA PO","start":"1868-01-01","end":"1955-
11-17"}],"D928":[{"name":"GARNIGA","end":"1923-02-13","start":"1920-10-16"},
{"start":"1923-02-14","end":"1928-05-23","name":"GARNIGA"},{"name":"GARNIGA"
,"start":"1947-06-23","end":"1993-04-13"},{"end":"9999-12-31","start":"1993-
04-14","name":"GARNIGA TERME"}],"I738":[{"name":"SILVANO D'ORBA","start":"18
61-03-17","end":"9999-12-31"}],"D706":[{"name":"FORMAZZA","end":"1992-05-22"
,"start":"1861-03-17"},{"name":"FORMAZZA","start":"1992-05-23","end":"9999-1
2-31"}],"D697":[{"name":"FORESTO SPARSO","end":"9999-12-31","start":"1863-07
-13"}],"H064":[{"start":"1921-01-05","end":"1923-02-07","name":"PRIMANO"},{"
name":"PRIMANO","end":"1924-03-08","start":"1923-02-08"},{"name":"PRIMANO","
start":"1924-03-09","end":"1924-10-30"},{"end":"1947-09-15","start":"1924-10
-31","name":"PRIMANO"}],"E008":[{"name":"GIACCIANO CON BARUCHELLA","start":"
1866-11-19","end":"9999-12-31"}],"F030":[{"start":"1861-03-17","end":"9999-1
2-31","name":"MASSA LUBRENSE"}],"E122":[{"name":"GRADARA","start":"1861-03-1
7","end":"1998-09-03"},{"start":"1998-09-04","end":"9999-12-31","name":"GRAD
ARA"}],"B365":[{"end":"1928-01-21","start":"1861-03-17","name":"CAINO"},{"en
d":"9999-12-31","start":"1956-03-25","name":"CAINO"}],"D525":[{"name":"FELET
TO UMBERTO","start":"1867-09-02","end":"1928-05-13"}],"I910":[{"name":"SPINE
TE","end":"1970-03-02","start":"1861-03-17"},{"name":"SPINETE","end":"9999-1
2-31","start":"1970-03-03"}],"A340":[{"start":"1861-03-17","end":"9999-12-31
","name":"APRIGLIANO"}],"G398":[{"name":"PECETTO TORINESE","end":"1956-04-24
","start":"1861-03-17"},{"start":"1956-04-25","end":"9999-12-31","name":"PEC
ETTO TORINESE"}],"Z300":[{"name":"Namibia","start":"0000-01-01","end":"9999-
12-31"}],"F755":[{"name":"MORTASO","start":"1920-10-16","end":"1923-02-13"},
{"name":"MORTASO","end":"1928-10-29","start":"1923-02-14"}],"L685":[{"name":
"VARISELLA","end":"1927-12-23","start":"1871-01-01"},{"name":"VARISELLA","st
art":"1954-06-30","end":"9999-12-31"}],"E978":[{"start":"1871-01-15","end":"
1927-01-11","name":"MARTA"},{"start":"1927-01-12","end":"9999-12-31","name":
"MARTA"}],"A173":[{"end":"1935-04-07","start":"1861-03-17","name":"ALBUGNANO
"},{"name":"ALBUGNANO","start":"1935-04-08","end":"9999-12-31"}],"E215":[{"n
ame":"GRUARO","start":"1866-11-19","end":"9999-12-31"}],"Z512":[{"name":"Bel
ize","end":"9999-12-31","start":"0000-01-01"}],"E692":[{"start":"1887-07-22"
,"end":"9999-12-31","name":"LORIA"}],"E844":[{"start":"1861-03-17","end":"99
99-12-31","name":"MALALBERGO"}],"Z227":[{"end":"9999-12-31","start":"0000-01
-01","name":"Kuwait"}],"F803":[{"start":"1920-10-16","end":"1923-02-13","nam
e":"MULES"},{"end":"1927-01-11","start":"1923-02-14","name":"MULES"},{"end":
"1928-12-09","start":"1927-01-12","name":"MULES"}],"D127":[{"start":"1890-06
-15","end":"9999-12-31","name":"COZZO"}],"I778":[{"start":"1861-03-17","end"
:"1927-07-27","name":"SODD\u00cc"},{"name":"SODD\u00cc","start":"1979-04-20"
,"end":"9999-12-31"}],"G650":[{"end":"9999-12-31","start":"1867-02-12","name
":"PIEVE PORTO MORONE"}],"D860":[{"end":"1923-02-13","start":"1920-10-16","n
ame":"GAIS"},{"name":"GAIS","end":"1927-01-11","start":"1923-02-14"},{"name"
:"GAIS","start":"1927-01-12","end":"1928-05-28"},{"name":"GAIS","end":"9999-
12-31","start":"1928-05-29"}],"A605":[{"start":"1861-03-17","end":"9999-12-3
1","name":"BALZOLA"}],"A962":[{"end":"9999-12-31","start":"1861-03-17","name
":"BONATE SOTTO"}],"C344":[{"name":"CASTRONOVO DI SICILIA","end":"9999-12-31
","start":"1862-11-10"}],"F250":[{"name":"MISTERBIANCO","start":"1861-03-17"
,"end":"9999-12-31"}],"D132":[{"name":"CRAVAGLIANA","end":"1927-01-11","star
t":"1861-03-17"},{"name":"CRAVAGLIANA","start":"1927-01-12","end":"9999-12-3
1"}],"Z232":[{"name":"Maldive","start":"0000-01-01","end":"9999-12-31"}],"E5
35":[{"end":"1927-01-11","start":"1861-03-17","name":"LEONESSA"},{"start":"1
927-01-12","end":"9999-12-31","name":"LEONESSA"}],"E710":[{"name":"LOZZO","e
nd":"1927-01-11","start":"1861-03-17"},{"name":"LOZZO","start":"1927-01-12",
"end":"1928-08-07"}],"B166":[{"start":"1861-03-17","end":"1927-01-11","name"
:"BREZZO DI BEDERO"},{"start":"1927-01-12","end":"1928-12-07","name":"BREZZO
 DI BEDERO"},{"end":"9999-12-31","start":"1953-07-24","name":"BREZZO DI BEDE
RO"}],"F900":[{"name":"NISSORIA","end":"1927-01-11","start":"1861-03-17"},{"
name":"NISSORIA","start":"1927-01-12","end":"9999-12-31"}],"C689":[{"name":"
CIMADOLMO","end":"9999-12-31","start":"1866-11-19"}],"H244":[{"start":"1866-
11-19","end":"1968-04-05","name":"RESIUTTA"},{"end":"9999-12-31","start":"19
68-04-06","name":"RESIUTTA"}],"C986":[{"end":"9999-12-31","start":"1880-03-2
6","name":"CORBETTA"}],"B465":[{"name":"CAMBURZANO","start":"1861-03-17","en
d":"1927-01-11"},{"name":"CAMBURZANO","end":"1992-04-15","start":"1927-01-12
"},{"start":"1992-04-16","end":"9999-12-31","name":"CAMBURZANO"}],"D963":[{"
name":"GEMONIO","end":"1927-01-11","start":"1861-03-17"},{"name":"GEMONIO","
start":"1927-01-12","end":"9999-12-31"}],"G817":[{"start":"1861-03-17","end"
:"1928-03-13","name":"POMPU"},{"start":"1970-12-09","end":"1974-08-19","name
":"POMPU"},{"start":"1974-08-20","end":"9999-12-31","name":"POMPU"}],"I026":
[{"end":"1992-04-15","start":"1863-07-13","name":"SAN MAURO MARCHESATO"},{"n
ame":"SAN MAURO MARCHESATO","end":"9999-12-31","start":"1992-04-16"}],"H699"
:[{"end":"1927-11-23","start":"1861-03-17","name":"SALE MARASINO"},{"end":"1
947-05-20","start":"1927-11-24","name":"SALE MARASINO"},{"name":"SALE MARASI
NO","start":"1947-05-21","end":"9999-12-31"}],"H996":[{"name":"SAN MARTINO D
I VENEZZE","end":"9999-12-31","start":"1867-07-22"}],"E938":[{"end":"1923-02
-13","start":"1920-10-16","name":"MAREBBE"},{"name":"MAREBBE","start":"1923-
02-14","end":"1927-01-11"},{"name":"MAREBBE","start":"1927-01-12","end":"192
9-02-13"},{"start":"1929-02-14","end":"9999-12-31","name":"MAREBBE"}],"G194"
:[{"name":"OTTOBIANO","start":"1861-03-17","end":"9999-12-31"}],"A133":[{"en
d":"1945-08-31","start":"1928-05-23","name":"ALBANOVA"},{"start":"1945-09-01
","end":"1946-03-25","name":"ALBANOVA"}],"E177":[{"end":"1928-01-21","start"
:"1861-03-17","name":"GRIGNASCO"},{"name":"GRIGNASCO","start":"1928-01-22","
end":"1933-01-31"},{"start":"1933-02-01","end":"9999-12-31","name":"GRIGNASC
O"}],"L312":[{"start":"1928-07-29","end":"9999-12-31","name":"TOSCOLANO-MADE
RNO"}],"H594":[{"name":"ROTZO","end":"1940-09-12","start":"1866-11-19"},{"st
art":"1940-09-13","end":"9999-12-31","name":"ROTZO"}],"F149":[{"name":"MERLI
NO","start":"1861-03-17","end":"1992-04-15"},{"end":"9999-12-31","start":"19
92-04-16","name":"MERLINO"}],"L500":[{"end":"1998-09-03","start":"1861-03-17
","name":"URBINO"},{"start":"1998-09-04","end":"9999-12-31","name":"URBINO"}
],"B151":[{"name":"BRENTINO","end":"1928-10-16","start":"1866-11-19"}],"C584
":[{"name":"CETARA","end":"9999-12-31","start":"1861-03-17"}],"G498":[{"name
":"PESCOROCCHIANO","start":"1861-03-17","end":"1927-01-11"},{"name":"PESCORO
CCHIANO","start":"1927-01-12","end":"9999-12-31"}],"Z400":[{"start":"0000-01
-01","end":"9999-12-31","name":"Bermuda"}],"F070":[{"name":"MAZZO DI VALTELL
INA","end":"9999-12-31","start":"1863-07-13"}],"A440":[{"name":"ARSAGO","sta
rt":"1861-03-17","end":"1929-07-01"},{"start":"1929-07-02","end":"9999-12-31
","name":"ARZAGO D'ADDA"}],"A066":[{"name":"AGARO","start":"1861-03-17","end
":"1928-10-16"}],"G003":[{"end":"9999-12-31","start":"1861-03-17","name":"OF
FAGNA"}],"G309":[{"end":"9999-12-31","start":"1871-06-20","name":"VILLARICCA
"}],"A271":[{"name":"ANCONA","start":"1861-03-17","end":"1925-10-15"},{"end"
:"1928-05-21","start":"1925-10-16","name":"ANCONA"},{"end":"1948-04-09","sta
rt":"1928-05-22","name":"ANCONA"},{"name":"ANCONA","end":"9999-12-31","start
":"1948-04-10"}],"H883":[{"name":"SAN GIORGIO BIGARELLO","start":"2019-01-01
","end":"9999-12-31"},{"name":"SAN GIORGIO DI MANTOVA","start":"1867-06-24",
"end":"1925-03-20"},{"start":"1925-03-21","end":"1943-02-23","name":"SAN GIO
RGIO DI MANTOVA"},{"name":"SAN GIORGIO DI MANTOVA","end":"1980-06-14","start
":"1943-02-24"},{"name":"SAN GIORGIO DI MANTOVA","start":"1980-06-15","end":
"2018-12-31"}],"M054":[{"name":"VIMOGNO","start":"1861-03-17","end":"1927-12
-24"}],"C893":[{"end":"9999-12-31","start":"1861-03-17","name":"COLOGNE"}],"
F361":[{"name":"MONGARDINO","start":"1861-03-17","end":"1935-04-07"},{"name"
:"MONGARDINO","end":"9999-12-31","start":"1935-04-08"}],"A497":[{"name":"AUR
ANO","start":"1861-03-17","end":"1992-05-22"},{"end":"9999-12-31","start":"1
992-05-23","name":"AURANO"}],"G794":[{"name":"POLLEIN","end":"1927-01-11","s
tart":"1861-03-17"},{"name":"POLLEIN","start":"1927-01-12","end":"1928-03-20
"},{"end":"9999-12-31","start":"1946-05-15","name":"POLLEIN"}],"I981":[{"nam
e":"STRONCONE","start":"1861-03-17","end":"1927-01-11"},{"end":"1927-01-12",
"start":"1927-01-12","name":"STRONCONE"},{"start":"1947-09-24","end":"9999-1
2-31","name":"STRONCONE"}],"E777":[{"start":"1861-03-17","end":"1992-04-15",
"name":"MACCASTORNA"},{"start":"1992-04-16","end":"9999-12-31","name":"MACCA
STORNA"}],"A733":[{"end":"1928-01-07","start":"1861-03-17","name":"BEE"},{"n
ame":"BEE","start":"1948-01-17","end":"1992-05-22"},{"name":"BEE","start":"1
992-05-23","end":"9999-12-31"}],"E099":[{"end":"1920-07-15","start":"1864-03
-11","name":"GORLA PRIMO"},{"end":"1923-10-13","start":"1920-07-16","name":"
GORLAPRECOTTO"}],"E393":[{"name":"LABRO","start":"1861-03-17","end":"1923-03
-18"},{"name":"LABRO","end":"1927-01-11","start":"1923-03-19"},{"start":"192
7-01-12","end":"1957-04-06","name":"LABRO"},{"name":"LABRO","start":"1957-04
-07","end":"9999-12-31"}],"A818":[{"start":"1869-02-24","end":"2009-06-29","
name":"BESANA IN BRIANZA"},{"end":"9999-12-31","start":"2009-06-30","name":"
BESANA IN BRIANZA"}],"F749":[{"name":"MORROVALLE","start":"1861-03-17","end"
:"9999-12-31"}],"L207":[{"name":"TORANO NUOVO","start":"1863-07-13","end":"9
999-12-31"}],"B751":[{"end":"1928-02-04","start":"1861-03-17","name":"CARCIN
A"}],"I885":[{"end":"1927-01-11","start":"1861-03-17","name":"SPARANISE"},{"
end":"1945-08-31","start":"1927-01-12","name":"SPARANISE"},{"name":"SPARANIS
E","start":"1945-09-01","end":"9999-12-31"}],"G973":[{"start":"1861-03-17","
end":"1934-11-22","name":"PRAGELATO"},{"start":"1934-11-23","end":"9999-12-3
1","name":"PRAGELATO"}],"L613":[{"start":"1861-03-17","end":"1974-08-19","na
me":"VALLERMOSA"},{"start":"1974-08-20","end":"2016-04-27","name":"VALLERMOS
A"},{"start":"2016-04-28","end":"9999-12-31","name":"VALLERMOSA"}],"L957":[{
"name":"VILLA LAGARINA","start":"1920-10-16","end":"1923-02-13"},{"name":"VI
LLA LAGARINA","start":"1923-02-14","end":"1929-02-20"},{"end":"1955-03-01","
start":"1929-02-21","name":"VILLA LAGARINA"},{"name":"VILLA LAGARINA","start
":"1955-03-02","end":"1967-08-29"},{"end":"9999-12-31","start":"1967-08-30",
"name":"VILLA LAGARINA"}],"C990":[{"name":"CORCIANO","end":"9999-12-31","sta
rt":"1861-03-17"}],"E283":[{"name":"ILBONO","start":"1878-11-23","end":"1927
-01-11"},{"end":"1958-08-14","start":"1927-01-12","name":"ILBONO"},{"start":
"1958-08-15","end":"2005-12-31","name":"ILBONO"},{"name":"ILBONO","end":"201
6-04-27","start":"2006-01-01"},{"name":"ILBONO","start":"2016-04-28","end":"
9999-12-31"}],"I334":[{"name":"SANT'EUFEMIA LAMEZIA","end":"1968-11-27","sta
rt":"1935-06-06"}],"A051":[{"name":"ACQUAVIVA D'ISERNIA","start":"1861-03-17
","end":"1928-05-31"},{"name":"ACQUAVIVA D'ISERNIA","start":"1946-04-26","en
d":"1970-03-02"},{"name":"ACQUAVIVA D'ISERNIA","start":"1970-03-03","end":"9
999-12-31"}],"B479":[{"start":"1861-03-17","end":"1923-07-22","name":"CAMINA
TA"},{"name":"CAMINATA","start":"1923-07-23","end":"1929-02-13"},{"name":"CA
MINATA","end":"2017-12-31","start":"1950-02-19"}],"I937":[{"name":"STALETT\u
00cc","end":"9999-12-31","start":"1861-03-17"}],"L420":[{"name":"TRICERRO","
start":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12
-31","name":"TRICERRO"}],"D727":[{"start":"1863-07-13","end":"1964-10-09","n
ame":"FORNOVO DI SAN GIOVANNI"},{"name":"FORNOVO SAN GIOVANNI","start":"1964
-10-10","end":"9999-12-31"}],"I178":[{"start":"1863-07-13","end":"1927-01-11
","name":"SANTA CROCE CAMERINA"},{"name":"SANTA CROCE CAMERINA","end":"9999-
12-31","start":"1927-01-12"}],"F356":[{"end":"1923-02-07","start":"1921-01-0
5","name":"MONFALCONE"},{"start":"1923-02-08","end":"1948-01-09","name":"MON
FALCONE"},{"start":"1948-01-10","end":"9999-12-31","name":"MONFALCONE"}],"L3
54":[{"name":"TRECASALI","start":"1867-06-03","end":"2013-12-31"}],"M274":[{
"name":"GOLFO ARANCI","end":"2005-12-31","start":"1979-04-20"},{"start":"200
6-01-01","end":"2016-04-27","name":"GOLFO ARANCI"},{"end":"9999-12-31","star
t":"2016-04-28","name":"GOLFO ARANCI"}],"H980":[{"name":"SAN MARCELLO PISTOI
ESE","end":"1927-01-11","start":"1863-04-13"},{"end":"1979-11-16","start":"1
927-01-12","name":"SAN MARCELLO PISTOIESE"},{"name":"SAN MARCELLO PISTOIESE"
,"end":"2016-12-31","start":"1979-11-17"}],"F017":[{"name":"MASLIANICO","sta
rt":"1861-03-17","end":"9999-12-31"}],"B033":[{"name":"BORGO SAN DALMAZZO","
start":"1861-03-17","end":"9999-12-31"}],"A511":[{"name":"AVERARA","end":"19
78-02-11","start":"1861-03-17"},{"start":"1978-02-12","end":"9999-12-31","na
me":"AVERARA"}],"B339":[{"name":"CA' DE' STEFANI","start":"1868-08-22","end"
:"1927-11-23"}],"C082":[{"end":"1970-03-02","start":"1861-03-17","name":"CAS
TEL DEL GIUDICE"},{"start":"1970-03-03","end":"9999-12-31","name":"CASTEL DE
L GIUDICE"}],"G870":[{"end":"1957-01-23","start":"1861-03-17","name":"PONTRE
MOLI"},{"name":"PONTREMOLI","end":"9999-12-31","start":"1957-01-24"}],"D305"
:[{"start":"1861-03-17","end":"1975-05-27","name":"DISO"},{"name":"DISO","en
d":"9999-12-31","start":"1975-05-28"}],"I922":[{"name":"SPOLTORE","end":"192
7-01-11","start":"1861-03-17"},{"name":"SPOLTORE","start":"1927-01-12","end"
:"1928-03-23"},{"name":"SPOLTORE","end":"2006-07-12","start":"1947-08-30"},{
"name":"SPOLTORE","end":"9999-12-31","start":"2006-07-13"}],"F557":[{"name":
"SANT'ARCANGELO TRIMONTE","end":"1978-08-01","start":"1864-11-07"},{"name":"
SANT'ARCANGELO TRIMONTE","end":"9999-12-31","start":"1978-08-02"}],"D732":[{
"start":"1861-03-17","end":"9999-12-31","name":"FORTUNAGO"}],"E110":[{"end":
"1928-04-06","start":"1861-03-17","name":"GORRINO"}],"I474":[{"start":"1861-
03-17","end":"1910-05-19","name":"SAVIGNO"},{"start":"1910-05-20","end":"201
3-12-31","name":"SAVIGNO"}],"B766":[{"name":"CARERI","start":"1861-03-17","e
nd":"9999-12-31"}],"H092":[{"name":"PUOS D'ALPAGO","start":"1866-11-19","end
":"2016-02-22"}],"E526":[{"name":"LENO","start":"1861-03-17","end":"1927-12-
08"},{"name":"LENO","start":"1927-12-09","end":"9999-12-31"}],"F518":[{"end"
:"1929-04-19","start":"1861-03-17","name":"MONTEGIOCO"},{"name":"MONTEGIOCO"
,"start":"1929-04-20","end":"9999-12-31"}],"H751":[{"name":"SAMO DI CALABRIA
","start":"1928-07-28","end":"1946-03-09"}],"A610":[{"start":"1861-03-17","e
nd":"1928-08-06","name":"BANNIO"},{"name":"BANNIO","start":"1928-08-07","end
":"1929-03-23"},{"end":"1992-05-22","start":"1929-03-24","name":"BANNIO ANZI
NO"},{"end":"9999-12-31","start":"1992-05-23","name":"BANNIO ANZINO"}],"E332
":[{"end":"9999-12-31","start":"1861-03-17","name":"ISCHITELLA"}],"C160":[{"
name":"CASTELLETTO MERLI","start":"1861-03-17","end":"1929-05-13"},{"start":
"1929-05-14","end":"9999-12-31","name":"CASTELLETTO MERLI"}],"E705":[{"name"
:"LOVERO","start":"1861-03-17","end":"9999-12-31"}],"D481":[{"end":"1992-05-
22","start":"1861-03-17","name":"FALMENTA"},{"end":"2018-12-31","start":"199
2-05-23","name":"FALMENTA"}],"C827":[{"name":"COIMO","start":"1861-03-17","e
nd":"1928-06-28"}],"F058":[{"name":"MATTIE","start":"1861-03-17","end":"1943
-06-02"},{"name":"MATTIE","start":"1943-06-03","end":"9999-12-31"}],"E327":[
{"start":"1861-03-17","end":"1928-03-28","name":"ISASCA"},{"name":"ISASCA","
start":"1947-11-26","end":"9999-12-31"}],"Z116":[{"start":"0000-01-01","end"
:"9999-12-31","name":"Irlanda"}],"D216":[{"name":"CUREGGIO","end":"9999-12-3
1","start":"1861-03-17"}],"E924":[{"end":"9999-12-31","start":"1910-02-17","
name":"MARCELLINA"}],"L918":[{"name":"VILLA D'ADIGE","end":"1928-04-06","sta
rt":"1867-07-22"}],"A792":[{"name":"BEREGUARDO","start":"1872-10-15","end":"
9999-12-31"}],"M355":[{"end":"9999-12-31","start":"2016-01-01","name":"CEMBR
A LISIGNAGO"}],"L275":[{"start":"1861-03-17","end":"1927-01-11","name":"TORR
E ORSINA"},{"name":"TORRE ORSINA","start":"1927-01-12","end":"1927-01-12"}],
"C832":[{"name":"COLCIAGO","end":"1883-09-25","start":"1861-03-17"}],"E472":
[{"start":"1945-04-24","end":"9999-12-31","name":"LATINA"},{"end":"1934-10-2
5","start":"1932-11-06","name":"LITTORIA"},{"start":"1934-10-26","end":"1935
-05-01","name":"LITTORIA"},{"name":"LITTORIA","start":"1935-05-02","end":"19
45-04-23"}],"B883":[{"start":"1863-04-23","end":"1928-05-22","name":"CASALEG
GIO NOVARA"},{"name":"CASALEGGIO NOVARA","end":"9999-12-31","start":"1948-01
-15"}],"D004":[{"name":"CORIANO","start":"1861-03-17","end":"1992-04-15"},{"
name":"CORIANO","start":"1992-04-16","end":"9999-12-31"}],"I750":[{"start":"
1861-03-17","end":"9999-12-31","name":"SINIO"}],"H766":[{"name":"SAN BASILIO
","end":"1928-07-27","start":"1861-03-17"},{"end":"1946-02-06","start":"1928
-07-28","name":"SAN BASILIO"},{"name":"SAN BASILIO","start":"1946-02-07","en
d":"1974-08-19"},{"end":"2016-04-27","start":"1974-08-20","name":"SAN BASILI
O"},{"name":"SAN BASILIO","end":"9999-12-31","start":"2016-04-28"}],"D576":[
{"name":"FIESSE","end":"1929-02-20","start":"1861-03-17"},{"name":"FIESSE","
end":"9999-12-31","start":"1950-08-03"}],"G678":[{"name":"PINO TORINESE","st
art":"1861-03-17","end":"9999-12-31"}],"B092":[{"name":"BOTTICINO MATTINA","
start":"1861-03-17","end":"1928-03-28"}],"F811":[{"name":"MURELLO","end":"99
99-12-31","start":"1861-03-17"}],"L730":[{"name":"VENDONE","start":"1861-03-
17","end":"1927-01-11"},{"name":"VENDONE","start":"1927-01-12","end":"9999-1
2-31"}],"D437":[{"name":"ESINO SUPERIORE","start":"1861-03-17","end":"1927-1
0-09"}],"H339":[{"name":"RIVAROLO LIGURE","start":"1869-10-11","end":"1926-0
2-05"}],"H033":[{"name":"PREMIA","end":"1928-10-15","start":"1861-03-17"},{"
name":"PREMIA","start":"1928-10-16","end":"1929-05-06"},{"name":"PREMIA","en
d":"1937-08-19","start":"1929-05-07"},{"name":"PREMIA","end":"1992-05-22","s
tart":"1937-08-20"},{"end":"9999-12-31","start":"1992-05-23","name":"PREMIA"
}],"A857":[{"name":"BIEDA","start":"1871-01-15","end":"1927-01-11"},{"start"
:"1927-01-12","end":"1927-10-18","name":"BIEDA"},{"start":"1927-10-19","end"
:"1945-09-14","name":"BIEDA"},{"start":"1945-09-15","end":"1952-10-06","name
":"BIEDA"},{"end":"9999-12-31","start":"1952-10-07","name":"BLERA"}],"D793":
[{"name":"FRAZZAN\u00d2","start":"1861-03-17","end":"9999-12-31"}],"C453":[{
"end":"9999-12-31","start":"1861-03-17","name":"CENADI"}],"L515":[{"name":"U
SSEAUX","start":"1861-03-17","end":"1927-11-05"},{"end":"9999-12-31","start"
:"1948-05-07","name":"USSEAUX"}],"F278":[{"start":"1863-01-19","end":"1928-1
2-29","name":"MOIO DELLA CIVITELLA"},{"name":"MOIO DELLA CIVITELLA","start":
"1945-07-19","end":"9999-12-31"}],"D377":[{"name":"DUBINO","start":"1861-03-
17","end":"1928-11-29"},{"name":"DUBINO","end":"9999-12-31","start":"1928-11
-30"}],"H479":[{"name":"RODI","end":"1927-03-23","start":"1923-11-28"},{"nam
e":"RODI","end":"1948-12-24","start":"1947-06-17"},{"start":"1948-12-25","en
d":"9999-12-31","name":"ROD\u00cc MILICI"}],"I035":[{"end":"9999-12-31","sta
rt":"1862-12-29","name":"SAN MICHELE DI GANZARIA"}],"B980":[{"name":"CASOLA 
DI NAPOLI","start":"1863-01-19","end":"9999-12-31"}],"F915":[{"end":"9999-12
-31","start":"1861-03-17","name":"NOCI"}],"A368":[{"start":"1863-12-28","end
":"1928-11-03","name":"SAN NICOL\u00d2 D'ARCIDANO"},{"name":"SAN NICOL\u00d2
 D'ARCIDANO","end":"1974-08-19","start":"1947-01-24"},{"start":"1974-08-20",
"end":"9999-12-31","name":"SAN NICOL\u00d2 D'ARCIDANO"}],"L055":[{"name":"TA
RSIA","end":"9999-12-31","start":"1861-03-17"}],"D422":[{"name":"ERCHIE","st
art":"1861-03-17","end":"1927-01-11"},{"name":"ERCHIE","start":"1927-01-12",
"end":"9999-12-31"}],"D974":[{"name":"GERA","start":"1861-03-17","end":"1929
-07-02"},{"name":"GERA LARIO","start":"1929-07-03","end":"9999-12-31"}],"I15
0":[{"end":"1928-12-18","start":"1863-02-23","name":"SAN SEBASTIANO CURONE"}
,{"name":"SAN SEBASTIANO CURONE","end":"1947-09-23","start":"1928-12-19"},{"
name":"SAN SEBASTIANO CURONE","start":"1947-09-24","end":"9999-12-31"}],"G96
4":[{"start":"1861-03-17","end":"1919-02-02","name":"POZZUOLI"},{"end":"1927
-11-07","start":"1919-02-03","name":"POZZUOLI"},{"end":"9999-12-31","start":
"1927-11-08","name":"POZZUOLI"}],"H166":[{"name":"RAIANO","start":"1861-03-1
7","end":"9999-12-31"}],"L130":[{"end":"1928-11-28","start":"1861-03-17","na
me":"TERRASA"}],"G367":[{"start":"1861-03-17","end":"9999-12-31","name":"PAS
TURANA"}],"C045":[{"name":"CASTAGNETO","end":"1913-07-14","start":"1895-12-0
9"},{"name":"CASTAGNETO PO","start":"1913-07-15","end":"1939-02-04"},{"end":
"1948-05-06","start":"1939-02-05","name":"CASTAGNETO PO"},{"end":"9999-12-31
","start":"1948-05-07","name":"CASTAGNETO PO"}],"D193":[{"name":"CUCCA","sta
rt":"1866-11-19","end":"1902-02-08"},{"end":"1981-04-20","start":"1902-02-09
","name":"VERONELLA"},{"end":"9999-12-31","start":"1981-04-21","name":"VERON
ELLA"}],"L601":[{"start":"1929-05-15","end":"9999-12-31","name":"VALLE DI CA
SIES"}],"I328":[{"name":"SAN TEODORO","end":"1929-01-23","start":"1861-03-17
"},{"name":"SAN TEODORO","end":"9999-12-31","start":"1940-08-29"}],"E406":[{
"name":"LAGNASCO","end":"9999-12-31","start":"1861-03-17"}],"L269":[{"end":"
9999-12-31","start":"1872-11-01","name":"TORRE D'ISOLA"}],"E633":[{"start":"
1861-03-17","end":"1935-04-07","name":"LOAZZOLO"},{"name":"LOAZZOLO","start"
:"1935-04-08","end":"9999-12-31"}],"A677":[{"end":"1974-08-19","start":"1861
-03-17","name":"BARRALI"},{"name":"BARRALI","start":"1974-08-20","end":"2016
-04-27"},{"start":"2016-04-28","end":"9999-12-31","name":"BARRALI"}],"L443":
[{"name":"TROBASO","end":"1927-12-03","start":"1861-03-17"}],"G858":[{"name"
:"PONTESTURA","start":"1861-03-17","end":"1928-06-26"},{"name":"PONTESTURA",
"start":"1928-06-27","end":"1948-01-14"},{"end":"9999-12-31","start":"1948-0
1-15","name":"PONTESTURA"}],"D668":[{"name":"FONTANA ELICE","end":"1910-02-1
0","start":"1884-06-18"},{"start":"1910-02-11","end":"1911-10-31","name":"FO
NTANA ELICE"},{"name":"FONTANELICE","end":"9999-12-31","start":"1911-11-01"}
],"G566":[{"name":"PIAN DEL VOGLIO","start":"1862-10-31","end":"1924-12-11"}
,{"start":"1924-12-12","end":"2004-10-25","name":"SAN BENEDETTO VAL DI SAMBR
O"},{"start":"2004-10-26","end":"9999-12-31","name":"SAN BENEDETTO VAL DI SA
MBRO"}],"A503":[{"end":"1927-01-11","start":"1861-03-17","name":"AUSTIS"},{"
name":"AUSTIS","start":"1927-01-12","end":"9999-12-31"}],"I694":[{"name":"SE
STRI PONENTE","start":"1861-03-17","end":"1923-11-28"},{"start":"1923-11-29"
,"end":"1926-02-05","name":"SESTRI PONENTE"}],"I842":[{"end":"1923-03-18","s
tart":"1861-03-17","name":"SORBANO"},{"name":"SORBANO","start":"1923-03-19",
"end":"1965-01-16"}],"D310":[{"name":"DIZZASCO","end":"9999-12-31","start":"
1861-03-17"}],"H151":[{"start":"1861-03-17","end":"9999-12-31","name":"RACCU
JA"}],"I463":[{"name":"SATRIANO","start":"1861-03-17","end":"9999-12-31"}],"
C760":[{"name":"CIVIDATE CAMUNO","end":"1928-05-21","start":"1887-02-14"},{"
name":"CIVIDATE CAMUNO","start":"1947-11-04","end":"9999-12-31"}],"D502":[{"
end":"9999-12-31","start":"1980-06-10","name":"FARINI"},{"end":"1927-01-26",
"start":"1868-01-01","name":"FARINI D'OLMO"},{"end":"1980-06-09","start":"19
27-01-27","name":"FARINI D'OLMO"}],"E105":[{"start":"1861-03-17","end":"1927
-01-11","name":"GORNATE-SUPERIORE"},{"end":"1928-01-11","start":"1927-01-12"
,"name":"GORNATE-SUPERIORE"}],"B594":[{"name":"CANELLI","end":"1935-04-07","
start":"1861-03-17"},{"end":"9999-12-31","start":"1935-04-08","name":"CANELL
I"}],"I249":[{"name":"SANTA MARIA MAGGIORE","end":"1928-12-13","start":"1928
-06-28"},{"start":"1928-12-14","end":"1992-05-22","name":"SANTA MARIA MAGGIO
RE"},{"end":"9999-12-31","start":"1992-05-23","name":"SANTA MARIA MAGGIORE"}
],"G011":[{"name":"OGLIASTRO CILENTO","end":"9999-12-31","start":"1862-12-29
"}],"Z716":[{"name":"Nuova Caledonia","end":"9999-12-31","start":"0000-01-01
"}],"B996":[{"name":"CASSAGO BRIANZA","end":"1992-04-15","start":"1927-06-07
"},{"start":"1992-04-16","end":"9999-12-31","name":"CASSAGO BRIANZA"}],"M263
":[{"name":"PORTO CESAREO","end":"9999-12-31","start":"1975-06-01"}],"B244":
[{"name":"BUCUIE","end":"1923-02-07","start":"1921-01-05"},{"name":"BUCUIE",
"end":"1947-09-15","start":"1923-02-08"}],"A192":[{"end":"9999-12-31","start
":"1861-03-17","name":"ALGHERO"}],"I599":[{"end":"9999-12-31","start":"1861-
03-17","name":"SEMIANA"}],"G551":[{"name":"PIANDIMELETO","end":"1958-03-28",
"start":"1861-03-17"},{"end":"1998-09-03","start":"1958-03-29","name":"PIAND
IMELETO"},{"name":"PIANDIMELETO","start":"1998-09-04","end":"9999-12-31"}],"
H465":[{"start":"1861-03-17","end":"1929-05-27","name":"ROCCHETTA LIGURE"},{
"start":"1929-05-28","end":"1947-09-03","name":"ROCCHETTA LIGURE"},{"name":"
ROCCHETTA LIGURE","end":"1955-11-25","start":"1947-09-04"},{"end":"9999-12-3
1","start":"1955-11-26","name":"ROCCHETTA LIGURE"}],"A637":[{"start":"1866-1
1-19","end":"9999-12-31","name":"BARBONA"}],"E673":[{"start":"1861-03-17","e
nd":"1927-11-23","name":"LONGHENA"},{"end":"9999-12-31","start":"1947-11-26"
,"name":"LONGHENA"}],"A764":[{"end":"9999-12-31","start":"1864-03-11","name"
:"BELMONTE MEZZAGNO"}],"G467":[{"name":"PERSICETO","end":"1928-10-06","start
":"1912-08-24"},{"name":"SAN GIOVANNI IN PERSICETO","start":"1861-03-17","en
d":"1912-08-23"},{"name":"SAN GIOVANNI IN PERSICETO","end":"9999-12-31","sta
rt":"1928-10-07"}],"C806":[{"name":"COCCAGLIO","end":"9999-12-31","start":"1
861-03-17"}],"F880":[{"start":"1871-01-15","end":"1934-10-25","name":"NETTUN
O"},{"name":"NETTUNO","end":"1939-12-11","start":"1934-10-26"},{"name":"NETT
UNO","end":"9999-12-31","start":"1945-06-05"}],"I124":[{"name":"SAN POLO DI 
PIAVE","start":"1867-11-25","end":"9999-12-31"}],"C265":[{"start":"1862-10-3
1","end":"1959-05-30","name":"CASTEL SAN PIETRO DELL'EMILIA"},{"end":"9999-1
2-31","start":"1959-05-31","name":"CASTEL SAN PIETRO TERME"}],"E507":[{"star
t":"1861-03-17","end":"1924-01-10","name":"LECCO"},{"name":"LECCO","start":"
1924-01-11","end":"1928-07-17"},{"end":"1992-04-15","start":"1928-07-18","na
me":"LECCO"},{"name":"LECCO","end":"9999-12-31","start":"1992-04-16"}],"G096
":[{"name":"PIEVE FISSIRAGA","start":"1879-05-26","end":"1978-02-11"},{"name
":"PIEVE FISSIRAGA","start":"1978-02-12","end":"1992-04-15"},{"start":"1992-
04-16","end":"9999-12-31","name":"PIEVE FISSIRAGA"}],"L343":[{"name":"TRAVER
SA","end":"1928-04-30","start":"1861-03-17"}],"B911":[{"name":"CASALROMANO",
"end":"9999-12-31","start":"1868-07-01"}],"L049":[{"name":"TARANTO","end":"1
919-11-28","start":"1861-03-17"},{"end":"1923-12-21","start":"1919-11-29","n
ame":"TARANTO"},{"start":"1923-12-22","end":"1933-06-22","name":"TARANTO"},{
"name":"TARANTO","start":"1933-06-23","end":"1993-04-30"},{"name":"TARANTO",
"start":"1993-05-01","end":"2000-02-10"},{"start":"2000-02-11","end":"2006-0
3-09","name":"TARANTO"},{"name":"TARANTO","start":"2006-03-10","end":"9999-1
2-31"}],"L170":[{"end":"1928-07-28","start":"1861-03-17","name":"TIMOLINE"}]
,"I428":[{"name":"SARDARA","end":"1974-08-19","start":"1861-03-17"},{"name":
"SARDARA","start":"1974-08-20","end":"2005-12-31"},{"name":"SARDARA","start"
:"2006-01-01","end":"2016-04-27"},{"end":"9999-12-31","start":"2016-04-28","
name":"SARDARA"}],"E306":[{"name":"INTROD","start":"1861-03-17","end":"1927-
01-11"},{"start":"1927-01-12","end":"1928-05-12","name":"INTROD"},{"end":"99
99-12-31","start":"1946-03-10","name":"INTROD"}],"B129":[{"end":"1928-06-26"
,"start":"1868-08-19","name":"CASTELVERDE"},{"name":"CASTELVERDE","end":"999
9-12-31","start":"1928-06-27"}],"D623":[{"start":"1862-12-29","end":"9999-12
-31","name":"FIUMEFREDDO DI SICILIA"}],"A681":[{"end":"1928-02-08","start":"
1861-03-17","name":"BARUMINI"},{"start":"1928-02-09","end":"1947-01-23","nam
e":"BARUMINI"},{"start":"1947-01-24","end":"1974-08-19","name":"BARUMINI"},{
"name":"BARUMINI","start":"1974-08-20","end":"2005-12-31"},{"name":"BARUMINI
","end":"2016-04-27","start":"2006-01-01"},{"name":"BARUMINI","end":"9999-12
-31","start":"2016-04-28"}],"I363":[{"end":"1923-12-21","start":"1863-04-23"
,"name":"SANTO STEFANO DI MAGRA"},{"start":"1923-12-22","end":"9999-12-31","
name":"SANTO STEFANO DI MAGRA"}],"I069":[{"name":"SAN PANTALEO","end":"1905-
07-10","start":"1861-03-17"}],"F983":[{"end":"1974-08-19","start":"1868-02-1
8","name":"NURAMINIS"},{"name":"NURAMINIS","start":"1974-08-20","end":"2016-
04-27"},{"end":"9999-12-31","start":"2016-04-28","name":"NURAMINIS"}],"D410"
:[{"end":"1947-09-15","start":"1940-05-07","name":"ENTRACQUE"},{"start":"194
7-09-16","end":"9999-12-31","name":"ENTRACQUE"},{"end":"1940-05-06","start":
"1861-03-17","name":"ENTRAQUE"}],"L717":[{"start":"1883-08-10","end":"1928-0
1-25","name":"VELLANO"},{"name":"VELLANO","end":"1929-02-13","start":"1928-0
1-26"}],"C652":[{"start":"1923-04-27","end":"1927-01-11","name":"CHIUSA"},{"
name":"CHIUSA","start":"1927-01-12","end":"1929-06-03"},{"name":"CHIUSA","en
d":"1960-11-22","start":"1929-06-04"},{"end":"9999-12-31","start":"1960-11-2
3","name":"CHIUSA"},{"end":"1923-02-13","start":"1920-10-16","name":"CHIUSA 
ALL'ISARCO"},{"end":"1923-04-26","start":"1923-02-14","name":"CHIUSA ALL'ISA
RCO"}],"H365":[{"name":"ROASIO","end":"1927-01-11","start":"1861-03-17"},{"e
nd":"1929-06-03","start":"1927-01-12","name":"ROASIO"},{"start":"1929-06-04"
,"end":"1947-12-21","name":"ROASIO"},{"name":"ROASIO","end":"9999-12-31","st
art":"1947-12-22"}],"M349":[{"name":"ALTAVALLE","end":"9999-12-31","start":"
2016-01-01"}],"M043":[{"name":"VILLESSE","start":"1921-01-05","end":"1923-02
-07"},{"name":"VILLESSE","end":"1927-01-11","start":"1923-02-08"},{"end":"19
28-05-01","start":"1927-01-12","name":"VILLESSE"},{"name":"VILLESSE","end":"
9999-12-31","start":"1954-06-30"}],"L583":[{"name":"VALGUARNERA CAROPEPE","s
tart":"1861-03-17","end":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-
12","name":"VALGUARNERA CAROPEPE"}],"B064":[{"name":"BORUTTA","start":"1861-
03-17","end":"9999-12-31"}],"A029":[{"start":"1861-03-17","end":"1951-05-05"
,"name":"ACI SANT'ANTONIO"},{"end":"9999-12-31","start":"1951-05-06","name":
"ACI SANT'ANTONIO"}],"A323":[{"end":"1939-12-11","start":"1871-01-15","name"
:"ANZIO"},{"start":"1945-06-05","end":"9999-12-31","name":"ANZIO"}],"B608":[
{"name":"CANNALONGA","start":"1861-03-17","end":"1928-12-29"},{"name":"CANNA
LONGA","end":"9999-12-31","start":"1946-04-02"}],"I255":[{"start":"1949-10-1
6","end":"9999-12-31","name":"SANTA MARINELLA"}],"D848":[{"end":"1927-01-11"
,"start":"1861-03-17","name":"GAGLIANICO"},{"end":"1992-04-15","start":"1927
-01-12","name":"GAGLIANICO"},{"name":"GAGLIANICO","start":"1992-04-16","end"
:"9999-12-31"}],"L642":[{"name":"VALNEGRA","start":"1861-03-17","end":"1927-
09-13"},{"start":"1956-08-26","end":"1978-02-11","name":"VALNEGRA"},{"name":
"VALNEGRA","end":"9999-12-31","start":"1978-02-12"}],"L894":[{"start":"1861-
03-17","end":"9999-12-31","name":"VIGOLO"}],"C279":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"CASTELVECCHIO SUBEQUO"}],"F896":[{"end":"1928-07-28
","start":"1861-03-17","name":"NIGOLINE"}],"L235":[{"name":"TORRALBA","start
":"1861-03-17","end":"9999-12-31"}],"H800":[{"name":"SAN CIPRIANO PICENTINO"
,"start":"1862-11-07","end":"1928-03-06"},{"name":"SAN CIPRIANO PICENTINO","
start":"1928-03-07","end":"1929-03-04"},{"end":"1946-04-25","start":"1929-03
-05","name":"SAN CIPRIANO PICENTINO"},{"name":"SAN CIPRIANO PICENTINO","end"
:"9999-12-31","start":"1946-04-26"}],"C872":[{"end":"1923-02-13","start":"19
20-10-16","name":"COLLE SANTA LUCIA"},{"end":"9999-12-31","start":"1923-02-1
4","name":"COLLE SANTA LUCIA"}],"E432":[{"name":"LAMPORECCHIO","start":"1897
-07-01","end":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":
"LAMPORECCHIO"}],"M161":[{"start":"1861-03-17","end":"9999-12-31","name":"ZE
ME"}],"L117":[{"end":"1927-01-11","start":"1861-03-17","name":"TERNI"},{"nam
e":"TERNI","start":"1927-01-12","end":"1927-06-16"},{"start":"1927-06-17","e
nd":"1929-02-23","name":"TERNI"},{"start":"1929-02-24","end":"1929-04-19","n
ame":"TERNI"},{"name":"TERNI","end":"1947-09-23","start":"1929-04-20"},{"nam
e":"TERNI","start":"1947-09-24","end":"9999-12-31"}],"G080":[{"end":"1996-08
-31","start":"1866-11-19","name":"OPPEANO"},{"start":"1996-09-01","end":"200
0-07-15","name":"OPPEANO"},{"start":"2000-07-16","end":"9999-12-31","name":"
OPPEANO"}],"E950":[{"name":"MARIANO AL BREMBO","start":"1863-07-13","end":"1
927-08-19"}],"E372":[{"start":"1938-06-02","end":"1945-01-11","name":"ISTONI
O"},{"name":"VASTO","end":"1938-06-01","start":"1861-03-17"},{"start":"1945-
01-12","end":"9999-12-31","name":"VASTO"}],"E427":[{"end":"1923-10-13","star
t":"1861-03-17","name":"LAMBRATE"}],"H253":[{"start":"1868-07-28","end":"999
9-12-31","name":"REVINE LAGO"}],"G992":[{"end":"9999-12-31","start":"1863-05
-06","name":"PRATA D'ANSIDONIA"}],"Z315":[{"end":"9999-12-31","start":"0000-
01-01","name":"Etiopia"}],"A164":[{"name":"ALBIOLO","start":"1861-03-17","en
d":"9999-12-31"}],"C353":[{"name":"CATENANUOVA","start":"1861-03-17","end":"
1927-01-11"},{"name":"CATENANUOVA","start":"1927-01-12","end":"9999-12-31"}]
,"H903":[{"name":"SAN GIOVANNI DI GERACE","end":"9999-12-31","start":"1863-0
4-10"}],"C059":[{"start":"1866-11-19","end":"9999-12-31","name":"CASTELBELFO
RTE"}],"D541":[{"name":"FERMIGNANO","end":"1998-09-03","start":"1861-03-17"}
,{"name":"FERMIGNANO","start":"1998-09-04","end":"9999-12-31"}],"F238":[{"na
me":"MIRADOLO","start":"1872-11-01","end":"1938-03-24"},{"name":"MIRADOLO TE
RME","start":"1938-03-25","end":"9999-12-31"}],"D337":[{"start":"1866-11-19"
,"end":"1929-01-13","name":"DONADA"},{"end":"1956-08-08","start":"1938-02-10
","name":"DONADA"},{"start":"1956-08-09","end":"1994-12-31","name":"DONADA"}
],"H439":[{"name":"ROCCA SAN GIOVANNI","start":"1861-03-17","end":"9999-12-3
1"}],"I724":[{"name":"SIDDI","start":"1861-03-17","end":"1927-08-06"},{"star
t":"1946-02-09","end":"1974-08-19","name":"SIDDI"},{"name":"SIDDI","start":"
1974-08-20","end":"2005-12-31"},{"name":"SIDDI","start":"2006-01-01","end":"
2016-04-27"},{"name":"SIDDI","start":"2016-04-28","end":"9999-12-31"}],"I801
":[{"name":"SOLFERINO","start":"1868-07-01","end":"9999-12-31"}],"M156":[{"e
nd":"1928-04-14","start":"1861-03-17","name":"ZELBIO"},{"name":"ZELBIO","end
":"9999-12-31","start":"1948-03-26"}],"D934":[{"start":"1861-03-17","end":"9
999-12-31","name":"GATTATICO"}],"E200":[{"name":"GROSIO","start":"1861-03-17
","end":"9999-12-31"}],"C614":[{"start":"1866-11-19","end":"9999-12-31","nam
e":"CHIARANO"}],"G638":[{"name":"PIEVE D'ALPAGO","start":"1866-11-19","end":
"2016-02-22"}],"D536":[{"start":"1861-03-17","end":"1928-12-13","name":"FENI
LS"}],"A468":[{"end":"9999-12-31","start":"1864-11-07","name":"SINALUNGA"}],
"D291":[{"start":"1863-01-26","end":"9999-12-31","name":"DIANO D'ALBA"}],"C5
19":[{"start":"1921-01-05","end":"1923-02-07","name":"CERNIZZA GORIZIANA"},{
"end":"1927-01-11","start":"1923-02-08","name":"CERNIZZA GORIZIANA"},{"start
":"1927-01-12","end":"1928-01-21","name":"CERNIZZA GORIZIANA"},{"end":"1947-
09-15","start":"1928-01-22","name":"CERNIZZA GORIZIANA"}],"B562":[{"name":"C
AMPOROTONDO DI FIASTRONE","end":"9999-12-31","start":"1863-02-23"}],"D322":[
{"name":"DOL GRANDE","start":"1923-04-27","end":"1927-01-11"},{"start":"1927
-01-12","end":"1928-08-23","name":"DOL GRANDE"},{"name":"VALGRANDE","start":
"1921-01-05","end":"1923-02-07"},{"name":"VALGRANDE","end":"1923-04-26","sta
rt":"1923-02-08"}],"E853":[{"start":"1861-03-17","end":"1928-06-29","name":"
MALESCO"},{"name":"MALESCO","end":"1992-05-22","start":"1928-06-30"},{"end":
"9999-12-31","start":"1992-05-23","name":"MALESCO"}],"I905":[{"start":"1861-
03-17","end":"1992-04-15","name":"SPILINGA"},{"start":"1992-04-16","end":"99
99-12-31","name":"SPILINGA"}],"I662":[{"start":"1867-06-24","end":"9999-12-3
1","name":"SERRAVALLE A PO"}],"L770":[{"start":"1861-03-17","end":"1928-02-2
2","name":"VERNA"}],"D477":[{"name":"FALERONE","end":"2009-07-11","start":"1
861-03-17"},{"name":"FALERONE","start":"2009-07-12","end":"9999-12-31"}],"H3
79":[{"start":"1877-12-21","end":"9999-12-31","name":"ROCCA PIETORE"}],"H073
":[{"name":"PROPATA","start":"1861-03-17","end":"9999-12-31"}],"B729":[{"end
":"2009-06-29","start":"1869-02-24","name":"CARATE BRIANZA"},{"end":"9999-12
-31","start":"2009-06-30","name":"CARATE BRIANZA"}],"B827":[{"name":"CARPINE
TO DELLA NORA","start":"1863-07-13","end":"1927-01-11"},{"end":"9999-12-31",
"start":"1927-01-12","name":"CARPINETO DELLA NORA"}],"F256":[{"name":"MOCONE
SI","start":"1861-03-17","end":"9999-12-31"}],"M374":[{"start":"2016-02-23",
"end":"9999-12-31","name":"VAL DI ZOLDO"}],"L254":[{"end":"9999-12-31","star
t":"1861-03-17","name":"TORRECUSO"}],"D018":[{"start":"1861-03-17","end":"19
37-05-12","name":"CORNAREDO"},{"name":"CORNAREDO","end":"9999-12-31","start"
:"1937-05-13"}],"I895":[{"start":"1861-03-17","end":"9999-12-31","name":"SPE
ZZANO ALBANESE"}],"E716":[{"name":"LUCERA","end":"9999-12-31","start":"1861-
03-17"}],"B160":[{"name":"BRESSANONE","end":"1923-02-13","start":"1920-10-16
"},{"start":"1923-02-14","end":"1927-01-11","name":"BRESSANONE"},{"name":"BR
ESSANONE","end":"1928-09-07","start":"1927-01-12"},{"name":"BRESSANONE","end
":"1941-01-06","start":"1928-09-08"},{"end":"2013-05-16","start":"1941-01-07
","name":"BRESSANONE"},{"end":"9999-12-31","start":"2013-05-17","name":"BRES
SANONE"}],"F728":[{"name":"MORI","end":"1923-02-13","start":"1920-10-16"},{"
name":"MORI","end":"1924-01-13","start":"1923-02-14"},{"start":"1924-01-14",
"end":"1971-08-31","name":"MORI"},{"end":"9999-12-31","start":"1971-09-01","
name":"MORI"}],"I234":[{"name":"SANTA MARIA CAPUA VETERE","end":"1927-01-11"
,"start":"1862-09-08"},{"name":"SANTA MARIA CAPUA VETERE","end":"1929-02-05"
,"start":"1927-01-12"},{"name":"SANTA MARIA CAPUA VETERE","end":"1945-08-31"
,"start":"1929-02-06"},{"name":"SANTA MARIA CAPUA VETERE","start":"1945-09-0
1","end":"1946-11-11"},{"start":"1946-11-12","end":"1946-12-18","name":"SANT
A MARIA CAPUA VETERE"},{"name":"SANTA MARIA CAPUA VETERE","end":"1947-09-24"
,"start":"1946-12-19"},{"name":"SANTA MARIA CAPUA VETERE","end":"9999-12-31"
,"start":"1947-09-25"}],"E089":[{"start":"1876-11-20","end":"9999-12-31","na
me":"GONZAGA"}],"M138":[{"start":"1918-12-13","end":"1992-04-15","name":"ZAC
CANOPOLI"},{"end":"9999-12-31","start":"1992-04-16","name":"ZACCANOPOLI"}],"
C883":[{"name":"COLLIO","end":"9999-12-31","start":"1861-03-17"}],"M007":[{"
start":"1861-03-17","end":"9999-12-31","name":"VILLAR FOCCHIARDO"}],"H021":[
{"name":"PREDOSA","end":"1929-06-14","start":"1861-03-17"},{"name":"PREDOSA"
,"end":"9999-12-31","start":"1929-06-15"}],"A487":[{"start":"1861-03-17","en
d":"1929-03-09","name":"ATRANI"},{"name":"ATRANI","start":"1945-07-19","end"
:"9999-12-31"}],"B832":[{"end":"1938-12-17","start":"1861-03-17","name":"CAR
RARA"},{"end":"9999-12-31","start":"1946-03-04","name":"CARRARA"}],"D866":[{
"end":"1929-02-07","start":"1861-03-17","name":"GALDO"}],"G656":[{"name":"PI
EVE TESINO","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14","
end":"1928-02-20","name":"PIEVE TESINO"},{"start":"1928-02-21","end":"1947-0
3-29","name":"PIEVE TESINO"},{"name":"PIEVE TESINO","start":"1947-03-30","en
d":"9999-12-31"}],"D558":[{"start":"1929-04-23","end":"1945-08-31","name":"F
ERTILIA"},{"start":"1945-09-01","end":"1946-12-21","name":"FERTILIA"}],"G959
":[{"start":"1861-03-17","end":"9999-12-31","name":"POZZOLENGO"}],"I991":[{"
end":"9999-12-31","start":"1861-03-17","name":"SUBBIANO"}],"G784":[{"name":"
POLI","start":"1871-01-15","end":"1929-06-15"},{"start":"1929-06-16","end":"
9999-12-31","name":"POLI"}],"D205":[{"end":"9999-12-31","start":"1871-07-10"
,"name":"CUNEO"}],"M434":[{"name":"MORANSENGO-TONENGO","end":"9999-12-31","s
tart":"2023-01-01"}],"H893":[{"name":"SAN GIORGIO DELLE PERTICHE","start":"1
866-11-19","end":"9999-12-31"}],"B239":[{"name":"BUCCIANO","start":"1861-03-
17","end":"9999-12-31"}],"A145":[{"name":"ALBENGA","start":"1861-03-17","end
":"1927-01-11"},{"name":"ALBENGA","end":"1929-05-27","start":"1927-01-12"},{
"name":"ALBENGA","start":"1929-05-28","end":"9999-12-31"}],"G802":[{"name":"
POLVERARA","end":"9999-12-31","start":"1866-11-19"}],"F504":[{"name":"MONTEF
LAVIO","end":"9999-12-31","start":"1871-01-15"}],"B453":[{"start":"1861-03-1
7","end":"9999-12-31","name":"CAMAGNA MONFERRATO"}],"L506":[{"name":"URZULEI
","start":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"2005
-12-31","name":"URZULEI"},{"name":"URZULEI","end":"2016-04-27","start":"2006
-01-01"},{"start":"2016-04-28","end":"9999-12-31","name":"URZULEI"}],"L074":
[{"name":"TAVIANO","end":"1954-11-04","start":"1885-01-15"},{"start":"1954-1
1-05","end":"1989-12-01","name":"TAVIANO"},{"name":"TAVIANO","start":"1989-1
2-02","end":"9999-12-31"}],"A446":[{"end":"9999-12-31","start":"1871-01-15",
"name":"ARSOLI"}],"F076":[{"end":"1923-02-13","start":"1920-10-16","name":"M
EANO"},{"name":"MEANO","start":"1923-02-14","end":"1926-09-30"}],"H082":[{"n
ame":"PROVEZZE","end":"1928-08-13","start":"1861-03-17"}],"F261":[{"name":"M
ODOLO","end":"1927-01-11","start":"1861-03-17"},{"name":"MODOLO","end":"2005
-12-31","start":"1927-01-12"},{"end":"9999-12-31","start":"2006-01-01","name
":"MODOLO"}],"E543":[{"name":"LERMA","end":"9999-12-31","start":"1861-03-17"
}],"L725":[{"end":"1927-10-12","start":"1861-03-17","name":"VENAFRO"},{"name
":"VENAFRO","start":"1927-10-13","end":"1937-02-23"},{"name":"VENAFRO","star
t":"1937-02-24","end":"1970-03-02"},{"name":"VENAFRO","start":"1970-03-03","
end":"1979-11-30"},{"start":"1979-12-01","end":"9999-12-31","name":"VENAFRO"
}],"G209":[{"end":"1934-03-16","start":"1921-05-12","name":"PACE DEL MELA"},
{"name":"PACE DEL MELA","start":"1934-03-17","end":"9999-12-31"}],"A371":[{"
start":"1861-03-17","end":"1927-01-11","name":"ARCISATE"},{"name":"ARCISATE"
,"start":"1927-01-12","end":"1928-10-16"},{"end":"1968-07-09","start":"1928-
10-17","name":"ARCISATE"},{"start":"1968-07-10","end":"9999-12-31","name":"A
RCISATE"}],"I020":[{"name":"SAN MARZANO MOASCA","end":"1935-04-07","start":"
1929-04-20"},{"name":"SAN MARZANO MOASCA","start":"1935-04-08","end":"1947-0
6-23"}],"D851":[{"start":"1862-11-28","end":"9999-12-31","name":"GAGLIANO DE
L CAPO"}],"G661":[{"name":"PIGNATARO MAGGIORE","end":"1927-01-11","start":"1
862-09-29"},{"name":"PIGNATARO MAGGIORE","end":"1929-03-23","start":"1927-01
-12"},{"start":"1929-03-24","end":"1945-08-31","name":"PIGNATARO MAGGIORE"},
{"name":"PIGNATARO MAGGIORE","start":"1945-09-01","end":"1947-10-09"},{"name
":"PIGNATARO MAGGIORE","start":"1947-10-10","end":"9999-12-31"}],"L307":[{"n
ame":"TORTORETO","end":"1929-05-29","start":"1861-03-17"},{"name":"TORTORETO
","end":"1956-07-28","start":"1929-05-30"},{"start":"1956-07-29","end":"9999
-12-31","name":"TORTORETO"}],"A431":[{"name":"ARPAIA","end":"9999-12-31","st
art":"1861-03-17"}],"H990":[{"name":"SAN MARTINO IN PENSILIS","end":"1970-03
-02","start":"1863-08-10"},{"start":"1970-03-03","end":"9999-12-31","name":"
SAN MARTINO IN PENSILIS"}],"L904":[{"name":"VIGUZZOLO","end":"1928-09-24","s
tart":"1861-03-17"},{"end":"1947-09-03","start":"1928-09-25","name":"VIGUZZO
LO"},{"start":"1947-09-04","end":"1959-12-03","name":"VIGUZZOLO"},{"name":"V
IGUZZOLO","start":"1959-12-04","end":"9999-12-31"}],"F906":[{"name":"NOASCA"
,"start":"1861-03-17","end":"1927-01-11"},{"name":"NOASCA","start":"1927-01-
12","end":"1929-03-02"},{"name":"NOASCA","end":"1945-10-04","start":"1935-06
-05"},{"end":"9999-12-31","start":"1945-10-05","name":"NOASCA"}],"C980":[{"s
tart":"1861-03-17","end":"1909-01-22","name":"COPPARO"},{"end":"9999-12-31",
"start":"1909-01-23","name":"COPPARO"}],"F609":[{"start":"1863-04-23","end":
"1923-12-21","name":"MONTEROSSO AL MARE"},{"name":"MONTEROSSO AL MARE","star
t":"1923-12-22","end":"9999-12-31"}],"E293":[{"name":"INCINO","end":"1906-11
-12","start":"1861-03-17"}],"H947":[{"name":"SAN LAZZARO ALBERONI","start":"
1862-11-28","end":"1923-07-22"}],"B045":[{"name":"BORGOUNITO","start":"1928-
06-30","end":"1947-11-04"}],"L125":[{"name":"TERRANOVA DEI PASSERINI","start
":"1863-01-26","end":"1992-04-15"},{"end":"9999-12-31","start":"1992-04-16",
"name":"TERRANOVA DEI PASSERINI"}],"D671":[{"name":"FONTANAROSA","end":"9999
-12-31","start":"1861-03-17"}],"H344":[{"name":"RIVAROSSA","start":"1861-03-
17","end":"1928-02-08"},{"name":"RIVAROSSA","start":"1946-11-17","end":"1999
-12-31"},{"name":"RIVAROSSA","end":"9999-12-31","start":"2000-01-01"}],"L819
":[{"name":"VEZZANO LIGURE","end":"1923-12-21","start":"1863-04-23"},{"start
":"1923-12-22","end":"1928-12-12","name":"VEZZANO LIGURE"},{"end":"9999-12-3
1","start":"1928-12-13","name":"VEZZANO LIGURE"}],"E682":[{"name":"LONIGO","
start":"1866-11-19","end":"9999-12-31"}],"A060":[{"name":"ADRO","start":"186
1-03-17","end":"1928-08-13"},{"name":"ADRO","end":"1962-05-25","start":"1928
-08-14"},{"name":"ADRO","start":"1962-05-26","end":"9999-12-31"}],"E415":[{"
name":"LAINATE","end":"1928-06-26","start":"1870-08-02"},{"name":"LAINATE","
start":"1928-06-27","end":"9999-12-31"}],"L212":[{"name":"TORCHIARA","end":"
1929-03-23","start":"1861-03-17"},{"name":"TORCHIARA","start":"1929-03-24","
end":"1947-05-12"},{"name":"TORCHIARA","start":"1947-05-13","end":"9999-12-3
1"}],"G488":[{"end":"1934-03-16","start":"1863-05-06","name":"PESCHIERA BORR
OMEO"},{"name":"PESCHIERA BORROMEO","end":"1995-05-04","start":"1934-03-17"}
,{"name":"PESCHIERA BORROMEO","end":"9999-12-31","start":"1995-05-05"}],"C59
4":[{"end":"9999-12-31","start":"1977-01-02","name":"CHALLAND-SAINT-VICTOR"}
,{"name":"CHALLANT SAINT VICTOR","end":"1927-01-11","start":"1861-03-17"},{"
start":"1927-01-12","end":"1928-06-14","name":"CHALLANT SAINT VICTOR"},{"nam
e":"CHALLANT SAINT VICTOR","end":"1977-01-01","start":"1946-09-27"}],"E116":
[{"name":"GOTTOLENGO","end":"9999-12-31","start":"1861-03-17"}],"Z327":[{"st
art":"0000-01-01","end":"9999-12-31","name":"Madagascar"}],"B760":[{"start":
"1861-03-17","end":"1923-11-23","name":"CAREGGINE"},{"name":"CAREGGINE","end
":"9999-12-31","start":"1923-11-24"}],"F128":[{"name":"MENTOULLES","end":"19
27-11-05","start":"1861-03-17"}],"F002":[{"name":"MARZIO","start":"1861-03-1
7","end":"1927-01-11"},{"name":"MARZIO","start":"1927-01-12","end":"1978-02-
11"},{"start":"1978-02-12","end":"9999-12-31","name":"MARZIO"}],"L572":[{"na
me":"VALERA FRATTA","start":"1861-03-17","end":"1992-04-15"},{"name":"VALERA
 FRATTA","start":"1992-04-16","end":"9999-12-31"}],"G876":[{"name":"PONZATE"
,"start":"1861-03-17","end":"1928-09-22"}],"E621":[{"start":"1861-03-17","en
d":"9999-12-31","name":"LIVIGNO"}],"H584":[{"end":"1928-03-03","start":"1927
-06-02","name":"ROTA"},{"end":"9999-12-31","start":"1928-03-04","name":"ROTA
 D'IMAGNA"}],"Z526":[{"end":"9999-12-31","start":"0000-01-01","name":"Domini
ca"}],"F424":[{"start":"1861-03-17","end":"9999-12-31","name":"MONTANERA"}],
"L426":[{"name":"TRIGOLO","start":"1861-03-17","end":"9999-12-31"}],"F972":[
{"start":"1861-03-17","end":"1928-05-01","name":"NUCETTO"},{"name":"NUCETTO"
,"start":"1928-05-02","end":"1947-01-09"},{"name":"NUCETTO","start":"1947-01
-10","end":"9999-12-31"}],"H689":[{"end":"9999-12-31","start":"1866-11-19","
name":"SALARA"}],"F350":[{"name":"MONDONIO","end":"1929-04-05","start":"1861
-03-17"}],"C244":[{"end":"9999-12-31","start":"1870-06-24","name":"CASTELNUO
VO DI VAL DI CECINA"}],"G184":[{"name":"OSTELLATO","end":"1986-10-18","start
":"1861-03-17"},{"name":"OSTELLATO","end":"9999-12-31","start":"1986-10-19"}
],"Z332":[{"name":"Maurizio","end":"9999-12-31","start":"0000-01-01"}],"H986
":[{"name":"SAN MARCO LA CATOLA","end":"9999-12-31","start":"1861-03-17"}],"
C699":[{"end":"1968-04-05","start":"1866-11-19","name":"CIMOLAIS"},{"start":
"1968-04-06","end":"9999-12-31","name":"CIMOLAIS"}],"C996":[{"name":"COREGLI
A ANTELMINELLI","end":"9999-12-31","start":"1863-01-05"}],"A745":[{"end":"19
92-04-15","start":"1861-03-17","name":"BELLANO"},{"end":"9999-12-31","start"
:"1992-04-16","name":"BELLANO"}],"H444":[{"end":"1928-10-30","start":"1872-1
0-18","name":"ROCCASECCA DEI VOLSCI"},{"name":"ROCCASECCA DEI VOLSCI","start
":"1947-06-24","end":"9999-12-31"}],"B265":[{"start":"1861-03-17","end":"999
9-12-31","name":"BULZI"}],"E191":[{"name":"GRONDONA","end":"1929-05-20","sta
rt":"1861-03-17"},{"end":"1962-05-25","start":"1929-05-21","name":"GRONDONA"
},{"end":"9999-12-31","start":"1962-05-26","name":"GRONDONA"}],"C815":[{"nam
e":"CODOGN\u00c8","start":"1869-08-11","end":"9999-12-31"}],"A590":[{"name":
"BALDISSERO CANAVESE","end":"1927-01-11","start":"1863-01-05"},{"name":"BALD
ISSERO CANAVESE","start":"1927-01-12","end":"1929-05-03"},{"start":"1947-01-
10","end":"9999-12-31","name":"BALDISSERO CANAVESE"}],"B806":[{"start":"1861
-03-17","end":"1927-01-11","name":"CARONNO CORBELLARO"},{"end":"1928-01-11",
"start":"1927-01-12","name":"CARONNO CORBELLARO"}],"G548":[{"end":"1927-01-1
1","start":"1861-03-17","name":"PIANCERI"},{"name":"PIANCERI","end":"1928-03
-13","start":"1927-01-12"}],"D646":[{"name":"FOGLIZZO","end":"9999-12-31","s
tart":"1861-03-17"}],"F698":[{"start":"1861-03-17","end":"1927-01-11","name"
:"MONTRESTA"},{"name":"MONTRESTA","end":"1997-02-06","start":"1927-01-12"},{
"start":"1997-02-07","end":"2005-12-31","name":"MONTRESTA"},{"name":"MONTRES
TA","start":"2006-01-01","end":"9999-12-31"}],"D949":[{"end":"9999-12-31","s
tart":"1879-08-04","name":"GAZOLDO DEGLI IPPOLITI"}],"E315":[{"end":"1928-04
-02","start":"1861-03-17","name":"INVORIO INFERIORE"}],"C129":[{"start":"186
3-02-06","end":"9999-12-31","name":"CASTELLAMMARE DI STABIA"}],"H148":[{"end
":"9999-12-31","start":"1861-03-17","name":"RACALMUTO"}],"C911":[{"start":"1
861-03-17","end":"1927-01-11","name":"COMABBIO"},{"end":"9999-12-31","start"
:"1927-01-12","name":"COMABBIO"}],"D100":[{"name":"COSSOINE","start":"1861-0
3-17","end":"9999-12-31"}],"M026":[{"name":"VILLASPECIOSA","end":"1974-08-19
","start":"1861-03-17"},{"name":"VILLASPECIOSA","start":"1974-08-20","end":"
2016-04-27"},{"name":"VILLASPECIOSA","end":"9999-12-31","start":"2016-04-28"
}],"G298":[{"start":"1861-03-17","end":"1928-05-18","name":"PALUDI"},{"start
":"1934-03-07","end":"9999-12-31","name":"PALUDI"}],"Z200":[{"end":"9999-12-
31","start":"0000-01-01","name":"Afghanistan"}],"A240":[{"start":"1861-03-17
","end":"9999-12-31","name":"ALTOMONTE"}],"D631":[{"start":"1920-10-16","end
":"1923-02-13","name":"FLAVON"},{"end":"1928-10-16","start":"1923-02-14","na
me":"FLAVON"},{"name":"FLAVON","end":"1947-03-29","start":"1928-10-17"},{"st
art":"1947-03-30","end":"2015-12-31","name":"FLAVON"}],"F932":[{"name":"NONI
O","start":"1861-03-17","end":"1928-02-21"},{"name":"NONIO","end":"1992-05-2
2","start":"1947-07-09"},{"name":"NONIO","start":"1992-05-23","end":"9999-12
-31"}],"L695":[{"name":"VASSENA","start":"1861-03-17","end":"1927-12-29"}],"
I012":[{"name":"SAN MARTINO IN STRADA","end":"1992-04-15","start":"1869-02-1
1"},{"start":"1992-04-16","end":"9999-12-31","name":"SAN MARTINO IN STRADA"}
],"L527":[{"end":"1923-02-13","start":"1920-10-16","name":"VADENA"},{"end":"
1927-01-11","start":"1923-02-14","name":"VADENA"},{"name":"VADENA","end":"99
99-12-31","start":"1927-01-12"}],"G841":[{"end":"1926-02-05","start":"1889-0
6-17","name":"PONTEDECIMO"}],"G388":[{"end":"1939-07-12","start":"1883-06-21
","name":"PAVIA"},{"start":"1939-07-13","end":"2005-02-18","name":"PAVIA"},{
"start":"2005-02-19","end":"9999-12-31","name":"PAVIA"}],"C064":[{"end":"193
5-04-07","start":"1863-02-16","name":"CASTEL BOGLIONE"},{"end":"9999-12-31",
"start":"1935-04-08","name":"CASTEL BOGLIONE"}],"F927":[{"name":"NOMAGLIO","
start":"1894-02-02","end":"1927-01-11"},{"name":"NOMAGLIO","start":"1927-01-
12","end":"1929-03-02"},{"name":"NOMAGLIO","end":"9999-12-31","start":"1954-
07-15"}],"A159":[{"name":"ALBIATE","start":"1861-03-17","end":"2009-06-29"},
{"name":"ALBIATE","end":"9999-12-31","start":"2009-06-30"}],"I552":[{"start"
:"1921-01-05","end":"1923-02-07","name":"SCRILIE"},{"name":"SCRILIE","start"
:"1923-02-08","end":"1923-04-26"},{"name":"SCRILLA","start":"1923-04-27","en
d":"1927-01-11"},{"start":"1927-01-12","end":"1928-03-12","name":"SCRILLA"}]
,"G945":[{"name":"POVEGLIANO VERONESE","end":"9999-12-31","start":"1866-11-1
9"}],"L532":[{"end":"9999-12-31","start":"1955-04-05","name":"VAGLIO BASILIC
ATA"},{"end":"1933-07-12","start":"1863-04-10","name":"VAGLIO DI BASILICATA"
},{"start":"1933-07-13","end":"1955-04-04","name":"VAGLIO LUCANO"}],"F450":[
{"end":"1998-09-03","start":"1861-03-17","name":"MONTECALVO IN FOGLIA"},{"na
me":"MONTECALVO IN FOGLIA","start":"1998-09-04","end":"9999-12-31"}],"G836":
[{"name":"PONTECCHIO","end":"1929-07-01","start":"1866-11-19"},{"start":"192
9-07-02","end":"9999-12-31","name":"PONTECCHIO POLESINE"}],"B652":[{"name":"
CAPIAGO","start":"1861-03-17","end":"1928-12-15"}],"F324":[{"name":"MONASTER
ACE","start":"1861-03-17","end":"9999-12-31"}],"L326":[{"name":"TRAMUTOLA","
start":"1861-03-17","end":"9999-12-31"}],"E483":[{"start":"1861-03-17","end"
:"9999-12-31","name":"LAURIA"}],"B872":[{"name":"CASAL DI PRINCIPE","end":"1
927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"1928-05-23","n
ame":"CASAL DI PRINCIPE"},{"end":"9999-12-31","start":"1946-03-25","name":"C
ASAL DI PRINCIPE"}],"B279":[{"name":"BUROLO","end":"1927-01-11","start":"186
1-03-17"},{"end":"1929-04-19","start":"1927-01-12","name":"BUROLO"},{"end":"
9999-12-31","start":"1946-12-24","name":"BUROLO"}],"L220":[{"name":"TORITTO"
,"end":"9999-12-31","start":"1861-03-17"}],"A586":[{"name":"BAISO","start":"
1861-03-17","end":"9999-12-31"}],"C608":[{"start":"1861-03-17","end":"1957-1
2-05","name":"CHIANCIANO"},{"start":"1957-12-06","end":"9999-12-31","name":"
CHIANCIANO TERME"}],"A984":[{"end":"1928-04-03","start":"1861-03-17","name":
"BORDIGHERA"},{"name":"BORDIGHERA","end":"9999-12-31","start":"1928-04-04"}]
,"A759":[{"start":"1861-03-17","end":"2009-06-29","name":"BELLUSCO"},{"start
":"2009-06-30","end":"9999-12-31","name":"BELLUSCO"}],"M334":[{"name":"VAL B
REMBILLA","start":"2014-02-04","end":"9999-12-31"}],"M178":[{"end":"1981-04-
20","start":"1866-11-19","name":"ZIMELLA"},{"end":"2007-07-31","start":"1981
-04-21","name":"ZIMELLA"},{"start":"2007-08-01","end":"9999-12-31","name":"Z
IMELLA"}],"I274":[{"end":"1978-02-11","start":"1864-03-11","name":"SANT'ANGE
LO LODIGIANO"},{"start":"1978-02-12","end":"1992-04-15","name":"SANT'ANGELO 
LODIGIANO"},{"name":"SANT'ANGELO LODIGIANO","start":"1992-04-16","end":"9999
-12-31"}],"A387":[{"name":"ARENA PO","start":"1863-08-10","end":"9999-12-31"
}],"B614":[{"end":"1927-08-02","start":"1861-03-17","name":"CANNITELLO"}],"F
041":[{"name":"MASSELLO","end":"1928-04-30","start":"1861-03-17"},{"end":"19
47-01-08","start":"1928-05-01","name":"MASSELLO"},{"name":"MASSELLO","start"
:"1947-01-09","end":"9999-12-31"}],"A471":[{"name":"ASOLO","end":"1928-06-18
","start":"1866-11-19"},{"start":"1928-06-19","end":"1947-04-30","name":"ASO
LO"},{"name":"ASOLO","start":"1947-05-01","end":"9999-12-31"}],"E791":[{"nam
e":"MADDALONI","start":"1861-03-17","end":"1904-02-25"},{"end":"1927-01-11",
"start":"1904-02-26","name":"MADDALONI"},{"name":"MADDALONI","start":"1927-0
1-12","end":"1928-08-17"},{"name":"MADDALONI","end":"1945-08-31","start":"19
28-08-18"},{"start":"1945-09-01","end":"1975-05-21","name":"MADDALONI"},{"st
art":"1975-05-22","end":"9999-12-31","name":"MADDALONI"}],"A346":[{"start":"
1921-01-05","end":"1923-02-07","name":"AQUILEIA"},{"name":"AQUILEIA","end":"
1924-01-13","start":"1923-02-08"},{"end":"1947-01-08","start":"1924-01-14","
name":"AQUILEIA"},{"name":"AQUILEIA","end":"1968-04-05","start":"1947-01-09"
},{"end":"9999-12-31","start":"1968-04-06","name":"AQUILEIA"}],"A297":[{"sta
rt":"1872-10-21","end":"1935-03-06","name":"ANGUILLARA SABAZIA"},{"start":"1
935-03-07","end":"9999-12-31","name":"ANGUILLARA SABAZIA"}],"B059":[{"name":
"BORSANO","start":"1913-01-03","end":"1927-01-11"},{"name":"BORSANO","end":"
1928-05-21","start":"1927-01-12"}],"I514":[{"start":"1861-03-17","end":"2013
-12-31","name":"SCARPERIA"}],"B353":[{"end":"1928-02-05","start":"1861-03-17
","name":"CAGLIANO"}],"Z306":[{"start":"0000-01-01","end":"9999-12-31","name
":"Camerun"}],"C729":[{"end":"1927-01-11","start":"1863-04-23","name":"CISAN
O SUL NEVA"},{"name":"CISANO SUL NEVA","start":"1927-01-12","end":"1929-06-0
3"},{"end":"9999-12-31","start":"1929-06-04","name":"CISANO SUL NEVA"}],"L03
4":[{"start":"1861-03-17","end":"1907-04-19","name":"TALAMELLO"},{"start":"1
907-04-20","end":"1928-08-16","name":"TALAMELLO"},{"name":"TALAMELLO","start
":"1946-04-17","end":"1998-09-03"},{"name":"TALAMELLO","end":"2009-08-14","s
tart":"1998-09-04"},{"start":"2009-08-15","end":"9999-12-31","name":"TALAMEL
LO"}],"F036":[{"name":"MASSA SUPERIORE","end":"1928-08-13","start":"1866-11-
19"}],"I916":[{"name":"SPINONE AL LAGO","end":"9999-12-31","start":"1960-12-
24"},{"end":"1947-05-24","start":"1928-07-29","name":"SPINONE DEI CASTELLI"}
,{"start":"1947-05-25","end":"1955-07-09","name":"SPINONE DEI CASTELLI"},{"n
ame":"SPINONE DEI CASTELLI","end":"1960-12-23","start":"1955-07-10"}],"H748"
:[{"end":"1928-01-26","start":"1861-03-17","name":"SAMBUGHETTO"}],"I619":[{"
name":"SEPPIANA","end":"1928-05-12","start":"1861-03-17"},{"name":"SEPPIANA"
,"end":"1992-05-22","start":"1956-05-20"},{"start":"1992-05-23","end":"2015-
12-31","name":"SEPPIANA"}],"L407":[{"end":"1920-04-11","start":"1866-11-19",
"name":"TREVISO"},{"start":"1920-04-12","end":"9999-12-31","name":"TREVISO"}
],"D700":[{"name":"FORGARIA","start":"1866-11-19","end":"1938-01-12"},{"name
":"FORGARIA NEL FRIULI","end":"1968-04-05","start":"1938-01-13"},{"start":"1
968-04-06","end":"1969-04-22","name":"FORGARIA NEL FRIULI"},{"start":"1969-0
4-23","end":"9999-12-31","name":"FORGARIA NEL FRIULI"}],"C562":[{"start":"18
67-04-12","end":"2019-01-29","name":"LENTIAI"}],"B519":[{"name":"CAMPOBASSO"
,"end":"1928-05-11","start":"1861-03-17"},{"name":"CAMPOBASSO","start":"1928
-05-12","end":"1947-05-21"},{"name":"CAMPOBASSO","start":"1947-05-22","end":
"1970-03-02"},{"name":"CAMPOBASSO","start":"1970-03-03","end":"9999-12-31"}]
,"I054":[{"name":"SAN NICANDRO","end":"1862-11-09","start":"1861-03-17"},{"e
nd":"9999-12-31","start":"1999-10-22","name":"SAN NICANDRO GARGANICO"},{"end
":"1999-10-21","start":"1862-11-10","name":"SANNICANDRO GARGANICO"}],"Z507":
[{"name":"Giamaica","start":"0000-01-01","end":"9999-12-31"}],"A331":[{"name
":"APPARIZIONE","end":"1926-02-05","start":"1861-03-17"}],"A547":[{"end":"99
99-12-31","start":"1861-03-17","name":"BAGNACAVALLO"}],"F763":[{"end":"1924-
03-08","start":"1923-04-27","name":"MOSCHIENA"},{"start":"1924-03-09","end":
"1929-05-20","name":"MOSCHIENA"},{"start":"1929-05-21","end":"1941-09-11","n
ame":"MOSCHIENA"},{"name":"MOSCHIENIZZE","start":"1921-01-05","end":"1922-02
-01"},{"name":"MOSCHIENIZZE","start":"1922-02-02","end":"1923-02-07"},{"name
":"MOSCHIENIZZE","start":"1923-02-08","end":"1923-04-26"}],"B163":[{"name":"
BRESTOVIZZA IN VALLE","start":"1921-01-05","end":"1923-02-07"},{"name":"BRES
TOVIZZA IN VALLE","start":"1923-02-08","end":"1927-01-11"},{"end":"1928-08-2
3","start":"1927-01-12","name":"BRESTOVIZZA IN VALLE"}],"E380":[{"end":"9999
-12-31","start":"1861-03-17","name":"IZANO"}],"G902":[{"end":"9999-12-31","s
tart":"1861-03-17","name":"PORTICI"}],"D855":[{"start":"1866-11-19","end":"9
999-12-31","name":"GAIBA"}],"G665":[{"end":"9999-12-31","start":"1861-03-17"
,"name":"PIGRA"}],"M144":[{"name":"ZANDOBBIO","end":"1928-10-31","start":"18
61-03-17"},{"end":"9999-12-31","start":"1948-03-17","name":"ZANDOBBIO"}],"A4
35":[{"name":"ARQU\u00c0 POLESINE","end":"9999-12-31","start":"1868-07-13"}]
,"I248":[{"end":"1968-04-05","start":"1866-11-19","name":"SANTA MARIA LA LON
GA"},{"name":"SANTA MARIA LA LONGA","start":"1968-04-06","end":"9999-12-31"}
],"L721":[{"start":"1861-03-17","end":"1927-12-03","name":"VELLO"}],"L804":[
{"name":"VERZUOLO","start":"1861-03-17","end":"1928-04-30"},{"name":"VERZUOL
O","start":"1928-05-01","end":"9999-12-31"}],"F806":[{"start":"1861-03-17","
end":"9999-12-31","name":"MURA"}],"C880":[{"start":"1957-04-07","end":"1962-
09-08","name":"COLLI DI LABRO"},{"start":"1962-09-09","end":"9999-12-31","na
me":"COLLI SUL VELINO"}],"G072":[{"name":"ONNO","end":"1927-12-29","start":"
1861-03-17"}],"B698":[{"name":"CAPRIANO DEL COLLE","end":"1927-12-26","start
":"1862-10-20"},{"name":"CAPRIANO DEL COLLE","start":"1957-07-12","end":"999
9-12-31"}],"A375":[{"name":"ARCONATE","end":"1958-01-21","start":"1869-07-22
"},{"name":"ARCONATE","start":"1958-01-22","end":"1978-02-11"},{"end":"1997-
07-25","start":"1978-02-12","name":"ARCONATE"},{"end":"9999-12-31","start":"
1997-07-26","name":"ARCONATE"}],"I598":[{"end":"9999-12-31","start":"1861-03
-17","name":"SEMESTENE"}],"D951":[{"start":"1927-07-01","end":"9999-12-31","
name":"GAZZADA SCHIANNO"}],"F265":[{"name":"MOGGIO","end":"1929-02-14","star
t":"1861-03-17"},{"name":"MOGGIO","start":"1948-03-21","end":"1992-04-15"},{
"name":"MOGGIO","start":"1992-04-16","end":"9999-12-31"}],"H890":[{"start":"
1863-01-05","end":"1927-01-11","name":"SAN GIORGIO CANAVESE"},{"name":"SAN G
IORGIO CANAVESE","start":"1927-01-12","end":"1945-10-04"},{"name":"SAN GIORG
IO CANAVESE","end":"9999-12-31","start":"1945-10-05"}],"Z101":[{"end":"9999-
12-31","start":"0000-01-01","name":"Andorra"}],"D201":[{"start":"1861-03-17"
,"end":"1927-01-11","name":"CUGNOLI"},{"start":"1927-01-12","end":"9999-12-3
1","name":"CUGNOLI"}],"B450":[{"name":"CALVISANO","end":"9999-12-31","start"
:"1861-03-17"}],"F652":[{"name":"MONTE URABICE","start":"1923-04-27","end":"
1927-01-11"},{"end":"1928-01-21","start":"1927-01-12","name":"MONTE URABICE"
},{"end":"1923-02-07","start":"1921-01-05","name":"VRABICE"},{"name":"VRABIC
E","start":"1923-02-08","end":"1923-04-26"}],"H774":[{"name":"SAN BENEDETTO 
ULLANO","start":"1861-03-17","end":"1928-05-11"},{"start":"1937-07-18","end"
:"9999-12-31","name":"SAN BENEDETTO ULLANO"}],"D564":[{"start":"1861-03-17",
"end":"9999-12-31","name":"FIASTRA"}],"L782":[{"name":"VERPOGLIANO","end":"1
923-02-07","start":"1921-01-05"},{"end":"1927-01-11","start":"1923-02-08","n
ame":"VERPOGLIANO"},{"start":"1927-01-12","end":"1928-04-28","name":"VERPOGL
IANO"}],"B927":[{"end":"1927-01-11","start":"1863-04-23","name":"CASANOVA LE
RRONE"},{"name":"CASANOVA LERRONE","end":"1929-04-11","start":"1927-01-12"},
{"name":"CASANOVA LERRONE","start":"1929-04-12","end":"9999-12-31"}],"H025":
[{"end":"1928-03-07","start":"1861-03-17","name":"PREGLIA"}],"E540":[{"name"
:"LEQUIO BERRIA","end":"1928-05-01","start":"1862-12-19"},{"start":"1928-05-
02","end":"1947-03-29","name":"LEQUIO BERRIA"},{"name":"LEQUIO BERRIA","star
t":"1947-03-30","end":"9999-12-31"}],"I995":[{"start":"1861-03-17","end":"19
27-08-27","name":"SUELLI"},{"start":"1947-04-06","end":"1974-08-19","name":"
SUELLI"},{"start":"1974-08-20","end":"2016-04-27","name":"SUELLI"},{"name":"
SUELLI","end":"9999-12-31","start":"2016-04-28"}],"G252":[{"end":"1907-07-01
","start":"1861-03-17","name":"PALAGIANO"},{"name":"PALAGIANO","end":"1923-1
2-21","start":"1907-07-02"},{"name":"PALAGIANO","start":"1923-12-22","end":"
9999-12-31"}],"I023":[{"end":"1970-03-02","start":"1861-03-17","name":"SAN M
ASSIMO"},{"name":"SAN MASSIMO","start":"1970-03-03","end":"9999-12-31"}],"I3
29":[{"name":"SAN TEODORO","end":"2005-12-31","start":"1959-05-03"},{"start"
:"2006-01-01","end":"2011-01-20","name":"SAN TEODORO"},{"start":"2011-01-21"
,"end":"2016-04-27","name":"SAN TEODORO"},{"end":"9999-12-31","start":"2016-
04-28","name":"SAN TEODORO"}],"C983":[{"name":"CORATO","start":"1861-03-17",
"end":"9999-12-31"}],"B932":[{"name":"CASAPE","end":"9999-12-31","start":"18
71-01-15"}],"D966":[{"end":"9999-12-31","start":"1861-03-17","name":"GENIVOL
TA"}],"E290":[{"name":"IMPERIA","end":"1925-08-21","start":"1923-11-29"},{"e
nd":"1928-12-14","start":"1925-08-22","name":"IMPERIA"},{"start":"1928-12-15
","end":"9999-12-31","name":"IMPERIA"}],"G859":[{"start":"1861-03-17","end":
"9999-12-31","name":"PONTEVICO"}],"I891":[{"name":"SPERLINGA","start":"1861-
03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31","name":"
SPERLINGA"}],"D669":[{"name":"FONTANA DEL CONTE","end":"1923-02-07","start":
"1921-01-05"},{"start":"1923-02-08","end":"1924-03-08","name":"FONTANA DEL C
ONTE"},{"start":"1924-03-09","end":"1927-10-21","name":"FONTANA DEL CONTE"},
{"name":"FONTANA DEL CONTE","end":"1947-09-15","start":"1927-10-22"}],"H174"
:[{"name":"RANCO","start":"1861-03-17","end":"1927-01-11"},{"name":"RANCO","
start":"1927-01-12","end":"1928-01-12"},{"name":"RANCO","end":"9999-12-31","
start":"1957-01-25"}],"G976":[{"name":"PRAIANO","end":"9999-12-31","start":"
1861-03-17"}],"A741":[{"start":"1872-11-01","end":"1929-04-08","name":"BELGI
OIOSO"},{"name":"BELGIOIOSO","end":"1942-11-29","start":"1929-04-09"},{"name
":"BELGIOIOSO","end":"1947-11-24","start":"1942-11-30"},{"name":"BELGIOIOSO"
,"start":"1947-11-25","end":"1948-01-19"},{"name":"BELGIOIOSO","start":"1948
-01-20","end":"1961-03-08"},{"end":"9999-12-31","start":"1961-03-09","name":
"BELGIOIOSO"}],"L182":[{"name":"TIVOLI","start":"1871-01-15","end":"1937-11-
02"},{"start":"1937-11-03","end":"9999-12-31","name":"TIVOLI"}],"E396":[{"st
art":"1863-01-19","end":"1938-11-12","name":"LACCO AMENO"},{"name":"LACCO AM
ENO","start":"1945-09-22","end":"9999-12-31"}],"H338":[{"start":"1861-03-17"
,"end":"1927-10-20","name":"RIVARA"},{"start":"1927-10-21","end":"1946-12-23
","name":"RIVARA"},{"start":"1946-12-24","end":"9999-12-31","name":"RIVARA"}
],"E247":[{"end":"1945-02-22","start":"1928-01-26","name":"GUARDIA PIEMONTES
E TERME"}],"H478":[{"name":"RODERO","start":"1861-03-17","end":"9999-12-31"}
],"C896":[{"name":"COLOGNOLA DEL PIANO","start":"1863-07-13","end":"1927-03-
11"}],"F279":[{"name":"MOIOLA","start":"1861-03-17","end":"1928-06-13"},{"en
d":"9999-12-31","start":"1946-07-03","name":"MOIOLA"}],"E625":[{"name":"LIVO
RNO","end":"1931-10-02","start":"1861-03-17"},{"name":"LIVORNO","end":"9999-
12-31","start":"1931-10-03"}],"A736":[{"end":"9999-12-31","start":"1861-03-1
7","name":"BELCASTRO"}],"G006":[{"name":"OFFLAGA","start":"1861-03-17","end"
:"1928-02-15"},{"name":"OFFLAGA","start":"1928-02-16","end":"9999-12-31"}],"
A369":[{"name":"ARCIDOSSO","end":"9999-12-31","start":"1861-03-17"}],"A063":
[{"name":"AFFORI","end":"1923-10-13","start":"1884-08-23"}],"F872":[{"end":"
1923-06-28","start":"1866-11-19","name":"NERVESA"},{"name":"NERVESA DELLA BA
TTAGLIA","start":"1923-06-29","end":"9999-12-31"}],"G574":[{"start":"1861-03
-17","end":"1929-05-13","name":"PIARIO"},{"name":"PIARIO","end":"9999-12-31"
,"start":"1958-09-05"}],"E597":[{"start":"1862-12-19","end":"1947-10-06","na
me":"LIMONE PIEMONTE"},{"end":"9999-12-31","start":"1947-10-07","name":"LIMO
NE PIEMONTE"}],"H886":[{"end":"1998-09-03","start":"1861-03-17","name":"SAN 
GIORGIO DI PESARO"},{"name":"SAN GIORGIO DI PESARO","end":"2016-12-31","star
t":"1998-09-04"}],"F519":[{"start":"1861-03-17","end":"9999-12-31","name":"M
ONTEGIORDANO"}],"B763":[{"end":"1992-04-15","start":"1861-03-17","name":"CAR
ENNO"},{"name":"CARENNO","start":"1992-04-16","end":"9999-12-31"}],"E411":[{
"end":"1923-02-07","start":"1921-01-05","name":"LAGOSTA"},{"name":"LAGOSTA",
"end":"1947-09-15","start":"1923-02-08"}],"H847":[{"name":"SAN FRANCESCO AL 
CAMPO","start":"1863-01-26","end":"9999-12-31"}],"B036":[{"start":"1861-03-1
7","end":"9999-12-31","name":"BORGO SAN LORENZO"}],"D062":[{"name":"CORTEMIL
IA","end":"1928-06-27","start":"1861-03-17"},{"name":"CORTEMILIA","start":"1
928-06-28","end":"1947-05-22"},{"name":"CORTEMILIA","end":"9999-12-31","star
t":"1947-05-23"}],"G214":[{"end":"1923-02-13","start":"1920-10-16","name":"P
ADERGNONE"},{"end":"1928-04-18","start":"1923-02-14","name":"PADERGNONE"},{"
name":"PADERGNONE","start":"1952-09-20","end":"2015-12-31"}],"D675":[{"end":
"9999-12-31","start":"1863-04-23","name":"FONTANETO D'AGOGNA"}],"L121":[{"na
me":"TERRAGNOLO","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-
14","end":"9999-12-31","name":"TERRAGNOLO"}],"C238":[{"name":"CASTELNUOVO D'
ISTRIA","start":"1921-01-05","end":"1923-02-07"},{"start":"1923-02-08","end"
:"1928-11-21","name":"CASTELNUOVO D'ISTRIA"},{"name":"CASTELNUOVO D'ISTRIA",
"end":"1947-09-15","start":"1928-11-22"}],"F059":[{"name":"MATTINATA","start
":"1955-08-19","end":"9999-12-31"}],"F353":[{"name":"MONDRONE","start":"1861
-03-17","end":"1928-05-11"}],"B041":[{"start":"1867-02-23","end":"1927-01-11
","name":"BORGOSESIA"},{"end":"1928-09-05","start":"1927-01-12","name":"BORG
OSESIA"},{"start":"1928-09-06","end":"1939-10-03","name":"BORGOSESIA"},{"sta
rt":"1939-10-04","end":"9999-12-31","name":"BORGOSESIA"}],"L919":[{"name":"V
ILLA CARCINA","end":"9999-12-31","start":"1928-02-05"}],"F614":[{"name":"MON
TERUBBIANO","end":"1910-07-25","start":"1868-08-27"},{"name":"MONTERUBBIANO"
,"start":"1910-07-26","end":"2009-07-11"},{"name":"MONTERUBBIANO","start":"2
009-07-12","end":"9999-12-31"}],"L616":[{"start":"1958-05-28","end":"9999-12
-31","name":"VALLESACCARDA"}],"E286":[{"end":"1928-10-30","start":"1861-03-1
7","name":"IMBERIDO"}],"A776":[{"name":"BELV\u00cc","start":"1861-03-17","en
d":"1927-01-11"},{"name":"BELV\u00cc","start":"1927-01-12","end":"9999-12-31
"}],"C058":[{"name":"CASTEL BARONIA","start":"1861-03-17","end":"9999-12-31"
}],"G941":[{"start":"1921-01-05","end":"1923-02-07","name":"POSTUMIA"},{"nam
e":"POSTUMIA","start":"1923-02-08","end":"1925-08-21"},{"name":"POSTUMIA","e
nd":"1935-03-01","start":"1925-08-22"},{"start":"1935-03-02","end":"1947-09-
15","name":"POSTUMIA GROTTE"}],"H438":[{"end":"9999-12-31","start":"1861-03-
17","name":"ROCCA SAN FELICE"}],"E496":[{"name":"LAVENO-MOMBELLO","start":"1
928-01-28","end":"9999-12-31"}],"F239":[{"name":"MIRANDA","end":"1928-08-07"
,"start":"1861-03-17"},{"name":"MIRANDA","end":"1970-03-02","start":"1937-02
-24"},{"start":"1970-03-03","end":"9999-12-31","name":"MIRANDA"}],"F832":[{"
start":"1867-09-02","end":"1968-04-05","name":"MUZZANA DEL TURGNANO"},{"end"
:"9999-12-31","start":"1968-04-06","name":"MUZZANA DEL TURGNANO"}],"G534":[{
"name":"PIACENZA D'ADIGE","start":"1867-08-26","end":"9999-12-31"}],"A593":[
{"start":"1869-07-22","end":"1905-04-06","name":"BALESTRINO"},{"name":"BALES
TRINO","end":"1927-01-11","start":"1905-04-07"},{"start":"1927-01-12","end":
"9999-12-31","name":"BALESTRINO"}],"E067":[{"name":"GIVOLETTO","end":"1928-0
4-02","start":"1861-03-17"},{"name":"GIVOLETTO","start":"1947-01-21","end":"
9999-12-31"}],"G845":[{"name":"PONTE DI NOSSA","end":"1919-12-14","start":"1
861-03-17"},{"name":"PONTE DI NOSSA","start":"1919-12-15","end":"1927-08-05"
},{"start":"1927-08-06","end":"1928-09-24","name":"PONTE DI NOSSA"}],"H134":
[{"name":"QUINTO VICENTINO","start":"1867-09-02","end":"9999-12-31"}],"G936"
:[{"name":"POSTAL","end":"1923-02-13","start":"1920-10-16"},{"name":"POSTAL"
,"end":"1927-01-11","start":"1923-02-14"},{"name":"POSTAL","end":"1993-04-13
","start":"1927-01-12"},{"name":"POSTAL","start":"1993-04-14","end":"9999-12
-31"}],"G639":[{"name":"PIEVE DEL CAIRO","end":"9999-12-31","start":"1890-11
-28"}],"A469":[{"end":"1928-12-02","start":"1861-03-17","name":"ASNAGO"}],"C
518":[{"end":"9999-12-31","start":"1886-07-19","name":"CERRETO LAZIALE"}],"Z
203":[{"name":"Arabia Saudita","start":"0000-01-01","end":"9999-12-31"}],"L6
91":[{"name":"VARZO","end":"1992-05-22","start":"1861-03-17"},{"end":"9999-1
2-31","start":"1992-05-23","name":"VARZO"}],"B256":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"BUGNARA"}],"F827":[{"start":"1869-02-01","end":"192
3-10-13","name":"MUSOCCO"}],"D103":[{"name":"COSTA VALLE IMAGNA","start":"18
64-11-07","end":"9999-12-31"}],"F160":[{"name":"MESTRIAGO","start":"1920-10-
16","end":"1923-02-13"},{"name":"MESTRIAGO","end":"1928-10-06","start":"1923
-02-14"}],"H378":[{"start":"1861-03-17","end":"1940-05-06","name":"ROBURENT"
},{"end":"9999-12-31","start":"1952-09-12","name":"ROBURENT"},{"name":"ROBUR
ENTO","start":"1940-05-07","end":"1952-09-11"}],"B728":[{"name":"CARATE LARI
O","end":"1927-12-23","start":"1862-12-29"}],"A243":[{"start":"1861-03-17","
end":"1927-01-11","name":"ALVIGNANO"},{"name":"ALVIGNANO","start":"1927-01-1
2","end":"1945-08-31"},{"name":"ALVIGNANO","start":"1945-09-01","end":"9999-
12-31"}],"B609":[{"start":"1869-05-17","end":"9999-12-31","name":"CANNARA"}]
,"A028":[{"end":"1934-04-06","start":"1861-03-17","name":"ACIREALE"},{"start
":"1934-04-07","end":"9999-12-31","name":"ACIREALE"}],"C915":[{"end":"1923-0
2-13","start":"1920-10-16","name":"COMANO"},{"start":"1923-02-14","end":"192
8-04-02","name":"COMANO"}],"B906":[{"start":"1936-07-29","end":"1962-03-27",
"name":"CASALNUOVO LUCANO"},{"name":"SAN PAOLO ALBANESE","end":"1936-07-28",
"start":"1863-04-10"},{"start":"1962-03-28","end":"9999-12-31","name":"SAN P
AOLO ALBANESE"}],"C278":[{"end":"9999-12-31","start":"1906-08-12","name":"CA
STELVECCHIO CALVISIO"}],"D635":[{"name":"FLORESTA","end":"9999-12-31","start
":"1861-03-17"}],"D849":[{"name":"GAGLIANO CASTELFERRATO","start":"1862-12-2
9","end":"1927-01-11"},{"name":"GAGLIANO CASTELFERRATO","end":"9999-12-31","
start":"1927-01-12"}],"D242":[{"end":"1928-02-25","start":"1861-03-17","name
":"DAGNENTE"}],"E311":[{"name":"INVERSO PINASCA","end":"1928-05-30","start":
"1861-03-17"},{"end":"9999-12-31","start":"1947-08-03","name":"INVERSO PINAS
CA"}],"I509":[{"name":"SCARIA","start":"1861-03-17","end":"1928-04-30"}],"A1
02":[{"name":"AIELLO CALABRO","end":"1934-03-06","start":"1928-05-19"},{"nam
e":"AIELLO CALABRO","start":"1934-03-07","end":"1937-07-17"},{"start":"1937-
07-18","end":"9999-12-31","name":"AIELLO CALABRO"}],"C811":[{"start":"1861-0
3-17","end":"9999-12-31","name":"COCULLO"}],"I604":[{"name":"SENALES","start
":"1923-04-27","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31",
"name":"SENALES"},{"name":"SENALES IN VENOSTA","end":"1923-02-13","start":"1
920-10-16"},{"name":"SENALES IN VENOSTA","start":"1923-02-14","end":"1923-04
-26"}],"B504":[{"name":"CAMPELLO SUL CLITUNNO","start":"1863-04-13","end":"1
925-11-16"},{"name":"CAMPELLO SUL CLITUNNO","end":"1927-04-07","start":"1925
-11-17"},{"name":"CAMPELLO SUL CLITUNNO","start":"1930-09-14","end":"9999-12
-31"}],"F453":[{"name":"MONTECAROTTO","start":"1861-03-17","end":"9999-12-31
"}],"B261":[{"start":"1861-03-17","end":"1992-04-15","name":"BULCIAGO"},{"na
me":"BULCIAGO","start":"1992-04-16","end":"9999-12-31"}],"E195":[{"name":"GR
OPELLO CAIROLI","start":"1888-03-12","end":"9999-12-31"}],"B076":[{"name":"B
OSCOREALE","start":"1877-03-09","end":"1928-04-27"},{"end":"1946-11-08","sta
rt":"1946-02-28","name":"BOSCOREALE"},{"name":"BOSCOREALE","end":"9999-12-31
","start":"1946-11-09"}],"D592":[{"name":"FILETTO","start":"1861-03-17","end
":"9999-12-31"}],"G032":[{"name":"OLIVA GESSI","end":"1928-10-15","start":"1
863-03-30"},{"name":"OLIVA GESSI","start":"1946-12-19","end":"9999-12-31"}],
"A335":[{"end":"9999-12-31","start":"1879-03-28","name":"APPIGNANO DEL TRONT
O"}],"E480":[{"name":"LAUREANA CILENTO","end":"9999-12-31","start":"1862-12-
29"}],"L223":[{"name":"TORNACO","start":"1861-03-17","end":"9999-12-31"}],"I
068":[{"name":"SAN PANCRAZIO PARMENSE","start":"1866-01-21","end":"1924-01-0
6"},{"end":"1943-05-28","start":"1924-01-07","name":"SAN PANCRAZIO PARMENSE"
}],"A702":[{"name":"BASSANO BRESCIANO","end":"9999-12-31","start":"1867-11-2
5"}],"M348":[{"end":"9999-12-31","start":"2015-01-30","name":"LA VALLETTA BR
IANZA"}],"G027":[{"start":"1928-02-21","end":"1953-09-12","name":"OLGIATE CA
LCO"}],"F045":[{"name":"MASSIMENO","end":"1923-02-13","start":"1920-10-16"},
{"end":"1928-04-02","start":"1923-02-14","name":"MASSIMENO"},{"name":"MASSIM
ENO","end":"9999-12-31","start":"1952-05-17"}],"A475":[{"name":"ASSISI","sta
rt":"1861-03-17","end":"1929-02-20"},{"start":"1929-02-21","end":"9999-12-31
","name":"ASSISI"}],"I816":[{"name":"SOMANA","start":"1861-03-17","end":"192
8-01-26"}],"E795":[{"name":"MADONNA DEL SASSO","start":"1928-03-07","end":"1
992-05-22"},{"name":"MADONNA DEL SASSO","end":"9999-12-31","start":"1992-05-
23"}],"A176":[{"start":"1861-03-17","end":"9999-12-31","name":"ALCAMO"}],"B4
24":[{"start":"1861-03-17","end":"1928-06-14","name":"CALOPEZZATI"},{"name":
"CALOPEZZATI","start":"1928-06-15","end":"1937-07-17"},{"name":"CALOPEZZATI"
,"start":"1937-07-18","end":"9999-12-31"}],"B350":[{"name":"CAFASSE","end":"
1928-04-14","start":"1861-03-17"},{"name":"CAFASSE","end":"9999-12-31","star
t":"1928-04-15"}],"H734":[{"name":"SALZA DI PINEROLO","end":"1928-04-30","st
art":"1892-04-29"},{"start":"1947-01-09","end":"9999-12-31","name":"SALZA DI
 PINEROLO"}],"B002":[{"name":"BORGIA","end":"9999-12-31","start":"1861-03-17
"}],"L048":[{"name":"TARANTASCA","start":"1861-03-17","end":"9999-12-31"}],"
D094":[{"end":"1927-01-11","start":"1861-03-17","name":"COSSATO"},{"end":"19
30-04-05","start":"1927-01-12","name":"COSSATO"},{"name":"COSSATO","start":"
1930-04-06","end":"1992-04-15"},{"name":"COSSATO","start":"1992-04-16","end"
:"9999-12-31"}],"A884":[{"end":"9999-12-31","start":"1861-03-17","name":"BIS
EGNA"}],"D703":[{"name":"FORL\u00cc DEL SANNIO","end":"1928-05-30","start":"
1863-02-06"},{"start":"1928-05-31","end":"1946-04-25","name":"FORL\u00cc DEL
 SANNIO"},{"name":"FORL\u00cc DEL SANNIO","start":"1946-04-26","end":"1970-0
3-02"},{"name":"FORL\u00cc DEL SANNIO","end":"9999-12-31","start":"1970-03-0
3"}],"F760":[{"name":"MORUZZO","start":"1866-11-19","end":"1968-04-05"},{"na
me":"MORUZZO","end":"9999-12-31","start":"1968-04-06"}],"B128":[{"end":"1907
-05-09","start":"1868-02-14","name":"BREDA DI PIAVE"},{"end":"9999-12-31","s
tart":"1907-05-10","name":"BREDA DI PIAVE"}],"I429":[{"name":"SARDIGLIANO","
end":"1928-12-14","start":"1861-03-17"},{"end":"1929-04-11","start":"1928-12
-15","name":"SARDIGLIANO"},{"end":"9999-12-31","start":"1929-04-12","name":"
SARDIGLIANO"}],"M366":[{"name":"DIMARO FOLGARIDA","end":"9999-12-31","start"
:"2016-01-01"}],"G489":[{"start":"1930-09-09","end":"9999-12-31","name":"PES
CHIERA DEL GARDA"},{"end":"1930-09-08","start":"1867-08-26","name":"PESCHIER
A SUL LAGO DI GARDA"}],"L051":[{"name":"TARCES","end":"1927-01-11","start":"
1923-04-27"},{"name":"TARCES","end":"1928-01-26","start":"1927-01-12"},{"sta
rt":"1920-10-16","end":"1923-02-13","name":"TARRES DI SOPRA"},{"end":"1923-0
4-26","start":"1923-02-14","name":"TARRES DI SOPRA"}],"A957":[{"end":"9999-1
2-31","start":"1911-04-18","name":"BOMPENSIERE"}],"A613":[{"name":"BAONE","e
nd":"9999-12-31","start":"1866-11-19"}],"A354":[{"end":"1959-04-07","start":
"1866-11-19","name":"ARBA"},{"end":"1968-04-05","start":"1959-04-08","name":
"ARBA"},{"name":"ARBA","end":"9999-12-31","start":"1968-04-06"}],"I031":[{"e
nd":"9999-12-31","start":"1861-03-17","name":"SAN MAURO CILENTO"}],"C163":[{
"name":"CASTELLETTO MONFORTE","start":"1861-03-17","end":"1930-12-31"}],"F91
1":[{"end":"9999-12-31","start":"1863-04-13","name":"NOCERA UMBRA"}],"A420":
[{"end":"1923-02-13","start":"1920-10-16","name":"ARNAGO"},{"name":"ARNAGO",
"start":"1923-02-14","end":"1928-09-05"}],"D186":[{"name":"CROTTA D'ADDA","s
tart":"1861-03-17","end":"9999-12-31"}],"H224":[{"end":"1927-08-02","start":
"1865-12-25","name":"REGGIO DI CALABRIA"},{"start":"1927-08-03","end":"1933-
01-25","name":"REGGIO DI CALABRIA"},{"start":"1933-01-26","end":"1977-02-26"
,"name":"REGGIO DI CALABRIA"},{"end":"9999-12-31","start":"1977-02-27","name
":"REGGIO DI CALABRIA"}],"L511":[{"start":"1930-06-30","end":"1963-02-19","n
ame":"USMATE VELATE"},{"name":"USMATE VELATE","start":"1963-02-20","end":"20
09-06-29"},{"start":"2009-06-30","end":"9999-12-31","name":"USMATE VELATE"},
{"end":"1930-06-29","start":"1869-03-11","name":"VELATE MILANESE"}],"D874":[
{"name":"GALLICANO","end":"1923-11-23","start":"1861-03-17"},{"start":"1923-
11-24","end":"1948-12-09","name":"GALLICANO"},{"name":"GALLICANO","end":"999
9-12-31","start":"1948-12-10"}],"H688":[{"end":"9999-12-31","start":"1861-03
-17","name":"SALAPARUTA"}],"C698":[{"start":"1861-03-17","end":"1927-05-22",
"name":"CIMMO"},{"end":"1927-12-30","start":"1927-05-23","name":"TAVERNOLE C
IMMO"},{"name":"TAVERNOLE SUL MELLA","start":"1927-12-31","end":"1955-03-11"
},{"start":"1955-03-12","end":"9999-12-31","name":"TAVERNOLE SUL MELLA"}],"I
046":[{"start":"1861-03-17","end":"1925-11-29","name":"SAN MINIATO"},{"start
":"1925-11-30","end":"1928-12-31","name":"SAN MINIATO"},{"name":"SAN MINIATO
","start":"1929-01-01","end":"9999-12-31"}],"B880":[{"end":"9999-12-31","sta
rt":"1861-03-17","name":"CASALECCHIO DI RENO"}],"F815":[{"name":"MURLO","sta
rt":"1861-03-17","end":"9999-12-31"}],"I753":[{"name":"SINOPOLI","start":"18
61-03-17","end":"9999-12-31"}],"F064":[{"name":"MAZZANO ROMANO","start":"187
2-08-26","end":"9999-12-31"}],"L066":[{"end":"1927-01-11","start":"1861-03-1
7","name":"TAVAGNASCO"},{"name":"TAVAGNASCO","end":"1929-05-04","start":"192
7-01-12"},{"start":"1947-01-15","end":"9999-12-31","name":"TAVAGNASCO"}],"L2
71":[{"name":"TORREGROTTA","end":"9999-12-31","start":"1923-11-28"}],"M351":
[{"end":"9999-12-31","start":"2016-01-01","name":"AMBLAR-DON"}],"C927":[{"na
me":"COMISO","end":"1927-01-11","start":"1861-03-17"},{"name":"COMISO","end"
:"1937-07-16","start":"1927-01-12"},{"name":"COMISO","start":"1937-07-17","e
nd":"9999-12-31"}],"H030":[{"name":"PREMENO","end":"1927-12-23","start":"186
1-03-17"},{"name":"PREMENO","start":"1927-12-24","end":"1928-01-25"},{"name"
:"PREMENO","start":"1928-01-26","end":"1992-05-22"},{"name":"PREMENO","start
":"1992-05-23","end":"9999-12-31"}],"L733":[{"end":"1927-01-11","start":"186
1-03-17","name":"VENEGONO INFERIORE"},{"name":"VENEGONO INFERIORE","end":"19
28-09-24","start":"1927-01-12"},{"name":"VENEGONO INFERIORE","end":"9999-12-
31","start":"1960-09-29"}],"C324":[{"name":"CASTIONE DELLA PRESOLANA","start
":"1863-07-13","end":"1927-11-19"},{"end":"1958-09-09","start":"1927-11-20",
"name":"CASTIONE DELLA PRESOLANA"},{"start":"1958-09-10","end":"9999-12-31",
"name":"CASTIONE DELLA PRESOLANA"}],"C450":[{"end":"1927-01-11","start":"186
5-12-14","name":"CELLIO"},{"name":"CELLIO","start":"1927-01-12","end":"2017-
12-31"}],"I087":[{"start":"1923-04-27","end":"1927-01-11","name":"SAN PIETRO
"},{"name":"SAN PIETRO","start":"1927-01-12","end":"1929-02-21"},{"end":"192
3-02-13","start":"1920-10-16","name":"SAN PIETRO IN VALLE AURINA"},{"end":"1
923-04-26","start":"1923-02-14","name":"SAN PIETRO IN VALLE AURINA"}],"L497"
:[{"end":"9999-12-31","start":"1866-11-19","name":"URBANA"}],"D790":[{"name"
:"FRATTAMINORE","start":"1886-02-12","end":"1928-05-28"},{"end":"9999-12-31"
,"start":"1928-05-29","name":"FRATTAMINORE"}],"E701":[{"start":"1863-02-23",
"end":"1928-11-09","name":"LOVENO GRUMELLO"}],"E824":[{"end":"1928-05-29","s
tart":"1861-03-17","name":"MAGOGNINO"}],"L818":[{"name":"VEZZANO","end":"192
7-01-11","start":"1923-04-27"},{"name":"VEZZANO","end":"1928-05-12","start":
"1927-01-12"},{"name":"VEZZANO IN VENOSTA","end":"1923-02-13","start":"1920-
10-16"},{"end":"1923-04-26","start":"1923-02-14","name":"VEZZANO IN VENOSTA"
}],"D612":[{"end":"1910-07-30","start":"1869-12-10","name":"FIRENZE"},{"star
t":"1910-07-31","end":"1928-12-15","name":"FIRENZE"},{"end":"1939-11-14","st
art":"1928-12-16","name":"FIRENZE"},{"end":"9999-12-31","start":"1939-11-15"
,"name":"FIRENZE"}],"C932":[{"start":"1861-03-17","end":"1928-04-15","name":
"COMNAGO"}],"D485":[{"name":"FAMIE","end":"1927-06-01","start":"1923-04-27"}
,{"name":"FAMLIE","start":"1921-01-05","end":"1923-02-07"},{"start":"1923-02
-08","end":"1923-04-26","name":"FAMLIE"}],"B983":[{"end":"2017-05-04","start
":"1864-11-07","name":"CASOLE BRUZIO"}],"H755":[{"end":"1955-08-11","start":
"1940-08-10","name":"SAMPEIRE"},{"start":"1955-08-12","end":"9999-12-31","na
me":"SAMPEYRE"},{"start":"1861-03-17","end":"1940-08-09","name":"SAMP\u00c8Y
RE"}],"D147":[{"name":"CREMIA","start":"1861-03-17","end":"9999-12-31"}],"Z2
47":[{"start":"0000-01-01","end":"9999-12-31","name":"Malaysia"}],"A207":[{"
end":"1927-01-11","start":"1887-07-22","name":"ALLERONA"},{"name":"ALLERONA"
,"end":"1938-03-17","start":"1927-01-12"},{"name":"ALLERONA","start":"1938-0
3-18","end":"9999-12-31"}],"I153":[{"name":"SAN SECONDO PARMENSE","start":"1
862-10-20","end":"9999-12-31"}],"G555":[{"end":"1927-01-11","start":"1861-03
-17","name":"PIANELLA"},{"end":"9999-12-31","start":"1927-01-12","name":"PIA
NELLA"}],"H461":[{"name":"ROCCHETTA DI VARA","start":"1863-03-30","end":"192
3-12-21"},{"end":"1929-05-18","start":"1923-12-22","name":"ROCCHETTA DI VARA
"},{"end":"9999-12-31","start":"1929-05-19","name":"ROCCHETTA DI VARA"}],"I9
42":[{"name":"STAZZEMA","end":"1929-06-01","start":"1861-03-17"},{"name":"ST
AZZEMA","start":"1929-06-02","end":"9999-12-31"}],"L133":[{"name":"TERRATI",
"end":"1928-01-17","start":"1861-03-17"}],"G208":[{"name":"PACECO","end":"19
38-09-19","start":"1861-03-17"},{"start":"1946-02-07","end":"1979-10-19","na
me":"PACECO"},{"end":"9999-12-31","start":"1979-10-20","name":"PACECO"}],"E1
01":[{"name":"GORLA MAGGIORE","start":"1916-05-01","end":"1927-01-11"},{"sta
rt":"1927-01-12","end":"9999-12-31","name":"GORLA MAGGIORE"}],"E630":[{"star
t":"1870-01-02","end":"1923-12-21","name":"LIZZANO"},{"name":"LIZZANO","end"
:"1954-10-28","start":"1923-12-22"},{"name":"LIZZANO","end":"9999-12-31","st
art":"1954-10-29"}],"A312":[{"start":"1880-10-08","end":"1931-06-02","name":
"ANTIGNANO"},{"start":"1931-06-03","end":"1935-04-07","name":"ANTIGNANO"},{"
start":"1935-04-08","end":"9999-12-31","name":"ANTIGNANO"}],"G015":[{"name":
"OLBIA","end":"1958-08-14","start":"1939-10-24"},{"start":"1958-08-15","end"
:"1960-03-11","name":"OLBIA"},{"name":"OLBIA","start":"1960-03-12","end":"19
79-04-19"},{"name":"OLBIA","end":"2005-12-31","start":"1979-04-20"},{"end":"
2016-04-27","start":"2006-01-01","name":"OLBIA"},{"start":"2016-04-28","end"
:"9999-12-31","name":"OLBIA"},{"start":"1862-09-22","end":"1939-01-25","name
":"TERRANOVA PAUSANIA"},{"end":"1939-10-23","start":"1939-01-26","name":"TER
RANOVA PAUSANIA"}],"F608":[{"start":"1862-12-19","end":"1929-04-19","name":"
MONTEROSSO GRANA"},{"name":"MONTEROSSO GRANA","start":"1929-04-20","end":"99
99-12-31"}],"L440":[{"start":"1861-03-17","end":"9999-12-31","name":"TRIVOLZ
IO"}],"H155":[{"name":"RADICENA","start":"1861-03-17","end":"1928-03-26"}],"
D747":[{"end":"1928-05-02","start":"1861-03-17","name":"FOSSENO"}],"A500":[{
"name":"AURISINA","end":"1928-09-08","start":"1923-04-27"},{"name":"NABRESIN
A","start":"1921-01-05","end":"1923-02-07"},{"end":"1923-04-26","start":"192
3-02-08","name":"NABRESINA"}],"B896":[{"end":"9999-12-31","start":"1861-03-1
7","name":"CASALINCONTRADA"}],"D313":[{"name":"DOCCIO","end":"1927-01-11","s
tart":"1861-03-17"},{"end":"1928-09-05","start":"1927-01-12","name":"DOCCIO"
}],"D019":[{"end":"1924-02-21","start":"1861-03-17","name":"CORNATE"},{"star
t":"1924-02-22","end":"2009-12-17","name":"CORNATE D'ADDA"},{"name":"CORNATE
 D'ADDA","end":"9999-12-31","start":"2009-12-18"}],"H456":[{"name":"ROCCELLA
 IONICA","end":"9999-12-31","start":"1863-04-10"}],"F729":[{"name":"MORIAGO"
,"start":"1866-11-19","end":"1962-08-15"},{"start":"1962-08-16","end":"9999-
12-31","name":"MORIAGO DELLA BATTAGLIA"}],"I460":[{"name":"SASSOFELTRIO","en
d":"1929-05-14","start":"1861-03-17"},{"name":"SASSOFELTRIO","end":"1998-09-
03","start":"1947-11-25"},{"start":"1998-09-04","end":"2021-06-16","name":"S
ASSOFELTRIO"},{"name":"SASSOFELTRIO","end":"9999-12-31","start":"2021-06-17"
}],"E088":[{"start":"1861-03-17","end":"1927-12-28","name":"GONNOSTRAMATZA"}
,{"name":"GONNOSTRAMATZA","start":"1927-12-29","end":"1946-02-27"},{"end":"1
947-01-23","start":"1946-02-28","name":"GONNOSTRAMATZA"},{"end":"1974-08-19"
,"start":"1947-01-24","name":"GONNOSTRAMATZA"},{"name":"GONNOSTRAMATZA","end
":"9999-12-31","start":"1974-08-20"}],"C763":[{"end":"9999-12-31","start":"1
861-03-17","name":"CIVITA"}],"F562":[{"name":"MONTEMARZINO","end":"1929-04-1
9","start":"1861-03-17"},{"name":"MONTEMARZINO","start":"1929-04-20","end":"
9999-12-31"}],"G864":[{"end":"9999-12-31","start":"1882-09-03","name":"PONTI
DA"}],"D654":[{"end":"9999-12-31","start":"1866-11-19","name":"FOLLINA"}],"M
139":[{"name":"ZAFFERANA ETNEA","start":"1861-03-17","end":"1934-04-06"},{"n
ame":"ZAFFERANA ETNEA","end":"1951-05-05","start":"1934-04-07"},{"name":"ZAF
FERANA ETNEA","start":"1951-05-06","end":"9999-12-31"}],"D786":[{"name":"UMB
ERTIDE","end":"9999-12-31","start":"1863-04-13"}],"M260":[{"start":"1973-03-
14","end":"9999-12-31","name":"CASAPESENNA"}],"L605":[{"end":"1927-01-11","s
tart":"1907-06-21","name":"VALLEFREDDA"},{"name":"VALLEFREDDA","end":"1932-1
2-17","start":"1927-01-12"},{"end":"9999-12-31","start":"1932-12-18","name":
"VALLEMAIO"}],"D559":[{"name":"FIACCONE","start":"1861-03-17","end":"1927-07
-01"},{"start":"1927-07-02","end":"9999-12-31","name":"FRACONALTO"}],"C041":
[{"start":"1866-11-19","end":"9999-12-31","name":"CASTAGNARO"}],"B238":[{"en
d":"9999-12-31","start":"1861-03-17","name":"BUCCHIANICO"}],"F883":[{"name":
"NEVIGLIE","start":"1861-03-17","end":"9999-12-31"}],"H361":[{"name":"ROANA"
,"start":"1869-05-11","end":"1906-04-17"},{"name":"ROANA","end":"1928-08-06"
,"start":"1906-04-18"},{"start":"1928-08-07","end":"9999-12-31","name":"ROAN
A"}],"B058":[{"start":"1861-03-17","end":"1927-01-11","name":"BORRIANA"},{"n
ame":"BORRIANA","start":"1927-01-12","end":"1992-04-15"},{"name":"BORRIANA",
"start":"1992-04-16","end":"9999-12-31"}],"C728":[{"end":"9999-12-31","start
":"1863-07-13","name":"CISANO BERGAMASCO"}],"L340":[{"start":"1937-06-16","e
nd":"1955-06-08","name":"TRAVE"},{"name":"TRAVES","end":"1937-06-15","start"
:"1861-03-17"},{"end":"9999-12-31","start":"1955-06-09","name":"TRAVES"}],"C
256":[{"name":"CASTELRUGGIERO","start":"1861-03-17","end":"1929-04-11"}],"I6
18":[{"name":"SEPINO","start":"1861-03-17","end":"1970-03-02"},{"end":"9999-
12-31","start":"1970-03-03","name":"SEPINO"}],"H749":[{"name":"SAMMICHELE DI
 BARI","start":"1863-01-19","end":"1929-07-09"},{"name":"SAMMICHELE DI BARI"
,"end":"9999-12-31","start":"1929-07-10"}],"B518":[{"name":"CAMPO","start":"
1920-10-16","end":"1923-02-13"},{"name":"CAMPO","end":"1928-04-02","start":"
1923-02-14"}],"A412":[{"name":"ARLENA DI CASTRO","start":"1876-09-23","end":
"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31","name":"ARLENA DI CAS
TRO"}],"L173":[{"name":"TIONE","end":"1928-05-03","start":"1861-03-17"},{"na
me":"TIONE DEGLI ABRUZZI","start":"1928-05-04","end":"1951-12-31"},{"name":"
TIONE DEGLI ABRUZZI","start":"1952-01-01","end":"1954-10-20"},{"end":"9999-1
2-31","start":"1954-10-21","name":"TIONE DEGLI ABRUZZI"}],"A685":[{"name":"B
ARZANIGA","end":"1928-06-27","start":"1868-01-22"}],"G157":[{"name":"OSIMO",
"end":"9999-12-31","start":"1861-03-17"}],"D620":[{"name":"FIUME","end":"194
7-09-15","start":"1924-03-09"}],"I360":[{"name":"SANTO STEFANO DI SESSANIO",
"start":"1863-05-06","end":"9999-12-31"}],"B278":[{"name":"BURIASCO","end":"
9999-12-31","start":"1861-03-17"}],"C906":[{"end":"1927-10-18","start":"1861
-03-17","name":"COL SAN GIOVANNI"}],"D413":[{"end":"1923-02-13","start":"192
0-10-16","name":"EORES"},{"start":"1923-02-14","end":"1927-01-11","name":"EO
RES"},{"name":"EORES","start":"1927-01-12","end":"1929-01-22"}],"B915":[{"en
d":"1928-03-23","start":"1863-04-10","name":"TRINITAPOLI"},{"name":"TRINITAP
OLI","start":"1928-03-24","end":"2009-07-26"},{"start":"2009-07-27","end":"9
999-12-31","name":"TRINITAPOLI"}],"H356":[{"end":"9999-12-31","start":"1867-
08-26","name":"RIVOLI VERONESE"}],"H017":[{"end":"1925-10-31","start":"1861-
03-17","name":"PREDAPPIO"},{"start":"1925-11-01","end":"1927-03-15","name":"
PREDAPPIO"},{"name":"PREDAPPIO","end":"1992-04-15","start":"1936-03-25"},{"n
ame":"PREDAPPIO","end":"9999-12-31","start":"1992-04-16"},{"end":"1936-03-24
","start":"1927-03-16","name":"PREDAPPIO NUOVA"}],"C609":[{"name":"CHIANNI",
"start":"1861-03-17","end":"9999-12-31"}],"F980":[{"name":"NURACHI","start":
"1861-03-17","end":"1927-10-11"},{"start":"1946-02-07","end":"1974-08-19","n
ame":"NURACHI"},{"name":"NURACHI","start":"1974-08-20","end":"9999-12-31"}],
"C076":[{"name":"CASTEL D'ARIO","start":"1867-06-24","end":"9999-12-31"}],"A
758":[{"name":"BELLUNO VERONESE","start":"1866-11-19","end":"1928-10-16"}],"
M040":[{"name":"VILLE SAN SEBASTIANO","end":"1928-12-07","start":"1861-03-17
"}],"C261":[{"end":"9999-12-31","start":"1861-03-17","name":"CASTEL SAN GIOV
ANNI"}],"C504":[{"name":"CERESOLE ALBA","start":"1871-06-01","end":"9999-12-
31"}],"A226":[{"end":"1927-01-11","start":"1861-03-17","name":"ALTARE"},{"na
me":"ALTARE","end":"1952-05-23","start":"1927-01-12"},{"name":"ALTARE","star
t":"1952-05-24","end":"1968-08-02"},{"name":"ALTARE","start":"1968-08-03","e
nd":"9999-12-31"}],"L580":[{"name":"VALGRANA","start":"1861-03-17","end":"99
99-12-31"}],"H557":[{"start":"1861-03-17","end":"1927-08-02","name":"ROSAL\u
00cc"}],"M179":[{"start":"1861-03-17","end":"1927-01-11","name":"ZIMONE"},{"
start":"1927-01-12","end":"1992-04-15","name":"ZIMONE"},{"name":"ZIMONE","st
art":"1992-04-16","end":"9999-12-31"}],"E742":[{"name":"LUNAMATRONA","end":"
1927-08-05","start":"1861-03-17"},{"start":"1927-08-06","end":"1927-11-02","
name":"LUNAMATRONA"},{"start":"1927-11-03","end":"1946-02-08","name":"LUNAMA
TRONA"},{"name":"LUNAMATRONA","start":"1946-02-09","end":"1946-02-27"},{"nam
e":"LUNAMATRONA","start":"1946-02-28","end":"1951-01-11"},{"start":"1951-01-
12","end":"1974-08-19","name":"LUNAMATRONA"},{"end":"2005-12-31","start":"19
74-08-20","name":"LUNAMATRONA"},{"name":"LUNAMATRONA","start":"2006-01-01","
end":"2016-04-27"},{"end":"9999-12-31","start":"2016-04-28","name":"LUNAMATR
ONA"}],"F092":[{"name":"MEGLIADINO SAN VITALE","start":"1866-11-19","end":"9
999-12-31"}],"B811":[{"name":"CARPANETA CON DOSIMO","end":"1928-12-14","star
t":"1868-01-01"}],"I901":[{"name":"SPIGNO MONFERRATO","start":"1863-04-13","
end":"1928-12-05"},{"name":"SPIGNO MONFERRATO","end":"1947-10-01","start":"1
928-12-06"},{"end":"1953-11-16","start":"1947-10-02","name":"SPIGNO MONFERRA
TO"},{"name":"SPIGNO MONFERRATO","end":"9999-12-31","start":"1953-11-17"}],"
G757":[{"end":"1923-03-18","start":"1861-03-17","name":"POGGIO CATINO"},{"st
art":"1923-03-19","end":"1927-01-11","name":"POGGIO CATINO"},{"end":"9999-12
-31","start":"1927-01-12","name":"POGGIO CATINO"}],"D834":[{"name":"GABBIONE
TA-BINANUOVA","end":"9999-12-31","start":"1928-04-07"}],"A320":[{"name":"ANZ
ANO DEGLI IRPINI","start":"1862-12-29","end":"1929-02-23"},{"end":"1931-05-2
1","start":"1929-02-24","name":"ANZANO DEGLI IRPINI"},{"name":"ANZANO DI PUG
LIA","end":"1939-01-07","start":"1931-05-22"},{"end":"9999-12-31","start":"1
939-01-08","name":"ANZANO DI PUGLIA"}],"A454":[{"end":"1927-01-11","start":"
1861-03-17","name":"ARZANA"},{"name":"ARZANA","end":"2005-12-31","start":"19
27-01-12"},{"start":"2006-01-01","end":"2016-04-27","name":"ARZANA"},{"start
":"2016-04-28","end":"9999-12-31","name":"ARZANA"}],"H803":[{"name":"SAN COL
OMBANO AL LAMBRO","start":"1863-01-26","end":"9999-12-31"}],"G389":[{"name":
"PAVIA DI UDINE","end":"1968-04-05","start":"1867-09-02"},{"start":"1968-04-
06","end":"2006-07-19","name":"PAVIA DI UDINE"},{"name":"PAVIA DI UDINE","st
art":"2006-07-20","end":"9999-12-31"}],"G083":[{"name":"ORA","end":"1923-02-
13","start":"1920-10-16"},{"name":"ORA","end":"1948-03-13","start":"1923-02-
14"},{"end":"1985-09-10","start":"1948-03-14","name":"ORA"},{"name":"ORA","s
tart":"1985-09-11","end":"9999-12-31"}],"D295":[{"name":"DIANO CALDERINA","s
tart":"1861-03-17","end":"1924-02-03"}],"E953":[{"start":"1861-03-17","end":
"9999-12-31","name":"MARIANOPOLI"}],"A158":[{"name":"ALBIANO","start":"1920-
10-16","end":"1923-02-13"},{"name":"ALBIANO","start":"1923-02-14","end":"192
8-09-08"},{"end":"1952-05-16","start":"1928-09-09","name":"ALBIANO"},{"name"
:"ALBIANO","start":"1952-05-17","end":"9999-12-31"}],"E617":[{"name":"LISSON
E","end":"2009-06-29","start":"1869-04-16"},{"name":"LISSONE","end":"9999-12
-31","start":"2009-06-30"}],"I805":[{"name":"SOLOFRA","end":"1927-10-18","st
art":"1861-03-17"},{"start":"1927-10-19","end":"9999-12-31","name":"SOLOFRA"
}],"L287":[{"name":"TORRI DEL BENACO","end":"9999-12-31","start":"1867-08-26
"}],"H250":[{"end":"1935-04-07","start":"1863-02-16","name":"REVIGLIASCO D'A
STI"},{"name":"REVIGLIASCO D'ASTI","start":"1935-04-08","end":"9999-12-31"}]
,"D545":[{"start":"1861-03-17","end":"9999-12-31","name":"FEROLETO DELLA CHI
ESA"}],"Z311":[{"end":"9999-12-31","start":"0000-01-01","name":"Congo"}],"E1
42":[{"name":"GRANITI","start":"1861-03-17","end":"9999-12-31"}],"I071":[{"n
ame":"SAN PAOLO DI JESI","end":"1928-05-28","start":"1863-02-23"},{"name":"S
AN PAOLO DI JESI","start":"1946-02-06","end":"9999-12-31"}],"I266":[{"name":
"SANT'ANDREA APOSTOLO DELLO IONIO","end":"9999-12-31","start":"1863-07-13"}]
,"F244":[{"name":"MISANO ADRIATICO","start":"1938-08-18","end":"1992-04-15"}
,{"end":"9999-12-31","start":"1992-04-16","name":"MISANO ADRIATICO"},{"name"
:"MISANO IN VILLA VITTORIA","start":"1870-01-02","end":"1938-08-17"}],"C350"
:[{"start":"1873-12-01","end":"1927-08-02","name":"CATAFORIO"}],"L246":[{"st
art":"1866-11-19","end":"1968-04-05","name":"TORREANO"},{"name":"TORREANO","
start":"1968-04-06","end":"9999-12-31"}],"C972":[{"start":"1861-03-17","end"
:"9999-12-31","name":"CONTROGUERRA"}],"H900":[{"end":"9999-12-31","start":"1
861-03-17","name":"SAN GIORIO DI SUSA"}],"L397":[{"name":"TREVI","end":"1925
-11-16","start":"1861-03-17"},{"start":"1925-11-17","end":"9999-12-31","name
":"TREVI"}],"C424":[{"start":"1988-10-25","end":"9999-12-31","name":"CEGLIE 
MESSAPICA"},{"start":"1864-03-31","end":"1927-01-11","name":"CEGLIE MESSAPIC
O"},{"start":"1927-01-12","end":"1988-10-24","name":"CEGLIE MESSAPICO"}],"E2
03":[{"start":"1861-03-17","end":"1928-03-13","name":"GROSSO"},{"start":"194
7-01-09","end":"9999-12-31","name":"GROSSO"}],"F699":[{"name":"MONTRIGIASCO"
,"end":"1928-02-22","start":"1861-03-17"}],"D948":[{"name":"GAVORRANO","star
t":"1861-03-17","end":"1960-07-18"},{"start":"1960-07-19","end":"9999-12-31"
,"name":"GAVORRANO"}],"L994":[{"name":"VILLANTERIO","start":"1872-10-15","en
d":"9999-12-31"}],"G549":[{"name":"PIANCOGNO","start":"1963-02-13","end":"99
99-12-31"}],"F996":[{"name":"OCRE","start":"1861-03-17","end":"9999-12-31"}]
,"L231":[{"end":"1927-01-11","start":"1861-03-17","name":"TORP\u00c8"},{"nam
e":"TORP\u00c8","start":"1927-01-12","end":"9999-12-31"}],"C128":[{"name":"C
ASTELLALTO","end":"9999-12-31","start":"1861-03-17"}],"G644":[{"end":"1923-0
2-13","start":"1920-10-16","name":"PIEVE DI LEDRO"},{"name":"PIEVE DI LEDRO"
,"start":"1923-02-14","end":"1928-02-08"},{"name":"PIEVE DI LEDRO","start":"
1952-09-20","end":"1955-03-01"},{"name":"PIEVE DI LEDRO","start":"1955-03-02
","end":"2009-12-31"}],"M165":[{"name":"ZERBA","end":"1923-07-22","start":"1
861-03-17"},{"name":"ZERBA","end":"9999-12-31","start":"1923-07-23"}],"F594"
:[{"name":"MONTERCHI","end":"1927-04-13","start":"1861-03-17"},{"name":"MONT
ERCHI","end":"1939-02-17","start":"1927-04-14"},{"end":"9999-12-31","start":
"1939-02-18","name":"MONTERCHI"}],"H149":[{"name":"RACCOLANA","end":"1928-05
-01","start":"1866-11-19"}],"L596":[{"start":"1861-03-17","end":"9999-12-31"
,"name":"VALLEBONA"}],"G299":[{"end":"1923-02-13","start":"1920-10-16","name
":"PALUS"},{"name":"PALUS","start":"1923-02-14","end":"1923-04-26"},{"name":
"PLAUS","end":"1927-01-11","start":"1923-04-27"},{"start":"1927-01-12","end"
:"1995-08-30","name":"PLAUS"},{"start":"1995-08-31","end":"9999-12-31","name
":"PLAUS"}],"I251":[{"end":"9999-12-31","start":"1861-03-17","name":"SANTA M
ARIA NUOVA"}],"D385":[{"end":"1927-01-11","start":"1861-03-17","name":"DUNO"
},{"name":"DUNO","end":"1928-02-15","start":"1927-01-12"},{"start":"1954-06-
04","end":"9999-12-31","name":"DUNO"}],"E850":[{"start":"1920-10-16","end":"
1923-02-13","name":"MAL\u00c9"},{"end":"1928-09-05","start":"1923-02-14","na
me":"MAL\u00c9"},{"name":"MAL\u00c9","end":"1953-01-03","start":"1928-09-06"
},{"end":"1970-08-04","start":"1953-01-04","name":"MAL\u00c9"},{"name":"MAL\
u00c9","end":"9999-12-31","start":"1970-08-05"}],"H070":[{"end":"9999-12-31"
,"start":"1861-03-17","name":"PRIZZI"}],"C002":[{"end":"9999-12-31","start":
"1863-01-19","name":"CASSANO ALL'IONIO"}],"L773":[{"end":"9999-12-31","start
":"1870-08-02","name":"VERNATE"}],"M329":[{"end":"9999-12-31","start":"2014-
01-01","name":"PRATOVECCHIO STIA"}],"M023":[{"start":"1861-03-17","end":"192
9-05-22","name":"VILLA SANT'ANGELO"},{"end":"9999-12-31","start":"1954-07-03
","name":"VILLA SANT'ANGELO"}],"B250":[{"name":"BUGGERRU","start":"1960-03-1
2","end":"1974-08-19"},{"start":"1974-08-20","end":"2005-12-31","name":"BUGG
ERRU"},{"end":"2016-04-27","start":"2006-01-01","name":"BUGGERRU"},{"end":"9
999-12-31","start":"2016-04-28","name":"BUGGERRU"}],"D401":[{"end":"9999-12-
31","start":"1861-03-17","name":"ELVA"}],"L164":[{"start":"1861-03-17","end"
:"9999-12-31","name":"TICENGO"}],"E785":[{"end":"1928-08-14","start":"1861-0
3-17","name":"MACERATA FELTRIA"},{"name":"MACERATA FELTRIA","start":"1928-08
-15","end":"1947-09-23"},{"end":"1998-09-03","start":"1947-09-24","name":"MA
CERATA FELTRIA"},{"start":"1998-09-04","end":"9999-12-31","name":"MACERATA F
ELTRIA"}],"G930":[{"name":"POSCANTE","end":"1926-02-18","start":"1861-03-17"
},{"name":"POSCANTE","end":"1928-03-27","start":"1926-02-19"}],"B803":[{"nam
e":"CARONA","end":"9999-12-31","start":"1861-03-17"}],"G452":[{"name":"P\u00
c8RGINE","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14","end
":"1929-01-24","name":"P\u00c8RGINE"},{"end":"1929-06-15","start":"1929-01-2
5","name":"PERGINE VALSUGANA"},{"name":"PERGINE VALSUGANA","start":"1929-06-
16","end":"1955-03-01"},{"name":"PERGINE VALSUGANA","start":"1955-03-02","en
d":"9999-12-31"}],"L468":[{"end":"1923-02-13","start":"1920-10-16","name":"T
URANO"},{"start":"1923-02-14","end":"1929-02-27","name":"TURANO"},{"name":"T
URANO","end":"1932-01-09","start":"1929-02-28"},{"end":"1934-05-03","start":
"1932-01-10","name":"VALVESTINO"},{"name":"VALVESTINO","end":"1947-10-01","s
tart":"1934-05-04"},{"name":"VALVESTINO","start":"1947-10-02","end":"9999-12
-31"}],"I877":[{"start":"1861-03-17","end":"9999-12-31","name":"SOVICILLE"}]
,"D643":[{"end":"9999-12-31","start":"1861-03-17","name":"FOGGIA"}],"A770":[
{"start":"1863-02-16","end":"1935-04-07","name":"BELVEGLIO"},{"name":"BELVEG
LIO","end":"9999-12-31","start":"1935-04-08"}],"E490":[{"name":"LA VALLE","e
nd":"1964-10-01","start":"1866-11-19"},{"end":"9999-12-31","start":"1964-10-
02","name":"LA VALLE AGORDINA"}],"A642":[{"name":"BARCONE","start":"1861-03-
17","end":"1927-12-24"}],"G833":[{"start":"1883-05-21","end":"1928-01-25","n
ame":"PONTE BUGGIANESE"},{"name":"PONTE BUGGIANESE","start":"1928-01-26","en
d":"9999-12-31"}],"F151":[{"name":"MERONE","end":"1928-12-15","start":"1861-
03-17"},{"name":"MERONE","end":"9999-12-31","start":"1928-12-16"}],"A894":[{
"end":"9999-12-31","start":"1861-03-17","name":"BITRITTO"}],"L323":[{"name":
"TRAMONTI","start":"1861-03-17","end":"9999-12-31"}],"C667":[{"start":"1920-
10-16","end":"1923-02-13","name":"CIAGO"},{"end":"1928-04-18","start":"1923-
02-14","name":"CIAGO"}],"L029":[{"name":"TAGLIUNO","start":"1861-03-17","end
":"1927-08-04"}],"C734":[{"name":"CISMON","end":"1928-03-03","start":"1866-1
1-19"},{"name":"CISMON DEL GRAPPA","end":"2019-01-29","start":"1928-03-04"}]
,"Z602":[{"name":"Brasile","end":"9999-12-31","start":"0000-01-01"}],"B070":
[{"name":"BOSCHI SANT'ANNA","end":"1928-04-14","start":"1866-11-19"},{"start
":"1947-09-30","end":"9999-12-31","name":"BOSCHI SANT'ANNA"}],"E658":[{"name
":"LOMASO","end":"1952-05-16","start":"1928-04-03"},{"name":"LOMASO","start"
:"1952-05-17","end":"2009-12-31"}],"A117":[{"start":"1864-03-11","end":"1928
-05-11","name":"ALA DI STURA"},{"name":"ALA DI STURA","start":"1928-05-12","
end":"9999-12-31"}],"A235":[{"name":"ALTINO","start":"1861-03-17","end":"999
9-12-31"}],"B149":[{"name":"BRENO","start":"1861-03-17","end":"1927-12-28"},
{"name":"BRENO","start":"1927-12-29","end":"1949-08-26"},{"name":"BRENO","en
d":"9999-12-31","start":"1949-08-27"}],"D084":[{"start":"1921-01-05","end":"
1923-02-07","name":"COSBANA"},{"name":"COSBANA","end":"1923-04-26","start":"
1923-02-08"},{"end":"1927-01-11","start":"1923-04-27","name":"COSBANA DEL CO
LLIO"},{"name":"COSBANA DEL COLLIO","end":"1928-03-06","start":"1927-01-12"}
],"I448":[{"end":"1927-01-11","start":"1861-03-17","name":"SARULE"},{"end":"
9999-12-31","start":"1927-01-12","name":"SARULE"}],"I144":[{"name":"SAN SALV
ATORE MONFERRATO","end":"1972-03-21","start":"1863-02-16"},{"name":"SAN SALV
ATORE MONFERRATO","start":"1972-03-22","end":"9999-12-31"}],"B900":[{"name":
"CASALMORANO","end":"9999-12-31","start":"1867-09-07"}],"G718":[{"name":"PIU
RO","end":"9999-12-31","start":"1861-03-17"}],"F465":[{"end":"9999-12-31","s
tart":"1866-11-19","name":"MONTECCHIO PRECALCINO"}],"L681":[{"name":"VARESE 
LIGURE","end":"1923-12-21","start":"1862-03-31"},{"end":"9999-12-31","start"
:"1923-12-22","name":"VARESE LIGURE"}],"F751":[{"end":"9999-12-31","start":"
1861-03-17","name":"MORSASCO"}],"H518":[{"name":"ROMENTINO","start":"1861-03
-17","end":"9999-12-31"}],"D342":[{"name":"DONIGALA FENUGHEDU","end":"1927-1
1-08","start":"1862-09-29"}],"C134":[{"name":"CASTELLANA","end":"1950-06-28"
,"start":"1861-03-17"},{"name":"CASTELLANA GROTTE","end":"9999-12-31","start
":"1950-06-29"}],"E211":[{"end":"1927-01-11","start":"1871-01-15","name":"GR
OTTE SANTO STEFANO"},{"name":"GROTTE SANTO STEFANO","end":"1928-05-19","star
t":"1927-01-12"}],"C378":[{"name":"CAVALLIRIO","start":"1861-03-17","end":"9
999-12-31"}],"A717":[{"name":"BATTIPAGLIA","start":"1929-05-21","end":"9999-
12-31"}],"B749":[{"start":"1866-11-19","end":"9999-12-31","name":"CARCERI"}]
,"G414":[{"start":"1874-10-26","end":"1926-02-05","name":"PEGLI"}],"B361":[{
"name":"CAIANELLO","start":"1861-03-17","end":"1927-01-11"},{"name":"CAIANEL
LO","start":"1927-01-12","end":"1945-08-31"},{"name":"CAIANELLO","end":"9999
-12-31","start":"1945-09-01"}],"A583":[{"name":"BAIO","start":"1861-03-17","
end":"1917-09-17"},{"name":"BAIO DORA","end":"1927-01-11","start":"1917-09-1
8"},{"name":"BAIO DORA","end":"1929-03-02","start":"1927-01-12"}],"C438":[{"
name":"CELLARENGO","start":"1861-03-17","end":"1935-04-07"},{"end":"9999-12-
31","start":"1935-04-08","name":"CELLARENGO"}],"D521":[{"name":"FAVRIA-OGLIA
NICO","end":"1946-07-18","start":"1927-11-02"}],"E878":[{"name":"MANDATORICC
IO","start":"1861-03-17","end":"1928-05-18"},{"name":"MANDATORICCIO","start"
:"1928-05-19","end":"1934-03-06"},{"name":"MANDATORICCIO","end":"9999-12-31"
,"start":"1934-03-07"}],"H058":[{"end":"1927-01-11","start":"1861-03-17","na
me":"PRIACCO"},{"name":"PRIACCO","start":"1927-01-12","end":"1928-09-25"}],"
I744":[{"end":"1992-04-15","start":"1861-03-17","name":"SIMBARIO"},{"name":"
SIMBARIO","start":"1992-04-16","end":"9999-12-31"}],"G118":[{"name":"ORNICA"
,"start":"1861-03-17","end":"9999-12-31"}],"E486":[{"end":"9999-12-31","star
t":"1861-03-17","name":"LAURITO"}],"B557":[{"name":"CAMPORGIANO","start":"18
61-03-17","end":"1923-11-23"},{"start":"1923-11-24","end":"9999-12-31","name
":"CAMPORGIANO"}],"H811":[{"end":"1935-04-07","start":"1863-02-16","name":"S
AN DAMIANO D'ASTI"},{"end":"9999-12-31","start":"1935-04-08","name":"SAN DAM
IANO D'ASTI"}],"L764":[{"name":"VERGHERETO","start":"1861-03-17","end":"1923
-03-18"},{"name":"VERGHERETO","end":"1992-04-15","start":"1923-03-19"},{"nam
e":"VERGHERETO","end":"9999-12-31","start":"1992-04-16"}],"E185":[{"end":"99
99-12-31","start":"1948-04-30","name":"GRISOLIA"}],"F766":[{"name":"MOSO","e
nd":"1927-01-11","start":"1923-04-27"},{"start":"1927-01-12","end":"1928-06-
14","name":"MOSO"},{"start":"1928-06-15","end":"1955-03-01","name":"MOSO"},{
"end":"9999-12-31","start":"1955-03-02","name":"MOSO IN PASSIRIA"},{"end":"1
923-02-13","start":"1920-10-16","name":"PIANO IN PASSERIA"},{"name":"PIANO I
N PASSERIA","start":"1923-02-14","end":"1923-04-26"}],"L637":[{"name":"VAL M
ARCHIROLO","start":"1928-10-07","end":"1955-08-18"}],"D582":[{"start":"1864-
01-29","end":"9999-12-31","name":"FIGLINE VEGLIATURO"}],"F033":[{"name":"MAS
SANZAGO","start":"1866-11-19","end":"9999-12-31"}],"E447":[{"name":"LAPEDONA
","end":"2009-07-11","start":"1861-03-17"},{"start":"2009-07-12","end":"9999
-12-31","name":"LAPEDONA"}],"F339":[{"name":"MONCHIERO NOVELLO","start":"192
8-04-15","end":"1932-11-08"},{"name":"NOVELLO MONCHIERO","end":"1947-03-15",
"start":"1932-11-09"}],"I657":[{"name":"SERRAVALLE LIBARNA","start":"1937-09
-08","end":"1947-05-07"},{"start":"1863-01-26","end":"1937-09-07","name":"SE
RRAVALLE SCRIVIA"},{"start":"1947-05-08","end":"9999-12-31","name":"SERRAVAL
LE SCRIVIA"}],"I913":[{"name":"SPINO AL BREMBO","start":"1863-07-13","end":"
1928-03-26"}],"H278":[{"name":"RICLARETTO","end":"1928-04-30","start":"1861-
03-17"}],"A049":[{"name":"ACQUAVIVA PLATANI","end":"9999-12-31","start":"186
2-10-31"}],"F479":[{"name":"MONTECORICE","end":"9999-12-31","start":"1930-07
-10"},{"name":"ORTODONICO","start":"1861-03-17","end":"1930-07-09"}],"A343":
[{"name":"AQUARA","start":"1861-03-17","end":"9999-12-31"}],"B017":[{"start"
:"1929-03-15","end":"1947-05-13","name":"BORGO LITTORIO"},{"name":"BORGO SAN
 GIOVANNI","start":"1947-05-14","end":"1992-04-15"},{"name":"BORGO SAN GIOVA
NNI","start":"1992-04-16","end":"9999-12-31"},{"name":"CAZZIMANI","start":"1
866-01-14","end":"1929-03-14"}],"E571":[{"name":"LIBRIZZI","start":"1861-03-
17","end":"9999-12-31"}],"H915":[{"end":"1926-03-17","start":"1861-03-17","n
ame":"SAN GIOVANNI DI GALERMO"}],"D828":[{"start":"1861-03-17","end":"9999-1
2-31","name":"FUSCALDO"}],"L622":[{"name":"VALLE SUPERIORE MOSSO","start":"1
861-03-17","end":"1927-01-11"},{"name":"VALLE SUPERIORE MOSSO","end":"1938-0
4-07","start":"1927-01-12"}],"A170":[{"start":"1921-01-05","end":"1923-02-07
","name":"ALBONA"},{"end":"1937-11-05","start":"1923-02-08","name":"ALBONA"}
,{"start":"1937-11-06","end":"1939-09-18","name":"ALBONA"},{"end":"1947-09-1
5","start":"1939-09-19","name":"ALBONA"}],"G314":[{"start":"1861-03-17","end
":"1928-04-16","name":"PANTASINA"}],"B155":[{"end":"1929-01-24","start":"186
6-11-19","name":"BREONIO"}],"E587":[{"start":"1929-01-24","end":"1946-03-09"
,"name":"LILLIANA"},{"name":"LILLIANES","start":"1861-03-17","end":"1927-01-
11"},{"name":"LILLIANES","end":"1929-01-23","start":"1927-01-12"},{"start":"
1946-03-10","end":"9999-12-31","name":"LILLIANES"}],"C478":[{"name":"CEPPO M
ORELLI","end":"1992-05-22","start":"1866-01-14"},{"name":"CEPPO MORELLI","en
d":"9999-12-31","start":"1992-05-23"}],"E838":[{"name":"MAIOLO","end":"1928-
08-16","start":"1861-03-17"},{"name":"MAIOLO","start":"1947-02-05","end":"19
98-09-03"},{"name":"MAIOLO","end":"2009-08-14","start":"1998-09-04"},{"name"
:"MAIOLO","end":"9999-12-31","start":"2009-08-15"}],"H896":[{"end":"9999-12-
31","start":"1861-03-17","name":"SAN GIORGIO DI PIANO"}],"D442":[{"end":"999
9-12-31","start":"1866-11-19","name":"ESTE"}],"C886":[{"end":"9999-12-31","s
tart":"1861-03-17","name":"COLMURANO"}],"G653":[{"start":"1861-03-17","end":
"9999-12-31","name":"PIEVE SANTO STEFANO"}],"D863":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"GALATONE"}],"G917":[{"end":"1925-03-20","start":"18
66-11-19","name":"PORTO MANTOVANO"},{"start":"1925-03-21","end":"1943-02-23"
,"name":"PORTO MANTOVANO"},{"name":"PORTO MANTOVANO","end":"9999-12-31","sta
rt":"1943-02-24"}],"C174":[{"end":"9999-12-31","start":"1861-03-17","name":"
CASTELLINA MARITTIMA"}],"E386":[{"end":"1927-01-11","start":"1907-07-01","na
me":"JERAGO CON ORAGO"},{"end":"9999-12-31","start":"1927-01-12","name":"JER
AGO CON ORAGO"}],"F253":[{"end":"1928-09-05","start":"1861-03-17","name":"MO
ANO"}],"C338":[{"end":"1927-04-13","start":"1872-10-18","name":"CASTRO DEI V
OLSCI"},{"start":"1927-04-14","end":"9999-12-31","name":"CASTRO DEI VOLSCI"}
],"B461":[{"start":"1861-03-17","end":"9999-12-31","name":"CAMBIAGO"}],"L192
":[{"name":"TOLFA","end":"1927-09-13","start":"1871-01-15"},{"name":"TOLFA",
"end":"1935-03-06","start":"1927-09-14"},{"start":"1935-03-07","end":"9999-1
2-31","name":"TOLFA"}],"E713":[{"name":"LUBRIANO","end":"1927-01-11","start"
:"1871-01-15"},{"end":"9999-12-31","start":"1927-01-12","name":"LUBRIANO"}],
"E531":[{"name":"LENTELLA","start":"1861-03-17","end":"9999-12-31"}],"E296":
[{"start":"1863-04-13","end":"2013-12-31","name":"INCISA IN VAL D'ARNO"}],"H
238":[{"end":"9999-12-31","start":"1866-11-19","name":"RESANA"}],"D960":[{"n
ame":"GELA","start":"1927-12-31","end":"9999-12-31"},{"end":"1927-12-30","st
art":"1862-10-31","name":"TERRANOVA DI SICILIA"}],"E347":[{"name":"ISOLA COM
ACINA","end":"1950-04-05","start":"1928-11-03"}],"A601":[{"start":"1861-03-1
7","end":"1927-01-11","name":"BALOCCO"},{"start":"1927-01-12","end":"1956-11
-09","name":"BALOCCO"},{"name":"BALOCCO","end":"9999-12-31","start":"1956-11
-10"}],"E944":[{"name":"MARETTO","end":"1928-12-30","start":"1861-03-17"},{"
start":"1947-06-23","end":"9999-12-31","name":"MARETTO"}],"A130":[{"name":"A
LBANO VERCELLESE","start":"1862-12-29","end":"1927-01-11"},{"name":"ALBANO V
ERCELLESE","start":"1927-01-12","end":"9999-12-31"}],"L677":[{"name":"VAREDO
","end":"2009-06-29","start":"1861-03-17"},{"name":"VAREDO","end":"9999-12-3
1","start":"2009-06-30"}],"E546":[{"start":"1861-03-17","end":"9999-12-31","
name":"LESEGNO"}],"A269":[{"end":"1927-01-11","start":"1871-01-15","name":"A
NAGNI"},{"name":"ANAGNI","end":"9999-12-31","start":"1927-01-12"}],"Z403":[{
"name":"Saint Pierre e Miquelon","end":"9999-12-31","start":"0000-01-01"}],"
A862":[{"name":"BIENO","end":"1928-10-28","start":"1880-06-25"}],"L503":[{"e
nd":"9999-12-31","start":"1861-03-17","name":"URI"}],"B456":[{"start":"1861-
03-17","end":"1938-02-18","name":"CAMAIRAGO"},{"start":"1938-02-19","end":"1
992-04-15","name":"CAMAIRAGO"},{"end":"2017-12-31","start":"1992-04-16","nam
e":"CAMAIRAGO"}],"C847":[{"end":"9999-12-31","start":"1862-10-06","name":"CO
LLE DI VAL D'ELSA"}],"F073":[{"end":"1927-01-11","start":"1862-09-29","name"
:"MEANA SARDO"},{"end":"9999-12-31","start":"1927-01-12","name":"MEANA SARDO
"}],"F379":[{"name":"MONSAMPIETRO MORICO","end":"2009-07-11","start":"1893-0
3-03"},{"end":"9999-12-31","start":"2009-07-12","name":"MONSAMPIETRO MORICO"
}],"A443":[{"name":"ARSI\u00c8","end":"9999-12-31","start":"1866-11-19"}],"I
588":[{"name":"SELLERO","start":"1861-03-17","end":"1927-12-26"},{"start":"1
948-12-21","end":"9999-12-31","name":"SELLERO"}],"I837":[{"start":"1861-03-1
7","end":"1928-10-08","name":"SOPRAZOCCO"}],"H880":[{"start":"1863-02-06","e
nd":"1927-01-11","name":"SAN GIORGIO A LIRI"},{"start":"1927-01-12","end":"9
999-12-31","name":"SAN GIORGIO A LIRI"}],"B224":[{"end":"1929-05-22","start"
:"1861-03-17","name":"BRUSASCHETTO"}],"L857":[{"start":"1861-03-17","end":"1
927-01-11","name":"VIDRACCO"},{"name":"VIDRACCO","end":"1945-10-04","start":
"1927-01-12"},{"end":"9999-12-31","start":"1945-10-05","name":"VIDRACCO"}],"
A515":[{"name":"AVEZZANO","end":"1940-02-07","start":"1861-03-17"},{"start":
"1940-02-08","end":"1954-10-25","name":"AVEZZANO"},{"name":"AVEZZANO","start
":"1954-10-26","end":"1959-12-28"},{"name":"AVEZZANO","end":"1960-08-12","st
art":"1959-12-29"},{"end":"9999-12-31","start":"1960-08-13","name":"AVEZZANO
"}],"C890":[{"name":"COLOGNA VENETA","start":"1867-08-26","end":"9999-12-31"
}],"M429":[{"start":"2020-01-01","end":"9999-12-31","name":"BORGO D'ANAUNIA"
}],"A730":[{"end":"1923-02-13","start":"1920-10-16","name":"BEDOLLO"},{"name
":"BEDOLLO","start":"1923-02-14","end":"9999-12-31"}],"B688":[{"name":"CAPRA
NICA","start":"1871-01-15","end":"1927-01-11"},{"name":"CAPRANICA","end":"99
99-12-31","start":"1927-01-12"}],"D301":[{"end":"1923-02-07","start":"1921-0
1-05","name":"DIGNANO D'ISTRIA"},{"start":"1923-02-08","end":"1947-09-15","n
ame":"DIGNANO D'ISTRIA"}],"I822":[{"name":"SOMMARIVA DEL BOSCO","end":"9999-
12-31","start":"1871-04-01"}],"E390":[{"start":"1923-11-28","end":"9999-12-3
1","name":"JOPPOLO GIANCAXIO"}],"A055":[{"start":"1927-11-08","end":"9999-12
-31","name":"ADELFIA"}],"G970":[{"start":"1940-05-07","end":"1956-03-28","na
me":"PRADLEVE"},{"name":"PRADLEVES","start":"1861-03-17","end":"1940-05-06"}
,{"name":"PRADLEVES","start":"1956-03-29","end":"9999-12-31"}],"L368":[{"end
":"1992-04-15","start":"1861-03-17","name":"TREMENICO"},{"name":"TREMENICO",
"end":"2017-12-31","start":"1992-04-16"}],"G352":[{"name":"PASIAN DI PRATO",
"end":"1968-04-05","start":"1866-11-19"},{"end":"9999-12-31","start":"1968-0
4-06","name":"PASIAN DI PRATO"}],"I229":[{"name":"SANTA MARGHERITA","start":
"1920-10-16","end":"1923-02-13"},{"end":"1928-09-05","start":"1923-02-14","n
ame":"SANTA MARGHERITA"}],"C993":[{"start":"1866-11-19","end":"1968-04-05","
name":"CORDOVADO"},{"end":"9999-12-31","start":"1968-04-06","name":"CORDOVAD
O"}],"L610":[{"name":"VALLENONCELLO","start":"1866-11-19","end":"1930-04-27"
}],"I348":[{"start":"1861-03-17","end":"1929-05-28","name":"SANT'OMERO"},{"n
ame":"SANT'OMERO","start":"1929-05-29","end":"9999-12-31"}],"B755":[{"start"
:"1861-03-17","end":"1911-02-07","name":"CARD\u00c8"},{"name":"CARD\u00c8","
end":"9999-12-31","start":"1911-02-08"}],"E280":[{"name":"IDRO","start":"186
1-03-17","end":"9999-12-31"}],"I881":[{"end":"1940-01-23","start":"1929-06-1
3","name":"SPADAFORA"},{"name":"SPADAFORA","start":"1940-01-24","end":"9999-
12-31"}],"L423":[{"name":"TRIEI","end":"1927-01-11","start":"1861-03-17"},{"
end":"1960-03-10","start":"1927-01-12","name":"TRIEI"},{"name":"TRIEI","end"
:"2005-12-31","start":"1960-03-11"},{"end":"2016-04-27","start":"2006-01-01"
,"name":"TRIEI"},{"start":"2016-04-28","end":"9999-12-31","name":"TRIEI"}],"
F365":[{"name":"MONGIARDINO LIGURE","start":"1863-01-26","end":"1954-04-23"}
,{"start":"1954-04-24","end":"9999-12-31","name":"MONGIARDINO LIGURE"}],"C77
4":[{"name":"CIVITELLA IN VAL DI CHIANA","start":"1862-11-28","end":"9999-12
-31"}],"B030":[{"name":"BORGORATTO MORMOROLO","start":"1863-03-30","end":"99
99-12-31"}],"I985":[{"name":"STROPPO","start":"1861-03-17","end":"9999-12-31
"}],"G873":[{"name":"PONZANO DI FERMO","start":"1862-11-24","end":"2009-07-1
1"},{"start":"2009-07-12","end":"9999-12-31","name":"PONZANO DI FERMO"}],"M0
68":[{"name":"VIRLE TREPONTI","end":"1928-05-11","start":"1862-12-19"}],"A27
5":[{"end":"9999-12-31","start":"1861-03-17","name":"ANDEZENO"}],"L792":[{"n
ame":"VERTEMATE CON MINOPRIO","start":"1929-05-30","end":"9999-12-31"}],"E11
3":[{"start":"1866-11-19","end":"9999-12-31","name":"GOSALDO"}],"A918":[{"en
d":"9999-12-31","start":"1946-10-27","name":"BODIO LOMNAGO"},{"name":"LOMNAG
O","start":"1861-03-17","end":"1927-01-11"},{"name":"LOMNAGO","start":"1927-
01-12","end":"1928-01-11"},{"end":"1946-10-26","start":"1928-01-12","name":"
LOMNAGO"}],"F947":[{"start":"1920-10-16","end":"1923-02-13","name":"NOVALEDO
"},{"start":"1923-02-14","end":"9999-12-31","name":"NOVALEDO"}],"E009":[{"na
me":"GIAGLIONE","end":"9999-12-31","start":"1861-03-17"}],"E303":[{"name":"I
NTRA","end":"1927-12-03","start":"1861-03-17"},{"name":"INTRA","end":"1929-0
5-24","start":"1927-12-04"},{"start":"1929-05-25","end":"1939-06-04","name":
"INTRA"}],"L471":[{"name":"TURBIGO","start":"1861-03-17","end":"1958-08-27"}
,{"name":"TURBIGO","end":"9999-12-31","start":"1958-08-28"}],"G925":[{"name"
:"PORTOVENERE","start":"1861-03-17","end":"1923-12-21"},{"name":"PORTOVENERE
","end":"9999-12-31","start":"1923-12-22"}],"C250":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"CASTELPOTO"}],"L346":[{"name":"TRAVERSETOLO","end":
"1951-12-31","start":"1861-03-17"},{"start":"1952-01-01","end":"9999-12-31",
"name":"TRAVERSETOLO"}],"L297":[{"name":"TORRI DI QUARTESOLO","end":"1988-01
-05","start":"1866-11-19"},{"start":"1988-01-06","end":"9999-12-31","name":"
TORRI DI QUARTESOLO"}],"G093":[{"name":"ORERO","end":"9999-12-31","start":"1
861-03-17"}],"G399":[{"name":"PECORARA","start":"1861-03-17","end":"2017-12-
31"}],"I351":[{"end":"1927-01-11","start":"1861-03-17","name":"SANTOPADRE"},
{"name":"SANTOPADRE","start":"1927-01-12","end":"9999-12-31"}],"A034":[{"end
":"2002-10-25","start":"1861-03-17","name":"ACQUAFREDDA"},{"start":"2002-10-
26","end":"9999-12-31","name":"ACQUAFREDDA"}],"D285":[{"name":"DES\u00c9RTES
","start":"1861-03-17","end":"1928-12-13"}],"G821":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"PONNA"}],"L547":[{"end":"1928-01-25","start":"1861-
03-17","name":"VALBRONA"},{"name":"VALBRONA","end":"9999-12-31","start":"192
8-01-26"}],"G704":[{"name":"PISONIANO","end":"9999-12-31","start":"1871-10-2
9"}],"E979":[{"start":"1861-03-17","end":"9999-12-31","name":"MARTANO"}],"E6
76":[{"end":"1923-02-13","start":"1920-10-16","name":"LONGIAR\u00d9"},{"name
":"LONGIAR\u00d9","start":"1923-02-14","end":"1927-01-11"},{"name":"LONGIAR\
u00d9","start":"1927-01-12","end":"1928-06-15"}],"A407":[{"name":"ARITZO","s
tart":"1861-03-17","end":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-
12","name":"ARITZO"}],"L331":[{"end":"1936-02-11","start":"1868-08-25","name
":"TRAPANI"},{"name":"TRAPANI","end":"1938-09-19","start":"1936-02-12"},{"st
art":"1938-09-20","end":"1946-02-06","name":"TRAPANI"},{"end":"1979-10-19","
start":"1946-02-07","name":"TRAPANI"},{"start":"1979-10-20","end":"9999-12-3
1","name":"TRAPANI"}],"M087":[{"name":"VITTADONE","start":"1861-03-17","end"
:"1929-05-28"}],"C803":[{"name":"COAZZE","start":"1861-03-17","end":"9999-12
-31"}],"G289":[{"name":"PALMIANO","end":"9999-12-31","start":"1861-03-17"}],
"C149":[{"name":"CASTELLAZZO NOVARESE","start":"1863-04-23","end":"1928-05-2
2"},{"name":"CASTELLAZZO NOVARESE","start":"1948-01-15","end":"9999-12-31"}]
,"D395":[{"name":"ELINI","start":"1958-08-15","end":"2005-12-31"},{"end":"20
16-04-27","start":"2006-01-01","name":"ELINI"},{"end":"9999-12-31","start":"
2016-04-28","name":"ELINI"}],"H128":[{"name":"QUINDICI","start":"1861-03-17"
,"end":"1928-03-12"},{"start":"1928-03-13","end":"1957-12-07","name":"QUINDI
CI"},{"start":"1957-12-08","end":"9999-12-31","name":"QUINDICI"}],"I567":[{"
name":"SEDRINA","start":"1861-03-17","end":"2002-10-25"},{"name":"SEDRINA","
start":"2002-10-26","end":"9999-12-31"}],"M046":[{"name":"VILLONGO SAN FILAS
TRO","start":"1861-03-17","end":"1927-11-02"}],"C070":[{"end":"2009-07-11","
start":"1863-01-26","name":"SERVIGLIANO"},{"start":"2009-07-12","end":"9999-
12-31","name":"SERVIGLIANO"}],"H002":[{"name":"PRATO CARNICO","end":"1968-04
-05","start":"1867-09-02"},{"start":"1968-04-06","end":"9999-12-31","name":"
PRATO CARNICO"}],"B734":[{"name":"CARAVONICA","end":"1929-03-15","start":"18
61-03-17"},{"name":"CARAVONICA","end":"9999-12-31","start":"1929-03-16"}],"B
667":[{"name":"CAPODRISE","start":"1861-03-17","end":"1927-01-11"},{"end":"1
928-04-19","start":"1927-01-12","name":"CAPODRISE"},{"end":"9999-12-31","sta
rt":"1946-08-24","name":"CAPODRISE"}],"L586":[{"name":"VALLANZENGO","end":"1
927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"1992-04-15","n
ame":"VALLANZENGO"},{"name":"VALLANZENGO","start":"1992-04-16","end":"9999-1
2-31"}],"A220":[{"name":"ALONTE","end":"9999-12-31","start":"1866-11-19"}],"
G882":[{"start":"1913-07-13","end":"9999-12-31","name":"PORCARI"}],"F584":[{
"name":"MONTE OSSOLANO","start":"1861-03-17","end":"1928-01-07"}],"H424":[{"
end":"9999-12-31","start":"1861-03-17","name":"ROCCAMONTEPIANO"}],"D111":[{"
name":"COSTA SERINA","start":"1861-03-17","end":"1927-11-20"},{"end":"9999-1
2-31","start":"1948-04-22","name":"COSTA SERINA"}],"H350":[{"name":"RIVERGAR
O","end":"1913-06-19","start":"1861-03-17"},{"end":"9999-12-31","start":"191
3-06-20","name":"RIVERGARO"}],"H972":[{"end":"1923-02-13","start":"1920-10-1
6","name":"SAN LUGANO"},{"name":"SAN LUGANO","start":"1923-02-14","end":"192
6-03-02"}],"C900":[{"end":"9999-12-31","start":"1871-01-15","name":"COLONNA"
}],"F689":[{"end":"1970-03-02","start":"1864-03-11","name":"MONTORIO NEI FRE
NTANI"},{"start":"1970-03-03","end":"9999-12-31","name":"MONTORIO NEI FRENTA
NI"}],"Z211":[{"end":"9999-12-31","start":"0000-01-01","name":"Cipro"}],"L75
9":[{"end":"1927-01-11","start":"1861-03-17","name":"VEREZZI"},{"name":"VERE
ZZI","end":"1933-05-23","start":"1927-01-12"}],"F986":[{"start":"1861-03-17"
,"end":"1927-01-11","name":"NURRI"},{"start":"1927-01-12","end":"2005-12-31"
,"name":"NURRI"},{"end":"2016-04-27","start":"2006-01-01","name":"NURRI"},{"
name":"NURRI","end":"9999-12-31","start":"2016-04-28"}],"L984":[{"name":"VIL
LANOVA D'ASTI","end":"9999-12-31","start":"1951-08-09"},{"start":"1863-05-06
","end":"1935-04-07","name":"VILLANUOVA D'ASTI"},{"name":"VILLANUOVA D'ASTI"
,"start":"1935-04-08","end":"1951-08-08"}],"L387":[{"start":"1906-04-18","en
d":"1928-08-06","name":"TRESCH\u00c8 CONCA"}],"M031":[{"end":"9999-12-31","s
tart":"1861-03-17","name":"VILLAVALLELONGA"}],"I739":[{"name":"SILVANO PIETR
A","end":"1932-12-27","start":"1861-03-17"},{"end":"9999-12-31","start":"193
2-12-28","name":"SILVANO PIETRA"}],"G528":[{"start":"1861-03-17","end":"1927
-01-11","name":"PEZZANA"},{"name":"PEZZANA","start":"1927-01-12","end":"9999
-12-31"}],"H256":[{"name":"REZZATO","end":"1928-05-11","start":"1861-03-17"}
,{"name":"REZZATO","start":"1928-05-12","end":"9999-12-31"}],"C749":[{"name"
:"CITTAREALE","end":"1927-01-11","start":"1861-03-17"},{"name":"CITTAREALE",
"end":"9999-12-31","start":"1927-01-12"}],"E861":[{"name":"MALLAS","end":"19
27-01-11","start":"1923-04-27"},{"end":"1928-01-26","start":"1927-01-12","na
me":"MALLAS"},{"name":"MALLES","start":"1920-10-16","end":"1923-02-13"},{"st
art":"1923-02-14","end":"1923-04-26","name":"MALLES"}],"H728":[{"name":"SALV
ATERRA","end":"1928-04-06","start":"1866-11-19"}],"I260":[{"start":"1881-03-
28","end":"9999-12-31","name":"SANTOMENNA"}],"A695":[{"end":"1929-02-07","st
art":"1861-03-17","name":"BASELICA BOLOGNA"}],"B378":[{"start":"1861-03-17",
"end":"1960-03-11","name":"CALANGIANUS"},{"name":"CALANGIANUS","end":"1963-0
9-26","start":"1960-03-12"},{"name":"CALANGIANUS","end":"1979-04-19","start"
:"1963-09-27"},{"name":"CALANGIANUS","end":"2005-12-31","start":"1979-04-20"
},{"name":"CALANGIANUS","start":"2006-01-01","end":"2016-04-27"},{"end":"999
9-12-31","start":"2016-04-28","name":"CALANGIANUS"}],"B134":[{"name":"BREGNA
NO","start":"1861-03-17","end":"1980-05-28"},{"name":"BREGNANO","end":"9999-
12-31","start":"1980-05-29"}],"A527":[{"start":"1904-06-25","end":"1929-06-1
3","name":"AZZANO D'ASTI"},{"name":"AZZANO D'ASTI","start":"1947-12-24","end
":"9999-12-31"},{"name":"AZZANO DEL TANARO","start":"1863-02-16","end":"1904
-06-24"}],"F021":[{"name":"MASSA FERMANA","start":"1862-11-24","end":"2009-0
7-11"},{"start":"2009-07-12","end":"9999-12-31","name":"MASSA FERMANA"}],"D7
11":[{"start":"1861-03-17","end":"9999-12-31","name":"FORMIGINE"}],"E602":[{
"name":"LINGUAGLOSSA","start":"1861-03-17","end":"9999-12-31"}],"L159":[{"na
me":"THURES","end":"1928-12-13","start":"1861-03-17"}],"E965":[{"name":"MARN
ATE","start":"1866-01-21","end":"1927-01-11"},{"end":"9999-12-31","start":"1
927-01-12","name":"MARNATE"}],"A326":[{"end":"1927-01-11","start":"1861-03-1
7","name":"AOSTA"},{"end":"1928-03-20","start":"1927-01-12","name":"AOSTA"},
{"name":"AOSTA","start":"1928-03-21","end":"1946-05-14"},{"end":"9999-12-31"
,"start":"1946-05-15","name":"AOSTA"}],"F893":[{"name":"NICOTERA","start":"1
861-03-17","end":"1992-04-15"},{"end":"9999-12-31","start":"1992-04-16","nam
e":"NICOTERA"}],"A532":[{"start":"1861-03-17","end":"1927-01-11","name":"AZZ
IO"},{"name":"AZZIO","start":"1927-01-12","end":"1927-08-18"},{"name":"AZZIO
","start":"1956-01-24","end":"9999-12-31"}],"I139":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"SAN ROBERTO"}],"B438":[{"name":"CALVARUSO","end":"1
929-06-01","start":"1866-01-14"}],"C361":[{"end":"9999-12-31","start":"1862-
11-07","name":"CAVA DE' TIRRENI"}],"H504":[{"start":"1920-10-16","end":"1923
-02-13","name":"ROMAGNANO"},{"name":"ROMAGNANO","start":"1923-02-14","end":"
1926-09-30"}],"E568":[{"end":"1928-05-25","start":"1861-03-17","name":"LEZZA
"}],"E856":[{"name":"MALGESSO","start":"1861-03-17","end":"1927-01-11"},{"en
d":"2022-12-31","start":"1927-01-12","name":"MALGESSO"}],"H076":[{"end":"193
4-10-25","start":"1871-01-15","name":"PROSSEDI"},{"name":"PROSSEDI","start":
"1934-10-26","end":"1934-12-17"},{"end":"1945-04-08","start":"1934-12-18","n
ame":"PROSSEDI"},{"end":"9999-12-31","start":"1945-04-09","name":"PROSSEDI"}
],"H261":[{"name":"RHEMES","start":"1928-05-12","end":"1939-10-20"},{"name":
"RHEMES","end":"1946-03-09","start":"1946-01-30"},{"name":"VAL DI REMA","end
":"1946-01-29","start":"1939-10-21"}],"F082":[{"name":"MEDESANO","start":"18
73-11-03","end":"9999-12-31"}],"C557":[{"start":"1861-03-17","end":"9999-12-
31","name":"CERVINARA"}],"G408":[{"start":"1866-11-19","end":"9999-12-31","n
ame":"PEDEROBBA"}],"L590":[{"name":"VALLE DI CADORE","start":"1867-06-02","e
nd":"1966-02-09"},{"start":"1966-02-10","end":"9999-12-31","name":"VALLE DI 
CADORE"}],"M153":[{"name":"ZEDDIANI","end":"1927-10-09","start":"1861-03-17"
},{"name":"ZEDDIANI","start":"1947-10-26","end":"1974-08-19"},{"name":"ZEDDI
ANI","end":"9999-12-31","start":"1974-08-20"}],"F990":[{"name":"NUVOLERA","e
nd":"9999-12-31","start":"1861-03-17"}],"H906":[{"end":"1947-01-08","start":
"1928-07-01","name":"SAN GIOVANNI AL NATISONE"},{"start":"1947-01-09","end":
"1968-04-05","name":"SAN GIOVANNI AL NATISONE"},{"start":"1968-04-06","end":
"2011-10-19","name":"SAN GIOVANNI AL NATISONE"},{"end":"9999-12-31","start":
"2011-10-20","name":"SAN GIOVANNI AL NATISONE"},{"end":"1928-05-04","start":
"1866-11-19","name":"SAN GIOVANNI DI MANZANO"},{"name":"SAN GIOVANNI DI MANZ
ANO","start":"1928-05-05","end":"1928-06-30"}],"L240":[{"end":"1927-12-26","
start":"1864-06-23","name":"TORRE DI RUGGIERO"},{"name":"TORRE DI RUGGIERO",
"start":"1927-12-27","end":"9999-12-31"}],"C356":[{"name":"CATTOLICA ERACLEA
","start":"1874-06-15","end":"9999-12-31"}],"H609":[{"start":"1867-09-02","e
nd":"1968-04-05","name":"ROVEREDO IN PIANO"},{"name":"ROVEREDO IN PIANO","st
art":"1968-04-06","end":"2003-04-16"},{"name":"ROVEREDO IN PIANO","end":"999
9-12-31","start":"2003-04-17"}],"I040":[{"end":"1929-04-25","start":"1867-08
-05","name":"SAN MICHELE AL TAGLIAMENTO"},{"end":"1930-03-31","start":"1929-
04-26","name":"SAN MICHELE AL TAGLIAMENTO"},{"end":"9999-12-31","start":"193
0-04-01","name":"SAN MICHELE AL TAGLIAMENTO"}],"B886":[{"end":"9999-12-31","
start":"1866-11-19","name":"CASALEONE"}],"A572":[{"end":"9999-12-31","start"
:"1862-11-28","name":"BAGNOLO DEL SALENTO"}],"I179":[{"end":"9999-12-31","st
art":"1883-02-23","name":"SANTA CROCE DEL SANNIO"}],"B478":[{"end":"1928-09-
24","start":"1861-03-17","name":"CAMIGNONE"}],"D213":[{"start":"1920-10-16",
"end":"1923-02-13","name":"CURACES"},{"start":"1923-02-14","end":"1923-04-26
","name":"CURACES"},{"start":"1923-04-27","end":"1923-11-08","name":"QUARAZZ
E"}],"C155":[{"start":"1861-03-17","end":"1927-01-11","name":"CASTELLETTO CE
RVO"},{"name":"CASTELLETTO CERVO","end":"1992-04-15","start":"1927-01-12"},{
"name":"CASTELLETTO CERVO","end":"9999-12-31","start":"1992-04-16"}],"Z113":
[{"end":"9999-12-31","start":"0000-01-01","name":"Gibilterra"}],"C461":[{"st
art":"1866-11-19","end":"9999-12-31","name":"CENESELLI"}],"D180":[{"name":"C
ROPALATI","start":"1861-03-17","end":"1928-05-18"},{"name":"CROPALATI","star
t":"1928-05-19","end":"1934-03-06"},{"name":"CROPALATI","end":"9999-12-31","
start":"1934-03-07"}],"A426":[{"name":"ARN\u00d2","start":"1929-01-01","end"
:"1931-05-19"},{"end":"1928-12-31","start":"1928-02-24","name":"BONDO BREGUZ
ZO"},{"name":"BONDO BREGUZZO","end":"1947-01-07","start":"1931-05-20"}],"L74
5":[{"name":"VERANO","end":"1923-02-13","start":"1920-10-16"},{"start":"1923
-02-14","end":"1927-01-11","name":"VERANO"},{"start":"1927-01-12","end":"999
9-12-31","name":"VERANO"}],"M360":[{"end":"9999-12-31","start":"2016-01-01",
"name":"SELLA GIUDICARIE"}],"E523":[{"name":"LENI","start":"1910-01-10","end
":"9999-12-31"}],"B174":[{"name":"BRIGA MARITTIMA","start":"1862-12-19","end
":"1947-09-15"}],"G308":[{"name":"PANICALE","end":"1937-03-11","start":"1861
-03-17"},{"start":"1937-03-12","end":"1982-04-21","name":"PANICALE"},{"start
":"1982-04-22","end":"9999-12-31","name":"PANICALE"}],"I765":[{"start":"1927
-09-15","end":"1974-08-19","name":"SIURGUS DONIGALA"},{"start":"1974-08-20",
"end":"2016-04-27","name":"SIURGUS DONIGALA"},{"end":"9999-12-31","start":"2
016-04-28","name":"SIURGUS DONIGALA"}],"A125":[{"name":"ALBA ADRIATICA","sta
rt":"1956-07-29","end":"9999-12-31"}],"E098":[{"name":"GORIZIA","start":"192
1-01-05","end":"1923-02-07"},{"end":"1927-01-11","start":"1923-02-08","name"
:"GORIZIA"},{"name":"GORIZIA","end":"1928-12-11","start":"1927-01-12"},{"nam
e":"GORIZIA","start":"1928-12-12","end":"1947-09-15"},{"name":"GORIZIA","end
":"9999-12-31","start":"1947-09-16"}],"H036":[{"name":"PREMOLO","start":"186
1-03-17","end":"1928-09-25"},{"name":"PREMOLO","start":"1947-09-30","end":"9
999-12-31"}],"B847":[{"name":"CARTOSIO","start":"1861-03-17","end":"9999-12-
31"}],"D607":[{"name":"FIORANO MODENESE","start":"1862-12-29","end":"9999-12
-31"}],"C456":[{"name":"CENATE SOPRA","start":"1861-03-17","end":"1927-11-30
"},{"name":"CENATE SOPRA","end":"9999-12-31","start":"1947-06-24"}],"F748":[
{"end":"1970-03-02","start":"1863-02-06","name":"MORRONE DEL SANNIO"},{"end"
:"9999-12-31","start":"1970-03-03","name":"MORRONE DEL SANNIO"}],"A819":[{"n
ame":"BESANO","end":"1927-01-11","start":"1861-03-17"},{"name":"BESANO","sta
rt":"1927-01-12","end":"1928-05-21"},{"end":"9999-12-31","start":"1958-09-05
","name":"BESANO"}],"D796":[{"start":"1861-03-17","end":"9999-12-31","name":
"FRESAGRANDINARIA"}],"A212":[{"start":"1861-03-17","end":"1928-05-30","name"
:"ALMA"}],"L060":[{"start":"1920-10-16","end":"1923-02-13","name":"TASSULLO"
},{"name":"TASSULLO","start":"1923-02-14","end":"1928-09-05"},{"name":"TASSU
LLO","end":"1947-03-14","start":"1928-09-06"},{"name":"TASSULLO","end":"2015
-12-31","start":"1947-03-15"}],"D573":[{"start":"1920-10-16","end":"1923-02-
13","name":"FIEROZZO"},{"start":"1923-02-14","end":"1929-03-24","name":"FIER
OZZO"},{"end":"9999-12-31","start":"1948-01-25","name":"FIEROZZO"}],"H763":[
{"name":"SAN BARTOLOMEO AL MARE","start":"1968-07-26","end":"9999-12-31"},{"
end":"1923-12-21","start":"1861-03-17","name":"SAN BARTOLOMEO DEL CERVO"},{"
name":"SAN BARTOLOMEO DEL CERVO","start":"1947-10-02","end":"1968-07-25"}],"
A074":[{"start":"1861-03-17","end":"1927-01-11","name":"AGLI\u00c8"},{"name"
:"AGLI\u00c8","end":"1929-06-14","start":"1927-01-12"},{"start":"1929-06-15"
,"end":"1945-10-04","name":"AGLI\u00c8"},{"name":"AGLI\u00c8","start":"1945-
10-05","end":"1947-01-09"},{"end":"1947-01-23","start":"1947-01-10","name":"
AGLI\u00c8"},{"name":"AGLI\u00c8","start":"1947-01-24","end":"9999-12-31"}],
"C688":[{"end":"1928-03-06","start":"1861-03-17","name":"CIMA"}],"A506":[{"s
tart":"1877-11-03","end":"9999-12-31","name":"AVEGNO"}],"B890":[{"name":"CAS
ALETTO DI SOPRA","start":"1861-03-17","end":"9999-12-31"}],"I165":[{"end":"9
999-12-31","start":"1861-03-17","name":"SAN SOSTI"}],"G563":[{"name":"PIANI"
,"start":"1861-03-17","end":"1907-04-18"},{"end":"1923-11-28","start":"1907-
04-19","name":"PIANI"}],"C224":[{"end":"1923-03-18","start":"1863-04-13","na
me":"CASTELNUOVO DI FARFA"},{"name":"CASTELNUOVO DI FARFA","end":"1927-01-11
","start":"1923-03-19"},{"name":"CASTELNUOVO DI FARFA","start":"1927-01-12",
"end":"1937-10-25"},{"end":"9999-12-31","start":"1937-10-26","name":"CASTELN
UOVO DI FARFA"}],"E403":[{"name":"LAGHI","start":"1866-11-19","end":"9999-12
-31"}],"L371":[{"name":"TREMEZZO","start":"1861-03-17","end":"1929-02-07"},{
"name":"TREMEZZO","start":"1947-10-03","end":"2014-02-03"}],"E939":[{"name":
"MARENE","start":"1861-03-17","end":"9999-12-31"}],"E636":[{"start":"1861-03
-17","end":"1927-01-11","name":"LOCARNO"},{"end":"1929-03-01","start":"1927-
01-12","name":"LOCARNO"}],"H698":[{"name":"SALECCHIO","end":"1928-10-16","st
art":"1861-03-17"}],"A725":[{"start":"1861-03-17","end":"1992-05-22","name":
"BAVENO"},{"start":"1992-05-23","end":"9999-12-31","name":"BAVENO"}],"A307":
[{"name":"ANTERSELVA","end":"1923-02-13","start":"1920-10-16"},{"start":"192
3-02-14","end":"1927-01-11","name":"ANTERSELVA"},{"name":"ANTERSELVA","end":
"1928-10-17","start":"1927-01-12"}],"Z531":[{"end":"9999-12-31","start":"000
0-01-01","name":"Montserrat"}],"L431":[{"name":"TRIPI","start":"1861-03-17",
"end":"9999-12-31"}],"F148":[{"start":"1866-11-19","end":"9999-12-31","name"
:"MERLARA"}],"D196":[{"start":"1861-03-17","end":"1928-12-01","name":"CUCCIA
GO"},{"end":"9999-12-31","start":"1928-12-02","name":"CUCCIAGO"}],"Z347":[{"
start":"0000-01-01","end":"9999-12-31","name":"Sudafrica"}],"G499":[{"start"
:"1861-03-17","end":"1927-01-11","name":"PESCOSANSONESCO"},{"end":"9999-12-3
1","start":"1927-01-12","name":"PESCOSANSONESCO"}],"I451":[{"start":"1861-03
-17","end":"9999-12-31","name":"SASSANO"}],"E641":[{"end":"1923-02-07","star
t":"1921-01-05","name":"LOCAVIZZA DI AIDUSSINA"},{"name":"LOCAVIZZA DI AIDUS
SINA","start":"1923-02-08","end":"1927-01-11"},{"end":"1928-07-27","start":"
1927-01-12","name":"LOCAVIZZA DI AIDUSSINA"}],"A904":[{"end":"9999-12-31","s
tart":"1861-03-17","name":"BLESSAGNO"}],"H163":[{"name":"RAGUSA","start":"18
65-12-25","end":"1927-01-11"},{"end":"1937-07-16","start":"1927-01-12","name
":"RAGUSA"},{"name":"RAGUSA","start":"1937-07-17","end":"1950-11-11"},{"name
":"RAGUSA","end":"9999-12-31","start":"1950-11-12"}],"H932":[{"name":"SAN GI
USEPPE DI CASTO","end":"1927-01-11","start":"1863-04-23"},{"name":"SAN GIUSE
PPE DI CASTO","start":"1927-01-12","end":"1929-05-21"}],"A254":[{"start":"18
66-11-19","end":"1968-04-05","name":"AMARO"},{"end":"9999-12-31","start":"19
68-04-06","name":"AMARO"}],"I779":[{"name":"SOGLIANO AL RUBICONE","end":"197
8-12-20","start":"1862-11-24"},{"name":"SOGLIANO AL RUBICONE","start":"1978-
12-21","end":"1992-04-15"},{"start":"1992-04-16","end":"9999-12-31","name":"
SOGLIANO AL RUBICONE"}],"C755":[{"end":"1928-11-09","start":"1861-03-17","na
me":"CIVEZZA"},{"end":"9999-12-31","start":"1946-07-11","name":"CIVEZZA"}],"
M266":[{"name":"ORDONA","start":"1975-05-20","end":"2003-09-03"},{"start":"2
003-09-04","end":"2008-06-03","name":"ORDONA"},{"name":"ORDONA","start":"200
8-06-04","end":"9999-12-31"}],"B993":[{"start":"1861-03-17","end":"9999-12-3
1","name":"CASPOGGIO"}],"Z713":[{"start":"0000-01-01","end":"9999-12-31","na
me":"Nauru"}],"M071":[{"name":"VISCHE","start":"1861-03-17","end":"1927-01-1
1"},{"name":"VISCHE","start":"1927-01-12","end":"1945-10-04"},{"start":"1945
-10-05","end":"9999-12-31","name":"VISCHE"}],"D495":[{"name":"FARA SAN MARTI
NO","start":"1861-03-17","end":"9999-12-31"}],"B025":[{"end":"9999-12-31","s
tart":"1861-03-17","name":"BORGONOVO VAL TIDONE"}],"D780":[{"name":"FRASSINE
TO PO","end":"9999-12-31","start":"1861-03-17"}],"L487":[{"end":"9999-12-31"
,"start":"1938-01-18","name":"UGGIATE-TREVANO"}],"H450":[{"name":"ROCCA SUSE
LLA","end":"1905-11-03","start":"1861-03-17"},{"name":"ROCCA SUSELLA","start
":"1905-11-04","end":"9999-12-31"}],"L145":[{"name":"TERZOLAS","end":"1923-0
2-13","start":"1920-10-16"},{"name":"TERZOLAS","start":"1923-02-14","end":"1
928-09-05"},{"name":"TERZOLAS","start":"1953-01-04","end":"9999-12-31"}],"C0
30":[{"name":"CASSINELLE","end":"1917-01-28","start":"1861-03-17"},{"name":"
CASSINELLE","end":"9999-12-31","start":"1917-01-29"}],"B774":[{"name":"CARIA
TI","start":"1861-03-17","end":"1921-05-11"},{"name":"CARIATI","end":"9999-1
2-31","start":"1921-05-12"}],"H324":[{"end":"1927-01-11","start":"1871-01-15
","name":"RIPI"},{"start":"1927-01-12","end":"9999-12-31","name":"RIPI"}],"I
466":[{"name":"SAUZE D'OULX","end":"1928-05-03","start":"1861-03-17"},{"name
":"SAUZE D'OULX","start":"1947-07-25","end":"9999-12-31"}],"L955":[{"name":"
VILLAGUARDIA","start":"1861-03-17","end":"1928-02-20"}],"D506":[{"end":"2016
-02-22","start":"1866-11-19","name":"FARRA D'ALPAGO"}],"G608":[{"start":"186
1-03-17","end":"9999-12-31","name":"PIETRACAMELA"}],"E169":[{"name":"GREVE",
"start":"1861-03-17","end":"1977-09-09"},{"name":"GREVE IN CHIANTI","start":
"1977-09-10","end":"9999-12-31"}],"D074":[{"end":"1924-01-06","start":"1861-
03-17","name":"CORTILE SAN MARTINO"},{"start":"1924-01-07","end":"1943-05-28
","name":"CORTILE SAN MARTINO"}],"M318":[{"name":"TRECASTELLI","start":"2014
-01-01","end":"9999-12-31"}],"D725":[{"name":"FORNO CANAVESE","end":"9999-12
-31","start":"1927-06-28"},{"name":"FORNO DI RIVARA","start":"1896-05-11","e
nd":"1927-06-27"}],"I846":[{"name":"SORBOLONGO","end":"1928-06-29","start":"
1861-03-17"}],"I935":[{"name":"STAGNO LOMBARDO","start":"1895-11-04","end":"
9999-12-31"}],"B080":[{"name":"BOSIO","end":"9999-12-31","start":"1948-04-07
"}],"F015":[{"name":"MASIO","end":"9999-12-31","start":"1861-03-17"}],"H933"
:[{"name":"SAN GIUSEPPE JATO","end":"9999-12-31","start":"1864-12-26"}],"L85
1":[{"name":"VICOVARO","start":"1871-01-15","end":"9999-12-31"}],"A196":[{"e
nd":"1927-01-11","start":"1861-03-17","name":"ALIANO"},{"start":"1927-01-12"
,"end":"9999-12-31","name":"ALIANO"}],"D307":[{"start":"1940-11-23","end":"1
947-09-15","name":"DIVACCIA GROTTE DEL TIMAVO"}],"F208":[{"name":"MILIS","en
d":"1928-05-11","start":"1861-03-17"},{"end":"1946-03-24","start":"1928-05-1
2","name":"MILIS"},{"name":"MILIS","start":"1946-03-25","end":"1951-01-11"},
{"end":"1974-08-19","start":"1951-01-12","name":"MILIS"},{"name":"MILIS","st
art":"1974-08-20","end":"9999-12-31"}],"H409":[{"name":"ROCCAFORZATA","start
":"1861-03-17","end":"1923-12-21"},{"start":"1923-12-22","end":"1957-05-02",
"name":"ROCCAFORZATA"},{"start":"1957-05-03","end":"1988-11-04","name":"ROCC
AFORZATA"},{"end":"2000-02-10","start":"1988-11-05","name":"ROCCAFORZATA"},{
"name":"ROCCAFORZATA","end":"2006-03-09","start":"2000-02-11"},{"end":"9999-
12-31","start":"2006-03-10","name":"ROCCAFORZATA"}],"F555":[{"name":"MONTEMA
GGIORE AL METAURO","start":"1862-11-07","end":"1940-08-14"},{"end":"1998-09-
03","start":"1940-08-15","name":"MONTEMAGGIORE AL METAURO"},{"end":"2016-12-
31","start":"1998-09-04","name":"MONTEMAGGIORE AL METAURO"}],"I831":[{"start
":"1861-03-17","end":"1928-12-13","name":"SONICO"},{"name":"SONICO","end":"9
999-12-31","start":"1948-03-26"}],"E230":[{"name":"GUALDO TADINO","end":"999
9-12-31","start":"1861-03-17"}],"B992":[{"name":"CASOTTO","end":"1923-02-13"
,"start":"1920-10-16"},{"start":"1923-02-14","end":"1929-07-25","name":"CASO
TTO"},{"name":"CASOTTO","end":"1940-09-12","start":"1929-07-26"}],"E024":[{"
start":"1861-03-17","end":"1927-01-11","name":"GIFFLENGA"},{"name":"GIFFLENG
A","end":"1992-04-15","start":"1927-01-12"},{"name":"GIFFLENGA","end":"9999-
12-31","start":"1992-04-16"}],"D254":[{"name":"DASIO","end":"1927-06-09","st
art":"1861-03-17"}],"L018":[{"end":"1968-04-05","start":"1866-11-19","name":
"SUTRIO"},{"name":"SUTRIO","start":"1968-04-06","end":"9999-12-31"}],"Z154":
[{"name":"Federazione russa","end":"9999-12-31","start":"0000-01-01"}],"A495
":[{"end":"9999-12-31","start":"1861-03-17","name":"AULETTA"}],"H162":[{"nam
e":"RAGOLI","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14","
end":"1928-02-08","name":"RAGOLI"},{"name":"RAGOLI","start":"1928-02-09","en
d":"1952-05-16"},{"end":"2015-12-31","start":"1952-05-17","name":"RAGOLI"}],
"F958":[{"end":"1928-10-29","start":"1861-03-17","name":"NOVEDRATE"},{"start
":"1950-04-05","end":"9999-12-31","name":"NOVEDRATE"}],"E775":[{"name":"MACC
AGNO","end":"1955-11-30","start":"1953-08-25"},{"end":"2014-02-03","start":"
1955-12-01","name":"MACCAGNO"},{"name":"MACCAGNO SUPERIORE","end":"1927-01-1
1","start":"1861-03-17"},{"end":"1928-01-06","start":"1927-01-12","name":"MA
CCAGNO SUPERIORE"},{"name":"MACCAGNO SUPERIORE","end":"1953-08-24","start":"
1928-01-07"}],"B638":[{"start":"1861-03-17","end":"1936-06-03","name":"CANTO
NALE"},{"name":"CANTONALE","end":"1936-06-04","start":"1936-06-04"}],"A780":
[{"end":"9999-12-31","start":"1895-04-05","name":"BENESTARE"}],"L866":[{"sta
rt":"1861-03-17","end":"1928-11-21","name":"VIGAN\u00d2"},{"name":"VIGAN\u00
d2","start":"1953-12-06","end":"1992-04-15"},{"start":"1992-04-16","end":"99
99-12-31","name":"VIGAN\u00d2"}],"F864":[{"end":"9999-12-31","start":"1861-0
3-17","name":"NEMBRO"}],"L558":[{"name":"VALDIERI","start":"1861-03-17","end
":"1928-12-31"},{"name":"VALDIERI","start":"1929-01-01","end":"1947-09-15"},
{"end":"9999-12-31","start":"1947-09-16","name":"VALDIERI"}],"G562":[{"end":
"1928-07-28","start":"1861-03-17","name":"PIAN GAIANO"}],"H830":[{"name":"SA
N FEDELE","end":"1940-07-01","start":"1861-03-17"},{"name":"SAN FEDELE INTEL
VI","end":"2017-12-31","start":"1940-07-02"}],"L205":[{"name":"TORA E PICCIL
LI","end":"1927-01-11","start":"1861-03-17"},{"name":"TORA E PICCILLI","end"
:"1945-08-31","start":"1927-01-12"},{"end":"9999-12-31","start":"1945-09-01"
,"name":"TORA E PICCILLI"}],"E402":[{"name":"LAGANADI","end":"9999-12-31","s
tart":"1861-03-17"}],"I887":[{"start":"1874-10-08","end":"9999-12-31","name"
:"SPECCHIA"}],"G729":[{"end":"9999-12-31","start":"1861-03-17","name":"PLACA
NICA"}],"I538":[{"name":"SCILLATO","start":"1961-04-11","end":"9999-12-31"}]
,"C548":[{"end":"1927-01-11","start":"1861-03-17","name":"CERVATTO"},{"start
":"1927-01-12","end":"9999-12-31","name":"CERVATTO"}],"H529":[{"name":"RONCE
LLO","end":"2009-12-17","start":"1861-03-17"},{"end":"9999-12-31","start":"2
009-12-18","name":"RONCELLO"}],"G815":[{"name":"POMPIANO","start":"1861-03-1
7","end":"1928-12-31"},{"name":"POMPIANO","end":"9999-12-31","start":"1929-0
1-01"}],"B467":[{"name":"CAMERANA","end":"1928-03-26","start":"1861-03-17"},
{"name":"CAMERANA","end":"1947-01-09","start":"1928-03-27"},{"name":"CAMERAN
A","end":"9999-12-31","start":"1947-01-10"}],"A213":[{"end":"1923-02-13","st
art":"1920-10-16","name":"ALMAZZAGO"},{"name":"ALMAZZAGO","end":"1928-10-06"
,"start":"1923-02-14"}],"H762":[{"name":"SAN BARTOLOMEO VALMARA","end":"1928
-02-09","start":"1864-03-11"}],"B096":[{"name":"BOUSSON","start":"1882-06-09
","end":"1928-12-13"}],"D572":[{"name":"FIERA DI PRIMIERO","end":"1923-02-13
","start":"1920-10-16"},{"start":"1923-02-14","end":"1927-11-23","name":"FIE
RA DI PRIMIERO"},{"end":"2015-12-31","start":"1947-01-24","name":"FIERA DI P
RIMIERO"}],"E175":[{"end":"1928-08-14","start":"1861-03-17","name":"GRIGNANO
"}],"D426":[{"name":"ERTO E CASSO","start":"1866-11-19","end":"1968-04-05"},
{"name":"ERTO E CASSO","end":"1971-07-09","start":"1968-04-06"},{"name":"ERT
O E CASSO","end":"9999-12-31","start":"1971-07-10"}],"A180":[{"end":"1927-07
-01","start":"1861-03-17","name":"ALES"},{"name":"ALES","start":"1927-07-02"
,"end":"1946-02-07"},{"start":"1946-02-08","end":"1974-08-19","name":"ALES"}
,{"end":"1979-04-19","start":"1974-08-20","name":"ALES"},{"end":"9999-12-31"
,"start":"1979-04-20","name":"ALES"}],"G911":[{"name":"PORTO CIVITANOVA","en
d":"1938-09-13","start":"1913-06-27"}],"C298":[{"name":"CASTIGLIONE MESSER M
ARINO","start":"1863-02-06","end":"9999-12-31"}],"F489":[{"start":"1868-03-2
7","end":"1911-04-17","name":"MONTEDORO"},{"start":"1911-04-18","end":"9999-
12-31","name":"MONTEDORO"}],"H288":[{"start":"1873-02-17","end":"1902-03-09"
,"name":"RIGNANO FLAMINIO"},{"start":"1902-03-10","end":"9999-12-31","name":
"RIGNANO FLAMINIO"}],"G129":[{"end":"9999-12-31","start":"1861-03-17","name"
:"ORSOMARSO"}],"H624":[{"name":"ROZZO","end":"1922-02-01","start":"1921-01-0
5"},{"name":"ROZZO","end":"1923-02-07","start":"1922-02-02"},{"end":"1947-09
-15","start":"1923-02-08","name":"ROZZO"}],"E581":[{"start":"1861-03-17","en
d":"1992-04-15","name":"LIERNA"},{"end":"9999-12-31","start":"1992-04-16","n
ame":"LIERNA"}],"E769":[{"name":"LUVINATE","start":"1861-03-17","end":"1927-
01-11"},{"name":"LUVINATE","end":"1927-07-26","start":"1927-01-12"},{"end":"
9999-12-31","start":"1957-08-13","name":"LUVINATE"}],"C388":[{"name":"CAVAZU
CCHERINA","start":"1866-11-19","end":"1930-12-06"},{"name":"JESOLO","start":
"1930-12-07","end":"9999-12-31"}],"C841":[{"start":"1893-05-26","end":"1923-
03-18","name":"COLLALTO SABINO"},{"start":"1923-03-19","end":"1927-01-11","n
ame":"COLLALTO SABINO"},{"name":"COLLALTO SABINO","end":"9999-12-31","start"
:"1927-01-12"}],"Z225":[{"name":"Iraq","start":"0000-01-01","end":"9999-12-3
1"}],"E522":[{"start":"1868-06-06","end":"1927-07-26","name":"LENDINARA"},{"
name":"LENDINARA","end":"9999-12-31","start":"1927-07-27"}],"D125":[{"end":"
1923-02-13","start":"1920-10-16","name":"COVELO"},{"start":"1923-02-14","end
":"1928-05-21","name":"COVELO"}],"L671":[{"name":"VARANO","end":"1906-09-29"
,"start":"1861-03-17"},{"name":"VARANO BORGHI","start":"1906-09-30","end":"1
927-01-11"},{"start":"1927-01-12","end":"9999-12-31","name":"VARANO BORGHI"}
],"H398":[{"start":"1861-03-17","end":"1927-01-11","name":"ROCCA D'EVANDRO"}
,{"start":"1927-01-12","end":"1945-08-31","name":"ROCCA D'EVANDRO"},{"name":
"ROCCA D'EVANDRO","start":"1945-09-01","end":"1963-08-26"},{"name":"ROCCA D'
EVANDRO","start":"1963-08-27","end":"9999-12-31"}],"E336":[{"end":"1927-01-1
1","start":"1861-03-17","name":"ISILI"},{"name":"ISILI","end":"2005-12-31","
start":"1927-01-12"},{"name":"ISILI","start":"2006-01-01","end":"2016-04-27"
},{"end":"9999-12-31","start":"2016-04-28","name":"ISILI"}],"D819":[{"name":
"FUMONE","end":"1927-01-11","start":"1871-01-15"},{"name":"FUMONE","start":"
1927-01-12","end":"9999-12-31"}],"A607":[{"name":"BANCHETTE","end":"1927-01-
11","start":"1861-03-17"},{"name":"BANCHETTE","start":"1927-01-12","end":"19
45-10-04"},{"start":"1945-10-05","end":"9999-12-31","name":"BANCHETTE"}],"A7
96":[{"end":"1927-01-11","start":"1861-03-17","name":"BERGEGGI"},{"name":"BE
RGEGGI","start":"1927-01-12","end":"9999-12-31"}],"C945":[{"end":"1929-01-22
","start":"1861-03-17","name":"CONCENEDO"}],"C836":[{"name":"COLFELICE","sta
rt":"1923-12-21","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31
","name":"COLFELICE"}],"E476":[{"name":"LAUCO","end":"1968-04-05","start":"1
866-11-19"},{"end":"9999-12-31","start":"1968-04-06","name":"LAUCO"}],"E888"
:[{"start":"1861-03-17","end":"1928-02-20","name":"MANGONE"},{"start":"1934-
03-07","end":"9999-12-31","name":"MANGONE"}],"A573":[{"start":"1862-10-20","
end":"9999-12-31","name":"BAGNOLO IN PIANO"}],"E537":[{"end":"1923-12-21","s
tart":"1861-03-17","name":"LEPORANO"},{"name":"LEPORANO","start":"1923-12-22
","end":"9999-12-31"}],"E341":[{"name":"ISOLA DEL CANTONE","end":"9999-12-31
","start":"1863-04-23"}],"G064":[{"start":"1895-10-10","end":"1927-01-11","n
ame":"ONAN\u00cc"},{"start":"1927-01-12","end":"9999-12-31","name":"ONAN\u00
cc"}],"Z112":[{"end":"9999-12-31","start":"0000-01-01","name":"Germania"}],"
D212":[{"end":"1928-12-14","start":"1861-03-17","name":"CUQUELLO"}],"M152":[
{"name":"ZECCONE","start":"1871-06-01","end":"1929-02-04"},{"name":"ZECCONE"
,"end":"9999-12-31","start":"1947-11-27"}],"D695":[{"start":"1861-03-17","en
d":"1928-06-25","name":"FORDONGIANUS"},{"end":"1951-01-11","start":"1928-06-
26","name":"FORDONGIANUS"},{"end":"1974-08-19","start":"1951-01-12","name":"
FORDONGIANUS"},{"name":"FORDONGIANUS","start":"1974-08-20","end":"9999-12-31
"}],"B964":[{"name":"CASEZ","start":"1920-10-16","end":"1923-02-13"},{"name"
:"CASEZ","start":"1923-02-14","end":"1928-11-26"},{"name":"CASEZ","start":"1
953-01-04","end":"1968-08-27"}],"D527":[{"name":"FELITTO","end":"9999-12-31"
,"start":"1861-03-17"}],"B367":[{"name":"CAIRANO","end":"9999-12-31","start"
:"1861-03-17"}],"I969":[{"end":"1927-01-11","start":"1861-03-17","name":"STR
AMBINELLO"},{"start":"1927-01-12","end":"1929-04-12","name":"STRAMBINELLO"},
{"start":"1947-10-11","end":"9999-12-31","name":"STRAMBINELLO"}],"A711":[{"n
ame":"BASTIDA DE' DOSSI","end":"2014-02-03","start":"1861-03-17"}],"I666":[{
"name":"SERRE","start":"1861-03-17","end":"9999-12-31"}],"H650":[{"start":"1
864-01-29","end":"1928-11-30","name":"SABBIO CHIESE"},{"name":"SABBIO CHIESE
","start":"1928-12-01","end":"1956-09-15"},{"name":"SABBIO CHIESE","start":"
1956-09-16","end":"9999-12-31"}],"F083":[{"start":"1861-03-17","end":"9999-1
2-31","name":"MEDICINA"}],"D532":[{"start":"1861-03-17","end":"1927-11-05","
name":"FENESTRELLE"},{"name":"FENESTRELLE","end":"1948-05-06","start":"1927-
11-06"},{"start":"1948-05-07","end":"9999-12-31","name":"FENESTRELLE"}],"F75
7":[{"start":"1920-10-16","end":"1923-02-13","name":"MORTER"},{"start":"1923
-02-14","end":"1927-01-11","name":"MORTER"},{"end":"1928-12-09","start":"192
7-01-12","name":"MORTER"}],"E135":[{"start":"1861-03-17","end":"1926-01-21",
"name":"GRANAGLIONE"},{"name":"GRANAGLIONE","end":"2015-12-31","start":"1926
-01-22"}],"L687":[{"start":"1920-10-16","end":"1923-02-13","name":"VARNA"},{
"name":"VARNA","start":"1923-02-14","end":"1927-01-11"},{"name":"VARNA","sta
rt":"1927-01-12","end":"1929-05-06"},{"end":"1973-08-14","start":"1929-05-07
","name":"VARNA"},{"name":"VARNA","start":"1973-08-15","end":"2013-05-16"},{
"start":"2013-05-17","end":"9999-12-31","name":"VARNA"}],"E217":[{"name":"GR
UMELLO CON FARVENGO CREMONESE","start":"1868-01-01","end":"1915-03-18"},{"en
d":"9999-12-31","start":"1915-03-19","name":"GRUMELLO CREMONESE ED UNITI"}],
"B566":[{"name":"CAMPOSANTO","start":"1867-01-01","end":"9999-12-31"}],"D326
":[{"end":"1923-02-07","start":"1921-01-05","name":"DOL-OTLIZA"},{"start":"1
923-02-08","end":"1923-04-26","name":"DOL-OTLIZA"},{"name":"DOL-OTTELZA","st
art":"1923-04-27","end":"1927-01-11"},{"end":"1928-07-27","start":"1927-01-1
2","name":"DOL-OTTELZA"}],"B858":[{"start":"1861-03-17","end":"1970-03-02","
name":"CASACALENDA"},{"name":"CASACALENDA","start":"1970-03-03","end":"9999-
12-31"}],"F299":[{"name":"PORTO EMPEDOCLE","start":"1863-01-19","end":"9999-
12-31"}],"C876":[{"name":"COLLEVECCHIO","end":"1923-03-18","start":"1871-03-
01"},{"start":"1923-03-19","end":"1927-01-11","name":"COLLEVECCHIO"},{"name"
:"COLLEVECCHIO","start":"1927-01-12","end":"9999-12-31"}],"E436":[{"start":"
1861-03-17","end":"9999-12-31","name":"LANDIONA"}],"H498":[{"end":"1928-05-1
2","start":"1861-03-17","name":"ROLETTO"},{"name":"ROLETTO","end":"9999-12-3
1","start":"1954-07-15"}],"E603":[{"name":"LINGUEGLIETTA","start":"1861-03-1
7","end":"1928-03-05"}],"F644":[{"name":"MONTESEGALE","end":"1905-11-03","st
art":"1861-03-17"},{"end":"9999-12-31","start":"1905-11-04","name":"MONTESEG
ALE"}],"L646":[{"start":"1866-11-19","end":"1938-11-23","name":"VALROVINA"}]
,"L949":[{"end":"9999-12-31","start":"1867-08-26","name":"VILLAFRANCA DI VER
ONA"}],"E577":[{"name":"LICIGNANO DI NAPOLI","start":"1886-02-12","end":"192
9-04-12"}],"A533":[{"name":"AZZONE","start":"1861-03-17","end":"1928-02-22"}
,{"name":"AZZONE","end":"9999-12-31","start":"1947-05-22"}],"B011":[{"name":
"BORGOFORTE","end":"2014-02-03","start":"1872-01-01"}],"C488":[{"start":"186
1-03-17","end":"1970-03-02","name":"CERCEPICCOLA"},{"name":"CERCEPICCOLA","e
nd":"9999-12-31","start":"1970-03-03"}],"G594":[{"name":"PIEDICAVALLO","end"
:"1906-08-25","start":"1861-03-17"},{"end":"1927-01-11","start":"1906-08-26"
,"name":"PIEDICAVALLO"},{"name":"PIEDICAVALLO","end":"1992-04-15","start":"1
927-01-12"},{"start":"1992-04-16","end":"9999-12-31","name":"PIEDICAVALLO"}]
,"F892":[{"start":"1861-03-17","end":"1927-01-11","name":"NICOSIA"},{"name":
"NICOSIA","start":"1927-01-12","end":"9999-12-31"}],"E441":[{"name":"LANUSEI
","end":"1927-01-11","start":"1861-03-17"},{"name":"LANUSEI","end":"2005-12-
31","start":"1927-01-12"},{"start":"2006-01-01","end":"2016-04-27","name":"L
ANUSEI"},{"start":"2016-04-28","end":"9999-12-31","name":"LANUSEI"}],"G699":
[{"name":"PIRAINO","end":"9999-12-31","start":"1861-03-17"}],"I651":[{"name"
:"SERRAPETRONA","start":"1861-03-17","end":"9999-12-31"}],"H194":[{"name":"R
AVANUSA","end":"1977-07-05","start":"1861-03-17"},{"start":"1977-07-06","end
":"9999-12-31","name":"RAVANUSA"}],"G996":[{"name":"PRATI","end":"1923-02-13
","start":"1920-10-16"},{"name":"PRATI","end":"1927-01-11","start":"1923-02-
14"},{"name":"PRATI","start":"1927-01-12","end":"1931-12-06"}],"F549":[{"sta
rt":"1861-03-17","end":"2009-07-11","name":"MONTELPARO"},{"end":"9999-12-31"
,"start":"2009-07-12","name":"MONTELPARO"}],"B551":[{"end":"9999-12-31","sta
rt":"1871-03-30","name":"CAMPOMORONE"}],"G244":[{"name":"PAINA","start":"186
1-03-17","end":"1869-02-28"}],"L100":[{"start":"1866-11-19","end":"9999-12-3
1","name":"TEOLO"}],"C184":[{"name":"CASTELLO D'AGOGNA","end":"9999-12-31","
start":"1861-03-17"}],"E376":[{"end":"9999-12-31","start":"1861-03-17","name
":"ITTIREDDU"}],"L631":[{"end":"9999-12-31","start":"1863-01-26","name":"VAL
LORIATE"}],"H817":[{"end":"1923-02-07","start":"1921-01-05","name":"SAN DANI
ELE DEL CARSO"},{"name":"SAN DANIELE DEL CARSO","end":"1927-01-11","start":"
1923-02-08"},{"name":"SAN DANIELE DEL CARSO","end":"1928-03-12","start":"192
7-01-12"},{"name":"SAN DANIELE DEL CARSO","end":"1947-09-15","start":"1928-0
3-13"}],"I005":[{"start":"1868-07-01","end":"9999-12-31","name":"SAN MARTINO
 DALL'ARGINE"}],"H973":[{"name":"SAN LUPO","start":"1861-03-17","end":"9999-
12-31"}],"M418":[{"start":"2019-01-01","end":"9999-12-31","name":"PIADENA DR
IZZONA"}],"I871":[{"name":"SOVER","start":"1920-10-16","end":"1923-02-13"},{
"name":"SOVER","start":"1923-02-14","end":"9999-12-31"}],"E270":[{"name":"GU
SPINI","start":"1861-03-17","end":"1974-08-19"},{"start":"1974-08-20","end":
"2005-12-31","name":"GUSPINI"},{"end":"2016-04-27","start":"2006-01-01","nam
e":"GUSPINI"},{"name":"GUSPINI","end":"9999-12-31","start":"2016-04-28"}],"A
285":[{"start":"1861-03-17","end":"2009-07-26","name":"ANDRIA"},{"name":"AND
RIA","start":"2009-07-27","end":"9999-12-31"}],"H794":[{"name":"SAN CESARIO 
SUL PANARO","end":"9999-12-31","start":"1862-12-29"}],"D407":[{"end":"9999-1
2-31","start":"1866-11-19","name":"ENEGO"}],"C656":[{"name":"CHIUSAFORTE","s
tart":"1867-09-02","end":"1928-04-30"},{"end":"1968-04-05","start":"1928-05-
01","name":"CHIUSAFORTE"},{"name":"CHIUSAFORTE","end":"9999-12-31","start":"
1968-04-06"}],"L700":[{"start":"1866-11-19","end":"9999-12-31","name":"VAZZO
LA"}],"I218":[{"start":"1927-01-12","end":"1957-07-21","name":"SANTA LUCE OR
CIANO"}],"C959":[{"name":"CONFIGNI","end":"1923-03-18","start":"1861-03-17"}
,{"name":"CONFIGNI","end":"1927-01-11","start":"1923-03-19"},{"name":"CONFIG
NI","start":"1927-01-12","end":"9999-12-31"}],"H003":[{"name":"PRATO ALLA DR
AVA","end":"1923-02-13","start":"1920-10-16"},{"name":"PRATO ALLA DRAVA","st
art":"1923-02-14","end":"1927-01-11"},{"start":"1927-01-12","end":"1928-12-0
9","name":"PRATO ALLA DRAVA"}],"C784":[{"name":"CIVITELLA SAN PAOLO","start"
:"1871-01-15","end":"9999-12-31"}],"M114":[{"name":"VOLDOMINO","start":"1861
-03-17","end":"1927-01-11"},{"end":"1928-02-08","start":"1927-01-12","name":
"VOLDOMINO"}],"I975":[{"end":"1923-02-13","start":"1920-10-16","name":"STREM
BO"},{"start":"1923-02-14","end":"1928-02-15","name":"STREMBO"},{"name":"STR
EMBO","start":"1928-02-16","end":"1947-01-14"},{"name":"STREMBO","end":"9999
-12-31","start":"1947-01-15"}],"C209":[{"name":"CASTELMEZZANO","start":"1861
-03-17","end":"1928-09-24"},{"start":"1932-02-07","end":"9999-12-31","name":
"CASTELMEZZANO"}],"A395":[{"name":"BRESSANA","end":"1928-10-06","start":"187
3-08-08"}],"G092":[{"start":"1861-03-17","end":"1929-05-28","name":"ORENO"}]
,"B678":[{"end":"9999-12-31","start":"1867-11-25","name":"CAPPELLA MAGGIORE"
}],"Z120":[{"name":"Lussemburgo","start":"0000-01-01","end":"9999-12-31"}],"
D220":[{"end":"1928-12-13","start":"1861-03-17","name":"CURNASCO"}],"I578":[
{"name":"SEGUSINO","start":"1866-11-19","end":"9999-12-31"}],"E050":[{"end":
"9999-12-31","start":"1861-03-17","name":"GIRIFALCO"}],"H870":[{"name":"SAN 
GIACOMO DELLE SEGNATE","start":"1922-09-19","end":"9999-12-31"}],"A111":[{"s
tart":"1861-03-17","end":"9999-12-31","name":"AIROLE"}],"E302":[{"name":"INT
IMIANO","start":"1861-03-17","end":"1928-12-15"}],"G980":[{"name":"PRALUNGO"
,"end":"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"1992-
04-15","name":"PRALUNGO"},{"name":"PRALUNGO","end":"9999-12-31","start":"199
2-04-16"}],"F157":[{"name":"MESORACA","start":"1861-03-17","end":"1992-04-15
"},{"name":"MESORACA","end":"9999-12-31","start":"1992-04-16"}],"A929":[{"en
d":"9999-12-31","start":"1861-03-17","name":"BOGOGNO"}],"Z358":[{"name":"Bot
swana","start":"0000-01-01","end":"9999-12-31"}],"E735":[{"start":"1861-03-1
7","end":"9999-12-31","name":"LUISAGO"}],"A626":[{"end":"9999-12-31","start"
:"1861-03-17","name":"BARBARA"}],"E510":[{"name":"LEGGIUNO","end":"1927-01-1
1","start":"1861-03-17"},{"name":"LEGGIUNO","start":"1927-01-12","end":"1928
-01-06"},{"end":"9999-12-31","start":"1963-09-14","name":"LEGGIUNO"}],"C661"
:[{"name":"CHIUSDINO","end":"9999-12-31","start":"1861-03-17"}],"F006":[{"na
me":"MASCHITO","end":"9999-12-31","start":"1861-03-17"}],"L004":[{"name":"SU
MMONTE","end":"9999-12-31","start":"1861-03-17"}],"I457":[{"start":"1863-02-
06","end":"9999-12-31","name":"SASSO DI CASTALDA"}],"C080":[{"end":"1928-12-
13","start":"1861-03-17","name":"CASTELDELCI"},{"start":"1928-12-14","end":"
1998-09-03","name":"CASTELDELCI"},{"start":"1998-09-04","end":"2009-08-14","
name":"CASTELDELCI"},{"name":"CASTELDELCI","start":"2009-08-15","end":"9999-
12-31"}],"L576":[{"name":"VALFURVA","end":"9999-12-31","start":"1861-03-17"}
],"F574":[{"name":"MONTENARS","start":"1866-11-19","end":"1928-12-09"},{"end
":"1959-03-17","start":"1947-01-09","name":"MONTENARS"},{"start":"1959-03-18
","end":"1968-04-05","name":"MONTENARS"},{"name":"MONTENARS","start":"1968-0
4-06","end":"9999-12-31"}],"G872":[{"start":"1863-02-16","end":"1928-07-28",
"name":"PONZANO MONFERRATO"},{"name":"PONZANO MONFERRATO","end":"9999-12-31"
,"start":"1928-07-29"}],"E647":[{"name":"LOD\u00c8","start":"1861-03-17","en
d":"1927-01-11"},{"name":"LOD\u00c8","end":"9999-12-31","start":"1927-01-12"
}],"D730":[{"name":"FORTE DEI MARMI","start":"1913-05-11","end":"9999-12-31"
}],"L437":[{"name":"TRIVIGLIANO","start":"1871-01-15","end":"1927-01-11"},{"
end":"9999-12-31","start":"1927-01-12","name":"TRIVIGLIANO"}],"I920":[{"end"
:"1927-12-31","start":"1861-03-17","name":"SPOCCIA"}],"A301":[{"start":"1863
-02-23","end":"1992-04-15","name":"ANNONE DI BRIANZA"},{"end":"9999-12-31","
start":"1992-04-16","name":"ANNONE DI BRIANZA"}],"G279":[{"name":"PALLANZA",
"start":"1880-06-25","end":"1927-11-01"},{"end":"1939-06-04","start":"1927-1
1-02","name":"PALLANZA"}],"Z341":[{"name":"Sao Tom\u00e9 e Principe","start"
:"0000-01-01","end":"9999-12-31"}],"L793":[{"name":"VERTENEGLIO","end":"1923
-02-07","start":"1921-01-05"},{"end":"1977-04-03","start":"1923-02-08","name
":"VERTENEGLIO"}],"E112":[{"end":"1929-05-03","start":"1861-03-17","name":"G
ORZONE"}],"H090":[{"name":"PULSANO","end":"1923-12-21","start":"1861-03-17"}
,{"name":"PULSANO","start":"1923-12-22","end":"9999-12-31"}],"D515":[{"end":
"1926-08-23","start":"1873-12-29","name":"FAVARO VENETO"}],"M257":[{"end":"9
999-12-31","start":"1975-03-23","name":"PORTOPALO DI CAPO PASSERO"}],"L377":
[{"end":"9999-12-31","start":"1861-03-17","name":"TRENTINARA"}],"C992":[{"st
art":"1866-11-19","end":"9999-12-31","name":"CORDIGNANO"}],"E225":[{"start":
"1920-10-16","end":"1923-02-13","name":"GRUMO"},{"start":"1923-02-14","end":
"1928-05-21","name":"GRUMO"}],"Z522":[{"name":"Barbados","start":"0000-01-01
","end":"9999-12-31"}],"D055":[{"name":"CORTE DE' CORTESI","end":"1867-12-31
","start":"1861-03-17"}],"L422":[{"name":"TRICHIANA","start":"1866-11-19","e
nd":"2019-01-29"}],"F976":[{"name":"NULE","end":"9999-12-31","start":"1861-0
3-17"}],"L974":[{"name":"VILLANOVA MONDOV\u00cc","end":"9999-12-31","start":
"1862-12-19"}],"F679":[{"end":"1938-12-17","start":"1861-03-17","name":"MONT
IGNOSO"},{"name":"MONTIGNOSO","start":"1946-03-04","end":"9999-12-31"}],"Z33
6":[{"start":"0000-01-01","end":"9999-12-31","name":"Egitto"}],"H982":[{"nam
e":"SAN MARCO D'ALUNZIO","start":"1867-11-25","end":"1929-04-10"},{"end":"19
80-10-18","start":"1933-02-04","name":"SAN MARCO D'ALUNZIO"},{"end":"1984-12
-08","start":"1980-10-19","name":"SAN MARCO D'ALUNZIO"},{"name":"SAN MARCO D
'ALUNZIO","end":"9999-12-31","start":"1984-12-09"}],"H119":[{"name":"QUARTUC
CIU","end":"1928-05-11","start":"1861-03-17"},{"name":"QUARTUCCIU","start":"
1983-02-15","end":"9999-12-31"}],"I823":[{"name":"SOMMARIVA PERNO","start":"
1871-04-01","end":"9999-12-31"}],"G614":[{"name":"SATRIANO DI LUCANIA","end"
:"9999-12-31","start":"1887-01-07"}],"M304":[{"name":"MOSSO","start":"1999-0
1-01","end":"2018-12-31"}],"H921":[{"end":"1992-04-15","start":"1895-12-20",
"name":"SAN GIOVANNI IN MARIGNANO"},{"name":"SAN GIOVANNI IN MARIGNANO","sta
rt":"1992-04-16","end":"9999-12-31"}],"L481":[{"start":"1921-01-05","end":"1
923-02-07","name":"UCOVIZZA"},{"name":"UCOVIZZA","start":"1923-02-08","end":
"1923-04-26"},{"name":"UGOVIZZA VALBRUNA","start":"1923-04-27","end":"1928-0
8-07"}],"G353":[{"end":"1968-04-05","start":"1866-11-19","name":"PASIANO DI 
PORDENONE"},{"name":"PASIANO DI PORDENONE","end":"9999-12-31","start":"1968-
04-06"}],"I091":[{"name":"SAN PIETRO","end":"1923-02-07","start":"1921-01-05
"},{"start":"1923-02-08","end":"1923-04-26","name":"SAN PIETRO"},{"end":"192
7-01-11","start":"1923-04-27","name":"SAN PIETRO DI GORIZIA"},{"start":"1927
-01-12","end":"1927-01-12","name":"SAN PIETRO DI GORIZIA"}],"E686":[{"name":
"LORENTINO","start":"1861-03-17","end":"1928-12-13"}],"E989":[{"end":"9999-1
2-31","start":"1963-04-28","name":"MARTINSICURO"}],"E754":[{"name":"LUSCIANO
","end":"9999-12-31","start":"1946-07-03"}],"M077":[{"start":"1861-03-17","e
nd":"1929-05-14","name":"VISONE"},{"name":"VISONE","start":"1947-11-26","end
":"9999-12-31"}],"F214":[{"name":"MILO","start":"1955-02-15","end":"9999-12-
31"}],"L216":[{"start":"1861-03-17","end":"9999-12-31","name":"TORGIANO"}],"
G519":[{"name":"PETRURO","end":"1928-01-27","start":"1861-03-17"},{"name":"P
ETRURO","end":"1950-05-22","start":"1945-08-28"},{"start":"1950-05-23","end"
:"9999-12-31","name":"PETRURO IRPINO"}],"D918":[{"end":"1927-12-03","start":
"1861-03-17","name":"GARDONE VAL TROMPIA"},{"end":"9999-12-31","start":"1927
-12-04","name":"GARDONE VAL TROMPIA"}],"H825":[{"end":"9999-12-31","start":"
1864-04-01","name":"SAN DONATO DI NINEA"}],"C638":[{"name":"CHIOGGIA","end":
"1937-09-02","start":"1866-11-19"},{"name":"CHIOGGIA","end":"9999-12-31","st
art":"1937-09-03"}],"H719":[{"name":"SALORNO","end":"1923-02-13","start":"19
20-10-16"},{"name":"SALORNO","end":"1927-01-12","start":"1923-02-14"},{"star
t":"1927-01-13","end":"1948-03-13","name":"SALORNO"},{"end":"2019-10-10","st
art":"1948-03-14","name":"SALORNO"},{"start":"2019-10-11","end":"9999-12-31"
,"name":"SALORNO SULLA STRADA DEL VINO"}],"L310":[{"name":"TOSCANELLA","end"
:"1911-10-21","start":"1871-01-15"},{"name":"TUSCANIA","start":"1911-10-22",
"end":"1927-01-11"},{"name":"TUSCANIA","start":"1927-01-12","end":"9999-12-3
1"}],"C096":[{"name":"CASTEL DI SANGRO","start":"1861-03-17","end":"9999-12-
31"}],"I648":[{"start":"1861-03-17","end":"9999-12-31","name":"SERRAMEZZANA"
}],"E458":[{"start":"1861-03-17","end":"1927-01-11","name":"LA SALLE"},{"end
":"1929-05-27","start":"1927-01-12","name":"LA SALLE"},{"end":"9999-12-31","
start":"1946-01-30","name":"LA SALLE"},{"name":"SALA DORA","start":"1935-06-
05","end":"1946-01-29"}],"G806":[{"name":"POMARICO","end":"1927-01-11","star
t":"1861-03-17"},{"end":"9999-12-31","start":"1927-01-12","name":"POMARICO"}
],"C467":[{"name":"CENTA SAN NICOL\u00d2","end":"1923-02-13","start":"1920-1
0-16"},{"name":"CENTA SAN NICOL\u00d2","end":"1929-01-23","start":"1923-02-1
4"},{"end":"1988-08-02","start":"1947-05-17","name":"CENTA SAN NICOL\u00d2"}
,{"name":"CENTA SAN NICOL\u00d2","start":"1988-08-03","end":"2015-12-31"}],"
M199":[{"name":"ZUGLIANO","start":"1866-11-19","end":"1943-06-02"},{"end":"9
999-12-31","start":"1943-06-03","name":"ZUGLIANO"}],"L502":[{"start":"1861-0
3-17","end":"9999-12-31","name":"URGNANO"}],"A863":[{"name":"BIENO","end":"1
923-02-13","start":"1920-10-16"},{"start":"1923-02-14","end":"1928-02-20","n
ame":"BIENO"},{"start":"1947-03-30","end":"9999-12-31","name":"BIENO"}],"Z40
2":[{"end":"9999-12-31","start":"0000-01-01","name":"Groenlandia"}],"A442":[
{"name":"ARSIA","start":"1937-11-06","end":"1939-09-18"},{"start":"1939-09-1
9","end":"1947-09-15","name":"ARSIA"}],"F072":[{"name":"MAZZUNNO","end":"192
8-05-12","start":"1861-03-17"}],"H086":[{"start":"1861-03-17","end":"1928-09
-24","name":"PUEGNAGO"},{"end":"1971-11-04","start":"1928-09-25","name":"PUE
GNAGO"},{"start":"1971-11-05","end":"9999-12-31","name":"PUEGNAGO DEL GARDA"
}],"F665":[{"name":"MONTE VIDON CORRADO","end":"2009-07-11","start":"1861-03
-17"},{"name":"MONTE VIDON CORRADO","start":"2009-07-12","end":"9999-12-31"}
],"F902":[{"name":"NIZZA MONFERRATO","end":"1929-05-27","start":"1861-03-17"
},{"name":"NIZZA MONFERRATO","end":"1935-04-07","start":"1929-05-28"},{"name
":"NIZZA MONFERRATO","start":"1935-04-08","end":"9999-12-31"}],"E154":[{"nam
e":"GRAVERE","end":"9999-12-31","start":"1861-03-17"}],"B298":[{"end":"1928-
07-29","start":"1864-11-07","name":"BUSSOLINO GASSINESE"}],"A960":[{"name":"
BONARCADO","end":"1974-08-19","start":"1861-03-17"},{"name":"BONARCADO","sta
rt":"1974-08-20","end":"9999-12-31"}],"F859":[{"end":"1928-05-02","start":"1
861-03-17","name":"NEBBIUNO"},{"end":"1950-09-08","start":"1928-05-03","name
":"NEBBIUNO"},{"name":"NEBBIUNO","start":"1950-09-09","end":"9999-12-31"}],"
M010":[{"name":"VILLA ROMAN\u00d2","start":"1861-03-17","end":"1929-05-15"}]
,"D130":[{"end":"1928-03-24","start":"1861-03-17","name":"CRANA GATTUGNO"}],
"F252":[{"name":"MIZZOLE","end":"1933-09-01","start":"1866-11-19"}],"B841":[
{"end":"9999-12-31","start":"1861-03-17","name":"CARR\u00d9"}],"E712":[{"nam
e":"LU","end":"1929-02-11","start":"1861-03-17"},{"start":"1929-02-12","end"
:"2019-01-31","name":"LU"}],"L193":[{"name":"TOLLEGNO","start":"1861-03-17",
"end":"1927-01-11"},{"start":"1927-01-12","end":"1992-04-15","name":"TOLLEGN
O"},{"start":"1992-04-16","end":"9999-12-31","name":"TOLLEGNO"}],"B388":[{"s
tart":"1871-01-15","end":"1927-01-11","name":"CALCATA"},{"end":"9999-12-31",
"start":"1927-01-12","name":"CALCATA"}],"D601":[{"start":"1861-03-17","end":
"1927-01-11","name":"FINALE MARINA"},{"name":"FINALE MARINA","start":"1927-0
1-12","end":"1927-01-12"}],"E690":[{"end":"9999-12-31","start":"1861-03-17",
"name":"LORETO"}],"D269":[{"end":"9999-12-31","start":"1861-03-17","name":"D
ELICETO"}],"H047":[{"end":"2019-05-14","start":"1861-03-17","name":"PRESICCE
"}],"B836":[{"name":"CARREGA LIGURE","end":"9999-12-31","start":"1895-01-21"
}],"D443":[{"start":"1861-03-17","end":"1927-01-11","name":"ESTERZILI"},{"na
me":"ESTERZILI","start":"1927-01-12","end":"2005-12-31"},{"name":"ESTERZILI"
,"start":"2006-01-01","end":"2016-04-27"},{"name":"ESTERZILI","start":"2016-
04-28","end":"9999-12-31"}],"B945":[{"end":"9999-12-31","start":"1877-03-09"
,"name":"CASATISMA"}],"D862":[{"start":"1861-03-17","end":"1979-01-13","name
":"GALATINA"},{"name":"GALATINA","start":"1979-01-14","end":"2004-05-11"},{"
end":"9999-12-31","start":"2004-05-12","name":"GALATINA"}],"L668":[{"end":"9
999-12-31","start":"1863-04-23","name":"VAPRIO D'AGOGNA"}],"A342":[{"name":"
APUANIA","start":"1938-12-18","end":"1946-03-04"}],"G045":[{"name":"OLLOMONT
","end":"1927-01-11","start":"1861-03-17"},{"name":"OLLOMONT","end":"1939-10
-20","start":"1927-01-12"},{"end":"9999-12-31","start":"1946-01-30","name":"
OLLOMONT"},{"name":"OLLOMONTE","start":"1939-10-21","end":"1946-01-29"}],"C3
67":[{"start":"1927-12-31","end":"1992-05-22","name":"CAVAGLIO-SPOCCIA"},{"n
ame":"CAVAGLIO-SPOCCIA","end":"2018-12-31","start":"1992-05-23"}],"L623":[{"
name":"VALLEVE","end":"9999-12-31","start":"1861-03-17"}],"Z302":[{"name":"A
ngola","start":"0000-01-01","end":"9999-12-31"}],"C964":[{"end":"9999-12-31"
,"start":"1866-11-19","name":"CONSELVE"}],"Z735":[{"name":"Stati Federati di
 Micronesia","start":"0000-01-01","end":"9999-12-31"}],"E358":[{"name":"ISOL
A RIZZA","start":"1872-07-08","end":"9999-12-31"}],"F027":[{"start":"1861-03
-17","end":"1923-12-21","name":"MASSAFRA"},{"end":"9999-12-31","start":"1923
-12-22","name":"MASSAFRA"}],"Z510":[{"name":"Haiti","start":"0000-01-01","en
d":"9999-12-31"}],"L410":[{"name":"TREZZO TINELLA","start":"1862-12-19","end
":"9999-12-31"}],"D717":[{"end":"1924-01-31","start":"1866-11-19","name":"FO
RNI"},{"end":"1926-03-05","start":"1924-02-01","name":"TONEZZA"},{"name":"TO
NEZZA","start":"1926-03-06","end":"1959-05-23"},{"name":"TONEZZA DEL CIMONE"
,"start":"1959-05-24","end":"9999-12-31"}],"C858":[{"name":"COLLEFERRO","sta
rt":"1935-07-23","end":"9999-12-31"}],"C566":[{"name":"CESANO MADERNO","end"
:"2009-06-29","start":"1869-02-24"},{"start":"2009-06-30","end":"9999-12-31"
,"name":"CESANO MADERNO"}],"A521":[{"end":"1927-01-11","start":"1861-03-17",
"name":"AVISE"},{"name":"AVISE","start":"1927-01-12","end":"1928-04-16"},{"s
tart":"1946-03-10","end":"9999-12-31","name":"AVISE"}],"D583":[{"start":"186
4-01-29","end":"2013-12-31","name":"FIGLINE VALDARNO"}],"E120":[{"start":"18
61-03-17","end":"1928-02-22","name":"GOZZANO"},{"end":"1947-05-22","start":"
1928-02-23","name":"GOZZANO"},{"name":"GOZZANO","start":"1947-05-23","end":"
9999-12-31"}],"F032":[{"start":"1861-03-17","end":"1920-10-18","name":"MASSA
 MARITTIMA"},{"name":"MASSA MARITTIMA","start":"1920-10-19","end":"1961-01-2
5"},{"name":"MASSA MARITTIMA","start":"1961-01-26","end":"9999-12-31"}],"I91
2":[{"name":"SPINETOLI","start":"1867-01-31","end":"9999-12-31"}],"E867":[{"
name":"MALPOTREMO","end":"1928-05-11","start":"1861-03-17"}],"B488":[{"start
":"1893-02-17","end":"1928-05-11","name":"CAMNAGO FALOPPIA"}],"A582":[{"star
t":"1861-03-17","end":"1928-01-18","name":"BAIEDO"}],"I189":[{"end":"1928-08
-07","start":"1861-03-17","name":"SANT'AGAPITO"},{"start":"1934-02-25","end"
:"1970-03-02","name":"SANT'AGAPITO"},{"end":"9999-12-31","start":"1970-03-03
","name":"SANT'AGAPITO"}],"G760":[{"name":"POGGIOFIORITO","end":"9999-12-31"
,"start":"1911-09-14"},{"end":"1911-09-13","start":"1861-03-17","name":"VILL
ARIELLI"}],"B876":[{"name":"CASALE CORTE CERRO","start":"1869-01-02","end":"
1913-01-02"},{"name":"CASALE CORTE CERRO","start":"1913-01-03","end":"1992-0
5-22"},{"start":"1992-05-23","end":"9999-12-31","name":"CASALE CORTE CERRO"}
],"A828":[{"name":"BESTAGNO","start":"1861-03-17","end":"1928-02-20"}],"B184
":[{"end":"1928-06-18","start":"1861-03-17","name":"BRIONE"},{"start":"1948-
02-13","end":"9999-12-31","name":"BRIONE"}],"D343":[{"start":"1862-09-29","e
nd":"1927-09-15","name":"DONIGALA SIURGUS"}],"D049":[{"name":"CORTALE","end"
:"9999-12-31","start":"1861-03-17"}],"C551":[{"name":"CERVESINA","start":"18
61-03-17","end":"1905-08-19"},{"end":"1959-01-06","start":"1905-08-20","name
":"CERVESINA"},{"end":"9999-12-31","start":"1959-01-07","name":"CERVESINA"}]
,"D170":[{"name":"CRISPANO","end":"9999-12-31","start":"1861-03-17"}],"H267"
:[{"start":"1871-01-15","end":"9999-12-31","name":"RIANO"}],"I954":[{"name":
"STIGLIANO","end":"1927-01-11","start":"1861-03-17"},{"name":"STIGLIANO","st
art":"1927-01-12","end":"9999-12-31"}],"G827":[{"name":"PONTE","end":"9999-1
2-31","start":"1913-01-01"}],"L337":[{"name":"TRATALIAS","end":"1974-08-19",
"start":"1861-03-17"},{"end":"2005-12-31","start":"1974-08-20","name":"TRATA
LIAS"},{"start":"2006-01-01","end":"2016-04-27","name":"TRATALIAS"},{"name":
"TRATALIAS","start":"2016-04-28","end":"9999-12-31"}],"M081":[{"start":"1861
-03-17","end":"9999-12-31","name":"VITA"}],"G160":[{"name":"OSIO SOTTO","sta
rt":"1861-03-17","end":"9999-12-31"}],"A401":[{"end":"1935-03-06","start":"1
871-01-15","name":"ARICCIA"},{"name":"ARICCIA","end":"9999-12-31","start":"1
935-03-07"}],"F936":[{"name":"NORIGLIO","start":"1920-10-16","end":"1923-02-
13"},{"end":"1927-07-14","start":"1923-02-14","name":"NORIGLIO"}],"L934":[{"
end":"9999-12-31","start":"1866-11-19","name":"VILLA DEL CONTE"}],"F845":[{"
name":"NARO","end":"1929-05-15","start":"1861-03-17"},{"name":"NARO","start"
:"1929-05-16","end":"1946-03-24"},{"name":"NARO","end":"1986-03-08","start":
"1946-03-25"},{"name":"NARO","end":"9999-12-31","start":"1986-03-09"}],"F639
":[{"name":"MONTESCHENO","start":"1861-03-17","end":"1928-05-11"},{"name":"M
ONTESCHENO","start":"1928-05-12","end":"1956-05-19"},{"end":"1992-05-22","st
art":"1956-05-20","name":"MONTESCHENO"},{"start":"1992-05-23","end":"9999-12
-31","name":"MONTESCHENO"}],"I016":[{"start":"1861-03-17","end":"9999-12-31"
,"name":"SAN MARTINO VALLE CAUDINA"}],"I357":[{"end":"9999-12-31","start":"1
863-01-19","name":"SANTO STEFANO DEL SOLE"}],"G832":[{"start":"1923-04-27","
end":"1924-08-30","name":"PONTEBBA NOVA"},{"start":"1921-01-05","end":"1923-
02-07","name":"PONTEFELLA"},{"name":"PONTEFELLA","start":"1923-02-08","end":
"1923-04-26"}],"F534":[{"name":"MONTELANICO","start":"1871-01-15","end":"999
9-12-31"}],"A643":[{"name":"BARD","start":"1861-03-17","end":"1927-01-11"},{
"name":"BARD","end":"1928-06-29","start":"1927-01-12"},{"name":"BARD","start
":"1928-06-30","end":"1946-03-24"},{"end":"9999-12-31","start":"1946-03-25",
"name":"BARD"}],"B784":[{"name":"CARLANTINO","end":"9999-12-31","start":"186
1-03-17"}],"B959":[{"end":"9999-12-31","start":"1863-02-06","name":"CASELLE 
IN PITTARI"}],"Z603":[{"name":"Cile","end":"9999-12-31","start":"0000-01-01"
}],"L291":[{"start":"1863-02-06","end":"1928-07-17","name":"TORRICELLA PELIG
NA"},{"end":"9999-12-31","start":"1928-07-18","name":"TORRICELLA PELIGNA"}],
"B656":[{"start":"1861-03-17","end":"9999-12-31","name":"CAPISTRELLO"}],"L32
2":[{"start":"1920-10-16","end":"1923-02-13","name":"TRAMBILENO"},{"end":"99
99-12-31","start":"1923-02-14","name":"TRAMBILENO"}],"I868":[{"name":"SOSTEG
NO","end":"1927-01-11","start":"1861-03-17"},{"end":"1992-04-15","start":"19
27-01-12","name":"SOSTEGNO"},{"name":"SOSTEGNO","start":"1992-04-16","end":"
9999-12-31"}],"F941":[{"start":"1928-09-25","end":"1947-09-29","name":"NOSSA
"},{"start":"1947-09-30","end":"1956-05-23","name":"NOSSA"},{"name":"PONTE N
OSSA","end":"9999-12-31","start":"1956-05-24"}],"I556":[{"name":"SEBORGA","e
nd":"9999-12-31","start":"1861-03-17"}],"H560":[{"start":"1861-03-17","end":
"9999-12-31","name":"ROSATE"}],"D770":[{"name":"FRASCARO","start":"1861-03-1
7","end":"9999-12-31"}],"L477":[{"name":"TURSI","start":"1871-03-01","end":"
1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":"TURSI"}],"L381
":[{"end":"1968-04-05","start":"1869-02-08","name":"TREPPO CARNICO"},{"start
":"1968-04-06","end":"2018-01-31","name":"TREPPO CARNICO"}],"M037":[{"name":
"VILLE DEL MONTE","end":"1923-02-13","start":"1920-10-16"},{"end":"1929-04-1
1","start":"1923-02-14","name":"VILLE DEL MONTE"}],"G453":[{"end":"1998-09-0
3","start":"1861-03-17","name":"PERGOLA"},{"end":"9999-12-31","start":"1998-
09-04","name":"PERGOLA"}],"B802":[{"start":"1861-03-17","end":"9999-12-31","
name":"CAROLEI"}],"L848":[{"end":"1928-03-05","start":"1888-06-15","name":"Z
IANO"},{"name":"ZIANO PIACENTINO","end":"9999-12-31","start":"1928-03-06"}],
"D642":[{"start":"1861-03-17","end":"1873-11-09","name":"FOCE"}],"D894":[{"n
ame":"GAMBASCA","end":"1928-03-12","start":"1861-03-17"},{"name":"GAMBASCA",
"end":"9999-12-31","start":"1946-11-12"}],"D235":[{"start":"1861-03-17","end
":"1927-01-11","name":"CUTIGLIANO"},{"name":"CUTIGLIANO","end":"1936-07-25",
"start":"1927-01-12"},{"name":"CUTIGLIANO","start":"1936-07-26","end":"1979-
11-16"},{"name":"CUTIGLIANO","start":"1979-11-17","end":"2016-12-31"}],"M404
":[{"start":"2019-01-01","end":"9999-12-31","name":"VALLE CANNOBINA"}],"C678
":[{"start":"1861-03-17","end":"9999-12-31","name":"CICOGNOLO"}],"A084":[{"e
nd":"9999-12-31","start":"1871-01-15","name":"AGOSTA"}],"B209":[{"name":"BRO
ZOLO","end":"1928-04-03","start":"1861-03-17"},{"start":"1947-01-21","end":"
9999-12-31","name":"BROZOLO"}],"Z217":[{"end":"9999-12-31","start":"0000-01-
01","name":"Taiwan"}],"D117":[{"start":"1884-09-05","end":"2006-03-03","name
":"COSTA VOLPINO"},{"end":"9999-12-31","start":"2006-03-04","name":"COSTA VO
LPINO"}],"B661":[{"start":"1861-03-17","end":"9999-12-31","name":"CAPIZZONE"
}],"M022":[{"start":"1861-03-17","end":"1928-11-02","name":"VILLA SANTA MARI
A"},{"name":"VILLA SANTA MARIA","end":"1947-05-22","start":"1928-11-03"},{"n
ame":"VILLA SANTA MARIA","start":"1947-05-23","end":"9999-12-31"}],"E720":[{
"start":"1921-01-05","end":"1923-02-07","name":"LUCINICO"},{"end":"1927-01-1
1","start":"1923-02-08","name":"LUCINICO"},{"name":"LUCINICO","start":"1927-
01-12","end":"1927-01-12"}],"I204":[{"end":"1927-01-11","start":"1880-06-25"
,"name":"SANTA GIULIA"},{"name":"SANTA GIULIA","start":"1927-01-12","end":"1
929-04-12"}],"I561":[{"name":"SECUGNAGO","end":"1992-04-15","start":"1861-03
-17"},{"name":"SECUGNAGO","end":"9999-12-31","start":"1992-04-16"}],"M108":[
{"name":"VODO CADORE","end":"9999-12-31","start":"1866-11-19"}],"D454":[{"en
d":"1927-01-11","start":"1869-05-24","name":"FABRO"},{"end":"1938-03-17","st
art":"1927-01-12","name":"FABRO"},{"start":"1938-03-18","end":"9999-12-31","
name":"FABRO"}],"C794":[{"end":"1923-02-13","start":"1920-10-16","name":"CLE
S"},{"end":"1928-09-08","start":"1923-02-14","name":"CLES"},{"name":"CLES","
start":"1928-09-09","end":"9999-12-31"}],"A834":[{"start":"1928-10-16","end"
:"1992-05-22","name":"BEURA-CARDEZZA"},{"end":"9999-12-31","start":"1992-05-
23","name":"BEURA-CARDEZZA"}],"D320":[{"end":"1921-12-30","start":"1921-01-0
5","name":"DOLE"},{"name":"DOLE","end":"1923-02-07","start":"1921-12-31"},{"
name":"DOLE","start":"1923-02-08","end":"1927-01-11"},{"start":"1927-01-12",
"end":"1928-08-23","name":"DOLE"}],"B560":[{"start":"1921-01-05","end":"1923
-02-07","name":"CAMPOROSSO"},{"name":"CAMPOROSSO","start":"1923-02-08","end"
:"1923-04-26"},{"name":"CAMPOROSSO IN VALCANALE","start":"1923-04-27","end":
"1928-06-15"}],"G021":[{"start":"1863-03-30","end":"9999-12-31","name":"OLEV
ANO DI LOMELLINA"}],"A295":[{"name":"ANGROGNA","start":"1861-03-17","end":"9
999-12-31"}],"I660":[{"name":"SERRAVALLE PISTOIESE","end":"1927-01-11","star
t":"1863-04-13"},{"start":"1927-01-12","end":"9999-12-31","name":"SERRAVALLE
 PISTOIESE"}],"L772":[{"start":"1861-03-17","end":"9999-12-31","name":"VERNA
SCA"}],"H656":[{"end":"1913-01-02","start":"1869-03-15","name":"SACCONAGO"},
{"name":"SACCONAGO","start":"1913-01-03","end":"1927-04-13"},{"end":"1928-05
-21","start":"1927-04-14","name":"SACCONAGO"}],"C003":[{"name":"CASSANO D'AD
DA","start":"1869-03-11","end":"9999-12-31"}],"H959":[{"end":"9999-12-31","s
tart":"1861-03-17","name":"SAN LORENZO"}],"H784":[{"end":"9999-12-31","start
":"1861-03-17","name":"SAN BUONO"}],"C309":[{"name":"CASTIGLIONE DEL LAGO","
end":"1937-03-11","start":"1875-05-11"},{"name":"CASTIGLIONE DEL LAGO","end"
:"9999-12-31","start":"1937-03-12"}],"Z726":[{"end":"9999-12-31","start":"00
00-01-01","name":"Samoa"}],"C973":[{"start":"1861-03-17","end":"9999-12-31",
"name":"CONTRONE"}],"D158":[{"end":"1981-12-19","start":"1861-03-17","name":
"CRESPELLANO"},{"start":"1981-12-20","end":"2013-12-31","name":"CRESPELLANO"
}],"E809":[{"end":"9999-12-31","start":"1910-03-24","name":"MAGLIANO ALFIERI
"},{"end":"1910-03-23","start":"1862-12-19","name":"MAGLIANO D'ALBA"}],"Z258
":[{"end":"9999-12-31","start":"0000-01-01","name":"Turkmenistan"}],"L405":[
{"end":"1927-01-11","start":"1861-03-17","name":"TREVISAGO"},{"name":"TREVIS
AGO","end":"1927-05-26","start":"1927-01-12"}],"Z505":[{"name":"Repubblica D
ominicana","start":"0000-01-01","end":"9999-12-31"}],"E202":[{"end":"1905-06
-29","start":"1861-03-17","name":"GROSSETO"},{"name":"GROSSETO","end":"1955-
11-25","start":"1905-06-30"},{"name":"GROSSETO","end":"9999-12-31","start":"
1955-11-26"}],"A545":[{"name":"BAGGIO","end":"1923-10-13","start":"1869-03-2
2"}],"L106":[{"end":"1923-02-13","start":"1920-10-16","name":"TERENTO"},{"na
me":"TERENTO","start":"1923-02-14","end":"1927-01-11"},{"end":"1929-03-25","
start":"1927-01-12","name":"TERENTO"},{"end":"9999-12-31","start":"1929-03-2
6","name":"TERENTO"}],"F104":[{"end":"9999-12-31","start":"1861-03-17","name
":"MELFI"}],"E370":[{"name":"ISSO","end":"9999-12-31","start":"1861-03-17"}]
,"E952":[{"end":"1923-02-07","start":"1921-01-05","name":"MARIANO"},{"start"
:"1923-02-08","end":"1923-04-26","name":"MARIANO"},{"end":"1927-01-11","star
t":"1923-04-27","name":"MARIANO DEL FRIULI"},{"name":"MARIANO DEL FRIULI","s
tart":"1927-01-12","end":"1928-09-08"},{"end":"9999-12-31","start":"1928-09-
09","name":"MARIANO DEL FRIULI"}],"E143":[{"start":"1861-03-17","end":"9999-
12-31","name":"GRANOZZO CON MONTICELLO"}],"A948":[{"name":"BOLOTANA","end":"
1927-01-11","start":"1861-03-17"},{"name":"BOLOTANA","end":"9999-12-31","sta
rt":"1927-01-12"}],"H661":[{"name":"SAGAMA","start":"1861-03-17","end":"1927
-01-11"},{"start":"1927-01-12","end":"1928-02-13","name":"SAGAMA"},{"name":"
SAGAMA","end":"2005-12-31","start":"1946-07-18"},{"name":"SAGAMA","start":"2
006-01-01","end":"9999-12-31"}],"G990":[{"name":"PRATA DI PRINCIPATO ULTRA",
"start":"1863-02-06","end":"9999-12-31"}],"F408":[{"start":"1861-03-17","end
":"9999-12-31","name":"MONTALDO ROERO"}],"L640":[{"name":"VALMOREA","end":"9
999-12-31","start":"1928-05-04"}],"I318":[{"start":"1863-07-13","end":"9999-
12-31","name":"SANT'EGIDIO ALLA VIBRATA"}],"H802":[{"start":"1863-04-23","en
d":"9999-12-31","name":"SAN COLOMBANO CERTENOLI"}],"C870":[{"name":"COLLE SA
N MAGNO","start":"1861-03-17","end":"1927-01-11"},{"end":"9999-12-31","start
":"1927-01-12","name":"COLLE SAN MAGNO"}],"E430":[{"name":"LA MORRA","end":"
9999-12-31","start":"1863-01-26"}],"G082":[{"name":"OPPIDO MAMERTINA","end":
"1927-08-23","start":"1863-04-10"},{"end":"9999-12-31","start":"1927-08-24",
"name":"OPPIDO MAMERTINA"}],"A385":[{"name":"ARDORE","end":"9999-12-31","sta
rt":"1861-03-17"}],"L706":[{"end":"9999-12-31","start":"1871-09-10","name":"
VEDELAGO"}],"C650":[{"start":"1911-04-18","end":"9999-12-31","name":"CHIUPPA
NO"}],"F821":[{"start":"1921-01-05","end":"1923-02-07","name":"MUSCOLI STRAS
SOLDO"},{"end":"1928-05-02","start":"1923-02-08","name":"MUSCOLI STRASSOLDO"
}],"F704":[{"end":"1929-01-21","start":"1867-01-14","name":"MONZA"},{"name":
"MONZA","end":"1935-03-28","start":"1929-01-22"},{"name":"MONZA","start":"19
35-03-29","end":"1935-07-12"},{"name":"MONZA","end":"1938-06-01","start":"19
35-07-13"},{"name":"MONZA","end":"2009-06-29","start":"1938-06-02"},{"end":"
9999-12-31","start":"2009-06-30","name":"MONZA"}],"L697":[{"end":"1923-02-13
","start":"1920-10-16","name":"VATTARO"},{"start":"1923-02-14","end":"1928-0
9-08","name":"VATTARO"},{"start":"1947-03-30","end":"1988-08-02","name":"VAT
TARO"},{"name":"VATTARO","start":"1988-08-03","end":"2015-12-31"}],"E743":[{
"start":"1861-03-17","end":"1998-09-03","name":"LUNANO"},{"name":"LUNANO","e
nd":"9999-12-31","start":"1998-09-04"}],"L525":[{"start":"1861-03-17","end":
"1923-03-18","name":"VACONE"},{"name":"VACONE","start":"1923-03-19","end":"1
927-01-11"},{"name":"VACONE","end":"1928-03-12","start":"1927-01-12"},{"name
":"VACONE","end":"9999-12-31","start":"1946-03-08"}],"F399":[{"name":"MONTAL
BANO JONICO","start":"1871-03-01","end":"1927-01-11"},{"start":"1927-01-12",
"end":"1959-04-24","name":"MONTALBANO JONICO"},{"start":"1959-04-25","end":"
1974-08-01","name":"MONTALBANO JONICO"},{"start":"1974-08-02","end":"9999-12
-31","name":"MONTALBANO JONICO"}],"F093":[{"name":"MEINA","end":"1928-05-02"
,"start":"1861-03-17"},{"name":"MEINA","start":"1928-05-03","end":"1929-05-2
0"},{"start":"1929-05-21","end":"9999-12-31","name":"MEINA"}],"F925":[{"star
t":"1861-03-17","end":"1927-12-28","name":"NOLE"},{"end":"1947-01-14","start
":"1927-12-29","name":"NOLE"},{"name":"NOLE","end":"9999-12-31","start":"194
7-01-15"}],"E061":[{"start":"1861-03-17","end":"9999-12-31","name":"GIURDIGN
ANO"}],"G344":[{"name":"PARRANO","end":"1927-01-11","start":"1861-03-17"},{"
name":"PARRANO","end":"1928-01-28","start":"1927-01-12"},{"name":"PARRANO","
start":"1928-01-29","end":"9999-12-31"}],"E276":[{"end":"1923-02-13","start"
:"1920-10-16","name":"IAVR\u00c8"},{"name":"IAVR\u00c8","end":"1928-06-27","
start":"1923-02-14"}],"G947":[{"name":"POVIGLIO","end":"1992-01-29","start":
"1861-03-17"},{"name":"POVIGLIO","start":"1992-01-30","end":"9999-12-31"}],"
D685":[{"name":"FONTEVIVO","start":"1861-03-17","end":"9999-12-31"}],"D412":
[{"start":"1861-03-17","end":"9999-12-31","name":"ENVIE"}],"C194":[{"end":"1
923-02-13","start":"1920-10-16","name":"CASTELLO TESINO"},{"name":"CASTELLO 
TESINO","end":"9999-12-31","start":"1923-02-14"}],"E516":[{"start":"1920-10-
16","end":"1923-02-13","name":"LEGOS"},{"name":"LEGOS","end":"1928-04-03","s
tart":"1923-02-14"}],"F528":[{"start":"1863-01-05","end":"1923-12-21","name"
:"MONTEGROSSO PIAN LATTE"},{"end":"1928-07-17","start":"1925-08-22","name":"
MONTEGROSSO PIAN LATTE"},{"start":"1950-02-17","end":"9999-12-31","name":"MO
NTEGROSSO PIAN LATTE"}],"B267":[{"end":"9999-12-31","start":"1861-03-17","na
me":"BUONALBERGO"}],"A413":[{"name":"ARLUNO","start":"1861-03-17","end":"199
7-07-25"},{"start":"1997-07-26","end":"2002-10-25","name":"ARLUNO"},{"end":"
9999-12-31","start":"2002-10-26","name":"ARLUNO"}],"C817":[{"name":"CODROIPO
","start":"1866-11-19","end":"1928-06-14"},{"name":"CODROIPO","start":"1928-
06-15","end":"1947-01-08"},{"name":"CODROIPO","start":"1947-01-09","end":"19
68-04-05"},{"end":"2012-07-18","start":"1968-04-06","name":"CODROIPO"},{"sta
rt":"2012-07-19","end":"9999-12-31","name":"CODROIPO"}],"G689":[{"start":"19
27-01-27","end":"1927-05-26","name":"PIONE"}],"L172":[{"name":"TINNURA","end
":"1927-01-11","start":"1861-03-17"},{"name":"TINNURA","start":"1927-01-12",
"end":"1928-02-13"},{"name":"TINNURA","start":"1946-07-18","end":"2005-12-31
"},{"name":"TINNURA","start":"2006-01-01","end":"9999-12-31"}],"H184":[{"nam
e":"RAPINO","start":"1861-03-17","end":"9999-12-31"}],"G986":[{"name":"PRARO
STINO","start":"1861-03-17","end":"1907-03-14"},{"start":"1907-03-15","end":
"1928-05-30","name":"PRAROSTINO"},{"start":"1959-06-27","end":"9999-12-31","
name":"PRAROSTINO"}],"E317":[{"name":"INZAGO","start":"1861-03-17","end":"99
99-12-31"}],"Z126":[{"name":"Paesi Bassi","end":"9999-12-31","start":"0000-0
1-01"}],"E056":[{"start":"1863-02-06","end":"9999-12-31","name":"GIULIANO TE
ATINO"}],"D226":[{"end":"9999-12-31","start":"1866-11-19","name":"CURTAROLO"
}],"D758":[{"start":"1861-03-17","end":"1923-12-21","name":"FRAMURA"},{"star
t":"1923-12-22","end":"9999-12-31","name":"FRAMURA"}],"H876":[{"end":"9999-1
2-31","start":"1861-03-17","name":"SAN GINESIO"}],"F882":[{"name":"NEVIANO D
EGLI ARDUINI","start":"1861-03-17","end":"1951-12-31"},{"start":"1952-01-01"
,"end":"9999-12-31","name":"NEVIANO DEGLI ARDUINI"}],"G584":[{"end":"1928-04
-30","start":"1861-03-17","name":"PIAZZO"}],"C498":[{"start":"1866-11-19","e
nd":"9999-12-31","name":"CEREA"}],"F289":[{"name":"MOLINI DI COLOGNOLA","sta
rt":"1861-03-17","end":"1927-02-17"}],"H488":[{"name":"ROGGIANO GRAVINA","st
art":"1864-06-23","end":"9999-12-31"}],"Z611":[{"start":"0000-01-01","end":"
9999-12-31","name":"Per\u00f9"}],"E914":[{"name":"MARANO MARCHESATO","start"
:"1861-03-17","end":"9999-12-31"}],"L928":[{"start":"1966-12-09","end":"1989
-09-27","name":"VILLA CORTESE"},{"end":"9999-12-31","start":"1989-09-28","na
me":"VILLA CORTESE"}],"F499":[{"name":"MONTEFIASCONE","end":"1927-01-11","st
art":"1871-01-15"},{"start":"1927-01-12","end":"9999-12-31","name":"MONTEFIA
SCONE"}],"H298":[{"end":"9999-12-31","start":"1864-03-11","name":"RIO SALICE
TO"}],"E236":[{"name":"GUARCINO","end":"1927-01-11","start":"1871-01-15"},{"
end":"9999-12-31","start":"1927-01-12","name":"GUARCINO"}],"Z325":[{"end":"9
999-12-31","start":"0000-01-01","name":"Liberia"}],"E591":[{"name":"LIMBIATE
","start":"1869-04-26","end":"2009-06-29"},{"end":"9999-12-31","start":"2009
-06-30","name":"LIMBIATE"}],"L963":[{"name":"VILLALVERNIA","end":"1953-07-15
","start":"1861-03-17"},{"name":"VILLALVERNIA","start":"1953-07-16","end":"9
999-12-31"}],"C624":[{"end":"1925-07-09","start":"1861-03-17","name":"CHIAVE
RANO"},{"name":"CHIAVERANO","start":"1925-07-10","end":"1927-01-11"},{"name"
:"CHIAVERANO","start":"1927-01-12","end":"1945-10-04"},{"name":"CHIAVERANO",
"start":"1945-10-05","end":"9999-12-31"}],"I714":[{"end":"1923-02-13","start
":"1920-10-16","name":"SFRUZ"},{"name":"SFRUZ","start":"1923-02-14","end":"1
929-01-23"},{"name":"SFRUZ","end":"9999-12-31","start":"1952-09-20"}],"G148"
:[{"name":"ORVIETO","end":"1927-01-11","start":"1861-03-17"},{"name":"ORVIET
O","end":"1937-08-21","start":"1927-01-12"},{"start":"1937-08-22","end":"999
9-12-31","name":"ORVIETO"}],"C288":[{"start":"1862-11-28","end":"1940-08-02"
,"name":"CASTELVETRO PIACENTINO"},{"end":"9999-12-31","start":"1940-08-03","
name":"CASTELVETRO PIACENTINO"}],"E241":[{"name":"GUARDEA","start":"1861-03-
17","end":"1927-01-11"},{"name":"GUARDEA","start":"1927-01-12","end":"9999-1
2-31"}],"C529":[{"end":"9999-12-31","start":"1861-03-17","name":"CERRETO GUI
DI"}],"D312":[{"end":"1923-02-07","start":"1921-01-05","name":"DOBERD\u00d2"
},{"name":"DOBERD\u00d2","start":"1923-02-08","end":"1923-04-26"},{"end":"19
47-09-15","start":"1923-04-27","name":"DOBERD\u00d2 DEL LAGO"},{"end":"1948-
01-09","start":"1947-09-16","name":"DOBERD\u00d2 DEL LAGO"},{"end":"1951-02-
14","start":"1948-01-10","name":"DOBERD\u00d2 DEL LAGO"},{"name":"DOBERD\u00
d2 DEL LAGO","end":"2009-01-28","start":"1951-02-15"},{"end":"9999-12-31","s
tart":"2009-01-29","name":"DOBERD\u00d2 DEL LAGO-DOBERDOB"}],"I840":[{"name"
:"SORAGNA","end":"9999-12-31","start":"1861-03-17"}],"H548":[{"end":"1927-07
-08","start":"1861-03-17","name":"RONGIO"}],"B086":[{"name":"BOTRUGNO","end"
:"9999-12-31","start":"1958-05-28"}],"C762":[{"name":"CIVIGLIO","end":"1943-
08-12","start":"1861-03-17"}],"G444":[{"name":"PERCILE","end":"9999-12-31","
start":"1899-07-26"}],"B719":[{"start":"1861-03-17","end":"9999-12-31","name
":"CARAGLIO"}],"A747":[{"name":"BELLARIA-IGEA MARINA","start":"1956-03-04","
end":"1992-04-15"},{"end":"9999-12-31","start":"1992-04-16","name":"BELLARIA
-IGEA MARINA"}],"F563":[{"name":"MONTEMESOLA","start":"1861-03-17","end":"19
23-12-21"},{"name":"MONTEMESOLA","start":"1923-12-22","end":"9999-12-31"}],"
H945":[{"name":"SAN LAZZARO DI SAVENA","end":"9999-12-31","start":"1862-10-3
1"}],"E898":[{"name":"MANZANO","end":"1923-02-13","start":"1920-10-16"},{"na
me":"MANZANO","end":"1924-01-14","start":"1923-02-14"}],"H836":[{"start":"18
72-12-16","end":"1928-09-24","name":"SAN FELICE CIRCEO"},{"name":"SAN FELICE
 CIRCEO","end":"1933-09-12","start":"1928-09-25"},{"name":"SAN FELICE CIRCEO
","start":"1933-09-13","end":"1934-10-25"},{"start":"1934-10-26","end":"1934
-12-17","name":"SAN FELICE CIRCEO"},{"name":"SAN FELICE CIRCEO","end":"1945-
04-08","start":"1934-12-18"},{"end":"9999-12-31","start":"1945-04-09","name"
:"SAN FELICE CIRCEO"}],"B047":[{"end":"1923-02-07","start":"1921-01-05","nam
e":"BERIE"},{"start":"1923-02-08","end":"1923-04-26","name":"BERIE"},{"name"
:"BORIANO","end":"1927-01-11","start":"1923-04-27"},{"start":"1927-01-12","e
nd":"1928-08-23","name":"BORIANO"}],"A019":[{"name":"ACCUMOLI","start":"1861
-03-17","end":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":
"ACCUMOLI"}],"A313":[{"name":"ANTILLO","start":"1861-03-17","end":"9999-12-3
1"}],"A654":[{"start":"1861-03-17","end":"1927-11-23","name":"BARESI"}],"A78
6":[{"name":"BERBENNO","start":"1861-03-17","end":"9999-12-31"}],"L127":[{"n
ame":"TERRANOVA SAPPO MINULIO","start":"1864-06-23","end":"1928-03-26"},{"en
d":"9999-12-31","start":"1946-04-26","name":"TERRANOVA SAPPO MINULIO"}],"L86
0":[{"end":"1960-08-18","start":"1862-12-29","name":"VIETRI SUL MARE"},{"sta
rt":"1960-08-19","end":"9999-12-31","name":"VIETRI SUL MARE"}],"I152":[{"end
":"9999-12-31","start":"1863-01-05","name":"SAN SEBASTIANO DA PO"}],"H841":[
{"name":"SAN FILI","start":"1861-03-17","end":"9999-12-31"}],"A559":[{"name"
:"BAGNI DI CASCIANA","start":"1927-12-04","end":"1956-08-14"},{"name":"CASCI
ANA TERME","start":"1956-08-15","end":"2013-12-31"}],"H388":[{"name":"ROCCAC
ARAMANICO","start":"1861-03-17","end":"1927-01-11"},{"end":"1929-03-23","sta
rt":"1927-01-12","name":"ROCCACARAMANICO"}],"L132":[{"start":"1867-08-05","e
nd":"9999-12-31","name":"TERRASSA PADOVANA"}],"I943":[{"end":"1928-04-16","s
tart":"1861-03-17","name":"STAZZONA"},{"name":"STAZZONA","end":"9999-12-31",
"start":"1954-06-30"}],"C398":[{"end":"9999-12-31","start":"1861-03-17","nam
e":"CAVEZZO"}],"E417":[{"name":"LAINO BORGO","start":"1861-03-17","end":"192
8-04-16"},{"end":"9999-12-31","start":"1947-11-25","name":"LAINO BORGO"}],"D
613":[{"end":"9999-12-31","start":"1861-03-17","name":"FIRENZUOLA"}],"D957":
[{"name":"GAZZO VERONESE","end":"1928-06-14","start":"1867-08-26"},{"start":
"1928-06-15","end":"9999-12-31","name":"GAZZO VERONESE"}],"C409":[{"name":"C
AZZAGO BRABBIA","start":"1863-02-23","end":"1927-01-11"},{"name":"CAZZAGO BR
ABBIA","start":"1927-01-12","end":"9999-12-31"}],"B982":[{"end":"9999-12-31"
,"start":"1861-03-17","name":"CASOLA VALSENIO"}],"C933":[{"name":"COMO","end
":"1937-05-17","start":"1884-12-22"},{"start":"1937-05-18","end":"1943-08-12
","name":"COMO"},{"end":"1947-11-02","start":"1943-08-13","name":"COMO"},{"s
tart":"1947-11-03","end":"9999-12-31","name":"COMO"}],"A186":[{"end":"9999-1
2-31","start":"1861-03-17","name":"ALFANO"}],"D420":[{"name":"ERBEZZO","star
t":"1866-11-19","end":"9999-12-31"}],"L727":[{"name":"VENARIA REALE","end":"
9999-12-31","start":"1861-03-17"}],"A874":[{"end":"1927-11-07","start":"1861
-03-17","name":"BINETTO"},{"name":"BINETTO","start":"1946-11-12","end":"9999
-12-31"}],"I752":[{"name":"SINNAI","end":"1928-06-29","start":"1861-03-17"},
{"name":"SINNAI","start":"1928-06-30","end":"1946-04-01"},{"name":"SINNAI","
end":"1974-08-19","start":"1946-04-02"},{"start":"1974-08-20","end":"9999-12
-31","name":"SINNAI"}],"L305":[{"start":"1861-03-17","end":"1928-05-11","nam
e":"TORTORA"},{"start":"1937-07-18","end":"9999-12-31","name":"TORTORA"}],"L
732":[{"name":"VENEGONO","end":"1960-09-29","start":"1928-09-25"}],"B825":[{
"name":"CARPINETI","end":"9999-12-31","start":"1884-06-20"}],"I897":[{"name"
:"SPEZZANO GRANDE","end":"1928-05-18","start":"1861-03-17"}],"A790":[{"name"
:"BERDO SAN GIOVANNI","start":"1921-01-05","end":"1923-02-07"},{"end":"1924-
03-08","start":"1923-02-08","name":"BERDO SAN GIOVANNI"},{"name":"BERDO SAN 
GIOVANNI","end":"1924-10-30","start":"1924-03-09"}],"C830":[{"name":"COLBORD
OLO","end":"1928-02-20","start":"1869-04-26"},{"name":"COLBORDOLO","start":"
1928-02-21","end":"1947-01-18"},{"name":"COLBORDOLO","start":"1947-01-19","e
nd":"1998-09-03"},{"end":"2013-12-31","start":"1998-09-04","name":"COLBORDOL
O"}],"I114":[{"name":"SAN PIETRO IN GUARANO","start":"1861-03-17","end":"999
9-12-31"}],"G748":[{"end":"1923-02-07","start":"1921-01-05","name":"PODRAGA"
},{"end":"1927-01-11","start":"1923-02-08","name":"PODRAGA"},{"end":"1928-01
-21","start":"1927-01-12","name":"PODRAGA"}],"E470":[{"end":"1927-01-11","st
art":"1861-03-17","name":"LA THUILE"},{"start":"1927-01-12","end":"1939-10-2
0","name":"LA THUILE"},{"name":"LA THUILE","start":"1946-01-30","end":"1947-
09-15"},{"end":"9999-12-31","start":"1947-09-16","name":"LA THUILE"},{"end":
"1946-01-29","start":"1939-10-21","name":"PORTA LITTORIA"}],"B921":[{"start"
:"1861-03-17","end":"1927-01-11","name":"CASALZUIGNO"},{"name":"CASALZUIGNO"
,"end":"1928-01-17","start":"1927-01-12"},{"end":"9999-12-31","start":"1928-
01-18","name":"CASALZUIGNO"}],"A485":[{"name":"ATESSA","end":"9999-12-31","s
tart":"1861-03-17"}],"A612":[{"name":"BANZI","start":"1901-07-16","end":"192
7-01-11"},{"name":"BANZI","end":"1944-04-29","start":"1927-01-12"},{"end":"9
999-12-31","start":"1944-04-30","name":"BANZI"}],"E330":[{"start":"1872-09-0
2","end":"1927-01-11","name":"ISCHIA DI CASTRO"},{"name":"ISCHIA DI CASTRO",
"start":"1927-01-12","end":"9999-12-31"}],"D207":[{"end":"1935-04-07","start
":"1861-03-17","name":"CUNICO"},{"end":"9999-12-31","start":"1935-04-08","na
me":"CUNICO"}],"F308":[{"name":"MOMBALDONE","start":"1861-03-17","end":"1935
-04-07"},{"name":"MOMBALDONE","end":"9999-12-31","start":"1935-04-08"}],"C16
2":[{"name":"CASTELLETTO MONFERRATO","end":"9999-12-31","start":"1937-10-16"
},{"start":"1861-03-17","end":"1937-10-15","name":"CASTELLETTO SCAZZOSO"}],"
I418":[{"name":"SAONARA","end":"9999-12-31","start":"1866-11-19"}],"Z107":[{
"start":"0000-01-01","end":"9999-12-31","name":"Danimarca"}],"B309":[{"name"
:"BUTTRIO","start":"1870-01-02","end":"1968-04-05"},{"name":"BUTTRIO","end":
"9999-12-31","start":"1968-04-06"}],"Z317":[{"end":"9999-12-31","start":"000
0-01-01","name":"Gambia"}],"B003":[{"start":"1861-03-17","end":"1927-01-11",
"name":"BORGIALLO"},{"end":"1928-09-25","start":"1927-01-12","name":"BORGIAL
LO"},{"start":"1948-01-15","end":"9999-12-31","name":"BORGIALLO"}],"E126":[{
"name":"GRADOLI","end":"1927-01-11","start":"1871-01-15"},{"name":"GRADOLI",
"start":"1927-01-12","end":"9999-12-31"}],"F118":[{"end":"1923-02-13","start
":"1920-10-16","name":"MELTINA"},{"name":"MELTINA","start":"1923-02-14","end
":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":"MELTINA"}],
"D335":[{"start":"1861-03-17","end":"1927-07-27","name":"DOMUSNOVAS CANALES"
}],"D702":[{"start":"1861-03-17","end":"1938-11-12","name":"FORIO"},{"start"
:"1945-09-22","end":"9999-12-31","name":"FORIO"}],"C560":[{"name":"CERZETO",
"start":"1861-03-17","end":"9999-12-31"}],"E611":[{"end":"9999-12-31","start
":"1861-03-17","name":"LISCIA"}],"L281":[{"name":"TORRESINA","end":"1928-05-
11","start":"1862-12-19"},{"start":"1947-01-15","end":"9999-12-31","name":"T
ORRESINA"}],"B794":[{"start":"1861-03-17","end":"1962-09-22","name":"CARMIGN
ANO"},{"end":"1992-04-15","start":"1962-09-23","name":"CARMIGNANO"},{"name":
"CARMIGNANO","start":"1992-04-16","end":"9999-12-31"}],"Z516":[{"name":"Pana
ma","end":"9999-12-31","start":"0000-01-01"}],"F414":[{"name":"MONTALLEGRO",
"end":"9999-12-31","start":"1861-03-17"}],"L416":[{"name":"TRIBOGNA","start"
:"1861-03-17","end":"9999-12-31"}],"I304":[{"end":"1913-06-16","start":"1864
-02-02","name":"SANT'ARCANGELO DI ROMAGNA"},{"start":"1913-06-17","end":"199
2-04-15","name":"SANTARCANGELO DI ROMAGNA"},{"name":"SANTARCANGELO DI ROMAGN
A","end":"9999-12-31","start":"1992-04-16"}],"G751":[{"end":"9999-12-31","st
art":"1861-03-17","name":"POGGIARDO"}],"I799":[{"end":"9999-12-31","start":"
1866-11-19","name":"SOLESINO"}],"I907":[{"start":"1861-03-17","end":"2009-07
-26","name":"SPINAZZOLA"},{"name":"SPINAZZOLA","end":"9999-12-31","start":"2
009-07-27"}],"M091":[{"end":"9999-12-31","start":"1861-03-17","name":"VITTUO
NE"}],"D475":[{"end":"1927-01-11","start":"1874-03-16","name":"FALERIA"},{"n
ame":"FALERIA","end":"9999-12-31","start":"1927-01-12"}],"B973":[{"end":"192
8-05-04","start":"1861-03-17","name":"CASLETTO"}],"I257":[{"start":"1863-02-
23","end":"1927-01-11","name":"SANT'AMBROGIO OLONA"},{"name":"SANT'AMBROGIO 
OLONA","start":"1927-01-12","end":"1927-12-28"}],"D176":[{"name":"CROCETTA",
"end":"1928-04-07","start":"1866-11-19"}],"L237":[{"name":"TORRAZZA COSTE","
end":"9999-12-31","start":"1861-03-17"}],"G479":[{"end":"1929-04-10","start"
:"1861-03-17","name":"PESARO"},{"name":"PESARO","end":"1998-09-03","start":"
1929-04-11"},{"name":"PESARO","start":"1998-09-04","end":"9999-12-31"}],"L11
5":[{"name":"TERNATE","end":"1927-01-11","start":"1861-03-17"},{"name":"TERN
ATE","start":"1927-01-12","end":"9999-12-31"}],"A094":[{"name":"AIAS","end":
"1946-01-29","start":"1939-10-21"},{"name":"AYAS","end":"1927-01-11","start"
:"1861-03-17"},{"name":"AYAS","start":"1927-01-12","end":"1939-10-20"},{"nam
e":"AYAS","start":"1946-01-30","end":"9999-12-31"}],"G339":[{"name":"PAROLDO
","end":"1928-05-11","start":"1861-03-17"},{"end":"9999-12-31","start":"1946
-11-12","name":"PAROLDO"}],"G033":[{"start":"1863-01-26","end":"1928-05-22",
"name":"TAVERNETTE"}],"L829":[{"name":"VIALE","end":"1935-04-07","start":"18
61-03-17"},{"start":"1935-04-08","end":"9999-12-31","name":"VIALE"}],"B870":
[{"end":"1929-06-14","start":"1861-03-17","name":"CASAL CERMELLI"},{"end":"1
994-12-13","start":"1929-06-15","name":"CASAL CERMELLI"},{"name":"CASAL CERM
ELLI","start":"1994-12-14","end":"9999-12-31"}],"E425":[{"end":"2005-12-31",
"start":"1861-03-17","name":"LA MADDALENA"},{"start":"2006-01-01","end":"201
6-04-27","name":"LA MADDALENA"},{"end":"9999-12-31","start":"2016-04-28","na
me":"LA MADDALENA"}],"L391":[{"start":"1861-03-17","end":"1927-08-23","name"
:"TRESILICO"}],"H678":[{"end":"2004-06-30","start":"1862-10-31","name":"SALA
 BOLOGNESE"},{"name":"SALA BOLOGNESE","end":"9999-12-31","start":"2004-07-01
"}],"G766":[{"name":"POGGIO PICENZE","start":"1861-03-17","end":"9999-12-31"
}],"A703":[{"name":"BASSANO","end":"1928-11-06","start":"1878-07-09"},{"star
t":"1928-11-07","end":"1938-11-23","name":"BASSANO DEL GRAPPA"},{"name":"BAS
SANO DEL GRAPPA","end":"9999-12-31","start":"1938-11-24"}],"D884":[{"start":
"1861-03-17","end":"1927-01-11","name":"GALLO"},{"name":"GALLO","start":"192
7-01-12","end":"1945-08-31"},{"end":"1986-03-05","start":"1945-09-01","name"
:"GALLO"},{"name":"GALLO MATESE","end":"9999-12-31","start":"1986-03-06"}],"
I550":[{"start":"1861-03-17","end":"1938-09-02","name":"SCORTICATA"},{"name"
:"TORRIANA","start":"1938-09-03","end":"1978-12-20"},{"name":"TORRIANA","end
":"1992-04-15","start":"1978-12-21"},{"start":"1992-04-16","end":"2013-12-31
","name":"TORRIANA"}],"E078":[{"start":"1872-01-01","end":"9999-12-31","name
":"GOITO"}],"H858":[{"start":"1920-10-16","end":"1923-02-13","name":"SAN GEN
ESIO"},{"start":"1923-02-14","end":"1927-01-11","name":"SAN GENESIO"},{"star
t":"1927-01-12","end":"1928-12-29","name":"SAN GENESIO"},{"name":"SAN GENESI
O ATESINO","start":"1928-12-30","end":"9999-12-31"}],"H566":[{"name":"ROSELL
O","end":"9999-12-31","start":"1861-03-17"}],"D776":[{"name":"FRASSINELLE PO
LESINE","end":"9999-12-31","start":"1878-12-23"}],"F452":[{"name":"MONTECARL
O","start":"1881-04-22","end":"9999-12-31"}],"L530":[{"start":"1861-03-17","
end":"1929-06-04","name":"VAGLIERANO"}],"D593":[{"start":"1952-01-01","end":
"9999-12-31","name":"FILIANO"}],"L715":[{"start":"1861-03-17","end":"1928-04
-14","name":"VELESO"},{"start":"1948-03-26","end":"9999-12-31","name":"VELES
O"}],"B650":[{"name":"CAPERGNANICA","end":"9999-12-31","start":"1869-05-17"}
],"H367":[{"end":"1953-08-05","start":"1861-03-17","name":"ROBASSOMERO"},{"n
ame":"ROBASSOMERO","start":"1953-08-06","end":"9999-12-31"}],"F930":[{"end":
"9999-12-31","start":"1861-03-17","name":"NONANTOLA"}],"D469":[{"name":"FAIC
CHIO","end":"9999-12-31","start":"1861-03-17"}],"I010":[{"start":"1920-10-16
","end":"1923-02-13","name":"SAN MARTINO IN CASIES"},{"end":"1927-01-11","st
art":"1923-02-14","name":"SAN MARTINO IN CASIES"},{"start":"1927-01-12","end
":"1929-05-15","name":"SAN MARTINO IN CASIES"}],"D243":[{"name":"DAIANO","en
d":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14","end":"1928-05-1
2","name":"DAIANO"},{"start":"1946-12-26","end":"2019-12-31","name":"DAIANO"
}],"H964":[{"end":"1923-02-07","start":"1921-01-05","name":"SAN LORENZO DI M
OSSA"},{"end":"1927-01-11","start":"1923-02-08","name":"SAN LORENZO DI MOSSA
"},{"end":"1928-01-01","start":"1927-01-12","name":"SAN LORENZO DI MOSSA"},{
"name":"SAN LORENZO DI MOSSA","end":"1968-11-11","start":"1955-08-19"},{"sta
rt":"1968-11-12","end":"9999-12-31","name":"SAN LORENZO ISONTINO"}],"G166":[
{"name":"OSPEDALETTO LODIGIANO","start":"1863-01-26","end":"1992-04-15"},{"n
ame":"OSPEDALETTO LODIGIANO","end":"9999-12-31","start":"1992-04-16"}],"A103
":[{"name":"AIELLO DEL FRIULI","start":"1921-01-05","end":"1923-02-07"},{"na
me":"AIELLO DEL FRIULI","start":"1923-02-08","end":"1931-07-01"},{"name":"AI
ELLO DEL FRIULI","start":"1931-07-02","end":"1968-04-05"},{"end":"9999-12-31
","start":"1968-04-06","name":"AIELLO DEL FRIULI"}],"E908":[{"end":"9999-12-
31","start":"1872-12-30","name":"MARANO EQUO"}],"A635":[{"name":"BARBIANO","
end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14","end":"1927-01
-11","name":"BARBIANO"},{"start":"1927-01-12","end":"9999-12-31","name":"BAR
BIANO"}],"E726":[{"name":"LUGAGNANO VAL D'ARDA","start":"1870-06-01","end":"
9999-12-31"}],"H551":[{"name":"RONZO","end":"1923-02-13","start":"1920-10-16
"},{"name":"RONZO","end":"1924-01-14","start":"1923-02-14"}],"F718":[{"name"
:"MORDANO","end":"9999-12-31","start":"1865-12-25"}],"A849":[{"start":"1861-
03-17","end":"1929-01-21","name":"BIASSONO"},{"start":"1929-01-22","end":"20
09-06-29","name":"BIASSONO"},{"start":"2009-06-30","end":"9999-12-31","name"
:"BIASSONO"}],"B817":[{"end":"9999-12-31","start":"1861-03-17","name":"CARPE
NEDOLO"}],"G465":[{"name":"PERRERO","start":"1895-03-04","end":"1928-04-30"}
,{"start":"1928-05-01","end":"9999-12-31","name":"PERRERO"}],"E258":[{"name"
:"GUDO VISCONTI","end":"9999-12-31","start":"1861-03-17"}],"D028":[{"name":"
CORNO GIOVINE","end":"1992-04-15","start":"1869-02-08"},{"end":"9999-12-31",
"start":"1992-04-16","name":"CORNO GIOVINE"}],"D102":[{"name":"COSTA VESCOVA
TO","start":"1863-02-23","end":"9999-12-31"}],"C267":[{"end":"1961-12-25","s
tart":"1913-01-01","name":"CASTELSANTANGELO"},{"name":"CASTELSANTANGELO SUL 
NERA","start":"1961-12-26","end":"9999-12-31"}],"E505":[{"name":"LECCE NEI M
ARSI","end":"9999-12-31","start":"1863-05-06"}],"A242":[{"start":"1861-03-17
","end":"1927-01-11","name":"ALVIANO"},{"end":"9999-12-31","start":"1927-01-
12","name":"ALVIANO"}],"B194":[{"start":"1861-03-17","end":"1992-04-15","nam
e":"BRIVIO"},{"end":"9999-12-31","start":"1992-04-16","name":"BRIVIO"}],"H01
1":[{"start":"1861-03-17","end":"1928-05-11","name":"PRAZZO"},{"end":"9999-1
2-31","start":"1928-05-12","name":"PRAZZO"}],"B498":[{"end":"1928-01-25","st
art":"1862-12-19","name":"CAMPAGNOLA CREMASCA"},{"start":"1953-08-26","end":
"9999-12-31","name":"CAMPAGNOLA CREMASCA"}],"I199":[{"end":"1929-04-10","sta
rt":"1861-03-17","name":"SANT'AGATA DI MILITELLO"},{"start":"1929-04-11","en
d":"1933-02-03","name":"SANT'AGATA DI MILITELLO"},{"end":"9999-12-31","start
":"1933-02-04","name":"SANT'AGATA DI MILITELLO"}],"A592":[{"start":"1861-03-
17","end":"1954-07-22","name":"BALESTRATE"},{"name":"BALESTRATE","end":"2008
-05-23","start":"1954-07-23"},{"name":"BALESTRATE","end":"9999-12-31","start
":"2008-05-24"}],"G151":[{"start":"1861-03-17","end":"1928-12-18","name":"OS
ASCO"},{"name":"OSASCO","end":"9999-12-31","start":"1947-08-03"}],"F833":[{"
name":"MUZZANO","start":"1861-03-17","end":"1927-01-11"},{"name":"MUZZANO","
start":"1927-01-12","end":"1992-04-15"},{"name":"MUZZANO","end":"9999-12-31"
,"start":"1992-04-16"}],"B288":[{"end":"9999-12-31","start":"1950-08-01","na
me":"BUSETO PALIZZOLO"}],"D741":[{"end":"9999-12-31","start":"1867-08-05","n
ame":"FOSSALTA DI PORTOGRUARO"}],"F352":[{"name":"MONDRAGONE","end":"1927-01
-11","start":"1861-03-17"},{"end":"1945-08-31","start":"1927-01-12","name":"
MONDRAGONE"},{"start":"1945-09-01","end":"9999-12-31","name":"MONDRAGONE"}],
"B624":[{"name":"CANSANO","start":"1904-06-06","end":"9999-12-31"}],"F970":[
{"end":"9999-12-31","start":"1861-03-17","name":"NOVOLI"}],"Z330":[{"end":"9
999-12-31","start":"0000-01-01","name":"Marocco"}],"I524":[{"end":"1923-02-0
7","start":"1921-01-05","name":"SCHERBINA"},{"name":"SCHERBINA","start":"192
3-02-08","end":"1927-01-11"},{"name":"SCHERBINA","end":"1928-08-23","start":
"1927-01-12"}],"E107":[{"name":"GORO","start":"1962-10-10","end":"9999-12-31
"}],"I926":[{"start":"1861-03-17","end":"1927-01-11","name":"SPOTORNO"},{"na
me":"SPOTORNO","end":"9999-12-31","start":"1927-01-12"}],"M189":[{"name":"ZO
PP\u00c8","start":"1866-11-19","end":"1955-06-07"},{"name":"ZOPP\u00c8 DI CA
DORE","start":"1955-06-08","end":"9999-12-31"}],"D736":[{"end":"9999-12-31",
"start":"1861-03-17","name":"FOSSA"}],"I629":[{"end":"1927-11-23","start":"1
861-03-17","name":"SERINA"},{"end":"1966-01-21","start":"1927-11-24","name":
"SERINA"},{"name":"SERINA","end":"9999-12-31","start":"1966-01-22"}],"C719":
[{"end":"9999-12-31","start":"1861-03-17","name":"CIRCELLO"}],"E038":[{"name
":"GIOIA DEL COLLE","end":"9999-12-31","start":"1863-01-19"}],"H096":[{"name
":"PUTIGNANO","start":"1861-03-17","end":"9999-12-31"}],"B762":[{"name":"CAR
EMA","end":"1925-03-19","start":"1861-03-17"},{"start":"1925-03-20","end":"1
927-01-11","name":"CAREMA"},{"name":"CAREMA","start":"1927-01-12","end":"192
9-05-02"},{"name":"CAREMA","start":"1929-05-03","end":"1941-09-27"},{"name":
"CAREMA","end":"1945-10-04","start":"1941-09-28"},{"start":"1945-10-05","end
":"9999-12-31","name":"CAREMA"}],"C086":[{"name":"CASTEL DEL RIO","end":"191
0-02-10","start":"1884-06-18"},{"end":"9999-12-31","start":"1910-02-11","nam
e":"CASTEL DEL RIO"}],"H467":[{"name":"ROCCHETTA SANT'ANTONIO","start":"1861
-03-17","end":"1939-02-17"},{"name":"ROCCHETTA SANT'ANTONIO","start":"1939-0
2-18","end":"9999-12-31"}],"L570":[{"end":"1938-06-18","start":"1861-03-17",
"name":"VALENZA"},{"start":"1938-06-19","end":"1938-07-25","name":"VALENZA"}
,{"end":"9999-12-31","start":"1938-07-26","name":"VALENZA"}],"D063":[{"start
":"1861-03-17","end":"1927-08-12","name":"CORTENEDOLO"}],"B529":[{"start":"1
928-12-09","end":"9999-12-31","name":"CAMPO DI TRENS"}],"D369":[{"name":"DRE
ZZO","end":"1928-06-13","start":"1861-03-17"},{"name":"DREZZO","start":"1956
-11-10","end":"2014-02-03"}],"":[{"end":"9999-12-31","start":"0000-01-01","n
ame":""},{"start":"0000-01-01","end":"9999-12-31","name":""},{"start":"0000-
01-01","end":"9999-12-31","name":""},{"name":"","end":"9999-12-31","start":"
0000-01-01"},{"name":"","start":"0000-01-01","end":"9999-12-31"},{"start":"0
000-01-01","end":"9999-12-31","name":""},{"name":"","start":"0000-01-01","en
d":"9999-12-31"},{"end":"9999-12-31","start":"0000-01-01","name":""},{"name"
:"","start":"0000-01-01","end":"9999-12-31"},{"name":"","start":"0000-01-01"
,"end":"9999-12-31"},{"name":"","start":"0000-01-01","end":"9999-12-31"},{"s
tart":"0000-01-01","end":"9999-12-31","name":""},{"name":"","end":"9999-12-3
1","start":"0000-01-01"},{"name":"","end":"9999-12-31","start":"0000-01-01"}
,{"start":"0000-01-01","end":"9999-12-31","name":""},{"name":"","end":"9999-
12-31","start":"0000-01-01"},{"name":"","start":"0000-01-01","end":"9999-12-
31"},{"name":"","start":"0000-01-01","end":"9999-12-31"},{"start":"0000-01-0
1","end":"9999-12-31","name":""},{"start":"0000-01-01","end":"9999-12-31","n
ame":""},{"end":"9999-12-31","start":"0000-01-01","name":""},{"name":"","end
":"9999-12-31","start":"0000-01-01"},{"start":"0000-01-01","end":"9999-12-31
","name":""},{"name":"","start":"0000-01-01","end":"9999-12-31"},{"end":"999
9-12-31","start":"0000-01-01","name":""},{"name":"","end":"9999-12-31","star
t":"0000-01-01"},{"name":"","start":"0000-01-01","end":"9999-12-31"},{"end":
"9999-12-31","start":"0000-01-01","name":""},{"end":"9999-12-31","start":"00
00-01-01","name":""},{"name":"","start":"0000-01-01","end":"9999-12-31"},{"s
tart":"0000-01-01","end":"9999-12-31","name":""},{"start":"0000-01-01","end"
:"9999-12-31","name":""}],"A062":[{"end":"9999-12-31","start":"1871-01-15","
name":"AFFILE"}],"E680":[{"name":"PORTO AZZURRO","start":"1947-07-06","end":
"9999-12-31"},{"end":"1906-04-17","start":"1873-06-23","name":"PORTO LONGONE
"},{"name":"PORTO LONGONE","end":"1947-07-05","start":"1906-04-18"}],"G365":
[{"start":"1866-11-19","end":"9999-12-31","name":"PASTRENGO"}],"C047":[{"nam
e":"CASTAGNOLE MONFERRATO","end":"1935-04-07","start":"1863-02-16"},{"end":"
9999-12-31","start":"1935-04-08","name":"CASTAGNOLE MONFERRATO"}],"L210":[{"
name":"TORBOLE CASAGLIA","start":"1872-01-19","end":"9999-12-31"}],"F873":[{
"end":"1926-02-05","start":"1861-03-17","name":"NERVI"}],"L183":[{"name":"TI
ZZANO VAL PARMA","end":"9999-12-31","start":"1863-08-10"}],"B398":[{"name":"
CALDAROLA","end":"9999-12-31","start":"1861-03-17"}],"A675":[{"name":"BARRA"
,"start":"1861-03-17","end":"1925-11-29"}],"I022":[{"start":"1867-08-26","en
d":"1927-02-15","name":"SAN MASSIMO ALL'ADIGE"}],"G253":[{"end":"9999-12-31"
,"start":"1861-03-17","name":"PALAGONIA"}],"H992":[{"end":"9999-12-31","star
t":"1863-02-06","name":"SAN MARTINO DI FINITA"}],"M204":[{"end":"1992-04-15"
,"start":"1861-03-17","name":"ZUNGRI"},{"name":"ZUNGRI","end":"9999-12-31","
start":"1992-04-16"}],"D435":[{"start":"1861-03-17","end":"1927-10-09","name
":"ESINO INFERIORE"}],"B933":[{"start":"1861-03-17","end":"1927-01-11","name
":"CASAPINTA"},{"end":"1992-04-15","start":"1927-01-12","name":"CASAPINTA"},
{"start":"1992-04-16","end":"9999-12-31","name":"CASAPINTA"}],"C982":[{"name
":"CORANA","start":"1869-01-31","end":"1932-12-27"},{"start":"1932-12-28","e
nd":"9999-12-31","name":"CORANA"}],"B409":[{"end":"1927-01-11","start":"1877
-05-18","name":"CALICE LIGURE"},{"end":"9999-12-31","start":"1927-01-12","na
me":"CALICE LIGURE"}],"A855":[{"name":"BICINICCO","end":"1968-04-05","start"
:"1866-11-19"},{"start":"1968-04-06","end":"2006-07-19","name":"BICINICCO"},
{"name":"BICINICCO","end":"9999-12-31","start":"2006-07-20"}],"L517":[{"end"
:"9999-12-31","start":"1913-01-01","name":"USSITA"}],"I108":[{"name":"SAN PI
ETRO IN AMANTEA","start":"1861-03-17","end":"1928-05-18"},{"end":"9999-12-31
","start":"1937-07-18","name":"SAN PIETRO IN AMANTEA"}],"D375":[{"name":"DRU
SACCO","end":"1927-01-11","start":"1861-03-17"},{"end":"1929-05-28","start":
"1927-01-12","name":"DRUSACCO"}],"G800":[{"name":"POLONGHERA","end":"1928-04
-02","start":"1861-03-17"},{"start":"1928-04-03","end":"1946-12-27","name":"
POLONGHERA"},{"end":"9999-12-31","start":"1946-12-28","name":"POLONGHERA"}],
"I037":[{"name":"SAN MICHELE MONDOV\u00cc","start":"1862-12-19","end":"1928-
05-02"},{"name":"SAN MICHELE MONDOV\u00cc","start":"1928-05-03","end":"1946-
12-04"},{"start":"1946-12-05","end":"9999-12-31","name":"SAN MICHELE MONDOV\
u00cc"}],"L783":[{"name":"VERRAYES","end":"1927-01-11","start":"1861-03-17"}
,{"end":"1928-06-13","start":"1927-01-12","name":"VERRAYES"},{"name":"VERRAY
ES","end":"9999-12-31","start":"1946-05-15"}],"F917":[{"start":"1863-04-10",
"end":"9999-12-31","name":"NOEPOLI"}],"F653":[{"start":"1861-03-17","end":"2
009-07-11","name":"MONTE URANO"},{"name":"MONTE URANO","end":"9999-12-31","s
tart":"2009-07-12"}],"H080":[{"name":"PROVAGLIO SOTTO","end":"1928-06-14","s
tart":"1861-03-17"}],"L057":[{"end":"1923-02-07","start":"1921-01-05","name"
:"TARVISIO"},{"name":"TARVISIO","start":"1923-02-08","end":"1928-06-14"},{"n
ame":"TARVISIO","start":"1928-06-15","end":"1968-04-05"},{"name":"TARVISIO",
"start":"1968-04-06","end":"9999-12-31"}],"F314":[{"start":"1863-07-13","end
":"1927-01-11","name":"MOMBELLO LAGO MAGGIORE"},{"end":"1928-01-27","start":
"1927-01-12","name":"MOMBELLO LAGO MAGGIORE"}],"C090":[{"start":"1861-03-17"
,"end":"9999-12-31","name":"CASTEL DI IERI"}],"I404":[{"name":"SAN VITO AL T
ORRE","start":"1921-01-05","end":"1923-02-07"},{"name":"SAN VITO AL TORRE","
start":"1923-02-08","end":"1968-04-05"},{"start":"1968-04-06","end":"9999-12
-31","name":"SAN VITO AL TORRE"}],"L316":[{"start":"1863-07-13","end":"9999-
12-31","name":"TOVO DI SANT'AGATA"}],"A951":[{"name":"BOLZANETO","start":"18
78-06-21","end":"1926-02-05"}],"Z241":[{"end":"9999-12-31","start":"0000-01-
01","name":"Thailandia"}],"A564":[{"start":"1869-12-10","end":"1928-12-15","
name":"BAGNO A RIPOLI"},{"start":"1928-12-16","end":"9999-12-31","name":"BAG
NO A RIPOLI"}],"D141":[{"end":"1938-02-18","start":"1928-07-29","name":"CRED
ERA RUBBIANO"},{"start":"1938-02-19","end":"9999-12-31","name":"CREDERA RUBB
IANO"}],"B830":[{"name":"CARPINONE","end":"1970-03-02","start":"1861-03-17"}
,{"name":"CARPINONE","start":"1970-03-03","end":"9999-12-31"}],"A201":[{"end
":"1928-10-07","start":"1910-07-26","name":"AL\u00cc MARINA"},{"name":"AL\u0
0cc MARINA","end":"1954-08-14","start":"1946-04-24"},{"start":"1954-08-15","
end":"9999-12-31","name":"AL\u00cc TERME"}],"G379":[{"name":"PAU","end":"192
7-07-02","start":"1861-03-17"},{"name":"PAU","end":"1974-08-19","start":"194
6-02-08"},{"start":"1974-08-20","end":"9999-12-31","name":"PAU"}],"G073":[{"
end":"1928-11-30","start":"1862-11-28","name":"ONO DEGNO"}],"H638":[{"name":
"RUMIANCA","start":"1861-03-17","end":"1928-07-29"}],"E696":[{"name":"LOSE",
"end":"1923-02-07","start":"1921-01-05"},{"start":"1923-02-08","end":"1927-0
1-11","name":"LOSE"},{"start":"1927-01-12","end":"1928-04-28","name":"LOSE"}
],"E707":[{"start":"1861-03-17","end":"1927-01-11","name":"LOZZA"},{"start":
"1927-01-12","end":"9999-12-31","name":"LOZZA"}],"E999":[{"name":"MARZANO","
end":"9999-12-31","start":"1872-11-01"}],"C825":[{"start":"1920-10-16","end"
:"1923-02-13","name":"COGOLO"},{"name":"COGOLO","end":"1928-08-11","start":"
1923-02-14"}],"D136":[{"start":"1866-11-19","end":"9999-12-31","name":"CREAZ
ZO"}],"Z236":[{"name":"Pakistan","end":"9999-12-31","start":"0000-01-01"}],"
E325":[{"start":"1861-03-17","end":"1927-12-26","name":"IRMA"},{"start":"195
6-01-06","end":"9999-12-31","name":"IRMA"}],"C119":[{"name":"CASTELGOMBERTO"
,"start":"1866-11-19","end":"9999-12-31"}],"L491":[{"end":"1923-02-07","star
t":"1921-01-05","name":"UMAGO"},{"start":"1923-02-08","end":"1977-04-03","na
me":"UMAGO"}],"I081":[{"start":"1861-03-17","end":"1926-02-05","name":"SAN P
IER D'ARENA"}],"B162":[{"end":"9999-12-31","start":"1884-08-23","name":"BRES
SO"}],"A966":[{"name":"BONDIONE","end":"1927-06-01","start":"1861-03-17"},{"
name":"BONDIONE","end":"1927-10-10","start":"1927-06-02"}],"A669":[{"start":
"1861-03-17","end":"2009-07-26","name":"BARLETTA"},{"start":"2009-07-27","en
d":"9999-12-31","name":"BARLETTA"}],"G903":[{"start":"1862-09-08","end":"192
7-01-11","name":"PORTICO DI CASERTA"},{"end":"1928-12-14","start":"1927-01-1
2","name":"PORTICO DI CASERTA"},{"name":"PORTICO DI CASERTA","start":"1946-0
4-26","end":"9999-12-31"}],"C921":[{"start":"1921-01-05","end":"1923-02-07",
"name":"COMEN"},{"name":"COMEN","start":"1923-02-08","end":"1923-04-26"},{"n
ame":"COMENO","end":"1927-01-11","start":"1923-04-27"},{"start":"1927-01-12"
,"end":"1928-08-23","name":"COMENO"},{"name":"COMENO","end":"1947-09-15","st
art":"1928-08-24"}],"L277":[{"name":"TORRE PELLICE","end":"9999-12-31","star
t":"1878-02-18"}],"M357":[{"name":"MADRUZZO","start":"2016-01-01","end":"999
9-12-31"}],"M016":[{"name":"VILLASALTO","end":"1974-08-19","start":"1861-03-
17"},{"name":"VILLASALTO","end":"2016-04-27","start":"1974-08-20"},{"end":"9
999-12-31","start":"2016-04-28","name":"VILLASALTO"}],"G439":[{"name":"PENTO
NE","end":"9999-12-31","start":"1861-03-17"}],"E421":[{"name":"LAIVES","end"
:"1923-02-13","start":"1920-10-16"},{"name":"LAIVES","end":"1927-01-11","sta
rt":"1923-02-14"},{"end":"9999-12-31","start":"1927-01-12","name":"LAIVES"}]
,"L395":[{"name":"TREVANO","end":"1938-01-17","start":"1861-03-17"}],"F529":
[{"name":"MONTEGROTTO TERME","end":"9999-12-31","start":"1934-09-27"},{"star
t":"1919-04-07","end":"1934-09-26","name":"SAN PIETRO MONTAGNON"}],"C563":[{
"end":"1927-11-08","start":"1863-02-23","name":"CESANA BRIANZA"},{"start":"1
955-08-19","end":"1992-04-15","name":"CESANA BRIANZA"},{"name":"CESANA BRIAN
ZA","end":"2004-06-07","start":"1992-04-16"},{"end":"9999-12-31","start":"20
04-06-08","name":"CESANA BRIANZA"}],"G224":[{"name":"PADOVA","start":"1866-1
1-19","end":"9999-12-31"}],"F762":[{"start":"1861-03-17","end":"1928-03-12",
"name":"MOSCHIANO"},{"name":"MOSCHIANO","start":"1957-12-08","end":"9999-12-
31"}],"D586":[{"end":"9999-12-31","start":"1871-01-15","name":"FILACCIANO"}]
,"G688":[{"start":"1867-08-26","end":"9999-12-31","name":"PIOMBINO DESE"}],"
L111":[{"name":"TERMENO","end":"1923-02-13","start":"1920-10-16"},{"end":"19
48-03-13","start":"1923-02-14","name":"TERMENO"},{"start":"1948-03-14","end"
:"1971-09-21","name":"TERMENO"},{"name":"TERMENO SULLA STRADA DEL VINO","end
":"1978-01-31","start":"1971-09-22"},{"start":"1978-02-01","end":"1990-11-27
","name":"TERMENO SULLA STRADA DEL VINO"},{"end":"9999-12-31","start":"1990-
11-28","name":"TERMENO SULLA STRADA DEL VINO"}],"M167":[{"start":"1861-03-17
","end":"9999-12-31","name":"ZERBOL\u00d2"}],"D759":[{"end":"9999-12-31","st
art":"1873-04-10","name":"FRANCAVILLA BISIO"}],"D984":[{"name":"GERMAGNO","s
tart":"1861-03-17","end":"1928-01-26"},{"name":"GERMAGNO","start":"1955-06-0
9","end":"1992-05-22"},{"end":"9999-12-31","start":"1992-05-23","name":"GERM
AGNO"}],"H489":[{"name":"ROGHUDI","start":"1861-03-17","end":"1977-09-16"},{
"name":"ROGHUDI","end":"9999-12-31","start":"1977-09-17"}],"F288":[{"name":"
MOLINELLA","end":"9999-12-31","start":"1861-03-17"}],"D387":[{"name":"DUSINO
","end":"1928-02-21","start":"1861-03-17"}],"L929":[{"name":"VILLA D'ADDA","
start":"1861-03-17","end":"9999-12-31"}],"B352":[{"start":"1861-03-17","end"
:"1998-09-03","name":"CAGLI"},{"start":"1998-09-04","end":"9999-12-31","name
":"CAGLI"}],"F624":[{"end":"1871-05-31","start":"1863-03-30","name":"MONTESA
NO AL PIANO"}],"B970":[{"end":"1880-03-15","start":"1861-03-17","name":"CASI
RAGO"}],"L626":[{"end":"1976-08-18","start":"1861-03-17","name":"VALLIO"},{"
name":"VALLIO TERME","start":"1976-08-19","end":"9999-12-31"}],"G755":[{"nam
e":"POGGIO BERNI","end":"1992-04-15","start":"1861-03-17"},{"name":"POGGIO B
ERNI","end":"2013-12-31","start":"1992-04-16"}],"D471":[{"start":"1964-09-01
","end":"9999-12-31","name":"FALCIANO DEL MASSICO"}],"M095":[{"end":"9999-12
-31","start":"1872-12-10","name":"VIVARO ROMANO"}],"D346":[{"name":"DORIO","
start":"1861-03-17","end":"1928-01-26"},{"end":"1992-04-15","start":"1950-04
-05","name":"DORIO"},{"start":"1992-04-16","end":"9999-12-31","name":"DORIO"
}],"D297":[{"name":"DIANO MARINA","start":"1871-05-01","end":"1924-02-03"},{
"name":"DIANO MARINA","start":"1924-02-04","end":"1925-08-21"},{"name":"DIAN
O MARINA","end":"9999-12-31","start":"1925-08-22"}],"F398":[{"end":"1917-06-
10","start":"1876-03-20","name":"MONTAIONE"},{"name":"MONTAIONE","start":"19
17-06-11","end":"9999-12-31"}],"E615":[{"name":"LISIO","start":"1861-03-17",
"end":"1929-04-10"},{"start":"1947-01-10","end":"9999-12-31","name":"LISIO"}
],"G030":[{"end":"1928-08-14","start":"1861-03-17","name":"OLGINATE"},{"end"
:"1992-04-15","start":"1928-08-15","name":"OLGINATE"},{"start":"1992-04-16",
"end":"9999-12-31","name":"OLGINATE"}],"I807":[{"end":"1928-12-13","start":"
1861-03-17","name":"SOLOMIAC"}],"E482":[{"name":"LAURENZANA","end":"9999-12-
31","start":"1861-03-17"}],"B873":[{"name":"CASALDUNI","start":"1892-03-28",
"end":"9999-12-31"}],"L285":[{"end":"9999-12-31","start":"1872-11-01","name"
:"TORREVECCHIA PIA"}],"D547":[{"name":"FERRANDINA","end":"1927-01-11","start
":"1861-03-17"},{"name":"FERRANDINA","end":"9999-12-31","start":"1927-01-12"
}],"A700":[{"start":"1923-07-13","end":"1968-04-05","name":"BASILIANO"},{"st
art":"1968-04-06","end":"2012-07-18","name":"BASILIANO"},{"start":"2012-07-1
9","end":"9999-12-31","name":"BASILIANO"},{"end":"1923-07-12","start":"1866-
11-19","name":"PASIAN SCHIAVONESCO"}],"D331":[{"name":"DOMICELLA","end":"999
9-12-31","start":"1861-03-17"}],"I553":[{"end":"1911-05-13","start":"1861-03
-17","name":"SCURCOLA"},{"name":"SCURCOLA MARSICANA","start":"1911-05-14","e
nd":"9999-12-31"}],"G155":[{"start":"1861-03-17","end":"1927-01-11","name":"
OSIGLIA"},{"name":"OSIGLIA","start":"1927-01-12","end":"9999-12-31"}],"B314"
:[{"name":"CABRAS","start":"1861-03-17","end":"1928-02-08"},{"name":"CABRAS"
,"end":"1974-08-19","start":"1928-02-09"},{"name":"CABRAS","start":"1974-08-
20","end":"9999-12-31"}],"A646":[{"name":"BARDI","start":"1861-03-17","end":
"1923-10-30"},{"name":"BARDI","start":"1923-10-31","end":"1927-01-26"},{"nam
e":"BARDI","end":"1927-05-26","start":"1927-01-27"},{"end":"9999-12-31","sta
rt":"1927-05-27","name":"BARDI"}],"L533":[{"start":"1861-03-17","end":"1923-
11-23","name":"VAGLI SOTTO"},{"end":"1929-06-01","start":"1923-11-24","name"
:"VAGLI SOTTO"},{"start":"1929-06-02","end":"9999-12-31","name":"VAGLI SOTTO
"}],"A949":[{"start":"1871-01-15","end":"1927-01-11","name":"BOLSENA"},{"sta
rt":"1927-01-12","end":"9999-12-31","name":"BOLSENA"}],"B917":[{"name":"CASA
LVECCHIO DI PUGLIA","start":"1862-11-10","end":"9999-12-31"}],"B653":[{"name
":"CAPIAGO INTIMIANO","end":"9999-12-31","start":"1929-04-12"},{"start":"192
8-12-15","end":"1929-04-11","name":"SALVADERA"}],"Z606":[{"end":"9999-12-31"
,"start":"0000-01-01","name":"Guyana"}],"D590":[{"end":"9999-12-31","start":
"1865-12-25","name":"FILATTIERA"}],"H015":[{"name":"PRECI","end":"9999-12-31
","start":"1861-03-17"}],"E501":[{"end":"1923-02-13","start":"1920-10-16","n
ame":"LAZFONS"},{"start":"1923-02-14","end":"1927-01-11","name":"LAZFONS"},{
"name":"LAZFONS","start":"1927-01-12","end":"1929-06-04"}],"F933":[{"name":"
NORAGUGUME","start":"1861-03-17","end":"1927-01-11"},{"end":"1928-03-12","st
art":"1927-01-12","name":"NORAGUGUME"},{"start":"1946-04-27","end":"9999-12-
31","name":"NORAGUGUME"}],"H208":[{"name":"REBBIO","start":"1861-03-17","end
":"1937-05-17"}],"F409":[{"name":"MONTALDO SCARAMPI","start":"1861-03-17","e
nd":"1935-04-07"},{"name":"MONTALDO SCARAMPI","end":"9999-12-31","start":"19
35-04-08"}],"I013":[{"name":"SAN MARTINO QUISCA","start":"1921-01-05","end":
"1923-02-07"},{"name":"SAN MARTINO QUISCA","start":"1923-02-08","end":"1927-
01-11"},{"start":"1927-01-12","end":"1927-12-28","name":"SAN MARTINO QUISCA"
},{"start":"1927-12-29","end":"1947-09-15","name":"SAN MARTINO QUISCA"}],"E2
64":[{"name":"GUIGLIA","start":"1880-05-21","end":"9999-12-31"}],"I319":[{"e
nd":"9999-12-31","start":"1866-11-19","name":"SANT'ELENA"}],"H555":[{"name":
"RORETO","end":"1939-04-22","start":"1937-06-16"},{"name":"RORETO CHISONE","
start":"1939-04-23","end":"1975-03-11"},{"name":"ROURE","end":"1937-06-15","
start":"1861-03-17"},{"end":"9999-12-31","start":"1975-03-12","name":"ROURE"
}],"Z140":[{"end":"9999-12-31","start":"0000-01-01","name":"Moldova"}],"A631
":[{"name":"BARBATA","end":"9999-12-31","start":"1861-03-17"}],"A100":[{"nam
e":"AIELLI","start":"1861-03-17","end":"9999-12-31"}],"G461":[{"name":"PERNU
MIA","end":"1993-07-13","start":"1866-11-19"},{"name":"PERNUMIA","start":"19
93-07-14","end":"9999-12-31"}],"H958":[{"name":"SAN LORENZO IN CAMPO","end":
"1928-11-09","start":"1869-06-11"},{"name":"SAN LORENZO IN CAMPO","end":"193
7-07-10","start":"1928-11-10"},{"end":"1998-09-03","start":"1937-07-11","nam
e":"SAN LORENZO IN CAMPO"},{"name":"SAN LORENZO IN CAMPO","start":"1998-09-0
4","end":"9999-12-31"}],"C308":[{"end":"1927-01-11","start":"1863-08-10","na
me":"CASTIGLIONE A CASAURIA"},{"end":"9999-12-31","start":"1927-01-12","name
":"CASTIGLIONE A CASAURIA"}],"F162":[{"name":"META","end":"1927-05-17","star
t":"1861-03-17"},{"end":"9999-12-31","start":"1946-04-24","name":"META"}],"L
711":[{"end":"1927-01-11","start":"1861-03-17","name":"VEGLIE"},{"name":"VEG
LIE","end":"1927-04-13","start":"1927-01-12"},{"start":"1927-04-14","end":"9
999-12-31","name":"VEGLIE"}],"Z259":[{"name":"Uzbekistan","end":"9999-12-31"
,"start":"0000-01-01"}],"H864":[{"name":"SAN GERVASIO D'ADDA","start":"1863-
07-13","end":"1928-02-22"}],"D159":[{"start":"1879-06-27","end":"1992-04-15"
,"name":"CRESPIATICA"},{"start":"1992-04-16","end":"9999-12-31","name":"CRES
PIATICA"}],"E808":[{"name":"MAGLIANO ALPI","start":"1862-12-19","end":"9999-
12-31"}],"F830":[{"name":"MUSSOMELI","start":"1861-03-17","end":"9999-12-31"
}],"G456":[{"name":"PERLEDO","start":"1861-03-17","end":"1928-08-15"},{"name
":"PERLEDO","end":"1992-04-15","start":"1953-06-17"},{"end":"9999-12-31","st
art":"1992-04-16","name":"PERLEDO"}],"A687":[{"start":"1861-03-17","end":"19
29-01-21","name":"BARZIO"},{"start":"1929-01-22","end":"1992-04-15","name":"
BARZIO"},{"name":"BARZIO","end":"9999-12-31","start":"1992-04-16"}],"Z333":[
{"end":"9999-12-31","start":"0000-01-01","name":"Mozambico"}],"F973":[{"name
":"NUCHIS","start":"1861-03-17","end":"1939-01-26"}],"B414":[{"end":"1927-11
-01","start":"1861-03-17","name":"CALINO"}],"G749":[{"end":"1927-01-11","sta
rt":"1871-01-15","name":"POFI"},{"start":"1927-01-12","end":"9999-12-31","na
me":"POFI"}],"A671":[{"end":"9999-12-31","start":"1861-03-17","name":"BAROLO
"}],"E044":[{"start":"1863-04-10","end":"1948-07-21","name":"GIOIOSA IONICA"
},{"name":"GIOIOSA IONICA","end":"9999-12-31","start":"1948-07-22"}],"G361":
[{"end":"1928-09-24","start":"1867-08-19","name":"PASSIRANO"},{"name":"PASSI
RANO","start":"1928-09-25","end":"1929-05-27"},{"name":"PASSIRANO","start":"
1929-05-28","end":"1995-05-04"},{"start":"1995-05-05","end":"9999-12-31","na
me":"PASSIRANO"}],"L607":[{"name":"VALLELONGA","start":"1861-03-17","end":"1
992-04-15"},{"end":"9999-12-31","start":"1992-04-16","name":"VALLELONGA"}],"
L796":[{"name":"VERTOVINO","end":"1923-02-07","start":"1921-01-05"},{"end":"
1927-01-11","start":"1923-02-08","name":"VERTOVINO"},{"start":"1927-01-12","
end":"1928-01-21","name":"VERTOVINO"}],"F794":[{"end":"1927-01-11","start":"
1861-03-17","name":"MUCENO"},{"name":"MUCENO","end":"1928-08-15","start":"19
27-01-12"}],"B118":[{"name":"BRANCALEONE","start":"1861-03-17","end":"9999-1
2-31"}],"F309":[{"name":"MOMBARCARO","end":"1928-06-27","start":"1861-03-17"
},{"end":"9999-12-31","start":"1946-12-28","name":"MOMBARCARO"}],"F003":[{"n
ame":"MASATE","start":"1861-03-17","end":"9999-12-31"}],"I419":[{"name":"SAO
NE","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14","end":"19
28-02-23","name":"SAONE"}],"L573":[{"name":"VALFABBRICA","start":"1861-03-17
","end":"1929-02-20"},{"name":"VALFABBRICA","end":"9999-12-31","start":"1929
-02-21"}],"G557":[{"name":"PIANELLO VAL TIDONE","end":"9999-12-31","start":"
1862-11-28"}],"C408":[{"name":"CAZZAGO SAN MARTINO","end":"1927-11-01","star
t":"1863-02-23"},{"name":"CAZZAGO SAN MARTINO","start":"1927-11-02","end":"2
000-04-13"},{"end":"9999-12-31","start":"2000-04-14","name":"CAZZAGO SAN MAR
TINO"}],"E683":[{"name":"LORANZ\u00c8","end":"1927-01-11","start":"1861-03-1
7"},{"end":"1929-04-12","start":"1927-01-12","name":"LORANZ\u00c8"},{"name":
"LORANZ\u00c8","start":"1947-10-11","end":"9999-12-31"}],"L213":[{"name":"TO
RCHIAROLO","end":"1927-01-11","start":"1861-03-17"},{"name":"TORCHIAROLO","s
tart":"1927-01-12","end":"9999-12-31"}],"F870":[{"start":"1861-03-17","end":
"9999-12-31","name":"NERETO"}],"C104":[{"start":"1861-03-17","end":"1927-01-
11","name":"CASTELFORTE"},{"start":"1927-01-12","end":"1928-10-27","name":"C
ASTELFORTE"},{"name":"CASTELFORTE","end":"1934-10-25","start":"1928-10-28"},
{"start":"1934-10-26","end":"1934-12-17","name":"CASTELFORTE"},{"end":"1945-
04-08","start":"1934-12-18","name":"CASTELFORTE"},{"name":"CASTELFORTE","end
":"1947-05-22","start":"1945-04-09"},{"name":"CASTELFORTE","end":"1984-03-05
","start":"1947-05-23"},{"start":"1984-03-06","end":"9999-12-31","name":"CAS
TELFORTE"}],"G356":[{"end":"1928-06-15","start":"1867-09-02","name":"RIVOLTO
"}],"G017":[{"name":"OLCIO","end":"1928-01-26","start":"1861-03-17"}],"L180"
:[{"name":"TISSI","end":"9999-12-31","start":"1861-03-17"}],"D745":[{"start"
:"1863-04-13","end":"9999-12-31","name":"FOSSATO DI VICO"}],"I826":[{"name":
"SONA","end":"1975-01-28","start":"1866-11-19"},{"name":"SONA","end":"9999-1
2-31","start":"1975-01-29"}],"H157":[{"end":"9999-12-31","start":"1861-03-17
","name":"RADICONDOLI"}],"G250":[{"end":"9999-12-31","start":"1958-01-25","n
ame":"PALAGANO"}],"E899":[{"name":"MANZANO","end":"1968-04-05","start":"1866
-11-19"},{"end":"9999-12-31","start":"1968-04-06","name":"MANZANO"}],"A018":
[{"name":"ACCIANO","start":"1861-03-17","end":"9999-12-31"}],"C925":[{"name"
:"COMEZZANO-CIZZAGO","start":"1927-12-04","end":"9999-12-31"}],"E292":[{"sta
rt":"1958-05-28","end":"9999-12-31","name":"INARZO"}],"I085":[{"end":"2013-1
2-31","start":"1861-03-17","name":"SAN PIERO A SIEVE"}],"E321":[{"start":"18
61-03-17","end":"1940-05-29","name":"IONADI"},{"start":"1940-05-30","end":"1
992-04-15","name":"IONADI"},{"name":"IONADI","start":"1992-04-16","end":"999
9-12-31"}],"A558":[{"name":"BAGNI DELLA PORRETTA","end":"1926-01-21","start"
:"1882-12-08"},{"end":"1931-09-26","start":"1926-01-22","name":"BAGNI DELLA 
PORRETTA"},{"name":"PORRETTA TERME","start":"1931-09-27","end":"2015-12-31"}
],"A866":[{"end":"1980-06-14","start":"1866-11-19","name":"BIGARELLO"},{"nam
e":"BIGARELLO","end":"2018-12-31","start":"1980-06-15"}],"B452":[{"name":"CA
LVIZZANO","end":"9999-12-31","start":"1861-03-17"}],"G803":[{"name":"POLVERI
GI","start":"1861-03-17","end":"9999-12-31"}],"H389":[{"name":"ROCCACASALE",
"start":"1861-03-17","end":"9999-12-31"}],"C821":[{"end":"1927-01-11","start
":"1861-03-17","name":"COGNE"},{"name":"COGNE","end":"9999-12-31","start":"1
927-01-12"}],"C704":[{"name":"CINGOLI","end":"9999-12-31","start":"1861-03-1
7"}],"H083":[{"start":"1861-03-17","end":"1970-03-02","name":"PROVVIDENTI"},
{"end":"9999-12-31","start":"1970-03-03","name":"PROVVIDENTI"}],"F650":[{"na
me":"MONTESTRUTTO","start":"1861-03-17","end":"1927-01-11"},{"end":"1929-05-
04","start":"1927-01-12","name":"MONTESTRUTTO"}],"L780":[{"start":"1871-01-1
5","end":"1927-01-11","name":"VEROLI"},{"name":"VEROLI","end":"9999-12-31","
start":"1927-01-12"}],"D487":[{"end":"1968-04-05","start":"1866-11-19","name
":"FANNA"},{"name":"FANNA","end":"9999-12-31","start":"1968-04-06"}],"C093":
[{"name":"CASTEL DI LAMA","end":"9999-12-31","start":"1862-11-24"}],"D145":[
{"end":"1929-02-13","start":"1861-03-17","name":"CREMENO"},{"end":"1948-03-2
0","start":"1929-02-14","name":"CREMENO"},{"start":"1948-03-21","end":"1992-
04-15","name":"CREMENO"},{"end":"9999-12-31","start":"1992-04-16","name":"CR
EMENO"}],"H757":[{"name":"SANARICA","start":"1861-03-17","end":"9999-12-31"}
],"E542":[{"start":"1861-03-17","end":"1923-12-21","name":"LERICI"},{"start"
:"1923-12-22","end":"1939-12-27","name":"LERICI"},{"name":"LERICI","end":"19
60-05-19","start":"1939-12-28"},{"start":"1960-05-20","end":"9999-12-31","na
me":"LERICI"}],"A205":[{"name":"ALLAIN","start":"1861-03-17","end":"1927-01-
11"},{"name":"ALLAIN","end":"1939-10-20","start":"1927-01-12"},{"end":"1977-
01-01","start":"1946-01-30","name":"ALLAIN"},{"start":"1977-01-02","end":"99
99-12-31","name":"ALLEIN"},{"end":"1946-01-29","start":"1939-10-21","name":"
ALLENO"}],"H299":[{"start":"1920-10-16","end":"1923-02-13","name":"RIO DI PU
STERIA"},{"start":"1923-02-14","end":"1927-01-11","name":"RIO DI PUSTERIA"},
{"end":"1929-05-29","start":"1927-01-12","name":"RIO DI PUSTERIA"},{"name":"
RIO DI PUSTERIA","end":"1956-01-03","start":"1929-05-30"},{"start":"1956-01-
04","end":"1982-02-16","name":"RIO DI PUSTERIA"},{"name":"RIO DI PUSTERIA","
start":"1982-02-17","end":"9999-12-31"}],"D446":[{"name":"FABIASCO","start":
"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"1928-10-07","n
ame":"FABIASCO"}],"B833":[{"start":"1866-11-19","end":"1995-04-07","name":"C
ARRARA SAN GIORGIO"}],"F498":[{"name":"MONTEFERRANTE","end":"1928-09-24","st
art":"1861-03-17"},{"name":"MONTEFERRANTE","start":"1933-05-19","end":"9999-
12-31"}],"C882":[{"name":"COLLINAS","end":"1928-02-04","start":"1863-12-28"}
,{"name":"COLLINAS","end":"1946-03-08","start":"1928-02-05"},{"end":"1974-08
-19","start":"1946-03-09","name":"COLLINAS"},{"end":"2005-12-31","start":"19
74-08-20","name":"COLLINAS"},{"start":"2006-01-01","end":"2016-04-27","name"
:"COLLINAS"},{"name":"COLLINAS","end":"9999-12-31","start":"2016-04-28"}],"A
955":[{"start":"1871-01-15","end":"1927-01-11","name":"BOMARZO"},{"start":"1
927-01-12","end":"1942-02-20","name":"BOMARZO"},{"end":"9999-12-31","start":
"1942-02-21","name":"BOMARZO"}],"G070":[{"end":"1927-01-11","start":"1861-03
-17","name":"ONIFAI"},{"start":"1927-01-12","end":"1928-02-01","name":"ONIFA
I"},{"name":"ONIFAI","end":"9999-12-31","start":"1946-03-08"}],"D371":[{"nam
e":"DRO","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14","end
":"1928-04-03","name":"DRO"},{"end":"1947-09-29","start":"1928-04-04","name"
:"DRO"},{"name":"DRO","start":"1947-09-30","end":"9999-12-31"}],"C289":[{"na
me":"CASTEL VISCARDO","end":"1927-01-11","start":"1879-08-29"},{"start":"192
7-01-12","end":"1937-08-21","name":"CASTEL VISCARDO"},{"name":"CASTEL VISCAR
DO","start":"1937-08-22","end":"9999-12-31"}],"H644":[{"name":"RUTINO","end"
:"9999-12-31","start":"1861-03-17"}],"H892":[{"end":"1936-10-27","start":"18
61-03-17","name":"SAN GIORGIO A CREMANO"},{"start":"1936-10-28","end":"9999-
12-31","name":"SAN GIORGIO A CREMANO"}],"G149":[{"end":"1927-12-23","start":
"1861-03-17","name":"ORZINUOVI"},{"end":"9999-12-31","start":"1927-12-24","n
ame":"ORZINUOVI"}],"D431":[{"start":"1861-03-17","end":"1927-01-11","name":"
ESCOLCA"},{"name":"ESCOLCA","start":"1927-01-12","end":"2005-12-31"},{"name"
:"ESCOLCA","end":"2016-04-27","start":"2006-01-01"},{"end":"9999-12-31","sta
rt":"2016-04-28","name":"ESCOLCA"}],"C528":[{"start":"1863-02-16","end":"192
8-08-07","name":"CERRETO D'ASTI"},{"name":"CERRETO D'ASTI","end":"1994-07-27
","start":"1947-10-11"},{"start":"1994-07-28","end":"9999-12-31","name":"CER
RETO D'ASTI"}],"L196":[{"name":"TOLMINO","end":"1923-02-07","start":"1921-01
-05"},{"name":"TOLMINO","end":"1927-01-11","start":"1923-02-08"},{"end":"192
8-05-11","start":"1927-01-12","name":"TOLMINO"},{"end":"1947-09-15","start":
"1928-05-12","name":"TOLMINO"}],"H549":[{"end":"1927-01-11","start":"1861-03
-17","name":"RONSECCO"},{"name":"RONSECCO","end":"9999-12-31","start":"1927-
01-12"}],"A851":[{"name":"BIBBIENA","end":"9999-12-31","start":"1861-03-17"}
],"F194":[{"end":"1928-10-06","start":"1861-03-17","name":"MIGGIANO"},{"star
t":"1928-10-07","end":"1947-05-11","name":"MIGGIANO"},{"name":"MIGGIANO","en
d":"9999-12-31","start":"1947-05-12"}],"G900":[{"name":"PORTE","end":"9999-1
2-31","start":"1861-03-17"}],"E382":[{"start":"1871-01-15","end":"1936-09-24
","name":"JENNE"},{"end":"9999-12-31","start":"1936-09-25","name":"JENNE"}],
"B718":[{"start":"1864-06-23","end":"1928-07-27","name":"CARAFFA DEL BIANCO"
},{"name":"CARAFFA DEL BIANCO","end":"9999-12-31","start":"1946-03-09"}],"F8
17":[{"name":"MURO LUCANO","end":"9999-12-31","start":"1863-05-06"}],"F719":
[{"start":"1861-03-17","end":"1869-03-04","name":"MOREGNANO"}],"A848":[{"end
":"9999-12-31","start":"1861-03-17","name":"BIANZONE"}],"E259":[{"name":"GUG
LIONESI","end":"1924-02-06","start":"1861-03-17"},{"name":"GUGLIONESI","star
t":"1924-02-07","end":"1970-03-02"},{"start":"1970-03-03","end":"9999-12-31"
,"name":"GUGLIONESI"}],"D323":[{"end":"1974-08-19","start":"1905-07-10","nam
e":"DOLIANOVA"},{"end":"2016-04-27","start":"1974-08-20","name":"DOLIANOVA"}
,{"end":"9999-12-31","start":"2016-04-28","name":"DOLIANOVA"}],"A381":[{"nam
e":"ARDENA","end":"1927-01-11","start":"1861-03-17"},{"name":"ARDENA","end":
"1928-09-05","start":"1927-01-12"}],"D029":[{"name":"CORNOVECCHIO","start":"
1866-08-16","end":"1992-04-15"},{"name":"CORNOVECCHIO","start":"1992-04-16",
"end":"9999-12-31"}],"B563":[{"name":"CAMPOSAMPIERO","start":"1866-11-19","e
nd":"9999-12-31"}],"G890":[{"end":"9999-12-31","start":"1861-03-17","name":"
PORNASSIO"}],"I663":[{"name":"SERRAVALLE SESIA","start":"1863-08-10","end":"
1927-01-11"},{"name":"SERRAVALLE SESIA","start":"1927-01-12","end":"1927-10-
08"},{"name":"SERRAVALLE SESIA","start":"1927-10-09","end":"9999-12-31"}],"E
852":[{"name":"MALEO","start":"1861-03-17","end":"1938-02-18"},{"name":"MALE
O","start":"1938-02-19","end":"1992-04-15"},{"end":"9999-12-31","start":"199
2-04-16","name":"MALEO"}],"H072":[{"start":"1861-03-17","end":"1907-02-10","
name":"PROCIDA"},{"name":"PROCIDA","end":"9999-12-31","start":"1907-02-11"}]
,"I494":[{"name":"SCANDALE","end":"1992-04-15","start":"1861-03-17"},{"name"
:"SCANDALE","end":"9999-12-31","start":"1992-04-16"}],"L084":[{"name":"TEGLI
O","start":"1861-03-17","end":"1924-01-03"},{"end":"1959-09-08","start":"192
4-01-04","name":"TEGLIO"},{"start":"1959-09-09","end":"9999-12-31","name":"T
EGLIO"}],"F086":[{"end":"9999-12-31","start":"1868-07-01","name":"MEDOLE"}],
"H902":[{"name":"SAN GIOVANNI","end":"1927-01-11","start":"1923-04-27"},{"na
me":"SAN GIOVANNI","end":"1929-02-21","start":"1927-01-12"},{"start":"1920-1
0-16","end":"1923-02-13","name":"SAN GIOVANNI IN VALLE AURINA"},{"end":"1923
-04-26","start":"1923-02-14","name":"SAN GIOVANNI IN VALLE AURINA"}],"C970":
[{"start":"1861-03-17","end":"1928-05-12","name":"CONTRA"}],"C352":[{"end":"
9999-12-31","start":"1861-03-17","name":"CATANZARO"}],"H665":[{"end":"1923-0
2-07","start":"1921-01-05","name":"SAGRADO"},{"name":"SAGRADO","start":"1923
-02-08","end":"1927-01-11"},{"name":"SAGRADO","end":"1947-09-15","start":"19
27-01-12"},{"end":"1951-02-14","start":"1947-09-16","name":"SAGRADO"},{"end"
:"9999-12-31","start":"1951-02-15","name":"SAGRADO"}],"I198":[{"start":"1861
-03-17","end":"1928-07-27","name":"SANT'AGATA DEL BIANCO"},{"name":"SANT'AGA
TA DEL BIANCO","start":"1946-03-09","end":"9999-12-31"}],"B499":[{"name":"CA
MPAGNOLA EMILIA","end":"9999-12-31","start":"1862-10-20"}],"M294":[{"end":"9
999-12-31","start":"1990-01-30","name":"BELLIZZI"}],"E373":[{"end":"9999-12-
31","start":"1866-11-19","name":"ISTRANA"}],"Z501":[{"end":"9999-12-31","sta
rt":"0000-01-01","name":"Aruba"}],"A337":[{"end":"1959-09-08","start":"1924-
01-04","name":"APRICA"},{"end":"9999-12-31","start":"1959-09-09","name":"APR
ICA"}],"E079":[{"name":"GOLASECCA","start":"1861-03-17","end":"1927-01-11"},
{"name":"GOLASECCA","start":"1927-01-12","end":"9999-12-31"}],"L401":[{"end"
:"9999-12-31","start":"1884-06-14","name":"TREVIGNANO ROMANO"}],"D164":[{"na
me":"CRETO","start":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-14","
end":"1928-03-06","name":"CRETO"}],"H859":[{"name":"SAN GENESIO","end":"1929
-03-07","start":"1871-06-01"},{"name":"SAN GENESIO ED UNITI","end":"1939-07-
12","start":"1929-03-08"},{"start":"1939-07-13","end":"2005-02-18","name":"S
AN GENESIO ED UNITI"},{"name":"SAN GENESIO ED UNITI","start":"2005-02-19","e
nd":"9999-12-31"}],"A541":[{"name":"BADIA TEDALDA","end":"9999-12-31","start
":"1861-03-17"}],"E140":[{"end":"1927-10-12","start":"1861-03-17","name":"GR
ANDOLA"}],"M384":[{"name":"VAL LIONA","end":"9999-12-31","start":"2017-02-17
"}],"G993":[{"name":"PRATA CAMPORTACCIO","end":"9999-12-31","start":"1863-07
-13"}],"A291":[{"name":"ANGHIARI","end":"9999-12-31","start":"1861-03-17"}],
"D468":[{"end":"1923-02-13","start":"1920-10-16","name":"FAI"},{"start":"192
3-02-14","end":"1928-05-23","name":"FAI"},{"start":"1948-03-02","end":"1952-
05-16","name":"FAI"},{"name":"FAI DELLA PAGANELLA","start":"1952-05-17","end
":"9999-12-31"}],"A322":[{"end":"1928-08-07","start":"1861-03-17","name":"AN
ZINO"}],"G025":[{"name":"OLGIATE COMASCO","end":"9999-12-31","start":"1862-1
2-29"}],"A536":[{"name":"BADALUCCO","end":"9999-12-31","start":"1861-03-17"}
],"L643":[{"name":"VALPELLINA","end":"1946-01-29","start":"1939-10-21"},{"st
art":"1861-03-17","end":"1927-01-11","name":"VALPELLINE"},{"name":"VALPELLIN
E","start":"1927-01-12","end":"1939-10-20"},{"name":"VALPELLINE","end":"9999
-12-31","start":"1946-01-30"}],"E433":[{"start":"1861-03-17","end":"1927-01-
11","name":"LAMPORO"},{"name":"LAMPORO","end":"9999-12-31","start":"1927-01-
12"}],"A477":[{"end":"1928-06-26","start":"1861-03-17","name":"ASSOLO"},{"en
d":"1974-08-19","start":"1946-02-08","name":"ASSOLO"},{"start":"1974-08-20",
"end":"9999-12-31","name":"ASSOLO"}],"F047":[{"end":"1942-12-14","start":"18
61-03-17","name":"MASSINO"},{"name":"MASSINO VISCONTI","end":"9999-12-31","s
tart":"1942-12-15"}],"C873":[{"end":"1927-01-11","start":"1861-03-17","name"
:"COLLESCIPOLI"},{"name":"COLLESCIPOLI","end":"1927-01-12","start":"1927-01-
12"}],"E797":[{"name":"MADRUZZO","start":"1928-07-29","end":"1953-09-12"}],"
G774":[{"start":"1861-03-17","end":"9999-12-31","name":"POGNANO"}],"E909":[{
"end":"1923-02-13","start":"1920-10-16","name":"MARANO"},{"name":"MARANO","s
tart":"1923-02-14","end":"1929-02-13"}],"E606":[{"name":"LIPARI","start":"18
67-03-15","end":"9999-12-31"}],"M042":[{"name":"VILLETTE","end":"1928-06-30"
,"start":"1861-03-17"},{"start":"1957-01-25","end":"1992-05-22","name":"VILL
ETTE"},{"start":"1992-05-23","end":"9999-12-31","name":"VILLETTE"}],"H006":[
{"name":"PRATOLA SERRA","start":"1862-10-24","end":"9999-12-31"}],"C653":[{"
name":"CHIUSA DI PESIO","start":"1862-12-19","end":"9999-12-31"}],"C917":[{"
end":"1992-04-15","start":"1861-03-17","name":"COMAZZO"},{"start":"1992-04-1
6","end":"9999-12-31","name":"COMAZZO"}],"H574":[{"name":"ROSOLINI","start":
"1861-03-17","end":"9999-12-31"}],"D764":[{"name":"FRANCAVILLA MARITTIMA","e
nd":"9999-12-31","start":"1863-01-19"}],"D637":[{"name":"FLORINAS","end":"99
99-12-31","start":"1861-03-17"}],"L582":[{"name":"VALGRISANCHE","end":"1927-
01-11","start":"1861-03-17"},{"end":"1939-10-20","start":"1927-01-12","name"
:"VALGRISANCHE"},{"start":"1946-01-30","end":"1977-01-01","name":"VALGRISANC
HE"},{"name":"VALGRISENCHE","start":"1977-01-02","end":"9999-12-31"},{"start
":"1939-10-21","end":"1946-01-29","name":"VALGRISENZA"}],"G886":[{"end":"196
8-04-05","start":"1866-11-19","name":"PORCIA"},{"name":"PORCIA","start":"196
8-04-06","end":"9999-12-31"}],"G478":[{"name":"PERUGIA","start":"1861-03-17"
,"end":"9999-12-31"}],"E740":[{"name":"LUMEZZANE SAN SEBASTIANO","start":"19
21-04-29","end":"1928-01-06"}],"C314":[{"name":"CASTIGLIONE FALLETTO","end":
"9999-12-31","start":"1861-03-17"}],"G338":[{"start":"1873-08-08","end":"194
8-04-06","name":"PARODI LIGURE"},{"start":"1948-04-07","end":"9999-12-31","n
ame":"PARODI LIGURE"}],"E814":[{"name":"MAGLIANO VETERE","start":"1861-03-17
","end":"9999-12-31"}],"L828":[{"end":"9999-12-31","start":"1861-03-17","nam
e":"VIAGRANDE"}],"D622":[{"name":"FIUMEDINISI","start":"1861-03-17","end":"9
999-12-31"}],"I362":[{"start":"1863-04-13","end":"1916-09-14","name":"SANTO 
STEFANO AL CORNO"},{"name":"SANTO STEFANO LODIGIANO","start":"1916-09-15","e
nd":"1992-04-15"},{"end":"9999-12-31","start":"1992-04-16","name":"SANTO STE
FANO LODIGIANO"}],"H976":[{"end":"9999-12-31","start":"1863-02-06","name":"S
AN MANGO D'AQUINO"}],"G174":[{"name":"OSSANESGA","start":"1861-03-17","end":
"1928-05-18"}],"F982":[{"name":"NURALLAO","end":"1927-01-11","start":"1861-0
3-17"},{"name":"NURALLAO","end":"2005-12-31","start":"1927-01-12"},{"end":"2
016-04-27","start":"2006-01-01","name":"NURALLAO"},{"name":"NURALLAO","end":
"9999-12-31","start":"2016-04-28"}],"H679":[{"start":"1863-02-23","end":"192
6-02-06","name":"SALA COMACINA"},{"name":"SALA COMACINA","end":"1928-11-02",
"start":"1926-02-07"},{"name":"SALA COMACINA","start":"1950-04-05","end":"99
99-12-31"}],"D681":[{"name":"FONTECCHIO","start":"1861-03-17","end":"1954-10
-20"},{"end":"9999-12-31","start":"1954-10-21","name":"FONTECCHIO"}],"F119":
[{"name":"MELZO","end":"9999-12-31","start":"1861-03-17"}],"B308":[{"name":"
BUTTOGNO","end":"1928-06-28","start":"1861-03-17"}],"L342":[{"end":"9999-12-
31","start":"1927-09-04","name":"TRAVEDONA-MONATE"}],"F921":[{"name":"NOGARO
LE ROCCA","end":"1929-04-04","start":"1867-08-26"},{"name":"NOGAROLE ROCCA",
"end":"9999-12-31","start":"1929-04-05"}],"A410":[{"name":"ARIZZANO INFERIOR
E","end":"1927-12-03","start":"1911-07-04"}],"E065":[{"start":"1920-10-16","
end":"1923-02-13","name":"GIUSTINO"},{"name":"GIUSTINO","start":"1923-02-14"
,"end":"1928-04-02"},{"start":"1952-05-17","end":"9999-12-31","name":"GIUSTI
NO"}],"L521":[{"name":"UTA","end":"1974-08-19","start":"1861-03-17"},{"end":
"9999-12-31","start":"1974-08-20","name":"UTA"}],"G847":[{"start":"1929-10-2
9","end":"9999-12-31","name":"PONTE LAMBRO"},{"start":"1928-05-25","end":"19
29-10-28","name":"PONTELEZZA"}],"E672":[{"name":"LONGARONE","end":"1986-05-2
6","start":"1866-11-19"},{"name":"LONGARONE","end":"2014-02-21","start":"198
6-05-27"}],"H214":[{"start":"1866-11-19","end":"1934-03-13","name":"RECOARO"
},{"end":"9999-12-31","start":"1934-03-14","name":"RECOARO TERME"}],"I798":[
{"name":"SOLERO","end":"1929-06-14","start":"1861-03-17"},{"end":"9999-12-31
","start":"1929-06-15","name":"SOLERO"}],"F825":[{"start":"1861-03-17","end"
:"1927-01-11","name":"MUSIGNANO"},{"end":"1928-01-07","start":"1927-01-12","
name":"MUSIGNANO"}],"C414":[{"end":"1928-08-06","start":"1861-03-17","name":
"CECIMA"},{"name":"CECIMA","start":"1956-06-06","end":"9999-12-31"}],"G378":
[{"start":"1861-03-17","end":"9999-12-31","name":"PAT\u00d9"}],"M262":[{"sta
rt":"1973-03-14","end":"9999-12-31","name":"CELLOLE"}],"L960":[{"start":"186
1-03-17","end":"1928-07-28","name":"VILL'ALBESE"}],"A193":[{"name":"ALGUA","
start":"1963-10-20","end":"9999-12-31"},{"end":"1928-03-23","start":"1927-11
-20","name":"ALGUA DI COSTA SERINA"},{"name":"ALGUA DI COSTA SERINA","start"
:"1948-04-21","end":"1963-10-19"}],"H845":[{"name":"SAN FLORIANO","end":"192
3-02-07","start":"1921-01-05"},{"start":"1923-02-08","end":"1923-04-26","nam
e":"SAN FLORIANO"},{"start":"1923-04-27","end":"1927-01-11","name":"SAN FLOR
IANO DEL COLLIO"},{"end":"1927-12-28","start":"1927-01-12","name":"SAN FLORI
ANO DEL COLLIO"},{"name":"SAN FLORIANO DEL COLLIO","end":"2009-01-28","start
":"1951-02-15"},{"name":"SAN FLORIANO DEL COLLIO-\u0160TEVERJAN","end":"9999
-12-31","start":"2009-01-29"}],"H936":[{"name":"SAN GIUSTO CANAVESE","start"
:"1863-01-05","end":"1927-01-11"},{"end":"1945-10-04","start":"1927-01-12","
name":"SAN GIUSTO CANAVESE"},{"name":"SAN GIUSTO CANAVESE","start":"1945-10-
05","end":"9999-12-31"}],"G134":[{"end":"1960-07-04","start":"1928-12-14","n
ame":"ORTA SAN GIULIO"},{"start":"1960-07-05","end":"9999-12-31","name":"ORT
A SAN GIULIO"}],"E998":[{"name":"MARZANO APPIO","start":"1863-04-10","end":"
1927-01-11"},{"name":"MARZANO APPIO","end":"1945-08-31","start":"1927-01-12"
},{"end":"9999-12-31","start":"1945-09-01","name":"MARZANO APPIO"}],"H639":[
{"start":"1920-10-16","end":"1923-02-13","name":"RUMO"},{"name":"RUMO","star
t":"1923-02-14","end":"1928-09-05"},{"end":"1948-03-13","start":"1928-09-06"
,"name":"RUMO"},{"start":"1948-03-14","end":"9999-12-31","name":"RUMO"}],"H5
34":[{"name":"RONCIGLIONE","start":"1871-01-15","end":"1927-01-11"},{"start"
:"1927-01-12","end":"9999-12-31","name":"RONCIGLIONE"}],"D677":[{"name":"FON
TANIGORDA","start":"1861-03-17","end":"1923-07-22"},{"start":"1923-07-23","e
nd":"9999-12-31","name":"FONTANIGORDA"}],"I843":[{"end":"1928-03-06","start"
:"1862-12-29","name":"SORBO SERPICO"},{"name":"SORBO SERPICO","start":"1945-
08-28","end":"9999-12-31"}],"C118":[{"start":"1873-10-18","end":"9999-12-31"
,"name":"CASTEL GOFFREDO"}],"I462":[{"name":"SASSUOLO","start":"1861-03-17",
"end":"9999-12-31"}],"A668":[{"start":"1901-07-22","end":"2009-06-29","name"
:"BARLASSINA"},{"name":"BARLASSINA","start":"2009-06-30","end":"9999-12-31"}
],"H941":[{"name":"SAN GREGORIO D'IPPONA","start":"1863-02-06","end":"1992-0
4-15"},{"start":"1992-04-16","end":"9999-12-31","name":"SAN GREGORIO D'IPPON
A"}],"D503":[{"name":"FARNESE","start":"1871-01-15","end":"1927-01-11"},{"en
d":"9999-12-31","start":"1927-01-12","name":"FARNESE"}],"G438":[{"name":"PEN
NE","start":"1861-03-17","end":"1927-01-11"},{"name":"PENNE","end":"9999-12-
31","start":"1927-01-12"}],"F560":[{"name":"MONTEMARCIANO","end":"2010-03-01
","start":"1861-03-17"},{"start":"2010-03-02","end":"9999-12-31","name":"MON
TEMARCIANO"}],"E632":[{"name":"LOANO","start":"1878-08-28","end":"1927-01-11
"},{"name":"LOANO","end":"9999-12-31","start":"1927-01-12"}],"A310":[{"name"
:"ANTICOLI DI CAMPAGNA","start":"1872-10-18","end":"1911-09-18"},{"name":"FI
UGGI","start":"1911-09-19","end":"1927-01-11"},{"end":"9999-12-31","start":"
1927-01-12","name":"FIUGGI"}],"E245":[{"start":"1861-03-17","end":"9999-12-3
1","name":"GUARDIA LOMBARDI"}],"I109":[{"end":"1929-01-23","start":"1866-11-
19","name":"SAN PIETRO IN CARIANO"},{"name":"SAN PIETRO IN CARIANO","start":
"1929-01-24","end":"9999-12-31"}],"A502":[{"end":"1927-01-11","start":"1862-
10-06","name":"AUSONIA"},{"start":"1927-01-12","end":"9999-12-31","name":"AU
SONIA"}],"B408":[{"end":"9999-12-31","start":"1891-11-13","name":"CALESTANO"
}],"L863":[{"name":"VIGALZANO","start":"1920-10-16","end":"1923-02-13"},{"st
art":"1923-02-14","end":"1929-01-24","name":"VIGALZANO"}],"E627":[{"start":"
1869-02-08","end":"1992-04-15","name":"LIVRAGA"},{"name":"LIVRAGA","start":"
1992-04-16","end":"9999-12-31"}],"B104":[{"name":"BOVINO","end":"9999-12-31"
,"start":"1861-03-17"}],"I940":[{"name":"STAVA","end":"1927-01-11","start":"
1923-04-27"},{"name":"STAVA","end":"1928-12-06","start":"1927-01-12"},{"star
t":"1920-10-16","end":"1923-02-13","name":"STAVE"},{"end":"1923-04-26","star
t":"1923-02-14","name":"STAVE"}],"D192":[{"end":"1927-01-11","start":"1861-0
3-17","name":"CUASSO AL MONTE"},{"start":"1927-01-12","end":"1985-03-04","na
me":"CUASSO AL MONTE"},{"start":"1985-03-05","end":"9999-12-31","name":"CUAS
SO AL MONTE"}],"Z321":[{"name":"Guinea equatoriale","start":"0000-01-01","en
d":"9999-12-31"}],"I394":[{"end":"9999-12-31","start":"1863-02-06","name":"S
AN VITO CHIETINO"}],"D610":[{"name":"FIORENZUOLA DI FOCARA","start":"1888-10
-07","end":"1929-04-10"}],"C930":[{"name":"COMMESSAGGIO","end":"9999-12-31",
"start":"1868-07-01"}],"D423":[{"name":"ERICE","end":"1948-12-24","start":"1
934-09-12"},{"name":"ERICE","start":"1948-12-25","end":"1950-07-31"},{"name"
:"ERICE","end":"1952-08-16","start":"1950-08-01"},{"start":"1952-08-17","end
":"1955-02-14","name":"ERICE"},{"start":"1955-02-15","end":"9999-12-31","nam
e":"ERICE"},{"start":"1861-03-17","end":"1934-09-11","name":"MONTE SAN GIULI
ANO"}],"A481":[{"name":"ATELETA","start":"1861-03-17","end":"9999-12-31"}],"
B925":[{"end":"9999-12-31","start":"1861-03-17","name":"CASANDRINO"}],"H027"
:[{"name":"PREL\u00c0","start":"1861-03-17","end":"1928-04-16"},{"start":"19
28-04-17","end":"9999-12-31","name":"PREL\u00c0"}],"I709":[{"end":"1901-07-2
1","start":"1869-03-01","name":"SEVESO"},{"end":"2009-06-29","start":"1901-0
7-22","name":"SEVESO"},{"end":"9999-12-31","start":"2009-06-30","name":"SEVE
SO"}],"I997":[{"name":"SUISIO","end":"9999-12-31","start":"1861-03-17"}],"A2
16":[{"end":"1929-05-16","start":"1861-03-17","name":"ALMENNO SAN BARTOLOMEO
"},{"end":"9999-12-31","start":"1929-05-17","name":"ALMENNO SAN BARTOLOMEO"}
],"B399":[{"end":"2004-06-30","start":"1862-10-31","name":"CALDERARA DI RENO
"},{"start":"2004-07-01","end":"9999-12-31","name":"CALDERARA DI RENO"}],"B0
93":[{"start":"1861-03-17","end":"1928-03-28","name":"BOTTICINO SERA"}],"B70
4":[{"start":"1862-10-27","end":"1927-01-11","name":"CAPRIATI A VOLTURNO"},{
"name":"CAPRIATI A VOLTURNO","start":"1927-01-12","end":"1945-08-31"},{"name
":"CAPRIATI A VOLTURNO","start":"1945-09-01","end":"9999-12-31"}],"B821":[{"
start":"1872-11-01","end":"1929-02-07","name":"CARPIGNAGO"}],"C452":[{"name"
:"CEMBRA","end":"1923-02-13","start":"1920-10-16"},{"name":"CEMBRA","end":"1
928-09-05","start":"1923-02-14"},{"end":"1952-05-16","start":"1928-09-06","n
ame":"CEMBRA"},{"name":"CEMBRA","end":"2015-12-31","start":"1952-05-17"}],"D
792":[{"end":"1923-02-13","start":"1920-10-16","name":"FRAVEGGIO"},{"end":"1
928-04-18","start":"1923-02-14","name":"FRAVEGGIO"}],"A576":[{"name":"BAGNON
E","start":"1894-03-09","end":"9999-12-31"}],"B289":[{"name":"BUSNAGO","end"
:"2009-12-17","start":"1861-03-17"},{"start":"2009-12-18","end":"9999-12-31"
,"name":"BUSNAGO"}],"A004":[{"name":"ABBADIA CERRETO","end":"1992-04-15","st
art":"1861-03-17"},{"name":"ABBADIA CERRETO","end":"9999-12-31","start":"199
2-04-16"}],"G667":[{"name":"PILCANTE","end":"1923-02-13","start":"1920-10-16
"},{"end":"1928-09-05","start":"1923-02-14","name":"PILCANTE"}],"A793":[{"st
art":"1861-03-17","end":"1928-11-30","name":"BERGAMASCO"},{"end":"1955-11-25
","start":"1928-12-01","name":"BERGAMASCO"},{"end":"9999-12-31","start":"195
5-11-26","name":"BERGAMASCO"}],"G734":[{"name":"PLATANIA","start":"1861-03-1
7","end":"1961-01-27"},{"name":"PLATANIA","end":"9999-12-31","start":"1961-0
1-28"}],"D857":[{"name":"GAIOLA MOIOLA","end":"1946-07-03","start":"1928-06-
14"}],"B882":[{"name":"CASALEGGIO BOIRO","start":"1863-01-26","end":"9999-12
-31"}],"E473":[{"name":"LATISANA","start":"1866-11-19","end":"1959-08-17"},{
"name":"LATISANA","start":"1959-08-18","end":"1968-04-05"},{"end":"9999-12-3
1","start":"1968-04-06","name":"LATISANA"}],"A437":[{"end":"9999-12-31","sta
rt":"1862-11-24","name":"ARQUATA DEL TRONTO"}],"C833":[{"name":"COLDERANO","
end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14","end":"1923-04
-26","name":"COLDERANO"},{"start":"1923-04-27","end":"1927-01-11","name":"CO
LDRANO"},{"start":"1927-01-12","end":"1928-12-09","name":"COLDRANO"}],"L301"
:[{"name":"TORRIONI","end":"1928-01-27","start":"1861-03-17"},{"start":"1947
-09-24","end":"9999-12-31","name":"TORRIONI"}],"I628":[{"name":"SERIATE","st
art":"1861-03-17","end":"9999-12-31"}],"M188":[{"name":"ZONE","start":"1861-
03-17","end":"9999-12-31"}],"A377":[{"name":"ARCUGNANO","end":"9999-12-31","
start":"1866-11-19"}],"I284":[{"start":"1885-06-12","end":"1935-03-06","name
":"SANT'ANGELO ROMANO"},{"start":"1935-03-07","end":"1937-11-02","name":"SAN
T'ANGELO ROMANO"},{"start":"1937-11-03","end":"9999-12-31","name":"SANT'ANGE
LO ROMANO"}],"E333":[{"name":"ISEO","end":"1927-12-28","start":"1861-03-17"}
,{"name":"ISEO","end":"9999-12-31","start":"1927-12-29"}],"E039":[{"name":"G
IOIA SANNITICA","start":"1862-10-27","end":"1927-01-11"},{"name":"GIOIA SANN
ITICA","start":"1927-01-12","end":"1945-08-31"},{"end":"9999-12-31","start":
"1945-09-01","name":"GIOIA SANNITICA"}],"C718":[{"name":"CIPRESSA","start":"
1861-03-17","end":"1928-03-05"},{"start":"1928-03-06","end":"1954-07-02","na
me":"CIPRESSA"},{"start":"1954-07-03","end":"9999-12-31","name":"CIPRESSA"}]
,"B528":[{"name":"CAMPODIPIETRA","end":"1970-03-02","start":"1861-03-17"},{"
name":"CAMPODIPIETRA","end":"9999-12-31","start":"1970-03-03"}],"F267":[{"na
me":"MOGLIA","start":"1876-11-20","end":"9999-12-31"}],"D368":[{"name":"DRES
ENZA","start":"1921-01-05","end":"1923-02-07"},{"start":"1923-02-08","end":"
1927-01-11","name":"DRESENZA"},{"end":"1928-05-01","start":"1927-01-12","nam
e":"DRESENZA"}],"A422":[{"start":"1861-03-17","end":"1927-01-11","name":"ARN
ASCO"},{"end":"9999-12-31","start":"1927-01-12","name":"ARNASCO"}],"C389":[{
"name":"CAVAZZO CARNICO","start":"1871-08-03","end":"1968-04-05"},{"start":"
1968-04-06","end":"9999-12-31","name":"CAVAZZO CARNICO"}],"H821":[{"name":"S
ANDIGLIANO","end":"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","
end":"1992-04-15","name":"SANDIGLIANO"},{"name":"SANDIGLIANO","start":"1992-
04-16","end":"9999-12-31"}],"H704":[{"name":"SALE SAN GIOVANNI","end":"9999-
12-31","start":"1948-04-28"}],"C083":[{"end":"9999-12-31","start":"1861-03-1
7","name":"CASTEL DEL MONTE"}],"E768":[{"start":"1920-10-16","end":"1923-02-
13","name":"LUTAGO"},{"name":"LUTAGO","start":"1923-02-14","end":"1927-01-11
"},{"start":"1927-01-12","end":"1929-02-21","name":"LUTAGO"}],"D066":[{"star
t":"1861-03-17","end":"9999-12-31","name":"CORTENUOVA"}],"D271":[{"name":"DE
MONTE","start":"1861-03-17","end":"9999-12-31"}],"D733":[{"start":"1861-03-1
7","end":"1948-06-25","name":"FORZA D'AGR\u00d2"},{"name":"FORZA D'AGR\u00d2
","end":"9999-12-31","start":"1948-06-26"}],"I923":[{"end":"9999-12-31","sta
rt":"1861-03-17","name":"SPONGANO"}],"H093":[{"start":"1861-03-17","end":"19
27-06-09","name":"PURIA"}],"H399":[{"name":"ROCCA DI BOTTE","start":"1908-01
-22","end":"9999-12-31"}],"C757":[{"name":"CIVIASCO","end":"1927-01-11","sta
rt":"1861-03-17"},{"name":"CIVIASCO","end":"1929-03-01","start":"1927-01-12"
},{"name":"CIVIASCO","end":"9999-12-31","start":"1947-11-26"}],"D497":[{"sta
rt":"1861-03-17","end":"1928-05-18","name":"FARDELLA"},{"end":"9999-12-31","
start":"1947-01-10","name":"FARDELLA"}],"L790":[{"name":"VERSCIACO","start":
"1920-10-16","end":"1923-02-13"},{"end":"1927-01-11","start":"1923-02-14","n
ame":"VERSCIACO"},{"name":"VERSCIACO","start":"1927-01-12","end":"1928-12-09
"}],"B027":[{"name":"BORGO PANIGALE","start":"1861-03-17","end":"1937-12-05"
}],"H925":[{"name":"SAN GIOVANNI REATINO","end":"1875-03-05","start":"1861-0
3-17"}],"D818":[{"end":"1929-01-23","start":"1866-11-19","name":"FUMANE"},{"
start":"1929-01-24","end":"9999-12-31","name":"FUMANE"}],"L612":[{"start":"1
871-01-15","end":"1927-01-11","name":"VALLERANO"},{"end":"9999-12-31","start
":"1927-01-12","name":"VALLERANO"}],"I095":[{"end":"9999-12-31","start":"186
3-08-10","name":"SAN PIETRO APOSTOLO"}],"E282":[{"name":"IGLIANO","start":"1
861-03-17","end":"9999-12-31"}],"L485":[{"name":"UGGIANO LA CHIESA","end":"9
999-12-31","start":"1861-03-17"}],"L147":[{"name":"TESERO","start":"1920-10-
16","end":"1923-02-13"},{"name":"TESERO","start":"1923-02-14","end":"1999-07
-13"},{"end":"9999-12-31","start":"1999-07-14","name":"TESERO"}],"E889":[{"e
nd":"1968-04-05","start":"1866-11-19","name":"MANIAGO"},{"start":"1968-04-06
","end":"1971-07-09","name":"MANIAGO"},{"start":"1971-07-10","end":"9999-12-
31","name":"MANIAGO"}],"E392":[{"end":"9999-12-31","start":"1880-02-06","nam
e":"LABICO"}],"I820":[{"name":"SOMMA VESUVIANA","start":"1863-01-19","end":"
9999-12-31"}],"E221":[{"name":"GRUMENTO NOVA","end":"9999-12-31","start":"19
32-11-22"},{"end":"1932-11-21","start":"1863-05-06","name":"SAPONARA DI GRUM
ENTO"}],"C549":[{"name":"CERVENO","start":"1861-03-17","end":"1927-12-03"},{
"name":"CERVENO","end":"9999-12-31","start":"1947-06-18"}],"D051":[{"end":"1
935-04-07","start":"1861-03-17","name":"CORTANZE"},{"name":"CORTANZE","start
":"1935-04-08","end":"9999-12-31"}],"I167":[{"end":"1928-08-11","start":"186
1-03-17","name":"SANT'ABBONDIO"},{"name":"SANT'ABBONDIO","end":"2002-12-31",
"start":"1957-05-02"}],"H528":[{"name":"RONCEGNO","start":"1920-10-16","end"
:"1923-02-13"},{"end":"2005-05-03","start":"1923-02-14","name":"RONCEGNO"},{
"start":"2005-05-04","end":"9999-12-31","name":"RONCEGNO TERME"}],"G424":[{"
name":"PELLEGRINO PARMENSE","start":"1870-07-01","end":"1919-08-31"},{"name"
:"PELLEGRINO PARMENSE","start":"1919-09-01","end":"9999-12-31"}],"G350":[{"n
ame":"PARZANICA","end":"9999-12-31","start":"1861-03-17"}],"G972":[{"name":"
PRADURO E SASSO","start":"1861-03-17","end":"1935-08-15"},{"name":"SASSO BOL
OGNESE","end":"1938-04-27","start":"1935-08-16"},{"end":"9999-12-31","start"
:"1938-04-28","name":"SASSO MARCONI"}],"A727":[{"name":"BEAULARD","start":"1
861-03-17","end":"1928-05-03"}],"F184":[{"end":"1951-03-03","start":"1861-03
-17","name":"MEZZOJUSO"},{"name":"MEZZOJUSO","end":"9999-12-31","start":"195
1-03-04"}],"L186":[{"start":"1863-02-06","end":"1927-01-11","name":"TOCCO DA
 CASAURIA"},{"start":"1927-01-12","end":"9999-12-31","name":"TOCCO DA CASAUR
IA"}],"A305":[{"name":"ANTEI SANT'ANDREA","end":"1946-01-29","start":"1939-1
0-21"},{"name":"ANTEY SAINT ANDR\u00c9","end":"1927-01-11","start":"1861-03-
17"},{"start":"1927-01-12","end":"1939-10-20","name":"ANTEY SAINT ANDR\u00c9
"},{"start":"1946-01-30","end":"1977-01-01","name":"ANTEY SAINT ANDR\u00c9"}
,{"start":"1977-01-02","end":"9999-12-31","name":"ANTEY-SAINT-ANDR\u00c9"}],
"G002":[{"end":"1910-07-25","start":"1861-03-17","name":"OFENA"},{"start":"1
910-07-26","end":"1951-02-01","name":"OFENA"},{"start":"1951-02-02","end":"9
999-12-31","name":"OFENA"}],"F876":[{"start":"1861-03-17","end":"1923-03-18"
,"name":"NESPOLO"},{"start":"1923-03-19","end":"1927-01-11","name":"NESPOLO"
},{"end":"9999-12-31","start":"1927-01-12","name":"NESPOLO"}],"L874":[{"name
":"VIGGIANO","start":"1861-03-17","end":"9999-12-31"}],"C299":[{"end":"2017-
12-31","start":"1863-07-13","name":"CASTIGLIONE D'INTELVI"}],"D511":[{"name"
:"FAULE","start":"1861-03-17","end":"1928-04-02"},{"end":"9999-12-31","start
":"1946-12-28","name":"FAULE"}],"Z345":[{"name":"Somalia","end":"9999-12-31"
,"start":"0000-01-01"}],"H882":[{"end":"1979-06-30","start":"1926-11-11","na
me":"SAN GIORGIO IONICO"},{"end":"9999-12-31","start":"1979-07-01","name":"S
AN GIORGIO IONICO"},{"name":"SAN GIORGIO SOTTO TARANTO","end":"1923-12-21","
start":"1862-11-28"},{"name":"SAN GIORGIO SOTTO TARANTO","start":"1923-12-22
","end":"1926-11-10"}],"H289":[{"end":"1968-04-05","start":"1866-11-19","nam
e":"RIGOLATO"},{"start":"1968-04-06","end":"9999-12-31","name":"RIGOLATO"}],
"C892":[{"start":"1920-10-16","end":"1923-02-13","name":"COLOGNA IN GIUDICAR
IE"},{"name":"COLOGNA IN GIUDICARIE","start":"1923-02-14","end":"1928-03-06"
}],"F488":[{"name":"MONTE DI PROCIDA","end":"9999-12-31","start":"1907-02-11
"}],"A732":[{"name":"BEDULITA","start":"1861-03-17","end":"9999-12-31"}],"G1
28":[{"name":"ORSOGNA","end":"9999-12-31","start":"1861-03-17"}],"L313":[{"n
ame":"TOSS","end":"1923-02-13","start":"1920-10-16"},{"end":"1928-12-21","st
art":"1923-02-14","name":"TOSS"}],"B941":[{"name":"CASASCO","start":"1861-03
-17","end":"1929-04-19"},{"name":"CASASCO","end":"9999-12-31","start":"1929-
04-20"}],"L019":[{"end":"1925-11-29","start":"1861-03-17","name":"SUVERETO"}
,{"start":"1925-11-30","end":"9999-12-31","name":"SUVERETO"}],"I634":[{"name
":"SERMONETA","start":"1871-01-15","end":"1932-11-05"},{"name":"SERMONETA","
start":"1932-11-06","end":"1934-10-25"},{"end":"1934-12-17","start":"1934-10
-26","name":"SERMONETA"},{"end":"1935-05-01","start":"1934-12-18","name":"SE
RMONETA"},{"end":"1945-04-08","start":"1935-05-02","name":"SERMONETA"},{"nam
e":"SERMONETA","end":"9999-12-31","start":"1945-04-09"}],"I767":[{"name":"SI
ZZANO","start":"1861-03-17","end":"9999-12-31"}],"B534":[{"name":"CAMPOFILON
E","end":"2009-07-11","start":"1861-03-17"},{"name":"CAMPOFILONE","end":"999
9-12-31","start":"2009-07-12"}],"F656":[{"name":"MONTEVARCHI","end":"9999-12
-31","start":"1861-03-17"}],"L654":[{"name":"VALTORNENZA","start":"1939-10-2
1","end":"1946-01-29"},{"start":"1861-03-17","end":"1927-01-11","name":"VALT
OURNANCHE"},{"end":"1939-10-20","start":"1927-01-12","name":"VALTOURNANCHE"}
,{"name":"VALTOURNANCHE","end":"1977-01-01","start":"1946-01-30"},{"end":"99
99-12-31","start":"1977-01-02","name":"VALTOURNENCHE"}],"F784":[{"name":"MOT
TOLA","start":"1861-03-17","end":"1923-12-21"},{"name":"MOTTOLA","end":"9999
-12-31","start":"1923-12-22"}],"A127":[{"start":"1861-03-17","end":"9999-12-
31","name":"ALBAIRATE"}],"L786":[{"start":"1863-04-13","end":"1929-04-18","n
ame":"VERRUA SICCOMARIO"}],"B639":[{"end":"1928-12-01","start":"1861-03-17",
"name":"CANT\u00d9"},{"end":"9999-12-31","start":"1928-12-02","name":"CANT\u
00d9"}],"B936":[{"start":"1885-01-15","end":"1921-05-11","name":"CASARANO"},
{"end":"9999-12-31","start":"1921-05-12","name":"CASARANO"}],"B845":[{"end":
"9999-12-31","start":"1861-03-17","name":"CARTIGNANO"}],"D962":[{"name":"GEM
ONA","end":"1935-03-09","start":"1866-11-19"},{"name":"GEMONA DEL FRIULI","e
nd":"1968-04-05","start":"1935-03-10"},{"name":"GEMONA DEL FRIULI","end":"99
99-12-31","start":"1968-04-06"}],"D605":[{"start":"1863-02-23","end":"2013-1
1-11","name":"FINO MORNASCO"},{"start":"2013-11-12","end":"9999-12-31","name
":"FINO MORNASCO"}],"L559":[{"end":"1923-02-13","start":"1920-10-16","name":
"VALDIGIOVO"},{"name":"VALDIGIOVO","end":"1923-04-26","start":"1923-02-14"},
{"name":"VALGIOVO","end":"1927-01-11","start":"1923-04-27"},{"end":"1928-04-
12","start":"1927-01-12","name":"VALGIOVO"}],"G256":[{"start":"1861-03-17","
end":"1928-05-11","name":"PALANZO"}],"I539":[{"name":"SCIOLZE","end":"9999-1
2-31","start":"1861-03-17"}],"A132":[{"name":"ALBANO LAZIALE","end":"1935-03
-06","start":"1872-10-02"},{"start":"1935-03-07","end":"9999-12-31","name":"
ALBANO LAZIALE"}],"G728":[{"start":"1861-03-17","end":"1992-04-15","name":"P
IZZONI"},{"start":"1992-04-16","end":"9999-12-31","name":"PIZZONI"}],"G906":
[{"name":"PORTO CERESIO","start":"1863-02-23","end":"1927-01-11"},{"end":"19
28-05-21","start":"1927-01-12","name":"PORTO CERESIO"},{"name":"PORTO CERESI
O","start":"1928-05-22","end":"1958-09-04"},{"start":"1958-09-05","end":"198
5-03-04","name":"PORTO CERESIO"},{"name":"PORTO CERESIO","end":"9999-12-31",
"start":"1985-03-05"}],"H104":[{"name":"QUARGNENTO","start":"1861-03-17","en
d":"9999-12-31"}],"A963":[{"name":"BONATE SOPRA","start":"1890-04-28","end":
"9999-12-31"}],"G609":[{"name":"PIETRACATELLA","start":"1861-03-17","end":"1
970-03-02"},{"start":"1970-03-03","end":"9999-12-31","name":"PIETRACATELLA"}
],"E168":[{"name":"GRESSONEY SAINT JEAN","start":"1861-03-17","end":"1927-01
-11"},{"end":"1928-06-30","start":"1927-01-12","name":"GRESSONEY SAINT JEAN"
},{"name":"GRESSONEY SAINT JEAN","start":"1946-05-15","end":"1977-01-01"},{"
name":"GRESSONEY-SAINT-JEAN","end":"9999-12-31","start":"1977-01-02"}],"M319
":[{"end":"9999-12-31","start":"2014-01-01","name":"FABBRICHE DI VERGEMOLI"}
],"M013":[{"name":"VILLAR PELLICE","end":"9999-12-31","start":"1861-03-17"}]
,"Z233":[{"name":"Mongolia","start":"0000-01-01","end":"9999-12-31"}],"F661"
:[{"name":"MONTEVERDI","start":"1861-03-17","end":"1920-07-04"},{"name":"MON
TEVERDI MARITTIMO","start":"1920-07-05","end":"9999-12-31"}],"D133":[{"name"
:"CRAVANZANA","start":"1861-03-17","end":"9999-12-31"}],"C157":[{"start":"19
28-12-09","end":"2006-05-11","name":"CASTELLETTO DI BRANDUZZO"},{"start":"20
06-05-12","end":"9999-12-31","name":"CASTELLETTO DI BRANDUZZO"}],"L190":[{"s
tart":"1861-03-17","end":"1905-04-06","name":"TOIRANO"},{"start":"1905-04-07
","end":"1927-01-11","name":"TOIRANO"},{"start":"1927-01-12","end":"1929-05-
13","name":"TOIRANO"},{"name":"TOIRANO","end":"1946-12-23","start":"1929-05-
14"},{"end":"9999-12-31","start":"1946-12-24","name":"TOIRANO"}],"E693":[{"n
ame":"LORO CIUFFENNA","start":"1863-04-13","end":"9999-12-31"}],"H408":[{"st
art":"1864-06-23","end":"9999-12-31","name":"ROCCAFORTE DEL GRECO"}],"E464":
[{"name":"LAS PLASSAS","end":"1928-02-09","start":"1861-03-17"},{"start":"19
47-01-24","end":"1974-08-19","name":"LAS PLASSAS"},{"name":"LAS PLASSAS","st
art":"1974-08-20","end":"2005-12-31"},{"end":"2016-04-27","start":"2006-01-0
1","name":"LAS PLASSAS"},{"name":"LAS PLASSAS","start":"2016-04-28","end":"9
999-12-31"}],"F209":[{"end":"9999-12-31","start":"1862-12-29","name":"MILITE
LLO IN VAL DI CATANIA"}],"D440":[{"end":"1927-01-11","start":"1868-01-01","n
ame":"ESPERIA"},{"end":"9999-12-31","start":"1927-01-12","name":"ESPERIA"}],
"L747":[{"name":"VERBICARO","start":"1861-03-17","end":"9999-12-31"}],"G504"
:[{"end":"9999-12-31","start":"1868-06-12","name":"PESSINA CREMONESE"}],"F80
2":[{"end":"9999-12-31","start":"1861-03-17","name":"MULAZZO"}],"G076":[{"na
me":"ONZO","start":"1861-03-17","end":"1927-01-11"},{"name":"ONZO","end":"99
99-12-31","start":"1927-01-12"}],"G261":[{"end":"1927-01-11","start":"1863-0
1-19","name":"PALAZZO SAN GERVASIO"},{"start":"1927-01-12","end":"1944-04-29
","name":"PALAZZO SAN GERVASIO"},{"name":"PALAZZO SAN GERVASIO","start":"194
4-04-30","end":"9999-12-31"}],"C208":[{"name":"CASTEL MELLA","start":"1864-0
6-23","end":"9999-12-31"}],"B679":[{"name":"CAPPELLA CANTONE","start":"1861-
03-17","end":"9999-12-31"}],"L620":[{"name":"VALLE SAN NICOLAO","end":"1927-
01-11","start":"1861-03-17"},{"name":"VALLE SAN NICOLAO","start":"1927-01-12
","end":"1965-02-15"},{"start":"1965-02-16","end":"1992-04-15","name":"VALLE
 SAN NICOLAO"},{"end":"9999-12-31","start":"1992-04-16","name":"VALLE SAN NI
COLAO"}],"B976":[{"name":"CASNATE","start":"1861-03-17","end":"1937-10-09"}]
,"I579":[{"start":"1920-10-16","end":"1923-02-13","name":"SEIO"},{"name":"SE
IO","start":"1923-02-14","end":"1928-08-06"}],"A172":[{"start":"1861-03-17",
"end":"9999-12-31","name":"ALBOSAGGIA"}],"L413":[{"start":"1861-03-17","end"
:"9999-12-31","name":"TREZZONE"}],"C555":[{"start":"1861-03-17","end":"1931-
03-05","name":"CERVIGNANO"},{"name":"CERVIGNANO D'ADDA","end":"1992-04-15","
start":"1931-03-06"},{"name":"CERVIGNANO D'ADDA","end":"2004-06-07","start":
"1992-04-16"},{"end":"9999-12-31","start":"2004-06-08","name":"CERVIGNANO D'
ADDA"}],"I674":[{"name":"SESANA","end":"1923-02-07","start":"1921-01-05"},{"
name":"SESANA","start":"1923-02-08","end":"1927-06-08"},{"start":"1927-06-09
","end":"1947-09-15","name":"SESANA"}],"Z359":[{"name":"Lesotho","start":"00
00-01-01","end":"9999-12-31"}],"B006":[{"name":"BORGO VALSUGANA","end":"1923
-02-13","start":"1920-10-16"},{"name":"BORGO VALSUGANA","start":"1923-02-14"
,"end":"1928-05-12"},{"name":"BORGO VALSUGANA","end":"1947-01-14","start":"1
928-05-13"},{"end":"9999-12-31","start":"1947-01-15","name":"BORGO VALSUGANA
"}],"A928":[{"end":"1927-01-11","start":"1861-03-17","name":"BOGNO"},{"name"
:"BOGNO","start":"1927-01-12","end":"1928-01-21"}],"B574":[{"name":"CANALE D
'AGORDO","start":"1964-05-14","end":"9999-12-31"},{"start":"1866-11-19","end
":"1964-05-13","name":"FORNO DI CANALE"}],"C864":[{"name":"COLLEPARDO","end"
:"1927-01-11","start":"1871-01-15"},{"name":"COLLEPARDO","start":"1927-01-12
","end":"9999-12-31"}],"A706":[{"name":"BASSANO IN TEVERINA","start":"1871-0
1-15","end":"1927-01-11"},{"name":"BASSANO IN TEVERINA","start":"1927-01-12"
,"end":"1927-10-21"},{"name":"BASSANO IN TEVERINA","end":"9999-12-31","start
":"1958-10-25"}],"G763":[{"end":"1923-03-18","start":"1861-03-17","name":"PO
GGIO MIRTETO"},{"name":"POGGIO MIRTETO","end":"1927-01-11","start":"1923-03-
19"},{"end":"9999-12-31","start":"1927-01-12","name":"POGGIO MIRTETO"}],"A69
7":[{"name":"BASIANO","end":"9999-12-31","start":"1861-03-17"}],"G036":[{"na
me":"OLIVERI","end":"9999-12-31","start":"1861-03-17"}],"M419":[{"end":"9999
-12-31","start":"2019-01-01","name":"COLLI VERDI"}],"B214":[{"name":"BRUGNAT
O","end":"1923-12-21","start":"1861-03-17"},{"name":"BRUGNATO","start":"1923
-12-22","end":"1956-07-24"},{"name":"BRUGNATO","start":"1956-07-25","end":"9
999-12-31"}],"A525":[{"start":"1861-03-17","end":"1927-01-11","name":"AZEGLI
O"},{"start":"1927-01-12","end":"1929-05-24","name":"AZEGLIO"},{"end":"1945-
10-04","start":"1929-05-25","name":"AZEGLIO"},{"name":"AZEGLIO","end":"1946-
12-22","start":"1945-10-05"},{"name":"AZEGLIO","end":"9999-12-31","start":"1
946-12-23"}],"Z731":[{"start":"0000-01-01","end":"9999-12-31","name":"Kiriba
ti"}],"I812":[{"name":"SOLTO COLLINA","end":"1948-06-25","start":"1928-03-07
"},{"name":"SOLTO COLLINA","start":"1948-06-26","end":"9999-12-31"}],"D340":
[{"start":"1861-03-17","end":"1929-05-29","name":"DONELASCO"}],"E967":[{"end
":"1927-12-03","start":"1861-03-17","name":"MARONE"},{"name":"MARONE","start
":"1927-12-04","end":"9999-12-31"}],"C958":[{"end":"9999-12-31","start":"186
1-03-17","name":"CONFIENZA"}],"H308":[{"name":"RIONERO SANNITICO","start":"1
864-03-11","end":"1970-03-02"},{"end":"9999-12-31","start":"1970-03-03","nam
e":"RIONERO SANNITICO"}],"E364":[{"start":"1861-03-17","end":"9999-12-31","n
ame":"ISORELLA"}],"G041":[{"name":"OLIVETTA SAN MICHELE","end":"1947-09-15",
"start":"1890-05-31"},{"start":"1947-09-16","end":"9999-12-31","name":"OLIVE
TTA SAN MICHELE"}],"I219":[{"start":"1864-09-12","end":"9999-12-31","name":"
SANTA LUCIA DI SERINO"}],"D173":[{"name":"CROCE","end":"1928-06-25","start":
"1861-03-17"}],"D246":[{"name":"DAMBEL","end":"1923-02-13","start":"1920-10-
16"},{"end":"1928-11-26","start":"1923-02-14","name":"DAMBEL"},{"name":"DAMB
EL","end":"9999-12-31","start":"1948-03-02"}],"F298":[{"name":"MOLLI\u00c8RE
S","start":"1861-03-17","end":"1928-12-13"}],"D397":[{"name":"ELLERA","start
":"1861-03-17","end":"1927-01-11"},{"name":"ELLERA","start":"1927-01-12","en
d":"1929-04-11"}],"Z146":[{"name":"Lituania","end":"9999-12-31","start":"000
0-01-01"}],"A106":[{"start":"1861-03-17","end":"1927-01-11","name":"AILANO"}
,{"name":"AILANO","start":"1927-01-12","end":"1945-08-31"},{"name":"AILANO",
"end":"9999-12-31","start":"1945-09-01"}],"G163":[{"name":"OSOPPO","end":"19
68-04-05","start":"1866-11-19"},{"name":"OSOPPO","start":"1968-04-06","end":
"9999-12-31"}],"I565":[{"start":"1861-03-17","end":"1961-06-13","name":"SEDI
NI"},{"name":"SEDINI","end":"1969-04-14","start":"1961-06-14"},{"start":"196
9-04-15","end":"9999-12-31","name":"SEDINI"}],"B665":[{"start":"1921-01-05",
"end":"1923-02-07","name":"CAPODISTRIA"},{"name":"CAPODISTRIA","start":"1923
-02-08","end":"1923-11-14"},{"name":"CAPODISTRIA","end":"1977-04-03","start"
:"1923-11-15"}],"B902":[{"name":"CASALNOCETO","start":"1861-03-17","end":"99
99-12-31"}],"L948":[{"name":"VILLAFRANCA PIEMONTE","end":"1934-03-28","start
":"1863-01-26"},{"name":"VILLAFRANCA PIEMONTE","end":"9999-12-31","start":"1
950-02-03"},{"name":"VILLAFRANCA SABAUDA","end":"1950-02-02","start":"1934-0
3-29"}],"C489":[{"name":"CERCHIARA DI CALABRIA","end":"9999-12-31","start":"
1864-04-01"}],"D994":[{"end":"1974-08-19","start":"1861-03-17","name":"GESIC
O"},{"end":"2016-04-27","start":"1974-08-20","name":"GESICO"},{"end":"9999-1
2-31","start":"2016-04-28","name":"GESICO"}],"Z131":[{"end":"9999-12-31","st
art":"0000-01-01","name":"Spagna"}],"A640":[{"start":"1866-11-19","end":"196
8-04-05","name":"BARCIS"},{"end":"9999-12-31","start":"1968-04-06","name":"B
ARCIS"}],"D231":[{"name":"CUSANO MILANINO","start":"1915-08-16","end":"1936-
07-07"},{"start":"1936-07-08","end":"9999-12-31","name":"CUSANO MILANINO"},{
"name":"CUSANO SUL SEVESO","end":"1915-08-15","start":"1862-12-29"}],"F548":
[{"start":"1861-03-17","end":"1970-03-02","name":"MONTELONGO"},{"end":"9999-
12-31","start":"1970-03-03","name":"MONTELONGO"}],"G698":[{"name":"PIPERNO",
"end":"1928-09-08","start":"1871-01-15"},{"name":"PRIVERNO","end":"1928-10-2
9","start":"1928-09-09"},{"end":"1934-10-25","start":"1928-10-30","name":"PR
IVERNO"},{"name":"PRIVERNO","start":"1934-10-26","end":"1934-12-17"},{"start
":"1934-12-18","end":"1935-07-17","name":"PRIVERNO"},{"end":"1939-12-07","st
art":"1935-07-18","name":"PRIVERNO"},{"name":"PRIVERNO","start":"1939-12-08"
,"end":"1945-04-08"},{"start":"1945-04-09","end":"1947-03-21","name":"PRIVER
NO"},{"name":"PRIVERNO","start":"1947-03-22","end":"1947-06-23"},{"name":"PR
IVERNO","end":"9999-12-31","start":"1947-06-24"}],"Z600":[{"name":"Argentina
","start":"0000-01-01","end":"9999-12-31"}],"D596":[{"name":"FILOGASO","end"
:"1992-04-15","start":"1861-03-17"},{"name":"FILOGASO","start":"1992-04-16",
"end":"9999-12-31"}],"B072":[{"end":"1927-01-11","start":"1863-02-23","name"
:"BOSCO VALTRAVAGLIA"},{"name":"BOSCO VALTRAVAGLIA","start":"1927-01-12","en
d":"1928-01-21"}],"L385":[{"name":"TRES","start":"1920-10-16","end":"1923-02
-13"},{"end":"1928-12-13","start":"1923-02-14","name":"TRES"},{"name":"TRES"
,"start":"1928-12-14","end":"1950-05-12"},{"start":"1950-05-13","end":"2014-
12-31","name":"TRES"}],"H563":[{"name":"ROSCIATE","start":"1861-03-17","end"
:"1927-08-18"}],"D773":[{"end":"1951-12-31","start":"1871-01-15","name":"FRA
SCATI"},{"name":"FRASCATI","end":"9999-12-31","start":"1952-01-01"}],"G450":
[{"name":"PERFUGAS","start":"1861-03-17","end":"1969-04-14"},{"name":"PERFUG
AS","start":"1969-04-15","end":"1988-08-05"},{"name":"PERFUGAS","start":"198
8-08-06","end":"9999-12-31"}],"F836":[{"name":"NALLES","end":"1923-02-13","s
tart":"1920-10-16"},{"name":"NALLES","end":"1927-01-11","start":"1923-02-14"
},{"start":"1927-01-12","end":"1928-12-31","name":"NALLES"},{"name":"NALLES"
,"start":"1929-01-01","end":"1953-05-22"},{"name":"NALLES","end":"1964-09-22
","start":"1953-05-23"},{"name":"NALLES","end":"9999-12-31","start":"1964-09
-23"}],"L834":[{"name":"VIARIGI","end":"1935-04-07","start":"1861-03-17"},{"
end":"9999-12-31","start":"1935-04-08","name":"VIARIGI"}],"F945":[{"name":"N
OVACELLA","end":"1923-02-13","start":"1920-10-16"},{"name":"NOVACELLA","star
t":"1923-02-14","end":"1927-01-11"},{"name":"NOVACELLA","end":"1929-05-07","
start":"1927-01-12"}],"G324":[{"name":"PARABIAGO","start":"1861-03-17","end"
:"9999-12-31"}],"A772":[{"name":"BELVEDERE DI SPINELLO","start":"1863-04-10"
,"end":"1992-04-15"},{"name":"BELVEDERE DI SPINELLO","start":"1992-04-16","e
nd":"9999-12-31"}],"I854":[{"name":"SORIANO CALABRO","end":"1964-10-02","sta
rt":"1877-11-18"},{"end":"1992-04-15","start":"1964-10-03","name":"SORIANO C
ALABRO"},{"start":"1992-04-16","end":"9999-12-31","name":"SORIANO CALABRO"}]
,"G927":[{"start":"1861-03-17","end":"1927-01-11","name":"PORTULA"},{"name":
"PORTULA","start":"1927-01-12","end":"1992-04-15"},{"name":"PORTULA","end":"
9999-12-31","start":"1992-04-16"}],"E492":[{"name":"LAVARONE","end":"1923-02
-13","start":"1920-10-16"},{"end":"1929-05-27","start":"1923-02-14","name":"
LAVARONE"},{"name":"LAVARONE","end":"9999-12-31","start":"1929-05-28"}],"L29
5":[{"end":"9999-12-31","start":"1869-01-28","name":"TORRICELLA SICURA"}],"B
252":[{"start":"1861-03-17","end":"1928-06-30","name":"BUGGIOLO"}],"I968":[{
"name":"STRADELLA","end":"9999-12-31","start":"1861-03-17"}],"F841":[{"name"
:"NARCAO","start":"1861-03-17","end":"1958-08-09"},{"name":"NARCAO","start":
"1958-08-10","end":"1965-01-07"},{"name":"NARCAO","start":"1965-01-08","end"
:"1974-08-19"},{"end":"2005-12-31","start":"1974-08-20","name":"NARCAO"},{"s
tart":"2006-01-01","end":"2016-04-27","name":"NARCAO"},{"end":"9999-12-31","
start":"2016-04-28","name":"NARCAO"}],"D287":[{"end":"1927-01-11","start":"1
861-03-17","name":"DESULO"},{"end":"9999-12-31","start":"1927-01-12","name":
"DESULO"}],"L545":[{"start":"1928-05-19","end":"1929-05-16","name":"VALBREMB
O"},{"name":"VALBREMBO","start":"1929-05-17","end":"1970-06-01"},{"name":"VA
LBREMBO","start":"1970-06-02","end":"9999-12-31"}],"G932":[{"name":"POSITANO
","end":"9999-12-31","start":"1861-03-17"}],"E723":[{"end":"9999-12-31","sta
rt":"1861-03-17","name":"LUCO DEI MARSI"}],"A405":[{"name":"ARIGNANO","start
":"1861-03-17","end":"1928-01-21"},{"name":"ARIGNANO","end":"1947-01-08","st
art":"1928-01-22"},{"start":"1947-01-09","end":"9999-12-31","name":"ARIGNANO
"}],"B859":[{"name":"CASACANDITELLA","end":"9999-12-31","start":"1861-03-17"
}],"M085":[{"name":"VITO D'ASIO","start":"1866-11-19","end":"1968-04-05"},{"
name":"VITO D'ASIO","end":"9999-12-31","start":"1968-04-06"}],"B757":[{"name
":"CARDEZZA","end":"1928-10-16","start":"1861-03-17"}],"L201":[{"name":"TONA
DICO","end":"1923-02-13","start":"1920-10-16"},{"name":"TONADICO","end":"192
7-11-23","start":"1923-02-14"},{"start":"1947-01-24","end":"2015-12-31","nam
e":"TONADICO"}],"F566":[{"start":"1861-03-17","end":"9999-12-31","name":"MON
TEMILETTO"}],"F858":[{"end":"1937-03-03","start":"1876-10-18","name":"NE"},{
"name":"NE","end":"9999-12-31","start":"1937-03-04"}],"L564":[{"end":"9999-1
2-31","start":"1931-12-06","name":"VAL DI VIZZE"}],"D782":[{"start":"1861-03
-17","end":"9999-12-31","name":"FRASSINO"}],"H452":[{"end":"1927-01-11","sta
rt":"1861-03-17","name":"ROCCAVIGNALE"},{"start":"1927-01-12","end":"1928-12
-12","name":"ROCCAVIGNALE"},{"name":"ROCCAVIGNALE","end":"9999-12-31","start
":"1928-12-13"}],"B083":[{"start":"1861-03-17","end":"9999-12-31","name":"BO
SSICO"}],"B389":[{"name":"CALCER\u00c0NICA","start":"1920-10-16","end":"1923
-02-13"},{"end":"1929-01-24","start":"1923-02-14","name":"CALCER\u00c0NICA"}
,{"start":"1947-05-17","end":"1957-12-31","name":"CALCER\u00c0NICA"},{"end":
"1988-08-02","start":"1958-01-01","name":"CALCERANICA AL LAGO"},{"start":"19
88-08-03","end":"9999-12-31","name":"CALCERANICA AL LAGO"}],"A491":[{"start"
:"1866-11-19","end":"1953-08-31","name":"ATTIMIS"},{"start":"1953-09-01","en
d":"1968-04-05","name":"ATTIMIS"},{"start":"1968-04-06","end":"9999-12-31","
name":"ATTIMIS"}],"H930":[{"end":"1943-06-02","start":"1893-04-30","name":"S
AN GIULIANO MILANESE"},{"end":"9999-12-31","start":"1943-06-03","name":"SAN 
GIULIANO MILANESE"}],"D268":[{"start":"1878-02-11","end":"9999-12-31","name"
:"DELIANUOVA"}],"F367":[{"end":"1946-01-29","start":"1929-10-03","name":"MON
GIOVE"},{"start":"1861-03-17","end":"1927-01-11","name":"MONTJOVET"},{"name"
:"MONTJOVET","end":"1928-05-01","start":"1927-01-12"},{"name":"MONTJOVET","s
tart":"1928-05-02","end":"1929-10-02"},{"end":"1946-05-14","start":"1946-01-
30","name":"MONTJOVET"},{"end":"9999-12-31","start":"1946-05-15","name":"MON
TJOVET"}],"I987":[{"name":"STRUPPA","start":"1861-03-17","end":"1926-02-05"}
],"I384":[{"end":"1927-10-08","start":"1861-03-17","name":"SAN VERO MILIS"},
{"name":"SAN VERO MILIS","end":"1947-10-25","start":"1927-10-09"},{"end":"19
74-08-19","start":"1947-10-26","name":"SAN VERO MILIS"},{"start":"1974-08-20
","end":"9999-12-31","name":"SAN VERO MILIS"}],"A277":[{"name":"ANDONNO","st
art":"1861-03-17","end":"1928-12-31"}],"E233":[{"end":"9999-12-31","start":"
1861-03-17","name":"GUALTIERI SICAMIN\u00d2"}],"L966":[{"end":"1974-08-19","
start":"1861-03-17","name":"VILLAMAR"},{"name":"VILLAMAR","start":"1974-08-2
0","end":"2005-12-31"},{"start":"2006-01-01","end":"2016-04-27","name":"VILL
AMAR"},{"name":"VILLAMAR","start":"2016-04-28","end":"9999-12-31"}],"F964":[
{"name":"NOVENTA VICENTINA","start":"1866-11-19","end":"9999-12-31"}],"C027"
:[{"name":"CASSINE","end":"9999-12-31","start":"1861-03-17"}],"L669":[{"end"
:"1927-01-11","start":"1861-03-17","name":"VARALLO"},{"name":"VARALLO","star
t":"1927-01-12","end":"1929-03-01"},{"name":"VARALLO","end":"1947-11-25","st
art":"1929-03-02"},{"start":"1947-11-26","end":"9999-12-31","name":"VARALLO"
}],"I649":[{"name":"SERRAMONACESCA","start":"1861-03-17","end":"1927-01-11"}
,{"start":"1927-01-12","end":"9999-12-31","name":"SERRAMONACESCA"}],"F551":[
{"name":"MONTELUPO FIORENTINO","end":"9999-12-31","start":"1863-04-13"}],"I9
46":[{"end":"1927-01-11","start":"1861-03-17","name":"STELLA"},{"start":"192
7-01-12","end":"1966-07-29","name":"STELLA"},{"start":"1966-07-30","end":"99
99-12-31","name":"STELLA"}],"H718":[{"start":"1928-05-12","end":"1947-09-15"
,"name":"SALONA D'ISONZO"}],"I835":[{"start":"1861-03-17","end":"1927-01-11"
,"name":"SOPRANA"},{"start":"1927-01-12","end":"1992-04-15","name":"SOPRANA"
},{"name":"SOPRANA","start":"1992-04-16","end":"2018-12-31"}],"E459":[{"end"
:"1927-10-20","start":"1861-03-17","name":"LASCARI"},{"name":"LASCARI","star
t":"1927-10-21","end":"9999-12-31"}],"L855":[{"name":"VIDOLASCO","start":"18
61-03-17","end":"1934-03-06"}],"M198":[{"name":"ZUCLO","end":"1923-02-13","s
tart":"1920-10-16"},{"name":"ZUCLO","start":"1923-02-14","end":"1928-02-23"}
,{"end":"2015-12-31","start":"1952-05-17","name":"ZUCLO"}],"A517":[{"start":
"1861-03-17","end":"9999-12-31","name":"AVIATICO"}],"B549":[{"name":"CAMPOMA
GGIORE","start":"1861-03-17","end":"9999-12-31"}],"I294":[{"name":"SANT'ANTI
OCO","end":"1974-08-19","start":"1861-03-17"},{"name":"SANT'ANTIOCO","start"
:"1974-08-20","end":"2005-12-31"},{"name":"SANT'ANTIOCO","start":"2006-01-01
","end":"2016-04-27"},{"end":"9999-12-31","start":"2016-04-28","name":"SANT'
ANTIOCO"}],"D804":[{"start":"1866-11-19","end":"1968-04-05","name":"FRISANCO
"},{"name":"FRISANCO","end":"9999-12-31","start":"1968-04-06"}],"D721":[{"na
me":"FORNI DI VAL D'ASTICO","end":"1940-09-12","start":"1926-03-06"}],"I931"
:[{"name":"STABELLO","end":"1928-03-26","start":"1861-03-17"}],"A783":[{"nam
e":"BENEVENTO","end":"9999-12-31","start":"1861-03-17"}],"F011":[{"name":"MA
SER\u00c0 DI PADOVA","end":"9999-12-31","start":"1868-02-14"}],"B892":[{"nam
e":"CASALFIUMANESE","end":"9999-12-31","start":"1861-03-17"}],"B644":[{"name
":"CAPACCIO","end":"2016-06-09","start":"1861-03-17"},{"name":"CAPACCIO PAES
TUM","end":"9999-12-31","start":"2016-06-10"}],"L951":[{"name":"VILLAFRATI",
"start":"1861-03-17","end":"9999-12-31"}],"H833":[{"name":"SAN FELICE DEL LI
TTORIO","start":"1929-08-23","end":"1949-09-16"},{"name":"SAN FELICE DEL MOL
ISE","end":"1970-03-02","start":"1949-09-17"},{"name":"SAN FELICE DEL MOLISE
","start":"1970-03-03","end":"9999-12-31"},{"name":"SAN FELICE SLAVO","end":
"1902-01-21","start":"1863-01-19"},{"end":"1929-08-22","start":"1902-01-22",
"name":"SAN FELICE SLAVO"}],"I544":[{"start":"1861-03-17","end":"1927-01-11"
,"name":"SCOPA"},{"name":"SCOPA","end":"9999-12-31","start":"1927-01-12"}],"
A057":[{"name":"ADRARA SAN MARTINO","start":"1861-03-17","end":"9999-12-31"}
],"A316":[{"end":"1928-05-31","start":"1861-03-17","name":"ANTRONAPIANA"}],"
H118":[{"start":"1862-09-29","end":"1935-04-11","name":"QUARTU SANT'ELENA"},
{"name":"QUARTU SANT'ELENA","end":"1974-08-19","start":"1935-04-12"},{"name"
:"QUARTU SANT'ELENA","start":"1974-08-20","end":"9999-12-31"}],"C534":[{"end
":"1970-03-02","start":"1863-04-10","name":"CERRO AL VOLTURNO"},{"name":"CER
RO AL VOLTURNO","start":"1970-03-03","end":"9999-12-31"}],"E345":[{"start":"
1861-03-17","end":"1928-12-29","name":"ISOLABELLA"},{"name":"ISOLABELLA","en
d":"9999-12-31","start":"1946-12-24"}],"A210":[{"name":"ALLUMIERE","end":"99
99-12-31","start":"1871-01-15"}],"L062":[{"end":"9999-12-31","start":"1861-0
3-17","name":"TAURASI"}],"G058":[{"name":"OLZAI","end":"1927-01-11","start":
"1861-03-17"},{"end":"9999-12-31","start":"1927-01-12","name":"OLZAI"}],"C94
1":[{"name":"CONCA CASALE","start":"1911-07-28","end":"1927-10-13"},{"start"
:"1937-02-24","end":"1970-03-02","name":"CONCA CASALE"},{"end":"9999-12-31",
"start":"1970-03-03","name":"CONCA CASALE"}],"L675":[{"name":"VARAZZE","star
t":"1861-03-17","end":"1927-01-11"},{"name":"VARAZZE","end":"9999-12-31","st
art":"1927-01-12"}],"D121":[{"name":"COTIGNOLA","start":"1861-03-17","end":"
9999-12-31"}],"E988":[{"start":"1862-12-19","end":"1928-03-12","name":"MARTI
NIANA PO"},{"start":"1928-03-13","end":"1946-11-11","name":"MARTINIANA PO"},
{"name":"MARTINIANA PO","end":"9999-12-31","start":"1946-11-12"}],"A183":[{"
end":"9999-12-31","start":"1863-02-06","name":"ALESSANDRIA DEL CARRETTO"}],"
H414":[{"name":"ROCCA GRIMALDA","end":"9999-12-31","start":"1861-03-17"}],"D
919":[{"end":"1869-01-31","start":"1861-03-17","name":"GAREGNANO"}],"D616":[
{"name":"FISTO","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-1
4","end":"1928-10-29","name":"FISTO"}],"G518":[{"name":"PETRON\u00c0","start
":"1861-03-17","end":"9999-12-31"}],"C845":[{"start":"1861-03-17","end":"999
9-12-31","name":"COLLAZZONE"}],"C936":[{"name":"COMUNE DELLA VALSOLDA","end"
:"1928-06-18","start":"1927-06-09"},{"name":"VALSOLDA","start":"1928-06-19",
"end":"1936-10-23"},{"name":"VALSOLDA","end":"9999-12-31","start":"1936-10-2
4"}],"C639":[{"end":"9999-12-31","start":"1861-03-17","name":"CHIOMONTE"}],"
B157":[{"start":"1880-06-25","end":"1928-11-26","name":"BRESCIA"},{"name":"B
RESCIA","start":"1928-11-27","end":"9999-12-31"}],"E585":[{"name":"LIGONCHIO
","start":"1861-03-17","end":"2015-12-31"}],"D182":[{"name":"CROSA","end":"1
927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"1992-04-15","n
ame":"CROSA"},{"end":"2015-12-31","start":"1992-04-16","name":"CROSA"}],"M36
2":[{"name":"VALLELAGHI","start":"2016-01-01","end":"9999-12-31"}],"G278":[{
"name":"PALLAGORIO","end":"1992-04-15","start":"1861-03-17"},{"name":"PALLAG
ORIO","start":"1992-04-16","end":"9999-12-31"}],"G915":[{"start":"1921-01-05
","end":"1923-02-07","name":"PORTOLE"},{"name":"PORTOLE","start":"1923-02-08
","end":"1947-09-15"}],"I042":[{"name":"SAN MICHELE ALL'ADIGE","start":"1920
-10-16","end":"1923-02-13"},{"name":"SAN MICHELE ALL'ADIGE","start":"1923-02
-14","end":"1928-05-21"},{"name":"SAN MICHELE ALL'ADIGE","end":"1953-01-03",
"start":"1928-05-22"},{"end":"9999-12-31","start":"1953-01-04","name":"SAN M
ICHELE ALL'ADIGE"}],"E171":[{"end":"1928-05-01","start":"1866-11-19","name":
"GREZZANA"},{"name":"GREZZANA","start":"1928-05-02","end":"1947-10-04"},{"na
me":"GREZZANA","end":"9999-12-31","start":"1947-10-05"}],"A570":[{"start":"1
862-12-19","end":"9999-12-31","name":"BAGNOLO CREMASCO"}],"G811":[{"name":"P
OMEZIA","end":"1970-05-26","start":"1938-07-27"},{"name":"POMEZIA","start":"
1970-05-27","end":"9999-12-31"}],"F678":[{"end":"1935-04-07","start":"1861-0
3-17","name":"MONTIGLIO"},{"start":"1935-04-08","end":"1998-08-31","name":"M
ONTIGLIO"}],"B555":[{"end":"9999-12-31","start":"1861-03-17","name":"CAMPORA
"}],"L635":[{"start":"1861-03-17","end":"1927-01-11","name":"VALMAGGIA"},{"e
nd":"1929-03-01","start":"1927-01-12","name":"VALMAGGIA"}],"E187":[{"name":"
GRIZZANA","start":"1882-12-25","end":"1985-05-10"},{"name":"GRIZZANA MORANDI
","start":"1985-05-11","end":"9999-12-31"}],"L849":[{"name":"VICONAGO","end"
:"1908-07-21","start":"1893-12-05"},{"start":"1908-07-22","end":"1927-01-11"
,"name":"VICONAGO"},{"name":"VICONAGO","start":"1927-01-12","end":"1928-08-1
5"}],"F992":[{"end":"1927-01-11","start":"1861-03-17","name":"OCCHIEPPO INFE
RIORE"},{"start":"1927-01-12","end":"1992-04-15","name":"OCCHIEPPO INFERIORE
"},{"start":"1992-04-16","end":"9999-12-31","name":"OCCHIEPPO INFERIORE"}],"
C976":[{"start":"1861-03-17","end":"9999-12-31","name":"CONZA DELLA CAMPANIA
"}],"Z723":[{"name":"Polinesia francese","end":"9999-12-31","start":"0000-01
-01"}],"C679":[{"end":"1927-01-11","start":"1861-03-17","name":"CICONIO"},{"
start":"1927-01-12","end":"1929-06-15","name":"CICONIO"},{"start":"1947-01-1
0","end":"9999-12-31","name":"CICONIO"}],"E445":[{"end":"9999-12-31","start"
:"1863-01-05","name":"LANZO TORINESE"}],"B208":[{"start":"1861-03-17","end":
"1927-01-11","name":"BROVIDA"},{"end":"1929-04-12","start":"1927-01-12","nam
e":"BROVIDA"}],"I655":[{"end":"9999-12-31","start":"1861-03-17","name":"SERR
ASTRETTA"}],"C574":[{"end":"1992-04-15","start":"1861-03-17","name":"CESENAT
ICO"},{"name":"CESENATICO","end":"9999-12-31","start":"1992-04-16"}],"B015":
[{"name":"BORGOFRANCO D'IVREA","start":"1894-02-02","end":"1927-01-11"},{"na
me":"BORGOFRANCO D'IVREA","start":"1927-01-12","end":"1929-03-01"},{"name":"
BORGOFRANCO D'IVREA","end":"1945-10-04","start":"1929-03-02"},{"start":"1945
-10-05","end":"1954-08-23","name":"BORGOFRANCO D'IVREA"},{"start":"1954-08-2
4","end":"9999-12-31","name":"BORGOFRANCO D'IVREA"}],"C006":[{"start":"1861-
03-17","end":"9999-12-31","name":"CASSARO"}],"F080":[{"start":"1861-03-17","
end":"1928-09-05","name":"MEDE"},{"start":"1928-09-06","end":"9999-12-31","n
ame":"MEDE"}],"H653":[{"name":"SABLE GRANDE","end":"1927-01-11","start":"192
3-04-27"},{"name":"SABLE GRANDE","start":"1927-01-12","end":"1928-03-12"},{"
start":"1921-01-05","end":"1923-02-07","name":"VELICHE-ZABLIE"},{"name":"VEL
ICHE-ZABLIE","start":"1923-02-08","end":"1923-04-26"}],"H917":[{"name":"SAN 
GIOVANNI INCARICO","end":"1927-01-11","start":"1861-03-17"},{"name":"SAN GIO
VANNI INCARICO","start":"1927-01-12","end":"1929-06-17"},{"name":"SAN GIOVAN
NI INCARICO","start":"1929-06-18","end":"9999-12-31"}],"H314":[{"end":"9999-
12-31","start":"1861-03-17","name":"RIPALTA ARPINA"}],"M109":[{"name":"VOGHE
RA","start":"1861-03-17","end":"1959-01-06"},{"name":"VOGHERA","end":"9999-1
2-31","start":"1959-01-07"}],"E131":[{"end":"1928-04-27","start":"1861-03-17
","name":"GRAGNANO"},{"name":"GRAGNANO","end":"1979-01-13","start":"1928-04-
28"},{"name":"GRAGNANO","end":"9999-12-31","start":"1979-01-14"}],"C214":[{"
start":"1862-12-19","end":"1929-03-25","name":"CASTELNUOVO DI CEVA"},{"start
":"1947-03-30","end":"9999-12-31","name":"CASTELNUOVO DI CEVA"}],"E600":[{"e
nd":"9999-12-31","start":"1872-11-01","name":"LINAROLO"}],"F638":[{"end":"19
29-06-12","start":"1861-03-17","name":"MONTESCANO"},{"end":"9999-12-31","sta
rt":"1948-12-26","name":"MONTESCANO"}],"A715":[{"name":"BATE","start":"1921-
01-05","end":"1923-02-07"},{"start":"1923-02-08","end":"1923-04-26","name":"
BATE"},{"end":"1927-01-11","start":"1923-04-27","name":"BATTAGLIA DELLA BAIN
SIZZA"},{"start":"1927-01-12","end":"1928-08-15","name":"BATTAGLIA DELLA BAI
NSIZZA"}],"A530":[{"start":"1867-09-02","end":"1968-04-05","name":"AZZANO DE
CIMO"},{"name":"AZZANO DECIMO","end":"9999-12-31","start":"1968-04-06"}],"B8
64":[{"name":"CASALBELTRAME","start":"1861-03-17","end":"9999-12-31"}],"E146
":[{"end":"9999-12-31","start":"1913-06-14","name":"GRANZE"}],"I262":[{"end"
:"9999-12-31","start":"1861-03-17","name":"SANT'ANASTASIA"}],"B958":[{"end":
"1984-09-26","start":"1863-04-13","name":"CASELLE LURANI"},{"name":"CASELLE 
LURANI","start":"1984-09-27","end":"1992-04-15"},{"start":"1992-04-16","end"
:"9999-12-31","name":"CASELLE LURANI"}],"I869":[{"name":"SOTTO IL MONTE","en
d":"1964-01-27","start":"1861-03-17"},{"name":"SOTTO IL MONTE GIOVANNI XXIII
","end":"9999-12-31","start":"1964-01-28"}],"D691":[{"name":"FORCE","end":"9
999-12-31","start":"1861-03-17"}],"L103":[{"end":"1929-06-17","start":"1861-
03-17","name":"TERAMO"},{"name":"TERAMO","end":"9999-12-31","start":"1929-06
-18"}],"G238":[{"end":"1928-05-13","start":"1866-11-19","name":"PAGNACCO"},{
"start":"1947-01-09","end":"1968-04-05","name":"PAGNACCO"},{"end":"9999-12-3
1","start":"1968-04-06","name":"PAGNACCO"}],"E731":[{"end":"9999-12-31","sta
rt":"1868-02-14","name":"LUGO DI VICENZA"}],"I188":[{"name":"SANTA FLAVIA","
end":"9999-12-31","start":"1880-03-19"}],"B489":[{"end":"1901-04-30","start"
:"1861-03-17","name":"CAMO"},{"start":"1901-05-01","end":"1928-06-13","name"
:"CAMO"},{"name":"CAMO","start":"1947-05-23","end":"2018-12-31"}],"H970":[{"
name":"SAN LUCA","end":"9999-12-31","start":"1861-03-17"}],"C902":[{"name":"
COLONNO","start":"1861-03-17","end":"1926-02-06"},{"name":"COLONNO","start":
"1926-02-07","end":"1928-11-02"},{"end":"9999-12-31","start":"1950-04-05","n
ame":"COLONNO"}],"C665":[{"name":"CHIVASSO","start":"1861-03-17","end":"9999
-12-31"}],"H352":[{"name":"RIVIGNANO","start":"1866-11-19","end":"1968-04-05
"},{"name":"RIVIGNANO","start":"1968-04-06","end":"2013-12-31"}],"M284":[{"e
nd":"9999-12-31","start":"1983-02-27","name":"SANTA MARIA COGHINAS"}],"A115"
:[{"start":"1864-01-14","end":"2005-12-31","name":"AL\u00c0 DEI SARDI"},{"en
d":"2016-04-27","start":"2006-01-01","name":"AL\u00c0 DEI SARDI"},{"name":"A
L\u00c0 DEI SARDI","end":"9999-12-31","start":"2016-04-28"}],"E273":[{"name"
:"H\u00d4NE","start":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12",
"end":"1928-06-29","name":"H\u00d4NE"},{"name":"H\u00d4NE","end":"9999-12-31
","start":"1946-03-10"}],"A237":[{"end":"9999-12-31","start":"1866-11-19","n
ame":"ALTIVOLE"}],"E746":[{"name":"LUOGOSANO","end":"9999-12-31","start":"18
61-03-17"}],"I484":[{"start":"1861-03-17","end":"1928-03-28","name":"SCAGNEL
LO"},{"name":"SCAGNELLO","start":"1947-01-10","end":"9999-12-31"}],"D048":[{
"name":"CORTACCIA","end":"1923-02-13","start":"1920-10-16"},{"name":"CORTACC
IA","end":"1948-03-13","start":"1923-02-14"},{"name":"CORTACCIA","end":"1971
-09-21","start":"1948-03-14"},{"end":"1978-01-31","start":"1971-09-22","name
":"CORTACCIA SULLA STRADA DEL VINO"},{"end":"1990-11-27","start":"1978-02-01
","name":"CORTACCIA SULLA STRADA DEL VINO"},{"name":"CORTACCIA SULLA STRADA 
DEL VINO","start":"1990-11-28","end":"9999-12-31"}],"F096":[{"start":"1861-0
3-17","end":"9999-12-31","name":"MELAZZO"}],"L094":[{"name":"TEM\u00d9","end
":"1927-12-03","start":"1861-03-17"},{"name":"TEM\u00d9","start":"1927-12-04
","end":"9999-12-31"}],"C072":[{"end":"9999-12-31","start":"1861-03-17","nam
e":"CASTELCOVATI"}],"L703":[{"end":"1927-01-11","start":"1863-02-23","name":
"VEDANO OLONA"},{"end":"9999-12-31","start":"1927-01-12","name":"VEDANO OLON
A"}],"A391":[{"start":"1861-03-17","end":"9999-12-31","name":"ARGEGNO"}],"F4
67":[{"end":"1998-09-03","start":"1861-03-17","name":"MONTE CERIGNONE"},{"na
me":"MONTE CERIGNONE","start":"1998-09-04","end":"9999-12-31"}],"G880":[{"st
art":"1920-10-16","end":"1923-02-13","name":"POR"},{"name":"POR","end":"1928
-03-06","start":"1923-02-14"}],"A222":[{"name":"ALPIGNANO","start":"1861-03-
17","end":"9999-12-31"}],"E787":[{"name":"MACLODIO","start":"1861-03-17","en
d":"1927-11-23"},{"name":"MACLODIO","end":"9999-12-31","start":"1947-08-15"}
],"I971":[{"start":"1928-06-15","end":"1945-10-04","name":"STRAMBINO ROMANO"
},{"name":"STRAMBINO ROMANO","end":"1946-12-24","start":"1945-10-05"}],"D223
":[{"name":"CURSI","start":"1861-03-17","end":"9999-12-31"}],"E359":[{"name"
:"ISOLA SAN GIULIO","end":"1928-08-11","start":"1861-03-17"},{"name":"ISOLA 
SAN GIULIO","end":"1928-12-13","start":"1928-08-12"}],"A281":[{"name":"ANDRA
NO","start":"1861-03-17","end":"9999-12-31"}],"E053":[{"name":"GIUGGIANELLO"
,"end":"9999-12-31","start":"1861-03-17"}],"Z123":[{"name":"Monaco","end":"9
999-12-31","start":"0000-01-01"}],"H873":[{"name":"SAN GILLIO","end":"1928-0
4-02","start":"1861-03-17"},{"name":"SAN GILLIO","end":"9999-12-31","start":
"1947-01-21"}],"A416":[{"name":"ARMIO","end":"1927-01-11","start":"1861-03-1
7"},{"start":"1927-01-12","end":"1928-08-06","name":"ARMIO"}],"M394":[{"star
t":"2018-01-01","end":"9999-12-31","name":"CENTRO VALLE INTELVI"}],"C859":[{
"name":"COLLEGIOVE","end":"1923-03-18","start":"1861-03-17"},{"start":"1923-
03-19","end":"1927-01-11","name":"COLLEGIOVE"},{"name":"COLLEGIOVE","start":
"1927-01-12","end":"9999-12-31"}],"G983":[{"name":"PRANDAGLIO","start":"1861
-03-17","end":"1928-09-24"}],"I875":[{"start":"1861-03-17","end":"1957-01-17
","name":"SOVERIA SIMERI"},{"name":"SOVERIA SIMERI","start":"1957-01-18","en
d":"9999-12-31"}],"I001":[{"name":"SAN MARTINO AL MONTE","end":"1923-02-13",
"start":"1920-10-16"},{"name":"SAN MARTINO AL MONTE","start":"1923-02-14","e
nd":"1927-01-11"},{"name":"SAN MARTINO AL MONTE","start":"1927-01-12","end":
"1928-12-09"}],"C252":[{"name":"CASTEL RITALDI","end":"9999-12-31","start":"
1930-09-14"},{"name":"CASTEL RITALDI E SAN GIOVANNI","start":"1875-06-24","e
nd":"1927-04-08"}],"I369":[{"name":"SANTO STEFANO DI BRIGA","end":"1928-04-1
6","start":"1861-03-17"}],"I063":[{"end":"9999-12-31","start":"1868-02-14","
name":"SAN NICOL\u00d2 DI COMELICO"}],"E214":[{"end":"9999-12-31","start":"1
861-03-17","name":"GROTTOLELLA"}],"C131":[{"end":"1872-10-31","start":"1861-
03-17","name":"CASTEL LAMBRO"}],"L228":[{"name":"TORNO","end":"9999-12-31","
start":"1861-03-17"}],"F754":[{"name":"MORTARA","end":"9999-12-31","start":"
1861-03-17"}],"L756":[{"start":"1905-04-24","end":"1992-04-15","name":"VERDE
RIO SUPERIORE"},{"start":"1992-04-16","end":"2014-02-03","name":"VERDERIO SU
PERIORE"}],"H672":[{"start":"1861-03-17","end":"1927-01-11","name":"SAINT NI
COLAS"},{"start":"1946-03-10","end":"1977-01-01","name":"SAINT NICOLAS"},{"n
ame":"SAINT-NICOLAS","start":"1927-01-12","end":"1928-05-12"},{"name":"SAINT
-NICOLAS","start":"1977-01-02","end":"9999-12-31"}],"F989":[{"name":"NUVOLEN
TO","start":"1861-03-17","end":"1928-08-15"},{"name":"NUVOLENTO","start":"19
28-08-16","end":"1947-05-11"},{"name":"NUVOLENTO","end":"9999-12-31","start"
:"1947-05-12"}],"C600":[{"start":"1861-03-17","end":"9999-12-31","name":"CHE
REMULE"}],"F686":[{"end":"1925-11-29","start":"1863-04-13","name":"MONTOPOLI
 IN VAL D'ARNO"},{"start":"1925-11-30","end":"1928-02-08","name":"MONTOPOLI 
IN VAL D'ARNO"},{"start":"1928-02-09","end":"9999-12-31","name":"MONTOPOLI I
N VAL D'ARNO"}],"G411":[{"name":"PEDIVIGLIANO","end":"1928-02-22","start":"1
861-03-17"},{"name":"PEDIVIGLIANO","end":"9999-12-31","start":"1937-07-18"}]
,"D629":[{"name":"FIVIZZANO","start":"1861-03-17","end":"1919-06-26"},{"name
":"FIVIZZANO","start":"1919-06-27","end":"1919-08-21"},{"start":"1919-08-22"
,"end":"1983-07-20","name":"FIVIZZANO"},{"start":"1983-07-21","end":"2006-01
-06","name":"FIVIZZANO"},{"end":"9999-12-31","start":"2006-01-07","name":"FI
VIZZANO"}],"I736":[{"name":"SILLAVENGO","end":"1928-02-22","start":"1861-03-
17"},{"end":"9999-12-31","start":"1948-01-17","name":"SILLAVENGO"}],"D926":[
{"start":"1861-03-17","end":"1928-05-01","name":"GARLATE"},{"start":"1928-05
-02","end":"1929-04-19","name":"GARLATE"},{"name":"GARLATE","start":"1929-04
-20","end":"1953-07-23"},{"name":"GARLATE","start":"1953-07-24","end":"1992-
04-15"},{"end":"9999-12-31","start":"1992-04-16","name":"GARLATE"}],"M170":[
{"name":"ZERMEGHEDO","start":"1866-11-19","end":"1929-05-29"},{"name":"ZERME
GHEDO","end":"9999-12-31","start":"1947-07-09"}],"C146":[{"end":"1986-05-26"
,"start":"1866-11-19","name":"CASTELLAVAZZO"},{"end":"2014-02-21","start":"1
986-05-27","name":"CASTELLAVAZZO"}],"G286":[{"name":"PALMAS ARBOREA","start"
:"1862-09-29","end":"1927-11-08"},{"start":"1957-07-02","end":"1974-08-19","
name":"PALMAS ARBOREA"},{"name":"PALMAS ARBOREA","end":"9999-12-31","start":
"1974-08-20"}],"D524":[{"name":"FELETTO","end":"9999-12-31","start":"1861-03
-17"}],"B364":[{"name":"CAINES","end":"1923-02-13","start":"1920-10-16"},{"n
ame":"CAINES","end":"1927-01-11","start":"1923-02-14"},{"end":"9999-12-31","
start":"1927-01-12","name":"CAINES"}],"H065":[{"end":"1929-04-13","start":"1
861-03-17","name":"PRIMEGLIO SCHIERANO"}],"M343":[{"name":"VALDAONE","start"
:"2015-01-01","end":"9999-12-31"}],"M049":[{"name":"VILMINORE","start":"1861
-03-17","end":"1922-12-06"},{"end":"1927-07-14","start":"1922-12-07","name":
"VILMINORE"}],"B967":[{"end":"1956-11-09","start":"1861-03-17","name":"CASIN
A"},{"name":"CASINA","end":"9999-12-31","start":"1956-11-10"}],"I741":[{"sta
rt":"1861-03-17","end":"1927-12-20","name":"SILVI"},{"end":"9999-12-31","sta
rt":"1929-05-03","name":"SILVI"}],"L589":[{"name":"VALLATA","end":"9999-12-3
1","start":"1861-03-17"}],"H814":[{"name":"SAN DAMIANO AL COLLE","start":"18
78-06-14","end":"9999-12-31"}],"L761":[{"name":"VERGANO NOVARESE","start":"1
863-04-23","end":"1928-04-18"}],"G247":[{"name":"PAISCO LOVENO","end":"9999-
12-31","start":"1928-11-10"}],"G396":[{"name":"PECCO","end":"1927-01-11","st
art":"1861-03-17"},{"start":"1927-01-12","end":"1929-05-14","name":"PECCO"},
{"name":"PECCO","start":"1947-01-10","end":"2018-12-31"}],"C187":[{"name":"C
ASTELLO DI BRIANZA","start":"1928-02-10","end":"1992-04-15"},{"name":"CASTEL
LO DI BRIANZA","start":"1992-04-16","end":"9999-12-31"}],"E679":[{"end":"999
9-12-31","start":"1863-02-23","name":"LONGONE AL SEGRINO"}],"C445":[{"name":
"CELLEDIZZO","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02-14",
"end":"1928-08-11","name":"CELLEDIZZO"}],"H197":[{"name":"RAVELLINO","end":"
1928-02-05","start":"1864-01-29"}],"E976":[{"start":"1861-03-17","end":"1973
-06-24","name":"MARSICO NUOVO"},{"start":"1973-06-25","end":"9999-12-31","na
me":"MARSICO NUOVO"}],"F225":[{"name":"MINUCCIANO","start":"1861-03-17","end
":"1923-11-23"},{"start":"1923-11-24","end":"9999-12-31","name":"MINUCCIANO"
}],"F647":[{"name":"MONTESILVANO MARINA","start":"1922-07-09","end":"1927-01
-11"},{"name":"MONTESILVANO MARINA","end":"1928-11-26","start":"1927-01-12"}
],"D708":[{"name":"FORMIA","end":"1927-01-11","start":"1862-03-28"},{"start"
:"1927-01-12","end":"1928-06-13","name":"FORMIA"},{"end":"1934-10-25","start
":"1928-06-14","name":"FORMIA"},{"name":"FORMIA","end":"1934-12-17","start":
"1934-10-26"},{"start":"1934-12-18","end":"1945-04-08","name":"FORMIA"},{"en
d":"9999-12-31","start":"1945-04-09","name":"FORMIA"}],"E574":[{"start":"186
9-04-01","end":"1933-10-05","name":"LICCIANA"},{"name":"LICCIANA NARDI","end
":"9999-12-31","start":"1933-10-06"}],"G597":[{"start":"1862-12-29","end":"9
999-12-31","name":"PIEDIMONTE ETNEO"}],"B123":[{"start":"1861-03-17","end":"
9999-12-31","name":"BRANZI"}],"E006":[{"end":"9999-12-31","start":"1861-03-1
7","name":"GHISALBA"}],"F112":[{"start":"1864-06-23","end":"1977-09-16","nam
e":"MELITO DI PORTO SALVO"},{"end":"9999-12-31","start":"1977-09-17","name":
"MELITO DI PORTO SALVO"}],"G625":[{"start":"1861-03-17","end":"1928-07-28","
name":"PIETRAPORZIO"},{"name":"PIETRAPORZIO","start":"1947-01-10","end":"999
9-12-31"}],"H740":[{"name":"SAMBASSO","end":"1927-01-11","start":"1923-04-27
"},{"start":"1927-01-12","end":"1928-01-06","name":"SAMBASSO"},{"name":"SAMB
ASSO","start":"1928-01-07","end":"1947-09-15"},{"name":"SEMPAS","end":"1923-
02-07","start":"1921-01-05"},{"start":"1923-02-08","end":"1923-04-26","name"
:"SEMPAS"}],"L349":[{"name":"TREBASELEGHE","end":"9999-12-31","start":"1866-
11-19"}],"L043":[{"name":"TAPIGLIANO","start":"1861-03-17","end":"1928-05-02
"}],"B492":[{"end":"9999-12-31","start":"1861-03-17","name":"CAMPAGNA"}],"L1
61":[{"end":"1947-07-09","start":"1928-02-23","name":"TIARNO"}],"M117":[{"st
art":"1921-01-05","end":"1923-02-07","name":"VOLOSCA-ABBAZIA"},{"name":"VOLO
SCA-ABBAZIA","end":"1924-03-08","start":"1923-02-08"},{"name":"VOLOSCA-ABBAZ
IA","start":"1924-03-09","end":"1931-12-21"}],"A598":[{"start":"1879-01-24",
"end":"1927-01-11","name":"BALLARATE"},{"start":"1927-01-12","end":"1928-01-
07","name":"BALLARATE"}],"D404":[{"name":"ENDENNA","start":"1861-03-17","end
":"1928-03-26"}],"G540":[{"end":"1929-02-09","start":"1873-01-28","name":"VA
LLE DELL'ANGELO"},{"start":"1945-09-22","end":"9999-12-31","name":"VALLE DEL
L'ANGELO"}],"F690":[{"start":"1894-03-05","end":"1929-06-17","name":"MONTORI
O AL VOMANO"},{"name":"MONTORIO AL VOMANO","start":"1929-06-18","end":"9999-
12-31"}],"C787":[{"start":"1861-03-17","end":"9999-12-31","name":"CLAINO CON
 OSTENO"}],"H606":[{"end":"9999-12-31","start":"1866-11-19","name":"ROVERCHI
ARA"}],"H797":[{"end":"9999-12-31","start":"1864-12-26","name":"SAN CIPIRELL
O"}],"C359":[{"name":"CAUTANO","start":"1861-03-17","end":"9999-12-31"}],"H9
09":[{"name":"SAN GIOVANNI BATTISTA","end":"1923-11-28","start":"1861-03-17"
}],"C053":[{"start":"1861-03-17","end":"9999-12-31","name":"CASTEGGIO"}],"D0
37":[{"end":"9999-12-31","start":"1861-03-17","name":"CORREGGIO"}],"H433":[{
"start":"1861-03-17","end":"1927-01-11","name":"ROCCARAINOLA"},{"end":"1928-
09-07","start":"1927-01-12","name":"ROCCARAINOLA"},{"name":"ROCCARAINOLA","s
tart":"1928-09-08","end":"1946-07-02"},{"end":"9999-12-31","start":"1946-07-
03","name":"ROCCARAINOLA"}],"I375":[{"name":"SANT'URBANO","end":"9999-12-31"
,"start":"1866-11-19"}],"E859":[{"name":"MALITO","start":"1861-03-17","end":
"1928-05-21"},{"name":"MALITO","end":"1937-07-17","start":"1928-05-22"},{"na
me":"MALITO","start":"1937-07-18","end":"9999-12-31"}],"D108":[{"name":"COST
ACCIARO","start":"1861-03-17","end":"9999-12-31"}],"Z208":[{"end":"9999-12-3
1","start":"0000-01-01","name":"Cambogia"}],"G290":[{"end":"9999-12-31","sta
rt":"1861-03-17","name":"PALMOLI"}],"A248":[{"start":"1861-03-17","end":"192
7-06-06","name":"ALZANO SOPRA"}],"H373":[{"name":"ROBECCO SUL NAVIGLIO","end
":"9999-12-31","start":"1870-08-02"}],"I435":[{"name":"SARMEDE","end":"9999-
12-31","start":"1866-11-19"}],"B723":[{"end":"1923-02-13","start":"1920-10-1
6","name":"CARANO"},{"start":"1923-02-14","end":"1928-05-12","name":"CARANO"
},{"name":"CARANO","start":"1946-12-26","end":"2019-12-31"}],"H079":[{"name"
:"PROVAGLIO SOPRA","end":"1928-06-14","start":"1861-03-17"}],"F712":[{"name"
:"MORBEGNO","end":"1938-07-08","start":"1861-03-17"},{"end":"1978-02-11","st
art":"1938-07-09","name":"MORBEGNO"},{"name":"MORBEGNO","start":"1978-02-12"
,"end":"2006-08-08"},{"name":"MORBEGNO","start":"2006-08-09","end":"9999-12-
31"}],"L455":[{"name":"TUBRE","start":"1920-10-16","end":"1923-02-13"},{"nam
e":"TUBRE","start":"1923-02-14","end":"1927-01-11"},{"name":"TUBRE","end":"9
999-12-31","start":"1927-01-12"}],"H140":[{"name":"QUINZANO D'OGLIO","start"
:"1862-11-07","end":"9999-12-31"}],"C513":[{"name":"CERIGNALE","end":"1923-0
7-22","start":"1861-03-17"},{"name":"CERIGNALE","end":"9999-12-31","start":"
1923-07-23"}],"D022":[{"name":"CORNELIANO D'ALBA","start":"1862-12-19","end"
:"1928-06-13"},{"name":"CORNELIANO D'ALBA","start":"1928-06-14","end":"1946-
11-11"},{"name":"CORNELIANO D'ALBA","start":"1946-11-12","end":"9999-12-31"}
],"E252":[{"name":"GUASILA","start":"1861-03-17","end":"1974-08-19"},{"start
":"1974-08-20","end":"2016-04-27","name":"GUASILA"},{"end":"9999-12-31","sta
rt":"2016-04-28","name":"GUASILA"}],"C731":[{"name":"CISERIIS","end":"1928-0
4-29","start":"1866-11-19"}],"A891":[{"start":"1861-03-17","end":"1927-01-11
","name":"BISUSCHIO"},{"end":"9999-12-31","start":"1927-01-12","name":"BISUS
CHIO"}],"F154":[{"start":"1861-03-17","end":"1927-01-11","name":"MESENZANA"}
,{"end":"1928-01-06","start":"1927-01-12","name":"MESENZANA"},{"name":"MESEN
ZANA","start":"1953-07-21","end":"9999-12-31"}],"L156":[{"start":"1866-11-19
","end":"1939-11-16","name":"TEZZE"},{"end":"9999-12-31","start":"1939-11-17
","name":"TEZZE SUL BRENTA"}],"E902":[{"end":"9999-12-31","start":"1861-03-1
7","name":"MARA"}],"E665":[{"end":"1927-01-11","start":"1861-03-17","name":"
LONATE CEPPINO"},{"end":"9999-12-31","start":"1927-01-12","name":"LONATE CEP
PINO"}],"A023":[{"end":"9999-12-31","start":"1861-03-17","name":"ACERNO"}],"
D081":[{"start":"1877-03-09","end":"1928-10-15","name":"CORVINO SAN QUIRICO"
},{"name":"CORVINO SAN QUIRICO","end":"1946-12-18","start":"1928-10-16"},{"e
nd":"9999-12-31","start":"1946-12-19","name":"CORVINO SAN QUIRICO"}],"A329":
[{"name":"APIRO","start":"1861-03-17","end":"1929-04-10"},{"name":"APIRO","s
tart":"1929-04-11","end":"1948-01-14"},{"name":"APIRO","start":"1948-01-15",
"end":"9999-12-31"}],"C273":[{"end":"1927-01-11","start":"1861-03-17","name"
:"CASTELSEPRIO"},{"start":"1927-01-12","end":"1928-06-14","name":"CASTELSEPR
IO"},{"start":"1947-06-17","end":"9999-12-31","name":"CASTELSEPRIO"}],"I136"
:[{"name":"SAN QUIRINO","end":"1968-04-05","start":"1866-11-19"},{"name":"SA
N QUIRINO","end":"9999-12-31","start":"1968-04-06"}],"F458":[{"end":"1929-04
-11","start":"1869-06-11","name":"MONTECATINI VAL DI CECINA"},{"end":"9999-1
2-31","start":"1929-04-12","name":"MONTECATINI VAL DI CECINA"}],"C746":[{"na
me":"CITTADUCALE","end":"1927-01-11","start":"1861-03-17"},{"name":"CITTADUC
ALE","end":"9999-12-31","start":"1927-01-12"}],"H259":[{"start":"1928-06-30"
,"end":"1929-04-10","name":"REZZO CAVARGNA"},{"start":"1929-04-11","end":"99
99-12-31","name":"VAL REZZO"}],"A995":[{"end":"1992-04-15","start":"1863-01-
26","name":"BORGHETTO LODIGIANO"},{"end":"9999-12-31","start":"1992-04-16","
name":"BORGHETTO LODIGIANO"}],"E072":[{"end":"2012-06-29","start":"1861-03-1
7","name":"GODIASCO"},{"name":"GODIASCO SALICE TERME","end":"9999-12-31","st
art":"2012-06-30"}],"H852":[{"name":"SANFRONT","start":"1861-03-17","end":"1
940-08-09"},{"name":"SANFRONT","end":"9999-12-31","start":"1951-11-18"},{"en
d":"1951-11-17","start":"1940-08-10","name":"SANFRONTE"}],"H333":[{"start":"
1861-03-17","end":"9999-12-31","name":"RIVALBA"}],"I475":[{"start":"1893-08-
30","end":"9999-12-31","name":"SAVIGNONE"}],"H039":[{"start":"1920-10-16","e
nd":"1923-02-13","name":"PREORE"},{"start":"1923-02-14","end":"1928-02-08","
name":"PREORE"},{"name":"PREORE","start":"1952-05-17","end":"2015-12-31"}],"
D907":[{"name":"GANGI","end":"9999-12-31","start":"1861-03-17"}],"D799":[{"s
tart":"1929-04-12","end":"1945-08-31","name":"FRIGNANO"},{"name":"FRIGNANO",
"start":"1945-09-01","end":"1946-11-08"},{"start":"1952-03-01","end":"9999-1
2-31","name":"FRIGNANO"}],"C459":[{"end":"9999-12-31","start":"1861-03-17","
name":"CENE"}],"A816":[{"name":"BERZO DEMO","end":"1927-12-26","start":"1861
-03-17"},{"name":"BERZO DEMO","end":"9999-12-31","start":"1948-12-21"}],"D30
4":[{"name":"DIPIGNANO","start":"1861-03-17","end":"9999-12-31"}],"B392":[{"
name":"CALCINAIA","start":"1861-03-17","end":"1928-11-06"},{"end":"1929-06-1
7","start":"1928-11-07","name":"CALCINAIA"},{"end":"2007-04-03","start":"192
9-06-18","name":"CALCINAIA"},{"start":"2007-04-04","end":"9999-12-31","name"
:"CALCINAIA"}],"B221":[{"name":"BRUNO","end":"1935-04-07","start":"1861-03-1
7"},{"end":"9999-12-31","start":"1935-04-08","name":"BRUNO"}],"L355":[{"star
t":"1861-03-17","end":"9999-12-31","name":"TRECASTAGNI"}],"M275":[{"name":"L
OIRI PORTO SAN PAOLO","end":"2005-12-31","start":"1979-04-20"},{"name":"LOIR
I PORTO SAN PAOLO","end":"2016-04-27","start":"2006-01-01"},{"start":"2016-0
4-28","end":"9999-12-31","name":"LOIRI PORTO SAN PAOLO"}],"A068":[{"name":"A
GEROLA","end":"9999-12-31","start":"1861-03-17"}],"I702":[{"start":"1929-05-
04","end":"1945-10-04","name":"SETTIMO TAVAGNASCO"},{"name":"SETTIMO TAVAGNA
SCO","end":"1947-01-14","start":"1945-10-05"},{"name":"SETTIMO TAVAGNASCO","
start":"1947-01-15","end":"1950-10-04"},{"name":"SETTIMO VITTONE","end":"192
7-01-11","start":"1861-03-17"},{"name":"SETTIMO VITTONE","end":"1929-05-03",
"start":"1927-01-12"},{"name":"SETTIMO VITTONE","end":"9999-12-31","start":"
1950-10-05"}],"D077":[{"start":"1861-03-17","end":"9999-12-31","name":"CORTO
NA"}],"H473":[{"name":"RODDINO","end":"9999-12-31","start":"1861-03-17"}],"I
335":[{"name":"SANT'EUSANIO DEL SANGRO","end":"9999-12-31","start":"1863-02-
06"}],"I884":[{"name":"SPADOLA","start":"1861-03-17","end":"1992-04-15"},{"n
ame":"SPADOLA","start":"1992-04-16","end":"9999-12-31"}],"A429":[{"name":"AR
ONA","end":"1928-02-25","start":"1861-03-17"},{"end":"1960-09-28","start":"1
928-02-26","name":"ARONA"},{"end":"9999-12-31","start":"1960-09-29","name":"
ARONA"}],"F867":[{"end":"1974-08-19","start":"1861-03-17","name":"NEONELI"},
{"start":"1974-08-20","end":"9999-12-31","name":"NEONELI"}],"G480":[{"name":
"PESCAGLIA","end":"1962-03-09","start":"1861-03-17"},{"name":"PESCAGLIA","st
art":"1962-03-10","end":"9999-12-31"}],"B768":[{"end":"1927-01-11","start":"
1861-03-17","name":"CARESANABLOT"},{"name":"CARESANABLOT","start":"1927-01-1
2","end":"9999-12-31"}],"E032":[{"name":"GIMINO","end":"1923-02-07","start":
"1921-01-05"},{"name":"GIMINO","start":"1923-02-08","end":"1947-09-15"}],"F1
20":[{"name":"MENAGGIO","start":"1861-03-17","end":"1928-06-25"},{"start":"1
928-06-26","end":"1929-05-13","name":"MENAGGIO"},{"name":"MENAGGIO","start":
"1929-05-14","end":"9999-12-31"}],"G795":[{"start":"1877-04-09","end":"9999-
12-31","name":"POLLENA TROCCHIA"}],"M055":[{"end":"1947-09-15","start":"1861
-03-17","name":"VINADIO"},{"name":"VINADIO","end":"9999-12-31","start":"1947
-09-16"}],"C771":[{"name":"CIVITAQUANA","end":"1927-01-11","start":"1861-03-
17"},{"name":"CIVITAQUANA","end":"9999-12-31","start":"1927-01-12"}],"B889":
[{"start":"1861-03-17","end":"1938-02-18","name":"CASALETTO CEREDANO"},{"nam
e":"CASALETTO CEREDANO","end":"9999-12-31","start":"1938-02-19"}],"B282":[{"
start":"1861-03-17","end":"9999-12-31","name":"BUSALLA"}],"I176":[{"end":"99
99-12-31","start":"1864-06-23","name":"SANTA CRISTINA D'ASPROMONTE"}],"E027"
:[{"start":"1861-03-17","end":"9999-12-31","name":"GIFFONI VALLE PIANA"}],"F
358":[{"end":"1930-12-31","start":"1862-12-19","name":"MONFORTE D'ALBA"},{"n
ame":"MONFORTE D'ALBA","start":"1931-01-01","end":"9999-12-31"}],"D257":[{"n
ame":"DAVOLI","start":"1861-03-17","end":"9999-12-31"}],"H627":[{"end":"9999
-12-31","start":"1861-03-17","name":"RUBIANA"}],"M130":[{"start":"1862-12-29
","end":"1929-05-13","name":"VOLTURARA IRPINA"},{"name":"VOLTURARA IRPINA","
start":"1929-05-14","end":"9999-12-31"}],"G311":[{"end":"9999-12-31","start"
:"1861-03-17","name":"PANNARANO"}],"I243":[{"name":"SANTA MARIA HO\u00c8","e
nd":"1928-12-12","start":"1861-03-17"},{"name":"SANTA MARIA HO\u00c8","end":
"1992-04-15","start":"1953-09-13"},{"start":"1992-04-16","end":"9999-12-31",
"name":"SANTA MARIA HO\u00c8"}],"C585":[{"end":"1927-12-03","start":"1861-03
-17","name":"CETO"},{"name":"CETO","start":"1947-06-18","end":"9999-12-31"}]
,"D010":[{"start":"1863-01-19","end":"9999-12-31","name":"CORLETO PERTICARA"
}],"I469":[{"start":"1867-08-26","end":"1927-01-11","name":"SAVIANO"},{"name
":"SAVIANO","start":"1927-01-12","end":"9999-12-31"}],"B168":[{"start":"1861
-03-17","end":"1928-02-10","name":"BRIANZOLA"}],"F720":[{"start":"1861-03-17
","end":"9999-12-31","name":"MORENGO"}],"H632":[{"name":"RUFFANO","start":"1
861-03-17","end":"1963-03-29"},{"name":"RUFFANO","start":"1963-03-30","end":
"9999-12-31"}],"I593":[{"name":"SELVA DEI MOLINI","end":"1923-02-13","start"
:"1920-10-16"},{"start":"1923-02-14","end":"1927-01-11","name":"SELVA DEI MO
LINI"},{"start":"1927-01-12","end":"1928-06-18","name":"SELVA DEI MOLINI"},{
"start":"1928-06-19","end":"9999-12-31","name":"SELVA DEI MOLINI"}],"G195":[
{"end":"1923-07-22","start":"1861-03-17","name":"OTTONE"},{"name":"OTTONE","
end":"9999-12-31","start":"1923-07-23"}],"C171":[{"start":"1928-08-12","end"
:"1947-01-14","name":"CASTELLI CUSIANI"},{"name":"CASTELLI CUSIANI","end":"1
948-05-07","start":"1947-01-15"}],"I776":[{"end":"1926-07-03","start":"1861-
03-17","name":"SOCCAVO"}],"H245":[{"start":"1861-03-17","end":"9999-12-31","
name":"RESUTTANO"}],"D550":[{"start":"1861-03-17","end":"1928-05-12","name":
"FERRAZZANO"},{"name":"FERRAZZANO","start":"1947-05-22","end":"1970-03-02"},
{"name":"FERRAZZANO","end":"9999-12-31","start":"1970-03-03"}],"B693":[{"sta
rt":"1861-03-17","end":"1913-03-19","name":"CAPRESE"},{"start":"1913-03-20",
"end":"9999-12-31","name":"CAPRESE MICHELANGELO"}],"M269":[{"name":"PATERNO"
,"start":"1973-06-25","end":"9999-12-31"}],"G067":[{"end":"1923-11-28","star
t":"1861-03-17","name":"ONEGLIA"}],"C345":[{"name":"CASTRONUOVO DI SANT'ANDR
EA","start":"1863-05-06","end":"9999-12-31"}],"D199":[{"name":"CUGLIATE-FABI
ASCO","end":"9999-12-31","start":"1955-08-19"}],"E534":[{"end":"1923-02-13",
"start":"1920-10-16","name":"LENZUMO"},{"name":"LENZUMO","end":"1928-02-08",
"start":"1923-02-14"}],"G496":[{"name":"PESCOPAGANO","start":"1861-03-17","e
nd":"9999-12-31"}],"E941":[{"end":"1928-01-17","start":"1861-03-17","name":"
MARENTINO"},{"name":"MARENTINO","end":"9999-12-31","start":"1928-01-18"}],"A
604":[{"end":"9999-12-31","start":"1861-03-17","name":"BALVANO"}],"C988":[{"
name":"CORCHIANO","start":"1871-01-15","end":"1927-01-11"},{"end":"9999-12-3
1","start":"1927-01-12","name":"CORCHIANO"}],"L449":[{"name":"TROMELLO","sta
rt":"1861-03-17","end":"1927-05-24"},{"end":"9999-12-31","start":"1927-05-25
","name":"TROMELLO"}],"I102":[{"name":"SAN PIETRO DI CARID\u00c0","start":"1
928-05-04","end":"9999-12-31"}],"A509":[{"name":"AVELLINO","end":"1938-08-31
","start":"1861-03-17"},{"name":"AVELLINO","start":"1938-09-01","end":"9999-
12-31"}],"I028":[{"name":"SAN MAURO CASTELVERDE","end":"9999-12-31","start":
"1863-04-10"}],"L263":[{"name":"TORRE DE' PASSERI","start":"1861-03-17","end
":"1927-01-11"},{"name":"TORRE DE' PASSERI","end":"9999-12-31","start":"1927
-01-12"}],"E639":[{"end":"9999-12-31","start":"1862-11-03","name":"LOCATE DI
 TRIULZI"}],"E936":[{"end":"9999-12-31","start":"1866-11-19","name":"MARCON"
}],"H998":[{"name":"SAN MARTINO DI PERRERO","start":"1863-01-26","end":"1928
-04-30"}],"H808":[{"end":"9999-12-31","start":"1863-01-19","name":"SAN COSTA
NTINO ALBANESE"}],"E864":[{"end":"9999-12-31","start":"1866-11-19","name":"M
ALO"}],"G088":[{"name":"ORBETELLO","start":"1861-03-17","end":"1960-09-28"},
{"name":"ORBETELLO","start":"1960-09-29","end":"9999-12-31"}],"I500":[{"end"
:"1929-05-03","start":"1861-03-17","name":"SCANNABUE"}],"B600":[{"end":"1923
-02-13","start":"1920-10-16","name":"CANEZZA"},{"start":"1923-02-14","end":"
1929-01-24","name":"CANEZZA"}],"D840":[{"name":"GADESCO","start":"1866-01-14
","end":"1928-05-18"}],"I312":[{"start":"1862-09-22","end":"2005-12-31","nam
e":"SANTA TERESA GALLURA"},{"end":"2016-04-27","start":"2006-01-01","name":"
SANTA TERESA GALLURA"},{"name":"SANTA TERESA GALLURA","start":"2016-04-28","
end":"9999-12-31"}],"B131":[{"start":"1861-03-17","end":"1927-01-11","name":
"BREGANO"},{"name":"BREGANO","start":"1927-01-12","end":"2022-12-31"}],"F402
":[{"name":"MONTALCINO","start":"1861-03-17","end":"2016-12-31"}],"A942":[{"
name":"NOVA SIRI","end":"1927-01-11","start":"1872-03-21"},{"end":"9999-12-3
1","start":"1927-01-12","name":"NOVA SIRI"}],"D714":[{"name":"FORNACE","end"
:"1923-02-13","start":"1920-10-16"},{"name":"FORNACE","end":"9999-12-31","st
art":"1923-02-14"}],"C967":[{"name":"CONTARINA","end":"1929-01-13","start":"
1873-04-10"},{"name":"CONTARINA","start":"1938-02-10","end":"1956-08-08"},{"
name":"CONTARINA","start":"1956-08-09","end":"1994-12-31"}],"L026":[{"name":
"TAGLIO DI PO","start":"1866-11-19","end":"1929-01-12"},{"name":"TAGLIO DI P
O","start":"1929-01-13","end":"1936-09-28"},{"end":"1938-02-09","start":"193
6-09-29","name":"TAGLIO DI PO"},{"end":"9999-12-31","start":"1938-02-10","na
me":"TAGLIO DI PO"}],"F024":[{"name":"MASSA MARTANA","end":"9999-12-31","sta
rt":"1863-04-13"}],"C364":[{"start":"1861-03-17","end":"1928-05-02","name":"
CAVAGLIETTO"},{"end":"9999-12-31","start":"1947-11-28","name":"CAVAGLIETTO"}
],"F775":[{"name":"MOTTAFOLLONE","start":"1861-03-17","end":"9999-12-31"}],"
E958":[{"name":"MARINO","end":"1974-10-14","start":"1871-01-15"},{"name":"MA
RINO","end":"1993-11-04","start":"1974-10-15"},{"end":"1995-09-20","start":"
1993-11-05","name":"MARINO"},{"end":"9999-12-31","start":"1995-09-21","name"
:"MARINO"}],"A153":[{"name":"ALBESE CON CASSANO","end":"9999-12-31","start":
"1928-09-23"}],"B146":[{"name":"BRENNO USERIA","end":"1927-01-11","start":"1
863-02-23"},{"name":"BRENNO USERIA","end":"1928-10-16","start":"1927-01-12"}
],"H264":[{"name":"RHO","start":"1861-03-17","end":"1928-06-26"},{"start":"1
928-06-27","end":"9999-12-31","name":"RHO"}],"E208":[{"name":"GROTTAZZOLINA"
,"end":"2009-07-11","start":"1861-03-17"},{"start":"2009-07-12","end":"9999-
12-31","name":"GROTTAZZOLINA"}],"H501":[{"name":"ROMA","start":"1884-09-11",
"end":"1902-03-09"},{"start":"1902-03-10","end":"1924-01-10","name":"ROMA"},
{"name":"ROMA","end":"1932-11-05","start":"1924-01-11"},{"start":"1932-11-06
","end":"1935-03-06","name":"ROMA"},{"name":"ROMA","start":"1935-03-07","end
":"1935-07-22"},{"name":"ROMA","start":"1935-07-23","end":"1936-04-24"},{"na
me":"ROMA","start":"1936-04-25","end":"1937-11-02"},{"end":"1938-07-26","sta
rt":"1937-11-03","name":"ROMA"},{"end":"1990-03-24","start":"1938-07-27","na
me":"ROMA"},{"name":"ROMA","end":"1992-04-03","start":"1990-03-25"},{"end":"
9999-12-31","start":"1992-04-04","name":"ROMA"}],"D943":[{"name":"GAVERINA",
"end":"1968-09-17","start":"1861-03-17"},{"name":"GAVERINA TERME","start":"1
968-09-18","end":"9999-12-31"}],"B445":[{"name":"CALVI RISORTA","start":"186
2-12-29","end":"1927-01-11"},{"end":"1945-08-31","start":"1927-01-12","name"
:"CALVI RISORTA"},{"end":"9999-12-31","start":"1945-09-01","name":"CALVI RIS
ORTA"}],"B187":[{"name":"BRIOSCO","end":"2009-06-29","start":"1869-02-13"},{
"start":"2009-06-30","end":"9999-12-31","name":"BRIOSCO"}],"F230":[{"name":"
MIRABELLA ECLANO","end":"9999-12-31","start":"1862-12-29"}],"D152":[{"start"
:"1861-03-17","end":"1924-01-14","name":"CRENNA"}],"Z252":[{"name":"Armenia"
,"end":"9999-12-31","start":"0000-01-01"}],"E555":[{"start":"1952-12-14","en
d":"9999-12-31","name":"LETOJANNI"}],"M326":[{"name":"SCARPERIA E SAN PIERO"
,"start":"2014-01-01","end":"9999-12-31"}],"L778":[{"end":"1928-01-25","star
t":"1861-03-17","name":"VEROLAVECCHIA"},{"end":"9999-12-31","start":"1948-03
-21","name":"VEROLAVECCHIA"}],"F169":[{"end":"1927-01-11","start":"1902-01-1
7","name":"MEZZANA SUPERIORE"},{"start":"1927-01-12","end":"1927-12-03","nam
e":"MEZZANA SUPERIORE"}],"G630":[{"name":"PIETRAVAIRANO","end":"1927-01-11",
"start":"1861-03-17"},{"name":"PIETRAVAIRANO","start":"1927-01-12","end":"19
45-08-31"},{"name":"PIETRAVAIRANO","start":"1945-09-01","end":"9999-12-31"}]
,"C123":[{"name":"CASTELGUIDONE","end":"9999-12-31","start":"1861-03-17"}],"
E015":[{"name":"GIAROLE","end":"9999-12-31","start":"1861-03-17"}],"A460":[{
"end":"9999-12-31","start":"1861-03-17","name":"ASCEA"}],"F476":[{"end":"190
7-09-29","start":"1861-03-17","name":"MONTE COLOMBO"},{"name":"MONTE COLOMBO
","end":"1992-04-15","start":"1907-09-30"},{"name":"MONTE COLOMBO","start":"
1992-04-16","end":"2015-12-31"}],"L474":[{"name":"TURRIACO","end":"1923-02-0
7","start":"1921-01-05"},{"name":"TURRIACO","start":"1923-02-08","end":"1948
-01-09"},{"name":"TURRIACO","end":"9999-12-31","start":"1948-01-10"}],"A046"
:[{"end":"1924-01-11","start":"1861-03-17","name":"ACQUATE"}],"H482":[{"end"
:"1923-02-07","start":"1921-01-05","name":"RODIC"},{"start":"1923-02-08","en
d":"1923-04-26","name":"RODIC"},{"name":"RODITTI","end":"1927-01-12","start"
:"1923-04-27"}],"B053":[{"name":"BORNATO","end":"1927-11-01","start":"1861-0
3-17"}],"B359":[{"start":"1861-03-17","end":"2018-12-31","name":"CAGNO"}],"H
615":[{"end":"1929-05-13","start":"1861-03-17","name":"ROVETTA"},{"name":"RO
VETTA","end":"9999-12-31","start":"1947-05-22"}],"L922":[{"start":"1914-01-0
1","end":"1927-01-11","name":"VILLA CELIERA"},{"end":"9999-12-31","start":"1
927-01-12","name":"VILLA CELIERA"}],"B787":[{"start":"1861-03-17","end":"999
9-12-31","name":"CARLENTINI"}],"F341":[{"name":"MONCLASSICO","start":"1920-1
0-16","end":"1923-02-13"},{"start":"1923-02-14","end":"1928-09-05","name":"M
ONCLASSICO"},{"name":"MONCLASSICO","end":"1970-08-04","start":"1953-09-12"},
{"name":"MONCLASSICO","start":"1970-08-05","end":"2015-12-31"}],"F537":[{"na
me":"MONTELEONE DI CALABRIA","end":"1928-01-18","start":"1863-02-06"},{"name
":"VIBO VALENTIA","start":"1928-01-19","end":"1937-05-27"},{"name":"VIBO VAL
ENTIA","end":"1940-05-29","start":"1937-05-28"},{"start":"1940-05-30","end":
"1992-04-15","name":"VIBO VALENTIA"},{"name":"VIBO VALENTIA","start":"1992-0
4-16","end":"9999-12-31"}],"D752":[{"name":"FRABOSA SOTTANA","end":"9999-12-
31","start":"1861-03-17"}],"C492":[{"name":"CERCHIO","start":"1861-03-17","e
nd":"9999-12-31"}],"F888":[{"start":"1861-03-17","end":"1935-06-05","name":"
NICASTRO"},{"name":"NICASTRO","start":"1935-06-06","end":"1968-11-27"}],"I35
4":[{"name":"SANT'ORSOLA","end":"1923-02-13","start":"1920-10-16"},{"start":
"1923-02-14","end":"1929-03-23","name":"SANT'ORSOLA"},{"name":"SANT'ORSOLA",
"start":"1929-03-24","end":"1929-06-15"},{"name":"SANT'ORSOLA","start":"1929
-06-16","end":"1948-01-24"},{"name":"SANT'ORSOLA","end":"1986-12-30","start"
:"1948-01-25"},{"name":"SANT'ORSOLA TERME","end":"9999-12-31","start":"1986-
12-31"}],"A031":[{"name":"ACQUACANINA","end":"2016-12-31","start":"1861-03-1
7"}],"B513":[{"end":"1934-02-24","start":"1861-03-17","name":"CAMPIONE"},{"e
nd":"9999-12-31","start":"1934-02-25","name":"CAMPIONE D'ITALIA"}],"L178":[{
"name":"TIROLO","start":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-1
4","end":"1927-01-11","name":"TIROLO"},{"start":"1927-01-12","end":"9999-12-
31","name":"TIROLO"}],"L937":[{"name":"VILLA ESTENSE","start":"1867-11-25","
end":"9999-12-31"}],"I420":[{"name":"SAPONARA","end":"9999-12-31","start":"1
952-12-14"},{"name":"SAPONARA VILLAFRANCA","end":"1929-06-02","start":"1863-
04-10"}],"F769":[{"start":"1861-03-17","end":"1927-01-11","name":"MOSSO SANT
A MARIA"},{"name":"MOSSO SANTA MARIA","start":"1927-01-12","end":"1938-04-06
"},{"end":"1992-04-15","start":"1938-04-07","name":"MOSSO SANTA MARIA"},{"en
d":"1998-12-31","start":"1992-04-16","name":"MOSSO SANTA MARIA"}],"F336":[{"
name":"MONCALVO","start":"1861-03-17","end":"1908-07-22"},{"name":"MONCALVO"
,"end":"1935-04-07","start":"1908-07-23"},{"name":"MONCALVO","end":"9999-12-
31","start":"1935-04-08"}],"C723":[{"name":"CIRIGLIANO","start":"1861-03-17"
,"end":"1927-01-11"},{"name":"CIRIGLIANO","start":"1927-01-12","end":"9999-1
2-31"}],"L334":[{"name":"TRASACCO","end":"9999-12-31","start":"1861-03-17"}]
,"I613":[{"start":"1861-03-17","end":"1927-01-11","name":"SENNARIOLO"},{"end
":"1974-08-19","start":"1927-01-12","name":"SENNARIOLO"},{"name":"SENNARIOLO
","end":"9999-12-31","start":"1974-08-20"}],"G824":[{"name":"PONTAGNA","end"
:"1927-12-03","start":"1861-03-17"}],"F522":[{"end":"2009-07-11","start":"18
61-03-17","name":"MONTEGRANARO"},{"name":"MONTEGRANARO","end":"9999-12-31","
start":"2009-07-12"}],"G701":[{"start":"1861-03-17","end":"1928-05-11","name
":"PIRRI"}],"B273":[{"start":"1866-11-19","end":"1924-01-13","name":"BURANO"
}],"L885":[{"start":"1861-03-17","end":"9999-12-31","name":"VIGNOLA"}],"D418
":[{"end":"1929-05-03","start":"1861-03-17","name":"ERBANNO"}],"I207":[{"end
":"9999-12-31","start":"1866-11-19","name":"SANTA GIUSTINA IN COLLE"}],"F581
":[{"start":"1882-04-03","end":"9999-12-31","name":"OSTRA VETERE"}],"B731":[
{"name":"CARAVAGGIO","end":"9999-12-31","start":"1861-03-17"}],"E489":[{"nam
e":"LAVAGNO","start":"1866-11-19","end":"9999-12-31"}],"L981":[{"start":"186
1-03-17","end":"1927-01-11","name":"VILLANEUVE"},{"end":"1928-05-11","start"
:"1927-01-12","name":"VILLANEUVE"},{"name":"VILLANEUVE","start":"1928-05-12"
,"end":"1928-11-06"},{"end":"1939-07-03","start":"1928-11-07","name":"VILLAN
OVA BALTEA"},{"name":"VILLANOVA BALTEA","start":"1939-07-04","end":"1946-01-
29"},{"name":"VILLENEUVE","start":"1946-01-30","end":"1946-03-24"},{"name":"
VILLENEUVE","start":"1946-03-25","end":"9999-12-31"}],"Z214":[{"end":"9999-1
2-31","start":"0000-01-01","name":"Corea del Nord"}],"D114":[{"name":"COSTAR
AINERA","end":"1928-03-05","start":"1861-03-17"},{"name":"COSTARAINERA","sta
rt":"1954-07-03","end":"9999-12-31"}],"H421":[{"name":"ROCCA MASSIMA","start
":"1871-01-15","end":"1934-10-25"},{"name":"ROCCA MASSIMA","end":"1934-12-17
","start":"1934-10-26"},{"name":"ROCCA MASSIMA","end":"1945-04-08","start":"
1934-12-18"},{"name":"ROCCA MASSIMA","start":"1945-04-09","end":"9999-12-31"
}],"M407":[{"name":"TERRE D'ADIGE","start":"2019-01-01","end":"9999-12-31"}]
,"A087":[{"end":"1935-07-12","start":"1869-02-04","name":"AGRATE BRIANZA"},{
"start":"1935-07-13","end":"2009-06-29","name":"AGRATE BRIANZA"},{"name":"AG
RATE BRIANZA","end":"9999-12-31","start":"2009-06-30"}],"F175":[{"start":"18
61-03-17","end":"1979-10-16","name":"MEZZANINO"},{"name":"MEZZANINO","start"
:"1979-10-17","end":"9999-12-31"}],"D897":[{"name":"GAMBELLARA","end":"9999-
12-31","start":"1866-11-19"}],"A753":[{"name":"BELLIZZI","end":"1934-06-12",
"start":"1861-03-17"},{"end":"1938-08-31","start":"1934-06-13","name":"BELLI
ZZI IRPINO"}],"M034":[{"name":"VILLA VICENTINA","end":"1923-02-07","start":"
1921-01-05"},{"start":"1923-02-08","end":"1928-04-15","name":"VILLA VICENTIN
A"},{"start":"1948-03-26","end":"1968-04-05","name":"VILLA VICENTINA"},{"sta
rt":"1968-04-06","end":"2012-06-20","name":"VILLA VICENTINA"},{"name":"VILLA
 VICENTINA","end":"2018-01-31","start":"2012-06-21"}],"L138":[{"start":"1861
-03-17","end":"9999-12-31","name":"TERRICCIOLA"}],"L977":[{"name":"VILLANOVA
 DEL SILLARO","start":"1878-06-21","end":"1992-04-15"},{"name":"VILLANOVA DE
L SILLARO","start":"1992-04-16","end":"9999-12-31"}],"E549":[{"end":"9999-12
-31","start":"1861-03-17","name":"LESINA"}],"C392":[{"name":"CAVEDAGO","star
t":"1920-10-16","end":"1923-02-13"},{"name":"CAVEDAGO","start":"1923-02-14",
"end":"1928-10-29"},{"start":"1947-07-09","end":"9999-12-31","name":"CAVEDAG
O"}],"A266":[{"start":"1920-10-16","end":"1923-02-13","name":"AMPEZZO"},{"na
me":"AMPEZZO","end":"1923-04-26","start":"1923-02-14"},{"end":"9999-12-31","
start":"1923-04-27","name":"CORTINA D'AMPEZZO"}],"A071":[{"end":"1927-01-11"
,"start":"1913-11-09","name":"AGLIANA"},{"start":"1927-01-12","end":"9999-12
-31","name":"AGLIANA"}],"F441":[{"name":"MONTEBELLO DI BERTONA","start":"186
3-07-13","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31","name"
:"MONTEBELLO DI BERTONA"}],"G203":[{"name":"OZIERI","start":"1861-03-17","en
d":"1988-08-05"},{"name":"OZIERI","start":"1988-08-06","end":"9999-12-31"}],
"A805":[{"name":"BERNEZZO","start":"1861-03-17","end":"9999-12-31"}],"B459":
[{"name":"CAMASCO","start":"1861-03-17","end":"1927-01-11"},{"start":"1927-0
1-12","end":"1929-03-01","name":"CAMASCO"}],"I158":[{"end":"9999-12-31","sta
rt":"1861-03-17","name":"SAN SEVERO"}],"F376":[{"start":"1895-01-28","end":"
1927-08-05","name":"MONOPOLI"},{"start":"1927-08-06","end":"9999-12-31","nam
e":"MONOPOLI"}],"L374":[{"name":"TRENS","start":"1920-10-16","end":"1923-02-
13"},{"name":"TRENS","end":"1927-01-11","start":"1923-02-14"},{"end":"1928-1
2-09","start":"1927-01-12","name":"TRENS"}],"M254":[{"name":"VALSANTAMARINA"
,"start":"1941-09-12","end":"1947-09-15"}],"H382":[{"start":"1861-03-17","en
d":"9999-12-31","name":"ROCCABASCERANA"}],"F603":[{"start":"1871-01-15","end
":"1927-04-13","name":"MONTE ROMANO"},{"start":"1927-04-14","end":"9999-12-3
1","name":"MONTE ROMANO"}],"F436":[{"end":"1923-02-13","start":"1920-10-16",
"name":"MONTE A MEZZOD\u00cc"},{"end":"1923-04-26","start":"1923-02-14","nam
e":"MONTE A MEZZOD\u00cc"},{"start":"1923-04-27","end":"1927-01-11","name":"
MONTE DI MEZZOD\u00cc"},{"end":"1928-05-12","start":"1927-01-12","name":"MON
TE DI MEZZOD\u00cc"}],"G850":[{"name":"PONTELONGO","end":"1996-12-06","start
":"1866-11-19"},{"end":"9999-12-31","start":"1996-12-07","name":"PONTELONGO"
}],"L434":[{"name":"TRIUGGIO","end":"2009-06-29","start":"1871-07-01"},{"sta
rt":"2009-06-30","end":"9999-12-31","name":"TRIUGGIO"}],"D660":[{"end":"1992
-04-15","start":"1861-03-17","name":"FOMBIO"},{"start":"1992-04-16","end":"9
999-12-31","name":"FOMBIO"}],"E299":[{"name":"INDUNO OLONA","end":"1927-01-1
1","start":"1863-02-23"},{"end":"1927-12-28","start":"1927-01-12","name":"IN
DUNO OLONA"},{"end":"9999-12-31","start":"1950-12-24","name":"INDUNO OLONA"}
],"E892":[{"end":"1927-01-11","start":"1861-03-17","name":"MANOPPELLO"},{"na
me":"MANOPPELLO","end":"9999-12-31","start":"1927-01-12"}],"F577":[{"name":"
MONTENERO D'IDRIA","start":"1921-01-05","end":"1921-12-30"},{"end":"1923-02-
07","start":"1921-12-31","name":"MONTENERO D'IDRIA"},{"name":"MONTENERO D'ID
RIA","start":"1923-02-08","end":"1927-01-11"},{"name":"MONTENERO D'IDRIA","s
tart":"1927-01-12","end":"1928-01-25"},{"name":"MONTENERO D'IDRIA","end":"19
47-09-15","start":"1928-01-26"}],"I320":[{"start":"1863-04-10","end":"1928-0
6-29","name":"SANT'ELIA A PIANISI"},{"end":"1946-04-08","start":"1928-06-30"
,"name":"SANT'ELIA A PIANISI"},{"end":"1970-03-02","start":"1946-04-09","nam
e":"SANT'ELIA A PIANISI"},{"end":"9999-12-31","start":"1970-03-03","name":"S
ANT'ELIA A PIANISI"}],"A901":[{"end":"1923-02-13","start":"1920-10-16","name
":"BLEGGIO INFERIORE"},{"name":"BLEGGIO INFERIORE","start":"1923-02-14","end
":"1928-06-26"},{"end":"2009-12-31","start":"1947-01-10","name":"BLEGGIO INF
ERIORE"}],"E644":[{"start":"1861-03-17","end":"1927-01-11","name":"LOCERI"},
{"end":"2005-12-31","start":"1927-01-12","name":"LOCERI"},{"name":"LOCERI","
start":"2006-01-01","end":"2016-04-27"},{"start":"2016-04-28","end":"9999-12
-31","name":"LOCERI"}],"L007":[{"name":"SUNO","end":"9999-12-31","start":"18
61-03-17"}],"I454":[{"start":"1861-03-17","end":"1925-11-29","name":"SASSETT
A"},{"end":"9999-12-31","start":"1925-11-30","name":"SASSETTA"}],"C768":[{"n
ame":"CIVITALUPARELLA","start":"1861-03-17","end":"1928-11-09"},{"name":"CIV
ITALUPARELLA","end":"1964-04-08","start":"1928-11-10"},{"name":"CIVITALUPARE
LLA","start":"1964-04-09","end":"9999-12-31"}],"E389":[{"name":"JOPPOLO","st
art":"1861-03-17","end":"1992-04-15"},{"start":"1992-04-16","end":"9999-12-3
1","name":"JOPPOLO"}],"A251":[{"end":"1929-03-09","start":"1861-03-17","name
":"AMALFI"},{"end":"1945-07-18","start":"1929-03-10","name":"AMALFI"},{"name
":"AMALFI","start":"1945-07-19","end":"1946-04-25"},{"name":"AMALFI","end":"
9999-12-31","start":"1946-04-26"}],"E083":[{"name":"GONARS","start":"1866-11
-19","end":"1940-12-19"},{"end":"1968-04-05","start":"1940-12-20","name":"GO
NARS"},{"start":"1968-04-06","end":"9999-12-31","name":"GONARS"}],"M074":[{"
name":"VISIGNANO","end":"1923-02-07","start":"1921-01-05"},{"start":"1923-02
-08","end":"1924-12-12","name":"VISIGNANO"},{"start":"1924-12-13","end":"194
7-09-15","name":"VISIGNANO D'ISTRIA"}],"F217":[{"name":"MINEO","end":"9999-1
2-31","start":"1861-03-17"}],"D318":[{"end":"9999-12-31","start":"1861-03-17
","name":"DOLCEACQUA"}],"H542":[{"name":"RONCOFREDDO","end":"1992-04-15","st
art":"1861-03-17"},{"name":"RONCOFREDDO","end":"9999-12-31","start":"1992-04
-16"}],"F135":[{"start":"1861-03-17","end":"1929-06-14","name":"MERCATELLO"}
,{"end":"1951-07-16","start":"1929-06-15","name":"MERCATELLO"},{"end":"1998-
09-03","start":"1951-07-17","name":"MERCATELLO SUL METAURO"},{"name":"MERCAT
ELLO SUL METAURO","start":"1998-09-04","end":"9999-12-31"}],"E757":[{"name":
"LUSERNA","end":"1923-02-13","start":"1920-10-16"},{"end":"9999-12-31","star
t":"1923-02-14","name":"LUSERNA"}],"H899":[{"end":"1935-04-07","start":"1861
-03-17","name":"SAN GIORGIO SCARAMPI"},{"start":"1935-04-08","end":"9999-12-
31","name":"SAN GIORGIO SCARAMPI"}],"G142":[{"name":"ORTONA DEI MARSI","star
t":"1861-03-17","end":"1948-07-02"},{"end":"9999-12-31","start":"1948-07-03"
,"name":"ORTONA DEI MARSI"}],"C282":[{"start":"1939-10-21","end":"1946-01-29
","name":"CASTEL VERR\u00c8S"},{"end":"1927-01-11","start":"1861-03-17","nam
e":"VERR\u00c8S"},{"start":"1927-01-12","end":"1928-08-13","name":"VERR\u00c
8S"},{"end":"1939-10-20","start":"1928-08-14","name":"VERR\u00c8S"},{"end":"
1946-03-24","start":"1946-01-30","name":"VERR\u00c8S"},{"name":"VERR\u00c8S"
,"end":"9999-12-31","start":"1946-03-25"}],"H321":[{"end":"9999-12-31","star
t":"1861-03-17","name":"RIPATRANSONE"}],"C889":[{"end":"1928-02-10","start":
"1861-03-17","name":"COLOGNA"}],"M307":[{"start":"1928-10-17","end":"1955-03
-02","name":"RASUN VALDAORA"}],"B771":[{"name":"CARFIZZI","start":"1904-06-0
6","end":"1992-04-15"},{"name":"CARFIZZI","end":"9999-12-31","start":"1992-0
4-16"}],"H292":[{"name":"RIMASCO","end":"1927-01-11","start":"1861-03-17"},{
"name":"RIMASCO","end":"2017-12-31","start":"1927-01-12"}],"G953":[{"end":"1
927-01-11","start":"1861-03-17","name":"POZZALLO"},{"start":"1927-01-12","en
d":"9999-12-31","name":"POZZALLO"}],"G617":[{"name":"PIETRA GAVINA","end":"1
872-12-15","start":"1861-03-17"}],"I412":[{"name":"SAN ZENO NAVIGLIO","start
":"1879-07-21","end":"9999-12-31"}],"F302":[{"name":"MONTEGRAZIE","end":"192
3-11-28","start":"1872-07-16"},{"name":"MONTEGRAZIE","start":"1925-08-22","e
nd":"1928-12-14"}],"C168":[{"name":"CASTELLETTO VILLA","start":"1861-03-17",
"end":"1927-01-11"},{"start":"1927-01-12","end":"1929-06-03","name":"CASTELL
ETTO VILLA"}],"M125":[{"start":"1872-01-01","end":"9999-12-31","name":"VOLTA
 MANTOVANA"}],"B585":[{"start":"1861-03-17","end":"1929-04-10","name":"CANDE
LARA"}],"F510":[{"end":"1927-01-11","start":"1861-03-17","name":"MONTEFRANCO
"},{"name":"MONTEFRANCO","end":"9999-12-31","start":"1927-01-12"}],"F735":[{
"name":"MORMANNO","start":"1861-03-17","end":"9999-12-31"}],"A618":[{"name":
"BARANZATE","end":"2003-03-05","start":"2001-12-12"},{"end":"9999-12-31","st
art":"2004-06-08","name":"BARANZATE"}],"L910":[{"name":"VILLA AGNEDO","start
":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-14","end":"1928-07-27",
"name":"VILLA AGNEDO"},{"end":"2015-12-31","start":"1947-01-14","name":"VILL
A AGNEDO"}],"M212":[{"name":"LADISPOLI","end":"9999-12-31","start":"1970-05-
27"}],"C464":[{"start":"1920-10-16","end":"1923-02-13","name":"CENGLES"},{"e
nd":"1927-01-11","start":"1923-02-14","name":"CENGLES"},{"name":"CENGLES","s
tart":"1927-01-12","end":"1929-05-14"}],"G742":[{"name":"POCAPAGLIA","start"
:"1861-03-17","end":"1928-06-26"},{"start":"1946-04-26","end":"9999-12-31","
name":"POCAPAGLIA"}],"B171":[{"name":"BRICHERASIO","end":"9999-12-31","start
":"1861-03-17"}],"F050":[{"name":"MASULLAS","end":"1928-03-12","start":"1861
-03-17"},{"name":"MASULLAS","end":"1962-05-21","start":"1928-03-13"},{"name"
:"MASULLAS","end":"1970-12-08","start":"1962-05-22"},{"name":"MASULLAS","sta
rt":"1970-12-09","end":"1974-08-19"},{"name":"MASULLAS","end":"9999-12-31","
start":"1974-08-20"}],"H683":[{"start":"1863-01-19","end":"9999-12-31","name
":"SALA CONSILINA"}],"L738":[{"end":"9999-12-31","start":"1861-03-17","name"
:"VENOSA"}],"H044":[{"start":"1861-03-17","end":"1928-11-09","name":"PRESEGN
O"}],"I758":[{"start":"1861-03-17","end":"1928-05-01","name":"SIROLO"},{"nam
e":"SIROLO","end":"9999-12-31","start":"1946-11-02"}],"B345":[{"start":"1880
-01-16","end":"9999-12-31","name":"CADONEGHE"}],"G670":[{"start":"1861-03-17
","end":"9999-12-31","name":"PIMONTE"}],"I226":[{"name":"SANTA MARGHERITA D'
ADIGE","end":"2018-02-16","start":"1867-08-26"}],"A360":[{"name":"ARCADE","s
tart":"1866-11-19","end":"1960-02-20"},{"name":"ARCADE","start":"1960-02-21"
,"end":"9999-12-31"}],"L813":[{"name":"VESTRENO","start":"1861-03-17","end":
"1992-04-15"},{"start":"1992-04-16","end":"2017-12-31","name":"VESTRENO"}],"
F270":[{"start":"1861-03-17","end":"1928-06-14","name":"MOGORELLA"},{"name":
"MOGORELLA","start":"1951-01-12","end":"1974-08-19"},{"start":"1974-08-20","
end":"9999-12-31","name":"MOGORELLA"}],"B988":[{"name":"CASORATE PRIMO","sta
rt":"1863-03-30","end":"9999-12-31"}],"M426":[{"end":"9999-12-31","start":"2
019-02-20","name":"COLCERESA"}],"A744":[{"end":"2014-02-03","start":"1861-03
-17","name":"BELLAGIO"}],"H171":[{"name":"RAMPONIO VERNA","end":"1936-10-23"
,"start":"1928-02-23"},{"name":"RAMPONIO VERNA","start":"1936-10-24","end":"
2016-12-31"}],"D912":[{"end":"9999-12-31","start":"1864-04-01","name":"GARBA
GNATE MILANESE"}],"C632":[{"name":"CHIETI","end":"9999-12-31","start":"1880-
02-13"}],"G447":[{"end":"1929-03-14","start":"1861-03-17","name":"PERDIFUMO"
},{"name":"PERDIFUMO","start":"1929-03-15","end":"9999-12-31"}],"Z704":[{"en
d":"9999-12-31","start":"0000-01-01","name":"Figi"}],"C245":[{"end":"9999-12
-31","start":"1861-03-17","name":"CASTELPAGANO"}],"G185":[{"name":"OSTIANO",
"start":"1868-07-01","end":"9999-12-31"}],"I583":[{"name":"SELINO","start":"
1861-03-17","end":"1927-08-03"}],"F425":[{"end":"1928-06-14","start":"1863-0
7-13","name":"MONTANO COMASCO"}],"I717":[{"name":"SIAMAGGIORE","start":"1861
-03-17","end":"1927-09-04"},{"end":"1974-08-19","start":"1951-01-12","name":
"SIAMAGGIORE"},{"end":"9999-12-31","start":"1974-08-20","name":"SIAMAGGIORE"
}],"H585":[{"name":"ROTA GRECA","end":"9999-12-31","start":"1863-04-10"}],"L
069":[{"end":"1970-03-02","start":"1861-03-17","name":"TAVENNA"},{"name":"TA
VENNA","end":"9999-12-31","start":"1970-03-03"}],"L363":[{"name":"TREGLIO","
start":"1861-03-17","end":"9999-12-31"}],"C627":[{"name":"CHIERI","start":"1
878-08-02","end":"9999-12-31"}],"G571":[{"name":"PIANSANO","start":"1871-01-
15","end":"1927-01-11"},{"name":"PIANSANO","start":"1927-01-12","end":"9999-
12-31"}],"C595":[{"end":"1927-01-11","start":"1861-03-17","name":"CHAMBAVE"}
,{"end":"1928-06-13","start":"1927-01-12","name":"CHAMBAVE"},{"name":"CHAMBA
VE","start":"1928-06-14","end":"1939-10-20"},{"name":"CHAMBAVE","start":"194
6-01-30","end":"1946-05-14"},{"name":"CHAMBAVE","end":"9999-12-31","start":"
1946-05-15"},{"name":"CIAMBAVE","end":"1946-01-29","start":"1939-10-21"}],"A
810":[{"end":"1968-04-05","start":"1866-11-19","name":"BERTIOLO"},{"name":"B
ERTIOLO","end":"2007-11-07","start":"1968-04-06"},{"name":"BERTIOLO","end":"
9999-12-31","start":"2007-11-08"}],"E414":[{"name":"LAIGUEGLIA","end":"1927-
01-11","start":"1861-03-17"},{"name":"LAIGUEGLIA","start":"1927-01-12","end"
:"9999-12-31"}],"Z528":[{"name":"Saint Vincent e Grenadine","end":"9999-12-3
1","start":"0000-01-01"}],"L428":[{"start":"1958-08-12","end":"2005-12-31","
name":"TRINIT\u00c0 D'AGULTU E VIGNOLA"},{"start":"2006-01-01","end":"2016-0
4-27","name":"TRINIT\u00c0 D'AGULTU E VIGNOLA"},{"name":"TRINIT\u00c0 D'AGUL
TU E VIGNOLA","end":"9999-12-31","start":"2016-04-28"}],"I170":[{"start":"18
63-04-10","end":"9999-12-31","name":"SANTA CATERINA DELLO IONIO"}],"H988":[{
"start":"1923-04-27","end":"1927-01-11","name":"SAN MARTINO"},{"name":"SAN M
ARTINO","start":"1927-01-12","end":"1928-06-14"},{"start":"1920-10-16","end"
:"1923-02-13","name":"SAN MARTINO IN BADIA"},{"name":"SAN MARTINO IN BADIA",
"end":"1923-04-26","start":"1923-02-14"},{"name":"SAN MARTINO IN BADIA","sta
rt":"1928-06-15","end":"1964-06-09"},{"name":"SAN MARTINO IN BADIA","end":"9
999-12-31","start":"1964-06-10"}],"G211":[{"start":"1861-03-17","end":"1975-
03-22","name":"PACHINO"},{"name":"PACHINO","start":"1975-03-23","end":"9999-
12-31"}],"I049":[{"end":"9999-12-31","start":"1958-05-28","name":"SAN NAZZAR
O"}],"I343":[{"name":"SANTINO","end":"1928-10-28","start":"1861-03-17"}],"C9
98":[{"name":"CORENO AUSONIO","end":"1927-01-11","start":"1862-12-29"},{"sta
rt":"1927-01-12","end":"9999-12-31","name":"CORENO AUSONIO"}],"H345":[{"name
":"RIVA SANTO STEFANO","start":"1928-03-07","end":"1947-03-27"},{"name":"RIV
A SANTO STEFANO","start":"1947-03-28","end":"1954-09-18"}],"M063":[{"name":"
VIOLA","start":"1861-03-17","end":"1929-04-10"},{"name":"VIOLA","start":"192
9-04-11","end":"1947-01-09"},{"name":"VIOLA","end":"9999-12-31","start":"194
7-01-10"}],"M369":[{"name":"ALTO RENO TERME","start":"2016-01-01","end":"999
9-12-31"}],"G486":[{"start":"1861-03-17","end":"1928-08-07","name":"PESCHE"}
,{"end":"1970-03-02","start":"1937-02-24","name":"PESCHE"},{"name":"PESCHE",
"end":"9999-12-31","start":"1970-03-03"}],"E118":[{"start":"1861-03-17","end
":"9999-12-31","name":"GOVONE"}],"A657":[{"name":"BARGA","end":"9999-12-31",
"start":"1861-03-17"}],"F126":[{"start":"1861-03-17","end":"9999-12-31","nam
e":"MENFI"}],"L124":[{"end":"9999-12-31","start":"1864-03-04","name":"TERRAN
OVA DA SIBARI"}],"D189":[{"end":"1992-04-15","start":"1861-03-17","name":"CR
UCOLI"},{"name":"CRUCOLI","end":"9999-12-31","start":"1992-04-16"}],"G878":[
{"end":"1927-01-11","start":"1861-03-17","name":"POPOLI"},{"end":"9999-12-31
","start":"1927-01-12","name":"POPOLI"}],"F611":[{"name":"MONTEROTONDO","end
":"9999-12-31","start":"1871-01-15"}],"B044":[{"start":"1954-07-03","end":"9
999-12-31","name":"BORGO TOSSIGNANO"},{"end":"1954-07-02","start":"1884-06-1
8","name":"TOSSIGNANO"}],"I770":[{"name":"SLIVIA","start":"1923-04-27","end"
:"1928-09-08"},{"end":"1923-02-07","start":"1921-01-05","name":"SLIVNO"},{"e
nd":"1923-04-26","start":"1923-02-08","name":"SLIVNO"}],"L662":[{"name":"VAN
DOIES DI SOTTO","start":"1920-10-16","end":"1923-02-13"},{"name":"VANDOIES D
I SOTTO","start":"1923-02-14","end":"1927-01-11"},{"name":"VANDOIES DI SOTTO
","start":"1927-01-12","end":"1929-03-24"}],"G658":[{"start":"1928-07-29","e
nd":"1992-05-22","name":"PIEVE VERGONTE"},{"name":"PIEVE VERGONTE","end":"99
99-12-31","start":"1992-05-23"}],"D868":[{"name":"GALGAGNANO","start":"1861-
03-17","end":"1927-12-26"},{"start":"1927-12-27","end":"1992-04-15","name":"
GALGAGNANO"},{"start":"1992-04-16","end":"9999-12-31","name":"GALGAGNANO"}],
"L905":[{"name":"VILLA SANTA LUCIA","start":"1863-02-06","end":"1927-01-11"}
,{"name":"VILLA SANTA LUCIA","end":"9999-12-31","start":"1927-01-12"}],"A877
":[{"name":"BIONA","end":"1946-01-29","start":"1939-10-21"},{"end":"1927-01-
11","start":"1861-03-17","name":"BIONAZ"},{"name":"BIONAZ","end":"1939-10-20
","start":"1927-01-12"},{"end":"9999-12-31","start":"1946-01-30","name":"BIO
NAZ"}],"E833":[{"start":"1866-11-19","end":"1961-03-08","name":"MAJANO"},{"n
ame":"MAJANO","start":"1961-03-09","end":"1968-04-05"},{"name":"MAJANO","sta
rt":"1968-04-06","end":"9999-12-31"}],"C473":[{"name":"CENTRISOLA","end":"19
48-03-30","start":"1928-01-18"}],"M141":[{"end":"1990-04-09","start":"1871-0
1-15","name":"ZAGAROLO"},{"start":"1990-04-10","end":"9999-12-31","name":"ZA
GAROLO"}],"D357":[{"end":"1923-03-18","start":"1861-03-17","name":"DOVADOLA"
},{"name":"DOVADOLA","start":"1923-03-19","end":"1992-04-15"},{"end":"9999-1
2-31","start":"1992-04-16","name":"DOVADOLA"}],"F258":[{"end":"1927-01-11","
start":"1861-03-17","name":"MODICA"},{"name":"MODICA","start":"1927-01-12","
end":"1950-11-11"},{"name":"MODICA","start":"1950-11-12","end":"9999-12-31"}
],"H459":[{"end":"1927-01-11","start":"1862-12-29","name":"ROCCHETTA E CROCE
"},{"start":"1927-01-12","end":"1945-08-31","name":"ROCCHETTA E CROCE"},{"na
me":"ROCCHETTA E CROCE","start":"1945-09-01","end":"9999-12-31"}],"D016":[{"
name":"CORNALBA","end":"1927-11-23","start":"1861-03-17"},{"end":"9999-12-31
","start":"1966-01-22","name":"CORNALBA"}],"B382":[{"start":"1861-03-17","en
d":"9999-12-31","name":"CALASCIO"}],"E718":[{"name":"LUCIGNANO","start":"186
1-03-17","end":"9999-12-31"}],"C333":[{"name":"CASTREZZONE","end":"1928-07-2
7","start":"1861-03-17"}],"F726":[{"name":"MORGEX","start":"1861-03-17","end
":"1927-01-11"},{"end":"1929-05-27","start":"1927-01-12","name":"MORGEX"},{"
name":"MORGEX","start":"1946-01-30","end":"9999-12-31"},{"name":"VALDIGNA D'
AOSTA","start":"1935-06-05","end":"1946-01-29"}],"L801":[{"name":"VERZEGNIS"
,"end":"1968-04-05","start":"1866-11-19"},{"end":"9999-12-31","start":"1968-
04-06","name":"VERZEGNIS"}],"L724":[{"name":"VELTURNO","end":"1923-02-13","s
tart":"1920-10-16"},{"end":"1927-01-11","start":"1923-02-14","name":"VELTURN
O"},{"end":"1929-06-04","start":"1927-01-12","name":"VELTURNO"},{"start":"19
60-11-23","end":"9999-12-31","name":"VELTURNO"}],"L075":[{"name":"TAVIGLIANO
","end":"1927-01-11","start":"1861-03-17"},{"name":"TAVIGLIANO","end":"1929-
05-21","start":"1927-01-12"},{"name":"TAVIGLIANO","start":"1955-01-19","end"
:"1992-04-15"},{"name":"TAVIGLIANO","end":"9999-12-31","start":"1992-04-16"}
],"D789":[{"end":"1934-12-14","start":"1861-03-17","name":"FRATTAMAGGIORE"},
{"name":"FRATTAMAGGIORE","end":"9999-12-31","start":"1934-12-15"}],"M136":[{
"end":"9999-12-31","start":"1861-03-17","name":"VOTTIGNASCO"}],"D954":[{"nam
e":"GAZZELLI","start":"1861-03-17","end":"1928-04-16"}],"F505":[{"start":"19
20-10-16","end":"1923-02-13","name":"MONTEFONTANA"},{"name":"MONTEFONTANA","
start":"1923-02-14","end":"1927-01-11"},{"end":"1929-02-14","start":"1927-01
-12","name":"MONTEFONTANA"}],"A144":[{"name":"ALBEGNO","end":"1928-12-13","s
tart":"1861-03-17"}],"B292":[{"start":"1861-03-17","end":"9999-12-31","name"
:"BUSSERO"}],"H771":[{"start":"1867-06-24","end":"9999-12-31","name":"SAN BE
NEDETTO PO"}],"D561":[{"start":"1872-09-02","end":"9999-12-31","name":"FIANO
 ROMANO"}],"B321":[{"start":"1861-03-17","end":"1928-03-07","name":"CADDO"}]
,"D204":[{"name":"CUNARDO","start":"1861-03-17","end":"1927-01-11"},{"end":"
9999-12-31","start":"1927-01-12","name":"CUNARDO"}],"Z104":[{"name":"Bulgari
a","end":"9999-12-31","start":"0000-01-01"}],"E930":[{"start":"1884-04-07","
end":"1951-03-09","name":"MARCIANA"},{"name":"MARCIANA","end":"9999-12-31","
start":"1951-03-10"}],"F908":[{"start":"1861-03-17","end":"1927-01-11","name
":"NOCCIANO"},{"name":"NOCCIANO","start":"1927-01-12","end":"9999-12-31"}],"
G785":[{"name":"POLIA","end":"1992-04-15","start":"1861-03-17"},{"start":"19
92-04-16","end":"9999-12-31","name":"POLIA"}],"H233":[{"start":"1861-03-17",
"end":"1928-12-29","name":"RENATE"},{"start":"1928-12-30","end":"1929-04-10"
,"name":"RENATE"},{"start":"1956-04-08","end":"2009-06-29","name":"RENATE"},
{"start":"2009-06-30","end":"9999-12-31","name":"RENATE"}],"B899":[{"end":"1
992-04-15","start":"1863-01-26","name":"CASALMAIOCCO"},{"name":"CASALMAIOCCO
","end":"9999-12-31","start":"1992-04-16"}],"L508":[{"start":"1861-03-17","e
nd":"1927-10-08","name":"USELLUS"},{"end":"1947-03-29","start":"1927-10-09",
"name":"USELLUS"},{"start":"1947-03-30","end":"1959-04-07","name":"USELLUS"}
,{"start":"1959-04-08","end":"1974-08-19","name":"USELLUS"},{"name":"USELLUS
","start":"1974-08-20","end":"9999-12-31"}],"I117":[{"name":"SAN PIETRO MUSS
OLINO","start":"1866-11-19","end":"9999-12-31"}],"G490":[{"name":"PESCHIERA 
MARAGLIO","start":"1863-07-13","end":"1928-02-15"}],"F078":[{"start":"1861-0
3-17","end":"2009-06-29","name":"MEDA"},{"start":"2009-06-30","end":"9999-12
-31","name":"MEDA"}],"A448":[{"end":"1928-12-08","start":"1866-11-19","name"
:"ARTEGNA"},{"start":"1928-12-09","end":"1947-01-08","name":"ARTEGNA"},{"nam
e":"ARTEGNA","start":"1947-01-09","end":"1968-04-05"},{"name":"ARTEGNA","sta
rt":"1968-04-06","end":"9999-12-31"}],"I235":[{"name":"SANTA MARIA DELLA CRO
CE","start":"1875-04-01","end":"1928-05-28"}],"B542":[{"name":"CAMPOLI DEL M
ONTE TABURNO","start":"1863-02-06","end":"9999-12-31"}],"I894":[{"name":"SPE
SSA","end":"9999-12-31","start":"1861-03-17"}],"I642":[{"start":"1861-03-17"
,"end":"1928-05-18","name":"SERRA D'AIELLO"},{"start":"1937-07-18","end":"99
99-12-31","name":"SERRA D'AIELLO"}],"M375":[{"name":"ALPAGO","end":"9999-12-
31","start":"2016-02-23"}],"E452":[{"end":"1923-02-13","start":"1920-10-16",
"name":"LARDARO"},{"start":"1923-02-14","end":"1928-04-28","name":"LARDARO"}
,{"end":"2015-12-31","start":"1958-01-01","name":"LARDARO"}],"C079":[{"name"
:"CASTELLI CALEPIO","end":"9999-12-31","start":"1927-08-05"}],"A229":[{"name
":"ALTAVILLA MILICIA","end":"9999-12-31","start":"1862-11-10"}],"C140":[{"st
art":"1861-03-17","end":"1928-05-02","name":"CASTELLAR"},{"name":"CASTELLAR"
,"start":"1946-11-12","end":"2018-12-31"}],"H513":[{"start":"1921-01-05","en
d":"1923-02-07","name":"ROMANS"},{"start":"1923-02-08","end":"1927-01-11","n
ame":"ROMANS"},{"start":"1927-01-12","end":"1928-05-01","name":"ROMANS"}],"M
176":[{"name":"ZIBIDO SAN GIACOMO","end":"1965-07-05","start":"1870-08-02"},
{"name":"ZIBIDO SAN GIACOMO","start":"1965-07-06","end":"1970-02-11"},{"name
":"ZIBIDO SAN GIACOMO","end":"9999-12-31","start":"1970-02-12"}],"A822":[{"n
ame":"BESENO","start":"1929-03-26","end":"1947-07-09"}],"G280":[{"end":"1928
-07-29","start":"1861-03-17","name":"PALLANZENO"},{"start":"1947-01-13","end
":"1992-05-22","name":"PALLANZENO"},{"end":"9999-12-31","start":"1992-05-23"
,"name":"PALLANZENO"}],"L035":[{"name":"TALAMONA","end":"9999-12-31","start"
:"1861-03-17"}],"G113":[{"end":"1927-11-07","start":"1861-03-17","name":"ORI
STANO"},{"start":"1927-11-08","end":"1931-02-19","name":"ORISTANO"},{"name":
"ORISTANO","end":"1947-06-06","start":"1931-02-20"},{"start":"1947-06-07","e
nd":"1957-07-01","name":"ORISTANO"},{"start":"1957-07-02","end":"1974-08-19"
,"name":"ORISTANO"},{"name":"ORISTANO","start":"1974-08-20","end":"9999-12-3
1"}],"I515":[{"start":"1861-03-17","end":"1927-12-27","name":"SCARPIZZOLO"}]
,"H787":[{"start":"1923-04-27","end":"1948-01-09","name":"SAN CANZIAN D'ISON
ZO"},{"name":"SAN CANZIAN D'ISONZO","end":"9999-12-31","start":"1948-01-10"}
,{"name":"SAN CANZIANO","end":"1923-02-07","start":"1921-01-05"},{"start":"1
923-02-08","end":"1923-04-26","name":"SAN CANZIANO"}],"D920":[{"name":"GARES
SIO","end":"9999-12-31","start":"1861-03-17"}],"I730":[{"name":"SILANUS","en
d":"1927-01-11","start":"1861-03-17"},{"end":"9999-12-31","start":"1927-01-1
2","name":"SILANUS"}],"B482":[{"name":"CAMINO","end":"1929-05-22","start":"1
861-03-17"},{"name":"CAMINO","start":"1929-05-23","end":"9999-12-31"}],"E873
":[{"name":"MAMMOLA","end":"9999-12-31","start":"1861-03-17"}],"A837":[{"sta
rt":"1866-11-19","end":"1928-04-14","name":"BEVILACQUA"},{"end":"9999-12-31"
,"start":"1947-09-30","name":"BEVILACQUA"}],"A588":[{"name":"BALDICHIERI D'A
STI","start":"1861-03-17","end":"1928-06-25"},{"start":"1928-06-26","end":"1
929-04-11","name":"BALDICHIERI D'ASTI"},{"name":"BALDICHIERI D'ASTI","end":"
1935-04-07","start":"1929-04-12"},{"name":"BALDICHIERI D'ASTI","start":"1935
-04-08","end":"1947-01-09"},{"name":"BALDICHIERI D'ASTI","start":"1947-01-10
","end":"9999-12-31"}],"C606":[{"end":"1928-01-27","start":"1861-03-17","nam
e":"CHIANCHE"},{"name":"CHIANCHE","end":"1945-08-27","start":"1928-01-28"},{
"start":"1945-08-28","end":"9999-12-31","name":"CHIANCHE"}],"L750":[{"start"
:"1865-12-14","end":"1927-01-11","name":"VERCELLI"},{"name":"VERCELLI","end"
:"9999-12-31","start":"1927-01-12"}],"D457":[{"name":"FAEDO","end":"1923-02-
13","start":"1920-10-16"},{"name":"FAEDO","start":"1923-02-14","end":"1928-0
5-21"},{"end":"2019-12-31","start":"1953-01-04","name":"FAEDO"}],"B615":[{"s
tart":"1861-03-17","end":"1928-02-08","name":"CANNOBIO"},{"start":"1928-02-0
9","end":"1992-05-22","name":"CANNOBIO"},{"name":"CANNOBIO","start":"1992-05
-23","end":"9999-12-31"}],"H359":[{"end":"9999-12-31","start":"1861-03-17","
name":"RIZZICONI"}],"C797":[{"name":"CLOZ","start":"1920-10-16","end":"1923-
02-13"},{"start":"1923-02-14","end":"1928-12-31","name":"CLOZ"},{"start":"19
47-01-10","end":"2019-12-31","name":"CLOZ"}],"H053":[{"name":"PRETURO","star
t":"1861-03-17","end":"1927-09-20"}],"F680":[{"name":"MONTIRONE","start":"18
61-03-17","end":"1928-01-06"},{"name":"MONTIRONE","end":"9999-12-31","start"
:"1955-11-26"}],"I275":[{"name":"SANT'ANGELO DI PIOVE DI SACCO","end":"9999-
12-31","start":"1867-08-26"}],"H746":[{"name":"SAMBUCO","start":"1861-03-17"
,"end":"1928-07-28"},{"end":"9999-12-31","start":"1947-01-10","name":"SAMBUC
O"}],"C259":[{"start":"1863-01-19","end":"1961-01-31","name":"CASTEL SAN GIO
RGIO"},{"name":"CASTEL SAN GIORGIO","start":"1961-02-01","end":"9999-12-31"}
],"M335":[{"name":"BELLAGIO","start":"2014-02-04","end":"9999-12-31"}],"C852
":[{"end":"9999-12-31","start":"1893-12-05","name":"COLLECCHIO"}],"B421":[{"
name":"CALOGNA","start":"1861-03-17","end":"1928-04-15"}],"A985":[{"start":"
1861-03-17","end":"1927-11-23","name":"BORDOGNA"}],"D091":[{"end":"1927-01-1
1","start":"1863-02-23","name":"DUE COSSANI"},{"end":"1928-05-21","start":"1
927-01-12","name":"DUE COSSANI"}],"E970":[{"name":"MAROSTICA","end":"1938-12
-21","start":"1866-11-19"},{"name":"MAROSTICA","start":"1938-12-22","end":"9
999-12-31"}],"H731":[{"end":"9999-12-31","start":"1904-03-11","name":"SALVIR
OLA"},{"end":"1904-03-10","start":"1868-08-01","name":"SALVIROLA CREMASCA"}]
,"A348":[{"name":"AQUINO","end":"1927-01-11","start":"1861-03-17"},{"name":"
AQUINO","start":"1927-01-12","end":"9999-12-31"}],"H273":[{"end":"1970-03-02
","start":"1861-03-17","name":"RICCIA"},{"name":"RICCIA","start":"1970-03-03
","end":"9999-12-31"}],"D823":[{"name":"FURCI","start":"1861-03-17","end":"9
999-12-31"}],"F107":[{"name":"MELILLI","end":"1979-08-09","start":"1861-03-1
7"},{"name":"MELILLI","end":"9999-12-31","start":"1979-08-10"}],"A881":[{"en
d":"9999-12-31","start":"1861-03-17","name":"BISACCIA"}],"Z308":[{"name":"Re
pubblica Centrafricana","end":"9999-12-31","start":"0000-01-01"}],"G390":[{"
end":"1928-10-12","start":"1863-02-16","name":"PAVONE D'ALESSANDRIA"}],"F325
":[{"name":"MONASTERO BORMIDA","start":"1863-02-16","end":"1927-07-05"},{"na
me":"MONASTERO BORMIDA","end":"1935-04-07","start":"1927-07-06"},{"name":"MO
NASTERO BORMIDA","end":"9999-12-31","start":"1935-04-08"}],"M102":[{"name":"
VIZZOLO PREDABISSI","start":"1863-08-10","end":"9999-12-31"}],"L599":[{"name
":"VALLECROSIA","end":"1923-12-05","start":"1861-03-17"},{"name":"VALLECROSI
A","start":"1923-12-06","end":"1925-08-21"},{"name":"VALLECROSIA","start":"1
925-08-22","end":"9999-12-31"}],"G944":[{"name":"POVEGLIANO","start":"1866-1
1-19","end":"9999-12-31"}],"H146":[{"start":"1920-10-16","end":"1923-02-13",
"name":"RABBI"},{"name":"RABBI","end":"9999-12-31","start":"1923-02-14"}],"G
347":[{"name":"PARTANNA","start":"1861-03-17","end":"9999-12-31"}],"G296":[{
"start":"1920-10-16","end":"1923-02-13","name":"PAL\u00d9"},{"start":"1923-0
2-14","end":"1929-03-24","name":"PAL\u00d9"},{"end":"1959-07-07","start":"19
48-01-25","name":"PAL\u00d9"},{"end":"9999-12-31","start":"1959-07-08","name
":"PAL\u00d9 DEL FERSINA"}],"M028":[{"start":"1861-03-17","end":"1927-01-11"
,"name":"VILLATA"},{"name":"VILLATA","end":"9999-12-31","start":"1927-01-12"
}],"C065":[{"end":"9999-12-31","start":"1861-03-17","name":"CASTEL BOLOGNESE
"}],"H600":[{"name":"ROVEGRO","end":"1928-10-28","start":"1861-03-17"}],"C67
2":[{"end":"1966-02-09","start":"1866-11-19","name":"CIBIANA"},{"start":"196
6-02-10","end":"1968-09-17","name":"CIBIANA"},{"start":"1968-09-18","end":"9
999-12-31","name":"CIBIANA DI CADORE"}],"L249":[{"name":"TORRE BERETTI","end
":"1928-07-27","start":"1861-03-17"}],"H131":[{"start":"1868-02-14","end":"9
999-12-31","name":"QUINTO DI TREVISO"}],"B808":[{"start":"1861-03-17","end":
"1923-12-21","name":"CAROSINO"},{"end":"9999-12-31","start":"1923-12-22","na
me":"CAROSINO"}],"L463":[{"end":"1928-06-13","start":"1861-03-17","name":"TU
ILI"},{"end":"1946-03-07","start":"1928-06-14","name":"TUILI"},{"name":"TUIL
I","end":"1958-08-09","start":"1946-03-08"},{"name":"TUILI","start":"1958-08
-10","end":"1974-08-19"},{"end":"2005-12-31","start":"1974-08-20","name":"TU
ILI"},{"name":"TUILI","end":"2016-04-27","start":"2006-01-01"},{"name":"TUIL
I","start":"2016-04-28","end":"9999-12-31"}],"D648":[{"end":"1923-02-13","st
art":"1920-10-16","name":"FOIANA"},{"start":"1923-02-14","end":"1927-01-11",
"name":"FOIANA"},{"end":"1929-05-18","start":"1927-01-12","name":"FOIANA"}],
"G546":[{"name":"PIAN CAMUNO","start":"1863-01-26","end":"1929-02-20"},{"nam
e":"PIAN CAMUNO","end":"9999-12-31","start":"1957-02-14"}],"F999":[{"end":"1
929-04-11","start":"1866-11-19","name":"ODERZO"},{"name":"ODERZO","start":"1
929-04-12","end":"9999-12-31"}],"L842":[{"end":"9999-12-31","start":"1862-11
-10","name":"VICO DEL GARGANO"}],"F707":[{"name":"MORANO SUL PO","end":"9999
-12-31","start":"1863-02-16"}],"L694":[{"end":"1923-02-13","start":"1920-10-
16","name":"VASIO"},{"start":"1923-02-14","end":"1928-06-29","name":"VASIO"}
],"F696":[{"name":"MONTORSO VICENTINO","end":"9999-12-31","start":"1866-11-1
9"}],"E653":[{"start":"1920-10-16","end":"1923-02-13","name":"LODRONE"},{"en
d":"1928-04-16","start":"1923-02-14","name":"LODRONE"}],"E917":[{"name":"MAR
ANZANA","start":"1861-03-17","end":"1935-04-07"},{"name":"MARANZANA","start"
:"1935-04-08","end":"9999-12-31"}],"I862":[{"end":"1927-05-17","start":"1861
-03-17","name":"SORRENTO"},{"name":"SORRENTO","end":"1946-04-23","start":"19
27-05-18"},{"start":"1946-04-24","end":"9999-12-31","name":"SORRENTO"}],"I44
3":[{"name":"SARROCH","start":"1861-03-17","end":"1974-08-19"},{"name":"SARR
OCH","start":"1974-08-20","end":"9999-12-31"}],"G838":[{"name":"PONTECORVO",
"end":"1927-01-11","start":"1861-03-17"},{"name":"PONTECORVO","end":"9999-12
-31","start":"1927-01-12"}],"C740":[{"name":"CISTERNA DI LATINA","end":"9999
-12-31","start":"1946-10-10"},{"name":"CISTERNA DI LITTORIA","end":"1945-04-
08","start":"1935-06-15"},{"name":"CISTERNA DI LITTORIA","start":"1945-04-09
","end":"1946-10-09"},{"name":"CISTERNA DI ROMA","end":"1932-11-05","start":
"1873-01-14"},{"name":"CISTERNA DI ROMA","end":"1933-09-12","start":"1932-11
-06"},{"end":"1934-10-25","start":"1933-09-13","name":"CISTERNA DI ROMA"},{"
name":"CISTERNA DI ROMA","end":"1934-12-17","start":"1934-10-26"},{"name":"C
ISTERNA DI ROMA","end":"1935-06-14","start":"1934-12-18"}],"G587":[{"name":"
PIAZZOLA SUL BRENTA","end":"1986-05-26","start":"1882-07-20"},{"name":"PIAZZ
OLA SUL BRENTA","start":"1986-05-27","end":"1988-08-26"},{"name":"PIAZZOLA S
UL BRENTA","start":"1988-08-27","end":"9999-12-31"}],"I269":[{"start":"1921-
01-05","end":"1923-02-07","name":"SANT'ANDREA"},{"name":"SANT'ANDREA","end":
"1923-04-26","start":"1923-02-08"},{"start":"1923-04-27","end":"1927-01-11",
"name":"SANT'ANDREA DI GORIZIA"},{"start":"1927-01-12","end":"1927-01-12","n
ame":"SANT'ANDREA DI GORIZIA"}],"E314":[{"start":"1928-04-03","end":"9999-12
-31","name":"INVORIO"}],"L328":[{"name":"TRANI","end":"2009-07-26","start":"
1861-03-17"},{"name":"TRANI","start":"2009-07-27","end":"9999-12-31"}],"G713
":[{"end":"1927-01-11","start":"1877-12-28","name":"PISTOIA"},{"name":"PISTO
IA","start":"1927-01-12","end":"1939-10-25"},{"name":"PISTOIA","end":"9999-1
2-31","start":"1939-10-26"}],"I601":[{"start":"1963-04-17","end":"9999-12-31
","name":"SEMPRONIANO"}],"H187":[{"start":"1861-03-17","end":"9999-12-31","n
ame":"RAPONE"}],"C814":[{"end":"9999-12-31","start":"1861-03-17","name":"COD
IGORO"}],"I130":[{"end":"1927-01-11","start":"1862-11-24","name":"SAN POTITO
 SANNITICO"},{"name":"SAN POTITO SANNITICO","end":"1945-08-31","start":"1927
-01-12"},{"start":"1945-09-01","end":"9999-12-31","name":"SAN POTITO SANNITI
CO"}],"B264":[{"end":"9999-12-31","start":"1861-03-17","name":"BULTEI"}],"H4
45":[{"name":"ROCCASICURA","end":"1970-03-02","start":"1861-03-17"},{"start"
:"1970-03-03","end":"9999-12-31","name":"ROCCASICURA"}],"L150":[{"name":"TES
SENNANO","start":"1871-01-15","end":"1927-01-11"},{"name":"TESSENNANO","end"
:"9999-12-31","start":"1927-01-12"}],"B501":[{"name":"CAMPARADA","end":"2009
-06-29","start":"1861-03-17"},{"name":"CAMPARADA","start":"2009-06-30","end"
:"9999-12-31"}],"C197":[{"start":"1885-02-06","end":"1970-03-02","name":"CAS
TELMAURO"},{"end":"9999-12-31","start":"1970-03-03","name":"CASTELMAURO"}],"
G386":[{"name":"PAUPISI","end":"1912-12-31","start":"1892-03-28"},{"name":"P
AUPISI","start":"1913-01-01","end":"9999-12-31"}],"H464":[{"start":"1861-03-
17","end":"1927-01-11","name":"ROCCHETTA CENGIO"},{"name":"ROCCHETTA CENGIO"
,"start":"1927-01-12","end":"1928-07-17"}],"I326":[{"name":"SANTE MARIE","en
d":"9999-12-31","start":"1869-01-08"}],"E408":[{"start":"1866-11-19","end":"
1867-11-24","name":"LAGO"},{"start":"1867-11-25","end":"1868-07-27","name":"
SAN GIORGIO DI LAGO"}],"H693":[{"start":"1861-03-17","end":"1970-03-02","nam
e":"SALCITO"},{"name":"SALCITO","start":"1970-03-03","end":"9999-12-31"}],"I
532":[{"start":"1866-11-19","end":"9999-12-31","name":"SCHIVENOGLIA"}],"A139
":[{"name":"ALBARETTO DELLA TORRE","start":"1862-12-19","end":"1928-05-01"},
{"name":"ALBARETTO DELLA TORRE","end":"9999-12-31","start":"1947-03-30"}],"B
245":[{"name":"BUDAGNE","start":"1921-01-05","end":"1923-02-07"},{"start":"1
923-02-08","end":"1927-01-11","name":"BUDAGNE"},{"name":"BUDAGNE","start":"1
927-01-12","end":"1928-04-28"}],"B632":[{"name":"MANDELA","end":"9999-12-31"
,"start":"1872-12-27"}],"L552":[{"end":"1923-02-13","start":"1920-10-16","na
me":"VALDAORA"},{"name":"VALDAORA","start":"1923-02-14","end":"1927-01-11"},
{"end":"1928-10-17","start":"1927-01-12","name":"VALDAORA"},{"end":"9999-12-
31","start":"1955-03-02","name":"VALDAORA"}],"D666":[{"start":"1861-03-17","
end":"1927-01-11","name":"FONTAINEMORE"},{"name":"FONTAINEMORE","start":"192
7-01-12","end":"1929-01-24"},{"start":"1946-03-10","end":"9999-12-31","name"
:"FONTAINEMORE"}],"G568":[{"name":"PIANO DI SORRENTO","start":"1865-10-25","
end":"1927-05-17"},{"name":"PIANO DI SORRENTO","end":"9999-12-31","start":"1
946-04-24"}],"F430":[{"name":"MONTASOLA","start":"1861-03-17","end":"1923-03
-18"},{"name":"MONTASOLA","end":"1927-01-11","start":"1923-03-19"},{"name":"
MONTASOLA","end":"1928-03-12","start":"1927-01-12"},{"name":"MONTASOLA","end
":"9999-12-31","start":"1946-03-08"}],"G856":[{"start":"1879-06-02","end":"1
928-01-06","name":"PONTE SAN PIETRO"},{"name":"PONTE SAN PIETRO","start":"19
28-01-07","end":"1948-02-25"},{"end":"1962-08-01","start":"1948-02-26","name
":"PONTE SAN PIETRO"},{"end":"1964-02-29","start":"1962-08-02","name":"PONTE
 SAN PIETRO"},{"start":"1964-03-01","end":"9999-12-31","name":"PONTE SAN PIE
TRO"}],"D969":[{"start":"1873-11-10","end":"1926-02-05","name":"GENOVA"},{"n
ame":"GENOVA","end":"9999-12-31","start":"1926-02-06"}],"B595":[{"name":"ORV
INIO","end":"1923-03-18","start":"1863-04-13"},{"start":"1923-03-19","end":"
1927-01-11","name":"ORVINIO"},{"name":"ORVINIO","end":"9999-12-31","start":"
1927-01-12"}],"F952":[{"start":"1861-03-17","end":"9999-12-31","name":"NOVAR
A"}],"H168":[{"name":"RAMACCA","start":"1861-03-17","end":"1934-03-06"},{"na
me":"RAMACCA","start":"1934-03-07","end":"9999-12-31"}],"I527":[{"start":"18
66-11-19","end":"9999-12-31","name":"SCHIAVON"}],"F370":[{"end":"9999-12-31"
,"start":"1861-03-17","name":"MONGRASSANO"}],"E104":[{"name":"GORNATE INFERI
ORE","start":"1861-03-17","end":"1927-01-11"},{"end":"1928-04-06","start":"1
927-01-12","name":"GORNATE INFERIORE"},{"name":"GORNATE OLONA","end":"9999-1
2-31","start":"1928-04-07"}],"B627":[{"start":"1861-03-17","end":"1927-01-11
","name":"CANTALICE"},{"name":"CANTALICE","start":"1927-01-12","end":"1928-0
2-20"},{"name":"CANTALICE","end":"9999-12-31","start":"1946-11-12"}],"L012":
[{"name":"SUS\u00c0","start":"1920-10-16","end":"1923-02-13"},{"start":"1923
-02-14","end":"1929-01-24","name":"SUS\u00c0"}],"A260":[{"name":"AMBLAR","en
d":"1923-02-13","start":"1920-10-16"},{"name":"AMBLAR","end":"1928-06-15","s
tart":"1923-02-14"},{"name":"AMBLAR","end":"2015-12-31","start":"1952-09-20"
}],"I695":[{"end":"1974-08-19","start":"1861-03-17","name":"SESTU"},{"start"
:"1974-08-20","end":"9999-12-31","name":"SESTU"}],"F143":[{"name":"MERCURAGO
","end":"1902-01-23","start":"1861-03-17"},{"name":"MERCURAGO","end":"1928-0
2-25","start":"1902-01-24"}],"F809":[{"start":"1887-06-06","end":"9999-12-31
","name":"MURAZZANO"}],"B998":[{"end":"9999-12-31","start":"1863-01-19","nam
e":"CASSANO DELLE MURGE"}],"D651":[{"name":"FOLGARIA","start":"1920-10-16","
end":"1923-02-13"},{"start":"1923-02-14","end":"1929-05-27","name":"FOLGARIA
"},{"name":"FOLGARIA","end":"9999-12-31","start":"1929-05-28"}],"G861":[{"na
me":"PONTI","end":"9999-12-31","start":"1861-03-17"}],"F202":[{"start":"1861
-03-17","end":"9999-12-31","name":"MIGNANEGO"}],"C044":[{"name":"CASTAGNETO"
,"start":"1861-03-17","end":"1900-03-24"},{"name":"CASTAGNETO CARDUCCI","end
":"1925-11-29","start":"1907-11-08"},{"end":"1929-06-03","start":"1925-11-30
","name":"CASTAGNETO CARDUCCI"},{"start":"1929-06-04","end":"9999-12-31","na
me":"CASTAGNETO CARDUCCI"},{"name":"CASTAGNETO MARITTIMO","end":"1907-11-07"
,"start":"1900-03-25"}],"G602":[{"name":"PIENZA","end":"9999-12-31","start":
"1861-03-17"}],"G965":[{"start":"1869-01-22","end":"9999-12-31","name":"POZZ
UOLO MARTESANA"}],"M312":[{"name":"LONATO DEL GARDA","start":"2007-11-02","e
nd":"9999-12-31"}],"D975":[{"name":"GERACE","end":"1905-08-07","start":"1861
-03-17"},{"end":"9999-12-31","start":"1941-01-23","name":"GERACE"},{"start":
"1905-08-08","end":"1941-01-22","name":"GERACE SUPERIORE"}],"B473":[{"name":
"CAMERI","end":"9999-12-31","start":"1861-03-17"}],"E882":[{"name":"MANDURIA
","start":"1861-03-17","end":"1923-12-21"},{"name":"MANDURIA","end":"9999-12
-31","start":"1923-12-22"}],"L054":[{"name":"TARRES","end":"1927-01-11","sta
rt":"1923-04-27"},{"name":"TARRES","end":"1928-12-09","start":"1927-01-12"},
{"name":"TARRES DI SOTTO","end":"1923-02-13","start":"1920-10-16"},{"name":"
TARRES DI SOTTO","end":"1923-04-26","start":"1923-02-14"}],"F056":[{"name":"
MATTARELLO","start":"1920-10-16","end":"1923-02-13"},{"name":"MATTARELLO","e
nd":"1926-09-30","start":"1923-02-14"}],"D218":[{"end":"1935-06-05","start":
"1861-03-17","name":"CURINGA"},{"end":"9999-12-31","start":"1935-06-06","nam
e":"CURINGA"}],"F317":[{"end":"1928-06-13","start":"1861-03-17","name":"MOMO
"},{"start":"1928-06-14","end":"9999-12-31","name":"MOMO"}],"I407":[{"start"
:"1952-08-17","end":"9999-12-31","name":"SAN VITO LO CAPO"}],"F914":[{"name"
:"NOCETO","start":"1861-03-17","end":"9999-12-31"}],"L916":[{"name":"VILLABA
TE","start":"1861-03-17","end":"9999-12-31"}],"E289":[{"start":"1865-12-25",
"end":"9999-12-31","name":"IMOLA"}],"I034":[{"start":"1863-05-06","end":"999
9-12-31","name":"SAN MICHELE DI SERINO"}],"A351":[{"name":"ARAGONA","end":"1
955-02-14","start":"1892-07-18"},{"name":"ARAGONA","start":"1955-02-15","end
":"9999-12-31"}],"E528":[{"name":"LENTA","start":"1861-03-17","end":"1927-01
-11"},{"name":"LENTA","end":"9999-12-31","start":"1927-01-12"}],"L514":[{"na
me":"USSASSAI","start":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12
","end":"2005-12-31","name":"USSASSAI"},{"end":"2016-04-27","start":"2006-01
-01","name":"USSASSAI"},{"start":"2016-04-28","end":"9999-12-31","name":"USS
ASSAI"}],"D871":[{"end":"1927-01-11","start":"1863-02-23","name":"GALLIATE L
OMBARDO"},{"start":"1927-01-12","end":"9999-12-31","name":"GALLIATE LOMBARDO
"}],"F516":[{"start":"1861-03-17","end":"9999-12-31","name":"MONTEGALLO"}],"
H221":[{"name":"REGALBUTO","end":"1927-01-11","start":"1861-03-17"},{"name":
"REGALBUTO","start":"1927-01-12","end":"9999-12-31"}],"M207":[{"end":"9999-1
2-31","start":"1967-10-15","name":"LARIANO"}],"H392":[{"name":"ROCCA D'ARAZZ
O","start":"1861-03-17","end":"1929-06-12"},{"name":"ROCCA D'ARAZZO","end":"
1935-04-07","start":"1929-06-13"},{"name":"ROCCA D'ARAZZO","end":"1947-12-23
","start":"1935-04-08"},{"end":"9999-12-31","start":"1947-12-24","name":"ROC
CA D'ARAZZO"}],"B039":[{"start":"1861-03-17","end":"1923-11-28","name":"BORG
O SANT'AGATA"}],"C382":[{"name":"CAVARIA CON PREMEZZO","end":"1927-01-11","s
tart":"1924-01-18"},{"end":"9999-12-31","start":"1927-01-12","name":"CAVARIA
 CON PREMEZZO"},{"start":"1892-04-04","end":"1924-01-17","name":"CAVARIA ED 
UNITI"}],"D005":[{"end":"2018-03-30","start":"1863-02-06","name":"CORIGLIANO
 CALABRO"}],"F276":[{"name":"MOIO DE' CALVI","start":"1863-07-13","end":"192
7-09-13"},{"name":"MOIO DE' CALVI","start":"1956-08-26","end":"9999-12-31"}]
,"M354":[{"start":"2016-01-01","end":"9999-12-31","name":"CASTEL IVANO"}],"L
274":[{"name":"TORRE ORSAIA","start":"1861-03-17","end":"1929-04-11"},{"star
t":"1929-04-12","end":"9999-12-31","name":"TORRE ORSAIA"}],"F061":[{"end":"1
980-08-09","start":"1863-07-13","name":"MAZARA DEL VALLO"},{"name":"MAZARA D
EL VALLO","start":"1980-08-10","end":"9999-12-31"}],"H282":[{"name":"RIETI",
"start":"1875-03-05","end":"1923-03-18"},{"name":"RIETI","start":"1923-03-19
","end":"1927-01-11"},{"name":"RIETI","start":"1927-01-12","end":"1928-02-20
"},{"name":"RIETI","start":"1928-02-21","end":"1939-11-14"},{"end":"1946-03-
08","start":"1939-11-15","name":"RIETI"},{"start":"1946-03-09","end":"1946-1
1-11","name":"RIETI"},{"name":"RIETI","start":"1946-11-12","end":"9999-12-31
"}],"E925":[{"end":"9999-12-31","start":"1861-03-17","name":"MARCELLINARA"}]
,"M420":[{"end":"9999-12-31","start":"2019-02-01","name":"LU E CUCCARO MONFE
RRATO"}],"A739":[{"end":"9999-12-31","start":"1955-07-10","name":"BELFORTE D
EL CHIENTI"},{"start":"1862-12-29","end":"1955-07-09","name":"BELFORTE DEL C
HIENTO"}],"A366":[{"end":"9999-12-31","start":"1861-03-17","name":"ARCEVIA"}
],"G009":[{"name":"OGGIONO","start":"1861-03-17","end":"1928-10-29"},{"name"
:"OGGIONO","start":"1928-10-30","end":"1992-04-15"},{"start":"1992-04-16","e
nd":"9999-12-31","name":"OGGIONO"}],"G303":[{"name":"PANCALIERI","start":"18
61-03-17","end":"1928-05-22"},{"start":"1928-05-23","end":"1946-12-18","name
":"PANCALIERI"},{"start":"1946-12-19","end":"9999-12-31","name":"PANCALIERI"
}],"H889":[{"end":"9999-12-31","start":"1864-06-23","name":"SAN GIORGIO MORG
ETO"}],"C321":[{"start":"1865-12-25","end":"9999-12-31","name":"CASTIGNANO"}
],"C292":[{"name":"CASTENASO","end":"9999-12-31","start":"1861-03-17"}],"I22
0":[{"end":"1921-05-11","start":"1863-04-10","name":"SANTA LUCIA DEL MELA"},
{"end":"1934-03-16","start":"1921-05-12","name":"SANTA LUCIA DEL MELA"},{"st
art":"1934-03-17","end":"9999-12-31","name":"SANTA LUCIA DEL MELA"}],"H768":
[{"start":"1866-11-19","end":"9999-12-31","name":"SAN BELLINO"}],"D578":[{"s
tart":"1867-08-05","end":"9999-12-31","name":"FIESSO D'ARTICO"}],"G676":[{"s
tart":"1863-02-16","end":"1929-04-05","name":"PINO D'ASTI"},{"name":"PINO D'
ASTI","end":"9999-12-31","start":"1947-06-23"}],"G979":[{"name":"PRALORMO","
end":"9999-12-31","start":"1861-03-17"}],"E821":[{"name":"MAGNANO","end":"19
27-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"1992-04-15","na
me":"MAGNANO"},{"name":"MAGNANO","start":"1992-04-16","end":"9999-12-31"}],"
E704":[{"end":"2006-03-03","start":"1861-03-17","name":"LOVERE"},{"end":"999
9-12-31","start":"2006-03-04","name":"LOVERE"}],"C542":[{"name":"CERVA","end
":"9999-12-31","start":"1861-03-17"}],"E093":[{"end":"1927-01-11","start":"1
861-03-17","name":"GORGOGLIONE"},{"name":"GORGOGLIONE","end":"9999-12-31","s
tart":"1927-01-12"}],"E399":[{"name":"LACIGNO","start":"1920-10-16","end":"1
923-02-13"},{"name":"LACIGNO","start":"1923-02-14","end":"1923-04-26"},{"sta
rt":"1923-04-27","end":"1927-01-11","name":"LACINIGO"},{"start":"1927-01-12"
,"end":"1929-02-14","name":"LACINIGO"}],"A567":[{"end":"1970-03-02","start":
"1863-04-10","name":"BAGNOLI DEL TRIGNO"},{"name":"BAGNOLI DEL TRIGNO","star
t":"1970-03-03","end":"9999-12-31"}],"F743":[{"name":"MOROZZO","start":"1861
-03-17","end":"9999-12-31"}],"A451":[{"end":"1929-02-21","start":"1861-03-17
","name":"ARTOGNE"},{"end":"1959-07-25","start":"1957-02-14","name":"ARTOGNE
"},{"name":"ARTOGNE","end":"9999-12-31","start":"1959-07-26"}],"D831":[{"end
":"1928-06-15","start":"1923-04-27","name":"FUSINE IN VALROMANA"},{"name":"R
OCCALBA","end":"1923-02-07","start":"1921-01-05"},{"name":"ROCCALBA","end":"
1923-04-26","start":"1923-02-08"}],"I904":[{"start":"1872-01-01","end":"1924
-02-21","name":"SPILIMBERGO"},{"end":"1968-04-05","start":"1924-02-22","name
":"SPILIMBERGO"},{"name":"SPILIMBERGO","end":"1990-11-28","start":"1968-04-0
6"},{"name":"SPILIMBERGO","end":"9999-12-31","start":"1990-11-29"}],"B140":[
{"start":"1861-03-17","end":"2002-10-25","name":"BREMBILLA"},{"name":"BREMBI
LLA","start":"2002-10-26","end":"2014-02-03"}],"F417":[{"start":"1863-03-30"
,"end":"1939-02-04","name":"MONTALTO PAVESE"},{"end":"9999-12-31","start":"1
939-02-05","name":"MONTALTO PAVESE"}],"I307":[{"start":"1861-03-17","end":"9
999-12-31","name":"SANT'ARSENIO"}],"H723":[{"end":"1927-01-11","start":"1861
-03-17","name":"SALTRIO"},{"name":"SALTRIO","end":"1928-01-12","start":"1927
-01-12"},{"start":"1953-10-15","end":"9999-12-31","name":"SALTRIO"}],"B079":
[{"name":"BOSIA","end":"1928-04-14","start":"1861-03-17"},{"name":"BOSIA","s
tart":"1947-01-10","end":"9999-12-31"}],"B373":[{"name":"CAJONVICO","end":"1
928-03-28","start":"1861-03-17"}],"L020":[{"name":"SUZZARA","start":"1866-11
-19","end":"9999-12-31"}],"B909":[{"end":"1873-10-17","start":"1861-03-17","
name":"CASALPOGLIO"}],"C615":[{"name":"CHIARAVALLE","end":"1928-05-21","star
t":"1865-12-11"},{"name":"CHIARAVALLE","start":"1928-05-22","end":"1928-06-2
6"},{"name":"CHIARAVALLE","start":"1928-06-27","end":"1947-11-28"},{"start":
"1947-11-29","end":"1948-04-09","name":"CHIARAVALLE"},{"start":"1948-04-10",
"end":"1956-08-08","name":"CHIARAVALLE"},{"start":"1956-08-09","end":"9999-1
2-31","name":"CHIARAVALLE"}],"B606":[{"name":"CANISTRO","end":"9999-12-31","
start":"1861-03-17"}],"D935":[{"name":"GATTEO","end":"1992-04-15","start":"1
861-03-17"},{"name":"GATTEO","start":"1992-04-16","end":"9999-12-31"}],"D846
":[{"name":"GAGGINO","end":"1928-05-11","start":"1861-03-17"}],"I725":[{"nam
e":"SIDERNO","end":"9999-12-31","start":"1861-03-17"}],"B433":[{"name":"CALT
RANO","start":"1866-11-19","end":"9999-12-31"}],"F898":[{"start":"1866-11-19
","end":"1968-04-05","name":"NIMIS"},{"end":"9999-12-31","start":"1968-04-06
","name":"NIMIS"}],"C482":[{"end":"9999-12-31","start":"1862-12-29","name":"
CERANO D'INTELVI"}],"H492":[{"end":"9999-12-31","start":"1861-03-17","name":
"ROGNO"}],"I074":[{"end":"1927-01-11","start":"1863-04-23","name":"SAN PAOLO
 CERVO"},{"end":"1992-04-15","start":"1927-01-12","name":"SAN PAOLO CERVO"},
{"name":"SAN PAOLO CERVO","start":"1992-04-16","end":"2015-12-31"}],"A165":[
{"name":"ALBISSOLA MARINA","start":"1861-03-17","end":"1927-01-11"},{"start"
:"1927-01-12","end":"9999-12-31","name":"ALBISSOLA MARINA"}],"Z314":[{"start
":"0000-01-01","end":"9999-12-31","name":"Benin"}],"I506":[{"name":"SCANZORO
SCIATE","end":"9999-12-31","start":"1927-08-19"}],"C421":[{"name":"CEFAL\u00
d9","end":"1927-10-20","start":"1861-03-17"},{"name":"CEFAL\u00d9","end":"99
99-12-31","start":"1927-10-21"}],"G636":[{"start":"1905-07-30","end":"1928-0
1-25","name":"PIEVE A NIEVOLE"},{"name":"PIEVE A NIEVOLE","end":"1934-11-02"
,"start":"1928-01-26"},{"name":"PIEVE A NIEVOLE","end":"1940-10-07","start":
"1934-11-03"},{"start":"1940-10-08","end":"9999-12-31","name":"PIEVE A NIEVO
LE"}],"D538":[{"name":"FERENTILLO","start":"1861-03-17","end":"1927-01-11"},
{"start":"1927-01-12","end":"9999-12-31","name":"FERENTILLO"}],"G939":[{"nam
e":"POSTIGLIONE","end":"9999-12-31","start":"1861-03-17"}],"D887":[{"name":"
GALLUZZO","end":"1928-12-16","start":"1865-08-13"}],"A466":[{"end":"1927-01-
11","start":"1861-03-17","name":"ASIGLIANO"},{"start":"1927-01-12","end":"19
28-03-08","name":"ASIGLIANO"},{"end":"9999-12-31","start":"1928-03-09","name
":"ASIGLIANO VERCELLESE"}],"B852":[{"name":"CARUGO AROSIO","start":"1929-12-
28","end":"1950-04-05"},{"name":"CARUGO CON AROSIO","start":"1929-05-16","en
d":"1929-12-27"}],"F241":[{"end":"9999-12-31","start":"1866-11-19","name":"M
IRANO"}],"G403":[{"name":"PEDASO","start":"1861-03-17","end":"2009-07-11"},{
"name":"PEDASO","start":"2009-07-12","end":"9999-12-31"}],"E563":[{"name":"L
EVERANO","end":"9999-12-31","start":"1861-03-17"}],"L991":[{"name":"VILLANOV
A TRUSCHEDU","end":"1928-06-26","start":"1861-03-17"},{"end":"1974-08-19","s
tart":"1951-01-12","name":"VILLANOVA TRUSCHEDU"},{"name":"VILLANOVA TRUSCHED
U","end":"9999-12-31","start":"1974-08-20"}],"B259":[{"name":"BUJA","end":"1
968-04-05","start":"1866-11-19"},{"start":"1968-04-06","end":"9999-12-31","n
ame":"BUJA"}],"M320":[{"start":"2014-01-01","end":"9999-12-31","name":"VALSA
MOGGIA"}],"A097":[{"name":"AIDOMAGGIORE","end":"1974-08-19","start":"1861-03
-17"},{"start":"1974-08-20","end":"9999-12-31","name":"AIDOMAGGIORE"}],"F591
":[{"start":"1861-03-17","end":"1935-09-05","name":"MONTEPRANDONE"},{"name":
"MONTEPRANDONE","end":"9999-12-31","start":"1935-09-06"}],"G641":[{"name":"P
IEVE DI BONO","start":"1928-03-07","end":"1952-09-19"},{"end":"2015-12-31","
start":"1952-09-20","name":"PIEVE DI BONO"}],"A779":[{"name":"BENE VAGIENNA"
,"start":"1862-12-19","end":"1928-02-09"},{"name":"BENE VAGIENNA","end":"194
7-05-22","start":"1928-02-10"},{"name":"BENE VAGIENNA","end":"9999-12-31","s
tart":"1947-05-23"}],"L234":[{"name":"TORRADELLO","end":"1872-10-29","start"
:"1861-03-17"}],"E499":[{"name":"LAVINA","start":"1861-03-17","end":"1928-04
-14"}],"M158":[{"name":"ZELO BUON PERSICO","start":"1869-02-13","end":"1992-
04-15"},{"name":"ZELO BUON PERSICO","start":"1992-04-16","end":"9999-12-31"}
],"L895":[{"name":"VIGOLO BASELGA","start":"1920-10-16","end":"1923-02-13"},
{"name":"VIGOLO BASELGA","end":"1928-05-21","start":"1923-02-14"},{"end":"19
68-07-02","start":"1947-01-15","name":"VIGOLO BASELGA"}],"I254":[{"name":"SA
NTA MARINA SALINA","start":"1910-01-10","end":"9999-12-31"}],"H967":[{"name"
:"SAN LORENZO MAGGIORE","end":"9999-12-31","start":"1861-03-17"}],"F330":[{"
end":"9999-12-31","start":"1862-12-19","name":"MONASTEROLO DI SAVIGLIANO"}],
"Z352":[{"name":"Tunisia","end":"9999-12-31","start":"0000-01-01"}],"B065":[
{"start":"1920-10-16","end":"1923-02-13","name":"BORZAGO"},{"end":"1928-10-2
9","start":"1923-02-14","name":"BORZAGO"}],"H364":[{"name":"ROASENDA","start
":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"1927-11-23",
"name":"ROASENDA"},{"start":"1927-11-24","end":"9999-12-31","name":"ROVASEND
A"}],"I426":[{"name":"SARCONI","end":"1929-01-23","start":"1861-03-17"},{"st
art":"1944-03-04","end":"9999-12-31","name":"SARCONI"}],"E308":[{"end":"1992
-04-15","start":"1861-03-17","name":"INTROZZO"},{"name":"INTROZZO","start":"
1992-04-16","end":"2017-12-31"}],"A179":[{"name":"ALDINO","end":"9999-12-31"
,"start":"1955-03-02"},{"start":"1920-10-16","end":"1923-02-13","name":"VALD
AGNO"},{"name":"VALDAGNO","start":"1923-02-14","end":"1928-08-15"},{"end":"1
955-03-01","start":"1948-03-14","name":"VALDAGNO"},{"start":"1928-08-16","en
d":"1948-03-13","name":"VALDAGNO DI TRENTO"}],"I572":[{"name":"SEGHEBBIA","e
nd":"1928-06-30","start":"1861-03-17"}],"A040":[{"name":"ACQUAPENDENTE","sta
rt":"1871-01-15","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31
","name":"ACQUAPENDENTE"}],"F470":[{"end":"1923-02-13","start":"1920-10-16",
"name":"MONTECHIARO"},{"name":"MONTECHIARO","start":"1923-02-14","end":"1927
-01-11"},{"start":"1927-01-12","end":"1929-05-14","name":"MONTECHIARO"}],"G0
98":[{"start":"1861-03-17","end":"1927-01-11","name":"ORIA"},{"start":"1927-
01-12","end":"9999-12-31","name":"ORIA"}],"B672":[{"end":"9999-12-31","start
":"1861-03-17","name":"CAPORCIANO"}],"B740":[{"name":"CARBONARA DI NOLA","en
d":"1927-01-11","start":"1862-12-29"},{"name":"CARBONARA DI NOLA","start":"1
927-01-12","end":"9999-12-31"}],"H123":[{"start":"1883-03-30","end":"1901-05
-09","name":"VIRGILIO"},{"name":"VIRGILIO","start":"1901-05-10","end":"1925-
03-20"},{"name":"VIRGILIO","end":"2014-02-03","start":"1925-03-21"}],"I819":
[{"name":"SOMMA LOMBARDO","end":"1927-01-11","start":"1862-11-07"},{"name":"
SOMMA LOMBARDO","start":"1927-01-12","end":"1927-12-03"},{"start":"1927-12-0
4","end":"9999-12-31","name":"SOMMA LOMBARDO"}],"B197":[{"end":"1992-04-15",
"start":"1861-03-17","name":"BROGNATURO"},{"name":"BROGNATURO","start":"1992
-04-16","end":"9999-12-31"}],"C501":[{"start":"1861-03-17","end":"1928-05-22
","name":"CERENZIA"},{"name":"CERENZIA","start":"1946-12-24","end":"1992-04-
15"},{"name":"CERENZIA","end":"1999-01-12","start":"1992-04-16"},{"name":"CE
RENZIA","end":"9999-12-31","start":"1999-01-13"}],"M412":[{"name":"SOLBIATE 
CON CAGNO","end":"9999-12-31","start":"2019-01-01"}],"C264":[{"start":"1863-
02-16","end":"1929-05-22","name":"CASTEL SAN PIETRO MONFERRATO"}],"I125":[{"
name":"SAN POLO DEI CAVALIERI","start":"1871-01-15","end":"1910-02-16"},{"st
art":"1910-02-17","end":"9999-12-31","name":"SAN POLO DEI CAVALIERI"}],"G523
":[{"start":"1937-02-24","end":"1970-03-02","name":"PETTORANELLO DEL MOLISE"
},{"name":"PETTORANELLO DEL MOLISE","start":"1970-03-03","end":"9999-12-31"}
,{"name":"PETTORANELLO DI MOLISE","end":"1928-08-07","start":"1863-02-06"}],
"B814":[{"start":"1861-03-17","end":"2017-12-31","name":"CARPASIO"}],"A765":
[{"name":"BELMONTE IN SABINA","end":"1923-03-18","start":"1872-09-02"},{"nam
e":"BELMONTE IN SABINA","start":"1923-03-19","end":"1927-01-11"},{"name":"BE
LMONTE IN SABINA","end":"9999-12-31","start":"1927-01-12"}],"G961":[{"end":"
1953-07-15","start":"1861-03-17","name":"POZZOLO FORMIGARO"},{"name":"POZZOL
O FORMIGARO","start":"1953-07-16","end":"9999-12-31"}],"H690":[{"name":"SALA
SCO","end":"1927-01-11","start":"1861-03-17"},{"name":"SALASCO","start":"192
7-01-12","end":"9999-12-31"}],"L604":[{"start":"1961-06-14","end":"1983-02-2
6","name":"VALLEDORIA"},{"name":"VALLEDORIA","start":"1983-02-27","end":"999
9-12-31"}],"F606":[{"start":"1871-01-15","end":"1928-12-29","name":"MONTEROS
I"},{"start":"1928-12-30","end":"1955-05-26","name":"MONTEROSI"},{"name":"MO
NTEROSI","start":"1955-05-27","end":"9999-12-31"}],"C680":[{"name":"CIGLIANO
","start":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"9999
-12-31","name":"CIGLIANO"}],"F909":[{"start":"1861-03-17","end":"1928-08-12"
,"name":"NOCCO"}],"F797":[{"name":"MUGGI\u00d2","end":"2009-06-29","start":"
1861-03-17"},{"name":"MUGGI\u00d2","end":"9999-12-31","start":"2009-06-30"}]
,"B898":[{"name":"CASALMAGGIORE","start":"1861-03-17","end":"1915-04-15"},{"
name":"CASALMAGGIORE","end":"9999-12-31","start":"1915-04-16"}],"E047":[{"na
me":"GIOVINAZZO","end":"1928-10-15","start":"1861-03-17"},{"end":"9999-12-31
","start":"1928-10-16","name":"GIOVINAZZO"}],"F433":[{"end":"9999-12-31","st
art":"1861-03-17","name":"MONTAZZOLI"}],"D655":[{"name":"FOLLO","end":"1923-
12-21","start":"1861-03-17"},{"name":"FOLLO","start":"1923-12-22","end":"192
9-05-28"},{"end":"9999-12-31","start":"1929-05-29","name":"FOLLO"}],"B417":[
{"end":"1927-01-11","start":"1861-03-17","name":"CALLABIANA"},{"end":"1992-0
4-15","start":"1927-01-12","name":"CALLABIANA"},{"end":"9999-12-31","start":
"1992-04-16","name":"CALLABIANA"}],"G865":[{"name":"PONTINIA","end":"1935-12
-30","start":"1935-07-18"},{"name":"PONTINIA","start":"1935-12-31","end":"19
37-05-28"},{"name":"PONTINIA","start":"1937-05-29","end":"1945-04-08"},{"nam
e":"PONTINIA","start":"1945-04-09","end":"9999-12-31"}],"L509":[{"start":"18
61-03-17","end":"9999-12-31","name":"USINI"}],"M192":[{"end":"1928-06-30","s
tart":"1861-03-17","name":"ZORNASCO"}],"F079":[{"end":"1923-02-07","start":"
1921-01-05","name":"MEDANA"},{"name":"MEDANA","end":"1927-01-11","start":"19
23-02-08"},{"name":"MEDANA","end":"1928-05-11","start":"1927-01-12"}],"F373"
:[{"end":"1928-04-07","start":"1861-03-17","name":"MONIGA"},{"end":"1952-01-
21","start":"1947-08-15","name":"MONIGA"},{"name":"MONIGA DEL GARDA","start"
:"1952-01-22","end":"9999-12-31"}],"A449":[{"end":"9999-12-31","start":"1873
-03-06","name":"ARTENA"}],"H712":[{"start":"1867-08-26","end":"1954-10-25","
name":"SAN MAURO DI SALINE"},{"name":"SAN MAURO DI SALINE","end":"9999-12-31
","start":"1954-10-26"}],"G206":[{"start":"1861-03-17","end":"9999-12-31","n
ame":"OZZERO"}],"F140":[{"start":"1861-03-17","end":"1927-01-11","name":"MER
CENASCO"},{"end":"1928-06-15","start":"1927-01-12","name":"MERCENASCO"},{"st
art":"1946-12-24","end":"9999-12-31","name":"MERCENASCO"}],"A263":[{"name":"
AMENDOLARA","start":"1861-03-17","end":"9999-12-31"}],"G956":[{"name":"POZZO
 ALTO","start":"1862-11-07","end":"1929-04-10"}],"H154":[{"name":"RADDUSA","
end":"9999-12-31","start":"1861-03-17"}],"G659":[{"name":"PIGLIO","start":"1
871-01-15","end":"1927-01-11"},{"start":"1927-01-12","end":"9999-12-31","nam
e":"PIGLIO"}],"D869":[{"start":"1869-03-11","end":"1924-01-13","name":"GALLA
RATE"},{"name":"GALLARATE","end":"1927-01-11","start":"1924-01-14"},{"start"
:"1927-01-12","end":"9999-12-31","name":"GALLARATE"}],"D262":[{"name":"DEGAG
NA","start":"1861-03-17","end":"1928-06-13"}],"C107":[{"name":"CASTELFRANCO 
DELL'EMILIA","end":"1929-02-23","start":"1862-10-31"},{"name":"CASTELFRANCO 
DELL'EMILIA","end":"1940-09-06","start":"1929-02-24"},{"name":"CASTELFRANCO 
EMILIA","start":"1940-09-07","end":"9999-12-31"}],"G014":[{"name":"OLBA","st
art":"1861-03-17","end":"1927-01-11"},{"name":"OLBA","start":"1927-01-12","e
nd":"1929-06-12"}],"B591":[{"end":"9999-12-31","start":"1861-03-17","name":"
CANDIDONI"}],"B236":[{"name":"BUBBIO","start":"1861-03-17","end":"1935-04-07
"},{"start":"1935-04-08","end":"9999-12-31","name":"BUBBIO"}],"H458":[{"name
":"ROCCHETTA A VOLTURNO","end":"1928-03-06","start":"1884-04-18"},{"start":"
1934-03-07","end":"1970-03-02","name":"ROCCHETTA A VOLTURNO"},{"name":"ROCCH
ETTA A VOLTURNO","start":"1970-03-03","end":"9999-12-31"}],"F259":[{"name":"
MODIGLIANA","end":"1923-03-18","start":"1861-03-17"},{"start":"1923-03-19","
end":"1992-04-15","name":"MODIGLIANA"},{"name":"MODIGLIANA","end":"9999-12-3
1","start":"1992-04-16"}],"C038":[{"name":"CASSOLNOVO","start":"1861-03-17",
"end":"9999-12-31"}],"E719":[{"name":"LUCINASCO","start":"1861-03-17","end":
"1923-12-21"},{"start":"1958-08-28","end":"9999-12-31","name":"LUCINASCO"}],
"E086":[{"name":"GONNESA","start":"1861-03-17","end":"1938-01-07"},{"name":"
GONNESA","start":"1938-01-08","end":"1940-02-12"},{"name":"GONNESA","end":"1
974-08-19","start":"1945-04-21"},{"name":"GONNESA","start":"1974-08-20","end
":"2005-12-31"},{"name":"GONNESA","start":"2006-01-01","end":"2016-04-27"},{
"name":"GONNESA","end":"9999-12-31","start":"2016-04-28"}],"G554":[{"name":"
PIANE DI SERRAVALLE SESIA","end":"1913-03-19","start":"1863-04-23"},{"start"
:"1913-03-20","end":"1927-01-11","name":"PIANE SESIA"},{"start":"1927-01-12"
,"end":"1927-10-08","name":"PIANE SESIA"}],"F852":[{"name":"NAVELLI","end":"
9999-12-31","start":"1861-03-17"}],"B241":[{"name":"BUCCINIGO","end":"1928-0
1-25","start":"1861-03-17"}],"D788":[{"end":"9999-12-31","start":"1867-07-22
","name":"FRATTA POLESINE"}],"L429":[{"end":"1927-01-11","start":"1870-09-01
","name":"TRINO"},{"end":"9999-12-31","start":"1927-01-12","name":"TRINO"}],
"B470":[{"name":"CAMERATA PICENA","end":"1925-10-15","start":"1863-02-23"},{
"name":"CAMERATA PICENA","end":"1928-06-26","start":"1925-10-16"},{"name":"C
AMERATA PICENA","end":"1989-01-23","start":"1947-11-29"},{"name":"CAMERATA P
ICENA","end":"9999-12-31","start":"1989-01-24"}],"Z529":[{"end":"9999-12-31"
,"start":"0000-01-01","name":"Anguilla"}],"H989":[{"name":"SAN MARTINO","sta
rt":"1923-04-27","end":"1927-01-11"},{"start":"1927-01-12","end":"1928-04-07
","name":"SAN MARTINO"},{"name":"SAN MARTINO IN PASSIRIA","start":"1920-10-1
6","end":"1923-02-13"},{"name":"SAN MARTINO IN PASSIRIA","start":"1923-02-14
","end":"1923-04-26"},{"start":"1953-11-25","end":"9999-12-31","name":"SAN M
ARTINO IN PASSIRIA"}],"F672":[{"end":"9999-12-31","start":"1861-03-17","name
":"MONTICELLI BRUSATI"}],"H754":[{"name":"SAMONE","end":"1923-02-13","start"
:"1920-10-16"},{"name":"SAMONE","start":"1923-02-14","end":"1928-07-27"},{"n
ame":"SAMONE","end":"9999-12-31","start":"1947-01-14"}],"H686":[{"end":"9999
-12-31","start":"1863-04-10","name":"SALENTO"}],"C707":[{"end":"9999-12-31",
"start":"1928-11-03","name":"CINISELLO BALSAMO"}],"C999":[{"name":"CORFINIO"
,"end":"9999-12-31","start":"1928-12-07"},{"name":"PENTIMA","end":"1928-12-0
6","start":"1861-03-17"}],"I048":[{"name":"SANNAZZARO DE' BURGONDI","start":
"1863-08-10","end":"1989-07-26"},{"end":"9999-12-31","start":"1989-07-27","n
ame":"SANNAZZARO DE' BURGONDI"}],"C696":[{"end":"9999-12-31","start":"1861-0
3-17","name":"CIMINNA"}],"D484":[{"name":"FALZES","end":"1923-02-13","start"
:"1920-10-16"},{"name":"FALZES","start":"1923-02-14","end":"1927-01-11"},{"s
tart":"1927-01-12","end":"1928-03-06","name":"FALZES"},{"name":"FALZES","sta
rt":"1928-03-07","end":"9999-12-31"}],"E825":[{"name":"MAGOMADAS","start":"1
861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"2005-12-31","nam
e":"MAGOMADAS"},{"name":"MAGOMADAS","start":"2006-01-01","end":"9999-12-31"}
],"M368":[{"start":"2016-01-01","end":"9999-12-31","name":"MONTESCUDO-MONTE 
COLOMBO"}],"G272":[{"name":"PALERMITI","end":"9999-12-31","start":"1861-03-1
7"}],"C325":[{"end":"9999-12-31","start":"1863-07-13","name":"CASTIONE ANDEV
ENNO"}],"E119":[{"name":"GOZZA","end":"1923-02-07","start":"1921-01-05"},{"s
tart":"1923-02-08","end":"1927-01-11","name":"GOZZA"},{"end":"1928-04-28","s
tart":"1927-01-12","name":"GOZZA"}],"F065":[{"end":"9999-12-31","start":"186
1-03-17","name":"MAZZARINO"}],"G879":[{"name":"POPPI","end":"9999-12-31","st
art":"1861-03-17"}],"E921":[{"name":"MARCALLO CON CASONE","end":"1925-04-16"
,"start":"1870-08-02"},{"start":"1925-04-17","end":"2005-02-18","name":"MARC
ALLO CON CASONE"},{"start":"2005-02-19","end":"9999-12-31","name":"MARCALLO 
CON CASONE"}],"D188":[{"start":"1920-10-16","end":"1923-02-13","name":"CROVI
ANA"},{"start":"1923-02-14","end":"1928-09-05","name":"CROVIANA"},{"name":"C
ROVIANA","start":"1953-01-04","end":"9999-12-31"}],"F814":[{"end":"1928-11-3
0","start":"1861-03-17","name":"MURISENGO"},{"name":"MURISENGO","start":"192
8-12-01","end":"9999-12-31"}],"G512":[{"name":"PETRELLA TIFERNINA","start":"
1863-02-06","end":"1970-03-02"},{"end":"1979-11-30","start":"1970-03-03","na
me":"PETRELLA TIFERNINA"},{"end":"9999-12-31","start":"1979-12-01","name":"P
ETRELLA TIFERNINA"}],"L816":[{"end":"9999-12-31","start":"1863-03-30","name"
:"VEZZA D'OGLIO"}],"M423":[{"end":"9999-12-31","start":"2019-01-30","name":"
VALBRENTA"}],"D875":[{"end":"9999-12-31","start":"1872-10-18","name":"GALLIC
ANO NEL LAZIO"}],"E982":[{"name":"MARTIGNACCO","start":"1866-11-19","end":"1
968-04-05"},{"name":"MARTIGNACCO","end":"9999-12-31","start":"1968-04-06"}],
"G300":[{"end":"1968-04-05","start":"1866-11-19","name":"PALUZZA"},{"end":"9
999-12-31","start":"1968-04-06","name":"PALUZZA"}],"H225":[{"end":"9999-12-3
1","start":"1861-03-17","name":"REGGIOLO"}],"H647":[{"name":"SABAUDIA","star
t":"1933-09-13","end":"1934-10-25"},{"start":"1934-10-26","end":"1934-12-17"
,"name":"SABAUDIA"},{"end":"1945-04-08","start":"1934-12-18","name":"SABAUDI
A"},{"start":"1945-04-09","end":"9999-12-31","name":"SABAUDIA"}],"G052":[{"n
ame":"OLTRESARCA","end":"1923-02-13","start":"1920-10-16"},{"start":"1923-02
-14","end":"1929-02-13","name":"OLTRESARCA"}],"L068":[{"name":"TAVAZZANO","s
tart":"1861-03-17","end":"1869-04-01"}],"A355":[{"end":"1941-05-22","start":
"1928-05-12","name":"ARBATAX DI TORTOL\u00cc"},{"end":"2005-12-31","start":"
1941-05-23","name":"TORTOL\u00cc"},{"end":"2016-04-27","start":"2006-01-01",
"name":"TORTOL\u00cc"},{"name":"TORTOL\u00cc","start":"2016-04-28","end":"99
99-12-31"}],"I223":[{"name":"SANTA MADDALENA IN CASIES","end":"1923-02-13","
start":"1920-10-16"},{"end":"1927-01-11","start":"1923-02-14","name":"SANTA 
MADDALENA IN CASIES"},{"name":"SANTA MADDALENA IN CASIES","end":"1929-05-15"
,"start":"1927-01-12"}],"E090":[{"start":"1861-03-17","end":"9999-12-31","na
me":"GORDONA"}],"D971":[{"name":"GENZANO","end":"1901-07-15","start":"1861-0
3-17"},{"end":"1927-01-11","start":"1901-07-16","name":"GENZANO"},{"name":"G
ENZANO","start":"1927-01-12","end":"1935-06-25"},{"start":"1935-06-26","end"
:"1944-04-29","name":"GENZANO DI LUCANIA"},{"name":"GENZANO DI LUCANIA","sta
rt":"1944-04-30","end":"9999-12-31"}],"H112":[{"end":"1869-02-11","start":"1
861-03-17","name":"QUARTIANO"}],"F740":[{"name":"MOROLO","end":"1927-01-11",
"start":"1871-01-15"},{"start":"1927-01-12","end":"9999-12-31","name":"MOROL
O"}],"L891":[{"name":"VIGO D'ANAUNIA","start":"1920-10-16","end":"1923-02-13
"},{"start":"1923-02-14","end":"1928-12-21","name":"VIGO D'ANAUNIA"}],"F627"
:[{"end":"1910-05-19","start":"1861-03-17","name":"MONTE SAN PIETRO"},{"star
t":"1910-05-20","end":"9999-12-31","name":"MONTE SAN PIETRO"}],"G232":[{"sta
rt":"1861-03-17","end":"1923-03-18","name":"PAGANICO SABINO"},{"name":"PAGAN
ICO SABINO","start":"1923-03-19","end":"1927-01-11"},{"end":"9999-12-31","st
art":"1927-01-12","name":"PAGANICO SABINO"}],"B143":[{"name":"BRENDOLA","end
":"9999-12-31","start":"1866-11-19"}],"D384":[{"end":"1927-01-11","start":"1
861-03-17","name":"DUMENZA"},{"end":"1928-05-21","start":"1927-01-12","name"
:"DUMENZA"},{"name":"DUMENZA","end":"9999-12-31","start":"1928-05-22"}],"G64
5":[{"start":"1866-11-19","end":"1928-07-27","name":"PIEVE DI SOLIGO"},{"sta
rt":"1928-07-28","end":"1945-04-30","name":"PIEVE DI SOLIGO"},{"start":"1945
-05-01","end":"9999-12-31","name":"PIEVE DI SOLIGO"}],"M164":[{"name":"ZEPPA
RA","start":"1861-03-17","end":"1927-07-02"}],"G839":[{"start":"1861-03-17",
"end":"9999-12-31","name":"PONTECURONE"}],"F595":[{"name":"MONTEREALE","end"
:"9999-12-31","start":"1861-03-17"}],"D987":[{"name":"GERMIGNAGA","start":"1
861-03-17","end":"1927-01-11"},{"end":"1928-02-08","start":"1927-01-12","nam
e":"GERMIGNAGA"},{"name":"GERMIGNAGA","end":"9999-12-31","start":"1947-12-22
"}],"L329":[{"start":"1920-10-16","end":"1923-02-13","name":"TRANSACQUA"},{"
name":"TRANSACQUA","end":"1927-11-23","start":"1923-02-14"},{"name":"TRANSAC
QUA","end":"2015-12-31","start":"1947-01-24"}],"L023":[{"end":"1927-07-27","
start":"1861-03-17","name":"TADASUNI"},{"name":"TADASUNI","start":"1958-08-1
5","end":"1974-08-19"},{"end":"9999-12-31","start":"1974-08-20","name":"TADA
SUNI"}],"B952":[{"name":"CASCINE SAN GIACOMO","end":"1927-01-11","start":"18
61-03-17"},{"name":"CASCINE SAN GIACOMO","start":"1927-01-12","end":"1964-10
-01"},{"name":"SAN GIACOMO VERCELLESE","start":"1964-10-02","end":"9999-12-3
1"}],"H720":[{"start":"1861-03-17","end":"1956-07-25","name":"SALSOMAGGIORE"
},{"name":"SALSOMAGGIORE TERME","end":"9999-12-31","start":"1956-07-26"}],"F
632":[{"name":"POTENZA PICENA","end":"9999-12-31","start":"1863-01-05"}],"L9
95":[{"start":"1862-11-28","end":"1928-09-24","name":"VILLANUOVA SUL CLISI"}
,{"name":"VILLANUOVA SUL CLISI","start":"1928-09-25","end":"9999-12-31"}],"B
430":[{"name":"CALTAVUTURO","end":"9999-12-31","start":"1861-03-17"}],"G227"
:[{"name":"PADULI","end":"9999-12-31","start":"1861-03-17"}],"I782":[{"start
":"1861-03-17","end":"1928-04-07","name":"SOIANO DEL LAGO"},{"name":"SOIANO 
DEL LAGO","start":"1947-10-04","end":"9999-12-31"}],"C425":[{"end":"1923-02-
07","start":"1921-01-05","name":"CEGLIE"},{"name":"CEGLIE","start":"1923-02-
08","end":"1924-03-08"},{"end":"1924-10-30","start":"1924-03-09","name":"CEG
LIE"}],"C126":[{"name":"CASTELLAFIUME","end":"9999-12-31","start":"1861-03-1
7"}],"D544":[{"end":"9999-12-31","start":"1861-03-17","name":"FEROLETO ANTIC
O"}],"G400":[{"start":"1861-03-17","end":"2017-05-04","name":"PEDACE"}],"A16
1":[{"start":"1866-11-19","end":"9999-12-31","name":"ALBIGNASEGO"}],"L598":[
{"name":"VALLECORSA","end":"1927-04-13","start":"1871-01-15"},{"name":"VALLE
CORSA","end":"9999-12-31","start":"1927-04-14"}],"M323":[{"end":"9999-12-31"
,"start":"2014-01-01","name":"FISCAGLIA"}],"C611":[{"end":"1923-02-07","star
t":"1921-01-05","name":"CHIAPOVANO"},{"name":"CHIAPOVANO","end":"1927-01-11"
,"start":"1923-02-08"},{"name":"CHIAPOVANO","end":"1928-01-25","start":"1927
-01-12"},{"start":"1928-01-26","end":"1947-09-15","name":"CHIAPOVANO"}],"M02
9":[{"end":"1928-04-16","start":"1861-03-17","name":"VILLATALLA"}],"E560":[{
"name":"LEVANTO","start":"1861-03-17","end":"1923-12-21"},{"name":"LEVANTO",
"end":"9999-12-31","start":"1923-12-22"}],"D931":[{"name":"GARZIGLIANA","sta
rt":"1861-03-17","end":"9999-12-31"}],"I804":[{"name":"SOLMONA","start":"186
1-03-17","end":"1902-05-21"},{"start":"1902-05-22","end":"9999-12-31","name"
:"SULMONA"}],"I721":[{"start":"1861-03-17","end":"1928-05-04","name":"SIAPIC
CIA"},{"name":"SIAPICCIA","end":"9999-12-31","start":"1974-09-13"}],"L248":[
{"name":"TORREBELVICINO","end":"9999-12-31","start":"1866-11-19"}],"D294":[{
"start":"1861-03-17","end":"1924-02-03","name":"DIANO BORELLO"}],"B202":[{"e
nd":"1981-05-02","start":"1861-03-17","name":"BRONTE"},{"name":"BRONTE","sta
rt":"1981-05-03","end":"9999-12-31"}],"B809":[{"end":"1927-01-11","start":"1
861-03-17","name":"CAROVIGNO"},{"start":"1927-01-12","end":"9999-12-31","nam
e":"CAROVIGNO"}],"A455":[{"name":"ARZANO","start":"1861-03-17","end":"9999-1
2-31"}],"D835":[{"name":"GABIANO","start":"1861-03-17","end":"1928-07-17"},{
"name":"GABIANO","start":"1928-07-18","end":"9999-12-31"}],"D946":[{"name":"
GAVIRATE","end":"1927-01-11","start":"1861-03-17"},{"name":"GAVIRATE","end":
"1928-01-11","start":"1927-01-12"},{"end":"1929-05-18","start":"1928-01-12",
"name":"GAVIRATE"},{"end":"9999-12-31","start":"1929-05-19","name":"GAVIRATE
"}],"D649":[{"name":"FOIANO DELLA CHIANA","start":"1862-11-28","end":"9999-1
2-31"}],"F998":[{"name":"ODALENGO PICCOLO","start":"1861-03-17","end":"9999-
12-31"}],"F039":[{"name":"MASSE DI SAN MARTINO","start":"1861-03-17","end":"
1869-04-01"}],"C726":[{"name":"CIR\u00d2 MARINA","end":"1992-04-15","start":
"1952-04-24"},{"start":"1992-04-16","end":"9999-12-31","name":"CIR\u00d2 MAR
INA"}],"F333":[{"name":"MONASTIR","end":"1974-08-19","start":"1861-03-17"},{
"name":"MONASTIR","end":"2016-04-27","start":"1974-08-20"},{"name":"MONASTIR
","start":"2016-04-28","end":"9999-12-31"}],"I616":[{"name":"SENOSECCHIA","e
nd":"1923-02-07","start":"1921-01-05"},{"start":"1923-02-08","end":"1926-10-
28","name":"SENOSECCHIA"},{"end":"1947-09-15","start":"1926-10-29","name":"S
ENOSECCHIA"}],"A761":[{"name":"BELMONTE DEL SANNIO","start":"1863-02-06","en
d":"1928-05-19"},{"name":"BELMONTE DEL SANNIO","start":"1934-03-07","end":"1
970-03-02"},{"start":"1970-03-03","end":"9999-12-31","name":"BELMONTE DEL SA
NNIO"}],"I919":[{"name":"SPIRANO","start":"1861-03-17","end":"9999-12-31"}],
"C505":[{"name":"CERESOLE REALE","start":"1863-01-05","end":"1927-01-11"},{"
name":"CERESOLE REALE","end":"1929-03-01","start":"1927-01-12"},{"start":"19
29-03-02","end":"1935-06-04","name":"CERESOLE REALE"},{"name":"CERESOLE REAL
E","start":"1935-06-05","end":"1945-10-04"},{"name":"CERESOLE REALE","end":"
9999-12-31","start":"1945-10-05"}],"B516":[{"name":"CAMPO CALABRO","end":"99
99-12-31","start":"1950-02-11"},{"name":"CAMPO DI CALABRIA","end":"1927-08-0
3","start":"1864-06-23"}],"I121":[{"end":"9999-12-31","start":"1861-03-17","
name":"SAN PIO DELLE CAMERE"}],"B914":[{"name":"POZZAGLIO ED UNITI","end":"9
999-12-31","start":"1888-03-12"}],"A043":[{"name":"ACQUARO","start":"1861-03
-17","end":"1928-08-11"},{"end":"1929-04-10","start":"1928-08-12","name":"AC
QUARO"},{"name":"ACQUARO","start":"1929-04-11","end":"1992-04-15"},{"start":
"1992-04-16","end":"9999-12-31","name":"ACQUARO"}],"A349":[{"start":"1861-03
-17","end":"1928-01-21","name":"ARA"}],"F473":[{"end":"9999-12-31","start":"
1861-03-17","name":"MONTECHIARUGOLO"}],"B317":[{"name":"CACCAVONE","start":"
1861-03-17","end":"1922-03-06"},{"end":"1970-03-02","start":"1922-03-07","na
me":"POGGIO SANNITA"},{"name":"POGGIO SANNITA","start":"1970-03-03","end":"9
999-12-31"}],"Z309":[{"name":"Ciad","start":"0000-01-01","end":"9999-12-31"}
],"B056":[{"name":"BORORE","end":"1927-01-11","start":"1861-03-17"},{"start"
:"1927-01-12","end":"1928-03-12","name":"BORORE"},{"name":"BORORE","end":"19
39-02-16","start":"1928-03-13"},{"start":"1939-02-17","end":"9999-12-31","na
me":"BORORE"}],"A756":[{"name":"BELLOSGUARDO","end":"9999-12-31","start":"18
61-03-17"}],"C078":[{"name":"CASTEL D'AZZANO","start":"1866-11-19","end":"19
41-12-09"},{"name":"CASTEL D'AZZANO","end":"9999-12-31","start":"1941-12-10"
}],"A684":[{"name":"BARZANA","start":"1861-03-17","end":"9999-12-31"}],"B743
":[{"name":"CARBONE","end":"9999-12-31","start":"1861-03-17"}],"A228":[{"sta
rt":"1862-12-29","end":"9999-12-31","name":"ALTAVILLA IRPINA"}],"H867":[{"na
me":"SAN GIACOMO DEGLI SCHIAVONI","start":"1864-03-11","end":"1928-05-03"},{
"start":"1946-03-25","end":"1970-03-02","name":"SAN GIACOMO DEGLI SCHIAVONI"
},{"start":"1970-03-03","end":"9999-12-31","name":"SAN GIACOMO DEGLI SCHIAVO
NI"}],"D042":[{"name":"CORRIDONIA","start":"1931-10-24","end":"9999-12-31"},
{"name":"PAUSULA","start":"1861-03-17","end":"1931-10-23"}],"H120":[{"name":
"QUASSOLO","start":"1861-03-17","end":"1927-01-11"},{"name":"QUASSOLO","end"
:"1929-03-02","start":"1927-01-12"},{"name":"QUASSOLO","start":"1954-08-24",
"end":"9999-12-31"}],"G520":[{"start":"1861-03-17","end":"1928-12-11","name"
:"PETTENASCO"},{"end":"9999-12-31","start":"1928-12-12","name":"PETTENASCO"}
],"A589":[{"name":"BALDISSERO D'ALBA","start":"1862-12-19","end":"9999-12-31
"}],"I182":[{"name":"SANTADI","start":"1861-03-17","end":"1958-01-25"},{"end
":"1974-08-19","start":"1958-01-26","name":"SANTADI"},{"name":"SANTADI","sta
rt":"1974-08-20","end":"1979-05-25"},{"end":"2005-12-31","start":"1979-05-26
","name":"SANTADI"},{"name":"SANTADI","end":"2016-04-27","start":"2006-01-01
"},{"end":"9999-12-31","start":"2016-04-28","name":"SANTADI"}],"B061":[{"end
":"1920-03-21","start":"1866-11-19","name":"BORSO"},{"end":"9999-12-31","sta
rt":"1920-03-22","name":"BORSO DEL GRAPPA"}],"B276":[{"end":"1928-11-29","st
art":"1861-03-17","name":"BURGOS"},{"name":"BURGOS","end":"1946-11-01","star
t":"1928-11-30"},{"start":"1946-11-02","end":"9999-12-31","name":"BURGOS"}],
"H358":[{"end":"1926-09-16","start":"1861-03-17","name":"RIVOLTELLA"}],"C908
":[{"end":"9999-12-31","start":"1861-03-17","name":"COLTURANO"}],"H476":[{"e
nd":"1927-11-23","start":"1869-06-07","name":"RODENGO"}],"F482":[{"start":"1
861-03-17","end":"9999-12-31","name":"MONTECOSARO"}],"C898":[{"start":"1861-
03-17","end":"1928-07-28","name":"COLOMBARO"}],"A738":[{"name":"BELFORTE MON
FERRATO","start":"1863-01-26","end":"1928-01-28"},{"name":"BELFORTE MONFERRA
TO","start":"1947-06-23","end":"9999-12-31"}],"B680":[{"start":"1868-01-01",
"end":"9999-12-31","name":"CAPPELLA DE' PICENARDI"}],"F615":[{"name":"MONTES
","end":"1923-02-13","start":"1920-10-16"},{"end":"1928-09-05","start":"1923
-02-14","name":"MONTES"}],"G122":[{"name":"ORROLI","end":"1927-01-11","start
":"1861-03-17"},{"name":"ORROLI","start":"1927-01-12","end":"2005-12-31"},{"
name":"ORROLI","end":"2016-04-27","start":"2006-01-01"},{"start":"2016-04-28
","end":"9999-12-31","name":"ORROLI"}],"G008":[{"end":"1927-01-11","start":"
1864-04-01","name":"OGGIONA CON SANTO STEFANO"},{"end":"9999-12-31","start":
"1927-01-12","name":"OGGIONA CON SANTO STEFANO"}],"I580":[{"end":"1928-05-11
","start":"1861-03-17","name":"SELARGIUS"},{"end":"1974-08-19","start":"1947
-04-06","name":"SELARGIUS"},{"name":"SELARGIUS","end":"9999-12-31","start":"
1974-08-20"}],"H341":[{"end":"9999-12-31","start":"1915-04-16","name":"RIVAR
OLO DEL RE ED UNITI"}],"H537":[{"start":"1909-07-04","end":"2009-06-29","nam
e":"RONCO BRIANTINO"},{"name":"RONCO BRIANTINO","end":"9999-12-31","start":"
2009-06-30"}],"D674":[{"name":"FONTANELLE","end":"9999-12-31","start":"1866-
11-19"}],"H888":[{"name":"SAN GIORGIO LUCANO","start":"1863-04-10","end":"19
27-01-11"},{"name":"SAN GIORGIO LUCANO","start":"1927-01-12","end":"9999-12-
31"}],"G215":[{"name":"PADERNA","end":"1928-06-18","start":"1861-03-17"},{"n
ame":"PADERNA","start":"1946-12-17","end":"9999-12-31"}],"G978":[{"start":"1
937-06-16","end":"9999-12-31","name":"PRALI"},{"end":"1937-06-15","start":"1
895-03-04","name":"PRALY"}],"G137":[{"start":"1861-03-17","end":"2009-07-11"
,"name":"ORTEZZANO"},{"name":"ORTEZZANO","start":"2009-07-12","end":"9999-12
-31"}],"L360":[{"name":"TREDOSSI","end":"1928-06-26","start":"1866-01-22"}],
"H769":[{"end":"1935-09-05","start":"1862-11-24","name":"SAN BENEDETTO DEL T
RONTO"},{"name":"SAN BENEDETTO DEL TRONTO","start":"1935-09-06","end":"9999-
12-31"}],"D579":[{"name":"FIGINO SERENZA","end":"9999-12-31","start":"1862-1
2-29"}],"H336":[{"end":"2009-07-29","start":"1861-03-17","name":"RIVANAZZANO
"},{"start":"2009-07-30","end":"9999-12-31","name":"RIVANAZZANO TERME"}],"E3
98":[{"start":"1920-10-16","end":"1923-02-13","name":"LACES"},{"name":"LACES
","end":"1927-01-11","start":"1923-02-14"},{"name":"LACES","end":"1928-12-08
","start":"1927-01-12"},{"name":"LACES","start":"1928-12-09","end":"9999-12-
31"}],"A813":[{"name":"BERZANO DI TORTONA","end":"1929-01-23","start":"1863-
02-23"},{"start":"1947-09-04","end":"9999-12-31","name":"BERZANO DI TORTONA"
}],"H522":[{"name":"RONC\u00c0","start":"1880-01-16","end":"9999-12-31"}],"C
417":[{"name":"CEDEGOLO","start":"1927-04-11","end":"1927-12-26"},{"name":"C
EDEGOLO","start":"1927-12-27","end":"1948-12-20"},{"name":"CEDEGOLO","start"
:"1948-12-21","end":"9999-12-31"},{"name":"GREVO","start":"1861-03-17","end"
:"1927-04-10"}],"I173":[{"start":"1923-04-27","end":"1927-01-11","name":"SAN
TA CRISTINA"},{"name":"SANTA CRISTINA","end":"1956-01-03","start":"1927-01-1
2"},{"name":"SANTA CRISTINA IN GARDENA","start":"1920-10-16","end":"1923-02-
13"},{"end":"1923-04-26","start":"1923-02-14","name":"SANTA CRISTINA IN GARD
ENA"},{"end":"9999-12-31","start":"1956-01-04","name":"SANTA CRISTINA VALGAR
DENA"}],"G575":[{"name":"PIASCO","end":"9999-12-31","start":"1861-03-17"}],"
C236":[{"start":"1861-03-17","end":"1923-11-23","name":"CASTELNUOVO DI GARFA
GNANA"},{"name":"CASTELNUOVO DI GARFAGNANA","end":"9999-12-31","start":"1923
-11-24"}],"D219":[{"start":"1861-03-17","end":"1927-01-11","name":"CURINO"},
{"end":"1992-04-15","start":"1927-01-12","name":"CURINO"},{"start":"1992-04-
16","end":"9999-12-31","name":"CURINO"}],"C591":[{"start":"1861-03-17","end"
:"1927-11-02","name":"CEVO"},{"name":"CEVO","start":"1954-08-05","end":"9999
-12-31"}],"Z119":[{"start":"0000-01-01","end":"9999-12-31","name":"Liechtens
tein"}],"B107":[{"name":"BOVOLONE","start":"1866-11-19","end":"1996-08-31"},
{"name":"BOVOLONE","end":"2000-07-15","start":"1996-09-01"},{"end":"9999-12-
31","start":"2000-07-16","name":"BOVOLONE"}],"H581":[{"name":"ROSSIGLIONE","
start":"1861-03-17","end":"9999-12-31"}],"E624":[{"name":"LIVO","end":"1923-
02-13","start":"1920-10-16"},{"name":"LIVO","start":"1923-02-14","end":"1928
-10-29"},{"end":"1947-03-14","start":"1928-10-30","name":"LIVO"},{"end":"999
9-12-31","start":"1947-03-15","name":"LIVO"}],"L618":[{"end":"1923-02-13","s
tart":"1920-10-16","name":"VALLE SAN FELICE"},{"start":"1923-02-14","end":"1
924-01-13","name":"VALLE SAN FELICE"}],"I340":[{"name":"SANT'ILARIO LIGURE",
"end":"1926-02-05","start":"1863-04-23"}],"D812":[{"start":"1861-03-17","end
":"1928-05-12","name":"FROSSASCO"},{"name":"FROSSASCO","start":"1928-05-13",
"end":"1954-07-14"},{"start":"1954-07-15","end":"9999-12-31","name":"FROSSAS
CO"}],"E288":[{"start":"1920-10-16","end":"1923-02-13","name":"IMER"},{"name
":"IMER","end":"1928-01-27","start":"1923-02-14"},{"end":"9999-12-31","start
":"1947-03-30","name":"IMER"}],"M060":[{"name":"VINOVO","end":"9999-12-31","
start":"1868-09-09"}],"G181":[{"start":"1909-10-01","end":"1925-04-16","name
":"OSSONA"},{"name":"OSSONA","end":"2005-02-18","start":"1925-04-17"},{"star
t":"2005-02-19","end":"9999-12-31","name":"OSSONA"}],"C241":[{"end":"1929-05
-02","start":"1928-12-19","name":"CASTELNUOVO NIGRA"},{"start":"1929-05-03",
"end":"1945-10-04","name":"CASTELNUOVO NIGRA"},{"name":"CASTELNUOVO NIGRA","
start":"1945-10-05","end":"1947-01-23"},{"name":"CASTELNUOVO NIGRA","end":"9
999-12-31","start":"1947-01-24"}],"E529":[{"end":"2019-02-14","start":"1970-
08-29","name":"OSMATE"},{"name":"OSMATE LENTATE","start":"1892-06-24","end":
"1927-01-11"},{"start":"1927-01-12","end":"1929-06-12","name":"OSMATE LENTAT
E"},{"end":"1970-08-28","start":"1929-06-13","name":"OSMATE LENTATE"}],"A910
":[{"start":"1861-03-17","end":"9999-12-31","name":"BOBBIO PELLICE"}],"H175"
:[{"start":"1861-03-17","end":"9999-12-31","name":"RANDAZZO"}],"B038":[{"nam
e":"BORGO SAN SIRO","start":"1861-03-17","end":"1927-05-24"},{"end":"9999-12
-31","start":"1927-05-25","name":"BORGO SAN SIRO"}],"E762":[{"name":"LUSIANA
","start":"1866-11-19","end":"2019-02-19"}],"F808":[{"end":"1974-08-19","sta
rt":"1861-03-17","name":"MURAVERA"},{"name":"MURAVERA","start":"1974-08-20",
"end":"1986-04-02"},{"start":"1986-04-03","end":"2016-04-27","name":"MURAVER
A"},{"name":"MURAVERA","start":"2016-04-28","end":"9999-12-31"}],"I773":[{"s
tart":"1921-01-05","end":"1923-02-07","name":"SMERIA"},{"name":"SMERIA","end
":"1924-03-08","start":"1923-02-08"},{"end":"1924-10-30","start":"1924-03-09
","name":"SMERIA"}],"B696":[{"end":"1927-06-01","start":"1861-03-17","name":
"CAPRI"},{"name":"CAPRI","start":"1927-06-02","end":"1946-04-01"},{"name":"C
APRI","end":"9999-12-31","start":"1946-04-02"}],"Z719":[{"name":"Nuova Zelan
da","end":"9999-12-31","start":"0000-01-01"}],"L251":[{"start":"1861-03-17",
"end":"9999-12-31","name":"TORRE BOLDONE"}],"B999":[{"start":"1863-02-23","e
nd":"1927-01-11","name":"CASSANO VALCUVIA"},{"end":"9999-12-31","start":"192
7-01-12","name":"CASSANO VALCUVIA"}],"M371":[{"end":"9999-12-31","start":"20
16-01-01","name":"LESSONA"}],"B707":[{"name":"CAPRIGLIO","start":"1861-03-17
","end":"1928-10-17"},{"name":"CAPRIGLIO","end":"9999-12-31","start":"1946-1
1-12"}],"I596":[{"start":"1863-04-23","end":"1927-01-11","name":"SELVE MARCO
NE"},{"end":"1992-04-15","start":"1927-01-12","name":"SELVE MARCONE"},{"name
":"SELVE MARCONE","end":"2016-12-31","start":"1992-04-16"}],"C470":[{"end":"
9999-12-31","start":"1861-03-17","name":"CENTOLA"}],"H402":[{"end":"9999-12-
31","start":"1861-03-17","name":"ROCCA DI MEZZO"}],"E830":[{"end":"1927-12-2
6","start":"1861-03-17","name":"MAGREGLIO"},{"name":"MAGREGLIO","end":"9999-
12-31","start":"1950-04-05"}],"G781":[{"end":"1867-09-05","start":"1861-03-1
7","name":"POLENGO"}],"I994":[{"name":"SUEGLIO","start":"1861-03-17","end":"
1992-04-15"},{"start":"1992-04-16","end":"9999-12-31","name":"SUEGLIO"}],"C3
30":[{"name":"CASTO","start":"1861-03-17","end":"1928-04-02"},{"start":"1928
-04-03","end":"9999-12-31","name":"CASTO"}],"H024":[{"name":"PREGHENA","star
t":"1920-10-16","end":"1923-02-13"},{"name":"PREGHENA","end":"1928-10-29","s
tart":"1923-02-14"}],"D565":[{"name":"FIAV\u00c8","start":"1920-10-16","end"
:"1923-02-13"},{"end":"1928-04-02","start":"1923-02-14","name":"FIAV\u00c8"}
,{"name":"FIAV\u00c8","start":"1952-05-17","end":"9999-12-31"}],"H775":[{"na
me":"SAN BENIGNO CANAVESE","end":"9999-12-31","start":"1861-03-17"}],"B325":
[{"end":"1927-01-11","start":"1908-07-22","name":"CADEGLIANO"},{"name":"CADE
GLIANO","end":"1928-08-15","start":"1927-01-12"}],"E162":[{"name":"GRECO MIL
ANESE","end":"1904-06-23","start":"1863-02-23"},{"end":"1923-10-13","start":
"1904-06-24","name":"GRECO MILANESE"}],"M431":[{"name":"VILLE DI FIEMME","st
art":"2020-01-01","end":"9999-12-31"}],"I246":[{"end":"1927-05-11","start":"
1866-11-19","name":"SANTA MARIA IN STELLE"}],"E409":[{"name":"LAGONEGRO","st
art":"1861-03-17","end":"9999-12-31"}],"F264":[{"name":"MOERNA","start":"192
0-10-16","end":"1923-02-13"},{"name":"MOERNA","end":"1929-02-28","start":"19
23-02-14"}],"L071":[{"name":"TAVERNERIO","start":"1861-03-17","end":"1928-09
-22"},{"end":"9999-12-31","start":"1928-09-23","name":"TAVERNERIO"}],"E933":
[{"start":"1861-03-17","end":"9999-12-31","name":"MARCIANO DELLA CHIANA"}],"
A977":[{"name":"BONO","start":"1861-03-17","end":"1928-11-29"},{"name":"BONO
","end":"1933-07-31","start":"1928-11-30"},{"end":"9999-12-31","start":"1933
-08-01","name":"BONO"}],"A138":[{"name":"ALBARETO","end":"9999-12-31","start
":"1946-02-27"},{"end":"1928-06-28","start":"1862-10-20","name":"ALBARETO DI
 BORGOTARO"}],"F501":[{"name":"MONTEFIORE DELL'ASO","end":"9999-12-31","star
t":"1862-11-24"}],"G722":[{"name":"PIZZO","end":"1928-06-26","start":"1861-0
3-17"},{"start":"1928-06-27","end":"1992-04-15","name":"PIZZO"},{"start":"19
92-04-16","end":"9999-12-31","name":"PIZZO"}],"I287":[{"name":"SANT'ANGELO I
N VADO","start":"1861-03-17","end":"1929-06-14"},{"name":"SANT'ANGELO IN VAD
O","end":"1998-09-03","start":"1929-06-15"},{"name":"SANT'ANGELO IN VADO","e
nd":"9999-12-31","start":"1998-09-04"}],"A374":[{"start":"1866-11-19","end":
"1981-04-20","name":"ARCOLE"},{"start":"1981-04-21","end":"2007-07-31","name
":"ARCOLE"},{"name":"ARCOLE","start":"2007-08-01","end":"9999-12-31"}],"H230
":[{"name":"REMEDELLO","start":"1928-01-22","end":"2002-10-25"},{"name":"REM
EDELLO","end":"9999-12-31","start":"2002-10-26"}],"D968":[{"name":"GENONI","
end":"1927-01-11","start":"1861-03-17"},{"end":"2005-12-31","start":"1927-01
-12","name":"GENONI"},{"name":"GENONI","start":"2006-01-01","end":"2016-04-2
7"},{"start":"2016-04-28","end":"9999-12-31","name":"GENONI"}],"L805":[{"nam
e":"VESCOVANA","end":"1913-06-13","start":"1866-11-19"},{"start":"1913-06-14
","end":"9999-12-31","name":"VESCOVANA"}],"G569":[{"end":"1923-02-13","start
":"1920-10-16","name":"PIANO"},{"name":"PIANO","start":"1923-02-14","end":"1
928-10-06"}],"A434":[{"start":"1868-02-14","end":"9999-12-31","name":"ARQU\u
00c0 PETRARCA"}],"G493":[{"end":"9999-12-31","start":"1861-03-17","name":"PE
SCOCOSTANZO"}],"G664":[{"end":"1923-12-21","start":"1861-03-17","name":"PIGN
ONE"},{"name":"PIGNONE","end":"9999-12-31","start":"1923-12-22"}],"M145":[{"
name":"ZAN\u00c8","start":"1866-11-19","end":"9999-12-31"}],"G737":[{"start"
:"1861-03-17","end":"1928-07-17","name":"PLESIO"},{"start":"1928-07-18","end
":"9999-12-31","name":"PLESIO"}],"D854":[{"name":"GAIARINE","start":"1869-08
-11","end":"9999-12-31"}],"H169":[{"name":"RAMO DI PALO","end":"1927-07-26",
"start":"1866-11-19"}],"A007":[{"name":"ABBASANTA","end":"1927-07-27","start
":"1861-03-17"},{"end":"1946-08-23","start":"1934-11-02","name":"ABBASANTA"}
,{"end":"1950-06-15","start":"1946-08-24","name":"ABBASANTA"},{"name":"ABBAS
ANTA","end":"1974-08-19","start":"1950-06-16"},{"name":"ABBASANTA","start":"
1974-08-20","end":"9999-12-31"}],"C370":[{"end":"9999-12-31","start":"1867-0
8-26","name":"CAVAION VERONESE"}],"H302":[{"name":"RIOLO","start":"1861-03-1
7","end":"1915-02-04"},{"name":"RIOLO DEI BAGNI","end":"1957-04-23","start":
"1915-02-05"},{"end":"1985-01-13","start":"1957-04-24","name":"RIOLO TERME"}
,{"name":"RIOLO TERME","end":"9999-12-31","start":"1985-01-14"}],"M346":[{"s
tart":"2015-01-01","end":"9999-12-31","name":"VALVASONE ARZENE"}],"C952":[{"
end":"1935-03-28","start":"1861-03-17","name":"CONCOREZZO"},{"end":"2009-06-
29","start":"1935-03-29","name":"CONCOREZZO"},{"start":"2009-06-30","end":"9
999-12-31","name":"CONCOREZZO"}],"M297":[{"name":"FIUMICINO","start":"1992-0
4-04","end":"9999-12-31"}],"A885":[{"name":"BISENTI","end":"9999-12-31","sta
rt":"1861-03-17"}],"G283":[{"name":"PALMA CAMPANIA","start":"1863-08-10","en
d":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":"PALMA CAMP
ANIA"}],"D095":[{"start":"1861-03-17","end":"1903-06-14","name":"COSSERIA"},
{"name":"COSSERIA","end":"1927-01-11","start":"1903-06-15"},{"name":"COSSERI
A","start":"1927-01-12","end":"1928-12-12"},{"name":"COSSERIA","start":"1928
-12-13","end":"9999-12-31"}],"C143":[{"name":"CASTELLARO","start":"1861-03-1
7","end":"1923-12-21"},{"end":"9999-12-31","start":"1925-08-22","name":"CAST
ELLARO"}],"H510":[{"start":"1862-12-29","end":"1929-05-15","name":"ROMAN\u00
d2 BRIANZA"}],"H735":[{"name":"SALZANO","end":"9999-12-31","start":"1866-11-
19"}],"I818":[{"name":"SOMENDENNA","end":"1928-03-26","start":"1861-03-17"}]
,"I497":[{"name":"SCANDOLARA RAVARA","end":"1934-11-19","start":"1861-03-17"
},{"name":"SCANDOLARA RAVARA","start":"1934-11-20","end":"9999-12-31"}],"M33
1":[{"name":"VALLEFOGLIA","start":"2014-01-01","end":"9999-12-31"}],"L087":[
{"name":"TELGATE","start":"1861-03-17","end":"9999-12-31"}],"G110":[{"name":
"ORIOLO","end":"9999-12-31","start":"1861-03-17"}],"D923":[{"start":"1920-10
-16","end":"1923-02-13","name":"GARGAZZONE"},{"start":"1923-02-14","end":"19
27-01-11","name":"GARGAZZONE"},{"start":"1927-01-12","end":"1993-04-13","nam
e":"GARGAZZONE"},{"name":"GARGAZZONE","start":"1993-04-14","end":"9999-12-31
"}],"I733":[{"name":"SILUN MONT'AQUILA","end":"1929-05-06","start":"1923-04-
27"},{"name":"SLUM","end":"1923-02-07","start":"1922-02-02"},{"name":"SLUM",
"end":"1923-04-26","start":"1923-02-08"}],"A981":[{"start":"1861-03-17","end
":"1927-01-11","name":"BORBONA"},{"name":"BORBONA","end":"9999-12-31","start
":"1927-01-12"}],"B425":[{"name":"CALOSSO","end":"1935-04-07","start":"1861-
03-17"},{"end":"9999-12-31","start":"1935-04-08","name":"CALOSSO"}],"C430":[
{"start":"1861-03-17","end":"9999-12-31","name":"CELICO"}],"I271":[{"end":"1
974-08-19","start":"1861-03-17","name":"SANT'ANDREA FRIUS"},{"name":"SANT'AN
DREA FRIUS","end":"2016-04-27","start":"1974-08-20"},{"name":"SANT'ANDREA FR
IUS","end":"9999-12-31","start":"2016-04-28"}],"E870":[{"end":"9999-12-31","
start":"1861-03-17","name":"MALVICINO"}],"I066":[{"name":"SAN PANCRAZIO SALE
NTINO","end":"1927-01-11","start":"1862-11-28"},{"name":"SAN PANCRAZIO SALEN
TINO","start":"1927-01-12","end":"9999-12-31"}],"H050":[{"name":"PRESTINE","
end":"1927-12-03","start":"1861-03-17"},{"name":"PRESTINE","end":"2016-04-23
","start":"1947-06-18"}],"F683":[{"name":"MONTONA","end":"1923-02-07","start
":"1921-01-05"},{"name":"MONTONA","start":"1923-02-08","end":"1947-09-15"}],
"L753":[{"end":"9999-12-31","start":"1861-03-17","name":"VERDELLO"}],"E794":
[{"end":"1928-01-17","start":"1861-03-17","name":"MADONE"},{"start":"1948-03
-30","end":"9999-12-31","name":"MADONE"}],"G777":[{"end":"1928-12-29","start
":"1861-03-17","name":"POIRINO"},{"end":"1946-12-23","start":"1928-12-30","n
ame":"POIRINO"},{"name":"POIRINO","start":"1946-12-24","end":"9999-12-31"}],
"A922":[{"start":"1861-03-17","end":"1928-02-23","name":"BOGLIASCO"},{"end":
"1956-11-07","start":"1946-12-19","name":"BOGLIASCO"},{"name":"BOGLIASCO","e
nd":"9999-12-31","start":"1956-11-08"}],"A474":[{"name":"ASSEMINI","end":"19
74-08-19","start":"1861-03-17"},{"name":"ASSEMINI","end":"9999-12-31","start
":"1974-08-20"}],"L046":[{"name":"TARANO","start":"1871-03-01","end":"1923-0
3-18"},{"end":"1927-01-11","start":"1923-03-19","name":"TARANO"},{"name":"TA
RANO","start":"1927-01-12","end":"9999-12-31"}],"F044":[{"name":"MASSIGNANO"
,"start":"1861-03-17","end":"9999-12-31"}],"B611":[{"name":"CANNETO DI BARI"
,"end":"1927-11-07","start":"1863-01-19"}],"I511":[{"name":"SCARMAGNO","star
t":"1861-03-17","end":"1927-01-11"},{"name":"SCARMAGNO","end":"1928-06-15","
start":"1927-01-12"},{"start":"1946-12-24","end":"9999-12-31","name":"SCARMA
GNO"}],"E003":[{"name":"GHIFFA","start":"1862-12-29","end":"1927-12-23"},{"n
ame":"GHIFFA","end":"1928-10-15","start":"1927-12-24"},{"start":"1928-10-16"
,"end":"1992-05-22","name":"GHIFFA"},{"end":"9999-12-31","start":"1992-05-23
","name":"GHIFFA"}],"B126":[{"name":"BREBBIA","end":"1927-01-11","start":"18
61-03-17"},{"start":"1927-01-12","end":"9999-12-31","name":"BREBBIA"}],"E309
":[{"start":"1861-03-17","end":"1929-05-15","name":"INVERIGO"},{"end":"9999-
12-31","start":"1929-05-16","name":"INVERIGO"}],"A937":[{"end":"9999-12-31",
"start":"1861-03-17","name":"BOLGARE"}],"E973":[{"end":"9999-12-31","start":
"1861-03-17","name":"MARSAGLIA"}],"A178":[{"name":"ALDENO","end":"1923-02-13
","start":"1920-10-16"},{"start":"1923-02-14","end":"1928-05-22","name":"ALD
ENO"},{"start":"1928-05-23","end":"1947-06-22","name":"ALDENO"},{"start":"19
47-06-23","end":"9999-12-31","name":"ALDENO"}],"C809":[{"name":"COCQUIO","st
art":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"1927-05-2
6","name":"COCQUIO"}],"M387":[{"end":"9999-12-31","start":"2018-01-01","name
":"MONTALTO CARPASIO"}],"D167":[{"name":"CREVENNA","start":"1861-03-17","end
":"1928-01-25"}],"H270":[{"end":"1927-01-11","start":"1861-03-17","name":"RI
BORDONE"},{"end":"1945-10-04","start":"1927-01-12","name":"RIBORDONE"},{"sta
rt":"1945-10-05","end":"9999-12-31","name":"RIBORDONE"}],"C202":[{"name":"CA
STELLUCCIO VALMAGGIORE","end":"9999-12-31","start":"1861-03-17"}],"I051":[{"
name":"SAN NAZZARO VAL CAVARGNA","end":"1928-11-26","start":"1863-07-13"},{"
name":"SAN NAZZARO VAL CAVARGNA","start":"1950-04-05","end":"9999-12-31"}],"
G099":[{"start":"1861-03-17","end":"1927-12-27","name":"ORIANO"}],"A334":[{"
name":"APPIGNANO","start":"1861-03-17","end":"9999-12-31"}],"G393":[{"name":
"PAVULLO NEL FRIGNANO","start":"1863-04-13","end":"9999-12-31"}],"D820":[{"n
ame":"FUNDOLES","end":"1923-02-13","start":"1920-10-16"},{"end":"1923-04-26"
,"start":"1923-02-14","name":"FUNDOLES"},{"end":"1927-01-11","start":"1923-0
4-27","name":"FUNDRES"},{"name":"FUNDRES","end":"1929-03-24","start":"1927-0
1-12"}],"G177":[{"name":"OSSERO","end":"1923-02-07","start":"1921-01-05"},{"
end":"1924-02-06","start":"1923-02-08","name":"OSSERO"},{"name":"OSSERO","st
art":"1924-02-07","end":"1947-09-15"}],"D539":[{"end":"1927-01-11","start":"
1871-01-15","name":"FERENTINO"},{"name":"FERENTINO","end":"9999-12-31","star
t":"1927-01-12"}],"H441":[{"name":"ROCCA SANTO STEFANO","end":"1928-10-29","
start":"1871-01-15"},{"name":"ROCCA SANTO STEFANO","start":"1946-04-26","end
":"9999-12-31"}],"C516":[{"start":"1861-03-17","end":"1928-12-01","name":"CE
RMENATE"},{"name":"CERMENATE","end":"9999-12-31","start":"1928-12-02"}],"B50
5":[{"name":"CAMPERTOGNO","start":"1861-03-17","end":"1927-01-11"},{"name":"
CAMPERTOGNO","end":"9999-12-31","start":"1927-01-12"}],"E817":[{"name":"MAGL
IONE","start":"1861-03-17","end":"1927-01-11"},{"end":"1929-03-02","start":"
1927-01-12","name":"MAGLIONE"},{"start":"1947-09-30","end":"9999-12-31","nam
e":"MAGLIONE"}],"I605":[{"name":"SENEGHE","end":"1974-08-19","start":"1861-0
3-17"},{"name":"SENEGHE","end":"9999-12-31","start":"1974-08-20"}],"I962":[{
"end":"9999-12-31","start":"1905-07-10","name":"STORNARA"}],"B258":[{"start"
:"1861-03-17","end":"1927-01-11","name":"BUGUGGIATE"},{"start":"1927-01-12",
"end":"1928-01-12","name":"BUGUGGIATE"},{"start":"1956-03-04","end":"9999-12
-31","name":"BUGUGGIATE"}],"B726":[{"name":"CARASCO","end":"9999-12-31","sta
rt":"1861-03-17"}],"H376":[{"name":"ROBELLA","end":"1935-04-07","start":"186
1-03-17"},{"name":"ROBELLA","start":"1935-04-08","end":"9999-12-31"}],"F382"
:[{"end":"9999-12-31","start":"1866-11-19","name":"MONSELICE"}],"C056":[{"na
me":"CASTEGNERO","start":"1866-11-19","end":"9999-12-31"}],"A778":[{"end":"1
927-10-12","start":"1862-12-29","name":"BENE LARIO"},{"name":"BENE LARIO","s
tart":"1950-04-05","end":"9999-12-31"}],"C317":[{"start":"1861-03-17","end":
"9999-12-31","name":"CASTIGLIONE TINELLA"}],"H603":[{"start":"1861-03-17","e
nd":"1929-05-02","name":"ROVENNA"}],"E498":[{"start":"1861-03-17","end":"193
8-10-16","name":"LAVIANO"},{"name":"LAVIANO","start":"1938-10-17","end":"999
9-12-31"}],"H436":[{"name":"ROCCAROMANA","start":"1861-03-17","end":"1927-01
-11"},{"end":"1945-08-31","start":"1927-01-12","name":"ROCCAROMANA"},{"end":
"9999-12-31","start":"1945-09-01","name":"ROCCAROMANA"}],"H577":[{"start":"1
861-03-17","end":"1927-01-11","name":"ROSSA"},{"name":"ROSSA","end":"9999-12
-31","start":"1927-01-12"}],"D767":[{"name":"FRANCICA","start":"1861-03-17",
"end":"1992-04-15"},{"start":"1992-04-16","end":"9999-12-31","name":"FRANCIC
A"}],"L460":[{"start":"1861-03-17","end":"1927-01-11","name":"TUFINO"},{"nam
e":"TUFINO","end":"1928-09-08","start":"1927-01-12"},{"end":"9999-12-31","st
art":"1946-07-03","name":"TUFINO"}],"D634":[{"start":"1861-03-17","end":"192
7-12-26","name":"FLERO"},{"end":"9999-12-31","start":"1956-10-07","name":"FL
ERO"}],"C914":[{"start":"1919-08-22","end":"9999-12-31","name":"COMANO"}],"I
440":[{"name":"SAROLA","end":"1924-02-03","start":"1861-03-17"}],"C743":[{"n
ame":"CITTADELLA","end":"9999-12-31","start":"1866-11-19"}],"H217":[{"end":"
1927-03-11","start":"1861-03-17","name":"REDONA"}],"H135":[{"name":"QUINTO A
L MARE","start":"1861-03-17","end":"1926-02-05"}],"F542":[{"end":"9999-12-31
","start":"1862-09-22","name":"MONTELEONE ROCCA DORIA"}],"G844":[{"start":"1
861-03-17","end":"9999-12-31","name":"PONTE DI LEGNO"}],"G692":[{"name":"PIO
V\u00c0","end":"1928-08-06","start":"1861-03-17"},{"start":"1928-08-07","end
":"1935-04-07","name":"PIOV\u00c0"},{"name":"PIOV\u00c0","start":"1935-04-08
","end":"1940-02-10"},{"name":"PIOV\u00c0 MASSAIA","start":"1940-02-11","end
":"1947-10-10"},{"name":"PIOV\u00c0 MASSAIA","start":"1947-10-11","end":"199
4-07-27"},{"name":"PIOV\u00c0 MASSAIA","end":"9999-12-31","start":"1994-07-2
8"}],"B078":[{"end":"1923-02-13","start":"1920-10-16","name":"BOSENTINO"},{"
end":"1928-09-08","start":"1923-02-14","name":"BOSENTINO"},{"name":"BOSENTIN
O","start":"1947-03-30","end":"2015-12-31"}],"G710":[{"name":"PISOGNE","end"
:"9999-12-31","start":"1861-03-17"}],"B908":[{"end":"1868-06-13","start":"18
61-03-17","name":"CASALORZO GEROLDI"}],"A026":[{"start":"1861-03-17","end":"
9999-12-31","name":"ACI CASTELLO"}],"F899":[{"end":"9999-12-31","start":"186
1-03-17","name":"NISCEMI"}],"C276":[{"name":"CASTELVECCHIO DI ROCCA BARBENA"
,"start":"1863-04-23","end":"1927-01-11"},{"name":"CASTELVECCHIO DI ROCCA BA
RBENA","end":"9999-12-31","start":"1927-01-12"}],"L942":[{"name":"VILLAFALLE
TTO","start":"1861-03-17","end":"9999-12-31"}],"I133":[{"end":"9999-12-31","
start":"1868-03-25","name":"SAN PROSPERO"}],"G535":[{"name":"PIACENZA","star
t":"1861-03-17","end":"1923-07-22"},{"name":"PIACENZA","end":"9999-12-31","s
tart":"1923-07-23"}],"C061":[{"name":"CASTELBELLO","end":"1923-02-13","start
":"1920-10-16"},{"name":"CASTELBELLO","start":"1923-02-14","end":"1927-01-11
"},{"name":"CASTELBELLO","start":"1927-01-12","end":"1929-02-14"}],"F292":[{
"name":"MOLINO ALZANO","start":"1928-06-19","end":"1946-07-18"}],"L153":[{"e
nd":"1927-01-11","start":"1861-03-17","name":"TETI"},{"name":"TETI","end":"9
999-12-31","start":"1927-01-12"}],"E011":[{"name":"GIANO VETUSTO","start":"1
863-02-06","end":"1927-01-11"},{"start":"1927-01-12","end":"1929-03-24","nam
e":"GIANO VETUSTO"},{"start":"1947-10-10","end":"9999-12-31","name":"GIANO V
ETUSTO"}],"B577":[{"end":"1923-02-13","start":"1920-10-16","name":"CANAL SAN
 BOVO"},{"start":"1923-02-14","end":"9999-12-31","name":"CANAL SAN BOVO"}],"
I503":[{"start":"1862-09-29","end":"1927-01-11","name":"SCANO DI MONTIFERRO"
},{"start":"1927-01-12","end":"1974-08-19","name":"SCANO DI MONTIFERRO"},{"n
ame":"SCANO DI MONTIFERRO","start":"1974-08-20","end":"9999-12-31"}],"G105":
[{"name":"ORINO","start":"1861-03-17","end":"1927-01-11"},{"name":"ORINO","e
nd":"1927-08-18","start":"1927-01-12"},{"start":"1956-01-24","end":"9999-12-
31","name":"ORINO"}],"B603":[{"name":"CANICATTINI BAGNI","start":"1861-03-17
","end":"9999-12-31"}],"D462":[{"name":"FAGGETO LARIO","end":"9999-12-31","s
tart":"1928-05-12"}],"A328":[{"name":"APICE","start":"1861-03-17","end":"999
9-12-31"}],"B436":[{"name":"CALVAGESE","end":"1925-10-24","start":"1861-03-1
7"},{"name":"CALVAGESE DELLA RIVIERA","start":"1925-10-25","end":"1928-02-15
"},{"start":"1928-02-16","end":"9999-12-31","name":"CALVAGESE DELLA RIVIERA"
}],"I677":[{"name":"SESSA CILENTO","start":"1861-03-17","end":"1929-03-14"},
{"end":"9999-12-31","start":"1929-03-15","name":"SESSA CILENTO"}],"D843":[{"
start":"1897-04-01","end":"1927-01-11","name":"GAETA"},{"start":"1927-01-12"
,"end":"1927-03-15","name":"GAETA"},{"name":"GAETA","end":"1934-10-25","star
t":"1927-03-16"},{"name":"GAETA","start":"1934-10-26","end":"1934-12-17"},{"
name":"GAETA","end":"1945-04-08","start":"1934-12-18"},{"end":"9999-12-31","
start":"1945-04-09","name":"GAETA"}],"H258":[{"name":"REZZOAGLIO","end":"999
9-12-31","start":"1918-05-13"}],"F459":[{"name":"MONTECATINI DI VAL DI NIEVO
LE","start":"1861-03-17","end":"1905-07-29"},{"start":"1905-07-30","end":"19
28-01-25","name":"MONTECATINI DI VAL DI NIEVOLE"},{"end":"1934-11-02","start
":"1928-01-26","name":"MONTECATINI DI VAL DI NIEVOLE"},{"name":"MONTECATINI 
DI VAL DI NIEVOLE","end":"1940-10-07","start":"1934-11-03"}],"E551":[{"end":
"1927-01-11","start":"1861-03-17","name":"LESSOLO"},{"name":"LESSOLO","start
":"1927-01-12","end":"1945-10-04"},{"start":"1945-10-05","end":"9999-12-31",
"name":"LESSOLO"}],"B376":[{"start":"1861-03-17","end":"1935-04-07","name":"
CALAMANDRANA"},{"name":"CALAMANDRANA","start":"1935-04-08","end":"9999-12-31
"}],"H726":[{"name":"SALUSSOLA","end":"1927-01-11","start":"1861-03-17"},{"n
ame":"SALUSSOLA","start":"1927-01-12","end":"1992-04-15"},{"start":"1992-04-
16","end":"9999-12-31","name":"SALUSSOLA"}],"A150":[{"name":"ALBERONA","end"
:"9999-12-31","start":"1861-03-17"}],"H505":[{"name":"ROMAGNESE","start":"18
61-03-17","end":"1923-07-22"},{"end":"1927-01-26","start":"1923-07-23","name
":"ROMAGNESE"},{"name":"ROMAGNESE","end":"9999-12-31","start":"1927-01-27"}]
,"B441":[{"start":"1866-11-19","end":"9999-12-31","name":"CALVENE"}],"A599":
[{"name":"BALME","end":"9999-12-31","start":"1861-03-17"}],"I192":[{"name":"
SANT'AGATA DI ESARO","start":"1862-12-12","end":"9999-12-31"}],"F771":[{"nam
e":"MOTTA BALUFFI","end":"9999-12-31","start":"1868-07-13"}],"D940":[{"start
":"1861-03-17","end":"1928-10-08","name":"GAVARDO"},{"end":"9999-12-31","sta
rt":"1928-10-09","name":"GAVARDO"}],"E367":[{"name":"ISPRA","end":"1927-01-1
1","start":"1861-03-17"},{"name":"ISPRA","start":"1927-01-12","end":"1928-06
-14"},{"start":"1928-06-15","end":"9999-12-31","name":"ISPRA"}],"C358":[{"na
me":"CAURIA","start":"1920-10-16","end":"1923-02-13"},{"name":"CAURIA","star
t":"1923-02-14","end":"1927-01-13"}],"H908":[{"name":"SAN GIOVANNI A TEDUCCI
O","end":"1925-11-29","start":"1861-03-17"}],"E964":[{"start":"1861-03-17","
end":"1929-04-13","name":"MARMORITO"}],"F233":[{"name":"MIRABELLO SANNITICO"
,"end":"1928-05-12","start":"1863-02-06"},{"start":"1947-05-22","end":"1970-
03-02","name":"MIRABELLO SANNITICO"},{"end":"9999-12-31","start":"1970-03-03
","name":"MIRABELLO SANNITICO"}],"B217":[{"name":"BRUMANO","end":"9999-12-31
","start":"1861-03-17"}],"Z209":[{"name":"Sri Lanka","end":"9999-12-31","sta
rt":"0000-01-01"}],"E858":[{"name":"MALGRATE","start":"1861-03-17","end":"19
92-04-15"},{"name":"MALGRATE","end":"9999-12-31","start":"1992-04-16"}],"E56
6":[{"name":"LEVONE","end":"9999-12-31","start":"1896-05-11"}],"D109":[{"end
":"9999-12-31","start":"1863-04-13","name":"COSTA DE' NOBILI"}],"H078":[{"en
d":"1928-08-13","start":"1861-03-17","name":"PROVAGLIO D'ISEO"},{"name":"PRO
VAGLIO D'ISEO","end":"1929-05-27","start":"1928-08-14"},{"end":"9999-12-31",
"start":"1929-05-28","name":"PROVAGLIO D'ISEO"}],"A249":[{"name":"ALZATE BRI
ANZA","end":"1928-09-24","start":"1919-09-22"},{"name":"ALZATE BRIANZA","sta
rt":"1928-09-25","end":"9999-12-31"},{"name":"ALZATE CON VERZAGO","start":"1
864-03-11","end":"1919-09-21"}],"B135":[{"name":"BREGUZZO","end":"1923-02-13
","start":"1920-10-16"},{"name":"BREGUZZO","start":"1923-02-14","end":"1928-
02-23"},{"start":"1947-01-07","end":"2015-12-31","name":"BREGUZZO"}],"A842":
[{"end":"9999-12-31","start":"1861-03-17","name":"BIANCHI"}],"C120":[{"name"
:"CASTELGRANDE","end":"9999-12-31","start":"1861-03-17"}],"G633":[{"name":"P
IEVE DI CORIANO","start":"1867-06-24","end":"2017-12-31"}],"A694":[{"end":"1
923-02-13","start":"1920-10-16","name":"BASELGA DI PIN\u00c8"},{"name":"BASE
LGA DI PIN\u00c8","end":"1928-09-08","start":"1923-02-14"},{"name":"BASELGA 
DI PIN\u00c8","start":"1928-09-09","end":"9999-12-31"}],"G406":[{"end":"1923
-02-13","start":"1920-10-16","name":"PEDEMONTE"},{"name":"PEDEMONTE","end":"
1929-07-25","start":"1923-02-14"},{"end":"1980-06-19","start":"1929-07-26","
name":"PEDEMONTE"},{"end":"9999-12-31","start":"1980-06-20","name":"PEDEMONT
E"}],"C867":[{"end":"1927-01-11","start":"1861-03-17","name":"COLLERETTO CAS
TELNUOVO"},{"name":"COLLERETTO CASTELNUOVO","start":"1927-01-12","end":"1928
-09-25"},{"name":"COLLERETTO CASTELNUOVO","end":"1956-11-09","start":"1948-0
1-15"},{"start":"1956-11-10","end":"9999-12-31","name":"COLLERETTO CASTELNUO
VO"}],"A463":[{"end":"9999-12-31","start":"1862-11-10","name":"ASCOLI SATRIA
NO"}],"I792":[{"name":"SOLBIATE","end":"2018-12-31","start":"1861-03-17"}],"
F171":[{"name":"MEZZANA RABATTONE","start":"1873-05-10","end":"9999-12-31"}]
,"E678":[{"name":"LONGOBUCCO","start":"1861-03-17","end":"9999-12-31"}],"L98
5":[{"name":"VILLANOVA DEL GHEBBO","start":"1866-11-19","end":"9999-12-31"}]
,"H425":[{"name":"ROCCAMORICE","start":"1861-03-17","end":"1927-01-11"},{"en
d":"1929-05-22","start":"1927-01-12","name":"ROCCAMORICE"},{"name":"ROCCAMOR
ICE","end":"9999-12-31","start":"1946-11-02"}],"B510":[{"name":"VALPRATO SOA
NA","end":"1945-10-04","start":"1929-01-22"},{"start":"1945-10-05","end":"99
99-12-31","name":"VALPRATO SOANA"}],"F585":[{"end":"1927-03-30","start":"186
1-03-17","name":"MONTEPAGANO"},{"end":"9999-12-31","start":"1927-03-31","nam
e":"ROSETO DEGLI ABRUZZI"}],"D997":[{"start":"1861-03-17","end":"1974-08-19"
,"name":"GESTURI"},{"end":"2005-12-31","start":"1974-08-20","name":"GESTURI"
},{"name":"GESTURI","start":"2006-01-01","end":"2016-04-27"},{"end":"9999-12
-31","start":"2016-04-28","name":"GESTURI"}],"D709":[{"end":"1927-01-11","st
art":"1861-03-17","name":"FORMICOLA"},{"name":"FORMICOLA","start":"1927-01-1
2","end":"1945-08-31"},{"name":"FORMICOLA","end":"9999-12-31","start":"1945-
09-01"}],"B735":[{"start":"1871-01-15","end":"1927-01-11","name":"CARBOGNANO
"},{"start":"1927-01-12","end":"9999-12-31","name":"CARBOGNANO"}],"I423":[{"
name":"SARACENA","start":"1861-03-17","end":"9999-12-31"}],"L881":[{"start":
"1861-03-17","end":"9999-12-31","name":"VIGNALE MONFERRATO"}],"C720":[{"name
":"CIRCHINA","end":"1923-02-07","start":"1921-01-05"},{"start":"1923-02-08",
"end":"1927-01-11","name":"CIRCHINA"},{"name":"CIRCHINA","start":"1927-01-12
","end":"1928-07-28"},{"name":"CIRCHINA","start":"1928-07-29","end":"1947-09
-15"}],"B302":[{"name":"BUTERA","start":"1861-03-17","end":"9999-12-31"}],"L
348":[{"name":"TRAVO","end":"1927-06-27","start":"1861-03-17"},{"name":"TRAV
O","start":"1927-06-28","end":"9999-12-31"}],"I610":[{"start":"1861-03-17","
end":"9999-12-31","name":"SENISE"}],"D394":[{"name":"ELICE","start":"1861-03
-17","end":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":"EL
ICE"}],"L229":[{"name":"TORNOLO","end":"9999-12-31","start":"1893-08-11"}],"
I368":[{"end":"1918-05-12","start":"1861-03-17","name":"SANTO STEFANO D'AVET
O"},{"name":"SANTO STEFANO D'AVETO","end":"9999-12-31","start":"1918-05-13"}
],"B270":[{"name":"BUONVICINO","start":"1861-03-17","end":"9999-12-31"}],"G7
05":[{"name":"PISCINA","start":"1893-05-26","end":"1960-09-28"},{"name":"PIS
CINA","end":"9999-12-31","start":"1960-09-29"}],"F988":[{"name":"NUSCO","end
":"9999-12-31","start":"1861-03-17"}],"A035":[{"name":"ACQUALAGNA","end":"19
98-09-03","start":"1861-03-17"},{"end":"9999-12-31","start":"1998-09-04","na
me":"ACQUALAGNA"}],"G332":[{"name":"PARENZO","end":"1923-02-07","start":"192
1-01-05"},{"name":"PARENZO","end":"1947-09-15","start":"1923-02-08"}],"D284"
:[{"end":"9999-12-31","start":"1926-09-17","name":"DESENZANO DEL GARDA"},{"n
ame":"DESENZANO SUL LAGO","start":"1862-09-22","end":"1926-09-16"}],"G526":[
{"start":"1861-03-17","end":"9999-12-31","name":"PEVERAGNO"}],"D628":[{"star
t":"1861-03-17","end":"9999-12-31","name":"FIUMINATA"}],"L822":[{"name":"VEZ
ZI","end":"1871-03-01","start":"1861-03-17"}],"G472":[{"end":"1923-02-07","s
tart":"1921-01-05","name":"PERTEOLE"},{"end":"1928-04-15","start":"1923-02-0
8","name":"PERTEOLE"}],"I857":[{"name":"SORISO","end":"9999-12-31","start":"
1861-03-17"}],"G924":[{"start":"1861-03-17","end":"9999-12-31","name":"PORTO
 TORRES"}],"H126":[{"start":"1861-03-17","end":"1927-01-11","name":"QUILIANO
"},{"name":"QUILIANO","start":"1927-01-12","end":"1952-05-23"},{"name":"QUIL
IANO","start":"1952-05-24","end":"9999-12-31"}],"M048":[{"start":"1866-11-19
","end":"9999-12-31","name":"VILLORBA"}],"A750":[{"name":"BELLINO","end":"99
99-12-31","start":"1861-03-17"}],"G327":[{"name":"PARATICO","end":"9999-12-3
1","start":"1861-03-17"}],"L837":[{"name":"VICARI","end":"9999-12-31","start
":"1861-03-17"}],"L588":[{"start":"1920-10-16","end":"1923-02-13","name":"VA
LLARSA"},{"name":"VALLARSA","end":"9999-12-31","start":"1923-02-14"}],"H325"
:[{"start":"1861-03-17","end":"1939-12-28","name":"RIPOSTO"},{"name":"RIPOST
O","end":"9999-12-31","start":"1945-10-25"}],"B775":[{"name":"CARID\u00c0","
end":"1928-05-03","start":"1861-03-17"}],"F146":[{"start":"1861-03-17","end"
:"1992-05-22","name":"MERGOZZO"},{"end":"9999-12-31","start":"1992-05-23","n
ame":"MERGOZZO"}],"G200":[{"end":"9999-12-31","start":"1861-03-17","name":"O
VINDOLI"}],"L144":[{"name":"TERZO D'AQUILEIA","end":"1923-02-07","start":"19
21-01-05"},{"start":"1923-02-08","end":"1924-01-14","name":"TERZO D'AQUILEIA
"},{"start":"1947-01-09","end":"1968-04-05","name":"TERZO D'AQUILEIA"},{"end
":"9999-12-31","start":"1968-04-06","name":"TERZO D'AQUILEIA"}],"D198":[{"na
me":"CUGGIONO","end":"9999-12-31","start":"1861-03-17"}],"B024":[{"name":"BO
RGONE SUSA","start":"1863-01-05","end":"9999-12-31"}],"L448":[{"name":"TROIN
A","end":"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"999
9-12-31","name":"TROINA"}],"F600":[{"end":"9999-12-31","start":"1861-03-17",
"name":"MONTE ROBERTO"}],"C989":[{"end":"1928-05-02","start":"1861-03-17","n
ame":"CORCIAGO"}],"C754":[{"name":"CIVENNA","start":"1861-03-17","end":"1927
-12-26"},{"name":"CIVENNA","start":"1927-12-27","end":"1950-04-04"},{"name":
"CIVENNA","start":"1950-04-05","end":"2014-02-03"}],"D494":[{"end":"9999-12-
31","start":"1861-03-17","name":"FARA FILIORUM PETRI"}],"C686":[{"name":"CIL
AVEGNA","end":"9999-12-31","start":"1861-03-17"}],"A508":[{"name":"AVELLA","
end":"9999-12-31","start":"1861-03-17"}],"G853":[{"start":"1861-03-17","end"
:"1970-03-05","name":"PONTERANICA"},{"name":"PONTERANICA","start":"1970-03-0
6","end":"9999-12-31"}],"D663":[{"start":"1920-10-16","end":"1923-02-13","na
me":"FONDO"},{"name":"FONDO","end":"1928-06-29","start":"1923-02-14"},{"end"
:"1947-03-26","start":"1928-06-30","name":"FONDO"},{"start":"1947-03-27","en
d":"2019-12-31","name":"FONDO"}],"F131":[{"name":"MERANA","start":"1861-03-1
7","end":"1928-12-05"},{"name":"MERANA","start":"1947-10-02","end":"1953-11-
16"},{"name":"MERANA","start":"1953-11-17","end":"9999-12-31"}],"B402":[{"na
me":"CALDIERO","end":"9999-12-31","start":"1866-11-19"}],"A255":[{"name":"AM
ARONI","end":"9999-12-31","start":"1861-03-17"}],"I029":[{"start":"1864-02-0
2","end":"1927-01-11","name":"SAN MAURO FORTE"},{"end":"9999-12-31","start":
"1927-01-12","name":"SAN MAURO FORTE"}],"H707":[{"name":"SALI VERCELLESE","e
nd":"1927-01-11","start":"1864-01-29"},{"name":"SALI VERCELLESE","end":"9999
-12-31","start":"1927-01-12"}],"H999":[{"name":"SAN MARTINO AL TAGLIAMENTO",
"end":"1928-12-15","start":"1867-09-02"},{"end":"1968-04-05","start":"1946-1
2-24","name":"SAN MARTINO AL TAGLIAMENTO"},{"name":"SAN MARTINO AL TAGLIAMEN
TO","end":"9999-12-31","start":"1968-04-06"}],"E638":[{"name":"LOCATE VARESI
NO","start":"1863-02-23","end":"1928-09-22"},{"name":"LOCATE VARESINO","end"
:"9999-12-31","start":"1950-04-05"}],"A905":[{"name":"BLEVIO","end":"1930-11
-05","start":"1861-03-17"},{"start":"1930-11-06","end":"9999-12-31","name":"
BLEVIO"}],"E080":[{"start":"1861-03-17","end":"1924-01-06","name":"GOLESE"},
{"name":"GOLESE","end":"1943-05-28","start":"1924-01-07"}],"A662":[{"end":"1
934-02-08","start":"1931-05-17","name":"BARI"},{"name":"BARI","end":"1937-04
-01","start":"1934-02-09"},{"end":"9999-12-31","start":"1937-04-02","name":"
BARI"},{"name":"BARI DELLE PUGLIE","start":"1863-01-19","end":"1928-03-23"},
{"start":"1928-03-24","end":"1928-10-15","name":"BARI DELLE PUGLIE"},{"end":
"1931-05-16","start":"1928-10-16","name":"BARI DELLE PUGLIE"}],"C647":[{"nam
e":"CHIRIGNAGO","start":"1866-11-19","end":"1926-08-23"}],"G432":[{"end":"19
27-01-11","start":"1863-04-13","name":"PENNA IN TEVERINA"},{"name":"PENNA IN
 TEVERINA","end":"9999-12-31","start":"1927-01-12"}],"B169":[{"name":"BRIATI
CO","start":"1861-03-17","end":"1992-04-15"},{"name":"BRIATICO","start":"199
2-04-16","end":"9999-12-31"}],"I468":[{"end":"1992-04-15","start":"1861-03-1
7","name":"SAVELLI"},{"name":"SAVELLI","start":"1992-04-16","end":"9999-12-3
1"}],"C112":[{"name":"CASTELFRANCO DI SOPRA","start":"1861-03-17","end":"201
3-12-31"}],"E992":[{"start":"1861-03-17","end":"9999-12-31","name":"MARTIS"}
],"A801":[{"name":"BERNALDA","start":"1861-03-17","end":"1927-01-11"},{"name
":"BERNALDA","start":"1927-01-12","end":"1933-01-31"},{"name":"BERNALDA","st
art":"1933-02-01","end":"9999-12-31"}],"A724":[{"start":"1861-03-17","end":"
1926-02-05","name":"BAVARI"}],"F187":[{"end":"1923-02-13","start":"1920-10-1
6","name":"MEZZOLOMBARDO"},{"name":"MEZZOLOMBARDO","start":"1923-02-14","end
":"9999-12-31"}],"B230":[{"name":"BRUSSON","end":"1927-01-11","start":"1861-
03-17"},{"name":"BRUSSON","start":"1927-01-12","end":"1939-10-20"},{"name":"
BRUSSON","end":"9999-12-31","start":"1946-01-30"},{"name":"BRUSSONE","start"
:"1939-10-21","end":"1946-01-29"}],"G427":[{"end":"2016-12-31","start":"1861
-03-17","name":"PELLIO INTELVI"}],"C225":[{"end":"9999-12-31","start":"1970-
08-04","name":"CASTELNUOVO DEL GARDA"},{"name":"CASTELNUOVO DI VERONA","end"
:"1970-08-03","start":"1867-08-26"}],"I164":[{"start":"1861-03-17","end":"99
99-12-31","name":"SAN SOSTENE"}],"G950":[{"name":"POZZA","start":"1920-10-16
","end":"1923-02-13"},{"end":"1926-07-22","start":"1923-02-14","name":"POZZA
"},{"start":"1952-05-17","end":"1956-01-03","name":"POZZA"},{"name":"POZZA D
I FASSA","end":"2017-12-31","start":"1956-01-04"}],"A075":[{"start":"1866-11
-19","end":"9999-12-31","name":"AGNA"}],"G372":[{"end":"9999-12-31","start":
"1863-05-06","name":"PATERNO CALABRO"}],"M268":[{"end":"9999-12-31","start":
"1972-04-05","name":"BLUFI"}],"F445":[{"end":"9999-12-31","start":"1861-03-1
7","name":"MONTEBRUNO"}],"D362":[{"end":"1923-02-07","start":"1922-02-02","n
ame":"DRAGUCCIO"},{"start":"1923-02-08","end":"1928-11-30","name":"DRAGUCCIO
"}],"B522":[{"name":"CAMPOCHIARO","end":"1970-03-02","start":"1861-03-17"},{
"name":"CAMPOCHIARO","end":"9999-12-31","start":"1970-03-03"}],"A428":[{"nam
e":"AROLO","start":"1861-03-17","end":"1927-01-11"},{"name":"AROLO","start":
"1927-01-12","end":"1928-01-07"}],"F513":[{"end":"1927-01-11","start":"1869-
07-12","name":"MONTEGABBIONE"},{"name":"MONTEGABBIONE","start":"1927-01-12",
"end":"1928-01-28"},{"name":"MONTEGABBIONE","start":"1928-01-29","end":"9999
-12-31"}],"I622":[{"name":"SERAVEZZA","end":"9999-12-31","start":"1861-03-17
"}],"M182":[{"end":"9999-12-31","start":"1861-03-17","name":"ZOAGLI"}],"B769
":[{"name":"CAREZZANO","end":"1928-08-06","start":"1928-03-07"},{"end":"1947
-09-30","start":"1928-08-07","name":"CAREZZANO"},{"start":"1947-10-01","end"
:"9999-12-31","name":"CAREZZANO"},{"name":"CAREZZANO INFERIORE","start":"186
1-03-17","end":"1927-05-09"},{"name":"CAREZZANO MAGGIORE","end":"1928-03-06"
,"start":"1927-05-10"}],"C712":[{"name":"CINTE TESINO","end":"1923-02-13","s
tart":"1920-10-16"},{"name":"CINTE TESINO","start":"1923-02-14","end":"1928-
02-20"},{"end":"9999-12-31","start":"1947-03-30","name":"CINTE TESINO"}],"L9
13":[{"end":"9999-12-31","start":"1890-12-01","name":"VILLA BASILICA"}],"L65
7":[{"name":"VALVASONE","end":"1928-12-14","start":"1866-11-19"},{"start":"1
928-12-15","end":"1946-12-23","name":"VALVASONE"},{"name":"VALVASONE","end":
"1968-04-05","start":"1946-12-24"},{"name":"VALVASONE","end":"2014-12-31","s
tart":"1968-04-06"}],"A124":[{"end":"1931-02-05","start":"1861-03-17","name"
:"ALBA"},{"name":"ALBA","start":"1931-02-06","end":"1947-12-21"},{"end":"195
7-07-19","start":"1947-12-22","name":"ALBA"},{"start":"1957-07-20","end":"99
99-12-31","name":"ALBA"}],"C690":[{"start":"1861-03-17","end":"1928-07-29","
name":"CIMAMULERA"}],"B888":[{"end":"9999-12-31","start":"1862-12-29","name"
:"CASALETTO SPARTANO"}],"B537":[{"start":"1861-03-17","end":"1924-01-13","na
me":"CAMPOFRANCO"},{"name":"CAMPOFRANCO","start":"1924-01-14","end":"9999-12
-31"}],"B476":[{"name":"CAMEROTA","end":"1928-10-06","start":"1861-03-17"},{
"start":"1928-10-07","end":"9999-12-31","name":"CAMEROTA"}],"I637":[{"start"
:"1861-03-17","end":"1935-04-07","name":"SEROLE"},{"name":"SEROLE","end":"99
99-12-31","start":"1935-04-08"}],"I764":[{"name":"SIURGUS","start":"1861-03-
17","end":"1927-09-15"}],"F053":[{"end":"1928-03-12","start":"1861-03-17","n
ame":"MATHI"},{"name":"MATHI","start":"1928-03-13","end":"1939-07-20"},{"end
":"9999-12-31","start":"1959-10-03","name":"MATHI"},{"name":"MATI","start":"
1939-07-21","end":"1947-01-08"},{"start":"1947-01-09","end":"1959-10-02","na
me":"MATI"}],"H680":[{"name":"SALA AL BARRO","end":"1928-02-09","start":"186
3-02-23"}],"F359":[{"name":"MONFORTE SAN GIORGIO","end":"9999-12-31","start"
:"1863-04-10"}],"E096":[{"end":"9999-12-31","start":"1861-03-17","name":"GOR
IANO SICOLI"}],"H038":[{"name":"PREONE","start":"1866-11-19","end":"1928-04-
07"},{"start":"1946-12-24","end":"1968-04-05","name":"PREONE"},{"end":"9999-
12-31","start":"1968-04-06","name":"PREONE"}],"B175":[{"end":"9999-12-31","s
tart":"1947-10-07","name":"BRIGA ALTA"}],"C458":[{"start":"1866-11-19","end"
:"1964-06-26","name":"CENCENIGHE"},{"name":"CENCENIGHE AGORDINO","end":"9999
-12-31","start":"1964-06-27"}],"F746":[{"name":"MORRO REATINO","end":"1923-0
3-18","start":"1863-04-13"},{"start":"1923-03-19","end":"1927-01-11","name":
"MORRO REATINO"},{"name":"MORRO REATINO","start":"1927-01-12","end":"9999-12
-31"}],"G507":[{"name":"PETESCIA","end":"1923-03-18","start":"1861-03-17"},{
"end":"1927-01-11","start":"1923-03-19","name":"PETESCIA"},{"name":"PETESCIA
","start":"1927-01-12","end":"1950-06-03"},{"name":"TURANIA","end":"9999-12-
31","start":"1950-06-04"}],"L744":[{"start":"1861-03-17","end":"1928-03-08",
"name":"VERANO"},{"end":"2009-06-29","start":"1928-03-09","name":"VERANO BRI
ANZA"},{"start":"2009-06-30","end":"9999-12-31","name":"VERANO BRIANZA"}],"D
798":[{"start":"1861-03-17","end":"9999-12-31","name":"FRIGENTO"}],"E467":[{
"name":"LATERA","end":"1927-01-11","start":"1871-01-15"},{"name":"LATERA","e
nd":"9999-12-31","start":"1927-01-12"}],"G673":[{"end":"9999-12-31","start":
"1878-12-23","name":"PINCARA"}],"G306":[{"start":"1868-07-29","end":"9999-12
-31","name":"PANDINO"}],"C154":[{"name":"CASTELLERO","end":"1928-06-26","sta
rt":"1861-03-17"},{"start":"1947-01-10","end":"9999-12-31","name":"CASTELLER
O"}],"B581":[{"start":"1861-03-17","end":"1927-01-11","name":"CANCELLO ED AR
NONE"},{"name":"CANCELLO ED ARNONE","end":"1945-08-31","start":"1927-01-12"}
,{"name":"CANCELLO ED ARNONE","start":"1945-09-01","end":"9999-12-31"}],"A36
3":[{"end":"1927-01-11","start":"1861-03-17","name":"ARCE"},{"name":"ARCE","
end":"1929-06-17","start":"1927-01-12"},{"start":"1929-06-18","end":"9999-12
-31","name":"ARCE"}],"A069":[{"end":"1958-08-11","start":"1861-03-17","name"
:"AGGIUS"},{"start":"1958-08-12","end":"1969-04-14","name":"AGGIUS"},{"name"
:"AGGIUS","end":"1975-05-03","start":"1969-04-15"},{"name":"AGGIUS","start":
"1975-05-04","end":"2005-12-31"},{"name":"AGGIUS","start":"2006-01-01","end"
:"2016-04-27"},{"name":"AGGIUS","end":"9999-12-31","start":"2016-04-28"}],"F
731":[{"name":"MORIGERATI","start":"1861-03-17","end":"9999-12-31"}],"L810":
[{"end":"9999-12-31","start":"1866-11-19","name":"VESTENANOVA"}],"F273":[{"n
ame":"MOIANA","end":"1928-12-15","start":"1861-03-17"}],"M121":[{"name":"VOL
PEGLINO","end":"1929-01-23","start":"1861-03-17"},{"name":"VOLPEGLINO","star
t":"1947-09-04","end":"9999-12-31"}],"H107":[{"end":"1928-03-07","start":"18
61-03-17","name":"QUARNA SOTTO"},{"name":"QUARNA SOTTO","end":"1992-05-22","
start":"1947-10-01"},{"name":"QUARNA SOTTO","start":"1992-05-23","end":"9999
-12-31"}],"I681":[{"start":"1861-03-17","end":"9999-12-31","name":"SESTINO"}
],"G621":[{"end":"1927-01-11","start":"1861-03-17","name":"PIETRANICO"},{"na
me":"PIETRANICO","end":"9999-12-31","start":"1927-01-12"}],"C436":[{"start":
"1861-03-17","end":"9999-12-31","name":"CELLAMARE"}],"E876":[{"start":"1861-
03-17","end":"9999-12-31","name":"MANDANICI"}],"I060":[{"start":"1912-01-01"
,"end":"9999-12-31","name":"SAN NICOLA ARCELLA"}],"D419":[{"start":"1866-11-
19","end":"9999-12-31","name":"ERB\u00c8"}],"H056":[{"name":"PREZZA","start"
:"1861-03-17","end":"9999-12-31"}],"C603":[{"name":"CHIAIANO ED UNITI","end"
:"1926-07-03","start":"1863-02-06"}],"H317":[{"name":"RIPALTA NUOVA","start"
:"1875-04-01","end":"1928-05-29"}],"H914":[{"end":"9999-12-31","start":"1879
-03-28","name":"SAN GIOVANNI GEMINI"}],"G116":[{"name":"ORNAGO","start":"186
1-03-17","end":"2009-06-29"},{"start":"2009-06-30","end":"9999-12-31","name"
:"ORNAGO"}],"C577":[{"start":"1867-04-12","end":"9999-12-31","name":"CESIOMA
GGIORE"}],"E488":[{"end":"9999-12-31","start":"1861-03-17","name":"LAVAGNA"}
],"H516":[{"name":"ROMBIOLO","start":"1861-03-17","end":"1992-04-15"},{"name
":"ROMBIOLO","end":"9999-12-31","start":"1992-04-16"}],"M173":[{"name":"ZIAN
O","end":"1923-02-13","start":"1920-10-16"},{"name":"ZIANO","end":"1929-01-2
4","start":"1923-02-14"},{"end":"1947-03-14","start":"1929-01-25","name":"ZI
ANO"},{"start":"1947-03-15","end":"1955-03-01","name":"ZIANO"},{"end":"9999-
12-31","start":"1955-03-02","name":"ZIANO DI FIEMME"}],"C441":[{"end":"1928-
05-30","start":"1862-12-19","name":"CELLE DI MACRA"},{"name":"CELLE DI MACRA
","end":"9999-12-31","start":"1946-12-28"}],"F221":[{"name":"MINERVINO DI LE
CCE","start":"1864-04-01","end":"1913-07-09"},{"name":"MINERVINO DI LECCE","
start":"1913-07-10","end":"9999-12-31"}],"F392":[{"name":"MONTAGNA","start":
"1920-10-16","end":"1923-02-13"},{"end":"1948-03-13","start":"1923-02-14","n
ame":"MONTAGNA"},{"end":"1985-09-10","start":"1948-03-14","name":"MONTAGNA"}
,{"end":"2005-05-03","start":"1985-09-11","name":"MONTAGNA"},{"end":"2023-04
-13","start":"2005-05-04","name":"MONTAGNA"},{"name":"MONTAGNA SULLA STRADA 
DEL VINO","end":"9999-12-31","start":"2023-04-14"}],"M340":[{"start":"2014-0
2-04","end":"9999-12-31","name":"BORGO VIRGILIO"}],"L765":[{"start":"1871-08
-03","end":"1927-01-11","name":"VERGIATE"},{"name":"VERGIATE","end":"9999-12
-31","start":"1927-01-12"}],"C376":[{"name":"CAVALLERMAGGIORE","end":"9999-1
2-31","start":"1861-03-17"}],"E184":[{"name":"GRISIGNANO DI ZOCCO","end":"99
99-12-31","start":"1867-09-02"}],"H276":[{"name":"RICENGO","end":"9999-12-31
","start":"1869-05-17"}],"F282":[{"name":"MOLASSANA","start":"1861-03-17","e
nd":"1926-02-05"}],"H061":[{"end":"9999-12-31","start":"1862-12-29","name":"
PRIGNANO SULLA SECCHIA"}],"D826":[{"name":"FURORE","start":"1861-03-17","end
":"1928-10-29"},{"name":"FURORE","end":"9999-12-31","start":"1947-10-26"}],"
I745":[{"start":"1861-03-17","end":"9999-12-31","name":"SIMERI CRICHI"}],"B3
58":[{"name":"CAGNANO AMITERNO","start":"1864-06-23","end":"9999-12-31"}],"F
889":[{"end":"9999-12-31","start":"1868-09-09","name":"NICHELINO"}],"G415":[
{"end":"1928-04-30","start":"1861-03-17","name":"PEGLIO"},{"start":"1948-03-
21","end":"9999-12-31","name":"PEGLIO"}],"B867":[{"start":"1895-12-09","end"
:"1937-10-22","name":"CASALBORGONE"},{"start":"1937-10-23","end":"9999-12-31
","name":"CASALBORGONE"}],"F768":[{"name":"MOSSANO","end":"1939-09-06","star
t":"1866-11-19"},{"name":"MOSSANO","end":"2018-02-16","start":"1947-12-22"}]
,"B120":[{"name":"BRANDICO","start":"1861-03-17","end":"1927-11-23"},{"end":
"9999-12-31","start":"1947-05-12","name":"BRANDICO"}],"L179":[{"end":"1923-0
2-13","start":"1920-10-16","name":"TISA"},{"name":"TISA","end":"1923-04-26",
"start":"1923-02-14"},{"name":"TISO","end":"1927-01-11","start":"1923-04-27"
},{"name":"TISO","end":"1929-04-11","start":"1927-01-12"}],"G682":[{"name":"
PIOBBICO","end":"1998-09-03","start":"1861-03-17"},{"name":"PIOBBICO","start
":"1998-09-04","end":"9999-12-31"}],"C135":[{"start":"1947-07-10","end":"198
1-04-15","name":"CASTELLANA SICULA"},{"start":"1981-04-16","end":"9999-12-31
","name":"CASTELLANA SICULA"}],"L040":[{"name":"TAMBRE","end":"9999-12-31","
start":"1866-11-19"}],"H743":[{"end":"9999-12-31","start":"1923-07-20","name
":"SAMBUCA DI SICILIA"},{"name":"SAMBUCA ZABUT","start":"1864-01-16","end":"
1923-07-19"}],"C217":[{"name":"CASTELNOVO DEL FRIULI","start":"1867-09-02","
end":"1968-04-05"},{"name":"CASTELNOVO DEL FRIULI","start":"1968-04-06","end
":"2009-04-15"},{"name":"CASTELNOVO DEL FRIULI","end":"9999-12-31","start":"
2009-04-16"}],"L466":[{"name":"TUORO","end":"1922-03-06","start":"1875-05-11
"},{"name":"TUORO SUL TRASIMENO","end":"1937-03-11","start":"1922-03-07"},{"
end":"9999-12-31","start":"1937-03-12","name":"TUORO SUL TRASIMENO"}],"E209"
:[{"end":"9999-12-31","start":"1861-03-17","name":"GROTTE"}],"F464":[{"start
":"1866-11-19","end":"9999-12-31","name":"MONTECCHIO MAGGIORE"}],"F693":[{"s
tart":"1861-03-17","end":"2013-12-02","name":"MONTORO INFERIORE"}],"I145":[{
"end":"1948-07-10","start":"1863-02-06","name":"SAN SALVATORE TELESINO"},{"e
nd":"9999-12-31","start":"1948-07-11","name":"SAN SALVATORE TELESINO"}],"G54
3":[{"name":"PIANA DEGLI ALBANESI","start":"1941-08-20","end":"9999-12-31"},
{"end":"1941-08-19","start":"1861-03-17","name":"PIANA DEI GRECI"}],"I487":[
{"end":"9999-12-31","start":"1861-03-17","name":"SCALDASOLE"}],"C050":[{"end
":"1929-06-12","start":"1861-03-17","name":"CASTANA"},{"name":"CASTANA","end
":"1948-12-25","start":"1929-06-13"},{"name":"CASTANA","end":"9999-12-31","s
tart":"1948-12-26"}],"L097":[{"start":"1920-10-16","end":"1923-02-13","name"
:"TENNO"},{"name":"TENNO","end":"1929-04-11","start":"1923-02-14"},{"name":"
TENNO","end":"1996-02-13","start":"1929-04-12"},{"name":"TENNO","end":"9999-
12-31","start":"1996-02-14"}],"H430":[{"name":"ROCCAPIATTA","start":"1861-03
-17","end":"1928-05-30"}],"A991":[{"start":"1861-03-17","end":"9999-12-31","
name":"BORGETTO"}],"E802":[{"name":"MAGGIANICO","end":"1924-01-10","start":"
1869-02-08"},{"name":"MAGGIANICO","end":"1928-07-18","start":"1924-01-11"}],
"A234":[{"name":"ALTILIA","end":"1928-05-21","start":"1861-03-17"},{"end":"9
999-12-31","start":"1937-07-18","name":"ALTILIA"}],"G293":[{"end":"9999-12-3
1","start":"1895-11-27","name":"PALOMBARA SABINA"}],"D085":[{"name":"COSEANO
","end":"1968-04-05","start":"1866-11-19"},{"name":"COSEANO","end":"2006-11-
29","start":"1968-04-06"},{"start":"2006-11-30","end":"2014-02-03","name":"C
OSEANO"},{"end":"9999-12-31","start":"2014-02-04","name":"COSEANO"}],"F168":
[{"end":"1923-02-13","start":"1920-10-16","name":"MEZZANA"},{"name":"MEZZANA
","start":"1923-02-14","end":"9999-12-31"}],"C302":[{"end":"9999-12-31","sta
rt":"1863-04-23","name":"CASTIGLIONE CHIAVARESE"}],"B720":[{"name":"CARAMAGN
A PIEMONTE","end":"9999-12-31","start":"1862-12-19"}],"H952":[{"end":"1953-1
1-24","start":"1928-04-07","name":"SAN LEONARDO IN PASSIRIA"},{"end":"9999-1
2-31","start":"1953-11-25","name":"SAN LEONARDO IN PASSIRIA"}],"L779":[{"end
":"1919-08-21","start":"1861-03-17","name":"VEROLENGO"},{"name":"VEROLENGO",
"end":"9999-12-31","start":"1919-08-22"}],"C735":[{"name":"CISON DI VALMARIN
O","end":"9999-12-31","start":"1867-11-25"}],"M287":[{"name":"RAGALNA","star
t":"1985-05-17","end":"9999-12-31"}],"E661":[{"start":"1861-03-17","end":"19
28-05-22","name":"LOMBRIASCO"},{"end":"9999-12-31","start":"1946-12-19","nam
e":"LOMBRIASCO"}],"A895":[{"start":"1874-04-22","end":"1927-01-11","name":"B
ITTI"},{"name":"BITTI","start":"1927-01-12","end":"9999-12-31"}],"H143":[{"s
tart":"1866-11-19","end":"1922-09-18","name":"QUISTELLO"},{"end":"9999-12-31
","start":"1922-09-19","name":"QUISTELLO"}],"C510":[{"name":"CERIALE","end":
"1927-01-11","start":"1861-03-17"},{"end":"9999-12-31","start":"1927-01-12",
"name":"CERIALE"}],"H809":[{"start":"1861-03-17","end":"1998-09-03","name":"
SAN COSTANZO"},{"name":"SAN COSTANZO","end":"9999-12-31","start":"1998-09-04
"}],"G383":[{"end":"1974-08-19","start":"1863-12-28","name":"SAN NICOL\u00d2
 GERREI"},{"name":"SAN NICOL\u00d2 GERREI","end":"2016-04-27","start":"1974-
08-20"},{"name":"SAN NICOL\u00d2 GERREI","end":"9999-12-31","start":"2016-04
-28"}],"G089":[{"end":"1998-09-03","start":"1861-03-17","name":"ORCIANO DI P
ESARO"},{"start":"1998-09-04","end":"2016-12-31","name":"ORCIANO DI PESARO"}
],"L451":[{"end":"1927-01-11","start":"1862-12-29","name":"TRONZANO VERCELLE
SE"},{"start":"1927-01-12","end":"9999-12-31","name":"TRONZANO VERCELLESE"}]
,"A020":[{"name":"ACERENZA","start":"1861-03-17","end":"9999-12-31"}],"G716"
:[{"end":"9999-12-31","start":"1861-03-17","name":"PITIGLIANO"}],"C270":[{"s
tart":"1928-04-17","end":"1934-03-06","name":"CASTEL SAN VINCENZO"},{"name":
"CASTEL SAN VINCENZO","start":"1934-03-07","end":"1970-03-02"},{"name":"CAST
EL SAN VINCENZO","end":"9999-12-31","start":"1970-03-03"}],"H202":[{"end":"1
927-01-11","start":"1861-03-17","name":"RAVISCANINA"},{"start":"1927-01-12",
"end":"1945-08-31","name":"RAVISCANINA"},{"start":"1945-09-01","end":"9999-1
2-31","name":"RAVISCANINA"}],"I431":[{"start":"1920-10-16","end":"1923-02-13
","name":"SARENTINO"},{"start":"1923-02-14","end":"1927-01-11","name":"SAREN
TINO"},{"name":"SARENTINO","start":"1927-01-12","end":"9999-12-31"}],"M397":
[{"end":"9999-12-31","start":"2018-01-01","name":"ALLUVIONI PIOVERA"}],"I371
":[{"start":"1861-03-17","end":"9999-12-31","name":"SANTO STEFANO IN ASPROMO
NTE"}],"E959":[{"start":"1920-10-16","end":"1923-02-13","name":"MARLENGO"},{
"name":"MARLENGO","start":"1923-02-14","end":"1927-01-11"},{"start":"1927-01
-12","end":"1974-09-17","name":"MARLENGO"},{"name":"MARLENGO","start":"1974-
09-18","end":"1985-10-22"},{"name":"MARLENGO","start":"1985-10-23","end":"99
99-12-31"}],"E784":[{"end":"1927-01-11","start":"1925-12-04","name":"MACERAT
A CAMPANIA"},{"end":"1928-12-14","start":"1927-01-12","name":"MACERATA CAMPA
NIA"},{"end":"9999-12-31","start":"1946-04-26","name":"MACERATA CAMPANIA"},{
"name":"MACERATA DI MARCIANISE","start":"1862-10-27","end":"1925-12-03"}],"L
165":[{"end":"9999-12-31","start":"1861-03-17","name":"TICINETO"}],"E656":[{
"end":"1992-04-15","start":"1861-03-17","name":"LOMAGNA"},{"start":"1992-04-
16","end":"9999-12-31","name":"LOMAGNA"}],"I446":[{"end":"1928-02-23","start
":"1863-02-23","name":"SARTIRANA BRIANTEA"}],"H330":[{"name":"RIVA","start":
"1920-10-16","end":"1923-02-13"},{"name":"RIVA","start":"1923-02-14","end":"
1929-05-27"},{"name":"RIVA","start":"1929-05-28","end":"1955-03-01"},{"start
":"1955-03-02","end":"1957-07-02","name":"RIVA"},{"start":"1957-07-03","end"
:"1969-12-23","name":"RIVA"},{"start":"1969-12-24","end":"1996-02-13","name"
:"RIVA DEL GARDA"},{"name":"RIVA DEL GARDA","end":"9999-12-31","start":"1996
-02-14"}],"F967":[{"name":"NOVI VELIA","start":"1862-12-29","end":"1928-12-2
9"},{"end":"9999-12-31","start":"1946-04-02","name":"NOVI VELIA"}],"L739":[{
"start":"1948-06-29","end":"9999-12-31","name":"VENTICANO"}],"C024":[{"name"
:"CASSINA VALSASSINA","end":"1929-02-14","start":"1861-03-17"},{"name":"CASS
INA VALSASSINA","start":"1948-03-21","end":"1992-04-15"},{"start":"1992-04-1
6","end":"9999-12-31","name":"CASSINA VALSASSINA"}],"A274":[{"start":"1920-1
0-16","end":"1923-02-13","name":"ANDALO"},{"name":"ANDALO","end":"1928-05-12
","start":"1923-02-14"},{"start":"1947-01-10","end":"9999-12-31","name":"AND
ALO"}],"I387":[{"name":"SAN VINCENZO AL VOLTURNO","end":"1928-04-17","start"
:"1863-02-06"}],"I984":[{"name":"STROPPIANA","end":"1927-01-11","start":"186
1-03-17"},{"start":"1927-01-12","end":"9999-12-31","name":"STROPPIANA"}],"I7
59":[{"name":"SIRONE","end":"1992-04-15","start":"1861-03-17"},{"name":"SIRO
NE","end":"9999-12-31","start":"1992-04-16"}],"G791":[{"name":"POLISTENA","s
tart":"1861-03-17","end":"1936-07-14"},{"name":"POLISTENA","end":"9999-12-31
","start":"1936-07-15"}],"L366":[{"name":"TREIA","start":"1861-03-17","end":
"9999-12-31"}],"F364":[{"start":"1861-03-17","end":"1978-12-28","name":"MONG
IANA"},{"end":"1992-04-15","start":"1978-12-29","name":"MONGIANA"},{"name":"
MONGIANA","end":"9999-12-31","start":"1992-04-16"}],"C402":[{"name":"CAVONA"
,"start":"1861-03-17","end":"1927-01-11"},{"name":"CAVONA","end":"1928-02-15
","start":"1927-01-12"}],"H470":[{"end":"2010-02-26","start":"1869-07-06","n
ame":"RODANO"},{"name":"RODANO","end":"9999-12-31","start":"2010-02-27"}],"L
567":[{"end":"9999-12-31","start":"1872-01-01","name":"VALEGGIO SUL MINCIO"}
],"B686":[{"name":"CAPRALBA","start":"1868-08-14","end":"9999-12-31"}],"B754
":[{"name":"CARDANO AL CAMPO","start":"1864-04-01","end":"1927-01-11"},{"nam
e":"CARDANO AL CAMPO","start":"1927-01-12","end":"9999-12-31"}],"B989":[{"st
art":"1869-03-11","end":"1909-09-30","name":"CASOREZZO"},{"start":"1909-10-0
1","end":"9999-12-31","name":"CASOREZZO"}],"B112":[{"end":"1927-11-20","star
t":"1861-03-17","name":"BRACCA"},{"name":"BRACCA","end":"9999-12-31","start"
:"1948-04-22"}],"A054":[{"name":"ACUTO","end":"1927-01-11","start":"1871-01-
15"},{"start":"1927-01-12","end":"9999-12-31","name":"ACUTO"}],"C169":[{"end
":"9999-12-31","start":"1861-03-17","name":"CASTELLI"}],"I547":[{"name":"SCO
PPO","end":"1923-02-07","start":"1921-01-05"},{"start":"1923-02-08","end":"1
927-08-19","name":"SCOPPO"}],"G483":[{"end":"9999-12-31","start":"1868-01-01
","name":"PESCAROLO ED UNITI"}],"M066":[{"end":"1923-02-07","start":"1921-01
-05","name":"VIPACCO"},{"end":"1927-01-11","start":"1923-02-08","name":"VIPA
CCO"},{"name":"VIPACCO","start":"1927-01-12","end":"1928-04-28"},{"name":"VI
PACCO","start":"1928-04-29","end":"1947-09-15"}],"M271":[{"name":"MAZZARRONE
","end":"9999-12-31","start":"1976-05-27"}],"L351":[{"name":"TREBECCO","end"
:"1923-07-22","start":"1861-03-17"},{"name":"TREBECCO","end":"1929-02-13","s
tart":"1923-07-23"}],"B647":[{"end":"9999-12-31","start":"1861-03-17","name"
:"CAPANNOLI"}],"D807":[{"name":"FRONTINO","end":"1928-09-24","start":"1861-0
3-17"},{"end":"1958-03-28","start":"1947-01-19","name":"FRONTINO"},{"start":
"1958-03-29","end":"1998-09-03","name":"FRONTINO"},{"end":"9999-12-31","star
t":"1998-09-04","name":"FRONTINO"}],"A619":[{"name":"BARASSO","start":"1861-
03-17","end":"1927-01-11"},{"name":"BARASSO","end":"1927-07-26","start":"192
7-01-12"},{"start":"1957-08-13","end":"9999-12-31","name":"BARASSO"}],"F123"
:[{"end":"1923-12-21","start":"1861-03-17","name":"MENDATICA"},{"end":"1928-
07-17","start":"1925-08-22","name":"MENDATICA"},{"name":"MENDATICA","end":"9
999-12-31","start":"1950-02-17"}],"A916":[{"start":"1920-10-16","end":"1923-
02-13","name":"BOCENAGO"},{"name":"BOCENAGO","start":"1923-02-14","end":"192
8-02-15"},{"end":"9999-12-31","start":"1947-01-15","name":"BOCENAGO"}],"I297
":[{"name":"SANT'ANTONIO D'ADDA","end":"1927-09-07","start":"1863-07-13"}],"
I346":[{"start":"1878-06-21","end":"9999-12-31","name":"SANT'OLCESE"}],"B225
":[{"end":"9999-12-31","start":"1957-07-21","name":"BRUSASCO"},{"name":"BRUS
SASCO","end":"1928-04-02","start":"1861-03-17"}],"A514":[{"name":"AVETRANA",
"start":"1861-03-17","end":"1923-12-21"},{"name":"AVETRANA","start":"1923-12
-22","end":"9999-12-31"}],"C230":[{"name":"CASTELNUOVO CALCEA","end":"1935-0
4-07","start":"1861-03-17"},{"name":"CASTELNUOVO CALCEA","start":"1935-04-08
","end":"9999-12-31"}],"I471":[{"start":"1862-10-24","end":"1963-07-30","nam
e":"SAVIGNANO DI PUGLIA"},{"end":"9999-12-31","start":"1963-07-31","name":"S
AVIGNANO IRPINO"}],"E388":[{"name":"JESI","start":"1865-12-11","end":"9999-1
2-31"}],"B712":[{"name":"CAPRIVA","start":"1921-01-05","end":"1923-02-07"},{
"end":"1923-04-26","start":"1923-02-08","name":"CAPRIVA"},{"start":"1954-09-
03","end":"1955-08-18","name":"CAPRIVA DEL FRIULI"},{"end":"9999-12-31","sta
rt":"1955-08-19","name":"CAPRIVA DEL FRIULI"},{"end":"1927-01-11","start":"1
923-04-27","name":"CAPRIVA DI CORMONS"},{"end":"1927-12-31","start":"1927-01
-12","name":"CAPRIVA DI CORMONS"},{"end":"1947-09-15","start":"1928-01-01","
name":"CAPRIVA DI CORMONS"},{"start":"1947-09-16","end":"1951-02-14","name":
"CAPRIVA DI CORMONS"},{"start":"1951-02-15","end":"1954-09-02","name":"CAPRI
VA DI CORMONS"}],"C769":[{"name":"CIVITANOVA DEL SANNIO","end":"1970-03-02",
"start":"1864-03-11"},{"name":"CIVITANOVA DEL SANNIO","start":"1970-03-03","
end":"9999-12-31"}],"E841":[{"end":"1927-11-23","start":"1884-11-06","name":
"MAIRANO"},{"start":"1927-11-24","end":"1947-05-11","name":"MAIRANO"},{"star
t":"1947-05-12","end":"1947-11-25","name":"MAIRANO"},{"end":"9999-12-31","st
art":"1947-11-26","name":"MAIRANO"}],"M133":[{"start":"1861-03-17","end":"99
99-12-31","name":"VOLVERA"}],"I240":[{"end":"9999-12-31","start":"1861-03-17
","name":"SANTA MARIA DI LICODIA"}],"D319":[{"end":"9999-12-31","start":"186
1-03-17","name":"DOLCEDO"}],"D013":[{"start":"1871-05-01","end":"1936-07-07"
,"name":"CORMANO"},{"name":"CORMANO","start":"1936-07-08","end":"9999-12-31"
}],"C522":[{"end":"1966-09-22","start":"1928-05-22","name":"CERNUSCO MONTEVE
CCHIA"}],"H417":[{"name":"ROCCALBEGNA","end":"1963-04-16","start":"1861-03-1
7"},{"name":"ROCCALBEGNA","start":"1963-04-17","end":"9999-12-31"}],"C336":[
{"end":"1984-04-20","start":"1861-03-17","name":"CASTRIGNANO DEL CAPO"},{"na
me":"CASTRIGNANO DEL CAPO","end":"9999-12-31","start":"1984-04-21"}],"F723":
[{"name":"MORETTA","end":"1928-05-01","start":"1861-03-17"},{"name":"MORETTA
","start":"1928-05-02","end":"1946-11-11"},{"name":"MORETTA","end":"9999-12-
31","start":"1946-11-12"}],"I590":[{"end":"9999-12-31","start":"1957-01-18",
"name":"SELLIA MARINA"}],"C476":[{"start":"1861-03-17","end":"1904-08-15","n
ame":"CEPPALONI"},{"name":"CEPPALONI","end":"1918-01-17","start":"1904-08-16
"},{"name":"CEPPALONI","start":"1918-01-18","end":"9999-12-31"}],"E945":[{"s
tart":"1861-03-17","end":"9999-12-31","name":"MARGARITA"}],"H898":[{"start":
"1861-03-17","end":"9999-12-31","name":"SAN GIORGIO LA MOLARA"}],"E836":[{"s
tart":"1861-03-17","end":"1928-06-26","name":"MAIERATO"},{"start":"1928-06-2
7","end":"1992-04-15","name":"MAIERATO"},{"end":"9999-12-31","start":"1992-0
4-16","name":"MAIERATO"}],"C537":[{"name":"CERRO MAGGIORE","end":"1928-10-06
","start":"1862-11-07"},{"name":"CERRO MAGGIORE","end":"9999-12-31","start":
"1928-10-07"}],"C888":[{"end":"1927-01-11","start":"1861-03-17","name":"COLO
BRARO"},{"name":"COLOBRARO","end":"9999-12-31","start":"1927-01-12"}],"F492"
:[{"name":"MONTEFALCO","start":"1861-03-17","end":"9999-12-31"}],"B690":[{"e
nd":"9999-12-31","start":"1861-03-17","name":"CAPRARICA DI LECCE"}],"C341":[
{"name":"CASTROFILIPPO","start":"1861-03-17","end":"9999-12-31"}],"D553":[{"
start":"1864-03-11","end":"1940-03-04","name":"FERRERA CENISIO"},{"name":"MO
NCENISIO","end":"1947-09-15","start":"1940-03-05"},{"name":"MONCENISIO","sta
rt":"1947-09-16","end":"9999-12-31"}],"L139":[{"end":"9999-12-31","start":"1
861-03-17","name":"TERRUGGIA"}],"E548":[{"end":"9999-12-31","start":"1897-01
-01","name":"TERENZO"}],"H241":[{"start":"1923-04-27","end":"1927-01-11","na
me":"RESIA"},{"start":"1927-01-12","end":"1928-06-15","name":"RESIA"},{"name
":"RESIA ALL'ADIGE","start":"1920-10-16","end":"1923-02-13"},{"name":"RESIA 
ALL'ADIGE","end":"1923-04-26","start":"1923-02-14"}],"I159":[{"name":"SAN SI
GISMONDO","start":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-14","en
d":"1927-01-11","name":"SAN SIGISMONDO"},{"start":"1927-01-12","end":"1929-0
3-24","name":"SAN SIGISMONDO"}],"G191":[{"name":"OTTANA","end":"1927-01-11",
"start":"1861-03-17"},{"end":"9999-12-31","start":"1927-01-12","name":"OTTAN
A"}],"A552":[{"start":"1864-06-23","end":"9999-12-31","name":"BAGNARA CALABR
A"}],"B458":[{"start":"1861-03-17","end":"1927-09-20","name":"CAMARDA"}],"C1
75":[{"name":"CASTELLINO DEL BIFERNO","end":"1970-03-02","start":"1863-02-06
"},{"name":"CASTELLINO DEL BIFERNO","start":"1970-03-03","end":"1979-11-30"}
,{"start":"1979-12-01","end":"9999-12-31","name":"CASTELLINO DEL BIFERNO"}],
"H591":[{"name":"ROTONDELLA","start":"1861-03-17","end":"1927-01-11"},{"name
":"ROTONDELLA","start":"1927-01-12","end":"9999-12-31"}],"E298":[{"start":"1
879-02-10","end":"1928-05-25","name":"INDOVERO"}],"H236":[{"name":"RENON","s
tart":"1920-10-16","end":"1923-02-13"},{"name":"RENON","end":"1927-01-11","s
tart":"1923-02-14"},{"end":"1928-06-14","start":"1927-01-12","name":"RENON"}
,{"name":"RENON","end":"9999-12-31","start":"1928-06-15"}],"L260":[{"end":"1
929-03-20","start":"1871-06-01","name":"TORRE DEL MANGANO"}],"C581":[{"name"
:"CESSANITI","start":"1861-03-17","end":"1992-04-15"},{"name":"CESSANITI","s
tart":"1992-04-16","end":"9999-12-31"}],"G315":[{"name":"PANTELLERIA","start
":"1861-03-17","end":"9999-12-31"}],"B154":[{"start":"1924-03-24","end":"201
4-03-07","name":"BRENZONE"},{"name":"BRENZONE SUL GARDA","end":"9999-12-31",
"start":"2014-03-08"},{"name":"CASTELLETTO DI BRENZONE","end":"1924-03-23","
start":"1870-10-17"}],"A012":[{"name":"ABETONE","start":"1936-07-26","end":"
2016-12-31"}],"I631":[{"start":"1861-03-17","end":"9999-12-31","name":"SERLE
"}],"I100":[{"end":"1923-02-07","start":"1921-01-05","name":"SAN PIETRO DEL 
CARSO"},{"name":"SAN PIETRO DEL CARSO","end":"1947-09-15","start":"1923-02-0
8"}],"F781":[{"name":"MOTTA SANT'ANASTASIA","end":"9999-12-31","start":"1861
-03-17"}],"B296":[{"end":"1975-01-28","start":"1866-11-19","name":"BUSSOLENG
O"},{"end":"9999-12-31","start":"1975-01-29","name":"BUSSOLENGO"}],"L651":[{
"start":"1928-01-26","end":"1929-04-12","name":"VALSTRONA"},{"name":"VALSTRO
NA","start":"1929-04-13","end":"1955-06-08"},{"name":"VALSTRONA","start":"19
55-06-09","end":"1992-05-22"},{"start":"1992-05-23","end":"9999-12-31","name
":"VALSTRONA"}],"B347":[{"name":"CADREZZATE","start":"1861-03-17","end":"192
7-01-11"},{"end":"2019-02-14","start":"1927-01-12","name":"CADREZZATE"}],"A0
13":[{"end":"9999-12-31","start":"1861-03-17","name":"ABRIOLA"}],"B531":[{"s
tart":"1867-08-26","end":"9999-12-31","name":"CAMPODORO"}],"A319":[{"end":"9
999-12-31","start":"1863-07-13","name":"ANZANO DEL PARCO"}],"I949":[{"name":
"STENICO","start":"1920-10-16","end":"1923-02-13"},{"name":"STENICO","end":"
1928-03-12","start":"1923-02-14"},{"name":"STENICO","start":"1928-03-13","en
d":"9999-12-31"}],"I646":[{"start":"1890-08-01","end":"9999-12-31","name":"S
ERRALUNGA D'ALBA"}],"C098":[{"start":"1864-03-11","end":"1923-03-18","name":
"CASTEL DI TORA"},{"name":"CASTEL DI TORA","start":"1923-03-19","end":"1927-
01-11"},{"name":"CASTEL DI TORA","start":"1927-01-12","end":"1948-03-20"},{"
name":"CASTEL DI TORA","start":"1948-03-21","end":"9999-12-31"}],"E456":[{"s
tart":"1861-03-17","end":"1970-03-02","name":"LARINO"},{"start":"1970-03-03"
,"end":"9999-12-31","name":"LARINO"}],"G808":[{"name":"POMAROLO","end":"1923
-02-13","start":"1920-10-16"},{"start":"1923-02-14","end":"1967-08-29","name
":"POMAROLO"},{"name":"POMAROLO","start":"1967-08-30","end":"9999-12-31"}],"
A553":[{"start":"1867-09-02","end":"1940-12-19","name":"BAGNARIA ARSA"},{"na
me":"BAGNARIA ARSA","end":"1968-04-05","start":"1940-12-20"},{"start":"1968-
04-06","end":"9999-12-31","name":"BAGNARIA ARSA"}],"H088":[{"start":"1861-03
-17","end":"1927-09-14","name":"PULA"},{"name":"PULA","end":"1948-05-06","st
art":"1927-09-15"},{"end":"1974-08-19","start":"1948-05-07","name":"PULA"},{
"end":"9999-12-31","start":"1974-08-20","name":"PULA"}],"B546":[{"name":"CAM
POLONGO MAGGIORE","end":"9999-12-31","start":"1867-08-05"}],"H630":[{"name":
"RUDIANO","start":"1861-03-17","end":"9999-12-31"}],"I687":[{"start":"1920-1
0-16","end":"1923-02-13","name":"SESTO"},{"start":"1923-02-14","end":"1927-0
1-11","name":"SESTO"},{"end":"9999-12-31","start":"1927-01-12","name":"SESTO
"}],"H101":[{"end":"1928-12-05","start":"1861-03-17","name":"QUALIANO"},{"na
me":"QUALIANO","end":"9999-12-31","start":"1928-12-06"}],"M127":[{"name":"VO
LTIDO","start":"1869-01-28","end":"9999-12-31"}],"F493":[{"start":"1870-09-3
0","end":"1919-08-03","name":"MONTEFALCONE APPENNINO"},{"start":"1919-08-04"
,"end":"2009-07-11","name":"MONTEFALCONE APPENNINO"},{"name":"MONTEFALCONE A
PPENNINO","end":"9999-12-31","start":"2009-07-12"}],"B587":[{"name":"CANDIA 
LOMELLINA","end":"1928-11-28","start":"1863-03-30"},{"name":"CANDIA LOMELLIN
A","start":"1928-11-29","end":"9999-12-31"}],"B838":[{"end":"1923-12-21","st
art":"1861-03-17","name":"CARRO"},{"name":"CARRO","start":"1923-12-22","end"
:"9999-12-31"}],"L666":[{"name":"VANZONE CON SAN CARLO","start":"1875-05-21"
,"end":"1992-05-22"},{"name":"VANZONE CON SAN CARLO","start":"1992-05-23","e
nd":"9999-12-31"}],"D552":[{"name":"FERRERA ERBOGNONE","end":"9999-12-31","s
tart":"1863-03-30"}],"F737":[{"end":"9999-12-31","start":"1861-03-17","name"
:"MORNESE"}],"F664":[{"name":"MONTE VIDON COMBATTE","start":"1861-03-17","en
d":"2009-07-11"},{"end":"9999-12-31","start":"2009-07-12","name":"MONTE VIDO
N COMBATTE"}],"E155":[{"name":"GRAVINA IN PUGLIA","start":"1863-01-19","end"
:"1957-05-14"},{"name":"GRAVINA IN PUGLIA","start":"1957-05-15","end":"9999-
12-31"}],"L969":[{"start":"1870-07-15","end":"9999-12-31","name":"VILLA MINO
ZZO"}],"E461":[{"name":"LASINO","end":"1923-02-13","start":"1920-10-16"},{"s
tart":"1923-02-14","end":"1928-07-28","name":"LASINO"},{"start":"1953-09-12"
,"end":"1978-07-12","name":"LASINO"},{"name":"LASINO","end":"2015-12-31","st
art":"1978-07-13"}],"B713":[{"name":"CAPRIVA","start":"1921-01-05","end":"19
23-02-07"},{"end":"1923-04-26","start":"1923-02-08","name":"CAPRIVA"},{"star
t":"1923-04-27","end":"1927-08-19","name":"CAPRIVA NEL CARSO"}],"M132":[{"en
d":"9999-12-31","start":"1861-03-17","name":"VOLTURINO"}],"F569":[{"name":"M
ONTEMITRO","end":"1970-03-02","start":"1902-01-22"},{"name":"MONTEMITRO","st
art":"1970-03-03","end":"9999-12-31"}],"G264":[{"name":"PALAZZOLO SULL'OGLIO
","start":"1862-09-08","end":"1962-05-25"},{"name":"PALAZZOLO SULL'OGLIO","s
tart":"1962-05-26","end":"9999-12-31"}],"C523":[{"start":"1867-01-14","end":
"9999-12-31","name":"CERNUSCO SUL NAVIGLIO"}],"D012":[{"start":"1939-10-21",
"end":"1946-01-29","name":"CORMAIORE"},{"end":"1927-01-11","start":"1861-03-
17","name":"COURMAYEUR"},{"end":"1939-10-20","start":"1927-01-12","name":"CO
URMAYEUR"},{"end":"9999-12-31","start":"1946-01-30","name":"COURMAYEUR"}],"B
386":[{"end":"1923-02-13","start":"1920-10-16","name":"CALAVINO"},{"name":"C
ALAVINO","start":"1923-02-14","end":"1928-07-28"},{"start":"1953-09-12","end
":"1978-07-12","name":"CALAVINO"},{"name":"CALAVINO","start":"1978-07-13","e
nd":"2015-12-31"}],"F722":[{"end":"2009-07-11","start":"1910-07-26","name":"
MORESCO"},{"name":"MORESCO","start":"2009-07-12","end":"9999-12-31"}],"D274"
:[{"name":"DERCOLO","end":"1923-02-13","start":"1920-10-16"},{"end":"1928-10
-29","start":"1923-02-14","name":"DERCOLO"}],"H824":[{"name":"SAN DONATO VAL
 DI COMINO","start":"1862-12-29","end":"1927-01-11"},{"name":"SAN DONATO VAL
 DI COMINO","end":"1948-05-05","start":"1927-01-12"},{"end":"9999-12-31","st
art":"1948-05-06","name":"SAN DONATO VAL DI COMINO"}],"H701":[{"end":"1992-0
4-15","start":"1863-04-13","name":"SALERANO SUL LAMBRO"},{"name":"SALERANO S
UL LAMBRO","start":"1992-04-16","end":"9999-12-31"}],"B280":[{"name":"BURONZ
O","start":"1861-03-17","end":"1927-01-11"},{"start":"1927-01-12","end":"999
9-12-31","name":"BURONZO"}],"F215":[{"end":"1927-12-08","start":"1861-03-17"
,"name":"MILZANELLO"}],"F137":[{"name":"MERCATINO MARECCHIA","start":"1907-0
4-20","end":"1928-08-15"},{"end":"1941-10-23","start":"1928-08-16","name":"M
ERCATINO MARECCHIA"},{"name":"NOVAFELTRIA","start":"1941-10-24","end":"1946-
04-16"},{"name":"NOVAFELTRIA","start":"1946-04-17","end":"1947-02-04"},{"nam
e":"NOVAFELTRIA","start":"1947-02-05","end":"1998-09-03"},{"name":"NOVAFELTR
IA","start":"1998-09-04","end":"2009-08-14"},{"name":"NOVAFELTRIA","start":"
2009-08-15","end":"9999-12-31"}],"F978":[{"name":"NUMANA","end":"1928-05-01"
,"start":"1870-01-02"},{"start":"1928-05-02","end":"1946-11-01","name":"NUMA
NA"},{"name":"NUMANA","end":"9999-12-31","start":"1946-11-02"}],"Z338":[{"st
art":"0000-01-01","end":"9999-12-31","name":"Ruanda"}],"E755":[{"name":"LUSC
IANO E DUCENTA","end":"1927-01-11","start":"1861-03-17"},{"start":"1927-01-1
2","end":"1929-04-12","name":"LUSCIANO E DUCENTA"}],"F008":[{"name":"MASCIAG
O MILANESE","end":"1929-03-02","start":"1862-07-28"}],"B113":[{"name":"BRACC
A DI COSTA SERINA","start":"1928-03-24","end":"1948-04-20"}],"L578":[{"start
":"1861-03-17","end":"1927-12-30","name":"VALGIOIE"},{"start":"1957-08-20","
end":"9999-12-31","name":"VALGIOIE"}],"M305":[{"end":"1948-03-25","start":"1
927-07-31","name":"ALM\u00c8 CON VILLA"},{"name":"ALM\u00c8 CON VILLA","end"
:"1951-09-10","start":"1948-03-26"}],"G482":[{"name":"PESCARA","start":"1879
-05-12","end":"1927-01-11"},{"name":"PESCARA","start":"1927-01-12","end":"19
28-03-23"},{"name":"PESCARA","start":"1928-03-24","end":"1928-12-04"},{"star
t":"1928-12-05","end":"1947-08-29","name":"PESCARA"},{"start":"1947-08-30","
end":"2006-07-12","name":"PESCARA"},{"name":"PESCARA","end":"9999-12-31","st
art":"2006-07-13"}],"F122":[{"end":"1929-05-27","start":"1861-03-17","name":
"MENCONICO"},{"name":"MENCONICO","start":"1929-05-28","end":"9999-12-31"}],"
G615":[{"end":"9999-12-31","start":"1893-10-26","name":"PIETRAFITTA"}],"E030
":[{"name":"GILDONE","end":"1970-03-02","start":"1861-03-17"},{"end":"9999-1
2-31","start":"1970-03-03","name":"GILDONE"}],"L975":[{"start":"1863-04-23",
"end":"1927-01-11","name":"VILLANOVA D'ALBENGA"},{"start":"1927-01-12","end"
:"9999-12-31","name":"VILLANOVA D'ALBENGA"}],"E688":[{"start":"1861-03-17","
end":"2013-12-31","name":"LORENZANA"}],"D054":[{"name":"CORTE BRUGNATELLA","
end":"1923-07-22","start":"1861-03-17"},{"end":"1952-10-17","start":"1923-07
-23","name":"CORTE BRUGNATELLA"},{"start":"1952-10-18","end":"9999-12-31","n
ame":"CORTE BRUGNATELLA"}],"E224":[{"name":"GRUMO NEVANO","start":"1864-02-0
2","end":"1934-12-14"},{"end":"9999-12-31","start":"1934-12-15","name":"GRUM
O NEVANO"}],"L218":[{"start":"1863-02-06","end":"9999-12-31","name":"TORINO 
DI SANGRO"}],"I700":[{"name":"SETTIMO MILANESE","start":"1862-11-03","end":"
9999-12-31"}],"D619":[{"name":"FIUMARA","end":"1927-08-02","start":"1861-03-
17"},{"start":"1948-05-08","end":"9999-12-31","name":"FIUMARA"}],"A807":[{"e
nd":"1928-03-12","start":"1861-03-17","name":"BERSEZIO"}],"F181":[{"name":"M
EZZEGRA","start":"1861-03-17","end":"1929-02-07"},{"name":"MEZZEGRA","start"
:"1947-10-03","end":"2014-02-03"}],"D916":[{"name":"GARDOLO AL PIANO","start
":"1920-10-16","end":"1923-02-13"},{"name":"GARDOLO AL PIANO","end":"1926-09
-30","start":"1923-02-14"}],"G421":[{"end":"1928-08-12","start":"1861-03-17"
,"name":"PELLA"},{"start":"1947-01-15","end":"1960-07-04","name":"PELLA"},{"
start":"1960-07-05","end":"9999-12-31","name":"PELLA"}],"C636":[{"end":"1927
-03-01","start":"1862-12-29","name":"CHIGNOLO VERBANO"}],"C939":[{"end":"192
7-01-11","start":"1862-11-24","name":"CONCA DELLA CAMPANIA"},{"start":"1927-
01-12","end":"1945-08-31","name":"CONCA DELLA CAMPANIA"},{"name":"CONCA DELL
A CAMPANIA","start":"1945-09-01","end":"9999-12-31"}],"D514":[{"name":"FAVAR
A","start":"1861-03-17","end":"9999-12-31"}],"L871":[{"start":"1869-03-01","
end":"1893-10-06","name":"QUINTOSOLE"},{"name":"VIGENTINO","end":"1923-10-13
","start":"1893-10-07"}],"B390":[{"name":"CALCI","start":"1884-09-16","end":
"9999-12-31"}],"L005":[{"name":"SUNA","end":"1927-11-01","start":"1861-03-17
"}],"C641":[{"name":"CHIOPRIS-VISCONE","start":"1921-01-05","end":"1923-02-0
7"},{"name":"CHIOPRIS-VISCONE","start":"1923-02-08","end":"1968-04-05"},{"st
art":"1968-04-06","end":"2011-10-19","name":"CHIOPRIS-VISCONE"},{"end":"9999
-12-31","start":"2011-10-20","name":"CHIOPRIS-VISCONE"}],"E149":[{"name":"GR
ATTERI","end":"1927-10-20","start":"1861-03-17"},{"name":"GRATTERI","start":
"1927-10-21","end":"9999-12-31"}],"L538":[{"name":"VAIE","end":"9999-12-31",
"start":"1937-06-16"},{"name":"VAYES","end":"1937-06-15","start":"1861-03-17
"}],"L887":[{"end":"9999-12-31","start":"1861-03-17","name":"VIGNOLE BORBERA
"}],"I205":[{"end":"1927-11-08","start":"1861-03-17","name":"SANTA GIUSTA"},
{"start":"1947-06-07","end":"1974-08-19","name":"SANTA GIUSTA"},{"name":"SAN
TA GIUSTA","end":"9999-12-31","start":"1974-08-20"}],"B658":[{"end":"9999-12
-31","start":"1861-03-17","name":"CAPITIGNANO"}],"D991":[{"name":"GEROSA","e
nd":"2014-02-03","start":"1861-03-17"}],"I558":[{"name":"SECINARO","end":"99
99-12-31","start":"1861-03-17"}],"I866":[{"name":"SOSPIROLO","end":"9999-12-
31","start":"1866-11-19"}],"E070":[{"name":"SESTA GODANO","start":"1875-11-1
0","end":"1923-12-21"},{"end":"9999-12-31","start":"1923-12-22","name":"SEST
A GODANO"}],"H850":[{"end":"1969-12-13","start":"1861-03-17","name":"SAN FRA
TELLO"},{"end":"9999-12-31","start":"1969-12-14","name":"SAN FRATELLO"}],"D2
34":[{"start":"1948-12-25","end":"9999-12-31","name":"CUSTONACI"}],"M405":[{
"start":"2019-01-01","end":"9999-12-31","name":"VAL DI CHY"}],"A085":[{"name
":"AGRA","end":"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end
":"1928-01-06","name":"AGRA"},{"name":"AGRA","start":"1928-01-07","end":"999
9-12-31"}],"G382":[{"name":"PAULI ARBAREI","end":"1927-08-06","start":"1861-
03-17"},{"name":"PAULI ARBAREI","end":"1974-08-19","start":"1951-01-12"},{"n
ame":"PAULI ARBAREI","start":"1974-08-20","end":"2005-12-31"},{"name":"PAULI
 ARBAREI","end":"2016-04-27","start":"2006-01-01"},{"end":"9999-12-31","star
t":"2016-04-28","name":"PAULI ARBAREI"}],"Z134":[{"name":"Ungheria","start":
"0000-01-01","end":"9999-12-31"}],"E900":[{"name":"MANZIANA","start":"1871-0
1-15","end":"9999-12-31"}],"F177":[{"end":"1947-03-30","start":"1928-01-28",
"name":"MEZZANO IMER"}],"D895":[{"start":"1917-06-11","end":"1977-05-11","na
me":"GAMBASSI"},{"start":"1977-05-12","end":"9999-12-31","name":"GAMBASSI TE
RME"}],"I018":[{"end":"1923-12-21","start":"1864-04-01","name":"SAN MARZANO 
DI SAN GIUSEPPE"},{"name":"SAN MARZANO DI SAN GIUSEPPE","start":"1923-12-22"
,"end":"9999-12-31"}],"H203":[{"end":"1928-06-29","start":"1861-03-17","name
":"RE"},{"name":"RE","start":"1928-06-30","end":"1957-01-24"},{"name":"RE","
end":"1992-05-22","start":"1957-01-25"},{"name":"RE","start":"1992-05-23","e
nd":"9999-12-31"}],"G292":[{"name":"PALOMONTE","end":"9999-12-31","start":"1
862-11-07"}],"L831":[{"start":"1861-03-17","end":"9999-12-31","name":"VIANO"
}],"H953":[{"start":"1861-03-17","end":"1928-04-03","name":"SAN LEUCIO"},{"n
ame":"SAN LEUCIO DEL SANNIO","end":"9999-12-31","start":"1928-04-04"}],"C303
":[{"name":"CASTIGLIONE DI GARFAGNANA","end":"1923-11-23","start":"1863-03-3
0"},{"start":"1923-11-24","end":"9999-12-31","name":"CASTIGLIONE DI GARFAGNA
NA"}],"B785":[{"name":"CARLAZZO","end":"9999-12-31","start":"1928-05-12"}],"
F710":[{"start":"1921-01-05","end":"1923-02-07","name":"MORARO"},{"name":"MO
RARO","end":"1927-01-11","start":"1923-02-08"},{"end":"1928-01-01","start":"
1927-01-12","name":"MORARO"},{"name":"MORARO","start":"1955-08-19","end":"99
99-12-31"}],"M106":[{"start":"1861-03-17","end":"1927-01-11","name":"VOCCA"}
,{"end":"1929-03-01","start":"1927-01-12","name":"VOCCA"},{"name":"VOCCA","s
tart":"1947-11-26","end":"9999-12-31"}],"F535":[{"name":"MONTELAPIANO","end"
:"1928-11-02","start":"1861-03-17"},{"name":"MONTELAPIANO","start":"1947-05-
23","end":"9999-12-31"}],"H142":[{"name":"QUINZANO VERONESE","end":"1927-02-
15","start":"1867-08-26"}],"I851":[{"end":"1927-01-11","start":"1861-03-17",
"name":"SORGONO"},{"name":"SORGONO","end":"9999-12-31","start":"1927-01-12"}
],"E250":[{"start":"1861-03-17","end":"9999-12-31","name":"GUARDISTALLO"}],"
D020":[{"name":"CORNEDO","start":"1866-11-19","end":"1928-05-02"},{"name":"C
ORNEDO VICENTINO","end":"9999-12-31","start":"1928-05-03"}],"B490":[{"name":
"CAMOGLI","end":"9999-12-31","start":"1861-03-17"}],"L935":[{"name":"VILLA D
EL NEVOSO","end":"1936-09-02","start":"1927-12-16"},{"name":"VILLA DEL NEVOS
O","end":"1947-09-15","start":"1936-09-03"}],"G542":[{"name":"PIANA CRIXIA",
"end":"1927-01-11","start":"1880-06-25"},{"name":"PIANA CRIXIA","start":"192
7-01-12","end":"1929-04-11"},{"end":"9999-12-31","start":"1929-04-12","name"
:"PIANA CRIXIA"}],"F844":[{"end":"1927-01-11","start":"1875-09-07","name":"N
ARNI"},{"name":"NARNI","start":"1927-01-12","end":"9999-12-31"}],"L846":[{"s
tart":"1861-03-17","end":"1927-01-11","name":"VICOLI"},{"name":"VICOLI","sta
rt":"1927-01-12","end":"9999-12-31"}],"F692":[{"name":"MONTORIO ROMANO","sta
rt":"1871-01-15","end":"9999-12-31"}],"C676":[{"name":"CICERALE","end":"9999
-12-31","start":"1861-03-17"}],"C979":[{"name":"COPIANO","end":"2002-10-25",
"start":"1861-03-17"},{"name":"COPIANO","end":"9999-12-31","start":"2002-10-
26"}],"E803":[{"name":"MAGGIORA","start":"1861-03-17","end":"9999-12-31"}],"
I955":[{"start":"1861-03-17","end":"9999-12-31","name":"STIGNANO"}],"C568":[
{"start":"1861-03-17","end":"1929-01-23","name":"CESAR\u00d2"},{"start":"192
9-01-24","end":"1940-08-28","name":"CESAR\u00d2"},{"name":"CESAR\u00d2","sta
rt":"1940-08-29","end":"9999-12-31"}],"C856":[{"name":"COLLEDIMEZZO","end":"
1928-09-24","start":"1861-03-17"},{"name":"COLLEDIMEZZO","end":"1933-05-18",
"start":"1928-09-25"},{"name":"COLLEDIMEZZO","end":"1963-03-30","start":"193
3-05-19"},{"start":"1963-03-31","end":"9999-12-31","name":"COLLEDIMEZZO"}],"
A419":[{"name":"ARMUNGIA","start":"1861-03-17","end":"1974-08-19"},{"name":"
ARMUNGIA","end":"2016-04-27","start":"1974-08-20"},{"name":"ARMUNGIA","end":
"9999-12-31","start":"2016-04-28"}],"G683":[{"start":"1862-12-19","end":"192
8-06-13","name":"PIOBESI D'ALBA"},{"name":"PIOBESI D'ALBA","end":"9999-12-31
","start":"1946-11-12"}],"B447":[{"end":"9999-12-31","start":"1861-03-17","n
ame":"CALVIGNANO"}],"B185":[{"name":"BRIONE","start":"1920-10-16","end":"192
3-02-13"},{"name":"BRIONE","start":"1923-02-14","end":"1928-03-12"},{"name":
"BRIONE","end":"2015-12-31","start":"1946-12-24"}],"F110":[{"name":"MELITO I
RPINO","end":"9999-12-31","start":"1923-06-01"},{"name":"MELITO VALLE BONITO
","end":"1923-05-31","start":"1862-11-10"}],"H742":[{"name":"SAMBIASE","end"
:"1935-06-05","start":"1861-03-17"},{"name":"SAMBIASE","end":"1968-11-27","s
tart":"1935-06-06"}],"E557":[{"name":"LETTERE","end":"1925-07-09","start":"1
861-03-17"},{"name":"LETTERE","start":"1925-07-10","end":"9999-12-31"}],"F28
3":[{"start":"1861-03-17","end":"1923-11-23","name":"MOLAZZANA"},{"name":"MO
LAZZANA","end":"9999-12-31","start":"1923-11-24"}],"I671":[{"name":"SERSALE"
,"end":"1929-07-02","start":"1861-03-17"},{"start":"1929-07-03","end":"9999-
12-31","name":"SERSALE"}],"B571":[{"start":"1861-03-17","end":"1938-07-08","
name":"CAMPOVICO"}],"E017":[{"end":"1924-01-03","start":"1861-03-17","name":
"GIARRE"},{"start":"1924-01-04","end":"1934-04-06","name":"GIARRE"},{"start"
:"1934-04-07","end":"1939-12-28","name":"GIARRE"},{"start":"1945-10-25","end
":"9999-12-31","name":"GIARRE"}],"E356":[{"start":"1861-03-17","end":"9999-1
2-31","name":"ISOLA DOVARESE"}],"E749":[{"start":"1883-09-26","end":"1967-03
-09","name":"LURAGO D'ERBA"},{"end":"9999-12-31","start":"1967-03-10","name"
:"LURAGO D'ERBA"}],"M172":[{"name":"ZEVIO","start":"1866-11-19","end":"1928-
04-06"},{"end":"1954-05-19","start":"1928-04-07","name":"ZEVIO"},{"name":"ZE
VIO","start":"1954-05-20","end":"9999-12-31"}],"A826":[{"name":"BESOZZO","en
d":"1927-01-11","start":"1861-03-17"},{"name":"BESOZZO","end":"1928-01-21","
start":"1927-01-12"},{"end":"1929-04-11","start":"1928-01-22","name":"BESOZZ
O"},{"name":"BESOZZO","end":"9999-12-31","start":"1929-04-12"}],"F393":[{"st
art":"1861-03-17","end":"1925-04-16","name":"MONTAGNA"},{"name":"MONTAGNA","
end":"1928-06-01","start":"1925-04-17"},{"end":"9999-12-31","start":"1928-06
-02","name":"MONTAGNA IN VALTELLINA"}],"C861":[{"end":"1923-02-13","start":"
1920-10-16","name":"COLLE IN CASIES"},{"name":"COLLE IN CASIES","end":"1927-
01-11","start":"1923-02-14"},{"name":"COLLE IN CASIES","end":"1929-05-15","s
tart":"1927-01-12"}],"B211":[{"end":"1927-12-23","start":"1861-03-17","name"
:"BROZZO"}],"B486":[{"name":"CAMMARATA","start":"1861-03-17","end":"9999-12-
31"}],"C602":[{"start":"1861-03-17","end":"1928-04-30","name":"CHIABRANO"}],
"H670":[{"end":"1927-01-11","start":"1861-03-17","name":"SAINT DENIS"},{"sta
rt":"1927-01-12","end":"1928-06-14","name":"SAINT DENIS"},{"end":"1977-01-01
","start":"1946-05-15","name":"SAINT DENIS"},{"start":"1977-01-02","end":"99
99-12-31","name":"SAINT-DENIS"}],"C965":[{"name":"CONSIGLIO DI RUMO","end":"
1928-12-11","start":"1861-03-17"},{"start":"1948-03-21","end":"2011-02-10","
name":"CONSIGLIO DI RUMO"}],"Z734":[{"end":"9999-12-31","start":"0000-01-01"
,"name":"Palau"}],"F777":[{"start":"1861-03-17","end":"9999-12-31","name":"M
OTTA MONTECORVINO"}],"G044":[{"start":"1861-03-17","end":"1927-01-11","name"
:"OLLOLAI"},{"name":"OLLOLAI","start":"1927-01-12","end":"9999-12-31"}],"E36
1":[{"name":"ISOLELLA","end":"1927-01-11","start":"1861-03-17"},{"start":"19
27-01-12","end":"1928-09-05","name":"ISOLELLA"}],"B878":[{"start":"1899-07-2
6","end":"9999-12-31","name":"CASALE MARITTIMO"}],"G065":[{"end":"1927-01-11
","start":"1871-01-15","name":"ONANO"},{"end":"9999-12-31","start":"1927-01-
12","name":"ONANO"}],"D428":[{"name":"ERVE","start":"1861-03-17","end":"1992
-04-15"},{"name":"ERVE","end":"9999-12-31","start":"1992-04-16"}],"C531":[{"
name":"CERRINA MONFERRATO","start":"1861-03-17","end":"1928-08-13"},{"end":"
9999-12-31","start":"1928-08-14","name":"CERRINA MONFERRATO"}],"C347":[{"nam
e":"CASTROREALE","start":"1861-03-17","end":"1923-11-27"},{"name":"CASTROREA
LE","start":"1923-11-28","end":"1927-03-22"},{"name":"CASTROREALE","start":"
1927-03-23","end":"1947-06-16"},{"end":"1966-07-16","start":"1947-06-17","na
me":"CASTROREALE"},{"end":"9999-12-31","start":"1966-07-17","name":"CASTRORE
ALE"}],"C296":[{"name":"CASTIGLIONE DEI PEPOLI","start":"1863-04-13","end":"
9999-12-31"}],"F879":[{"start":"1939-12-12","end":"1945-06-05","name":"NETTU
NIA"}],"F272":[{"name":"MOGORO","end":"1974-08-19","start":"1861-03-17"},{"n
ame":"MOGORO","start":"1974-08-20","end":"9999-12-31"}],"C944":[{"name":"CON
CEI","end":"2009-12-31","start":"1952-09-20"}],"H286":[{"name":"RIGNANO SULL
'ARNO","start":"1863-04-13","end":"9999-12-31"}],"H411":[{"name":"ROCCAGIOVI
NE","end":"9999-12-31","start":"1871-01-15"}],"C546":[{"start":"1861-03-17",
"end":"1927-01-11","name":"CERVAROLO"},{"end":"1929-03-01","start":"1927-01-
12","name":"CERVAROLO"}],"Z224":[{"name":"Iran","start":"0000-01-01","end":"
9999-12-31"}],"D124":[{"start":"1861-03-17","end":"1923-03-18","name":"COTTA
NELLO"},{"end":"1927-01-11","start":"1923-03-19","name":"COTTANELLO"},{"name
":"COTTANELLO","end":"1928-03-12","start":"1927-01-12"},{"name":"COTTANELLO"
,"start":"1928-03-13","end":"1946-03-07"},{"name":"COTTANELLO","start":"1946
-03-08","end":"9999-12-31"}],"L189":[{"start":"1876-09-10","end":"1923-03-18
","name":"TOFFIA"},{"start":"1923-03-19","end":"1927-01-11","name":"TOFFIA"}
,{"end":"1949-07-09","start":"1927-01-12","name":"TOFFIA"},{"name":"TOFFIA",
"end":"9999-12-31","start":"1949-07-10"}],"G672":[{"start":"1861-03-17","end
":"1928-05-30","name":"PINASCA"},{"start":"1928-05-31","end":"1947-08-02","n
ame":"PINASCA"},{"start":"1947-08-03","end":"9999-12-31","name":"PINASCA"}],
"B098":[{"name":"BOVALINO","end":"9999-12-31","start":"1861-03-17"}],"E847":
[{"name":"MALBORGHETTO VALBRUNA","start":"1928-08-07","end":"1968-04-05"},{"
name":"MALBORGHETTO VALBRUNA","end":"9999-12-31","start":"1968-04-06"}],"H62
5":[{"start":"1866-11-19","end":"9999-12-31","name":"RUBANO"}],"M210":[{"nam
e":"TERME VIGLIATORE","start":"1966-07-17","end":"9999-12-31"}],"A798":[{"na
me":"BERGOLO","start":"1861-03-17","end":"1928-06-27"},{"end":"9999-12-31","
start":"1947-05-23","name":"BERGOLO"}],"L912":[{"name":"VILLA BARTOLOMEA","s
tart":"1866-11-19","end":"9999-12-31"}],"C838":[{"name":"COLI","start":"1868
-01-01","end":"1952-10-17"},{"name":"COLI","start":"1952-10-18","end":"9999-
12-31"}],"G740":[{"name":"PLOAGHE","start":"1861-03-17","end":"9999-12-31"}]
,"E478":[{"start":"1921-01-05","end":"1923-02-07","name":"LAURANA"},{"name":
"LAURANA","start":"1923-02-08","end":"1924-03-08"},{"end":"1947-09-15","star
t":"1924-03-09","name":"LAURANA"}],"C587":[{"name":"CETONA","end":"9999-12-3
1","start":"1861-03-17"}],"H597":[{"end":"1927-01-11","start":"1861-03-17","
name":"ROVATE"},{"end":"1928-06-14","start":"1927-01-12","name":"ROVATE"}],"
F052":[{"end":"1927-01-11","start":"1861-03-17","name":"MATERA"},{"end":"999
9-12-31","start":"1927-01-12","name":"MATERA"}],"F300":[{"name":"MOLO DI BOR
BERA","end":"1928-06-27","start":"1863-01-26"}],"I410":[{"start":"1861-03-17
","end":"9999-12-31","name":"SANZA"}],"G197":[{"start":"1861-03-17","end":"9
999-12-31","name":"OVADA"}],"E174":[{"end":"1925-01-17","start":"1923-04-27"
,"name":"GRIES"},{"name":"GR\u00ccES DI BOLZANO","start":"1920-10-16","end":
"1923-02-13"},{"start":"1923-02-14","end":"1923-04-26","name":"GR\u00ccES DI
 BOLZANO"}],"D069":[{"name":"CORTERANZO","end":"1928-11-30","start":"1861-03
-17"}],"B523":[{"start":"1861-03-17","end":"1927-01-11","name":"CAMPOCHIESA"
},{"name":"CAMPOCHIESA","start":"1927-01-12","end":"1929-05-27"}],"D363":[{"
end":"1927-06-09","start":"1861-03-17","name":"DRANO"}],"M183":[{"name":"ZOC
CA","end":"9999-12-31","start":"1880-05-21"}],"I623":[{"name":"SERBARIU","en
d":"1938-01-08","start":"1861-03-17"}],"G814":[{"start":"1861-03-17","end":"
1923-12-21","name":"POMPEIANA"},{"end":"1928-03-06","start":"1925-08-22","na
me":"POMPEIANA"},{"name":"POMPEIANA","end":"9999-12-31","start":"1947-03-28"
}],"F512":[{"end":"1928-01-27","start":"1861-03-17","name":"MONTEFUSCO"},{"s
tart":"1928-01-28","end":"1947-09-23","name":"MONTEFUSCO"},{"end":"9999-12-3
1","start":"1947-09-24","name":"MONTEFUSCO"}],"C713":[{"name":"CINTO EUGANEO
","start":"1867-08-26","end":"9999-12-31"}],"H247":[{"end":"9999-12-31","sta
rt":"1861-03-17","name":"REVELLO"}],"E338":[{"name":"ISOLA D'ASTI","start":"
1863-02-16","end":"1931-06-02"},{"name":"ISOLA D'ASTI","start":"1931-06-03",
"end":"1935-04-07"},{"start":"1935-04-08","end":"9999-12-31","name":"ISOLA D
'ASTI"}],"H396":[{"name":"ROCCA DE' GIORGI","start":"1861-03-17","end":"1939
-02-04"},{"end":"9999-12-31","start":"1939-02-05","name":"ROCCA DE' GIORGI"}
],"L561":[{"name":"VALDINA","end":"1929-06-12","start":"1861-03-17"},{"start
":"1948-12-25","end":"9999-12-31","name":"VALDINA"}],"A198":[{"name":"ALICE 
CASTELLO","start":"1862-12-29","end":"1927-01-11"},{"name":"ALICE CASTELLO",
"end":"9999-12-31","start":"1927-01-12"}],"E993":[{"start":"1861-03-17","end
":"9999-12-31","name":"MARTONE"}],"L204":[{"name":"TONENGO","end":"1928-11-3
0","start":"1861-03-17"},{"start":"1947-01-10","end":"2022-12-31","name":"TO
NENGO"}],"G140":[{"name":"ORTISEI","start":"1920-10-16","end":"1923-02-13"},
{"start":"1923-02-14","end":"1927-01-11","name":"ORTISEI"},{"name":"ORTISEI"
,"start":"1927-01-12","end":"1986-09-16"},{"name":"ORTISEI","end":"9999-12-3
1","start":"1986-09-17"}],"F206":[{"name":"MILAZZO","end":"9999-12-31","star
t":"1879-08-29"}],"C280":[{"start":"1861-03-17","end":"9999-12-31","name":"C
ASTELVENERE"}],"F865":[{"name":"NEMI","start":"1871-01-15","end":"1935-03-21
"},{"name":"NEMI","start":"1935-03-22","end":"9999-12-31"}],"H290":[{"name":
"RIGOSA","end":"1927-11-20","start":"1861-03-17"}],"G373":[{"start":"1920-10
-16","end":"1923-02-13","name":"PATONE"},{"start":"1923-02-14","end":"1929-0
2-13","name":"PATONE"}],"G079":[{"end":"9999-12-31","start":"1861-03-17","na
me":"OPI"}],"D508":[{"name":"FASANO","end":"1927-01-11","start":"1861-03-17"
},{"end":"1927-08-05","start":"1927-01-12","name":"FASANO"},{"name":"FASANO"
,"start":"1927-08-06","end":"1935-06-04"},{"start":"1935-06-05","end":"9999-
12-31","name":"FASANO"}],"G606":[{"start":"1861-03-17","end":"1928-06-13","n
ame":"PIETRABBONDANTE"},{"name":"PIETRABBONDANTE","start":"1928-06-14","end"
:"1946-10-11"},{"name":"PIETRABBONDANTE","start":"1946-10-12","end":"1970-03
-02"},{"end":"9999-12-31","start":"1970-03-03","name":"PIETRABBONDANTE"}],"A
663":[{"name":"BARI SARDO","end":"1927-01-11","start":"1862-09-29"},{"end":"
2005-12-31","start":"1927-01-12","name":"BARI SARDO"},{"end":"2016-04-27","s
tart":"2006-01-01","name":"BARI SARDO"},{"name":"BARI SARDO","start":"2016-0
4-28","end":"9999-12-31"}],"G797":[{"name":"POLLINA","start":"1861-03-17","e
nd":"9999-12-31"}],"E774":[{"name":"LUZZOGNO","end":"1928-01-26","start":"18
61-03-17"}],"G909":[{"name":"PORTOBUFFOL\u00c8","end":"9999-12-31","start":"
1866-11-19"}],"M057":[{"start":"1861-03-17","end":"1970-03-02","name":"VINCH
IATURO"},{"name":"VINCHIATURO","end":"9999-12-31","start":"1970-03-03"}],"G4
33":[{"name":"PENNABILLI","start":"1861-03-17","end":"1928-12-13"},{"end":"1
998-09-03","start":"1928-12-14","name":"PENNABILLI"},{"name":"PENNABILLI","s
tart":"1998-09-04","end":"2009-08-14"},{"name":"PENNABILLI","start":"2009-08
-15","end":"9999-12-31"}],"M316":[{"name":"MAPPANO","start":"2017-04-18","en
d":"9999-12-31"}],"A494":[{"name":"AUGUSTA","start":"1861-03-17","end":"9999
-12-31"}],"F961":[{"name":"NOVELLO","start":"1861-03-17","end":"1928-04-15"}
,{"start":"1947-03-15","end":"9999-12-31","name":"NOVELLO"}],"I848":[{"name"
:"SORDIO","end":"1992-04-15","start":"1861-03-17"},{"name":"SORDIO","start":
"1992-04-16","end":"9999-12-31"}],"D255":[{"name":"DAVAGNA","end":"9999-12-3
1","start":"1891-04-10"}],"C113":[{"start":"1861-03-17","end":"1910-08-27","
name":"CASTELFRANCO DI SOTTO"},{"end":"1925-11-29","start":"1910-08-28","nam
e":"CASTELFRANCO DI SOTTO"},{"name":"CASTELFRANCO DI SOTTO","start":"1925-11
-30","end":"9999-12-31"}],"H540":[{"name":"RONCO ALL'ADIGE","start":"1867-08
-26","end":"9999-12-31"}],"E025":[{"start":"1861-03-17","end":"9999-12-31","
name":"GIFFONE"}],"Z155":[{"end":"9999-12-31","start":"0000-01-01","name":"S
lovacchia"}],"I381":[{"name":"SAN VENANZO","end":"1927-01-11","start":"1861-
03-17"},{"name":"SAN VENANZO","end":"1929-02-04","start":"1927-01-12"},{"nam
e":"SAN VENANZO","end":"9999-12-31","start":"1929-02-05"}],"B939":[{"name":"
CASARZA LIGURE","end":"9999-12-31","start":"1878-06-10"}],"I477":[{"name":"S
AVOCA","end":"1929-01-23","start":"1861-03-17"},{"name":"SAVOCA","start":"19
48-06-26","end":"9999-12-31"}],"B636":[{"start":"1861-03-17","end":"1998-09-
03","name":"CANTIANO"},{"end":"9999-12-31","start":"1998-09-04","name":"CANT
IANO"}],"L556":[{"end":"1927-01-11","start":"1861-03-17","name":"VALDENGO"},
{"end":"1992-04-15","start":"1927-01-12","name":"VALDENGO"},{"end":"9999-12-
31","start":"1992-04-16","name":"VALDENGO"}],"B403":[{"name":"CALDOGNO","sta
rt":"1866-11-19","end":"1987-10-20"},{"start":"1987-10-21","end":"9999-12-31
","name":"CALDOGNO"}],"D662":[{"start":"1861-03-17","end":"1927-01-11","name
":"FONDI"},{"start":"1927-01-12","end":"1934-10-25","name":"FONDI"},{"name":
"FONDI","start":"1934-10-26","end":"1934-12-17"},{"name":"FONDI","start":"19
34-12-18","end":"1945-04-08"},{"start":"1945-04-09","end":"9999-12-31","name
":"FONDI"}],"D905":[{"end":"1927-12-26","start":"1861-03-17","name":"GANDINO
"},{"name":"GANDINO","end":"9999-12-31","start":"1927-12-27"}],"L868":[{"nam
e":"VIGARANO MAINARDA","end":"9999-12-31","start":"1902-02-01"}],"G852":[{"n
ame":"PONTENURE","end":"9999-12-31","start":"1861-03-17"}],"I322":[{"name":"
VALLEFIORITA","end":"9999-12-31","start":"1863-04-10"}],"E890":[{"name":"MAN
IGLIA","end":"1928-04-30","start":"1861-03-17"}],"I291":[{"name":"SANTA NINF
A","end":"9999-12-31","start":"1861-03-17"}],"G259":[{"start":"1861-03-17","
end":"9999-12-31","name":"PALAZZAGO"}],"I536":[{"name":"SCIDO","end":"9999-1
2-31","start":"1861-03-17"}],"L357":[{"name":"TRECCHINA","end":"9999-12-31",
"start":"1861-03-17"}],"M277":[{"end":"9999-12-31","start":"1977-12-18","nam
e":"SAN FERDINANDO"}],"B641":[{"name":"CANZO","end":"9999-12-31","start":"18
61-03-17"}],"L016":[{"end":"1924-01-13","start":"1861-03-17","name":"SUTERA"
},{"start":"1924-01-14","end":"9999-12-31","name":"SUTERA"}],"C390":[{"start
":"1890-05-02","end":"1909-07-21","name":"CAVE"},{"end":"9999-12-31","start"
:"1909-07-22","name":"CAVE"}],"F014":[{"name":"MASI DI VIGO","end":"1923-02-
13","start":"1920-10-16"},{"start":"1923-02-14","end":"1928-12-21","name":"M
ASI DI VIGO"}],"D724":[{"name":"FORNO ALPI GRAJE","end":"1928-01-06","start"
:"1873-04-25"}],"D801":[{"name":"FRIGNANO PICCOLO","end":"1927-01-11","start
":"1861-03-17"},{"start":"1927-01-12","end":"1929-04-12","name":"FRIGNANO PI
CCOLO"},{"end":"1950-12-01","start":"1946-11-08","name":"FRIGNANO PICCOLO"},
{"start":"1950-12-02","end":"9999-12-31","name":"VILLA DI BRIANO"}],"D075":[
{"start":"1920-10-16","end":"1923-02-13","name":"CORTINA ALL'ADIGE"},{"end":
"1927-01-11","start":"1923-02-14","name":"CORTINA ALL'ADIGE"},{"name":"CORTI
NA ALL'ADIGE","end":"1928-06-27","start":"1927-01-12"},{"name":"CORTINA ALL'
ADIGE","start":"1952-05-17","end":"1971-09-21"},{"name":"CORTINA SULLA STRAD
A DEL VINO","start":"1971-09-22","end":"9999-12-31"}],"I337":[{"start":"1861
-03-17","end":"1927-01-11","name":"SANTHI\u00c0"},{"start":"1927-01-12","end
":"9999-12-31","name":"SANTHI\u00c0"}],"F956":[{"end":"9999-12-31","start":"
1863-07-13","name":"NOVATE MEZZOLA"}],"L789":[{"name":"VERSA","end":"1923-02
-07","start":"1921-01-05"},{"end":"1927-01-11","start":"1923-02-08","name":"
VERSA"},{"name":"VERSA","end":"1928-05-01","start":"1927-01-12"}],"I541":[{"
name":"SCLAFANI","start":"1861-03-17","end":"1954-01-02"},{"start":"1954-01-
03","end":"9999-12-31","name":"SCLAFANI BAGNI"}],"H380":[{"name":"ROCCAVALDI
NA","end":"1923-11-27","start":"1863-04-10"},{"name":"ROCCAVALDINA","start":
"1923-11-28","end":"1940-01-23"},{"name":"ROCCAVALDINA","end":"1948-12-24","
start":"1940-01-24"},{"name":"ROCCAVALDINA","start":"1948-12-25","end":"9999
-12-31"}],"F659":[{"end":"2013-12-31","start":"1861-03-17","name":"MONTEVEGL
IO"}],"M281":[{"end":"9999-12-31","start":"1980-08-10","name":"PETROSINO"}],
"E667":[{"start":"1861-03-17","end":"2007-11-01","name":"LONATO"}],"E734":[{
"start":"1889-02-11","end":"1927-01-11","name":"LUINO"},{"end":"1928-02-08",
"start":"1927-01-12","name":"LUINO"},{"name":"LUINO","start":"1928-02-09","e
nd":"1928-12-07"},{"start":"1928-12-08","end":"1947-12-21","name":"LUINO"},{
"name":"LUINO","start":"1947-12-22","end":"1953-07-23"},{"start":"1953-07-24
","end":"1955-11-30","name":"LUINO"},{"start":"1955-12-01","end":"9999-12-31
","name":"LUINO"}],"C658":[{"name":"CHIUSANO D'ASTI","start":"1863-02-16","e
nd":"1935-04-07"},{"name":"CHIUSANO D'ASTI","start":"1935-04-08","end":"9999
-12-31"}],"I216":[{"name":"SANT'ALFIO","start":"1924-01-04","end":"1955-02-1
4"},{"name":"SANT'ALFIO","start":"1955-02-15","end":"9999-12-31"}],"Z361":[{
"name":"Gibuti","end":"9999-12-31","start":"0000-01-01"}],"L091":[{"name":"T
ELVES","start":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-14","end":
"1927-01-11","name":"TELVES"},{"name":"TELVES","start":"1927-01-12","end":"1
928-04-12"}],"I481":[{"start":"1861-03-17","end":"1928-05-03","name":"SAVOUL
X"}],"A709":[{"start":"1862-12-19","end":"9999-12-31","name":"BASTIA MONDOV\
u00cc"}],"A997":[{"name":"BORGHETTO","start":"1920-10-16","end":"1923-02-13"
},{"start":"1923-02-14","end":"1928-04-16","name":"BORGHETTO"}],"G333":[{"en
d":"1927-01-11","start":"1861-03-17","name":"PARETE"},{"name":"PARETE","star
t":"1927-01-12","end":"1945-08-31"},{"name":"PARETE","end":"9999-12-31","sta
rt":"1945-09-01"}],"M416":[{"name":"GATTICO-VERUNO","end":"9999-12-31","star
t":"2019-01-01"}],"G039":[{"start":"1863-01-19","end":"9999-12-31","name":"O
LIVETO CITRA"}],"A394":[{"name":"ARGENTERA","start":"1861-03-17","end":"1928
-03-12"},{"end":"9999-12-31","start":"1928-03-13","name":"ARGENTERA"}],"L823
":[{"name":"VEZZI PORTIO","end":"1927-01-11","start":"1871-03-01"},{"name":"
VEZZI PORTIO","end":"9999-12-31","start":"1927-01-12"}],"M115":[{"end":"9999
-12-31","start":"1953-06-18","name":"VOLLA"}],"I974":[{"name":"STREGNA","end
":"1968-04-05","start":"1866-11-19"},{"start":"1968-04-06","end":"9999-12-31
","name":"STREGNA"}],"I422":[{"name":"SAPRI","start":"1861-03-17","end":"999
9-12-31"}],"C785":[{"name":"CIVO","start":"1861-03-17","end":"9999-12-31"}],
"B303":[{"name":"BUTI","start":"1867-06-24","end":"9999-12-31"}],"B009":[{"e
nd":"1927-01-11","start":"1861-03-17","name":"BORGO D'ALE"},{"name":"BORGO D
'ALE","start":"1927-01-12","end":"9999-12-31"}],"H795":[{"name":"SAN CHIRICO
 NUOVO","start":"1861-03-17","end":"9999-12-31"}],"A628":[{"end":"1927-01-11
","start":"1872-09-16","name":"BARBARANO ROMANO"},{"name":"BARBARANO ROMANO"
,"start":"1927-01-12","end":"9999-12-31"}],"F520":[{"name":"MONTEGIORGIO","e
nd":"2009-07-11","start":"1898-08-13"},{"name":"MONTEGIORGIO","end":"9999-12
-31","start":"2009-07-12"}],"I377":[{"name":"SAN VALENTINO TORIO","end":"999
9-12-31","start":"1863-02-06"}],"A284":[{"name":"ANDRETTA","start":"1861-03-
17","end":"9999-12-31"}],"I793":[{"name":"SOLBIATE ARNO","start":"1957-12-26
","end":"9999-12-31"},{"name":"SOLBIATE ARNO E MONTE","end":"1927-01-11","st
art":"1905-04-24"},{"name":"SOLBIATE ARNO E MONTE","end":"1957-12-25","start
":"1927-01-12"}],"H480":[{"name":"RODI GARGANICO","start":"1861-03-17","end"
:"9999-12-31"}],"M391":[{"end":"9999-12-31","start":"2018-01-01","name":"RIO
"}],"B979":[{"start":"1863-03-30","end":"2006-01-06","name":"CASOLA IN LUNIG
IANA"},{"name":"CASOLA IN LUNIGIANA","end":"9999-12-31","start":"2006-01-07"
}],"I437":[{"name":"SARNICO","end":"1927-11-02","start":"1861-03-17"},{"name
":"SARNICO","start":"1927-11-03","end":"9999-12-31"}],"L920":[{"start":"1926
-03-18","end":"1927-01-11","name":"VILLA CASTELLI"},{"name":"VILLA CASTELLI"
,"start":"1927-01-12","end":"9999-12-31"}],"B676":[{"end":"9999-12-31","star
t":"1907-11-11","name":"CAPOVALLE"},{"end":"1907-11-10","start":"1861-03-17"
,"name":"HANO"}],"I576":[{"end":"1923-02-13","start":"1920-10-16","name":"SE
GONZANO"},{"start":"1923-02-14","end":"1928-09-08","name":"SEGONZANO"},{"nam
e":"SEGONZANO","end":"9999-12-31","start":"1928-09-09"}],"D750":[{"name":"FO
ZA","end":"9999-12-31","start":"1866-11-19"}],"L457":[{"name":"TUENNO","star
t":"1920-10-16","end":"1923-02-13"},{"name":"TUENNO","end":"2015-12-31","sta
rt":"1923-02-14"}],"I668":[{"end":"1927-01-11","start":"1861-03-17","name":"
SERRI"},{"end":"2005-12-31","start":"1927-01-12","name":"SERRI"},{"name":"SE
RRI","start":"2006-01-01","end":"2016-04-27"},{"end":"9999-12-31","start":"2
016-04-28","name":"SERRI"}],"C185":[{"start":"1861-03-17","end":"9999-12-31"
,"name":"CASTELLO D'ARGILE"}],"C447":[{"name":"CELLERE","end":"1927-01-11","
start":"1871-01-15"},{"end":"9999-12-31","start":"1927-01-12","name":"CELLER
E"}],"G632":[{"name":"PIEVE DI TECO","end":"1928-09-05","start":"1862-11-10"
},{"start":"1928-09-06","end":"1929-03-15","name":"PIEVE DI TECO"},{"name":"
PIEVE DI TECO","start":"1929-03-16","end":"1949-03-23"},{"start":"1949-03-24
","end":"9999-12-31","name":"PIEVE DI TECO"}],"A843":[{"start":"1861-03-17",
"end":"1978-06-09","name":"BIANCO"},{"end":"1980-06-11","start":"1978-06-10"
,"name":"BIANCO"},{"end":"9999-12-31","start":"1980-06-12","name":"BIANCO"}]
,"H195":[{"start":"1861-03-17","end":"9999-12-31","name":"RAVARINO"}],"A462"
:[{"start":"1867-01-17","end":"9999-12-31","name":"ASCOLI PICENO"}],"D328":[
{"end":"9999-12-31","start":"1861-03-17","name":"DOMANICO"}],"F227":[{"name"
:"MIOLA","start":"1920-10-16","end":"1923-02-13"},{"start":"1923-02-14","end
":"1928-09-08","name":"MIOLA"}],"B856":[{"name":"CARZANO","start":"1920-10-1
6","end":"1923-02-13"},{"name":"CARZANO","start":"1923-02-14","end":"1928-05
-12"},{"start":"1947-01-15","end":"1980-01-01","name":"CARZANO"},{"name":"CA
RZANO","end":"9999-12-31","start":"1980-01-02"}],"B568":[{"end":"1942-11-29"
,"start":"1928-11-10","name":"CAMPOSPINOSO ALBAREDO"},{"name":"CAMPOSPINOSO 
ALBAREDO","end":"1948-01-20","start":"1942-11-30"}],"F645":[{"end":"1928-05-
21","start":"1861-03-17","name":"MONTESICURO"}],"G595":[{"start":"1861-03-17
","end":"1927-01-11","name":"PIEDILUCO"},{"name":"PIEDILUCO","end":"1927-01-
12","start":"1927-01-12"}],"I193":[{"end":"9999-12-31","start":"1862-11-10",
"name":"SANT'AGATA DI PUGLIA"}],"F839":[{"start":"1865-12-14","end":"1925-11
-29","name":"NAPOLI"},{"name":"NAPOLI","start":"1925-11-30","end":"1926-07-0
3"},{"name":"NAPOLI","end":"1927-11-07","start":"1926-07-04"},{"end":"1929-0
4-12","start":"1927-11-08","name":"NAPOLI"},{"start":"1929-04-13","end":"999
9-12-31","name":"NAPOLI"}],"G627":[{"end":"1928-08-14","start":"1861-03-17",
"name":"PIETRARUBBIA"},{"start":"1947-09-24","end":"1998-09-03","name":"PIET
RARUBBIA"},{"end":"9999-12-31","start":"1998-09-04","name":"PIETRARUBBIA"}],
"D150":[{"start":"1861-03-17","end":"1920-02-26","name":"CREMONA"},{"name":"
CREMONA","start":"1940-08-28","end":"9999-12-31"},{"end":"1940-08-27","start
":"1920-02-27","name":"CREMONA ED UNITI"}],"H311":[{"name":"RIPABOTTONI","st
art":"1861-03-17","end":"1970-03-02"},{"name":"RIPABOTTONI","start":"1970-03
-03","end":"9999-12-31"}],"F232":[{"name":"MIRABELLO MONFERRATO","end":"1929
-02-11","start":"1863-02-16"},{"start":"1929-02-12","end":"9999-12-31","name
":"MIRABELLO MONFERRATO"}],"A940":[{"name":"BOLLATE","start":"1869-04-01","e
nd":"2001-12-11"},{"name":"BOLLATE","start":"2001-12-12","end":"2003-03-05"}
,{"name":"BOLLATE","start":"2003-03-06","end":"2004-06-07"},{"name":"BOLLATE
","end":"9999-12-31","start":"2004-06-08"}],"C211":[{"end":"1927-01-11","sta
rt":"1862-11-10","name":"CASTEL MORRONE"},{"name":"CASTEL MORRONE","start":"
1927-01-12","end":"1945-08-31"},{"end":"9999-12-31","start":"1945-09-01","na
me":"CASTEL MORRONE"}],"D599":[{"name":"FINALE EMILIA","start":"1940-08-24",
"end":"9999-12-31"},{"start":"1863-08-10","end":"1940-08-23","name":"FINALE 
NELL'EMILIA"}],"E134":[{"end":"1935-04-07","start":"1861-03-17","name":"GRAN
A"},{"name":"GRANA","start":"1935-04-08","end":"2023-01-16"},{"name":"GRANA 
MONFERRATO","start":"2023-01-17","end":"9999-12-31"}],"B861":[{"end":"9999-1
2-31","start":"1861-03-17","name":"CASALANGUIDA"}],"E378":[{"end":"1923-02-1
3","start":"1920-10-16","name":"IVANO-FRACENA"},{"end":"1928-07-27","start":
"1923-02-14","name":"IVANO-FRACENA"},{"name":"IVANO-FRACENA","start":"1947-0
1-14","end":"2016-07-01"}],"C878":[{"name":"COLLI A VOLTURNO","start":"1863-
08-10","end":"1928-03-05"},{"start":"1928-03-06","end":"1934-03-06","name":"
COLLI A VOLTURNO"},{"name":"COLLI A VOLTURNO","start":"1934-03-07","end":"19
46-04-01"},{"name":"COLLI A VOLTURNO","start":"1946-04-02","end":"1970-03-02
"},{"name":"COLLI A VOLTURNO","end":"9999-12-31","start":"1970-03-03"}],"Z14
9":[{"name":"Croazia","end":"9999-12-31","start":"0000-01-01"}],"H496":[{"na
me":"ROIO PIANO","start":"1863-05-06","end":"1927-09-20"}],"D249":[{"start":
"1920-10-16","end":"1923-02-13","name":"DARDINE"},{"end":"1929-01-23","start
":"1923-02-14","name":"DARDINE"}],"E438":[{"name":"LANGHIRANO","start":"1861
-03-17","end":"9999-12-31"}],"A109":[{"name":"AIRASCA","start":"1893-05-26",
"end":"1960-09-28"},{"name":"AIRASCA","end":"9999-12-31","start":"1960-09-29
"}],"I502":[{"name":"SCANO AL BREMBO","end":"1928-05-18","start":"1863-08-10
"}],"H067":[{"name":"PRI\u00d2","end":"1923-02-13","start":"1920-10-16"},{"n
ame":"PRI\u00d2","start":"1923-02-14","end":"1928-12-13"}],"B965":[{"start":
"1866-11-19","end":"9999-12-31","name":"CASIER"}],"B602":[{"end":"1929-05-15
","start":"1861-03-17","name":"CANICATT\u00cc"},{"start":"1929-05-16","end":
"9999-12-31","name":"CANICATT\u00cc"}],"D463":[{"name":"FAGGIANO","start":"1
861-03-17","end":"1923-12-21"},{"name":"FAGGIANO","start":"1923-12-22","end"
:"9999-12-31"}],"F400":[{"end":"1955-05-07","start":"1862-11-10","name":"MON
TALBANO DI ELICONA"},{"name":"MONTALBANO ELICONA","start":"1955-05-08","end"
:"9999-12-31"},{"end":"1862-11-09","start":"1861-03-17","name":"MONTE ALBANO
"}],"L648":[{"name":"VALSAVIORE","end":"1954-08-05","start":"1927-11-03"}],"
D842":[{"start":"1861-03-17","end":"1927-01-11","name":"GADONI"},{"name":"GA
DONI","start":"1927-01-12","end":"9999-12-31"}],"I310":[{"start":"1861-03-17
","end":"1923-03-18","name":"SANTA SOFIA"},{"end":"1924-01-13","start":"1923
-03-19","name":"SANTA SOFIA"},{"name":"SANTA SOFIA","end":"1925-10-31","star
t":"1924-01-14"},{"start":"1925-11-01","end":"1926-08-05","name":"SANTA SOFI
A"},{"start":"1926-08-06","end":"1929-06-17","name":"SANTA SOFIA"},{"start":
"1929-06-18","end":"1992-04-15","name":"SANTA SOFIA"},{"start":"1992-04-16",
"end":"9999-12-31","name":"SANTA SOFIA"}],"D694":[{"end":"1999-07-26","start
":"1861-03-17","name":"FORCOLA"},{"end":"9999-12-31","start":"1999-07-27","n
ame":"FORCOLA"}],"C486":[{"name":"CERCEMAGGIORE","end":"1927-01-11","start":
"1861-03-17"},{"name":"CERCEMAGGIORE","start":"1927-01-12","end":"1970-03-02
"},{"end":"9999-12-31","start":"1970-03-03","name":"CERCEMAGGIORE"}],"A539":
[{"end":"1928-04-06","start":"1867-07-22","name":"BADIA POLESINE"},{"name":"
BADIA POLESINE","start":"1928-04-07","end":"9999-12-31"}],"L943":[{"name":"V
ILLA FARALDI","start":"1861-03-17","end":"1923-12-21"},{"name":"VILLA FARALD
I","end":"9999-12-31","start":"1947-10-02"}],"I132":[{"name":"SAN PROCOPIO",
"end":"9999-12-31","start":"1861-03-17"}],"B311":[{"name":"CABELLA LIGURE","
start":"1895-01-21","end":"1929-05-27"},{"name":"CABELLA LIGURE","end":"1954
-04-23","start":"1929-05-28"},{"start":"1954-04-24","end":"9999-12-31","name
":"CABELLA LIGURE"}],"F293":[{"name":"MOLINO DEI TORTI","start":"1891-05-11"
,"end":"1928-06-18"},{"name":"MOLINO DEI TORTI","start":"1946-07-18","end":"
9999-12-31"}],"G168":[{"start":"1920-10-16","end":"1923-02-13","name":"OSPED
ALETTO"},{"name":"OSPEDALETTO","end":"1928-04-16","start":"1923-02-14"},{"en
d":"9999-12-31","start":"1958-01-01","name":"OSPEDALETTO"}],"L152":[{"start"
:"1861-03-17","end":"1927-01-11","name":"TESTICO"},{"start":"1927-01-12","en
d":"1929-04-11","name":"TESTICO"},{"start":"1929-04-12","end":"9999-12-31","
name":"TESTICO"}],"E609":[{"start":"1861-03-17","end":"1927-01-11","name":"L
ISANZA"},{"end":"1928-04-02","start":"1927-01-12","name":"LISANZA"}],"E906":
[{"name":"MARANO DI NAPOLI","end":"1948-03-10","start":"1863-02-06"},{"start
":"1948-03-11","end":"9999-12-31","name":"MARANO DI NAPOLI"}],"B195":[{"star
t":"1861-03-17","end":"1927-01-11","name":"BROCCO"},{"end":"1954-08-13","sta
rt":"1927-01-12","name":"BROCCO"},{"start":"1954-08-14","end":"9999-12-31","
name":"BROCCOSTELLA"}],"E261":[{"start":"1868-07-01","end":"9999-12-31","nam
e":"GUIDIZZOLO"}],"E076":[{"name":"GOGLIONE SOTTO","start":"1861-03-17","end
":"1928-08-14"}],"I860":[{"end":"1928-04-02","start":"1861-03-17","name":"SO
RMANO"},{"name":"SORMANO","end":"9999-12-31","start":"1947-10-01"}],"I127":[
{"end":"1928-08-06","start":"1863-03-30","name":"SAN PONZO SEMOLA"}],"H568":
[{"name":"ROSETO VALFORTORE","end":"9999-12-31","start":"1862-11-10"}],"D778
":[{"name":"FRASSINELLO-OLIVOLA","end":"1950-02-10","start":"1928-08-16"}],"
H856":[{"name":"SAN GAVINO MONREALE","end":"1974-08-19","start":"1863-12-28"
},{"name":"SAN GAVINO MONREALE","start":"1974-08-20","end":"2005-12-31"},{"e
nd":"2016-04-27","start":"2006-01-01","name":"SAN GAVINO MONREALE"},{"end":"
9999-12-31","start":"2016-04-28","name":"SAN GAVINO MONREALE"}],"E504":[{"st
art":"1905-06-23","end":"2004-04-09","name":"LAZZATE"},{"start":"2004-04-10"
,"end":"2009-06-29","name":"LAZZATE"},{"name":"LAZZATE","end":"9999-12-31","
start":"2009-06-30"}],"G464":[{"name":"PERRA","end":"1923-02-13","start":"19
20-10-16"},{"end":"1926-07-21","start":"1923-02-14","name":"PERRA"}],"C742":
[{"start":"1861-03-17","end":"9999-12-31","name":"CITERNA"}],"A634":[{"start
":"1861-03-17","end":"2008-08-15","name":"BARBIANELLO"},{"name":"BARBIANELLO
","end":"9999-12-31","start":"2008-08-16"}],"A767":[{"name":"BEL PRATO","end
":"1928-11-26","start":"1863-01-26"}],"G693":[{"start":"1866-11-19","end":"9
999-12-31","name":"PIOVE DI SACCO"}],"F543":[{"name":"MONTELEONE D'ORVIETO",
"start":"1863-04-13","end":"1927-01-11"},{"name":"MONTELEONE D'ORVIETO","sta
rt":"1927-01-12","end":"9999-12-31"}],"H602":[{"start":"1877-06-30","end":"1
926-07-15","name":"ROVELLO"},{"name":"ROVELLO PORRO","end":"1928-05-12","sta
rt":"1926-07-16"},{"name":"ROVELLO PORRO","end":"1982-02-05","start":"1939-1
1-04"},{"name":"ROVELLO PORRO","start":"1982-02-06","end":"9999-12-31"}],"C6
70":[{"name":"CROCETTA DEL MONTELLO","end":"9999-12-31","start":"1929-03-02"
},{"name":"CROCETTA TREVIGIANA","end":"1929-03-01","start":"1902-06-01"}],"B
067":[{"name":"BORZONASCA","start":"1861-03-17","end":"9999-12-31"}],"B496":
[{"end":"1924-01-10","start":"1872-09-16","name":"CAMPAGNANO DI ROMA"},{"nam
e":"CAMPAGNANO DI ROMA","end":"1958-03-28","start":"1924-01-11"},{"start":"1
958-03-29","end":"9999-12-31","name":"CAMPAGNANO DI ROMA"}],"L840":[{"name":
"VICENZA","end":"1987-10-20","start":"1866-11-19"},{"name":"VICENZA","end":"
1988-01-05","start":"1987-10-21"},{"name":"VICENZA","end":"9999-12-31","star
t":"1988-01-06"}],"E728":[{"start":"1863-05-06","end":"1927-01-11","name":"L
UGNANO DI VILLA TROIANA"},{"name":"LUGNANO DI VILLA TROIANA","end":"1927-10-
18","start":"1927-01-12"},{"start":"1927-10-19","end":"1928-02-20","name":"V
AZIA"}],"F716":[{"name":"MORCIANO DI LEUCA","end":"1984-04-20","start":"1865
-02-27"},{"end":"9999-12-31","start":"1984-04-21","name":"MORCIANO DI LEUCA"
}],"L714":[{"start":"1861-03-17","end":"1927-01-11","name":"VELATE"},{"start
":"1927-01-12","end":"1927-12-28","name":"VELATE"}],"H861":[{"end":"1927-01-
11","start":"1891-03-06","name":"SAN GERMANO VERCELLESE"},{"end":"9999-12-31
","start":"1927-01-12","name":"SAN GERMANO VERCELLESE"}],"M100":[{"end":"999
9-12-31","start":"1861-03-17","name":"VIZZINI"}],"D026":[{"name":"CORNIGLIO"
,"end":"9999-12-31","start":"1861-03-17"}],"E256":[{"end":"1929-02-20","star
t":"1861-03-17","name":"GUBBIO"},{"end":"9999-12-31","start":"1929-02-21","n
ame":"GUBBIO"}],"I963":[{"name":"STORNARELLA","start":"1861-03-17","end":"19
05-07-09"},{"end":"9999-12-31","start":"1905-07-10","name":"STORNARELLA"}],"
F089":[{"start":"1866-11-19","end":"1928-05-28","name":"MEDUNO"},{"start":"1
928-05-29","end":"1968-04-05","name":"MEDUNO"},{"end":"9999-12-31","start":"
1968-04-06","name":"MEDUNO"}],"F383":[{"end":"1928-05-11","start":"1888-03-1
9","name":"MONSERRATO"},{"start":"1991-12-10","end":"9999-12-31","name":"MON
SERRATO"},{"start":"1881-11-28","end":"1888-03-18","name":"PAULI MONSERRATO"
},{"start":"1861-03-17","end":"1881-11-27","name":"PAULI PIRRI"}],"D885":[{"
end":"9999-12-31","start":"1952-12-14","name":"GALLODORO"}],"E350":[{"start"
:"1861-03-17","end":"9999-12-31","name":"ISOLA DELLE FEMMINE"}],"E972":[{"na
me":"MARRUBIU","end":"1928-11-03","start":"1861-03-17"},{"end":"1974-08-19",
"start":"1948-05-27","name":"MARRUBIU"},{"name":"MARRUBIU","end":"9999-12-31
","start":"1974-08-20"}],"E424":[{"end":"9999-12-31","start":"1863-02-06","n
ame":"LAMA DEI PELIGNI"}],"L418":[{"name":"TRICARICO","end":"1927-01-11","st
art":"1861-03-17"},{"name":"TRICARICO","start":"1927-01-12","end":"9999-12-3
1"}],"C203":[{"end":"1938-12-26","start":"1871-01-15","name":"CASTEL MADAMA"
},{"name":"CASTEL MADAMA","end":"9999-12-31","start":"1938-12-27"}],"G221":[
{"name":"PADERNO D'ASOLO","start":"1867-11-25","end":"1920-03-14"},{"end":"2
019-01-29","start":"1920-03-15","name":"PADERNO DEL GRAPPA"}],"G392":[{"star
t":"1863-01-05","end":"1927-01-11","name":"PAVONE CANAVESE"},{"name":"PAVONE
 CANAVESE","start":"1927-01-12","end":"1945-10-04"},{"name":"PAVONE CANAVESE
","end":"1968-10-21","start":"1945-10-05"},{"name":"PAVONE CANAVESE","end":"
9999-12-31","start":"1968-10-22"}],"A095":[{"name":"AIBA","end":"1923-02-07"
,"start":"1921-01-05"},{"name":"AIBA","end":"1927-01-11","start":"1923-02-08
"},{"name":"AIBA","start":"1927-01-12","end":"1928-03-12"}],"E128":[{"start"
:"1871-01-15","end":"1927-01-11","name":"GRAFFIGNANO"},{"name":"GRAFFIGNANO"
,"start":"1927-01-12","end":"1928-05-18"},{"end":"9999-12-31","start":"1928-
05-19","name":"GRAFFIGNANO"}],"D981":[{"name":"GERENZANO","start":"1861-03-1
7","end":"1927-01-11"},{"end":"1928-02-15","start":"1927-01-12","name":"GERE
NZANO"},{"name":"GERENZANO","start":"1950-07-19","end":"9999-12-31"}],"A923"
:[{"end":"1946-12-19","start":"1928-02-24","name":"BOGLIASCO PIEVE"}],"F116"
:[{"name":"MELMA","end":"1935-03-14","start":"1866-11-19"},{"end":"9999-12-3
1","start":"1935-03-15","name":"SILEA"}],"L114":[{"name":"TERMON","start":"1
920-10-16","end":"1923-02-13"},{"end":"1928-10-29","start":"1923-02-14","nam
e":"TERMON"}],"C850":[{"name":"COLLEBEATO","end":"9999-12-31","start":"1861-
03-17"}],"F621":[{"start":"1863-03-30","end":"9999-12-31","name":"MONTE SAN 
GIUSTO"}],"L897":[{"start":"1861-03-17","end":"9999-12-31","name":"VIGOLZONE
"}],"E002":[{"end":"1928-05-02","start":"1861-03-17","name":"GHEVIO"}],"D474
":[{"start":"1861-03-17","end":"9999-12-31","name":"FALCONE"}],"I732":[{"end
":"9999-12-31","start":"1861-03-17","name":"SILIGO"}],"D922":[{"name":"GARGA
RO","end":"1923-02-07","start":"1921-01-05"},{"name":"GARGARO","start":"1923
-02-08","end":"1927-01-11"},{"name":"GARGARO","end":"1928-08-15","start":"19
27-01-12"},{"name":"GARGARO","end":"1947-09-15","start":"1928-08-16"}],"B480
":[{"start":"1920-10-16","end":"1923-02-13","name":"CAMINATA IN TURES"},{"na
me":"CAMINATA IN TURES","end":"1927-01-11","start":"1923-02-14"},{"end":"192
9-02-08","start":"1927-01-12","name":"CAMINATA IN TURES"}],"I305":[{"name":"
SANT'ARCANGELO","end":"9999-12-31","start":"1861-03-17"}],"F415":[{"start":"
1867-01-31","end":"9999-12-31","name":"MONTALTO DELLE MARCHE"}],"L752":[{"en
d":"9999-12-31","start":"1861-03-17","name":"VERDELLINO"}],"H676":[{"name":"
SAINT VINCENT","end":"1927-01-11","start":"1861-03-17"},{"name":"SAINT VINCE
NT","start":"1927-01-12","end":"1939-10-20"},{"end":"1977-01-01","start":"19
46-01-30","name":"SAINT VINCENT"},{"end":"9999-12-31","start":"1977-01-02","
name":"SAINT-VINCENT"},{"name":"SAN VINCENZO DELLA FONTE","start":"1939-10-2
1","end":"1946-01-29"}],"G768":[{"end":"9999-12-31","start":"1861-03-17","na
me":"POGGIO RENATICO"}],"F682":[{"name":"MONTOGGIO","start":"1893-08-30","en
d":"9999-12-31"}],"H979":[{"name":"SAN MARCELLO","end":"9999-12-31","start":
"1861-03-17"}],"B795":[{"name":"CARMIGNANO DI BRENTA","start":"1867-08-26","
end":"9999-12-31"}],"H009":[{"name":"PRATOVECCHIO STIA","start":"1929-03-12"
,"end":"1934-09-27"}],"C953":[{"start":"1920-10-16","end":"1923-02-13","name
":"CONDINO"},{"start":"1923-02-14","end":"1928-03-12","name":"CONDINO"},{"st
art":"1928-03-13","end":"1946-12-23","name":"CONDINO"},{"end":"2015-12-31","
start":"1946-12-24","name":"CONDINO"}],"C617":[{"end":"1923-10-13","start":"
1870-08-02","name":"CHIARAVALLE MILANESE"}],"H303":[{"name":"RIOLUNATO","sta
rt":"1861-03-17","end":"9999-12-31"}],"G889":[{"name":"PORLEZZA","end":"1928
-03-06","start":"1861-03-17"},{"name":"PORLEZZA","start":"1928-03-07","end":
"9999-12-31"}],"I727":[{"name":"SIGILLO","end":"9999-12-31","start":"1861-03
-17"}],"D937":[{"name":"GATTICO","end":"2018-12-31","start":"1877-10-31"}],"
D334":[{"name":"DOMUSNOVAS","end":"1974-08-19","start":"1861-03-17"},{"name"
:"DOMUSNOVAS","start":"1974-08-20","end":"2005-12-31"},{"end":"2016-04-27","
start":"2006-01-01","name":"DOMUSNOVAS"},{"name":"DOMUSNOVAS","end":"9999-12
-31","start":"2016-04-28"}],"C142":[{"end":"1928-09-24","start":"1861-03-17"
,"name":"CASTELLAR GUIDOBONO"},{"end":"1959-12-03","start":"1947-09-04","nam
e":"CASTELLAR GUIDOBONO"},{"name":"CASTELLAR GUIDOBONO","start":"1959-12-04"
,"end":"9999-12-31"}],"G282":[{"name":"PALMA DI MONTECHIARO","end":"9999-12-
31","start":"1863-01-19"}],"A820":[{"start":"1861-03-17","end":"9999-12-31",
"name":"BESATE"}],"A167":[{"name":"ALBIZZATE","start":"1869-03-11","end":"19
05-04-23"},{"end":"1927-01-11","start":"1905-04-24","name":"ALBIZZATE"},{"na
me":"ALBIZZATE","start":"1927-01-12","end":"9999-12-31"}],"G492":[{"end":"19
45-10-24","start":"1861-03-17","name":"PESCINA"},{"end":"9999-12-31","start"
:"1945-10-25","name":"PESCINA"}],"E927":[{"name":"MARCETELLI","start":"1861-
03-17","end":"1923-03-18"},{"start":"1923-03-19","end":"1927-01-11","name":"
MARCETELLI"},{"end":"9999-12-31","start":"1927-01-12","name":"MARCETELLI"}],
"B540":[{"start":"1939-10-21","end":"1946-01-29","name":"CAMPO LARIS"},{"nam
e":"CHAMPORCHER","start":"1861-03-17","end":"1927-01-11"},{"name":"CHAMPORCH
ER","end":"1939-10-20","start":"1927-01-12"},{"name":"CHAMPORCHER","start":"
1946-01-30","end":"9999-12-31"}],"E324":[{"start":"1928-02-02","end":"1946-0
3-08","name":"IRGOLI DI GALTELLI"}],"L318":[{"name":"TRABUCHELLO","start":"1
861-03-17","end":"1928-08-14"}],"I640":[{"end":"9999-12-31","start":"1877-09
-08","name":"SERRA RICC\u00d2"}],"E450":[{"start":"1861-03-17","end":"9999-1
2-31","name":"LAPPANO"}],"B290":[{"end":"1927-09-03","start":"1867-07-22","n
ame":"BUSO SARZANO"}],"C824":[{"start":"1866-11-19","end":"1924-04-17","name
":"COGOLLO"},{"name":"COGOLLO DEL CENGIO","end":"9999-12-31","start":"1924-0
4-18"}],"C701":[{"end":"1929-06-03","start":"1861-03-17","name":"CINAGLIO"},
{"name":"CINAGLIO","end":"1935-04-07","start":"1929-06-04"},{"start":"1935-0
4-08","end":"9999-12-31","name":"CINAGLIO"}],"G723":[{"name":"PIZZOCORNO","e
nd":"1928-08-06","start":"1861-03-17"}],"E932":[{"name":"MARCIANISE","end":"
1927-01-11","start":"1861-03-17"},{"name":"MARCIANISE","end":"1928-04-18","s
tart":"1927-01-12"},{"name":"MARCIANISE","end":"1945-08-31","start":"1928-04
-19"},{"name":"MARCIANISE","end":"1946-08-23","start":"1945-09-01"},{"name":
"MARCIANISE","start":"1946-08-24","end":"9999-12-31"}],"A565":[{"name":"BAGN
O DI ROMAGNA","start":"1861-03-17","end":"1923-03-18"},{"end":"1992-04-15","
start":"1923-03-19","name":"BAGNO DI ROMAGNA"},{"name":"BAGNO DI ROMAGNA","e
nd":"9999-12-31","start":"1992-04-16"}],"I106":[{"name":"SAN PIETRO D'ORZIO"
,"end":"1928-04-16","start":"1861-03-17"}],"D138":[{"name":"CREDA","end":"19
23-02-07","start":"1921-01-05"},{"name":"CREDA","end":"1927-01-11","start":"
1923-02-08"},{"end":"1928-05-01","start":"1927-01-12","name":"CREDA"}],"D977
":[{"name":"GERACI SICULO","end":"9999-12-31","start":"1863-07-13"}],"B380":
[{"end":"1928-09-05","start":"1861-03-17","name":"CALASCA"},{"name":"CALASCA
-CASTIGLIONE","start":"1928-09-06","end":"1992-05-22"},{"name":"CALASCA-CAST
IGLIONE","end":"9999-12-31","start":"1992-05-23"}],"I405":[{"name":"SAN VITO
 DI FAGAGNA","end":"1928-11-10","start":"1866-11-19"},{"end":"1968-04-05","s
tart":"1946-12-24","name":"SAN VITO DI FAGAGNA"},{"name":"SAN VITO DI FAGAGN
A","start":"1968-04-06","end":"1971-02-01"},{"start":"1971-02-02","end":"999
9-12-31","name":"SAN VITO DI FAGAGNA"}],"F315":[{"end":"1928-01-21","start":
"1863-01-05","name":"MOMBELLO DI TORINO"},{"name":"MOMBELLO DI TORINO","star
t":"1947-01-09","end":"9999-12-31"}],"A968":[{"name":"BONDONE","start":"1920
-10-16","end":"1923-02-13"},{"name":"BONDONE","start":"1923-02-14","end":"19
28-04-16"},{"start":"1953-09-12","end":"9999-12-31","name":"BONDONE"}],"E163
":[{"name":"GREGGIO","start":"1861-03-17","end":"1927-01-11"},{"start":"1927
-01-12","end":"9999-12-31","name":"GREGGIO"}],"H641":[{"start":"1861-03-17",
"end":"9999-12-31","name":"RUOTI"}],"D374":[{"end":"1928-06-27","start":"186
1-03-17","name":"DRUOGNO"},{"start":"1928-06-28","end":"1992-05-22","name":"
DRUOGNO"},{"start":"1992-05-23","end":"9999-12-31","name":"DRUOGNO"}],"M018"
:[{"name":"VILLA SAN GIOVANNI","start":"1867-01-25","end":"1927-08-02"},{"en
d":"1948-05-07","start":"1933-01-26","name":"VILLA SAN GIOVANNI"},{"start":"
1948-05-08","end":"1950-02-10","name":"VILLA SAN GIOVANNI"},{"name":"VILLA S
AN GIOVANNI","start":"1950-02-11","end":"9999-12-31"}],"A854":[{"name":"BICC
ARI","end":"9999-12-31","start":"1861-03-17"}],"F191":[{"name":"MIASINO","en
d":"9999-12-31","start":"1861-03-17"}],"I772":[{"end":"1923-02-13","start":"
1920-10-16","name":"SMARANO"},{"end":"1929-01-23","start":"1923-02-14","name
":"SMARANO"},{"start":"1952-09-20","end":"2014-12-31","name":"SMARANO"}],"D4
34":[{"name":"ESINE","end":"9999-12-31","start":"1861-03-17"}],"L660":[{"sta
rt":"1929-03-24","end":"1982-02-16","name":"VANDOIES"},{"start":"1982-02-17"
,"end":"9999-12-31","name":"VANDOIES"}],"E698":[{"name":"LOSINE","start":"18
61-03-17","end":"1927-12-28"},{"name":"LOSINE","end":"9999-12-31","start":"1
949-08-27"}],"H636":[{"start":"1861-03-17","end":"1929-05-28","name":"RUGINE
LLO"}],"H939":[{"start":"1861-03-17","end":"1927-01-11","name":"SAN GREGORIO
"},{"name":"SAN GREGORIO","end":"1945-08-31","start":"1927-01-12"},{"name":"
SAN GREGORIO","end":"1954-06-24","start":"1945-09-01"},{"name":"SAN GREGORIO
 MATESE","start":"1954-06-25","end":"9999-12-31"}],"D007":[{"name":"CORINALD
O","end":"9999-12-31","start":"1861-03-17"}],"H403":[{"end":"1992-04-15","st
art":"1863-05-06","name":"ROCCA DI NETO"},{"name":"ROCCA DI NETO","start":"1
992-04-16","end":"9999-12-31"}],"D738":[{"name":"FOSSACESIA","start":"1863-0
2-06","end":"9999-12-31"}],"I928":[{"name":"SPRIANA","start":"1861-03-17","e
nd":"9999-12-31"}],"H098":[{"name":"QUADRI","end":"9999-12-31","start":"1861
-03-17"}],"E036":[{"name":"GINOSA","start":"1861-03-17","end":"1923-12-21"},
{"start":"1923-12-22","end":"9999-12-31","name":"GINOSA"}],"A674":[{"name":"
BARONISSI","end":"9999-12-31","start":"1861-03-17"}],"E763":[{"end":"1927-01
-11","start":"1861-03-17","name":"LUSIGLI\u00c8"},{"name":"LUSIGLI\u00c8","s
tart":"1927-01-12","end":"1945-10-04"},{"start":"1945-10-05","end":"9999-12-
31","name":"LUSIGLI\u00c8"}],"F791":[{"end":"1927-10-12","start":"1861-03-17
","name":"MOZZO"},{"end":"9999-12-31","start":"1947-10-20","name":"MOZZO"}],
"B286":[{"name":"BUSCATE","start":"1861-03-17","end":"9999-12-31"}],"A579":[
{"start":"1861-03-17","end":"1927-01-11","name":"BAIA E LATINA"},{"name":"BA
IA E LATINA","end":"1945-08-31","start":"1927-01-12"},{"name":"BAIA E LATINA
","end":"9999-12-31","start":"1945-09-01"}],"I172":[{"end":"1928-05-23","sta
rt":"1913-07-10","name":"SANTA CESAREA"},{"end":"9999-12-31","start":"1928-0
5-24","name":"SANTA CESAREA TERME"}],"G364":[{"name":"PASTORANO","start":"18
61-03-17","end":"1927-01-11"},{"name":"PASTORANO","end":"1928-07-19","start"
:"1927-01-12"},{"name":"PASTORANO","end":"9999-12-31","start":"1946-12-19"}]
,"E041":[{"start":"1863-04-10","end":"9999-12-31","name":"GIOIA TAURO"}],"G9
67":[{"end":"1926-02-05","start":"1861-03-17","name":"PRA"}],"D813":[{"name"
:"FRUGAROLO","end":"1929-06-14","start":"1861-03-17"},{"name":"FRUGAROLO","e
nd":"9999-12-31","start":"1929-06-15"}],"A219":[{"name":"ALONE","end":"1928-
04-02","start":"1861-03-17"}],"B396":[{"name":"CALCO","end":"1927-06-17","st
art":"1861-03-17"},{"name":"CALCO","start":"1953-09-13","end":"1992-04-15"},
{"name":"CALCO","end":"9999-12-31","start":"1992-04-16"}],"B247":[{"name":"B
UDOIA","end":"1968-04-05","start":"1866-11-19"},{"end":"9999-12-31","start":
"1968-04-06","name":"BUDOIA"}],"H523":[{"start":"1877-02-15","end":"9999-12-
31","name":"RONCADE"}],"A812":[{"start":"1863-02-16","end":"1935-04-07","nam
e":"BERZANO DI SAN PIETRO"},{"start":"1935-04-08","end":"9999-12-31","name":
"BERZANO DI SAN PIETRO"}],"B597":[{"end":"1927-01-11","start":"1871-01-15","
name":"CANEPINA"},{"name":"CANEPINA","end":"9999-12-31","start":"1927-01-12"
}],"F483":[{"end":"1992-05-22","start":"1861-03-17","name":"MONTECRESTESE"},
{"name":"MONTECRESTESE","end":"9999-12-31","start":"1992-05-23"}],"D910":[{"
start":"1861-03-17","end":"1928-07-31","name":"GARBAGNA"},{"start":"1928-08-
01","end":"9999-12-31","name":"GARBAGNA"}],"C101":[{"end":"9999-12-31","star
t":"1876-03-20","name":"CASTELFIORENTINO"}],"I525":[{"start":"1861-03-17","e
nd":"1927-01-11","name":"SCHIANNO"},{"name":"SCHIANNO","end":"1927-06-30","s
tart":"1927-01-12"}],"G123":[{"end":"9999-12-31","start":"1866-11-19","name"
:"ORSAGO"}],"C630":[{"end":"9999-12-31","start":"1866-11-19","name":"CHIES D
'ALPAGO"}],"B625":[{"name":"CANSERO","start":"1861-03-17","end":"1867-12-31"
}],"I697":[{"end":"1927-01-11","start":"1861-03-17","name":"SETTEFRATI"},{"e
nd":"1963-04-27","start":"1927-01-12","name":"SETTEFRATI"},{"name":"SETTEFRA
TI","end":"9999-12-31","start":"1963-04-28"}],"I706":[{"start":"1861-03-17",
"end":"1927-01-11","name":"SEUI"},{"start":"1927-01-12","end":"2005-12-31","
name":"SEUI"},{"start":"2006-01-01","end":"2016-04-27","name":"SEUI"},{"star
t":"2016-04-28","end":"9999-12-31","name":"SEUI"}],"E915":[{"start":"1861-03
-17","end":"9999-12-31","name":"MARANO PRINCIPATO"}],"M410":[{"start":"2019-
01-01","end":"9999-12-31","name":"RIVA DEL PO"}],"D761":[{"start":"1864-04-0
1","end":"1926-03-17","name":"FRANCAVILLA FONTANA"},{"start":"1926-03-18","e
nd":"1927-01-11","name":"FRANCAVILLA FONTANA"},{"name":"FRANCAVILLA FONTANA"
,"end":"9999-12-31","start":"1927-01-12"}],"H571":[{"end":"1928-08-13","star
t":"1861-03-17","name":"ROSINGO"}],"E278":[{"name":"IDRIA","end":"1923-02-07
","start":"1921-01-05"},{"name":"IDRIA","end":"1927-01-11","start":"1923-02-
08"},{"start":"1927-01-12","end":"1928-08-23","name":"IDRIA"},{"start":"1928
-08-24","end":"1947-09-15","name":"IDRIA"}],"C311":[{"name":"CASTIGLIONE DEL
LA VALLE","start":"1870-01-02","end":"1928-03-26"},{"name":"COLLEDARA","star
t":"1928-03-27","end":"9999-12-31"}],"G585":[{"start":"1861-03-17","end":"19
27-02-19","name":"PIAZZO ALTO"}],"I183":[{"end":"9999-12-31","start":"1864-0
4-01","name":"SANTA DOMENICA TALAO"}],"I210":[{"name":"SANT'ALBANO STURA","s
tart":"1862-12-19","end":"9999-12-31"}],"H185":[{"name":"RAPOLANO","start":"
1861-03-17","end":"1949-09-08"},{"end":"9999-12-31","start":"1949-09-09","na
me":"RAPOLANO TERME"}],"E811":[{"end":"9999-12-31","start":"1869-01-08","nam
e":"MAGLIANO DE' MARSI"}],"B742":[{"name":"CARBONATE","start":"1861-03-17","
end":"1928-09-22"},{"end":"9999-12-31","start":"1953-09-13","name":"CARBONAT
E"}],"G171":[{"name":"OSSAGO LODIGIANO","start":"1867-01-02","end":"1992-04-
15"},{"name":"OSSAGO LODIGIANO","start":"1992-04-16","end":"9999-12-31"}],"H
447":[{"start":"1861-03-17","end":"9999-12-31","name":"ROCCASPARVERA"}],"D34
9":[{"start":"1920-10-16","end":"1923-02-13","name":"DORSINO"},{"name":"DORS
INO","start":"1923-02-14","end":"1928-01-25"},{"name":"DORSINO","end":"2014-
12-31","start":"1954-07-24"}],"D043":[{"name":"CORROPOLI","end":"9999-12-31"
,"start":"1861-03-17"}],"C195":[{"end":"9999-12-31","start":"1872-01-01","na
me":"CASTELLUCCHIO"}],"Z128":[{"start":"0000-01-01","end":"9999-12-31","name
":"Portogallo"}],"I570":[{"name":"SEGARIU","end":"1927-11-02","start":"1861-
03-17"},{"end":"1974-08-19","start":"1946-02-08","name":"SEGARIU"},{"start":
"1974-08-20","end":"2005-12-31","name":"SEGARIU"},{"end":"2016-04-27","start
":"2006-01-01","name":"SEGARIU"},{"name":"SEGARIU","start":"2016-04-28","end
":"9999-12-31"}],"F327":[{"start":"1863-01-05","end":"9999-12-31","name":"MO
NASTERO DI LANZO"}],"D228":[{"name":"CURTI","start":"1861-03-17","end":"1927
-01-11"},{"end":"1929-02-06","start":"1927-01-12","name":"CURTI"},{"name":"C
URTI","start":"1946-11-12","end":"9999-12-31"}],"E058":[{"end":"1928-02-23",
"start":"1861-03-17","name":"GIULIANOVA"},{"end":"1929-05-29","start":"1928-
02-24","name":"GIULIANOVA"},{"start":"1929-05-30","end":"9999-12-31","name":
"GIULIANOVA"}],"H878":[{"end":"9999-12-31","start":"1863-02-16","name":"SAN 
GIORGIO MONFERRATO"}],"D756":[{"start":"1861-03-17","end":"9999-12-31","name
":"FRAGNETO MONFORTE"}],"C496":[{"name":"CERDA","start":"1861-03-17","end":"
9999-12-31"}],"D684":[{"name":"FONTENO","start":"1861-03-17","end":"1928-03-
06"},{"name":"FONTENO","start":"1948-06-26","end":"9999-12-31"}],"G345":[{"n
ame":"PARRAVICINO","start":"1861-03-17","end":"1928-12-13"}],"F472":[{"name"
:"MONTECHIARO DENICE","start":"1929-06-02","end":"1946-07-03"}],"A042":[{"na
me":"ACQUARICA DEL CAPO","end":"2019-05-14","start":"1861-03-17"}],"H486":[{
"start":"1861-03-17","end":"1928-05-03","name":"ROGENO"},{"name":"ROGENO","s
tart":"1928-05-04","end":"1992-04-15"},{"name":"ROGENO","start":"1992-04-16"
,"end":"9999-12-31"}],"L629":[{"name":"VALLO TORINESE","start":"1863-01-05",
"end":"1927-12-23"},{"start":"1954-06-30","end":"9999-12-31","name":"VALLO T
ORINESE"}],"C067":[{"end":"9999-12-31","start":"1861-03-17","name":"CASTELBU
ONO"}],"F924":[{"end":"1927-01-11","start":"1861-03-17","name":"NOLA"},{"sta
rt":"1927-01-12","end":"9999-12-31","name":"NOLA"}],"B670":[{"name":"CAPOLON
A","start":"1861-03-17","end":"9999-12-31"}],"L926":[{"start":"1861-03-17","
end":"1923-11-23","name":"VILLA COLLEMANDINA"},{"name":"VILLA COLLEMANDINA",
"start":"1923-11-24","end":"9999-12-31"}],"H211":[{"end":"9999-12-31","start
":"1893-01-30","name":"RECANATI"}],"F332":[{"start":"1868-02-14","end":"9999
-12-31","name":"MONASTIER DI TREVISO"}],"E518":[{"name":"LEINI","end":"9999-
12-31","start":"1939-07-21"},{"start":"1861-03-17","end":"1939-07-20","name"
:"LEYNI"}],"L524":[{"name":"VACCARIZZO ALBANESE","end":"9999-12-31","start":
"1864-01-29"}],"F526":[{"name":"MONTEGRINO VALTRAVAGLIA","end":"1957-02-13",
"start":"1928-01-22"},{"name":"MONTEGRINO VALTRAVAGLIA","end":"1987-10-28","
start":"1957-02-14"},{"start":"1987-10-29","end":"9999-12-31","name":"MONTEG
RINO VALTRAVAGLIA"}],"F705":[{"name":"MONZAMBANO","end":"9999-12-31","start"
:"1872-01-01"}],"D589":[{"start":"1861-03-17","end":"1992-04-15","name":"FIL
ANDARI"},{"name":"FILANDARI","start":"1992-04-16","end":"9999-12-31"}],"G988
":[{"name":"PRASCORSANO","start":"1861-03-17","end":"1927-01-11"},{"end":"19
28-09-25","start":"1927-01-12","name":"PRASCORSANO"},{"end":"1996-02-29","st
art":"1948-01-15","name":"PRASCORSANO"},{"end":"9999-12-31","start":"1996-03
-01","name":"PRASCORSANO"}],"Z728":[{"start":"0000-01-01","end":"9999-12-31"
,"name":"Tonga"}],"I277":[{"name":"SANT'ANGELO A CUPOLO","end":"9999-12-31",
"start":"1865-12-25"}],"B203":[{"name":"BRONZOLO","end":"1923-02-13","start"
:"1920-10-16"},{"start":"1923-02-14","end":"1948-03-13","name":"BRONZOLO"},{
"start":"1948-03-14","end":"9999-12-31","name":"BRONZOLO"}],"D156":[{"end":"
9999-12-31","start":"1866-11-19","name":"CRESPADORO"}],"Z256":[{"name":"Kirg
hizistan","start":"0000-01-01","end":"9999-12-31"}],"L081":[{"start":"1940-0
7-23","end":"1998-09-03","name":"TAVULLIA"},{"name":"TAVULLIA","start":"1998
-09-04","end":"9999-12-31"},{"end":"1940-07-22","start":"1861-03-17","name":
"TOMBA DI PESARO"}],"M337":[{"name":"VERDERIO","end":"9999-12-31","start":"2
014-02-04"}],"G459":[{"end":"1925-03-19","start":"1861-03-17","name":"PERLOZ
"},{"end":"1927-01-11","start":"1925-03-20","name":"PERLOZ"},{"end":"1929-05
-03","start":"1927-01-12","name":"PERLOZ"},{"name":"PERLOZ","end":"9999-12-3
1","start":"1946-03-10"}],"I491":[{"start":"1862-12-19","end":"1929-03-23","
name":"SCALETTA UZZONE"}],"A987":[{"name":"BORE","start":"1932-07-06","end":
"9999-12-31"},{"start":"1919-09-01","end":"1932-07-05","name":"METTI E POZZO
LO"}],"B850":[{"start":"1861-03-17","end":"9999-12-31","name":"CARUGATE"}],"
M291":[{"end":"2005-12-31","start":"1988-08-06","name":"PISCINAS"},{"end":"2
016-04-27","start":"2006-01-01","name":"PISCINAS"},{"start":"2016-04-28","en
d":"9999-12-31","name":"PISCINAS"}],"M322":[{"name":"CASTELFRANCO PIANDISC\u
00d2","end":"9999-12-31","start":"2014-01-01"}],"F105":[{"end":"9999-12-31",
"start":"1861-03-17","name":"MELICUCC\u00c0"}],"H658":[{"end":"9999-12-31","
start":"1928-10-09","name":"SACROFANO"},{"start":"1871-01-15","end":"1928-10
-08","name":"SCROFANO"}],"A544":[{"name":"BAGALADI","start":"1861-03-17","en
d":"9999-12-31"}],"D161":[{"name":"CRESPINO","start":"1866-11-19","end":"999
9-12-31"}],"F633":[{"start":"1861-03-17","end":"1863-04-12","name":"MONTESAN
TO"},{"name":"MONTESANTO VIEZI","end":"1864-01-25","start":"1863-04-13"},{"s
tart":"1864-01-26","end":"1879-08-28","name":"MONTESANTO VIGI"}],"I316":[{"n
ame":"SANTA VITTORIA D'ALBA","end":"1947-10-28","start":"1862-12-19"},{"star
t":"1947-10-29","end":"9999-12-31","name":"SANTA VITTORIA D'ALBA"}],"L404":[
{"start":"1861-03-17","end":"1928-12-13","name":"TREVIOLO"},{"start":"1928-1
2-14","end":"1983-05-19","name":"TREVIOLO"},{"start":"1983-05-20","end":"999
9-12-31","name":"TREVIOLO"}],"F406":[{"start":"1864-02-21","end":"2017-12-31
","name":"MONTALTO LIGURE"}],"C480":[{"end":"1927-01-11","start":"1861-03-17
","name":"CERAMI"},{"name":"CERAMI","start":"1927-01-12","end":"9999-12-31"}
],"Z504":[{"name":"Cuba","end":"9999-12-31","start":"0000-01-01"}],"I057":[{
"end":"1904-06-05","start":"1863-07-13","name":"SAN NICOLA DELL'ALTO"},{"end
":"1992-04-15","start":"1904-06-06","name":"SAN NICOLA DELL'ALTO"},{"name":"
SAN NICOLA DELL'ALTO","end":"9999-12-31","start":"1992-04-16"}],"I783":[{"na
me":"SOLAGNA","start":"1866-11-19","end":"9999-12-31"}],"H490":[{"end":"1928
-02-20","start":"1861-03-17","name":"ROGLIANO"},{"end":"1934-03-06","start":
"1928-02-21","name":"ROGLIANO"},{"name":"ROGLIANO","start":"1934-03-07","end
":"1937-07-17"},{"name":"ROGLIANO","end":"9999-12-31","start":"1937-07-18"}]
,"A931":[{"end":"1927-01-11","start":"1861-03-17","name":"BOISSANO"},{"start
":"1927-01-12","end":"1929-05-13","name":"BOISSANO"},{"start":"1946-12-24","
end":"9999-12-31","name":"BOISSANO"}],"L037":[{"name":"TALEGGIO","end":"9999
-12-31","start":"1861-03-17"}],"M381":[{"end":"9999-12-31","start":"2017-01-
01","name":"TERRE DEL RENO"}],"L108":[{"end":"1923-02-13","start":"1920-10-1
6","name":"TERLANO"},{"end":"1927-01-11","start":"1923-02-14","name":"TERLAN
O"},{"name":"TERLANO","end":"9999-12-31","start":"1927-01-12"}],"I517":[{"st
art":"1861-03-17","end":"1928-12-14","name":"SCAVOLINO"}],"H785":[{"start":"
1861-03-17","end":"1992-04-15","name":"SAN CALOGERO"},{"name":"SAN CALOGERO"
,"start":"1992-04-16","end":"9999-12-31"}],"G233":[{"name":"PAGAZZANO","end"
:"9999-12-31","start":"1861-03-17"}],"B142":[{"name":"BREME","end":"9999-12-
31","start":"1861-03-17"}],"A294":[{"end":"1928-04-27","start":"1861-03-17",
"name":"ANGRI"},{"name":"ANGRI","end":"1929-05-14","start":"1928-04-28"},{"s
tart":"1929-05-15","end":"1946-04-25","name":"ANGRI"},{"name":"ANGRI","start
":"1946-04-26","end":"9999-12-31"}],"A946":[{"end":"9999-12-31","start":"188
3-01-14","name":"BOLOGNETTA"}],"A835":[{"end":"9999-12-31","start":"1861-03-
17","name":"BEVAGNA"}],"G771":[{"end":"1928-05-11","start":"1861-03-17","nam
e":"POGGIO SAN MARCELLO"},{"end":"9999-12-31","start":"1946-11-02","name":"P
OGGIO SAN MARCELLO"}],"A649":[{"end":"1927-01-11","start":"1861-03-17","name
":"BARDINO VECCHIO"},{"name":"BARDINO VECCHIO","end":"1929-01-23","start":"1
927-01-12"}],"D455":[{"name":"FAEDIS","end":"1953-08-31","start":"1866-11-19
"},{"end":"1968-04-05","start":"1953-09-01","name":"FAEDIS"},{"start":"1968-
04-06","end":"9999-12-31","name":"FAEDIS"}],"B617":[{"start":"1861-03-17","e
nd":"9999-12-31","name":"CANOLO"}],"Z609":[{"name":"Isole Falkland (Malvine)
","end":"9999-12-31","start":"0000-01-01"}],"B953":[{"name":"CASCINETTE D'IV
REA","end":"1927-01-11","start":"1925-07-10"},{"name":"CASCINETTE D'IVREA","
start":"1927-01-12","end":"1945-10-04"},{"start":"1945-10-05","end":"9999-12
-31","name":"CASCINETTE D'IVREA"}],"L022":[{"start":"1861-03-17","end":"1992
-04-15","name":"TACENO"},{"end":"9999-12-31","start":"1992-04-16","name":"TA
CENO"}],"C795":[{"name":"CLETO","start":"1863-01-19","end":"1928-05-18"},{"n
ame":"CLETO","start":"1934-03-07","end":"9999-12-31"}],"I222":[{"start":"192
1-01-05","end":"1923-02-07","name":"SANTA LUCIA DI TOLMINO"},{"start":"1923-
02-08","end":"1927-01-11","name":"SANTA LUCIA DI TOLMINO"},{"name":"SANTA LU
CIA DI TOLMINO","start":"1927-01-12","end":"1928-05-11"},{"start":"1928-05-1
2","end":"1935-03-01","name":"SANTA LUCIA DI TOLMINO"},{"end":"1947-09-15","
start":"1935-03-02","name":"SANTA LUCIA D'ISONZO"}],"I391":[{"name":"SAN VIT
ALIANO","end":"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end"
:"9999-12-31","name":"SAN VITALIANO"}],"G359":[{"end":"1918-06-02","start":"
1861-03-17","name":"PASSIGNANO"},{"start":"1918-06-03","end":"1937-03-11","n
ame":"PASSIGNANO SUL TRASIMENO"},{"end":"9999-12-31","start":"1937-03-12","n
ame":"PASSIGNANO SUL TRASIMENO"}],"G053":[{"end":"1923-02-07","start":"1921-
01-05","name":"OLTRESONZIA"},{"end":"1927-01-11","start":"1923-02-08","name"
:"OLTRESONZIA"},{"start":"1927-01-12","end":"1928-05-03","name":"OLTRESONZIA
"}],"C540":[{"name":"CERTALDO","start":"1861-03-17","end":"9999-12-31"}],"G7
87":[{"start":"1863-01-19","end":"9999-12-31","name":"POLIGNANO A MARE"}],"I
829":[{"name":"SONDRIO","end":"9999-12-31","start":"1861-03-17"}],"M004":[{"
start":"1861-03-17","end":"1927-01-11","name":"VILLAREGGIA"},{"start":"1927-
01-12","end":"1928-07-18","name":"VILLAREGGIA"},{"name":"VILLAREGGIA","start
":"1947-01-24","end":"9999-12-31"}],"A484":[{"end":"9999-12-31","start":"188
0-05-24","name":"ATENA LUCANA"}],"I115":[{"name":"SAN PIETRO IN LAMA","end":
"9999-12-31","start":"1861-03-17"}],"G513":[{"start":"1863-05-06","end":"192
7-01-11","name":"PETRELLA SALTO"},{"end":"9999-12-31","start":"1927-01-12","
name":"PETRELLA SALTO"}],"H280":[{"name":"RIESE","end":"1952-03-27","start":
"1866-11-19"},{"name":"RIESE PIO X","end":"9999-12-31","start":"1952-03-28"}
],"I237":[{"name":"SANTA MARIA DELLA VERSA","start":"1893-09-17","end":"1929
-05-29"},{"end":"9999-12-31","start":"1929-05-30","name":"SANTA MARIA DELLA 
VERSA"}],"M422":[{"name":"PIEVE DEL GRAPPA","end":"9999-12-31","start":"2019
-01-30"}],"E983":[{"name":"MARTIGNANA DI PO","start":"1863-01-05","end":"999
9-12-31"}],"A188":[{"name":"ALFIANELLO","end":"9999-12-31","start":"1861-03-
17"}],"B824":[{"end":"1939-07-12","start":"1872-11-01","name":"CURA CARPIGNA
NO"},{"end":"9999-12-31","start":"1939-07-13","name":"CURA CARPIGNANO"}],"B7
01":[{"start":"1863-01-26","end":"1929-06-14","name":"CAPRIATA D'ORBA"},{"st
art":"1929-06-15","end":"9999-12-31","name":"CAPRIATA D'ORBA"}],"M377":[{"na
me":"SAN MARCELLO PITEGLIO","end":"9999-12-31","start":"2017-01-01"}],"L257"
:[{"name":"TORRE DE' BUSI","start":"1861-03-17","end":"1992-04-15"},{"name":
"TORRE DE' BUSI","end":"2018-01-26","start":"1992-04-16"},{"end":"9999-12-31
","start":"2018-01-27","name":"TORRE DE' BUSI"}],"C290":[{"end":"9999-12-31"
,"start":"1861-03-17","name":"CASTELVISCONTI"}],"G273":[{"end":"1929-06-17",
"start":"1861-03-17","name":"PALERMO"},{"name":"PALERMO","end":"9999-12-31",
"start":"1929-06-18"}],"A001":[{"end":"1924-11-13","start":"1866-11-19","nam
e":"ABANO"},{"name":"ABANO TERME","start":"1924-11-14","end":"9999-12-31"}],
"L907":[{"end":"9999-12-31","start":"1863-07-13","name":"VILLA DI CHIAVENNA"
}],"L799":[{"name":"VERVIO","end":"9999-12-31","start":"1861-03-17"}],"H390"
:[{"name":"ROCCAFLUVIONE","start":"1882-03-17","end":"9999-12-31"}],"I416":[
{"end":"9999-12-31","start":"1863-03-30","name":"SAN ZENONE AL PO"}],"L304":
[{"name":"TORTONA","end":"9999-12-31","start":"1861-03-17"}],"F306":[{"name"
:"MOLVENA","start":"1889-07-15","end":"2019-02-19"}],"C380":[{"name":"CAVARE
NO","start":"1920-10-16","end":"1923-02-13"},{"name":"CAVARENO","start":"192
3-02-14","end":"1928-08-06"},{"end":"1953-01-03","start":"1928-08-07","name"
:"CAVARENO"},{"start":"1953-01-04","end":"1964-06-09","name":"CAVARENO"},{"e
nd":"1989-09-05","start":"1964-06-10","name":"CAVARENO"},{"name":"CAVARENO",
"end":"9999-12-31","start":"1989-09-06"}],"G731":[{"name":"PLANINA","end":"1
923-02-07","start":"1921-01-05"},{"start":"1923-02-08","end":"1927-01-11","n
ame":"PLANINA"},{"start":"1927-01-12","end":"1928-07-27","name":"PLANINA"}],
"D355":[{"name":"DOSSO DEL LIRO","start":"1861-03-17","end":"1928-04-30"},{"
end":"9999-12-31","start":"1948-03-21","name":"DOSSO DEL LIRO"}],"E880":[{"e
nd":"1928-12-18","start":"1863-04-23","name":"MANDELLO VITTA"},{"start":"195
5-03-19","end":"9999-12-31","name":"MANDELLO VITTA"}],"I281":[{"name":"SANT'
ANGELO DEI LOMBARDI","start":"1861-03-17","end":"9999-12-31"}],"L077":[{"end
":"1928-04-16","start":"1861-03-17","name":"TAVOLE"}],"F507":[{"start":"1862
-12-29","end":"9999-12-31","name":"MONTEFORTE CILENTO"}],"G746":[{"start":"1
881-04-22","end":"9999-12-31","name":"PODENZANA"}],"A971":[{"start":"1861-03
-17","end":"1970-03-02","name":"BONEFRO"},{"name":"BONEFRO","start":"1970-03
-03","end":"9999-12-31"}],"H546":[{"end":"1923-07-22","start":"1861-03-17","
name":"RONDANINA"},{"end":"9999-12-31","start":"1923-07-23","name":"RONDANIN
A"}],"B088":[{"start":"1861-03-17","end":"9999-12-31","name":"BOTTANUCO"}],"
L199":[{"name":"TOMBOLO","start":"1866-11-19","end":"9999-12-31"}],"G600":[{
"end":"1928-07-28","start":"1861-03-17","name":"PIEDIMULERA"},{"name":"PIEDI
MULERA","start":"1928-07-29","end":"1947-01-12"},{"start":"1947-01-13","end"
:"1992-05-22","name":"PIEDIMULERA"},{"name":"PIEDIMULERA","start":"1992-05-2
3","end":"9999-12-31"}],"G131":[{"name":"ORTA NOVA","start":"1862-11-10","en
d":"1958-01-13"},{"name":"ORTA NOVA","start":"1958-01-14","end":"1975-05-19"
},{"end":"2003-09-03","start":"1975-05-20","name":"ORTA NOVA"},{"start":"200
3-09-04","end":"2008-06-03","name":"ORTA NOVA"},{"start":"2008-06-04","end":
"9999-12-31","name":"ORTA NOVA"}],"F853":[{"name":"NAVE SAN ROCCO","start":"
1920-10-16","end":"1923-02-13"},{"end":"1928-05-22","start":"1923-02-14","na
me":"NAVE SAN ROCCO"},{"name":"NAVE SAN ROCCO","end":"1958-06-17","start":"1
948-03-02"},{"name":"NAVE SAN ROCCO","end":"2018-12-31","start":"1958-06-18"
}],"H347":[{"end":"1968-04-05","start":"1866-11-19","name":"RIVE D'ARCANO"},
{"start":"1968-04-06","end":"2006-11-29","name":"RIVE D'ARCANO"},{"start":"2
006-11-30","end":"9999-12-31","name":"RIVE D'ARCANO"}],"D449":[{"end":"2013-
12-31","start":"1950-08-06","name":"FABBRICHE DI VALLICO"},{"end":"1923-11-2
3","start":"1861-03-17","name":"TRASSILICO"},{"name":"TRASSILICO","end":"194
8-12-09","start":"1923-11-24"},{"end":"1950-08-05","start":"1948-12-10","nam
e":"TRASSILICO"}],"H296":[{"name":"RINCO","end":"1928-11-30","start":"1861-0
3-17"}],"E238":[{"name":"GUARDAMIGLIO","start":"1861-03-17","end":"1992-04-1
5"},{"start":"1992-04-16","end":"1997-07-25","name":"GUARDAMIGLIO"},{"name":
"GUARDAMIGLIO","end":"9999-12-31","start":"1997-07-26"}],"A655":[{"name":"BA
RESSA","end":"1927-10-08","start":"1861-03-17"},{"name":"BARESSA","start":"1
927-10-09","end":"1928-05-30"},{"start":"1928-05-31","end":"1946-02-06","nam
e":"BARESSA"},{"end":"1947-10-16","start":"1946-02-07","name":"BARESSA"},{"n
ame":"BARESSA","start":"1947-10-17","end":"1958-08-09"},{"name":"BARESSA","e
nd":"1974-08-19","start":"1958-08-10"},{"end":"9999-12-31","start":"1974-08-
20","name":"BARESSA"}],"H531":[{"name":"RONCHI","start":"1921-01-05","end":"
1923-02-07"},{"start":"1923-02-08","end":"1923-04-26","name":"RONCHI"},{"nam
e":"RONCHI DEI LEGIONARI","end":"1948-01-09","start":"1925-11-17"},{"end":"9
999-12-31","start":"1948-01-10","name":"RONCHI DEI LEGIONARI"},{"end":"1925-
11-16","start":"1923-04-27","name":"RONCHI DI MONFALCONE"}],"D263":[{"start"
:"1920-10-16","end":"1923-02-13","name":"DEGGIANO"},{"name":"DEGGIANO","end"
:"1928-10-06","start":"1923-02-14"}],"F200":[{"start":"1861-03-17","end":"99
99-12-31","name":"MIGLIERINA"}],"G146":[{"name":"ORTUERI","end":"1927-01-11"
,"start":"1861-03-17"},{"start":"1927-01-12","end":"9999-12-31","name":"ORTU
ERI"}],"H944":[{"name":"SANGUINETTO","end":"1929-05-29","start":"1866-11-19"
},{"name":"SANGUINETTO","end":"1948-03-30","start":"1929-05-30"},{"name":"SA
NGUINETTO","start":"1948-03-31","end":"9999-12-31"}],"C286":[{"start":"1861-
03-17","end":"1955-02-14","name":"CASTELVETRANO"},{"start":"1955-02-15","end
":"9999-12-31","name":"CASTELVETRANO"}],"M193":[{"name":"ZORZINO","end":"192
8-06-14","start":"1861-03-17"}],"A869":[{"name":"BIGLIO","end":"1927-10-09",
"start":"1861-03-17"}],"G445":[{"name":"PERDASDEFOGU","start":"1861-03-17","
end":"1927-01-11"},{"name":"PERDASDEFOGU","end":"2005-12-31","start":"1927-0
1-12"},{"start":"2006-01-01","end":"2016-04-27","name":"PERDASDEFOGU"},{"sta
rt":"2016-04-28","end":"9999-12-31","name":"PERDASDEFOGU"}],"F372":[{"name":
"MONGUZZO","end":"9999-12-31","start":"1861-03-17"}],"F950":[{"end":"1923-02
-13","start":"1920-10-16","name":"NOVA PONENTE"},{"start":"1923-02-14","end"
:"1927-01-11","name":"NOVA PONENTE"},{"name":"NOVA PONENTE","end":"2011-04-1
3","start":"1927-01-12"},{"name":"NOVA PONENTE","end":"9999-12-31","start":"
2011-04-14"}],"H386":[{"name":"ROCCA CANAVESE","end":"9999-12-31","start":"1
884-05-02"}],"H713":[{"name":"SALISANO","end":"1923-03-18","start":"1861-03-
17"},{"name":"SALISANO","end":"1927-01-11","start":"1923-03-19"},{"end":"999
9-12-31","start":"1927-01-12","name":"SALISANO"}],"C247":[{"end":"1970-03-02
","start":"1861-03-17","name":"CASTELPIZZUTO"},{"start":"1970-03-03","end":"
9999-12-31","name":"CASTELPIZZUTO"}],"C396":[{"name":"CAVERNAGO","end":"9999
-12-31","start":"1861-03-17"}],"L010":[{"name":"SURANO","end":"9999-12-31","
start":"1861-03-17"}],"G187":[{"name":"OSTUNI","start":"1861-03-17","end":"1
927-01-11"},{"name":"OSTUNI","start":"1927-01-12","end":"1935-06-04"},{"star
t":"1935-06-05","end":"9999-12-31","name":"OSTUNI"}],"A262":[{"end":"1927-01
-11","start":"1876-06-30","name":"AMELIA"},{"name":"AMELIA","end":"9999-12-3
1","start":"1927-01-12"}],"F427":[{"name":"MONTANO LUCINO","start":"1928-06-
15","end":"9999-12-31"}],"E896":[{"name":"MANTELLO","start":"1861-03-17","en
d":"9999-12-31"}],"H838":[{"name":"SAN FELICE DEL BENACO","start":"1928-06-3
0","end":"9999-12-31"}],"I715":[{"start":"1921-01-05","end":"1923-02-07","na
me":"SGONICO"},{"start":"1923-02-08","end":"2009-01-28","name":"SGONICO"},{"
name":"SGONICO-ZGONIK","start":"2009-01-29","end":"9999-12-31"}],"H587":[{"e
nd":"1923-06-28","start":"1861-03-17","name":"ROTA FUORI"},{"start":"1923-06
-29","end":"1927-06-02","name":"ROTA FUORI"}],"C625":[{"name":"CHIENES","end
":"1923-02-13","start":"1920-10-16"},{"name":"CHIENES","end":"1927-01-11","s
tart":"1923-02-14"},{"end":"1929-03-23","start":"1927-01-12","name":"CHIENES
"},{"end":"9999-12-31","start":"1929-03-24","name":"CHIENES"}],"I530":[{"nam
e":"SCHILPARIO","start":"1861-03-17","end":"9999-12-31"}],"A788":[{"name":"B
ERCETO","end":"9999-12-31","start":"1891-11-13"}],"B630":[{"name":"CANTALUPO
 NEL SANNIO","start":"1864-03-11","end":"1970-03-02"},{"start":"1970-03-03",
"end":"9999-12-31","name":"CANTALUPO NEL SANNIO"}],"F432":[{"start":"1861-03
-17","end":"9999-12-31","name":"MONTAURO"}],"B101":[{"name":"BOVES","start":
"1861-03-17","end":"9999-12-31"}],"L550":[{"end":"1923-02-13","start":"1920-
10-16","name":"VALDA"},{"start":"1923-02-14","end":"1928-09-05","name":"VALD
A"},{"name":"VALDA","end":"2015-12-31","start":"1952-05-17"}],"C597":[{"star
t":"1861-03-17","end":"1928-12-13","name":"CHAMPLAS-DU-COL"}],"M413":[{"name
":"SASSOCORVARO AUDITORE","end":"9999-12-31","start":"2019-01-01"}],"G336":[
{"name":"PARLASCO","start":"1861-03-17","end":"1992-04-15"},{"name":"PARLASC
O","start":"1992-04-16","end":"9999-12-31"}],"F701":[{"end":"9999-12-31","st
art":"1861-03-17","name":"MONT\u00d9 BECCARIA"}],"G522":[{"start":"1861-03-1
7","end":"9999-12-31","name":"PETTINEO"}],"L826":[{"start":"1868-07-01","end
":"9999-12-31","name":"VIADANA"}],"I180":[{"end":"1923-02-07","start":"1921-
01-05","name":"SANTA CROCE DI AIDUSSINA"},{"end":"1927-01-11","start":"1923-
02-08","name":"SANTA CROCE DI AIDUSSINA"},{"name":"SANTA CROCE DI AIDUSSINA"
,"start":"1927-01-12","end":"1928-03-12"},{"name":"SANTA CROCE DI AIDUSSINA"
,"end":"1947-09-15","start":"1928-03-13"}],"H978":[{"end":"1927-01-11","star
t":"1861-03-17","name":"SAN MARCELLINO"},{"name":"SAN MARCELLINO","start":"1
927-01-12","end":"1929-04-12"},{"name":"SAN MARCELLINO","end":"9999-12-31","
start":"1946-12-23"}],"G769":[{"name":"POGGIORSINI","start":"1957-05-15","en
d":"9999-12-31"}],"H008":[{"end":"1929-03-11","start":"1861-03-17","name":"P
RATOVECCHIO"},{"end":"2013-12-31","start":"1934-09-27","name":"PRATOVECCHIO"
}],"G341":[{"name":"PARONA ALL'ADIGE","start":"1867-08-26","end":"1905-03-19
"},{"start":"1905-03-20","end":"1927-05-11","name":"PARONA DI VALPOLICELLA"}
],"E064":[{"end":"1903-02-01","start":"1861-03-17","name":"GIUSTENICE"},{"en
d":"1927-01-11","start":"1903-02-02","name":"GIUSTENICE"},{"start":"1927-01-
12","end":"9999-12-31","name":"GIUSTENICE"}],"I213":[{"start":"1863-03-30","
end":"1929-05-13","name":"SANT'ALESSIO CON VIALONE"},{"start":"1947-11-28","
end":"9999-12-31","name":"SANT'ALESSIO CON VIALONE"}],"G537":[{"end":"1998-0
9-03","start":"1861-03-17","name":"PIAGGE"},{"end":"2016-12-31","start":"199
8-09-04","name":"PIAGGE"}],"D179":[{"name":"CROGNALETO","start":"1861-03-17"
,"end":"9999-12-31"}],"G888":[{"name":"PORDENONE","start":"1866-11-19","end"
:"1930-04-26"},{"name":"PORDENONE","start":"1930-04-27","end":"1968-04-05"},
{"start":"1968-04-06","end":"9999-12-31","name":"PORDENONE"}],"G476":[{"name
":"PERTOSA","end":"9999-12-31","start":"1861-03-17"}],"D040":[{"name":"CORRE
ZZOLA","end":"9999-12-31","start":"1866-11-19"}],"H122":[{"name":"QUATTRO CA
STELLA","start":"1872-03-03","end":"9999-12-31"}],"G175":[{"name":"OSSECCA V
ITTUGLIE","start":"1923-04-27","end":"1927-01-11"},{"start":"1927-01-12","en
d":"1928-01-06","name":"OSSECCA VITTUGLIE"},{"end":"1923-02-07","start":"192
1-01-05","name":"OSSECH-VITTUGLIE"},{"end":"1923-04-26","start":"1923-02-08"
,"name":"OSSECH-VITTUGLIE"}],"I573":[{"name":"SEGNI","end":"9999-12-31","sta
rt":"1871-01-15"}],"L419":[{"name":"TRICASE","start":"1874-10-08","end":"999
9-12-31"}],"C191":[{"name":"CASTELLO DI SERRAVALLE","start":"1862-12-19","en
d":"2013-12-31"}],"B507":[{"start":"1864-06-23","end":"1928-12-15","name":"C
AMPI BISENZIO"},{"name":"CAMPI BISENZIO","start":"1928-12-16","end":"9999-12
-31"}],"Z519":[{"end":"9999-12-31","start":"0000-01-01","name":"Isole Turks 
e Caicos"}],"E815":[{"name":"MAGLIE","start":"1861-03-17","end":"9999-12-31"
}],"H181":[{"end":"1923-02-13","start":"1920-10-16","name":"RANZO"},{"name":
"RANZO","start":"1923-02-14","end":"1928-04-18"}],"I796":[{"start":"1861-03-
17","end":"1923-12-21","name":"SOLDANO"},{"name":"SOLDANO","end":"1928-05-22
","start":"1925-08-22"},{"start":"1946-12-23","end":"9999-12-31","name":"SOL
DANO"}],"I607":[{"start":"1861-03-17","end":"9999-12-31","name":"SENIGA"}],"
B673":[{"name":"CAPORETTO","end":"1923-02-07","start":"1921-01-05"},{"start"
:"1923-02-08","end":"1927-01-11","name":"CAPORETTO"},{"start":"1927-01-12","
end":"1928-05-01","name":"CAPORETTO"},{"end":"1947-09-15","start":"1928-05-0
2","name":"CAPORETTO"}],"Z353":[{"name":"Uganda","end":"9999-12-31","start":
"0000-01-01"}],"G581":[{"name":"PIAZZA SANTO STEFANO","end":"1929-05-02","st
art":"1863-02-23"}],"C315":[{"start":"1871-01-15","end":"1927-01-11","name":
"CASTIGLIONE IN TEVERINA"},{"start":"1927-01-12","end":"9999-12-31","name":"
CASTIGLIONE IN TEVERINA"}],"E129":[{"name":"GRAGLIA PIANA","start":"1862-12-
29","end":"1928-09-25"}],"B306":[{"end":"1935-04-07","start":"1861-03-17","n
ame":"BUTTIGLIERA D'ASTI"},{"name":"BUTTIGLIERA D'ASTI","end":"9999-12-31","
start":"1935-04-08"}],"H575":[{"name":"ROSORA","start":"1861-03-17","end":"1
928-07-27"},{"end":"9999-12-31","start":"1946-03-08","name":"ROSORA"}],"D765
":[{"start":"1863-04-10","end":"9999-12-31","name":"FRANCAVILLA DI SICILIA"}
],"E911":[{"end":"9999-12-31","start":"1867-08-26","name":"MARANO DI VALPOLI
CELLA"}],"Z614":[{"name":"Venezuela","end":"9999-12-31","start":"0000-01-01"
}],"G775":[{"start":"1861-03-17","end":"1928-08-12","name":"POGNO"},{"start"
:"1947-01-15","end":"9999-12-31","name":"POGNO"}],"A831":[{"name":"BETTOLA",
"end":"9999-12-31","start":"1885-04-27"}],"C791":[{"end":"1968-04-05","start
":"1866-11-19","name":"CLAUZETTO"},{"name":"CLAUZETTO","end":"2009-04-15","s
tart":"1968-04-06"},{"name":"CLAUZETTO","start":"2009-04-16","end":"9999-12-
31"}],"B200":[{"end":"1928-05-02","start":"1861-03-17","name":"BRONDELLO"},{
"name":"BRONDELLO","end":"9999-12-31","start":"1946-11-12"}],"D451":[{"start
":"1861-03-17","end":"1937-05-28","name":"FABRIANO"},{"name":"FABRIANO","end
":"9999-12-31","start":"1937-05-29"}],"H781":[{"start":"1877-02-15","end":"1
902-02-06","name":"SAN BIAGIO DI CALLALTA"},{"start":"1902-02-07","end":"190
7-05-12","name":"SAN BIAGIO DI CALLALTA"},{"name":"SAN BIAGIO DI CALLALTA","
start":"1907-05-13","end":"9999-12-31"}],"I196":[{"start":"1863-04-13","end"
:"9999-12-31","name":"SANT'AGATA SUL SANTERNO"}],"G024":[{"name":"OLGIA","st
art":"1861-03-17","end":"1928-06-30"}],"A846":[{"start":"1861-03-17","end":"
1928-07-29","name":"BIANZANO"},{"name":"BIANZANO","start":"1955-07-10","end"
:"9999-12-31"}],"A935":[{"start":"1920-10-16","end":"1923-02-13","name":"BOL
ENTINA"},{"name":"BOLENTINA","end":"1928-09-05","start":"1923-02-14"}],"E729
":[{"name":"LUGNANO IN TEVERINA","start":"1863-04-13","end":"1927-01-11"},{"
start":"1927-01-12","end":"9999-12-31","name":"LUGNANO IN TEVERINA"}],"G402"
:[{"end":"9999-12-31","start":"1861-03-17","name":"PEDARA"}],"B853":[{"name"
:"CARUNCHIO","end":"9999-12-31","start":"1861-03-17"}],"M385":[{"name":"CASA
LI DEL MANCO","end":"9999-12-31","start":"2017-05-05"}],"D165":[{"start":"18
61-03-17","end":"1927-01-11","name":"CREVACUORE"},{"name":"CREVACUORE","star
t":"1927-01-12","end":"1992-04-15"},{"name":"CREVACUORE","end":"9999-12-31",
"start":"1992-04-16"}],"E562":[{"start":"1861-03-17","end":"9999-12-31","nam
e":"LEVATE"}],"F088":[{"name":"MEDUNA DI LIVENZA","start":"1884-08-23","end"
:"9999-12-31"}],"D466":[{"name":"FAGNANO","start":"1861-03-17","end":"1863-0
1-11"},{"end":"1869-03-21","start":"1863-01-12","name":"FAGNANO SUL NAVIGLIO
"}],"M295":[{"name":"SAN CESAREO","start":"1990-04-10","end":"9999-12-31"}],
"A538":[{"end":"1928-07-31","start":"1861-03-17","name":"BADIA"},{"name":"BA
DIA PAVESE","end":"9999-12-31","start":"1928-08-01"}],"C483":[{"name":"CERAN
O","start":"1861-03-17","end":"9999-12-31"}],"B432":[{"name":"CALTO","end":"
9999-12-31","start":"1866-11-19"}],"F101":[{"name":"MELENDUGNO","end":"9999-
12-31","start":"1861-03-17"}],"A887":[{"start":"1861-03-17","end":"9999-12-3
1","name":"BISIGNANO"}],"I780":[{"end":"1979-01-13","start":"1862-11-28","na
me":"SOGLIANO CAVOUR"},{"name":"SOGLIANO CAVOUR","end":"2004-05-11","start":
"1979-01-14"},{"start":"2004-05-12","end":"9999-12-31","name":"SOGLIANO CAVO
UR"}],"H493":[{"name":"ROGOLO","end":"9999-12-31","start":"1861-03-17"}],"D0
97":[{"start":"1861-03-17","end":"1927-01-11","name":"COSSILA"},{"name":"COS
SILA","start":"1927-01-12","end":"1940-08-02"}],"H664":[{"end":"1923-02-07",
"start":"1921-01-05","name":"SAGORIA SAN MARTINO"},{"start":"1923-02-08","en
d":"1924-03-08","name":"SAGORIA SAN MARTINO"},{"name":"SAGORIA SAN MARTINO",
"end":"1927-10-21","start":"1924-03-09"}],"H737":[{"name":"SAMARIA","end":"1
923-02-07","start":"1921-01-05"},{"name":"SAMARIA","start":"1923-02-08","end
":"1927-01-11"},{"end":"1928-01-25","start":"1927-01-12","name":"SAMARIA"}],
"G169":[{"end":"1949-09-08","start":"1866-11-19","name":"OSPITALE"},{"start"
:"1949-09-09","end":"9999-12-31","name":"OSPITALE DI CADORE"}],"E608":[{"sta
rt":"1861-03-17","end":"9999-12-31","name":"LIRIO"}],"I495":[{"start":"1861-
03-17","end":"1928-11-29","name":"SCANDELUZZA"},{"name":"SCANDELUZZA","start
":"1928-11-30","end":"1935-04-07"},{"name":"SCANDELUZZA","end":"1947-12-23",
"start":"1935-04-08"},{"start":"1947-12-24","end":"1998-08-31","name":"SCAND
ELUZZA"}],"L085":[{"end":"9999-12-31","start":"1868-02-14","name":"TEGLIO VE
NETO"}],"G230":[{"end":"9999-12-31","start":"1861-03-17","name":"PAGANI"}],"
B427":[{"name":"CALTABELLOTTA","start":"1861-03-17","end":"9999-12-31"}],"D7
79":[{"name":"FRASSINERE","end":"1936-07-07","start":"1861-03-17"}],"H569":[
{"name":"ROSIGNANO MONFERRATO","start":"1863-02-16","end":"9999-12-31"}],"B3
72":[{"start":"1861-03-17","end":"1924-01-14","name":"CAJELLO"}],"H722":[{"n
ame":"SALTO","end":"1927-01-11","start":"1861-03-17"},{"name":"SALTO","end":
"1928-09-25","start":"1927-01-12"}],"B950":[{"name":"CASCINA","end":"1929-06
-17","start":"1861-03-17"},{"end":"9999-12-31","start":"1929-06-18","name":"
CASCINA"}],"A218":[{"start":"1861-03-17","end":"1928-03-05","name":"ALMESE"}
,{"end":"1955-07-09","start":"1928-03-06","name":"ALMESE"},{"name":"ALMESE",
"start":"1955-07-10","end":"9999-12-31"}],"A975":[{"name":"BONITO","end":"99
99-12-31","start":"1861-03-17"}],"G050":[{"start":"1861-03-17","end":"9999-1
2-31","name":"OLTRE IL COLLE"}],"E092":[{"end":"9999-12-31","start":"1886-03
-01","name":"GORGO AL MONTICANO"}],"I285":[{"start":"1861-03-17","end":"1998
-09-03","name":"SANT'ANGELO IN LIZZOLA"},{"end":"2013-12-31","start":"1998-0
9-04","name":"SANT'ANGELO IN LIZZOLA"}],"D351":[{"name":"DOSOLO","start":"18
68-07-01","end":"9999-12-31"}],"H110":[{"start":"1861-03-17","end":"1927-01-
11","name":"QUART"},{"name":"QUART","end":"1928-10-15","start":"1927-01-12"}
,{"end":"1929-04-11","start":"1928-10-16","name":"QUART"},{"end":"1946-03-24
","start":"1946-01-30","name":"QUART"},{"name":"QUART","start":"1946-03-25",
"end":"1946-05-14"},{"start":"1946-05-15","end":"9999-12-31","name":"QUART"}
,{"end":"1946-01-29","start":"1929-04-12","name":"QUARTO PRAETORIA"}],"C543"
:[{"start":"1881-06-24","end":"9999-12-31","name":"CERVARA DI ROMA"}],"L807"
:[{"end":"1935-04-07","start":"1861-03-17","name":"VESIME"},{"name":"VESIME"
,"start":"1935-04-08","end":"9999-12-31"}],"F742":[{"name":"MOROSOLO","end":
"1927-01-11","start":"1861-03-17"},{"start":"1927-01-12","end":"1929-06-18",
"name":"MOROSOLO"}],"H283":[{"name":"REIFEMBERG","end":"1923-02-07","start":
"1921-01-05"},{"start":"1923-02-08","end":"1923-04-26","name":"REIFEMBERG"},
{"name":"RIFEMBERGO","end":"1927-01-11","start":"1923-04-27"},{"name":"RIFEM
BERGO","start":"1927-01-12","end":"1928-01-25"},{"name":"RIFEMBERGO","start"
:"1928-01-26","end":"1947-09-15"}],"G510":[{"end":"1972-04-04","start":"1861
-03-17","name":"PETRALIA SOPRANA"},{"name":"PETRALIA SOPRANA","end":"9999-12
-31","start":"1972-04-05"}],"G735":[{"name":"PLAT\u00cc","start":"1895-04-05
","end":"9999-12-31"}],"M147":[{"end":"9999-12-31","start":"1861-03-17","nam
e":"ZANICA"}],"A871":[{"name":"BINANUOVA","start":"1861-03-17","end":"1928-0
4-06"}],"E980":[{"start":"1866-11-19","end":"9999-12-31","name":"MARTELLAGO"
}],"G302":[{"end":"9999-12-31","start":"1861-03-17","name":"PAMPARATO"}],"A0
05":[{"start":"1928-03-06","end":"1992-04-15","name":"ABBADIA LARIANA"},{"na
me":"ABBADIA LARIANA","start":"1992-04-16","end":"9999-12-31"}],"C293":[{"na
me":"CASTENEDOLO","start":"1861-03-17","end":"9999-12-31"}],"I626":[{"name":
"SEREN","start":"1866-11-19","end":"1923-05-20"},{"end":"9999-12-31","start"
:"1923-05-21","name":"SEREN DEL GRAPPA"}],"G270":[{"name":"PALAZZUOLO DI ROM
AGNA","end":"1951-09-26","start":"1861-03-17"},{"end":"9999-12-31","start":"
1951-09-27","name":"PALAZZUOLO SUL SENIO"}],"D739":[{"end":"1868-01-01","sta
rt":"1861-03-17","name":"FOSSA GUAZZONA"}],"M186":[{"end":"1923-02-07","star
t":"1921-01-05","name":"ZOLLA"},{"name":"ZOLLA","start":"1923-02-08","end":"
1927-01-11"},{"name":"ZOLLA","end":"1928-01-25","start":"1927-01-12"},{"star
t":"1928-01-26","end":"1947-09-15","name":"ZOLLA"}],"I929":[{"end":"9999-12-
31","start":"1861-03-17","name":"SQUILLACE"}],"H099":[{"end":"1928-04-14","s
tart":"1861-03-17","name":"QUAGLIETTA"}],"H393":[{"end":"1923-12-20","start"
:"1861-03-17","name":"ROCCA D'ARCE"},{"name":"ROCCA D'ARCE","start":"1923-12
-21","end":"1927-01-11"},{"name":"ROCCA D'ARCE","end":"9999-12-31","start":"
1927-01-12"}],"B705":[{"name":"CAPRIE","end":"9999-12-31","start":"1937-06-1
6"},{"name":"CHIAVRIE","end":"1937-06-15","start":"1861-03-17"}],"C716":[{"n
ame":"CIORLANO","start":"1898-04-15","end":"1927-01-11"},{"start":"1927-01-1
2","end":"1945-08-31","name":"CIORLANO"},{"start":"1945-09-01","end":"9999-1
2-31","name":"CIORLANO"}],"C383":[{"name":"CAVARZERE","start":"1866-11-19","
end":"9999-12-31"}],"C089":[{"name":"CASTELDIDONE","end":"9999-12-31","start
":"1861-03-17"}],"B332":[{"start":"1861-03-17","end":"9999-12-31","name":"CA
DEO"}],"D366":[{"end":"1968-04-05","start":"1866-11-19","name":"DRENCHIA"},{
"end":"9999-12-31","start":"1968-04-06","name":"DRENCHIA"}],"B526":[{"end":"
1904-06-05","start":"1861-03-17","name":"CAMPO DI GIOVE"},{"end":"9999-12-31
","start":"1904-06-06","name":"CAMPO DI GIOVE"}],"A578":[{"start":"1861-03-1
7","end":"9999-12-31","name":"BAGOLINO"}],"B472":[{"end":"9999-12-31","start
":"1872-12-27","name":"CAMERATA NUOVA"}],"E883":[{"name":"MANERBA","start":"
1861-03-17","end":"1953-02-02"},{"start":"1953-02-03","end":"9999-12-31","na
me":"MANERBA DEL GARDA"}],"B924":[{"name":"CASAMICCIOLA","start":"1861-03-17
","end":"1938-11-13"},{"start":"1945-09-22","end":"1956-08-14","name":"CASAM
ICCIOLA"},{"name":"CASAMICCIOLA TERME","start":"1956-08-15","end":"9999-12-3
1"}],"F670":[{"end":"9999-12-31","start":"1863-03-30","name":"MONTICELLI PAV
ESE"}],"H777":[{"start":"1928-10-28","end":"1992-05-22","name":"SAN BERNARDI
NO VERBANO"},{"start":"1992-05-23","end":"9999-12-31","name":"SAN BERNARDINO
 VERBANO"}],"D567":[{"start":"1861-03-17","end":"9999-12-31","name":"FICARAZ
ZI"}],"I395":[{"end":"1862-12-18","start":"1861-03-17","name":"SAN VITO"},{"
start":"1862-12-19","end":"1867-09-05","name":"SAN VITO E MODESTO"}],"G577":
[{"name":"PIATTO","end":"1927-01-11","start":"1861-03-17"},{"name":"PIATTO",
"start":"1927-01-12","end":"1992-04-15"},{"name":"PIATTO","end":"9999-12-31"
,"start":"1992-04-16"}],"E594":[{"end":"9999-12-31","start":"1861-03-17","na
me":"LIMINA"}],"D139":[{"name":"CREDARO","start":"1861-03-17","end":"9999-12
-31"}],"B105":[{"name":"BOVISIO","end":"1947-08-07","start":"1861-03-17"},{"
end":"2009-06-29","start":"1947-08-08","name":"BOVISIO-MASCIAGO"},{"name":"B
OVISIO-MASCIAGO","end":"9999-12-31","start":"2009-06-30"}],"C116":[{"name":"
CASTEL GANDOLFO","end":"9999-12-31","start":"1871-01-15"}],"A666":[{"name":"
BARILE","end":"9999-12-31","start":"1861-03-17"}],"A969":[{"start":"1861-03-
17","end":"1928-02-15","name":"BONDO PETELLO"}],"I711":[{"name":"SEZZADIO","
end":"1928-12-31","start":"1916-11-10"},{"name":"SEZZADIO","end":"1954-01-23
","start":"1929-01-01"},{"end":"9999-12-31","start":"1954-01-24","name":"SEZ
ZADIO"},{"end":"1916-11-09","start":"1861-03-17","name":"SEZZ\u00c8"}],"G603
":[{"end":"1931-09-23","start":"1861-03-17","name":"PIERANICA"},{"end":"1947
-10-20","start":"1931-09-24","name":"PIERANICA"},{"name":"PIERANICA","end":"
9999-12-31","start":"1947-10-21"}],"M313":[{"end":"9999-12-31","start":"2010
-01-01","name":"LEDRO"}],"C621":[{"start":"1861-03-17","end":"9999-12-31","n
ame":"CHIAVARI"}],"M019":[{"name":"VILLA SAN SECONDO","end":"1928-07-18","st
art":"1861-03-17"},{"start":"1947-10-20","end":"9999-12-31","name":"VILLA SA
N SECONDO"}],"F850":[{"name":"NAVA","start":"1861-03-17","end":"1928-02-05"}
],"G436":[{"name":"PENNA SAN GIOVANNI","start":"1861-03-17","end":"9999-12-3
1"}],"G376":[{"end":"9999-12-31","start":"1876-01-10","name":"PATTADA"}],"E2
44":[{"start":"1861-03-17","end":"1970-03-02","name":"GUARDIALFIERA"},{"name
":"GUARDIALFIERA","start":"1970-03-03","end":"9999-12-31"}],"H938":[{"name":
"SAN GREGORIO NELLE ALPI","start":"1867-04-12","end":"9999-12-31"}],"H177":[
{"name":"RANZANICO","start":"1861-03-17","end":"1928-07-28"},{"name":"RANZAN
ICO","start":"1929-05-23","end":"9999-12-31"}],"D260":[{"end":"1974-08-19","
start":"1861-03-17","name":"DECIMOPUTZU"},{"end":"2016-04-27","start":"1974-
08-20","name":"DECIMOPUTZU"},{"name":"DECIMOPUTZU","end":"9999-12-31","start
":"2016-04-28"}],"E699":[{"name":"LOTTULO","end":"1929-01-21","start":"1861-
03-17"}],"Z160":[{"end":"9999-12-31","start":"0000-01-01","name":"Kosovo"}],
"F203":[{"end":"1927-01-11","start":"1861-03-17","name":"MIGNANO"},{"end":"1
945-08-31","start":"1927-01-12","name":"MIGNANO"},{"name":"MIGNANO","end":"1
948-01-17","start":"1945-09-01"},{"name":"MIGNANO MONTE LUNGO","start":"1948
-01-18","end":"9999-12-31"}],"G441":[{"end":"9999-12-31","start":"1861-03-17
","name":"PERANO"}],"M190":[{"name":"ZOPPOLA","start":"1866-11-19","end":"19
68-04-05"},{"start":"1968-04-06","end":"9999-12-31","name":"ZOPPOLA"}],"F617
":[{"name":"MONTE SAN CANDIDO","start":"1920-10-16","end":"1923-02-13"},{"na
me":"MONTE SAN CANDIDO","start":"1923-02-14","end":"1927-01-11"},{"start":"1
927-01-12","end":"1928-12-09","name":"MONTE SAN CANDIDO"}],"F953":[{"end":"1
927-01-11","start":"1861-03-17","name":"NOVAREGLIA"},{"end":"1929-05-28","st
art":"1927-01-12","name":"NOVAREGLIA"}],"L319":[{"start":"1861-03-17","end":
"1927-01-11","name":"TRADATE"},{"name":"TRADATE","end":"1928-06-18","start":
"1927-01-12"},{"end":"9999-12-31","start":"1928-06-19","name":"TRADATE"}],"L
013":[{"name":"SUSA","end":"9999-12-31","start":"1861-03-17"}],"H710":[{"nam
e":"SALICETO","end":"9999-12-31","start":"1861-03-17"}],"H535":[{"name":"RON
COBELLO","end":"1927-11-23","start":"1863-07-13"},{"start":"1927-11-24","end
":"9999-12-31","name":"RONCOBELLO"}],"A651":[{"name":"BARDONECCHIA","start":
"1861-03-17","end":"1927-07-26"},{"name":"BARDONECCHIA","end":"1947-09-15","
start":"1927-07-27"},{"name":"BARDONECCHIA","start":"1947-09-16","end":"9999
-12-31"}],"G217":[{"end":"9999-12-31","start":"1862-12-19","name":"PADERNO F
RANCIACORTA"}],"G135":[{"name":"ORTE","start":"1871-01-15","end":"1927-01-11
"},{"name":"ORTE","start":"1927-01-12","end":"1927-10-21"},{"start":"1927-10
-22","end":"1958-10-24","name":"ORTE"},{"end":"9999-12-31","start":"1958-10-
25","name":"ORTE"}],"I533":[{"name":"SCIACCA","start":"1861-03-17","end":"99
99-12-31"}],"H844":[{"start":"1861-03-17","end":"1992-04-15","name":"SAN FIO
RANO"},{"end":"9999-12-31","start":"1992-04-16","name":"SAN FIORANO"}],"H692
":[{"start":"1921-01-05","end":"1923-02-07","name":"SALCANO"},{"end":"1927-0
1-11","start":"1923-02-08","name":"SALCANO"},{"name":"SALCANO","start":"1927
-01-12","end":"1927-01-12"}],"B633":[{"name":"CANTARANA","start":"1861-03-17
","end":"1928-08-12"},{"start":"1947-06-23","end":"9999-12-31","name":"CANTA
RANA"}],"C682":[{"end":"1928-02-15","start":"1861-03-17","name":"CIGNANO"}],
"C415":[{"name":"CECINA","start":"1892-06-20","end":"1906-04-17"},{"start":"
1906-04-18","end":"1925-11-29","name":"CECINA"},{"name":"CECINA","start":"19
25-11-30","end":"9999-12-31"}],"L553":[{"name":"SUSGNEVIZZA","start":"1922-0
2-02","end":"1922-11-29"},{"name":"SUSGNEVIZZA","start":"1922-11-30","end":"
1923-02-07"},{"name":"SUSGNEVIZZA","end":"1923-04-26","start":"1923-02-08"},
{"start":"1923-04-27","end":"1947-09-15","name":"VALDARSA"}],"B406":[{"name"
:"CALENZANO","start":"1861-03-17","end":"9999-12-31"}],"G712":[{"end":"1927-
01-11","start":"1861-03-17","name":"PISTICCI"},{"name":"PISTICCI","end":"193
3-01-31","start":"1927-01-12"},{"end":"9999-12-31","start":"1933-02-01","nam
e":"PISTICCI"}],"H206":[{"name":"REANA DEL ROJALE","end":"1968-04-05","start
":"1867-09-02"},{"name":"REANA DEL ROJALE","end":"9999-12-31","start":"1968-
04-06"}],"H865":[{"start":"1862-11-24","end":"9999-12-31","name":"SAN GERVAS
IO BRESCIANO"}],"F290":[{"name":"MOLINI DI TRIORA","end":"9999-12-31","start
":"1904-01-22"}],"I863":[{"end":"9999-12-31","start":"1861-03-17","name":"SO
RSO"}],"L109":[{"name":"TERLIZZI","start":"1861-03-17","end":"9999-12-31"}],
"E652":[{"start":"1861-03-17","end":"9999-12-31","name":"LODRINO"}],"I442":[
{"start":"1861-03-17","end":"1927-01-11","name":"SARRE"},{"name":"SARRE","en
d":"1928-03-20","start":"1927-01-12"},{"name":"SARRE","start":"1946-05-15","
end":"9999-12-31"}],"A648":[{"name":"BARDINO NUOVO","end":"1927-01-11","star
t":"1861-03-17"},{"start":"1927-01-12","end":"1929-01-23","name":"BARDINO NU
OVO"}],"H961":[{"name":"SAN LORENZO BELLIZZI","end":"9999-12-31","start":"18
61-03-17"}],"F540":[{"end":"9999-12-31","start":"1863-04-13","name":"MONTELE
ONE DI SPOLETO"}],"G690":[{"name":"PIORACO","start":"1861-03-17","end":"9999
-12-31"}],"Z608":[{"start":"0000-01-01","end":"9999-12-31","name":"Suriname"
}],"C673":[{"name":"CICAGNA","start":"1861-03-17","end":"9999-12-31"}],"Z729
":[{"start":"0000-01-01","end":"9999-12-31","name":"Wallis e Futuna"}],"H554
":[{"name":"ROR\u00c0","start":"1861-03-17","end":"1928-03-26"},{"end":"9999
-12-31","start":"1947-01-09","name":"ROR\u00c0"}],"E806":[{"name":"MAGISANO"
,"end":"1946-02-27","start":"1861-03-17"},{"end":"1957-01-17","start":"1946-
02-28","name":"MAGISANO"},{"name":"MAGISANO","start":"1957-01-18","end":"999
9-12-31"}],"G458":[{"name":"PERLO","start":"1861-03-17","end":"1928-05-01"},
{"name":"PERLO","end":"9999-12-31","start":"1947-01-10"}],"L462":[{"end":"19
28-08-23","start":"1861-03-17","name":"TUGLIE"},{"name":"TUGLIE","end":"9999
-12-31","start":"1928-08-24"}],"C507":[{"start":"1863-02-23","end":"1930-11-
05","name":"CERETTO GRUE"},{"start":"1930-11-06","end":"9999-12-31","name":"
CERRETO GRUE"}],"B191":[{"start":"1928-01-07","end":"1953-07-20","name":"BRI
SSAGO-VALTRAVAGLIA"},{"start":"1953-07-21","end":"9999-12-31","name":"BRISSA
GO-VALTRAVAGLIA"}],"E265":[{"name":"SIZIANO","start":"1871-06-01","end":"199
8-04-18"},{"name":"SIZIANO","end":"9999-12-31","start":"1998-04-19"}],"L843"
:[{"start":"1872-10-18","end":"1927-01-11","name":"VICO NEL LAZIO"},{"start"
:"1927-01-12","end":"9999-12-31","name":"VICO NEL LAZIO"}],"H014":[{"start":
"1866-11-19","end":"1968-04-05","name":"PRECENICCO"},{"name":"PRECENICCO","e
nd":"9999-12-31","start":"1968-04-06"}],"M103":[{"start":"1866-11-19","end":
"9999-12-31","name":"VO'"}],"I960":[{"end":"9999-12-31","start":"1861-03-17"
,"name":"STIO"}],"C306":[{"start":"1862-11-07","end":"1929-03-04","name":"CA
STIGLIONE DEL GENOVESI"},{"name":"CASTIGLIONE DEL GENOVESI","start":"1946-04
-26","end":"9999-12-31"}],"H956":[{"start":"1923-04-27","end":"1927-01-11","
name":"SAN LORENZO"},{"name":"SAN LORENZO","end":"1928-04-03","start":"1927-
01-12"},{"end":"1928-10-27","start":"1928-04-04","name":"SAN LORENZO"},{"sta
rt":"1939-12-01","end":"9999-12-31","name":"SAN LORENZO DI SEBATO"},{"end":"
1923-02-13","start":"1920-10-16","name":"SAN LORENZO IN PUSTERIA"},{"start":
"1923-02-14","end":"1923-04-26","name":"SAN LORENZO IN PUSTERIA"},{"end":"19
28-12-13","start":"1928-10-28","name":"SAN LORENZO IN PUSTERIA"},{"name":"SA
N LORENZO IN PUSTERIA","start":"1928-12-14","end":"1939-11-30"}],"G154":[{"e
nd":"1927-01-11","start":"1861-03-17","name":"OSIDDA"},{"name":"OSIDDA","end
":"9999-12-31","start":"1927-01-12"}],"B315":[{"name":"CACCAMO","start":"186
1-03-17","end":"9999-12-31"}],"H659":[{"start":"1861-03-17","end":"1927-01-1
1","name":"SADALI"},{"name":"SADALI","end":"2005-12-31","start":"1927-01-12"
},{"end":"2016-04-27","start":"2006-01-01","name":"SADALI"},{"name":"SADALI"
,"end":"9999-12-31","start":"2016-04-28"}],"F380":[{"name":"MONSAMPOLO DEL T
RONTO","start":"1862-11-24","end":"9999-12-31"}],"E353":[{"start":"1928-08-1
5","end":"9999-12-31","name":"ISOLA DI FONDRA"}],"D229":[{"name":"CUSAGO","e
nd":"9999-12-31","start":"1861-03-17"}],"E059":[{"end":"1923-11-23","start":
"1861-03-17","name":"GIUNCUGNANO"},{"end":"2014-12-31","start":"1923-11-24",
"name":"GIUNCUGNANO"}],"Z129":[{"name":"Romania","end":"9999-12-31","start":
"0000-01-01"}],"H879":[{"name":"SAN GIORGIO","start":"1923-04-27","end":"192
7-01-11"},{"end":"1928-04-04","start":"1927-01-12","name":"SAN GIORGIO"},{"n
ame":"SAN GIORGIO IN VAL TURES","end":"1923-02-13","start":"1920-10-16"},{"e
nd":"1923-04-26","start":"1923-02-14","name":"SAN GIORGIO IN VAL TURES"}],"C
200":[{"start":"1893-11-13","end":"1928-06-14","name":"CASTEL VERRINO"},{"st
art":"1946-10-12","end":"1970-03-02","name":"CASTELVERRINO"},{"name":"CASTEL
VERRINO","end":"9999-12-31","start":"1970-03-03"}],"H272":[{"name":"RICALDON
E","end":"9999-12-31","start":"1861-03-17"}],"B791":[{"end":"9999-12-31","st
art":"1861-03-17","name":"CARMAGNOLA"}],"F286":[{"start":"1920-10-16","end":
"1923-02-13","name":"MOLINA DI LEDRO"},{"end":"1928-04-03","start":"1923-02-
14","name":"MOLINA DI LEDRO"},{"name":"MOLINA DI LEDRO","start":"1928-04-04"
,"end":"1955-03-01"},{"name":"MOLINA DI LEDRO","end":"2009-12-31","start":"1
955-03-02"}],"L284":[{"name":"TORREVECCHIA TEATINA","end":"9999-12-31","star
t":"1879-05-12"}],"E614":[{"start":"1920-10-16","end":"1923-02-13","name":"L
ISIGNAGO"},{"name":"LISIGNAGO","end":"1928-09-05","start":"1923-02-14"},{"st
art":"1952-05-17","end":"2015-12-31","name":"LISIGNAGO"}],"L628":[{"end":"19
28-12-29","start":"1862-12-29","name":"VALLO DELLA LUCANIA"},{"end":"1945-07
-18","start":"1928-12-30","name":"VALLO DELLA LUCANIA"},{"end":"1946-04-01",
"start":"1945-07-19","name":"VALLO DELLA LUCANIA"},{"name":"VALLO DELLA LUCA
NIA","start":"1946-04-02","end":"9999-12-31"}],"I301":[{"name":"SANTA PAOLIN
A","start":"1861-03-17","end":"9999-12-31"}],"F411":[{"start":"1861-03-17","
end":"1927-01-11","name":"MONTALENGHE"},{"start":"1927-01-12","end":"1945-10
-04","name":"MONTALENGHE"},{"name":"MONTALENGHE","start":"1945-10-05","end":
"9999-12-31"}],"A920":[{"start":"1862-12-01","end":"1980-05-13","name":"BOFF
ALORA SOPRA TICINO"},{"end":"9999-12-31","start":"1980-05-14","name":"BOFFAL
ORA SOPRA TICINO"}],"E519":[{"end":"9999-12-31","start":"1934-10-05","name":
"LEIVI"},{"end":"1934-10-04","start":"1861-03-17","name":"SAN RUFINO"}],"A45
7":[{"name":"ARZENO D'ONEGLIA","start":"1889-01-28","end":"1923-12-21"},{"na
me":"ARZENO D'ONEGLIA","start":"1925-08-22","end":"1929-03-16"}],"C853":[{"s
tart":"1861-03-17","end":"1927-01-11","name":"COLLECORVINO"},{"name":"COLLEC
ORVINO","end":"9999-12-31","start":"1927-01-12"}],"M094":[{"name":"VI\u00d9"
,"start":"1861-03-17","end":"1927-10-18"},{"start":"1927-10-19","end":"9999-
12-31","name":"VI\u00d9"}],"G989":[{"name":"PRASO","start":"1920-10-16","end
":"1923-02-13"},{"end":"1928-03-06","start":"1923-02-14","name":"PRASO"},{"n
ame":"PRASO","start":"1952-09-20","end":"2014-12-31"}],"G754":[{"name":"POGG
IO A CAIANO","start":"1962-09-23","end":"1992-04-15"},{"start":"1992-04-16",
"end":"9999-12-31","name":"POGGIO A CAIANO"}],"D837":[{"name":"GABERIE","end
":"1923-02-07","start":"1921-01-05"},{"start":"1923-02-08","end":"1923-04-26
","name":"GABERIE"},{"start":"1923-04-27","end":"1927-01-11","name":"GABRIA"
},{"start":"1927-01-12","end":"1928-03-12","name":"GABRIA"}],"D588":[{"name"
:"FILAGO","end":"1928-01-25","start":"1861-03-17"},{"start":"1958-09-10","en
d":"9999-12-31","name":"FILAGO"},{"end":"1928-01-25","start":"1861-03-17","n
ame":"MARNE"},{"name":"MARNE","start":"1928-01-26","end":"1958-09-09"}],"G68
6":[{"end":"9999-12-31","start":"1869-07-06","name":"PIOLTELLO"}],"G112":[{"
name":"ORIS","end":"1923-02-13","start":"1920-10-16"},{"name":"ORIS","end":"
1927-01-11","start":"1923-02-14"},{"name":"ORIS","start":"1927-01-12","end":
"1929-05-14"}],"F625":[{"start":"1862-12-29","end":"9999-12-31","name":"MONT
ESANO SULLA MARCELLANA"}],"E279":[{"name":"IDRIA DI SOTTO","start":"1921-01-
05","end":"1921-12-30"},{"end":"1923-02-07","start":"1921-12-31","name":"IDR
IA DI SOTTO"},{"start":"1923-02-08","end":"1927-01-11","name":"IDRIA DI SOTT
O"},{"name":"IDRIA DI SOTTO","end":"1928-08-23","start":"1927-01-12"}],"G647
":[{"start":"1867-08-30","end":"9999-12-31","name":"PIEVE D'OLMI"}],"C432":[
{"name":"CELLA MONTE","start":"1863-02-16","end":"9999-12-31"}],"D985":[{"na
me":"GERMANEDO","end":"1924-01-11","start":"1861-03-17"}],"F597":[{"name":"M
ONTERENZIO","start":"1866-01-01","end":"1931-05-20"},{"end":"9999-12-31","st
art":"1931-05-21","name":"MONTERENZIO"}],"E872":[{"name":"MALVITO","start":"
1861-03-17","end":"1928-07-17"},{"start":"1934-03-07","end":"9999-12-31","na
me":"MALVITO"}],"B483":[{"name":"CAMINO AL TAGLIAMENTO","start":"1949-12-11"
,"end":"1968-04-05"},{"end":"9999-12-31","start":"1968-04-06","name":"CAMINO
 AL TAGLIAMENTO"},{"start":"1866-11-19","end":"1928-06-15","name":"CAMINO DI
 CODROIPO"},{"name":"CAMINO DI CODROIPO","end":"1949-12-10","start":"1947-01
-09"}],"H052":[{"name":"PRETORO","start":"1861-03-17","end":"9999-12-31"}],"
C950":[{"name":"CONCORDIA SAGITARIA","start":"1868-12-31","end":"1965-09-10"
},{"start":"1965-09-11","end":"9999-12-31","name":"CONCORDIA SAGITTARIA"}],"
C372":[{"start":"1920-10-16","end":"1923-02-13","name":"CAVALESE"},{"name":"
CAVALESE","start":"1923-02-14","end":"1928-05-12"},{"start":"1928-05-13","en
d":"1946-12-25","name":"CAVALESE"},{"start":"1946-12-26","end":"1999-07-13",
"name":"CAVALESE"},{"name":"CAVALESE","start":"1999-07-14","end":"9999-12-31
"}],"H300":[{"end":"9999-12-31","start":"1871-01-15","name":"RIOFREDDO"}],"L
709":[{"name":"VEDUGGIO CON COLZANO","start":"1866-01-14","end":"1928-12-30"
},{"name":"VEDUGGIO CON COLZANO","end":"2009-06-29","start":"1956-04-08"},{"
name":"VEDUGGIO CON COLZANO","start":"2009-06-30","end":"9999-12-31"}],"A091
":[{"name":"AGROPOLI","start":"1861-03-17","end":"9999-12-31"}],"G225":[{"na
me":"PADRIA","start":"1861-03-17","end":"9999-12-31"}],"D881":[{"name":"GALL
INARO","end":"1963-04-27","start":"1948-05-06"},{"start":"1963-04-28","end":
"9999-12-31","name":"GALLINARO"}],"A823":[{"start":"1861-03-17","end":"9999-
12-31","name":"BESENZONE"}],"H512":[{"end":"9999-12-31","start":"1868-02-14"
,"name":"ROMANO D'EZZELINO"}],"C427":[{"name":"CELENTINO","end":"1923-02-13"
,"start":"1920-10-16"},{"start":"1923-02-14","end":"1928-08-11","name":"CELE
NTINO"}],"F247":[{"end":"1905-06-22","start":"1882-01-09","name":"MISINTO"},
{"end":"2009-06-29","start":"1905-06-23","name":"MISINTO"},{"name":"MISINTO"
,"end":"9999-12-31","start":"2009-06-30"}],"D348":[{"start":"1866-01-21","en
d":"9999-12-31","name":"DORNO"}],"L394":[{"name":"TRETTO","end":"1969-08-27"
,"start":"1866-11-19"}],"F396":[{"start":"1920-10-16","end":"1923-02-13","na
me":"MONTAGNE"},{"name":"MONTAGNE","end":"1928-02-08","start":"1923-02-14"},
{"name":"MONTAGNE","end":"2015-12-31","start":"1952-05-17"}],"A868":[{"name"
:"BIGLIANA","end":"1923-02-07","start":"1921-01-05"},{"end":"1927-01-11","st
art":"1923-02-08","name":"BIGLIANA"},{"name":"BIGLIANA","end":"1928-05-11","
start":"1927-01-12"}],"A556":[{"end":"1928-06-14","start":"1861-03-17","name
":"BAGNASCO D'ASTI"}],"B543":[{"end":"1927-01-11","start":"1863-08-10","name
":"CAMPOLI APPENNINO"},{"end":"9999-12-31","start":"1927-01-12","name":"CAMP
OLI APPENNINO"}],"D877":[{"name":"GALLICO","start":"1861-03-17","end":"1927-
08-02"}],"I643":[{"start":"1861-03-17","end":"9999-12-31","name":"SERRA DE' 
CONTI"}],"F195":[{"start":"1861-03-17","end":"1928-07-29","name":"MIGIANDONE
"}],"H227":[{"end":"9999-12-31","start":"1861-03-17","name":"REINO"}],"M201"
:[{"name":"ZUMAGLIA","start":"1880-12-13","end":"1927-01-11"},{"name":"ZUMAG
LIA","start":"1927-01-12","end":"1992-04-15"},{"end":"9999-12-31","start":"1
992-04-16","name":"ZUMAGLIA"}],"H645":[{"name":"RUVO DI PUGLIA","end":"9999-
12-31","start":"1863-01-19"}],"H839":[{"end":"1928-03-23","start":"1863-01-1
9","name":"SAN FERDINANDO DI PUGLIA"},{"name":"SAN FERDINANDO DI PUGLIA","st
art":"1928-03-24","end":"2009-07-26"},{"name":"SAN FERDINANDO DI PUGLIA","en
d":"9999-12-31","start":"2009-07-27"}],"B293":[{"name":"BUSSETO","end":"9999
-12-31","start":"1861-03-17"}],"A357":[{"name":"ARBOREA","end":"1948-05-26",
"start":"1944-03-08"},{"end":"1974-08-19","start":"1948-05-27","name":"ARBOR
EA"},{"end":"2011-01-20","start":"1974-08-20","name":"ARBOREA"},{"end":"9999
-12-31","start":"2011-01-21","name":"ARBOREA"},{"name":"MUSSOLINIA DI SARDEG
NA","end":"1944-03-07","start":"1931-02-20"}],"G720":[{"start":"1861-03-17",
"end":"9999-12-31","name":"PIZZALE"}],"A016":[{"start":"1861-03-17","end":"9
999-12-31","name":"ACCEGLIO"}],"A954":[{"start":"1867-09-02","end":"9999-12-
31","name":"BOLZANO VICENTINO"}],"A789":[{"end":"2005-12-31","start":"1861-0
3-17","name":"BERCHIDDA"},{"name":"BERCHIDDA","start":"2006-01-01","end":"20
16-04-27"},{"name":"BERCHIDDA","start":"2016-04-28","end":"9999-12-31"}],"I4
01":[{"name":"SAN VITO DI LEGUZZANO","end":"9999-12-31","start":"1867-09-02"
}],"H232":[{"end":"1928-01-21","start":"1861-03-17","name":"REMEDELLO SOTTO"
}],"F311":[{"name":"MOMBARUZZO","start":"1861-03-17","end":"1935-04-07"},{"e
nd":"9999-12-31","start":"1935-04-08","name":"MOMBARUZZO"}],"A204":[{"name":
"ALLAI","end":"1974-08-19","start":"1861-03-17"},{"name":"ALLAI","start":"19
74-08-20","end":"9999-12-31"}],"C526":[{"start":"1863-04-23","end":"1927-01-
11","name":"CERRETO CASTELLO"},{"name":"CERRETO CASTELLO","end":"1992-04-15"
,"start":"1927-01-12"},{"start":"1992-04-16","end":"2018-12-31","name":"CERR
ETO CASTELLO"}],"C332":[{"name":"CASTREZZATO","start":"1861-03-17","end":"99
99-12-31"}],"D144":[{"end":"1927-01-11","start":"1893-12-05","name":"CREMENA
GA"},{"start":"1927-01-12","end":"9999-12-31","name":"CREMENAGA"}],"B089":[{
"end":"1928-10-06","start":"1895-07-29","name":"BOTTARONE"}],"L198":[{"end":
"1927-10-21","start":"1923-04-27","name":"TOMADIO"},{"name":"TOMADIO","end":
"1947-09-15","start":"1927-10-22"},{"name":"TOMAI","end":"1923-02-07","start
":"1921-01-05"},{"name":"TOMAI","start":"1923-02-08","end":"1923-04-26"}],"B
383":[{"end":"1974-08-19","start":"1861-03-17","name":"CALASETTA"},{"name":"
CALASETTA","end":"2005-12-31","start":"1974-08-20"},{"start":"2006-01-01","e
nd":"2016-04-27","name":"CALASETTA"},{"start":"2016-04-28","end":"9999-12-31
","name":"CALASETTA"}],"A561":[{"name":"BAGNI DI MONTECATINI","start":"1905-
07-30","end":"1928-01-25"},{"end":"1928-12-07","start":"1928-01-26","name":"
BAGNI DI MONTECATINI"},{"end":"1934-11-02","start":"1928-12-08","name":"MONT
ECATINI-TERME"},{"start":"1934-11-03","end":"1940-10-07","name":"MONTECATINI
-TERME"},{"name":"MONTECATINI-TERME","end":"9999-12-31","start":"1940-10-08"
}],"B716":[{"start":"1861-03-17","end":"9999-12-31","name":"CAPURSO"}],"C705
":[{"name":"CINIGIANO","start":"1861-03-17","end":"1956-06-12"},{"end":"9999
-12-31","start":"1956-06-13","name":"CINIGIANO"}],"E160":[{"end":"1923-03-18
","start":"1861-03-17","name":"GRECCIO"},{"start":"1923-03-19","end":"1927-0
1-11","name":"GRECCIO"},{"start":"1927-01-12","end":"9999-12-31","name":"GRE
CCIO"}],"E827":[{"end":"1923-02-13","start":"1920-10-16","name":"MAGR\u00c8"
},{"name":"MAGR\u00c8","end":"1927-01-11","start":"1923-02-14"},{"name":"MAG
R\u00c8","end":"1928-06-27","start":"1927-01-12"}],"E239":[{"start":"1861-03
-17","end":"9999-12-31","name":"GUARDAVALLE"}],"D448":[{"name":"FABBRICA DUR
INI","end":"1928-09-24","start":"1863-02-23"}],"I084":[{"end":"9999-12-31","
start":"1873-01-20","name":"SAN PIER NICETO"}],"L494":[{"name":"URAGO D'OGLI
O","start":"1861-03-17","end":"9999-12-31"}],"F496":[{"start":"1861-03-17","
end":"9999-12-31","name":"MONTEFANO"}],"C327":[{"start":"1866-11-19","end":"
1968-04-05","name":"CASTIONS DI STRADA"},{"name":"CASTIONS DI STRADA","end":
"9999-12-31","start":"1968-04-06"}],"L663":[{"end":"1923-02-13","start":"192
0-10-16","name":"VANGA"},{"name":"VANGA","end":"1927-01-11","start":"1923-02
-14"},{"start":"1927-01-12","end":"1928-06-15","name":"VANGA"}],"F067":[{"en
d":"1927-01-11","start":"1861-03-17","name":"MAZZ\u00c8"},{"name":"MAZZ\u00c
8","end":"1928-07-17","start":"1927-01-12"},{"start":"1928-07-18","end":"194
5-10-04","name":"MAZZ\u00c8"},{"name":"MAZZ\u00c8","end":"1947-01-23","start
":"1945-10-05"},{"name":"MAZZ\u00c8","end":"9999-12-31","start":"1947-01-24"
}],"C924":[{"name":"COMEZZANO","end":"1927-12-03","start":"1861-03-17"}],"C4
72":[{"start":"1861-03-17","end":"9999-12-31","name":"CENTRACHE"}],"H400":[{
"name":"ROCCA DI CAMBIO","start":"1861-03-17","end":"9999-12-31"}],"E832":[{
"end":"1923-02-13","start":"1920-10-16","name":"MAIA BASSA"},{"name":"MAIA B
ASSA","start":"1923-02-14","end":"1923-11-08"}],"M062":[{"start":"1861-03-17
","end":"9999-12-31","name":"VINZAGLIO"}],"B621":[{"end":"1928-02-20","start
":"1861-03-17","name":"CANOSIO"},{"end":"9999-12-31","start":"1946-04-26","n
ame":"CANOSIO"}],"D744":[{"end":"9999-12-31","start":"1864-01-29","name":"FO
SSATO SERRALTA"}],"A912":[{"end":"9999-12-31","start":"1861-03-17","name":"B
OCCHIGLIERO"}],"L798":[{"end":"2018-12-31","start":"1861-03-17","name":"VERU
NO"}],"B116":[{"start":"1920-10-16","end":"1923-02-13","name":"BRAIES"},{"en
d":"1927-01-11","start":"1923-02-14","name":"BRAIES"},{"name":"BRAIES","end"
:"9999-12-31","start":"1927-01-12"}],"C105":[{"end":"9999-12-31","start":"18
61-03-17","name":"CASTELFRANCI"}],"E760":[{"end":"1959-03-17","start":"1866-
11-19","name":"LUSEVERA"},{"end":"1968-04-05","start":"1959-03-18","name":"L
USEVERA"},{"end":"9999-12-31","start":"1968-04-06","name":"LUSEVERA"}],"I521
":[{"name":"SCHEGGIA","end":"1878-11-22","start":"1861-03-17"}],"I342":[{"en
d":"9999-12-31","start":"1862-10-20","name":"SANT'ILARIO D'ENZA"}],"D810":[{
"start":"1871-01-15","end":"1927-01-11","name":"FROSINONE"},{"end":"9999-12-
31","start":"1927-01-12","name":"FROSINONE"}],"G358":[{"start":"1929-04-13",
"end":"1935-04-07","name":"PASSERANO MARMORITO"},{"end":"9999-12-31","start"
:"1935-04-08","name":"PASSERANO MARMORITO"}],"I828":[{"end":"9999-12-31","st
art":"1861-03-17","name":"SONDALO"}],"F795":[{"name":"MUGGIA","end":"1923-02
-07","start":"1921-01-05"},{"end":"1923-11-14","start":"1923-02-08","name":"
MUGGIA"},{"start":"1923-11-15","end":"9999-12-31","name":"MUGGIA"}],"E045":[
{"name":"GIOVE","end":"1927-01-11","start":"1861-03-17"},{"name":"GIOVE","en
d":"9999-12-31","start":"1927-01-12"}],"C406":[{"name":"CAVRIANA","end":"999
9-12-31","start":"1868-07-01"}],"G867":[{"name":"PONTIROLO NUOVO","end":"999
9-12-31","start":"1863-07-13"}],"D913":[{"name":"GARBAGNATE MONASTERO","end"
:"1992-04-15","start":"1861-03-17"},{"name":"GARBAGNATE MONASTERO","start":"
1992-04-16","end":"9999-12-31"}],"D657":[{"name":"FOLSOGNO","start":"1861-03
-17","end":"1928-06-30"}],"B415":[{"name":"CALITRI","end":"9999-12-31","star
t":"1861-03-17"}],"F480":[{"name":"MONTECORVINO PUGLIANO","start":"1861-03-1
7","end":"1911-07-26"},{"name":"MONTECORVINO PUGLIANO","end":"9999-12-31","s
tart":"1911-07-27"}],"B682":[{"name":"CAPRACOTTA","start":"1861-03-17","end"
:"1970-03-02"},{"end":"9999-12-31","start":"1970-03-03","name":"CAPRACOTTA"}
],"G120":[{"name":"OROTELLI","start":"1861-03-17","end":"1927-01-11"},{"star
t":"1927-01-12","end":"9999-12-31","name":"OROTELLI"}],"C633":[{"end":"9999-
12-31","start":"1861-03-17","name":"CHIEUTI"}],"I582":[{"end":"1928-11-26","
start":"1868-02-09","name":"SELEGAS"},{"start":"1928-11-27","end":"1946-03-2
4","name":"SELEGAS"},{"name":"SELEGAS","start":"1946-03-25","end":"1948-05-0
6"},{"name":"SELEGAS","end":"1974-08-19","start":"1948-05-07"},{"name":"SELE
GAS","end":"2016-04-27","start":"1974-08-20"},{"end":"9999-12-31","start":"2
016-04-28","name":"SELEGAS"}],"A189":[{"name":"ALFIANO NATTA","start":"1863-
02-16","end":"1928-01-28"},{"start":"1928-01-29","end":"9999-12-31","name":"
ALFIANO NATTA"}],"E986":[{"end":"1923-12-21","start":"1864-04-01","name":"MA
RTINA FRANCA"},{"end":"9999-12-31","start":"1923-12-22","name":"MARTINA FRAN
CA"}],"E689":[{"name":"LOREO","end":"1929-01-12","start":"1866-11-19"},{"sta
rt":"1929-01-13","end":"1929-04-10","name":"LOREO"},{"end":"1937-09-02","sta
rt":"1929-04-11","name":"LOREO"},{"name":"LOREO","end":"1947-12-21","start":
"1937-09-03"},{"start":"1947-12-22","end":"9999-12-31","name":"LOREO"}],"L21
9":[{"start":"1889-08-12","end":"9999-12-31","name":"TORINO"}],"G516":[{"end
":"2009-07-11","start":"1869-03-04","name":"PETRITOLI"},{"start":"2009-07-12
","end":"9999-12-31","name":"PETRITOLI"}],"D618":[{"end":"1925-11-01","start
":"1861-03-17","name":"FIUMANA"}],"L812":[{"name":"VESTONE","end":"1928-09-2
4","start":"1861-03-17"},{"name":"VESTONE","start":"1928-09-25","end":"9999-
12-31"}],"C938":[{"name":"CONA","end":"9999-12-31","start":"1866-11-19"}],"C
177":[{"name":"CASTELLIRI","end":"1927-01-11","start":"1878-08-13"},{"name":
"CASTELLIRI","start":"1927-01-12","end":"9999-12-31"}],"G914":[{"start":"186
6-11-19","end":"9999-12-31","name":"PORTOGRUARO"}],"E584":[{"name":"LIGNANO 
SABBIADORO","end":"1968-04-05","start":"1959-08-18"},{"start":"1968-04-06","
end":"9999-12-31","name":"LIGNANO SABBIADORO"}],"H621":[{"start":"1861-03-17
","end":"9999-12-31","name":"ROVITO"}],"G056":[{"name":"OLTRONA DI SAN MAMET
TE","start":"1863-02-23","end":"9999-12-31"}],"G317":[{"start":"1861-03-17",
"end":"9999-12-31","name":"PAOLA"}],"M213":[{"start":"1970-05-27","end":"999
9-12-31","name":"ARDEA"}],"C844":[{"start":"1861-03-17","end":"9999-12-31","
name":"COLLARMELE"}],"C692":[{"name":"CIMBERGO-PASPARDO","start":"1927-12-24
","end":"1947-12-22"}],"G743":[{"start":"1866-11-19","end":"1968-04-05","nam
e":"POCENIA"},{"name":"POCENIA","end":"9999-12-31","start":"1968-04-06"}],"H
415":[{"end":"1868-01-01","start":"1861-03-17","name":"ROCCAGUGLIELMA"}],"F6
76":[{"name":"MONTICIANO","start":"1861-03-17","end":"9999-12-31"}],"Z339":[
{"name":"Sahara occidentale","end":"9999-12-31","start":"0000-01-01"}],"L674
":[{"name":"VARARO","start":"1861-03-17","end":"1927-01-11"},{"end":"1928-01
-17","start":"1927-01-12","name":"VARARO"}],"F979":[{"start":"1861-03-17","e
nd":"1927-01-11","name":"NUORO"},{"start":"1927-01-12","end":"9999-12-31","n
ame":"NUORO"}],"H682":[{"end":"9999-12-31","start":"1869-03-11","name":"SALA
 BAGANZA"}],"F303":[{"end":"1923-11-28","start":"1871-08-03","name":"MOLTEDO
 SUPERIORE"},{"start":"1925-08-22","end":"1928-12-14","name":"MOLTEDO SUPERI
ORE"}],"F009":[{"name":"MASER","end":"9999-12-31","start":"1866-11-19"}],"B5
20":[{"name":"CAMPOBELLO DI LICATA","end":"1977-07-05","start":"1861-03-17"}
,{"name":"CAMPOBELLO DI LICATA","end":"9999-12-31","start":"1977-07-06"}],"D
360":[{"start":"1861-03-17","end":"9999-12-31","name":"DOZZA"}],"L579":[{"en
d":"1927-11-29","start":"1861-03-17","name":"VALGOGLIO"},{"end":"9999-12-31"
,"start":"1954-10-29","name":"VALGOGLIO"}],"E947":[{"start":"1861-03-17","en
d":"1928-05-30","name":"MARGNO"},{"name":"MARGNO","end":"1957-05-02","start"
:"1928-05-31"},{"start":"1957-05-03","end":"1992-04-15","name":"MARGNO"},{"e
nd":"9999-12-31","start":"1992-04-16","name":"MARGNO"}],"C535":[{"name":"CER
RO LAGO MAGGIORE","end":"1927-01-11","start":"1863-07-13"},{"name":"CERRO LA
GO MAGGIORE","end":"1928-01-27","start":"1927-01-12"}],"I620":[{"name":"SEPR
IO","start":"1928-09-23","end":"1950-04-04"},{"start":"1950-04-05","end":"19
53-09-13","name":"SEPRIO"}],"G276":[{"name":"PALIANO","start":"1871-01-15","
end":"1927-01-11"},{"name":"PALIANO","end":"9999-12-31","start":"1927-01-12"
}],"M180":[{"end":"1905-08-19","start":"1885-04-17","name":"ZINASCO"},{"name
":"ZINASCO","end":"9999-12-31","start":"1905-08-20"}],"G061":[{"name":"OME",
"end":"1928-06-18","start":"1861-03-17"},{"name":"OME","start":"1928-06-19",
"end":"1948-02-12"},{"start":"1948-02-13","end":"9999-12-31","name":"OME"}],
"E344":[{"name":"ISOLA BALBA","start":"1861-03-17","end":"1869-02-11"}],"C71
0":[{"name":"CINQUEFRONDI","end":"9999-12-31","start":"1861-03-17"}],"D266":
[{"name":"DELEBIO","end":"1928-11-29","start":"1861-03-17"},{"start":"1928-1
1-30","end":"9999-12-31","name":"DELEBIO"}],"E990":[{"name":"MARTIRANO","end
":"9999-12-31","start":"1957-01-18"}],"C283":[{"name":"CASTELVETERE DI CALOR
E","end":"1950-07-19","start":"1862-12-29"},{"end":"9999-12-31","start":"195
0-07-20","name":"CASTELVETERE SUL CALORE"}],"G143":[{"start":"2017-04-20","e
nd":"9999-12-31","name":"LUNI"},{"name":"ORTONOVO","end":"1923-12-21","start
":"1861-03-17"},{"name":"ORTONOVO","start":"1923-12-22","end":"2017-04-19"}]
,"I545":[{"start":"1861-03-17","end":"1927-01-11","name":"SCOPELLO"},{"start
":"1927-01-12","end":"9999-12-31","name":"SCOPELLO"}],"B232":[{"end":"9999-1
2-31","start":"1861-03-17","name":"BRUZOLO"}],"B839":[{"name":"CARRODANO","s
tart":"1861-03-17","end":"1923-12-21"},{"name":"CARRODANO","start":"1923-12-
22","end":"9999-12-31"}],"H293":[{"name":"RIMELLA","start":"1861-03-17","end
":"1927-01-11"},{"end":"9999-12-31","start":"1927-01-12","name":"RIMELLA"}],
"B645":[{"name":"CAPACI","start":"1861-03-17","end":"9999-12-31"}],"G370":[{
"name":"PATERNOPOLI","end":"9999-12-31","start":"1864-01-29"}],"D805":[{"sta
rt":"1861-03-17","end":"1928-02-20","name":"FRONT"},{"name":"FRONT","end":"1
999-12-31","start":"1947-01-21"},{"end":"9999-12-31","start":"2000-01-01","n
ame":"FRONT"}],"L968":[{"end":"1974-08-19","start":"1861-03-17","name":"VILL
AMASSARGIA"},{"start":"1974-08-20","end":"2005-12-31","name":"VILLAMASSARGIA
"},{"end":"2016-04-27","start":"2006-01-01","name":"VILLAMASSARGIA"},{"name"
:"VILLAMASSARGIA","end":"9999-12-31","start":"2016-04-28"}],"E082":[{"name":
"GOMBITO","start":"1861-03-17","end":"1938-02-18"},{"name":"GOMBITO","start"
:"1938-02-19","end":"1984-07-26"},{"end":"9999-12-31","start":"1984-07-27","
name":"GOMBITO"}],"A660":[{"name":"BARGE","start":"1861-03-17","end":"1911-0
2-07"},{"name":"BARGE","end":"9999-12-31","start":"1911-02-08"}],"G430":[{"n
ame":"PENANGO","start":"1861-03-17","end":"1908-07-22"},{"end":"1935-04-07",
"start":"1908-07-23","name":"PENANGO"},{"start":"1935-04-08","end":"9999-12-
31","name":"PENANGO"}],"F856":[{"name":"NAZ-SCIAVES","start":"1929-02-14","e
nd":"9999-12-31"}],"B227":[{"end":"1927-01-11","start":"1861-03-17","name":"
BRUSCIANO"},{"name":"BRUSCIANO","end":"9999-12-31","start":"1927-01-12"}],"L
854":[{"name":"VIDIGULFO","end":"9999-12-31","start":"1872-11-01"}],"F568":[
{"name":"MONTEMILONE","end":"9999-12-31","start":"1861-03-17"}],"I834":[{"na
me":"SOPRAMONTE","start":"1920-10-16","end":"1923-02-13"},{"end":"1926-11-18
","start":"1923-02-14","name":"SOPRAMONTE"}],"C110":[{"end":"9999-12-31","st
art":"1862-11-10","name":"CASTEL VITTORIO"}],"H543":[{"end":"1923-02-13","st
art":"1920-10-16","name":"RONCOGNO"},{"name":"RONCOGNO","end":"1929-01-24","
start":"1923-02-14"}],"D901":[{"start":"1861-03-17","end":"9999-12-31","name
":"GAMBOL\u00d2"}],"F965":[{"start":"1863-01-26","end":"9999-12-31","name":"
NOVI LIGURE"}],"I385":[{"name":"SAN VIGILIO","end":"1928-02-04","start":"186
1-03-17"}],"D251":[{"end":"1929-05-02","start":"1861-03-17","name":"DARFO"},
{"name":"DARFO","start":"1929-05-03","end":"1959-07-25"},{"start":"1959-07-2
6","end":"1969-03-29","name":"DARFO"},{"end":"9999-12-31","start":"1969-03-3
0","name":"DARFO BOARIO TERME"}],"B400":[{"name":"CALDES","start":"1920-10-1
6","end":"1923-02-13"},{"end":"1928-12-31","start":"1923-02-14","name":"CALD
ES"},{"start":"1929-01-01","end":"1956-11-19","name":"CALDES"},{"name":"CALD
ES","end":"9999-12-31","start":"1956-11-20"}],"E021":[{"end":"9999-12-31","s
tart":"1960-02-21","name":"GIAVERA DEL MONTELLO"}],"E893":[{"name":"MANSU\u0
0c8","end":"9999-12-31","start":"1866-11-19"}],"C777":[{"name":"CIVITELLA DI
 ROMAGNA","end":"1925-10-31","start":"1862-09-08"},{"end":"1926-02-20","star
t":"1925-11-01","name":"CIVITELLA DI ROMAGNA"},{"start":"1926-02-21","end":"
1926-08-05","name":"CIVITELLA DI ROMAGNA"},{"start":"1926-08-06","end":"1927
-03-15","name":"CIVITELLA DI ROMAGNA"},{"start":"1927-03-16","end":"1929-06-
17","name":"CIVITELLA DI ROMAGNA"},{"name":"CIVITELLA DI ROMAGNA","start":"1
929-06-18","end":"1933-07-26"},{"end":"1992-04-15","start":"1933-07-27","nam
e":"CIVITELLA DI ROMAGNA"},{"name":"CIVITELLA DI ROMAGNA","start":"1992-04-1
6","end":"9999-12-31"}],"A318":[{"start":"1861-03-17","end":"1926-11-10","na
me":"ANVERSA"},{"start":"1926-11-11","end":"9999-12-31","name":"ANVERSA DEGL
I ABRUZZI"}],"C099":[{"end":"1947-09-15","start":"1928-05-12","name":"CASTEL
 DOBRA"}],"C393":[{"name":"CAVEDINE","start":"1920-10-16","end":"1923-02-13"
},{"name":"CAVEDINE","start":"1923-02-14","end":"9999-12-31"}],"F861":[{"end
":"1927-10-06","start":"1866-11-19","name":"NEGRAR"},{"name":"NEGRAR","start
":"1927-10-07","end":"1929-01-23"},{"name":"NEGRAR","start":"1929-01-24","en
d":"2019-02-22"},{"start":"2019-02-23","end":"9999-12-31","name":"NEGRAR DI 
VALPOLICELLA"}],"I948":[{"name":"STELVIO","start":"1920-10-16","end":"1923-0
2-13"},{"name":"STELVIO","end":"1927-01-11","start":"1923-02-14"},{"start":"
1927-01-12","end":"1929-05-14","name":"STELVIO"},{"name":"STELVIO","end":"99
99-12-31","start":"1953-05-23"}],"H716":[{"name":"SALMORE","start":"1940-08-
10","end":"1951-09-10"},{"name":"SALMOUR","start":"1861-03-17","end":"1940-0
8-09"},{"name":"SALMOUR","end":"9999-12-31","start":"1951-09-11"}],"G202":[{
"start":"1861-03-17","end":"1927-01-11","name":"OZEGNA"},{"name":"OZEGNA","e
nd":"1929-06-15","start":"1927-01-12"},{"name":"OZEGNA","end":"9999-12-31","
start":"1947-01-24"}],"L565":[{"name":"VALDOBBIADENE","start":"1866-11-19","
end":"1929-02-20"},{"start":"1929-02-21","end":"9999-12-31","name":"VALDOBBI
ADENE"}],"M196":[{"name":"ZUBIENA","end":"1927-01-11","start":"1861-03-17"},
{"name":"ZUBIENA","start":"1927-01-12","end":"1992-04-15"},{"end":"9999-12-3
1","start":"1992-04-16","name":"ZUBIENA"}],"G809":[{"name":"POMBIA","end":"9
999-12-31","start":"1861-03-17"}],"H383":[{"name":"ROCCABERNARDA","start":"1
861-03-17","end":"1992-04-15"},{"start":"1992-04-16","end":"9999-12-31","nam
e":"ROCCABERNARDA"}],"H089":[{"start":"1929-01-24","end":"1968-04-05","name"
:"PULFERO"},{"name":"PULFERO","end":"9999-12-31","start":"1968-04-06"}],"G47
0":[{"start":"1920-10-16","end":"1923-02-13","name":"PERSONE"},{"name":"PERS
ONE","end":"1929-02-28","start":"1923-02-14"}],"E748":[{"end":"1970-03-02","
start":"1861-03-17","name":"LUPARA"},{"name":"LUPARA","start":"1970-03-03","
end":"9999-12-31"}],"D046":[{"end":"1928-07-18","start":"1861-03-17","name":
"CORSIONE"},{"name":"CORSIONE","end":"9999-12-31","start":"1947-10-20"}],"F0
98":[{"name":"MELE","end":"9999-12-31","start":"1861-03-17"}],"A752":[{"name
":"BELLINZAGO NOVARESE","end":"9999-12-31","start":"1863-04-23"}],"I874":[{"
end":"9999-12-31","start":"1871-01-01","name":"SOVERIA MANNELLI"}],"M395":[{
"name":"VALVARRONE","start":"2018-01-01","end":"9999-12-31"}],"I186":[{"name
":"SANT'EUFEMIA DELLA FONTE","start":"1862-09-22","end":"1928-11-26"}],"H791
":[{"name":"SAN CASCIANO IN VAL DI PESA","start":"1865-08-13","end":"9999-12
-31"}],"B272":[{"name":"BURAGO DI MOLGORA","end":"1978-02-15","start":"1862-
12-29"},{"start":"1978-02-16","end":"2009-06-29","name":"BURAGO DI MOLGORA"}
,{"name":"BURAGO DI MOLGORA","end":"9999-12-31","start":"2009-06-30"}],"D031
":[{"end":"1923-02-07","start":"1921-01-05","name":"CORONA"},{"start":"1923-
02-08","end":"1927-01-11","name":"CORONA"},{"start":"1927-01-12","end":"1928
-09-08","name":"CORONA"}],"L167":[{"name":"TIGLIETO","end":"1927-01-11","sta
rt":"1861-03-17"},{"name":"TIGLIETO","end":"1933-05-19","start":"1927-01-12"
},{"start":"1933-05-20","end":"9999-12-31","name":"TIGLIETO"}],"M111":[{"nam
e":"VOGOGNA","start":"1861-03-17","end":"1992-05-22"},{"start":"1992-05-23",
"end":"9999-12-31","name":"VOGOGNA"}],"G330":[{"name":"PARELLA","end":"1927-
01-11","start":"1861-03-17"},{"name":"PARELLA","start":"1927-01-12","end":"1
929-04-12"},{"start":"1947-10-11","end":"9999-12-31","name":"PARELLA"}],"C78
1":[{"name":"CIVITELLA DEL TRONTO","start":"1861-03-17","end":"9999-12-31"}]
,"B879":[{"start":"1868-02-14","end":"9999-12-31","name":"CASALE SUL SILE"}]
,"L820":[{"name":"VEZZANO SUL CROSTOLO","start":"1872-03-03","end":"9999-12-
31"}],"B512":[{"end":"1927-10-20","start":"1861-03-17","name":"CAMPIGLIONE"}
,{"name":"CAMPIGLIONE","start":"1927-10-21","end":"1929-04-12"},{"end":"9999
-12-31","start":"1929-04-13","name":"CAMPIGLIONE FENILE"}],"C569":[{"name":"
CESATE","end":"9999-12-31","start":"1861-03-17"}],"A418":[{"name":"ARMO","en
d":"1928-09-05","start":"1861-03-17"},{"end":"9999-12-31","start":"1949-03-2
4","name":"ARMO"}],"I147":[{"end":"9999-12-31","start":"1863-04-10","name":"
SAN SALVATORE DI FITALIA"}],"I485":[{"name":"SCALA COELI","start":"1863-04-1
0","end":"9999-12-31"}],"L095":[{"start":"1861-03-17","end":"1947-09-15","na
me":"TENDA"}],"B300":[{"name":"BUSTO ARSIZIO","end":"1927-04-13","start":"18
61-03-17"},{"name":"BUSTO ARSIZIO","end":"1928-05-21","start":"1927-04-14"},
{"name":"BUSTO ARSIZIO","start":"1928-05-22","end":"9999-12-31"}],"C722":[{"
name":"CIRI\u00c8","start":"1861-03-17","end":"1927-12-30"},{"name":"CIRI\u0
0c8","end":"1947-01-08","start":"1927-12-31"},{"name":"CIRI\u00c8","end":"99
99-12-31","start":"1947-01-09"}],"F523":[{"name":"MONTEGRIDOLFO","end":"1992
-04-15","start":"1861-03-17"},{"name":"MONTEGRIDOLFO","start":"1992-04-16","
end":"9999-12-31"}],"I612":[{"name":"SENNA LODIGIANA","start":"1869-03-19","
end":"1992-04-15"},{"name":"SENNA LODIGIANA","start":"1992-04-16","end":"999
9-12-31"}],"I790":[{"end":"1928-05-03","start":"1868-01-01","name":"SOLAROLO
 RAINERIO"},{"name":"SOLAROLO RAINERIO","end":"9999-12-31","start":"1947-09-
30"}],"D087":[{"end":"1923-12-05","start":"1862-11-10","name":"COSIO D'ARROS
CIA"},{"start":"1923-12-06","end":"1925-08-21","name":"COSIO D'ARROSCIA"},{"
name":"COSIO D'ARROSCIA","start":"1925-08-22","end":"9999-12-31"}],"L923":[{
"name":"VILLACHIARA","end":"9999-12-31","start":"1861-03-17"}],"A114":[{"sta
rt":"1861-03-17","end":"1927-11-08","name":"AIZURRO"}],"B052":[{"name":"BORN
ATE","end":"1927-01-11","start":"1861-03-17"},{"end":"1927-10-08","start":"1
927-01-12","name":"BORNATE"}],"C737":[{"name":"CISSICO","start":"1920-10-16"
,"end":"1923-02-13"},{"name":"CISSICO","start":"1923-02-14","end":"1923-04-2
6"},{"start":"1923-04-27","end":"1927-01-11","name":"ISSENGO"},{"name":"ISSE
NGO","start":"1927-01-12","end":"1928-03-07"}],"M285":[{"start":"1984-05-24"
,"end":"2005-12-31","name":"CARDEDU"},{"end":"2016-04-27","start":"2006-01-0
1","name":"CARDEDU"},{"name":"CARDEDU","start":"2016-04-28","end":"9999-12-3
1"}],"C493":[{"name":"CERCINO","end":"9999-12-31","start":"1861-03-17"}],"A8
97":[{"start":"1861-03-17","end":"9999-12-31","name":"BIVONGI"}],"B961":[{"e
nd":"1992-04-15","start":"1861-03-17","name":"CASELLE LANDI"},{"start":"1992
-04-16","end":"9999-12-31","name":"CASELLE LANDI"}],"I747":[{"name":"SINAGRA
","start":"1861-03-17","end":"9999-12-31"}],"A840":[{"name":"BIACESA","end":
"1923-02-13","start":"1920-10-16"},{"name":"BIACESA","end":"1925-09-14","sta
rt":"1923-02-14"},{"name":"BIACESA","start":"1925-09-15","end":"1928-04-03"}
],"C122":[{"name":"CASTELGUGLIELMO","end":"9999-12-31","start":"1866-11-19"}
],"I190":[{"name":"SANT'AGATA FOSSILI","end":"9999-12-31","start":"1863-02-2
3"}],"B865":[{"end":"9999-12-31","start":"1861-03-17","name":"CASALBORDINO"}
],"G417":[{"name":"PEGOGNAGA","start":"1876-11-20","end":"9999-12-31"}],"D94
2":[{"start":"1866-11-19","end":"9999-12-31","name":"GAVELLO"}],"A714":[{"en
d":"1919-04-06","start":"1866-11-19","name":"BATTAGLIA"},{"name":"BATTAGLIA"
,"start":"1919-04-07","end":"1925-04-27"},{"start":"1925-04-28","end":"1993-
07-13","name":"BATTAGLIA TERME"},{"start":"1993-07-14","end":"9999-12-31","n
ame":"BATTAGLIA TERME"}],"C978":[{"name":"COPERTINO","start":"1861-03-17","e
nd":"9999-12-31"}],"C137":[{"name":"CASTELLANIA","end":"1928-08-06","start":
"1861-03-17"},{"name":"CASTELLANIA","end":"2019-05-30","start":"1947-10-01"}
,{"end":"9999-12-31","start":"2019-05-31","name":"CASTELLANIA COPPI"}],"Z253
":[{"start":"0000-01-01","end":"9999-12-31","name":"Azerbaigian"}],"D153":[{
"start":"1921-01-05","end":"1923-02-07","name":"CRENOVIZZA"},{"start":"1923-
02-08","end":"1947-09-15","name":"CRENOVIZZA"}],"C215":[{"start":"1867-07-22
","end":"1928-08-13","name":"CASTELNOVO BARIANO"},{"start":"1946-12-19","end
":"9999-12-31","name":"CASTELNOVO BARIANO"}],"B206":[{"name":"BROVELLO","sta
rt":"1861-03-17","end":"1928-09-25"}],"E148":[{"end":"9999-12-31","start":"1
861-03-17","name":"GRASSOBBIO"}],"A943":[{"end":"1923-02-13","start":"1920-1
0-16","name":"BOLLONE"},{"name":"BOLLONE","start":"1923-02-14","end":"1929-0
2-28"}],"L539":[{"start":"1861-03-17","end":"9999-12-31","name":"VAILATE"}],
"H315":[{"name":"RIPALTA CREMASCA","end":"1955-03-11","start":"1928-05-29"},
{"end":"9999-12-31","start":"1955-03-12","name":"RIPALTA CREMASCA"}],"B956":
[{"name":"CASELLA","start":"1893-08-30","end":"9999-12-31"}],"I559":[{"name"
:"SECL\u00cc","start":"1861-03-17","end":"1928-10-06"},{"name":"SECL\u00cc",
"start":"1947-09-24","end":"9999-12-31"}],"G591":[{"end":"1927-01-11","start
":"1861-03-17","name":"PICINISCO"},{"start":"1927-01-12","end":"9999-12-31",
"name":"PICINISCO"}],"A152":[{"end":"1928-09-22","start":"1861-03-17","name"
:"ALBESE"}],"B014":[{"start":"1863-03-30","end":"9999-12-31","name":"SUARDI"
}],"F641":[{"name":"MONTESCUDO","end":"1920-03-14","start":"1862-08-15"},{"n
ame":"MONTESCUDO","end":"1992-04-15","start":"1920-03-15"},{"name":"MONTESCU
DO","start":"1992-04-16","end":"2015-12-31"}],"C575":[{"start":"1875-09-07",
"end":"1927-01-11","name":"CESI"},{"name":"CESI","end":"1927-01-12","start":
"1927-01-12"}],"G236":[{"name":"PAGLIERES","start":"1861-03-17","end":"1929-
01-21"}],"I786":[{"end":"9999-12-31","start":"1919-08-04","name":"SOLARO"}],
"H191":[{"start":"1920-10-16","end":"1923-02-13","name":"RASUN DI SOTTO"},{"
name":"RASUN DI SOTTO","start":"1923-02-14","end":"1927-01-11"},{"name":"RAS
UN DI SOTTO","end":"1928-10-17","start":"1927-01-12"}],"I654":[{"end":"1928-
09-24","start":"1861-03-17","name":"SERRA SANT'ABBONDIO"},{"name":"SERRA SAN
T'ABBONDIO","end":"1998-09-03","start":"1946-02-06"},{"name":"SERRA SANT'ABB
ONDIO","end":"9999-12-31","start":"1998-09-04"}],"E444":[{"name":"LANZO D'IN
TELVI","start":"1862-12-29","end":"1928-04-30"},{"name":"LANZO D'INTELVI","e
nd":"1936-10-23","start":"1928-05-01"},{"end":"2016-12-31","start":"1936-10-
24","name":"LANZO D'INTELVI"}],"F636":[{"start":"1867-02-24","end":"9999-12-
31","name":"MONTESARCHIO"}],"L767":[{"start":"1861-03-17","end":"1927-01-11"
,"name":"VERGOBBIO"},{"name":"VERGOBBIO","end":"1928-02-15","start":"1927-01
-12"}],"L634":[{"name":"VALMADRERA","end":"1992-04-15","start":"1861-03-17"}
,{"name":"VALMADRERA","start":"1992-04-16","end":"9999-12-31"}],"D460":[{"na
me":"FAETTO","start":"1861-03-17","end":"1928-04-30"}],"F939":[{"end":"1937-
06-09","start":"1861-03-17","name":"NOSATE"},{"name":"NOSATE","start":"1937-
06-10","end":"9999-12-31"}],"C181":[{"name":"CASTELLO VALTRAVAGLIA","end":"1
927-01-11","start":"1863-07-13"},{"start":"1927-01-12","end":"1928-07-28","n
ame":"CASTELLO VALTRAVAGLIA"},{"name":"CASTELVECCANA","start":"1928-07-29","
end":"9999-12-31"}],"G241":[{"name":"PAGNONA","end":"1992-04-15","start":"18
61-03-17"},{"end":"9999-12-31","start":"1992-04-16","name":"PAGNONA"}],"I019
":[{"start":"1862-11-07","end":"9999-12-31","name":"SAN MARZANO SUL SARNO"}]
,"F403":[{"name":"MONTALDEO","end":"9999-12-31","start":"1861-03-17"}],"I313
":[{"name":"SANTA VALERIA","start":"1928-04-03","end":"1947-10-01"}],"B554":
[{"start":"1866-11-19","end":"9999-12-31","name":"CAMPONOGARA"}],"B177":[{"n
ame":"BRIGNANO DEL CURONE","end":"1928-06-18","start":"1863-02-23"}],"B938":
[{"start":"1861-03-17","end":"9999-12-31","name":"CASARILE"}],"I103":[{"end"
:"9999-12-31","start":"1866-11-19","name":"SAN PIETRO DI FELETTO"}],"G505":[
{"name":"PESSINETTO","end":"1927-12-23","start":"1861-03-17"},{"end":"9999-1
2-31","start":"1934-02-01","name":"PESSINETTO"}],"L869":[{"end":"1941-12-09"
,"start":"1866-11-19","name":"VIGASIO"},{"name":"VIGASIO","start":"1941-12-1
0","end":"9999-12-31"}],"G258":[{"name":"PALAU","end":"2005-12-31","start":"
1959-04-08"},{"start":"2006-01-01","end":"2016-04-27","name":"PALAU"},{"end"
:"9999-12-31","start":"2016-04-28","name":"PALAU"}],"L262":[{"end":"1929-04-
08","start":"1861-03-17","name":"TORRE DE' NEGRI"},{"start":"1947-11-25","en
d":"9999-12-31","name":"TORRE DE' NEGRI"}],"E465":[{"name":"LASTEBASSE","sta
rt":"1866-11-19","end":"9999-12-31"}],"A010":[{"end":"9999-12-31","start":"1
869-04-05","name":"ABBIATEGRASSO"}],"G726":[{"end":"1927-09-19","start":"186
1-03-17","name":"PIZZOLI"},{"start":"1927-09-20","end":"9999-12-31","name":"
PIZZOLI"}],"E151":[{"end":"1928-04-30","start":"1861-03-17","name":"GRAVEDON
A"},{"start":"1928-05-01","end":"1928-12-11","name":"GRAVEDONA"},{"name":"GR
AVEDONA","start":"1928-12-12","end":"1948-03-20"},{"end":"2011-02-10","start
":"1948-03-21","name":"GRAVEDONA"}],"A550":[{"name":"BAGNARIA","start":"1863
-03-30","end":"1929-05-27"},{"end":"9999-12-31","start":"1946-12-19","name":
"BAGNARIA"}],"H105":[{"start":"1928-03-07","end":"1947-10-01","name":"QUARNA
"}],"L788":[{"name":"VERRUA PO","start":"1929-04-19","end":"1954-07-14"},{"n
ame":"VERRUA PO","end":"9999-12-31","start":"1954-07-15"}],"I592":[{"end":"1
903-10-19","start":"1866-11-19","name":"SELVA BELLUNESE"},{"start":"1903-10-
20","end":"9999-12-31","name":"SELVA DI CADORE"}],"A199":[{"start":"1861-03-
17","end":"1927-01-11","name":"ALICE SUPERIORE"},{"name":"ALICE SUPERIORE","
start":"1927-01-12","end":"1929-05-13"},{"name":"ALICE SUPERIORE","end":"194
5-10-04","start":"1929-05-14"},{"name":"ALICE SUPERIORE","start":"1945-10-05
","end":"1947-01-09"},{"name":"ALICE SUPERIORE","end":"2018-12-31","start":"
1947-01-10"}],"H633":[{"name":"RUFFIA","start":"1861-03-17","end":"1928-07-2
7"},{"end":"9999-12-31","start":"1947-05-23","name":"RUFFIA"}],"H406":[{"end
":"1929-05-20","start":"1863-01-26","name":"ROCCAFORTE LIGURE"},{"end":"1962
-05-25","start":"1929-05-21","name":"ROCCAFORTE LIGURE"},{"end":"9999-12-31"
,"start":"1962-05-26","name":"ROCCAFORTE LIGURE"}],"D604":[{"name":"FINO DEL
 MONTE","start":"1863-07-13","end":"1929-05-13"},{"name":"FINO DEL MONTE","e
nd":"9999-12-31","start":"1947-05-22"}],"B692":[{"name":"CAPRAUNA","start":"
1861-03-17","end":"9999-12-31"}],"G078":[{"end":"9999-12-31","start":"1861-0
3-17","name":"OPERA"}],"B844":[{"name":"CARTIGLIANO","start":"1866-11-19","e
nd":"9999-12-31"}],"L655":[{"name":"VALTORTA","start":"1861-03-17","end":"99
99-12-31"}],"G908":[{"start":"1861-03-17","end":"1862-11-02","name":"PORTO"}
,{"end":"1870-08-02","start":"1862-11-03","name":"PORTO D'ADDA"}],"B710":[{"
name":"CAPRINO BERGAMASCO","start":"1861-03-17","end":"1927-09-07"},{"end":"
9999-12-31","start":"1927-09-08","name":"CAPRINO BERGAMASCO"}],"E166":[{"nam
e":"GRESSONEI","end":"1946-01-29","start":"1939-10-21"},{"name":"GRESSONEY",
"end":"1939-10-20","start":"1928-06-30"},{"name":"GRESSONEY","end":"1946-05-
15","start":"1946-01-30"}],"D509":[{"name":"FASCIA","end":"1923-07-22","star
t":"1861-03-17"},{"name":"FASCIA","end":"9999-12-31","start":"1923-07-23"}],
"F785":[{"end":"9999-12-31","start":"1861-03-17","name":"MOZZAGROGNA"}],"B53
5":[{"name":"CAMPOFIORITO","start":"1861-03-17","end":"2008-05-23"},{"name":
"CAMPOFIORITO","end":"9999-12-31","start":"2008-05-24"}],"I242":[{"end":"999
9-12-31","start":"1866-11-19","name":"SANTA MARIA DI SALA"}],"I849":[{"end":
"9999-12-31","start":"1868-01-01","name":"SORESINA"}],"C520":[{"name":"CERNO
BBIO","start":"1861-03-17","end":"1929-05-02"},{"name":"CERNOBBIO","start":"
1929-05-03","end":"9999-12-31"}],"I635":[{"name":"SERNAGLIA","start":"1866-1
1-19","end":"1924-04-24"},{"name":"SERNAGLIA DELLA BATTAGLIA","start":"1924-
04-25","end":"9999-12-31"}],"A799":[{"name":"BERLINGO","start":"1861-03-17",
"end":"9999-12-31"}],"A907":[{"end":"1927-09-03","start":"1868-07-13","name"
:"BOARA POLESINE"}],"D816":[{"name":"FUIPIANO AL BREMBO","end":"1915-02-07",
"start":"1864-06-23"},{"end":"1928-04-17","start":"1915-02-08","name":"FUIPI
ANO AL BREMBO"}],"E479":[{"start":"1863-04-10","end":"1930-11-05","name":"LA
UREANA DI BORELLO"},{"name":"LAUREANA DI BORRELLO","end":"9999-12-31","start
":"1930-11-06"}],"C839":[{"name":"COLICO","end":"1992-04-15","start":"1861-0
3-17"},{"end":"9999-12-31","start":"1992-04-16","name":"COLICO"}],"L875":[{"
end":"1928-09-09","start":"1861-03-17","name":"VIGGIONA"}],"C232":[{"start":
"1861-03-17","end":"1929-04-04","name":"CASTELNUOVO D'ASTI"},{"end":"1930-03
-29","start":"1929-04-05","name":"CASTELNUOVO D'ASTI"},{"start":"1930-03-30"
,"end":"1935-04-07","name":"CASTELNUOVO DON BOSCO"},{"name":"CASTELNUOVO DON
 BOSCO","end":"1947-06-22","start":"1935-04-08"},{"name":"CASTELNUOVO DON BO
SCO","end":"9999-12-31","start":"1947-06-23"}],"Z344":[{"start":"0000-01-01"
,"end":"9999-12-31","name":"Sierra Leone"}],"B283":[{"end":"2015-12-31","sta
rt":"1861-03-17","name":"BUSANA"}],"A304":[{"name":"ANTEGNATE","start":"1861
-03-17","end":"9999-12-31"}],"B110":[{"name":"BOZZOLO","end":"9999-12-31","s
tart":"1868-07-01"}],"E766":[{"end":"1923-02-07","start":"1921-01-05","name"
:"LUSSINPICCOLO"},{"start":"1923-02-08","end":"1947-09-15","name":"LUSSINPIC
COLO"}],"F185":[{"name":"MEZZOLAGO","start":"1920-10-16","end":"1923-02-13"}
,{"name":"MEZZOLAGO","end":"1928-04-03","start":"1923-02-14"}],"D068":[{"nam
e":"CORTE PALASIO","start":"1861-03-17","end":"1992-04-15"},{"start":"1992-0
4-16","end":"9999-12-31","name":"CORTE PALASIO"}],"G425":[{"name":"PELLESTRI
NA","end":"1923-06-10","start":"1866-11-19"}],"L971":[{"end":"1923-02-13","s
tart":"1920-10-16","name":"VILLANDRO"},{"name":"VILLANDRO","end":"1927-01-11
","start":"1923-02-14"},{"name":"VILLANDRO","start":"1927-01-12","end":"1929
-06-03"},{"name":"VILLANDRO","start":"1929-06-04","end":"9999-12-31"}],"C227
":[{"start":"1861-03-17","end":"9999-12-31","name":"CASTELNUOVO BERARDENGA"}
],"E339":[{"end":"1992-04-15","start":"1863-02-06","name":"ISOLA DI CAPO RIZ
ZUTO"},{"start":"1992-04-16","end":"9999-12-31","name":"ISOLA DI CAPO RIZZUT
O"}],"E033":[{"end":"9999-12-31","start":"1965-08-04","name":"GINESTRA"}],"A
077":[{"end":"9999-12-31","start":"1861-03-17","name":"AGNANA CALABRA"}],"D4
29":[{"name":"ESANATOGLIA","end":"9999-12-31","start":"1862-09-29"}],"B777":
[{"name":"CARIGNANO","start":"1861-03-17","end":"9999-12-31"}],"H327":[{"sta
rt":"1867-05-18","end":"1964-06-26","name":"RIVAMONTE"},{"end":"9999-12-31",
"start":"1964-06-27","name":"RIVAMONTE AGORDINO"}],"M301":[{"start":"1996-01
-24","end":"2005-12-31","name":"PADRU"},{"start":"2006-01-01","end":"2016-04
-27","name":"PADRU"},{"name":"PADRU","start":"2016-04-28","end":"9999-12-31"
}],"I703":[{"end":"1957-07-25","start":"1861-03-17","name":"SETTIMO TORINESE
"},{"name":"SETTIMO TORINESE","end":"9999-12-31","start":"1957-07-26"}],"G61
1":[{"name":"PIETRADEFUSI","end":"1948-06-28","start":"1861-03-17"},{"start"
:"1948-06-29","end":"9999-12-31","name":"PIETRADEFUSI"}],"F878":[{"name":"NE
TRO","end":"1927-01-11","start":"1870-01-11"},{"name":"NETRO","end":"1992-04
-15","start":"1927-01-12"},{"name":"NETRO","start":"1992-04-16","end":"9999-
12-31"}],"I094":[{"end":"1925-11-29","start":"1861-03-17","name":"SAN PIETRO
 A PATIERNO"}],"H472":[{"name":"RODDI","end":"9999-12-31","start":"1861-03-1
7"}],"C400":[{"name":"CAVIZZANA","start":"1920-10-16","end":"1923-02-13"},{"
end":"1928-12-31","start":"1923-02-14","name":"CAVIZZANA"},{"start":"1956-11
-20","end":"9999-12-31","name":"CAVIZZANA"}],"F486":[{"name":"MONTE DI MALO"
,"start":"1866-11-19","end":"9999-12-31"}],"L484":[{"end":"9999-12-31","star
t":"1861-03-17","name":"UGENTO"}],"G126":[{"name":"ORSENIGO","start":"1861-0
3-17","end":"9999-12-31"}],"H924":[{"name":"SAN GIOVANNI LUPATOTO","end":"19
27-05-11","start":"1866-11-19"},{"end":"9999-12-31","start":"1932-08-19","na
me":"SAN GIOVANNI LUPATOTO"}],"F211":[{"name":"MILLAN-SARNES","start":"1920-
10-16","end":"1923-02-13"},{"end":"1927-01-11","start":"1923-02-14","name":"
MILLAN-SARNES"},{"end":"1928-09-08","start":"1927-01-12","name":"MILLAN-SARN
ES"}],"H332":[{"start":"1923-04-27","end":"1927-01-11","name":"RIVA DI TURES
"},{"start":"1927-01-12","end":"1929-02-08","name":"RIVA DI TURES"},{"name":
"RIVA IN VALLE","end":"1923-02-13","start":"1920-10-16"},{"end":"1923-04-26"
,"start":"1923-02-14","name":"RIVA IN VALLE"}],"E751":[{"end":"9999-12-31","
start":"1861-03-17","name":"LURANO"}],"A257":[{"end":"9999-12-31","start":"1
861-03-17","name":"AMATO"}],"H705":[{"name":"SALETTO","end":"2018-02-16","st
art":"1866-11-19"}],"L188":[{"end":"9999-12-31","start":"1861-03-17","name":
"TODI"}],"B393":[{"end":"9999-12-31","start":"1861-03-17","name":"CALCINATE"
}],"B099":[{"name":"BOVA MARINA","end":"9999-12-31","start":"1908-04-24"}],"
A404":[{"name":"ARIENZO SAN FELICE","start":"1928-09-25","end":"1945-08-31"}
,{"start":"1945-09-01","end":"1946-12-21","name":"ARIENZO SAN FELICE"}],"M08
4":[{"name":"VITICUSO ED ACQUAFONDATA","end":"1902-07-23","start":"1863-02-0
6"}],"G707":[{"name":"PISCIOTTA","end":"9999-12-31","start":"1861-03-17"}],"
L544":[{"start":"1927-10-11","end":"9999-12-31","name":"VALBONDIONE"}],"F546
":[{"name":"MONTELLA","end":"9999-12-31","start":"1861-03-17"}],"I951":[{"na
me":"STEZZANO","end":"1963-02-06","start":"1861-03-17"},{"name":"STEZZANO","
start":"1963-02-07","end":"9999-12-31"}],"G999":[{"name":"PRATO","end":"1949
-09-07","start":"1932-02-18"},{"start":"1949-09-08","end":"1992-04-15","name
":"PRATO"},{"end":"9999-12-31","start":"1992-04-16","name":"PRATO"},{"start"
:"1863-04-13","end":"1932-02-17","name":"PRATO IN TOSCANA"}],"D598":[{"start
":"1877-05-18","end":"1927-01-11","name":"FINALBORGO"},{"start":"1927-01-12"
,"end":"1927-01-12","name":"FINALBORGO"}],"G696":[{"name":"PIOZZANO","start"
:"1877-04-16","end":"9999-12-31"}],"E379":[{"start":"1861-03-17","end":"1927
-01-11","name":"IVREA"},{"name":"IVREA","start":"1927-01-12","end":"1945-10-
04"},{"end":"1968-10-21","start":"1945-10-05","name":"IVREA"},{"name":"IVREA
","start":"1968-10-22","end":"9999-12-31"}],"E073":[{"name":"GODOVICI","star
t":"1921-01-05","end":"1921-12-30"},{"start":"1921-12-31","end":"1923-02-07"
,"name":"GODOVICI"},{"end":"1927-01-11","start":"1923-02-08","name":"GODOVIC
I"},{"start":"1927-01-12","end":"1928-01-25","name":"GODOVICI"}],"L931":[{"n
ame":"VILLADEATI","start":"1861-03-17","end":"1928-01-28"},{"name":"VILLADEA
TI","end":"9999-12-31","start":"1928-01-29"}],"D248":[{"end":"1923-02-13","s
tart":"1920-10-16","name":"DAONE"},{"name":"DAONE","end":"1928-03-06","start
":"1923-02-14"},{"end":"2014-12-31","start":"1952-09-20","name":"DAONE"}],"F
347":[{"name":"MONDAVIO","start":"1861-03-17","end":"1998-09-03"},{"name":"M
ONDAVIO","end":"9999-12-31","start":"1998-09-04"}],"E439":[{"name":"LANGOSCO
","end":"9999-12-31","start":"1861-03-17"}],"Z148":[{"end":"9999-12-31","sta
rt":"0000-01-01","name":"Macedonia del Nord"}],"L294":[{"name":"TORRICELLA",
"end":"9999-12-31","start":"1954-10-29"}],"C879":[{"name":"COLLIANO","start"
:"1861-03-17","end":"9999-12-31"}],"F296":[{"start":"1920-10-16","end":"1923
-02-13","name":"MOLLARO"},{"end":"1929-01-23","start":"1923-02-14","name":"M
OLLARO"}],"B781":[{"name":"CARINOLA","end":"1927-01-11","start":"1861-03-17"
},{"name":"CARINOLA","end":"1945-08-31","start":"1927-01-12"},{"name":"CARIN
OLA","end":"1964-08-31","start":"1945-09-01"},{"start":"1964-09-01","end":"9
999-12-31","name":"CARINOLA"}],"A108":[{"name":"AIMAVILLA","end":"1946-01-29
","start":"1939-07-04"},{"name":"AYMAVILLE","start":"1946-01-30","end":"1977
-01-01"},