import hashlib
import json
import locale
import logging
import platform

import requests

from homematicip.base.base_connection import BaseConnection, HmipThrottlingError

logger = logging.getLogger(__name__)


class Connection(BaseConnection):
    def init(
        self,
        accesspoint_id,
        lookup=True,
        lookup_url="https://lookup.homematic.com:48335/getHost",
        **kwargs,
    ):
        self.set_token_and_characteristics(accesspoint_id)

        if lookup:
            while True:
                try:
                    result = requests.post(
                        lookup_url, json=self.clientCharacteristics, timeout=3
                    )
                    js = json.loads(result.text)
                    self._urlREST = js["urlREST"]
                    self._urlWebSocket = js["urlWebSocket"]
                    break
                except Exception as e:
                    pass
        else:  # pragma: no cover
            self._urlREST = "https://ps1.homematic.com:6969"
            self._urlWebSocket = "wss://ps1.homematic.com:8888"

    def _rest_call(self, path, body=None):
        result = None
        requestPath = "{}/hmip/{}".format(self._urlREST, path)
        logger.debug("_restcall path(%s) body(%s)", requestPath, body)
        for i in range(0, self._restCallRequestCounter):
            try:
                result = requests.post(
                    requestPath,
                    data=body,
                    headers=self.headers,
                    timeout=self._restCallTimout,
                )

                # The HMIP cloud returns a 429 Error when it has received too
                # many requests from a given client
                if result.status_code == 429:
                    logger.warning(f"_restcall {requestPath} returned 429 status code.")
                    raise HmipThrottlingError

                ret = result.json() if len(result.content) != 0 else ""
                logger.debug(
                    "_restcall result: Errorcode=%s content(%s)",
                    result.status_code,
                    ret,
                )
                return ret
            except requests.Timeout:
                logger.error("call to '%s' failed due Timeout", requestPath)
                pass
        return {"errorCode": "TIMEOUT"}
