# Copyright (c) 2025 Thomas Goirand
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

try:
    from unittest import mock
except ImportError:
    import mock
from osc_lib.cli import parseractions
from osc_lib import utils

from vmmsclient.v2.client import AddVMCommand, ListVMsCommand, RemoveVMCommand, ShowVMCommand, SetVMCommand, UnsetVMCommand

DEFAULT_API_VERSION = '2'
API_VERSION_OPTION = 'os_vmms_api_version'
API_NAME = 'vmms'
API_VERSIONS = {
    '2': 'vmmsclient.v2.client.Client',
}

__all__ = [
    'AddVMCommand',
    'ListVMsCommand', 
    'RemoveVMCommand',
    'ShowVMCommand',
    'SetVMCommand',
    'UnsetVMCommand'
]

def make_client(instance):
    """Returns a client to the ClientManager."""
    # Safely get the configuration value
    config = instance.get_configuration()
    requested_api_version = config.get(API_VERSION_OPTION, DEFAULT_API_VERSION)
    
    client_class = utils.get_client_class(API_NAME, requested_api_version, API_VERSIONS)
    
    kwargs = {
        'session': instance.session,
        'service_type': 'vmms',
        'region_name': instance.region_name,
        'interface': instance.interface,
    }
    
    client = client_class(**kwargs)
    return client

def build_option_parser(parser):
    """Hook to add global options."""
    parser.add_argument(
        '--os-vmms-api-version',
        metavar='<vmms-api-version>',
        default=utils.env('OS_VMMS_API_VERSION') or DEFAULT_API_VERSION,
        help='VM Migration Scheduler API version (default: %(default)s)')
    return parser
