/* Public include file for server support */
/*
 * This file is part of the SSH Library
 *
 * Copyright (c) 2003-2008 by Aris Adamantiadis
 *
 * The SSH Library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The SSH Library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the SSH Library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/**
 * @defgroup libssh_server The libssh server API
 *
 * @{
 */

#ifndef SERVER_H
#define SERVER_H

#include "libssh/libssh.h"
#define SERVERBANNER CLIENTBANNER

#ifdef __cplusplus
extern "C" {
#endif

enum ssh_bind_options_e {
  SSH_BIND_OPTIONS_BINDADDR,
  SSH_BIND_OPTIONS_BINDPORT,
  SSH_BIND_OPTIONS_BINDPORT_STR,
  SSH_BIND_OPTIONS_HOSTKEY,
  SSH_BIND_OPTIONS_DSAKEY,
  SSH_BIND_OPTIONS_RSAKEY,
  SSH_BIND_OPTIONS_BANNER,
  SSH_BIND_OPTIONS_LOG_VERBOSITY,
  SSH_BIND_OPTIONS_LOG_VERBOSITY_STR
};

typedef struct ssh_bind_struct* ssh_bind;

/* Callback functions */

/**
 * @brief Incoming connection callback. This callback is called when a ssh_bind
 *        has a new incoming connection.
 * @param sshbind Current sshbind session handler
 * @param message the actual message
 * @param userdata Userdata to be passed to the callback function.
 */
typedef void (*ssh_bind_incoming_connection_callback) (ssh_bind sshbind,
    void *userdata);

/**
 * @brief These are the callbacks exported by the ssh_bind structure.
 *
 * They are called by the server module when events appear on the network.
 */
struct ssh_bind_callbacks_struct {
  /** DON'T SET THIS use ssh_callbacks_init() instead. */
  size_t size;
  /** A new connection is available. */
  ssh_bind_incoming_connection_callback incoming_connection;
};
typedef struct ssh_bind_callbacks_struct *ssh_bind_callbacks;

/**
 * @brief Creates a new SSH server bind.
 *
 * @return A newly allocated ssh_bind session pointer.
 */
LIBSSH_API ssh_bind ssh_bind_new(void);

/**
 * @brief Set the options for the current SSH server bind.
 *
 * @param  sshbind     The ssh server bind to configure.
 *
 * @param  type The option type to set. This could be one of the
 *              following:
 *
 *              - SSH_BIND_OPTIONS_BINDADDR
 *                The ip address to bind (const char *).
 *
 *              - SSH_BIND_OPTIONS_BINDPORT
 *                The port to bind (unsigned int).
 *
 *              - SSH_BIND_OPTIONS_BINDPORT_STR
 *                The port to bind (const char *).
 *
 *              - SSH_BIND_OPTIONS_HOSTKEY
 *                This specifies the file containing the private host key used
 *                by SSHv1. (const char *).
 *
 *              - SSH_BIND_OPTIONS_DSAKEY
 *                This specifies the file containing the private host dsa key
 *                used by SSHv2. (const char *).
 *
 *              - SSH_BIND_OPTIONS_RSAKEY
 *                This specifies the file containing the private host dsa key
 *                used by SSHv2. (const char *).
 *
 *              - SSH_BIND_OPTIONS_BANNER
 *                That the server banner (version string) for SSH.
 *                (const char *).
 *
 *              - SSH_BIND_OPTIONS_LOG_VERBOSITY
 *                Set the session logging verbosity (int).\n
 *                \n
 *                The verbosity of the messages. Every log smaller or
 *                equal to verbosity will be shown.
 *                - SSH_LOG_NOLOG: No logging
 *                - SSH_LOG_RARE: Rare conditions or warnings
 *                - SSH_LOG_ENTRY: API-accessible entrypoints
 *                - SSH_LOG_PACKET: Packet id and size
 *                - SSH_LOG_FUNCTIONS: Function entering and leaving
 *
 *              - SSH_BIND_OPTIONS_LOG_VERBOSITY_STR
 *                Set the session logging verbosity (const char *).\n
 *                \n
 *                The verbosity of the messages. Every log smaller or
 *                equal to verbosity will be shown.
 *                - SSH_LOG_NOLOG: No logging
 *                - SSH_LOG_RARE: Rare conditions or warnings
 *                - SSH_LOG_ENTRY: API-accessible entrypoints
 *                - SSH_LOG_PACKET: Packet id and size
 *                - SSH_LOG_FUNCTIONS: Function entering and leaving
 *                \n
 *                See the corresponding numbers in libssh.h.
 *
 * @param  value The value to set. This is a generic pointer and the
 *               datatype which is used should be set according to the
 *               type set.
 *
 * @returns     SSH_OK on success, SSH_ERROR on invalid option or parameter.
 */
LIBSSH_API int ssh_bind_options_set(ssh_bind sshbind,
    enum ssh_bind_options_e type, const void *value);

/**
 * @brief Start listening to the socket.
 *
 * @param  ssh_bind_o     The ssh server bind to use.
 *
 * @return 0 on success, < 0 on error.
 */
LIBSSH_API int ssh_bind_listen(ssh_bind ssh_bind_o);

/**
 * @brief Set the callback for this bind.
 *
 * @param[in] sshbind   The bind to set the callback on.
 *
 * @param[in] callbacks An already set up ssh_bind_callbacks instance.
 *
 * @param[in] userdata  A pointer to private data to pass to the callbacks.
 *
 * @return              SSH_OK on success, SSH_ERROR if an error occured.
 *
 * @code
 *     struct ssh_callbacks_struct cb = {
 *         .userdata = data,
 *         .auth_function = my_auth_function
 *     };
 *     ssh_callbacks_init(&cb);
 *     ssh_bind_set_callbacks(session, &cb);
 * @endcode
 */
LIBSSH_API int ssh_bind_set_callbacks(ssh_bind sshbind, ssh_bind_callbacks callbacks,
    void *userdata);

/**
 * @brief  Set the session to blocking/nonblocking mode.
 *
 * @param  ssh_bind_o     The ssh server bind to use.
 *
 * @param  blocking     Zero for nonblocking mode.
 */
LIBSSH_API void ssh_bind_set_blocking(ssh_bind ssh_bind_o, int blocking);

/**
 * @brief Recover the file descriptor from the session.
 *
 * @param  ssh_bind_o     The ssh server bind to get the fd from.
 *
 * @return The file descriptor.
 */
LIBSSH_API socket_t ssh_bind_get_fd(ssh_bind ssh_bind_o);

/**
 * @brief Set the file descriptor for a session.
 *
 * @param  ssh_bind_o     The ssh server bind to set the fd.
 *
 * @param  fd           The file descriptssh_bind B
 */
LIBSSH_API void ssh_bind_set_fd(ssh_bind ssh_bind_o, socket_t fd);

/**
 * @brief Allow the file descriptor to accept new sessions.
 *
 * @param  ssh_bind_o     The ssh server bind to use.
 */
LIBSSH_API void ssh_bind_fd_toaccept(ssh_bind ssh_bind_o);

/**
 * @brief Accept an incoming ssh connection and initialize the session.
 *
 * @param  ssh_bind_o     The ssh server bind to accept a connection.
 * @param  session			A preallocated ssh session
 * @see ssh_new
 * @return SSH_OK when a connection is established
 */
LIBSSH_API int ssh_bind_accept(ssh_bind ssh_bind_o, ssh_session session);

/**
 * @brief Handles the key exchange and set up encryption
 *
 * @param  session			A connected ssh session
 * @see ssh_bind_accept
 * @return SSH_OK if the key exchange was successful
 */
LIBSSH_API int ssh_handle_key_exchange(ssh_session session);

/**
 * @brief Free a ssh servers bind.
 *
 * @param  ssh_bind_o     The ssh server bind to free.
 */
LIBSSH_API void ssh_bind_free(ssh_bind ssh_bind_o);

/* messages.c */
LIBSSH_API int ssh_message_reply_default(ssh_message msg);

LIBSSH_API char *ssh_message_auth_user(ssh_message msg);
LIBSSH_API char *ssh_message_auth_password(ssh_message msg);
LIBSSH_API ssh_public_key ssh_message_auth_publickey(ssh_message msg);
LIBSSH_API enum ssh_publickey_state_e ssh_message_auth_publickey_state(ssh_message msg);
LIBSSH_API int ssh_message_auth_reply_success(ssh_message msg,int partial);
LIBSSH_API int ssh_message_auth_reply_pk_ok(ssh_message msg, ssh_string algo, ssh_string pubkey);
LIBSSH_API int ssh_message_auth_reply_pk_ok_simple(ssh_message msg);

LIBSSH_API int ssh_message_auth_set_methods(ssh_message msg, int methods);

LIBSSH_API int ssh_message_service_reply_success(ssh_message msg);
LIBSSH_API char *ssh_message_service_service(ssh_message msg);

LIBSSH_API int ssh_message_global_request_reply_success(ssh_message msg,
                                                        uint16_t bound_port);

LIBSSH_API void ssh_set_message_callback(ssh_session session,
    int(*ssh_bind_message_callback)(ssh_session session, ssh_message msg, void *data),
    void *data);
LIBSSH_API int ssh_execute_message_callbacks(ssh_session session);

LIBSSH_API char *ssh_message_channel_request_open_originator(ssh_message msg);
LIBSSH_API int ssh_message_channel_request_open_originator_port(ssh_message msg);
LIBSSH_API char *ssh_message_channel_request_open_destination(ssh_message msg);
LIBSSH_API int ssh_message_channel_request_open_destination_port(ssh_message msg);

LIBSSH_API ssh_channel ssh_message_channel_request_channel(ssh_message msg);

LIBSSH_API char *ssh_message_channel_request_pty_term(ssh_message msg);
LIBSSH_API int ssh_message_channel_request_pty_width(ssh_message msg);
LIBSSH_API int ssh_message_channel_request_pty_height(ssh_message msg);
LIBSSH_API int ssh_message_channel_request_pty_pxwidth(ssh_message msg);
LIBSSH_API int ssh_message_channel_request_pty_pxheight(ssh_message msg);

LIBSSH_API char *ssh_message_channel_request_env_name(ssh_message msg);
LIBSSH_API char *ssh_message_channel_request_env_value(ssh_message msg);

LIBSSH_API char *ssh_message_channel_request_command(ssh_message msg);

LIBSSH_API char *ssh_message_channel_request_subsystem(ssh_message msg);

LIBSSH_API char *ssh_message_global_request_address(ssh_message msg);
LIBSSH_API int ssh_message_global_request_port(ssh_message msg);

LIBSSH_API int ssh_channel_open_reverse_forward(ssh_channel channel, const char *remotehost,
    int remoteport, const char *sourcehost, int localport);

LIBSSH_API int ssh_channel_request_send_exit_status(ssh_channel channel,
                                                int exit_status);
LIBSSH_API int ssh_channel_request_send_exit_signal(ssh_channel channel,
                                                const char *signum,
                                                int core,
                                                const char *errmsg,
                                                const char *lang);
LIBSSH_API int ssh_channel_write_stderr(ssh_channel channel,
                                                const void *data,
                                                uint32_t len);

/* deprecated functions */
SSH_DEPRECATED LIBSSH_API int ssh_accept(ssh_session session);
SSH_DEPRECATED LIBSSH_API int channel_write_stderr(ssh_channel channel,
        const void *data, uint32_t len);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* SERVER_H */

/** @} */
