/*
 * Read-Copy Update mechanism for mutual exclusion 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright (c) International Business Machines Corp., 2001
 *
 * Author: Dipankar Sarma <dipankar@in.ibm.com>
 * 
 * Based on the original work by Paul McKenney <paul.mckenney@us.ibm.com>
 * and inputs from Andrea Arcangeli, Rusty Russell, Andi Kleen etc.
 * Papers:
 * http://www.rdrop.com/users/paulmck/paper/rclockpdcsproof.pdf
 * http://lse.sourceforge.net/locking/rclock_OLS.2001.05.01c.sc.pdf (OLS2001)
 *
 * For detailed explanation of Read-Copy Update mechanism see -
 * 		http://lse.sourceforge.net/locking/rcupdate.html
 *
 */

#ifndef __LINUX_RCUPDATE_H
#define __LINUX_RCUPDATE_H

#include <linux/list.h>

/*
 * Callback structure for use with call_rcu(). 
 */
struct rcu_head {
	struct list_head list;
	void (*func)(void *obj);
	void *arg;
};

#define RCU_HEAD_INIT(head) { LIST_HEAD_INIT(head.list), NULL, NULL }
#define RCU_HEAD(head) struct rcu_head head = RCU_HEAD_INIT(head)
#define INIT_RCU_HEAD(ptr) do { \
	INIT_LIST_HEAD(&(ptr)->list); (ptr)->func = NULL; (ptr)->arg = NULL; \
} while (0)

#ifdef CONFIG_PREEMPT
#define rcu_read_lock()        preempt_disable()
#define rcu_read_unlock()      preempt_enable()
#else
#define rcu_read_lock()                do {} while(0)
#define rcu_read_unlock()      do {} while(0)
#endif
	
extern void FASTCALL(call_rcu(struct rcu_head *head, void (*func)(void *arg), void *arg));
extern void synchronize_kernel(void);

extern void rcu_init(void);

#endif /* __LINUX_RCUPDATE_H */
