/*
 * Network disk client
 * Server RPC info, XDR routines
 *
 * Copyright (c) 1996 Petr Salinger
 * Copyright (c) 2001 Lubomir Bulej <pallas@kadan.cz>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <linux/kernel.h>
#include <linux/sunrpc/clnt.h>

#include "nwd_xdr.h"
#include "nwd_dbg.h"


/* 
 * Space requirements for procedure arguments and 
 * return types, all sizes are in 32-bit words
 */

#define NWD_null_sz		0
#define NWD_res_sz		1
#define NWD_handle_sz		1
#define NWD_readargs1_sz	3
#define NWD_readargs2_sz	4
#define NWD_readres_sz		2
#define NWD_writeargs1_sz	3
#define NWD_writeargs2_sz	4
#define NWD_lookupres1_sz	3
#define NWD_lookupres2_sz	4


/*****************************************************************************\
| XDR HELPER FUNCTIONS                                                        |
\*****************************************************************************/

static inline
u32 * xdr_encode_int (u32 * p, __u32 val)
{
	* p++ = htonl (val);
	return p;
} /* xdr_encode_int */


static inline
u32 * xdr_decode_int (u32 * p, __u32 * val)
{
	* val = ntohl (* p++);
	return p;
} /* xdr_decode_int */


static inline
u32 * xdr_encode_h2int (u32 * p, __u64 val)
{
	* p++ = htonl (val & 0xFFFFFFFF);
	return p;
} /* xdr_encode_h2int */


static inline
u32 * xdr_decode_int2h (u32 * p, __u64 * val)
{
	* val = (__u64) ntohl (* p++);
	return p;
} /* xdr_decode_int2h */


/*****************************************************************************\
| NWD XDR FUNCTIONS                                                           |
\*****************************************************************************/

static
int nwd_xdr_null (struct rpc_rqst * req, u32 * p, void * dummy)
{
	return -EIO;
} /* nwd_xdr_null */


/*
 * Decode result status
 */
 
static
int nwd_xdr_res (struct rpc_rqst * req, u32 * p, nwd_res * res)
{
	p = xdr_decode_int (p, res);
	if (* res != NWD_OK)
		return -(* res);
	return 0;
} /* nwd_xdr_res */


/*
 * Encode remote file handle
 */

static
int nwd_xdr_handle (struct rpc_rqst * req, u32 * p, nwd_handle * hnd)
{
	p = xdr_encode_int (p, * hnd);
	req->rq_slen = xdr_adjust_iovec (req->rq_svec, p);
	return 0;
} /* nwd_xdr_handle */


/*
 * Encode READ arguments
 *
 * We read data directly into request buffer so we setup the
 * recv iovec [1] buffer base to point at the request buffer.
 *
 * Note: auth->au_rslack contains estimate of the size of verf part of RPC
 * reply. Together with RPC reply header size and the size of (non file-data) 
 * fields of procedure result it determines the "offset" of file data in the
 * reply. Knowing that we can align the receive iovecs so that the file-data 
 * part of the reply ends up in the request buffer.
 */
 
static inline
int nwd_xdr_readargs (struct rpc_rqst * req, u32 * p, nwd_rwargs * args, const int ver)
{
	struct rpc_auth * auth = req->rq_task->tk_auth;
	int replen = (RPC_REPHDRSIZE + auth->au_rslack + NWD_readres_sz) << 2;
	
	p = xdr_encode_int (p, args->file);
	if (ver == NWD_VERSION_1)
		p = xdr_encode_h2int (p, args->offset);
	else if (ver == NWD_VERSION_2)
		p = xdr_encode_hyper (p, args->offset);
	else {
		eprintk ("cannot encode readargs for version %d\n", ver);
		return -EINVAL;
	}
		
	p = xdr_encode_int (p, args->count);
	req->rq_slen = xdr_adjust_iovec (req->rq_svec, p);


	/* set up reply iovec */
	req->rq_rcv_buf.head [0].iov_len = replen;
	req->rq_rcv_buf.tail [0].iov_len = args->count;
	req->rq_rcv_buf.tail [0].iov_base = args->buffer;
	req->rq_rcv_buf.len = replen + args->count;
	req->rq_rcv_buf.page_len = 0;
	return 0;
} /* nwd_xdr_readargs */


static
int nwd_xdr_readargs1 (struct rpc_rqst * req, u32 * p, nwd_rwargs * args)
{
	return nwd_xdr_readargs (req, p, args, NWD_VERSION_1);
} /* nwd_xdr_readargs1 */


static
int nwd_xdr_readargs2 (struct rpc_rqst * req, u32 * p, nwd_rwargs * args)
{
	return nwd_xdr_readargs (req, p, args, NWD_VERSION_2);
} /* nwd_xdr_readargs2 */


/* 
 * Decode READ result 
 *
 * Note: The file data are optional and valid only if the operation 
 * result status was NWD_OK. In that case, the file data were written
 * directly into request buffer, for which recv iovec [1] was configured.
 */

static
int nwd_xdr_readres (struct rpc_rqst * req, u32 * p, nwd_readres * res)
{
	int hdlen, recvd;
	
	p = xdr_decode_int (p, & res->status);
	if (res->status != NWD_OK)
		return -res->status;
		
	p = xdr_decode_int (p, & res->count);
	
	/* check length of returned data */
	hdlen = (u8 *) p - (u8 *) req->rq_rvec [0].iov_base;
	recvd = req->rq_rlen - hdlen;
	if (res->count > recvd) {
		wprintk ("server cheating in read reply: count (%d) > recvd (%d)\n", res->count, recvd);
		res->count = recvd;
	}
	
	return res->count;
} /* nwd_xdr_readres */


/* 
 * Encode WRITE arguments
 *
 * We send the data directly from request buffer, so we setup the
 * send iovec [1] buffer base to point at the request buffer.
 */
 
static inline
int nwd_xdr_writeargs (struct rpc_rqst * req, u32 * p, nwd_rwargs * args, const int ver)
{
	p = xdr_encode_int (p, args->file);
	if (ver == NWD_VERSION_1)
		p = xdr_encode_h2int (p, args->offset);
	else if (ver == NWD_VERSION_2)
		p = xdr_encode_hyper (p, args->offset);
	else {
		eprintk ("cannot encode writeargs for version %d\n", ver);
		return -EINVAL;
	}
	
	p = xdr_encode_int (p, args->count);
	req->rq_slen = xdr_adjust_iovec (req->rq_svec, p);
	
	req->rq_snd_buf.tail [0].iov_base = args->buffer;
	req->rq_snd_buf.tail [0].iov_len = args->count;
	req->rq_snd_buf.len += args->count;
	req->rq_snd_buf.page_len = 0;
	return 0;
	
} /* nwd_xdr_writeargs */


static
int nwd_xdr_writeargs1 (struct rpc_rqst * req, u32 * p, nwd_rwargs * args)
{
	return nwd_xdr_writeargs (req, p, args, NWD_VERSION_1);
} /* nwd_xdr_writeargs1 */


static
int nwd_xdr_writeargs2 (struct rpc_rqst * req, u32 * p, nwd_rwargs * args)
{
	return nwd_xdr_writeargs (req, p, args, NWD_VERSION_2);
} /* nwd_xdr_writeargs2 */


/* 
 * Decode LOOKUP result 
 *
 * Note: The file lookup info is optional and valid only if the 
 * operation result status was NWD_OK.
 */

static inline
int nwd_xdr_lookupres (struct rpc_rqst * req, u32 * p, nwd_lookupres * res, const int ver)
{
	p = xdr_decode_int (p, & res->status);
	if (res->status != NWD_OK)
		return -res->status;
	
	p = xdr_decode_int (p, & res->file);
	if (ver == NWD_VERSION_1)
		p = xdr_decode_int2h (p, & res->size);
	else if (ver == NWD_VERSION_2)
		p = xdr_decode_hyper (p, & res->size);
	else {
		eprintk ("cannot decode lookupres for version %d\n", ver);
		return -EINVAL;
	}
	
	return 0;
} /* nwd_xdr_lookupres */


static
int nwd_xdr_lookupres1 (struct rpc_rqst * req, u32 * p, nwd_lookupres * res)
{
	return nwd_xdr_lookupres (req, p, res, NWD_VERSION_1);
} /* nwd_xdr_lookupres1 */


static
int nwd_xdr_lookupres2 (struct rpc_rqst * req, u32 * p, nwd_lookupres * res)
{
	return nwd_xdr_lookupres (req, p, res, NWD_VERSION_2);
} /* nwd_xdr_lookupres2 */


/*****************************************************************************\
| XDR ENCODE/DECODE STRUCTURES                                                |
\*****************************************************************************/

#ifndef MAX
# define MAX(a, b)		(((a) > (b)) ? (a) : (b))
#endif

/* 
 * Template for RPC procinfo structure
 */

#define PROC(proc, atype, rtype)						\
{ 									\
	p_procname:	"nwdproc_" #proc,				\
	p_encode:	(kxdrproc_t) nwd_xdr_##atype, 			\
	p_decode:	(kxdrproc_t) nwd_xdr_##rtype,			\
	p_bufsiz:	MAX (NWD_##atype##_sz, NWD_##rtype##_sz) << 2,	\
	p_count:	0						\
}


static struct rpc_procinfo nwd_procedures_1 [] = {
	PROC (null,	null,		null),
	PROC (dispose,	handle,		res),
	PROC (read,	readargs1,	readres),
	PROC (write,	writeargs1,	res),
	PROC (lookup,	handle,		lookupres1)
}; /* nwd_procedures_1 */


static struct rpc_procinfo nwd_procedures_2 [] = {
	PROC (null,	null,		null),
	PROC (dispose,	handle,		res),
	PROC (read,	readargs2,	readres),
	PROC (write,	writeargs2,	res),
	PROC (lookup,	handle,		lookupres2),
	PROC (sync,	handle,		res)
}; /* nwd_procedures_2 */


static struct rpc_version nwd_version_1 = {
	number:		NWD_VERSION_1,
	nrprocs:	ARRAY_SIZE (nwd_procedures_1),
	procs:		nwd_procedures_1
}; /* nwd_version_1 */


static struct rpc_version nwd_version_2 = {
	number:		NWD_VERSION_2,
	nrprocs:	ARRAY_SIZE (nwd_procedures_2),
	procs:		nwd_procedures_2
}; /* nwd_version_2 */



static struct rpc_version * nwd_versions [] = {
	NULL,
	& nwd_version_1,
	& nwd_version_2
}; /* nwd_versions */


static struct rpc_stat nwd_rpcstat;


struct rpc_program nwd_program = {
	name:		"nwd",
	number:		NWD_PROGRAM,
	nrvers:		ARRAY_SIZE (nwd_versions),
	version:	nwd_versions,
	stats:		& nwd_rpcstat
}; /* nwd_program */
