/*
 *  linux/fs/proc/proc_misc.c
 *
 *  linux/fs/proc/array.c
 *  Copyright (C) 1992  by Linus Torvalds
 *  based on ideas by Darren Senn
 *
 *  This used to be the part of array.c. See the rest of history and credits
 *  there. I took this into a separate file and switched the thing to generic
 *  proc_file_inode_operations, leaving in array.c only per-process stuff.
 *  Inumbers allocation made dynamic (via create_proc_entry()).  AV, May 1999.
 *
 * Changes:
 * Fulton Green      :  Encapsulated position metric calculations.
 *			<kernel@FultonGreen.com>
 */

#include <linux/types.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/kernel_stat.h>
#include <linux/tty.h>
#include <linux/string.h>
#include <linux/mman.h>
#include <linux/proc_fs.h>
#include <linux/ioport.h>
#include <linux/config.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/swap.h>
#include <linux/slab.h>
#include <linux/smp.h>
#include <linux/signal.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/smp_lock.h>
#include <linux/seq_file.h>
#include <linux/mm_inline.h>
#include <linux/sysrq.h>

#include <asm/uaccess.h>
#include <asm/pgtable.h>
#include <asm/io.h>

#define LOAD_INT(x) ((x) >> FSHIFT)
#define LOAD_FRAC(x) LOAD_INT(((x) & (FIXED_1-1)) * 100)
/*
 * Warning: stuff below (imported functions) assumes that its output will fit
 * into one page. For some of those functions it may be wrong. Moreover, we
 * have a way to deal with that gracefully. Right now I used straightforward
 * wrappers, but this needs further analysis wrt potential overflows.
 */
extern int get_hardware_list(char *);
extern int get_stram_list(char *);
extern int get_device_list(char *);
extern int get_filesystem_list(char *);
extern int get_exec_domain_list(char *);
#ifndef CONFIG_X86
extern int get_irq_list(char *);
#endif
extern int get_dma_list(char *);
extern int get_locks_status (char *, char **, off_t, int);
extern int get_swaparea_info (char *);
#ifdef CONFIG_SGI_DS1286
extern int get_ds1286_status(char *);
#endif

extern int module_loading_enabled;

void proc_sprintf(char *page, off_t *off, int *lenp, const char *format, ...)
{
	int len = *lenp;
	va_list args;

	/* try to only print whole lines */
	if (len > PAGE_SIZE-512)
		return;

	va_start(args, format);
	len += vsnprintf(page + len, PAGE_SIZE-len, format, args);
	va_end(args);

	if (len <= *off) {
		*off -= len;
		len = 0;
	}

	*lenp = len;
}

static int proc_calc_metrics(char *page, char **start, off_t off,
				 int count, int *eof, int len)
{
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static int loadavg_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int a, b, c;
	int len;

	a = avenrun[0] + (FIXED_1/200);
	b = avenrun[1] + (FIXED_1/200);
	c = avenrun[2] + (FIXED_1/200);
	len = sprintf(page,"%d.%02d %d.%02d %d.%02d %ld/%d %d\n",
		LOAD_INT(a), LOAD_FRAC(a),
		LOAD_INT(b), LOAD_FRAC(b),
		LOAD_INT(c), LOAD_FRAC(c),
		nr_running(), nr_threads, last_pid);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int uptime_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	unsigned long uptime;
	unsigned long idle;
	int len;

	uptime = jiffies;
	idle = init_task.times.tms_utime + init_task.times.tms_stime;

	/* The formula for the fraction parts really is ((t * 100) / HZ) % 100, but
	   that would overflow about every five days at HZ == 100.
	   Therefore the identity a = (a / b) * b + a % b is used so that it is
	   calculated as (((t / HZ) * 100) + ((t % HZ) * 100) / HZ) % 100.
	   The part in front of the '+' always evaluates as 0 (mod 100). All divisions
	   in the above formulas are truncating. For HZ being a power of 10, the
	   calculations simplify to the version in the #else part (if the printf
	   format is adapted to the same number of digits as zeroes in HZ.
	 */
#if HZ!=100
	len = sprintf(page,"%lu.%02lu %lu.%02lu\n",
		uptime / HZ,
		(((uptime % HZ) * 100) / HZ) % 100,
		idle / HZ,
		(((idle % HZ) * 100) / HZ) % 100);
#else
	len = sprintf(page,"%lu.%02lu %lu.%02lu\n",
		uptime / HZ,
		uptime % HZ,
		idle / HZ,
		idle % HZ);
#endif
	return proc_calc_metrics(page, start, off, count, eof, len);
}

struct vmalloc_info {
	unsigned long used;
	unsigned long largest_chunk;
};

static struct vmalloc_info get_vmalloc_info(void)
{
	unsigned long prev_end = VMALLOC_START;
	struct vm_struct* vma;
	struct vmalloc_info vmi;
	vmi.used = 0;

	read_lock(&vmlist_lock);

	if(!vmlist)
		vmi.largest_chunk = (VMALLOC_END-VMALLOC_START);
	else
		vmi.largest_chunk = 0;

	for (vma = vmlist; vma; vma = vma->next) {
		unsigned long free_area_size =
			(unsigned long)vma->addr - prev_end;
		vmi.used += vma->size;
		if (vmi.largest_chunk < free_area_size )
			vmi.largest_chunk = free_area_size;
		prev_end = vma->size + (unsigned long)vma->addr;
	}
	if(VMALLOC_END-prev_end > vmi.largest_chunk)
		vmi.largest_chunk = VMALLOC_END-prev_end;

	read_unlock(&vmlist_lock);
	return vmi;
}

static int meminfo_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	struct sysinfo i;
	int len, nid;
	int pg_size ;
	unsigned long vmtot;
	struct vmalloc_info vmi;

/*
 * display in kilobytes.
 */
#define K(x) ((x) << (PAGE_SHIFT - 10))
#define B(x) ((unsigned long long)(x) << PAGE_SHIFT)
	si_meminfo(&i);
	si_swapinfo(&i);
	pg_size = atomic_read(&page_cache_size) - i.bufferram ;
	vmtot = (VMALLOC_END-VMALLOC_START)>>10;
	vmi = get_vmalloc_info();
	vmi.used >>= 10;
	vmi.largest_chunk >>= 10;

	len = sprintf(page, "        total:    used:    free:  shared: buffers:  cached:\n"
		"Mem:  %8Lu %8Lu %8Lu %8Lu %8Lu %8Lu\n"
		"Swap: %8Lu %8Lu %8Lu\n",
		B(i.totalram), B(i.totalram-i.freeram), B(i.freeram),
		B(i.sharedram), B(i.bufferram),
		B(pg_size), B(i.totalswap),
		B(i.totalswap-i.freeswap), B(i.freeswap));
	/*
	 * Tagged format, for easy grepping and expansion.
	 * The above will go away eventually, once the tools
	 * have been updated.
	 */
#ifndef CONFIG_PROC_FS_ENHANCED_MEMINFO
	len += sprintf(page+len,
		"MemTotal:      %8lu kB\n"
		"MemFree:       %8lu kB\n"
		"Buffers:       %8lu kB\n"
		"Cached:        %8lu kB\n"
		"SwapCached:    %8lu kB\n"
		"Active:        %8u kB\n"
		"Inactive:      %8u kB\n"
		"HighTotal:     %8lu kB\n"
		"HighFree:      %8lu kB\n"
		"LowTotal:      %8lu kB\n"
		"LowFree:       %8lu kB\n"
		"SwapTotal:     %8lu kB\n"
		"SwapFree:      %8lu kB\n",
		K(i.totalram),
		K(i.freeram),
		K(i.bufferram),
		K(pg_size - swapper_space.nrpages),
		K(swapper_space.nrpages),
		K(nr_active_anon_pages()) + K(nr_active_cache_pages()),
		K(nr_inactive_dirty_pages()) + K(nr_inactive_laundry_pages()) + K(nr_inactive_clean_pages()),
		K(i.totalhigh),
		K(i.freehigh),
		K(i.totalram-i.totalhigh),
		K(i.freeram-i.freehigh),
		K(i.totalswap),
		K(i.freeswap));

#else /* CONFIG_PROC_FS_ENHANCED_MEMINFO */

	len += sprintf(page+len,
		"MemTotal:      %8lu kB\n"
		"MemFree:       %8lu kB\n"
		"MemUsed:       %8lu kB\n"
		"Buffers:       %8lu kB\n"
		"Cached:        %8lu kB\n"
		"SwapCached:    %8lu kB\n"
		"Active:        %8u kB\n"
		"ActiveAnon:    %8u kB\n"
		"ActiveCache:   %8u kB\n"
		"Inactive:      %8u kB\n"
		"Inact_dirty:   %8u kB\n"
		"Inact_laundry: %8u kB\n"
		"Inact_clean:   %8u kB\n"
		"Inact_target:  %8u kB\n"
		"HighTotal:     %8lu kB\n"
		"HighFree:      %8lu kB\n"
		"LowTotal:      %8lu kB\n"
		"LowFree:       %8lu kB\n"
		"SwapTotal:     %8lu kB\n"
		"SwapFree:      %8lu kB\n"
		"SwapUsed:      %8lu kB\n"
		"VmallocTotal:  %8lu kB\n"
		"VmallocUsed:   %8lu kB\n"
		"VmallocChunk:  %8lu kB\n",
		K(i.totalram),
		K(i.freeram),
		K(i.totalram-i.freeram),
		K(i.bufferram),
		K(pg_size - swapper_space.nrpages),
		K(swapper_space.nrpages),
		K(nr_active_anon_pages()) + K(nr_active_cache_pages()),
		K(nr_active_anon_pages()),
		K(nr_active_cache_pages()),
		K(nr_inactive_dirty_pages()) + K(nr_inactive_laundry_pages()) + K(nr_inactive_clean_pages()),
		K(nr_inactive_dirty_pages()),
		K(nr_inactive_laundry_pages()),
		K(nr_inactive_clean_pages()),
		K(inactive_target()),
		K(i.totalhigh),
		K(i.freehigh),
		K(i.totalram-i.totalhigh),
		K(i.freeram-i.freehigh),
		K(i.totalswap),
		K(i.freeswap),
		K(i.totalswap-i.freeswap),
		vmtot,
		vmi.used,
		vmi.largest_chunk);

#endif /* CONFIG_PROC_FS_ENHANCED_MEMINFO */

#ifdef CONFIG_DISCONTIGMEM
	for (nid = 0; nid < numnodes; ++nid) {
		si_meminfo_node(&i, nid);
		len += sprintf(page+len, "\n"
			"Node %d MemTotal:     %8lu kB\n"
			"Node %d MemFree:     %8lu kB\n"
			"Node %d MemUsed:     %8lu kB\n"
			"Node %d HighTotal:    %8lu kB\n"
			"Node %d HighFree:     %8lu kB\n"
			"Node %d LowTotal:     %8lu kB\n"
			"Node %d LowFree:      %8lu kB\n",
			nid, K(i.totalram),
			nid, K(i.freeram),
			nid, K(i.totalram-i.freeram),
			nid, K(i.totalhigh),
			nid, K(i.freehigh),
			nid, K(i.totalram-i.totalhigh),
			nid, K(i.freeram-i.freehigh));
	}
#endif

	return proc_calc_metrics(page, start, off, count, eof, len);
#undef B
#undef K
}

extern struct seq_operations fragmentation_op;
static int fragmentation_open(struct inode *inode, struct file *file)
{
	(void)inode;
	return seq_open(file, &fragmentation_op);
}

static struct file_operations fragmentation_file_operations = {
	open:		fragmentation_open,
	read:		seq_read,
	llseek:		seq_lseek,
	release:	seq_release,
};

static int version_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	extern char *linux_banner;
	int len;

	strcpy(page, linux_banner);
	len = strlen(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

extern struct seq_operations cpuinfo_op;
static int cpuinfo_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &cpuinfo_op);
}
static struct file_operations proc_cpuinfo_operations = {
	open:		cpuinfo_open,
	read:		seq_read,
	llseek:		seq_lseek,
	release:	seq_release,
};

#ifdef CONFIG_PROC_HARDWARE
static int hardware_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_hardware_list(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}
#endif

#ifdef CONFIG_STRAM_PROC
static int stram_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_stram_list(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}
#endif

extern struct seq_operations partitions_op;
static int partitions_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &partitions_op);
}
static struct file_operations proc_partitions_operations = {
	open:		partitions_open,
	read:		seq_read,
	llseek:		seq_lseek,
	release:	seq_release,
};

#ifdef CONFIG_MODULES
extern struct seq_operations modules_op;
static int modules_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &modules_op);
}

static int modules_write(struct file *file, const char *buffer,
			unsigned long count, void *data)
{
		char tmp[4];

		if (!count)
			return -EINVAL;

		if (count > 4)
			count = 4;

		if (copy_from_user(&tmp[0], buffer, count))
			return -EFAULT;

		if (tmp[0]=='o') {
			printk(KERN_INFO "Disabled module (un)loading\n");
			module_loading_enabled = 0;
		}

		return count;
}

static struct file_operations proc_modules_operations = {
	open:		modules_open,
	read:		seq_read,
	write:		modules_write,
	llseek:		seq_lseek,
	release:	seq_release,
};

extern struct seq_operations ksyms_op;
static int ksyms_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &ksyms_op);
}
static struct file_operations proc_ksyms_operations = {
	open:		ksyms_open,
	read:		seq_read,
	llseek:		seq_lseek,
	release:	seq_release,
};
#endif

extern struct seq_operations slabinfo_op;
extern ssize_t slabinfo_write(struct file *, const char *, size_t, loff_t *);
static int slabinfo_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &slabinfo_op);
}
static struct file_operations proc_slabinfo_operations = {
	open:		slabinfo_open,
	read:		seq_read,
	write:		slabinfo_write,
	llseek:		seq_lseek,
	release:	seq_release,
};

#if 0
#ifdef CONFIG_TUX
/*
 * print out TUX internal statistics into /proc/stat.
 * (Most of them are not maintained if CONFIG_TUX_DEBUG is off.)
 */

static int print_tux_procinfo (char *page)
{
	unsigned int len = 0, i;

#define P(x) \
	do { len += sprintf(page + len, #x ": %u\n", x); } while(0)

	P(kstat.input_fastpath);
	P(kstat.input_slowpath);
	P(kstat.inputqueue_got_packet);
	P(kstat.inputqueue_no_packet);
	P(kstat.nr_keepalive_optimized);
	P(kstat.parse_static_incomplete);
	P(kstat.parse_static_redirect);
	P(kstat.parse_static_cachemiss);
	P(kstat.parse_static_nooutput);
	P(kstat.parse_static_normal);
	P(kstat.parse_dynamic_incomplete);
	P(kstat.parse_dynamic_redirect);
	P(kstat.parse_dynamic_cachemiss);
	P(kstat.parse_dynamic_nooutput);
	P(kstat.parse_dynamic_normal);
	P(kstat.complete_parsing);
	P(kstat.nr_free_pending);
	P(kstat.nr_allocated);
	P(kstat.nr_idle_input_pending);
	P(kstat.nr_output_space_pending);
	P(kstat.nr_input_pending);
	P(kstat.nr_cachemiss_pending);
	P(kstat.nr_secondary_pending);
	P(kstat.nr_output_pending);
	P(kstat.nr_redirect_pending);
	P(kstat.nr_finish_pending);
	P(kstat.nr_userspace_pending);
	P(kstat.nr_postpone_pending);
	P(kstat.static_lookup_cachemisses);
	P(kstat.static_sendfile_cachemisses);
	P(kstat.user_lookup_cachemisses);
	P(kstat.user_fetch_cachemisses);
	P(kstat.user_sendobject_cachemisses);
	P(kstat.user_sendobject_write_misses);
	P(kstat.nr_keepalive_reqs);
	P(kstat.nr_nonkeepalive_reqs);

	len += sprintf(page + len, "keephist: ");
	for (i = 0; i < KEEPALIVE_HIST_SIZE; i++)
		if (kstat.keepalive_hist[i])
			len += sprintf(page + len, "%d(%d) ",
					i, kstat.keepalive_hist[i]);
	len += sprintf(page + len, "\n");
#undef P

	return len;
}
#endif /* CONFIG_TUX */

/* mess++ */
static int kstat_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int i, len = 0;
	extern unsigned long total_forks;
	unsigned long jif = jiffies_to_clock_t(jiffies);
	unsigned int sum = 0, user = 0, nice = 0, system = 0;
	int major, disk;

	for (i = 0 ; i < smp_num_cpus; i++) {
		int cpu = cpu_logical_map(i), j;

		user += jiffies_to_clock_t(kstat.per_cpu_user[cpu]);
		nice += jiffies_to_clock_t(kstat.per_cpu_nice[cpu]);
		system += jiffies_to_clock_t(kstat.per_cpu_system[cpu]);
#if !defined(CONFIG_ARCH_S390)
		for (j = 0 ; j < NR_IRQS ; j++)
			sum += kstat.irqs[cpu][j];
#endif
	}

	proc_sprintf(page, &off, &len,
		      "cpu  %u %u %u %lu\n", user, nice, system,
		      jif * smp_num_cpus - (user + nice + system));
	for (i = 0 ; i < smp_num_cpus; i++)
		proc_sprintf(page, &off, &len,
			"cpu%d %u %u %u %lu\n",
			i,
			jiffies_to_clock_t(kstat.per_cpu_user[cpu_logical_map(i)]),
			jiffies_to_clock_t(kstat.per_cpu_nice[cpu_logical_map(i)]),
			jiffies_to_clock_t(kstat.per_cpu_system[cpu_logical_map(i)]),
			jif - jiffies_to_clock_t(kstat.per_cpu_user[cpu_logical_map(i)] \
				   + kstat.per_cpu_nice[cpu_logical_map(i)] \
				   + kstat.per_cpu_system[cpu_logical_map(i)]));
	proc_sprintf(page, &off, &len,
		"page %u %u\n"
		"swap %u %u\n"
		"intr %u",
			kstat.pgpgin >> 1,
			kstat.pgpgout >> 1,
			kstat.pswpin,
			kstat.pswpout,
			sum
	);
#if !defined(CONFIG_ARCH_S390)
	for (i = 0 ; i < NR_IRQS ; i++)
		proc_sprintf(page, &off, &len,
			     " %u", kstat_irqs(i));
#endif

	proc_sprintf(page, &off, &len, "\ndisk_io: ");

	for (major = 0; major < DK_MAX_MAJOR; major++) {
		for (disk = 0; disk < DK_MAX_DISK; disk++) {
			int active = kstat.dk_drive[major][disk] +
				kstat.dk_drive_rblk[major][disk] +
				kstat.dk_drive_wblk[major][disk];
			if (active)
				proc_sprintf(page, &off, &len,
					"(%u,%u):(%u,%u,%u,%u,%u) ",
					major, disk,
					kstat.dk_drive[major][disk],
					kstat.dk_drive_rio[major][disk],
					kstat.dk_drive_rblk[major][disk],
					kstat.dk_drive_wio[major][disk],
					kstat.dk_drive_wblk[major][disk]
			);
		}
	}

	len += sprintf(page + len,
		"\nctxt %lu\n"
		"btime %lu\n"
		"processes %lu\n",
		nr_context_switches(),
		xtime.tv_sec - jif / HZ,
		total_forks);

#ifdef CONFIG_TUX
	len += print_tux_procinfo(page+len);
#endif

	return proc_calc_metrics(page, start, off, count, eof, len);
}
#else /* seq_file mess++ */
#ifdef CONFIG_TUX
void seq_print_tux_procinfo(struct seq_file *p, void *v)
{
	unsigned int len = 0, i;

#define P(x) \
	do { seq_printf(p, #x ": %u\n", x); } while(0)

	P(kstat.input_fastpath);
	P(kstat.input_slowpath);
	P(kstat.inputqueue_got_packet);
	P(kstat.inputqueue_no_packet);
	P(kstat.nr_keepalive_optimized);
	P(kstat.parse_static_incomplete);
	P(kstat.parse_static_redirect);
	P(kstat.parse_static_cachemiss);
	P(kstat.parse_static_nooutput);
	P(kstat.parse_static_normal);
	P(kstat.parse_dynamic_incomplete);
	P(kstat.parse_dynamic_redirect);
	P(kstat.parse_dynamic_cachemiss);
	P(kstat.parse_dynamic_nooutput);
	P(kstat.parse_dynamic_normal);
	P(kstat.complete_parsing);
	P(kstat.nr_free_pending);
	P(kstat.nr_allocated);
	P(kstat.nr_idle_input_pending);
	P(kstat.nr_output_space_pending);
	P(kstat.nr_input_pending);
	P(kstat.nr_cachemiss_pending);
	P(kstat.nr_secondary_pending);
	P(kstat.nr_output_pending);
	P(kstat.nr_redirect_pending);
	P(kstat.nr_finish_pending);
	P(kstat.nr_userspace_pending);
	P(kstat.nr_postpone_pending);
	P(kstat.static_lookup_cachemisses);
	P(kstat.static_sendfile_cachemisses);
	P(kstat.user_lookup_cachemisses);
	P(kstat.user_fetch_cachemisses);
	P(kstat.user_sendobject_cachemisses);
	P(kstat.user_sendobject_write_misses);
	P(kstat.nr_keepalive_reqs);
	P(kstat.nr_nonkeepalive_reqs);

	seq_printf(p, "keephist: ");
	for (i = 0; i < KEEPALIVE_HIST_SIZE; i++)
		if (kstat.keepalive_hist[i])
			seq_printf(p, "%d(%d) ",
					i, kstat.keepalive_hist[i]);
	seq_printf(p, "\n");
#undef P

	return;
}
#endif /* CONFIG_TUX */
int show_stat(struct seq_file *p, void *v)
{
	int i;
	extern unsigned long total_forks;
	unsigned long jif = jiffies;
	unsigned int sum = 0, user = 0, nice = 0, system = 0;
	int major, disk;

	for (i = 0 ; i < smp_num_cpus; i++) {
		int cpu = cpu_logical_map(i), j;

		user += kstat.per_cpu_user[cpu];
		nice += kstat.per_cpu_nice[cpu];
		system += kstat.per_cpu_system[cpu];
#if !defined(CONFIG_ARCH_S390)
		for (j = 0 ; j < NR_IRQS ; j++)
			sum += kstat.irqs[cpu][j];
#endif
	}

	seq_printf(p,
		      "cpu  %u %u %u %lu\n", user, nice, system,
		      jif * smp_num_cpus - (user + nice + system));
	for (i = 0 ; i < smp_num_cpus; i++)
		seq_printf(p,
			"cpu%d %u %u %u %lu\n",
			i,
			kstat.per_cpu_user[cpu_logical_map(i)],
			kstat.per_cpu_nice[cpu_logical_map(i)],
			kstat.per_cpu_system[cpu_logical_map(i)],
			jif - (  kstat.per_cpu_user[cpu_logical_map(i)] \
				   + kstat.per_cpu_nice[cpu_logical_map(i)] \
				   + kstat.per_cpu_system[cpu_logical_map(i)]));
	seq_printf(p,
		"page %u %u\n"
		"swap %u %u\n"
		"intr %u",
			kstat.pgpgin >> 1,
			kstat.pgpgout >> 1,
			kstat.pswpin,
			kstat.pswpout,
			sum
	);
#if !defined(CONFIG_ARCH_S390) && !defined(CONFIG_ALPHA) && !defined(CONFIG_PPC64)
	for (i = 0 ; i < NR_IRQS ; i++)
		seq_printf(p,
			     " %u", kstat_irqs(i));
#endif

	seq_printf(p, "\ndisk_io: ");

	for (major = 0; major < DK_MAX_MAJOR; major++) {
		for (disk = 0; disk < DK_MAX_DISK; disk++) {
			int active = kstat.dk_drive[major][disk] +
				kstat.dk_drive_rblk[major][disk] +
				kstat.dk_drive_wblk[major][disk];
			if (active)
				seq_printf(p,
					"(%u,%u):(%u,%u,%u,%u,%u) ",
					major, disk,
					kstat.dk_drive[major][disk],
					kstat.dk_drive_rio[major][disk],
					kstat.dk_drive_rblk[major][disk],
					kstat.dk_drive_wio[major][disk],
					kstat.dk_drive_wblk[major][disk]
			);
		}
	}

	seq_printf(p,
		"\nctxt %lu\n"
		"btime %lu\n"
		"processes %lu\n",
		nr_context_switches(),
		xtime.tv_sec - jif / HZ,
		total_forks);

#ifdef CONFIG_TUX
	seq_print_tux_procinfo(p, v);
#endif

	return 0;

}

static void *single_start(struct seq_file *p, loff_t *pos)
{
	return (void *)(*pos == 0);
}

static void *single_next(struct seq_file *p, void *v, loff_t *pos)
{
	++*pos;
	return NULL;
}

static void single_stop(struct seq_file *p, void *v)
{
}

static struct seq_operations proc_stat_op = {
	start:	single_start,
	next:	single_next,
	stop:	single_stop,
	show:	show_stat,
};

static int stat_open(struct inode *inode, struct file *file)
{
        unsigned size = 4096 * (1 + smp_num_cpus / 32);
        char *buf;
        struct seq_file *m;
        int res;

        /* don't ask for more than the kmalloc() max size, currently 128 KB */
        if (size > 128 * 1024)
                size = 128 * 1024;
        buf = kmalloc(size, GFP_KERNEL);
        if (!buf)
                return -ENOMEM;

        res = seq_open(file, &proc_stat_op);
        if (!res) {
                m = file->private_data;
                m->buf = buf;
                m->size = size;
        } else
                kfree(buf);
        return res;
}

static struct file_operations proc_stat_operations = {
        .open           = stat_open,
        .read           = seq_read,
        .llseek         = seq_lseek,
        .release        = seq_release,
};
#endif /* seq stat mess++ */

static int devices_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_device_list(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

#if !defined(CONFIG_X86) && !defined(CONFIG_ARCH_S390) && !defined(CONFIG_PPC64) && !defined(CONFIG_IA64)
static int interrupts_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_irq_list(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

#else /* !CONFIG_X86 !CONFIG_ARCH_S390 !CONFIG_PPC64 !CONFIG_IA64 */
#if !defined(CONFIG_IA64) && !defined(CONFIG_ARCH_S390)

extern int show_interrupts(struct seq_file *p, void *v);
static struct seq_operations proc_interrupts_op = {
	start:	single_start,
	next:	single_next,
	stop:	single_stop,
	show:	show_interrupts,
};

static int interrupts_open(struct inode *inode, struct file *file)
{
	unsigned size = PAGE_SIZE * (1 + smp_num_cpus / 8);
	/*
	* probably should depend on NR_CPUS, but that's only rough estimate;
	* if we'll need more it will be given,
	*/
	char *buf = kmalloc(size, GFP_KERNEL);
	struct seq_file *m;
	int res;

	if (!buf)
		return -ENOMEM;
	res = seq_open(file, &proc_interrupts_op);
	if (!res) {
		m = file->private_data;
		m->buf = buf;
		m->size = size;
	} else
		kfree(buf);
	return res;
}
static struct file_operations proc_interrupts_operations = {
	open:			interrupts_open,
	read:			seq_read,
	llseek:			seq_lseek,
	release:		seq_release,
};
#endif /* !CONFIG_X86 */
#endif /* !CONFIG_IA64 !CONFIG_ARCH_S390 */

static int filesystems_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_filesystem_list(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int dma_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_dma_list(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int ioports_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_ioport_list(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int cmdline_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	extern char saved_command_line[];
	int len = 0;

	proc_sprintf(page, &off, &len, "%s\n", saved_command_line);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

#ifdef CONFIG_SGI_DS1286
static int ds1286_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_ds1286_status(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}
#endif

static int locks_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len;
	lock_kernel();
	len = get_locks_status(page, start, off, count);
	unlock_kernel();
	if (len < count) *eof = 1;
	return len;
}

static int execdomains_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_exec_domain_list(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int swaps_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_swaparea_info(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int memory_read_proc(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	int len = get_mem_list(page);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

/*
 * This function accesses profiling information. The returned data is
 * binary: the sampling step and the actual contents of the profile
 * buffer. Use of the program readprofile is recommended in order to
 * get meaningful info out of these data.
 */
static ssize_t read_profile(struct file *file, char *buf,
			    size_t count, loff_t *ppos)
{
	loff_t n = *ppos;
	unsigned p = n;
	ssize_t read;
	char * pnt;
	unsigned int sample_step = 1 << prof_shift;

	if (p != n || p >= (prof_len+1)*sizeof(unsigned int))
		return 0;
	if (count > (prof_len+1)*sizeof(unsigned int) - p)
		count = (prof_len+1)*sizeof(unsigned int) - p;
	read = 0;

	while (p < sizeof(unsigned int) && count > 0) {
		put_user(*((char *)(&sample_step)+p),buf);
		buf++; p++; count--; read++;
	}
	pnt = (char *)prof_buffer + p - sizeof(unsigned int);
	if (copy_to_user(buf,(void *)pnt,count))
		return -EFAULT;
	read += count;
	*ppos = n + read;
	return read;
}

/*
 * Writing to /proc/profile resets the counters
 *
 * Writing a 'profiling multiplier' value into it also re-sets the profiling
 * interrupt frequency, on architectures that support this.
 */
static ssize_t write_profile(struct file * file, const char * buf,
			     size_t count, loff_t *ppos)
{
#ifdef CONFIG_SMP
	extern int setup_profiling_timer (unsigned int multiplier);

	if (count==sizeof(int)) {
		unsigned int multiplier;

		if (copy_from_user(&multiplier, buf, sizeof(int)))
			return -EFAULT;

		if (setup_profiling_timer(multiplier))
			return -EINVAL;
	}
#endif

	memset(prof_buffer, 0, prof_len * sizeof(*prof_buffer));
	return count;
}

static struct file_operations proc_profile_operations = {
	read:		read_profile,
	write:		write_profile,
};

#ifdef CONFIG_MAGIC_SYSRQ
/*
 * writing 'C' to /proc/sysrq-trigger is like sysrq-C
 */
static ssize_t write_sysrq_trigger(struct file *file, const char *buf,
				     size_t count, loff_t *ppos)
{
	if (count) {
		char c;

		if (get_user(c, buf))
			return -EFAULT;
		handle_sysrq(c, NULL, NULL, NULL);
	}
	return count;
}

static struct file_operations proc_sysrq_trigger_operations = {
	.write		= write_sysrq_trigger,
};
#endif

struct proc_dir_entry *proc_root_kcore;

static void create_seq_entry(char *name, mode_t mode, struct file_operations *f)
{
	struct proc_dir_entry *entry;
	entry = create_proc_entry(name, mode, NULL);
	if (entry)
		entry->proc_fops = f;
}

void __init proc_misc_init(void)
{
	struct proc_dir_entry *entry;
	int gr_mode = 0;
	static struct {
		char *name;
		int (*read_proc)(char*,char**,off_t,int,int*,void*);
	} *p, simple_ones[] = {
		{"loadavg",     loadavg_read_proc},
		{"uptime",	uptime_read_proc},
		{"meminfo",	meminfo_read_proc},
		{"version",	version_read_proc},
#ifdef CONFIG_PROC_HARDWARE
		{"hardware",	hardware_read_proc},
#endif
#ifdef CONFIG_STRAM_PROC
		{"stram",	stram_read_proc},
#endif
#if 0
		/* mess++ */
		{"stat",	kstat_read_proc},
#endif
#ifndef CONFIG_GRKERNSEC_PROC_ADD
		{"devices",	devices_read_proc},
#endif
#if !defined(CONFIG_ARCH_S390) && !defined(CONFIG_PPC64) && !defined(CONFIG_X86)
		{"interrupts",	interrupts_read_proc},
#endif
		{"filesystems",	filesystems_read_proc},
#ifndef CONFIG_GRKERNSEC_PROC_ADD
		{"dma",		dma_read_proc},
		{"ioports",	ioports_read_proc},
		{"cmdline",	cmdline_read_proc},
#endif
#ifdef CONFIG_SGI_DS1286
		{"rtc",		ds1286_read_proc},
#endif
		{"locks",	locks_read_proc},
		{"swaps",	swaps_read_proc},
#ifndef CONFIG_GRKERNSEC_PROC_ADD
		{"iomem",	memory_read_proc},
#endif
		{"execdomains",	execdomains_read_proc},
		{NULL,}
	};
	for (p = simple_ones; p->name; p++)
		create_proc_read_entry(p->name, 0, NULL, p->read_proc, NULL);

#ifdef CONFIG_GRKERNSEC_PROC_USER
	gr_mode = S_IRUSR;
#elif CONFIG_GRKERNSEC_PROC_USERGROUP
	gr_mode = S_IRUSR | S_IRGRP;
#endif

#ifdef CONFIG_GRKERNSEC_PROC_ADD
	create_proc_read_entry("devices", gr_mode, NULL, &devices_read_proc, NULL);
	create_proc_read_entry("dma", gr_mode, NULL, &dma_read_proc, NULL);
	create_proc_read_entry("ioports", gr_mode, NULL, &ioports_read_proc, NULL);
	create_proc_read_entry("cmdline", gr_mode, NULL, &cmdline_read_proc, NULL);
	create_proc_read_entry("iomem", gr_mode, NULL, &memory_read_proc, NULL);
#endif
	proc_symlink("mounts", NULL, "self/mounts");

	/* And now for trickier ones */
	entry = create_proc_entry("kmsg", S_IRUSR, &proc_root);
	if (entry)
		entry->proc_fops = &proc_kmsg_operations;
#ifdef CONFIG_GRKERNSEC_PROC_ADD
	create_seq_entry("cpuinfo", gr_mode, &proc_cpuinfo_operations);
	create_seq_entry("slabinfo", gr_mode,&proc_slabinfo_operations);
#else
	create_seq_entry("cpuinfo", 0, &proc_cpuinfo_operations);
	create_seq_entry("slabinfo",S_IWUSR|S_IRUGO,&proc_slabinfo_operations);
#endif
	create_seq_entry("stat", 0, &proc_stat_operations);
	create_seq_entry("partitions", 0, &proc_partitions_operations);
#if defined(CONFIG_X86) && !defined(CONFIG_GRKERNSEC_PROC_ADD)
	create_seq_entry("interrupts", 0, &proc_interrupts_operations);
#else
	create_seq_entry("interrupts", gr_mode, &proc_interrupts_operations);
#endif
	create_seq_entry("buddyinfo",S_IRUGO, &fragmentation_file_operations);
#if defined(CONFIG_MODULES) && !defined(CONFIG_GRKERNSEC_PROC)
	create_seq_entry("ksyms", 0, &proc_ksyms_operations);
	create_seq_entry("modules", 0, &proc_modules_operations);
#elif defined(CONFIG_MODULES) && defined(CONFIG_GRKERNSEC_PROC)
	create_seq_entry("ksyms", gr_mode, &proc_ksyms_operations);
	create_seq_entry("modules", gr_mode, &proc_modules_operations);
#endif
#ifndef CONFIG_GRKERNSEC_PROC_ADD
	proc_root_kcore = create_proc_entry("kcore", S_IRUSR, NULL);
	if (proc_root_kcore) {
		proc_root_kcore->proc_fops = &proc_kcore_operations;
		proc_root_kcore->size =
				(size_t)high_memory - PAGE_OFFSET + PAGE_SIZE;
	}
#endif
	if (prof_shift) {
		entry = create_proc_entry("profile", S_IWUSR | S_IRUGO, NULL);
		if (entry) {
			entry->proc_fops = &proc_profile_operations;
			entry->size = (1+prof_len) * sizeof(unsigned int);
		}
	}
#ifdef CONFIG_MAGIC_SYSRQ
	entry = create_proc_entry("sysrq-trigger", S_IWUSR, NULL);
	if (entry)
		entry->proc_fops = &proc_sysrq_trigger_operations;
#endif
#ifdef CONFIG_PPC32
	{
		extern struct file_operations ppc_htab_operations;
		entry = create_proc_entry("ppc_htab", S_IRUGO|S_IWUSR, NULL);
		if (entry)
			entry->proc_fops = &ppc_htab_operations;
	}
#endif
}
