
/******************************************************************************
  
  Copyright(c) 2003 - 2004 Intel Corporation. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  James P. Ketrenos <ipw2100-admin@linux.intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

******************************************************************************/
#include <linux/802_11.h>
#include <linux/compiler.h>
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/if_arp.h>
#include <linux/in6.h>
#include <linux/in.h>
#include <linux/ip.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/skbuff.h>
#include <linux/slab.h>
#include <linux/tcp.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/wireless.h>
#include <linux/etherdevice.h>
#include <asm/uaccess.h>

#include "ieee80211.h"


/*


802.11 Data Frame 

      ,-------------------------------------------------------------------.
Bytes |  2   |  2   |    6    |    6    |    6    |  2   | 0..2312 |   4  |
      |------|------|---------|---------|---------|------|---------|------|
Desc. | ctrl | dura |  DA/RA  |   TA    |    SA   | Sequ |  Frame  |  fcs |
      |      | tion | (BSSID) |         |         | ence |  data   |      |
      `--------------------------------------------------|         |------'
Total: 28 non-data bytes                                 `----.----'  
                                                              |
       .- 'Frame data' expands to <---------------------------'
       |
       V
      ,---------------------------------------------------.
Bytes |  1   |  1   |    1    |    3     |  2   |  0-2304 |
      |------|------|---------|----------|------|---------|
Desc. | SNAP | SNAP | Control |Eth Tunnel| Type | IP      |
      | DSAP | SSAP |         |          |      | Packet  |
      | 0xAA | 0xAA |0x03 (UI)|0x00-00-F8|      |         |
      `-----------------------------------------|         |
Total: 8 non-data bytes                         `----.----'
                                                     |
       .- 'IP Packet' expands, if WEP enabled, to <--'
       |
       V
      ,-----------------------.
Bytes |  4  |   0-2296  |  4  |
      |-----|-----------|-----|
Desc. | IV  | Encrypted | ICV |
      |     | IP Packet |     |
      `-----------------------'
Total: 8 non-data bytes


802.3 Ethernet Data Frame 

      ,-----------------------------------------.
Bytes |   6   |   6   |  2   |  Variable |   4  |
      |-------|-------|------|-----------|------|
Desc. | Dest. | Source| Type | IP Packet |  fcs |
      |  MAC  |  MAC  |      |           |      |
      `-----------------------------------------'
Total: 18 non-data bytes

*/

static u8 P802_1H_OUI[P80211_OUI_LEN] = { 0x00, 0x00, 0xf8 };
static u8 RFC1042_OUI[P80211_OUI_LEN] = { 0x00, 0x00, 0x00 };

static inline int ieee80211_add_snap(void *data, u16 h_proto)
{
	struct ieee_802_11_snap_header *snap;
	u8 *oui;

	snap = data;
	snap->dsap = 0xaa;
	snap->ssap = 0xaa;
	snap->ctrl = 0x03;

	if (ntohs(h_proto) == 0x8137 || ntohs(h_proto) == 0x80f3)
		oui = P802_1H_OUI;
	else
		oui = RFC1042_OUI;
	snap->oui[0] = oui[0];
	snap->oui[1] = oui[1];
	snap->oui[2] = oui[2];

	*(u16 *)(data + SNAP_SIZE) = h_proto;

	return ntohs(h_proto);
}

static inline struct sk_buff *ieee80211_tx_frame_encrypt(
	struct ieee80211_device *ieee, 
	u16 fragment, u16 frag_size,
	struct sk_buff *skb)
{
	struct ieee80211_crypt_data* crypt = ieee->crypt;
	struct sk_buff *wep_skb;
	int wep_len, olen;
	int in_place = 0;
	struct ieee80211_hdr *header;
	
	/* SKB visualization
	 * 
	 *  ,- skb->data 
	 * |
	 * |    ETHERNET HEADER        ,-<-- PAYLOAD
	 * |                           |     14 bytes from skb->data 
	 * |  2 bytes for Type --> ,T. |     (sizeof ethhdr)
	 * |                       | | |    
	 * |,-Dest.--. ,--Src.---. | | |     
	 * |  6 bytes| | 6 bytes | | | |
	 * v         | |         | | | |
	 * 0         | v       1 | v | v           2       
	 * 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5
	 *     ^     | ^         | ^ | 
	 *     |     | |         | | |  
	 *     |     | |         | `T' <---- 2 bytes for Type
	 *     |     | |         |
	 *     |     | '---SNAP--' <-------- 6 bytes for SNAP
	 *     |     |  
	 *     `-IV--' <-------------------- 4 bytes for IV (WEP)
	 *
	 *      SNAP HEADER
	 */
#if 0
	if (fragment == 0 && ieee->tx_payload_only && 
	    (skb_tailroom(skb) >= crypt->ops->extra_postfix_len) &&
	    (skb_headroom(skb) >= crypt->ops->extra_prefix_len)) {
		/* - IF -
		 * this is the first fragment in a fragment chain
		 * - AND -
		 * the underlying hardware does not require a full 802.11 
		 * - AND -
		 * there is enough tail room for ICV (extra_postfix)--note that
		 * skb_alloc() usually gives more than requested due to 
		 * alignment -- see include/linux/skbuff.h
		 * - AND -
		 * there is enough head room for the IV (extra_prefix)--note
		 * that a SNAP header is 6 bytes smaller than a standard 
		 * ethernet header
		 * - THEN -
		 * we can encrypt the packet inplace.  This saves us from 
		 * an SKB allocation, and also from having to do an extra 
		 * memcpy since the Host AP decryption does encryption in place
		 */
		in_place = 1;
		wep_skb = skb;

		/* Set skb->data to the start of where the IV will be. */
		skb_push(wep_skb, crypt->ops->extra_prefix_len);
	} else {
#endif
		/* There is not enough room in the provided skb.  Allocate
		 * a new one and copy the IP payload fragment into it. */
		wep_len = crypt->ops->extra_prefix_len + 
			frag_size +
			crypt->ops->extra_postfix_len;

		if (!ieee->tx_payload_only)
			wep_len += sizeof(struct ieee80211_hdr);

		wep_skb = dev_alloc_skb(wep_len);
		if (wep_skb == NULL) {
			printk(KERN_WARNING "%s: Could not allocate SKB for "
			       "WEP encryption.  Dropping packet.\n", 
			       ieee->dev->name);
			ieee->ieee_stats.tx_discards++;
			return NULL;
		}

		if (!ieee->tx_payload_only) {
			header = (struct ieee80211_hdr *)
				skb_put(wep_skb, sizeof(struct ieee80211_hdr));
			printk(KERN_DEBUG "%s: TODO -- implement 802.11 "
			       "header construction...\n", ieee->dev->name);
		}

		/* Put space at the start of wep_skb for the IV */
		skb_put(wep_skb, crypt->ops->extra_prefix_len);

		/* Copy the unencrypted payload into the wep_skb 
		 *
		 * NOTE: As of HostAP v0.1.3, the encryption system used
		 * performs encryption in place.  This means we need to copy
		 * the non encrypted data into the new SKB vs. having Host AP
		 * transform the encrypted data across buffers */
		memcpy(skb_put(wep_skb, frag_size),
		       skb->data + fragment * ieee->fts, frag_size);
#if 0
	} 
#endif

	/* Advance wep_skb storage to hold the ICV */
	skb_put(wep_skb, crypt->ops->extra_postfix_len);

	/* To encrypt, frame format is:
	 * IV (4 bytes), clear payload (including SNAP), ICV (4 bytes) */
	atomic_inc(&crypt->refcnt);
	olen = crypt->ops->encrypt(wep_skb->data, wep_skb->len - 
				   (crypt->ops->extra_prefix_len +
				    crypt->ops->extra_postfix_len), 
				   crypt->priv);
	atomic_dec(&crypt->refcnt);
	if (olen != wep_skb->len) {
		printk(KERN_INFO "%s: Encryption failed: %d vs. %d.\n",
		       ieee->dev->name, olen, wep_skb->len);
		if (!in_place)
			dev_kfree_skb(wep_skb);
		ieee->ieee_stats.tx_discards++;
		return NULL;
	}

	if (!in_place && fragment == 0) {
		/* If this is the last fragment to extract then we are done
		 * placing the rest of this SKB into the tx_queue so 
		 * we're done with the original SKB--free it... */
		dev_kfree_skb(skb);
	}

	return wep_skb;
}


static inline struct sk_buff *ieee80211_add_fragment(
	struct ieee80211_device *ieee,
	int fragment, int frag_size,
	struct sk_buff *skb,
	int encrypt)
{
	int new_len = 0;
	struct sk_buff *new_skb;
	struct ieee80211_hdr *header;

	if (encrypt)
		return ieee80211_tx_frame_encrypt(ieee, fragment, frag_size, 
						  skb);

	if (fragment == 0 && ieee->tx_payload_only) {
		/* Set the end of the payload == the end of this fragment */
		skb_trim(skb, frag_size);
		return skb;
	}

	/* If this isn't the first fragment in the series or if
	 * we need to store the entire 802.11 header then we need
	 * to allocate a new SKB for this fragment, copy the data,
	 * etc. 
	 *
	 * NOTE:  This isn't the optimal approach. Ideally we should
	 * support the case where 802.11 headers are not needed, and
	 * there is no encryption--which means we can fit everything
	 * within the original SKB */
	
	new_len = frag_size;
	if (!ieee->tx_payload_only)
		new_len = sizeof(struct ieee80211_hdr);
	
	new_skb = dev_alloc_skb(new_len);
	if (new_skb == NULL) {
		printk(KERN_WARNING "%s: Could not allocate SKB for "
		       "802.11 encapsulation.  Dropping packet.\n", 
		       ieee->dev->name);
		ieee->ieee_stats.tx_discards++;
		return NULL;
	}

	if (!ieee->tx_payload_only) {
		header = (struct ieee80211_hdr *)skb_put(
			new_skb, sizeof(struct ieee80211_hdr));
		
		printk(KERN_DEBUG "%s: TODO--implement 802.11 header "
		       "construction.\n", ieee->dev->name);
	}
	
	memcpy(skb_put(new_skb, frag_size), 
	       skb->data + ieee->fts * fragment,
	       frag_size);

	/* If this is the last fragment to extract then we are done
	 * placing the rest of this SKB into the tx_queue so 
	 * we're done with the original SKB--free it... */
	if (fragment == 0)
		dev_kfree_skb(skb);

	return new_skb;
}


/* SKBs are added to the ieee->tx_queue. */
int ieee80211_tx(struct ieee80211_device *ieee, struct sk_buff *skb)
{
	char src[ETH_ALEN];
	char dest[ETH_ALEN];
	int i, frag_size, frag_total;
	struct sk_buff *frag, *last_frag = NULL;
	unsigned long flags;
	struct net_device_stats *stats = &ieee->stats;
	int ether_type, encrypt;
	struct sk_buff *old_skb = NULL;

	spin_lock_irqsave(&ieee->lock, flags);

	if (!ieee->tx_payload_only) {
		memcpy(src, skb->data, ETH_ALEN);
		memcpy(dest, skb->data + ETH_ALEN, ETH_ALEN);
		printk(KERN_INFO "%s: TODO Add support for building full "
		       "802.11 headers\n", ieee->dev->name);
	}

	/* Put a SNAP header where the ethernet header used to be 
	 * and update skb->data pointer - the original ethernet header is
	 * lost.
	 *
	 * An assumption is that SNAP header is smaller than 
	 * ethernet header.  Drop if not. */
	if (unlikely(skb->len < SNAP_SIZE + sizeof(u16))) {
		printk(KERN_WARNING "%s: skb too small for SNAP "
		       "header (%d bytes).\n", ieee->dev->name, skb->len);
		goto failed;
	}

	/* HACK TEST */
	old_skb = skb;
	skb = dev_alloc_skb(old_skb->len);
	if (skb == NULL) {
		printk(KERN_WARNING "%s: Could not allocate SKB.  "
		       "Dropping packet.\n", 
		       ieee->dev->name);
		ieee->ieee_stats.tx_discards++;
		goto failed;
	}
	memcpy(skb_put(skb, old_skb->len), old_skb->data, old_skb->len);

	/* move data pointer forward to where the SNAP goes */
	skb_pull(skb, sizeof(struct ethhdr) - SNAP_SIZE - sizeof(u16));
	
        /* Overwrite the ethernet header with the SNAP 
	 * h_proto is SNAP_SIZE bytes from the start of skb->data */
	ether_type = ieee80211_add_snap(skb->data , 
					*(u16 *)(skb->data + SNAP_SIZE));
		
	frag_total = skb->len / ieee->fts;
	frag_size = skb->len % ieee->fts;
	i = frag_total;
	encrypt = (ieee->host_encrypt && ether_type != ETH_P_PAE && 
		   ieee->crypt && ieee->crypt->ops);
	if (frag_size) {
		frag = ieee80211_add_fragment(ieee, i, frag_size, skb, 
					      encrypt);
		if (frag == NULL)
			goto failed;

		skb_queue_tail(&ieee->tx_queue, frag);
		last_frag = frag;
		frag_total++;
	} else if (frag_total == 0)
		goto failed;

	/* Add fragments to the tx_queue in reverse order.  This allows us
	 * to potentially use the currently allocated SKB for the first
	 * fragment without having to re-allocate or perform a memcpy */
	while (i > 0) {
		i--;
		frag = ieee80211_add_fragment(ieee, i, ieee->fts, skb, 
					      encrypt);
		if (frag == NULL)
			goto failed;

		if (last_frag != NULL) 
			skb_insert(last_frag, frag);
		else 
			skb_queue_tail(&ieee->tx_queue, frag);

		last_frag = frag;
	}
	
	spin_unlock_irqrestore(&ieee->lock, flags);

	if (old_skb) {
		if (in_interrupt())
			dev_kfree_skb_irq(old_skb);
		else 
			dev_kfree_skb(old_skb);
	}

	return frag_total;

 failed:
	/* If we have completed the SKB copy then nuke the new copy */
	if (old_skb)
		dev_kfree_skb(skb);

        if (last_frag != NULL) {
		/* At least one fragment was added to the tx_queue, so pull
		 * all the SKBs from the tail until we have pulled the 
		 * last frag added */
		do {
			frag = skb_dequeue_tail(&ieee->tx_queue);
			dev_kfree_skb(frag);
		} while (frag != last_frag);
	}

	stats->tx_errors++;
	spin_unlock_irqrestore(&ieee->lock, flags);
	return -1;
}

/*EXPORT_SYMBOL(ieee80211_tx);*/
