// SPDX-License-Identifier: GPL-2.0
#include "tests.h"
#include "util/blake2s.h"
#include "util/debug.h"

#include <linux/compiler.h>
#include <stdlib.h>
#include <string2.h>

static int test_strreplace(char needle, const char *haystack,
			   const char *replace, const char *expected)
{
	char *new = strreplace_chars(needle, haystack, replace);
	int ret = strcmp(new, expected);

	free(new);
	return ret == 0;
}

/* Maximum data length tested by test_blake2s() */
#define MAX_DATA_LEN 512

/*
 * Hash length tested by test_blake2s().  BLAKE2s supports variable-length
 * hashes.  However, the only user of BLAKE2s in 'perf' uses 20-byte hashes,
 * matching the length of the ELF build ID field.  So that's the length we test.
 */
#define HASH_LEN 20

/* Test the implementation of the BLAKE2s hash algorithm. */
static int test_blake2s(void)
{
	u8 data[MAX_DATA_LEN];
	u8 hash[HASH_LEN];
	u8 hash2[HASH_LEN];
	struct blake2s_ctx main_ctx;
	/*
	 * This value was generated by the following Python code:
	 *
	 * import hashlib
	 *
	 * data = bytes(i % 256 for i in range(513))
	 * h = hashlib.blake2s(digest_size=20)
	 * for i in range(513):
	 *     h.update(hashlib.blake2s(data=data[:i], digest_size=20).digest())
	 * print(h.hexdigest())
	 */
	static const u8 expected_hash_of_hashes[20] = {
		0xef, 0x9b, 0x13, 0x98, 0x78, 0x8e, 0x74, 0x59, 0x9c, 0xd5,
		0x0c, 0xf0, 0x33, 0x97, 0x79, 0x3d, 0x3e, 0xd0, 0x95, 0xa6
	};
	size_t i;

	/* Generate MAX_DATA_LEN bytes of data. */
	for (i = 0; i < MAX_DATA_LEN; i++)
		data[i] = i;

	blake2s_init(&main_ctx, sizeof(hash));
	for (i = 0; i <= MAX_DATA_LEN; i++) {
		struct blake2s_ctx ctx;

		/* Compute the BLAKE2s hash of 'i' data bytes. */
		blake2s_init(&ctx, HASH_LEN);
		blake2s_update(&ctx, data, i);
		blake2s_final(&ctx, hash);

		/* Verify that multiple updates produce the same result. */
		blake2s_init(&ctx, HASH_LEN);
		blake2s_update(&ctx, data, i / 2);
		blake2s_update(&ctx, &data[i / 2], i - (i / 2));
		blake2s_final(&ctx, hash2);
		TEST_ASSERT_VAL("inconsistent BLAKE2s hashes",
				memcmp(hash, hash2, HASH_LEN) == 0);

		/*
		 * Pass the hash to another BLAKE2s context, so that we
		 * incrementally compute the hash of all the hashes.
		 */
		blake2s_update(&main_ctx, hash, HASH_LEN);
	}

	/* Verify the hash of all the hashes. */
	blake2s_final(&main_ctx, hash);
	TEST_ASSERT_VAL("wrong BLAKE2s hashes",
			memcmp(hash, expected_hash_of_hashes, HASH_LEN) == 0);
	return 0;
}

static int test__util(struct test_suite *t __maybe_unused, int subtest __maybe_unused)
{
	TEST_ASSERT_VAL("empty string", test_strreplace(' ', "", "123", ""));
	TEST_ASSERT_VAL("no match", test_strreplace('5', "123", "4", "123"));
	TEST_ASSERT_VAL("replace 1", test_strreplace('3', "123", "4", "124"));
	TEST_ASSERT_VAL("replace 2", test_strreplace('a', "abcabc", "ef", "efbcefbc"));
	TEST_ASSERT_VAL("replace long", test_strreplace('a', "abcabc", "longlong",
							"longlongbclonglongbc"));

	return test_blake2s();
}

DEFINE_SUITE("util", util);
