/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_run_result.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::unittest::Test_Run_Result
 *
 * $Id: test_run_result.t.cpp,v 1.18 2005/06/23 09:54:27 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @todo add a sane comparison test
 */
#include <diagnostics/unittest.hpp>

// used components
#include <diagnostics/unittest/test_system/test_run_result.hpp>
#include <diagnostics/util/to_string.hpp>
#include <diagnostics/unittest/test_system_exception.hpp>

// test support
#include <diagnostics/util/record_vectors.ts.hpp>
#include <diagnostics/util/compilation_mode.ts.hpp>
#include <diagnostics/util/dummy_values_for_tid_pid.ts.hpp>

#include <string>

#define TEST_COMPONENT_NAME Test_Run_Result
#define TEST_COMPONENT_NAMESPACE diagnostics::unittest

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;

/**
 * @brief [PRO] fct/abnormal: invalid build_level and target_level, with NO records
 */
void invalid_cstr(Test_Data & test_data)
{
    typedef Test_Run_Result::Records_t Records_t;

    // LEVEL_TEST
    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_TEST,LEVEL_DEBUG);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_DEBUG,LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_TEST,LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // LEVEL_SYSTEM
    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_SYSTEM,LEVEL_DEBUG);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_DEBUG,LEVEL_SYSTEM);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_SYSTEM,LEVEL_SYSTEM);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // lower build_level than target_level
    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_PROD,LEVEL_DEBUG);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_PROD,LEVEL_AUDIT);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_DEBUG,LEVEL_AUDIT);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // LEVEL_TEST
    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_TEST,LEVEL_DEBUG,Records_t());
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_DEBUG,LEVEL_TEST,Records_t());
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_TEST,LEVEL_TEST,Records_t());
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // LEVEL_TEST
    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_SYSTEM,LEVEL_DEBUG,Records_t());
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_DEBUG,LEVEL_SYSTEM,Records_t());
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_SYSTEM,LEVEL_SYSTEM,Records_t());
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // lower build_level than target_level
    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_PROD,LEVEL_DEBUG,Records_t());
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_PROD,LEVEL_AUDIT,Records_t());
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result r1(LEVEL_DEBUG,LEVEL_AUDIT,Records_t());
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}



/**
 * @brief [PRO] fct/normal: all cstrs, assignment, comparision, with NO records
 */
void cstr_copy_assign_compare_recordless(Test_Data & test_data)
{
    using ::diagnostics::internal::to_string;
    typedef Test_Run_Result::Records_t Records_t;
    
    // Test_Run_Result(Level_t,Level_t)
    Test_Run_Result r1(LEVEL_PROD,LEVEL_PROD);
    TEST_ASSERT(r1.build_level()==LEVEL_PROD);
    TEST_ASSERT(r1.target_level()==LEVEL_PROD);
    TEST_ASSERT(r1.abstract_state()==Test_Run_Result::STATE_EMPTY);
    TEST_ASSERT(r1.failure_count()==0);
    TEST_ASSERT(r1.invalidation_count()==0);
    TEST_ASSERT(r1.records().size()==0);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r1)));
    // self cmp and assign
    TEST_ASSERT(r1==r1);
    r1=r1;
    TEST_ASSERT(r1.build_level()==LEVEL_PROD);
    TEST_ASSERT(r1.target_level()==LEVEL_PROD);
    TEST_ASSERT(r1.abstract_state()==Test_Run_Result::STATE_EMPTY);
    TEST_ASSERT(r1.failure_count()==0);
    TEST_ASSERT(r1.invalidation_count()==0);
    TEST_ASSERT(r1.records().size()==0);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r1)));

    // Test_Run_Result(Level_t,Level_t)
    Test_Run_Result r2(LEVEL_DEBUG,LEVEL_PROD);
    TEST_ASSERT(r2.build_level()==LEVEL_DEBUG);
    TEST_ASSERT(r2.target_level()==LEVEL_PROD);
    TEST_ASSERT(r2.abstract_state()==Test_Run_Result::STATE_EMPTY);
    TEST_ASSERT(r2.failure_count()==0);
    TEST_ASSERT(r2.invalidation_count()==0);
    TEST_ASSERT(r2.records().size()==0);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r2",to_string(r2)));
    // self cmp and assign
    TEST_ASSERT(r2==r2);
    r2=r2;
    TEST_ASSERT(r2.build_level()==LEVEL_DEBUG);
    TEST_ASSERT(r2.target_level()==LEVEL_PROD);
    TEST_ASSERT(r2.abstract_state()==Test_Run_Result::STATE_EMPTY);
    TEST_ASSERT(r2.failure_count()==0);
    TEST_ASSERT(r2.invalidation_count()==0);
    TEST_ASSERT(r2.records().size()==0);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r2",to_string(r2)));


    TEST_ASSERT(r1!=r2);

    // Test_Run_Result(level,records)
    Test_Run_Result r3(LEVEL_AUDIT,LEVEL_DEBUG,Records_t());
    TEST_ASSERT(r3.build_level()==LEVEL_AUDIT);
    TEST_ASSERT(r3.target_level()==LEVEL_DEBUG);
    TEST_ASSERT(r3.abstract_state()==Test_Run_Result::STATE_EMPTY);
    TEST_ASSERT(r3.failure_count()==0);
    TEST_ASSERT(r3.invalidation_count()==0);
    TEST_ASSERT(r3.records().size()==0);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r3",to_string(r3)));
    // self cmp and assign
    TEST_ASSERT(r3==r3);
    r3=r3;
    TEST_ASSERT(r3.build_level()==LEVEL_AUDIT);
    TEST_ASSERT(r3.target_level()==LEVEL_DEBUG);
    TEST_ASSERT(r3.abstract_state()==Test_Run_Result::STATE_EMPTY);
    TEST_ASSERT(r3.failure_count()==0);
    TEST_ASSERT(r3.invalidation_count()==0);
    TEST_ASSERT(r3.records().size()==0);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r3",to_string(r3)));


    TEST_ASSERT(r1!=r3);
    TEST_ASSERT(r2!=r3);

    // copy constr
    Test_Run_Result r4(r3);
    TEST_ASSERT(r4.build_level()==LEVEL_AUDIT);
    TEST_ASSERT(r4.target_level()==LEVEL_DEBUG);
    TEST_ASSERT(r4.abstract_state()==Test_Run_Result::STATE_EMPTY);
    TEST_ASSERT(r4.failure_count()==0);
    TEST_ASSERT(r4.invalidation_count()==0);
    TEST_ASSERT(r4.records().size()==0);
    TEST_ASSERT(r4==r3);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r3",to_string(r4)));

    // assign
    TEST_ASSERT(r1!=r3);
    r1=r3;
    TEST_ASSERT(r1==r3);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r3",to_string(r1)));
}

/**
 * @brief [PRO] fct/normal: cstrs with records, assignment, comparision, with records
 */
void cstr_copy_assign_compare_with_records(Test_Data & test_data)
{
    using ::diagnostics::internal::to_string;
    typedef Test_Run_Result::Records_t Records_t;

    // Test_Run_Result(level,records)
    Test_Run_Result r0(LEVEL_PROD,LEVEL_PROD,Empty_Trace_1sec_1usec_No_Carry().records());
    TEST_ASSERT(r0.build_level()==LEVEL_PROD);
    TEST_ASSERT(r0.target_level()==LEVEL_PROD);
    Empty_Trace_1sec_1usec_No_Carry().check(r0,false,LEVEL_PROD);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r0",to_string(r0)));
    // self cmp and assign
    TEST_ASSERT(r0==r0);
    r0=r0;
    TEST_ASSERT(r0.build_level()==LEVEL_PROD);
    TEST_ASSERT(r0.target_level()==LEVEL_PROD);
    Empty_Trace_1sec_1usec_No_Carry().check(r0,false,LEVEL_PROD);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r0",to_string(r0)));

    
    // Test_Run_Result(level,records)
    Test_Run_Result r1(LEVEL_AUDIT,LEVEL_DEBUG,Empty_Trace_0sec_0usec().records());
    TEST_ASSERT(r1.build_level()==LEVEL_AUDIT);
    TEST_ASSERT(r1.target_level()==LEVEL_DEBUG);
    Empty_Trace_0sec_0usec().check(r1,false,LEVEL_DEBUG);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r1)));
    // self cmp and assign
    TEST_ASSERT(r1==r1);
    r1=r1;
    TEST_ASSERT(r1.build_level()==LEVEL_AUDIT);
    TEST_ASSERT(r1.target_level()==LEVEL_DEBUG);
    Empty_Trace_0sec_0usec().check(r1,false,LEVEL_DEBUG);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r1)));


    // copy constr
    Test_Run_Result r2(r1);
    TEST_ASSERT(r2.build_level()==LEVEL_AUDIT);
    TEST_ASSERT(r2.target_level()==LEVEL_DEBUG);
    Empty_Trace_0sec_0usec().check(r2,false,LEVEL_DEBUG);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r2)));
    TEST_ASSERT(r2==r2);
    TEST_ASSERT(r1==r2);


    // assign
    Test_Run_Result r3(LEVEL_DEBUG,LEVEL_DEBUG);
    TEST_ASSERT(r3!=r2);
    r3=r2;
    TEST_ASSERT(r2==r3);
    Empty_Trace_0sec_0usec().check(r3,false,LEVEL_DEBUG);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r3)));


    // different level, different records...
    TEST_ASSERT(r0.records()!=r1.records());
    TEST_ASSERT(r0.build_level()!=r1.build_level());
    TEST_ASSERT(r0!=r1);

    TEST_ASSERT(r0!=r3);
    r3=r0;
    TEST_ASSERT(r0==r3);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r0",to_string(r3)));
    Empty_Trace_1sec_1usec_No_Carry().check(r3,false,LEVEL_PROD);
}


/**
 * @brief [PRO] fct/normal: correct traces and their evaluation 
 *
 * Copies the vector of Records m_record_vector->records() as a whole
 * and pice by pice.  Checks in both cases operator<<, whether the
 * right set of Records has been copied, whether the final result is
 * correct (m_record_vector->check), and whether it is impossible to
 * prolong the sequence after the closing of the Test_Case. The whole
 * test is repeated for all three possible target_levels.
 *
 * Checks again the copy cstr and assignment. 
 */
class Correct_Trace_Processing
{
    Record_Vector * const m_record_vector;
public:
    typedef Test_Run_Result::Records_t Records_t;
    typedef Records_t::const_iterator Const_Iterator_t;

    Correct_Trace_Processing(Record_Vector * const r)
	: m_record_vector(r)
    {
    }

    Correct_Trace_Processing(Correct_Trace_Processing const & other)
	: m_record_vector(other.m_record_vector->clone())
    {
    }
    
    ~Correct_Trace_Processing()
    {
	delete m_record_vector;
    }
    

    void operator()(Test_Data & test_data) const;

private:
    void copy_as_whole(Records_t const & orig,Test_Data & test_data,Level_t const target_level) const;
    void manual_copy(Records_t const & orig,Test_Data & test_data,Level_t const target_level) const;
    void check_impossible_adding(Type_t const type,
				 Test_Run_Result & result) const;

    void cmp_records_and_completion(Records_t const & orig,
				    Test_Run_Result & result) const;
    void cmp_records(Records_t const & orig,
		     Records_t const & copy) const;
};

void Correct_Trace_Processing::operator()(Test_Data & test_data) const
{
    Records_t const records(m_record_vector->records());
    copy_as_whole(records,test_data,LEVEL_AUDIT);
    manual_copy(records,test_data,LEVEL_AUDIT);

    copy_as_whole(records,test_data,LEVEL_DEBUG);
    manual_copy(records,test_data,LEVEL_DEBUG);

    copy_as_whole(records,test_data,LEVEL_PROD);
    manual_copy(records,test_data,LEVEL_PROD);
}

void Correct_Trace_Processing::copy_as_whole(Records_t const & orig,Test_Data & test_data,Level_t const target_level) const
{
    // copy as whole here
    Test_Run_Result result(LEVEL_AUDIT,target_level,orig); 
    // check the level
    TEST_ASSERT(result.build_level()==LEVEL_AUDIT);
    TEST_ASSERT(result.target_level()==target_level);
    // check the operator<<
    TEST_ASSERT(
		test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"result-" + ::std::string(level_to_string(target_level)),
		 ::diagnostics::internal::to_string<Test_Run_Result>(result))); 

    // check the records iteself
    cmp_records_and_completion(orig,result);

    // copy cstr
    Test_Run_Result result1(result);
    TEST_ASSERT(result==result1);
    cmp_records_and_completion(orig,result1);
    // assignment
    Test_Run_Result result2(LEVEL_AUDIT,LEVEL_AUDIT);
    TEST_ASSERT(result!=result2);
    result2=result;
    TEST_ASSERT(result==result2);
    cmp_records_and_completion(orig,result2);	
}


void Correct_Trace_Processing::manual_copy(Records_t const & orig,Test_Data & test_data,Level_t const target_level) const
{
    // empty copy
    Test_Run_Result result(LEVEL_AUDIT,target_level); 
    // check the level
    TEST_ASSERT(result.build_level()==LEVEL_AUDIT); 
    TEST_ASSERT(result.target_level()==target_level); 
    // really empty?
    TEST_ASSERT(result.abstract_state()==Test_Run_Result::STATE_EMPTY); 
	
    // do a manual copy just leaving out the last record
    Const_Iterator_t cur(orig.begin());
    Const_Iterator_t const end(orig.end()-1);
    for(;cur!=end;++cur) {
	result.add_record(*cur);
	// check incomplete state
	TEST_ASSERT(result.abstract_state()==Test_Run_Result::STATE_INCOMPLETE); 
    }
    // add last (check for completion (if complete) is in cmp_records_and_completion... by m_record_vector->check)
    result.add_record(*cur);

    // check operator<<
    TEST_ASSERT(
		test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"result-" + ::std::string(level_to_string(target_level)),
		 ::diagnostics::internal::to_string<Test_Run_Result>(result))); 

    // check the records itself 
    cmp_records_and_completion(orig,result);
}

void Correct_Trace_Processing::check_impossible_adding(Type_t const type,
						       Test_Run_Result & result) const
{
    TEST_THROWING_BLOCK_ENTER; 
    result.add_record(CSTR_RECORD("invalid add",LEVEL_PROD,type)); 
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception); 

    TEST_THROWING_BLOCK_ENTER; 
    result.add_record(CSTR_RECORD("invalid add",LEVEL_DEBUG,type)); 
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception); 

    TEST_THROWING_BLOCK_ENTER; 
    result.add_record(CSTR_RECORD("invalid add",LEVEL_AUDIT,type)); 
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception); 

    TEST_THROWING_BLOCK_ENTER; 
    result.add_record(CSTR_RECORD("invalid add",LEVEL_TEST,type)); 
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception); 

    result.add_record(CSTR_RECORD("ignored",LEVEL_SYSTEM,type)); 
}


void Correct_Trace_Processing::cmp_records_and_completion(Records_t const & orig,
							  Test_Run_Result & result) const
{
    // check the records iteself
    cmp_records(orig,result.records());
    // check the derived variables
    m_record_vector->check(result,false,result.target_level()); 
	
    // check whether TYPE_LOG_OPEN and TYPE_LOG_CLOSE are really ignored 
    result.add_record(CSTR_RECORD("ignored",LEVEL_PROD,TYPE_LOG_OPEN));
    result.add_record(CSTR_RECORD("ignored",LEVEL_PROD,TYPE_LOG_CLOSE));
    result.add_record(CSTR_RECORD("ignored",LEVEL_DEBUG,TYPE_LOG_OPEN));
    result.add_record(CSTR_RECORD("ignored",LEVEL_DEBUG,TYPE_LOG_CLOSE));
    result.add_record(CSTR_RECORD("ignored",LEVEL_AUDIT,TYPE_LOG_OPEN));
    result.add_record(CSTR_RECORD("ignored",LEVEL_AUDIT,TYPE_LOG_CLOSE));
    result.add_record(CSTR_RECORD("ignored",LEVEL_TEST,TYPE_LOG_OPEN));
    result.add_record(CSTR_RECORD("ignored",LEVEL_TEST,TYPE_LOG_CLOSE));

    check_impossible_adding(TYPE_BLOCK_ENTER,result);
    check_impossible_adding(TYPE_BLOCK_EXIT,result);
    check_impossible_adding(TYPE_PROCEDURE_ENTER,result);
    check_impossible_adding(TYPE_PROCEDURE_EXIT,result);
    check_impossible_adding(TYPE_METHOD_ENTER,result);
    check_impossible_adding(TYPE_METHOD_EXIT,result);
    check_impossible_adding(TYPE_TRACE,result);
    check_impossible_adding(TYPE_TRACE_BINARY,result);
    check_impossible_adding(TYPE_FAILED_CHECK,result);
    check_impossible_adding(TYPE_UNEXPECTED_EXCEPTION,result);
    check_impossible_adding(TYPE_WRONG_EXCEPTION,result);
    check_impossible_adding(TYPE_MISSING_EXCEPTION,result);
    check_impossible_adding(TYPE_TESTCASE_ENTER,result);
    check_impossible_adding(TYPE_TESTCASE_EXIT,result);

    // check the records again
    cmp_records(orig,result.records());
    // check the derived variables
    m_record_vector->check(result,false,result.target_level()); 
}


void Correct_Trace_Processing::cmp_records(Records_t const & orig,
					   Records_t const & copy) const
{
    Const_Iterator_t cur(orig.begin());
    Const_Iterator_t const end(orig.end());
    Const_Iterator_t cur2(copy.begin());
    Const_Iterator_t const end2(copy.end());
    for(;cur!=end;++cur) {
	if(cur->type()==TYPE_LOG_OPEN) continue;
	if(cur->type()==TYPE_LOG_CLOSE) continue;
	TEST_ASSERT(*cur==*cur2++);
    }
    TEST_ASSERT(cur2==end2);
}


/**
 * @brief [PRO] fct/abnormal: incorrect traces and their evaluation 
 *
 * Copies the vector of Records m_record_vector->records() as a whole and pice by
 * pice. When copied as whole, it is only checked that
 * Test_System_Exception is thrown. 
 *
 * When copied pice by pice, all records but the last must be
 * successfully copied. The last one must cause an error.  Then, it is
 * checked that the final result is correct (m_record_vector->check),
 * and that the right set of records has been copied. This is only
 * done for target_level @ref LEVEL_PROD.
 */
class Incorrect_Trace_Processing 
{
    Record_Vector * const m_record_vector;
public:
    typedef Test_Run_Result::Records_t Records_t;
    typedef Records_t::const_iterator Const_Iterator_t;

    Incorrect_Trace_Processing(Record_Vector * const r)
	: m_record_vector(r)
    {
    }
    
    Incorrect_Trace_Processing(Incorrect_Trace_Processing const & other)
	: m_record_vector(other.m_record_vector->clone())
    {
    }

    ~Incorrect_Trace_Processing()
    {
	delete m_record_vector;
    }


    void operator()(Test_Data & test_data) const;
    
private:
    void copy_as_whole(Records_t const & orig,Test_Data & test_data) const;
    void manual_copy(Records_t const & orig,Test_Data & test_data) const;
    void check_result(Records_t const & orig,
		      Test_Run_Result const & result) const;
};


void Incorrect_Trace_Processing::operator()(Test_Data & test_data) const
{
    Records_t const records(m_record_vector->records());
    copy_as_whole(records,test_data);
    manual_copy(records,test_data);
}

void Incorrect_Trace_Processing::copy_as_whole(Records_t const & orig,Test_Data & test_data) const
{
    // copy as whole here
    TEST_THROWING_BLOCK_ENTER;
    Test_Run_Result result(LEVEL_PROD,LEVEL_PROD,orig); 
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}


void Incorrect_Trace_Processing::manual_copy(Records_t const & orig,Test_Data & test_data) const
{
    // empty copy
    Test_Run_Result result(LEVEL_PROD,LEVEL_PROD); 
    // check the level
    TEST_ASSERT(result.build_level()==LEVEL_PROD); 
    TEST_ASSERT(result.target_level()==LEVEL_PROD); 
    // really empty?
    TEST_ASSERT(result.abstract_state()==Test_Run_Result::STATE_EMPTY); 
    
    
    // do a manual copy just leaving out the last record
    Const_Iterator_t cur(orig.begin());
    Const_Iterator_t const end(orig.end()-1);
    for(;cur!=end;++cur) {
	result.add_record(*cur);
	// check (in)complete state
	TEST_ASSERT(result.abstract_state()==
		    (cur->type()==TYPE_TESTCASE_EXIT 
		     ? Test_Run_Result::STATE_COMPLETE 
		     : Test_Run_Result::STATE_INCOMPLETE)); 
    }
    
    // add last -- must fail
    TEST_THROWING_BLOCK_ENTER;
    result.add_record(*cur);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // check operator<<
    TEST_ASSERT(
		test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"result",
		 ::diagnostics::internal::to_string<Test_Run_Result>(result))); 
    
    // check the records itself 
    check_result(orig,result);
}

void Incorrect_Trace_Processing::check_result(Records_t const & orig,
					      Test_Run_Result const & result) const
{
    Const_Iterator_t cur(orig.begin());
    Const_Iterator_t const end(orig.end()-1); // NOT the last one
    Const_Iterator_t cur2(result.records().begin());
    Const_Iterator_t const end2(result.records().end());
    for(;cur!=end;++cur) {
	if(cur->type()==TYPE_LOG_OPEN) continue;
	if(cur->type()==TYPE_LOG_CLOSE) continue;
	TEST_ASSERT(*cur==*cur2++);
    }
    TEST_ASSERT(cur2==end2);
    m_record_vector->check(result,false,LEVEL_PROD);
}

/**
 * @brief [PRO] impl: The implementation for deciding whether the
 * target_level and the build_level are in the correct relationship to
 * each other depends on the following ordering.
 */
void level_odering(Test_Data &)
{
    TEST_ASSERT(LEVEL_PROD<LEVEL_DEBUG);
    TEST_ASSERT(LEVEL_DEBUG<LEVEL_AUDIT);
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


TEST_SUITE_BEGIN;
TEST_ABNORMAL_CASE(&invalid_cstr,LEVEL_PROD);
TEST_NORMAL_CASE(&cstr_copy_assign_compare_recordless,LEVEL_PROD);
TEST_NORMAL_CASE(&cstr_copy_assign_compare_with_records,LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Empty_Trace_0sec_0usec()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Empty_Trace_1sec_1usec_No_Carry()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Empty_Trace_1sec_1usec_With_Carry()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_One_Failure()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_One_Invalidation()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_One_Failed_Check()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_All_Level_Type_Combinations()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_All_Level_Type_Combinations_Times_2()),LEVEL_PROD);
TEST_ABNORMAL_CASE(Incorrect_Trace_Processing(new Trace_With_Two_Test_Case_Enter_Msgs()),LEVEL_PROD);
TEST_ABNORMAL_CASE(Incorrect_Trace_Processing(new Trace_With_No_Test_Case_Enter()),LEVEL_PROD);
TEST_ABNORMAL_CASE(Incorrect_Trace_Processing(new Trace_With_Two_Test_Case_Exits()),LEVEL_PROD);
TEST_IMPL_CASE(&level_odering);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
