/* Copyright 1992 John Bovey, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */
/*
 * This module has been very heavily modified by R. Nation
 * (nation@rocket.sanders.lockheed.com).
 * No additional restrictions are applied
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 */

#include <stdarg.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <grp.h>
#include <string.h>
#include <sys/time.h>
#ifdef SVR4
#include <sys/stropts.h>
#endif

#include <termios.h>

#ifdef FREEBSD
#include <sys/ioctl.h>
#endif

#include <sys/wait.h>
#include <sys/stat.h>
#include <stdlib.h>

#include "rxvt.h"
#include "command.h"
#include "screen.h"
#include "xsetup.h"
#include "sbar.h"

int size_set = 0;      /* flag set once the window size has been set */

#define KBUFSIZE	256	/* size of keyboard mapping buffer */
#define COM_BUF_SIZE   4000	/* size of buffer used to read from command */
#define COM_PUSH_MAX	20	/* max number of chars to push back in queue */
#define STRING_MAX     4000     /* maximum size of strings to process */
#define ARGS_MAX         20     /* maximum # of args for esc sequences */

/*  Special character returned by get_com_char().
 */
#define GCC_NULL	0x100		/* Input buffer is empty */
#define ESC		033

/*  Flags used to control get_com_char();
 */
#define BUF_ONLY	1

/*  Global variables that are set up at the beginning and then not changed
 */
extern Display		*display;
extern Window		vt_win;
extern Window		main_win;
extern struct sbar_info sbar;

extern int login_shell;

int comm_fd = -1;/* file descriptor connected to the command */
static int comm_pid;	/* process id if child */
static int x_fd;	/* file descriptor of the X server connection */
static int fd_width;	/* width of file descriptors being used */
static int app_cur_keys = 0;/* flag to set cursor keys in application mode */
static int app_kp_keys = 0; /* flag to set application keypad keys */
#ifndef SVR4
static char ptynam[] = "/dev/ptyxx";
static char ttynam[] = "/dev/ttyxx";
#endif
static Atom wm_del_win;

extern int fat_sbar, cheight,cwidth,needs_complete_refresh, sline_top,offset;
int console;

KeySym SecureKeysym = XK_s;
KeySym BigFontKeysym = XK_greater;
KeySym SmallFontKeysym = XK_less;
KeySym PageUpKeysym = XK_Prior;
KeySym PageDownKeysym = XK_Next;

int refresh_nl_count;

/*  Terminal mode structures.
 */
static struct termios ttmode;
#undef CTRL
#define CTRL(c) ((c) - '@')

#ifndef CINTR
#ifndef _POSIX_VDISABLE
#define _POSIX_VDISABLE 0
#endif

#ifdef SVR4
#define CINTR 0177
#define CQUIT CTRL('U')
#define CERASE CTRL('H')
#else
#define CINTR CTRL('C')
#define CQUIT CTRL('\\')
#define CERASE 0177
#endif

#define CEOF CTRL('D')
#define CKILL CTRL('U')
#define CEOL _POSIX_VDISABLE
#define CEOL2 _POSIX_VDISABLE
#define CNSWTCH _POSIX_VDISABLE
#define CSTART CTRL('Q')
#define CSTOP CTRL('S')
#define CSUSP CTRL('Z');
#define CDSUSP _POSIX_VDISABLE
#define CRPRNT CTRL('R')
#define CFLUSH CTRL('O')
#define CWERASE CTRL('W')
#define CLNEXT CTRL('V')
#endif


/*  Variables used for buffered command input.
 */
static unsigned char com_buf[COM_BUF_SIZE];
static unsigned char *com_buf_next, *com_buf_top;
static unsigned char com_stack[COM_PUSH_MAX];
static unsigned char *com_stack_top;
unsigned char mask = 0xff;

static void catch_child(int);
static void catch_sig(int);
static int run_command(unsigned char *,unsigned char **);
static unsigned char *lookup_key(XEvent *,int *);
static int get_com_char(int);
static void push_com_char(int);
void get_X_event(void);
void process_string(int);
void process_escape_sequence(void);
void process_csi_sequence(void);
void process_xterm_sequence(void);
void process_terminal_mode(int,int,int,int *);
void process_sgr_mode(int,int,int,int *);


#ifdef SVR4
#define _NEW_TTY_CTRL
#include <sys/resource.h>
int static getdtablesize() 
{
  struct rlimit   rlp;
  getrlimit(RLIMIT_NOFILE, &rlp);
  return(rlp.rlim_cur);
}
#endif


/*  Catch a SIGCHLD signal and exit if the direct child has died.
 */
static void catch_child(int nonsense)
{
  if (wait((int *)NULL) == comm_pid)
    clean_exit(0);
}

/*  Catch a fatal signal and tidy up before quitting
 */
static void catch_sig(int sig)
{
  signal(sig,SIG_DFL);
  setuid(getuid());
  cleanutent();
  kill(getpid(),sig);
}

/*  Run the command in a subprocess and return a file descriptor for the
 *  master end of the pseudo-teletype pair with the command talking to
 *  the slave.
 */
static int run_command(unsigned char *command,unsigned char **argv)
{
  int ptyfd, ttyfd;
  int uid, gid;
  unsigned char *s3, *s4;
  int i;
  int width, height;
#ifdef SVR4
  char *ttynam, *ptsname();
#else
  static char ptyc3[] = "pqrstuvwxyz";
  static char ptyc4[] = "0123456789abcdef";
 #endif
  unsigned char argv0[256];

  /*  First find a master pty that we can open.  
   */
#ifdef SVR4
  ptyfd = open("/dev/ptmx",O_RDWR);
  if (ptyfd < 0) 
    {
      error("Can't open a pseudo teletype");
      return(-1);
    }
  grantpt(ptyfd);
  unlockpt(ptyfd);
  fcntl(ptyfd,F_SETFL,O_NDELAY);
  ttynam=ptsname(ptyfd);
#else
  ptyfd = -1;
  for (s3 = ptyc3; *s3 != 0; s3++) 
    {
      for (s4 = ptyc4; *s4 != 0; s4++) 
	{
	  ptynam[8] = ttynam[8] = *s3;
	  ptynam[9] = ttynam[9] = *s4;
	  if ((ptyfd = open(ptynam,O_RDWR)) >= 0) 
	    {
	      if (geteuid() == 0 || access(ttynam,R_OK|W_OK) == 0)
		break;
	      else 
		{
		  close(ptyfd);
		  ptyfd = -1;
		}
	    }
	}
      if (ptyfd >= 0)
	break;
    }
  if (ptyfd < 0) 
    {
      error("Can't open a pseudo teletype");
      return(-1);
    }
  fcntl(ptyfd,F_SETFL,O_NDELAY);
#endif  
  for (i = 1; i <= 15; i++)
    signal(i,catch_sig);
  signal(SIGCHLD,catch_child);
  makeutent(&ttynam[5]);	/* stamp /etc/utmp */  
  comm_pid = fork();
  if (comm_pid < 0) 
    {
      error("Can't fork");
      return(-1);
    }
  if (comm_pid == 0) 
    {
      struct group *gr;
      
      if (setsid() < 0)
	error("failed to set process group");
      
      if ((ttyfd = open(ttynam,O_RDWR)) < 0) 
	{
	  error("could not open slave tty %s",ttynam);
	  clean_exit(1);
	}
#ifdef SVR4
      ioctl(ttyfd,I_PUSH,"ptem");
      ioctl(ttyfd,I_PUSH,"ldterm");
#endif
#if defined(TIOCSCTTY)
      ioctl(ttyfd, TIOCSCTTY, 0) ;
#endif

      uid = getuid();
#ifndef SVR4
      if ((gr = getgrnam("tty")) != NULL)
	gid = gr->gr_gid;
      else
	gid = -1;
      fchown(ttyfd,uid,gid);
      fchmod(ttyfd,0600);
#endif
#ifdef TIOCCONS
     if (console) 
	{
	  int on = 1;
	  if (ioctl (ttyfd, TIOCCONS, (unsigned char *)&on) == -1)
	    fprintf(stderr, "xvt: cannot open console\n");
	}
#endif  /* TIOCCONS */

      for (i = 0; i < getdtablesize(); i++)
	if (i != ttyfd)
	  close(i);
      dup(ttyfd);
      dup(ttyfd);
      dup(ttyfd);

      if (ttyfd > 2)
	close(ttyfd);
      
      /* init of termios structure		*/
      ttmode.c_iflag = BRKINT | IGNPAR | ICRNL| IXON | IMAXBEL;
      ttmode.c_oflag = OPOST | ONLCR ;
      ttmode.c_cflag = B9600 | CS8 | CREAD;
      ttmode.c_lflag = ISIG|IEXTEN|ICANON|ECHO|ECHOE|ECHOK|ECHOCTL|ECHOKE;

      ttmode.c_cc[VEOF] = CEOF;
      ttmode.c_cc[VEOL] = CEOL; 
      ttmode.c_cc[VINTR] = CINTR;
      ttmode.c_cc[VQUIT] = CQUIT;
      ttmode.c_cc[VERASE] = CERASE;
      ttmode.c_cc[VKILL] = CKILL;
      ttmode.c_cc[VSUSP] = CSUSP;
#ifdef VDSUSP
      ttmode.c_cc[VDSUSP] = CDSUSP;
#endif
      ttmode.c_cc[VSTART] = CSTART;
      ttmode.c_cc[VSTOP] = CSTOP;
      ttmode.c_cc[VREPRINT] = CRPRNT;
      ttmode.c_cc[VDISCARD] = CFLUSH;
      ttmode.c_cc[VWERASE] = CWERASE;
      ttmode.c_cc[VLNEXT] = CLNEXT;
#ifdef VSWTC
      ttmode.c_cc[VSWTC] = 0;
#endif 
#ifdef VSWTCH
      ttmode.c_cc[VSWTCH] = 0;
#endif 
#if VMIN != VEOF
      ttmode.c_cc[VMIN] = 1;
#endif
#if VTIME != VEOL
      ttmode.c_cc[VTIME] = 0;
#endif


      if(mask == 0x7f)
	ttmode.c_cflag = B9600 | PARENB | CS7 | CREAD;
#ifdef FREEBSD
      ioctl(0,TIOCSETA,(char *)&ttmode);
#else
      ioctl(0,TCSETS,(char *)&ttmode);
#endif
      scr_get_size(&width,&height);
      tty_set_size(0,width,height);
      setgid(getgid());
      setuid(uid);
      if(login_shell)
	{
	  strcpy(argv0,"-");
	  strcat(argv0,argv[0]);
	  argv[0] = argv0;
	}
      execvp(command,(char **)argv);
      error("Couldn't execute %s",command);
      clean_exit(1);
    }
  return(ptyfd);
}

/*  Tell the teletype handler what size the window is.  Called after a window
 *  size change.
 */
void tty_set_size(int fd,int width,int height)
{
  struct winsize wsize;
  
  if(fd < 0)
    return;

  wsize.ws_row = (unsigned short)height;
  wsize.ws_col = (unsigned short)width;
  ioctl(fd,TIOCSWINSZ,(char *)&wsize);
}

/*  Initialise the command connection.  This should be called after the X
 *  server connection is established.
 */
void init_command(unsigned char *command,unsigned char **argv)
{
  /*  Enable the delete window protocol.
   */
  wm_del_win = XInternAtom(display,"WM_DELETE_WINDOW",False);
  XSetWMProtocols(display,main_win,&wm_del_win,1);
  
  if ((comm_fd = run_command(command,argv)) < 0) 
    {
      error("Quitting");
      clean_exit(1);
    }

  x_fd = XConnectionNumber(display);
  fd_width = getdtablesize();
  com_buf_next = com_buf_top = com_buf;
  com_stack_top = com_stack;
}

/*  Convert the keypress event into a string.
 */
static unsigned char *lookup_key(XEvent *ev,int *pcount)
{
  KeySym keysym;
  XComposeStatus compose;
  int count;
  static unsigned char kbuf[KBUFSIZE];
  unsigned char *s, *c;
  int meta;

  count = XLookupString(&ev->xkey,kbuf,KBUFSIZE-1,&keysym,&compose);
  kbuf[count] = (unsigned char)0;
  meta = ev->xkey.state & Mod1Mask;
  s = NULL;
  if((keysym == SecureKeysym)&&((ev->xkey.state &Mod1Mask) == Mod1Mask))
    {
      scr_secure();
      count = 0;
    }
  else if((keysym == BigFontKeysym)&&((ev->xkey.state &Mod1Mask) == Mod1Mask))
    {
      NewFont(BIGGER);
      count = 0;
    }
  else if((keysym==SmallFontKeysym)&&((ev->xkey.state &Mod1Mask) == Mod1Mask))
    {
      NewFont(SMALLER);
      count = 0;
    }
  else if((keysym == PageUpKeysym)&&((ev->xkey.state &Mod1Mask) == Mod1Mask))
    {
      scr_move_up_down(UP);
      count = 0;
    }
  else if((keysym == PageDownKeysym)&&((ev->xkey.state &Mod1Mask) == Mod1Mask))
    {
      scr_move_up_down(DOWN);
      count = 0;
    }
  else switch(keysym)
    {
    case XK_Up :
      strcpy(kbuf,(app_cur_keys ? "\033OA" : "\033[A"));
      count = strlen(kbuf);
      break;
    case XK_Down :
      strcpy(kbuf,app_cur_keys ? "\033OB" : "\033[B");
      count = strlen(kbuf);
      break;
    case XK_Right :
      strcpy(kbuf,app_cur_keys ? "\033OC" : "\033[C");
      count = strlen(kbuf);
      break;
    case XK_Left :
      strcpy(kbuf,app_cur_keys ? "\033OD" : "\033[D");
      count = strlen(kbuf);
      break;
    case XK_KP_F1 :
      strcpy(kbuf,"\033OP");
      count = 3;
      break;
    case XK_KP_F2 :
      strcpy(kbuf,"\033OQ");
      count = 3;
      break;
    case XK_KP_F3 :
      strcpy(kbuf,"\033OR");
      count = 3;
      break;
    case XK_KP_F4 :
      strcpy(kbuf,"\033OS");
      count = 3;
      break;
    case XK_KP_0 :
      strcpy(kbuf,app_kp_keys ? "\033Op" : "0");
      count = strlen(kbuf);
      break;
    case XK_KP_1 :
      strcpy(kbuf,app_kp_keys ? "\033Oq" : "1");
      count = strlen(kbuf);
      break;
    case XK_KP_2 :
      strcpy(kbuf,app_kp_keys ? "\033Or" : "2");
      count = strlen(kbuf);
      break;
    case XK_KP_3 :
      strcpy(kbuf,app_kp_keys ? "\033Os" : "3");
      count = strlen(kbuf);
      break;
    case XK_KP_4 :
      strcpy(kbuf,app_kp_keys ? "\033Ot" : "4");
      count = strlen(kbuf);
      break;
    case XK_KP_5 :
      strcpy(kbuf,app_kp_keys ? "\033Ou" : "5");
      count = strlen(kbuf);
      break;
    case XK_KP_6 :
      strcpy(kbuf,app_kp_keys ? "\033Ov" : "6");
      count = strlen(kbuf);
      break;
    case XK_KP_7 :
      strcpy(kbuf,app_kp_keys ? "\033Ow" : "7");
      count = strlen(kbuf);
      break;
    case XK_KP_8 :
      strcpy(kbuf,app_kp_keys ? "\033Ox" : "8");
      count = strlen(kbuf);
      break;
    case XK_KP_9 :
      strcpy(kbuf,app_kp_keys ? "\033Oy" : "9");
      count = strlen(kbuf);
      break;
    case XK_KP_Subtract :
      strcpy(kbuf,app_kp_keys ? "\033Om" : "-");
      count = strlen(kbuf);
      break;
    case XK_KP_Separator :
      strcpy(kbuf,app_kp_keys ? "\033Ol" : ",");
      count = strlen(kbuf);
      break;
    case XK_KP_Decimal :
      strcpy(kbuf,app_kp_keys ? "\033On" : ".");
      count = strlen(kbuf);
      break;
    case XK_KP_Enter :
      strcpy(kbuf,app_kp_keys ? "\033OM" : "\r");
      count = strlen(kbuf);
      break;
    case XK_Home :
      strcpy(kbuf,"\033Oq");
      count = 3;
      break;
    case XK_End :
      strcpy(kbuf,"\033Ow");
      count = 3;
      break;
    case XK_F1 :
      strcpy(kbuf,"\033[11~");
      XMoveWindow(display,main_win,100,100);
      count = 5;
      break;
    case XK_F2 :
      strcpy(kbuf,"\033[12~");
      count = 5;
      break;
    case XK_F3 :
      strcpy(kbuf,"\033[13~");
      count = 5;
      break;
    case XK_F4 :
      strcpy(kbuf,"\033[14~");
      count = 5;
      break;
    case XK_F5 :
      strcpy(kbuf,"\033[15~");
      count = 5;
      break;
    case XK_F6 :
      strcpy(kbuf,"\033[17~");
      count = 5;
      break;
    case XK_F7 :
      strcpy(kbuf,"\033[18~");
      count = 5;
      break;
    case XK_F8 :
      strcpy(kbuf,"\033[19~");
      count = 5;
      break;
    case XK_F9 :
      strcpy(kbuf,"\033[20~");
      count = 5;
      break;
    case XK_F10 :
      strcpy(kbuf,"\033[21~");
      count = 5;
      break;
    case XK_F11 :
      strcpy(kbuf,"\033[23~");
      count = 5;
      break;
    case XK_F12 :
      strcpy(kbuf,"\033[24~");
      count = 5;
      break;
    case XK_F13 :
      strcpy(kbuf,"\033[25~");
      count = 5;
      break;
    case XK_F14 :
      strcpy(kbuf,"\033[26~");
      count = 5;
      break;
    case XK_Help:
    case XK_F15:
      strcpy(kbuf,"\033[28~");
      count = 5;
      break;
    case XK_Menu:
    case XK_F16:
      strcpy(kbuf,"\033[29~");
      count = 5;
      break;
    case XK_F17 :
      strcpy(kbuf,"\033[31~");
      count = 5;
      break;
    case XK_F18 :
      strcpy(kbuf,"\033[32~");
      count = 5;
      break;
    case XK_F19 :
      strcpy(kbuf,"\033[33~");
      count = 5;
      break;
    case XK_F20 :
      strcpy(kbuf,"\033[34~");
      count = 5;
      break;
    case XK_Find :
      strcpy(kbuf,"\033[1~");
      count = 4;
      break;
    case XK_Insert :
      strcpy(kbuf,"\033[2~");
      count = 4;
      break;
    case XK_Execute :
      strcpy(kbuf,"\033[3~");
      count = 4;
      break;
    case XK_Select :
      strcpy(kbuf,"\033[4~");
      count = 4;
      break;
    case XK_Prior :
      strcpy(kbuf,"\033[5~");
      count = 4;
      break;
    case XK_Next:
      strcpy(kbuf,"\033[6~");
      count = 4;
      break;
    }
  *pcount = count;
  /* if meta is pressed, precede message with ESC */
  /* used to set the 8th bit high, but this confuses things
   * for some non-english character sets */
  if(meta &&(count > 0))
    {
      *pcount = count + 1;
      for(c = (kbuf + (count)); c > kbuf ; c--)
	{
	  *c = *(c-1);
	}
      *kbuf = 27;
      *(kbuf+count+1)=0;
    }
  
  return (kbuf);
}

/*  Return the next input character after first passing any keyboard input
 *  to the command.  If flags & BUF_ONLY is true then only buffered 
 *  characters are returned and once the buffer is empty the special value 
 *  GCC_NULL is returned.  
 */
static int get_com_char(int flags)
{
  fd_set in_fdset;
  int count;
  unsigned char val;
  static int returned = 0;

  if (com_stack_top > com_stack)
    {
      returned++;
      return(*--com_stack_top);
    }
  
  if (com_buf_next < com_buf_top)
    {
      returned++;
      return(*com_buf_next++ & mask);
    }
  else if (flags & BUF_ONLY)
    {
      if((returned < STRING_MAX)&&(refresh_nl_count < cheight))
	{
	  count = read(comm_fd,com_buf,COM_BUF_SIZE);
	  if (count > 0)
	    {
	      com_buf_next = com_buf;
	      com_buf_top = com_buf + count;
	      val = *com_buf_next;
	      com_buf_next++ ;
	      returned++;
	      return(val & mask);	  
	    }
	}
      return (GCC_NULL);
    }

  returned = 0;
  for (;;) 
    {
      /* try reading from each descriptor before going into select(); */
      while(XPending(display))
	get_X_event();
      
      if(needs_complete_refresh == 1)
	{
	  refresh_nl_count = 0;
	  refresh(0,cheight-1,0,cwidth-1);
	  cursor(ON);
	  XFlush(display);
	}

      FD_ZERO(&in_fdset);
      FD_SET(comm_fd,&in_fdset);
      FD_SET(x_fd,&in_fdset);
      select(fd_width,&in_fdset,NULL,NULL,NULL);

      if(FD_ISSET(comm_fd,&in_fdset))
	{
	  count = read(comm_fd,com_buf,COM_BUF_SIZE);
	  if (count > 0)
	    {
	      com_buf_next = com_buf;
	      com_buf_top = com_buf + count;
	      val = *com_buf_next;
	      com_buf_next++ ;
	      return(val & mask);	  
	    }
	}
    }
}
	
void get_X_event(void)
{
  XEvent event;
  unsigned char *s;
  int count;
  Window root, child;
  int root_x, root_y, x, y,tl,lk;
  unsigned int mods;
  XEvent dummy;

  XNextEvent(display,&event);
  switch(event.type)
    {
    case KeyPress:
      s = lookup_key(&event,&count);
      send_string(s,count);
      return;
    case ClientMessage:
      if (event.xclient.format == 32 && event.xclient.data.l[0] == wm_del_win)
	clean_exit(0);
      return;
    case MappingNotify:
      XRefreshKeyboardMapping(&event.xmapping);
      return;
    case GraphicsExpose:
    case Expose:
      if (!size_set) 
	{	      
	  /*  Force a window resize if an exposure event
	   *  arrives before the first resize event.
	   */
	  resize_window();
	  size_set = 1;
	}
      if(event.xany.window==vt_win)
	{
	  scr_refresh(event.xexpose.x,event.xexpose.y,
		      event.xexpose.width,event.xexpose.height);
	  return;
	}
      else 
	{
	  while(XCheckTypedWindowEvent (display, event.xany.window, Expose,
					&dummy));
	  while(XCheckTypedWindowEvent (display, event.xany.window, 
					GraphicsExpose, &dummy));

	  if(event.xany.window==sbar.sb_win)	  
	    {
	      sbar_show(-1,-1,-1);
	      return;
	    }
	  else if(event.xany.window==sbar.sb_up_win)	  
	    {
	      sbar_up_reset();
	      return;
	    }
	  else if(event.xany.window==sbar.sb_down_win)	  
	    {
	      sbar_down_reset();
	      return;
	    }
	}
      return;
    case FocusIn:
      scr_focus(1);
      return;
    case FocusOut:
      scr_focus(0);
      return;
    case ConfigureNotify:
      resize_window();
      size_set = 1;
      return;
    case SelectionClear:
      scr_clear_selection();
      return;
    case SelectionNotify:
      scr_paste_primary(event.xselection.requestor,
			 event.xselection.property);
      return;
    case SelectionRequest:
      scr_send_selection(event.xselectionrequest.time,
			 event.xselectionrequest.requestor,
			 event.xselectionrequest.target,
			 event.xselectionrequest.property);
      return;
    case ButtonPress:
      if (event.xany.window == vt_win)
	{
	  switch (event.xbutton.button)
	    {
	    case Button1 :
	    case Button3 :
	      scr_start_selection(event.xbutton.x,event.xbutton.y);
	      return;
	    default:
	      return;
	    }
	}
      if (event.xany.window == sbar.sb_win)
	{
	  if(!fat_sbar)
	    scr_move_to(event.xbutton.y);
	  else
	    {
	      switch (event.xbutton.button)
		{
		case Button1 :
		  tl =cheight + sline_top ;
		  lk = event.xbutton.y * (cheight) / (sbar.height - 1);
		  event.xbutton.y = (tl-offset-cheight+lk)*(sbar.height-1)/tl;
		  scr_move_to(event.xbutton.y);
		  break;
		case Button2 :
		  scr_move_to(event.xbutton.y);
		  break;
		case Button3 :
		  tl =cheight + sline_top ;
		  lk = event.xbutton.y * (cheight) / (sbar.height - 1);
		  event.xbutton.y = (tl-offset-cheight-lk)*(sbar.height-1)/tl;
		  scr_move_to(event.xbutton.y);
		  break;
		default :
		  break;
		}
	    }
	  return;
	}
      if (event.xany.window == sbar.sb_up_win)
	{
	  scr_move_up_down(UP);
	  return;
	}
      if (event.xany.window == sbar.sb_down_win)
	{
	  scr_move_up_down(DOWN);
	  return;
	}
      return;
    case ButtonRelease :
      if (event.xany.window == vt_win)
	{
	  switch (event.xbutton.button)
	    {
	    case Button1 :
	    case Button3 :
	      scr_make_selection(event.xbutton.time);
	      return;
	    case Button2 :
	      scr_request_selection(event.xbutton.time,event.xbutton.x,
				    event.xbutton.y);
	      return;
	    }
	}
      return;
    case MotionNotify :
      if ((event.xany.window == sbar.sb_win)
	  &&((event.xbutton.state&Button1Mask)||
	     (event.xbutton.state&Button2Mask)||
	     (event.xbutton.state&Button3Mask)))
	{
	  XQueryPointer(display,sbar.sb_win,&root,&child,
			&root_x,&root_y,&x,&y,&mods);
	  if((!fat_sbar)||(event.xbutton.state&Button2Mask))
	    scr_move_to(y);
	  return;
	}

      if (event.xany.window == vt_win && 
	  ((event.xbutton.state == Button1Mask)
	   ||(event.xbutton.state == Button3Mask)))
	{
	  scr_extend_selection(event.xbutton.x,event.xbutton.y);
	  return;
	}
      return;
    default:
      return;
    }
}


/*  Push an input character back into the input queue.
 */
static void push_com_char(int c)
{
  if (com_stack_top < com_stack + COM_PUSH_MAX)
    *com_stack_top++ = c;
}

/*  Send count characters directly to the command.
 */
void send_string(unsigned char *buf,int count)
{
#ifdef 0
  int i;

  for(i=0;i<count;i++)
    {
      if (write(comm_fd,buf,1) <= 0) 
	XBell(display,0);
      buf++;
    }
#endif
  int i, write_bytes, wrote;
  
  if (count <= 0)
    return;
  
  i = 0;
  write_bytes = count;
  
  while(1)
    {
      wrote = write(comm_fd, buf, write_bytes);
      
      if ((wrote == write_bytes) && (i + wrote == count))
	break;
      
      if (wrote <= 0)
	{
	  if (write_bytes == 1)   /* If we wrote the smallest chunk    */
	    {                     /* possible and it still didn't like */
	      XBell(display,0);   /* it, then just beep and give up    */
	      return;
	    }
	  else                    /* If it completely errored out,    */
	    {                     /* then try again at half the size  */
	      write_bytes /= 2;
	    }
	}
      else if (wrote < write_bytes)
	{                         /* If wrote less than wanted to,    */
	  i   += wrote;           /* increment pointers, and change   */
	  buf += wrote;           /* the size of the writes to what   */
	  write_bytes = wrote;    /* just worked.                     */
	}
      else if (wrote == write_bytes)
	{                         /* If we wrote all we tried to,     */
	  i += wrote;           /* then just increment the pointers */
	  buf += wrote;
	}
    }
}




/*  Send printf formatted output to the command.  Only used for small ammounts
 *  of data.
 */
void cprintf(unsigned char *fmt,...)
{
  va_list args;
  static unsigned char buf[1024];
  
  va_start(args,fmt);
  

  vsprintf(buf,fmt,args);
  va_end(args);
  send_string(buf,strlen(buf));
}

/*  Return an input token
 */
void get_token()
{
  int c;
  
  while(1)
    {
      while((c = get_com_char(0)) == GCC_NULL);

     if (c >= ' ' || c == '\n' || c == '\r' || c == '\t') 
       process_string(c);
      else 
	switch(c)
	  {
	  case 0:         /* NUL does nothing */
	    break;
	  case EOF:
	    clean_exit(0);
	    break;
	  case 5:
	    cprintf("\033[?6c");	/* I am a VT102 */
	    break;
	  case 0xb:
	    scr_index(1);
	    break;
	  case 0xc:
	    scr_index(1);
	    break;
	  case ESC:
	    process_escape_sequence();
	    break;
	  case '\b' :
	    scr_backspace();
	    break;
	  case '\007' :	/* bell */
	    XBell(display,0);
	    break;
	  case '\016':
	    scr_choose_charset(1);
	    break;
	  case '\017':
	    scr_choose_charset(0);
	    break;
	  }
    }
}


void process_string(int c)
{
  int i,nlcount;
  unsigned char string[STRING_MAX];
  int nlmax = cheight -2;

  if(nlmax <2)
    nlmax = 2;

  i = 0;
  if(c == '\n')
    nlcount = 1;
  else
    nlcount = 0;
  do 
    {
      string[i++] = c;
      c = get_com_char(BUF_ONLY);
      if (c == '\n')
	{
	  if(nlcount >nlmax)
	    break;
	  else
	    nlcount++;
	}

    } while (c != GCC_NULL && (c >= ' ' || c == '\n' || c == '\r' 
			      || c == '\t') && (i < (STRING_MAX-1)));
  if (c != GCC_NULL)
    {
      push_com_char(c);
    }
  string[i]=0;
  refresh_nl_count += nlcount;
  scr_string(string,i,nlcount);
} 

  
void process_escape_sequence()
{
  int c;
  c = get_com_char(0);
  switch(c)
    {
    case '[':
      process_csi_sequence();
      break;
    case ']':
      process_xterm_sequence();
      break;
    case '(':
      scr_set_charset(0,get_com_char(0));
      break;
    case ')':
      scr_set_charset(1,get_com_char(0));
      break;
    case '7':
      scr_save_cursor();
      break;
    case '8' :
      scr_restore_cursor();
      break;
    case '=' :
      app_kp_keys = 1;
      break;
    case '>' :
      app_kp_keys = 0;
      break;
    case 'D' :
      scr_index(1);
      break;
    case 'E':
      scr_string("\n\r",2,1);
      break;
    case 'M' :
      scr_index(-1);
      break;
    case 'Z' :
      cprintf("\033[?6c");	/* I am a VT102 */
      break;
    case 'c' :                  /* restore power-on values */
      scr_power_on();
    case '#':                /* set character size */
    case 'H' :               /* horizontal tab set */
    default:
      return;
    }
} 


void process_csi_sequence()
{
  int c,n,i,nargs,private=0;
  int arg[ARGS_MAX];

  arg[0]=0;
  arg[1]=0;

  c = get_com_char(0);
  if (c >= '<' && c <= '?') 
    {
      private = c;
      c = get_com_char(0);
    }

  /*  read any numerical arguments
   */
  i = 0;
  do 
    {
      n = 0;
      while (c >= '0' && c <= '9') 
	{
	  n = n * 10 + c - '0';
	  c = get_com_char(0);
	}
      if (i < ARGS_MAX)
	arg[i++] = n;
      if (c == ESC)
	push_com_char(c);
      if (c < ' ')
	return;
      if (c < '@')
	c = get_com_char(0);
    } while (c < '@' && c >= ' ');
  if (c == ESC)
    push_com_char(c);
  if (c < ' ')
    return;
  nargs = i;

  switch (c) 
    {
    case 'A':	/* cursor up */
      scr_move(0,((arg[0] == 0) ? -1 : -arg[0]),ROW_RELATIVE | COL_RELATIVE);
      break;
    case 'B' :	/* cursor down */
      scr_move(0,((arg[0] == 0) ? 1 : arg[0]),ROW_RELATIVE | COL_RELATIVE);
      break;
    case 'C' :	/* cursor forward */
      scr_move(((arg[0] == 0) ? 1 : arg[0]),0,ROW_RELATIVE | COL_RELATIVE);
      break;
    case 'D' :	/* cursor back */
      scr_move(((arg[0] == 0) ? -1 : -arg[0]),0,ROW_RELATIVE | COL_RELATIVE);
      break;
    case 'f' :
    case 'H' :	/* position cursor */
      if(nargs==0)
	scr_move(0,0,0);
      else if (nargs == 1)
	scr_move(0,((arg[0]==0)? 0 : (arg[0]-1)),0);
      else 
	scr_move(arg[1] - 1,arg[0] - 1,0);
      break;
    case 'J' :
      scr_erase_screen(arg[0]);
      break;
    case 'K' :
      scr_erase_line(arg[0]);
      break;
    case 'L' :
      scr_insert_delete_lines( (arg[0]==0) ? 1 : arg[0],INSERT);
      break;
    case 'M' :
      scr_insert_delete_lines( (arg[0]==0) ? 1 : arg[0],DELETE);
      break;
    case 'P' :
      scr_insert_delete_characters( (arg[0]==0) ? 1 : arg[0],DELETE);
      break;
    case '@' :
      scr_insert_delete_characters( (arg[0]==0) ? 1 : arg[0],INSERT);
      break;
    case 'c' :
      cprintf("\033[?6c");	/* I am a VT102 */
      break;
    case 'h' :
    case 'l' :
      process_terminal_mode(c,private,nargs,arg);
      break;
    case 'm' :
      process_sgr_mode(c,private,nargs,arg);
      break;
    case 'n' :		/* request for information */
      switch (arg[0]) 
	{
	case 6 :
	  scr_report_position();
	  break;
	}
      break;
    case 'r' :		/* set top and bottom margins */
      /* what's this about? something to do with vi on ESIX systems */
      if (private == '?') break;
      if (nargs < 2 || arg[0] >= arg[1])
	scr_set_margins(0,10000);
      else
	scr_set_margins(arg[0] - 1,arg[1] - 1);
      break;
    case 'g' :                  /* tab clear */
      break;
    }
}



void process_xterm_sequence()
{
  int c,n,i,arg;
  unsigned char string[STRING_MAX];

  c = get_com_char(0);
  n = 0;
  while (c >= '0' && c <= '9') 
    {
      n = n * 10 + c - '0';
      c = get_com_char(0);
    }
  arg = n;

  c = get_com_char(0);
  i = 0;
  while ( c != 7 && i < STRING_MAX)
    {
      if (c >= ' ')
	string[i++] = c;
      c = get_com_char(0);
    }
  string[i] = 0;
  switch (arg) 
    {
    case 0 :
      change_window_name(string);
      change_icon_name(string);
      break;
    case 1 :
      change_icon_name(string);
      break;
    case 2 :
      change_window_name(string);
      break;
    }
}


void process_terminal_mode(int c,int private,int nargs,int *arg)
{
  int mode;

  mode = (c == 'h') ? HIGH : LOW;
  if (private == '?') 
    {
      switch (arg[0]) 
	{
	case 1 :
	  app_cur_keys = (mode == HIGH);
	  break;
	case 3:                 /* 132 columns mode is N/A */
	  break;
	case 4:                 /* Smooth scrolling is N/A */
	  break;
	case 6 :
	  scr_set_decom(mode);
	  break;
	case 7 :
	  scr_set_wrap(mode);
	  break;
	case 8:                 /* auto repeat is N/A */
	  break;
	case 47 :		/* switch to main screen */
	  scr_change_screen(mode);
	  break;
	}
    }
  else if (private == 0) 
    {
      if(arg[0]==4)
	scr_set_insert(mode);
    }
}


void process_sgr_mode(int c,int private,int nargs,int *arg)
{
  int i;

  if (nargs == 0)
    scr_change_rendition(1,~RS_NONE);
  else
    for (i = 0; i < nargs; i++) 
      switch (arg[i]) 
	{
	case 0 :
	  scr_change_rendition(1,~RS_NONE);
	  break;
	case 1 :
	  scr_change_rendition(0,RS_BOLD);
	  break;
	case 4 :
	  scr_change_rendition(0,RS_ULINE);
	  break;
	case 5 :
	  scr_change_rendition(0,RS_BLINK);
	  break;
	case 7 :
	  scr_change_rendition(0,RS_RVID);
	  break;
	case 22 :
	  scr_change_rendition(1,RS_BOLD);
	  break;
	case 24 :
	  scr_change_rendition(1,RS_ULINE);
	  break;
	case 25 :
	  scr_change_rendition(1,RS_BLINK);
	  break;
	case 27 :
	  scr_change_rendition(1,RS_RVID);
	  break;
    }
}

#ifdef NO_SETSID
setsid()
{
  if (setpgrp(0,0) < 0) return(-1);
  return(0);
}
#endif

