/*
 *  texrep.C from ObjectProDSP 0.1
 *  Copyright (C) 1994, Mountain Math Software. All rights reserved.
 *  
 *  This file is part of ObjectProDSP, a tool for Digital Signal
 *  Processing design, development and implementation. It is free
 *  software provided you use and distribute it under the terms of
 *  version 2 of the GNU General Public License as published
 *  by the Free Software Foundation. You may NOT distribute it or
 *  works derived from it or code that it generates under ANY
 *  OTHER terms.  In particular NONE of the ObjectProDSP system is
 *  licensed for use under the GNU General Public LIBRARY License.
 *  Mountain Math Software plans to offer a commercial version of
 *  ObjectProDSP for a fee. That version will allow redistribution
 *  of generated code under standard commercial terms.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of version 2 of the GNU General
 *  Public License along with this program. See file COPYING. If not
 *  or if you wish information on commercial versions and licensing
 *  write Mountain Math Software, P. O. Box 2124, Saratoga, CA 95070,
 *  USA, or send us e-mail at: support@mtnmath.com.
 *  
 *  You may also obtain the GNU General Public License by writing the
 *  Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 *  USA.  However if you received a copy of this program without the
 *  file COPYING or without a copyright notice attached to all text
 *  files, libraries and executables please inform Mountain Math Software.
 *  
 *  ObjectProDSP is a trademark of Mountain Math Software.
 */
#include <iostream.h>
#include <ctype.h>
#include <string.h>
#include "mkstr.h"

struct TeXMacro {
	const char * Text ;
	const char * Macro ;
};

class ReplaceWithTeX {
	TeXMacro * TheMacros ;
	const char * Special ;

	int IsSpecial(char C);
	int IsSeparator(char C) ;

	const char * GetQuoteReplace(const char * Start, const char * End) ;
	const char * CheckForQuoted(const char * Start) ;
	TeXMacro * CheckForMacro(const char * Start) ;
	TeXMacro * FindReplacement(const char * Word);
public:
	ReplaceWithTeX(TeXMacro * the_macros, const char * spec);
	char * ReplaceMacros(const char * Orig);
	char * ReplaceQuoted(const char * Orig);
	char * Replace(const char * Orig);
};

static TeXMacro TheReplacements[] = {
	{"DSP++", "\\dpp{}"},
	{"ObjectPro++", "\\IntPp{}"},
	{"ObjectProDSP", "\\Dpp{}"},
	{"C++", "\\Cpp{}"},
	{0}
};

static const char * TheSpecialCharacters = "+" ;

ReplaceWithTeX TheReplacer(TheReplacements,TheSpecialCharacters) ;

ReplaceWithTeX::ReplaceWithTeX(TeXMacro * the_macros, const char * spec):
	TheMacros(the_macros),
	Special(spec)
{
}

int ReplaceWithTeX::IsSeparator(char C)
{
	if (!C) return 1 ;
	if (isalnum(C)) return 0 ;
	if (IsSpecial(C)) return 0;
	return 1 ;
}

int ReplaceWithTeX::IsSpecial(char C)
{
	for (const char * Spec = Special; *Spec; Spec++)
		if (C == *Spec) return 1 ;
	return 0 ;
}

TeXMacro * ReplaceWithTeX::FindReplacement(const char * Word)
{
	for (TeXMacro * Ck = TheMacros; Ck->Text; Ck++) 
		if (!strcmp(Ck->Text,Word)) return Ck ;
	return 0 ;
}

static int tex_quote_alnum(char c)
{
	if (c == '\'') return 0 ;
	if (c == ' ') return 1 ;
	return isgraph(c);

}

const char * ReplaceWithTeX::CheckForQuoted(const char * Start)
{
	if (*Start++ != '`') return 0 ;
	for (const char * Pt = Start; *Pt; Pt++) if (!tex_quote_alnum(*Pt)) break ;
	if (*Pt != '\'') return 0 ;
	return Pt ;
}

const char * ReplaceWithTeX::GetQuoteReplace(const char * Start,
	const char * End)
{
	const MaxReplaceString= 64 ;
	static char Buf[MaxReplaceString+1] ;
	const char * Prefix = "\\DppNm{" ;
	const char * Suffix = "}" ;
	int ReplaceLength = End - Start - 1 ;

	int PrefixLength = strlen(Prefix);
	int SuffixLength = strlen(Suffix);

	if (ReplaceLength < 1) {
		Buf[0] = '\0' ;
		return Buf ;
	}
	if (ReplaceLength + PrefixLength + SuffixLength + 1 >=
		MaxReplaceString) return 0 ;
	strcpy(Buf,Prefix);
	strncpy(Buf+PrefixLength,Start+1,ReplaceLength);
	strcpy(Buf+PrefixLength+ReplaceLength,Suffix);
	return Buf ;
}

TeXMacro * ReplaceWithTeX::CheckForMacro(const char * Start)
{
	const MaxWordSize = 32 ;
	char Word[MaxWordSize + 1] ;
	for (const char * Pt = Start; *Pt; Pt++) if (IsSeparator(*Pt)) break;
	int Length = Pt - Start ;
	if (Length < 1) return 0 ;
	if (Length >= MaxWordSize) return 0 ;
	strncpy(Word,Start,Length);
	Word[Length] = '\0' ;
	return FindReplacement(Word);
}

char * ReplaceWithTeX::Replace(const char * Orig)
{
	char * Temp = ReplaceMacros(Orig);
	char * Temp2 = ReplaceQuoted(Temp);
	delete Temp ;
	return Temp2 ;
}

char * ReplaceWithTeX::ReplaceQuoted(const char * Orig)
{
	const MaxLine = 1024 ;
	char Temp[MaxLine] ;
	int PreviousSpace = 1 ;
	int index = 0 ;
	for (const char * Pt = Orig ; *Pt ; Pt++) {
		if (isspace(*Pt) || *Pt == '(') PreviousSpace = 1 ;
		else {
			const char * Replace = 0 ;
			if (PreviousSpace) Replace = CheckForQuoted(Pt) ;
			PreviousSpace = 0 ;
			if (Replace) {
				const char * Sub = GetQuoteReplace(Pt,Replace);
				if (Sub) {
					int Length = strlen(Sub);
					if (Length + index > MaxLine - 2)
						return Concatenate(Orig);
					Pt = Replace ;
					strcpy(Temp+index,Sub);
					index+= Length ;
					continue ;
				}
			}
		}
		if (index > MaxLine -2) return Concatenate(Orig);
		Temp[index++]= *Pt ;
	}
	Temp[index] = '\0' ;
	return Concatenate(Temp);
}

char * ReplaceWithTeX::ReplaceMacros(const char * Orig)
{
	const MaxLine = 1024 ;
	char Temp[MaxLine] ;
	int PreviousSpace = 1 ;
	int index = 0 ;
	int DidReplace = 0 ;
	for (const char * Pt = Orig ; *Pt ; Pt++) {
		if (isspace(*Pt)) PreviousSpace = 1 ;
		else {
			TeXMacro * Replace = 0 ;
/*
 *			if (PreviousSpace)
 *				cerr << "Checking: `" <<  Pt << "'.\n" ;
 */
			if (PreviousSpace) Replace = CheckForMacro(Pt) ;
			PreviousSpace = 0 ;
			if (Replace) {
				int Length = strlen(Replace->Macro) ;
				if (index + Length > MaxLine - 2) return
					Concatenate(Orig) ;
				strcpy(Temp+index,Replace->Macro);
				index += Length ;
				Pt += strlen(Replace->Text) - 1;
				DidReplace = 1 ;
				continue ;
			}
		}
		if (index > MaxLine -2) return Concatenate(Orig) ;
		Temp[index++]= *Pt ;
	}
	Temp[index] = '\0' ;
	if (!DidReplace) return Concatenate(Orig) ;
	// cerr << "Returning `" << Temp << "'.\n" ;
	return Concatenate(Temp);
}
 

char * TeX(const char * Text)
{
	return TheReplacer.Replace(Text);
}
