/*
 *  (C) 2004  Dominik Brodowski <linux@dominikbrodowski.de>
 *
 *  Licensed under the terms of the GNU GPL License version 2.
 */


#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#define _GNU_SOURCE
#include <getopt.h>

#include <linux/threads.h> /* for NR_CPUS */

#include "cpufreq.h"
#include "config.h"

static void print_header(void) {
        printf(PACKAGE_STRING ": cpufreq-set (C) Dominik Brodowski 2004\n");
	printf("Report errors and bugs to "PACKAGE_BUGREPORT", please\n");
}

static void print_help(void) {
	printf("Usage: cpufreq-set [options]\n");
	printf("Options:\n");
	printf("  -c CPU, --cpu CPU        number of CPU where cpufreq settings shall be modified\n");
	printf("  -d FREQ, --min FREQ      new minimum CPU frequency the governor may select\n");
	printf("  -u FREQ, --max FREQ      new maximum CPU frequency the governor may select\n");
	printf("  -g GOV, --governor GOV   new cpufreq governor\n");
	printf("  -f FREQ, --freq FREQ     specific frequency to be set. Requires userspace\n"
	       "                           governor to be available and loaded\n");
	printf("  -h, --help           Prints out this screen\n");
	printf("\n");
	printf("Notes:\n");
	printf("1. Omitting the -c or --cpu argument is equivalent to setting it to zero\n");
	printf("2. The -f FREQ, --freq FREQ parameter cannot be combined with any other parameter\n");
	printf("   except the -c CPU, --cpu CPU parameter\n");
	printf("3. FREQuencies must be passed in kHz = 1000 MHz = 1000000 GHz.\n");

}

static struct option set_opts[] = {
	{ .name="cpu",		.has_arg=required_argument,	.flag=NULL,	.val='c'},
	{ .name="min",		.has_arg=required_argument,	.flag=NULL,	.val='d'},
	{ .name="max",		.has_arg=required_argument,	.flag=NULL,	.val='u'},
	{ .name="governor",	.has_arg=required_argument,	.flag=NULL,	.val='g'},
	{ .name="freq",		.has_arg=required_argument,	.flag=NULL,	.val='f'},
	{ .name="help",		.has_arg=no_argument,		.flag=NULL,	.val='h'},
};

static void print_unknown_arg(void) {
	print_header();
	printf("unknown or invalid parameter\n");
	print_help();
}

int main(int argc, char **argv) {
	extern char *optarg;
	extern int optind, opterr, optopt;
	int ret = 0, cont = 1;
	unsigned int cpu = 0;
	unsigned long min;
	unsigned long max;
	unsigned long freq;
	char gov[20];
	int freq_is_set = 0;
	int min_is_set = 0;
	int max_is_set = 0;
	int gov_is_set = 0;
	int cpu_is_set = 0;
	int double_parm = 0;

	do {
		ret = getopt_long(argc, argv, "c:d:u:g:f:h", set_opts, NULL);
		switch (ret) {
		case '?':
			print_unknown_arg();
			return -EINVAL;
		case 'h':
			print_header();
			print_help();
			return 0;
		case -1:
			cont = 0;
			break;
		case 'c':
			if (cpu_is_set)
				double_parm++;
			cpu_is_set++;
			if ((sscanf(optarg, "%d ", &cpu)) != 1) {
				print_unknown_arg();
				return -EINVAL;
                        }
			break;
		case 'd':
			if (min_is_set)
				double_parm++;
			min_is_set++;
			if ((sscanf(optarg, "%lu ", &min)) != 1) {
				print_unknown_arg();
				return -EINVAL;
                        }
			break;
		case 'u':
			if (max_is_set)
				double_parm++;
			max_is_set++;
			if ((sscanf(optarg, "%lu ", &max)) != 1) {
				print_unknown_arg();
				return -EINVAL;
                        }
			break;
		case 'f':
			if (freq_is_set)
				double_parm++;
			freq_is_set++;
			if ((sscanf(optarg, "%lu ", &freq)) != 1) {
				print_unknown_arg();
				return -EINVAL;
                        }
			break;
		case 'g':
			if (gov_is_set)
				double_parm++;
			gov_is_set++;
			if ((strlen(optarg) < 3) || (strlen(optarg) > 18)) {
				print_unknown_arg();
				return -EINVAL;
                        }
			if ((sscanf(optarg, "%s", gov)) != 1) {
				print_unknown_arg();
				return -EINVAL;
                        }
			break;
		}
	} while(cont);

	if (double_parm) {
		print_header();
		printf("the same parameter was passed more than once\n");
		return -EINVAL;
	}

	if (freq_is_set) {
		if ((min_is_set) || (max_is_set) || (gov_is_set)) {
			printf("the -f/--freq parameter cannot be combined with -d/--min, -u/--max or\n");
			printf("-g/--governor parameters\n");
			return -EINVAL;
		}
		ret = cpufreq_set_frequency(cpu, freq);
		goto out;
	}

	ret = min_is_set + max_is_set + gov_is_set;

	if (!ret) {
			printf("At least one parameter out of -f/--freq, -d/--min, -u/--max, and\n");
			printf("-g/--governor must be passed\n");
			return -EINVAL;
	}

	if (ret == 1) {
		if (min_is_set)
			ret = cpufreq_modify_policy_min(cpu, min);
		else if (max_is_set)
			ret = cpufreq_modify_policy_max(cpu, max);
		else if (gov_is_set)
			ret = cpufreq_modify_policy_governor(cpu, gov);
		if (!ret)
			return 0;
	}

	{
		struct cpufreq_policy *cur_pol = cpufreq_get_policy(cpu);
		struct cpufreq_policy new_pol;
		if (!cur_pol) {
			printf("wrong, unknown or unhandled CPU?");
			return -EINVAL;
		}

		if (min_is_set)
			new_pol.min = min;
		else
			new_pol.min = cur_pol->min;

		if (max_is_set)
			new_pol.max = max;
		else
			new_pol.max = cur_pol->max;

		new_pol.governor = gov;
		if (!gov_is_set)
			strncpy(gov, cur_pol->governor, 20);

		cpufreq_put_policy(cur_pol);

		ret = cpufreq_set_policy(cpu, &new_pol);
	}
 out:
	if (ret) {
		printf("Error setting new values. Common errors:\n");
		printf("- Do you have proper administration rights? (super-user?)\n");
		printf("- Is the governor you requested available and modprobed?\n");
		printf("- Trying to set an invalid policy?\n");
		printf("- Trying to set a specific frequency, but userspace governor is not availble,\n");
		printf("    for example on hardware, where setting from userspace is impossible, or\n");
		printf("    simply not loaded?\n");
	}
	return (ret);
}
