;; Mouse handling for NeXT "Emacs" front end.
;; Ripped off of sun-mouse.el
;; Copyright (C) 1987 Free Software Foundation, Inc.

;; This file is NOT part of the standard GNU Emacs distribution.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

;;; converted from sun-mouse.el by John G. Myers, Feb 1990.
;;; sun-mouse by Jeff Peck, Sun Microsystems, Jan 1987.
;;; Original idea by Stan Jefferson

;;(provide 'eterm-mouse)

;;;
;;;     Modelled after the GNUEMACS keymap interface.
;;;
;;; User Functions:
;;;   make-mousemap, copy-mousemap, 
;;;   define-mouse, global-set-mouse, local-set-mouse,
;;;   use-global-mousemap, use-local-mousemap,
;;;   mouse-lookup, describe-mouse-bindings
;;;
;;; Options:
;;;   scrollbar-width
;;;

(defvar scrollbar-width 5
  "*The character width of the scrollbar.
The cursor is deemed to be in the right edge scrollbar if it is this near the
right edge, and more than two chars past the end of the indicated line.
Setting to nil limits the scrollbar to the edge or vertical dividing bar.")

;;;
;;; Mousemaps
;;;
(defun make-mousemap ()
  "Returns a new mousemap."
  (cons 'mousemap nil))

(defun copy-mousemap (mousemap)
  "Return a copy of mousemap."
  (copy-alist mousemap))

(defun define-mouse (mousemap mouse-list def)
  "Args MOUSEMAP, MOUSE-LIST, DEF.  Define MOUSE-LIST in MOUSEMAP as DEF.
MOUSE-LIST is a list of atoms specifing a mouse hit according to these rules:
  * One of these atoms specifies the active region of the definition.
	text, scrollbar, modeline, minibuffer
  * One or two or these atoms specify the button or button combination.
        left, middle, right, double
  * Any combination of these atoms specify the active shift keys.
        control, shift, meta
  * With a single unshifted button, you can add
	up
    to indicate an up-click.
The atom `double' is used with a button designator to denote a double click.
Two button chords are denoted by listing the two buttons.
See eterm-mouse-handler for the treatment of the form DEF."
  (mousemap-set (mouse-list-to-mouse-code mouse-list) mousemap def))

(defun global-set-mouse (mouse-list def)
  "Give MOUSE-EVENT-LIST a local definition of DEF.
See define-mouse for a description of MOUSE-EVENT-LIST and DEF.
Note that if MOUSE-EVENT-LIST has a local definition in the current buffer,
that local definition will continue to shadow any global definition."
  (interactive "xMouse event: \nxDefinition: ")
  (define-mouse current-global-mousemap mouse-list def))

(defun local-set-mouse (mouse-list def)
  "Give MOUSE-EVENT-LIST a local definition of DEF.
See define-mouse for a description of the arguments.
The definition goes in the current buffer's local mousemap.
Normally buffers in the same major mode share a local mousemap."
  (interactive "xMouse event: \nxDefinition: ")
  (if (null current-local-mousemap)
      (setq current-local-mousemap (make-mousemap)))
  (define-mouse current-local-mousemap mouse-list def))

(defun use-global-mousemap (mousemap)
  "Selects MOUSEMAP as the global mousemap."
  (setq current-global-mousemap mousemap))

(defun use-local-mousemap (mousemap)
  "Selects MOUSEMAP as the local mousemap.
nil for MOUSEMAP means no local mousemap."
  (setq current-local-mousemap mousemap))


;;;
;;; Interface to the Mouse encoding defined in Emacstool.c
;;;
;;; Called when mouse-prefix is sent to emacs, additional
;;; information is read in as a list (button x y time-delta)
;;;
;;; First, some generally useful functions:
;;;

(defun logtest (x y)
  "True if any bits set in X are also set in Y.
Just like the Common Lisp function of the same name."
  (not (zerop (logand x y))))


;;;
;;; Hit accessors.
;;;

(defconst em::ButtonBits 7)		; Lowest 3 bits.
(defconst em::ShiftmaskBits 56)		; Second lowest 3 bits (56 = 63 - 7).
(defconst em::DoubleBits 64)		; Bit 7.
(defconst em::UpBits 128)		; Bit 8.

;;; All the useful code bits
(defmacro em::hit-code (hit)
  (` (nth 0 (, hit))))
;;; The button, or buttons if a chord.
(defmacro em::hit-button (hit)
  (` (logand em::ButtonBits (nth 0 (, hit)))))
;;; The shift, control, and meta flags.
(defmacro em::hit-shiftmask (hit)
  (` (logand em::ShiftmaskBits (nth 0 (, hit)))))
;;; Set if a double click (but not a chord).
(defmacro em::hit-double (hit)
  (` (logand em::DoubleBits (nth 0 (, hit)))))
;;; Set on button release (as opposed to button press).
(defmacro em::hit-up (hit)
  (` (logand em::UpBits (nth 0 (, hit)))))
;;; Screen x position.
(defmacro em::hit-x (hit) (list 'nth 1 hit))
;;; Screen y position.
(defmacro em::hit-y (hit) (list 'nth 2 hit))
;;; Millisconds since last hit.
(defmacro em::hit-delta (hit) (list 'nth 3 hit))

(defmacro em::hit-up-p (hit)		; A predicate.
  (` (not (zerop (em::hit-up (, hit))))))

;;;
;;; Loc accessors.  for em::window-xy
;;;
(defmacro em::loc-w (loc) (list 'nth 0 loc))
(defmacro em::loc-x (loc) (list 'nth 1 loc))
(defmacro em::loc-y (loc) (list 'nth 2 loc))

(defmacro eval-in-buffer (buffer &rest forms)
  "Macro to switches to BUFFER, evaluates FORMS, returns to original buffer."
  ;; When you don't need the complete window context of eval-in-window
  (` (let ((StartBuffer (current-buffer)))
    (unwind-protect
	(progn
	  (set-buffer (, buffer))
	  (,@ forms))
    (set-buffer StartBuffer)))))

(put 'eval-in-buffer 'lisp-indent-hook 1)

;;; this is used extensively by sun-fns.el
;;;
(defmacro eval-in-window (window &rest forms)
  "Switch to WINDOW, evaluate FORMS, return to original window."
  (` (let ((OriginallySelectedWindow (selected-window)))
       (unwind-protect
	   (progn
	     (select-window (, window))
	     (,@ forms))
	 (select-window OriginallySelectedWindow)))))
(put 'eval-in-window 'lisp-indent-hook 1)

;;;
;;; handy utility, generalizes window_loop
;;;

;;; It's a macro (and does not evaluate its arguments).
(defmacro eval-in-windows (form &optional yesmini)
  "Switches to each window and evaluates FORM.  Optional argument
YESMINI says to include the minibuffer as a window.
This is a macro, and does not evaluate its arguments."
  (` (let ((OriginallySelectedWindow (selected-window)))
       (unwind-protect 
	   (while (progn
		    (, form)
		    (not (eq OriginallySelectedWindow
			     (select-window
			      (next-window nil (, yesmini)))))))
	 (select-window OriginallySelectedWindow)))))
(put 'eval-in-window 'lisp-indent-hook 0)

(defun move-to-loc (x y)
  "Move cursor to window location X, Y.
Handles wrapped and horizontally scrolled lines correctly."
  (move-to-window-line y)
  ;; window-line-end expects this to return the window column it moved to.
  (let ((cc (current-column))
	(nc (move-to-column
	     (if (zerop (window-hscroll))
		 (+ (current-column)
		    (min (- (window-width) 2)	; To stay on the line.
			 x))
	       (+ (window-hscroll) -1
		  (min (1- (window-width))	; To stay on the line.
		       x))))))
    (- nc cc)))


(defun minibuffer-window-p (window)
  "True iff this WINDOW is minibuffer."
  (= (screen-height)
     (nth 3 (window-edges window))	; The bottom edge.
     ))


(defun eterm-mouse-handler (&optional hit)
  "Evaluates the function or list associated with a mouse hit.
Expecting to read a hit, which is a list: (button x y unused).  
A form bound to button by define-mouse is found by mouse-lookup. 
The variables: *mouse-window*, *mouse-x*, *mouse-y* are bound.  
If the form is a symbol (symbolp), it is funcall'ed with *mouse-window*,
*mouse-x*, and *mouse-y* as arguments; if the form is a list (listp),
the form is eval'ed; if the form is neither of these, it is an error.
Returns nil."
  (interactive)
  (if (null hit) (setq hit (mouse-hit-read)))
  (let ((loc (em::window-xy (em::hit-x hit) (em::hit-y hit))))
    (let ((*mouse-window* (em::loc-w loc))
	  (*mouse-x* (em::loc-x loc))
	  (*mouse-y* (em::loc-y loc))
	  (mouse-code (mouse-event-code hit loc)))
      (let ((form (eval-in-buffer (window-buffer *mouse-window*)
		    (mouse-lookup mouse-code))))
	(cond ((null form)
	       (if (not (em::hit-up-p hit))	; undefined up hits are ok.
		   (error "Undefined mouse event: %s" 
			  (prin1-to-string 
			   (mouse-code-to-mouse-list mouse-code)))))
	      ((symbolp form)
	       (setq this-command form)
	       (funcall form *mouse-window* *mouse-x* *mouse-y*))
	      ((listp form)
	       (setq this-command (car form))
	       (eval form))
	      (t
	       (error "Mouse action must be symbol or list, but was: %s"
		      form))))))
  ;; Don't let 'eterm-mouse-handler get on last-command,
  ;; since this function should be transparent.
  (if (eq this-command 'eterm-mouse-handler)
      (setq this-command last-command))
  ;; (message (prin1-to-string this-command))	; to see what your buttons did
  nil)

(defun mouse-hit-read ()
  "Read mouse-hit list from keyboard.  Like (read 'read-char),
but that uses minibuffer, and mucks up last-command."
  (let ((char-list nil) (char nil))
    (while (not (equal 13		; Carriage return.
		       (prog1 (setq char (read-char)) 
			 (setq char-list (cons char char-list))))))
    (read (mapconcat 'char-to-string (nreverse char-list) ""))
    ))


(defun em::window-xy (x y)
  "Find window containing screen coordinates X and Y.
Returns list (window x y) where x and y are relative to window."
  (or
   (catch 'found
     (eval-in-windows 
      (let ((we (window-edges (selected-window))))
	(let ((le (nth 0 we))
	      (te (nth 1 we))
	      (re (nth 2 we))
	      (be (nth 3 we)))
	  (if (= re (screen-width))
	      ;; include the continuation column with this window
	      (setq re (1+ re)))
	  (if (= be (screen-height))
	      ;; include partial line at bottom of screen with this window
	      ;; id est, if window is not multple of char size.
	      (setq be (1+ be)))

	  (if (and (>= x le) (< x re)
		   (>= y te) (< y be))
	      (throw 'found 
		     (list (selected-window) (- x le) (- y te))))))
      t))				; include minibuffer in eval-in-windows
   ;;If x,y from a real mouse click, we shouldn't get here.
   (list nil x y)
   ))

(defun em::window-region (loc)
  "Parse LOC into a region symbol.
Returns one of (text scrollbar modeline minibuffer)"
  (let ((w (em::loc-w loc))
	(x (em::loc-x loc))
	(y (em::loc-y loc)))
    (let ((right (1- (window-width w)))
	  (bottom (1- (window-height w))))
      (cond ((minibuffer-window-p w) 'minibuffer)
	    ((>= y bottom) 'modeline)
	    ((>= x right) 'scrollbar)
	    ;; far right column (window seperator) is always a scrollbar
	    ((and scrollbar-width
		  ;; mouse within scrollbar-width of edge.
		  (>= x (- right scrollbar-width))
		  ;; mouse a few chars past the end of line.
		  (>= x (+ 2 (window-line-end w x y))))
	     'scrollbar)
	    (t 'text)))))

(defun window-line-end (w x y)
  "Return WINDOW column (ignore X) containing end of line Y"
  (eval-in-window w (save-excursion (move-to-loc (screen-width) y))))

;;;
;;; The encoding of mouse events into a mousemap.
;;; These values must agree with coding in emacstool:
;;;
(defconst em::keyword-alist 
  '((left . 1) (middle . 2) (right . 4)
    (shift . 8) (control . 16) (meta . 32) (double . 64) (up . 128)
    (text . 256) (scrollbar . 512) (modeline . 1024) (minibuffer . 2048)
    ))

(defun mouse-event-code (hit loc)
  "Maps MOUSE-HIT and LOC into a mouse-code."
;;;Region is a code for one of text, modeline, scrollbar, or minibuffer.
  (logior (em::hit-code hit)
	  (mouse-region-to-code (em::window-region loc))))

(defun mouse-region-to-code (region)
  "Returns partial mouse-code for specified REGION."
  (cdr (assq region em::keyword-alist)))

(defun mouse-list-to-mouse-code (mouse-list)
  "Map a MOUSE-LIST to a mouse-code."
  (apply 'logior
	 (mapcar (function (lambda (x)
			     (cdr (assq x em::keyword-alist))))
		  mouse-list)))

(defun mouse-code-to-mouse-list (mouse-code)
  "Map a MOUSE-CODE to a mouse-list."
  (apply 'nconc (mapcar
		 (function (lambda (x)
			     (if (logtest mouse-code (cdr x))
				 (list (car x)))))
		 em::keyword-alist)))

(defun mousemap-set (code mousemap value)
  (let* ((alist (cdr mousemap))
	 (assq-result (assq code alist)))
    (if assq-result
	(setcdr assq-result value)
      (setcdr mousemap (cons (cons code value) alist)))))

(defun mousemap-get (code mousemap)
  (cdr (assq code (cdr mousemap))))

(defun mouse-lookup (mouse-code)
  "Look up MOUSE-EVENT and return the definition. nil means undefined."
  (or (mousemap-get mouse-code current-local-mousemap)
      (mousemap-get mouse-code current-global-mousemap)))

;;;
;;; I (jpeck) don't understand the utility of the next four functions
;;; ask Steven Greenbaum <froud@kestrel>
;;;
(defun mouse-mask-lookup (mask list)
  "Args MASK (a bit mask) and LIST (a list of (code . form) pairs).
Returns a list of elements of LIST whose code or'ed with MASK is non-zero."
  (let ((result nil))
    (while list
      (if (logtest mask (car (car list)))
	  (setq result (cons (car list) result)))
      (setq list (cdr list)))
    result))

(defun mouse-union (l l-unique)
  "Return the union of list of mouse (code . form) pairs L and L-UNIQUE,
where L-UNIQUE is considered to be union'ized already."
  (let ((result l-unique))
    (while l
      (let ((code-form-pair (car l)))
	(if (not (assq (car code-form-pair) result))
	    (setq result (cons code-form-pair result))))
      (setq l (cdr l)))
    result))

(defun mouse-union-first-prefered (l1 l2)
  "Return the union of lists of mouse (code . form) pairs L1 and L2,
based on the code's, with preference going to elements in L1."
  (mouse-union l2 (mouse-union l1 nil)))

(defun mouse-code-function-pairs-of-region (region)
  "Return a list of (code . function) pairs, where each code is
currently set in the REGION."
  (let ((mask (mouse-region-to-code region)))
    (mouse-union-first-prefered
     (mouse-mask-lookup mask (cdr current-local-mousemap))
     (mouse-mask-lookup mask (cdr current-global-mousemap))
     )))

;;;
;;; Functions for DESCRIBE-MOUSE-BINDINGS
;;; And other mouse documentation functions
;;; Still need a good procedure to print out a help sheet in readable format.
;;;

(defun one-line-doc-string (function)
  "Returns first line of documentation string for FUNCTION.
If there is no documentation string, then the string
\"No documentation\" is returned."
  (while (consp function) (setq function (car function)))
  (let ((doc (documentation function)))
    (if (null doc)
	"No documentation."
      (string-match "^.*$" doc)
      (substring doc 0 (match-end 0)))))

(defun print-mouse-format (binding)
  (princ (car binding))
  (princ ": ")
  (mapcar (function
	   (lambda (mouse-list)
	     (princ mouse-list)
	     (princ " ")))
	  (cdr binding))
  (terpri)
  (princ "  ")
  (princ (one-line-doc-string (car binding)))
  (terpri)
  )

(defun print-mouse-bindings (region)
  "Prints mouse-event bindings for REGION."
  (mapcar 'print-mouse-format (em::event-bindings region)))

(defun em::event-bindings (region)
  "Returns an alist of (function . (mouse-list1 ... mouse-listN)) for REGION,
where each mouse-list is bound to the function in REGION."
  (let ((mouse-bindings (mouse-code-function-pairs-of-region region))
	(result nil))
    (while mouse-bindings
      (let* ((code-function-pair (car mouse-bindings))
	     (current-entry (assoc (cdr code-function-pair) result)))
	(if current-entry
	    (setcdr current-entry
		    (cons (mouse-code-to-mouse-list (car code-function-pair))
			  (cdr current-entry)))
	  (setq result (cons (cons (cdr code-function-pair)
				   (list (mouse-code-to-mouse-list
					  (car code-function-pair))))
			     result))))
      (setq mouse-bindings (cdr mouse-bindings))
      )
    result))

(defun describe-mouse-bindings ()
  "Lists all current mouse-event bindings."
  (interactive)
  (with-output-to-temp-buffer "*Help*"
    (princ "Text Region") (terpri)
    (princ "---- ------") (terpri)
    (print-mouse-bindings 'text) (terpri)
    (princ "Modeline Region") (terpri)
    (princ "-------- ------") (terpri)
    (print-mouse-bindings 'modeline) (terpri)
    (princ "Scrollbar Region") (terpri)
    (princ "--------- ------") (terpri)
    (print-mouse-bindings 'scrollbar)))

(defun describe-mouse-briefly (mouse-list)
  "Print a short description of the function bound to MOUSE-LIST."
  (interactive "xDescibe mouse list briefly: ")
  (let ((function (mouse-lookup (mouse-list-to-mouse-code mouse-list))))
    (if function
	(message "%s runs the command %s" mouse-list function)
      (message "%s is undefined" mouse-list))))


;;;
;;; initialize mouse maps
;;;

(make-variable-buffer-local 'current-local-mousemap)
(setq-default current-local-mousemap nil)
(defvar current-global-mousemap (make-mousemap))
