/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Pat Hanrahan, Stuart Levy, Tamara Munzner, Mark Phillips */

#include <math.h>
#include "radians.h"
#include "transform3.h"

/* 
 * Pre-multiply a matrix by a rotate about the x axis.
 *
 *       (  1  0  0  0 )      
 *       (  0  c  s  0 )      
 * [a] = (  0 -s  c  0 )  * [a]
 *       (  0  0  0  1 )      
 */
void
Ctm3RotateX( T, angle )
    Transform3 T;
    float angle;
{
    register int i;
    double t,s,c;

    s=sin(angle);
    c=cos(angle);
    for (i=0; i<4; ++i) {
      t = T[1][i] * c + T[2][i] * s;
      T[2][i] = T[2][i] * c - T[1][i] * s;
      T[1][i] = t;
    }
}

/* 
 * Pre-multiply a matrix by a rotate about the y axis.
 *
 *       (  c  0  s  0 )      
 *       (  0  1  0  0 )      
 * [a] = ( -s  0  c  0 ) * [a]
 *       (  0  0  0  1 )      
 */
void
Ctm3RotateY( T, angle )
    Transform3 T;
    float angle;
{
    register int i;
    double t,s,c;

    s=sin(angle);
    c=cos(angle);
    for (i=0; i<4; ++i) {
      t = T[2][i] * c - T[0][i] * s;
      T[0][i] = T[0][i] * c + T[2][i] * s;
      T[2][i] = t;
    }
}

/* 
 * Pre-multiply a matrix by a rotate about the z axis.
 *
 *       (  c  s  0  0 )      
 *       ( -s  c  0  0 )      
 * [a] = (  0  0  1  0 ) * [a]
 *       (  0  0  0  1 )      
 */
void
Ctm3RotateZ( T, angle )
    Transform3 T;
    float angle;
{
    register int i;
    double t,s,c;

    s=sin(angle);
    c=cos(angle);
    for (i=0; i<4; ++i) {
      t = T[0][i] * c + T[1][i] * s;
      T[1][i] = T[1][i] * c - T[0][i] * s;
      T[0][i] = t;
    }
}

/* 
 * Pre-multiply a matrix by a rotate about an arbitrary axis
 *
 * [a] = [ rotation] * [a]
 *             
 */
void
Ctm3Rotate( T, angle, axis )
    Transform3 T;
    float angle;
    HPoint3 *axis;
{
    if(      axis == (HPoint3 *)&TM3_XAXIS ) Ctm3RotateX( T, angle );
    else if( axis == (HPoint3 *)&TM3_YAXIS ) Ctm3RotateY( T, angle );
    else if( axis == (HPoint3 *)&TM3_ZAXIS ) Ctm3RotateZ( T, angle );
    else {
	Transform3 Ta;

	Tm3Rotate( Ta, angle, axis );
	Tm3Concat(Ta, T, T);
    }
}
