/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include <ctype.h>
#include "quadP.h"

static int
getquads( file, pquad, off, binary, dimn )
    FILE	*file;
    register Quad *pquad;
    int		off;
    int		binary;
    int 	dimn;
{
    HPoint3 *p;
    Point3 *n;
    ColorA *c;
    register int k;

    p = &pquad->p[off][0];
    n = (pquad->flag & QUAD_N) ? &pquad->n[off][0] : NULL;
    c = (pquad->flag & QUAD_C) ? &pquad->c[off][0] : NULL;
    for(k = 4 * (pquad->maxquad - off); --k >= 0; ) {
	if(fgetnf(file, dimn, (float *)p, binary) < dimn)
	    break;
	/* set the w-coordinate if the points are 3 dim'nal */
	if (dimn == 3) p->w = 1.0;
	p++;
	if(n != NULL) {
	    if(fgetnf(file, 3, (float *)n, binary) < 3)
		return -1;
	    n++;
	}
	if(c != NULL) {
	    if(fgetnf(file, 4, (float *)c, binary) < 4)
		return -1;
	    c++;
	}
    }
    k++;
    if(k % 4 != 0)
	return(-1);
    return(pquad->maxquad - k/4);
}


Quad *
QuadFLoad( FILE *file, char *fname )
{
    Quad quad;
    int binary = 0;
    register int ch;
    int ngot;
    int dimn = 3;

    quad.flag = 0;
    quad.p = NULL;
    quad.n = NULL;
    quad.c = NULL;

    (void) fnextc(file, 0);	/* Skip white space and comments */

    /* Parse [C][N][4]{QUAD|POLY}[ BINARY]\n */

    if((ch = getc(file)) == 'C') {
	quad.flag = QUAD_C;
	ch = getc(file);
    }
    if(ch == 'N') {
	quad.flag |= QUAD_N;
	ch = getc(file);
    }

    if(ch == '4') {
	quad.flag |= QUAD_4D;
	dimn = 4;
	ch = getc(file);
    }

    if(!( (ch == 'P' && fexpectstr(file, "OLY") == 0) ||
	 (ch == 'Q' && fexpectstr(file, "UAD") == 0))) {
	return NULL;
    }

    if(fnextc(file, 1) == 'B' && fexpectstr(file, "BINARY") == 0) {
	binary = 1;
	if(fnextc(file, 1) != '\n') {	/* Expect \n after BINARY */
            GeomError(1,"QuadFLoad: bad QUAD file heading on %s", fname);
	    return NULL;
        }
	(void) getc(file);		/* Toss \n */
    }

    if(binary) {
	/*
	 * Binary case is easy.
	 * Read the quad count, then the P, N and C arrays.
	 * Error if we get less than the quad count promised.
	 */
	if(fgetni(file, 1, &quad.maxquad, 1) <= 0) 
	    return NULL;
	if(quad.maxquad <= 0 || quad.maxquad > 10000000) {
	    OOGLError(0, "Reading QUAD BINARY from \"%s\": incredible quad count 0x%x",
		fname, quad.maxquad);
	    return NULL;
	}
	quad.p = OOGLNewNE(QuadP, quad.maxquad, "QuadFLoad: vertices");
	if(quad.flag & QUAD_N) quad.n = OOGLNewNE(QuadN, quad.maxquad, "QuadFLoad: normals");
	if(quad.flag & QUAD_C) quad.c = OOGLNewNE(QuadC, quad.maxquad, "QuadFLoad: colors");
	ngot = getquads(file, &quad, 0, 1, dimn);
	if(ngot != quad.maxquad)
	    goto discard;
    } else {
	/*
	 * ASCII case is harder.
	 * Guess how many quads we need, and realloc() if we need more.
	 */

	quad.maxquad = 1000;	/* Trial value */
	quad.p = OOGLNewNE(QuadP, quad.maxquad, "QuadFLoad: vertices");
	if(quad.flag & QUAD_N)
	  quad.n = OOGLNewNE(QuadN, quad.maxquad, "QuadFLoad: normals");
	if(quad.flag & QUAD_C)
	  quad.c = OOGLNewNE(QuadC, quad.maxquad, "QuadFLoad: colors");
	ngot = 0;
	for(;;) {
	    ngot = getquads(file, &quad, ngot, 0, dimn);
	    if(ngot < quad.maxquad)
		break;

	    quad.maxquad *= 2;
	    quad.p = OOGLRenewNE(QuadP, quad.p, quad.maxquad,
			"QuadFLoad: more space for vertices");
	    if(quad.n)
              quad.n = OOGLRenewNE(QuadN, quad.n, quad.maxquad,
			"QuadFLoad: more space for normals");
	    if(quad.c)
              quad.c = OOGLRenewNE(QuadC, quad.c, quad.maxquad,
			"QuadFLoad: more space for colors");
	}
	/* Release unneeded storage */
	if(ngot <= 0)
	    goto discard;

	quad.maxquad = ngot;
	quad.p = OOGLRenewN(QuadP, quad.p, quad.maxquad);
	if (quad.n)
	    quad.n = OOGLRenewN(QuadN, quad.n, quad.maxquad);
	if (quad.c)
	    quad.c = OOGLRenewN(QuadC, quad.c, quad.maxquad);
    }

    return (Quad *) GeomCCreate(NULL, QuadMethods(),
	CR_4D, (dimn == 4) ? 1 : 0,
	CR_NOCOPY, CR_FLAG, quad.flag, CR_NELEM, quad.maxquad,
	CR_POINT4, quad.p, CR_NORMAL, quad.n, CR_COLOR, quad.c, 0);

  discard:
    OOGLFree(quad.p);
    if (quad.n) OOGLFree(quad.n);
    if (quad.c) OOGLFree(quad.c);
    OOGLSyntax(file, "Error reading QUADs in \"%s\"", fname);
    return NULL;
}
