/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include "tlistP.h"
#include "inst.h"
#include "list.h"
#include "handleP.h"

Geom *
TlistImport( Pool *p )
{
    Geom *subtlist = NULL;
    Geom *list = NULL;
    Geom *groupinst = NULL;
    Geom *geom = NULL;
    Geom *g = NULL;
    Transform tfm;
    Handle *geomhandle = NULL, *tfmhandle = NULL, *subtlisthandle = NULL;
    static char errfmt[] = "Reading TLIST from \"%s\": %s";
    Transform *t;
    int i, c, nel;
    int more;
    vvec v;
    vvec vinst;
    int brack = 0;
    int binary = 0;
    int sawheader = 1;
    FILE *file;
    Transform TT[100];


    if((file = PoolInputFile(p)) == NULL)
	return NULL;

    switch(fnextc(file, 0)) {
    case 'G':
	if(fexpectstr (file, "GROUP")) return NULL;
	break;
    case 'T':
        if(fexpectstr (file, "TLIST")) return NULL;
	break;
    default:
	sawheader = 0;
    }
    VVINIT(vinst, Geom *, 7);
    if (sawheader && (fnextc(file, 1) == 'B')) {
	if(fexpectstr(file, "BINARY")) return NULL;
	if(fnextc(file, 1) == '\n') (void) fgetc(file);

	/*
	 * TLIST BINARY ... \n
	 * <number of tlist elements as a 32-bit int>
	 * <that many 4x4 matrices>
	 */
	if(fgetni(file, 1, &nel, binary) != 1 || nel < 0 || nel > 9999999)
	    return NULL;
	VVINIT(v, Transform, 1);
	vvneeds(&v, nel);
	if((VVCOUNT(v) = fgettransform(file, nel, (float *)t, 1)) < nel) {
	    vvfree(&v);
	    OOGLError(0, "Reading binary TLIST from \"%s\": got only %d of %d transforms expected",
		PoolName(p), VVCOUNT(v), nel);
	    return NULL;
	}
    } else {
	VVINIT(v, Transform, 100);
	vvuse(&v, TT, 100);
    }

    for(;;) {
	switch(fnextc(file, 0)) {
	case ':': case '<': case '@': case 'd':
	tfmobject:
	    if(TransStreamIn(p, &tfmhandle, tfm)) {
		*VVAPPEND(vinst, Geom *) = GeomCCreate(NULL, InstMethods(),
			    CR_AXIS, tfm, CR_AXISHANDLE, tfmhandle, CR_END);
	    }
	    break;

	case CBRA: fgetc(file); brack++; break;
	case CKET:
	    if(--brack >= 0) {
		fgetc(file);
		break;
	    }
	    /* If brackets matched, fall into... */

	case EOF: case ';': case ')':
		goto done;

	case 't':
	    fgetc(file);
	    if(fexpectstr(file, /*t*/"ransform") == 0)
		goto tfmobject;
	    if(fexpecttoken(file, /*t*/"list"))
		goto syntax;
	    if(!GeomStreamIn(p, &subtlisthandle, &subtlist)) {
		OOGLSyntax(file, errfmt, PoolName(p), "failed reading 'tlist'");
		return NULL;
	    }
	    break;

	case 'u':
	case 'g':
	    if(fexpecttoken(file, "unit") && fexpecttoken(file, "geom"))
		goto syntax;
	    /*
	     * "GROUP" Compatibility mode -- create an Inst comprising our
	     * Tlist and the given object.
	     */
	    if(!GeomStreamIn(p, &geomhandle, &geom))
		OOGLSyntax(file, errfmt, PoolName(p), "failed reading 'geom'");
	    else
		groupinst = GeomCCreate(NULL, InstMethods(),
		    CR_HANDLE_GEOM, geomhandle, geom, CR_END);
	    break;

	default:
	    if(fgettransform(file, 1, VVAPPEND(v, float), 0) > 0)
		break;
	    /* Else fall into... */

	 syntax:
	    if(sawheader)
		OOGLSyntax(file, errfmt, PoolName(p), "unknown keyword");
	    vvfree(&v);
	    return NULL;
	}
    }
  done:
    if(brack > 0)
	OOGLSyntax(file, errfmt, PoolName(p), "unclosed '{'");  /* } */

    if(VVCOUNT(v) > 0) {
	vvtrim(&v);
	*VVAPPEND(vinst, Geom *) = GeomCCreate(NULL,TlistMethods(),
	    CR_NOCOPY, CR_NELEM, VVCOUNT(v), CR_ELEM, VVEC(v,float), CR_END);
    }

    for(i = 0, list = g = NULL; i < VVCOUNT(vinst); i++) {
	g = VVEC(vinst, Geom *)[i];
	GeomSet(g, CR_HANDLE_GEOM, subtlisthandle, subtlist, CR_END);
	if(VVCOUNT(vinst) > 1)
	    list = ListAppend(list, g);
    }
    if(list) g = list;

    if(groupinst) {
	GeomSet(groupinst, CR_TLIST, g, CR_END);
	g = groupinst;
    }

    return g;
}

int
TlistExport( Tlist *tl, Pool *p )
{
    FILE *outf;

    if(tl == NULL || tl->elements == NULL
			|| (outf = PoolOutputFile(p)) == NULL)
	return 0;
    fprintf(outf, "TLIST\n");
    if(fputtransform(outf, tl->nelements, (float *)tl->elements, 0)
		!= tl->nelements)
	return 0;
   if(tl->tlist != NULL) {
	fprintf(outf, " tlist ");
	GeomStreamOut( p, tl->tlisthandle, tl->tlist );
   }
   return 1;
}
