static char rcsid[] = "$Id: source.c,v 1.5 1992/03/25 17:48:26 jtsillas Exp $";

/*****************************************************************************
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *  Copyright 1990 Thomson Consumer Electronics, Inc.
 *  Copyright 1991 Bull HN Worldwide Info Systems, Inc.
 *
 *****************************************************************************/

/*  source.c
 *
 *    Create the source window and handle display of file.
 *
 *    source_init(): 	Initialization routine.
 *    Update():		Action proc to update source window on scrollbar 
 *                      action.
 *    ScrollTxt():      Callback to scroll text in source window (static).
 *    UpdateSigns():    Update all the signs (static).
 *    SelectWord():     Replaces the standard select translation (static).
 *    NotifyResize():	Action proc to update source window on resize.
 *    CreateSourceWindow(): Create the source window.
 *    BuildLinePos():	Build an array of starting text position of each line
 *                      (static).
 *    LookUpFileTable(): Check out source file info from a file table (static).
 *    SaveDisplayedFileInfo(): Records displayed file info into file table
 *                      (static).
 *    DisplayFile():	Display a file on the source window (static).
 *    LoadFile():	Search for a file and open it for display.
 *    LoadCurrentFile(): Load the CurrentFile.
 *    resetLineBounds(): Resets the top and bottom lines for resize (static).
 *    CreateCommandButtons(): Create the command buttons (static).
 *    PopupCommands():  Manage the commands dialog.
 *    MakeDirList():    Create a list of directories to search for source.
 *    GetPathname():    Get a path for a file using the dirList.
 *    UpdateScrollV():  Update all the properties of a scroll bar.
 */

#include <X11/Xos.h>
#include <sys/stat.h>
#include <pwd.h>
#include "global.h"
#include "bitmaps.h"
#if defined(SCO)
#   include <fcntl.h>
#endif

#include <Xm/Form.h>
#include <Xm/Text.h>
#include <Xm/ScrolledW.h>
#include <Xm/ScrollBar.h>
#include <Xm/ArrowB.h>
#include <Xm/PushB.h>

#define	MAXDIRS	256			/* max number of dirs in dirList */

static void SelectWord(Widget, XEvent*, String*, Cardinal*);
static void ScrollTxt(Widget, unsigned int *, XmScrollBarCallbackStruct*);
static void UpdateSigns(void);
static void resetLineBounds(Widget, XEvent*, String*, Cardinal*);
static void CreateCommandButtons(Widget);
static void DisplayFile(FileRec*);
static void SaveDisplayedFileInfo(void);
static int LookUpFileTable(char*, char*, FileRec**);
static void BuildLinePos(FileRec*);


char		CurrentFile[MAXNAME];	/* current contents of file variable */
Widget		sourceWindow,		/* text window for source file */
                sourceBBW,
                sourceScroll,
                scrollV,
                sourceForm,
                commandDialog;

FileRec  	*displayedFile;		/* pointer to table entry of currently
					   displayed file */

static FileRec	**fileTable;		/* table of file records */
static int	fileTableSize;		/* size of file table */
static char 	*dirList[MAXDIRS];	/* list of dirs for searching files */
static short    maxcol;
static short    currvalue;

void source_init()
{
    dirList[0] = NULL;
}

/*
 *  Update topline, bottomline, arrow sign, updown sign, stop signs, and
 *  line label.
 */
void Update(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  XmTextPosition     pos;
  int			topline;
  FileRec 		*file;
  int Py;
  Arg args[5];
  
  if (displayedFile) 
    {
      file = displayedFile;
      pos = XmTextGetTopCharacter(sourceWindow);
      file->topPosition = pos;
      topline = TextPositionToLine(pos);
      /* Update the symbols only if the text scrolls */
      if (file->topline != topline) 
	{
	  file->topline = topline;
	  file->bottomline = MIN (file->topline + file->lines - 1, 
				  file->lastline);
	  if (*num_params == 1 && strcmp(params[0], "warp") == 0)
	    {
	      XmTextSetInsertionPosition(sourceWindow, 
					 file->linepos[file->topline]);
	    }

	  UpdateLineLabel(file->topline);
	  UpdateStops(file);
	  UpdateArrow(file);
	  UpdateUpdown(file);
	  UpdateBomb(file);
	  UpdateScrollV(file);
	}
      else {/* Update caret position only */
	pos = XmTextGetInsertionPosition(sourceWindow);
	UpdateLineLabel(TextPositionToLine(pos));
      }
    }
}

static void UpdateSigns()
{
  int pos;
  FileRec *file;
  Arg args[MAXARGS];

  file = displayedFile;
  pos = XmTextGetTopCharacter(sourceWindow);
  file->topline = TextPositionToLine(pos);

  file->bottomline = MIN (file->topline + file->lines - 1, 
			  file->lastline);
  UpdateStops(file);
  UpdateArrow(file);
  UpdateUpdown(file);
  UpdateBomb(file);

}  

/*  This is my own select word routine to replace the standard action
 *  procedure provided by the Text widget.
 *  It selects a word delimited by DELIMITERS, not whitespace.
 */
static void SelectWord(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  XmTextPosition left, right, center, last;
  char *valuestring;
  
  XtCallActionProc(w, "set-insertion-point", event, params, *num_params);
  center = XmTextGetInsertionPosition(w);
  valuestring = XmTextGetString(w);
  last = XmTextGetLastPosition(w);
  
  if(strchr(DELIMITERS, valuestring[center])) return;
  left = center;
  right = center;
  
  while(left>0 && !strchr(DELIMITERS, valuestring[left-1])) left--;
  while(right<last && !strchr(DELIMITERS, valuestring[right])) right++;
  
  XmTextSetSelection(w, left, right, 0);
  XtFree(valuestring);
}

static void ScrollTxt(w, client_data, call_data)
     Widget w;
     unsigned int *client_data;
     XmScrollBarCallbackStruct* call_data;
{
  Arg args[5];
  int Py;

  if(!displayedFile) return;

  switch(*client_data)
    {
    case _PAGERIGHT:
      XtCallActionProc(sourceWindow, "page-right", 
		       call_data->event, 
		       NULL, 0);
      break;
    case _PAGELEFT:
      XtCallActionProc(sourceWindow, "page-left", 
		       call_data->event, 
		       NULL, 0);
      break;
    case _SCROLLUPPAGE:
      Py = TextPositionToLine(XmTextGetTopCharacter(sourceWindow));
      Py = (Py - displayedFile->lines >= 1 ?
	     Py - displayedFile->lines : 1);
      XmTextSetTopCharacter(sourceWindow, displayedFile->linepos[Py]);
      Py = TextPositionToLine(XmTextGetTopCharacter(sourceWindow));
      XtSetArg(args[0], XmNvalue, Py);
      XtSetValues(scrollV, args, 1);
      break;
    case _SCROLLDOWNPAGE:
      Py = TextPositionToLine(XmTextGetTopCharacter(sourceWindow));
      Py = (displayedFile->lines + Py <= displayedFile->lastline ?
	     displayedFile->lines + Py : displayedFile->lastline);
      XmTextSetTopCharacter(sourceWindow, displayedFile->linepos[Py]);
      XtSetArg(args[0], XmNvalue, Py);
      XtSetValues(scrollV, args, 1);
      break;
    default:
      XmTextSetTopCharacter(sourceWindow, 
        displayedFile->linepos[call_data->value]);
    }
  UpdateSigns();
}

static void resetLineBounds(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  Arg args[MAXARGS];
  short file_lines;

  if(displayedFile)
    {
      XtSetArg(args[0], XmNrows, &file_lines);
      XtGetValues(sourceWindow, args, 1);
      displayedFile->lines = file_lines;
      displayedFile->bottomline = MIN (displayedFile->topline + 
				       displayedFile->lines - 1, 
				       displayedFile->lastline);
      UpdateSigns();
      UpdateScrollV(displayedFile);
    }
}

#define BUTTONSNUM 18		/* Number of buttons */
#define NCOLUMNS 2		/* Should divide BUTTONSNUM evenly */
#define SPACEFACTOR 32		/* These number can be diddled. */
#define BUTTONWIDTH 32
#define FRACTIONBASE 	     ((BUTTONSNUM * SPACEFACTOR) / NCOLUMNS) 

static void CreateCommandButtons(parent)
     Widget parent;
{
  static unsigned int scroll_client_data[] = {
    _PAGELEFT, _PAGERIGHT,
  };

  static buttonstruct commandbuttons[] = {
    { NULL, "stackbutton", stack_bits, stack_width, stack_height, 
	&app_resources.button_color, DoIt, "info stack\n" },
    { NULL, "runbutton", run_bits, run_width, run_height, 
	&app_resources.button_color, DoIt, "run\n" },
    { NULL, "upbutton", up_bits, up_width, up_height, 
	&app_resources.updown_color, DoIt, "up\n" },
    { NULL, "localsbutton", locals_bits, locals_width, locals_height, 
	&app_resources.button_color, DoIt, "info locals\n" },
    { NULL, "downbutton", down_bits, down_width, down_height, 
	&app_resources.updown_color, DoIt, "down\n" },
    { NULL, "argsbutton", args_bits, args_width, args_height, 
	&app_resources.button_color, DoIt, "info args\n" },
    { NULL, "printbutton", print_bits, print_width, print_height, 
	&app_resources.button_color, Print, NULL },
    { NULL, "printsbutton", prints_bits, prints_width, prints_height, 
	&app_resources.button_color, Print, (void *) 1 },
    { NULL, "nextbutton", next_bits, next_width, next_height, 
	&app_resources.button_color, DoIt, "next\n" },
    { NULL, "contbutton", cont_bits, cont_width, cont_height, 
	&app_resources.button_color, DoIt, "cont\n" },
    { NULL, "stepbutton", step_bits, step_width, step_height, 
	&app_resources.button_color, DoIt, "step\n" },
    { NULL, "finishbutton", finish_bits, finish_width, finish_height, 
	&app_resources.button_color, DoIt, "finish\n" },
    { NULL, "breakbutton", stop_bits, stop_width, stop_height, 
	&app_resources.stop_color, Break, "break" },
    { NULL, "clearbutton", clear_bits, clear_width, clear_height, 
	&app_resources.stop_color, Clear, NULL },
    { NULL, "dispbutton", disp_bits, disp_width, disp_height, 
	&app_resources.button_color, Display_, NULL },
    { NULL, "undispbutton", undisp_bits, undisp_width, undisp_height, 
	&app_resources.button_color, Undisplay, NULL },
    { NULL, "pageRbutton", pageR_bits, pageR_width, pageR_height, 
	&app_resources.arrow_color, ScrollTxt,
	(XtPointer) &scroll_client_data[0]},
    { NULL, "pageLbutton", pageL_bits, pageL_width, pageL_height, 
	&app_resources.arrow_color, ScrollTxt, (void *)
	  (XtPointer) &scroll_client_data[1]},
  };
  
  Pixel bg;
  int screen;
  Display *topdisplay;
  int window1;
  int depth1;
  int buttonindex;
  XtCallbackList buttoncalls;
  Arg args[10];

  static Arg buttonargs[] = {
    { XmNbottomAttachment, XmATTACH_POSITION },
    { XmNtopAttachment, XmATTACH_POSITION },
    { XmNrightAttachment, XmATTACH_FORM },
    { XmNwidth, BUTTONWIDTH },
    { XmNlabelType, XmPIXMAP },
    { XmNhighlightOnEnter, True },
    { 0, 0 },
    { 0, 0 },
    { 0, 0 },
  };

  XtSetArg(args[0], XmNbackground, &bg);
  XtGetValues(sourceForm, args, 1);
  topdisplay = XtDisplay(toplevel);
  screen = DefaultScreen(topdisplay);
  window1 = DefaultRootWindow(topdisplay);
  depth1 = DefaultDepth(topdisplay, screen);
  
  for (buttonindex = 0; buttonindex < BUTTONSNUM; buttonindex++)
    {
      XtSetArg(buttonargs[6], XmNlabelPixmap, 
	       XCreatePixmapFromBitmapData(topdisplay, window1,
				    commandbuttons[buttonindex].labelbits,
				    commandbuttons[buttonindex].labelwidth,
				    commandbuttons[buttonindex].labelheight, 
				    *(commandbuttons[buttonindex].label_color),
				    bg, depth1));

      XtSetArg(buttonargs[7], XmNrightOffset,
	       (buttonindex%NCOLUMNS)?(BUTTONWIDTH*(buttonindex%NCOLUMNS)):0);
      XtSetArg(buttonargs[8], XmNtopPosition, 
	       ((buttonindex/NCOLUMNS) * SPACEFACTOR));
      XtSetArg(buttonargs[9], XmNbottomPosition,
	       (((buttonindex/NCOLUMNS) + 1) * SPACEFACTOR));

      commandbuttons[buttonindex].button = 
	XtCreateManagedWidget(commandbuttons[buttonindex].buttonname,
			      xmPushButtonWidgetClass, parent,
			      buttonargs, 10);
      XtAddCallback(commandbuttons[buttonindex].button, 
		    XmNactivateCallback, commandbuttons[buttonindex].CB, 
		    commandbuttons[buttonindex].CBarg);
    }
}

/* 
 *  On top of a form widget, we have a text widget, label
 *  widgets for the stop sign, arrow sign, and updown signs.
 */
void CreateSourceWindow(parent)
     Widget parent;
{
  int ctr;
  Arg args[MAXARGS];
  
  static XtActionsRec command_actions[] = {
    { "SelectWord", (XtActionProc) SelectWord },
    { "Update", (XtActionProc) Update },
    { "resetLineBounds", (XtActionProc) resetLineBounds },
    { NULL, NULL }
  };

  static Arg fractionbase_args[] = {
    { XmNfractionBase, FRACTIONBASE },
  };

				/* The following arrays are used */
				/* to construct callbacks for the scroll */
				/* bar. NCALLS is the number of callbacks */
				/* to use. */
#define NCALLS 5
  static unsigned int scroll_client_data[] = {
    _SCROLLUP, _SCROLLDOWN, _SCROLLDOWNPAGE, _SCROLLUPPAGE, _SCROLLDRAG,
  };
  static XtPointer scroll_call_list[] = {
    XmNdecrementCallback, XmNincrementCallback, XmNpageIncrementCallback,
    XmNpageDecrementCallback, XmNdragCallback,
  };

  sourceScroll = XtCreateManagedWidget("sourceScroll", 
				       xmScrolledWindowWidgetClass,
				       parent, NULL, 0);   
  
  scrollV = XtCreateWidget("scrollV", xmScrollBarWidgetClass, sourceScroll,
			   NULL, 0);
  
  sourceForm = XtCreateManagedWidget("sourceForm", xmFormWidgetClass,
				     sourceScroll, fractionbase_args, 1);
  
  XmScrolledWindowSetAreas(sourceScroll, NULL, scrollV, sourceForm);
  
  for (ctr = 0; ctr < NCALLS; ctr++) {
    XtRemoveAllCallbacks(scrollV, scroll_call_list[ctr]);
    XtAddCallback(scrollV, scroll_call_list[ctr], ScrollTxt,
		  (XtPointer) &scroll_client_data[ctr]);
  }

  if(!app_resources.useCommandDialog)
    {
      XtSetArg(args[0], XmNrightOffset, BUTTONWIDTH * NCOLUMNS);
      sourceWindow = XtCreateManagedWidget("sourceWindow", xmTextWidgetClass,
					   sourceForm, args, 1);
    }
  else
    sourceWindow = XtCreateManagedWidget("sourceWindow", xmTextWidgetClass,
					 sourceForm, NULL, 0);
    
  XtAppAddActions(app_context, command_actions, XtNumber(command_actions));

  XtSetArg(args[0], XmNrightWidget, sourceWindow);
  sourceBBW = XtCreateManagedWidget("sourceBBW", xmBulletinBoardWidgetClass,
				    sourceForm, args, 1);
  if(app_resources.useCommandDialog)
    {
      commandDialog = XmCreateFormDialog(parent, "commandDialog",
					 fractionbase_args, 1);
      CreateCommandButtons(commandDialog);
    }
  else
    CreateCommandButtons(sourceForm);
}

/* 
 * Popup the commandDialog Box if we are using it.
 */
void PopupCommands(w, client_data, call_data)
     Widget w;
     XtPointer client_data;
     XmPushButtonCallbackStruct *call_data;
{
  if(app_resources.useCommandDialog)
    XtManageChild(commandDialog);
  else
    {
      UpdateMessageWindow("Not using command button dialog", NULL);
      bell(0);
    }
}

/*
 *  Build the array which gives the starting text position of each line.
 *  > Estimate the number of lines in the file and allocate memory buffer.
 *  > Starting position of line #1 is 0, and is stored in linepos[1].
 *  > Search for '\n' till end of buffer.
 */
static void BuildLinePos(file)
FileRec *file;
{
    char *p;
    int	 line, nlines;

    nlines = MAX(2, file->filesize/CHARS_PER_LINE);
    file->linepos = (XmTextPosition *)
		    XtMalloc ((nlines+1) * sizeof(XmTextPosition));
    p = file->buf;

    file->linepos[0] = 0;
    file->linepos[1] = 0;
    line = 2;
   
    while (*p) {
	if (*p++ == '\n') {
	    if (line == nlines) { 	/* buffer full, need more memory */
                file->linepos = (XmTextPosition *) XtRealloc (file->linepos, 
			  (nlines + ADD_LINES) * sizeof(XmTextPosition));
		nlines += ADD_LINES;
            }
            file->linepos[line++] = p - file->buf;
	}
    }
    file->lastline = line - 2;
    file->linepos = (XmTextPosition *) XtRealloc       /* shrink to min size */
			(file->linepos, line * sizeof(XmTextPosition));
}


/*
 * Look up the file table for an entry with "filename"
 * If not found, create an entry and initialize proper fields,
 * else, return pointer to entry found.
 */
static int LookUpFileTable(pathname, filename, file)
char *pathname, *filename;
FileRec **file;
{
    struct stat fileinfo;
    int  	fd;
    int 	i, j, n;

    for (i=0; fileTable && fileTable[i] && i<fileTableSize; i++) {
	if (!strcmp(fileTable[i]->pathname, pathname)) { /* file found */
	    if (stat(pathname, &fileinfo) == -1) {
		UpdateMessageWindow("Error: cannot stat file %s", pathname);
	        *file = fileTable[i];
		return 0;
	    }
	    if (fileinfo.st_mtime > fileTable[i]->mtime) { /* file modified */
		XtFree((char *)fileTable[i]->buf);
		XtFree((char *)fileTable[i]->linepos);
		XtFree((char *)fileTable[i]);
		displayedFile = NULL;
		break;
	    }
	    else if (displayedFile && 		/* same as displayed file */
		     strcmp(pathname, displayedFile->pathname) == 0) {
		*file = NULL;
		return 0;
	    }
	    else {
	    	*file = fileTable[i];
		return 0;
	    }
	}
    }

    /* Record file into file table */

    if (i == fileTableSize) {		/* file table full, enlarge it */
	fileTableSize += ADD_SIZE;
	fileTable = (FileRec **) 
		     XtRealloc (fileTable, fileTableSize * sizeof(FileRec *));
	for (j=i; j<fileTableSize; j++)
	    fileTable[j] = NULL;
    }
    if ((fd = open(pathname, O_RDONLY)) == -1) {
	UpdateMessageWindow("Error: cannot open file %s", pathname);
	return -1;
    }
    if (fstat(fd, &fileinfo) == -1) {
	UpdateMessageWindow("Error: cannot fstat file %s", pathname);
	close(fd);
	return -1;
    }
    fileTable[i] = (FileRec *) XtMalloc (sizeof(FileRec));
    fileTable[i]->filesize = fileinfo.st_size + 1;
    fileTable[i]->mtime = fileinfo.st_mtime;
    fileTable[i]->buf = XtMalloc((int)fileTable[i]->filesize);
    if ((n = read(fd, fileTable[i]->buf, (int) fileTable[i]->filesize)) == -1) {
	UpdateMessageWindow("Error: cannot read file %s", pathname);
	XtFree(fileTable[i]->buf);
	XtFree(fileTable[i]);
	fileTable[i] = NULL;
	close(fd);
	return -1;
    }
    fileTable[i]->buf[n] = '\0';
    fileTable[i]->pathname = XtNewString(pathname);
    fileTable[i]->filename = XtNewString(filename);
    fileTable[i]->currentline = 1;
    fileTable[i]->topline = 1;
    fileTable[i]->bottomline = 0;
    fileTable[i]->topPosition = 0;
    BuildLinePos(fileTable[i]);
    close(fd);
    *file = fileTable[i];
    return 0;
}

/*  
 *  Remember file position and current line before closing.
 */
static void SaveDisplayedFileInfo()
{
    XmTextPosition pos;
    Arg args[MAXARGS];

    if (displayedFile) {
      XtSetArg(args[0], XmNtopPosition, &(displayedFile->topPosition));
      XtGetValues(sourceWindow, args, 1);
      XtSetArg(args[0], XmNcursorPosition, &pos);
      XtGetValues(sourceWindow, args, 1);
      displayedFile->currentline = TextPositionToLine(pos);
    }
}

/* Update the Vertical scroll bar for a given file and rows */
void UpdateScrollV(file)
     FileRec *file;
{
  
  Arg args[MAXARGS];
  int slidersize;
  Dimension height;
  int value, maximum;

  
  if(file->lastline < file->lines)
    {
      if(XtIsManaged(scrollV))
	XtUnmanageChild(scrollV);
    }
  else
    {
      XtSetArg(args[0], XmNheight, &height);
      XtSetArg(args[1], XmNmaximum, &maximum);
      XtGetValues(scrollV, args, 2);
      slidersize = (file->lines * height)/file->lastline + 10;

      value = file->topline;
      XtSetArg(args[0], XmNmaximum, file->lastline + slidersize);
      XtSetArg(args[1], XmNminimum, 0);
      if (slidersize+value > maximum) /* Set new max first */
	{
	  XtSetValues(scrollV, args, 2);
	  XmScrollBarSetValues(scrollV, value, slidersize, 1, 0, True);
	}
      else			/* First set value then set max */
	{
	  XmScrollBarSetValues(scrollV, value, slidersize, 1, 0, True);
	  XtSetValues(scrollV, args, 2);
	}
      XtManageChild(scrollV);
    }
}

/*   DisplayFile() displays the file onto the source window.  It
 *     uses topPosition to remember where it was last opened.  But it
 *     must recalculate bottomline because the window size might be
 *     different.
 */
static void DisplayFile(file)
FileRec *file;
{
    Arg 	args[MAXARGS];
    short       rows;

    XtSetArg(args[0], XmNeditable, False);
    XtSetArg(args[1], XmNvalue, file->buf);
    XtSetValues(sourceWindow, args, 2);

    XtSetArg(args[0], XmNrows, &rows);
    XtGetValues(sourceWindow, args, 1);

    file->lines = rows;
    file->bottomline = MIN (file->topline + file->lines - 1, file->lastline);
    UpdateScrollV(file);
}


/*  Create a list of directories for searching source files.
 *  It reads the list of directories specified by the user, adding
 *  the current directory into the list if it is not already there.
 */
void MakeDirList(output)
char *output;
{
    char *s, list[LINESIZ], command[LINESIZ];
    int  i;

    for (i=0; dirList[i]; i++)			/* remove old list */
	XtFree(dirList[i]);
    i = 0;

    if (output) {                                        /* create list */
        s = (char *) strtok(output, ":");
        while (s) {
            dirList[i] = XtNewString(s);
            i++;
            s = (char *) strtok(NULL, ":");
        }
        dirList[i] = NULL;
    }
}


/*  Returns the full pathname of a given file.
 *  It searches for the file from a list of directories.
 */
char *GetPathname(filename)
     char *filename;
{
  char pathname[LINESIZ];
  char *normal_file;
  int i;
  
  /* Normalize the file name using normal_file to point to
   * the last element of a path.
   */
  for (normal_file = filename + strlen(filename); normal_file != filename 
       && *normal_file != '/'; normal_file--);
  if (*normal_file == '/') normal_file++;
  if (normal_file == NULL || !strcmp(normal_file, ""))
    return NULL;
  
  for (i=0; dirList[i]; i++) 
    {
      if (!strcmp(dirList[i], "$cwd"))
	sprintf(pathname, "%s/%s", cwd, normal_file);
      else if (!strcmp(dirList[i], "$cdir"))
	sprintf(pathname, "%s/%s", cwd, filename);
      else
	sprintf(pathname, "%s/%s", dirList[i], normal_file);
      if (access(pathname, R_OK) == 0)
	return XtNewString(pathname);
    }

  if (access(filename, R_OK) == 0)
    return XtNewString(filename); /* When all else fails try the obvious. */

  UpdateMessageWindow("File not found: %s", filename);
  return NULL;
}

/*
 * Given a file name, LoadFile attempts to open it and displays it onto
 * the source window:
 *   1. get the full pathname of the file
 *   2. LookUpFileTable() returns a pointer to the file's entry if it's
 *      already in the table; else, creates an entry and return a pointer.
 *   3. save the current displayedFile info
 *   4. display the file
 *   5. update the file label and the various signs on the source window.
 *  LoadFile returns 0 upon successful completion, -1 otherwise.
 */
int LoadFile(filename)
char *filename;
{
    FileRec 	*file;
    char	*pathname;
    Arg         args[5];
    Position    Py;
    

    pathname = GetPathname(filename);
    if (pathname == NULL) { 
	return -1;
    }
    if (LookUpFileTable(pathname, filename, &file) != -1) {
	if (file) {	/* load new file */
	    SaveDisplayedFileInfo();
	    DisplayFile(file);
	    UpdateFileLabel(pathname);
	    XmTextClearSelection(sourceWindow,0);
	    XmTextSetInsertionPosition(sourceWindow, 
				       file->linepos[file->currentline]);
	    UpdateLineLabel(file->currentline);
	    UpdateStops(file);
	    UpdateArrow(file);
	    UpdateUpdown(file);
	    UpdateBomb(file);
	    displayedFile = file;
	}
    	return 0;
    }
    else {		/* LookUpFileTable() fails */
    	return -1;
    }
}

int LoadCurrentFile()
{
    query_gdb("info line\n");
    return LoadFile(CurrentFile);
}


