/*	$NetBSD: adiantum_selftest.c,v 1.1 2020/06/29 23:44:01 riastradh Exp $	*/

/*-
 * Copyright (c) 2020 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(1, "$NetBSD: adiantum_selftest.c,v 1.1 2020/06/29 23:44:01 riastradh Exp $");

#include <sys/types.h>

#ifdef _KERNEL

#include <sys/systm.h>

#include <lib/libkern/libkern.h>

#else

#include <string.h>
#include <stdio.h>

#include <openssl/aes.h>

struct aesenc {
	AES_KEY enckey;
};

struct aesdec {
	AES_KEY deckey;
};

static void
hexdump(int (*prf)(const char *, ...) __printflike(1,2), const char *prefix,
    const void *buf, size_t len)
{
	const uint8_t *p = buf;
	size_t i;

	(*prf)("%s (%zu bytes)\n", prefix, len);
	for (i = 0; i < len; i++) {
		if (i % 16 == 8)
			(*prf)("  ");
		else
			(*prf)(" ");
		(*prf)("%02hhx", p[i]);
		if ((i + 1) % 16 == 0)
			(*prf)("\n");
	}
	if (i % 16)
		(*prf)("\n");
}

#endif

#include "adiantum.h"

/* https://github.com/google/adiantum/blob/aab35db7bfb6e05d5ad0b41b5088a9f5a840bde3/test_vectors/ours/Adiantum/Adiantum_XChaCha12_32_AES256.json */

int
adiantum_selftest(void)
{
	static const struct {
		uint8_t k[32];
		unsigned tlen;
		uint8_t t[64];
		unsigned len;
		uint8_t p[4096];
		uint8_t c[4096];
	} C[] = {
		[0] = {
			.k = {
				0x7f,0xc7,0x15,0x2a, 0xe1,0xf5,0xfd,0xa4,
				0x17,0x67,0x69,0xae, 0xc9,0x2b,0xba,0x82,
				0xa3,0x14,0xe7,0xcf, 0xad,0xfd,0x85,0x40,
				0xda,0x7b,0x7d,0x24, 0xbd,0xf1,0x7d,0x07,
			},
			.tlen = 0,
			.len = 16,
			.p = {
				0x9b,0xe3,0x82,0xc6, 0x5a,0xc1,0x9f,0xad,
				0x46,0x59,0xb8,0x0b, 0xac,0xc8,0x57,0xa0,
			},
			.c = {
				0x82,0x0a,0xe4,0x44, 0x77,0xdd,0x9a,0x18,
				0x6f,0x80,0x28,0x8b, 0x25,0x07,0x0e,0x85,
			},
		},
		[1] = {
			.k = {
				0x26,0x6a,0xf9,0x4a, 0x21,0x49,0x6b,0x4e,
				0x3e,0xff,0x43,0x46, 0x9c,0xc1,0xfa,0x72,
				0x0e,0x77,0x9a,0xd5, 0x37,0x47,0x00,0x38,
				0xb3,0x6f,0x58,0x6c, 0xde,0xc0,0xa6,0x74,
			},
			.tlen = 0,
			.len = 128,
			.p = {
				0xdd,0x07,0xfe,0x61, 0x97,0x0c,0x31,0x48,
				0x09,0xbf,0xdb,0x9b, 0x4b,0x7d,0x9c,0x80,
				0xe6,0x11,0xe5,0x76, 0x5b,0xcc,0x76,0xdf,
				0x34,0xd5,0x23,0xcd, 0xe1,0xdc,0x4e,0x4f,
				0x65,0x20,0x58,0x8e, 0xe8,0x2c,0xc2,0x64,
				0x32,0x83,0x7a,0xbf, 0xe1,0xca,0x0b,0x4b,
				0xc6,0xec,0x0d,0xc5, 0x4a,0xb6,0x9b,0xa5,
				0xc4,0x01,0x54,0xf5, 0xb5,0xfa,0x8f,0x58,
				0x45,0x72,0x28,0xd8, 0x55,0x21,0xa2,0x5c,
				0x7d,0xc8,0x0c,0x3c, 0x3c,0x99,0xc4,0x1a,
				0xc2,0xe7,0x1c,0x0c, 0x14,0x72,0x1d,0xf8,
				0x45,0xb7,0x9c,0x97, 0x07,0x04,0x9b,0x91,
				0x5e,0x95,0xef,0x5f, 0xe6,0xad,0xbd,0xbb,
				0xe7,0xd1,0x22,0xc3, 0x98,0x44,0x89,0x05,
				0xe8,0x63,0x0d,0x44, 0xcb,0x36,0xd5,0x43,
				0xcc,0x05,0x7c,0x31, 0xd3,0xbc,0x17,0x7f,
			},
			.c = {
				0xba,0xd3,0xbf,0xbf, 0xb2,0x4e,0x1a,0xfd,
				0x59,0xbe,0x9d,0x40, 0xe0,0x27,0x94,0xdd,
				0x5c,0x08,0x1c,0xa5, 0xd0,0x25,0x87,0xca,
				0x15,0x6a,0x35,0xe9, 0x8a,0x05,0x67,0x53,
				0x04,0x4d,0xdf,0x35, 0x07,0x19,0x25,0xa0,
				0x44,0x1a,0x5b,0xd6, 0x8b,0x0f,0xd3,0x36,
				0x8a,0x60,0x8c,0x6b, 0x53,0xdb,0x69,0xb0,
				0x37,0x69,0xb5,0x1b, 0x1f,0xf5,0xd5,0xab,
				0x47,0x3a,0x45,0xb2, 0x37,0x6c,0xc3,0xc1,
				0x1f,0xdb,0x74,0x6b, 0x1f,0x3b,0x2c,0x1a,
				0xee,0xff,0xe9,0x28, 0xfe,0xa3,0x49,0x96,
				0x7a,0xb3,0x68,0x4e, 0xb1,0xc4,0x85,0xdc,
				0x18,0x87,0xfd,0xbf, 0x84,0x39,0xb2,0x20,
				0x29,0x46,0x8a,0x3e, 0xa9,0xf9,0xcc,0x56,
				0x6b,0x2f,0x43,0x4a, 0x1b,0x48,0x6b,0xd6,
				0x03,0x1d,0x66,0xa1, 0x49,0xba,0xe9,0xf5,
			},
		},
		[2] = {
			.k = {
				0x7c,0xab,0xc4,0x63, 0xc0,0x40,0x5e,0xad,
				0x8f,0x02,0x5a,0xa9, 0xba,0x68,0x58,0xe3,
				0xb6,0xbb,0x03,0xc9, 0xe6,0x1e,0xe7,0xc3,
				0xd7,0x2c,0xf7,0x7a, 0xf7,0x2c,0xd1,0x07,
			},
			.tlen = 0,
			.len = 512,
			.p = {
				0x4f,0xc9,0x8f,0xa7, 0x81,0x81,0x3a,0xb7,
				0x3c,0x55,0x8f,0x8f, 0x18,0xc4,0x7a,0xd2,
				0x13,0x70,0x94,0x0f, 0x46,0xb2,0x0f,0x53,
				0xde,0xdf,0x06,0xf8, 0x60,0x34,0xad,0x39,
				0xe9,0x47,0x23,0x31, 0x94,0xf3,0x59,0x88,
				0x96,0x14,0x52,0x3b, 0x88,0xb7,0x55,0xe9,
				0x4a,0xbc,0x41,0xea, 0x24,0x03,0x35,0x78,
				0xb7,0x4b,0x9f,0x8b, 0xe4,0x36,0x77,0x0a,
				0x70,0x19,0x90,0x9b, 0xb1,0x70,0x27,0x23,
				0x31,0xd9,0xe5,0x26, 0x36,0x71,0x06,0xc7,
				0xd3,0xb1,0xb8,0x52, 0x6a,0xe1,0x95,0x86,
				0x76,0xc3,0x02,0x2c, 0xd2,0xe7,0xc2,0x1c,
				0x6f,0xcb,0x61,0x56, 0xfc,0x5e,0xf2,0x57,
				0x90,0x46,0xfb,0x6a, 0xc1,0x5e,0x56,0x5b,
				0x18,0x8d,0x0e,0x4f, 0x4e,0x14,0x4c,0x6d,
				0x97,0xf9,0x73,0xed, 0xc5,0x41,0x94,0x24,
				0xaa,0x35,0x2f,0x01, 0xef,0x8f,0xb2,0xfd,
				0xc2,0xc7,0x8b,0x9c, 0x9b,0x10,0x89,0xec,
				0x64,0xbb,0x54,0xa5, 0x01,0xdc,0x51,0x57,
				0xc8,0x5a,0x03,0xcb, 0x91,0x73,0xb2,0x08,
				0xc3,0xcc,0x3c,0x1b, 0xae,0x3e,0x0f,0xf3,
				0x93,0xb9,0xc3,0x27, 0xd7,0x88,0x66,0xa2,
				0x40,0xf9,0xfd,0x02, 0x61,0xe1,0x2b,0x5d,
				0xc9,0xe8,0xd6,0xac, 0xf0,0xd0,0xe3,0x79,
				0x94,0xff,0x50,0x09, 0x4e,0x68,0xe8,0x5e,
				0x3f,0x58,0xc8,0xb8, 0x0f,0xd7,0xc2,0x2d,
				0x91,0x3e,0x47,0x10, 0x50,0x98,0xa6,0xf9,
				0x37,0xd6,0x90,0xed, 0xb7,0x5e,0x3a,0xd0,
				0xd7,0x50,0xc4,0x69, 0xe6,0x29,0xb8,0x9a,
				0xc1,0x5c,0x2b,0x34, 0x6d,0x44,0x58,0xd6,
				0xd4,0x7e,0xe2,0x42, 0x67,0x45,0xe5,0x64,
				0x48,0xac,0x00,0xe9, 0xb6,0xd0,0xc3,0xc5,
				0x5d,0x9e,0x95,0x4e, 0x10,0x18,0x29,0x86,
				0xaa,0x37,0xa3,0x3c, 0xe1,0xd6,0x5d,0x6d,
				0x4a,0xca,0xc3,0xe2, 0x25,0xb7,0x49,0x4a,
				0x36,0x67,0xc0,0xe1, 0x02,0x45,0xcc,0xd4,
				0x11,0x37,0x11,0x8e, 0x54,0xf5,0xea,0x80,
				0x04,0x72,0x06,0x36, 0x8f,0xf9,0x1e,0xed,
				0x91,0x14,0x9d,0x42, 0x59,0xc1,0x87,0xb8,
				0xf1,0xce,0xb2,0x17, 0x42,0xa1,0x2f,0x96,
				0xa3,0x50,0xe9,0x01, 0x24,0x9e,0xe5,0xbb,
				0x97,0x83,0x31,0x12, 0xa8,0x7c,0xca,0x7b,
				0x90,0x33,0xad,0x1c, 0x99,0x81,0x1a,0xb8,
				0xa1,0xe0,0xf1,0x5a, 0xbc,0x08,0xde,0xab,
				0x69,0x0a,0x89,0xa0, 0x9f,0x02,0x5e,0x3a,
				0xf3,0xba,0xb9,0x6e, 0x34,0xdf,0x15,0x13,
				0x64,0x51,0xa9,0x55, 0x67,0xa3,0xba,0x6b,
				0x35,0xb0,0x8a,0x05, 0xf5,0x79,0x84,0x97,
				0x92,0x8e,0x11,0xeb, 0xef,0xec,0x65,0xb5,
				0xe6,0x42,0xfb,0x06, 0x33,0x93,0x6b,0xff,
				0xc2,0x49,0x15,0x71, 0xb0,0xca,0x62,0xd1,
				0x81,0x40,0xd2,0xab, 0x0b,0x7d,0x7e,0x1a,
				0xe9,0xec,0xfc,0xde, 0xdb,0xd5,0xa7,0x56,
				0x83,0x25,0x0e,0x5e, 0xac,0x0c,0x42,0x26,
				0x00,0x59,0x55,0x17, 0x8b,0x5a,0x03,0x7b,
				0x85,0xe9,0xc1,0xa3, 0xe4,0xeb,0xd3,0xde,
				0xd8,0x81,0xf5,0x31, 0x2c,0xda,0x21,0xbc,
				0xb5,0xd9,0x7a,0xd0, 0x1e,0x2a,0x6b,0xcf,
				0xad,0x06,0x3c,0xf2, 0xf7,0x5c,0x3a,0xf1,
				0xa7,0x0f,0x5f,0x53, 0xe9,0x3f,0x3c,0xf1,
				0xb7,0x47,0x53,0x16, 0x19,0xd9,0xef,0xf0,
				0xcb,0x16,0xe4,0xc9, 0xa3,0x8f,0xd6,0x3f,
				0xf8,0xb2,0x22,0x65, 0xf9,0xa1,0xa3,0x03,
				0xe4,0x06,0x75,0x69, 0xf5,0x32,0x48,0x80,
			},
			.c = {
				0x66,0x3f,0xf7,0x7a, 0x20,0xa4,0x35,0xd6,
				0x0e,0xe8,0x17,0x32, 0x84,0xae,0xee,0x18,
				0x0f,0x64,0x83,0x66, 0xa4,0xf4,0x24,0x53,
				0xe6,0x58,0x2e,0xd5, 0x61,0x58,0xdd,0x5f,
				0x1d,0xb9,0xba,0x34, 0xd0,0xd3,0x64,0xde,
				0x99,0x47,0x92,0x3a, 0x26,0x90,0xbb,0x98,
				0xb0,0xbd,0xf4,0x5e, 0x26,0x57,0xe0,0xe1,
				0x09,0x27,0xc1,0xc4, 0x86,0x2b,0x4b,0x48,
				0xbb,0xcd,0xec,0x2f, 0xd1,0x54,0xe9,0x21,
				0xa0,0x40,0x76,0x01, 0x2d,0xb1,0xe7,0x75,
				0xa1,0xd7,0x04,0x23, 0x9d,0xd3,0x0f,0x3b,
				0x7e,0xb8,0xd0,0x37, 0xe4,0xd9,0x48,0xaa,
				0xe1,0x4d,0x0f,0xf6, 0xae,0x29,0x20,0xae,
				0xda,0x35,0x18,0x97, 0x2c,0xc2,0xa9,0xdd,
				0x6e,0x50,0x73,0x52, 0x0a,0x8a,0x2a,0xd2,
				0x2a,0xf4,0x12,0xe9, 0x7d,0x88,0x37,0xae,
				0x12,0x81,0x92,0x96, 0xbe,0xea,0x15,0xa4,
				0x3c,0x53,0xad,0x1f, 0x75,0x54,0x24,0x81,
				0xaa,0x1b,0x92,0x84, 0x7c,0xb2,0xd7,0x10,
				0x5e,0xb6,0xab,0x83, 0x25,0xf7,0x03,0x2b,
				0xd9,0x53,0x4d,0xf9, 0x41,0x21,0xef,0xef,
				0x40,0x3a,0x2d,0x54, 0xa9,0xf0,0x72,0xff,
				0x03,0x59,0x2e,0x91, 0x07,0xff,0xe2,0x86,
				0x33,0x59,0x98,0xdf, 0xa4,0x7d,0x9e,0x52,
				0x95,0xd9,0x77,0x4b, 0xdf,0x93,0xc8,0x2d,
				0xbc,0x81,0x2b,0x77, 0x89,0xae,0x52,0xdc,
				0xfc,0xb7,0x22,0xf0, 0x1a,0x9d,0xc1,0x28,
				0x70,0xe2,0x15,0xe4, 0x77,0x11,0x49,0x09,
				0x89,0xf4,0x06,0x00, 0x64,0x78,0xb6,0x3f,
				0x63,0x36,0xfd,0x9f, 0x35,0x33,0x85,0x52,
				0x18,0x26,0xc1,0x0d, 0xf7,0xab,0x5a,0x06,
				0x9c,0x3a,0xab,0x5f, 0x81,0x36,0x39,0xe3,
				0xe6,0xf7,0x33,0xb0, 0xec,0xe6,0x8d,0x05,
				0xbd,0xc7,0xbd,0x20, 0x5f,0x74,0xdf,0x98,
				0x3a,0xa9,0xde,0xae, 0x89,0xee,0xcc,0x60,
				0x8b,0x23,0xed,0x0f, 0x55,0x4d,0x56,0xd2,
				0x69,0xa5,0xf8,0xff, 0x94,0x62,0x99,0xc6,
				0xd4,0x02,0x0b,0xcf, 0xe4,0x86,0x23,0x5e,
				0xed,0x12,0x12,0x2e, 0x0a,0x0f,0xda,0x12,
				0x0a,0x68,0x56,0xea, 0x16,0x92,0xa5,0xdb,
				0xf5,0x9d,0x0e,0xe6, 0x39,0x5d,0x76,0x50,
				0x41,0x85,0xb4,0xcc, 0xb3,0x9e,0x84,0x46,
				0xd3,0x93,0xcf,0xa1, 0xee,0x5b,0x51,0x94,
				0x05,0x46,0x16,0xbb, 0xd1,0xae,0x94,0xe4,
				0x1c,0x3d,0xeb,0xf4, 0x09,0x00,0xf7,0x86,
				0x57,0x60,0x49,0x94, 0xf5,0xa7,0x7e,0x4b,
				0x32,0x4a,0x6a,0xae, 0x2c,0x5f,0x30,0x2d,
				0x7c,0xa1,0x71,0x5e, 0x63,0x7a,0x70,0x56,
				0x1f,0xaf,0x3e,0xf3, 0x46,0xb5,0x68,0x61,
				0xe2,0xd4,0x16,0x6b, 0xaf,0x94,0x07,0xa9,
				0x5d,0x7a,0xee,0x4c, 0xad,0x85,0xcc,0x3e,
				0x99,0xf3,0xfa,0x21, 0xab,0x9d,0x12,0xdf,
				0x33,0x32,0x23,0x68, 0x96,0x8f,0x8f,0x78,
				0xb3,0x63,0xa0,0x83, 0x16,0x06,0x64,0xbd,
				0xea,0x1f,0x69,0x73, 0x9c,0x54,0xe1,0x60,
				0xe8,0x98,0xc9,0x94, 0xe9,0xdf,0x0c,0xee,
				0xf4,0x38,0x1e,0x9f, 0x26,0xda,0x3f,0x4c,
				0xfd,0x6d,0xf5,0xee, 0x75,0x91,0x7c,0x4f,
				0x4d,0xc2,0xe8,0x1a, 0x7b,0x1b,0xa9,0x52,
				0x1e,0x24,0x22,0x5a, 0x73,0xa5,0x10,0xa2,
				0x37,0x39,0x1e,0xd2, 0xf7,0xe0,0xab,0x77,
				0xb7,0x93,0x5d,0x30, 0xd2,0x5a,0x33,0xf4,
				0x63,0x98,0xe8,0x6d, 0x3f,0x34,0x4a,0xb9,
				0x44,0x57,0x39,0xe7, 0xa9,0xdd,0xac,0x91,
			},
		},
		[3] = {
			.k = {
				0xac,0x95,0xec,0x00, 0xa5,0x57,0x8e,0x99,
				0x14,0x54,0x95,0x60, 0xdc,0xae,0x56,0x66,
				0x03,0x22,0xa1,0x55, 0xbf,0xa5,0x2b,0x1c,
				0x02,0xc9,0x0c,0x2f, 0xa1,0x5d,0x1b,0x84,
			},
			.tlen = 0,
			.len = 1536,
			.p = {
				0xd2,0x80,0x06,0x95, 0xcd,0xe1,0x71,0x2c,
				0xcf,0x89,0xa6,0xc7, 0x8b,0xa7,0xe3,0xcb,
				0x66,0x3e,0x6b,0x58, 0x2a,0x20,0xd1,0xc4,
				0x07,0xd6,0x3b,0x03, 0xdc,0x26,0xda,0x1b,
				0xe0,0x51,0xd5,0x1c, 0x4c,0xed,0xd0,0xf5,
				0xe2,0x7f,0x89,0xe8, 0x3d,0x41,0x1a,0xa0,
				0xb1,0xed,0x61,0xa8, 0xc7,0x0a,0xe8,0x69,
				0x4d,0xb8,0x18,0x81, 0x6c,0x76,0x67,0x83,
				0x8a,0x47,0xa2,0x4b, 0xfb,0xfd,0x6f,0x65,
				0x88,0xa8,0xf6,0x6d, 0x9f,0x71,0x6e,0x33,
				0x4f,0x82,0xee,0x8f, 0x38,0x5c,0xe4,0x9b,
				0x45,0x29,0xca,0xda, 0x9b,0x5d,0x65,0x06,
				0xab,0xf5,0x86,0x28, 0x8c,0x3e,0x20,0x38,
				0x1a,0x4c,0xb2,0xd9, 0x1f,0xc0,0x10,0x59,
				0x6b,0x2c,0xb5,0x41, 0x41,0xc5,0xd9,0xb7,
				0x4f,0xc3,0x36,0x08, 0xd4,0xdc,0xff,0x57,
				0xd7,0x97,0x77,0x45, 0xc4,0x28,0x93,0x2c,
				0xbe,0xdc,0xae,0x1d, 0x18,0xc8,0xfa,0x9a,
				0xd4,0x41,0x2e,0x5a, 0x26,0x03,0xae,0x7a,
				0xb2,0x6a,0xc0,0x0c, 0xb6,0x3e,0xf0,0x73,
				0x36,0xed,0xea,0xc1, 0xae,0x9d,0xc9,0xa1,
				0x85,0x4c,0x57,0x14, 0xb0,0xf3,0xf8,0x4e,
				0x91,0x99,0x06,0x65, 0x17,0x66,0xc2,0x9a,
				0x7a,0x4f,0x39,0x77, 0x32,0x44,0xc8,0x3f,
				0xe2,0x3c,0xc2,0x31, 0x0b,0x40,0x84,0xee,
				0xa1,0xeb,0xc6,0xc2, 0xb4,0x48,0xe6,0x09,
				0xc5,0xf5,0x3d,0x96, 0x90,0xa2,0x1d,0xf2,
				0x89,0x26,0x9f,0x10, 0x49,0x30,0x0f,0xe1,
				0x5e,0xca,0x1c,0x3f, 0x82,0xda,0xcb,0x8d,
				0x91,0x6d,0x08,0x96, 0x9e,0x57,0x88,0x16,
				0xee,0xa7,0x9e,0xe8, 0x1b,0xc1,0x63,0xb0,
				0x57,0xfa,0xfd,0x56, 0x49,0xec,0x51,0x1d,
				0x34,0x2e,0xc6,0xda, 0xc0,0x1d,0x02,0x3e,
				0x52,0xaf,0x44,0x24, 0xc6,0x80,0x12,0x64,
				0xbe,0x44,0xa8,0x46, 0xb5,0x8d,0x80,0xfd,
				0x95,0x4a,0xeb,0x3d, 0x4f,0x85,0x1f,0x1c,
				0xa4,0x3f,0x5c,0x0c, 0x71,0xed,0x96,0x41,
				0xde,0xb0,0xbd,0x08, 0xf3,0x4d,0x37,0xd2,
				0xb1,0x4f,0x71,0x04, 0xf1,0x14,0x66,0x4a,
				0x59,0x73,0xdc,0x98, 0x5b,0x61,0x56,0xfd,
				0x50,0xe5,0x76,0xd9, 0x6a,0x9f,0x30,0x82,
				0x6f,0xdf,0x6e,0x7b, 0x91,0xc2,0x5e,0x4f,
				0x74,0x92,0x92,0xb8, 0x24,0xd3,0x30,0x21,
				0x5d,0x4b,0xb1,0x01, 0xf7,0x62,0x27,0x94,
				0xb3,0x88,0x86,0x75, 0xe8,0xab,0xe8,0x42,
				0x50,0x15,0xb7,0xde, 0xc0,0xc4,0x8d,0x4e,
				0x08,0x17,0xcb,0xf9, 0x4a,0x2e,0xe3,0x69,
				0xbd,0xe7,0xdb,0xd1, 0xf1,0xfa,0x47,0xed,
				0x78,0xa9,0x26,0xf0, 0xd1,0xbb,0x02,0xa1,
				0x07,0x5c,0x1f,0xe8, 0x2f,0x52,0xd8,0x95,
				0xd7,0xa9,0x2b,0x79, 0x77,0xf4,0xee,0xee,
				0xbc,0x1f,0xaa,0x46, 0xe7,0x66,0x75,0xb1,
				0x43,0x01,0x35,0xac, 0xc6,0x85,0xad,0x44,
				0x23,0x59,0x50,0x0b, 0x39,0x47,0x51,0x54,
				0x68,0x92,0x89,0x00, 0x08,0xa3,0xaa,0x24,
				0x03,0x3f,0xf6,0xab, 0x19,0x42,0xff,0x0c,
				0xc5,0xa3,0x96,0xcb, 0xd9,0x6d,0xa0,0xcc,
				0x24,0x9e,0x71,0xb1, 0x87,0x95,0x7a,0x2e,
				0x31,0x5e,0x17,0x26, 0x5a,0x1b,0xa1,0x33,
				0x10,0x3f,0xd7,0xce, 0xa0,0xd9,0xbc,0xd8,
				0x72,0xbe,0x75,0xc4, 0x78,0x3b,0x67,0xf5,
				0xc3,0x82,0x2d,0x21, 0x49,0x74,0x2e,0xd5,
				0x63,0xaa,0xa2,0x54, 0xc5,0xe2,0x98,0x82,
				0x39,0xd9,0xda,0x14, 0x3c,0x75,0x18,0xc8,
				0x75,0x6a,0xa1,0x7d, 0xfa,0x72,0x0f,0x9b,
				0x5a,0xb3,0x7c,0x15, 0xc2,0xa5,0x6d,0x98,
				0x02,0x6c,0xa2,0x26, 0xaa,0xc0,0x69,0xc5,
				0xa7,0xa2,0xca,0xf5, 0xf3,0x8c,0x80,0x4e,
				0x7e,0x47,0xc9,0x87, 0x47,0x36,0xd6,0xc6,
				0xe8,0x49,0xb5,0x97, 0xa8,0xdc,0x4a,0x55,
				0x6f,0x02,0x79,0x83, 0xe4,0x7c,0x4c,0x69,
				0xa6,0x4d,0x4f,0x8a, 0x48,0x18,0x00,0xf9,
				0xad,0xd1,0xb2,0xca, 0xc4,0x50,0x47,0x21,
				0x4e,0xa7,0xce,0x6e, 0xdf,0xbd,0x2a,0x4d,
				0xca,0x13,0x33,0xde, 0xa2,0x30,0xe1,0x03,
				0xcd,0x2c,0x74,0xd3, 0x30,0x0d,0x61,0xe6,
				0x9d,0xf3,0x09,0xc5, 0x27,0x99,0x0e,0x23,
				0xbc,0x21,0xdb,0xdb, 0xeb,0x77,0xea,0xd4,
				0x4b,0xbf,0x9b,0x49, 0x30,0xd4,0xc2,0xe7,
				0x5e,0x85,0xe8,0xb6, 0xa5,0xe3,0x4e,0x64,
				0xf0,0x45,0x95,0x04, 0x9a,0xed,0xaa,0x4d,
				0xbd,0x5e,0x03,0x9f, 0xd4,0x2b,0xae,0x14,
				0x1a,0x3d,0x49,0x92, 0xd6,0x6f,0x64,0xc7,
				0xca,0x18,0x32,0x16, 0xf6,0x07,0x00,0x22,
				0xfd,0xe1,0x45,0xe6, 0x19,0x24,0x5b,0x6e,
				0xd3,0x67,0xf2,0x60, 0x36,0xf5,0x22,0xeb,
				0x5f,0x42,0xba,0x70, 0x38,0xfc,0x98,0x96,
				0x58,0x72,0xbf,0x13, 0x60,0xcc,0x32,0x45,
				0x8d,0x00,0x44,0x60, 0xaf,0x7a,0x19,0xd6,
				0xc0,0x14,0x33,0x96, 0xf3,0x33,0xc3,0xa8,
				0x34,0x77,0x69,0x0c, 0x50,0xe5,0xfc,0x1b,
				0x42,0x39,0x96,0x24, 0x3a,0x3a,0x47,0x0e,
				0x27,0x66,0xa8,0x18, 0x50,0xdf,0x6d,0xa7,
				0xad,0x4f,0xe5,0x88, 0x79,0xea,0x30,0xe2,
				0xcd,0x27,0x05,0x36, 0x0c,0x3c,0x97,0x12,
				0x69,0xa6,0xc0,0xa2, 0xa7,0x58,0x82,0x20,
				0x68,0xfc,0xd0,0x81, 0x49,0xc0,0xcf,0xba,
				0x90,0xe1,0x03,0xce, 0x70,0xd6,0x94,0x1a,
				0xc0,0x22,0x3b,0xdc, 0x7f,0x63,0x6b,0xc4,
				0x91,0xc2,0x21,0xdc, 0x84,0x42,0x80,0x04,
				0x6f,0x14,0xc3,0x2c, 0x79,0x49,0x3c,0xb1,
				0x5f,0xc7,0x69,0x4a, 0x4f,0xf5,0xd5,0x4b,
				0x7c,0xe7,0x83,0x79, 0x30,0xff,0x74,0xe0,
				0xf7,0xd3,0x6c,0x95, 0xef,0x77,0xe8,0x7b,
				0x1f,0x54,0xad,0xc7, 0x4b,0xe8,0x5a,0x37,
				0xd7,0xe9,0xfe,0xcb, 0x11,0x7b,0x54,0xb8,
				0xd2,0xc7,0x80,0x1d, 0x80,0x17,0xdd,0x21,
				0xa6,0xed,0x20,0x2c, 0x8a,0xa1,0x0b,0x3a,
				0x08,0xde,0x34,0xe4, 0xa0,0xff,0x68,0xfa,
				0x4a,0x01,0xcc,0x4f, 0x57,0x5f,0x84,0x95,
				0x88,0xe2,0x7f,0xb7, 0x5d,0x35,0x36,0xe2,
				0xa1,0xca,0xc0,0x9b, 0x4a,0xb0,0x6f,0x35,
				0xef,0x08,0xd7,0x5a, 0xec,0x4f,0x97,0x20,
				0x92,0x2a,0x63,0x1d, 0x15,0x07,0x73,0x1f,
				0x97,0xcf,0x28,0x41, 0x65,0x0d,0x41,0xee,
				0xca,0xd8,0x90,0x65, 0xaa,0x3d,0x04,0x7f,
				0x35,0x4b,0x9e,0xe9, 0x96,0xa9,0x61,0xcb,
				0x43,0xc9,0xfa,0x1d, 0xc8,0x85,0x40,0x64,
				0x88,0x89,0xea,0xb5, 0xf7,0xe5,0xe4,0xfe,
				0xaf,0x8e,0x52,0xf9, 0x7e,0x7d,0x83,0x92,
				0x90,0x51,0x4c,0xf0, 0x49,0x52,0x5e,0x56,
				0xc9,0xb7,0x4c,0xca, 0x57,0x01,0x3d,0x28,
				0xe2,0x7d,0xaa,0x96, 0xd7,0xad,0xad,0xd9,
				0xd5,0x1a,0xd5,0xc2, 0xd0,0x5a,0xd3,0x7a,
				0x9a,0x91,0xa0,0xb8, 0x6f,0x28,0xff,0xa0,
				0x1c,0x1d,0xf1,0x5e, 0x45,0x53,0x3f,0x85,
				0x1b,0xc2,0x76,0x51, 0xbf,0x25,0x02,0xf7,
				0x10,0xde,0xb7,0x1a, 0x04,0x6c,0x9a,0xeb,
				0xb9,0x4b,0x67,0xfb, 0xa1,0x5b,0xa8,0x02,
				0x01,0x1f,0x38,0xa9, 0x9d,0x96,0x50,0x07,
				0xef,0xa7,0xc3,0xb4, 0x0f,0xcd,0x1b,0x9f,
				0xd2,0x08,0x87,0xca, 0xd5,0x65,0x1a,0x5e,
				0x1a,0xff,0x97,0xb0, 0x4b,0x43,0x67,0x51,
				0x22,0xfd,0x49,0xcd, 0x54,0x2f,0xf8,0x9b,
				0xed,0x46,0x7e,0x00, 0x5b,0x67,0x06,0xeb,
				0xb7,0x4d,0x1c,0x72, 0x74,0xdd,0xbd,0xb1,
				0x71,0x0a,0x28,0xc7, 0x7b,0xa8,0x12,0xac,
				0x58,0x53,0xa4,0xfb, 0x41,0x74,0xb4,0x52,
				0x95,0x99,0xf6,0x38, 0x53,0xff,0x2d,0x26,
				0xef,0x12,0x91,0xc6, 0x52,0xe1,0xa9,0x50,
				0xfa,0x8e,0x2e,0x82, 0x8b,0x4f,0xb7,0xad,
				0xe1,0x74,0x0d,0xbf, 0x73,0x04,0xdf,0x3f,
				0xf6,0xf8,0x09,0x9d, 0xdf,0x18,0x07,0x13,
				0xe6,0x60,0xf0,0x6a, 0x98,0x22,0x15,0xdf,
				0x0c,0x72,0x6a,0x9d, 0x6e,0x67,0x76,0x61,
				0xda,0xbe,0x10,0xd6, 0xf0,0x5f,0x06,0x74,
				0x76,0xce,0x63,0xee, 0x91,0x39,0x24,0xa9,
				0xcf,0xc7,0xca,0xd5, 0xb4,0xff,0x30,0x6e,
				0x05,0x32,0x0c,0x9d, 0xeb,0xfb,0xc6,0x3e,
				0xe4,0xc6,0x20,0xc5, 0x3e,0x1d,0x5c,0xd6,
				0x05,0xbe,0xb8,0xc3, 0x44,0xe3,0xc9,0xc1,
				0x38,0xaa,0xc5,0xc8, 0xe3,0x11,0x8d,0xde,
				0xdc,0x48,0x8e,0xe9, 0x38,0xe5,0x80,0xec,
				0x82,0x17,0xf2,0xcf, 0x26,0x55,0xf7,0xdc,
				0x78,0x7f,0xfb,0xc1, 0xb4,0x6c,0x80,0xcc,
				0xf8,0x5a,0xbc,0x8f, 0x9d,0x62,0xfe,0x35,
				0x17,0x7c,0x10,0xb7, 0x4a,0x0f,0x81,0x43,
				0x11,0xbd,0x33,0x47, 0x9c,0x61,0x02,0xec,
				0xab,0xde,0xb2,0x3f, 0x73,0x48,0xfb,0x5c,
				0x84,0x4a,0xeb,0xab, 0x58,0x07,0x18,0xdc,
				0x57,0x85,0xb8,0xe7, 0xff,0x9c,0xc2,0xc8,
				0xb3,0xef,0x5b,0x50, 0x16,0xb1,0x38,0x6e,
				0xa7,0xd7,0x9c,0xb1, 0x29,0x6b,0x74,0x9c,
				0x50,0xcc,0x90,0xee, 0x86,0x2a,0x7c,0x07,
				0xd4,0xcb,0xc2,0x24, 0x53,0xb0,0x3f,0x4f,
				0x9b,0xc4,0x62,0x73, 0x85,0x3d,0x1e,0x54,
				0x86,0xda,0x1e,0x5e, 0x70,0x73,0x6a,0x2a,
				0x29,0x75,0xb7,0x18, 0x1a,0x72,0x81,0x64,
				0x58,0xa0,0xb3,0x70, 0x61,0x9f,0x22,0x37,
				0xac,0xdc,0xe8,0xaf, 0xe2,0x74,0xe4,0xa7,
				0xed,0x92,0x5c,0x47, 0xff,0xc3,0xaf,0x9e,
				0x59,0xe1,0x09,0x22, 0x72,0x18,0x96,0x35,
				0x23,0x91,0x00,0xa3, 0x7d,0x95,0x25,0x95,
				0xd5,0xad,0xf8,0x6e, 0xcc,0x14,0x31,0xb2,
				0x52,0x20,0x2a,0x41, 0xf1,0xaf,0x9a,0xaf,
				0xdd,0xbd,0x04,0x5a, 0xcd,0x1a,0x86,0xb1,
				0x45,0x1b,0x6f,0x7a, 0x02,0x45,0x05,0xef,
				0x74,0xdf,0xe8,0x72, 0x1c,0x82,0x57,0xea,
				0x2a,0x24,0x1b,0x46, 0x3f,0x66,0x89,0x9f,
				0x00,0xb9,0xec,0xf7, 0x59,0x6d,0xeb,0xac,
				0xca,0x82,0x14,0x79, 0xbf,0x7f,0xd5,0x18,
				0x26,0x6b,0xee,0x34, 0x44,0xee,0x6d,0x8a,
				0x82,0x8f,0x4f,0xa3, 0x1a,0xc3,0x9b,0x2e,
				0x57,0x83,0xb8,0x7d, 0xa0,0x21,0xc6,0x66,
				0x96,0x7d,0x30,0x81, 0x29,0xc7,0x05,0x46,
				0x99,0xd4,0x35,0x7b, 0x40,0xe8,0x87,0x60,
				0x13,0xa5,0xa6,0xb9, 0x24,0x59,0xca,0xa8,
				0xcd,0x62,0xeb,0xc5, 0x22,0xff,0x49,0x64,
				0x03,0x2d,0x42,0x01, 0xa2,0x09,0x4a,0x45,
				0x41,0x34,0x88,0x44, 0xf4,0xe1,0xa3,0x48,
				0xcf,0x2d,0xee,0xee, 0xbf,0x83,0x1a,0x42,
				0x8d,0xa4,0x15,0x3d, 0xfc,0x92,0x67,0x91,
			},
			.c = {
				0x5c,0xb9,0xab,0x7c, 0xe4,0x0b,0xbe,0xa5,
				0x17,0x18,0xdf,0xd7, 0x17,0x13,0x98,0xbd,
				0xcb,0x1c,0xa3,0x39, 0x9c,0xbc,0x19,0x1f,
				0xca,0xcb,0x50,0x89, 0x1d,0x69,0xc3,0xcb,
				0xd1,0x76,0x70,0x6b, 0x7c,0x62,0x49,0xe8,
				0xb1,0xa8,0xb7,0x58, 0x87,0xf6,0x79,0xf7,
				0xf2,0xc1,0xd8,0xb2, 0x1d,0xd2,0x1a,0xf5,
				0xa0,0x41,0xda,0x17, 0x3f,0xaa,0xdb,0xf6,
				0xa9,0xf2,0x49,0x1c, 0x6f,0x20,0xf3,0xae,
				0x4a,0x5e,0x55,0xdd, 0xa6,0x9e,0xc4,0x03,
				0x07,0x22,0xc0,0xbe, 0x5e,0x58,0xdd,0xf0,
				0x7e,0xfe,0xcf,0x2c, 0x96,0x33,0x32,0xbd,
				0xe8,0xdf,0x84,0x71, 0x45,0x35,0x40,0x48,
				0xcf,0x10,0x45,0x47, 0x97,0x4c,0x20,0x6b,
				0x3a,0xdd,0x73,0xd0, 0xce,0x0c,0x4c,0xf1,
				0x78,0xcd,0x93,0xd2, 0x21,0x70,0xeb,0x2f,
				0x23,0x99,0x64,0xbb, 0x97,0x28,0xe9,0xde,
				0xef,0x9c,0xf2,0x7f, 0x4b,0x4d,0x2c,0x66,
				0x7b,0x6e,0x70,0xf7, 0x25,0x68,0xea,0x93,
				0x3a,0x27,0xbd,0x04, 0x8b,0xcd,0xd9,0xed,
				0x1a,0x9d,0xca,0x8f, 0x15,0x2d,0xa1,0x25,
				0xb8,0x66,0x1b,0x3d, 0xd4,0xd4,0x9b,0xab,
				0x3a,0xa8,0xe8,0x88, 0xc6,0xd2,0x5a,0x28,
				0x51,0x4d,0x11,0xb6, 0x4a,0x2b,0x6d,0xe4,
				0xc9,0xc1,0x20,0x6f, 0xba,0x23,0x72,0xc9,
				0x6d,0x44,0xf0,0xaa, 0x06,0x8c,0x9b,0xbb,
				0x4b,0xd2,0xa0,0x94, 0x5f,0x0b,0xc8,0xa3,
				0x4c,0xe9,0xe2,0x8a, 0xe5,0xf9,0xe3,0x2c,
				0xc7,0x87,0x75,0xc1, 0xc9,0x62,0xb5,0xb4,
				0x04,0x86,0x6a,0x31, 0x54,0x0e,0x31,0xf7,
				0xad,0xea,0xbb,0xa6, 0x8e,0x6c,0xac,0x24,
				0x52,0x2c,0x9d,0x1f, 0xde,0x70,0xfd,0xc4,
				0x93,0x8b,0x75,0x6c, 0xef,0xa7,0x89,0xaf,
				0x2c,0x4c,0xf6,0x38, 0xdd,0x79,0xfa,0x70,
				0x54,0x1e,0x92,0xd4, 0xb4,0x04,0x69,0x8e,
				0x6b,0x9e,0x12,0xfe, 0x15,0x15,0xf7,0x99,
				0xb6,0x2f,0xfc,0xfa, 0x66,0xe9,0x40,0xb5,
				0xd3,0x10,0xbb,0x42, 0xf9,0x68,0x64,0xd4,
				0x2a,0xcd,0x43,0x75, 0xb0,0x9c,0x61,0x34,
				0xc1,0xc4,0x42,0xf3, 0xf1,0xa7,0x65,0xf4,
				0xcb,0x42,0xe9,0xc2, 0x5a,0x05,0xdf,0x98,
				0xa3,0xba,0xf7,0xe0, 0x15,0xa1,0xdf,0xf7,
				0xce,0xd5,0xf0,0x62, 0x89,0xe1,0x44,0x3a,
				0x4f,0x6f,0x75,0x3e, 0xfc,0x19,0xe3,0x5f,
				0x36,0x48,0xc1,0x95, 0x08,0x22,0x09,0xf9,
				0x07,0x74,0x1c,0xa4, 0x1b,0x7e,0xa8,0x82,
				0xca,0x0b,0xd9,0x1e, 0xe3,0x5b,0x1c,0xb5,
				0x57,0x13,0x7d,0xbd, 0xbd,0x16,0x88,0xd4,
				0xb1,0x8e,0xdb,0x6f, 0x2f,0x7b,0x55,0x72,
				0x79,0xc9,0x49,0x7b, 0xf7,0x86,0xa9,0x3d,
				0x2d,0x11,0x33,0x7d, 0x82,0x38,0xc7,0xb5,
				0x7c,0x6b,0x0b,0x28, 0x42,0x50,0x47,0x69,
				0xd8,0x48,0xc6,0x85, 0x0b,0x1b,0xca,0x08,
				0x85,0x36,0x6d,0x97, 0xe9,0x3e,0xeb,0xe2,
				0x28,0x6a,0x17,0x61, 0x7d,0xcb,0xb6,0xb3,
				0x23,0x44,0x76,0xd3, 0x57,0x39,0x9b,0x1d,
				0x69,0x30,0xd8,0x3f, 0x21,0xe8,0x68,0x94,
				0x82,0x85,0x97,0xb1, 0x1f,0x0c,0x99,0x6e,
				0x6e,0x44,0xa6,0x82, 0xd0,0xa2,0xe6,0xfe,
				0xff,0x08,0x41,0x49, 0x54,0x18,0x51,0x88,
				0x23,0xd5,0x14,0xbd, 0xfe,0xea,0x5d,0x15,
				0xd4,0x0b,0x2d,0x92, 0x94,0x8d,0xd4,0xe5,
				0xaf,0x60,0x88,0x2b, 0x67,0xae,0xbb,0xa8,
				0xec,0xae,0x9b,0x35, 0xa2,0xd7,0xe8,0xb6,
				0xe5,0xaa,0x12,0xd5, 0xef,0x05,0x5a,0x64,
				0xe0,0xff,0x79,0x16, 0xb6,0xa3,0xdb,0x1e,
				0xee,0xe8,0xb7,0xd6, 0x71,0xbd,0x76,0xbf,
				0x66,0x2a,0x9c,0xec, 0xbe,0x8c,0xb5,0x8e,
				0x8e,0xc0,0x89,0x07, 0x5d,0x22,0xd8,0xe0,
				0x27,0xcf,0x58,0x8a, 0x8c,0x4d,0xc7,0xa4,
				0x45,0xfc,0xe5,0xa4, 0x32,0x7c,0xbf,0x86,
				0xf0,0x82,0x96,0x05, 0x1e,0x86,0x03,0x0f,
				0x1f,0x0d,0xf2,0xfc, 0x28,0x62,0x90,0x53,
				0xfe,0xd4,0x28,0x52, 0x4f,0xa6,0xbc,0x4d,
				0xba,0x5d,0x04,0xc0, 0x83,0x61,0xf6,0x41,
				0xc8,0x58,0x40,0x49, 0x1d,0x27,0xd5,0x9f,
				0x93,0x4f,0xb5,0x7a, 0xea,0x7b,0x86,0x31,
				0x2b,0xe5,0x92,0x51, 0x3e,0x7a,0xbe,0xdb,
				0x04,0xae,0x21,0x71, 0x5a,0x70,0xf9,0x9b,
				0xa8,0xb6,0xdb,0xcd, 0x21,0x56,0x75,0x2e,
				0x98,0x38,0x78,0x4d, 0x51,0x4a,0xa6,0x03,
				0x8a,0x84,0xb2,0xf9, 0x6b,0x98,0x6d,0xf3,
				0x12,0xaa,0xd4,0xea, 0xb3,0x7c,0xb0,0xd9,
				0x5e,0x1c,0xb0,0x69, 0x48,0x67,0x13,0x26,
				0xf0,0x25,0x04,0x93, 0x6d,0xc6,0x6c,0xb2,
				0xcd,0x7c,0x36,0x62, 0x6d,0x38,0x44,0xe9,
				0x6b,0xe2,0x7f,0xc1, 0x40,0xdb,0x55,0xe1,
				0xa6,0x71,0x94,0x0a, 0x13,0x5f,0x9e,0x66,
				0x3b,0xb3,0x11,0x90, 0xbb,0x68,0xd4,0x11,
				0xf2,0xb7,0x61,0xbd, 0xac,0x4a,0x56,0xf4,
				0x9e,0xe2,0xd0,0x1e, 0xb4,0xa1,0xb8,0x4e,
				0xbb,0xc2,0x73,0x63, 0x04,0x99,0x97,0x9f,
				0x76,0x18,0x82,0x11, 0x7e,0xe1,0xcc,0x58,
				0xb7,0xb5,0x37,0x78, 0x60,0x19,0x6c,0x2b,
				0x6e,0x65,0x15,0x10, 0x3c,0x93,0xf0,0xc5,
				0x3d,0x9e,0xeb,0x77, 0x72,0x25,0x95,0xf0,
				0x27,0xe8,0xbd,0x81, 0x9c,0x22,0x38,0xa7,
				0x8d,0xe9,0x94,0xf2, 0x27,0x8d,0x3a,0x34,
				0x36,0xba,0x26,0xa0, 0xd7,0x3e,0xd8,0xbe,
				0x60,0xd1,0x53,0x58, 0x56,0xe6,0xf3,0xa1,
				0x0d,0x62,0x5e,0x44, 0xd3,0x7c,0xc9,0x25,
				0x87,0xc8,0x1a,0x57, 0x7f,0xfa,0x79,0x4a,
				0x15,0xf6,0x3e,0x2e, 0xd0,0x6b,0x83,0x9b,
				0xe6,0xfe,0x6c,0xd3, 0x8e,0x40,0x4a,0x12,
				0x57,0x41,0xc9,0x5a, 0x42,0x91,0x0b,0x28,
				0x56,0x38,0xfc,0x45, 0x4b,0x26,0xbf,0x3a,
				0xa3,0x46,0x75,0x73, 0xde,0x7e,0x18,0x7c,
				0x82,0x92,0x73,0xe6, 0xb5,0xd2,0x1f,0x1c,
				0xdd,0xb3,0xd5,0x71, 0x9f,0xd2,0xa5,0xf4,
				0xf1,0xcb,0xfe,0xfb, 0xd3,0xb6,0x32,0xbd,
				0x8e,0x0d,0x73,0x0a, 0xb6,0xb1,0xfd,0x31,
				0xa5,0xa4,0x7a,0xb1, 0xa1,0xbb,0xf0,0x0b,
				0x97,0x21,0x27,0xe1, 0xbb,0x6a,0x2a,0x5b,
				0x95,0xda,0x01,0xd3, 0x06,0x8e,0x53,0xd8,
				0x23,0xa3,0xa9,0x82, 0x8a,0xa2,0x8f,0xdb,
				0x87,0x37,0x41,0x41, 0x2b,0x36,0xf3,0xb3,
				0xa6,0x32,0x5f,0x3e, 0xbf,0x70,0x3a,0x13,
				0xba,0x11,0xa1,0x4e, 0x11,0xa8,0xc0,0xb7,
				0xb2,0x1b,0xab,0xc8, 0xcb,0x38,0x35,0x2e,
				0x76,0xa7,0x0b,0x5a, 0x6c,0x53,0x83,0x60,
				0x4f,0xee,0x91,0xe8, 0xca,0x1e,0x7f,0x76,
				0x2b,0x4c,0xe7,0xd4, 0xcb,0xf8,0xeb,0x94,
				0x76,0x17,0x68,0x23, 0x95,0x93,0x7f,0x60,
				0x80,0x7a,0x85,0x70, 0x95,0x56,0xb9,0x76,
				0x76,0xb6,0x8f,0xe2, 0x93,0x60,0xfc,0x70,
				0x57,0x4a,0x27,0xc0, 0xfb,0x49,0x2f,0xac,
				0xde,0x87,0x2f,0x1a, 0x80,0xca,0x68,0x5e,
				0xc6,0x18,0x4e,0x3a, 0x4b,0x36,0xdc,0x24,
				0x78,0x7e,0xb0,0x58, 0x85,0x4d,0xa9,0xbc,
				0x0d,0x87,0xdd,0x02, 0xa6,0x0d,0x46,0xae,
				0xf7,0x2f,0x8e,0xeb, 0xf4,0x29,0xe0,0xbc,
				0x9a,0x34,0x30,0xc3, 0x29,0xea,0x2c,0xb3,
				0xb4,0xa2,0x9c,0x45, 0x6e,0xcb,0xa4,0x9d,
				0x22,0xe6,0x71,0xe0, 0xcb,0x9f,0x05,0xef,
				0x2f,0xf7,0x12,0xfd, 0x5d,0x48,0x6c,0x9e,
				0x8b,0xaa,0x90,0xb6, 0xa8,0x78,0xeb,0xde,
				0xeb,0x4c,0xce,0x7b, 0x62,0x60,0x69,0xc0,
				0x54,0xc3,0x13,0x76, 0xdc,0x7e,0xd1,0xc3,
				0x8e,0x24,0x58,0x43, 0x3c,0xbc,0xa0,0x75,
				0xf2,0x7c,0x2d,0x1e, 0x94,0xec,0x40,0x15,
				0xe1,0x78,0xac,0x4a, 0x93,0xef,0x87,0xec,
				0x99,0x94,0xcb,0x65, 0xde,0xcb,0x38,0xd7,
				0x89,0x90,0xa2,0x68, 0xcf,0xfd,0x98,0xf8,
				0x1f,0x06,0xd5,0x6c, 0x53,0x1d,0xd3,0xa7,
				0x06,0x0b,0xa9,0x92, 0xbb,0x6e,0x6f,0xaa,
				0x5a,0x54,0x71,0xb7, 0x90,0x00,0x06,0x6b,
				0xf9,0x34,0xba,0x41, 0x73,0x58,0x98,0xfc,
				0xca,0x98,0xbd,0xd3, 0x7d,0xa4,0x49,0xcc,
				0xa8,0x19,0xc1,0x40, 0x75,0x81,0x02,0x33,
				0xac,0x90,0xcd,0x58, 0xeb,0x1b,0xb4,0x4e,
				0xe0,0x8a,0xa9,0x0f, 0x15,0x8e,0x51,0x85,
				0x06,0x09,0x92,0x40, 0xe3,0x75,0x60,0x64,
				0xcf,0x9b,0x88,0xc7, 0xb0,0xab,0x37,0x5d,
				0x43,0x21,0x18,0x09, 0xff,0xec,0xa0,0xb3,
				0x47,0x09,0x22,0x4c, 0x55,0xc2,0x2d,0x2b,
				0xce,0xb9,0x3a,0xcc, 0xd7,0x0c,0xb2,0x9a,
				0xff,0x2a,0x73,0xac, 0x7a,0xf2,0x11,0x73,
				0x94,0xd9,0xbe,0x31, 0x9f,0xae,0x62,0xab,
				0x03,0xac,0x5f,0xe2, 0x99,0x90,0xfb,0xa5,
				0x74,0xc0,0xfa,0xb9, 0x3c,0x96,0x7c,0x36,
				0x25,0xab,0xff,0x2f, 0x24,0x65,0x73,0x21,
				0xc3,0x21,0x73,0xc9, 0x23,0x06,0x22,0x6c,
				0xb2,0x22,0x26,0x1d, 0x88,0x6f,0xd3,0x5f,
				0x6f,0x4d,0xf0,0x6d, 0x13,0x70,0x7d,0x67,
				0xe8,0x5c,0x3b,0x35, 0x27,0x8a,0x8c,0x65,
				0xae,0x50,0x78,0xe1, 0x26,0x07,0xf8,0x18,
				0xfc,0xea,0xa3,0x58, 0x73,0x2b,0xca,0x92,
				0x10,0xdc,0xb5,0x39, 0xd5,0x2d,0x21,0xfe,
				0x79,0xac,0x7d,0xe8, 0x0c,0xe9,0x6d,0x3e,
				0xb4,0x8a,0x23,0x65, 0x08,0xbc,0x57,0x51,
				0xe1,0xf8,0x8d,0x5b, 0xe4,0xfe,0x14,0x60,
				0x02,0xe7,0xd1,0xc2, 0xd2,0x2c,0x3f,0x4d,
				0x08,0xd1,0xd0,0xe7, 0x3b,0xcb,0x85,0x84,
				0x32,0xd6,0xb9,0xfb, 0xf7,0x45,0xa1,0xaf,
				0x9c,0xa3,0x8d,0x37, 0xde,0x03,0x6b,0xf4,
				0xae,0x58,0x03,0x26, 0x58,0x4f,0x73,0x49,
				0xc8,0x7f,0xa3,0xdd, 0x51,0xf2,0xec,0x34,
				0x8f,0xd5,0xe0,0xc2, 0xe5,0x33,0xf7,0x31,
				0x33,0xe7,0x98,0x5f, 0x26,0x14,0x4f,0xbb,
				0x88,0x1f,0xb3,0x92, 0x4e,0x97,0x2d,0xee,
				0x08,0x5f,0x9c,0x14, 0x5f,0xaf,0x6c,0x10,
				0xf9,0x47,0x41,0x81, 0xe9,0x99,0x49,0x52,
				0x86,0x29,0x55,0xba, 0x2e,0xb6,0x62,0x24,
				0x58,0xf7,0x4d,0x99, 0xce,0x75,0xa8,0x45,
				0x66,0x27,0x48,0x3f, 0x78,0xe3,0x48,0x7c,
				0xd7,0x1a,0x6c,0x89, 0x9d,0xb2,0x6a,0x23,
				0x9d,0xd7,0xed,0x82, 0x31,0x94,0x40,0x66,
				0xc8,0x28,0x52,0x23, 0xe7,0x61,0xde,0x71,
				0x69,0xf2,0x53,0x43, 0x30,0xce,0x6a,0x1a,
				0xfe,0x1e,0xeb,0xc2, 0x9f,0x61,0x81,0x94,
				0x18,0xed,0x58,0xbb, 0x01,0x13,0x92,0xb3,
				0xa6,0x90,0x7f,0xb5, 0xf4,0xbd,0xff,0xae,
			},
		},
		[4] = {
			.k = {
				0x7f,0x56,0x7d,0x15, 0x77,0xe6,0x83,0xac,
				0xd3,0xc5,0xb7,0x39, 0x9e,0x9f,0xf9,0x17,
				0xc7,0xff,0x50,0xb0, 0x33,0xee,0x8f,0xd7,
				0x3a,0xab,0x0b,0xfe, 0x6d,0xd1,0x41,0x8a,
			},
			.tlen = 0,
			.len = 4096,
			.p = {
				0x95,0x96,0x98,0xef, 0x73,0x92,0xb5,0x20,
				0xec,0xfc,0x4d,0x91, 0x54,0xbf,0x8d,0x9d,
				0x54,0xbc,0x4f,0x0f, 0x94,0xfc,0x94,0xcf,
				0x07,0xf6,0xef,0xbb, 0xed,0x3f,0xd3,0x60,
				0xba,0x85,0x1d,0x04, 0x08,0x54,0x92,0x08,
				0x06,0x52,0x7f,0x33, 0xfd,0xf3,0xdf,0x2a,
				0x17,0x2d,0xda,0x73, 0x03,0x56,0x21,0xa9,
				0xa3,0xab,0xf7,0x24, 0x17,0x39,0x7e,0x0f,
				0x00,0xdd,0xac,0x55, 0xb0,0x8b,0x2d,0x72,
				0x3b,0x9a,0x36,0x5a, 0xd9,0x0a,0x8e,0x0f,
				0xe2,0x1d,0xe8,0x85, 0xc3,0xc1,0x17,0x11,
				0xa7,0x2c,0x87,0x77, 0x9d,0x6c,0x3a,0xa6,
				0x90,0x59,0x10,0x24, 0xb0,0x92,0xe1,0xb6,
				0xa9,0x89,0x7c,0x95, 0x0a,0xf2,0xb2,0xa3,
				0x4a,0x40,0x88,0x35, 0x71,0x4e,0xa5,0xc9,
				0xde,0xba,0xd7,0x62, 0x56,0x46,0x40,0x1e,
				0xda,0x80,0xaf,0x28, 0x5d,0x40,0x36,0xf6,
				0x09,0x06,0x29,0x6e, 0xaa,0xca,0xe3,0x9e,
				0x9a,0x4f,0x4c,0x7e, 0x71,0x81,0x6f,0x9e,
				0x50,0x05,0x91,0x58, 0x13,0x6c,0x75,0x6a,
				0xd3,0x0e,0x7e,0xaf, 0xe1,0xbc,0xd9,0x38,
				0x18,0x47,0x73,0x3a, 0xf3,0x78,0x6f,0xcc,
				0x3e,0xea,0x52,0x82, 0xb9,0x0a,0xc5,0xfe,
				0x77,0xd6,0x25,0x56, 0x2f,0xec,0x04,0x59,
				0xda,0xd0,0xc9,0x22, 0xb1,0x01,0x60,0x7c,
				0x48,0x1a,0x31,0x3e, 0xcd,0x3d,0xc4,0x87,
				0xe4,0x83,0xc2,0x06, 0x91,0xf7,0x02,0x86,
				0xd2,0x9b,0xfd,0x26, 0x5b,0x9b,0x32,0xd1,
				0x5c,0xfd,0xb4,0xa8, 0x58,0x3f,0xd8,0x10,
				0x8a,0x56,0xee,0x04, 0xd0,0xbc,0xaa,0xa7,
				0x62,0xfd,0x9a,0x52, 0xec,0xb6,0x80,0x52,
				0x39,0x9e,0x07,0xc8, 0xb4,0x50,0xba,0x5a,
				0xb4,0x9a,0x27,0xdb, 0x93,0xb6,0x98,0xfe,
				0x52,0x08,0xa9,0x45, 0xeb,0x03,0x28,0x89,
				0x26,0x3c,0x9e,0x97, 0x0f,0x0d,0x0b,0x67,
				0xb0,0x00,0x01,0x71, 0x4b,0xa0,0x57,0x62,
				0xfe,0xb2,0x6d,0xbb, 0xe6,0xe4,0xdf,0xe9,
				0xbf,0xe6,0x21,0x58, 0xd7,0xf6,0x97,0x69,
				0xce,0xad,0xd8,0xfa, 0xce,0xe6,0x80,0xa5,
				0x60,0x10,0x2a,0x13, 0xb2,0x0b,0xbb,0x88,
				0xfb,0x64,0x66,0x00, 0x72,0x8c,0x4e,0x21,
				0x47,0x33,0x00,0x1f, 0x85,0xa6,0x3a,0xd3,
				0xe2,0x6c,0xc7,0x42, 0xb6,0x7b,0xc0,0x56,
				0x75,0xe2,0x61,0x72, 0x15,0xd1,0x88,0x08,
				0x3f,0x4d,0xfd,0xe2, 0x68,0x64,0xe5,0x7a,
				0x23,0x9b,0x3f,0x6c, 0xc3,0xd6,0x51,0x08,
				0x24,0x33,0x24,0x47, 0x7e,0xea,0x23,0xdc,
				0x07,0x41,0x66,0xa2, 0xa4,0xeb,0x23,0xa1,
				0x37,0x31,0xc0,0x7a, 0xe6,0xa4,0x63,0x05,
				0x20,0x44,0xe2,0x70, 0xd3,0x3e,0xee,0xd8,
				0x24,0x34,0x5d,0x80, 0xde,0xc2,0x34,0x66,
				0x5a,0x2b,0x6a,0x20, 0x4c,0x99,0x0d,0xbc,
				0x37,0x59,0xc5,0x8b, 0x70,0x4d,0xb4,0x0e,
				0x51,0xec,0x59,0xf6, 0x4f,0x08,0x1e,0x54,
				0x3d,0x45,0x31,0x99, 0x4d,0x5e,0x29,0x5f,
				0x12,0x57,0x46,0x09, 0x33,0xb9,0xf2,0x66,
				0xb4,0xc2,0xfa,0x63, 0xbe,0x42,0x6c,0x21,
				0x68,0x33,0x40,0xc6, 0xbd,0xd8,0x8a,0x55,
				0xd7,0x90,0x27,0x25, 0x7d,0x1e,0xed,0x02,
				0x50,0xd8,0xb1,0xac, 0xfa,0xd9,0xd4,0xcb,
				0x1c,0xc9,0x43,0x60, 0x44,0xab,0xd8,0x97,
				0x04,0xac,0xef,0x72, 0xa3,0x88,0xdc,0xb0,
				0xb0,0xb6,0xc6,0xd4, 0xd0,0x38,0xaf,0xc7,
				0xcd,0x8d,0x2a,0xa4, 0x13,0x53,0xd9,0xfd,
				0x2d,0x0b,0x91,0xb4, 0x3c,0x3a,0x72,0x11,
				0x6c,0x8b,0x96,0xa3, 0xc6,0x0b,0xd6,0x9a,
				0xa2,0xb9,0xae,0x76, 0xad,0xfd,0x01,0x90,
				0xab,0x93,0x9c,0x4b, 0xde,0x7e,0xf2,0x82,
				0x96,0xb9,0x98,0x55, 0xe2,0x68,0xe0,0xd8,
				0x61,0xb8,0x91,0x9a, 0xaf,0x92,0xd7,0xe5,
				0xeb,0x88,0xc5,0xb0, 0xcb,0x75,0x55,0xa9,
				0x94,0x7c,0x9c,0x11, 0x14,0x81,0x1a,0x09,
				0x61,0xd8,0x22,0x44, 0x13,0xba,0xe8,0x06,
				0x78,0xfd,0xd5,0x82, 0x73,0x19,0x9a,0xd1,
				0x5d,0x16,0xf5,0xd8, 0x86,0x7e,0xe3,0xcd,
				0xdc,0xe8,0x6a,0x18, 0x05,0xba,0x10,0xe4,
				0x06,0xc7,0xb2,0xf3, 0xb2,0x3e,0x1c,0x74,
				0x86,0xdd,0xad,0x8c, 0x82,0xf0,0x73,0x15,
				0x34,0xac,0x1d,0x95, 0x5e,0xba,0x2a,0xba,
				0xf8,0xac,0xbd,0xd7, 0x28,0x74,0x28,0xc7,
				0x29,0xa0,0x00,0x11, 0xda,0x31,0x7c,0xab,
				0x66,0x4d,0xb2,0x5e, 0xae,0x71,0xc5,0x31,
				0xcc,0x2b,0x9f,0x36, 0x2e,0xe6,0x97,0xa4,
				0xe1,0xb8,0x4b,0xc9, 0x00,0x87,0x7b,0x54,
				0xaa,0xeb,0xff,0x1a, 0x15,0xe8,0x3e,0x11,
				0xf7,0x25,0x3a,0xce, 0x94,0x23,0x27,0x44,
				0x77,0x80,0x6e,0xdd, 0x3f,0x8e,0x5a,0x92,
				0xae,0xee,0xb9,0x00, 0x79,0xc3,0x1d,0xab,
				0x17,0xb8,0x2b,0xff, 0x0d,0x64,0x29,0xb7,
				0x61,0x4d,0xd0,0x8d, 0x3d,0x36,0x3d,0x13,
				0xed,0x12,0xe8,0x08, 0xdd,0x4b,0x37,0xf7,
				0x2b,0xe7,0xeb,0x92, 0x78,0x98,0xc2,0xd6,
				0x13,0x15,0x94,0xff, 0xef,0xdc,0xda,0x27,
				0x7b,0xf9,0x58,0x5b, 0x90,0xf3,0xcd,0x1b,
				0x38,0x8a,0x00,0x38, 0x9b,0x95,0xcb,0x18,
				0x1f,0x97,0xd2,0x1f, 0x60,0x9d,0x6c,0xac,
				0xb8,0x72,0x08,0xd9, 0xc1,0xf4,0x98,0x72,
				0xf9,0x44,0xf2,0x2b, 0xe1,0x6e,0x76,0x15,
				0x63,0xfc,0x57,0x12, 0x23,0x4a,0xff,0xd3,
				0x1f,0x0d,0x0c,0xb9, 0x14,0xf9,0x98,0x52,
				0xce,0x90,0x34,0x8c, 0xd4,0x54,0x14,0x9e,
				0xf7,0x2c,0xba,0x5f, 0x80,0xb0,0x02,0x68,
				0x4f,0xca,0xb0,0xda, 0x44,0x11,0xb4,0xbd,
				0x12,0x14,0x80,0x6b, 0xc1,0xce,0xa7,0xfe,
				0x0e,0x16,0x69,0x19, 0x3c,0xe7,0xb6,0xfe,
				0x5a,0x59,0x02,0xf6, 0x78,0x3e,0xa4,0x65,
				0x57,0xa1,0xf2,0x65, 0xad,0x64,0xfc,0xba,
				0xd8,0x47,0xc8,0x8d, 0x11,0xf9,0x6a,0x25,
				0x22,0xa7,0x7f,0xa9, 0x43,0xe4,0x07,0x6b,
				0x49,0x26,0x42,0xe4, 0x03,0x1f,0x56,0xcd,
				0xf1,0x49,0xf8,0x0d, 0xea,0x1d,0x4f,0x77,
				0x5c,0x3c,0xcd,0x6d, 0x58,0xa8,0x92,0x6d,
				0x50,0x4a,0x81,0x6e, 0x09,0x2a,0x15,0x9e,
				0x3b,0x56,0xd3,0xb4, 0xef,0xe6,0x12,0xaf,
				0x60,0x3b,0x73,0xe7, 0xd8,0x2e,0xab,0x13,
				0xfb,0x7e,0xea,0xb1, 0x7b,0x54,0xc5,0x26,
				0x41,0x93,0x31,0xda, 0xb5,0x7a,0xe3,0x46,
				0x7a,0x8a,0xb0,0x81, 0xab,0xd5,0x90,0x85,
				0x4b,0xef,0x30,0x11, 0xb8,0x00,0x19,0x39,
				0xd3,0x11,0x54,0x53, 0x48,0x7a,0x7e,0xc5,
				0x4e,0x52,0xe5,0x4c, 0xeb,0xa2,0x9f,0x7a,
				0xdc,0xb5,0xc8,0x4e, 0x3b,0x5c,0x92,0x0f,
				0x19,0xcb,0x0a,0x9d, 0xda,0x01,0xfc,0x17,
				0x62,0xc3,0x46,0x63, 0x8b,0x4e,0x85,0x92,
				0x75,0x01,0x00,0xb3, 0x74,0xa8,0x23,0xd1,
				0xd2,0x91,0x53,0x0f, 0xd0,0xe9,0xed,0x90,
				0xde,0x9c,0x8c,0xb7, 0xf1,0x6a,0xd6,0x49,
				0x3c,0x22,0x2b,0xd7, 0x73,0x76,0x38,0x79,
				0xb5,0x88,0x1e,0xee, 0xdf,0xed,0x9f,0xfd,
				0x1a,0x0e,0xe7,0xd5, 0xc6,0xc9,0xfb,0x03,
				0xcc,0x84,0xb5,0xd2, 0x49,0xca,0x49,0x0a,
				0x1b,0x7c,0x78,0xe4, 0xd1,0x2e,0x7c,0x14,
				0x80,0x38,0x9d,0xba, 0x64,0x13,0xd3,0xf8,
				0x8e,0x05,0x4a,0xd6, 0x0d,0x73,0x09,0x1e,
				0xf1,0x75,0x63,0x59, 0xed,0xfc,0xbe,0x83,
				0x56,0x91,0x22,0x84, 0xd2,0x1e,0xf2,0x61,
				0x12,0x3d,0x50,0x6c, 0x9f,0xea,0x6b,0xcd,
				0x8c,0xac,0x28,0x0d, 0xad,0xf4,0xfd,0x77,
				0x45,0x68,0x17,0xb6, 0x03,0x13,0x54,0x7a,
				0xc0,0x8e,0x6b,0x56, 0x8a,0xd2,0xc6,0x1b,
				0xb3,0x3e,0x4f,0x68, 0x91,0x2e,0x2d,0x35,
				0x2a,0x32,0x27,0x86, 0x67,0x36,0x73,0xb8,
				0xfc,0x08,0xb8,0xf8, 0x1f,0x67,0x0b,0x32,
				0x89,0x00,0xfb,0x2d, 0xbe,0x74,0xae,0x41,
				0x3a,0xd3,0xed,0xf1, 0x67,0xee,0xe5,0x26,
				0xd4,0x59,0xdc,0x3b, 0x6b,0xf7,0x33,0x67,
				0xed,0xef,0xb0,0x5d, 0x5e,0x43,0x34,0xa2,
				0x3d,0x55,0x16,0x99, 0x4b,0x90,0x49,0x40,
				0x82,0x35,0x0d,0x82, 0xa6,0x16,0xd2,0x41,
				0xc8,0x65,0xd4,0xe7, 0x1a,0xdb,0xad,0xe6,
				0x48,0x5e,0xeb,0x94, 0xa6,0x9f,0x97,0x1e,
				0xd4,0x38,0x5d,0xff, 0x6e,0x17,0x0c,0xd0,
				0xb3,0xd5,0xb4,0x06, 0xd7,0xcb,0x8e,0xa3,
				0x27,0x75,0x24,0xb5, 0x14,0xe9,0x55,0x94,
				0x51,0x14,0xaf,0x15, 0x02,0xd3,0x9c,0x5f,
				0x43,0xfe,0x97,0xf4, 0x0b,0x4e,0x4d,0x89,
				0x15,0x33,0x4a,0x04, 0x10,0xf3,0xeb,0x13,
				0x71,0x86,0xb4,0x8a, 0x2c,0x75,0x04,0x47,
				0xb9,0x60,0xe9,0x2a, 0x5a,0xe8,0x7e,0x8b,
				0x91,0xa7,0x01,0x49, 0xcf,0xfc,0x48,0x83,
				0xa7,0x42,0xc8,0x2f, 0x80,0x92,0x04,0x64,
				0x03,0xf7,0x9f,0x1d, 0xc2,0x82,0x0b,0x14,
				0x65,0x4d,0x04,0x09, 0x13,0x5f,0xb8,0x66,
				0x19,0x14,0x7a,0x09, 0xa7,0xf8,0x73,0x2d,
				0x4d,0x90,0x86,0x14, 0x25,0xd6,0xd6,0xf5,
				0x82,0x9c,0x32,0xab, 0x5c,0x37,0x12,0x28,
				0xd1,0xfe,0xfa,0x0d, 0x90,0x8d,0x28,0x20,
				0xb1,0x1e,0xbe,0x30, 0x80,0xd7,0xb1,0x63,
				0xd9,0x23,0x83,0x0b, 0x9d,0xf5,0x0e,0x9c,
				0xa2,0x88,0x5f,0x2c, 0xf2,0xa6,0x9d,0x23,
				0x45,0x1c,0x9b,0x7a, 0xd2,0x60,0xa6,0x0f,
				0x44,0xba,0x91,0x3d, 0xc6,0xf7,0xef,0x2f,
				0x5c,0xa8,0x5e,0x2b, 0x50,0xd3,0xd1,0x85,
				0xfd,0xed,0x52,0x48, 0xe2,0xd9,0xd2,0x12,
				0x4e,0x03,0xc9,0x3d, 0x8f,0x8d,0x1f,0x8e,
				0x6b,0xd8,0xe3,0x32, 0xa7,0x5b,0x39,0x57,
				0x91,0x08,0x52,0x09, 0xa4,0x7a,0x40,0xc6,
				0xcf,0xcf,0x68,0xba, 0xb1,0x97,0xf8,0x38,
				0x94,0x1d,0x18,0x69, 0x80,0x6a,0x11,0x15,
				0xc2,0xfb,0x2d,0x6c, 0xd1,0xd4,0x88,0x50,
				0xbb,0xca,0x8c,0x56, 0x36,0xb6,0xc4,0x41,
				0x97,0xe6,0xb0,0x5c, 0x7f,0x51,0x00,0x6f,
				0x17,0xe5,0xde,0x27, 0xf7,0xb4,0x85,0x3b,
				0xc5,0xa1,0x60,0x1c, 0xba,0x21,0xd6,0xed,
				0xd5,0x08,0x62,0x80, 0xb4,0x85,0x52,0x15,
				0x5c,0x94,0x19,0x3a, 0x10,0x92,0xa4,0x06,
				0xf1,0x86,0x02,0xce, 0x94,0xd3,0xd5,0x33,
				0xe7,0x59,0x47,0x72, 0x12,0xf4,0x8b,0x06,
				0x29,0xa3,0xb0,0x39, 0x78,0x8f,0x46,0x56,
				0x4a,0x42,0x4f,0x89, 0x1b,0x3f,0x09,0x12,
				0xc4,0x24,0x0b,0x22, 0xf0,0x27,0x04,0x4d,
				0x39,0xd8,0x59,0xc8, 0x7c,0x59,0x18,0x0a,
				0x36,0xa8,0x3c,0xba, 0x42,0xe2,0xf7,0x7a,
				0x23,0x90,0x73,0xff, 0xd6,0xa3,0xb2,0xcf,
				0x60,0xc6,0x62,0x76, 0x61,0xa3,0xcd,0x53,
				0x94,0x37,0x3c,0x24, 0x4b,0xc1,0xc5,0x3b,
				0x26,0xf8,0x67,0x1d, 0xca,0xdd,0x08,0xcb,
				0xdb,0x00,0x96,0x34, 0xd0,0x5d,0xef,0x4e,
				0x64,0x18,0xb1,0xdc, 0x46,0x13,0xc1,0x8c,
				0x87,0xbf,0xa3,0xfe, 0xd7,0x49,0x7e,0xb3,
				0x94,0xe4,0x38,0x70, 0x2a,0xde,0xaf,0x73,
				0x46,0xda,0xff,0xec, 0xfc,0x18,0xe2,0x02,
				0x64,0x5f,0x9b,0xd2, 0xdf,0x8b,0xa8,0xd0,
				0x4c,0xd7,0x5c,0xc7, 0x80,0x59,0x4d,0x66,
				0x68,0xd3,0x4a,0x51, 0xc3,0x68,0xe2,0x0a,
				0x17,0x31,0x4b,0xd7, 0x23,0x28,0x25,0x26,
				0x4a,0xef,0x02,0xd7, 0x3a,0x53,0xdb,0x09,
				0x19,0x85,0x68,0xab, 0xa9,0x8c,0xff,0x7e,
				0x30,0xfb,0x42,0x08, 0xa1,0x5a,0xd1,0xc9,
				0x3f,0xc9,0x00,0xfb, 0xd4,0x3e,0xb0,0x1c,
				0x99,0xba,0xdc,0xb4, 0x69,0xe7,0xe1,0xb0,
				0x67,0x53,0x46,0xa6, 0xc6,0x34,0x5c,0x94,
				0xfa,0xd3,0x9b,0x48, 0x92,0xa1,0xd3,0xe5,
				0xa7,0xea,0xe1,0x86, 0x5e,0x90,0x26,0x2d,
				0x4b,0x85,0xe1,0x68, 0xee,0xc2,0xf1,0x25,
				0xb7,0xff,0x01,0x96, 0x61,0x54,0xba,0xf3,
				0x09,0x62,0x7f,0xa3, 0x92,0x6b,0xe7,0x00,
				0xfc,0xd4,0x04,0xfd, 0x2d,0x42,0x7e,0x56,
				0x91,0x33,0x6e,0xf8, 0x08,0x94,0xff,0xce,
				0x03,0x7e,0x4d,0x0a, 0x91,0x41,0x4f,0xaa,
				0xdd,0xd1,0x8c,0x34, 0x99,0x46,0xb5,0xfb,
				0x0e,0x09,0x26,0xcc, 0x6d,0x35,0x58,0x0a,
				0xc6,0xc0,0x89,0xa0, 0xbd,0xb6,0x89,0xd1,
				0x51,0x64,0x85,0x96, 0x4d,0x6a,0x16,0x26,
				0x30,0xb7,0xb3,0xe4, 0x80,0x46,0xaa,0x37,
				0x4c,0x9b,0x2b,0xa3, 0x76,0x5e,0x8b,0x52,
				0x13,0x42,0xe5,0xe3, 0xa8,0xe9,0xaf,0x83,
				0x60,0xc0,0xb0,0xf8, 0x3d,0x82,0x0a,0x21,
				0x60,0xd2,0x3f,0x1c, 0xb4,0xb5,0x53,0x31,
				0x2e,0x16,0xfd,0xf3, 0xc3,0x46,0xfa,0xcc,
				0x45,0x1f,0xd1,0xac, 0x22,0xe2,0x41,0xb5,
				0x21,0xf3,0xdd,0x1f, 0x81,0xbf,0x03,0xaf,
				0xd6,0x31,0xc1,0x6a, 0x2e,0xff,0xc1,0x2d,
				0x44,0x53,0xd0,0xb5, 0xa2,0x7c,0x5f,0xf4,
				0x47,0xf7,0x4d,0x1e, 0x77,0xe2,0x29,0xcc,
				0xd2,0x46,0x85,0xfa, 0xdb,0x7f,0x46,0xf5,
				0xc9,0x60,0x4a,0x2c, 0xb7,0xf2,0xa2,0x2c,
				0x9d,0x76,0xcd,0x82, 0x67,0xae,0xbb,0xe0,
				0x92,0x56,0x48,0xcb, 0xe5,0xf5,0x3c,0x2c,
				0xe0,0xe8,0x6a,0x6a, 0x5a,0x0a,0x20,0x7c,
				0xa6,0x9d,0x8e,0x84, 0xfa,0xfe,0x61,0x13,
				0x54,0x79,0xe0,0x83, 0xd2,0x15,0xe0,0x33,
				0xe4,0xf9,0xad,0xb8, 0x1e,0x75,0x35,0xd3,
				0xee,0x7e,0x4a,0x63, 0x2f,0xeb,0xf1,0xe6,
				0x22,0xac,0x77,0x74, 0xa1,0xc0,0xa0,0x21,
				0x66,0x59,0x7c,0x48, 0x7f,0xaa,0x05,0xe8,
				0x51,0xd9,0xc7,0xed, 0xb9,0xea,0x7a,0xdd,
				0x23,0x53,0xea,0x8f, 0xef,0xaa,0xe6,0x9e,
				0x19,0x21,0x84,0x27, 0xc5,0x78,0x2e,0x8c,
				0x52,0x40,0x15,0x1c, 0x2b,0x91,0xb3,0x4c,
				0xe8,0xfa,0xd3,0x64, 0x0f,0xf9,0xf4,0xb8,
				0x59,0x4d,0x6b,0x2d, 0x44,0x6c,0x8d,0xb2,
				0xdb,0x73,0x29,0x66, 0xb1,0xc2,0x28,0xfc,
				0x85,0xba,0x60,0x5e, 0x27,0x8f,0xfb,0xb3,
				0xc9,0x20,0x43,0xb1, 0x3e,0x18,0x97,0x42,
				0x63,0x2d,0x0c,0x97, 0xf2,0xcc,0xcd,0x90,
				0x46,0x5f,0x1a,0x85, 0xca,0x44,0x2a,0x1a,
				0x52,0xf7,0xbb,0x4e, 0xd1,0xab,0xd5,0xa3,
				0x58,0x6b,0xb6,0x5a, 0x88,0x1c,0x9d,0x3b,
				0xe2,0x46,0xe4,0x3b, 0x33,0x64,0x6c,0xfd,
				0xeb,0x36,0x8e,0x32, 0x1f,0x71,0xbd,0x95,
				0xb6,0xfd,0x1a,0xcb, 0xfb,0x4a,0x88,0x27,
				0xd6,0x28,0x7b,0x5e, 0xa3,0x8a,0x0c,0x36,
				0xa8,0x5d,0x2f,0x28, 0xa9,0xad,0xb2,0x88,
				0x9e,0x62,0x9d,0x4a, 0x07,0x74,0x00,0x04,
				0x0c,0xc1,0x6a,0x09, 0xe1,0x0b,0xfa,0xf3,
				0xd1,0x41,0xdd,0x94, 0x52,0x06,0xb8,0x9e,
				0xba,0x81,0xe0,0x52, 0xdf,0x52,0x5d,0x74,
				0x40,0x59,0x36,0x05, 0xf2,0x30,0xc4,0x84,
				0x85,0xdc,0xb8,0xba, 0xd9,0xf4,0x5f,0x11,
				0x83,0xce,0x25,0x57, 0x97,0xf5,0x0f,0xb5,
				0x0b,0xd6,0x6d,0x1c, 0xfb,0xf2,0x30,0xda,
				0xc2,0x05,0xa8,0xe1, 0xc2,0x57,0x0a,0x05,
				0x2d,0x4c,0x8b,0xb7, 0x5a,0xc0,0x8a,0xba,
				0xa9,0x85,0x7c,0xf0, 0xb8,0xce,0x72,0x79,
				0xf5,0x27,0x99,0xd7, 0xed,0xcf,0x85,0xfa,
				0x92,0x15,0xf1,0x47, 0x02,0x24,0x39,0x07,
				0x89,0xb6,0xdd,0x4a, 0xb8,0xbc,0xd5,0x9d,
				0x4c,0x03,0x8b,0x1d, 0x45,0x58,0x1c,0x86,
				0x46,0x71,0x0a,0x0d, 0x7c,0x5b,0xf9,0xdc,
				0x60,0xb5,0xb0,0x00, 0x70,0x47,0x83,0xa6,
				0x8e,0x79,0xba,0x1d, 0x21,0x20,0xc0,0x24,
				0x56,0x35,0x6a,0x49, 0xb6,0xa3,0x58,0x87,
				0x16,0xae,0xd9,0x77, 0x62,0xa0,0x61,0xce,
				0x3d,0xe6,0x77,0x9e, 0x83,0xec,0xc2,0x04,
				0x8c,0xba,0x62,0xac, 0x32,0xda,0xf0,0x89,
				0x7b,0x2b,0xb0,0xa3, 0x3a,0x5f,0x8b,0x0d,
				0xbd,0xe9,0x14,0xcd, 0x5b,0x7a,0xde,0xd5,
				0x0d,0xc3,0x4b,0x38, 0x92,0x31,0x97,0xd8,
				0xae,0x89,0x17,0x2c, 0xc9,0x54,0x96,0x66,
				0xd0,0x9f,0x60,0x7a, 0x7d,0x63,0x67,0xfc,
				0xb6,0x02,0xce,0xcc, 0x97,0x36,0x9c,0x3c,
				0x1e,0x69,0x3e,0xdb, 0x54,0x84,0x0a,0x77,
				0x6d,0x0b,0x6e,0x10, 0x9f,0xfb,0x2a,0xb1,
				0x49,0x31,0x71,0xf2, 0xd1,0x1e,0xea,0x87,
				0xb9,0xd6,0x4a,0x4c, 0x57,0x17,0xbc,0x8b,
				0x38,0x66,0x2d,0x5f, 0x25,0xca,0x6d,0x10,
				0xc6,0x2e,0xd7,0x2c, 0x89,0xf1,0x4c,0x1d,
				0xc9,0x9c,0x02,0x23, 0xc6,0x1f,0xd6,0xc3,
				0xb8,0xc7,0x85,0x29, 0x75,0x40,0x1e,0x04,
				0x6e,0xc7,0xb4,0x60, 0xfc,0xea,0x30,0x8b,
				0x4d,0x9d,0xb7,0x5d, 0x91,0xfb,0x8e,0xb8,
				0xc2,0x54,0xdf,0xdb, 0x79,0x58,0x32,0xda,
				0xd0,0xa1,0xd6,0xd6, 0xc4,0xc8,0xa4,0x16,
				0x95,0xbb,0xe5,0x58, 0xd2,0xb6,0x83,0x76,
				0x1d,0xd7,0x45,0xbc, 0xb8,0x14,0x79,0x3b,
				0x4e,0x1a,0x0b,0x5c, 0xfc,0xa5,0xa0,0xc3,
				0xf1,0x64,0x74,0xb0, 0x0d,0x82,0x90,0x62,
				0x87,0x02,0x0f,0x71, 0xc7,0xab,0x7d,0x2b,
				0x70,0xf1,0x9b,0x9e, 0xe7,0x6b,0x99,0x18,
				0x6c,0x54,0x17,0x0b, 0xf5,0x44,0x58,0x54,
				0x44,0x9b,0x54,0x30, 0x5e,0xaf,0xa6,0xfa,
				0x42,0x37,0xe8,0x67, 0xbf,0xf7,0x6c,0x1e,
				0x73,0xd8,0xc7,0x5c, 0xfa,0x51,0xd5,0x1f,
				0xab,0xfc,0x91,0x03, 0xc1,0xc1,0x22,0x58,
				0xc7,0xe8,0x60,0xae, 0xb6,0x58,0x44,0xad,
				0x1e,0x07,0x5d,0x3c, 0x90,0x33,0x43,0xe0,
				0x67,0x44,0x9f,0x8c, 0xf3,0xef,0xce,0x3a,
				0x22,0x2b,0x1b,0x97, 0x83,0x6f,0x9f,0xd3,
				0x46,0xc3,0xa1,0xdf, 0xde,0x60,0xf0,0x32,
				0x2e,0xcf,0xed,0x72, 0x27,0x0d,0xa7,0xd0,
				0x91,0x6a,0xf0,0x6d, 0x41,0xfa,0x77,0x2e,
				0xd8,0x43,0xce,0xe2, 0xf5,0x7a,0x9e,0x04,
				0x30,0x4c,0xe7,0x08, 0xf3,0x2e,0x13,0x05,
				0x5e,0xfa,0x16,0x2c, 0x6c,0x53,0x02,0xb5,
				0x2f,0x2c,0x7d,0x86, 0x61,0x0e,0x5f,0x96,
				0xe1,0x1c,0x37,0x87, 0xf0,0x84,0xe4,0x1d,
				0x53,0x4d,0xb1,0x13, 0xe2,0xcb,0x71,0x6e,
				0x86,0x7b,0xad,0x97, 0x3e,0x16,0xb3,0xb4,
				0x0f,0x32,0x01,0x69, 0x31,0x1f,0x49,0x99,
				0x7a,0x46,0xd9,0x9b, 0x5f,0x17,0x3d,0xcb,
				0xe4,0xfd,0xbc,0xbb, 0xe3,0xec,0x8c,0x54,
				0xc4,0x14,0x44,0x89, 0xa3,0x65,0x25,0xc0,
				0x06,0x9b,0x7d,0x9b, 0x7f,0x15,0x8f,0x84,
				0xe1,0x08,0x0d,0x2c, 0x0a,0x91,0x9a,0x85,
				0x4e,0xa1,0x50,0xee, 0x72,0x70,0xf4,0xd2,
				0x1c,0x67,0x20,0x1f, 0xe6,0xb2,0x9d,0x95,
				0x85,0x7e,0xf2,0x9d, 0xf0,0x73,0x10,0xe7,
				0xfc,0x62,0x9d,0xea, 0x8d,0x63,0xdc,0x70,
				0xe0,0x2b,0x30,0x01, 0x7c,0xcd,0x24,0x22,
				0x03,0xf9,0x8b,0xe4, 0x77,0xef,0x2c,0xdc,
				0xa5,0xfb,0x29,0x66, 0x50,0x1c,0xd7,0x4e,
				0x8f,0x0f,0xbf,0x61, 0x0c,0xea,0xc0,0xe6,
				0xc6,0xc3,0xa1,0xae, 0xf3,0xea,0x4c,0xfb,
				0x21,0x96,0xd1,0x38, 0x64,0xe0,0xdd,0xa8,
				0xa4,0xd0,0x33,0x82, 0xf0,0xdd,0x91,0x6e,
				0x88,0x27,0xe1,0x0d, 0x8b,0xfb,0xc6,0x36,
				0xc5,0x9a,0x9d,0xbc, 0x32,0x8f,0x8a,0x3a,
				0xfb,0xd0,0x88,0x1e, 0xe5,0xb8,0x68,0x35,
				0x4b,0x22,0x72,0x55, 0x9e,0x77,0x39,0x1d,
				0x64,0x81,0x6e,0xfd, 0xe3,0x29,0xb8,0xa5,
				0x3e,0xc8,0x4c,0x6f, 0x41,0xc2,0xbd,0xb6,
				0x15,0xd1,0xd5,0xe9, 0x77,0x97,0xb6,0x54,
				0x9e,0x60,0xdd,0xf3, 0x48,0xdb,0x65,0x04,
				0x54,0xa2,0x93,0x12, 0xf0,0x66,0x6c,0xae,
				0xa2,0x2c,0xb9,0xeb, 0xf0,0x7c,0x9c,0xae,
				0x8e,0x49,0xf5,0x0f, 0xfc,0x4b,0x2a,0xdb,
				0xaf,0xff,0x96,0x0d, 0xa6,0x05,0xe9,0x37,
				0x81,0x43,0x41,0xb2, 0x69,0x88,0xd5,0x2c,
				0xa2,0xa9,0x9b,0xf2, 0xf1,0x77,0x68,0x05,
				0x84,0x0f,0x6a,0xee, 0xd0,0xb5,0x65,0x4b,
				0x35,0x18,0xeb,0x34, 0xba,0x09,0x4f,0xc3,
				0x5a,0xac,0x44,0x5b, 0x03,0xf5,0xf5,0x1d,
				0x10,0x04,0xfd,0xb5, 0xc4,0x26,0x84,0x13,
				0x8a,0xde,0x8d,0xbb, 0x51,0xd0,0x6f,0x58,
				0xc1,0xe5,0x9e,0x12, 0xe6,0xba,0x13,0x73,
				0x27,0x3e,0x3f,0xf0, 0x4f,0x0f,0x64,0x6c,
				0x0e,0x36,0xe9,0xcc, 0x38,0x93,0x9b,0xda,
				0xf9,0xfd,0xc2,0xe9, 0x44,0x7a,0x93,0xa6,
				0x73,0xf6,0x2a,0xc0, 0x21,0x42,0xbc,0x58,
				0x9e,0xe3,0x0c,0x6f, 0xa1,0xd0,0xdd,0x67,
				0x14,0x3d,0x49,0xf1, 0x5b,0xc3,0xc3,0xa4,
				0x52,0xa3,0xe7,0x0f, 0xb4,0x26,0xf4,0x62,
				0x73,0xf5,0x9f,0x75, 0x5b,0x6e,0x38,0xc8,
				0x4a,0xcc,0xf6,0xfa, 0xcf,0xfb,0x28,0x02,
				0x8a,0xdb,0x6b,0x63, 0x52,0x17,0x94,0x87,
				0x71,0xa2,0xf5,0x5a, 0x1d,0x94,0xe3,0xcd,
				0x28,0x70,0x96,0xd5, 0xb1,0xaf,0xec,0xd6,
				0xea,0xf4,0xfc,0xe9, 0x10,0x66,0xd9,0x8a,
				0x1e,0x03,0x03,0xf1, 0x54,0x2d,0xc5,0x8c,
				0x85,0x71,0xed,0xa7, 0xa4,0x1e,0x5a,0xff,
				0xab,0xb8,0x07,0xb3, 0x0b,0x84,0x00,0x0a,
				0x7f,0xa5,0x38,0x20, 0x66,0x33,0x84,0x2f,
				0xec,0x16,0x94,0x78, 0xa8,0x42,0x98,0x55,
				0xa3,0xe5,0xd3,0x62, 0x2a,0xfc,0xed,0xec,
				0x7a,0x96,0x41,0x35, 0xc0,0xd2,0xe6,0x53,
				0xf8,0x0f,0x59,0x94, 0x0a,0xa0,0x50,0xef,
				0x0d,0x9f,0x04,0x1c, 0x5f,0x48,0xfe,0x33,
				0x20,0xca,0x8d,0x09, 0xdd,0x0b,0xf8,0x59,
				0xd3,0x63,0x8a,0xa4, 0xf5,0x73,0x6b,0x3e,
				0x7e,0x0f,0xff,0xdb, 0x96,0x62,0x4d,0x3a,
				0xdb,0x8d,0x8c,0x9b, 0x8c,0xb3,0xa1,0xff,
				0x16,0xb9,0x2c,0x8c, 0xf6,0xbb,0x0d,0x9e,
				0x6f,0xff,0x24,0x6f, 0x59,0xee,0x02,0xe6,
				0x57,0x38,0xbd,0x5f, 0xbd,0xd4,0xe5,0x74,
				0x14,0xea,0x85,0xbb, 0x0c,0xfe,0xad,0xad,
				0x98,0x82,0x8a,0x81, 0x0b,0x37,0xdc,0x7d,
				0xda,0x13,0x74,0x8a, 0xa5,0xaf,0x74,0x82,
				0x95,0x35,0x1f,0x0b, 0x03,0x88,0x17,0xf3,
				0x67,0x11,0x40,0xd1, 0x9d,0x48,0xec,0x9b,
				0xc8,0xb2,0xcc,0xb4, 0x93,0xd2,0x0b,0x0a,
				0xd6,0x6f,0x34,0x32, 0xd1,0x9a,0x0d,0x89,
				0x93,0x1f,0x96,0x5a, 0x7a,0x57,0x06,0x02,
				0x1d,0xbf,0x57,0x3c, 0x9e,0xca,0x5d,0x68,
				0xe8,0x4e,0xea,0x4f, 0x0b,0x11,0xf0,0x35,
				0x73,0x5a,0x77,0x24, 0x29,0xc3,0x60,0x51,
				0xf0,0x15,0x93,0x45, 0x6b,0xb1,0x70,0xe0,
				0xda,0xf7,0xf4,0x0a, 0x70,0xd1,0x73,0x3f,
				0x9c,0x9d,0x07,0x19, 0xad,0xb2,0x28,0xae,
				0xf2,0xe2,0xb6,0xf4, 0xbc,0x71,0x63,0x00,
				0xde,0xe3,0xdc,0xb1, 0xa3,0xd5,0x4c,0x34,
				0xf8,0x6b,0x68,0x4c, 0x73,0x84,0xab,0xd4,
				0x89,0xae,0x07,0x1a, 0x0d,0x3d,0x8e,0xaa,
				0x6c,0xa2,0x54,0xb3, 0xd9,0x46,0x81,0x87,
				0xe2,0xdc,0x49,0xb1, 0x14,0x5c,0xcc,0x72,
				0x56,0xf0,0x0f,0xa9, 0x3d,0x31,0x2f,0x08,
				0xbc,0x15,0xb7,0xd3, 0x0d,0x4f,0xd1,0xc9,
				0x4e,0xde,0x1c,0x03, 0xd1,0xae,0xaf,0x14,
				0x62,0xbc,0x1f,0x33, 0x5c,0x00,0xeb,0xf4,
				0x8e,0xf6,0x3e,0x13, 0x6a,0x64,0x42,0x07,
				0x60,0x71,0x35,0xf1, 0xd0,0xff,0x8d,0x1f,
				0x88,0xc0,0x1c,0x3c, 0x6c,0x1c,0x54,0x71,
				0x6b,0x65,0x4a,0xe2, 0xe3,0x5f,0x77,0x56,
				0x1c,0x8d,0x2a,0x8d, 0xef,0x92,0x4a,0xa9,
				0xf6,0xcf,0xa5,0x67, 0x89,0x8e,0x5a,0xd9,
				0x60,0xaa,0x94,0x14, 0x55,0x66,0x8a,0xb0,
				0x18,0x4f,0x9e,0x8e, 0xf4,0xdb,0xc1,0x88,
				0x9b,0xf0,0x84,0x33, 0x2f,0xcd,0x2c,0xeb,
				0x65,0xe6,0x5d,0xde, 0x30,0x97,0xad,0xe6,
				0xbc,0xcb,0x83,0x93, 0xf3,0xfd,0x65,0xdc,
				0x07,0x27,0xf9,0x0f, 0x4a,0x56,0x5c,0xf7,
				0xff,0xa3,0xd1,0xad, 0xd4,0xd1,0x38,0x13,
				0x71,0xc9,0x42,0x0f, 0x0d,0x35,0x12,0x32,
				0xd2,0x2d,0x2b,0x96, 0xe4,0x01,0xdc,0x55,
				0xd8,0x71,0x2c,0x0c, 0xc4,0x55,0x3f,0x16,
				0xe8,0xaa,0xe7,0xe8, 0x45,0xfa,0x23,0x23,
				0x5e,0x21,0x02,0xab, 0xc8,0x6b,0x88,0x5e,
				0xdc,0x90,0x13,0xb5, 0xe7,0x47,0xfa,0x12,
				0xd5,0xa7,0x0a,0x06, 0xd2,0x7c,0x62,0x80,
				0xb7,0x8e,0x4f,0x77, 0x88,0xb7,0xa2,0x12,
				0xdb,0x19,0x1f,0xd8, 0x00,0x82,0xf5,0xf2,
				0x59,0x34,0xec,0x91, 0xa8,0xc1,0xd7,0x6e,
				0x76,0x10,0xf3,0x15, 0xa6,0x86,0xfa,0xfd,
				0x45,0x2f,0x86,0x18, 0x16,0x83,0x16,0x8c,
				0x6e,0x99,0x7e,0x43, 0x3f,0x0a,0xba,0x32,
				0x94,0x5b,0x15,0x32, 0x66,0xc2,0x3a,0xdc,
				0xf3,0xd3,0x1d,0xd1, 0x5d,0x6f,0x5f,0x9a,
				0x7f,0xa2,0x90,0xf1, 0xa1,0xd0,0x17,0x33,
				0xdf,0x9a,0x2e,0xa2, 0xdc,0x89,0xe6,0xb0,
				0xda,0x23,0x2b,0xf6, 0xe9,0x1f,0x82,0x3c,
				0x07,0x90,0xab,0x3a, 0xb9,0x87,0xb0,0x02,
				0xcc,0xb9,0xe7,0x2e, 0xe7,0xc6,0xee,0xfa,
				0xe2,0x16,0xc8,0xc3, 0xd0,0x40,0x15,0xc5,
				0xa7,0xc8,0x20,0x42, 0xb7,0x09,0xf8,0x66,
				0xeb,0x0e,0x4b,0xd7, 0x91,0x74,0xa3,0x8b,
				0x17,0x2a,0x0c,0xee, 0x7f,0xc1,0xea,0x63,
				0xc6,0x3c,0x1e,0xea, 0x8b,0xa2,0xd1,0x2e,
				0xf3,0xa6,0x0f,0x36, 0xff,0xdd,0x81,0x06,
				0xe3,0x63,0xfc,0x0c, 0x38,0xb0,0x23,0xfb,
				0x83,0x66,0x81,0x73, 0x5c,0x0b,0x9c,0xd4,
				0x23,0xdc,0x7f,0x5c, 0x00,0x8c,0xa6,0xa7,
				0x52,0xd4,0xc1,0x00, 0xea,0x99,0x6b,0x59,
				0x19,0x8e,0x34,0x32, 0x24,0xea,0x0c,0x61,
				0x95,0x9d,0xdb,0xf0, 0x63,0xcc,0xa9,0xfd,
				0x1b,0xeb,0xd7,0xbc, 0x0c,0xa4,0x74,0x24,
				0xfd,0xfa,0x32,0x58, 0xe3,0x74,0x1c,0x8f,
				0x76,0xa6,0x53,0x0d, 0xea,0xde,0x50,0x92,
				0xbd,0x3f,0x3d,0x56, 0x8f,0x48,0x4e,0xb7,
				0x8c,0x5e,0x83,0x2c, 0xf7,0xec,0x04,0x2c,
				0x35,0xdf,0xa9,0x72, 0xc0,0x77,0xf5,0x44,
				0xe5,0xa7,0x56,0x3e, 0xa4,0x8d,0xb8,0x6e,
				0x31,0x86,0x15,0x1d, 0xc4,0x66,0x86,0x75,
				0xf8,0x1a,0xea,0x2f, 0x3a,0xb7,0xbf,0x97,
				0xe9,0x11,0x53,0x64, 0xa8,0x71,0xc6,0x78,
				0x8a,0x70,0xb5,0x18, 0xd7,0x9c,0xe3,0x44,
				0x1a,0x7c,0x6b,0x1b, 0x41,0xe1,0x1c,0x0d,
				0x98,0x43,0x67,0x28, 0xb8,0x14,0xb4,0x48,
				0x01,0x85,0x79,0x20, 0x94,0x36,0x25,0x3a,
				0x5c,0x48,0xd2,0x2e, 0x91,0x91,0xfd,0x85,
				0x38,0xc1,0xc5,0xa5, 0x4d,0x52,0x1f,0xb4,
				0xe7,0x44,0x7a,0xff, 0xb1,0x65,0xdf,0x53,
				0x86,0x2a,0xff,0x25, 0x2b,0xeb,0x3e,0xdc,
				0x3d,0xec,0x72,0xae, 0xa9,0xd1,0xdf,0xe9,
				0x4a,0x3e,0xe8,0xf1, 0x74,0xe0,0xee,0xd6,
				0x0b,0xba,0x9b,0x14, 0x9b,0x0c,0x4a,0xf9,
				0x55,0xee,0x7e,0x82, 0xa4,0xb5,0xa5,0xb7,
				0x2f,0x75,0x48,0x51, 0x60,0xcc,0x41,0x8e,
				0x65,0xe3,0xb7,0x29, 0xe0,0x32,0xe7,0x1b,
				0x2f,0xa0,0x80,0xce, 0x73,0x28,0x6c,0xf4,
				0xd0,0xc7,0x05,0x69, 0xbd,0x3e,0x2e,0x77,
				0x1a,0x7f,0x9a,0x98, 0x60,0x31,0xdb,0x47,
				0xc2,0xa2,0x12,0xcb, 0x8c,0x35,0xff,0x58,
				0xe3,0x07,0x22,0xe4, 0x2f,0x26,0x87,0x30,
				0x16,0xea,0x64,0x4f, 0x44,0x64,0x3d,0xe4,
				0x7b,0x41,0x06,0xca, 0xee,0x02,0xcf,0xf3,
				0x26,0x4c,0xfe,0x9c, 0xf6,0x64,0x96,0xd4,
				0xd9,0x7e,0x04,0x47, 0x1d,0xdb,0xc7,0x8c,
				0xae,0xd7,0x9d,0xea, 0xe3,0x3a,0xee,0x24,
				0xa9,0x2d,0x65,0xba, 0xd5,0x9f,0x38,0x81,
				0x61,0x42,0x15,0xdf, 0xcc,0x29,0xd9,0xf7,
				0xd4,0x30,0xb9,0xc9, 0x86,0x76,0xdc,0xee,
				0xa5,0x27,0xa6,0x27, 0xa3,0xbb,0x8f,0x3b,
				0xaa,0xca,0x01,0x52, 0x37,0x12,0xc0,0x55,
				0x39,0x4a,0xb2,0xce, 0x85,0x73,0xf2,0x10,
				0x9c,0x7f,0xa6,0x34, 0x7f,0x0f,0x69,0x63,
				0x03,0xc4,0xde,0xe2, 0x7b,0x10,0xbf,0x91,
				0x3e,0x7e,0xad,0xb7, 0xa8,0x85,0xc7,0x99,
				0xae,0x8e,0x7c,0x2e, 0x02,0x25,0x5b,0xd5,
				0xf4,0x46,0xd1,0x49, 0x48,0xa0,0x12,0x6a,
				0x6a,0x01,0x23,0xb9, 0x7e,0x67,0x8b,0x48,
				0xac,0xf7,0x88,0x88, 0xeb,0xd9,0x39,0x3a,
				0xc8,0xa0,0x06,0xd9, 0x0b,0x80,0xc4,0x84,
			},
			.c = {
				0x10,0x46,0xb6,0xc8, 0xaa,0x83,0x67,0x7b,
				0xc5,0x9a,0x9a,0x0d, 0xe2,0xec,0x6f,0x9a,
				0x3e,0x74,0xa7,0xfa, 0x43,0x93,0x9d,0xc5,
				0x23,0x27,0xad,0x99, 0x74,0xb4,0xc0,0xe4,
				0xd7,0x70,0x5c,0x95, 0x58,0xe3,0x8f,0x72,
				0xe3,0x03,0x3d,0xc2, 0xd9,0x69,0x37,0x3e,
				0x8e,0x2a,0x0c,0x2b, 0x75,0x59,0x05,0x18,
				0x4a,0x50,0x67,0xd4, 0xf5,0x4b,0xb0,0x59,
				0x08,0xaf,0xbc,0x6f, 0xb1,0x95,0xa1,0x32,
				0xe7,0x77,0x1a,0xfd, 0xaf,0xe8,0x4d,0x32,
				0x87,0x9c,0x87,0x90, 0x5e,0xe8,0x08,0xc3,
				0xb4,0x0c,0x80,0x9a, 0x9e,0x23,0xeb,0x5a,
				0x5c,0x18,0x4a,0x7c, 0xd0,0x4a,0x91,0x57,
				0x7e,0x6c,0x53,0xde, 0x98,0xc0,0x09,0x80,
				0x8d,0x41,0x0b,0xbc, 0x56,0x5e,0x69,0x61,
				0xd3,0x56,0x48,0x43, 0x19,0x49,0x49,0xaf,
				0xcf,0xad,0x98,0x3e, 0x88,0x4b,0x44,0x69,
				0x73,0xd2,0xcb,0xdf, 0x30,0xdb,0x76,0x1d,
				0xfb,0x4b,0xc5,0x66, 0x22,0x34,0x6f,0x07,
				0x0b,0xcd,0x1c,0xed, 0x88,0xd9,0x0d,0x30,
				0xe9,0x96,0xcb,0xf5, 0xde,0x57,0x5f,0x0b,
				0x12,0x11,0xcf,0x52, 0xf5,0x0d,0xf8,0x29,
				0x39,0x87,0xb2,0xa5, 0x7f,0x7a,0x2b,0x9d,
				0x66,0x11,0x32,0xf4, 0xd4,0x37,0x16,0x75,
				0xe3,0x0b,0x55,0x98, 0x44,0x6f,0xc7,0x5c,
				0xd4,0x89,0xf8,0xb3, 0xee,0xe4,0x5e,0x45,
				0x34,0xc2,0xc0,0xef, 0xdd,0x4d,0xbb,0xb4,
				0x0a,0x7b,0xda,0xe3, 0x6e,0x41,0xe1,0xb4,
				0x73,0xf8,0x9b,0x65, 0x1c,0x5f,0xdf,0x9c,
				0xd7,0x71,0x91,0x72, 0x6f,0x9e,0x8f,0x96,
				0x5d,0x45,0x11,0xd1, 0xb9,0x99,0x63,0x50,
				0xda,0x36,0xe9,0x75, 0x21,0x9a,0xce,0xc5,
				0x1a,0x8a,0x12,0x81, 0x8b,0xeb,0x51,0x7c,
				0x00,0x5f,0x58,0x5a, 0x3e,0x65,0x10,0x9e,
				0xe3,0x9e,0xf0,0x6b, 0xfe,0x49,0x50,0x2a,
				0x2a,0x3b,0xa5,0x42, 0x1b,0x15,0x2b,0x5b,
				0x88,0xb8,0xfb,0x6f, 0x0c,0x5d,0x16,0x76,
				0x48,0x77,0x4d,0x22, 0xb9,0xf0,0x0a,0x3f,
				0xa6,0xdd,0xc8,0x32, 0xcc,0x98,0x76,0x41,
				0x84,0x36,0x24,0x6d, 0x88,0x62,0x65,0x40,
				0xa4,0x55,0xdc,0x39, 0x74,0xed,0x0f,0x50,
				0x08,0xcf,0x69,0x5f, 0x1d,0x31,0xd6,0xb4,
				0x39,0x94,0x5b,0x18, 0x88,0x0f,0xcb,0x56,
				0xfb,0xf7,0x19,0xe0, 0x80,0xe0,0x4f,0x67,
				0x9c,0xab,0x35,0x78, 0xc9,0xca,0x95,0xfa,
				0x31,0xf0,0x5f,0xa6, 0xf9,0x71,0xbd,0x7f,
				0xb1,0xe2,0x42,0x67, 0x9d,0xfb,0x7f,0xde,
				0x41,0xa6,0x7f,0xc7, 0x7f,0x75,0xd8,0x8d,
				0x43,0xce,0xe6,0xeb, 0x74,0xee,0x4e,0x35,
				0xbc,0x7b,0x7c,0xfc, 0x8b,0x4f,0x1f,0xa2,
				0x5e,0x34,0x3b,0x5f, 0xd0,0x05,0x9d,0x4f,
				0xfe,0x47,0x59,0xa3, 0xf6,0xb7,0x27,0xb0,
				0xa1,0xec,0x1d,0x09, 0x86,0x70,0x48,0x00,
				0x03,0x0a,0x15,0x98, 0x2e,0x6d,0x48,0x2a,
				0x81,0xa2,0xde,0x11, 0xe4,0xde,0x8b,0xb0,
				0x06,0x28,0x03,0x82, 0xe4,0x6e,0x40,0xfb,
				0x3c,0x35,0x2d,0x1b, 0x62,0x56,0x87,0xd4,
				0xd6,0x06,0x36,0xce, 0x70,0x26,0x2f,0x21,
				0xf5,0x47,0x3f,0xf8, 0x57,0x17,0xa9,0x15,
				0x30,0xfd,0x1f,0xa6, 0x7a,0x24,0x1c,0xf8,
				0x33,0xf3,0xef,0xe1, 0x6c,0xb5,0x0b,0x04,
				0x21,0x5d,0xb5,0xff, 0x4f,0xdb,0xd1,0x3d,
				0x8f,0x01,0x56,0x7f, 0x0b,0xa4,0xf1,0xf9,
				0xdd,0xa3,0x38,0xcb, 0xa9,0xd3,0xdd,0xe3,
				0x29,0x5b,0x2b,0x22, 0xd7,0xe8,0x4f,0x02,
				0xb1,0x73,0x83,0x80, 0xda,0xd0,0x8e,0x11,
				0x9f,0x4d,0xd4,0x0a, 0x86,0x45,0x11,0xa1,
				0x9e,0x2e,0xa9,0x59, 0x6d,0x95,0x49,0xc5,
				0xc9,0xcd,0x7c,0x71, 0x81,0xac,0x6b,0xb8,
				0x1b,0x94,0xe8,0xe3, 0xb2,0xb7,0x8a,0x9b,
				0xda,0x5b,0xb7,0xc6, 0x00,0xcb,0x40,0x47,
				0x0c,0x38,0x75,0xb8, 0xba,0x6f,0x2b,0x9d,
				0x01,0xf3,0xf2,0xc8, 0xf7,0xde,0xcf,0xfb,
				0x82,0xa8,0x8f,0x10, 0x75,0x0e,0x27,0xc5,
				0x4b,0x9f,0xfe,0x1d, 0x60,0x84,0x69,0x96,
				0xac,0xb1,0xd3,0xdd, 0x07,0x4c,0x50,0x94,
				0xb1,0x17,0x53,0x23, 0x98,0xbf,0x22,0xf9,
				0x2c,0xb0,0x3f,0x62, 0x16,0xa7,0x8f,0xea,
				0x43,0x25,0xfb,0x21, 0x18,0xec,0x1a,0xf6,
				0x5e,0x64,0xbd,0x3d, 0xcf,0x27,0xf5,0x02,
				0xf2,0xaf,0x1b,0x2d, 0x2c,0xcb,0xaa,0x6d,
				0x7d,0xa0,0xae,0x31, 0x05,0x51,0x80,0x7f,
				0x99,0xcf,0xbd,0x0f, 0x12,0x5a,0xda,0x4a,
				0x56,0x22,0xd4,0x22, 0x95,0x2c,0x46,0x5a,
				0xb3,0x5a,0x5e,0xd4, 0x27,0x7f,0x06,0xbd,
				0x3c,0xf6,0xf2,0x0f, 0x9d,0xbb,0x0c,0x14,
				0x8c,0xb1,0x72,0xf2, 0xb0,0xaf,0xda,0xf7,
				0x05,0x33,0x78,0x9c, 0x79,0xe9,0xe0,0xc5,
				0x8c,0x4b,0x23,0x65, 0xd1,0x70,0x81,0x3d,
				0x74,0xfa,0xb6,0xff, 0xf2,0x65,0x21,0x3f,
				0xe4,0xc2,0x9e,0x9d, 0x49,0x0e,0xad,0xaf,
				0xc2,0x21,0x18,0xa8, 0x19,0xa8,0x69,0x32,
				0xcb,0x8e,0xc2,0x9d, 0xf5,0xbd,0x50,0x60,
				0x72,0xa2,0xa6,0xad, 0xe6,0x6b,0xd2,0x01,
				0x52,0xf9,0xac,0x18, 0xfa,0xe8,0x8d,0x4a,
				0x98,0x25,0xd3,0xa8, 0x0e,0x97,0x2d,0xa3,
				0xf6,0xf1,0x34,0x7c, 0xf0,0x15,0x06,0x05,
				0x31,0xdf,0xc7,0x86, 0x54,0xfb,0x62,0xe2,
				0xd5,0x3b,0x72,0xd2, 0x70,0x7c,0x3c,0x62,
				0x2f,0xbd,0x47,0x0d, 0x20,0x97,0xf5,0x1f,
				0xa1,0xe8,0x4c,0x3e, 0x13,0xec,0xb3,0xcc,
				0xc9,0x15,0x01,0x23, 0xe5,0x1f,0x3b,0x2e,
				0xc5,0xdd,0x71,0xe3, 0xfa,0x6a,0x44,0x07,
				0x25,0x64,0xa5,0xa5, 0x16,0x64,0x14,0xb8,
				0x86,0xb1,0xae,0x6f, 0xc5,0xdb,0x6b,0xfa,
				0x0f,0x8f,0xc5,0x89, 0x57,0x52,0xeb,0xb3,
				0xca,0x4e,0x23,0xac, 0xbd,0xad,0xf5,0x77,
				0x58,0x72,0x18,0x2c, 0xb8,0x37,0x0b,0xfd,
				0xfd,0x04,0x49,0x4a, 0x7b,0x11,0x82,0x1b,
				0xc4,0x5f,0x54,0x46, 0x97,0xe9,0xac,0x64,
				0xa7,0x13,0x04,0x56, 0x5a,0x3b,0x17,0x2c,
				0x08,0xff,0xa4,0xe2, 0xe4,0x43,0x05,0xfa,
				0x94,0x3a,0xbc,0x24, 0xec,0xa8,0x89,0x02,
				0xd0,0xbc,0xcf,0x4a, 0xef,0x0f,0x90,0x50,
				0xfb,0x6a,0x25,0x4f, 0xdb,0x67,0x5b,0xd8,
				0xa1,0x1e,0x95,0x4d, 0xe5,0xd6,0xf3,0x22,
				0x2e,0x6f,0x01,0x50, 0xd8,0x2f,0x91,0x47,
				0x82,0x0e,0xae,0x18, 0xbf,0x3a,0xc9,0x5a,
				0x71,0xcf,0x5e,0xbf, 0x9e,0xec,0x1d,0x11,
				0x96,0x33,0x32,0x5e, 0x5e,0xee,0xc8,0xee,
				0x52,0x03,0xbc,0x8d, 0x97,0xd2,0x55,0xc5,
				0xaf,0x52,0xb0,0x55, 0x8f,0xb8,0x9b,0x83,
				0x60,0x9f,0x60,0x92, 0x47,0x1d,0xf2,0x6e,
				0xd1,0x93,0xfe,0xc2, 0x77,0x8c,0xb6,0x49,
				0x5e,0x3e,0xdb,0xb9, 0x7a,0x58,0x4d,0x18,
				0x66,0xc8,0xc2,0x67, 0xf8,0x37,0x7d,0x06,
				0x50,0xcc,0x42,0xab, 0x08,0x27,0x8e,0x81,
				0x6f,0xb3,0x03,0xbd, 0x41,0x11,0xeb,0x13,
				0xf1,0xaf,0xee,0x56, 0xae,0xb3,0x36,0x41,
				0xb8,0xc9,0x0a,0x96, 0x88,0x1d,0x98,0x25,
				0xc6,0x45,0xeb,0x76, 0x07,0xc1,0xfe,0xae,
				0xbc,0x26,0x1f,0xc4, 0x5f,0x70,0x0c,0xae,
				0x70,0x00,0xcf,0xc6, 0x77,0x5c,0x9c,0x24,
				0x8b,0x4b,0x83,0x32, 0x09,0xb7,0xb1,0x43,
				0x4a,0x01,0x42,0x04, 0x4d,0xca,0x5f,0x4e,
				0x9b,0x2b,0xa9,0xcb, 0x99,0x0b,0x0e,0x57,
				0x09,0xd6,0xe2,0xa0, 0xc1,0x12,0x79,0xf2,
				0x6f,0xe1,0x6c,0x7f, 0x0a,0x1a,0xec,0xc1,
				0x82,0x4a,0xf8,0x98, 0x22,0xc9,0x81,0x81,
				0x5d,0xf8,0x7d,0x9d, 0x86,0x97,0xdd,0x9e,
				0x8a,0xb5,0xce,0x6c, 0xfb,0x06,0xc3,0x8a,
				0x0d,0x53,0xda,0x12, 0x0c,0x4b,0x6f,0xa0,
				0x3f,0x8d,0xc3,0x07, 0x27,0x10,0xaf,0xc5,
				0x27,0xfe,0x64,0x17, 0x18,0xa5,0x3a,0xfe,
				0x9b,0x91,0xae,0xd0, 0x2d,0x34,0x34,0x9e,
				0x9f,0x31,0x5d,0x3e, 0x4c,0x26,0x1e,0xcb,
				0x62,0x05,0xd2,0x83, 0x8d,0x71,0xb8,0x57,
				0xef,0x3a,0x94,0xb3, 0x3a,0x67,0x1b,0x21,
				0x33,0x1f,0x7f,0x10, 0xd8,0xd7,0x89,0x1b,
				0x4f,0x51,0x74,0x97, 0x4a,0x0e,0x74,0x59,
				0x74,0x66,0xef,0xdd, 0x26,0xb6,0xa1,0x53,
				0xd4,0x2f,0xd7,0x76, 0x51,0x27,0xcc,0xe4,
				0x94,0xe3,0xed,0x26, 0x13,0x4e,0xe8,0x2c,
				0x11,0x6e,0xb3,0x63, 0x51,0x36,0x9c,0x91,
				0x2d,0x66,0x2c,0x3e, 0x0a,0xf7,0xa4,0x97,
				0x70,0x6d,0x04,0xaa, 0x89,0xe8,0x2c,0x5e,
				0xdd,0x01,0x46,0xfc, 0x99,0xce,0xe6,0x32,
				0x8a,0x85,0xe6,0x07, 0x1e,0x71,0x5d,0x29,
				0x07,0x16,0x0e,0xf9, 0xd4,0xdf,0x54,0xb4,
				0x7b,0x7b,0x3f,0xe0, 0xeb,0x73,0xe0,0xe1,
				0x92,0x51,0x50,0x74, 0xb5,0x6e,0x08,0x7e,
				0x57,0x70,0xb2,0x1b, 0x9c,0xf2,0xa2,0x6b,
				0x52,0xa3,0x35,0xf7, 0x22,0x40,0xa6,0x11,
				0x30,0xd3,0x5b,0x4b, 0x78,0xc9,0xd7,0x84,
				0x9a,0x88,0x9a,0x44, 0xb4,0x88,0xfe,0x8c,
				0x3f,0x10,0xab,0xc7, 0xc9,0xb6,0x59,0x9a,
				0xf3,0xe6,0xe6,0x4d, 0xea,0x3e,0xe0,0xeb,
				0x9e,0xb4,0x41,0xf6, 0xcb,0xfc,0x04,0x73,
				0x7d,0xc8,0x00,0xc6, 0xf2,0x10,0x00,0xcf,
				0x59,0xed,0x05,0x2a, 0x6a,0xde,0x7a,0xdf,
				0x7d,0xa9,0x25,0xc8, 0x6e,0x08,0x60,0xf9,
				0xd8,0x23,0x9b,0x20, 0xe5,0x93,0x9c,0x90,
				0x3d,0xe0,0xd0,0x33, 0x2d,0xce,0x86,0x93,
				0xdc,0xb3,0x9c,0x40, 0x33,0x9a,0xf0,0x71,
				0x47,0x0e,0xc4,0xb9, 0x58,0xc4,0x36,0xf1,
				0x4c,0x82,0xcf,0x91, 0x9f,0x16,0xce,0x43,
				0x58,0x72,0x54,0x51, 0x0d,0x8e,0x1e,0x3d,
				0x5e,0x67,0x7e,0x96, 0x6e,0x12,0xb8,0xee,
				0x1f,0x8b,0x15,0x3b, 0x49,0x95,0x2f,0xd9,
				0xec,0x63,0x56,0xec, 0x4e,0x88,0x37,0x2f,
				0xa7,0xd5,0xe5,0x4a, 0x97,0x1f,0x6f,0xa0,
				0x40,0x68,0x69,0xee, 0x6a,0xc6,0xbe,0x83,
				0xba,0x69,0xb8,0x08, 0x0a,0x5c,0x2f,0xd2,
				0x3e,0x3b,0x73,0x40, 0x9c,0x62,0xcc,0xe1,
				0x99,0x44,0xa2,0xaa, 0xb8,0xe9,0x48,0xf4,
				0x79,0x07,0xe8,0xe8, 0x16,0x99,0x84,0x7b,
				0x3d,0x53,0xb2,0x5d, 0x2d,0xa4,0xb0,0x12,
				0xb9,0xa9,0x0d,0x77, 0x98,0xa1,0x98,0x90,
				0x4e,0xe2,0x14,0xd4, 0x15,0x35,0xd0,0x85,
				0xbf,0xa1,0x0f,0x54, 0x05,0xa0,0x90,0x2a,
				0x74,0xe3,0xd3,0x1b, 0x5e,0x16,0x07,0xcf,
				0x36,0xbd,0xea,0x9b, 0x2d,0x35,0x47,0xea,
				0xea,0xb7,0xd1,0xda, 0x66,0x47,0x42,0x47,
				0x4e,0x76,0xe5,0x90, 0x0c,0x82,0x15,0x3f,
				0x17,0x1b,0xa6,0x04, 0xb6,0x58,0x67,0x42,
				0xfb,0x19,0x2a,0xc2, 0xd7,0x6a,0x48,0x36,
				0x87,0x53,0x90,0x95, 0x53,0xb7,0xf1,0xbe,
				0x0d,0x9f,0xa3,0x74, 0x5f,0x3d,0x89,0xef,
				0x29,0x07,0xe1,0xc1, 0x13,0xe0,0xc7,0xf6,
				0x53,0xc2,0xe5,0x7e, 0x96,0xdf,0x1f,0x12,
				0x98,0xd6,0x7b,0x2d, 0xdb,0x3e,0x01,0x03,
				0x05,0xbe,0x66,0x29, 0x42,0xeb,0x5d,0xab,
				0xa8,0x13,0x78,0x7f, 0x1e,0x0e,0xfd,0x7f,
				0xf1,0xd2,0x59,0xb2, 0x46,0x13,0x1c,0xb8,
				0x42,0x4f,0x87,0xb3, 0x26,0x0b,0xed,0x26,
				0xb2,0xd5,0x27,0xfc, 0xf1,0xec,0x32,0x66,
				0xe1,0x2d,0x27,0x2a, 0xe2,0x80,0xf2,0x72,
				0x90,0x3c,0x54,0xfa, 0xaa,0xe6,0x31,0xb0,
				0xb7,0xdd,0x97,0x0d, 0x22,0xb5,0x16,0x46,
				0x66,0x6d,0x02,0x13, 0x9a,0x7c,0x52,0xfc,
				0xf8,0x73,0x0c,0x81, 0xac,0xa3,0x8f,0x40,
				0x50,0x2e,0x80,0x3b, 0xb6,0xdf,0x88,0xbb,
				0xb5,0xa8,0x13,0xfa, 0xd2,0xd6,0xb8,0x07,
				0x47,0x7b,0xa0,0x09, 0x9f,0xc3,0x42,0xab,
				0xb8,0xd6,0xca,0xfa, 0x41,0xdc,0x9a,0xb5,
				0x96,0xf4,0xfa,0xfd, 0x09,0xca,0x8e,0x47,
				0x1d,0x8f,0x8d,0x54, 0x3f,0xbf,0xfd,0x22,
				0x30,0x25,0xbd,0xea, 0xb3,0xf6,0x90,0x68,
				0x6e,0x2b,0x78,0x8e, 0xc4,0x58,0x1c,0xbd,
				0x6b,0x36,0xdc,0x9d, 0x9f,0x27,0xce,0xf6,
				0x4f,0x1b,0xeb,0x41, 0x2c,0x07,0xa1,0x1f,
				0xaa,0xc3,0x65,0xe0, 0x78,0x85,0x80,0x22,
				0x00,0x94,0x1a,0x9f, 0x34,0x2b,0x2b,0x51,
				0x94,0x93,0x23,0x20, 0x48,0x2e,0x16,0xd6,
				0xdf,0x09,0xa2,0xfa, 0xb8,0x9b,0xf0,0x64,
				0x18,0x36,0x78,0xbc, 0xb8,0x5b,0x87,0x90,
				0xba,0xd2,0x2e,0x30, 0xe6,0xc5,0xe0,0x0c,
				0x81,0x32,0x69,0x9a, 0x8a,0x5a,0x3d,0x6f,
				0x06,0xe1,0x3f,0xa9, 0xf2,0x0e,0x21,0xfe,
				0x9e,0x63,0x31,0xa9, 0xc3,0x3e,0xb4,0xcd,
				0xcb,0x60,0xd9,0x45, 0xc6,0x5f,0xc5,0xca,
				0x9e,0xd8,0x40,0x72, 0x39,0x04,0x59,0x2d,
				0x4c,0xac,0xdf,0xea, 0x4a,0x78,0xa9,0xd5,
				0x87,0xb1,0xd6,0x59, 0x77,0x58,0x4d,0xa7,
				0xd3,0x9b,0xfc,0xe3, 0xdd,0x8d,0xf5,0x57,
				0x06,0xb3,0x96,0xf1, 0xbe,0xd9,0x07,0x54,
				0x36,0xa4,0x8b,0xaa, 0x0b,0xcb,0xd3,0x80,
				0x13,0xa6,0x53,0x8e, 0xcc,0x23,0x15,0x02,
				0x1e,0x1b,0x2f,0x0a, 0x02,0x5b,0xca,0x50,
				0x11,0x28,0x27,0x0e, 0xbe,0xfe,0x76,0x60,
				0x1b,0x78,0x58,0x9b, 0xe6,0x0a,0x0a,0xef,
				0xa3,0xa5,0x33,0x0d, 0x5b,0x65,0xe1,0x03,
				0x38,0xdd,0xf8,0x22, 0x92,0xcd,0x50,0x87,
				0x02,0xbc,0x91,0x16, 0xfd,0x05,0x9c,0xcd,
				0x72,0xae,0x4c,0xd7, 0xef,0xb3,0x57,0x1a,
				0x3f,0x79,0x23,0xfd, 0xf0,0xc3,0xfb,0x68,
				0xb4,0xc9,0x93,0x22, 0x33,0xd3,0x01,0x74,
				0xe3,0x00,0x31,0xcf, 0x0f,0x23,0xc5,0xf7,
				0x09,0x95,0x5a,0xa0, 0x56,0xf9,0xb0,0x20,
				0xb1,0xcc,0x8d,0x88, 0xd6,0x27,0x97,0x8d,
				0x0e,0xa3,0x3d,0x33, 0x94,0x04,0x44,0x93,
				0x67,0x10,0xb6,0xa0, 0x0c,0x2a,0x28,0xd4,
				0x1b,0x41,0x86,0xe7, 0x29,0x2c,0x68,0x2a,
				0x94,0xf3,0x4f,0x20, 0xa1,0xb4,0x6c,0x9d,
				0x85,0x6b,0xa0,0x31, 0xa2,0xbd,0x74,0xf0,
				0x0b,0xe5,0x2f,0xb7, 0x8a,0x33,0xd9,0x1f,
				0xf2,0xb5,0xad,0x85, 0xc3,0xad,0x47,0x2f,
				0x27,0x2a,0xc9,0x32, 0xd8,0xd9,0x05,0xc2,
				0x9d,0xbf,0x21,0x88, 0x02,0x05,0x12,0x6e,
				0x0f,0xb6,0x64,0x43, 0xa8,0xc3,0x87,0xea,
				0xb0,0x81,0x5b,0x51, 0x51,0xf1,0x83,0x7d,
				0x94,0x46,0x7f,0x0a, 0x9a,0xef,0xcc,0x68,
				0x73,0xef,0x9d,0x3c, 0x0e,0xfc,0x37,0x91,
				0xca,0x36,0x2d,0x1d, 0x72,0x7e,0x39,0x9e,
				0xad,0xd3,0x55,0x1b, 0x10,0x1e,0xff,0x00,
				0xc1,0x45,0x80,0xe7, 0xb4,0xcc,0xc8,0xb0,
				0x62,0xbd,0xf9,0xa5, 0x8f,0x05,0xaa,0x3b,
				0x86,0x73,0x14,0xf9, 0xee,0x95,0xd0,0xfd,
				0x95,0x30,0x68,0x22, 0xc9,0x70,0x66,0x1d,
				0x91,0x3f,0xc0,0x19, 0x93,0x07,0x19,0x2d,
				0x3c,0x21,0x6b,0xc1, 0x2a,0xeb,0xaa,0xf2,
				0xa4,0x45,0x35,0xff, 0x8f,0x24,0x46,0x2c,
				0xc8,0x75,0x58,0x68, 0x0f,0x3b,0x87,0x11,
				0xcb,0x9f,0xf7,0x28, 0xbd,0x66,0x91,0x01,
				0xeb,0x70,0x8e,0x8d, 0xe6,0x01,0xc8,0x48,
				0x94,0xfe,0x4e,0xa8, 0xeb,0x90,0xbf,0xd1,
				0xcd,0x89,0xc2,0x98, 0x34,0x92,0xf9,0x08,
				0xb9,0xbc,0xd4,0x34, 0x1a,0x59,0xcc,0x80,
				0x9a,0xe6,0xbc,0xbb, 0x23,0x12,0x9c,0xa4,
				0x5b,0x79,0xc6,0x8a, 0xc0,0x03,0x2b,0x16,
				0xe5,0x1c,0x0f,0x02, 0x37,0x4f,0x3e,0xc2,
				0xf3,0x4d,0x7c,0xcb, 0xde,0x9b,0x66,0x52,
				0xf3,0xdd,0x86,0x42, 0x4a,0x81,0x5b,0x96,
				0x83,0x2a,0xb1,0x48, 0x31,0x42,0x16,0x16,
				0xf8,0x97,0xa3,0x52, 0xeb,0xb6,0xbe,0x99,
				0xe1,0xbc,0xa1,0x3a, 0xdd,0xea,0x00,0xfa,
				0x11,0x2f,0x0b,0xf8, 0xc7,0xcc,0xba,0x1a,
				0xf3,0x36,0x20,0x3f, 0x59,0xea,0xf1,0xc8,
				0x08,0xd0,0x6d,0x8e, 0x91,0x1e,0x90,0x91,
				0x7b,0x80,0xdc,0xcb, 0x5c,0x94,0x74,0x26,
				0xd3,0x5d,0x1a,0x2d, 0xad,0xcf,0xef,0xfa,
				0xe9,0xa0,0x17,0xb7, 0x2b,0x7c,0x37,0x83,
				0x31,0x78,0x1a,0xcf, 0x04,0xa0,0xe7,0x83,
				0x66,0x12,0x4f,0x9d, 0x31,0x6b,0x4d,0xc5,
				0x31,0x1b,0x3a,0xd9, 0x79,0x76,0x49,0xc3,
				0x19,0xf0,0x3f,0xb5, 0xbc,0x7d,0xa4,0xa7,
				0x24,0x44,0x75,0xbb, 0x6d,0x65,0x59,0xf8,
				0xe0,0xb9,0xd7,0x29, 0x79,0xce,0x14,0x32,
				0xd2,0x3e,0xb8,0x22, 0x4a,0x0a,0x2a,0x6c,
				0xb2,0xbd,0xa5,0xd4, 0xc4,0xc5,0x68,0xb3,
				0x63,0xe7,0x46,0x05, 0x3a,0x18,0xa5,0xad,
				0xcc,0x61,0xc3,0xec, 0x3d,0x42,0xb0,0xa7,
				0x23,0x72,0x1e,0x14, 0xd8,0x7e,0x68,0x60,
				0xec,0xe9,0x1d,0x5b, 0x1f,0x86,0xda,0x5e,
				0x34,0x74,0x00,0xd3, 0x98,0x98,0x7e,0xbd,
				0x6a,0x8b,0xd3,0x6f, 0x31,0xf1,0x62,0xb3,
				0xa3,0x86,0x95,0x02, 0x76,0x7d,0x58,0xbc,
				0xf8,0xb1,0x52,0xc3, 0x0b,0xd5,0x6b,0x74,
				0xa5,0x84,0xef,0xf2, 0x31,0xc1,0xe4,0x83,
				0x42,0x12,0xb5,0xe7, 0x61,0xdd,0xba,0x43,
				0x39,0xf2,0x44,0x0a, 0xb4,0x62,0x06,0x32,
				0x5b,0x33,0x67,0x2e, 0x7a,0x93,0x85,0x1a,
				0x07,0x36,0x9f,0xab, 0xf7,0x2a,0x6e,0x3d,
				0x3e,0xe3,0x59,0x1b, 0xf8,0xd3,0xe8,0x5f,
				0xe5,0x24,0xb3,0x59, 0x80,0xd5,0x11,0x14,
				0x98,0x3a,0xb4,0x7d, 0x8f,0x37,0x18,0xb2,
				0xa7,0x25,0xf4,0x31, 0x74,0x61,0x3a,0x42,
				0x62,0x77,0x37,0x3d, 0x72,0x1b,0x67,0x87,
				0xb3,0x59,0x4b,0x08, 0x07,0xdb,0x0b,0x57,
				0xfd,0x61,0x99,0x28, 0x3b,0xe5,0x7a,0xb4,
				0x6c,0x06,0x95,0x65, 0x2c,0x1c,0x41,0x71,
				0x21,0xd7,0x94,0x51, 0x1c,0x8d,0xe6,0x38,
				0xc5,0x95,0x7f,0x30, 0xd5,0xc5,0xcc,0xd2,
				0x03,0x7f,0x69,0x2e, 0xae,0xc7,0x28,0x2e,
				0xc6,0xa9,0x28,0x4b, 0x77,0xc3,0xcf,0xa3,
				0xc3,0xd3,0x2d,0x43, 0x47,0x87,0xde,0x38,
				0xeb,0x3a,0xb6,0xf9, 0xe7,0x3c,0xb6,0x92,
				0x19,0x42,0xf8,0xc2, 0x87,0x50,0xed,0xe6,
				0x3d,0x2b,0xb5,0xf8, 0x89,0x14,0x42,0xf7,
				0x2c,0x7a,0xbe,0xdc, 0x2f,0x5d,0x49,0x83,
				0xf5,0x60,0xe0,0xcf, 0xbc,0x23,0x13,0x4f,
				0xb3,0x16,0xd7,0x9a, 0xca,0x16,0x8b,0xa5,
				0x08,0x80,0xcf,0x21, 0xbb,0xd8,0x32,0x5e,
				0x07,0x8a,0xb3,0x48, 0xba,0x99,0xd4,0xd7,
				0x6a,0xae,0x4b,0x9b, 0xb4,0xd7,0x2f,0x87,
				0xb0,0x0a,0xd1,0x1b, 0xf1,0x8b,0xf6,0x21,
				0x81,0x8e,0xc4,0x79, 0x9a,0x5c,0x75,0xbe,
				0x87,0x99,0xe5,0x11, 0xf9,0x9a,0xe1,0xf9,
				0x76,0xa2,0x92,0xc6, 0xc0,0xd8,0x05,0xc9,
				0x7d,0x8c,0x27,0xc2, 0x7f,0xf4,0xe9,0x4f,
				0xb7,0xbc,0xa3,0x3e, 0x66,0x3b,0xaf,0xed,
				0x7a,0xd9,0x78,0x20, 0x6b,0xd5,0xe1,0xfe,
				0xd5,0x06,0x65,0x11, 0x49,0xac,0x22,0x38,
				0x02,0x80,0xec,0x91, 0x11,0x18,0x1a,0x61,
				0x3c,0x59,0x4e,0x7a, 0xd8,0xca,0xda,0xd4,
				0x27,0xbd,0xf4,0x00, 0x9c,0x1b,0xde,0xf3,
				0x6c,0x1f,0x20,0x9a, 0x30,0xc9,0x9b,0x3c,
				0xe5,0x55,0xb7,0xb3, 0xc8,0x52,0x9c,0x05,
				0xad,0xe8,0x13,0x9e, 0x31,0xc2,0x2c,0xd4,
				0x3f,0x18,0x00,0xc4, 0xcf,0x08,0x05,0x7b,
				0x5e,0x2a,0x8e,0x11, 0x61,0x03,0xc8,0x39,
				0x2b,0x54,0x1a,0xd9, 0x08,0x04,0xc6,0xe9,
				0xda,0x69,0xb3,0x0c, 0x83,0x44,0xcd,0xe8,
				0x50,0x04,0x72,0xa2, 0xb4,0x10,0x17,0x39,
				0x68,0x32,0xdb,0xab, 0xe3,0xee,0x57,0x1b,
				0x05,0x45,0x1f,0x5a, 0xdc,0xdc,0x56,0x81,
				0x98,0x20,0xfe,0x69, 0x0a,0xa4,0xd6,0x9d,
				0x25,0xdd,0x7e,0xd0, 0x2b,0x33,0x41,0x75,
				0xf6,0x59,0xa8,0xa3, 0x3c,0xdd,0xd9,0x6b,
				0xa8,0xcd,0x1d,0x1f, 0xc5,0x78,0x5b,0x93,
				0xdf,0x10,0x71,0xeb, 0xcc,0xbd,0x35,0x4c,
				0x07,0x21,0x5f,0xb7, 0x47,0x21,0x6d,0x55,
				0x8b,0x72,0x0e,0x4a, 0x2c,0x17,0xfc,0x75,
				0x21,0xdd,0x76,0xfd, 0x34,0xfc,0x0f,0x1b,
				0xa6,0x77,0x53,0xf9, 0xdb,0x09,0x07,0x58,
				0xb0,0x18,0x32,0x03, 0x98,0x79,0xdf,0x55,
				0xd3,0x95,0xba,0xa9, 0xb6,0x9f,0xad,0xc4,
				0x9d,0xba,0x76,0x36, 0x47,0xb1,0xde,0x78,
				0x18,0xa0,0x2f,0x16, 0x41,0xeb,0x4a,0x96,
				0x82,0xc4,0xa4,0xde, 0x4b,0xdf,0xee,0xc7,
				0x33,0xdf,0xb7,0xde, 0xd3,0xa7,0x0f,0xc7,
				0x23,0x61,0x6b,0xd9, 0x15,0xc8,0x09,0xf7,
				0xe7,0xf9,0x44,0xba, 0x14,0xdc,0x94,0x5e,
				0xd9,0xcc,0x74,0xb2, 0x3d,0xef,0x78,0x15,
				0xb5,0xb9,0x56,0xd5, 0xfb,0x47,0x49,0x3a,
				0xbc,0x53,0x71,0x8b, 0x72,0x8b,0xb2,0xe3,
				0x58,0xbf,0xea,0x47, 0x7a,0x76,0x03,0x48,
				0xdd,0x8c,0x30,0x99, 0x81,0x2c,0x5f,0xf6,
				0xd3,0x9b,0x8e,0x77, 0x1c,0xb7,0xbd,0x1e,
				0xd4,0x28,0x05,0xf7, 0xff,0xdf,0xd6,0xb9,
				0x83,0x99,0xbc,0x94, 0xb7,0x41,0x93,0xc4,
				0x66,0xff,0x29,0x4d, 0x5c,0xba,0x79,0xd9,
				0x6e,0x79,0x47,0x45, 0xd6,0x2d,0xcd,0x79,
				0xa1,0xfa,0x49,0xee, 0x8e,0x7f,0x2b,0x08,
				0x3f,0x60,0x56,0xcf, 0xcb,0xe8,0x0d,0x55,
				0xee,0xa5,0xaf,0x04, 0xde,0x01,0xde,0xce,
				0xb6,0x9c,0x68,0x4e, 0xb0,0x88,0xcd,0x89,
				0x83,0x6b,0x01,0xb5, 0x78,0xac,0x85,0x3c,
				0x2c,0xcf,0x39,0xb6, 0xc8,0x5f,0x0e,0xac,
				0x02,0x08,0x56,0xbe, 0xd1,0x8d,0x7d,0x55,
				0x69,0x0c,0x33,0x33, 0xff,0x1a,0xd6,0x0b,
				0xcf,0x57,0x18,0x01, 0x56,0x5f,0x9c,0x6f,
				0xe2,0x24,0xda,0xc3, 0x9f,0x81,0xc3,0x27,
				0x46,0x7a,0xb4,0xae, 0xec,0xa4,0x0e,0x41,
				0x8b,0xb7,0x16,0xe3, 0x9b,0x2e,0x32,0x75,
				0xd9,0x86,0xa2,0x13, 0x68,0x4e,0xbc,0x43,
				0xa2,0x78,0x64,0x1a, 0x7c,0xac,0x13,0x70,
				0x1c,0x23,0x15,0x5b, 0xda,0x99,0xa5,0x24,
				0x3d,0xcf,0x29,0xf7, 0xbc,0x1d,0x10,0xe8,
				0x95,0x1a,0x11,0xec, 0xfc,0xfb,0x20,0x1f,
				0x09,0x1b,0xe3,0x3d, 0xae,0x82,0x70,0xd7,
				0x9e,0xf3,0x18,0x97, 0x89,0xfa,0x42,0x67,
				0x70,0x9c,0xc8,0xbe, 0x62,0x98,0xf1,0x82,
				0xfc,0x2b,0xf0,0x40, 0xaa,0xdc,0x27,0xf9,
				0x21,0x5a,0xc1,0x25, 0x8b,0xef,0xd5,0x48,
				0x6c,0x68,0xae,0xbc, 0xcd,0xa9,0x3c,0x1e,
				0xe9,0xcf,0xe2,0xd1, 0xc0,0x98,0xa9,0x62,
				0x5d,0x1f,0x57,0x7a, 0xca,0x8a,0x0f,0xfb,
				0xe3,0xc9,0x7e,0x98, 0x44,0x84,0x67,0x12,
				0x60,0x60,0xe5,0xc7, 0xcc,0x72,0x90,0x64,
				0x67,0x30,0x6a,0xd8, 0xa1,0x11,0xd5,0x7e,
				0x5e,0x0c,0x74,0xa2, 0x6f,0x0a,0xff,0x41,
				0xd3,0x9a,0x30,0x56, 0xd4,0xec,0x9a,0x5f,
				0x22,0x71,0x6b,0x4e, 0xe6,0xe0,0x19,0x69,
				0x56,0x4a,0xba,0x9d, 0x50,0x8a,0x73,0x6a,
				0xf1,0x59,0x48,0xd6, 0xcd,0xfa,0xaa,0x0c,
				0xbb,0x7c,0xa4,0xbc, 0xf5,0x32,0x95,0x55,
				0x1c,0xe9,0x9a,0x60, 0x43,0x10,0xbd,0x27,
				0x88,0x2f,0x05,0xcf, 0xce,0x21,0x25,0x3a,
				0x07,0xab,0x37,0xfd, 0xf6,0x2f,0xd6,0x51,
				0xbe,0xe6,0xcc,0x58, 0x3a,0xab,0x60,0x23,
				0x45,0xa0,0xe5,0x79, 0xe5,0xaa,0xed,0xa4,
				0x28,0xd0,0x4d,0x37, 0x9c,0x6a,0xd7,0xc2,
				0x39,0x22,0xb9,0x3e, 0x0d,0xb8,0x94,0x65,
				0x48,0x4d,0x4c,0x02, 0x31,0x7e,0x9c,0xc9,
				0xb7,0xd6,0x23,0x1a, 0x94,0x5a,0x13,0x55,
				0x78,0x7a,0x29,0x4a, 0xa2,0xfd,0x37,0x24,
				0xd8,0xd0,0x9e,0x47, 0x24,0xab,0x26,0x34,
				0x28,0xb5,0x2d,0x82, 0x9a,0x4d,0xdd,0x17,
				0x68,0xe0,0x07,0x5d, 0xb9,0x2d,0xff,0xa9,
				0x0c,0x11,0x59,0x75, 0xda,0x98,0xe9,0xd5,
				0xfa,0xb5,0x18,0x16, 0x28,0x17,0x7c,0xad,
				0xab,0xee,0x65,0x10, 0x13,0x0d,0x26,0xfa,
				0x7f,0xac,0x06,0x43, 0x4d,0x5d,0x3a,0xf4,
				0x77,0xe7,0x03,0x17, 0x39,0x9f,0xbe,0x52,
				0x9b,0x68,0x2b,0x7f, 0xd3,0xa2,0x7e,0x5c,
				0x78,0x22,0xc5,0xe3, 0x17,0x73,0xc6,0x9e,
				0x68,0x17,0x74,0x50, 0xf4,0xc5,0xa8,0xc3,
				0x66,0xe1,0x05,0xed, 0xdd,0xdb,0xd3,0x11,
				0x16,0xad,0x05,0x3a, 0x38,0x55,0x1c,0xf0,
				0x93,0x0b,0x22,0x83, 0xc8,0x34,0xc5,0x43,
				0x4d,0x65,0x57,0xf3, 0x03,0x56,0x21,0xa9,
				0xbd,0x04,0x41,0x49, 0x62,0xfd,0xcc,0xc2,
				0x75,0x59,0x09,0xb9, 0x28,0x38,0xcf,0xfb,
				0x54,0x64,0x51,0xc2, 0x3e,0xad,0x35,0x3e,
				0x31,0x87,0x6e,0xfe, 0xf0,0x41,0xef,0x1d,
				0xb8,0x46,0xbe,0x85, 0xb9,0xff,0xa3,0xdb,
				0x87,0xf9,0x65,0x95, 0x60,0x53,0x7c,0x9d,
				0x26,0x83,0xfc,0xa7, 0xad,0x5a,0xcb,0x8d,
				0x81,0xec,0x28,0xeb, 0xdd,0x96,0x25,0x31,
				0x24,0x3f,0x59,0x28, 0x60,0x0b,0xc0,0x59,
				0xea,0x36,0x15,0xad, 0x70,0xd8,0x70,0xff,
				0x9b,0x15,0x76,0xc5, 0x84,0xe6,0x81,0x75,
				0x1a,0x1e,0xc9,0xec, 0x33,0xbe,0x10,0xd4,
				0x6f,0x10,0x1b,0xa2, 0xdb,0xc6,0x1b,0x0a,
				0xfb,0xe9,0x3f,0x4d, 0x04,0x4e,0x33,0x87,
				0xb3,0x21,0xad,0x41, 0xbe,0xce,0x26,0x0c,
				0x0c,0x84,0x0f,0x9a, 0xb9,0xa7,0xa2,0x36,
				0x70,0x49,0xce,0x25, 0x0f,0x69,0x4a,0x4a,
				0x3d,0xf5,0xa0,0x9e, 0xad,0x69,0x2d,0x79,
				0xdb,0x8b,0x85,0xf6, 0xb8,0x55,0xcd,0xf1,
				0xbb,0x04,0x35,0xad, 0xa8,0xb6,0x0d,0x3f,
				0x23,0xec,0x39,0xd7, 0xef,0x02,0x95,0x42,
				0x11,0xc9,0x70,0xc6, 0xa4,0x65,0x37,0x4d,
				0x9f,0x51,0x99,0xd6, 0x9e,0xb1,0x18,0xcf,
				0x31,0x81,0xde,0x95, 0x0a,0x8c,0x0c,0x80,
				0xdc,0xf7,0x19,0x5d, 0xdc,0x3e,0xee,0x0c,
				0x17,0xaf,0xc4,0x9c, 0xbf,0x65,0xf2,0xe1,
				0xc9,0xdb,0xc0,0x2a, 0xd0,0xbd,0xa1,0x7f,
				0x4b,0x9c,0x5b,0xe6, 0x91,0x98,0xa6,0xdb,
				0x72,0xef,0x14,0x38, 0x24,0x77,0x1e,0x71,
				0x74,0x63,0x0c,0xd9, 0x16,0x90,0x23,0x4a,
				0xe6,0xa4,0xc1,0x53, 0x8b,0xb4,0x7e,0x90,
				0x1b,0x68,0x32,0x48, 0x93,0xd8,0x72,0x43,
				0x8e,0x32,0x09,0x1e, 0x48,0xfc,0x3a,0xc6,
				0x15,0xb9,0x79,0x57, 0x02,0x61,0xc6,0x4b,
				0x56,0x1e,0x68,0x4e, 0x65,0x26,0xe5,0x1c,
				0xb1,0xd1,0x86,0x1d, 0xea,0x93,0x5a,0x88,
				0x4c,0x3b,0x10,0xd1, 0xf7,0x5a,0x4c,0xa3,
				0xe7,0x59,0xf5,0x04, 0x7d,0xd7,0xe3,0x2e,
				0x2c,0x3e,0x14,0x14, 0x83,0xed,0x3d,0x0b,
				0xa4,0xab,0x65,0xcf, 0x39,0xee,0xbe,0x0c,
				0x5e,0x4b,0x62,0x5e, 0xb4,0xd2,0x16,0xc7,
				0xe0,0x71,0x2b,0x92, 0x1e,0x21,0x45,0x02,
				0xfd,0xa1,0xda,0x0b, 0xbe,0xa6,0xe5,0x7f,
				0x31,0x8b,0x5a,0xcb, 0x8f,0xb8,0x0c,0xfb,
				0x7f,0x2d,0x7e,0xa2, 0x14,0xfd,0xe0,0xbb,
				0xa4,0x1b,0xce,0x81, 0x6f,0x25,0xbd,0x72,
				0x44,0x00,0x13,0x18, 0x75,0x04,0xf3,0x06,
				0xdc,0xf1,0x5b,0xa0, 0xb1,0x5a,0x9a,0xd8,
				0x4f,0xe7,0x94,0xe1, 0x65,0xe5,0xb2,0xd1,
				0x47,0x6d,0xd8,0x81, 0x22,0x96,0x09,0xd8,
				0x5e,0x12,0x73,0x62, 0xd6,0x2c,0xcb,0x45,
				0x71,0xa9,0xc1,0x21, 0x16,0x6f,0xf0,0xaa,
				0xce,0x19,0x1f,0x68, 0xee,0x17,0x07,0x94,
				0x4f,0x93,0x9a,0x12, 0xf7,0x91,0xe1,0xc6,
				0x9c,0x29,0xe5,0x06, 0x7a,0x40,0xf5,0xf6,
				0x51,0xc8,0x32,0x94, 0x52,0xd9,0x6b,0x9b,
				0x3e,0xb5,0xcf,0x1a, 0xf1,0x6c,0x7b,0x0a,
				0x16,0x47,0xee,0xa6, 0x46,0x0f,0xed,0xe0,
				0x1b,0x3f,0x39,0xfa, 0x4c,0x69,0xeb,0xfb,
				0xd0,0x36,0x3b,0x3a, 0x04,0x94,0xa4,0x2f,
				0x51,0xe1,0x1a,0x47, 0xc9,0xdb,0xf6,0x09,
				0xab,0x35,0x46,0x2c, 0x2f,0xb7,0x19,0xed,
				0x55,0x7e,0xa3,0x2c, 0xec,0xff,0x39,0xba,
				0x0f,0xfb,0x4f,0x8b, 0xfc,0x36,0x4e,0x5e,
				0xa1,0xe8,0x49,0x15, 0x65,0xd2,0xfb,0x11,
				0x4b,0x10,0xe6,0x07, 0x82,0x3a,0x5d,0x3f,
				0xeb,0xc0,0x0b,0x76, 0x66,0xb5,0xed,0x65,
				0xb3,0x9d,0x06,0x13, 0x3b,0x18,0x70,0x7a,
				0xbd,0xf7,0xd8,0x20, 0x81,0xc7,0x76,0x2e,
				0x21,0x6f,0xdb,0x8e, 0xba,0x83,0x42,0xb1,
			},
		},
		[5] = {
			.k = {
				0x79,0xce,0xb0,0x8e, 0xf8,0x7a,0x67,0xc6,
				0x48,0x2c,0x2a,0xc0, 0xa5,0x45,0x06,0x49,
				0xc8,0x90,0xb8,0xe9, 0xc6,0xb6,0xb3,0x50,
				0xbd,0x9e,0x46,0x56, 0x26,0xf2,0xb0,0x3b,
			},
			.tlen = 17,
			.t = {
				0xe6,0x93,0xbe,0x89, 0xf5,0xee,0x40,0xde,
				0xf2,0x9c,0xb5,0xec, 0x6a,0x37,0x23,0x46,
				0x0e,
			},
			.len = 16,
			.p = {
				0x5d,0x83,0x98,0x37, 0xc6,0x33,0x9e,0x7e,
				0x59,0xad,0xd2,0x5b, 0x8a,0x3a,0x9d,0x03,
			},
			.c = {
				0x96,0x23,0x2f,0x7d, 0x52,0xfc,0x98,0x63,
				0x98,0xa5,0x8b,0xdf, 0xca,0xbc,0x85,0x2f,
			},
		},
		[6] = {
			.k = {
				0x9f,0xd3,0x36,0xb1, 0x85,0x07,0xdf,0x19,
				0x01,0xea,0xf9,0x52, 0x68,0xbf,0xce,0xe7,
				0xd0,0x49,0xf3,0xba, 0x58,0xfb,0x87,0x18,
				0x9f,0xca,0x24,0xca, 0x61,0xa3,0xf0,0xda,
			},
			.tlen = 17,
			.t = {
				0xea,0xc6,0x72,0x5e, 0x66,0xd4,0xc7,0xbd,
				0xa1,0x6e,0xab,0x09, 0xb5,0x58,0x39,0xae,
				0x40,
			},
			.len = 128,
			.p = {
				0xc7,0xd6,0x73,0x65, 0xcb,0xf3,0xf5,0x3e,
				0xb9,0xa7,0xbf,0xb1, 0x54,0xcb,0xac,0x01,
				0xee,0xb5,0x94,0x17, 0x40,0x92,0xfd,0xad,
				0x8f,0xdb,0x27,0x22, 0x3d,0xb1,0x0b,0xf7,
				0xa7,0x46,0x70,0xd0, 0x31,0xdb,0xf9,0xdb,
				0xb9,0xb9,0x40,0x4a, 0x0a,0xba,0x77,0x6f,
				0x35,0x36,0x9e,0xeb, 0x68,0xe2,0x9e,0xd7,
				0xef,0xc2,0x5e,0x21, 0x0d,0xb3,0xb0,0x87,
				0xd6,0x43,0x35,0x6e, 0x22,0xa0,0xb7,0xec,
				0x26,0xe0,0x7d,0x48, 0xf5,0x5d,0x58,0xd3,
				0x29,0xb7,0x1f,0x7e, 0xe9,0x5a,0x02,0xa4,
				0xb1,0xde,0x10,0x9f, 0xe1,0xa8,0x5e,0x05,
				0xb6,0xa2,0x59,0xca, 0x3e,0xbc,0xd1,0x94,
				0x09,0x4e,0x1b,0x37, 0x29,0x9c,0x15,0xef,
				0x8c,0x72,0x53,0xbe, 0x6f,0x25,0x2c,0x68,
				0x88,0x08,0x0c,0x00, 0x80,0x7a,0x85,0x64,
			},
			.c = {
				0x49,0x36,0x97,0xd2, 0xde,0xa4,0xde,0x92,
				0x7d,0x30,0x08,0xc3, 0xd9,0x47,0xd4,0xcb,
				0x5b,0x41,0x27,0x2c, 0x06,0xb8,0x2b,0xef,
				0x7b,0x57,0x59,0xb7, 0x5b,0x81,0x38,0xb4,
				0xd1,0x81,0xb3,0xe8, 0xac,0xf0,0xa0,0x06,
				0xcb,0x74,0x31,0x01, 0xe1,0x3d,0xcf,0x6d,
				0x57,0xd1,0x65,0xcd, 0xe7,0x33,0x6c,0x03,
				0x54,0xf0,0x2c,0x41, 0xb8,0x75,0x07,0x1d,
				0x70,0xf0,0x9c,0xbd, 0x8f,0x6b,0xdb,0x76,
				0x86,0x5b,0xe0,0xfd, 0xad,0x61,0x7a,0x4c,
				0xd6,0xf1,0x85,0x0b, 0xfd,0x0b,0x3a,0x5f,
				0xcf,0xfc,0xb0,0x0b, 0x2b,0xc7,0x31,0x07,
				0x9d,0x75,0x82,0xd9, 0x14,0xd4,0x33,0xd3,
				0xff,0x20,0xf7,0x14, 0xcf,0xe4,0xda,0xca,
				0x11,0xcc,0x57,0x8f, 0x51,0x52,0x9d,0x90,
				0x01,0xc8,0x4e,0x1f, 0x2a,0x89,0xe2,0x52,
			},
		},
	};
	static struct adiantum A;
	static uint8_t buf[4096];
	unsigned i;
	int result = 0;

	for (i = 0; i < __arraycount(C); i++) {
		adiantum_init(&A, C[i].k);
		adiantum_enc(buf, C[i].p, C[i].len, C[i].t, C[i].tlen, &A);
		if (memcmp(buf, C[i].c, C[i].len)) {
			char prefix[16];
			snprintf(prefix, sizeof prefix, "adiantum enc %u", i);
			hexdump(printf, prefix, buf, C[i].len);
			result = -1;
		}
		memset(buf, 0, sizeof buf); /* paranoia */
		adiantum_dec(buf, C[i].c, C[i].len, C[i].t, C[i].tlen, &A);
		if (memcmp(buf, C[i].p, C[i].len)) {
			char prefix[16];
			snprintf(prefix, sizeof prefix, "adiantum dec %u", i);
			hexdump(printf, prefix, buf, C[i].len);
			result = -1;
		}
	}

	return result;
}
