/*
 * java.lang.Object.c
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "lib-license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>, 1996.
 */

/*** CHANGELOG ***
 *
 * 21.1.1998   Vesa Karpijoki   Added #includes <u.h> and <libc.h>.
 *                              Removed some #includes.
 *
 *  2.2.1998   Vesa Karpijoki   Fixed bugs.
 *
 * 18.3.1998   Vesa Karpijoki   Added Dokumatic documentation.
 *                              Changed #include:s.
 */

#include <u.h>
#include <libc.h>

#include "config.h"
#include "config-std.h"
#include "config-mem.h"
#include "../../../src/gtypes.h"
#include "../../../src/locks.h"
#include "../../../src/object.h"
#include "../../../src/soft.h"
#include <native.h>
#include "../../../src/constants.h"
#include "../../../src/access.h"
#include "../../../src/classMethod.h"	/* Don't need java.lang.Object.h */

/**
  @title java_lang_Object
  @desc Native methods of the Java API class java.lang.Object.
  @funcidx
  @end
  */

/**
  @function java_lang_Object_hashCode
  @description Generate object hash code.
  @parameter Reference to the object.
  @rvalue Hash value of the object.
  @end
  */
jint java_lang_Object_hashCode(struct Hjava_lang_Object* o)
{
	/* Hash code is object's address */
	return ((jint)(jword)o);
}

/**
  @function java_lang_Object_getClass
  @description Return class object for this object.
  @parameter Reference to the object.
  @rvalue Reference to the class.
  @end
  */
struct Hjava_lang_Class* java_lang_Object_getClass(struct Hjava_lang_Object* o)
{
	return (OBJECT_CLASS(o));
}

/**
  @function java_lang_Object_notifyAll
  @description Notify threads waiting here.
  @parameter Reference to the object.
  @end
  */
void java_lang_Object_notifyAll(struct Hjava_lang_Object* o)
{
	broadcastCond(o);
}

/**
  @function java_lang_Object_notify
  @description Notify a thread waiting here.
  @parameter Reference to the object.
  @end
  */
void java_lang_Object_notify(struct Hjava_lang_Object* o)
{
	signalCond(o);
}

/**
  @function java_lang_Object_clone
  @description Clones the given object.
  @parameter Reference to the object to be cloned.
  @rvalue Reference to the cloned object.
  @end
  */
struct Hjava_lang_Object* java_lang_Object_clone(struct Hjava_lang_Object* o)
{
	Hjava_lang_Object* obj;
	Hjava_lang_Class* class;
	static Hjava_lang_Class* cloneClass;
	int i;

	SET(i);
	USED(i);

	/* * Lookup cloneable class interface if we don't have it  */
	if (cloneClass == 0) {
		cloneClass = lookupClass("java/lang/Cloneable");
	}

	class = OBJECT_CLASS(o);

	if (!CLASS_IS_ARRAY(class)) {
		/* Check class is cloneable and throw exception if it isn't */
		if (soft_instanceof(cloneClass, o) == 0) {
			SignalError(0, "java.lang.CloneNotSupportedException", class->name->data);
		}
		/* Clone an object */
		obj = newObject(class);
		memcpy(OBJECT_DATA(obj), OBJECT_DATA(o), CLASS_FSIZE(class) - sizeof(Hjava_lang_Object));
	}
	else {
		/* Clone an array */
		obj = newArray(CLASS_ELEMENT_TYPE(class), ARRAY_SIZE(o));
		memcpy(ARRAY_DATA(obj), ARRAY_DATA(o),
		       ARRAY_SIZE(o) * TYPE_SIZE(CLASS_ELEMENT_TYPE(class)));
	}
	return (obj);
}

/**
  @function java_lang_Object_wait
  @description Wait for this object to be notified.
  @parameter Reference to the object that must wait.
  @parameter Timeout value.
  @end
  */
void java_lang_Object_wait(struct Hjava_lang_Object* o, jlong timeout)
{
	waitCond(o, timeout);
}
