/*
 * java.lang.UNIXProcess.c
 *
 * Copyright (c) 1996-7 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "lib-license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>
 **/

/*** CHANGELOG ***
 *
 * 21.1.1998   Vesa Karpijoki   Added #includes <u.h> and <libc.h>.
 *                              In Plan9 the forked child process inherits
 *                              the environment (variables) automatically,
 *                              so we cannot excplicitly give the child
 *                              certain environment variables (such system
 *                              call as execve() simply does not exist).
 *                              Replaced UNIX's kill() with Plan9's proc.
 *
 *  2.2.1998   Vesa Karpijoki   Replaced waitpid() with Plan9's call wait()
 *                              and Waitmsg struct. Added SET- and USED-calls.
 *
 * 18.3.1998   Vesa Karpijoki   Added Dokumatic documentation.
 *                              Changed #include:s.
 */

#define	DBG(s)

#include <u.h>
#include <libc.h>

#include "plan9interface.h"
#include "config.h"
#include "config-std.h"
#include "config-mem.h"
#include "config-io.h"
#include "config-signal.h"
#include "files.h"
#include "defs.h"
#include "../java.io.stubs/FileDescriptor.h"
#include "../java.lang.stubs/UNIXProcess.h"
#include "../../../src/support.h"

/**
  @title java_lang_UNIXProcess
  @desc Native methods of the Java API class java.lang.UNIXProcess.
  @funcidx
  @end
  */

extern void* gc_malloc_fixed(unsigned int);

/**
  @function java_lang_UNIXProcess_exec
  @description Execute a process.
  @parameter Reference to the current process (this). 
  @parameter Reference to the argument array.
  @parameter Reference to the environment variable array (ignored in Plan9).
  @end
  */
void java_lang_UNIXProcess_exec(struct Hjava_lang_UNIXProcess* this, HArrayOfObject* args, HArrayOfObject* envs)
{
	char** argv;
	int i;
	char* path;
	int arglen;

	/* In Plan9 the child process inherits the environment automagically,
	   so "envs" is not needed. USED() and SET() prevent warnings given by
	   the Plan9 compiler */
SET(this);
USED(this);
SET(envs);
USED(envs);

	arglen = (args ? obj_length(args) : 0);

DBG(	printf("args %d\n", arglen); fflush(stdout);	)

	if (arglen < 1) {
		SignalError(0, "java.io.IOException", "No such file");
	}
	path = makeCString((struct Hjava_lang_String*)unhand(args)->body[0]);

	/* Check program exists and we can execute it **/
        i = access(path, X_OK);
	if (i < 0) {
		gc_free(path);
		SignalError(0, "java.io.IOException", SYS_ERROR);
	}

	/* Build arguments **/
	argv = calloc(arglen + 1, sizeof(char*));
	for (i = 0; i < arglen; i++) {
		argv[i] = makeCString((struct Hjava_lang_String*)unhand(args)->body[i]);
	}

	/* Execute program **/
	exec(path, argv);

	exit(-1);
}

/**
  @function java_lang_UNIXProcess_fork
  @description Fork a process.
  @parameter Reference to the current process (this).
  @rvalue Id of the new forked process.
  @end
  */
jint java_lang_UNIXProcess_fork(struct Hjava_lang_UNIXProcess* this)
{
	int pid;
	int in[2];
	int out[2];
	int err[2];
	int sync[2];
	char b;

SET(this);
USED(this);

	/* Create the pipes to communicate with the child **/
	pipe(in);
	pipe(out);
	pipe(err);
	pipe(sync);

	pid = fork();

	switch (pid) {
	case 0:
		/* Child **/
		dup(in[0], 0);
		dup(out[1], 1);
		dup(err[1], 2);

		/* What is sync about anyhow?  Well my current guess is that
		 * the parent writes a single byte to it when it's ready to
		 * proceed.  So here I wait until I get it before doing
		 * anything.
		 **/
		read(sync[0], &b, 1);

		close(in[0]);
		close(in[1]);
		close(out[0]);
		close(out[1]);
		close(err[0]);
		close(err[1]);
		close(sync[0]);
		close(sync[1]);
		break;
	case -1:
		/* Error **/
		close(in[0]);
		close(in[1]);
		close(out[0]);
		close(out[1]);
		close(err[0]);
		close(err[1]);
		close(sync[0]);
		close(sync[1]);
		SignalError(0, "java.io.IOException", "Fork failed");
		break;
	default:
		/* Parent **/
		unhand(unhand(this)->stdin_fd)->fd = in[1];
		unhand(unhand(this)->stdout_fd)->fd = out[0];
		unhand(unhand(this)->stderr_fd)->fd = err[0];
		unhand(unhand(this)->sync_fd)->fd = sync[1];
		close(in[0]);
		close(out[1]);
		close(err[1]);
		close(sync[0]);
		unhand(this)->isalive = 1;
		unhand(this)->exit_code = 0;
		unhand(this)->pid = pid;
		break;
	}
	return (pid);
}

/**
  @function java_lang_UNIXProcess_forkAndExec
  @description Fork a process.
  @parameter Reference to the current process (this).
  @parameter Reference to the argument array.
  @parameter Reference to the environment variable array (ignored in Plan9).
  @rvalue Id of the new forked and executed process.
  @end
  */
jint java_lang_UNIXProcess_forkAndExec(struct Hjava_lang_UNIXProcess* this, HArrayOfObject* args, HArrayOfObject* envs)
{
	int pid;
	int in[2];
	int out[2];
	int err[2];
	int sync[2];
	char** argv;
	int i;
	char* path;
	int arglen;
	char b;

	/* In Plan9 the child process inherits the environment automagically,
	   so "envs" is not needed. USED() and SET() prevent warnings given by
	   the Plan9 compiler */
SET(this);
USED(this);
SET(envs);
USED(envs);

	/* Create the pipes to communicate with the child **/
	pipe(in);
	pipe(out);
	pipe(err);
	pipe(sync);

	pid = fork();

	switch (pid) {
	case 0:
		/* Child **/
		dup(in[0], 0);
		dup(out[1], 1);
		dup(err[1], 2);

		/* What is sync about anyhow?  Well my current guess is that
		 * the parent writes a single byte to it when it's ready to
		 * proceed.  So here I wait until I get it before doing
		 * anything.
		 **/
		read(sync[0], &b, 1);

		close(in[0]);
		close(in[1]);
		close(out[0]);
		close(out[1]);
		close(err[0]);
		close(err[1]);
		close(sync[0]);
		close(sync[1]);
		break;
	case -1:
		/* Error **/
		close(in[0]);
		close(in[1]);
		close(out[0]);
		close(out[1]);
		close(err[0]);
		close(err[1]);
		close(sync[0]);
		close(sync[1]);
		SignalError(0, "java.io.IOException", "Fork failed");
		return -1;
	default:
		/* Parent **/
		unhand(unhand(this)->stdin_fd)->fd = in[1];
		unhand(unhand(this)->stdout_fd)->fd = out[0];
		unhand(unhand(this)->stderr_fd)->fd = err[0];
		unhand(unhand(this)->sync_fd)->fd = sync[1];
		close(in[0]);
		close(out[1]);
		close(err[1]);
		close(sync[0]);
		unhand(this)->isalive = 1;
		unhand(this)->exit_code = 0;
		unhand(this)->pid = pid;
		return (pid);
	}

	arglen = (args ? obj_length(args) : 0);

DBG(	printf("args %d\n", arglen); fflush(stdout);	)

	if (arglen < 1) {
		SignalError(0, "java.io.IOException", "No such file");
	}
	path = makeCString((struct Hjava_lang_String*)unhand(args)->body[0]);

	/* Check program exists and we can execute it **/
        i = access(path, X_OK);
	if (i < 0) {
		gc_free(path);
		SignalError(0, "java.io.IOException", SYS_ERROR);
	}

	/* Build arguments **/
	argv = calloc(arglen + 1, sizeof(char*));
	for (i = 0; i < arglen; i++) {
		argv[i] = makeCString((struct Hjava_lang_String*)unhand(args)->body[i]);
	}

	/* Execute program **/
	exec(path, argv);

	exit(-1);
	return 0;
}


/**
  @function java_lang_UNIXProcess_waitForUNIXProcess
  @description Wait for a process.
  @parameter Reference to the current process (this).
  @rvalue Exit code of this process.
  @end
  */
jint java_lang_UNIXProcess_waitForUNIXProcess(struct Hjava_lang_UNIXProcess* this)
{
	Waitmsg msg;
	sprint(msg.pid, "%d", unhand(this)->pid);

	if (unhand(this)->isalive == 0) {
		if (wait(&msg) != -1) {
		    unhand(this)->isalive = 1;
		    unhand(this)->exit_code = 0;
		}

	}
	return (unhand(this)->exit_code);
}

/**
  @function java_lang_UNIXProcess_destroy
  @description Destroy a process.
  @parameter Reference to the current process (this).
  @end
  */
void java_lang_UNIXProcess_destroy(struct Hjava_lang_UNIXProcess* this)
{
    /*	kill(unhand(this)->pid, SIGTERM); */

   char procfile[32];  /* pid takes max 10 chars */
   int fd;

   sprintf(procfile, "/proc/%d/ctl", unhand(this)->pid);
   fd = open(procfile, OWRITE);

   if (fd < 0) SignalError (0, "java.io.IOException", SYS_ERROR);

   if (write(fd, "kill", 4) != 4) SignalError (0, "java.io.IOException",
					       SYS_ERROR);

   close(fd);
}

/**
  @function java_lang_UNIXProcess_run
  @description Run a process. Unimplemented!
  @parameter Reference to the current process (this).
  @end
  */
void java_lang_UNIXProcess_run(struct Hjava_lang_UNIXProcess* this)
{
    SET(this);
    USED(this);

    unimp("java.lang.UNIXProcess:run not implemented"); /* FIXME **/
}

/**
  @function java_lang_UNIXProcess_notifyReaders
  @description Notify readers of a process. Unimplemented!
  @parameter Reference to the current process (this).
  @end
  */
void java_lang_UNIXProcess_notifyReaders(struct Hjava_lang_UNIXProcess* this)
{
    SET(this);
    USED(this);

    unimp("java.lang.UNIXProcess:notifyReaders not implemented"); /* FIXME **/
}
