/*
 * java.net.InetAddress.c
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "lib-license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>, 1996.
 */

/*** CHANGELOG ***
 *
 * 13.1.1998   Olli-Pekka Auvinen   Removed #include <netinet/in.h> and
 *                                  #include <netdb.h>.
 *                                  Rewrote getLocalHostName,
 *                                  Checked makeAnyLocalAddress, getInetFamily.
 *
 * 17.1.1998   Olli-Pekka Auvinen   Rewrote lookupHostAddr, lookupAllHostAddr
 *                                  and getHostByAddr as unimplemented in
 *                                  plan9.
 *
 * 30.1.1998   Olli-Pekka Auvinen   Added #include "plan9interfaces.h"
 *                                  #include "java.net.stubs/hostresolution.h"
 *                                  Added lookupHostAddr, lookupAllHostAddr
 *                                  and getHostByAddr functions.
 *
 * 10.2.1998   Olli-Pekka Auvinen   Added the dokumatic documentation. 
 *
 */

#include <u.h>
#include <libc.h>

#include "config.h"
#include "config-std.h"
#include "config-mem.h"
#include "config-io.h"
#include "../../gtypes.h"
#include "../../object.h"
#include <native.h>
#include "../../itypes.h"
#include "../java.net.stubs/InetAddress.h"
#include "../java.net.stubs/InetAddressImpl.h"
#include "nets.h"
#include "hostresolution.h"
#include "plan9interface.h"

#define	HOSTNMSZ	80

/**
  @title java_net_InetAddressImpl
  @desc Instances of this class represent plain internet addresses.
  @funcidx
  @end
  */

/**
  @function    java_net_InetAddressImpl_getLocalHostName
  @description Return the name of the local host as a String.
  @parameter   Reference to the current (this) object.
  @rvalue      Java String holding the local host name.
  @end
  */

struct Hjava_lang_String* java_net_InetAddressImpl_getLocalHostName(struct Hjava_net_InetAddressImpl* none)
{
  char *hostName;
  USED(none);
  
  /* The localhost name is retrieved from the Plan9 environment variable
     "sysname" */
  
  hostName = getenv("sysname");
  
  if (hostName == 0) {
    strcpy("localhost", hostName);
  }
  return (makeJavaString(hostName, strlen(hostName)));
}

/**
  @function    java_net_InetAddressImpl_makeAnyLocalAddress
  @description Set the local address as anything possible.  
  @parameter   Reference to the current (this) object.
  @parameter   InetAddress object of the local host.
  @end
  */
 
void java_net_InetAddressImpl_makeAnyLocalAddress(struct Hjava_net_InetAddressImpl* none, struct Hjava_net_InetAddress* this)
{
  USED(none);
  unhand(this)->hostName = 0;
  unhand(this)->address = INADDR_ANY;
  unhand(this)->family = AF_INET;
}

/**
  @function    java_net_InetAddressImpl_lookupHostAddr
  @description Converts a hostname to the primary host address.
  @parameter   Reference to the current (this) object.
  @parameter   The name of the host to convert from.
  @rvalue      Java byte array holding (4 bytes long) the asked host address.
  @end
  */

HArrayOfByte* java_net_InetAddressImpl_lookupHostAddr(struct Hjava_net_InetAddressImpl* none, struct Hjava_lang_String* str)
{
  char name[MAXHOSTNAME];
  int ent;

  Hjava_lang_Object* obj;
  
  USED(none);

  javaString2CString(str, name, sizeof(name));
  
  ent = gethostbyname(name);
  if (ent == 0) {
    SignalError(0, "java.net.UnknownHostException", SYS_HERROR);
  }
  
  /* Copy in the network address */
  obj = AllocArray(sizeof(jint), TYPE_Byte);
  assert(obj != 0);
  
  *(jint*)((void*)((Array*)(obj)+1)) = ent;
  
  return ((HArrayOfByte*)obj);
}

/**
  @function    java_net_InetAddressImpl_lookupAllHostAddr
  @description Converts a hostname to an array of host addresses.
  @parameter   Reference to the current (this) object.
  @parameter   The name of the host to convert from.
  @rvalue      Java Array of byte arrays holding the host addresses.
  @end
  */

HArrayOfArray* java_net_InetAddressImpl_lookupAllHostAddr(struct Hjava_net_InetAddressImpl* none, struct Hjava_lang_String* str)
{
  char name[MAXHOSTNAME];
  int ent;
  Hjava_lang_Object* obj;
  Hjava_lang_Object* array;
  int alength;
  
  USED(none);

  javaString2CString(str, name, sizeof(name));
  
  ent = gethostbyname(name);
  if (ent == 0) {
    SignalError(0, "java.net.UnknownHostException", SYS_HERROR);
  }

  alength = 1;
  
  array = AllocObjectArray(alength, "[[B");
  assert(array != 0);
  
  obj = AllocArray(sizeof(jint), TYPE_Byte);
  assert(obj != 0);
  
  *(jint*)((void*)((Array*)(obj)+1)) = ent;

  ((Hjava_lang_Object**)((Array*)(array)+1))[0] = obj;
  
  return ((HArrayOfArray*)array);
}

/**
  @function    java_net_InetAddressImpl_lookupAllHostAddr
  @description Converts a network order address into a hostname.
  @parameter   Reference to the current (this) object.
  @parameter   The network order address to convert from.
  @rvalue      The asked host name as a Java String.
  @end
  */

struct Hjava_lang_String* java_net_InetAddressImpl_getHostByAddr(struct Hjava_net_InetAddressImpl* none, jint addr)
{
  char hostName[MAXHOSTNAME];
  char *name;

  USED(none);

  name = gethostbyaddr(addr, hostName);

  if (name == 0) {
    SignalError(0, "java.net.UnknownHostException", SYS_HERROR);
  }
  
  return (makeJavaString(name, strlen(name)));
}

/**
  @function    java_net_InetAddressImpl_getInetFamily
  @description Return the address family.
  @parameter   Reference to the current (this) object.
  @rvalue      The internet address family of this address implementation.
  @end
  */

jint java_net_InetAddressImpl_getInetFamily(struct Hjava_net_InetAddressImpl* none)
{
  USED(none);
  return (AF_INET);
}


