/*
 * java.lang.System.c
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "lib-license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>, 1996.
 **/

/*** CHANGELOG
 *
 * 9.1.1998    Vesa Karpijoki   The Plan9 system uses different kind of
 *                              calls in getting system properties:
 *                              getcwd(char*, int) is replaced by
 *                              getwd(char*, int), but the calls getuid(),
 *                              getpwuid() and uname(struct utsname*) do not
 *                              exist in Plan9 at all, so these must be
 *                              handled in other way.  
 *                                      
 *                              Note: environment variables KAFFEHOME and
 *                                    KAFFECLASSPATH must be set in Plan9.
 *
 * 21.1.1998   Vesa Karpijoki   Added #includes <u.h>, <libc.h> and
 *                              "plan9interface.h". Removed some #includes.
 *
 *  2.2.1998   Vesa Karpijoki   Fixed some bugs.
 *
 * 18.3.1998   Vesa Karpijoki   Added Dokumatic documentation.
 *                              Changed #include:s.
 */

#define	DBG(s)

#include <u.h>
#include <libc.h>

#include "config.h"
#include "config-std.h"
#include "config-io.h"
#include "config-mem.h"

#include "../../../src/gtypes.h"
#include "../../../src/object.h"
#include "../../../src/constants.h"
#include "../../../src/access.h"
#include "../../../src/baseClasses.h"
#include "../../../src/classMethod.h"
#include "../../../src/support.h"
#include "../../../src/soft.h"
#include "system.h"
#include "defs.h"
#include "../java.io.stubs/InputStream.h"
#include "../java.io.stubs/PrintStream.h"
#include "../java.lang.stubs/System.h"
#include <native.h>

/**
  @title java_lang_System
  @desc Native methods of the Java API class java.lang.System.
  @funcidx
  @end
  */

static char cwdpath[MAXPATHLEN];

extern jlong currentTime(void);
extern userProperty* userProperties;
extern Hjava_lang_Class SystemClass;

extern jint java_lang_Object_hashCode(struct Hjava_lang_Object*);

/**
  @function java_lang_System_arraycopy
  @description Copy one part of an array to another starting from a specific position.
  @parameter Reference to the source Object.
  @parameter The starting position of the source.
  @parameter Reference to the target Object.
  @parameter The starting position of the target.
  @end
  */
void java_lang_System_arraycopy(struct Hjava_lang_Object* src, jint srcpos, struct Hjava_lang_Object* dst, jint dstpos, jint len)
{
	char* in;
	char* out;
	int elemsz;
	Hjava_lang_Class* sclass;
	Hjava_lang_Class* dclass;

	/*	printf("src: %p src->table: %p\n", src, src->dtable); /********************/
	
	sclass = OBJECT_CLASS(src);
	
	/*	printf(" dst: %p dst->table: %p\n", dst, dst->dtable);   /*********************/
	
	dclass = OBJECT_CLASS(dst);

	/* Must be arrays **/
	if (! CLASS_IS_ARRAY(sclass) || ! CLASS_IS_ARRAY(dclass)) {
		SignalError(0, "java.lang.ArrayStoreException", "");
	}

	/* Make sure we'll keep in the array boundaries **/
	if ((srcpos < 0 || srcpos + len > ARRAY_SIZE(src)) ||
	    (dstpos < 0 || dstpos + len > ARRAY_SIZE(dst)) ||
	    (len < 0)) {
		SignalError(0, "java.lang.ArrayIndexOutOfBoundsException", "");
	}

	sclass = CLASS_ELEMENT_TYPE(sclass);
	dclass = CLASS_ELEMENT_TYPE(dclass);
	elemsz = TYPE_SIZE(sclass);

	len *= elemsz;
	srcpos *= elemsz;
	dstpos *= elemsz;

	in = &((char*)ARRAY_DATA(src))[srcpos];
	out = &((char*)ARRAY_DATA(dst))[dstpos];

	if (sclass == dclass) {
		memmove((void*)out, (void*)in, len);

		return;
	}

	if (CLASS_IS_PRIMITIVE(sclass) || CLASS_IS_PRIMITIVE(dclass))
		SignalError(0, "java.lang.ArrayStoreException", "");
	
	for (; len > 0; len -= sizeof(Hjava_lang_Object**)) {
		Hjava_lang_Object* val = *(Hjava_lang_Object**)in;
		if (! soft_instanceof(dclass, val)) {
			SignalError(0, "java.lang.ArrayStoreException", "");
		}
		*(Hjava_lang_Object**)out = val;
		in += sizeof(Hjava_lang_Object*);
		out += sizeof(Hjava_lang_Object*);
	}
}

/**
  @function java_lang_System_initProperties
  @description Initialise system properties to their defaults.
  @parameter Reference to a Properties object.
  @rvalue Reference to the Properties object that was initialised.
  @end
  */
struct Hjava_util_Properties* java_lang_System_initProperties(struct Hjava_util_Properties* p)
{
	char* jhome;
	char* cpath;
	char* dir;
	userProperty* prop;

	/* Add the default properties:
	 * java.version		Java version number
	 * java.vendor          Java vendor specific string
	 * java.vendor.url      Java vendor URL
	 * java.home            Java installation directory
	 * java.class.version   Java class version number
	 * java.class.path      Java classpath
	 * os.name              Operating System Name
	 * os.arch              Operating System Architecture
	 * os.version           Operating System Version
	 * file.separator       File separator ("/" on Unix)
	 * path.separator       Path separator (":" on Unix)
	 * line.separator       Line separator ("\n" on Unix)
	 * user.name            User account name
	 * user.home            User home directory
	 * user.dir             User's current working directory
	 **/

	setProperty(p, "java.version", kaffe_version);
	setProperty(p, "java.vendor", kaffe_vendor);
	setProperty(p, "java.vendor.url", kaffe_vendor_url);

	jhome = getenv(KAFFEHOME);
	if (jhome == 0) {
		jhome = ".";
	}
	setProperty(p, "java.home", jhome);

	setProperty(p, "java.class.version", kaffe_class_version);

	cpath = getenv(KAFFECLASSPATH);
	if (cpath == 0) {
		cpath = ".";
	}
	setProperty(p, "java.class.path", cpath);

	setProperty(p, "file.separator", file_seperator);
	setProperty(p, "path.separator", path_seperator);
	setProperty(p, "line.separator", line_seperator);

	dir = getwd(cwdpath, MAXPATHLEN);
	if (dir == 0) {
		dir = ".";
	}
	setProperty(p, "user.dir", dir);

	setProperty(p, "os.name",    "Plan9");
	setProperty(p, "os.arch",    getenv("cputype"));
	setProperty(p, "os.version", "1.0");

	setProperty(p, "user.name", getenv("user"));
	setProperty(p, "user.home", getenv("home"));

	/* JDK 1.1 **/
	setProperty(p, "file.encoding.pkg", "sun.io");

	/* Add in the awt.Toolkit we will be using **/



	/* Now process user defined properties **/
	for (prop = userProperties; prop != 0; prop = prop->next) {
		setProperty(p, prop->key, prop->value);
	}

	return (p);
}

/**
  @function java_lang_System_currentTimeMillis
  @description Return current time in milliseconds.
  @rvalue The current time in milliseconds.
  @end
  */
jlong java_lang_System_currentTimeMillis(void)
{
	return (currentTime());
}

/**
  @function java_lang_System_setIn0
  @description Set the stdin stream.
  @parameter Reference to an InputStream object.
  @end
  */
void java_lang_System_setIn0(struct Hjava_io_InputStream* stream)
{
	*(struct Hjava_io_InputStream**)FIELD_ADDRESS(&CLASS_SFIELDS(&SystemClass)[0]) = stream;
}

/**
  @function java_lang_System_setOut0
  @description Set the stdout stream.
  @parameter Reference to a Print Stream object.
  @end
  */
void java_lang_System_setOut0(struct Hjava_io_PrintStream* stream)
{
	*(struct Hjava_io_PrintStream**)FIELD_ADDRESS(&CLASS_SFIELDS(&SystemClass)[1]) = stream;
}

/**
  @function java_lang_System_setErr0
  @description Set the error stream.
  @parameter Reference to a PrintStream object.
  @end
  */
void java_lang_System_setErr0(struct Hjava_io_PrintStream* stream)
{
        *(struct Hjava_io_PrintStream**)FIELD_ADDRESS(&CLASS_SFIELDS(&SystemClass)[2]) = stream;
}

/**
  @function java_lang_System_identityHashCode
  @description Return a hashcode of an object.
  @parameter Reference to an Object.
  @rvalue The hash value of the Object.
  @end
  */
jint java_lang_System_identityHashCode(struct Hjava_lang_Object* o)
{
       return (java_lang_Object_hashCode(o));
}
