/* icode.c
 * Define the instructions.
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>
 */


/*** CHANGELOG ****
 *
 * 12.3.1998   Teemu Ikonen          Some clean-up and standard headers
 *
 * 14.3.1998   Teemu Ikonen          save registers between function calls
 *
 * 18.3.1998   Teemu Ikonen          Other hack to save registers
 *
 * 23.3.1998   Teemu Ikonen          fixed some bugs in native function invocation.
 *
 */

#include <u.h>
#include <libc.h>
#include <plan9interface.h>
#include "config.h"
#include "config-std.h"
#include "gtypes.h"
#include "slots.h"
#include "seq.h"
#include "registers.h"
#include "basecode.h"
#include "labels.h"
#include "constpool.h"
#include "icode.h"
#include "soft.h"
#include "access.h"
#include "object.h"
#include "constants.h"
#include "classMethod.h"
#include "gc.h"
#include "flags.h"
#include "itypes.h"
#include "md.h"
#include "codeproto.h"


#define	LSLOT(_s)	(_s)
#define	HSLOT(_s)	((_s)+1)


void startBlock(sequence*);
void endBlock(sequence*);
void prepareFunctionCall(sequence*);
void fixupFunctionCall(sequence*);
void syncRegisters(sequence*);
void nowritebackSlot(sequence*);

extern uint32 pc;
extern uint32 npc;
extern int maxPush;
extern int maxArgs;
extern int maxTemp;
extern int maxLocal;
extern int maxStack;
extern int isStatic;

bool used_ieee_rounding;
bool used_ieee_division;

#define	MAXLABTAB	64
label* labtab[MAXLABTAB];

/* ----------------------------------------------------------------------- */
/* Register loads and spills.						   */
/*									   */

void
spill_int(SlotInfo* src)
{
	void HAVE_spill_int(sequence*);
	sequence s;
	seq_dst(&s) = src;
	seq_value(&s, 1) = SLOT2FRAMEOFFSET(src);
	HAVE_spill_int(&s);
}


void
reload_int(SlotInfo* dst)
{
	void HAVE_reload_int(sequence*);
	sequence s;
	seq_dst(&s) = dst;
	seq_value(&s, 1) = SLOT2FRAMEOFFSET(dst);
	HAVE_reload_int(&s);
}

void
spill_float(SlotInfo* src)
{
	void HAVE_spill_float(sequence*);
	sequence s;
	seq_dst(&s) = src;
	seq_value(&s, 1) = SLOT2FRAMEOFFSET(src);
	HAVE_spill_float(&s);
}


void
reload_float(SlotInfo* dst)
{
	void HAVE_reload_float(sequence*);
	sequence s;
	seq_dst(&s) = dst;
	seq_value(&s, 1) = SLOT2FRAMEOFFSET(dst);
	HAVE_reload_float(&s);
}


void
spill_double(SlotInfo* src)
{
	void HAVE_spill_double(sequence*);
	sequence s;
	seq_dst(&s) = src;
	seq_value(&s, 1) = SLOT2FRAMEOFFSET(src);
	HAVE_spill_double(&s);
}


void
reload_double(SlotInfo* dst)
{
	void HAVE_reload_double(sequence*);
	sequence s;
	seq_dst(&s) = dst;
	seq_value(&s, 1) = SLOT2FRAMEOFFSET(dst);
	HAVE_reload_double(&s);
}


/* ----------------------------------------------------------------------- */
/* Prologues and epilogues.						   */
/*									   */

void
prologue(Method* meth)
{
	label* l;

	used_ieee_rounding = false;

	l = newLabel();
	l->type = Lnull; /* Lframe|Labsolute|Lgeneral */
	l->at = 0;
	l->to = 0;
	l->from = 0;

	/* Emit prologue code */
	slot_const_const(0, (jword)l, (jword)meth, HAVE_prologue, Tnull);
}

void
exception_prologue(void)
{
	label* l;

	l = newLabel();
	l->type = Lnull; /* Lframe|Labsolute|Lgeneral */
	l->at = 0;
	l->to = 0;
	l->from = 0;

	/* Emit exception prologue code */
	slot_const_const(0, (jword)l, 0, HAVE_exception_prologue, Tnull);
}

void
epilogue(void)
{
	slot_slot_slot(0, 0, 0, HAVE_epilogue, Tnull);
}


/* ----------------------------------------------------------------------- */
/* Conditional monitor management.					   */
/*									   */

void
mon_enter(methods* meth, SlotInfo* obj)
{
	/* Emit monitor entry if required */
	end_basic_block();
	saveframe();
	if ((meth->accflags & ACC_STATIC) != 0) {
		pusharg_ref_const(meth->class, 0);
	}
	else {
		pusharg_ref(obj, 0);
	}
	call_soft(soft_monitorenter);
	popargs();
	restoreframe();
	start_basic_block();
}

void
mon_exit(methods* meth, SlotInfo* obj)
{
	end_basic_block();
	saveframe();
	if ((meth->accflags & ACC_STATIC) != 0) {
		pusharg_ref_const(meth->class, 0);
	}
	else {
		pusharg_ref(obj, 0);
	}
	call_soft(soft_monitorexit);
	popargs();
	restoreframe();
	start_basic_block();
}

/* ----------------------------------------------------------------------- */
/* Basic block and instruction management.				   */
/*									   */

void
_start_basic_block(void)
{
	_slot_const_const(0, 0, 0, startBlock, Tnull);
}

void
_end_basic_block(uintp stk, uintp temp)
{
	_slot_const_const(0, stk, temp, endBlock, Tnull);
}

void
_start_instruction(uintp pc)
{
	void startInsn(sequence*);

	_slot_const_const(0, 0, pc, startInsn, Tnull);
}

void
_start_exception_block(uintp stk)
{
	/* Exception blocks act like function returns - the return
	 * value is the exception object.
	 */
	start_basic_block();
	exception_prologue();
	return_ref(&localinfo[stk]);
}

void
_fixup_function_call(void)
{
	_slot_const_const(0, 0, 0, fixupFunctionCall, Tnull);
}

void
_prepare_function_call(uintp stk, uintp temp)
{
	_slot_const_const(0, stk, temp, prepareFunctionCall, Tnull);
}

void
_slot_nowriteback(SlotInfo* slt)
{
	_slot_const_const(slt, 0, 0, nowritebackSlot, Tnull);
}

void 
_syncRegisters(uintp stk, uintp temp)
{
	_slot_const_const(0, stk, temp, syncRegisters, Tnull);
}


/* ----------------------------------------------------------------------- */
/* Moves.								   */
/*									   */

void
move_int_const(SlotInfo* dst, jint val)
{
	if (HAVE_move_int_const_rangecheck(val)) {
		slot_slot_const(dst, 0, val, HAVE_move_int_const, Tconst);
		dst->v.tint = val;
	}
	else

	{
		constpool *c;
		label* l;
		SlotInfo* tmp;

		c = newConstant(CPint, val);
		l = newLabel();
		l->type = Lconstant;
		l->at = 0;
		l->to = (uintp)c;
		l->from = 0;

		slot_alloctmp(tmp);
		move_label_const(tmp, l);
		load_int(dst, tmp);
	}
}

void
move_ref_const(SlotInfo* dst, void *val)
{
	if (HAVE_move_ref_const_rangecheck(val)) {
		slot_slot_const(dst, 0, (jword)val, HAVE_move_ref_const, Tconst);
		dst->v.tint = (jword)val;
	}
	else

	{
		constpool *c;
		label* l;
		SlotInfo* tmp;

		c = newConstant(CPref, val);
		l = newLabel();
		l->type = Lconstant;
		l->at = 0;
		l->to = (uintp)c;
		l->from = 0;

		slot_alloctmp(tmp);
		move_label_const(tmp, l);
		load_ref(dst, tmp);
	}
}

void
move_long_const(SlotInfo* dst, jlong val)
{

	move_int_const(LSLOT(dst), (jint)(val & 0xFFFFFFFF));
	move_int_const(HSLOT(dst), (jint)((val >> 32) & 0xFFFFFFFF));

}

void
move_float_const(SlotInfo* dst, float val)
{
	if (HAVE_move_float_const_rangecheck(val)) {
		slot_slot_fconst(dst, 0, val, HAVE_move_float_const, Tconst);
		dst->v.tdouble = val;
	}
	else

	{
		constpool *c;
		label* l;
		SlotInfo* tmp;

		c = newConstant(CPfloat, val);
		l = newLabel();
		l->type = Lconstant;
		l->at = 0;
		l->to = (uintp)c;
		l->from = 0;

		slot_alloctmp(tmp);
		move_label_const(tmp, l);
		load_float(dst, tmp);
	}
}

void
move_double_const(SlotInfo* dst, jdouble val)
{
	if (HAVE_move_double_const_rangecheck(val)) {
		lslot_slot_fconst(dst, 0, val, HAVE_move_double_const, Tconst);
		dst->v.tdouble = val;
	}
	else

	{
		constpool *c;
		label* l;
		SlotInfo* tmp;

		c = newConstant(CPdouble, val);
		l = newLabel();
		l->type = Lconstant;
		l->at = 0;
		l->to = (uintp)c;
		l->from = 0;

		slot_alloctmp(tmp);
		move_label_const(tmp, l);
		load_double(dst, tmp);
	}
}

void
move_any(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_move_any, Tcopy);
}


void
move_int(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_move_int, Tcopy);
}


void
move_ref(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_move_ref, Tcopy);
}

void
move_anylong(SlotInfo* dst, SlotInfo* src)
{

	assert(LSLOT(dst) != HSLOT(src));
	move_any(LSLOT(dst), LSLOT(src));
	move_any(HSLOT(dst), HSLOT(src));

}

void
move_long(SlotInfo* dst, SlotInfo* src)
{

	assert(LSLOT(dst) != HSLOT(src));
	move_int(LSLOT(dst), LSLOT(src));
	move_int(HSLOT(dst), HSLOT(src));

}

void
move_float(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_move_float, Tcopy);
}


void
move_double(SlotInfo* dst, SlotInfo* src)
{
	lslot_lslot_lslot(dst, 0, src, HAVE_move_double, Tcopy);
}


void
move_label_const(SlotInfo* dst, label* lab)
{
	slot_slot_const(dst, 0, (jword)lab, HAVE_move_label_const, Tnull);
}


void
swap_any(SlotInfo* src, SlotInfo* src2)
{

	SlotInfo* tmp;
	slot_alloctmp(tmp);
	move_ref(tmp, src);
	move_ref(src, src2);
	move_ref(src2, tmp);

}

/* ----------------------------------------------------------------------- */
/* Arithmetic operators - add, sub, etc.				   */
/*									   */


void
adc_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_adc_int, Tcomplex);
}


void
add_int_const(SlotInfo* dst, SlotInfo* src, jint val)
{
	if (HAVE_add_int_const_rangecheck(val)) {
		slot_slot_const(dst, src, val, HAVE_add_int_const, Taddregconst);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		add_int(dst, src, tmp);
	}
}

void
add_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_add_int, Tcomplex);
}


void
add_ref(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_add_ref, Tcomplex);
}


void
add_ref_const(SlotInfo* dst, SlotInfo* src, jint val)
{
	if (HAVE_add_ref_const_rangecheck(val)) {
		slot_slot_const(dst, src, val, HAVE_add_ref_const, Taddregconst);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		add_ref(dst, src, tmp);
	}
}

void
add_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	add_int(LSLOT(dst), LSLOT(src), LSLOT(src2));
	adc_int(HSLOT(dst), HSLOT(src), HSLOT(src2));

}



void
add_float(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_add_float, Tcomplex);
}


void
add_double(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	lslot_lslot_lslot(dst, src, src2, HAVE_add_double, Tcomplex);
}


void
sbc_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_sbc_int, Tcomplex);
}


void
sub_int_const(SlotInfo* dst, SlotInfo* src, jint val)
{
	if (HAVE_sub_int_const_rangecheck(val)) {
		slot_slot_const(dst, src, val, HAVE_sub_int_const, Tcomplex);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		sub_int(dst, src, tmp);
	}
}

void
sub_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_sub_int, Tcomplex);
}


void
sub_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	sub_int(LSLOT(dst), LSLOT(src), LSLOT(src2));
	sbc_int(HSLOT(dst), HSLOT(src), HSLOT(src2));

}



void
sub_float(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_sub_float, Tcomplex);
}


void
sub_double(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	lslot_lslot_lslot(dst, src, src2, HAVE_sub_double, Tcomplex);
}


void
mul_int_const(SlotInfo* dst, SlotInfo* src, jint val)
{

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		mul_int(dst, src, tmp);
	}
}

void
mul_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_mul_int, Tcomplex);

}

void
mul_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	preparelonglong();
	pusharg_long(src2, pusharg_long_idx_inc);
	pusharg_long(src, 0);
	pushlonglong();
	call_soft(soft_lmul);
	argcount++;
	popargs();
	poplonglong();
	restoreframe();
	start_basic_block();
	return_long(dst);

}



void
mul_float(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_mul_float, Tcomplex);
}


void
mul_double(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	lslot_lslot_lslot(dst, src, src2, HAVE_mul_double, Tcomplex);
}


void
div_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_div_int, Tcomplex);

}

void
div_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	preparelonglong();
	pusharg_long(src2, pusharg_long_idx_inc);
	pusharg_long(src, 0);
	pushlonglong();
	call_soft(soft_ldiv);
	argcount++;
	popargs();
	poplonglong();
	restoreframe();
	start_basic_block();
	return_long(dst);

}

void
div_float(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	pusharg_float(src2, 1);
	pusharg_float(src, 0);
	call_soft(soft_fdiv);
	popargs();
	restoreframe();
	start_basic_block();
	return_float(dst);

}

void
div_double(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	pusharg_double(src2, pusharg_long_idx_inc);
	pusharg_double(src, 0);
	call_soft(soft_fdivl);
	popargs();
	restoreframe();
	start_basic_block();
	return_double(dst);

}

void
rem_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_rem_int, Tcomplex);

}

void
rem_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	preparelonglong();
	pusharg_long(src2, pusharg_long_idx_inc);
	pusharg_long(src, 0);
	pushlonglong();
	call_soft(soft_lrem);
	argcount++;
	popargs();
	poplonglong();
	restoreframe();
	start_basic_block();
	return_long(dst);

}

void
rem_float(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	pusharg_float(src2, 1);
	pusharg_float(src, 0);
	call_soft(soft_frem);
	popargs();
	restoreframe();
	end_basic_block();
	return_float(dst);

}

void
rem_double(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	pusharg_double(src2, pusharg_long_idx_inc);
	pusharg_double(src, 0);
	call_soft(soft_freml);
	popargs();
	restoreframe();
	end_basic_block();
	return_double(dst);

}

void
neg_int(SlotInfo* dst, SlotInfo* src)
{

	SlotInfo* zero;
	slot_alloctmp(zero);
	move_int_const(zero, 0);
	sub_int(dst, zero, src);

}



void
neg_long(SlotInfo* dst, SlotInfo* src)
{
	SlotInfo* zero;
	slot_alloctmp(zero);
	move_int_const(zero, 0);
	sub_int(LSLOT(dst), zero, LSLOT(src));
	sbc_int(HSLOT(dst), zero, HSLOT(src));

}

void
neg_float(SlotInfo* dst, SlotInfo* src)
{

	SlotInfo* tmp;
	slot_alloctmp(tmp);
	move_float_const(tmp, 0);
	sub_float(dst, tmp, src);

}

void
neg_double(SlotInfo* dst, SlotInfo* src)
{

	SlotInfo* tmp;
	slot_alloc2tmp(tmp);
	move_double_const(tmp, 0);
	sub_double(dst, tmp, src);

}


/* ----------------------------------------------------------------------- */
/* Logical operators - and, or, etc.					   */
/*									   */

void
and_int_const(SlotInfo* dst, SlotInfo* src, jint val)
{

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		and_int(dst, src, tmp);
	}
}

void
and_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_and_int, Tcomplex);
}


void
and_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	and_int(LSLOT(dst), LSLOT(src), LSLOT(src2));
	and_int(HSLOT(dst), HSLOT(src), HSLOT(src2));

}



void
or_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_or_int, Tcomplex);
}


void
or_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	or_int(LSLOT(dst), LSLOT(src), LSLOT(src2));
	or_int(HSLOT(dst), HSLOT(src), HSLOT(src2));

}

void
xor_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_xor_int, Tcomplex);
}


void
xor_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	xor_int(LSLOT(dst), LSLOT(src), LSLOT(src2));
	xor_int(HSLOT(dst), HSLOT(src), HSLOT(src2));

}

void
lshl_int_const(SlotInfo* dst, SlotInfo* src, jint val)
{
	if (HAVE_lshl_int_const_rangecheck(val)) {
		slot_slot_const(dst, src, val, HAVE_lshl_int_const, Tcomplex);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		lshl_int(dst, src, tmp);
	}
}

void
lshl_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_lshl_int, Tcomplex);
}




void
lshl_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	preparelonglong();
	pusharg_int(src2, pusharg_long_idx_inc);
	pusharg_long(src, 0);
	pushlonglong();
	call_soft(soft_lshll);
	argcount++;
	popargs();
	poplonglong();
	restoreframe();
	start_basic_block();
	return_long(dst);

}

void
ashr_int_const(SlotInfo* dst, SlotInfo* src, jint val)
{

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		ashr_int(dst, src, tmp);
	}
}

void
ashr_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_ashr_int, Tcomplex);
}


void
ashr_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	preparelonglong();
	pusharg_int(src2, pusharg_long_idx_inc);
	pusharg_long(src, 0);
	pushlonglong();
	call_soft(soft_ashrl);
	argcount++;
	popargs();
	poplonglong();
	restoreframe();
	start_basic_block();
	return_long(dst);

}

void
lshr_int_const(SlotInfo* dst, SlotInfo* src, jint val)
{

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		lshr_int(dst, src, tmp);
	}
}

void
lshr_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_lshr_int, Tcomplex);
}


void
lshr_long(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	preparelonglong();
	pusharg_int(src2, pusharg_long_idx_inc);
	pusharg_long(src, 0);
	pushlonglong();
	call_soft(soft_lshrl);
	argcount++;
	popargs();
	poplonglong();
	restoreframe();
	start_basic_block();
	return_long(dst);

}




/* ----------------------------------------------------------------------- */
/* Load and store.							   */
/*									   */

void
load_int(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_load_int, Tload);
}


void
load_offset_int(SlotInfo* dst, SlotInfo* src, jint offset)
{
	if (offset == 0) {
		load_int(dst, src);
	}
	else
	if (HAVE_load_offset_int_rangecheck(offset)) {
		slot_slot_const(dst, src, offset, HAVE_load_offset_int, Tload);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		add_ref_const(tmp, src, offset);
		load_int(dst, tmp);
	}
}

void
load_ref(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_load_ref, Tload);
}


void
load_offset_ref(SlotInfo* dst, SlotInfo* src, jint offset)
{
	if (offset == 0) {
		load_ref(dst, src);
	}
	else
	if (HAVE_load_offset_ref_rangecheck(offset)) {
		slot_slot_const(dst, src, offset, HAVE_load_offset_ref, Tload);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		add_ref_const(tmp, src, offset);
		load_ref(dst, tmp);
	}
}


void
load_long(SlotInfo* dst, SlotInfo* src)
{

	SlotInfo* tmp;

	slot_alloctmp(tmp);
	add_ref_const(tmp, src, 4);
	/* Don't use LSLOT & HSLOT here */
	load_int(dst, src);
	load_int(dst+1, tmp);

}



void
load_float(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_load_float, Tload);
}


void
load_double(SlotInfo* dst, SlotInfo* src)
{
	lslot_lslot_slot(dst, 0, src, HAVE_load_double, Tload);
}


void
load_byte(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_load_byte, Tload);

}

void
load_char(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_load_char, Tload);

}

void
load_short(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_load_short, Tload);

}

void
load_code_ref(SlotInfo* dst, SlotInfo* src)
{

	load_ref(dst, src);

}

void
load_key(SlotInfo* dst, SlotInfo* src)
{
	load_int(dst, src);
}

void
store_int(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(0, dst, src, HAVE_store_int, Tstore);
}


void
store_offset_int(SlotInfo* dst, jint offset, SlotInfo* src)
{
	if (offset == 0) {
		store_int(dst, src);
	}
	else
	if (HAVE_store_offset_int_rangecheck(offset)) {
		slot_slot_const(src, dst, offset, HAVE_store_offset_int, Tstore);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		add_ref_const(tmp, dst, offset);
		store_int(tmp, src);
	}
}

void
store_ref(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(0, dst, src, HAVE_store_ref, Tstore);
}


void
store_offset_ref(SlotInfo* dst, jint offset, SlotInfo* src)
{
	if (offset == 0) {
		store_ref(dst, src);
	}
	else
	if (HAVE_store_offset_ref_rangecheck(offset)) {
		slot_slot_const(src, dst, offset, HAVE_store_offset_ref, Tstore);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		add_ref_const(tmp, dst, offset);
		store_ref(tmp, src);
	}
}

void
store_long(SlotInfo* dst, SlotInfo* src)
{

	SlotInfo* tmp;

	slot_alloctmp(tmp);
	add_ref_const(tmp, dst, 4);
	/* Don't use LSLOT & HSLOT here */
	store_int(dst, src);
	store_int(tmp, src+1);

}



void
store_float(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(0, dst, src, HAVE_store_float, Tstore);
}


void
store_double(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_lslot(0, dst, src, HAVE_store_double, Tstore);
}


void
store_byte(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(0, dst, src, HAVE_store_byte, Tstore);

}

void
store_char(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(0, dst, src, HAVE_store_char, Tstore);

}

void
store_short(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(0, dst, src, HAVE_store_short, Tstore);

}


/* ----------------------------------------------------------------------- */
/* Function argument management.					   */
/*									   */

void
pusharg_int_const(int val, int idx)
{
	if (HAVE_pusharg_int_const_rangecheck(val)) {
		slot_const_const(0, val, idx, HAVE_pusharg_int_const, Tnull);
		argcount += 1;
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		pusharg_int(tmp, idx);
	}
}

void
pusharg_int(SlotInfo* src, int idx)
{
	slot_slot_const(0, src, idx, HAVE_pusharg_int, Tnull);
	argcount += 1;
}


void
pusharg_ref(SlotInfo* src, int idx)
{
	slot_slot_const(0, src, idx, HAVE_pusharg_ref, Tnull);
	argcount += 1;
}


void
pusharg_ref_const(void* val, int idx)
{

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_ref_const(tmp, val);
		pusharg_ref(tmp, idx);
	}
}

void
pusharg_float(SlotInfo* src, int idx)
{
	slot_slot_const(0, src, idx, HAVE_pusharg_float, Tnull);
	argcount += 1;
}


void
pusharg_double(SlotInfo* src, int idx)
{
	lslot_lslot_const(0, src, idx, HAVE_pusharg_double, Tnull);
	argcount += pusharg_long_idx_inc;
}


void
pusharg_long(SlotInfo* src, int idx)
{

	/* Don't use LSLOT & HSLOT here */
	pusharg_int(src+1, idx+1);
	pusharg_int(src, idx);

}

/* used to get longlong type parameters right when calling
   8c generated code
   */
void preparelonglong(void)
{
    slot_slot_slot(0, 0, 0, HAVE_llpush, Tnull);
}

void pushlonglong(void)
{
    slot_slot_const(0, 0, argcount, HAVE_llpush2, Tnull);
}

void poplonglong(void)
{
    slot_slot_slot(0, 0, 0, HAVE_llpop, Tnull);
}

/* these two functions are used to save and restore some
 * registers JIT-compiler assumes subfunctions to save
 */
void 
saveframe(void)
{
    slot_slot_slot(0, 0, 0, HAVE_saveframe, Tnull);
}

void
restoreframe(void)
{
    slot_slot_slot(0, 0, 0, HAVE_restoreframe, Tnull);
}


void
popargs(void)
{
    if (argcount != 0) {
	slot_slot_const(0, 0, argcount, HAVE_popargs, Tnull);
	
	if (argcount > maxPush) {
	    maxPush = argcount;
	}
	argcount = 0;
    }
}



/* ----------------------------------------------------------------------- */
/* Control flow changes.						   */
/*									   */

void
branch(label* dst, int type)
{
	slot_const_const(0, (jword)dst, type, HAVE_branch, Tnull);
}


void
cbranch_int(SlotInfo* s1, SlotInfo* s2, label* dst, int type)
{

	cmp_int(0, s1, s2);
	branch(dst, type);

}

void
cbranch_int_const(SlotInfo* s1, jint val, label* dst, int type)
{

	cmp_int_const(0, s1, val);
	branch(dst, type);

}

void
cbranch_ref(SlotInfo* s1, SlotInfo* s2, label* dst, int type)
{

	cmp_ref(0, s1, s2);
	branch(dst, type);

}

void
cbranch_ref_const(SlotInfo* s1, void *val, label* dst, int type)
{

	cmp_ref_const(0, s1, val);
	branch(dst, type);

}

void
branch_indirect(SlotInfo* dst)
{
	slot_slot_const(0, dst, ba, HAVE_branch_indirect, Tnull);
}


void
call(SlotInfo* dst)
{
	slot_slot_const(0, dst, ba, HAVE_call, Tnull);
}


void
call_ref(void *routine)
{
	label* l = newLabel();

	l->type = Lexternal;
	l->at = 0;
	l->to = (uintp)routine;	/* What place does it goto */
	l->from = 0;

	slot_const_const(0, (jword)l, ba, HAVE_call_ref, Tnull);

}

void
call_indirect_const(void *ptr)
{
	slot_const_const(0, (jword)ptr, ba, HAVE_call_indirect_const, Tnull);

}

void
call_soft(void *routine)
{

	call_ref(routine);

}


void
ret(void)
{
	slot_slot_slot(0, 0, 0, HAVE_ret, Tnull);
}


void
return_int(SlotInfo* dst)
{
	slot_slot_slot(dst, 0, 0, HAVE_return_int, Tnull);
}


void
return_ref(SlotInfo* dst)
{
	slot_slot_slot(dst, 0, 0, HAVE_return_ref, Tnull);
}


void
return_long(SlotInfo* dst)
{
	lslot_lslot_lslot(dst, 0, 0, HAVE_return_long, Tnull);
}


void
return_float(SlotInfo* dst)
{
	slot_slot_slot(dst, 0, 0, HAVE_return_float, Tnull);
}


void
return_double(SlotInfo* dst)
{
	lslot_lslot_lslot(dst, 0, 0, HAVE_return_double, Tnull);
}


void
returnarg_int(SlotInfo* src)
{
	slot_slot_slot(0, 0, src, HAVE_returnarg_int, Tcopy);
}


void
returnarg_ref(SlotInfo* src)
{
	slot_slot_slot(0, 0, src, HAVE_returnarg_ref, Tcopy);
}


void
returnarg_long(SlotInfo* src)
{
	lslot_lslot_lslot(0, 0, src, HAVE_returnarg_long, Tcopy);
}


void
returnarg_float(SlotInfo* src)
{
	slot_slot_slot(0, 0, src, HAVE_returnarg_float, Tcopy);
}


void
returnarg_double(SlotInfo* src)
{
	lslot_lslot_lslot(0, 0, src, HAVE_returnarg_double, Tcopy);
}



/* ----------------------------------------------------------------------- */
/* Labels.								   */
/*									   */

label*
reference_label(int32 i, int32 n)
{
	label* l;

	assert(n < MAXLABTAB);
	if (labtab[n] == 0) {
		l = newLabel();
		labtab[n] = l;
		l->type = Lnull;
		l->at = 0;
		l->from = 0;
		l->to = 0;
	}
	else {
		l = labtab[n];
		labtab[n] = 0;
	}
	return (l);
}

label*
reference_code_label(uintp offset)
{
	label* l = newLabel();
	l->at = 0;		/* Where is the jump */
	l->to = offset;		/* What place does it goto */
	l->from = 0;
	l->type = Lcode;
	return (l);
}

label*
reference_table_label(int32 n)
{
	label* l;

	assert(n < MAXLABTAB);
	if (labtab[n] == 0) {
		l = newLabel();
		labtab[n] = l;
		l->type = Lnull;
		l->at = 0;
		l->from = 0;
		l->to = 0;
	}
	else {
		l = labtab[n];
		labtab[n] = 0;
	}
	return (l);
}

SlotInfo*
stored_code_label(SlotInfo* dst)
{
	return (dst);
}

SlotInfo*
table_code_label(SlotInfo* dst)
{
	return (dst);
}

void
set_label(int i, int n)
{
	assert(n < MAXLABTAB);
	if (labtab[n] == 0) {
		labtab[n] = newLabel();
		labtab[n]->type = Linternal;
		labtab[n]->at = 0;
		labtab[n]->from = 0;
		labtab[n]->to = 0;
		slot_slot_const(0, 0, (jword)labtab[n], HAVE_set_label, Tnull);
	}
	else {
		assert(labtab[n]->type == Lnull);
		labtab[n]->type = Linternal;
		slot_slot_const(0, 0, (jword)labtab[n], HAVE_set_label, Tnull);
		labtab[n] = 0;
	}
}


label*
build_code_ref(uint8* pos, uintp pc)
{
	label* l;
	jint offset;

	offset = (pos[0] * 0x01000000 + pos[1] * 0x00010000 +
		  pos[2] * 0x00000100 + pos[3] * 0x00000001);
	l = reference_code_label(pc+offset);

	slot_slot_const(0, 0, (jword)l, HAVE_build_code_ref, Tnull);
	return (l);
}


void
build_key(uint8* pos)
{
	jint val = (pos[0] * 0x01000000 + pos[1] * 0x00010000 +
		    pos[2] * 0x00000100 + pos[3] * 0x00000001);

	slot_slot_const(0, 0, val, HAVE_build_key, Tnull);
}



/* ----------------------------------------------------------------------- */
/* Comparisons.								   */
/*									   */

void
cmp_int_const(SlotInfo* dst, SlotInfo* src, jint val)
{
	if (HAVE_cmp_int_const_rangecheck(val)) {
		slot_slot_const(dst, src, val, HAVE_cmp_int_const, Tcomplex);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_int_const(tmp, val);
		cmp_int(dst, src, tmp);
	}
}

void
cmp_int(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_cmp_int, Tcomplex);
}


void
cmp_ref_const(SlotInfo* dst, SlotInfo* src, void* val)
{
	if (HAVE_cmp_ref_const_rangecheck((jword)val)) {
		slot_slot_const(dst, src, (jword)val, HAVE_cmp_ref_const, Tcomplex);
	}
	else

	{
		SlotInfo* tmp;
		slot_alloctmp(tmp);
		move_ref_const(tmp, val);
		cmp_ref(dst, src, tmp);
	}
}

void
cmp_ref(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{
	slot_slot_slot(dst, src, src2, HAVE_cmp_ref, Tcomplex);
}


void
lcmp(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	pusharg_long(src2, pusharg_long_idx_inc);
	pusharg_long(src, 0);
	call_soft(soft_lcmp);
	popargs();
	restoreframe();
	start_basic_block();
	return_int(dst);

}

void
cmpl_float(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	pusharg_float(src2, 1);
	pusharg_float(src, 0);
	call_soft(soft_fcmpl);
	popargs();
	restoreframe();
	start_basic_block();
	return_int(dst);

}

void
cmpl_double(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	pusharg_double(src2, pusharg_long_idx_inc);
	pusharg_double(src, 0);
	call_soft(soft_dcmpl);
	popargs();
	restoreframe();
	start_basic_block();
	return_int(dst);

}

void
cmpg_float(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	pusharg_float(src2, 1);
	pusharg_float(src, 0);
	call_soft(soft_fcmpg);
	popargs();
	restoreframe();
	start_basic_block();
	return_int(dst);

}

void
cmpg_double(SlotInfo* dst, SlotInfo* src, SlotInfo* src2)
{

	end_basic_block();
	saveframe();
	pusharg_double(src2, pusharg_long_idx_inc);
	pusharg_double(src, 0);
	call_soft(soft_dcmpg);
	popargs();
	restoreframe();
	start_basic_block();
	return_int(dst);

}

/* ----------------------------------------------------------------------- */
/* Conversions.								   */
/*									   */

void
cvt_int_long(SlotInfo* dst, SlotInfo* src)
{

	move_int(LSLOT(dst), src);
	ashr_int_const(HSLOT(dst), src, (8 * sizeof(jint)) - 1);

}

void
cvt_int_float(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_slot(dst, 0, src, HAVE_cvt_int_float, Tcomplex);

}

void
cvt_int_double(SlotInfo* dst, SlotInfo* src)
{
	lslot_lslot_slot(dst, 0, src, HAVE_cvt_int_double, Tcomplex);

}

void
cvt_long_int(SlotInfo* dst, SlotInfo* src)
{

	move_int(dst, LSLOT(src));

}

void
cvt_long_float(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_lslot(dst, 0, src, HAVE_cvt_long_float, Tcomplex);

}

void
cvt_long_double(SlotInfo* dst, SlotInfo* src)
{
	lslot_lslot_lslot(dst, 0, src, HAVE_cvt_long_double, Tcomplex);

}

void
cvt_float_int(SlotInfo* dst, SlotInfo* src)
{

	end_basic_block();
	saveframe();
	pusharg_float(src, 0);
	call_soft(soft_cvtfi);
	popargs();
	restoreframe();
	start_basic_block();
	return_int(dst);

}

void
cvt_float_long(SlotInfo* dst, SlotInfo* src)
{

	end_basic_block();
	saveframe();
	preparelonglong();
	pusharg_float(src, 0);
	pushlonglong();
	call_soft(soft_cvtfl);
	argcount++;
	popargs();
	poplonglong();
	restoreframe();
	start_basic_block();
	return_long(dst);

}

void
cvt_float_double(SlotInfo* dst, SlotInfo* src)
{
	lslot_lslot_slot(dst, 0, src, HAVE_cvt_float_double, Tcomplex);

}

void
cvt_double_int(SlotInfo* dst, SlotInfo* src)
{

	end_basic_block();
	saveframe();
	pusharg_double(src, 0);
	call_soft(soft_cvtdi);
	popargs();
	restoreframe();
	start_basic_block();
	return_int(dst);

}

void
cvt_double_long(SlotInfo* dst, SlotInfo* src)
{

	end_basic_block();
	saveframe();
	preparelonglong();
	pusharg_double(src, 0);
	pushlonglong();
	call_soft(soft_cvtdl);
	argcount++;
	popargs();
	poplonglong();
	restoreframe();
	start_basic_block();
	return_long(dst);

}

void
cvt_double_float(SlotInfo* dst, SlotInfo* src)
{
	slot_slot_lslot(dst, 0, src, HAVE_cvt_double_float, Tcomplex);

}

void
cvt_int_byte(SlotInfo* dst, SlotInfo* src)
{

	lshl_int_const(dst, src, 8 * (sizeof(jint) - sizeof(jbyte)));
	ashr_int_const(dst, dst, 8 * (sizeof(jint) - sizeof(jbyte)));

}

void
cvt_int_char(SlotInfo* dst, SlotInfo* src)
{
	and_int_const(dst, src, (1 << (8 * sizeof(jchar))) - 1);
}

void
cvt_int_short(SlotInfo* dst, SlotInfo* src)
{

	lshl_int_const(dst, src, 8 * (sizeof(jint) - sizeof(jshort)));
	ashr_int_const(dst, dst, 8 * (sizeof(jint) - sizeof(jshort)));

}



/* ----------------------------------------------------------------------- */
/* Breakpoints.								   */
/*									   */

void
breakpoint()
{
	ABORT();
}


/* ----------------------------------------------------------------------- */
/* Soft calls.								   */
/*									   */

void
softcall_lookupmethod(SlotInfo* dst, Method* meth, SlotInfo* obj)
{
	/* 'obj' must be written back since it will be reused */
	prepare_function_call();
	saveframe();
	pusharg_ref_const(meth, 1);
	pusharg_ref(obj, 0);
	call_soft(soft_lookupmethod);
	popargs();
	restoreframe();
	fixup_function_call();
	return_ref(dst);
}

void
softcall_badarrayindex(void)
{
	sync_registers();
	saveframe();
	call_soft(soft_badarrayindex);
	restoreframe();
}

void
softcall_nullpointer(void)
{
	sync_registers();
	saveframe();
	call_soft(soft_nullpointer);
	restoreframe();
}

void
softcall_new(SlotInfo* dst, Hjava_lang_Class* classobj)
{
	prepare_function_call();
	saveframe();
	pusharg_ref_const(classobj, 0);
	call_soft(soft_new);
	popargs();
	restoreframe();
	fixup_function_call();
	return_ref(dst);
}

void
softcall_newarray(SlotInfo* dst, SlotInfo* size, int type)
{
	slot_nowriteback(size);
	prepare_function_call();
	saveframe();
	pusharg_int(size, 1);
	pusharg_int_const(type, 0);
	call_soft(soft_newarray);
	popargs();
	restoreframe();
	fixup_function_call();
	return_ref(dst);
}

void
softcall_anewarray(SlotInfo* dst, SlotInfo* size, Hjava_lang_Class* type)
{
	slot_nowriteback(size);
	prepare_function_call();
	saveframe();
	pusharg_int(size, 1);
	pusharg_ref_const(type, 0);
	call_soft(soft_anewarray);
	popargs();
	restoreframe();
	fixup_function_call();
	return_ref(dst);
}

void
softcall_multianewarray(SlotInfo* dst, int size, SlotInfo* stktop, Hjava_lang_Class* classobj)
{
	int i;

	prepare_function_call();
	saveframe();
	for (i = 0; i < size; i++) {
		pusharg_int(&stktop[i], 1+size-i);
	}
	pusharg_int_const(size, 1);
	pusharg_ref_const(classobj, 0);
	call_soft(soft_multianewarray);
	popargs();
	restoreframe();
	fixup_function_call();
	return_ref(dst);
}

void
softcall_athrow(SlotInfo* obj)
{
	slot_nowriteback(obj);
	prepare_function_call();
	saveframe();
	pusharg_ref(obj, 0);
	call_soft(soft_athrow);
	popargs();
	restoreframe();
	fixup_function_call();
}

void
softcall_checkcast(SlotInfo* obj, Hjava_lang_Class* class)
{
	/* Must keep 'obj' */
	prepare_function_call();
	saveframe();
	pusharg_ref(obj, 1);
	pusharg_ref_const(class, 0);
	call_soft(soft_checkcast);
	popargs();
	restoreframe();
	fixup_function_call();
}



void
softcall_instanceof(SlotInfo* dst, SlotInfo* obj, Hjava_lang_Class* class)
{
	slot_nowriteback(obj);
	prepare_function_call();
	saveframe();
	pusharg_ref(obj, 1);
	pusharg_ref_const(class, 0);
	call_soft(soft_instanceof);
	popargs();
	restoreframe();
	fixup_function_call();
	return_int(dst);
}

void
softcall_monitorenter(SlotInfo* mon)
{
	slot_nowriteback(mon);
	prepare_function_call();
	saveframe();
	pusharg_ref(mon, 0);
	call_soft(soft_monitorenter);
	popargs();
	restoreframe();
	fixup_function_call();
}

void
softcall_monitorexit(SlotInfo* mon)
{
	slot_nowriteback(mon);
	prepare_function_call();
	saveframe();
	pusharg_ref(mon, 0);
	call_soft(soft_monitorexit);
	popargs();
	restoreframe();
	fixup_function_call();
}

void
softcall_initialise_class(Hjava_lang_Class* c)
{
	prepare_function_call();
	saveframe();
	pusharg_ref_const(c, 0);
	call_soft(soft_initialise_class);
	popargs();
	restoreframe();
	fixup_function_call();
}

void
softcall_checkarraystore(SlotInfo* array, SlotInfo* obj)
{
	prepare_function_call();
	saveframe();
	pusharg_ref(obj, 1);
	pusharg_ref(array, 0);
	call_soft(soft_checkarraystore);
	popargs();
	restoreframe();
	fixup_function_call();
}






