/*****************************************************************************
 * DESCRIPTION 
 *
 * threadPool.c   -  I/O-blocking eliminator
 *
 * CHANGELOG
 *
 * 16.2.1998   Teemu Ikonen           First and hopefully final version 
 *
 * 17.2.1998   Teemu Ikonen           Minor bugfixs, statistics collection and
 *                                    rendezvous calls for sync. Removed some
 *                                    unnecessary locks.
 * 
 * 18.2.1998   Teemu Ikonen           minor modifications to statistics
 *
 * 19.2.1998   Teemu Ikonen           Removed erranous asserts from open and 
 *                                    close.
 *
 * 20.2.1998   Teemu Ikonen           Removed fdpool as unneeded.
 *
 *****************************************************************************/
/*****************************************************************************
 * TESTTABLE
 *
 * 17.2.1998   Teemu Ikonen           Tested with common I/O-operations 
 *                                    - Needs some major optimizations
 *
 * 18.2.1998   Teemu Ikonen           Tested optimizations and behaviour with
 *                                    multithreaded applications.
 *                                    - All OK.
 *
 *
 * 20.2.1998   Teemu Ikonen           Tested with multithreaded web-server,
 *                                    works OK.
 *
 ****************************************************************************/
/*****************************************************************************
 * NOTES: different operations use memory allocations, as the argument
 *        buffer may be in calling threads stack-space. In that case the 
 *        I/O-thread is unable to access it.        
 *****************************************************************************/

#include <u.h>
#include <libc.h>
#include "plan9interface.h"
#include "config.h"
#include "config-std.h"
#include "config-io.h"
#include "config-mem.h"
#include "lerrno.h"
#include "object.h"
#include "thread.h"
#include "locks.h"
#include "flags.h"
#include "md.h"

#undef IODEBUG

/* every number is one process more, so don't mess with this.
 * Now the idea is, 2 for stdin, stdout and stderr, 1 for files
 * and 1 for net
 */
#define THREAD_POOL_SIZE 4

/* time slice to sleep when waiting I/O-thread to complete */
#define IO_WAITTIME  10

/* rendezvous identifiers */
#define SLAVE  0x0f
#define MASTER 0xf0

/* wrappers to lock */
#define LOCK_TH() lockMutex( &thpoolLock )
#define UNLOCK_TH() unlockMutex( &thpoolLock )

/* different I/O-operation identifiers */
#define OPEN  1
#define CLOSE 2
#define READ  3
#define WRITE 4

/* lock to achieve mutual exclusion when accessing threadpool */
static quickLock thpoolLock;

/* some profiling stuff */
static struct IOStat {
    int opened;
    int closed;
    int bread, bwrote;
    int reads;
    int writes;
    int errors;
    int jobs;
};

/* thread in threadpool */
typedef struct IOThread {
    long np, nr; /* these two are for read and write */
    int fd;      /* for every operation */
    int mode;    /* for open */
    int pid;     /* indicates pid of this thread */
    int code;    /* the operation */
    int job;     /* indicate job */
    int reserved; /* is this thread reserver */
    char *buff; /* buffer used for operations */
    struct IOStat *stat; /* statistics to this thread */
};

/* thread- and and statisticspool */
volatile static struct IOThread threadPool[THREAD_POOL_SIZE];
volatile static struct IOStat threadStat[THREAD_POOL_SIZE];

int killSlave(int);
void init_tio(void);
void clean_tio(void);
int launchIOSlave( struct IOThread *);
static void IOSlave( struct IOThread *);
int ioSynchronizer( struct IOThread * );

/* kill process */
int killSlave( int pid )
{
    char buffer[256];
    int fd;

    sprint( buffer, "/proc/%d/ctl", pid );
    if( (fd = open( buffer, OWRITE )) > 0 ) {
	write( fd, "kill", 4 );
	close( fd );
	return 0;
    } 
    return (-1);
}
      
/* this must be called before any threaded I/O-operation */
void init_tio()
{
    struct IOThread *t;
    int idx;

    /* format tables */
    memset( threadPool, 0, THREAD_POOL_SIZE * sizeof(struct IOThread));
    memset( threadStat, 0, THREAD_POOL_SIZE * sizeof(struct IOStat));

    /* launch threads */
    for( idx = 0 ; idx < THREAD_POOL_SIZE ; idx++ ) {

	t = &threadPool[idx];
	t->stat = &threadStat[idx];

	if( launchIOSlave( t ) <= 0 ) {
	    print("kaffe: threadlauncher failed\n");
	    exit(1);
	}
    }
}

/* called at system exit */
void clean_tio()
{
    int idx;
    struct IOThread *t;
    struct IOStat *stat;

    /* clean slaves */
    for( idx = 0 ; idx < THREAD_POOL_SIZE ; idx++ ) {
	t = &threadPool[idx];
	killSlave( t->pid );
    };

    /* print statistics */
    if( flag_iostat ) {
	print("kaffe - I/O-statistics\n");
	/* print out statistics */
	for( idx = 0 ; idx < THREAD_POOL_SIZE ; idx++ ) {
	    stat = &threadStat[idx];
	    t = &threadPool[idx];

	    print("slave: %d\n", t->pid );
	    print("\tfiles opened: %d\n", stat->opened );
	    print("\t      closed: %d\n", stat->closed );
	    print("\treads:  %d\n", stat->reads );
	    print("\twrites: %d\n", stat->writes );
	    print("\ttotal bytes read : %d\n", stat->bread );
	    print("\t            wrote: %d\n", stat->bwrote );
	    print("\ttotal jobs  : %d\n", stat->jobs );
	    print("\t      errors: %d\n\n", stat->errors );	
	}
    }
}

/* function to launch slaves */
int launchIOSlave( struct IOThread *t )
{
    int pid;

    pid = rfork( RFPROC | RFNOWAIT | RFNOTEG | RFMEM );

    /* check result */
    switch( pid ) {

	case 0: /* child, launch thread */
	    IOSlave( t );	    

	    /* above call should never return  */
	    assert(0);
	    break;

	case (-1): /* something failed */
	    return -1;
	    break;
	    
	default: /* parent, set and return value*/
	    t->pid = pid;
	    return t->pid;
	    break;
    }
    /* keep compiler happy */
    return (-1);
}

/* general io-thread */
static void IOSlave( struct IOThread *thisThread ) {

    volatile struct IOThread *this = thisThread;

    while( 1 ) {
	
	/* start waiting for job */
	while( MASTER != rendezvous( (ulong)this, SLAVE ));

	this->stat->jobs++;
	
	/* we have something to do, choose operation */
	switch( this->code ) {

	    case OPEN:

		this->stat->opened++;
#ifdef IODEBUG
		print("open_start: \"%s\"\n", this->buff );
#endif
		/* open call, can block e.g. when accepting tcp-connections */
		this->fd = open( this->buff, this->mode );
#ifdef IODEBUG
		print("open_end: \"%s\" %d\n", this->buff, this->fd );
#endif
		if( this->fd < 0 )
		    this->stat->errors++;
		break;

	    case CLOSE:
		/* close file, this may actually block if fd is 
		 * some tcp/udp filedescriptor, as buffers are 
		 * flushed to the big bad internet 
		 */
		this->stat->closed++;
#ifdef IODEBUG
		print("close_start: %d\n", this->fd );
#endif
		this->fd = close( this->fd );
#ifdef IODEBUG
		print("close_end: %d\n", this->result );
#endif
		if( this->fd < 0 )
		    this->stat->errors++;
		break;

	    case READ:
		this->stat->reads++;
#ifdef IODEBUG
		print("read_start: %d %d %d\n", this->fd, this->buff, this->nr ); 
#endif
		/* yea, it can block */
		this->np = read( this->fd, this->buff, this->nr );
#ifdef IODEBUG
		print("read_end: %d\n", this->np );
#endif
		if( this->np > 0 )
		    this->stat->bread += this->np;
		else this->stat->errors++;
		break;
		
	    case WRITE:
		this->stat->writes++;
#ifdef IODEBUG
		print("write_start: %d %d %d\n", this->fd, this->buff, this->nr ); 
#endif
		/* this can block at least if this is an tcp/udp filedescriptor */
		this->np = write( this->fd, this->buff, this->nr );
#ifdef IODEBUG
		print("write_end: %d\n", this->np );
#endif
		if( this->np > 0 )
		    this->stat->bwrote += this->np;
		else this->stat->errors++;
		break;

		
	    default:
		/* unable to find appropriate handler, abandon ship */
		assert(0);
	 		
	} /* switch( this->code ) */

	/* indicate that we are done */
	this->job = false;
    
    } /* while( 1 ) */
}

/* every I/O-operation call this to achieve synchronized I/O-operations */
int ioSynchronizer( struct IOThread *arg )
{
    volatile struct IOThread *t;
    int idx;

    /* first we need to find free thread to 
     * do this operation 
     */
    while( 1 ) {
	/* achieve mutual exclusion */
	LOCK_TH();

	/* walk threadpool */
	for( idx = 0 ; idx < THREAD_POOL_SIZE && threadPool[idx].reserved != 0 ; idx++ );
	
	if( idx == THREAD_POOL_SIZE ) {
	    /* we didn't found any free thread, free mutex and wait */
	    UNLOCK_TH();
	    sleepThread(IO_WAITTIME);

	} else {
	    /* free found, now proceed with plan  */
	    t = &threadPool[idx];

	    /* reserve thread and free mutex */
	    t->reserved = true;
	    UNLOCK_TH();
	
	    /* init structure */
	    t->np = arg->np;
	    t->nr = arg->nr;
	    t->fd = arg->fd;
	    t->mode = arg->mode;
	    t->code = arg->code;
	    t->buff = arg->buff;
	    t->job = true;

	    /* wake up thread */
	    while( SLAVE != rendezvous( (ulong)t, MASTER ));
	    
	    /* this is native yield */
	    sleep(0);

	    /* now the thread is executing request, wait while it's finished */
	    while( t->job == true ) {
		sleepThread( IO_WAITTIME );
	    }

	    /* thread is done, clean up and return result */
	    LOCK_TH();

	    arg->np = t->np;
	    arg->nr = t->nr;
	    arg->fd = t->fd;

	    t->reserved = false;

	    UNLOCK_TH();

	    /* done */
	    return ( 0 );
	} /* else */

    } /* while( 1 ) */

    /* this should be neverever reached, return is just to reduce compiler warnings */
    assert(0);
    return (-1);  
}

/* threaded open, this opens file and never blocks */
int threadedOpen( char *path, int mode )
{
    struct IOThread t;

    /* paranoid... */
    memset( &t, 0, sizeof( struct IOThread ) );

    /* init structure */
    t.code = OPEN;
    t.mode = mode;
    t.buff = gc_malloc_fixed( strlen( path ) + 1 );
    strcpy( t.buff, path );

    /* do the actual operation */
    ioSynchronizer( &t );

    gc_free( t.buff );

    /* finally done */
    return t.fd;
}

/* close file */
int threadedClose( int fd )
{
    struct IOThread t;

    /* paranoid... */
    memset( &t, 0, sizeof( struct IOThread ) );

    if( fd < 0 || fd > FOPEN_MAX ) 
	return -1;


    /* guess the fd is valid */

    /* init structure */
    t.code = CLOSE;
    t.fd = fd;

    /* do the actual operation */
    ioSynchronizer( &t );

    /* finally done */
    return t.fd;
}

/* read */
long threadedRead( int fd, void *buff, unsigned int len )
{
    struct IOThread t;

    /* paranoid... */
    memset( &t, 0, sizeof( struct IOThread ) );

    if( fd < 0 || fd > FOPEN_MAX ) 
	return -1;
    

    /* guess the fd is valid */

    /* init structure */
    t.code = READ;
    t.fd = fd;
    t.nr = len;
    t.buff = gc_malloc_fixed( len );

    /* do the actual operation */
    ioSynchronizer( &t );

    /* transfer data and free temporary buffer */
    memcpy( buff, t.buff, len );
    gc_free( t.buff );

    /* result of read/write is stored to np */
    return t.np;
}

/* write */
long threadedWrite( int fd, void *buff, unsigned int len )
{
    struct IOThread t;

    /* paranoid... */
    memset( &t, 0, sizeof( struct IOThread ) );

    if( fd < 0 || fd > FOPEN_MAX ) 
	return -1;


    /* guess the fd is valid */

    /* init structure */
    t.code = WRITE;
    t.fd = fd;
    t.nr = len;
    t.buff = gc_malloc_fixed( len );
    
    /* transfer data */
    memcpy( t.buff, buff, len );

    /* do the actual operation */
    ioSynchronizer( &t );

    /* free temporary buffer */
    gc_free( t.buff );

    /* result of read/write is stored to np */
    return t.np;
}

/* now just dummy function */
int threadedFileDescriptor(int i )
{
    return (i);
}



