#include  <stdio.h>
#include  <string.h>
#include  <time.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "and3.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "and3.c"                                           */


/* Module internal constants. */
#define        E_AND3_WIDTH         2
#define        E_AND3_SLICE         3
#define        E_AND3_DRIVE         4
#define        E_AND3_NOVIEW        5
#define        E_AND3_OPENVBE       6
#define        E_AND3_MODELNAME     7


/* Names of and3 leaf cells. */
static char *TablLeafCellAnd3[2] = {  "a3_fp",
									 "a3p_fp" };


/* Interface function of module.
   extern       char *Dgn_DispachAnd3();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorAnd3();
static       void  Dgn_VhdlAnd3();
static       void  Dgn_NetlistAnd3();


/*  Function  :  'Dgn_DispatchAnd3'                               */
/* -------------------------------------------------------------- */
/*  Interface of and3 generator.                                  */

extern char *Dgn_DispatchAnd3(and3_modelname,
							  and3_width    ,
							  and3_slice    ,
							  and3_drive    ,
							  and3_msb0     ,
							  and3_vhdl     ,
							  and3_netlist  ,
							  and3_icon     )
	char *and3_modelname;  /* Model name.                    */
	long  and3_width;      /* Bus wide.                      */
	long  and3_slice;      /* LSB starting slice.            */
	long  and3_drive;      /* Output drive capacity.         */
	long  and3_msb0;       /* Zero as MSB index.             */
	long  and3_vhdl;       /* Generation of behavioral view. */
	long  and3_netlist;    /* Generation of netlist view.    */
	long  and3_icon;       /* Generation of icon.            */
{
	static short  DispachAnd3_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachAnd3_FirstCall)
	{
		Dgn_ErrorAnd3();
		DispachAnd3_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (and3_netlist == FALSE)
		 &&(and3_vhdl    == FALSE)
		 &&(and3_icon    == FALSE) )
		Dut_PrintError( E_AND3_ERROR, E_AND3_NOVIEW);

	/* Check the bus wide. */
	if ((and3_width < 1) || (and3_width > DP_WIDTH))
		Dut_PrintError( E_AND3_ERROR, E_AND3_WIDTH, and3_width);

	/* Check the output drive capacity. */
	if ((and3_drive < 1) || (and3_drive > C_AND3_NBDRIVE))
		Dut_PrintError( E_AND3_ERROR, E_AND3_DRIVE, and3_drive);

	/* Check the LSB starting slice. */
	if ((and3_slice < 0) || ( and3_slice
							 +and3_width > DP_WIDTH))
		Dut_PrintError( E_AND3_ERROR, E_AND3_SLICE, and3_slice);

	/* Check the model name. */
	if (and3_modelname == (char*)NULL)
#ifndef  CRUNCH_FILENAME
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "and3_%ldx%ldx%ldx%ld%c_cl",
					                   DP_WIDTH              ,
					                 and3_width              ,
					                 and3_slice              ,
					                 and3_drive              ,
					               ((and3_msb0 ) ? 'm' : 'l'));
#else
		/* Build the crunched model name. */
		strcpy( ModelName, Dut_MakeCrunchModelName(
									    "od"       ,
								   8L,   DP_WIDTH-1,
								   8L, and3_width-1,
								   8L, and3_slice  ,
								   8L, and3_drive  ,
								   0L, and3_msb0   ));
#endif
	else
		strcpy( ModelName, and3_modelname);


	/* Generation of behaviorial view. */
	if (and3_vhdl)
		Dgn_VhdlAnd3( ModelName,
			   		  and3_width,
					  and3_slice,
			   		  and3_drive,
			   		  and3_msb0 );

	/* Generation of netlist view. */
	if (and3_netlist)
		Dgn_NetlistAnd3( ModelName,
						 and3_width,
						 and3_slice,
					     and3_drive,
						 and3_msb0 );

	/* Generation of icone. */
	/* if (and3_icon) */
	
	/* Return name of the generated model. */
	return((and3_modelname == (char*)NULL) ?      ModelName 
		                                   : and3_modelname);
}


/*  Function  :  'Dgn_ErrorAnd3'                                  */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorAnd3()
{
	Dut_AddError( E_AND3_ERROR, E_AND3_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_AND3_ERROR, E_AND3_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_AND3_ERROR, E_AND3_DRIVE,
				  "invalid output drive : %ld.");
	Dut_AddError( E_AND3_ERROR, E_AND3_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_AND3_ERROR, E_AND3_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_AND3_ERROR, E_AND3_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListAnd3'                                */
/* -------------------------------------------------------------- */
/*  Generation of and3 behavioral view.                           */

static void  Dgn_VhdlAnd3(and3_modelname,
					      and3_width    ,
					      and3_slice    ,
					      and3_drive    ,
						  and3_msb0     )
	char *and3_modelname;  /* Nom du modele.           */
	long  and3_width;      /* Largeur du bus.          */
	long  and3_slice;      /* LSB starting slice.      */
	long  and3_drive;      /* Output drive power.      */
	long  and3_msb0;       /* Zero comme index du MSB. */
{
	FILE *And3File;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((And3File=mbkfopen(and3_modelname,
						            "vbe",
						       WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_AND3_ERROR, E_AND3_OPENVBE,
					                  and3_modelname);

	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( And3File, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( And3File, 
"-- #                                                                 #\n"
			);
	fprintf( And3File,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( And3File,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( And3File,
"-- #                                                                 #\n"
			);
    fprintf( And3File,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( And3File,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( And3File, 
"-- # *************************************************************** #\n"
			);
	fprintf( And3File,
"-- #                                                                 #\n"
			);
    fprintf( And3File,
"-- # Generator : \"and3g\" (and3)                                      #\n"
			);
	fprintf( And3File, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf( TmpString, "\"%s.vbe\"."           , and3_modelname);
    fprintf( And3File , "-- # File : %-55s  #\n",     TmpString);
	fprintf( And3File ,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( And3File , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"and3g %ld %ld %ld %ld%s -vhdl -o %s\"" ,
			                         DP_WIDTH                      ,
			                       and3_width                      ,
			                       and3_slice                      ,
			                       and3_drive                      ,
			                      (and3_msb0     ) ? " -msb0"  : "",
			                       and3_modelname                  );
    fprintf( And3File , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( And3File, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "and3_msb0")            */
	if (!and3_msb0)
		sprintf( BusWide, "(%ld downto %ld)", and3_width-1
				                             +and3_slice  ,
				                              and3_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-and3_slice
				                                      -and3_width  ,
				                              DP_WIDTH-and3_slice-1);


	/* *** Interface description. *** */
	fprintf( And3File, "\nENTITY %s IS\n\tPORT(\n", and3_modelname);

	/* Terminals of data buses. */
	fprintf( And3File, "%14s :in    BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( And3File, "%14s :in    BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( And3File, "%14s :in    BIT_VECTOR%s;\n", "i2", BusWide);
	fprintf( And3File, "%14s :inout BIT_VECTOR%s;\n", "no", BusWide);
	fprintf( And3File, "%14s :  out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( And3File, "%14s :  in BIT;\n", "vdd");
	fprintf( And3File, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( And3File, "\t);\nEND %s;\n\n\n", and3_modelname);


	/* *** Behavioral view *** */
	fprintf( And3File, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            and3_modelname);
	fprintf( And3File, "BEGIN\n\n");
	fprintf( And3File, "\to%s <= not no%s;\n" , BusWide,
			                                    BusWide);
	fprintf( And3File, "\tno%s <= not(i0%s and i1%s and i2%s);\n\n" , BusWide,
			                                                BusWide,
			                                                BusWide,
			                                                BusWide);

	/* Check supplies connections. */
	fprintf( And3File, "\t-- ****************>");
	fprintf( And3File, " Supplies Check "      );
	fprintf( And3File, "<****************\n\n" );

	fprintf( And3File, "\tASSERT((vdd = '1') and (vss = '0'))\n");
	fprintf( And3File, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              and3_modelname);
	fprintf( And3File, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( And3File, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( And3File);
}


/*  Function  :  'Dgn_NetListAnd3'                                */
/* -------------------------------------------------------------- */
/*  Generation of and3erter/buffer netlist view.                  */

static void  Dgn_NetlistAnd3(and3_modelname,
						     and3_width    ,
						     and3_slice    ,
							 and3_drive    ,
						     and3_msb0     )
	char *and3_modelname;  /* Model name.            */
	long  and3_width;      /* Bus wide.              */
	long  and3_slice;      /* LSB starting slice.    */
	long  and3_drive;      /* Output drive capacity. */
	long  and3_msb0;       /* Zero as MSB index.     */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameI2[C_SUPP_MBKSZNAME];
	char  BusNameNO[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameI2[C_SUPP_MBKSZNAME];
	char  PinNameNO[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Select name of the leaf cell model. */
	/* (according to output drive power)   */
	LeafCellSliceName = TablLeafCellAnd3[and3_drive-1];

	/* Creation of the new logical figure. */
	DEF_LOFIG( and3_modelname);

	/* Name of data bus terminals. */
	if (!and3_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", and3_width-1+and3_slice, and3_slice);
		sprintf( BusNameI1,"i1[%ld:%ld]", and3_width-1+and3_slice, and3_slice);
		sprintf( BusNameI2,"i2[%ld:%ld]", and3_width-1+and3_slice, and3_slice);
		sprintf( BusNameNO,"no[%ld:%ld]", and3_width-1+and3_slice, and3_slice);
		sprintf( BusNameO , "o[%ld:%ld]", and3_width-1+and3_slice, and3_slice);
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-and3_slice
				                                  -and3_width  ,
				                          DP_WIDTH-and3_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-and3_slice
				                                  -and3_width  ,
				                          DP_WIDTH-and3_slice-1);
		sprintf( BusNameI2,"i2[%ld:%ld]", DP_WIDTH-and3_slice
				                                  -and3_width  ,
				                          DP_WIDTH-and3_slice-1);
		sprintf( BusNameNO,"no[%ld:%ld]", DP_WIDTH-and3_slice
				                                  -and3_width  ,
				                          DP_WIDTH-and3_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-and3_slice
				                                  -and3_width  ,
				                          DP_WIDTH-and3_slice-1);
	}

 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameI2, IN   , BusNameI2);
	LOCON( BusNameNO, INOUT, BusNameNO);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Data bus terminals **** */
	LOCON(     "vdd", INOUT,     "vdd"); 
	LOCON(     "vss", INOUT,     "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=and3_slice;
		 LeafCellIndex<and3_slice+and3_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "and3_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!and3_msb0) BusIndex =              LeafCellIndex;
		else            BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameI2, "i2[%ld]", BusIndex);
		sprintf( PinNameNO, "no[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameI2 ,
				      PinNameNO , 		
			          PinNameO  , "vdd", "vss", EOL);
	}

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
