#include  <time.h>
#include  <stdio.h>
#include  <string.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "mux2cs.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "mux2cs.c"                                         */


/* Module internal constants. */
#define        E_MUX2CS_WIDTH         2
#define        E_MUX2CS_SLICE         3
#define        E_MUX2CS_NOVIEW        4
#define        E_MUX2CS_OPENVBE       5
#define        E_MUX2CS_MODELNAME     6


/* Names of tristate leaf cells. */
static char *TablLeafCellMux2cs[2] = { "mx2sb0_fp",
									    "mx2sc_fp"};


/* Interface function of module.
   extern       char *Dgn_DispachMux2cs();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorMux2cs();
static       void  Dgn_VhdlMux2cs();
static       void  Dgn_NetlistMux2cs();


/*  Function  :  'Dgn_DispatchMux2cs'                             */
/* -------------------------------------------------------------- */
/*  Interface of a two input multiplexer generator.               */

extern char *Dgn_DispatchMux2cs(mux2cs_modelname,
								mux2cs_width    ,
								mux2cs_slice    ,
								mux2cs_msb0     ,
								mux2cs_vhdl     ,
								mux2cs_netlist  ,
								mux2cs_icon     )
	char *mux2cs_modelname;  /* Model name.                    */
	long  mux2cs_width;      /* Bus wide.                      */
	long  mux2cs_slice;      /* LSB starting slice.            */
	long  mux2cs_msb0;       /* Zero as MSB index.             */
	long  mux2cs_vhdl;       /* Generation of behavioral view. */
	long  mux2cs_netlist;    /* Generation of netlist view.    */
	long  mux2cs_icon;       /* Generation of icon.            */
{
	static short  DispachMux2cs_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachMux2cs_FirstCall)
	{
		Dgn_ErrorMux2cs();
		DispachMux2cs_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (mux2cs_netlist == FALSE)
		 &&(mux2cs_vhdl    == FALSE)
		 &&(mux2cs_icon    == FALSE) )
		Dut_PrintError( E_MUX2CS_ERROR, E_MUX2CS_NOVIEW);

	/* Check the bus wide. */
	if ((mux2cs_width < 1) || (mux2cs_width > DP_WIDTH))
		Dut_PrintError( E_MUX2CS_ERROR, E_MUX2CS_WIDTH, mux2cs_width);

	/* Check the LSB starting slice. */
	if ((mux2cs_slice < 0) || ( mux2cs_slice
							 +mux2cs_width > DP_WIDTH))
		Dut_PrintError( E_MUX2CS_ERROR, E_MUX2CS_SLICE, mux2cs_slice);

	/* Check the model name. */
	if (mux2cs_modelname == (char*)NULL)
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "mux2cs_%ldx%ldx%ld%c_cl"  ,
					                    DP_WIDTH              ,
					                mux2cs_width              ,
					                mux2cs_slice              ,
					              ((mux2cs_msb0 ) ? 'm' : 'l'));
	else
		strcpy( ModelName, mux2cs_modelname);

	/* Generation of behaviorial view. */
	if (mux2cs_vhdl)
		Dgn_VhdlMux2cs(ModelName    ,
					   mux2cs_width,
					   mux2cs_slice,
					   mux2cs_msb0 );

	/* Generation of netlist view. */
	if (mux2cs_netlist)
		Dgn_NetlistMux2cs(ModelName    ,
						  mux2cs_width,
						  mux2cs_slice,
						  mux2cs_msb0 );

	/* Generation of icon. */
	/* if (mux2cs_icon) */
	
	/* Return name of the generated model. */
	return((mux2cs_modelname == (char*)NULL) ?        ModelName 
		                                     : mux2cs_modelname);
}


/*  Function  :  'Dgn_ErrorMux2cs'                                */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorMux2cs()
{
	Dut_AddError( E_MUX2CS_ERROR, E_MUX2CS_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_MUX2CS_ERROR, E_MUX2CS_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_MUX2CS_ERROR, E_MUX2CS_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_MUX2CS_ERROR, E_MUX2CS_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_MUX2CS_ERROR, E_MUX2CS_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListMux2cs'                              */
/* -------------------------------------------------------------- */
/*  Generation of multiplexer behavioral view.                    */

static void  Dgn_VhdlMux2cs(mux2cs_modelname,
							mux2cs_width    ,
							mux2cs_slice    ,
							mux2cs_msb0     )
	char *mux2cs_modelname;  /* Model name.         */
	long  mux2cs_width;      /* Bus width.          */
	long  mux2cs_slice;      /* LSB starting slice. */
	long  mux2cs_msb0;       /* Zero as MSB index.  */
{
	FILE *Mux2csFile;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Mux2csFile=mbkfopen(mux2cs_modelname,
							            "vbe",
						           WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_MUX2CS_ERROR, E_MUX2CS_OPENVBE,
					                      mux2cs_modelname);


	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Mux2csFile, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Mux2csFile, 
"-- #                                                                 #\n"
			);
	fprintf( Mux2csFile,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Mux2csFile,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Mux2csFile,
"-- #                                                                 #\n"
			);
    fprintf( Mux2csFile,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Mux2csFile,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Mux2csFile, 
"-- # *************************************************************** #\n"
			);
	fprintf( Mux2csFile,
"-- #                                                                 #\n"
			);
    fprintf( Mux2csFile,
"-- # Generator : \"mux2csg\" (non-inverter tristate)                  #\n"
			);
	fprintf( Mux2csFile, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf(  TmpString, "\"%s.vbe\"."           , mux2cs_modelname);
    fprintf( Mux2csFile  , "-- # File : %-55s  #\n",      TmpString);
	fprintf( Mux2csFile,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Mux2csFile  , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"mux2csg %ld %ld %ld%s -vhdl -o %s\"" ,
			                     DP_WIDTH                        ,
			                 mux2cs_width                        ,
			                 mux2cs_slice                        ,
			                (mux2cs_msb0     ) ? " -msb0"  : ""  ,
			                 mux2cs_modelname                    );
    fprintf( Mux2csFile  , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Mux2csFile, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "mux2cs_msb0")            */
	if (!mux2cs_msb0)
		sprintf( BusWide, "(%ld downto %ld)", mux2cs_width-1
				                             +mux2cs_slice  ,
				                              mux2cs_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-mux2cs_slice
				                                      -mux2cs_width  ,
				                              DP_WIDTH-mux2cs_slice-1);


	/* *** Interface description. *** */
	fprintf( Mux2csFile, "\nENTITY %s IS\n\tPORT(\n", mux2cs_modelname);

	/* Controls terminals. */
	fprintf( Mux2csFile, "%14s :  in BIT;\n",  "sel");

	/* Terminals of data mux2css. */
	fprintf( Mux2csFile, "%14s :  in BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Mux2csFile, "%14s :  in BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( Mux2csFile, "%14s : out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Mux2csFile, "%14s :  in BIT;\n", "vdd");
	fprintf( Mux2csFile, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Mux2csFile, "\t);\nEND %s;\n\n\n", mux2cs_modelname);


	/* *** Behavioral view *** */
	fprintf( Mux2csFile, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            mux2cs_modelname);
	fprintf( Mux2csFile, "BEGIN\n\n");

	fprintf( Mux2csFile,     "\tWITH  sel  SELECT\n");
	fprintf( Mux2csFile,   "\t\to%s <= \n"           , BusWide);
	fprintf( Mux2csFile, "\t\t\ti0%s  WHEN \'0\',\n" , BusWide);
	fprintf( Mux2csFile, "\t\t\ti1%s  WHEN \'1\';\n" , BusWide);

	/* Check supplies connections. */
	fprintf( Mux2csFile, "\t-- ****************>");
	fprintf( Mux2csFile, " Supplies Check "      );
	fprintf( Mux2csFile, "<****************\n\n" );

	fprintf( Mux2csFile, "\tASSERT ((vdd = '1') and (vss = '0'))\n");
	fprintf( Mux2csFile, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              mux2cs_modelname);
	fprintf( Mux2csFile, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Mux2csFile, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Mux2csFile);
}


/*  Fonction  :  'Dgn_NetListMux2cs'                              */
/* -------------------------------------------------------------- */
/*  Generation of Multiplexer Netlist.                            */

static void  Dgn_NetlistMux2cs(mux2cs_modelname,
							   mux2cs_width    ,
							   mux2cs_slice    ,
							   mux2cs_msb0     )
	char *mux2cs_modelname;  /* Model name.         */
	long  mux2cs_width;      /* Bus wide.           */
	long  mux2cs_slice;      /* LSB starting slice. */
	long  mux2cs_msb0;       /* Zero as MSB index.  */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char *LeafCellBuffrName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Model names (slices and buffer). */
	LeafCellSliceName = TablLeafCellMux2cs[1];
	LeafCellBuffrName = TablLeafCellMux2cs[0];

	/* Creation of the new logical figure. */
	DEF_LOFIG( mux2cs_modelname);

	/* Name of data bus terminals. */
	if (!mux2cs_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", mux2cs_width-1
				                         +mux2cs_slice  ,
				                          mux2cs_slice  );
		sprintf( BusNameI1,"i1[%ld:%ld]", mux2cs_width-1
				                         +mux2cs_slice  ,
				                          mux2cs_slice  );
		sprintf( BusNameO , "o[%ld:%ld]", mux2cs_width-1
				                         +mux2cs_slice  ,
				                          mux2cs_slice  );
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-mux2cs_slice
				                                  -mux2cs_width  ,
				                          DP_WIDTH-mux2cs_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-mux2cs_slice
				                                  -mux2cs_width  ,
				                          DP_WIDTH-mux2cs_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-mux2cs_slice
				                                  -mux2cs_width  ,
				                          DP_WIDTH-mux2cs_slice-1);
	}
 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON(     "sel", IN   ,      "sel");

	/* **** Data bus terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Supply terminals **** */
	LOCON(     "vdd", INOUT,      "vdd"); 
	LOCON(     "vss", INOUT,      "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=mux2cs_slice; 
		 LeafCellIndex<mux2cs_slice+mux2cs_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "mux2cs_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!mux2cs_msb0) BusIndex =              LeafCellIndex;
		else               BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			                "s0",
			                "s1",
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameO  , "vdd", "vss", EOL);
	}


	/* Creation of buffer instance. */

	/* Build buffer instance name. */
	sprintf( LeafCellInstaName, "bmux2cs_%ld", DP_WIDTH);

	/* Buffer instanciation. */
	LOINS( LeafCellBuffrName,
		   LeafCellInstaName,
		               "sel",
		                "s0",
		                "s1", "vdd", "vss", VSS, EOL);

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
