#include  <time.h>
#include  <stdio.h>
#include  <string.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "mux3cd.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "mux3cd.c"                                         */


/* Module internal constants. */
#define        E_MUX3CD_WIDTH         2
#define        E_MUX3CD_SLICE         3
#define        E_MUX3CD_NOVIEW        4
#define        E_MUX3CD_OPENVBE       5
#define        E_MUX3CD_MODELNAME     6


/* Names of tristate leaf cells. */
static char *TablLeafCellMux3cd[2] = { "mx3db0_fp",
									    "mx3dc_fp"};


/* Interface function of module.
   extern       char *Dgn_DispachMux3cd();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorMux3cd();
static       void  Dgn_VhdlMux3cd();
static       void  Dgn_NetlistMux3cd();


/*  Function  :  'Dgn_DispatchMux3cd'                             */
/* -------------------------------------------------------------- */
/*  Interface of a two input multiplexer generator.               */

extern char *Dgn_DispatchMux3cd(mux3cd_modelname,
								mux3cd_width    ,
								mux3cd_slice    ,
								mux3cd_msb0     ,
								mux3cd_vhdl     ,
								mux3cd_netlist  ,
								mux3cd_icon     )
	char *mux3cd_modelname;  /* Model name.                    */
	long  mux3cd_width;      /* Bus wide.                      */
	long  mux3cd_slice;      /* LSB starting slice.            */
	long  mux3cd_msb0;       /* Zero as MSB index.             */
	long  mux3cd_vhdl;       /* Generation of behavioral view. */
	long  mux3cd_netlist;    /* Generation of netlist view.    */
	long  mux3cd_icon;       /* Generation of icon.            */
{
	static short  DispachMux3cd_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachMux3cd_FirstCall)
	{
		Dgn_ErrorMux3cd();
		DispachMux3cd_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (mux3cd_netlist == FALSE)
		 &&(mux3cd_vhdl    == FALSE)
		 &&(mux3cd_icon    == FALSE) )
		Dut_PrintError( E_MUX3CD_ERROR, E_MUX3CD_NOVIEW);

	/* Check the bus wide. */
	if ((mux3cd_width < 1) || (mux3cd_width > DP_WIDTH))
		Dut_PrintError( E_MUX3CD_ERROR, E_MUX3CD_WIDTH, mux3cd_width);

	/* Check the LSB starting slice. */
	if ((mux3cd_slice < 0) || ( mux3cd_slice
							 +mux3cd_width > DP_WIDTH))
		Dut_PrintError( E_MUX3CD_ERROR, E_MUX3CD_SLICE, mux3cd_slice);

	/* Check the model name. */
	if (mux3cd_modelname == (char*)NULL)
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "mux3cd_%ldx%ldx%ld%c_cl"  ,
					                    DP_WIDTH              ,
					                mux3cd_width              ,
					                mux3cd_slice              ,
					              ((mux3cd_msb0 ) ? 'm' : 'l'));
	else
		strcpy( ModelName, mux3cd_modelname);

	/* Generation of behaviorial view. */
	if (mux3cd_vhdl)
		Dgn_VhdlMux3cd(ModelName    ,
					   mux3cd_width,
					   mux3cd_slice,
					   mux3cd_msb0 );

	/* Generation of netlist view. */
	if (mux3cd_netlist)
		Dgn_NetlistMux3cd(ModelName    ,
						  mux3cd_width,
						  mux3cd_slice,
						  mux3cd_msb0 );

	/* Generation of icon. */
	/* if (mux3cd_icon) */
	
	/* Return name of the generated model. */
	return((mux3cd_modelname == (char*)NULL) ?        ModelName 
		                                     : mux3cd_modelname);
}


/*  Function  :  'Dgn_ErrorMux3cd'                                */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorMux3cd()
{
	Dut_AddError( E_MUX3CD_ERROR, E_MUX3CD_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_MUX3CD_ERROR, E_MUX3CD_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_MUX3CD_ERROR, E_MUX3CD_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_MUX3CD_ERROR, E_MUX3CD_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_MUX3CD_ERROR, E_MUX3CD_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListMux3cd'                              */
/* -------------------------------------------------------------- */
/*  Generation of multiplexer behavioral view.                    */

static void  Dgn_VhdlMux3cd(mux3cd_modelname,
							mux3cd_width    ,
							mux3cd_slice    ,
							mux3cd_msb0     )
	char *mux3cd_modelname;  /* Model name.         */
	long  mux3cd_width;      /* Bus width.          */
	long  mux3cd_slice;      /* LSB starting slice. */
	long  mux3cd_msb0;       /* Zero as MSB index.  */
{
	FILE *Mux3cdFile;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Mux3cdFile=mbkfopen(mux3cd_modelname,
							            "vbe",
						           WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_MUX3CD_ERROR, E_MUX3CD_OPENVBE,
					                      mux3cd_modelname);


	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Mux3cdFile, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Mux3cdFile, 
"-- #                                                                 #\n"
			);
	fprintf( Mux3cdFile,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Mux3cdFile,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Mux3cdFile,
"-- #                                                                 #\n"
			);
    fprintf( Mux3cdFile,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Mux3cdFile,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Mux3cdFile, 
"-- # *************************************************************** #\n"
			);
	fprintf( Mux3cdFile,
"-- #                                                                 #\n"
			);
    fprintf( Mux3cdFile,
"-- # Generator : \"mux3cdg\" (non-inverter tristate)                  #\n"
			);
	fprintf( Mux3cdFile, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf(  TmpString, "\"%s.vbe\"."           , mux3cd_modelname);
    fprintf( Mux3cdFile  , "-- # File : %-55s  #\n",      TmpString);
	fprintf( Mux3cdFile,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Mux3cdFile  , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"mux3cdg %ld %ld %ld%s -vhdl -o %s\"" ,
			                     DP_WIDTH                        ,
			                 mux3cd_width                        ,
			                 mux3cd_slice                        ,
			                (mux3cd_msb0     ) ? " -msb0"  : ""  ,
			                 mux3cd_modelname                    );
    fprintf( Mux3cdFile  , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Mux3cdFile, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "mux3cd_msb0")            */
	if (!mux3cd_msb0)
		sprintf( BusWide, "(%ld downto %ld)", mux3cd_width-1
				                             +mux3cd_slice  ,
				                              mux3cd_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-mux3cd_slice
				                                      -mux3cd_width  ,
				                              DP_WIDTH-mux3cd_slice-1);


	/* *** Interface description. *** */
	fprintf( Mux3cdFile, "\nENTITY %s IS\n\tPORT(\n", mux3cd_modelname);

	/* Controls terminals. */
	fprintf( Mux3cdFile, "%14s :  in BIT_VECTOR(1 downto 0);\n", "sel");

	/* Terminals of data mux3cds. */
	fprintf( Mux3cdFile, "%14s :  in BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Mux3cdFile, "%14s :  in BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( Mux3cdFile, "%14s :  in BIT_VECTOR%s;\n", "i2", BusWide);
	fprintf( Mux3cdFile, "%14s : out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Mux3cdFile, "%14s :  in BIT;\n", "vdd");
	fprintf( Mux3cdFile, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Mux3cdFile, "\t);\nEND %s;\n\n\n", mux3cd_modelname);


	/* *** Behavioral view *** */
	fprintf( Mux3cdFile, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            mux3cd_modelname);
	fprintf( Mux3cdFile, "BEGIN\n\n");

	fprintf( Mux3cdFile,     "\tWITH  sel  SELECT\n");
	fprintf( Mux3cdFile,   "\t\to%s <= \n"             , BusWide);
	fprintf( Mux3cdFile, "\t\t\ti0%s  WHEN  \"00\",\n" , BusWide);
	fprintf( Mux3cdFile, "\t\t\ti1%s  WHEN  \"01\",\n" , BusWide);
	fprintf( Mux3cdFile, "\t\t\ti2%s  WHEN  \"10\"\n"  , BusWide);
	fprintf( Mux3cdFile, "\t\t\t          | \"11\";\n" , BusWide);

	/* Check supplies connections. */
	fprintf( Mux3cdFile, "\t-- ****************>");
	fprintf( Mux3cdFile, " Supplies Check "      );
	fprintf( Mux3cdFile, "<****************\n\n" );

	fprintf( Mux3cdFile, "\tASSERT ((vdd = '1') and (vss = '0'))\n");
	fprintf( Mux3cdFile, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              mux3cd_modelname);
	fprintf( Mux3cdFile, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Mux3cdFile, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Mux3cdFile);
}


/*  Fonction  :  'Dgn_NetListMux3cd'                              */
/* -------------------------------------------------------------- */
/*  Generation of Multiplexer Netlist.                            */

static void  Dgn_NetlistMux3cd(mux3cd_modelname,
							   mux3cd_width    ,
							   mux3cd_slice    ,
							   mux3cd_msb0     )
	char *mux3cd_modelname;  /* Model name.         */
	long  mux3cd_width;      /* Bus wide.           */
	long  mux3cd_slice;      /* LSB starting slice. */
	long  mux3cd_msb0;       /* Zero as MSB index.  */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char *LeafCellBuffrName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameI2[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameI2[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Model names (slices and buffer). */
	LeafCellSliceName = TablLeafCellMux3cd[1];
	LeafCellBuffrName = TablLeafCellMux3cd[0];

	/* Creation of the new logical figure. */
	DEF_LOFIG( mux3cd_modelname);

	/* Name of data bus terminals. */
	if (!mux3cd_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", mux3cd_width-1
				                         +mux3cd_slice  ,
				                          mux3cd_slice  );
		sprintf( BusNameI1,"i1[%ld:%ld]", mux3cd_width-1
				                         +mux3cd_slice  ,
				                          mux3cd_slice  );
		sprintf( BusNameI2,"i2[%ld:%ld]", mux3cd_width-1
				                         +mux3cd_slice  ,
				                          mux3cd_slice  );
		sprintf( BusNameO , "o[%ld:%ld]", mux3cd_width-1
				                         +mux3cd_slice  ,
				                          mux3cd_slice  );
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-mux3cd_slice
				                                  -mux3cd_width  ,
				                          DP_WIDTH-mux3cd_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-mux3cd_slice
				                                  -mux3cd_width  ,
				                          DP_WIDTH-mux3cd_slice-1);
		sprintf( BusNameI2,"i2[%ld:%ld]", DP_WIDTH-mux3cd_slice
				                                  -mux3cd_width  ,
				                          DP_WIDTH-mux3cd_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-mux3cd_slice
				                                  -mux3cd_width  ,
				                          DP_WIDTH-mux3cd_slice-1);
	}
 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON("sel[1:0]", IN   ,"sel[1:0]");

	/* **** Data bus terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameI2, IN   , BusNameI2);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Supply terminals **** */
	LOCON(     "vdd", INOUT,      "vdd"); 
	LOCON(     "vss", INOUT,      "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=mux3cd_slice; 
		 LeafCellIndex<mux3cd_slice+mux3cd_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "mux3cd_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!mux3cd_msb0) BusIndex =              LeafCellIndex;
		else              BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameI2, "i2[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			                "s0",
			                "s1",
			                "s2",
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameI2 ,
			          PinNameO  , "vdd", "vss", EOL);
	}


	/* Creation of buffer instance. */

	/* Build buffer instance name. */
	sprintf( LeafCellInstaName, "bmux3cd_%ld", DP_WIDTH);

	/* Buffer instanciation. */
	LOINS( LeafCellBuffrName,
		   LeafCellInstaName,
		            "sel[1]",
		            "sel[0]",
		                "s0",
		                "s1",
		                "s2", "vdd", "vss", VSS, EOL);

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
