#include  <time.h>
#include  <stdio.h>
#include  <string.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "nand2mask.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "nand2mask.c"                                      */


/* Module internal constants. */
#define        E_NAND2MASK_WIDTH         2
#define        E_NAND2MASK_SLICE         3
#define        E_NAND2MASK_NOVIEW        4
#define        E_NAND2MASK_OPENVBE       5
#define        E_NAND2MASK_MODELNAME     6


/* Names of tristate leaf cells. */
static char *TablLeafCellNand2mask[3] = { "bna2x_fp",
									       "na2x_fp",
										 "n1na2x_fp"};


/* Interface function of module.
   extern       char *Dgn_DispachNand2mask();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorNand2mask();
static       void  Dgn_VhdlNand2mask();
static       void  Dgn_NetlistNand2mask();


/*  Function  :  'Dgn_DispatchNand2mask'                          */
/* -------------------------------------------------------------- */
/*  Interface of controllable NAND2 mask generator.               */

extern char *Dgn_DispatchNand2mask(nand2mask_modelname,
								   nand2mask_width    ,
								   nand2mask_slice    ,
								   nand2mask_xlmask   ,
								   nand2mask_msb0     ,
								   nand2mask_vhdl     ,
								   nand2mask_netlist  ,
								   nand2mask_icon     )
	char *nand2mask_modelname;  /* Model name.                    */
	long  nand2mask_width;      /* Bus wide.                      */
	long  nand2mask_slice;      /* LSB starting slice.            */
	long *nand2mask_xlmask;     /* Mask to apply.                 */
	long  nand2mask_msb0;       /* Zero as MSB index.             */
	long  nand2mask_vhdl;       /* Generation of behavioral view. */
	long  nand2mask_netlist;    /* Generation of netlist view.    */
	long  nand2mask_icon;       /* Generation of icon.            */
{
	static short  DispachNand2mask_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachNand2mask_FirstCall)
	{
		Dgn_ErrorNand2mask();
		DispachNand2mask_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (nand2mask_netlist == FALSE)
		 &&(nand2mask_vhdl    == FALSE)
		 &&(nand2mask_icon    == FALSE) )
		Dut_PrintError( E_NAND2MASK_ERROR, E_NAND2MASK_NOVIEW);

	/* Check the bus wide. */
	if ((nand2mask_width < 1) || (nand2mask_width > DP_WIDTH))
		Dut_PrintError( E_NAND2MASK_ERROR, E_NAND2MASK_WIDTH, nand2mask_width);

	/* Check the LSB starting slice. */
	if ((nand2mask_slice < 0) || ( nand2mask_slice
							      +nand2mask_width > DP_WIDTH))
		Dut_PrintError( E_NAND2MASK_ERROR, E_NAND2MASK_SLICE, nand2mask_slice);

	/* Check the model name. */
	if (nand2mask_modelname == (char*)NULL)
#ifndef  CRUNCH_FILENAME
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "nand2m%s_%ldx%ldx%ld%c_cl"   ,
					    Dut_XlToStr(nand2mask_xlmask,
									       DP_WIDTH ,
							  C_SUPP_BaseHexadecimal)            ,
					                       DP_WIDTH              ,
					                nand2mask_width              ,
					                nand2mask_slice              ,
					              ((nand2mask_msb0 ) ? 'm' : 'l'));
#else
		/* Build the crunched model name. */
		strcpy( ModelName, Dut_MakeCrunchModelName(
									    "ma"       ,
									 8L,   DP_WIDTH-1,
									 8L, nand2mask_width-1,
									 8L, nand2mask_slice  ,
									 8L,         1L  ,
									 0L, nand2mask_msb0   ));
#endif
	else
		strcpy( ModelName, nand2mask_modelname);

	/* Generation of behaviorial view. */
	if (nand2mask_vhdl)
		Dgn_VhdlNand2mask( ModelName       ,
						   nand2mask_width ,
						   nand2mask_slice ,
						   nand2mask_xlmask,
						   nand2mask_msb0  );

	/* Generation of netlist view. */
	if (nand2mask_netlist)
		Dgn_NetlistNand2mask( ModelName       ,
							  nand2mask_width ,
							  nand2mask_slice ,
							  nand2mask_xlmask,
							  nand2mask_msb0  );

	/* Generation of icon. */
	/* if (nand2mask_icon) */
	
	/* Return name of the generated model. */
	return((nand2mask_modelname == (char*)NULL) ?           ModelName 
		                                        : nand2mask_modelname);
}


/*  Function  :  'Dgn_ErrorNand2mask'                             */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorNand2mask()
{
	Dut_AddError( E_NAND2MASK_ERROR, E_NAND2MASK_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_NAND2MASK_ERROR, E_NAND2MASK_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_NAND2MASK_ERROR, E_NAND2MASK_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_NAND2MASK_ERROR, E_NAND2MASK_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_NAND2MASK_ERROR, E_NAND2MASK_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListNand2mask'                           */
/* -------------------------------------------------------------- */
/*  Generation of NAND2 cotrollable mask behavioral view.         */

static void  Dgn_VhdlNand2mask(nand2mask_modelname,
							   nand2mask_width    ,
							   nand2mask_slice    ,
							   nand2mask_xlmask   ,
							   nand2mask_msb0     )
	char *nand2mask_modelname;  /* Model name.         */
	long  nand2mask_width;      /* Bus width.          */
	long  nand2mask_slice;      /* LSB starting slice. */
	long *nand2mask_xlmask;     /* Mask to apply.      */
	long  nand2mask_msb0;       /* Zero as MSB index.  */
{
	FILE *Nand2maskFile;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Nand2maskFile=mbkfopen(nand2mask_modelname,
						            "vbe",
						       WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_NAND2MASK_ERROR, E_NAND2MASK_OPENVBE,
					                  nand2mask_modelname);


	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Nand2maskFile, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Nand2maskFile, 
"-- #                                                                 #\n"
			);
	fprintf( Nand2maskFile,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Nand2maskFile,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Nand2maskFile,
"-- #                                                                 #\n"
			);
    fprintf( Nand2maskFile,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Nand2maskFile,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Nand2maskFile, 
"-- # *************************************************************** #\n"
			);
	fprintf( Nand2maskFile,
"-- #                                                                 #\n"
			);
    fprintf( Nand2maskFile,
"-- # Generator : \"nand2maskg\" (non-inverter tristate)                #\n"
			);
	fprintf( Nand2maskFile, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	fprintf( Nand2maskFile, 
"-- # File  :                                                         #\n"
			);
	sprintf(  TmpString, "\"%s.vbe\"."        , nand2mask_modelname);
    fprintf( Nand2maskFile  , "-- # %62s  #\n",           TmpString);
	fprintf( Nand2maskFile,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Nand2maskFile  , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"nand2maskg %ld %ld %ld",
			                     DP_WIDTH          ,
			                   nand2mask_width     ,
			                   nand2mask_slice     );
    fprintf( Nand2maskFile  , "-- # Generator Call : %-45s  #\n", TmpString);
	sprintf(  TmpString, "%s%s -vhdl"                                 ,
			       Dut_XlToStr(nand2mask_xlmask,
							   nand2mask_width ,
						 C_SUPP_BaseHexadecimal)                      ,
			                  (nand2mask_msb0     ) ? " -msb0"  : ""  );
    fprintf( Nand2maskFile  , "-- # %62s  #\n",    TmpString);
	sprintf(     TmpString  , "-o %s\"", nand2mask_modelname);
    fprintf( Nand2maskFile  , "-- # %62s  #\n",    TmpString);

	fprintf( Nand2maskFile, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "nand2mask_msb0")            */
	if (!nand2mask_msb0)
		sprintf( BusWide, "(%ld downto %ld)", nand2mask_width-1
				                             +nand2mask_slice  ,
				                              nand2mask_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-nand2mask_slice
				                                      -nand2mask_width  ,
				                              DP_WIDTH-nand2mask_slice-1);


	/* *** Interface description. *** */
	fprintf( Nand2maskFile, "\nENTITY %s IS\n\tPORT(\n", nand2mask_modelname);

	/* Controls terminals. */
	fprintf( Nand2maskFile, "%14s :  in BIT;\n",  "msk");

	/* Terminals of data nand2masks. */
	fprintf( Nand2maskFile, "%14s :  in BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Nand2maskFile, "%14s : out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Nand2maskFile, "%14s :  in BIT;\n", "vdd");
	fprintf( Nand2maskFile, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Nand2maskFile, "\t);\nEND %s;\n\n\n", nand2mask_modelname);


	/* *** Behavioral view *** */
	fprintf( Nand2maskFile, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            nand2mask_modelname);
	fprintf( Nand2maskFile, "BEGIN\n\n");

	fprintf( Nand2maskFile,     "\tWITH msk SELECT\n"              );
	fprintf( Nand2maskFile,   "\t\to%s <=\n"                      , BusWide);
	fprintf( Nand2maskFile, "\t\t\tnot  i0%s         WHEN '0',\n"   , BusWide);
	fprintf( Nand2maskFile, "\t\t\tnot (i0%s and %s) WHEN '1';\n\n" , BusWide,
			               Dut_XlToStrVHDL(nand2mask_xlmask,nand2mask_width));

	/* Check supplies connections. */
	fprintf( Nand2maskFile, "\t-- ****************>");
	fprintf( Nand2maskFile, " Supplies Check "      );
	fprintf( Nand2maskFile, "<****************\n\n" );

	fprintf( Nand2maskFile, "\tASSERT ((vdd = '1') and (vss = '0'))\n");
	fprintf( Nand2maskFile, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              nand2mask_modelname);
	fprintf( Nand2maskFile, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Nand2maskFile, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Nand2maskFile);
}


/*  Fonction  :  'Dgn_NetListNand2mask'                           */
/* -------------------------------------------------------------- */
/*  Generation of NAND2 cotrollable mask Netlist.                 */

static void  Dgn_NetlistNand2mask(nand2mask_modelname,
								  nand2mask_width    ,
								  nand2mask_slice    ,
								  nand2mask_xlmask   ,
								  nand2mask_msb0     )
	char *nand2mask_modelname;  /* Model name.         */
	long  nand2mask_width;      /* Bus wide.           */
	long  nand2mask_slice;      /* LSB starting slice. */
	long *nand2mask_xlmask;     /* Mask to apply.      */
	long  nand2mask_msb0;       /* Zero as MSB index.  */
{
	long       BitNumber;
	long       MskValue;
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char *LeafCellBuffrName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Model names (slices and buffer). */
	LeafCellBuffrName = TablLeafCellNand2mask[0];

	/* Creation of the new logical figure. */
	DEF_LOFIG( nand2mask_modelname);

	/* Name of data bus terminals. */
	if (!nand2mask_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", nand2mask_width-1
				                         +nand2mask_slice  ,
				                          nand2mask_slice  );
		sprintf( BusNameO , "o[%ld:%ld]", nand2mask_width-1
				                         +nand2mask_slice  ,
				                          nand2mask_slice  );
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-nand2mask_slice
				                                  -nand2mask_width  ,
				                          DP_WIDTH-nand2mask_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-nand2mask_slice
				                                  -nand2mask_width  ,
				                          DP_WIDTH-nand2mask_slice-1);
	}
 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON(     "msk", IN   ,      "msk");

	/* **** Data bus terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Supply terminals **** */
	LOCON(     "vdd", INOUT,      "vdd"); 
	LOCON(     "vss", INOUT,      "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=nand2mask_slice,BitNumber=nand2mask_width; 
		 LeafCellIndex<nand2mask_slice          +nand2mask_width;
		 LeafCellIndex++              ,BitNumber--              )
	{
	    /* Select the correct mask cells. */
	    LeafCellSliceName = TablLeafCellNand2mask[
			Dut_XlGetBit(     nand2mask_xlmask,
		        LeafCellIndex-nand2mask_slice )+1];

		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "nand2mask_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!nand2mask_msb0) BusIndex =              LeafCellIndex;
		else                 BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
				       PinNameI0,
				      "sign_msk",
				       PinNameO , "vdd", "vss", EOL);
	}


	/* Creation of buffer instance. */

	/* Build buffer instance name. */
	sprintf( LeafCellInstaName, "bnand2mask_%ld", DP_WIDTH);

	/* Buffer instanciation. */
	LOINS( LeafCellBuffrName,
		   LeafCellInstaName,
		               "msk",
		          "sign_msk", "vdd", "vss", VSS, EOL);

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
