#include  <time.h>
#include  <stdio.h>
#include  <string.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "nor2mask.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "nor2mask.c"                                       */


/* Module internal constants. */
#define        E_NOR2MASK_WIDTH         2
#define        E_NOR2MASK_SLICE         3
#define        E_NOR2MASK_NOVIEW        4
#define        E_NOR2MASK_OPENVBE       5
#define        E_NOR2MASK_MODELNAME     6


/* Names of tristate leaf cells. */
static char *TablLeafCellNor2mask[3] = { "bno2x_fp",
									      "no2x_fp",
										"n1no2x_fp"};


/* Interface function of module.
   extern       char *Dgn_DispachNor2mask();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorNor2mask();
static       void  Dgn_VhdlNor2mask();
static       void  Dgn_NetlistNor2mask();


/*  Function  :  'Dgn_DispatchNor2mask'                           */
/* -------------------------------------------------------------- */
/*  Interface of controllable NOR2 mask generator.                */

extern char *Dgn_DispatchNor2mask(nor2mask_modelname,
								  nor2mask_width    ,
								  nor2mask_slice    ,
								  nor2mask_xlmask   ,
								  nor2mask_msb0     ,
								  nor2mask_vhdl     ,
								  nor2mask_netlist  ,
								  nor2mask_icon     )
	char *nor2mask_modelname;  /* Model name.                    */
	long  nor2mask_width;      /* Bus wide.                      */
	long  nor2mask_slice;      /* LSB starting slice.            */
	long *nor2mask_xlmask;     /* Mask to apply.                 */
	long  nor2mask_msb0;       /* Zero as MSB index.             */
	long  nor2mask_vhdl;       /* Generation of behavioral view. */
	long  nor2mask_netlist;    /* Generation of netlist view.    */
	long  nor2mask_icon;       /* Generation of icon.            */
{
	static short  DispachNor2mask_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachNor2mask_FirstCall)
	{
		Dgn_ErrorNor2mask();
		DispachNor2mask_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (nor2mask_netlist == FALSE)
		 &&(nor2mask_vhdl    == FALSE)
		 &&(nor2mask_icon    == FALSE) )
		Dut_PrintError( E_NOR2MASK_ERROR, E_NOR2MASK_NOVIEW);

	/* Check the bus wide. */
	if ((nor2mask_width < 1) || (nor2mask_width > DP_WIDTH))
		Dut_PrintError( E_NOR2MASK_ERROR, E_NOR2MASK_WIDTH, nor2mask_width);

	/* Check the LSB starting slice. */
	if ((nor2mask_slice < 0) || ( nor2mask_slice
							     +nor2mask_width > DP_WIDTH))
		Dut_PrintError( E_NOR2MASK_ERROR, E_NOR2MASK_SLICE, nor2mask_slice);

	/* Check the model name. */
	if (nor2mask_modelname == (char*)NULL)
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "nor2m%s_%ldx%ldx%ld%c_cl"   ,
					    Dut_XlToStr(nor2mask_xlmask,
									      DP_WIDTH ,
							 C_SUPP_BaseHexadecimal)            ,
					                      DP_WIDTH              ,
					                nor2mask_width              ,
					                nor2mask_slice              ,
					              ((nor2mask_msb0 ) ? 'm' : 'l'));
	else
		strcpy( ModelName, nor2mask_modelname);

	/* Generation of behaviorial view. */
	if (nor2mask_vhdl)
		Dgn_VhdlNor2mask( ModelName      ,
						  nor2mask_width ,
						  nor2mask_slice ,
						  nor2mask_xlmask,
						  nor2mask_msb0  );

	/* Generation of netlist view. */
	if (nor2mask_netlist)
		Dgn_NetlistNor2mask( ModelName      ,
							 nor2mask_width ,
							 nor2mask_slice ,
							 nor2mask_xlmask,
							 nor2mask_msb0  );

	/* Generation of icon. */
	/* if (nor2mask_icon) */
	
	/* Return name of the generated model. */
	return((nor2mask_modelname == (char*)NULL) ?          ModelName 
		                                       : nor2mask_modelname);
}


/*  Function  :  'Dgn_ErrorNor2mask'                              */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorNor2mask()
{
	Dut_AddError( E_NOR2MASK_ERROR, E_NOR2MASK_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_NOR2MASK_ERROR, E_NOR2MASK_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_NOR2MASK_ERROR, E_NOR2MASK_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_NOR2MASK_ERROR, E_NOR2MASK_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_NOR2MASK_ERROR, E_NOR2MASK_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListNor2mask'                           */
/* -------------------------------------------------------------- */
/*  Generation of NOR2 cotrollable mask behavioral view.         */

static void  Dgn_VhdlNor2mask(nor2mask_modelname,
							  nor2mask_width    ,
							  nor2mask_slice    ,
							  nor2mask_xlmask   ,
							  nor2mask_msb0     )
	char *nor2mask_modelname;  /* Model name.         */
	long  nor2mask_width;      /* Bus width.          */
	long  nor2mask_slice;      /* LSB starting slice. */
	long *nor2mask_xlmask;     /* Mask to apply.      */
	long  nor2mask_msb0;       /* Zero as MSB index.  */
{
	FILE *Nor2maskFile;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Nor2maskFile=mbkfopen(nor2mask_modelname,
							                "vbe",
						               WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_NOR2MASK_ERROR, E_NOR2MASK_OPENVBE,
					                      nor2mask_modelname);


	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Nor2maskFile, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Nor2maskFile, 
"-- #                                                                 #\n"
			);
	fprintf( Nor2maskFile,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Nor2maskFile,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Nor2maskFile,
"-- #                                                                 #\n"
			);
    fprintf( Nor2maskFile,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Nor2maskFile,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Nor2maskFile, 
"-- # *************************************************************** #\n"
			);
	fprintf( Nor2maskFile,
"-- #                                                                 #\n"
			);
    fprintf( Nor2maskFile,
"-- # Generator : \"nor2maskg\" (non-inverter tristate)                 #\n"
			);
	fprintf( Nor2maskFile, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	fprintf( Nor2maskFile, 
"-- # File  :                                                         #\n"
			);
	sprintf(      TmpString, "\"%s.vbe\"."   , nor2mask_modelname);
    fprintf( Nor2maskFile  , "-- # %62s  #\n",          TmpString);
	fprintf( Nor2maskFile  ,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Nor2maskFile, "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"nor2maskg %ld %ld %ld",
			                      DP_WIDTH        ,
			                nor2mask_width        ,
			                nor2mask_slice        );
    fprintf( Nor2maskFile  , "-- # Generator Call : %-45s  #\n", TmpString);
	sprintf(  TmpString, "%s%s -vhdl"                        ,
			  Dut_XlToStr(nor2mask_xlmask,
						  nor2mask_width ,
				   C_SUPP_BaseHexadecimal)                   ,
			             (nor2mask_msb0  ) ? " -msb0"  : ""  );
    fprintf( Nor2maskFile  , "-- # %62s  #\n",          TmpString);
	sprintf(      TmpString, "-o %s\""       , nor2mask_modelname);
    fprintf( Nor2maskFile  , "-- # %62s  #\n",          TmpString);

	fprintf( Nor2maskFile, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "nor2mask_msb0")            */
	if (!nor2mask_msb0)
		sprintf( BusWide, "(%ld downto %ld)", nor2mask_width-1
				                             +nor2mask_slice  ,
				                              nor2mask_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-nor2mask_slice
				                                      -nor2mask_width  ,
				                              DP_WIDTH-nor2mask_slice-1);


	/* *** Interface description. *** */
	fprintf( Nor2maskFile, "\nENTITY %s IS\n\tPORT(\n", nor2mask_modelname);

	/* Controls terminals. */
	fprintf( Nor2maskFile, "%14s :  in BIT;\n",  "msk");

	/* Terminals of data nor2masks. */
	fprintf( Nor2maskFile, "%14s :  in BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Nor2maskFile, "%14s : out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Nor2maskFile, "%14s :  in BIT;\n", "vdd");
	fprintf( Nor2maskFile, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Nor2maskFile, "\t);\nEND %s;\n\n\n", nor2mask_modelname);


	/* *** Behavioral view *** */
	fprintf( Nor2maskFile, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            nor2mask_modelname);
	fprintf( Nor2maskFile, "BEGIN\n\n");

	fprintf( Nor2maskFile,     "\tWITH msk SELECT\n"              );
	fprintf( Nor2maskFile,   "\t\to%s <=\n"                       , BusWide);
	fprintf( Nor2maskFile, "\t\t\tnot  i0%s        WHEN '0',\n"   , BusWide);
	fprintf( Nor2maskFile, "\t\t\tnot (i0%s or %s) WHEN '1';\n\n" , BusWide,
			                 Dut_XlToStrVHDL(nor2mask_xlmask,nor2mask_width));

	/* Check supplies connections. */
	fprintf( Nor2maskFile, "\t-- ****************>");
	fprintf( Nor2maskFile, " Supplies Check "      );
	fprintf( Nor2maskFile, "<****************\n\n" );

	fprintf( Nor2maskFile, "\tASSERT ((vdd = '1') and (vss = '0'))\n");
	fprintf( Nor2maskFile, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              nor2mask_modelname);
	fprintf( Nor2maskFile, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Nor2maskFile, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Nor2maskFile);
}


/*  Fonction  :  'Dgn_NetListNor2mask'                           */
/* -------------------------------------------------------------- */
/*  Generation of NOR2 cotrollable mask Netlist.                 */

static void  Dgn_NetlistNor2mask(nor2mask_modelname,
								 nor2mask_width    ,
								 nor2mask_slice    ,
								 nor2mask_xlmask   ,
								 nor2mask_msb0     )
	char *nor2mask_modelname;  /* Model name.         */
	long  nor2mask_width;      /* Bus wide.           */
	long  nor2mask_slice;      /* LSB starting slice. */
	long *nor2mask_xlmask;     /* Mask to apply.      */
	long  nor2mask_msb0;       /* Zero as MSB index.  */
{
	long       BitNumber;
	long       MskValue;
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char *LeafCellBuffrName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Model names (slices and buffer). */
	LeafCellBuffrName = TablLeafCellNor2mask[0];

	/* Creation of the new logical figure. */
	DEF_LOFIG( nor2mask_modelname);

	/* Name of data bus terminals. */
	if (!nor2mask_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", nor2mask_width-1
				                         +nor2mask_slice  ,
				                          nor2mask_slice  );
		sprintf( BusNameO , "o[%ld:%ld]", nor2mask_width-1
				                         +nor2mask_slice  ,
				                          nor2mask_slice  );
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-nor2mask_slice
				                                  -nor2mask_width  ,
				                          DP_WIDTH-nor2mask_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-nor2mask_slice
				                                  -nor2mask_width  ,
				                          DP_WIDTH-nor2mask_slice-1);
	}
 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON(     "msk", IN   ,      "msk");

	/* **** Data bus terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Supply terminals **** */
	LOCON(     "vdd", INOUT,      "vdd"); 
	LOCON(     "vss", INOUT,      "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=nor2mask_slice,BitNumber=nor2mask_width; 
		 LeafCellIndex<nor2mask_slice          +nor2mask_width;
		 LeafCellIndex++             ,BitNumber--             )
	{
	    /* Select the correct mask cells. */
	    LeafCellSliceName = TablLeafCellNor2mask[
			2-Dut_XlGetBit(     nor2mask_xlmask,
		          LeafCellIndex-nor2mask_slice )];

		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "nor2mask_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!nor2mask_msb0) BusIndex =              LeafCellIndex;
		else                BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
				       PinNameI0,
				      "sign_msk",
				       PinNameO , "vdd", "vss", EOL);
	}


	/* Creation of buffer instance. */

	/* Build buffer instance name. */
	sprintf( LeafCellInstaName, "bnor2mask_%ld", DP_WIDTH);

	/* Buffer instanciation. */
	LOINS( LeafCellBuffrName,
		   LeafCellInstaName,
		               "msk",
		          "sign_msk", "vdd", "vss", VSS, EOL);

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
