#include  <stdio.h>
#include  <string.h>
#include  <time.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "or3.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "or3.c"                                            */


/* Module internal constants. */
#define        E_OR3_WIDTH         2
#define        E_OR3_SLICE         3
#define        E_OR3_DRIVE         4
#define        E_OR3_NOVIEW        5
#define        E_OR3_OPENVBE       6
#define        E_OR3_MODELNAME     7


/* Names of or3 leaf cells. */
static char *TablLeafCellOr3[2] = {  "o3_fp",
									"o3p_fp" };


/* Interface function of module.
   extern       char *Dgn_DispachOr3();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorOr3();
static       void  Dgn_VhdlOr3();
static       void  Dgn_NetlistOr3();


/*  Function  :  'Dgn_DispatchOr3'                               */
/* -------------------------------------------------------------- */
/*  Interface of or3 generator.                                  */

extern char *Dgn_DispatchOr3(or3_modelname,
							  or3_width    ,
							  or3_slice    ,
							  or3_drive    ,
							  or3_msb0     ,
							  or3_vhdl     ,
							  or3_netlist  ,
							  or3_icon     )
	char *or3_modelname;  /* Model name.                    */
	long  or3_width;      /* Bus wide.                      */
	long  or3_slice;      /* LSB starting slice.            */
	long  or3_drive;      /* Output drive capacity.         */
	long  or3_msb0;       /* Zero as MSB index.             */
	long  or3_vhdl;       /* Generation of behavioral view. */
	long  or3_netlist;    /* Generation of netlist view.    */
	long  or3_icon;       /* Generation of icon.            */
{
	static short  DispachOr3_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachOr3_FirstCall)
	{
		Dgn_ErrorOr3();
		DispachOr3_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (or3_netlist == FALSE)
		 &&(or3_vhdl    == FALSE)
		 &&(or3_icon    == FALSE) )
		Dut_PrintError( E_OR3_ERROR, E_OR3_NOVIEW);

	/* Check the bus wide. */
	if ((or3_width < 1) || (or3_width > DP_WIDTH))
		Dut_PrintError( E_OR3_ERROR, E_OR3_WIDTH, or3_width);

	/* Check the output drive capacity. */
	if ((or3_drive < 1) || (or3_drive > C_OR3_NBDRIVE))
		Dut_PrintError( E_OR3_ERROR, E_OR3_DRIVE, or3_drive);

	/* Check the LSB starting slice. */
	if ((or3_slice < 0) || ( or3_slice
							 +or3_width > DP_WIDTH))
		Dut_PrintError( E_OR3_ERROR, E_OR3_SLICE, or3_slice);

	/* Check the model name. */
	if (or3_modelname == (char*)NULL)
#ifndef  CRUNCH_FILENAME
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "or3_%ldx%ldx%ldx%ld%c_cl",
					                   DP_WIDTH              ,
					                 or3_width              ,
					                 or3_slice              ,
					                 or3_drive              ,
					               ((or3_msb0 ) ? 'm' : 'l'));
#else
		/* Build the crunched model name. */
		strcpy( ModelName, Dut_MakeCrunchModelName(
									    "od"       ,
								   8L,   DP_WIDTH-1,
								   8L, or3_width-1,
								   8L, or3_slice  ,
								   8L, or3_drive  ,
								   0L, or3_msb0   ));
#endif
	else
		strcpy( ModelName, or3_modelname);


	/* Generation of behaviorial view. */
	if (or3_vhdl)
		Dgn_VhdlOr3( ModelName,
			   		  or3_width,
					  or3_slice,
			   		  or3_drive,
			   		  or3_msb0 );

	/* Generation of netlist view. */
	if (or3_netlist)
		Dgn_NetlistOr3( ModelName,
						 or3_width,
						 or3_slice,
					     or3_drive,
						 or3_msb0 );

	/* Generation of icone. */
	/* if (or3_icon) */
	
	/* Return name of the generated model. */
	return((or3_modelname == (char*)NULL) ?      ModelName 
		                                   : or3_modelname);
}


/*  Function  :  'Dgn_ErrorOr3'                                  */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorOr3()
{
	Dut_AddError( E_OR3_ERROR, E_OR3_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_OR3_ERROR, E_OR3_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_OR3_ERROR, E_OR3_DRIVE,
				  "invalid output drive : %ld.");
	Dut_AddError( E_OR3_ERROR, E_OR3_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_OR3_ERROR, E_OR3_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_OR3_ERROR, E_OR3_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListOr3'                                */
/* -------------------------------------------------------------- */
/*  Generation of or3 behavioral view.                           */

static void  Dgn_VhdlOr3(or3_modelname,
					      or3_width    ,
					      or3_slice    ,
					      or3_drive    ,
						  or3_msb0     )
	char *or3_modelname;  /* Nom du modele.           */
	long  or3_width;      /* Largeur du bus.          */
	long  or3_slice;      /* LSB starting slice.      */
	long  or3_drive;      /* Output drive power.      */
	long  or3_msb0;       /* Zero comme index du MSB. */
{
	FILE *Or3File;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Or3File=mbkfopen(or3_modelname,
						            "vbe",
						       WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_OR3_ERROR, E_OR3_OPENVBE,
					                  or3_modelname);

	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Or3File, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Or3File, 
"-- #                                                                 #\n"
			);
	fprintf( Or3File,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Or3File,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Or3File,
"-- #                                                                 #\n"
			);
    fprintf( Or3File,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Or3File,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Or3File, 
"-- # *************************************************************** #\n"
			);
	fprintf( Or3File,
"-- #                                                                 #\n"
			);
    fprintf( Or3File,
"-- # Generator : \"or3g\" (or3)                                      #\n"
			);
	fprintf( Or3File, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf( TmpString, "\"%s.vbe\"."           , or3_modelname);
    fprintf( Or3File , "-- # File : %-55s  #\n",     TmpString);
	fprintf( Or3File ,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Or3File , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"or3g %ld %ld %ld %ld%s -vhdl -o %s\"" ,
			                         DP_WIDTH                      ,
			                       or3_width                      ,
			                       or3_slice                      ,
			                       or3_drive                      ,
			                      (or3_msb0     ) ? " -msb0"  : "",
			                       or3_modelname                  );
    fprintf( Or3File , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Or3File, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "or3_msb0")            */
	if (!or3_msb0)
		sprintf( BusWide, "(%ld downto %ld)", or3_width-1
				                             +or3_slice  ,
				                              or3_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-or3_slice
				                                      -or3_width  ,
				                              DP_WIDTH-or3_slice-1);


	/* *** Interface description. *** */
	fprintf( Or3File, "\nENTITY %s IS\n\tPORT(\n", or3_modelname);

	/* Terminals of data buses. */
	fprintf( Or3File, "%14s :in    BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Or3File, "%14s :in    BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( Or3File, "%14s :in    BIT_VECTOR%s;\n", "i2", BusWide);
	fprintf( Or3File, "%14s :inout BIT_VECTOR%s;\n", "no", BusWide);
	fprintf( Or3File, "%14s :  out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Or3File, "%14s :  in BIT;\n", "vdd");
	fprintf( Or3File, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Or3File, "\t);\nEND %s;\n\n\n", or3_modelname);


	/* *** Behavioral view *** */
	fprintf( Or3File, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            or3_modelname);
	fprintf( Or3File, "BEGIN\n\n");
	fprintf( Or3File, "\to%s <= not no%s;\n" , BusWide,
			                                    BusWide);
	fprintf( Or3File, "\tno%s <= not(i0%s or i1%s or i2%s);\n\n" , BusWide,
			                                                BusWide,
			                                                BusWide,
			                                                BusWide);

	/* Check supplies connections. */
	fprintf( Or3File, "\t-- ****************>");
	fprintf( Or3File, " Supplies Check "      );
	fprintf( Or3File, "<****************\n\n" );

	fprintf( Or3File, "\tASSERT((vdd = '1') and (vss = '0'))\n");
	fprintf( Or3File, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              or3_modelname);
	fprintf( Or3File, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Or3File, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Or3File);
}


/*  Function  :  'Dgn_NetListOr3'                                */
/* -------------------------------------------------------------- */
/*  Generation of or3erter/buffer netlist view.                  */

static void  Dgn_NetlistOr3(or3_modelname,
						     or3_width    ,
						     or3_slice    ,
							 or3_drive    ,
						     or3_msb0     )
	char *or3_modelname;  /* Model name.            */
	long  or3_width;      /* Bus wide.              */
	long  or3_slice;      /* LSB starting slice.    */
	long  or3_drive;      /* Output drive capacity. */
	long  or3_msb0;       /* Zero as MSB index.     */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameI2[C_SUPP_MBKSZNAME];
	char  BusNameNO[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameI2[C_SUPP_MBKSZNAME];
	char  PinNameNO[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Select name of the leaf cell model. */
	/* (according to output drive power)   */
	LeafCellSliceName = TablLeafCellOr3[or3_drive-1];

	/* Creation of the new logical figure. */
	DEF_LOFIG( or3_modelname);

	/* Name of data bus terminals. */
	if (!or3_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", or3_width-1+or3_slice, or3_slice);
		sprintf( BusNameI1,"i1[%ld:%ld]", or3_width-1+or3_slice, or3_slice);
		sprintf( BusNameI2,"i2[%ld:%ld]", or3_width-1+or3_slice, or3_slice);
		sprintf( BusNameNO,"no[%ld:%ld]", or3_width-1+or3_slice, or3_slice);
		sprintf( BusNameO , "o[%ld:%ld]", or3_width-1+or3_slice, or3_slice);
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-or3_slice
				                                  -or3_width  ,
				                          DP_WIDTH-or3_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-or3_slice
				                                  -or3_width  ,
				                          DP_WIDTH-or3_slice-1);
		sprintf( BusNameI2,"i2[%ld:%ld]", DP_WIDTH-or3_slice
				                                  -or3_width  ,
				                          DP_WIDTH-or3_slice-1);
		sprintf( BusNameNO,"no[%ld:%ld]", DP_WIDTH-or3_slice
				                                  -or3_width  ,
				                          DP_WIDTH-or3_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-or3_slice
				                                  -or3_width  ,
				                          DP_WIDTH-or3_slice-1);
	}

 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameI2, IN   , BusNameI2);
	LOCON( BusNameNO, INOUT, BusNameNO);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Data bus terminals **** */
	LOCON(     "vdd", INOUT,     "vdd"); 
	LOCON(     "vss", INOUT,     "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=or3_slice;
		 LeafCellIndex<or3_slice+or3_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "or3_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!or3_msb0) BusIndex =              LeafCellIndex;
		else            BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameI2, "i2[%ld]", BusIndex);
		sprintf( PinNameNO, "no[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameI2 ,
				      PinNameNO , 		
			          PinNameO  , "vdd", "vss", EOL);
	}

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
