#include  <time.h>
#include  <stdio.h>
#include  <string.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "pdff.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "pdff.c"                                          */


/* Module internal constants. */
#define        E_PDFF_WIDTH         2
#define        E_PDFF_SLICE         3
#define        E_PDFF_NOVIEW        4
#define        E_PDFF_OPENVBE       5
#define        E_PDFF_MODELNAME     6


/* Names of tristate leaf cells. */
static char *TablLeafCellPdff[2] = { "pdffb3_fp",
									 "pdffa3_fp"};


/* Interface function of module.
   extern       char *Dgn_DispachPdff();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorPdff();
static       void  Dgn_VhdlPdff();
static       void  Dgn_NetlistPdff();


/*  Function  :  'Dgn_DispatchPdff'                              */
/* -------------------------------------------------------------- */
/*  Interface scannable D flip-flops generator.                   */

extern char *Dgn_DispatchPdff(pdff_modelname,
							  pdff_width    ,
							  pdff_slice    ,
							  pdff_msb0     ,
							  pdff_vhdl     ,
							  pdff_netlist  ,
							  pdff_icon     )
	char *pdff_modelname;  /* Model name.                    */
	long  pdff_width;      /* Bus wide.                      */
	long  pdff_slice;      /* LSB starting slice.            */
	long  pdff_msb0;       /* Zero as MSB index.             */
	long  pdff_vhdl;       /* Generation of behavioral view. */
	long  pdff_netlist;    /* Generation of netlist view.    */
	long  pdff_icon;       /* Generation of icon.            */
{
	static short  DispachPdff_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachPdff_FirstCall)
	{
		Dgn_ErrorPdff();
		DispachPdff_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (pdff_netlist == FALSE)
		 &&(pdff_vhdl    == FALSE)
		 &&(pdff_icon    == FALSE) )
		Dut_PrintError( E_PDFF_ERROR, E_PDFF_NOVIEW);

	/* Check the bus wide. */
	if ((pdff_width < 2) || (pdff_width > DP_WIDTH))
		Dut_PrintError( E_PDFF_ERROR, E_PDFF_WIDTH, pdff_width);

	/* Check the LSB starting slice. */
	if ((pdff_slice < 0) || ( pdff_slice
							 +pdff_width > DP_WIDTH))
		Dut_PrintError( E_PDFF_ERROR, E_PDFF_SLICE, pdff_slice);

	/* Check the model name. */
	if (pdff_modelname == (char*)NULL)
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "pdff_%ldx%ldx%ld%c_cl"  ,
					                   DP_WIDTH              ,
					                pdff_width              ,
					                pdff_slice              ,
					              ((pdff_msb0 ) ? 'm' : 'l'));
	else
		strcpy( ModelName, pdff_modelname);

	/* Generation of behaviorial view. */
	if (pdff_vhdl)
		Dgn_VhdlPdff(ModelName  ,
					 pdff_width,
					 pdff_slice,
					 pdff_msb0 );

	/* Generation of netlist view. */
	if (pdff_netlist)
		Dgn_NetlistPdff(ModelName  ,
						pdff_width,
						pdff_slice,
						pdff_msb0 );

	/* Generation of icon. */
	/* if (pdff_icon) */
	
	/* Return name of the generated model. */
	return((pdff_modelname == (char*)NULL) ?      ModelName 
		                                   : pdff_modelname);
}


/*  Function  :  'Dgn_ErrorPdff'                                  */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorPdff()
{
	Dut_AddError( E_PDFF_ERROR, E_PDFF_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_PDFF_ERROR, E_PDFF_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_PDFF_ERROR, E_PDFF_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_PDFF_ERROR, E_PDFF_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_PDFF_ERROR, E_PDFF_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_VhdlPdff'                                   */
/* -------------------------------------------------------------- */
/*  Generation scannable D flip-flops behavioral view.            */

static void  Dgn_VhdlPdff(pdff_modelname,
						  pdff_width    ,
						  pdff_slice    ,
						  pdff_msb0     )
	char *pdff_modelname;  /* Model name.         */
	long  pdff_width;      /* Bus width.          */
	long  pdff_slice;      /* LSB starting slice. */
	long  pdff_msb0;       /* Zero as MSB index.  */
{
	FILE *PdffFile;
	long  ConIndex;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[  256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((PdffFile=mbkfopen(pdff_modelname,
						              "vbe",
						         WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_PDFF_ERROR, E_PDFF_OPENVBE,
					                   pdff_modelname);


	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( PdffFile, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( PdffFile, 
"-- #                                                                 #\n"
			);
	fprintf( PdffFile,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( PdffFile,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( PdffFile,
"-- #                                                                 #\n"
			);
    fprintf( PdffFile,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( PdffFile,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( PdffFile, 
"-- # *************************************************************** #\n"
			);
	fprintf( PdffFile,
"-- #                                                                 #\n"
			);
    fprintf( PdffFile,
"-- # Generator : \"pdffg\" (Scannable D Flip-Flop)                    #\n"
			);
	fprintf( PdffFile, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf(  TmpString, "\"%s.vbe\"."            ,pdff_modelname);
    fprintf( PdffFile  , "-- # File : %-55s  #\n",      TmpString);
	fprintf( PdffFile,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( PdffFile  , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"pdffg %ld %ld %ld%s -vhdl -o %s\"" ,
			                      DP_WIDTH                      ,
			                   pdff_width                      ,
			                   pdff_slice                      ,
			                  (pdff_msb0     ) ? " -msb0"  : "",
			                   pdff_modelname                  );
    fprintf( PdffFile  , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( PdffFile, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "pdff_msb0")            */
	if (!pdff_msb0)
	{
		sprintf( BusWide ,  "(%ld downto %ld)", pdff_width-1
				                               +pdff_slice  ,
				                                pdff_slice  );
	}
	else
	{
		sprintf( BusWide  ,   "(%ld to %ld)", DP_WIDTH-pdff_slice
				                                      -pdff_width  ,
				                              DP_WIDTH-pdff_slice-1);
	}


	/* *** Interface description. *** */
	fprintf( PdffFile, "\nENTITY %s IS\n\tPORT(\n", pdff_modelname);

	/* Controls terminals. */
	fprintf( PdffFile, "%14s : in    BIT;\n",   "wen");
	fprintf( PdffFile, "%14s : in    BIT;\n",    "ck");

	/* Terminals of data pdffs. */
	fprintf( PdffFile, "%14s : in    BIT_VECTOR%s;\n",  "d", BusWide  );
	fprintf( PdffFile, "%14s :   out BIT_VECTOR%s;\n",  "q", BusWide );
	fprintf( PdffFile, "%14s :   out BIT_VECTOR%s;\n", "nq", BusWide  );

	/* Supplies terminals. */
	fprintf( PdffFile, "%14s :  in BIT;\n", "vdd");
	fprintf( PdffFile, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( PdffFile, "\t);\nEND %s;\n\n\n", pdff_modelname);


	/* *** Behavioral view *** */
	fprintf( PdffFile, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            pdff_modelname);
	
	/* Internal signals declaration. */
	fprintf( PdffFile, "\tSIGNAL den : BIT;\n");
	fprintf( PdffFile, "\tSIGNAL cks : BIT;\n");
	/* Internal register declaration. */
	for( BitIndex=pdff_slice; 
		 BitIndex<pdff_slice+pdff_width; BitIndex++)
	{
		fprintf( PdffFile, "\tSIGNAL inst_pdff_%ld_memm_s", BitIndex);
	    fprintf( PdffFile, ": REG_BIT REGISTER;\n"      );
		fprintf( PdffFile, "\tSIGNAL inst_pdff_%ld_mems_s", BitIndex);
	    fprintf( PdffFile, ": REG_BIT REGISTER;\n"      );
	}

	fprintf( PdffFile, "\nBEGIN\n\n");

	/* *** Compute internal signals *** */
	fprintf( PdffFile, "\tden <=     ck and wen;\n");
	fprintf( PdffFile, "\tcks <= not ck;\n\n");

	/* *** Write D flip-flops when in normal mode *** */
	fprintf( PdffFile, "\twd:BLOCK(den) BEGIN\n");
	/* Loop on bus wide. */
	for( BitIndex=pdff_slice; 
		 BitIndex<pdff_slice+pdff_width; BitIndex++)
	{
		if (!pdff_msb0) ConIndex =              BitIndex;
		else             ConIndex = DP_WIDTH -1 -BitIndex;

		fprintf( PdffFile, "\t\tinst_pdff_%ld_memm_s <= GUARDED ", ConIndex);
		fprintf( PdffFile,              "d(%ld);\n"               , ConIndex);
	}
	fprintf( PdffFile, "\tEND BLOCK wd;\n\n");

	/* *** Write slave registers *** */
	fprintf( PdffFile, "\tws:BLOCK(cks) BEGIN\n");
	for( BitIndex=pdff_slice; 
		 BitIndex<pdff_slice+pdff_width; BitIndex++)
	{
		fprintf( PdffFile, "\t\tinst_pdff_%ld_mems_s <= GUARDED ", BitIndex);
		fprintf( PdffFile,     "inst_pdff_%ld_memm_s;\n"         , BitIndex);
	}
	fprintf( PdffFile, "\tEND BLOCK ws;\n\n");

	/* *** Affect the outputs. *** */

	/* Loop on bus wide. */
	for( BitIndex=pdff_slice; 
		 BitIndex<pdff_slice+pdff_width; BitIndex++)
	{
		if (!pdff_msb0) ConIndex =              BitIndex;
		else             ConIndex = DP_WIDTH -1 -BitIndex;

		fprintf( PdffFile, "\t q(%ld) <=     inst_pdff_%ld_mems_s;\n",
				                                               ConIndex,
				                                               ConIndex);
		fprintf( PdffFile, "\tnq(%ld) <= not inst_pdff_%ld_mems_s;\n",
				                                               ConIndex,
				                                               ConIndex);
	}


	/* Check supplies connections. */
	fprintf( PdffFile, "\n\t-- ****************>");
	fprintf( PdffFile,   " Supplies Check "      );
	fprintf( PdffFile,   "<****************\n\n" );

	fprintf( PdffFile, "\tASSERT ((vdd = '1') and (vss = '0'))\n");
	fprintf( PdffFile, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              pdff_modelname);
	fprintf( PdffFile, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( PdffFile, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( PdffFile);
}


/*  Fonction  :  'Dgn_NetListPdff'                                */
/* -------------------------------------------------------------- */
/*  Generation of Scannable D Flip-Flop Netlist.                  */

static void  Dgn_NetlistPdff(pdff_modelname,
							 pdff_width    ,
							 pdff_slice    ,
							 pdff_msb0     )
	char *pdff_modelname;  /* Model name.         */
	long  pdff_width;      /* Bus wide.           */
	long  pdff_slice;      /* LSB starting slice. */
	long  pdff_msb0;       /* Zero as MSB index.  */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char *LeafCellBuffrName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameD[  C_SUPP_MBKSZNAME];
	char  BusNameNQ[ C_SUPP_MBKSZNAME];
	char  BusNameQ[  C_SUPP_MBKSZNAME];
	char  PinNameD[  C_SUPP_MBKSZNAME];
	char  PinNameNQ[ C_SUPP_MBKSZNAME];
	char  PinNameQ[  C_SUPP_MBKSZNAME];


	/* Model names (slices and buffer). */

	/* Creation of the new logical figure. */
	DEF_LOFIG( pdff_modelname);

	/* Name of data bus terminals. */
	if (!pdff_msb0)
	{
		sprintf( BusNameD , "d[%ld:%ld]", pdff_width-1
				                         +pdff_slice  , 
				                          pdff_slice  );
		sprintf( BusNameQ , "q[%ld:%ld]", pdff_width-1
				                         +pdff_slice  ,
				                          pdff_slice  );
		sprintf( BusNameNQ,"nq[%ld:%ld]", pdff_width-1
				                         +pdff_slice  ,
				                          pdff_slice  );
	}
	else
	{
		sprintf( BusNameD , "d[%ld:%ld]", DP_WIDTH-pdff_slice
				                                  -pdff_width  ,
				                          DP_WIDTH-pdff_slice-1);
		sprintf( BusNameQ , "q[%ld:%ld]", DP_WIDTH-pdff_slice
				                                  -pdff_width  ,
				                          DP_WIDTH-pdff_slice-1);
		sprintf( BusNameNQ,"nq[%ld:%ld]", DP_WIDTH-pdff_slice
				                                  -pdff_width  ,
				                          DP_WIDTH-pdff_slice-1);
	}
 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON(     "wen", IN   ,      "wen");
	LOCON(      "ck", IN   ,       "ck");

	/* **** Data bus terminals **** */
	LOCON( BusNameD , IN   , BusNameD );
	LOCON( BusNameQ ,   OUT, BusNameQ );
	LOCON( BusNameNQ,   OUT, BusNameNQ);

	/* **** Supply terminals **** */
	LOCON(     "vdd", INOUT,      "vdd"); 
	LOCON(     "vss", INOUT,      "vss"); 


	/* Instance of bottom column. */	
	LeafCellIndex=pdff_slice;

	/* Select leaf cell model name. */
	LeafCellSliceName = TablLeafCellPdff[1];

	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=pdff_slice; 
		 LeafCellIndex<pdff_slice+pdff_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "pdff_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!pdff_msb0) BusIndex =              LeafCellIndex;
		else            BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameD  ,   "d[%ld]", BusIndex  );
		sprintf( PinNameQ  ,   "q[%ld]", BusIndex  );
		sprintf( PinNameNQ ,  "nq[%ld]", BusIndex  );

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			               "den",
			          PinNameD  ,
			          PinNameQ  ,
			          PinNameNQ ,
			               "cks", "vdd", "vss", EOL);
	}


	/* Creation of buffer instance. */

	/* Select buffer model name. */
	LeafCellBuffrName = TablLeafCellPdff[0];
	/* Build buffer instance name. */
	sprintf( LeafCellInstaName, "bpdff_%ld", DP_WIDTH);

	/* Buffer instanciation. */
	LOINS( LeafCellBuffrName,
		   LeafCellInstaName,
		               "wen",
		                "ck",
		               "den",
		               "cks", "vdd", "vss", "vss", EOL);

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
