/*------------------------------------------------------------\
|                                                             |
| Tool    :                     RDS                           |
|                                                             |
| File    :                  RprParse.c                       |
|                                                             |
| Authors :                Jacomme Ludovic                    |
|                                                             |
| Date    :                   01.08.93                        |
|                                                             |
\------------------------------------------------------------*/

/*------------------------------------------------------------\
|                                                             |
|                         Include Files                       |
|                                                             |
\------------------------------------------------------------*/

# include MUT_H
# include MPH_H
# include RDS_H
# include RUT_H
# include RPR_H
# include RTL_H
# include <math.h>
# include <ctype.h>
# include <stdlib.h>
# include <string.h>
# include "rprerror.h"
# include "rprparse.h"

/*------------------------------------------------------------\
|                                                             |
|                           Constants                         |
|                                                             |
\------------------------------------------------------------*/
/*------------------------------------------------------------\
|                                                             |
|                            Types                            |
|                                                             |
\------------------------------------------------------------*/
/*------------------------------------------------------------\
|                                                             |
|                          Variables                          |
|                                                             |
\------------------------------------------------------------*/

  long   RDS_PHYSICAL_GRID;
  long   RDS_LAMBDA;
  long   RDS_LAMBDA2;
  long   RDS_UNIT;

/*------------------------------------------------------------\
|                                                             |
|                       Table variables                       |
|                                                             |
\------------------------------------------------------------*/

  char  RDS_S2R_POST_TREAT_TABLE  [ RDS_MAX_LAYER ][ RDS_S2R_POST_TREAT_FIELD  ];
  long  RDS_S2R_OVERSIZE_TABLE    [ RDS_MAX_LAYER ][ RDS_S2R_OVERSIZE_FIELD    ];
  long  RDS_S2R_RING_WIDTH_TABLE  [ RDS_MAX_LAYER ][ RDS_S2R_RING_WIDTH_FIELD  ];
  long  RDS_S2R_LAYER_WIDTH_TABLE [ RDS_MAX_LAYER ][ RDS_S2R_LAYER_WIDTH_FIELD ];

  long  RDS_SEGMENT_TABLE     [ MBK_MAX_LAYER     ][ RDS_SEGMENT_FIELD   ];
  long  RDS_VIA_TABLE         [ MBK_MAX_VIA       ][ RDS_VIA_FIELD       ];
  long  RDS_CONNECTOR_TABLE   [ MBK_MAX_LAYER     ][ RDS_CONNECTOR_FIELD ];
  long  RDS_REFERENCE_TABLE   [ MBK_MAX_REFERENCE ][ RDS_REFERENCE_FIELD ];

  char *RDS_CIF_LAYER_TABLE   [ RDS_MAX_LAYER ][ RDS_CIF_LAYER_FIELD ];
  char  RDS_GDS_LAYER_TABLE   [ RDS_MAX_LAYER ][ RDS_GDS_LAYER_FIELD ];

  char  RDS_LYNX_GRAPH_TABLE   [ RDS_MAX_LAYER ][ RDS_LYNX_GRAPH_FIELD ];
  float RDS_LYNX_CAPA_TABLE    [ RDS_MAX_LAYER ][ RDS_LYNX_CAPA_FIELD  ];

/*------------------------------------------------------------\
|                                                             |
|                      Keywords variables                     |
|                                                             |
\------------------------------------------------------------*/

  static char  KeywordDefined = 0;

  static char *DefineKeyword;
  static char *TableKeyword;
  static char *EndTableKeyword;
  static char *EndRecordKeyword;
  static char *LambdaKeyword;
  static char *PhysicalGridKeyword;
  static char *SegmentKeyword;
  static char *ViaKeyword;
  static char *ConnectorKeyword;
  static char *ReferenceKeyword;
  static char *CifLayerKeyword;
  static char *GdsLayerKeyword;
  static char *S2RPostTreatKeyword;
  static char *S2ROversizeKeyword;
  static char *S2RRingWidthKeyword;
  static char *S2RLayerWidthKeyword;
  static char *LynxGraphKeyword;
  static char *LynxCapaKeyword;

  static keyword KeywordDefine[ RPR_MAX_KEYWORD ] =
 
  {
    { "all",          RDS_USE_ALL     },
    { "alu1",         ALU1            },
    { "alu2",         ALU2            },
    { "alu3",         ALU3            },
    { "c_x_n",        C_X_N           },
    { "c_x_p",        C_X_P           },
    { "cont_body_n",  CONT_BODY_N     },
    { "cont_body_p",  CONT_BODY_P     },
    { "cont_dif_n",   CONT_DIF_N      },
    { "cont_dif_p",   CONT_DIF_P      },
    { "cont_poly",    CONT_POLY       },
    { "cont_via",     CONT_VIA        },
    { "cont_via2",    CONT_VIA2       },
    { "cont_via3",    CONT_VIA3       },
    { "drc",          RDS_USE_DRC     },
    { "ext",          RDS_USE_EXTRACT },
    { "lcw",          RDS_TRANS_LCW   },
    { "ndif",         NDIF            },
    { "notreat",      S2R_NOTREAT     },
    { "ntie",         NTIE            },
    { "ntrans",       NTRANS          },
    { "null",         (char)RPR_NULL  },
    { "nwell",        NWELL           },
    { "pdif",         PDIF            },
    { "poly",         POLY            },
    { "ptie",         PTIE            },
    { "ptrans",       PTRANS          },
    { "pwell",        PWELL           },
    { "rcw",          RDS_TRANS_RCW   },
    { "rds_abox",     RDS_ABOX        },
    { "rds_activ",    RDS_ACTIV       },
    { "rds_alu1",     RDS_ALU1        },
    { "rds_alu2",     RDS_ALU2        },
    { "rds_alu3",     RDS_ALU3        },
    { "rds_alu4",     RDS_ALU4        },
    { "rds_cont",     RDS_CONT        },
    { "rds_cpas",     RDS_CPAS        },
    { "rds_gate",     RDS_GATE        },
    { "rds_ndif",     RDS_NDIF        },
    { "rds_nimp",     RDS_NIMP        },
    { "rds_ntie",     RDS_NTIE        },
    { "rds_nwell",    RDS_NWELL       },
    { "rds_pdif",     RDS_PDIF        },
    { "rds_pimp",     RDS_PIMP        },
    { "rds_poly",     RDS_POLY        },
    { "rds_ptie",     RDS_PTIE        },
    { "rds_pwell",    RDS_PWELL       },
    { "rds_ref",      RDS_REF         },
    { "rds_talu1",    RDS_TALU1       },
    { "rds_talu2",    RDS_TALU2       },
    { "rds_talu3",    RDS_TALU3       },
    { "rds_tpoly",    RDS_TPOLY       },
    { "rds_user0",    RDS_USER0       },
    { "rds_user1",    RDS_USER1       },
    { "rds_user2",    RDS_USER2       },
    { "rds_via1",     RDS_VIA1        },
    { "rds_via2",     RDS_VIA2        },
    { "rds_via3",     RDS_VIA3        },
    { "ref_con",      MBK_REF_CON     },
    { "ref_ref",      MBK_REF_REF     },
    { "talu1",        TALU1           },
    { "talu2",        TALU2           },
    { "talu3",        TALU3           },
    { "tpoly",        TPOLY           },
    { "treat",        S2R_TREAT       },
    { "vw",           RDS_TRANS_VW    }
  };

  char RDS_TRANS_NAME [ RDS_MAX_TRANS ][ 4 ] =
  
  {
    "VW ",
    "LCW",
    "RCW"
  };

  char RDS_USE_NAME [ RDS_MAX_USE ][ 4 ] =

  {
    "ALL",
    "EXT",
    "DRC"
  };
 
  char MBK_VIA_NAME [ MBK_MAX_VIA ][ 12 ] =

  {  
    "CONT_POLY  ",
    "CONT_VIA   ",
    "CONT_DIF_N ",
    "CONT_DIF_P ",
    "CONT_BODY_N",
    "CONT_BODY_P",
    "CONT_VIA2  ",
    "C_X_N      ",
    "C_X_P      ",
    "CONT_VIA3  " 
  };

  char MBK_LAYER_NAME [ MBK_MAX_LAYER ][ 7 ] =

  {
    "NWELL ",
    "PWELL ",
    "NTIE  ",
    "PTIE  ",
    "NDIF  ",
    "PDIF  ",
    "NTRANS",
    "PTRANS",
    "POLY  ",
    "ALU1  ",
    "ALU2  ",
    "ALU3  ",
    "TPOLY ",
    "TALU1 ",
    "TALU2 ",
    "TALU3 "
  };

/*------------------------------------------------------------\
|                                                             |
|                       File variables                        |
|                                                             |
\------------------------------------------------------------*/

  static  FILE *RprFile;
  static  char  RprBuffer[ RPR_MAX_BUFFER ];
  static  long  RprCurrentLine = 0;

/*------------------------------------------------------------\
|                                                             |
|                          Functions                          |
|                                                             |
\------------------------------------------------------------*/

/*------------------------------------------------------------\
|                                                             |
|              Rpr Ascii to float (small numbers)             |
|                                                             |
\------------------------------------------------------------*/

  double RprAtof( integer )

    char * integer;
  {
    char * decimal;
    long shift;
    long i;

    if (decimal = strchr( integer, '.')) 
    {
      * decimal = '\0';

      decimal++;

      for(i=strlen(decimal),shift=1 ; i; shift*=10, i--);

      if (atol(integer) < 0)
      {
        return (double)(atol(integer)*shift - atol(decimal))/(double)(shift);
      }
      else
      {
        return (double)(atol(integer)*shift + atol(decimal))/(double)(shift);
      }
    } 
    else 
    {
      return (double)(atol(integer));
    }
  }

/*------------------------------------------------------------\
|                                                             |
|                       Rpr Upper Case                        |
|                                                             |
\------------------------------------------------------------*/

  char *RprUpperCase( Name )

    char *Name;
  {
    char *Target;
    int   Index;

    if ( Name == (char *)NULL ) return( (char *)NULL );

    Target = (char *)mbkalloc( strlen( Name ) + 1 );

    for ( Index = 0; Name[ Index ]; Index++ )
    {
      Target[ Index ] = toupper( Name[ Index ] );
    }
    Target[ Index ] = '\0';

    return( Target );
  }

/*------------------------------------------------------------\
|                                                             |
|                       Rpr File Get String                   |
|                                                             |
\------------------------------------------------------------*/

 char *RprFileGetString( String, Size )

   char *String;
   int   Size;
 {
   register char *RegisterString;
   register       Register;

   RegisterString = String;

   while (--Size > 0 && (Register = getc( RprFile )) != EOF )
   {
     if ((*RegisterString++ = Register) == '\\') 
     {
       if ((Register = getc( RprFile )) == '\n') 
       {
         RprCurrentLine++;

         *(RegisterString - 1) = ' ';
       } 
       else
       {
         ungetc( Register, RprFile );
       }
     }
     else
     {
       if ( Register == '\n') break;
     }
   }

   *RegisterString = '\0';

   return ( ( Register       == EOF    ) && 
            ( RegisterString == String ) ) ? (char *)NULL : String;
 }

/*------------------------------------------------------------\
|                                                             |
|                       Rpr File Get Line                     |
|                                                             |
\------------------------------------------------------------*/

 void RprGetLine( Buffer )

   char   *Buffer;
 {
   char   *Check;
   char   *String;
   char    OneComment;

   do 
   {
     OneComment = 0;

     Check = RprFileGetString( Buffer, RPR_MAX_BUFFER );

     if ( Check != (char *)NULL ) 
     {
       RprCurrentLine++;
     }
     else 
     {
       RprError( UNEXPECTED_EOF, (char *)NULL, RprCurrentLine );
     }

     if ( String = strchr( Buffer, RPR_COMMENT_CHAR ))
     {
       if ( String == Buffer )
       {
         OneComment = 1;
       }
       else
       {
         *(String - 1) = '\0'; 
       }
     }
 
     while (*Buffer != '\0' && strchr( RPR_SEPARATORS_STRING, *Buffer))
     {
       Buffer = Buffer + 1;;
     }

     if (*Buffer == '\0') OneComment = 1;

   } 
   while ( OneComment == 1);
 }      

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Check Keyword                    |
|                                                             |
\------------------------------------------------------------*/

 char RprCheckKeyword( Word )

    char *Word;
 {
   if ( ( Word == DefineKeyword ) ||
        ( Word == TableKeyword  ) )
   {
     return 1;
   }
   else
   {
     return 0;
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Get First word                   |
|                                                             |
\------------------------------------------------------------*/

 char *RprGetFirstWord( Buffer, IsKeyword )
 
       char *Buffer;
       char  IsKeyword;
 {
   char   *String;

   if ( String = (char *)strtok( Buffer, RPR_SEPARATORS_STRING )) 
   {
     String = namealloc( String );

     if ( ( IsKeyword ) && RprCheckKeyword( String ) )
     {
       RprError( UNEXPECTED_LINE, String, RprCurrentLine );
     }
   }
   return( String );
 }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Get Next word                    |
|                                                             |
\------------------------------------------------------------*/

 char *RprGetNextWord( IsKeyword )
 
       char IsKeyword;
 {
   char   *String;

   if ( String = (char *)strtok( (char *)NULL, RPR_SEPARATORS_STRING )) 
   {
      String = namealloc( String );
  
      if ( ( IsKeyword ) && RprCheckKeyword( String ) )
      {
        RprError( UNEXPECTED_LINE, String, RprCurrentLine );
      }
   }

   return( String );
 }

/*------------------------------------------------------------\
|                                                             |
|                      Rpr Translate Param                    |
|                                                             |
\------------------------------------------------------------*/

 long RprTranslateParam( Param )

      double   Param;
 {
   double Multiple; 
 
   Param    = Param * RDS_UNIT;
   Multiple = Param / RDS_PHYSICAL_GRID;

   if ( RDS_UNIT == -1 )
   {
     RprError( NOT_DEFINED, PhysicalGridKeyword, RprCurrentLine );
   }
 
   if ( (Multiple < (floor(Multiple + (double).5) - RPR_EPSILON )) ||
        (Multiple > (floor(Multiple + (double).5) + RPR_EPSILON )) ) 
   {
     sprintf( RprBuffer, "%g", Param / RDS_UNIT );

     RprError( MULTIPLE_GRID, RprBuffer, RprCurrentLine );
   }

   return( ( long )Param );
 }

/*------------------------------------------------------------\
|                                                             |
|                      Rpr Compute Rds Unit                   |
|                                                             |
\------------------------------------------------------------*/

 void RprComputeRdsUnit( PhysicalGrid )
  
      double PhysicalGrid;
 {
   long Number;
   long RdsUnit;

   if ( PhysicalGrid <= 0 )
   {
     RprError( NEGATIVE_VALUE, PhysicalGridKeyword, RprCurrentLine );
   }

   RdsUnit = 1;

   for ( Number = 1; 
         ! RprEqual( PhysicalGrid * (double)RdsUnit, (double)Number ); 
         Number = Number + 1 )
   {
     if ( RdsUnit > RPR_MAX_RDS_UNIT ) 
     {
       RprError( TOO_SMAL, PhysicalGridKeyword, RprCurrentLine );
     }

     for ( RdsUnit = RdsUnit - 1; 
           PhysicalGrid * (double)RdsUnit + RPR_EPSILON <  (double)Number; 
           RdsUnit = RdsUnit + 1 );

     if ( RprEqual( PhysicalGrid * (double)RdsUnit, (double)Number )) 
  
        break;
   }

   RDS_UNIT          = RdsUnit << 1;
   RDS_PHYSICAL_GRID = Number  << 1;
 }

/*------------------------------------------------------------\
|                                                             |
|                      Rpr Keyword Compare                    |
|                                                             |
\------------------------------------------------------------*/

  int RprKeywordCompare( FirstKey, SecondKey )

      keyword *FirstKey;
      keyword *SecondKey;
  {
    return strcmp( FirstKey->NAME, SecondKey->NAME );
  }

/*------------------------------------------------------------\
|                                                             |
|                      Rpr Get String Value                   |
|                                                             |
\------------------------------------------------------------*/

 long RprGetStringValue( String )

    char     *String;
 {
   double    FloatValue;
   long      Value;
   keyword  *Keyword;
   keyword   Entry;

   if ( strchr( String, '.' )) 
   {
     if ( sscanf( String, "%lf", &FloatValue) ) 
      
       return( RprTranslateParam( FloatValue ) );

     RprError( ILLEGAL_FLOAT, String, RprCurrentLine );
   } 
   else
   {
     if ( sscanf( String, "%d", &Value) )

      return ( RprTranslateParam( (double)Value) );

     Entry.NAME = String;

     Keyword = (keyword *)bsearch( (char *)(&Entry), 
                                   (char *)KeywordDefine, 
                                   RPR_MAX_KEYWORD, sizeof( keyword ),
                                   RprKeywordCompare );

     if ( Keyword != (keyword *)NULL ) return( Keyword->VALUE );

     RprError( UNEXPECTED_LINE, String, RprCurrentLine );
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                      Rpr Get String Float                   |
|                                                             |
\------------------------------------------------------------*/

 float RprGetStringFloat( String )

    char     *String;
 {
   float FloatValue;

   if ( sscanf( String, "%f", &FloatValue) )

     return( FloatValue );

   RprError( ILLEGAL_FLOAT, String, RprCurrentLine );
 }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Get Number                       |
|                                                             |
\------------------------------------------------------------*/

  long RprGetNumber( String )

       char *String;
  {
    long Value;

    if ( sscanf( String, "%d", &Value )) return Value;

    RprError( UNEXPECTED_LINE, "number", RprCurrentLine );
  }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Read Segment                     |
|                                                             |
\------------------------------------------------------------*/

 void RprReadSegment()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord; 
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= MBK_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < MBK_MAX_LAYER )
     {
       Layer     = RprGetStringValue( FirstWord );

       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                 ) &&
               ( Field     <  RDS_SEGMENT_FIELD ) )
       {
         FirstWord = RprGetNextWord( 1 );

         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field % 6 != 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_SEGMENT_TABLE [ Layer ][ Field ] = RprGetStringValue( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_SEGMENT_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Read Connector                   |
|                                                             |
\------------------------------------------------------------*/

 void RprReadConnector()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= MBK_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < MBK_MAX_LAYER )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                   ) &&
               ( Field     <  RDS_CONNECTOR_FIELD ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field % 3 != 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_CONNECTOR_TABLE [ Layer ][ Field ] = RprGetStringValue( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_CONNECTOR_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Read Reference                   |
|                                                             |
\------------------------------------------------------------*/

 void RprReadReference()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1                 ) &&
           ( LayerCount <= MBK_MAX_REFERENCE ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < MBK_MAX_REFERENCE )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                   ) &&
               ( Field     <  RDS_REFERENCE_FIELD ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field & 1 != 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_REFERENCE_TABLE [ Layer ][ Field ] = RprGetStringValue( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_REFERENCE_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                          Rpr Read Via                       |
|                                                             |
\------------------------------------------------------------*/

 void RprReadVia()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1           ) &&
           ( LayerCount <= MBK_MAX_VIA ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < MBK_MAX_VIA )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1               ) &&
               ( Field     <  RDS_VIA_FIELD ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field & 1 != 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_VIA_TABLE[ Layer ][ Field ] = RprGetStringValue( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_VIA_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         } 
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                       Rpr Read Cif Layer                    |
|                                                             |
\------------------------------------------------------------*/

 void RprReadCifLayer()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= RDS_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < RDS_MAX_LAYER )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                   ) &&
               ( Field     <  RDS_CIF_LAYER_FIELD ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field == 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_CIF_LAYER_TABLE[ Layer ][ Field ] = RprUpperCase( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_CIF_LAYER_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Read Gds Layer                   |
|                                                             |
\------------------------------------------------------------*/

 void RprReadGdsLayer()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= RDS_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < RDS_MAX_LAYER )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                   ) &&
               ( Field     <  RDS_GDS_LAYER_FIELD ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field == 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_GDS_LAYER_TABLE[ Layer ][ Field ] = atoi( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_GDS_LAYER_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Read Post Treat                  |
|                                                             |
\------------------------------------------------------------*/

 void RprReadS2RPostTreat()

 {
   char         Layer;
   char         LayerCount;
   char         EndTable;
   char         EndRecord;
   char         Field;
   char        *FirstWord; 

   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= RDS_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );

     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < RDS_MAX_LAYER ) 
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                    ) &&
               ( Field     <  RDS_S2R_POST_TREAT_FIELD ) )
       {
         FirstWord = RprGetNextWord( 1 );
  
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field == 0 ) 
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
  
           EndRecord = 1;
         }
         else 
         {
           RDS_S2R_POST_TREAT_TABLE[ Layer ][ Field ] = RprGetStringValue( FirstWord );

           Field = Field + 1;
         }
       } 
  
       if ( Field == RDS_S2R_POST_TREAT_FIELD ) 
       {
         FirstWord = RprGetNextWord( 0 );
  
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       } 
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Read Over Size                   |
|                                                             |
\------------------------------------------------------------*/

 void RprReadS2ROversize()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= RDS_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < RDS_MAX_LAYER )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                   ) &&
               ( Field     <  RDS_S2R_OVERSIZE_FIELD  ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field == 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_S2R_OVERSIZE_TABLE[ Layer ][ Field ] = RprGetStringValue( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_S2R_OVERSIZE_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                        Rpr Read Ring Width                  |
|                                                             |
\------------------------------------------------------------*/

 void RprReadS2RRingWidth()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= RDS_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < RDS_MAX_LAYER )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                     ) &&
               ( Field     <  RDS_S2R_RING_WIDTH_FIELD  ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field == 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_S2R_RING_WIDTH_TABLE[ Layer ][ Field ] = RprGetStringValue( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_S2R_RING_WIDTH_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                       Rpr Read Layer Width                  |
|                                                             |
\------------------------------------------------------------*/

 void RprReadS2RLayerWidth()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= RDS_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < RDS_MAX_LAYER )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                      ) &&
               ( Field     <  RDS_S2R_LAYER_WIDTH_FIELD  ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field == 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_S2R_LAYER_WIDTH_TABLE[ Layer ][ Field ] = RprGetStringValue( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_S2R_LAYER_WIDTH_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                       Rpr Read Lynx Graph                   |
|                                                             |
\------------------------------------------------------------*/

 void RprReadLynxGraph()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= RDS_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < RDS_MAX_LAYER )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                    ) &&
               ( Field     <  RDS_LYNX_GRAPH_FIELD ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field == 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_LYNX_GRAPH_TABLE [ Layer ][ Field ] = RprGetStringValue( FirstWord );
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_LYNX_GRAPH_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                       Rpr Read Lynx Capa                    |
|                                                             |
\------------------------------------------------------------*/

 void RprReadLynxCapa()

 {
   char  Layer;
   char  LayerCount;
   char  Field;
   char  EndTable;
   char  EndRecord;
   char *FirstWord;
  
   EndTable   = 0;
   LayerCount = 0;

   while ( ( EndTable   != 1             ) &&
           ( LayerCount <= RDS_MAX_LAYER ) )
   {
     RprGetLine( RprBuffer );

     FirstWord  = RprGetFirstWord( RprBuffer, 1 );
     
     if ( FirstWord == EndTableKeyword )
     {
       EndTable = 1;
     }
     else
     if ( LayerCount < RDS_MAX_LAYER )
     {
       Layer     = RprGetStringValue( FirstWord );
       EndRecord = 0;
       Field     = 0;

       while ( ( EndRecord != 1                   ) &&
               ( Field     <  RDS_LYNX_CAPA_FIELD ) )
       {
         FirstWord = RprGetNextWord( 1 );
 
         if ( FirstWord == EndRecordKeyword )
         {
           if ( Field == 0 )
           {
             RprError( MISSING_VALUE, (char *)NULL, RprCurrentLine );
           }
 
           EndRecord = 1;
         }
         else
         {
           RDS_LYNX_CAPA_TABLE [ Layer ][ Field ] = 

             RprGetStringFloat( FirstWord ) / RDS_LAMBDA2;
 
           Field = Field + 1;
         }
       }
 
       if ( Field == RDS_LYNX_CAPA_FIELD )
       {
         FirstWord = RprGetNextWord( 0 );
 
         if ( FirstWord != EndRecordKeyword )
         {
           RprError( TOO_MANY_WORDS, FirstWord, RprCurrentLine );
         }
       }
     }
 
     LayerCount = LayerCount + 1;
   }

   if ( EndTable == 0 )
   {
     RprError( LINE_EXPECTED, EndTableKeyword, RprCurrentLine);
   }
 }

/*------------------------------------------------------------\
|                                                             |
|                      Rpr Read Parameters                    |
|                                                             |
\------------------------------------------------------------*/

  void RprReadParam()

  {
    char   *FirstWord;
    char   *SecondWord;
    long    Continue;
    double  FloatValue;

    Continue = 0;

    while ( Continue != RPR_ALL_DEFINED_MASK )  
    {
      RprGetLine( RprBuffer );

      FirstWord = RprGetFirstWord( RprBuffer, 0);

      if ( FirstWord == DefineKeyword ) 
      {
        FirstWord = RprGetNextWord( 1 );

        if (! FirstWord )
        {
          RprError( MISSING_NAME, DefineKeyword, RprCurrentLine );
        }

        SecondWord = RprGetNextWord(1);

        if (! SecondWord ) 
        {
          RprError( MISSING_VALUE, SecondWord );
        }

        if ( FirstWord == PhysicalGridKeyword )
        {
          sscanf( SecondWord, "%lf", &FloatValue); 
  
          RprComputeRdsUnit( FloatValue );

          Continue |= RPR_PHYSICAL_GRID_MASK;
        }
        else
        if ( FirstWord == LambdaKeyword )
        {
          sscanf( SecondWord, "%lf", &FloatValue); 

          if ( FloatValue <= 0 )
          {
            RprError( NEGATIVE_VALUE, LambdaKeyword, RprCurrentLine );
          }

          RDS_LAMBDA  = RprTranslateParam( FloatValue );
          RDS_LAMBDA2 = RDS_LAMBDA * RDS_LAMBDA;

          Continue |= RPR_LAMBDA_MASK;
        }
        else
        {
          RprError( UNKNOWN_DEFINE, FirstWord, RprCurrentLine );
        }
      } 
      else 
      if ( FirstWord == TableKeyword )
      {
        if (!( FirstWord = RprGetNextWord(1)))
        {
          RprError( MISSING_NAME, TableKeyword );
        }

        if ( FirstWord == S2RPostTreatKeyword )
        {
          RprReadS2RPostTreat();

          Continue |= RPR_S2R_POST_TREAT_MASK;
        }
        else
        if ( FirstWord == SegmentKeyword )
        {
          RprReadSegment();

          Continue |= RPR_SEGMENT_MASK;
        } 
        else
        if ( FirstWord == ConnectorKeyword )
        {
          RprReadConnector();

          Continue |= RPR_CONNECTOR_MASK;
        }else
        if ( FirstWord == ReferenceKeyword )
        {
          RprReadReference();

          Continue |= RPR_REFERENCE_MASK;
        }  
        else
        if ( FirstWord == ViaKeyword )
        {
          RprReadVia();

          Continue |= RPR_VIA_MASK;
        }  
        else
        if ( FirstWord == CifLayerKeyword )
        {
          RprReadCifLayer();

          Continue |= RPR_CIF_LAYER_MASK;
        }  
        else
        if ( FirstWord == GdsLayerKeyword )  
        {
          RprReadGdsLayer();

          Continue |= RPR_GDS_LAYER_MASK;
        }
        else
        if ( FirstWord == S2ROversizeKeyword )
        {
          RprReadS2ROversize();

          Continue |= RPR_S2R_OVERSIZE_MASK;
        }
        else
        if ( FirstWord == S2RRingWidthKeyword )
        {
          RprReadS2RRingWidth();

          Continue |= RPR_S2R_RING_WIDTH_MASK;
        }
        else
        if ( FirstWord == S2RLayerWidthKeyword )
        {
          RprReadS2RLayerWidth();

          Continue |= RPR_S2R_LAYER_WIDTH_MASK;
        }
        else
        if ( FirstWord == LynxGraphKeyword )
        {
          RprReadLynxGraph();

          Continue |= RPR_LYNX_GRAPH_MASK;
        }
        else
        if ( FirstWord == LynxCapaKeyword )
        {
          RprReadLynxCapa();

          Continue |= RPR_LYNX_CAPA_MASK;
        }
        else
        {
          RprError( UNKNOWN_TABLE, FirstWord, RprCurrentLine );
        }
      } 
      else
      { 
        RprError( SYNTAX_ERROR, FirstWord, RprCurrentLine  );
      } 
    }       
  }

/*------------------------------------------------------------\
|                                                             |
|                      Rds Load Parameters                    |
|                                                             |
\------------------------------------------------------------*/

  void loadrdsparam()

  {
    char Field;
    char Layer;

    if ( ! KeywordDefined )
    {
      DefineKeyword        = namealloc( DEFINE_KEYWORD          );
      TableKeyword         = namealloc( TABLE_KEYWORD           );
      EndTableKeyword      = namealloc( END_TABLE_KEYWORD       );
      EndRecordKeyword     = namealloc( END_RECORD_KEYWORD      );
      LambdaKeyword        = namealloc( LAMBDA_KEYWORD          );
      PhysicalGridKeyword  = namealloc( PHYSICAL_GRID_KEYWORD   );
      SegmentKeyword       = namealloc( SEGMENT_KEYWORD         );
      ViaKeyword           = namealloc( VIA_KEYWORD             );
      ConnectorKeyword     = namealloc( CONNECTOR_KEYWORD       );
      ReferenceKeyword     = namealloc( REFERENCE_KEYWORD       );
      CifLayerKeyword      = namealloc( CIF_LAYER_KEYWORD       );
      GdsLayerKeyword      = namealloc( GDS_LAYER_KEYWORD       );
      S2ROversizeKeyword   = namealloc( S2R_OVERSIZE_KEYWORD    );
      S2RRingWidthKeyword  = namealloc( S2R_RING_WIDTH_KEYWORD  );
      S2RLayerWidthKeyword = namealloc( S2R_LAYER_WIDTH_KEYWORD );
      S2RPostTreatKeyword  = namealloc( S2R_POST_TREAT_KEYWORD  );
      LynxGraphKeyword     = namealloc( LYNX_GRAPH_KEYWORD      );
      LynxCapaKeyword      = namealloc( LYNX_CAPA_KEYWORD       );

      KeywordDefined = 1;
    }

    RDS_UNIT = -1;

    for ( Layer = 0; Layer < RDS_MAX_LAYER; Layer++ )
    {
      for ( Field = 0; Field < RDS_S2R_POST_TREAT_FIELD; Field++ )
      {
        RDS_S2R_POST_TREAT_TABLE [ Layer ][ Field ] = RDS_S2R_POST_TREAT_EMPTY;
      }

      for ( Field = 0; Field < RDS_GDS_LAYER_FIELD; Field++ )
      {
        RDS_GDS_LAYER_TABLE [ Layer ][ Field ] = RDS_GDS_LAYER_EMPTY;
      }

      for ( Field = 0; Field < RDS_CIF_LAYER_FIELD; Field++ )
      {
        RDS_CIF_LAYER_TABLE [ Layer ][ Field ] = RDS_CIF_LAYER_EMPTY;
      }

      for ( Field = 0; Field < RDS_S2R_OVERSIZE_FIELD; Field++ )
      {
        RDS_S2R_OVERSIZE_TABLE [ Layer ][ Field ] = RDS_S2R_OVERSIZE_EMPTY;
      }

      for ( Field = 0; Field < RDS_S2R_RING_WIDTH_FIELD; Field++ )
      {
        RDS_S2R_RING_WIDTH_TABLE [ Layer ][ Field ] = RDS_S2R_RING_WIDTH_EMPTY;
      }

      for ( Field = 0; Field < RDS_S2R_LAYER_WIDTH_FIELD; Field++ )
      {
        RDS_S2R_LAYER_WIDTH_TABLE [ Layer ][ Field ] = RDS_S2R_LAYER_WIDTH_EMPTY;
      }

      for ( Field = 0; Field < RDS_LYNX_GRAPH_FIELD; Field++ )
      {
        RDS_LYNX_GRAPH_TABLE [ Layer ][ Field ] = RDS_LYNX_GRAPH_EMPTY;
      }

      for ( Field = 0; Field < RDS_LYNX_CAPA_FIELD; Field++ )
      {
        RDS_LYNX_CAPA_TABLE[ Layer ][ Field ] = RDS_LYNX_CAPA_EMPTY;
      }
    }

    for ( Layer = 0; Layer < MBK_MAX_LAYER; Layer++ )
    {
      for ( Field = 0; Field < RDS_SEGMENT_FIELD; Field++ )
      {
        RDS_SEGMENT_TABLE [ Layer ][ Field ] = RDS_SEGMENT_EMPTY;
      }

      for ( Field = 0; Field < RDS_CONNECTOR_FIELD; Field++ )
      {
        RDS_CONNECTOR_TABLE [ Layer ][ Field ] = RDS_CONNECTOR_EMPTY;
      }
    }
    
    for ( Layer = 0; Layer < MBK_MAX_REFERENCE; Layer++ )
    {
      for ( Field = 0; Field < RDS_REFERENCE_FIELD; Field++ )
      {
        RDS_REFERENCE_TABLE [ Layer ][ Field ] = RDS_REFERENCE_EMPTY;
      }
    }

    for ( Layer = 0; Layer < MBK_MAX_VIA; Layer++ )
    {
      for ( Field = 0; Field < RDS_VIA_FIELD; Field++ )
      {
        RDS_VIA_TABLE [ Layer ][ Field ] = RDS_VIA_EMPTY;
      }
    }

    if ( !( RprFile = fopen( RDS_TECHNO_NAME, "r")))

      RprError( OPEN_FILE, RDS_TECHNO_NAME, 0);

    RprReadParam();
   
    fclose( RprFile );
  }

/*------------------------------------------------------------\
|                                                             |
|                      Rds view Parameters                    |
|                                                             |
\------------------------------------------------------------*/

  void viewrdsparam()

  {
    char  Layer;
    char  Index;

    fprintf( stdout, "\n  DEFINE PHYSICAL_GRID %d", RDS_PHYSICAL_GRID );
    fprintf( stdout, "\n  DEFINE LAMBDA %d"       , RDS_LAMBDA );
    fprintf( stdout, "\n# DEFINE RDS_UNIT %d"     , RDS_UNIT );

    fprintf( stdout, "\n\nTABLE MBK_TO_RDS_SEGMENT\n" );

    for ( Layer = 0; Layer < MBK_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", MBK_LAYER_NAME[ Layer ] );

      Index = 0;

      while ( GET_SEGMENT_LAYER( Layer, Index ) != RDS_SEGMENT_EMPTY )
      {
        fprintf( stdout, "  %s %s %2d %2d %2d %s", 
                 RDS_LAYER_NAME[ GET_SEGMENT_LAYER( Layer, Index ) ],
                 RDS_TRANS_NAME[ GET_SEGMENT_TRANS( Layer, Index ) ],
                 GET_SEGMENT_DLR   ( Layer, Index ),
                 GET_SEGMENT_DWR   ( Layer, Index ),
                 GET_SEGMENT_OFFSET( Layer, Index ),
                 RDS_USE_NAME[ GET_SEGMENT_USE( Layer, Index ) ] );

        Index = Index + 1;
      }
    }

    fprintf( stdout, "\n\nEND" );

    fprintf( stdout, "\n\nTABLE MBK_TO_RDS_CONNECTOR\n" );

    for ( Layer = 0; Layer < MBK_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", MBK_LAYER_NAME[ Layer ] );

      if ( GET_CONNECTOR_LAYER( Layer ) != RDS_CONNECTOR_EMPTY )
      {
        fprintf( stdout, " %s %2d %2d", 
                 RDS_LAYER_NAME[ GET_CONNECTOR_LAYER( Layer ) ],
                 GET_CONNECTOR_DER  ( Layer ),
                 GET_CONNECTOR_DWR  ( Layer ) );
      }
    } 

    fprintf( stdout, "\n\nEND" );

    fprintf( stdout, "\n\nTABLE MBK_TO_RDS_REFERENCE\n" );

    for ( Layer = 0; Layer < MBK_MAX_REFERENCE; Layer++ )
    {
      if ( Layer == 0 ) fprintf( stdout, "\nREF_REF" );
      else              fprintf( stdout, "\nREF_CON" );

      if ( GET_REFERENCE_LAYER( Layer ) != RDS_REFERENCE_EMPTY )
      {
        fprintf( stdout, " %s %2d", 
                 RDS_LAYER_NAME[ GET_REFERENCE_LAYER( Layer ) ],
                 GET_REFERENCE_SIZE( Layer ) );
      }
    }

    fprintf( stdout, "\n\nEND" );

    fprintf( stdout, "\n\nTABLE MBK_TO_RDS_VIA\n" );

    for ( Layer = 0; Layer < MBK_MAX_VIA; Layer++ )
    {
      Index = 0;

      fprintf( stdout, "\n%s", MBK_VIA_NAME[ Layer ] );

      while ( GET_VIA_LAYER( Layer, Index ) != RDS_VIA_EMPTY )
      {
        fprintf( stdout, "  %s %2d", 
                 RDS_LAYER_NAME[ GET_VIA_LAYER( Layer, Index ) ],
                 GET_VIA_SIZE( Layer, Index ) );

        Index = Index + 1;
      }
    }

    fprintf( stdout, "\n\nEND" );

    fprintf( stdout, "\n\nTABLE CIF_LAYER\n" );

    for ( Layer = 0; Layer < RDS_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", RDS_LAYER_NAME[ Layer ] );

      if ( GET_CIF_LAYER ( Layer ) != RDS_CIF_LAYER_EMPTY )
      {
        fprintf( stdout, "%s", GET_CIF_LAYER ( Layer ) );
      }
    }

    fprintf( stdout, "\n\nEND" );

    fprintf( stdout, "\n\nTABLE GDS_LAYER\n" );

    for ( Layer = 0; Layer < RDS_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", RDS_LAYER_NAME[ Layer ] );

      if ( GET_GDS_LAYER( Layer ) != RDS_GDS_LAYER_EMPTY )
      {
        fprintf( stdout, "%d", GET_GDS_LAYER ( Layer ) );
      }
    }
 
    fprintf( stdout, "\n\nEND" );
 
    fprintf( stdout, "\n\nTABLE S2R_OVERSIZE_DENOTCH\n" );

    for ( Layer = 0; Layer < RDS_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", RDS_LAYER_NAME[ Layer ] );

      if ( GET_S2R_OVERSIZE( Layer ) != RDS_S2R_OVERSIZE_EMPTY )
      {
        fprintf( stdout, "%ld", GET_S2R_OVERSIZE ( Layer ) );
      }
    }
 
    fprintf( stdout, "\n\nEND" );
 
    fprintf( stdout, "\n\nTABLE S2R_BLOC_RING_WIDTH\n" );

    for ( Layer = 0; Layer < RDS_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", RDS_LAYER_NAME[ Layer ] );

      if ( GET_S2R_RING_WIDTH( Layer ) != RDS_S2R_RING_WIDTH_EMPTY )
      {
        fprintf( stdout, "%ld", GET_S2R_RING_WIDTH ( Layer ) );
      }
    }
 
    fprintf( stdout, "\n\nEND" );
 
    fprintf( stdout, "\n\nTABLE S2R_MINIMUM_LAYER_WIDTH\n" );

    for ( Layer = 0; Layer < RDS_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", RDS_LAYER_NAME[ Layer ] );

      if ( GET_S2R_LAYER_WIDTH( Layer ) != RDS_S2R_LAYER_WIDTH_EMPTY )
      {
        fprintf( stdout, "%ld", GET_S2R_LAYER_WIDTH ( Layer ) );
      }
    }
 
    fprintf( stdout, "\n\nEND" );
 
    fprintf( stdout, "\n\nTABLE S2R_POST_TREAT\n" );

    for ( Layer = 0; Layer < RDS_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", RDS_LAYER_NAME[ Layer ] );

      if ( GET_S2R_POST_Y_OR_N( Layer ) != RDS_S2R_POST_TREAT_EMPTY )
      {
        fprintf( stdout, "%ld", GET_S2R_POST_Y_OR_N ( Layer ) );
      }
    }
 
    fprintf( stdout, "\n\nEND" );

    fprintf( stdout, "\n\nTABLE LYNX_GRAPH\n" );

    for ( Layer = 0; Layer < RDS_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", RDS_LAYER_NAME[ Layer ] );

      Index = 0;

      while ( GET_LYNX_GRAPH_LAYER( Layer, Index ) != RDS_LYNX_GRAPH_EMPTY )
      {
        fprintf( stdout, "  %s ", 
                 RDS_LAYER_NAME[ GET_LYNX_GRAPH_LAYER( Layer, Index ) ] );

        Index = Index + 1;
      }
    }

    fprintf( stdout, "\n\nEND" );

    fprintf( stdout, "\n\nTABLE LYNX_CAPA\n" );

    for ( Layer = 0; Layer < RDS_MAX_LAYER; Layer++ )
    {
      fprintf( stdout, "\n%s", RDS_LAYER_NAME[ Layer ] );

      fprintf( stdout, "  %g ", GET_LYNX_CAPA_LAYER( Layer ) * RDS_LAMBDA2 );
    }

    fprintf( stdout, "\n\nEND\n\n" );
  }
