/**********************************************************************/
/* mprims.c :                                                         */
/*                                                                    */
/* Mesh some basic primtives into an initial mesh of 4-vertex         */
/* polygons with associated normals                                   */
/*                                                                    */
/* Primitives supported same as those in QuickModel program. Output   */
/* is a set of patches                                                */
/*                                                                    */
/* - Quickmodel primitives support: cones, cubes, cylinders, spheres  */
/* - Option of normals facing in or out                               */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#include <stdio.h>
#include <math.h>
#include "mprims.h"

/**********************************************************************/
/* Print a polygonal patch out */
/**********************************************************************/
void print_polygon(p, num)
     Polygon *p;
     int num;
{
  int i;

  /* Print normal */
  printf("Patch norm%d %d {",num, p->size);
  for(i=0;i<p->size;i++) {
    if (normal_direction == NORMAL_OUT) 
      printf(" { %g %g %g }", p->normal[i].x, p->normal[i].y, 
	     p->normal[i].z);
    else
      printf(" { %g %g %g }", -p->normal[i].x, -p->normal[i].y, 
	     -p->normal[i].z);
  }
  printf(" }\n");

  /* Print vertex */
  printf("Patch vert%d %d {",num, p->size);     
  for(i=0;i<p->size;i++)
    printf(" { %g %g %g }", p->vertex[i].x, p->vertex[i].y, 
	   p->vertex[i].z);
  printf(" }\n");
}
    

/*====================================================================*/
/* Cube */
/*====================================================================*/
double faces[6][4][3] = { /* [face][vertex][x/y/z] */
  {{ -1, 1, -1}, {1, 1, -1,}, {1, 1, 1}, {-1, 1, 1}},    /* Face 0 */
  {{ 1, 1, -1,}, {1, -1, -1}, {1, -1, 1}, {1, 1, 1}},    /* Face 1 */
  {{ -1, -1, -1}, {-1, 1, -1}, {-1, 1, 1}, {-1, -1, 1}}, /* Face 2 */
  {{ -1, 1, 1}, {1, 1, 1}, {1, -1, 1}, {-1, -1, 1}},     /* Face 3 */
  {{ -1, 1, -1}, {1, 1, -1}, {1, -1, -1,}, {-1, -1, -1}},/* Face 4 */
  {{ 1,-1, -1 }, {-1, -1, -1 }, {-1, -1, 1}, {1, -1, 1}} /* Face 5 */
};
void mesh_cube();
void print_cube();

/**********************************************************************/
/* Mesh a cube */
/**********************************************************************/
void mesh_cube(cube, du, dv, id, sid)
     MCube *cube;
     int id, du, dv;
     char *sid;
{
  int i,j,k;
  CPolygon *pptr;
  int u,v,w;
  float face_normal[3]; 
  float tempv[2];

  /* Set u and v subdivision and id */
  cube->du = du; cube->dv = dv;
  cube->id = id; cube->sid = sid;
  cube->num_polys = du * dv;

  /* Create space for du*dv patches per each of 6 faces */
  for(i=0;i<6;i++) 
    if ((cube->faces[i] = (CPolygon *) malloc( (du*dv) *sizeof(CPolygon)))
	== NIL) { 
      fprintf(stderr,"No more memory left for malloc\n");
      exit(-1);
    }
  
  for (i=0;i<6;i++) {
    pptr = cube->faces[i];
    
    /* Set face normals and direction of normals {w=0=x, w=1=y, w=2=z} */
    /* (normals facting in */
    face_normal[0] = 0; face_normal[1] = 0; face_normal[2] = 0;
    switch(i) {
    case 0 : face_normal[Y] = 1; w = Y; u=X; v=Z; break;
    case 1 : face_normal[X] = 1; w = X; u=Y; v=Z; break;
    case 2 : face_normal[X] = -1; w = X; u=Y; v=Z; break;
    case 3 : face_normal[Z] = 1; w = Z; u=X; v=Y; break;
    case 4 : face_normal[Z] = -1; w = Z; u=X; v=Y; break;
    case 5 : face_normal[Y] = -1; w = Y; u=X; v=Z; break;
    default: break;
    }  
    
    dv = cube->dv;
    du = cube->du; 
    if (faces[i][0][v] > faces[i][3][v]) dv = -dv;
    if (faces[i][0][u] > faces[i][1][u]) du = -du; 
    
    /* Subdivide along 2 directions (u,v) of plane of face */
    for(j=0;j < _ABS(dv);j++) {
      tempv[0] = faces[i][0][v] + (2.0*j/dv);
      tempv[1] = faces[i][0][v] + (2.0*(j+1)/dv);
      
      for(k=0;k < _ABS(du);k++) {
	/* Set u,v,w coords of vertices of patch */
	pptr->vertex[0][u] = pptr->vertex[3][u] = faces[i][0][u]+(2.0*k/du);
	pptr->vertex[2][u] = pptr->vertex[1][u] = 
	  faces[i][0][u]+(2.0*(k+1)/du);

	pptr->vertex[0][v] = pptr->vertex[1][v] = tempv[0];
	pptr->vertex[3][v] = pptr->vertex[2][v] = tempv[1];
	
	pptr->vertex[0][w] = pptr->vertex[1][w] = pptr->vertex[2][w] 
	  = pptr->vertex[3][w] = faces[i][0][w];
	
	/* Set vertex normals */
	pptr->normal[0][u] = pptr->normal[1][u] = pptr->normal[2][u] = 
	  pptr->normal[3][u] = pptr->normal[0][v] = pptr->normal[1][v] = 
	    pptr->normal[2][v] = pptr->normal[3][v] = 0.0;
	pptr->normal[0][w] = pptr->normal[1][w] = pptr->normal[2][w] = 
	  pptr->normal[3][w] = face_normal[w];
	
	/* Goto next patch. */
	pptr++;
      }
    }
  }
}

/**********************************************************************/
/* Output mesh for a unit cube */
/**********************************************************************/
void print_cube(cube)
     MCube *cube;
{
  int i,j, k;
  CPolygon *p;
  int npatches;

  npatches = cube->du * cube->dv;

  printf("NumMeshes %d\n", 6);
  /* Print mesh for each face of cube */
  for (i=0;i<6;i++) {
    p = cube->faces[i];
    printf("Mesh mesh%s%d %d {\n", cube->sid, i, npatches);

    /* Print normals for each patch in face i */
    for (j=0;j<(cube->num_polys);j++) {
      printf("Patch norm%d %d {", j+1, 4);
      for(k=0;k<4;k++) {
	if (normal_direction == NORMAL_OUT)
	  printf(" { %g %g %g }", p->normal[k][0],
		 p->normal[k][1], p->normal[k][2]);
	else
	  printf(" { %g %g %g }", -p->normal[k][0],
		 -p->normal[k][1], -p->normal[k][2]);
      }
      printf(" }\n");

    /* Print vertices for each patch in face i */
      printf("Patch vert%d %d {", j+1, 4);
      for(k=0;k<4;k++) printf(" { %g %g %g }", p->vertex[k][0],
			      p->vertex[k][1], p->vertex[k][2]);
      printf(" }\n");
      p++;
    }

    printf("}\n");
  }
}

/*====================================================================*/
/* Sphere */
/*====================================================================*/
void mesh_sphere();
void print_sphere();

double *sine, *cosine;
/**********************************************************************/
/* Calculate meshed sphere, and output */
/**********************************************************************/
void mesh_sphere(sphere, r, num_sub, centre, id, sid)
     MSphere *sphere;
     double r;
     Vector centre;
     int num_sub, id;
     char *sid;
{
  register int i;
  register int longitude, latitude, size, latP1, longP1;
  Polygon *p;
  
  /* Store id, radius, centre, and # subdivisions */
  sphere->num_sub = num_sub;
  sphere->num_polys = 0;
  sphere->r = r;
  sphere->centre.x = centre.x; sphere->centre.y = centre.y; 
  sphere->centre.z = centre.z;
  sphere->id = id; sphere->sid = sid;

  /* Set sphere subdivision */
  if (sine != NIL) free((char *) sine);  
  if ((sine = (double *) malloc((num_sub + 1) * 4 * (sizeof(double))))
      == NIL){ 
    fprintf(stderr,"No more memory left for malloc\n");
    exit(-1);
  }
  cosine = sine + 2 * (num_sub + 1);
  for (i = 0; i < 2 * num_sub; i++) {
    sine[i] = sin(M_PI / num_sub * i);
    cosine[i] = cos(M_PI / num_sub * i);
  }
  sine[num_sub] = 0.;
  cosine[num_sub] = -1.;
  sine[2 * num_sub] = 0.;
  cosine[2 * num_sub] = 1.;

  /* Calculate vertices and normals */
  if ((p = sphere->poly = 
    (Polygon *) malloc(2 * num_sub * num_sub * sizeof(Polygon))) == NIL) {
    fprintf(stderr,"No more memory left for malloc\n");
    exit(-1);
  }
  for (longitude = 0; longitude < 2 * num_sub; longitude++) {
    longP1 = longitude + 1;
    for (latitude = 0; latitude < num_sub; latitude++) {
      size = 0;
      latP1 = latitude + 1;
      sphere->num_polys++;

      /* Calculate normals */
      p->normal[size].x = sine[latitude] * cosine[longitude];
      p->normal[size].y = sine[latitude] * sine[longitude];
      p->normal[size].z = cosine[latitude];
      size++;

      p->normal[size].x = sine[latP1] * cosine[longitude];
      p->normal[size].y = sine[latP1] * sine[longitude];
      p->normal[size].z = cosine[latP1];
      size++;

      if (latitude < num_sub - 1) {
	p->normal[size].x = sine[latP1] * cosine[longP1];
	p->normal[size].y = sine[latP1] * sine[longP1];
	p->normal[size].z = cosine[latP1];
	size++;
      }
      if (latitude != 0) {
	p->normal[size].x = sine[latitude] * cosine[longP1];
	p->normal[size].y = sine[latitude] * sine[longP1];
	p->normal[size].z = cosine[latitude];
	size++;
      }
      
      /* Keep track of number of vertices */
      p->size = size;

      /* Calculate vertices */
      for (i = 0; i < size; i++) {
	p->vertex[i].x = 
	  p->normal[i].x * r + centre.x;
	p->vertex[i].y = 
	  p->normal[i].y * r + centre.y;
	p->vertex[i].z = 
	  p->normal[i].z * r + centre.z;
      }
      p++;
    }
  }
}

/**********************************************************************/
/* Output sphere patch */
/**********************************************************************/
void print_sphere(sphere)
     MSphere *sphere;
{
  int i,j;
  Polygon *p;

  p = sphere->poly;

  printf("NumMeshes %d\n", 1);  
  printf("Mesh mesh%s%d %d {\n", sphere->sid, 1, sphere->num_polys);

  for(i=0;i<(sphere->num_polys);i++) {
    
    printf("Patch norm%d %d{",i+1, p->size);
    for(j=0;j<(p->size);j++) {
      if (normal_direction == NORMAL_OUT)
	printf(" { %g %g %g }", p->normal[j].x, 
	       p->normal[j].y, p->normal[j].z);
      else
	printf(" { %g %g %g }", -p->normal[j].x, 
	       -p->normal[j].y, -p->normal[j].z);
    }
    printf(" }\n");
    
    printf("Patch vert%d %d{", i+1, p->size);
    for(j=0;j<(p->size);j++) 
      printf(" { %g %g %g }", p->vertex[j].x, p->vertex[j].y, 
	     p->vertex[j].z);
    printf(" }\n");
    p++;
  }
  printf("}\n");
}

/*====================================================================*/
/* Cone */
/*====================================================================*/
double Unit();
Vector cNormal();
void mesh_cone();

/**********************************************************************/
#define MIN_LENGTH 1.e-7
double Unit(vp)
     register Vector *vp;
{
  double length, temp;
	
  temp= vp->x*vp->x + vp->y*vp->y + vp->z*vp->z;
  if(temp <= MIN_LENGTH)
    return(-1.);
  length= sqrt(temp);
  vp->x /= length;
  vp->y /= length;
  vp->z /= length;
  return(length);
}

/**********************************************************************/
/* Vector cross product                                               */
/**********************************************************************/
Vector Cross(a,b)
	Vector a,b;
{
  Vector c;
  
  c.x= a.y*b.z - a.z*b.y;
  c.y= a.z*b.x - a.x*b.z;
  c.z= a.x*b.y - a.y*b.x;
  return(c);
}

/**********************************************************************/
/* Find normal along circle of cone */
/**********************************************************************/
Vector cNormal(v1, base, top)
     Vector v1, base, top;
{
  Vector v2, v3, result;
  
  v2.x= top.x-base.x;
  v2.y= top.y-base.y;
  v2.z= top.z-base.z;
  
  v3= Cross(v2, v1);
  result= Cross(v3, v2);
  if(Unit(&result) < 0.) {
    fprintf(stderr, "couldn't normalize vector\n");
    exit(1);
  }
  return(result);
}

/**********************************************************************/
/* Create meshed cone */
/**********************************************************************/
void mesh_cone(cone, rb, rt, h, n1, n2, id, sid)
     MCone *cone;
     double rb, rt, h; /* radiuse of base and top, and height */
     int n1, n2; /* subdivision for base and sides resp. */
     int id; 
     char *sid;
{
  double angle, s1, s2, u1, u2;
  int i, j, k, ii;
  Vector base, top, *circle, *normal, cNormal();
  Polygon *bp, *tp;

  if(rb <= 0. || rb > 100. || rt <= 0. || rt > 100. || h <= 0. || h > 100.
     || n1 < 4 || n1 > 100 || n2 < 4 || n2 > 100) 
    fprintf(stderr, "Sorry, args out of range\n");
  
  cone->rb = rb;
  cone->rt = rt;
  cone->h = h;
  cone->num_side_polys = n1 * n2;
  cone->num_disc_polys = n2;
  cone->id = id;
  cone->sid = sid;

  if ((bp = cone->bpoly = (Polygon *) malloc( n1 * sizeof(Polygon) )) == NIL) {
    fprintf(stderr,"No more memory left for malloc\n");
    exit(-1);
  }
  if ((tp = cone->tpoly = (Polygon *) malloc( n1 * sizeof(Polygon) )) == NIL) {
    fprintf(stderr,"No more memory left for malloc\n");
    exit(-1);
  }
  
  if ((circle = (Vector *) malloc(n1*sizeof(Vector))) == NIL) {
    fprintf(stderr,"No more memory left for malloc\n");
    exit(-1);
  }
  if ((normal= (Vector *) malloc(n1*sizeof(Vector))) == NIL) {
    fprintf(stderr,"No more memory left for malloc\n");
    exit(-1);
  }
  
  /* Calculate base and top circular points and normals */
  for(i=0;i<n1;i++) {
    angle= (((double) i) / n1) * 2.0 * M_PI;
    circle[i].x= cos(angle);
    circle[i].y= sin(angle);
    circle[i].z= 0.;
    base.x = circle[i].x * rb;
    base.y = circle[i].y * rb;
    base.z = 0.;
    top.x = circle[i].x * rt;
    top.y = circle[i].y * rt;
    top.z = h;
    normal[i]= cNormal(circle[i], base, top);
  }

  printf("NumMeshes 3\n");

  /* Calculate bottom */
  printf("Mesh top_%s%d %d {\n", cone->sid, cone->id, cone->num_disc_polys);
  k = 1;
  for(i=0;i<n1;i++) {
    ii= (i+1) % n1;
    bp->size = 3;
    bp->vertex[0].x = circle[i].x * rb; 
    bp->vertex[0].y = circle[i].y * rb;
    bp->vertex[1].x = circle[ii].x * rb; 
    bp->vertex[1].y = circle[ii].y * rb;
    bp->vertex[0].z = 0.0; bp->vertex[1].z = 0.0;
    bp->vertex[2].x = 0.0; bp->vertex[2].y = 0.0; bp->vertex[2].z = 0.0;
    
    bp->normal[0].x = 0.0; bp->normal[0].y = 0.0; bp->normal[0].z = -1.0;
    bp->normal[1].x = 0.0; bp->normal[1].y = 0.0; bp->normal[1].z = -1.0;
    bp->normal[2].x = 0.0; bp->normal[2].y = 0.0; bp->normal[2].z = -1.0;

    print_polygon(bp, k);
    k++; bp++;
  }
  printf("}\n");

  /* Calculate top */
  printf("Mesh bot_%s%d %d {\n", cone->sid, cone->id, cone->num_disc_polys);
  k = 1;
  for(i=0;i<n1;i++) {
    ii= (i+1) % n1;
    tp->size = 3;
    tp->vertex[0].x= circle[i].x * rt; tp->vertex[0].y= circle[i].y * rt;
    tp->vertex[1].x= circle[ii].x * rt; tp->vertex[1].y= circle[ii].y * rt;
    tp->vertex[0].z= h; tp->vertex[1].z= h;
    tp->vertex[2].x= 0.0; tp->vertex[2].y= 0.0; tp->vertex[2].z= h;

    tp->normal[0].x = 0.0; tp->normal[0].y = 0.0; tp->normal[0].z = 1.0;
    tp->normal[1].x = 0.0; tp->normal[1].y = 0.0; tp->normal[1].z = 1.0;
    tp->normal[2].x = 0.0; tp->normal[2].y = 0.0; tp->normal[2].z = 1.0;

    print_polygon(tp, k);
    k++; tp++;
  }
  printf("}\n");

  printf("Mesh side_%s%d %d {\n", cone->sid, cone->id, cone->num_side_polys);
  k = 1;
  /* Calculate sides */
  for(i=0;i<n1;i++) {
    ii= (i+1) % n1;
    for(j=0;j<n2;j++) {
      u1= ((double) j)/n2;
      u2= ((double) j+1)/n2;
      s1= rb + u1*(rt-rb);
      s2= rb + u2*(rt-rb);
      
      printf("Patch norm%d 4 {",k);
      if (normal_direction == NORMAL_OUT) {
	printf(" { %g %g %g }", normal[i].x, normal[i].y, normal[i].z);
	printf(" { %g %g %g }", normal[ii].x, normal[ii].y, normal[ii].z);
	printf(" { %g %g %g }", normal[ii].x, normal[ii].y, normal[ii].z);
	printf(" { %g %g %g } }\n", normal[i].x, normal[i].y, normal[i].z);
      } else {
	printf(" { %g %g %g }", -normal[i].x, -normal[i].y, -normal[i].z);
	printf(" { %g %g %g }", -normal[ii].x, -normal[ii].y, -normal[ii].z);
	printf(" { %g %g %g }", -normal[ii].x, -normal[ii].y, -normal[ii].z);
	printf(" { %g %g %g } }\n", -normal[i].x, -normal[i].y, -normal[i].z);
      }

      printf("Patch vert%d 4 {",k);     
      printf(" { %g %g %g }", s1*circle[i].x, s1*circle[i].y, u1*h);
      printf(" { %g %g %g }", s1*circle[ii].x, s1*circle[ii].y, u1*h);
      printf(" { %g %g %g }", s2*circle[ii].x, s2*circle[ii].y, u2*h);
      printf(" { %g %g %g } }\n", s2*circle[i].x, s2*circle[i].y, u2*h);
      
      k++;
    }
  }
  printf("}\n");
}

/*====================================================================*/
/* Plane */
/*====================================================================*/
void mesh_plane();
void print_plane();
void mesh_plane() {}
void print_plane() {}

/*====================================================================*/
/* Meshed Cylinder */
/*====================================================================*/
void mesh_cylinder();
void print_cylinder();

/**********************************************************************/
/* Create a polygonally tesselated cylinder                           */
/**********************************************************************/
void mesh_cylinder(cyl, n, id, sid, radius, height)
     MCylinder *cyl;          /* Cylinder to mesh */
     int id;                 /* Cylinder id */
     char *sid;              /* Cylinder string id */
     int n;                  /* number of subdivisions */
     float radius,height;    /* radius and height */
{
  double theta, dtheta, z, dz;
  int vertex_i;
  Polygon *bp, *tp, *sp;

  cyl->id = id;
  cyl->sid = sid;
  cyl->r = radius;
  cyl->h = height;

  /* Calculate # of side and disc polys and allocate space */

  cyl->num_side_polys = 2 * n * n;
  cyl->num_disc_polys = 2 * n;
  if ((sp = cyl->spoly = 
       (Polygon *) malloc(cyl->num_side_polys * sizeof(Polygon))) == NIL) {
    fprintf(stderr,"No more memory left for malloc\n");
    exit(-1);
  }
  if ((tp = cyl->tpoly = 
       (Polygon *) malloc(cyl->num_disc_polys * sizeof(Polygon))) == NIL) {
    fprintf(stderr,"No more memory left for malloc\n");
    exit(-1);
  }
  if ((bp = cyl->bpoly = 
       (Polygon *) malloc(cyl->num_disc_polys * sizeof(Polygon))) == NIL) {
    fprintf(stderr,"No more memory left for malloc\n");
    exit(-1);
  }
  dz = height / (double) n;
  dtheta = M_PI / (double) n;

  for (z=(-height/2.0); z < (height/2.0); z=z+dz) {
    for (theta = 0.0; theta < (2 * M_PI); theta += dtheta) {
      
      /* Four vertex points of side patches */
      sp->size = 4;
      sp->vertex[0].z = sp->vertex[1].z = z;
      sp->vertex[0].y = sp->vertex[3].y = radius * cos(theta);
      sp->vertex[0].x = sp->vertex[3].x = radius * sin(theta);
      sp->vertex[1].y = sp->vertex[2].y = radius * cos(theta+dtheta);
      sp->vertex[1].x = sp->vertex[2].x = radius * sin(theta+dtheta);
      sp->vertex[2].z = sp->vertex[3].z = z + dz;

      /* Four normals of unit length for side vertices */
      for (vertex_i = 0; vertex_i < 4; vertex_i++) {
	sp->normal[vertex_i].z = 0;
	sp->normal[vertex_i].y = sp->vertex[vertex_i].y / radius;
	sp->normal[vertex_i].x = sp->vertex[vertex_i].x / radius;
      }

      /* Top and bottom triangular patches and normals */
      if (z == -1) {
	bp->size = tp->size = 3;
	bp->vertex[0].x = tp->vertex[0].x = sp->vertex[0].x; 
	bp->vertex[0].y = tp->vertex[0].y = sp->vertex[0].y;
	bp->vertex[0].z = -1.; tp->vertex[0].z = 1.;
	bp->vertex[1].x = tp->vertex[1].x = sp->vertex[1].x;
	bp->vertex[1].y = tp->vertex[1].y = sp->vertex[1].y;
	bp->vertex[1].z = -1.; tp->vertex[1].z = 1.;
	bp->vertex[2].x = 0.0; bp->vertex[2].y = 0.0; bp->vertex[2].z = -1.0;
	tp->vertex[2].x = 0.0; tp->vertex[2].y = 0.0; tp->vertex[2].z = 1.0;
	
	bp->normal[0].z = -1.; bp->normal[1].z = -1.; bp->normal[2].z = -1.;
	bp->normal[0].x = 0.; bp->normal[1].x = 0.; bp->normal[2].x = 0.;
	bp->normal[0].y = 0.; bp->normal[1].y = 0.; bp->normal[2].y = 0.;
	tp->normal[0].z = 1.; tp->normal[1].z = 1.; tp->normal[2].z = 1.;
	tp->normal[0].x = 0.; tp->normal[1].x = 0.; tp->normal[2].x = 0.;
	tp->normal[0].y = 0.; tp->normal[1].y = 0.; tp->normal[2].y = 0.;

	/* printf("TOP IS"); print_polygon(tp);
	printf("BOT IS"); print_polygon(bp); */
	bp++;
	tp++;
      }
      sp++;
    }
  }
}

/**********************************************************************/
/* Print Cylinder normals and points,and colours */
/**********************************************************************/
void print_cylinder(cyl)
     MCylinder *cyl;
{
  int i,j, k;
  Polygon *tp, *bp, *p;

  printf("NumMeshes 3\n");
  
  /* Print top */
  k = 1;
  printf("Mesh %s_top%d %d {\n", cyl->sid, k, cyl->num_disc_polys);
  tp = cyl->tpoly;
  for (i=0; i<(cyl->num_disc_polys); i++) {
    printf("Patch norm%d %d {", i+1, tp->size);
    for(j=0;j<(tp->size);j++) {
      if (normal_direction == NORMAL_OUT) 
	printf(" { %g %g %g }", tp->normal[j].x, 
	       tp->normal[j].y, tp->normal[j].z);
      else
	printf(" { %g %g %g }", -tp->normal[j].x, 
	       -tp->normal[j].y, -tp->normal[j].z);
    }
    printf(" }\n");

    printf("Patch vert%d %d {", i+1, tp->size);
    for(j=0;j<(tp->size);j++) 
      printf(" { %g %g %g }", tp->vertex[j].x, tp->vertex[j].y, 
	     tp->vertex[j].z);
    printf(" }\n");
    tp++;
  }

  /* Print bottom */
  k = 2;
  printf("Mesh %s_bot%d %d {\n", cyl->sid, k, cyl->num_disc_polys);
  bp = cyl->bpoly;
  for (i=0; i<(cyl->num_disc_polys); i++) {
    printf("Patch norm%d %d {", i+1, bp->size);
    for(j=0;j<(bp->size);j++) {
      if (normal_direction == NORMAL_OUT) 
	printf(" { %g %g %g }", bp->normal[j].x, 
	       bp->normal[j].y, bp->normal[j].z);
      else
	printf(" { %g %g %g }", -bp->normal[j].x, 
	       -bp->normal[j].y, -bp->normal[j].z);
    }
    printf(" }\n");

    printf("Patch vert%d %d {", i+1, bp->size);
    for(j=0;j<(bp->size);j++) 
      printf(" { %g %g %g }", bp->vertex[j].x, bp->vertex[j].y, 
	     bp->vertex[j].z);
    printf(" }\n");
    bp++;
  }

  /* Print sides */
  k = 3;
  printf("Mesh %s_side%d %d {\n", cyl->sid, k, cyl->num_side_polys);
  p = cyl->spoly;
  for (i=0; i<(cyl->num_side_polys); i++) {
    printf("Patch norm%d %d{", i+1, p->size);
    for(j=0;j<(p->size);j++) {
      if (normal_direction == NORMAL_OUT) 
	printf(" { %g %g %g }", p->normal[j].x, 
	       p->normal[j].y, p->normal[j].z);
      else
	printf(" { %g %g %g }", p->normal[j].x, 
	       p->normal[j].y, p->normal[j].z);
    }
    printf(" }\n");

    printf("Patch vert%d %d {", i+1, p->size);
    for(j=0;j<(p->size);j++) 
      printf(" { %g %g %g }", p->vertex[j].x, p->vertex[j].y, 
	     p->vertex[j].z);
    printf(" }\n");
    p++;
  }

  printf("}\n");
}

/**********************************************************************/
/* Tesselate a primitive into some fixed mesh and output it */
/**********************************************************************/
void mesh_primitive(ptype)
     int ptype;
{
  MSphere s;
  MCylinder cy;
  MCube cu;
  MCone co;
  Vector cu_centre;
  int i;

  switch (ptype) {
  case MCONE: 
    /* void mesh_cone(cone, rb, rt, h, n1, n2, id, sid) */
    mesh_cone(&co, 1.0, 0.01, 2.0, 16, 16,  5,"MCone");
    break;
  case MCUBE:
    mesh_cube(&cu, 2, 2, 1, "MCube");
    print_cube(&cu);
    for (i=0;i<6;i++) free(cu.faces[i]);
    break;
  case MCYLINDER:
    /* void mesh_cylinder(cyl, n, id, sid, radius, height) */
    mesh_cylinder(&cy, 8, 2,"MCyl", 1.0, 2.0);
    print_cylinder(&cy); 
    free(cy.spoly);
    free(cy.tpoly);
    free(cy.bpoly);
    break;
  case MSPHERE: 
    cu_centre.x = 0.;
    cu_centre.y = 0.;
    cu_centre.z = 0.;
    /* void mesh_sphere(sphere, r, num_sub, centre, id, sid) */
    mesh_sphere(&s, 1.0, 8, cu_centre, 2, "MSphere");
    printf("\n");
    print_sphere(&s);
    free(&s);
    break;
  default: 
    fprintf(stderr, "Somethings wrong. No such primitive exists !\n");
    break;
  }
}

/**********************************************************************/
/* main */
/**********************************************************************/
main(argc, argv)
     int argc;
     char *argv[];
{
  int ptype;

  if (argc < 3) {
    fprintf(stderr,
	    "usage %s [1:CONE | 2:CUBE | 3:CYLINDER | 4:SPHERE]\n",
	    argv[0]);
    fprintf(stderr,"\t[0|1 = Normals facing in or out (default)]\n");
    exit(1);
  }
  ptype = atoi(argv[1]);               /* Set type of primitive      */
  normal_direction = atoi(argv[2]);    /* Set direction of normals   */
  switch(ptype) {
    case 1 : ptype = MCONE; break;
    case 2 : ptype = MCUBE; break;
    case 3 : ptype = MCYLINDER; break;
    case 4 : ptype = MSPHERE; break;
    default : break;
    }
  mesh_primitive(ptype);
}
