/*
 *	Controls everything.  See design.
 *	Copyright 1989, Soren K. Lundsgaard.
 */

#include <stdio.h>
#include <sys/types.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "checkpid.h"
#include "fileactivity.h"
#include "getloadav.h"
#include "getnum.h"
#include "pid-job.h"
#include "subsjobno.h"
#include "ttyactivity.h"
#include "sendit.h"
#include "startmpqs.h"
#include "file-name.h"

#ifndef MPQS
#define MPQS	"./MPQS"
#endif

#ifndef RESTART
#define RESTART	"./RESTART"
#endif

int	pausetty;
float	pauseload, resumeload;
char	*mpqsarg[] = {
	{ MPQS },
	{ RESTART },
	{ 0  }
};

static int stopmpqs();

main(int argc, char *argv[])
{
	int	c;
	int	errflg = 0;
	extern	int	optind, opterr;
	extern	char	*optarg;
	char	**filearg;
	int	mpqspid;
	int	mpqsuid;
	int	runfactorpid;
	int	runrunning, mpqsrunning;
	int	killflag = 0;
	int	mpqs_stopped = 0;
	char    buf[BUFSIZ];
        char    pid_jobname[BUFSIZ];
	int     pid_jobfd;
	int     lockres;

	while ((c = getopt(argc, argv, "kl:L:t:m:")) != EOF ) {
		switch (c) {
		case 't':
			pausetty = atoi(optarg);
			if (pausetty < 120) {
				pausetty *= 60;
			} 
			break;
		case 'k':
			killflag = 1;
			break;
		case 'm':
			mpqsarg[0] = optarg;
			break;
		case 'l':
			pauseload = atof(optarg);
			break;
		case 'L':
			resumeload = atof(optarg);
			break;
		case '?':
			errflg++;
			break;
		default:
			break;
		}
	}
	filearg = &argv[optind];
	
	if (pauseload && !resumeload) resumeload = pauseload/2;

	if (errflg) {
		fputs("usage: runfactor -l pauseload -t pausetty [-L resumeload] [-m mpqs]\n", stderr);
		fputs("exiting: bad usage\n",stderr);
		exit(1);
	}

	/*
	 * First, check for the existence of the ID files.  If a file
	 * exists, check its owner.  If I am not the owner, then exit
	 * now.
	 *
	 * Then check to see if there is another runfactor process
	 * currently running, and see if there is an mpqs process
	 * currently running.
	 * 
	 * If there is a runfactor process running, but no mpqs process
	 * running, and I am supposed to kill mpqs instead of stopping it,
	 * then I should exit, since I assume that other runfactor has the
	 * same idea I do, and must have killed off the mpqs process.
	 *
	 * If there is a runfactor but no mpqs process, and I am supposed
	 * to stop the mpqs process, then I should kill the runfactor
	 * and then start an mpqs process.
	 *
	 * If there is an mpqs process but no runfactor process, then I
	 * should just attach myself to that mpqs process and continue
	 * running.
	 *
	 * -derek <warlord@MIT.EDU>
	 */
	 
	if (((mpqsuid = getmpqsuid()) != -1) && (mpqsuid != getuid())) {
		fputs("exiting: mpqs running, not controlled by me\n",stderr);
		exit(1);
	}

	runfactorpid = getrunfactorpid();
	mpqspid = getmpqspid();

	if (runfactorpid == 0) 
	    runrunning = 0;
	else 
	    /* See if runfactor is really running */
	    switch(kill(runfactorpid, 0)) {
	    case 0:
		runrunning = 1;
		break;
	    default:
		if (errno == ESRCH) 
		    runrunning = 0;
		else {
		    fputs("exiting: weird error from kill()\n",stderr);
		    exit(1);
		}
	    }

	if (mpqspid == 0) 
	    mpqsrunning = 0;
	else
	    /* See if mpqs is really running */
	    switch(kill(mpqspid, 0)) {
	    case 0:
		mpqsrunning = 1;
		break;
	    default:
		if (errno == ESRCH) {
		    sendit(getmpqsjob());
		    mpqsrunning = 0;
		} else {
		    fputs("exiting: weird error from kill()\n",stderr);
		    exit(1);
		}
	    }

	/* both running normally, exit */
	if (runrunning && mpqsrunning) {
	    fputs("Running normally.\n",stderr);
	    exit(0);
	}

	/* runfactor by no mpqs.  exit if in kill-mode.  kill runfactor
	 * and start mpqs if in stop-mode.
	 */	   
	if (runrunning && !mpqsrunning) {
	    if (killflag) {
		fputs("Another runfactor running\n",stderr);
		exit(0);
	    } else {
		kill(runfactorpid,SIGTERM);
		mpqspid = startmpqs(mpqsarg);
	    }
	}

	/* 
	 * Write out my PID ***before*** trying to start mpqs.  We
	 * do this in case we cannot start mpqs, so another starting
	 * runfactor knows that we are running and will either try to
	 * kill me, or realize I am running and die.
	 */
	writepid(getpid());

	/* I am the first!  Start up mpqs! */
	if (!runrunning && !mpqsrunning) 
	    mpqspid = startmpqs(mpqsarg);

	/* if there is mpqs but no runfactor, then just over! 
	 * this requires no new work on my part, really!
	 */

	/* Get the name of the pid-job file */
	buildfilename(pid_jobname, PIDDIR, MPQS_PID, NULL);
	if ((pid_jobfd = open(pid_jobname,O_RDWR)) == -1) {
		fprintf(stderr, "Cannot open %s\n", pid_jobname);
		return 0;
	}

	for (;;) {
		time_t	now;
		time(&now);

		if (pausetty && ttyactivity() + pausetty >= now) {
			if (stopmpqs(mpqspid, mpqs_stopped, killflag) == -1) {
				fputs("exiting: bad pausetty kill\n",stderr);
				exit(1);
			} else {
				lockres = lockf(pid_jobfd,F_ULOCK,(off_t) 4);
				if (lockres != 0) {
					fputs("error: unable to unlock (tty), exiting\n",stderr);
					exit(1);
				}
				if (!(mpqs_stopped & 01)) {
					fputs("sleeping due to tty actvity:",
					    stderr);
					fputs(ctime(&now), stderr);
					mpqs_stopped = 01;
				}
				sleep(DEFAULT_SLEEP);
				continue;
			}
		}
		if (filearg && fileactivity(filearg) + pausetty > now) {
			if (stopmpqs(mpqspid, mpqs_stopped, killflag) == -1) {
				fputs("exiting: bad fileact kill\n",stderr);
				exit(1);
			} else {
				lockres = lockf(pid_jobfd,F_ULOCK,(off_t) 4);
				if (lockres != 0) {
					fputs("error: unable to unlock (file), exiting\n",stderr);
					exit(1);
				}
				if (!(mpqs_stopped & 02)) {
					fputs("sleeping due to file actvity:",
					    stderr);
					fputs(ctime(&now), stderr);
					mpqs_stopped = 02;
				}
				sleep(DEFAULT_SLEEP);
				continue;
			}
		}
		if (pauseload && mpqs_stopped && getloadav() >= resumeload) {
			sleep(DEFAULT_SLEEP);
			continue;
		}
		if (pauseload && getloadav() >= pauseload) {
			if (stopmpqs(mpqspid, mpqs_stopped, killflag) == -1) {
				fputs("exiting: bad pauseload kill\n",stderr);
				exit(1);
			} else {
				lockres = lockf(pid_jobfd,F_ULOCK,(off_t) 4);
				if (lockres != 0) {
					fputs("error: unable to unlock (load), exiting\n",stderr);
					exit(1);
				}
				if (!(mpqs_stopped & 04)) {
					fputs("sleeping due to load average:",
					    stderr);
					fputs(ctime(&now), stderr);
					mpqs_stopped = 04;
				}
				sleep(DEFAULT_SLEEP);
				continue;
			}
		}
		lockres = lockf(pid_jobfd,F_TLOCK,(off_t) 4);
		if (lockres != 0) {
			fputs("exiting: tried to continue but could not lock pid-job file\n",stderr);
			exit(1);
		} else {
		    if (!killflag) {
			if (kill(mpqspid, SIGCONT) == -1) {
			    fputs("exiting: bad continue\n",stderr);
			    exit(1);
			} 
		    } else {
			if (mpqs_stopped) {
			    close(pid_jobfd);
			    mpqspid = startmpqs(mpqsarg);
			    if ((pid_jobfd = open(pid_jobname,O_RDWR)) == -1) {
				fprintf(stderr, "Cannot open %s\n", pid_jobname);
				return 0;
			    }
			}
		    }
		}
		if (mpqs_stopped) {
			fputs("continuing:", stderr);
			fputs(ctime(&now), stderr);
			mpqs_stopped = 0;
		}
		sleep(DEFAULT_SLEEP);
	}
}

/* This function stops the mpqs process. */
static int
stopmpqs(mpqspid, mpqs_stopped, killflag)
int mpqspid, mpqs_stopped, killflag;
{
    if (!killflag)
	return(kill(mpqspid, SIGSTOP));

    if (killflag)
	if (!mpqs_stopped)
	    return(kill(mpqspid, SIGTERM));

    return 0;
}
