/* file easynum.c  4.7.1995

title:     EASYNUM                 
author:    Pavel Pokorny           
           Prague Institute of Chemical Technology 
e-mail:    Pavel.Pokorny@vscht.cz  
publisher: Pavel Pokorny Praha  
year:      1995                      
ISBN:      80-901886-1-3             

available: tiger.vscht.cz(147.33.5.11):/pub/xpplot/easynum.tar.Z  

Copyright (c) 1995 Pavel Pokorny   

This software can be used free of charge under the condition that  
it is explicitly cited in every work  
for the preparation of which it has been used.  

This software can be copied, distributed, modified and compiled   
free of charge under the condition that this notice is preserved.  


List of functions:

seteps0 (x)
seth0   (x)
real myrand ()
called (s,n)
leave (s)

timeaction (argc,argv,action)
real usedtime ()

wrla()
wri (s,i)
wrf (s,x)
wrl ()
wrli (s,i)
wrlf (s,x)

vali (s)
long vall (s)
real valf (s)

timesmm (A,B,C,d1,d2,d3)     A = B.C  
timesmv (y,A,x,d1,d2)        y = A.x  
timessv (y,s,x,d)            y = s x  
timessm (A,s,B,d1,d2)        A = s B  
plusmm  (A,B,C,d1,d2)        A = B + C   
plusvv  (x,y,z,d)            x = y + z  
minusvv (x,y,z,d)            x = y - z
copy  (x,y,d)                x = y

fwritematrix (f,header,A,d1,d2)
swritematrix (fn,header,A,d1,d2)
writematrix (s,a,d1,d2)
fwritevector (f,header,v,d)
swritevector (fn,header,v,d)
writevector (s,a,d)
wvector (x,d)

solve (A,x,y,d)
invmat (a,d)
invmat1 (a,d)
invmat2 (a,d)
invmatn (A,d)
make1matrix (E,d)
makerandmatrix (E,d)
makesymrandmatrix (E,d)
makezerovector (x,d)
sqrvector (x,d)
sqrtvector (x,d)
sumnormvector (x,d)

real norm (x,d)
normalize (x,d)
real scalprod (x,y,d)
transpose (a,b,d)
readmatrix (A,d,s)
copymatrix (A,B,d1,d2)
extractdiag (x,A,d)

checkstep (ax0,yo,ah0,acor,f,p,d)
rk (x0,yo,h0,e0,y,ah,ae,f,p,d)
integ1h  (at, y,f,p,d)
integdt  (t,T,y,f,p,d)
integdtb (t,T,y,f,p,d)
integdte (t,T,y,pne,f,p,d)

kub  (t0,x0,h,f,g,J, at,x,d)
kub3e (t0,x0,h0,eps0, x,ah,aeps,F,p,d)

ppopenr  (af,s)
ppopenww (af,s)
ppopenw  (af,s)
ppopena  (af,s)
fileexists (s)

real det (A,d)
outer (z,A,d)
subdet (S,A,d,m,n)

newton (x,pf,d)
jacob (J,pf,x0,d)
er (s)
erp ()
grad (g,pf,x0,d)
difference  (x,y,d)
real distance (x,y,d)
in (x,x1,x2,d)
tangent (t,x,f,d)
eigv2 (eig,j)
solve2 (x,a)

real rotrat (T,A,f,p,d)
setstac (y,T,f,p,d)
perturbation (y,A)
oneyear (y,T,A,pne,f,p,d)

long wc (f)

prepranddata (x,n)
prepndata (x,n)
printdata (x,n)
sprintdata (x,n,s)
sprintidata (x,n,s)
print2data (x,y,n)
revdata (x,n)
revmatrix (A,d)
logdata (x,n)
windowdata (x,n)

sortdata (x,n)
subsortdata (x,l,r)
sortdatai (x,index,n)
subsortdatai (x,index,l,r)
sortdata3 (x,n,y,z)
subsortdata3 (x,l,r,y,z)
sortdatavm (x,A,d)
subsortdatavm (x,A,d,l,r)

allocdata (x,n)
alloclong (x,n)
readdata (x,n,s)
freaddata (x,n,f)
minmax (x,dl,pxmin,pxmax)
filtrdata (x,y,n,m)
standardize (x,n)
autocor (x,y,n)

Round       (wx)
long Roundl (wx)

real log2 (wx)
sqrtc (y,x)
sqrc  (y,x)
rk6 (t,x,h,f)
perpen (a,b,x,d)
vectprod (x,a,b,d)
real lnf (x)
real lnfs (x)

# ifdef HIGH_PRECISION

long double ldfrac (x)

long double ldsin (x)
long double ldcos (x)
long double ldrsin (x)
long double ldrcos (x)

long double ldsqrt (x)

long double ldSin (x)
long double ldCos (x)
long double ldmsin (v)
long double ldmcos (v)

ldwrite(x)
long double ldabs (x)
long double ldlog (x)

# endif

rktesttab (to,yo,f,p,d)
rktest (to,yo,h,ea,eb,ec,f,p,d)
prk (t0,x0,h0,eps0, x,ah,aeps,f,p,d)
rkr (x,yo,y,e,h,f,p,d)
grk (xo,yo,y,h,f,p,d)

orthonormalizen (e0,D,e,n,d)
real cosfi (a,b,d)
v2M (M,v,n,d)
M2v (v,M,n,d)
LE (x,le,F,p,T,TT,o,steps,nle,d)
LEd (x,le,F,p,iT,iTT,o,steps,nle,d)

binprintshort (b)
binprintdouble (x)

ilog2 (x)
ildlog2 (x)

long two2n (n)
long bitinv (n,size,columns)
reorder(x,size,columns)
TrigTable (csn,size)
fht (x,size)
MakeSpectrum (spectrum,x,size)

lookline (T,mgn,pos,i,d)
checkline (T,mgn,pos,i,p,q)
firstlook (T,mgn,pos,d)
look (T,mgn,pos,p,q,nar,d)
diagiter (T, R, mgn, pos, nar, pp, pq, d)
diagonalize (A,Q,D,d)

lsm (x,y,n,c,d,e,r,L)
real corcoef (x,y,n)
real Nad (a,b)
real Bezier (t,a,N)
curve (x,x1,x2,t,N,h,hmin,hmax,f,fout,p,d)
curvecorrector (x0,t,f,p,d)
verbq_conti (t,eig)
verbq_discr (t,eig)
real defint (f,a,b,n,m)
definttest ()
findcycle (x,n,i1,i2,f,p,dim)

End of list of functions.

*/
/*******************************************/
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <stdlib.h>
#include <sys/times.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <limits.h>

# ifndef DIM
# define DIM 20
# endif /* DIM */

# ifndef DIM2
# define DIM2 100
# endif /* DIM2 */

# ifdef HIGH_PRECISION
/*-----------------------------------------------------*/
# define EPS 1E-34L
# define ldeps  1E-35L

# define PiHalf 1.5707963267948966192313216916397514420986L
# define Pi     3.1415926535897932384626433832795028841972L
# define Pi3_2  4.712388980384689857693965074919254326296L
# define twoPi  6.283185307179586476925286766559005768394L
# define Pito2  9.869604401089358618834490999876151135314L
# define Pito3 31.00627668029982017547631506710139520223L
# define Pito4 97.4090910340024372364403326887051112497L
 /* 40 digits */

# define real long double
# define Sin(x)    (ldsin((real)(x)))
# define Cos(x)    (ldcos((real)(x)))
# define Sqrt(x)   (ldsqrt((real)(x)))

# define Abs(x)    (ldabs((real)(x)))
# define Log(x)    (ldlog((real)(x)))

long double ldfrac (long double);
long double ldsin  (long double);
long double ldcos  (long double);
long double ldSin  (long double);
long double ldCos  (long double);

long double ldsqrt (long double);
long double ldabs  (long double);
long double ldlog  (long double);

/*-----------------------------------------------------*/
# else /*  not  HIGH_PRECISION */
/*-----------------------------------------------------*/
# define EPS 1E-16

# define PiHalf 1.5707963267948966192313216916397514420986
# define Pi     3.1415926535897932384626433832795028841972
# define Pi3_2  4.712388980384689857693965074919254326296
# define twoPi  6.283185307179586476925286766559005768394
# define Pito2  9.869604401089358618834490999876151135314
# define Pito3 31.00627668029982017547631506710139520223
# define Pito4 97.4090910340024372364403326887051112497
 /* 40 digits */

# define real  double
# define Sin(x)    (sin((real)(x)))
# define Cos(x)    (cos((real)(x)))
# define Sqrt(x)   (sqrt((real)(x)))

# define Abs(x)    (fabs((real)(x)))
# define Log(x)    (log((real)(x)))

/*
real Sqrt (x)
 real x;
{
 (void) fprintf (stderr,"Sqrt : x = %G\n",x);
 return (sqrt((double)x));
}
*/
/*-----------------------------------------------------*/
# endif  /*  HIGH_PRECISION */

typedef real   vector [DIM];
typedef vector matrix [DIM];
typedef vector vectors[DIM2];

extern void exit();

int debug  = 0, /* flag to print debug info */
    debugl = 0; /* function calling level   */

# define FAST    0  /* local error = h^5 */
# define STIFF   1  /*               h^4 */
# define PRECISE 2  /*               h^8 */

int intmethod = FAST;

real
 geps0   = 1E-7,
 gh0     = 1E-5;
int gcormax = 0;
/************************************************/
# define seteps0(x) geps0 = x;
# define seth0(x)   gh0   = x;
/************************************************/
real myrand() { return ( rand() / (real)32768 ); }
/*
extern double drand48();
# define myrand drand48
*/
/************************************************/
called (s,n)
 char *s;
 long *n;
{
 int i;

 ++ *n;
 for (i=0;i<debugl;i++)
  (void)fprintf(stderr," ");
 (void)fprintf(stderr,"%s called %ld\n",s,*n);
 debugl++;
} /* called */
/************************************************/
leave (s)
 char *s;
{
 int i;

 debugl--;
 for (i=0;i<debugl;i++)
  (void)fprintf(stderr," ");
 (void)fprintf(stderr,"%s O.K.\n\n",s);
} /* leave */
/************************************************/

# ifndef HIGH_PRECISION

/*
timeaction (argc,argv,action)
 int argc;
 char **argv;
 int (*action)();
{
 time_t t1,t2;
 char s1[50],s2[50];
 int i;
 extern real usedtime();

 t1 = time ((time_t *)NULL);
 (void) strcpy (s1,ctime(&t1));
 for (i=0;i<50;i++) if (s1[i]==10) s1[i]=0;
 action (argc,argv);
 t2 = time ((time_t *)NULL);
 (void) strcpy (s2,ctime(&t2));
 for (i=0;i<50;i++) if (s2[i]==10) s2[i]=0;

 (void)fprintf (stderr,
   "program = %8s,  start = %s, end = %s, used = %G s \n",
   argv[0], s1, s2, usedtime());
 }

real usedtime ()
{
 struct tms b;

 (void) times (&b);
 return (
  (b.tms_utime +
   b.tms_stime +
   b.tms_cutime +
   b.tms_cstime) / (real) CLK_TCK
 );
}
*/
# endif

/************************************************/
wrla()    /* writeln asterisk */
{
  (void) printf (" * \n");
} /* wrla */
/************************************************/
wri (s,i)  /* write int */
 char *s;
 int i;
{
 (void) printf ("%s%3d",s,i);
} /* wri */
/************************************************/
wrf (s,x)  /* write float or real */
 char *s;
 real x;
{
 (void) printf ("%s%8.5G",s,x);
} /* wrf */
/************************************************/
wrl ()   /* writeln */
{
 (void) printf ("\n");
} /* wrl */
/************************************************/
wrli (s,i)  /* writeln int */
 char *s;
 int i;
{
 wri (s,i);
 wrl ();
} /* wrli */
/************************************************/
wrlf (s,x)  /* writeln float or real */
 char *s;
 real x;
{
 wrf (s,x);
 wrl ();
} /* wrlf */
/************************************************/
vali (s)
char *s;
{
 int i,is;

 is = sscanf(s,"%d",&i);
 if(is!=1){
  (void) fprintf (stderr,"error reading string %s\n",s);
  exit(1);
 };
 return (i);
} /* vali */
/************************************************/
long vall (s)
 char *s;
 {
  int is;
  long i;

  is = sscanf(s,"%ld",&i);
  if(is!=1){
   (void) fprintf (stderr,"error reading string %s\n",s);
   exit(1);
  };
  return (i);
 } /* vali */
/************************************************/
 real valf (s)
 char *s;
 {
  int is;
  double x;

  is = sscanf(s,"%lf",&x);
  if(is!=1){
   (void) fprintf (stderr,"error reading string %s\n",s);
   exit(1);
  };
  return ((real)x);
 } /* valf */
/************************************************/
/************************************************/
 timesmm (A,B,C,d1,d2,d3)   /*  A = B.C  */
  real A [DIM][DIM],B[DIM][DIM],C[DIM][DIM];
  int d1,d2,d3;
  {
   int i,j,n;
   real w,wA[DIM][DIM];

   for(i=0;i<d1;i++)
    for(j=0;j<d3;j++) {
     w = 0;
     for (n=0;n<d2;n++) w += B[i][n] * C[n][j];
     wA [i][j] = w;
    };
   for(i=0;i<d1;i++)
    for(j=0;j<d3;j++) A[i][j]=wA[i][j];
  } /* timesmm  */
/***************************************************/
 timesmv (y,A,x,d1,d2)   /*  y = A.x  */
 real A[DIM][DIM], x[DIM], y[DIM];
 int d1,d2;
 {
  int i,j;
  real w,wv[DIM];

  for(i=0;i<d1;i++){
   w = 0;
   for(j=0;j<d2;j++) w += A[i][j]*x[j];
   wv[i] = w;
  };
  for (i=0;i<d1;i++) y[i] = wv[i];
 } /* timesmv */
/***************************************************/
 timessv (y,s,x,d)   /*  y = s x  */
  real s,x[DIM],y[DIM];
  int d;
  {
   int i;

   for (i=0;i<d;i++) y[i] = s * x[i];
  }  /* timessv  */
/***************************************************/
 timessm (A,s,B,d1,d2)     /*  A = s B  */
  real s,A[DIM][DIM],B[DIM][DIM];
  int d1,d2;

  {
   int i,j;

   for (i=0;i<d1;i++)
    for (j=0;j<d2;j++) A[i][j] = s * B[i][j];
  }  /*  timessm  */
/***************************************************/
 plusmm (A,B,C,d1,d2)    /*  A = B + C   */
  real A[DIM][DIM],B[DIM][DIM],C[DIM][DIM];
  int d1,d2;
  {
   int i,j;

   for (i=0;i<d1;i++)
    for (j=0;j<d2;j++) A[i][j] = B[i][j]+C[i][j];
  }  /*  plusmm  */
/***************************************************/
 minusmm (A,B,C,d1,d2)    /*  A = B - C   */
  real A[DIM][DIM],B[DIM][DIM],C[DIM][DIM];
  int d1,d2;
  {
   int i,j;

   for (i=0;i<d1;i++)
    for (j=0;j<d2;j++) A[i][j] = B[i][j]-C[i][j];
  }  /*  minusmm  */
/***************************************************/
 plusvv (x,y,z,d)      /*   x = y + z  */
  real *x,*y,*z;
  int d;
  {
   int i;

   for (i=0;i<d;i++) x[i] = y[i] + z[i];
  }  /*  plusvv  */
/***************************************************/
 minusvv (x,y,z,d)      /*   x = y - z  */
  real *x,*y,*z;
  int d;
  {
   int i;

   for (i=0;i<d;i++) x[i] = y[i] - z[i];
  }  /*  minusvv  */
/***************************************************/
 copy  (x,y,d)
 real *x,*y;
 int d;
 {
  int i;

  for (i=0;i<d;i++) x[i] = y[i];
 } /* copy */
/***************************************************/
fwritematrix (f,header,A,d1,d2)
 FILE *f;
 char *header;
 matrix A;
 int d1,d2;
{
 int i,j;

 (void) fprintf (f,"%s\n",header);

 for(i=0;i<d1;i++){
  for(j=0;j<d2;j++) {
   (void) fprintf (f,"%10G ",(double) A[i][j]);
   if (d2 > 10) (void) fprintf (f,"\n");
  };
  (void) fprintf (f,"\n");
 };
} /* fwritematrix */
/***************************************************/
swritematrix (fn,header,A,d1,d2)
 char *fn,*header;
 matrix A;
 int d1,d2;
{
 FILE *f;

 ppopenww (&f,fn);
 fwritematrix (f,header,A,d1,d2);
 (void) fclose (f);
} /* swritematrix */
/***************************************************/
writematrix (header,A,d1,d2)
 char *header;
 matrix A;
 int d1,d2;
{
 fwritematrix (stdout,header,A,d1,d2);
} /* writematrix */
/************************************************/
fwritevector (f,header,v,d)
 FILE *f;
 char *header;
 vector v;
 int d;
{
 int i;

 (void) fprintf (f,"%s",header);

 for(i=0;i<d;i++) {
  (void) fprintf (f,"%10G ",(double) v[i]);
  if (d > 10) (void) fprintf (f,"\n");
 };

 (void) fprintf (f,"\n");
 } /* fwritevector */
/************************************************/
swritevector (fn,header,v,d)
 char *fn,*header;
 vector v;
 int d;
{
 FILE *f;

 ppopenww (&f,fn);
 fwritevector (f,header,v,d);
 (void) fclose (f);
} /* swritevector */
/************************************************/
writevector (header,v,d)
 char *header;
 vector v;
 int d;
{
 fwritevector (stdout,header,v,d);
} /* writevector */
/************************************************/
wvector (x,d)
 vector x;
 int d;
{
 int i;

 (void) printf ("\n");
 for (i=0;i<d;i++) (void) printf ("%G\n",x[i]);
} /* wvector */
/************************************************/
/************************************************/
solve (A,x,y,d)  /* A.x = y      */
 vector x,y;     /*  x  = Ainv.y */ 
 matrix A;
 int d;
{
 matrix Ainv;

 copymatrix (Ainv,A,d,d);
 invmat (Ainv,d);
 timesmv (x,Ainv,y,d,d);
} /* solve */
/************************************************/
invmat (a,d)
 real a[DIM][DIM];
 int d;
{
 if (d == 1) invmat1(a,d);
  else if (d==2) invmat2(a,d);
   else invmatn (a,d);
} /* invmat */
/**************************************************/
 invmat1 (a,d)
  real a[DIM][DIM];
  int d;
 {
/*  extern void exit(); */

  if ((*a)==0){
   (void) printf("error inverting singular 1D matrix\n");
   writematrix("matrix = ",a,d,d);
   exit(1);
  };
  a[0][0] = 1 / a[0][0];
 }
/**************************************************/
 invmat2 (a,d)
  real a[DIM][DIM];
  int d;
 {
  real Det, w;

  Det = a[0][0]*a[1][1]-a[0][1]*a[1][0];

  if (Det==0){
   (void) printf("error inverting singular 2D matrix\n");
   writematrix("matrix = ",a,d,d);
   exit(1);
  };
  w       =  a[0][0];
  a[0][0] =  a[1][1]/Det;
  a[1][1] =  w      /Det;
  a[0][1] = -a[0][1]/Det;
  a[1][0] = -a[1][0]/Det;
 } /* invmat2 */
/**************************************************/
 invmatn (A,d)
  real A[DIM][DIM];
  int    d;
 {
  int    i,j,k,n;
  real w,wv[DIM],B[DIM][DIM];

  if(d>DIM){
   (void) fprintf(stderr,"sorry DIM = %d \n",DIM);
   exit (1);
  };

   for (i=0;i<d;i++)
    for (j=0;j<d;j++)
     B[i][j] = 0;
   for (i=0;i<d;i++)
    B [i][i] = 1;
   for (k=0;k<d;k++) {
     i = k;
     while ((A[i][k]==0) && (i<d)) i++;
     if (A[i][k] == 0) {
       (void) printf("Error : inverting singular nD matrix\n");
       writematrix("matrix = ",A,d,d);
       exit(1);
      };
     if (i>k) {
       for(n=0;n<d;n++) {
        wv[n]   = A[i][n];
        A[i][n] = A[k][n];
        A[k][n] =   wv[n];
        wv[n]   = B[i][n];
        B[i][n] = B[k][n];
        B[k][n] =   wv[n];
       };
      };
     w = A[k][k];
     for (i=k;i<d;i++) A[k][i] = A[k][i] / w;
     for (i=0;i<d;i++) B[k][i] = B[k][i] / w;
     for (j=k+1;j<d;j++) {
       w = A[j][k];
       for(i=k;i<d;i++)A[j][i] = A[j][i] - A[k][i] * w;
       for(i=0;i<d;i++)B[j][i] = B[j][i] - B[k][i] * w;
      };
    };
   for (k=d-2;k>=0;k--)
    for (j=k+1;j<d;j++)
     for (i=0;i<d;i++)
      B[k][i] = B[k][i] - A[k][j] * B[j][i];
  for(i=0;i<d;i++)
   for(j=0;j<d;j++)
    A[i][j] = B[i][j];
 } /* invmatn */
/**************************************************/
 make1matrix (E,d)
  real E [DIM][DIM];
  int d;
  {
   int i,j;

   for (i=0;i<d;i++)
    for (j=0;j<d;j++) E[i][j] = 0;
   for (i=0;i<d;i++)  E[i][i] = 1;
  }  /*  make1matrix  */
/**************************************************/
makerandmatrix (E,d)
 real E [DIM][DIM];
 int d;
{
 int i,j;

 for (i=0;i<d;i++)
  for (j=0;j<d;j++)
   E[i][j] = (real) myrand();
}  /*  makerandmatrix  */
/**************************************************/
makesymrandmatrix (E,d)
 real E [DIM][DIM];
 int d;
{
 int i,j;

 for (i=0;i<d;i++)
  for (j=0;j<=i;j++) {
   E[i][j] = (real) myrand();
   E[j][i] = E[i][j];
  };
}  /*  makesymrandmatrix  */
/***************************************************/
makezerovector (x,d)
 vector x;
 int d;
{
 int i;

 for (i=0;i<d;i++) x[i]=0;
} /* makezerovector */
/***************************************************/
sqrvector (x,d)
 vector x;
 int d;
{
 int i;

 for (i=0;i<d;i++) x[i] *= x[i];
} /* sqrvector */
/***************************************************/
sqrtvector (x,d)
 vector x;
 int d;
{
 int i;

 for (i=0;i<d;i++) x[i] = sqrt((double)x[i]);
} /* sqrtvector */
/***************************************************/
sumnormvector (x,d)
 vector x;
 int d;
{
 int i;
 real s=0;

 for (i=0;i<d;i++) s += x[i];
 for (i=0;i<d;i++) x[i] /= s;
} /* sumnormvector */
/***************************************************/
real norm (x,d)
 vector x;
 int d;
{
 int i;
 real a=0;

 for (i=0;i<d;i++) a += x[i] * x[i];
 a = Sqrt (a);
 return (a);
} /* norm */
/***************************************************/
real anorm (x,d)
 vector x;
 int d;
{
 int i;
 real a=0;

 for (i=0;i<d;i++) a += fabs(x[i]);
 return (a);
} /* anorm */
/***************************************************/
normalize (x,d)
 real *x;
 int d;
{
 real a;
 int i;

 a = norm (x,d);
 for (i=0;i<d;i++) x[i] /= a;
} /* normalize */
/***************************************************/
anormalize (x,d)
 real *x;
 int d;
{
 real a;
 int i;

 a = anorm (x,d);
 for (i=0;i<d;i++) x[i] /= a;
} /* anormalize */
/***************************************************/
real scalprod (x,y,d)
 real x[],y[];
 int d;
{
 int i;
 real w;

 w = 0;
 for(i=0;i<d;i++)w+=x[i]*y[i];
 return (w);
} /* scalprod */
/************************************************/
transpose (a,b,d)
 real a[DIM][DIM],b[DIM][DIM];
 int d;
{
 int i,j;

 for (i=0;i<d;i++)
 for (j=0;j<d;j++) 
  a[i][j] = b[j][i];
} /* transpose */
/***********************************/
readmatrix (A,d,s)
 matrix A;
 int *d;
 char *s;
{
 int i,j;
 real *x;
 long n;

 readdata (&x,&n,s);
 *d = Round ( Sqrt (n) );

 if (*d > DIM) {
  (void) fprintf (stderr,
   "sorry, dim = %1d > maxdim = %1d\n", *d,DIM);
  exit (0);
 };

 for (i=0;i<*d;i++)
 for (j=0;j<*d;j++)
  A[i][j] = x [i * *d + j];
 (void) free ((void*)x);
} /* readm */
/***********************************/
copymatrix (A,B,d1,d2)
 matrix A,B;
 int d1,d2;
{
 int i,j;

 for (i=0;i<d1;i++)
 for (j=0;j<d2;j++)
  A[i][j] = B[i][j];
} /* copymatrix */
/***********************************/
extractdiag (x,A,d)
 vector x;
 matrix A;
 int d;
{
 int i;

 for (i=0;i<d;i++)
  x[i] = A[i][i];
} /* extractdiag */
/***********************************/
/***********************************/
checkstep (ax0,yo,ah0,acor,f,p,d)
 real *ax0,*ah0;
 int (*f)();
 vector yo,p;
 int *acor,d;
{
 vector y;
 real h,e,
  hmin = 1E-10,
  hmax = 10;
 int i,iter=0,itermax=10;
 static int nerout=0, neroutmax = 10;
 static long ncalled = 0;

 if (debug) {
  called ("checkstep",&ncalled);
  (void) fprintf (stderr,"to = %.15G\n",(double)*ax0);
  (void) fprintf (stderr,"yo = ");
  for(i=0;i<d;i++)
   (void)fprintf(stderr,"%.15G ",(double)yo[i]);
  (void) fprintf (stderr,"\n");
  (void) fprintf (stderr,"ho = %.15G\n",(double)*ah0);
 };

 h = *ah0;

 do {
  *ah0  = h / ++iter;

  switch (intmethod) {
   case FAST   :
    rk   (*ax0,yo,*ah0,geps0,y,&h,&e,f,p,d); break;
   case STIFF  :
    kub3e(*ax0,yo,*ah0,geps0,y,&h,&e,f,p,d); break;
   case PRECISE:
    prk  (*ax0,yo,*ah0,geps0,y,&h,&e,f,p,d); break;
  };

  if (debug) (void) fprintf (stderr,
   "iter = %2d h0 = %G h = %G e = %G\n",
   iter, (double)*ah0, (double)h, (double)e);

  if (!debug && iter > itermax && nerout < neroutmax) {
   nerout++;
   if (nerout==neroutmax)
    (void) fprintf (stderr, "\nfurther output on \
         checkstep konvergence suppressed\n\n");
   else {
    (void) fprintf (stderr, "\ncheckstep warning \n");
    (void) fprintf (stderr,"iter  = %d \n",iter);
    (void) fprintf (stderr,"e     = %G \n",(double)e);
    (void) fprintf (stderr,"geps0 = %G \n",(double)geps0);
    (void) fprintf (stderr,"t     = %.10G \n",(double)*ax0);
    for(i=0;i<d;i++)
    (void) fprintf (stderr,"x[%1d]  = %.10G \n",i,(double)yo[i]);
    (void) fprintf (stderr,"h0    = %G \n",(double)*ah0);
    (void) fprintf (stderr,"h     = %G \n",(double)  h );
   };
  };
 } while (e > 10 * geps0);

 for (i=0;i<d;i++) yo[i] = y[i];
 *ax0 += *ah0;
 if (Abs(h) > hmax) h = hmax * Abs(h) / h;
 if (Abs(h) < hmin) h = hmin * Abs(h) / h;
 *ah0  = h;
 *acor = iter-1;
 if (*acor > gcormax) gcormax = *acor;

 if (debug) {
  (void) fprintf (stderr,"t  = %.15G\n",(double)*ax0);
  (void) fprintf (stderr,"y  = ");
  for(i=0;i<d;i++)
   (void)fprintf(stderr,"%.15G ",(double)yo[i]);
  (void) fprintf (stderr,"\n");
  (void) fprintf (stderr,"h  = %.15G\n",(double)*ah0);
  leave ("checkstep");
 };
} /* checkstep */
/************************************************/
rk (x0,yo,h0,eps0,y,ah,aeps,F,p,d)
 real x0,yo[],h0,eps0,y[],*ah,*aeps;
 int (*F)();
 vector p;
 int d;
 {
  int i;
  real f  [DIM],
         k1 [DIM],
         k2 [DIM],
         k3 [DIM],
         k4 [DIM],
         k5 [DIM],
         yl [DIM],
         y2 [DIM],
         y3 [DIM],
         y4 [DIM],
         y5 [DIM],
         eps,h;
  static int nrk = 0;
  real g[DIM],J[DIM][DIM];
  static long ncalled = 0;

 if (debug) {
  called ("rk",&ncalled);
  (void)fprintf(stderr,"x0   = %G\n",x0);
  fwritevector (stderr,"yo   = ",yo,d);
  (void)fprintf(stderr,"h0   = %G\n",h0);
  (void)fprintf(stderr,"eps0 = %G\n",eps0);
 };

 nrk++;
 F(x0,yo,p, f,g,J, d);
 for (i=0; i<d; i++) k1[i] = h0 * f[i];
 for (i=0; i<d; i++) yl[i] = yo[i] + k1[i] / 3;
 F(x0+h0/3,yl ,p, f,g,J, d);
 for (i=0; i<d; i++) k2[i] = h0 * f[i];
 for (i=0; i<d; i++) y2[i] = yo[i] + (k1[i]+k2[i]) / 6;
 F(x0+h0/3,y2,p, f,g,J, d);
 for (i=0; i<d; i++) k3[i] = h0 * f[i];
 for (i=0; i<d; i++) y3[i] = yo[i]+0.125*k1[i]+0.375*k3[i];
 F(x0+h0/2,y3,p, f,g,J, d);
 for (i=0; i<d; i++) k4[i] = h0 * f[i];
 for (i=0; i<d; i++) y4[i] = yo[i]+0.5*k1[i]-1.5*k3[i]+2*k4[i];
 F(x0+h0,y4,p, f,g,J, d);
 for (i=0; i<d; i++) k5[i] = h0 * f[i];
 for (i=0; i<d; i++) y5[i] = yo[i]+(k1[i]+4*k4[i]+k5[i])/6;
 eps = 0;
 for (i=0; i<d; i++) eps += (y4[i]-y5[i]) * 
                            (y4[i]-y5[i]);
 eps = Sqrt (eps) / 5;

 if (debug) {
  fwritevector (stderr,"yl = ",yl ,d);
  fwritevector (stderr,"y2 = ",y2,d);
  fwritevector (stderr,"y3 = ",y3,d);
  fwritevector (stderr,"y4 = ",y4,d);
  fwritevector (stderr,"y5 = ",y5,d);
  (void)fprintf(stderr,"eps  = %G\n",eps );
 };

 if (eps>0)h = h0 * exp ((double)(Log(eps0/eps)/4));
 else      h = h0 * 2;
 *ah   = h;
 *aeps = eps;
 for (i=0;i<d;i++) y[i] = y5[i];

 if (debug) {
   (void)fprintf(stderr,"h    = %G\n",h   );
   leave ("rk");
 };
} /* rk */
 /************************************************/
 integ1h (at,y,f,p,d)
  real *at,y[];
  int (*f)();
  vector p;
  int d;
 {
  extern real gh0;
  int cor;
  static long ncalled = 0;

 if (debug) called ("integ1h",&ncalled);

 checkstep (at ,y  ,&gh0 ,&cor,f,p,d);
 if (debug) leave ("integ1h");
 } /* integ1h */
 /**************************************************/
 integdt (t,T,y,f,p,d)
  real y[],*t,T;
  int (*f)();
  vector p;
  int d;
 {
  real tmax = *t+T;
  extern real gh0;
  int cor;

  if (T > 0) {
   if (gh0 < 0) gh0 = -gh0;

   while ((*t+gh0) < tmax) {
    checkstep (t ,y  ,&gh0 ,&cor,f,p,d);
   }
   do {
    gh0 = tmax - *t;
    checkstep (t ,y  ,&gh0 ,&cor,f,p,d);
   }
   while (cor);
  };
 } /* integdt */
 /**************************************************/
 integdtb (t,T,y,f,p,d)
  real y[],*t,T;
  int (*f)();
  vector p;
  int d;
 {
  real tend = *t-T;
  extern real gh0;
  int cor;

  if (gh0 > 0) gh0 = -gh0;

  while ((*t+gh0) > tend) {
   checkstep (t ,y  ,&gh0 ,&cor,f,p,d);
  }
  do {
   gh0 = tend - *t;
   checkstep (t ,y  ,&gh0 ,&cor,f,p,d);
  }
  while (cor);
 } /* integdtb */
/**************************************************/
 integdte (t,T,y,pne,f,p,d)
  real *t,T,y[];
  int (*f)();
  vector p;
  int *pne,d;
 {
  real  tmax = *t + T,
         ay2 = 4,
         ay1 = 3,
         ay;
  extern real gh0;
  int cor;
  static int yf=1;

  while ((*t+gh0) < tmax) {
   checkstep (t ,y  ,&gh0 ,&cor,f,p,d);
   ay = y[1];
   if ((yf==1)&&(ay<ay1)) (*pne)++;
   if (ay>ay2) yf=1;
   if (ay<ay1) yf=0;
  }
  do {
   gh0 = tmax - *t;
   checkstep (t ,y  ,&gh0 ,&cor,f,p,d);
   ay = y[1];
   if ((yf==1)&&(ay<ay1)) (*pne)++;
   if (ay>ay2) yf=1;
   if (ay<ay1) yf=0;
  } while (cor);
 } /* integdte */
/**************************************************/
/**************************************************/
kub (t0,x0,h,f,g,J,  at,x,d)
 real t0,h;
 vector x0,f,g,x;
 matrix J;
 real *at;
 int d;
{
 real hh = h*h;
 vector wv,ww;
 matrix wA,wB,E,JJ;

 timesmm (JJ,J,J,d,d,d);
 make1matrix (E,d);

 timessm (wA,-h,J,d,d);    /*    -h J   */
 timessm (wB,hh/2,JJ,d,d); /* hh/2 JJ   */
 plusmm  (wA,E,wA,d,d);    /* E-hJ      */
 plusmm  (wA,wA,wB,d,d);   /* E-hJ+hhJJ */
 invmat  (wA,d);           /* inv       */

 timessm (wB,-h/2,J,d,d);  /* -h/2J       */ 
 plusmm  (wB,E,wB,d,d);    /* E-h/2J      */
 timessv (wv,h,f,d);       /* h f         */
 timesmv (wv,wB,wv,d,d);   /* (E-h/2J).hf */
   
 timessv (ww,hh/2,g,d);    /* hh/2 g      */
 plusvv  (wv,wv,ww,d);     /* (E-h/2J).hf+hh/2g */

 timesmv (wv,wA,wv,d,d);   /*  dx  */

 plusvv  (x,x0,wv,d);      /* x = x0 + dx  */

 *at = t0 + h;
}  /*  kub  */
/**************************************************/
 kub3e (t0,x0,h0,eps0, x,ah,aeps,F,p,d)
  /*  pars like in rk  */
  real t0,x0[DIM],h0,eps0;
  real x[DIM],*ah,*aeps;
  int (*F)();
  vector p;
  int d;
  {
   real h2 = h0/2, t1,t2,t3, eps,h;
   vector wv, x1,x2,x3, f0,f2, g0,g2;
   matrix J0,J2;

   F (t0,x0,p, f0,g0,J0, d);
   kub (t0,x0,h0,f0,g0,J0,&t1,x1,d);
   kub (t0,x0,h2,f0,g0,J0,&t2,x2,d);
   F (t2,x2,p, f2,g2,J2, d);
   kub (t2,x2,h2,f2,g2,J2,&t3,x3,d);
   timessv (wv,(real)-4.0 ,x3,d);
   plusvv  (wv,x1   ,wv,d);
   timessv (x ,(real)(-1/3.),wv,d);

   timessv (wv,(real)-1.0 ,x1,d);
   plusvv  (wv,x3   ,wv,d);
   eps = norm (wv,d);

   if (eps>0)h = h0 * exp ((double)(Log(eps0/eps)/3));
   else      h = h0 * 2;
   *ah   = h;
   *aeps = eps;
  }  /*  kub3e  */
/**************************************************/
/**************************************************/
 ppopenr (af,s)
  FILE **af;
  char *s;
 {
  *af = fopen (s,"r");

  if (*af == NULL) {
   (void) fprintf (stderr,
    "error opening file %s for reading \n\n", s);
   exit (1);
  }
 }  /* ppopenr */
 /************************************************/
 ppopenww (af,s)
  FILE **af;
  char *s;
 {
  *af = fopen (s,"w");
  if (*af == NULL) {
   (void) fprintf (stderr,
    "error opening file %s for writing \n\n", s);
   exit (1);
  }
 } /* ppopenww */
 /************************************************/
 ppopenw (af,s)
  FILE **af;
  char *s;
 {
  char c;

  if (fileexists (s)) {
   (void) fprintf (stderr,
    "warning : file %s exists, overwrite ? (y/n) ",s);
   c = getchar();
   if ((c != 'y')&&(c!='Y')) exit (1);
   while (getchar() != 10);
  }
  ppopenww (af,s);
 } /* ppopenw */
 /************************************************/
 ppopena (af,s)
  FILE **af;
  char *s;
 {
  *af = fopen (s,"a");
  if (*af == NULL) {
   (void) fprintf (stderr,
    "error opening file %s for append \n\n", s);
   exit (1);
  }
 } /* ppopena */
 /************************************************/
 fileexists (s)
  char *s;
 {
  FILE *f;
  int r;

  f = fopen (s,"r");
  if (f != NULL) (void)fclose (f);
  if (f == NULL) r = 0;
  else           r = 1;

  return (r);
 } /* fileexists */
/************************************************/
real det (A,d)
 matrix A;
 int d;
{
 real w=0;
 matrix S;
 int s=1,i,j=0;

 if (d==1) w = A[0][0];
 else if (d==2) w = A[0][0]*A[1][1]-A[0][1]*A[1][0];
 else if (d==3) w =  
  -A[0][2]*A[1][1]*A[2][0] + A[0][1]*A[1][2]*A[2][0] + 
   A[0][2]*A[1][0]*A[2][1] - A[0][0]*A[1][2]*A[2][1] - 
   A[0][1]*A[1][0]*A[2][2] + A[0][0]*A[1][1]*A[2][2];
 else {
  for(i=0;i<d;i++){
   subdet (S,A,d,i,j);
   w += s * A[i][j] * det (S,d-1);
   s *= -1;
  };
 };
 return (w);
} /* det */
/**************************************************/
 outer (z,A,d)
 real A[DIM][DIM],z[DIM];
 int d;
 {
  real S[DIM][DIM];
  int s=1,i=d-1,j;

  for(j=0;j<d;j++){
   subdet (S,A,d,i,j);
   z[j] = s * det (S,d-1);
   s *= -1;
  };
 } /* outer */
/**************************************************/
 subdet (S,A,d,m,n)
 real S[DIM][DIM],
        A[DIM][DIM];
 int d,m,n;
 {
  int i,j,di,dj;

  di = 0;
  for(i=0;i<d-1;i++){
   if(i==m)di=1;
   dj=0;
   for(j=0;j<d-1;j++){
    if(j==n)dj=1;
    S[i][j] = A[i+di][j+dj];
   };
  }
 } /* subdet */
/**************************************************/
newton (x,pf,pfjac,d)
 vector x;
 int (*pf)(),(*pfjac)(),d;
{
 int i,ii,n=0,nmax = 100,nsign=1;
 vector dx,x0,y,y00;
 matrix J;
 real w,ay,ay0;
 static long ncalled = 0;
 real adetJ,k1,k2,amin,amax;
 vector Jnorms;

 if (debug) called ("newton",&ncalled);

 copy (x0,x,d);
 (*pf) (x0,y00,d);
 
 ay0 = norm (y00,d);

 if (debug) {
  writevector ("x0 = ",x0,d);
  writevector ("y0 = ",y00,d);
  wrlf ("norm y0 = ",ay0);
 };

 if (pfjac) pfjac (x0,J,d);
 else jacob (J,pf,x0,d);

 adetJ = Abs (det (J,d));
 for (i=0;i<d;i++) Jnorms[i] = norm (J[i],d);

 w = 1;
 for (i=0;i<d;i++) w *= Jnorms[i];
 k1 = adetJ / w;

 amin = Jnorms [0];
 amax = Jnorms [0];
 for (i=1;i<d;i++) {
  if (Jnorms [i] < amin) amin = Jnorms[i];
  if (Jnorms [i] > amax) amax = Jnorms[i];
 }; 
 k2 = amin / amax;

 if (debug) {
  wrlf ("adetJ = ",adetJ);
  wrlf ("k1    = ",k1);
  wrlf ("k2    = ",k2);
 };

 if (k1 > 1E-3 && k2 > 1E-3) {
  invmat (J,d);
  timesmv (dx,J,y00,d,d);
  if (debug) writematrix ("J inv = ",J,d,d);
 } 
 else {
  nsign = -1;
  ii = 0;
  w = Abs (y00[ii]);
  for (i=1;i<d;i++)
   if (Abs (y00[i]) > w) {ii=i;w=Abs (y00[i]);};

  w = 0;
  for (i=0;i<d;i++) w += J [ii][i] * J [ii][i];
  w = y00[ii] / w;
  timessv (dx,w,J [ii], d);
 };
 if (debug) writevector ("dx    = ",dx,d);

 do {
  n++;
  minusvv (x,x0,dx,d);
  (*pf) (x,y,d);
  ay = norm (y,d);
  timessv (dx,0.5,dx,d);
 } while ( ay > ay0 && ay > 1000 * EPS && n < nmax );

 if (n == nmax) er ("Newton not converging");
 if (debug){
  wrli (" n= ",n);
  writevector ("x = ",x,d);
  (void)printf("residuum = %G\n",ay);
  leave ("newton");
 };
 return (nsign*n);
} /* newton */
/**************************************************/
 jacob (J,pf,x0,d)
 real J [DIM][DIM],x0[];
 int (*pf)(),d;
 {
  int i,j;
  real dx,
         xw  [DIM],
         y00 [DIM],
         yw  [DIM];

  for(i=0;i<d;i++) xw[i] = x0[i];
  (*pf) (x0,y00,d);
  for(j=0;j<d;j++){
   dx = 1E-8 + 1E-5 *Abs(xw[j]);
   xw[j] += dx;
   (*pf) (xw,yw,d);
   xw[j] = x0[j];
   for(i=0;i<d;i++)
    J[i][j] = (yw[i]-y00[i]) / dx;
  };
 } /* jacob */
 /**************************************************/
 er (s)
 char *s;
 {
  (void) fprintf (stderr," error : %s \n",s);
  exit (1);
 } /* er */
 /************************************************/
 erp ()
 {
  er ("reading paramfile");
 } /* erp */
 /************************************************/
 grad (g,pf,x0,d)
 real g [],x0[],(*pf)();
 int d;
 {
  int j;
  real dx, xw [DIM], y00, yw;

  for(j=0;j<d;j++) xw[j] = x0[j];
  y00 = (*pf) (x0);
  for(j=0;j<d;j++){
   dx = 1E-8 + 1E-5 *Abs(xw[j]);
   xw[j] += dx;
   yw = (*pf) (xw);
   xw[j] = x0[j];
   g[j] = (yw-y00) / dx;
  };
 } /* grad */
 /**************************************************/
 difference  (x,y,d)
 real *x,*y;
 int d;
 {
  int i;

  for (i=0;i<d;i++) x[i] -= y[i];
 } /* difference */
 /***************************************************/
 real distance (x,y,d)
 real *x,*y;
 int d;
 {
  real wv [DIM];

  copy (wv,x,d);
  difference (wv,y,d);
  return (norm (wv,d));

 } /* distance */
/***************************************************/
real adistance (x,y,d)
 real *x,*y;
 int d;
{
 int i;
 real s=0;

 for (i=0;i<d;i++) s += fabs(x[i]-y[i]);
 return (s);
} /* adistance */
/***************************************************/
 in (x,x1,x2,d)
 real *x,*x1,*x2;
 int d;
 {
  int i,iw = 1;

  for (i=0; (i<d) && iw; i++)
   iw = iw && ((x[i]-x1[i])*(x[i]-x2[i]) <= 0);
  return (iw);
 } /* in */
/***************************************************/
tangent (t,x,f,d)
 real *t,*x;
 int (*f)();
 int d;
{
 real J [DIM][DIM],
        S [DIM][DIM];
 extern real det ();

 int i, sign = 1;
 static long ncalled = 0;

 if (debug) called ("tangent",&ncalled);
 jacob (J,f,x,d);
 for (i=0;i<d;i++) {
  subdet (S,J,d,d-1,i);
  t[i] = sign * det (S,d-1);
  sign *= -1;
 };
 if (debug) leave ("tangent");
} /* tangent */
 /************************************************/
eigv2 (eig,j)
 real eig[4],j[DIM][DIM];
{
 real Tr,Det,a[3];

 Tr = j[0][0] + j[1][1];
 Det = j[0][0] * j[1][1] - j[0][1] * j[1][0];
 a[0] = Det;
 a[1] = - Tr;
 a[2] = 1;
 solve2 (eig,a);
} /* eigv2 */
 /************************************************/
solve2 (x,a)
 real x[4],a[3];
{
 real d = a[1] * a[1] - 4 * a[2] * a[0];

 if (d>=0) {
  x[0] = (-a[1] + Sqrt (d)) / (2*a[2]);
  x[1] = 0;
  x[2] = (-a[1] - Sqrt (d)) / (2*a[2]);
  x[3] = 0;
 }
 else {
  x[0] = - a[1]    / (2 * a[2]);
  x[1] = Sqrt (-d) / (2 * a[2]);
  x[2] = x[0];
  x[3] = - x[1];
 };
} /* solve2 */
/************************************************/
real rotrat (T,A,f,p,d)
 real T,A;
 int (*f)();
 vector p;
 int d;
{
#define SDIM 10000
 extern real distance ();
 struct {real c[DIM];int ne;} s [SDIM];
 real y[DIM],
        epsy = 1E-3, /* to close a cycle */
        w,r,
        dist;

 int n  = 0,  /* current point    */
     ne = 0,  /* # excitations    */
     nn,      /* nearest neighbor */
     i,       /* coordinate index */
     j,       /* to find nearest neighbor */
     Ne,Np;

 setstac (y,(real)200.0,f,p,d);

 do{
  for(i=0;i<d;i++)(s[n].c)[i]=y[i];
  s[n].ne = ne;
  n++;
   
  oneyear (y,T,A,&ne,f,p,d);

  dist = distance (y,s[0].c,d);
  nn   = 0;
  for(j=1;j<n;j++){
   w = distance (y,s[j].c,d);
   if (w < dist) {dist = w; nn = j;};
  };

 }while ((dist > epsy)&&(n < SDIM));

 Ne = ne - s[nn].ne;
 Np = n  - nn;
 r  = (real) Ne / Np;
/* printf ("%G %G %d / %d = %G \n",T,A,Ne,Np,r); */
 return (r);
# undef SDIM
} /* rotrat */
/**************************************************/
setstac (y,T,f,p,d)
 real y[],T;
 int (*f)();
 vector p;
 int d;
{
 static long ncalled = 0;
 static vector gystac;
 real t=0;

 if (!ncalled) {
  ncalled = 1;
  y [0] = 5.5; 
  y [1] = 6.4; 
  integdt (&t,T,y,f,p,d);
  copy (gystac,y,d);
 }
 else copy (y,gystac,d);
} /* setstac */
/**************************************************/
perturbation (y,A)
 real y[],A;
{
 y[0] += A;
/* y[0] = pow (10.0,-(A-log10(y[0]))); */
} /* perturbation */
/**************************************************/
oneyear (y,T,A,pne,f,p,d)
 real *y,T,A;
 int (*f)();
 vector p;
 int *pne,d;
{
 real t = 0;

 seth0(1E-5);
 perturbation (y,A);
 integdte (&t,T,y,pne,f,p,d);

/* alfa = alfa1;
 integdte (T/2,y,pne);
 alfa = alfa2;
 integdte (T/2,y,pne);
*/
} /* oneyear */
 /**************************************************/
 long wc (f)
 FILE *f;
 {
  long n = 0;
  int c;

  c=getc(f);
  do{
   while((c==' ')||(c=='\t')||(c=='\n'))c=getc(f);
   if(c!=EOF)n++;
   while((c!=' ')&&(c!='\t')&&(c!='\n')&&(c!=EOF))
    c=getc(f);
  }while(c!=EOF);
  rewind (f);
  return (n);
 } /* wc */
/**************************************************/
/**************************************************/
prepranddata (x,n)
 real *x;
 long n;
{
 long i;

 for (i=0;i<n;i++) x[i] = myrand();
} /* prepranddata */
/*********************************************/
prepndata (x,n)
 real **x;
 long n;
{
 long i;

 allocdata (x,n);
 for (i=0;i<n;i++) (*x)[i] = i;
} /* prepndata */
/*********************************************/
printdata (x,n)
 real *x;
 long n;
{
 long i;
 
 for (i=0;i<n;i++) (void) printf ("%.15G\n",(double)x[i]);
} /* printdata */
/*********************************************/
sprintdata (x,n,s)
 real *x;
 long n;
 char* s;
{
 long i;
 FILE *f;

 ppopenww (&f,s);
 for(i=0;i<n;i++) (void)fprintf(f,"%.15G\n",(double)x[i]);
 (void)fclose(f);
} /* sprintdata */
/*********************************************/
sprintldata (x,n,s)
 long *x,n;
 char* s;
{
 long i;
 FILE *f;

 ppopenww (&f,s);
 for(i=0;i<n;i++) (void)fprintf(f,"%d\n",x[i]);
 (void)fclose(f);
} /* sprintidata */
/*********************************************/
print2data (x,y,n)
 real *x,*y;
 long n;
{
 long i;

 for (i=0;i<n;i++)
  (void) printf ("%.15G %.15G\n",(double)x[i],(double)y[i]);
} /* print2data */
/*********************************************/
revdata (x,n)
 real *x;
 long n;
{
 long i,j;
 real w;

 for (i=0, j = n-1; i<n/2; i++,j--){
  w    = x[i];
  x[i] = x[j];
  x[j] = w;
 };
} /* revdata */
/*********************************************/
revmatrix (A,d)
 matrix A;
 int d;
{
 int i,j;
 vector w;

 for (i=0, j = d-1; i<d/2; i++,j--){
  copy (w   ,A[i],d);
  copy (A[i],A[j],d);
  copy (A[j],w   ,d);
 };
} /* revmatrix */
/*********************************************/
logdata (x,n)
 real *x;
 long n;
{
 long i;

 for (i=0;i<n;i++) x[i] = log10 (fabs ((double)x[i]));
} /* logdata */
/*********************************************/
windowdata (x,n)
 real *x;
 long n;
{
 long i;
 real w;

 for (i=0;i<n;i++) {
  w = i / (real)n ; /* 0 ... 1 */
  w = (1-Cos(twoPi*w))/2;
  x[i] *= w;
 };
} /* windowdata  */
/*********************************************/
sortdata (x,n)
 real *x;
 long n;
{
 subsortdata (x,0L,n-1);
} /* sortdata */
/*********************************************/
subsortdata (x,l,r)
 real *x;
 long l,r;
{
 long i=l,j=r;
 real v,w;

 v = x[(l+r)/2];
 do {
  while (x[i]<v) i++;
  while (x[j]>v) j--;
  if (i<=j) {
   w = x[i];
   x[i] = x[j];
   x[j] = w;
   i++;
   j--;
  };
 } while (i<=j);
 if (l<j) subsortdata (x,l,j);
 if (i<r) subsortdata (x,i,r);
} /* subsortdata */
/*********************************************/
sortdatai (x,index,n)
 real *x,*index;
 long n;
{
 subsortdatai (x,index,0L,n-1);
} /* sortdatai */
/*********************************************/
subsortdatai (x,index,l,r)
 real *x,*index;
 long l,r;
{
 long i=l,j=r;
 real v,w;

 v = x[(l+r)/2];
 do {
  while (x[i]<v) i++;
  while (x[j]>v) j--;
  if (i<=j) {
   w = x[i];
   x[i] = x[j];
   x[j] = w;
   w = index[i];
   index[i] = index[j];
   index[j] = w;
   i++;
   j--;
  };
 } while (i<=j);
 if (l<j) subsortdatai (x,index,l,j);
 if (i<r) subsortdatai (x,index,i,r);
} /* subsortdatai */
/*********************************************/
sortdata3 (x,n,y,z)
 real *x,*y,*z;
 long n;
{
 subsortdata3 (x,0L,n-1,y,z);
} /* sortdata3 */
/*********************************************/
subsortdata3 (x,l,r,y,z)
 real *x,*y,*z;
 long l,r;
{
 long i=l,j=r;
 real v,w;

 v = x[(l+r)/2];
 do {
  while (x[i]<v) i++;
  while (x[j]>v) j--;
  if (i<=j) {
   w = x[i]; x[i] = x[j]; x[j] = w;
   w = y[i]; y[i] = y[j]; y[j] = w;
   w = z[i]; z[i] = z[j]; z[j] = w;
   i++;
   j--;
  };
 } while (i<=j);
 if (l<j) subsortdata3 (x,l,j,y,z);
 if (i<r) subsortdata3 (x,i,r,y,z);
} /* subsortdata3 */
/*********************************************/
sortdatavm (x,A,d)
 vector x;
 matrix A;
 int d;
{
 subsortdatavm (x,A,d,0,d-1);
} /* sortdatavm */
/*********************************************/
subsortdatavm (x,A,d,l,r)
 vector x;
 matrix A;
 int d,l,r;
{
 int i=l,j=r;
 real v,w;
 vector wv;

 v = x[(l+r)/2];
 do {
  while (x[i]<v) i++;
  while (x[j]>v) j--;
  if (i<=j) {
                     /*  x[i]  <->  x[j]   */
   w = x[i];
   x[i] = x[j];
   x[j] = w;
                     /*  A[i]  <->  A[j]   */
   copy (wv,A[i],d);
   copy (A[i],A[j],d);
   copy (A[j],wv,d);

   i++;
   j--;
  };
 } while (i<=j);
 if (l<j) subsortdatavm (x,A,d,l,j);
 if (i<r) subsortdatavm (x,A,d,i,r);
} /* subsortdatavm */
/*********************************************/
/*********************************************/
allocdata (x,n)
 real **x;
 long n;
{
 static long ncalled = 0;

 if (debug) called ("allocdata",&ncalled);

 *x=(real*) calloc ((size_t)n,(size_t)sizeof(real));
 if (*x==NULL) {
  (void) fprintf (stderr,
   "error in allocdata : \
     cannot allocate memory for %ld reals",n);
  exit (1);
 };

 if (debug) leave ("allocdata");
} /* allocdata */
/*********************************************/
alloclong (x,n)
 long **x;
 long n;
{
 *x=(long*) calloc ((size_t)n,(size_t)sizeof(long));
 if (*x==NULL) {
  (void) fprintf (stderr,
   "error in alloclong : cannot allocate memory for %ld longs",n);
  exit (1);
 };
} /* alloclong */
/*******************************************/
readdata (x,n,s)
 real **x;
 long *n;
 char *s;
{
 FILE *f;

 ppopenr (&f,s);
 freaddata (x,n,f);
 (void) fclose (f);
} /* readdata */
/*******************************************/
# define N 10000
freaddata (x,n,f)
 real **x;
 long *n;
 FILE *f;
{
 long i=0;
 real w;
 static long ncalled = 0;

 if (debug) called ("freaddata",&ncalled);

 *n = N;
 if (debug) (void) fprintf (stderr,"n = %ld\n",*n);
 allocdata (x,*n);

 while (fscanf(f,"%lf",&w)==1) {

  if (debug) (void) fprintf (stderr,
   "i = %8ld  w = %G \n",i, w);

  if (i==*n) *x = (real*) realloc 
   ((void*)*x, (size_t) ((*n+=N) * sizeof(real)) );
 
  (*x)[i++] = w;
 };
 if (i < *n) {
  *n = i;
  *x = (real*) realloc 
   ((void*)*x, (size_t) ((*n) * sizeof(real)) );
 };
 (void) fclose (f);

 if (*n == 0) er ("empty input file");

 if (debug) leave ("freaddata");
} /* freaddata */
# undef N
/*******************************************/
 minmax (x,dl,pxmin,pxmax)
 real *x,*pxmin,*pxmax;
 long dl;
 {
  long i;

  *pxmin = x[0];
  *pxmax = x[0];
  for(i=1;i<dl;i++){
   if (x[i] < *pxmin) *pxmin = x[i];
   if (x[i] > *pxmax) *pxmax = x[i];
  };
 } /* minmax */
/*******************************************/
sfiltrdata (x,n)
 real *x;
 long n;
{
 real v = x[0], w;
 long i;

 for (i=1;i<n-1;i++) {
  w = (v + x[i] + x[i+1]) / 3;
  v = x[i];
  x[i] = w;
 };
} /* sfiltrdata */
/*******************************************/
s2filtrdata (x,y,n)
 real *x,*y;
 long n;
{
 real v = y[0],
  x1,x2,x3,xs,yl,y2,y3,z1,z2,z3,a,b,w;
 long i;

 for (i=1;i<n-1;i++) {
  x1 = x [i-1];
  x2 = x [i  ];
  x3 = x [i+1];
  yl = v;
  y2 = y [i  ];
  y3 = y [i+1];
  xs = (x1 + x2 + x3) / 3;
  z1 = x1 - xs;
  z2 = x2 - xs;
  z3 = x3 - xs;
  a  = (z1 * yl + z2 * y2 + z3 * y3) / 
       (z1 * z1 + z2 * z2 + z3 * z3);
  b = (yl + y2 + y3) / 3;
  w = a * z2 + b;
  v = y[i];
  y[i] = w;
 };
} /* s2filtrdata */
/*******************************************/
filtrdata (x,y,n,m)
 real *x,*y;
 long n,m;
{
 real *a, w=0, c = 0.4 * pow ((double)m,1.9);
 long i,j,k;
 static long ncalled = 0;

 if (debug) called ("filtrdata",&ncalled);

 allocdata (&a,m+1);
 for (i=0;i<=m;i++) a[i] = exp ((double)(-i*i/c));
 for (i=-m;i<=m;i++) w += a[labs(i)];
 for (i=0;i<=m;i++) a[i] /= w;

 if (debug) 
  for (i=0;i<=m;i++)
   (void)fprintf (stderr,"%G\n",a[i]);

 for (i=0;i<n;i++) {
  w = 0;
  for (j=-m;j<=m;j++) {
   k = i+j;
   if (k < 0  ) k = 0;
   if (k > n-1) k = n-1;
   w += a [labs(j)] * x[k];
  };
  y[i] = w;
 };
 free ((void*)a);
 if (debug) leave ("filtrdata");
} /* filtrdata */
/*******************************************/
standardize (x,n)
 real *x;
 long n;
{
 real m=0,ms=0,msd,rmsd;
 long i;

 for (i=0;i<n;i++) {
  m  += x[i];
  ms += x[i] * x[i];
 }; 
 m   /= n;          /* mean                       */
 ms  /= n;          /* mean square                */
 msd  = ms - m*m;   /* mean square deviation      */
 rmsd = Sqrt (msd); /* root mean square deviation */

 for (i=0;i<n;i++) x[i] = (x[i]-m)/rmsd;

} /* standardize */
/*******************************************/
autocor (x,y,n)
 real *x,*y;
 long n;
{
 long i,j;
 real w;

 for (i=0;i<n/2;i++) {
  w = 0;
  for (j=0; j < n-i; j++) w += x[j] * x[j+i];
  w /= n-i; 
  y[i] = w; 
 };
} /* autocor */ 
/*******************************************/
accumdata (x,n)
 real *x;
 long n;
{
 long i;
 real w=0;

 for (i=0;i<n;i++) {
  w += x[i];
  x[i] = w;
 };
} /* accumdata */
/*******************************************/
accumpdata (x,n,p)
 real *x,p;
 long n;
{
 long i;
 real w=0;

 for (i=0;i<n;i++) {
  w += pow ((double)x[i],(double)p);
  x[i] = w;
 };
} /* accumpdata */
/*******************************************/
/*******************************************/
Round (wx)
 real  wx;
{
 int y;

 if (wx >= 0) y = (int) (wx + 0.5);
 else         y = (int) (wx - 0.5);

 return (y);
} /* Round */
/*******************************************/
long Roundl (wx)
 real  wx;
{
 long y;

 if (wx >= 0) y = (long) (wx + 0.5);
 else         y = (long) (wx - 0.5);

 return (y);
} /* Roundl */
/*******************************************/
real Log2 (wx) real wx; {return (Log(wx)/Log(2));}
/*******************************************/
sqrtc (y,x)
 real x[2],y[2];
{
 real r  = Sqrt (Sqrt ((x[0]*x[0]+x[1]*x[1]))),
      fi = atan2 ((double)x[1],(double)x[0])/2;
 y[0] = r * Cos (fi);
 y[1] = r * Sin (fi);
}
/*******************************************/
sqrc (y,x)
 real x[2],y[2];
{
 real w = x[0]*x[0]-x[1]*x[1];
 y[1] = 2 * x[0] * x[1];
 y[0] = w;
}
/*******************************************/
rk6 (t,x,h,f)
/* local error = O(h^6) */
 real *t,*x,h,(*f)();
{
# define a1    0.25
# define b11   0.25
# define a2    0.25
# define b21   0.125
# define b22   0.125
# define a3    0.5
# define b31   0.0
# define b32  -0.5
# define b33   1.0
# define a4    0.75
# define b41   0.1875
# define b42   0.0
# define b43   0.0
# define b44   0.5625
# define a5    1.0
# define b51  -3.0/7.0
# define b52   2.0/7.0
# define b53  12.0/7.0
# define b54 -12.0/7.0
# define b55   8.0/7.0
# define cc   90.0
# define c1    7.0
# define c2    0.0
# define c3   32.0
# define c4   12.0
# define c5   32.0
# define c6    7.0
 
 real k1,k2,k3,k4,k5,k6;
 k1 = h * f (*t,     *x);
 k2 = h * f (*t+a1*h,*x+b11*k1);
 k3 = h * f (*t+a2*h,*x+b21*k1+b22*k2);
 k4 = h * f (*t+a3*h,*x+b31*k1+b32*k2+b33*k3);
 k5 = h * f (*t+a4*h,*x+b41*k1+b42*k2+b43*k3+b44*k4);
 k6 = h * f (*t+a5*h,*x+b51*k1+b52*k2+b53*k3+b54*k4+b55*k5);
 *x += (c1*k1+c2*k2+c3*k3+c4*k4+c5*k5+c6*k6) / cc;
 *t += h;

# undef a1
# undef b11
# undef a2
# undef b21
# undef b22
# undef a3
# undef b31
# undef b32
# undef b33
# undef a4
# undef b4
# undef b42
# undef b43
# undef b44
# undef a5
# undef b51
# undef b52
# undef b53
# undef b54
# undef b55
# undef cc
# undef c1
# undef c2
# undef c3
# undef c4
# undef c5
# undef c6
} /*  rk6 */
/*********************************************/
perpen (a,b,x,d)
 real *a,*b,*x;
 int d;
{
 real v[DIM];
 real w = Abs (x[0]);
 int i,m=0;

 for (i=1;i<d;i++)
  if (Abs(x[i]) < w) {
   m = i;
   w = Abs (x[i]);
  };
 for (i=0;i<d;i++) v[i] = 0;
 v[m] = 1;
 vectprod (a,v,x,d);
 vectprod (b,a,x,d);
 normalize (a,d);
 normalize (b,d);
} /* perpen */
/*********************************************/
vectprod (x,a,b,d)
 real *x,*a,*b;
 int d;
{
 if (d!=3) er ("dim <> 3 in vectprod");
 x[0] = a[1]*b[2] - a[2]*b[1];
 x[1] = a[2]*b[0] - a[0]*b[2];
 x[2] = a[0]*b[1] - a[1]*b[0];
} /* vectprod */
/*********************************************/
real lnf (x) /* log factorial */
 real x;
{
 real w = 0;

 while (x>1)  w += Log(x--);
 return (w);
} /* lnf */
/*********************************************/
real lnfs (x) /* log factorial ala Stirling */
 real x;
/* 
    n   error 
    1    0
    2    0.04
   10    0.07
 10^7    0.08
*/
{
 if (x<1) 
  return (0.0);
 else 
  return (x*Log(x) - x + 0.5*Log(x) + 1);
} /* lnfs */
/*********************************************/

# ifdef HIGH_PRECISION

/*********************************************/
long double ldfrac (x)
long double x;
{ return (x-floor((double)x));}
/*************************************/
long double ldsin (x)
long double x;
{
 long double result;
 int p;
 extern long double ldfrac();
 extern long double ldrsin();
 extern long double ldrcos();
 
 x = twoPi * ldfrac (x/twoPi);
 p = (int) (4 * x / Pi);

 switch (p) {
  case 0 : result = ldrsin (x);        break;
  case 1 : result = ldrcos (PiHalf-x); break;
  case 2 : result = ldrcos (x-PiHalf); break;
  case 3 : result = ldrsin (Pi-x);     break;
  case 4 : result =-ldrsin (x-Pi);     break;
  case 5 : result =-ldrcos (Pi3_2-x);  break;
  case 6 : result =-ldrcos (x-Pi3_2);  break;
  case 7 : result =-ldrsin (twoPi-x);  break;
 };
 return (result);
} /* ldsin */
/*************************************/
long double ldcos (x)
long double x;
{
 long double result;
 int p;
 extern long double ldfrac();
 extern long double ldrsin();
 extern long double ldrcos();
 
 x = twoPi * ldfrac (x/twoPi);
 p = (int) (4 * x / Pi);

 switch (p) {
  case 0: result = ldrcos (x);        break;
  case 1: result = ldrsin (PiHalf-x); break;
  case 2: result =-ldrsin (x-PiHalf); break;
  case 3: result =-ldrcos (Pi-x);     break;
  case 4: result =-ldrcos (x-Pi);     break;
  case 5: result =-ldrsin (Pi3_2-x);  break;
  case 6: result = ldrsin (x-Pi3_2);  break;
  case 7: result = ldrcos (twoPi-x);  break;
 };
 return (result);
} /* ldcos */
/*************************************/
long double ldrsin (x)
long double x;
{
 int n = 2;
 long double q = x * x,
             a = x,
             s = a;
 do {
  a *= q / n / (n+1); s -= a; n+= 2;
  a *= q / n / (n+1); s += a; n+= 2;
 } while (a > ldeps);
 return (s);
} /* ldrsin */
/*************************************/
long double ldrcos (x)
long double x;
{
 int n = 1;
 long double q = x * x,
             a = 1,
             s = a;
 do {
  a *= q / n / (n+1); s -= a; n+= 2;
  a *= q / n / (n+1); s += a; n+= 2;
 } while (a > ldeps);
 return (s);
} /* ldrcos */
/*************************************/
long double ldsqrt (x)
long double x;
{
 long double y = sqrt ((double)x);
 return ((y+x/y)/2);
} /* ldsqrt */
/*************************************/
long double ldSin (x)
long double x;
{
 long double result;
 int p;
 extern long double ldfrac();
 extern long double ldmsin();
 extern long double ldmcos();
 
 x = twoPi * ldfrac (x/twoPi);
 p = (int) (4 * x / Pi);

 switch (p) {
  case 0 : result = ldmsin (x);        break;
  case 1 : result = ldmcos (PiHalf-x); break;
  case 2 : result = ldmcos (x-PiHalf); break;
  case 3 : result = ldmsin (Pi-x);     break;
  case 4 : result =-ldmsin (x-Pi);     break;
  case 5 : result =-ldmcos (Pi3_2-x);  break;
  case 6 : result =-ldmcos (x-Pi3_2);  break;
  case 7 : result =-ldmsin (twoPi-x);  break;
 };
 return (result);
} /* ldSin */
/*************************************/
long double ldCos (x)
long double x;
{
 long double result;
 int p;
 extern long double ldfrac();
 extern long double ldmsin();
 extern long double ldmcos();
 
 x = twoPi * ldfrac (x/twoPi);
 p = (int) (4 * x / Pi);

 switch (p) {
  case 0: result = ldmcos (x);        break;
  case 1: result = ldmsin (PiHalf-x); break;
  case 2: result = ldmsin (x-PiHalf); break;
  case 3: result = ldmcos (Pi-x);     break;
  case 4: result = ldmcos (x-Pi);     break;
  case 5: result = ldmsin (Pi3_2-x);  break;
  case 6: result = ldmsin (x-Pi3_2);  break;
  case 7: result = ldmcos (twoPi-x);  break;
 };
 return (result);
} /* ldCos */
/*************************************/
long double ldmsin (v)
 long double v;
{ return (   0.0L 
   + v * (   1.0000000000000000000000000000014969422553L
   + v * ( - 3.0753090525658862328276038068746432165638E-28L
   + v * ( - 0.1666666666666666666666666444130624110775L
   + v * ( - 8.2183682211363006305699370999438974749279E-25L
   + v * (   0.0083333333333333333333515878642613724527L
   + v * ( - 2.6763582353725325354151118770490586227841E-22L
   + v * ( - 0.0001984126984126984099511885497321328819L
   + v * ( - 2.0539724813299525149331974120227134242916E-20L
   + v * (   2.7557319223987039973026373844162083339977E-6L
   + v * ( - 4.9027610011883610269670595138158626794292E-19L
   + v * ( - 2.5052108383828584168686412338554905798622E-8L
   + v * ( - 4.1181180080527276750007730252357619244739E-18L
   + v * ( + 1.6059044652799793873073251478754898600218E-10L
   + v * ( - 1.2477111281397993887853615749630408741498E-17L
   + v * ( - 7.6470184663500111891550489996709795684464E-13L
   + v * ( - 1.2562265321511996734510153144313468872043E-17L
   + v * (   2.8191756951445983281881220625103568515828E-15L
   + v * ( - 3.0872648822844969560074110458425104667095E-18L
   + v * ( - 7.5808594872384426843178059430192804235541E-18L
   ))))))))))))))))))));
} /* ldmsin */
/*************************************/
long double ldmcos (v)
 long double v;
{ return (   1.0L
   + v * (   3.8141589308706127892482895395392602961106E-30L
   + v * ( - 0.5000000000000000000000000007841124870749L
   + v * (   5.6811210203820346053029881907699596219625E-26L
   + v * (   0.0416666666666666666666645647021610480067L
   + v * (   4.6806066371539432478057434733023489784591E-23L
   + v * ( - 0.0013888888888888888895773634056693376707L
   + v * (   7.0960983155075070488138819611080350866462E-21L
   + v * (   0.0000248015873015872482611445421801843918L
   + v * (   3.0028974580766492097552903296198319207695E-19L
   + v * ( - 2.7557319224115001162681672537210947768027E-7L
   + v * (   4.2901640710174387334132098788191879710501E-18L
   + v * (   2.0876756876968156928088977580415949919368E-9L
   + v * (   2.2331975668036383786210402901152278144429E-17L
   + v * ( - 1.1470780485851844018990931907324617635166E-11L
   + v * (   4.1837997258345421644472503466560658425415E-17L
   + v * (   4.7757000391701367704939705078746211813607E-14L
   + v * (   2.4844033745203587751286030030690857241006E-17L
   + v * ( - 1.6743745529348078486393762777832279729211E-16L
   + v * (   3.1400948140589100279540916711707589022011E-18L
   ))))))))))))))))))));
} /* ldmcos */
/*************************************/
ldwrite(x)
 long double x;
{
 int f = floor ((double)x);
 int n,i=0,im=35;

 if (x<0) {(void)printf ("-");x=-x;}
 else      (void)printf (" ");
 (void)printf ("%d.",f);
 x = x - f;

 while ((x > 0) && (i++ < im)) {
  x *= 10;
  n = (int) x;
  (void)printf ("%d",n);
  x -= n;
 };

 (void)printf ("\n");

} /* ldwrite */
/*************************************/
long double ldabs (x)
 long double x;
{
 if (x < 0) return (-x);
 else       return ( x);
} /* ldabs */
/*************************************/
long double ldlog (x)
 long double x;
{
 return ((long double)log((double)x));
} /* ldlog */
/*************************************/

# endif  /*  HIGH_PRECISION  */

/*************************************/
rktesttab (to,yo,f,p,d)
 real to;
 vector yo;
 int (*f)();
 vector p;
 int d;
{
 real h1,h2,da1,da2,db1,db2,dc1,dc2,q = 1.2,ra,rb,rc;
 int i,im=28;

 (void)printf ("               grk <->            rkr <->             Richardson \n");
 (void)printf ("  h           d      r           d       r            d       r\n\n");

 h1 = 0.2;
 rktest (to,yo,h1,&da1,&db1,&dc1,f,p,d);
 
 for (i=0; i<im && da1>0 && db1>0 && dc1>0; i++) {
  h2  = h1;
  da2 = da1;
  db2 = db1;
  dc2 = dc1;

  h1 = h1 / q;  
  rktest (to,yo,h1,&da1,&db1,&dc1,f,p,d);

  if (da1>0 && db1>0 && dc1>0) {
   ra = Log (da1/da2) / Log (h1/h2);
   rb = Log (db1/db2) / Log (h1/h2);
   rc = Log (dc1/dc2) / Log (h1/h2);

   (void)printf (
   "%-8.3G  %-8.3G  %-8.3G  %-8.3G  %-8.3G  %-8.3G  %-8.3G\n",
     (double)h1,
     (double)da1,(double)ra,
     (double)db1,(double)rb,
     (double)dc1,(double)rc);
  };
 };
} /* rktesttab */
/*******************************************/
rktest (to,yo,h,ea,eb,ec,f,p,d)
 real to;
 vector yo;
 real h,*ea,*eb,*ec;
 int (*f)();
 vector p;
 int d;

{
 vector y,yoo,dy;
 extern real norm();
 real r;
 static long ncalled = 0;

 if (debug) called ("rktest",&ncalled);

 grk (to  ,yo,y  , h,f,p,d);
 grk (to+h,y ,yoo,-h,f,p,d);
 minusvv (dy,yoo,yo,d);
/*
writevector ("yoo",yoo,d);
writevector ("yo ",yo ,d);
writevector ("dy ",dy ,d);
*/
*ea = norm (dy,d);

 rkr (to  ,yo,y  ,ec, h,f,p,d);
 rkr (to+h,y ,yoo,&r,-h,f,p,d);
 minusvv (dy,yoo,yo,d);
 *eb = norm (dy,d);
 if (debug) leave ("rktest");
} /* rktest */
/*******************************************/
prk (t0,x0,h0,eps0, x,ah,aeps,f,p,d)
 /*  pars like in rk  */
 real t0,x0[DIM],h0,eps0;
 real x[DIM],*ah,*aeps;
 int (*f)();
 vector p;
 int d;
{
 real  eps,h;
 static long ncalled = 0;

 if (debug) called ("prk",&ncalled);

 rkr (t0,x0,x,&eps,h0,f,p,d);

 if (eps > 0) h = h0 * exp (log((double)(eps0/eps))/7);
 else         h = 2  * h0;
 *ah   = h;
 *aeps = eps;
 if (debug) leave ("prk");
}  /*  prk */
/**************************************************/
rkr (x,yo,y,e,h,f,p,d) /* Runge-Kutta Richardson */
 real x;
 vector yo,y;
 real *e,h;
 vector p;
 int (*f)(),d;
{
 vector yl,y2,y3;
 int i;
 real w;
 static long ncalled = 0;

 if (debug) called ("rkr",&ncalled);
 grk (x    ,yo,yl,h  ,f,p,d);
 grk (x    ,yo,y2,h/2,f,p,d);
 grk (x+h/2,y2,y3,h/2,f,p,d);

 w = 0;
 for (i=0;i<d;i++) {
  y[i] = (64*y3[i]-yl[i]) / 63;
  w += (yl[i]-y3[i]) * (yl[i]-y3[i]);
 };
 *e = Sqrt (w);
 if (debug) leave ("rkr");
} /* rkr */
/*******************************************/
grk (xo,yo,y,h,f,p,d) /* general Runge-Kutta */
 real xo;
 vector yo,y;
 real h;
 vector p;
 int (*f)(),d;
/* Hut Runge-Kutta method 
   Vitasek
   order = 6
   local error = O(h^7)
*/
{
# define S 8
 static real 
  a [S] = {0,1,1,1,1,2,5,1},
  aa[S] = {1,9,6,3,2,3,6,1},
  b [S][S] = {{   1},
              {   1,    3},
              {   1,   -3,   4},
              {  -5,   27, -24,   6},
              { 221, -981, 867,-102,   1},
              {-183,  678,-472, -66,  80, 3},
              { 716,-2079,1002, 834,-454,-9, 72},
              {  41,    0, 216,  27, 272,27,216, 41}},
  bb[S] =     {   9,   24,   6,   8,   9,48, 82,840};
 real x,w,k[S][S];
 int s,n,i;
 vector g;
 matrix J;
 static long ncalled = 0;

 if (debug) called ("grk",&ncalled); 
 for (i=0;i<d;i++) y[i] = yo[i];

 for (s=0;s<S;s++) {
  x = xo + h * a[s]/aa[s];
  f (x,y,p, k[s],g,J, d);
   
  for (i=0;i<d;i++) {
   w = 0; for (n=0;n<=s;n++) w += b[s][n]*k[n][i];
   y[i] = yo[i] + h * w / bb[s];
  };
 };
 if (debug) leave ("grk");
# undef S
} /* grk */
/*******************************************/
/*******************************************/
orthonormalizen (D,e,n,d)
 matrix D;
 vector e;
 int n,d;
{
 real s;
 vector v;
 int j,k;

 if (debug) (void) fprintf (stderr,"orthonormalizen called\n");
/*
writematrix ("D = ",D,d,d);
(void) printf ("n = %d\n",n);
(void) printf ("d = %d\n",d);
*/
 for (j=0;j<n;j++) {
  for (k=0;k<j;k++) {
   s = scalprod (D[j], D[k], d);
   timessv (v,s,D[k],d);
   minusvv (D[j],D[j],v,d);
  };
  e[j] = norm (D[j],d);
  normalize (D[j],d);
 };
 if (debug) (void) fprintf (stderr,"orthonormalizen O.K.\n");
/*
writevector ("e = ",e,n);
*/
} /* orthonormalizen */
/*********************************************/
real cosfi (a,b,d)
 vector a,b;
 int d;
{
 real A=0,B=0,AB=0;
 int i;

 for (i=0;i<d;i++) {
  A  += a[i]*a[i]; 
  B  += b[i]*b[i];
  AB += a[i]*b[i];
 };
 A = Sqrt (A);
 B = Sqrt (B);
 return (AB/A/B);
} /* cosfi */
/*********************************************/
v2M (M,v,n,d)
 matrix M;
 vector v;
 int n,d;
{
 int i;
 
 for (i=0;i<n;i++) copy (M[i],v+d*(i+1),d);
} /* v2M */
/*********************************************/
M2v (v,M,n,d)
 vector v;
 matrix M;
 int n,d;
{
 int i;
 
 for (i=0;i<n;i++) copy (v+d*(i+1),M[i],d);
} /* M2v */
/*********************************************/
LE (x,le,F,p,T,TT,o,steps,nle,d)
 vector x,le,p;
 int (*F)(); 
 real T,TT;
 int o,steps,nle,d;
{
 real t = 0;
 vector e,s;
 int i, dim = d+nle*d;
 matrix D;

 if (dim > DIM) {
  (void) fprintf (stderr,
   "sorry, dim = %1d > maxdim = %1d\n", dim,DIM);
  exit (0);
 };

 for (i=0;i<nle;i++) s[i] = 0;
 for (i=d;i<DIM;i++) x[i] = 0;
 for (i=d;i<dim;i++) x[i] = myrand();

 v2M (D,x,nle,d);
 orthonormalizen (D,e,nle,d);
 M2v (x,D,nle,d);

 seth0 (1E-6);
 integdt (&t,TT,x,F,p,d);
/*------------------------------------------------*/
 while (t < T + TT) {

  for (i=0;i<steps;i++) integ1h (&t,x,F,p,dim);

  v2M (D,x,nle,d);
  orthonormalizen (D,e,nle,d);
  M2v (x,D,nle,d);
  for (i=0;i<nle;i++) s[i] += Log (e[i]);

  if (o==1) {
   (void)printf("%G ",t);
   for (i=0;i<nle;i++)
    (void)printf("%G ",s[i]);
   (void)printf("\n");
  };

  if (o==2) {
   (void)printf("%G ",t);
   for (i=0;i<dim;i++)
    (void)printf("%G ",x[i]);
   for (i=0;i<nle;i++)
    (void)printf("%G ",s[i]);
   (void)printf("\n");
  };

 }; /* t */
/*------------------------------------------------*/
 for (i=0;i<nle;i++) le[i] = s [i] / (t-TT);
} /* LE */
/*********************************************/
LEd (x,le,F,p,iT,iTT,o,steps,nle,d)
 vector x,le,p;
 int (*F)(); 
 int  iT,iTT;
 int o,steps,nle,d;
{
 int it = 0;
 vector e,s;
 int i, dim = d+nle*d;
 matrix D;

 if (debug) (void) fprintf (stderr,"LEd called\n");

 if (dim > DIM) {
  (void) fprintf (stderr,
   "sorry, dim = %1d > maxdim = %1d\n", dim,DIM);
  exit (0);
 };

 for (i=0;i<nle;i++) s[i] = 0;
 for (i=d;i<DIM;i++) x[i] = 0;
 for (i=d;i<dim;i++) x[i] = myrand();

 v2M (D,x,nle,d);
 orthonormalizen (D,e,nle,d);
 M2v (x,D,nle,d);

 if (debug){
  (void) fprintf(stderr,"iTT=%d\n",iTT);
  writevector ("x = ",x,d);
  writevector ("p = ",p,d);
  (void) fprintf(stderr,"d =%d\n",d );
 }; 
for (it=0;it<iTT;it++) F(x,p,d); 
/*------------------------------------------------*/
 for (it=0;it<iT;it++) {
/*
(void) printf ("iTT = %d\n",iTT);
(void) printf ("dim = %d\n",dim);
writevector ("x = ",x,dim);
writevector ("p = ",p,4);
*/
  for (i=0;i<steps;i++) F(x,p,dim); 
/*
writevector ("x = ",x,dim);
*/
  v2M (D,x,nle,d);
  orthonormalizen (D,e,nle,d);
  M2v (x,D,nle,d);

  for (i=0;i<nle;i++) s[i] += Log (e[i]);

 if (o==1) {
   (void)printf("%d ",it);
   for (i=0;i<nle;i++)
    (void)printf("%G ",s[i]);
   (void)printf("\n");
  };

  if (o==2) {
   (void)printf("%d ",it);
   for (i=0;i<dim;i++)
    (void)printf("%G ",x[i]);
   for (i=0;i<nle;i++)
    (void)printf("%G ",s[i]);
   (void)printf("\n");
  };

 }; /* it */
/*------------------------------------------------*/
 for (i=0;i<nle;i++) le[i] = s [i] / it;
 if (debug) (void) fprintf (stderr,"LEd O.K.\n");
} /* LEd */
/*********************************************/
/*********************************************/
binprintshort (b)
short b;
{
 int i;

 for (i=0;i<16;i++) {
  (void) printf ("%c",'0'+!!(b << i & 0x8000));
  if (i%8 == 7) (void) printf (" ");
 };
} /* binprint */
/**************************************/
binprintdouble (x)
 double x;
{
 int i;
 short* ps;

 for (i=0;i<4;i++) {
  ps = (short*) &x;
  ps = ps + i;
  binprintshort (*ps);
 };
 (void) printf ("\n");
} /* binprintdouble */
/**************************************/
/**************************************/
ilog2 (x)
 double x;
{
 return (
  (int) (* (short*) &x  >> 4   &   0x07FF) - 0x03FF
 );
} /* ilog2 */
/**************************************/
# ifdef HIGH_PRECISION
ildlog2 (x)
 long double x;
{
 return (
  (int) (* (short*) &x         &   0x7FFF) - 0x3FFF
 );
} /* ildlog2 */
# endif
/************************************************/
/************************************************/
long two2n (n)
 long n;
{
 long nn,nnn;
 extern long Roundl();
 static long ncalled = 0;

 if (debug) called ("two2n",&ncalled);
 nn  = Roundl (Log2 ((real)n));
 nnn = Roundl ((real)pow (2.0,(double)nn));
 if (n < nnn) nn--;
 n   = Roundl ((real)pow (2.0,(double)nn));
 if (debug) leave ("two2n");
 return (n);
} /* two2n */
/************************************************/
long bitinv (n,size,columns)
 long n,size,columns;
{
 long r,i,b;

 r = size / 2;
 b = 0;
 for (i=0;i<columns;i++) {
  if (n & 1L) b += r;
  n /= 2;
  r /= 2;
 };
 return (b);
} /* bitinv */
/************************************************/
reorder(x,size,columns)
 real *x;
 long size,columns;
{
 long i,b;
 real w;
 extern long bitinv ();

 for (i=0;i<size;i++) {
  b = bitinv (i,size,columns);
  if (b>i) {
   w = x[i];
   x[i] = x[b];
   x[b] = w;
  };
 };
} /* reorder */
/************************************************/
TrigTable (csn,size)
 real *csn;
 long size;
{
 long i;
 real angle, omega;

   angle = 0;
   omega = 2*Pi/size;
   for (i=0;i<=size/4;i++) {
     csn[i] = Cos (angle);
     angle += omega;
   };
} /* TrigTable */
/************************************************/
fht (x,size)  /* 128 k ... 4.75 s */
 real *x;
 long size;
{
 long c,columns,g,groups,b,n,n2,n4,org,i1,i2,i3,i4,i;
 real cm,sm,z1,z2,w1,w2,w3,w4,yl,y2,scale;
 real *csn;

 allocdata (&csn,size/4+1);
 columns = Round (Log2 ((real)size));
 reorder (x,size,columns);
 TrigTable (csn,size);
 
  groups = size;
  for (c=1;c<=columns;c++) {
    groups /= 2;
    n = size / groups;
    n2 = n / 2;
    n4 = n / 4;
    for (g = 0; g < groups; g++) {
      org = g * n;
      yl = x[org] + x[org + n2];
      y2 = x[org] - x[org + n2];
      x[org] = yl;
      x[org + n2] = y2;
      for (b = 1; b <= n4; b++) {
	i1 = org + b;
	i2 = i1 + n2;
	i3 = org + n - b;
	i4 = i3 - n2;
	cm = csn[b * groups];
	sm = csn[size / 4 - b * groups];
	z1 = cm * x[i2] + sm * x[i3];
	z2 = cm * x[i3] - sm * x[i2];
	w1 = x[i1] + z1;
	w2 = x[i1] - z1;
	w3 = x[i4] + z2;
	w4 = x[i4] - z2;
	x[i1] = w1;
	x[i2] = w2;
	x[i3] = w3;
	x[i4] = w4;
      }  /* of butterfly */
    }  /* of group     */
  }  /* of column    */
  scale = Sqrt(size);
  for(i=0;i<size;i++) x[i] /= scale;
  free ((void*)csn);
}  /* fht */
/************************************************/
MakeSpectrum (spectrum,x,size)
 real *spectrum,*x;
 long size;
{
 long i;

  spectrum[0] = x[0]*x[0];
  for (i=1;i<=size/2;i++)
   spectrum [i] = x[i]*x[i] + x[size-i]*x[size-i];
  i = size / 2;
  spectrum[i] = spectrum[i] / 2;

  for (i = 0; i <=  size / 2; i++)
   spectrum [i] /= size;
} /* MakeSpectrum */
/************************************************/
/************************************************/
lookline (T,mgn,pos,i,d)
 matrix T;
 vector mgn;
 int pos[DIM], i, d;
{
 int j;

 if (i < d-1) {
  mgn[i] = Abs (T[i][i+1]);
  pos[i] = i+1;
  for (j=i+2;j<d;j++)
   if ( Abs (T[i][j]) > mgn[i] ) {
    mgn[i] = Abs (T[i][j]);
    pos[i] = j;
   };
 };
} /* lookline */
/***********************************/
checkline (T,mgn,pos,i,p,q)
 matrix T;
 vector mgn;
 int pos[DIM], i, p, q;
{
 if ( Abs (T[i][p]) > mgn[i] ) {
  mgn[i] = Abs (T[i][p]);
  pos[i] = p;
 };
 if ( Abs (T[i][q]) > mgn[i] ) {
  mgn[i] = Abs (T[i][q]);
  pos[i] = q;
 };
} /* checkline */
/***********************************/
firstlook (T,mgn,pos,d)
 matrix T;
 vector mgn;
 int pos[DIM], d;
{
 int i;

 for (i=0;i<d-1;i++)
  lookline (T,mgn,pos,i,d);
} /* firstlook */
/***********************************/
look (T,mgn,pos,p,q,nar,d)
 matrix T;
 vector mgn;
 int pos[DIM], *p, *q, *nar, d;
{
 real w = 0, wT = Abs (T[0][0]);
 int i;

 for (i=0;i<d-1;i++)
  if (mgn[i] > w) {
   w  = mgn [i];
   *p = i;
   *q = pos[i];
  };

 for (i=1;i<d;i++)
  if (Abs (T[i][i]) > wT) wT = Abs (T[i][i]);

 *nar = w / wT < EPS;
 if (debug) (void) printf ("w = %G\n",(double) w);
} /* look */
/***********************************/
diagiter (T, R, mgn, pos, nar, pp, pq, d)
 matrix T,R;
 vector mgn;
 int pos[DIM], *nar, *pp, *pq, d; 
{
 matrix Rt,RT,RTRt;
 real lambda,mi,ni,c,cc,s,ss,cs,w1,w2;
 int j, p = *pp, q = *pq;
 
 if (debug) {
  (void) printf ("------------------------\n");
  writematrix ("R = ",R,d,d);
  writematrix ("T = ",T,d,d);
  (void) printf ("p = %d  q = %d \n",p,q);
  writevector ("mgn = ",mgn,d-1);
  for (j=0;j<d-1;j++) (void) printf ("%10d ",pos[j]);
  (void) printf ("\n\n\n");
 };

 lambda = -T[p][q];
 mi = (T[p][p] - T[q][q]) / 2;
 ni = Sqrt (lambda*lambda+mi*mi);
 c  = Sqrt ( (Abs(mi) / ni + 1) / 2 );
 s  = lambda / 2 / ni / c;
 if (mi < 0) s = -s;
 cc = c * c;
 ss = s * s;
 cs = c * s;
 for (j=0;j<d;j++)
  if ( j != p && j != q ) {
   w1 = T [j] [p] * c - T [j] [q] * s;
   w2 = T [j] [p] * s + T [j] [q] * c;
   T [p] [j] = w1;
   T [q] [j] = w2;
   T [j] [p] = w1;
   T [j] [q] = w2;
  };

 w1 = T [p][p]*cc + T [q][q]*ss - 2 * T [p][q]*cs;
 w2 = T [p][p]*ss + T [q][q]*cc + 2 * T [p][q]*cs;
 T[p][p] = w1;
 T[q][q] = w2;

 T [p] [q] = 0;
 T [q] [p] = 0;

 for (j=0;j<d;j++) {
  w1 = R [j] [p] * c - R [j] [q] * s;
  w2 = R [j] [p] * s + R [j] [q] * c;
  R [j] [p] = w1;
  R [j] [q] = w2;
 };

 for (j=0;j<d-1;j++)
  if (j==p || j==q || pos[j]==p || pos[j]==q)
   lookline (T,mgn,pos,j,d);
  else 
   checkline (T,mgn,pos,j,p,q);
 look (T,mgn,pos,pp,pq,nar,d);

 if (debug) {
  transpose (Rt,R,d);
  timesmm (RT,R,T,d,d,d);
  timesmm (RTRt,RT,Rt,d,d,d);
  writematrix ("R     = ", R   ,d,d);
  writematrix ("T     = ", T   ,d,d);
  writematrix ("RTRt  = ", RTRt,d,d);
  (void) printf ("p = %d  q = %d \n",*pp,*pq);
  (void) printf ("------------------------\n");
 };

} /* diagiter */
/***********************************/
diagonalize (A,Q,s,d)
 matrix A,Q;
 vector s;
 int d;
{
 int pos[DIM], p, q, narrow;
 vector mgn;
 matrix T, R;

/*  A must be a symmetric matrix       */
/*  1 -> R -> Qt                       */
/*  A -> T -> D -> s                   */
/*  rows of Q are eigenvectors of A    */
/*  elements of s are eigenvalues of A */
/*  A = Qt.D.Q                         */
/*  50x50 in 4 seconds                 */

 make1matrix (R,d);
 copymatrix (T,A,d,d);
 firstlook (T,mgn,pos,d);
 look (T,mgn,pos,&p,&q,&narrow,d);
 while (!narrow)
  diagiter (T,R, mgn, pos, &narrow, &p, &q, d);

 transpose (Q,R,d);
 extractdiag (s,T,d);

 sortdatavm (s,Q,d);
 revdata (s,(long)d);
 revmatrix (Q,d);

 sqrtvector (s,d);
 sumnormvector (s,d);
} /* diagonalize */
/***********************************/
/***********************************/
lsm (x,y,n,c,d,e,r,L)
 real *x,*y; /* input data   */
 long n;
 vector c;   /* coefficients */
 real *d;    /* deviations   */ 
 real *e;    /* rmsd         */
 real *r;    /* correlation  */
 int L;
{
 vector P;
 matrix R;
 long i;
 int j,k,m;
 real w,v;
 extern real corcoef ();

 for (k=0;k<L;k++) {
  w = 0;
  for (i=0;i<n;i++) {
   v = y[i];
   for (m=0;m<k;m++) v *= x[i];
   w += v;
  };
  P[k] = w / n;
 };

 for (j=0;j<2*L-1;j++) {
  w = 0;
  for (i=0;i<n;i++) {
   v = 1;
   for (m=0;m<j;m++) v *= x[i];
   w += v;
  };
  for (k=0;k<=j;k++)
   if (k<L && (j-k)<L)
    R[k][j-k] = w / n;
 };

 invmat (R,L);
 timesmv (c,R,P,L,L);

 *e = 0;
 for (i=0;i<n;i++) {
  w = 0;
  for (k=0;k<L;k++) {
   v = c[k];
   for (m=0;m<k;m++) v *= x[i];
   w += v;
  };
  d[i] = y[i] - w;
  *e += d[i] * d[i];
 };
 *e = Sqrt ((double)*e);
 *r = corcoef (x,y,n);
} /* lsm */
/************************************************/
real corcoef (x,y,n)
 real *x,*y;
 long n;
{
 real w;
 long i;

 standardize (x,n);
 standardize (y,n);

 w = 0;
 for (i=0;i<n;i++) w += x[i] * y[i];

 return (w/n);
} /* corcoef */
/************************************************/
real Nad (a,b)
 long a,b;
{
 real r = 1;

 if (a<0 || b<0 || b>a) return ((real)0);
 if (2*b > a) b = a-b;
 while (b) r *= (a--)/(real)(b--);
 return (r);
} /* Nad */
/**********************************************/
real Bezier1 (t,a,N)
 real t,*a;
 long N;
{
 long i;
 real s;

/* here N is the number if input points */

 if (N==0) er ("Bezier curve from 0 points ?");
 if (N==1) return (a[0]);

 N--;
/* here N is the index of the last input point */

 if (t==0) return (a[0]);
 if (t==1) return (a[N]);

 s = a[0] * pow (1-t,(double)N) +
     a[N] * pow (  t,(double)N);

 for (i=1;i<N;i++)
  s += Nad (N,i) * a[i] * pow (  t,(double)   i ) * 
                          pow (1-t,(double)(N-i));
 return (s);
} /* Bezier1 */
/**********************************************/ 
/**********************************************/
curve (x,x1,x2,t,noutmax,h,hmin,hmax,f,fout,p,d)
 vector x,x1,x2,t,p;
 int noutmax,(*f)(),(*fout)(),d;
 real h,hmin,hmax;
{
 int ncor,nout=0,niter=0,nitermax=10*noutmax;
 vector x0,dx,xprev;

 copy (xprev,x,d);
 (void) curvecorrector (x,t,f,p,d);

 do {

  if(niter==0 || distance(x,xprev,d) > hmin) {
   fout (x,p,f,d);
   nout++;
   copy (xprev,x,d);
  };

  copy (x0,x,d);
  normalize (t,d);
  timessv (dx,h,t,d);
  plusvv (x,x0,dx,d);

  if (in(x,x1,x2,d)) {
   ncor = curvecorrector (x,t,f,p,d);
   if (ncor < 2) h*= 2;
   if (ncor > 2) h/= 2;

   if (h > hmax) h = hmax;

   minusvv (t,x,x0,d);
  };
 } while (in(x,x1,x2,d) && niter++<nitermax && nout<noutmax);
} /* curve */
/*******************************************/
curvecorrector (x0,t,f,p,d)
 vector x0,t,p;
 int (*f)(),d;
{
 int n,N=10,m=0,M=10;
 real eps = 1E-11, v1,v2,tim=0;
 vector x1,x2,dx,yl,y2,g;
 matrix J;

 copy (x1,x0,d);
 do {

  f (tim,x1,p, yl,g,J,d);
  yl [d-1] = scalprod (x1,t,d) - scalprod (x0,t,d);
  v1 = norm (yl,d);
  copy (J[d-1],t,d);

  if (debug) {
   writevector ("x1 = ",x1,d);
   writevector ("p  = ",p ,d);
   writevector ("yl = ",yl,d);
   (void) printf ("v1  = %G\n",v1 );
   writematrix ("J  = ",J,d,d);
  };

  invmat (J,d);
  timesmv (dx,J,yl,d,d);

  if (debug) {
   writematrix ("Jinv = ",J,d,d);
   writevector ("dx = ",dx,d);
  };

  n=0;
  do {
   minusvv (x2,x1,dx,d);
   f (tim,x2,p, y2,g,J,d);
   y2 [d-1] = scalprod (x2,t,d) - scalprod (x0,t,d);
   v2 = norm (y2,d);
   timessv (dx,0.5,dx,d);
   n++;
  } while ( v2>v1 && v2>eps && n<N );
  if (n==N) {
   (void) printf ("\n");
   (void) printf ("eps = %G\n",eps);
   (void) printf ("v1  = %G\n",v1 );
   (void) printf ("v2  = %G\n",v2 );
   writevector ("p  = ",p ,6);
   writevector ("x1 = ",x1,d);
   writevector ("x2 = ",x2,d);
   writevector ("dx = ",dx,d);
   writevector ("y2 = ",y2,d);
   (void) printf ("n   = %d\n",n  );
   (void) printf ("N   = %d\n",N  );
   (void) printf ("m   = %d\n",m  );
   (void) printf ("M   = %d\n",M  );
   er ("\ncurvecorrector 1 not converging\n");
  };
  copy (x1,x2,d);

  m++;
 } while (v2 > eps && m<M);
 if (m==M) {
   (void) printf ("\n");
   (void) printf ("eps = %G\n",eps);
   (void) printf ("v1  = %G\n",v1 );
   (void) printf ("v2  = %G\n",v2 );
   writevector ("p  = ",p ,6);
   writevector ("x1 = ",x1,d);
   writevector ("x2 = ",x2,d);
   writevector ("dx = ",dx,d);
   writevector ("y2 = ",y2,d);
   (void) printf ("n   = %d\n",n  );
   (void) printf ("N   = %d\n",N  );
   (void) printf ("m   = %d\n",m  );
   (void) printf ("M   = %d\n",M  );
   er ("\ncurvecorrector 2 not converging\n");
 };
 copy (x0,x1,d);
 return (m);
} /* curvecorrector */
/*******************************************/
verbq_conti (t,eig)
 int *t;
 vector eig;
{
 real r1 = eig[0],
      i1 = eig[1],
      r2 = eig[2];
 
 if (i1 != 0) {
  if (r1 < 0) *t = 0;           /* SF */
  else        *t = 1;           /* UF */
 } else {
  if      (r1 < 0 && r2 < 0) *t = 2;  /* SN */
  else if (r1 > 0 && r2 > 0) *t = 3;  /* UN */
  else                       *t = 4;  /* S  */
 };
} /* verbq_conti */
/*******************************************/
verbq_discr (t,eig)
 int *t;
 vector eig;
{
 real r1 = fabs(eig[0]),
      i1 =      eig[1],
      r2 = fabs(eig[2]);
 
 if (i1 != 0) {
  if (r1 < 1) *t = 0;           /* SF */
  else        *t = 1;           /* UF */
 } else {
  if      (r1 < 1 && r2 < 1) *t = 2;  /* SN */
  else if (r1 > 1 && r2 > 1) *t = 3;  /* UN */
  else                       *t = 4;  /* S  */
 };
} /* verbq_discr */
/*******************************************/
real defint (f,a,b,p,n,m)
 real (*f)(),a,b;
 vector p;
 int n, /* number of divisions  */
   m;   /* method: 1 = trapezoid; global error = O(h^2)
                   2 = Simpson                       4
                   3 = cubic                         4
                   4 = bi-quadratic                  6  */
{
 real
  h = (b-a)/n/m,
  s = f(a,p),
  x = a + h;
 int i;

 for (i=1;i<n;i++) {
  switch (m) {
   case 1: s += 2       * f(x,p); x += h; break;
   case 2: s += 4       * f(x,p); x += h;
           s += 2       * f(x,p); x += h; break;
   case 3: s += 3       * f(x,p); x += h;
           s += 3       * f(x,p); x += h;
           s += 2       * f(x,p); x += h; break;
   case 4: s += 64.0/14 * f(x,p); x += h;
           s += 24.0/14 * f(x,p); x += h;
           s += 64.0/14 * f(x,p); x += h;
           s += 2       * f(x,p); x += h; break;
  };
 };
 switch (m) {
  case 2: s += 4       * f(x,p);         break;
  case 3: s += 3       * f(x,p); x += h;
          s += 3       * f(x,p);         break;
  case 4: s += 64.0/14 * f(x,p); x += h;
          s += 24.0/14 * f(x,p); x += h;
          s += 64.0/14 * f(x,p);         break;
 };
 s += f(b,p);
 switch (m) {
  case 1: s *=  1.0/2  * h; break;
  case 2: s *=  1.0/3  * h; break;
  case 3: s *=  3.0/8  * h; break;
  case 4: s *= 14.0/45 * h; break; 
  default: er("illegal def int method");
 };
 return (s);
} /* defint */
/******************************/
/*
real f (x) real x; { return (1/x);}
real F (a,b) real a,b; { return (log(b/a));}
*/
/******************************/
definttest (f,F)
 real (*f)(), (*F)();
{
 real a=2.48,b=4.31,d,dd,r;
 int n,n1=1,n2=5000,q=2,m;
 vector p;

 for (m=1;m<5;m++) {
  dd = defint (f,a,b,p,n1,m) - F(a,b);
  for (n=n1*q;n<n2;n*=q) {
   d  = defint (f,a,b,p,n,m) - F(a,b);
   r  = log(fabs(dd/d)) / log ((double)q);
   dd = d;
   (void) printf ("%6d %10.3G %10.3f\n",n,d,r);
  };
  (void) printf ("\n");
 };
} /* definttest */
/***********************************************/
findcycle (x,n,i1,i2,f,p,dim)
 vectors x;
 int n,*i1,*i2,(*f)(),dim;
 vector p;
{
 int per=0,i=0,j;

 while (i++, per==0 && i<n) {
  copy (x[i],x[i-1],dim);
  f(x[i],p,dim);
  j=i;
  while (per==0 && j>0)
   if (j--, adistance(x[i],x[j],dim)<1e-10) per = i-j;
 };
 *i1 = j;
 *i2 = i-1;
} /* findcycle */
/***********************************************/
